import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _regeneratorRuntime from "@babel/runtime/regenerator";
import { bind } from 'bind-event-listener';
import noop from '@atlaskit/ds-lib/noop';
import { getBooleanFF } from '@atlaskit/platform-feature-flags';
import { COLOR_MODE_ATTRIBUTE, CUSTOM_THEME_ATTRIBUTE, THEME_DATA_ATTRIBUTE } from './constants';
import { themeIdsWithOverrides } from './theme-config';
import { isValidBrandHex } from './utils/color-utils';
import { findMissingCustomStyleElements } from './utils/custom-theme-loading-utils';
import { hash } from './utils/hash';
import { loadAndAppendThemeCss, loadThemeCss } from './utils/theme-loading';
import { themeObjectToString } from './utils/theme-state-transformer';

// Represents theme state once mounted to the page (auto is hidden from observers)

var defaultColorMode = 'light';
var isMatchMediaAvailable = typeof window !== 'undefined' && 'matchMedia' in window;
var darkModeMediaQuery = '(prefers-color-scheme: dark)';
var darkModeMql = isMatchMediaAvailable && window.matchMedia(darkModeMediaQuery);
var unbindThemeChangeListener = noop;
export var themeStateDefaults = {
  colorMode: 'auto',
  dark: 'dark',
  light: 'light',
  shape: undefined,
  spacing: undefined,
  typography: undefined,
  UNSAFE_themeOptions: undefined
};

/**
 * Updates the current theme when the system theme changes. Should be bound
 * to an event listener listening on the '(prefers-color-scheme: dark)' query
 * @param e The event representing a change in system theme.
 */
var checkNativeListener = function checkNativeListener(e) {
  var element = document.documentElement;
  element.setAttribute(COLOR_MODE_ATTRIBUTE, e.matches ? 'dark' : 'light');
};
var getThemePreferences = function getThemePreferences(themeState) {
  var colorMode = themeState.colorMode,
    dark = themeState.dark,
    light = themeState.light,
    shape = themeState.shape,
    spacing = themeState.spacing,
    typography = themeState.typography;
  var themePreferences = colorMode === 'auto' ? [light, dark] : [themeState[colorMode]];
  [shape, spacing, typography].forEach(function (themeId) {
    if (themeId) {
      themePreferences.push(themeId);
    }
  });
  if (getBooleanFF('platform.design-system-team.border-checkbox_nyoiu')) {
    themePreferences.push("".concat(themePreferences.includes('dark') ? 'dark' : 'light', "-new-input-border"));
  }

  // Load shape and spacing by default, currently behind a feature flag
  if (getBooleanFF('platform.design-system-team.space-and-shape-tokens_q5me6')) {
    if (!themePreferences.includes('spacing')) {
      themePreferences.push('spacing');
    }
  }
  return _toConsumableArray(new Set(themePreferences));
};

/**
 * Sets the theme globally at runtime. This updates the `data-theme` and `data-color-mode` attributes on your page's <html> tag.
 *
 * @param {Object<string, string>} themeState The themes and color mode that should be applied.
 * @param {string} themeState.colorMode Determines which color theme is applied. If set to `auto`, the theme applied will be determined by the OS setting.
 * @param {string} themeState.dark The color theme to be applied when the color mode resolves to 'dark'.
 * @param {string} themeState.light The color theme to be applied when the color mode resolves to 'light'.
 * @param {string} themeState.shape The shape theme to be applied.
 * @param {string} themeState.spacing The spacing theme to be applied.
 * @param {string} themeState.typography The typography theme to be applied.
 * @param {Object} themeState.UNSAFE_themeOptions The custom branding options to be used for custom theme generation
 * @param {function} themeLoader Callback function used to override the default theme loading functionality.
 *
 * @returns A Promise of an unbind function, that can be used to stop listening for changes to system theme.
 *
 * @example
 * ```
 * setGlobalTheme({colorMode: 'auto', light: 'light', dark: 'dark', spacing: 'spacing'});
 * ```
 */
var setGlobalTheme = /*#__PURE__*/function () {
  var _ref = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee3() {
    var _ref2,
      _ref2$colorMode,
      colorMode,
      _ref2$dark,
      dark,
      _ref2$light,
      light,
      _ref2$shape,
      shape,
      _ref2$spacing,
      spacing,
      _ref2$typography,
      typography,
      _ref2$UNSAFE_themeOpt,
      UNSAFE_themeOptions,
      themeLoader,
      themePreferences,
      loadingStrategy,
      themeAttributes,
      _args3 = arguments;
    return _regeneratorRuntime.wrap(function _callee3$(_context3) {
      while (1) switch (_context3.prev = _context3.next) {
        case 0:
          _ref2 = _args3.length > 0 && _args3[0] !== undefined ? _args3[0] : {}, _ref2$colorMode = _ref2.colorMode, colorMode = _ref2$colorMode === void 0 ? themeStateDefaults['colorMode'] : _ref2$colorMode, _ref2$dark = _ref2.dark, dark = _ref2$dark === void 0 ? themeStateDefaults['dark'] : _ref2$dark, _ref2$light = _ref2.light, light = _ref2$light === void 0 ? themeStateDefaults['light'] : _ref2$light, _ref2$shape = _ref2.shape, shape = _ref2$shape === void 0 ? themeStateDefaults['shape'] : _ref2$shape, _ref2$spacing = _ref2.spacing, spacing = _ref2$spacing === void 0 ? themeStateDefaults['spacing'] : _ref2$spacing, _ref2$typography = _ref2.typography, typography = _ref2$typography === void 0 ? themeStateDefaults['typography'] : _ref2$typography, _ref2$UNSAFE_themeOpt = _ref2.UNSAFE_themeOptions, UNSAFE_themeOptions = _ref2$UNSAFE_themeOpt === void 0 ? themeStateDefaults['UNSAFE_themeOptions'] : _ref2$UNSAFE_themeOpt;
          themeLoader = _args3.length > 1 ? _args3[1] : undefined;
          themePreferences = getThemePreferences({
            colorMode: colorMode,
            dark: dark,
            light: light,
            shape: shape,
            spacing: spacing,
            typography: typography
          });
          loadingStrategy = themeLoader ? themeLoader : loadAndAppendThemeCss;
          _context3.next = 6;
          return Promise.all([].concat(_toConsumableArray(themePreferences.map( /*#__PURE__*/function () {
            var _ref3 = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee(themeId) {
              return _regeneratorRuntime.wrap(function _callee$(_context) {
                while (1) switch (_context.prev = _context.next) {
                  case 0:
                    _context.next = 2;
                    return loadingStrategy(themeId);
                  case 2:
                    return _context.abrupt("return", _context.sent);
                  case 3:
                  case "end":
                    return _context.stop();
                }
              }, _callee);
            }));
            return function (_x) {
              return _ref3.apply(this, arguments);
            };
          }())), [_asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee2() {
            var mode, attrOfMissingCustomStyles, _yield$import, loadAndAppendCustomThemeCss;
            return _regeneratorRuntime.wrap(function _callee2$(_context2) {
              while (1) switch (_context2.prev = _context2.next) {
                case 0:
                  if (!(!themeLoader && UNSAFE_themeOptions && isValidBrandHex(UNSAFE_themeOptions === null || UNSAFE_themeOptions === void 0 ? void 0 : UNSAFE_themeOptions.brandColor))) {
                    _context2.next = 11;
                    break;
                  }
                  mode = colorMode || themeStateDefaults['colorMode'];
                  attrOfMissingCustomStyles = findMissingCustomStyleElements(UNSAFE_themeOptions, mode);
                  if (!(attrOfMissingCustomStyles.length === 0)) {
                    _context2.next = 5;
                    break;
                  }
                  return _context2.abrupt("return", false);
                case 5:
                  _context2.next = 7;
                  return import( /* webpackChunkName: "@atlaskit-internal_atlassian-custom-theme" */
                  './custom-theme');
                case 7:
                  _yield$import = _context2.sent;
                  loadAndAppendCustomThemeCss = _yield$import.loadAndAppendCustomThemeCss;
                  _context2.next = 11;
                  return loadAndAppendCustomThemeCss({
                    colorMode: attrOfMissingCustomStyles.length === 2 ? 'auto' :
                    // only load the missing custom theme styles
                    attrOfMissingCustomStyles[0],
                    UNSAFE_themeOptions: UNSAFE_themeOptions
                  });
                case 11:
                case "end":
                  return _context2.stop();
              }
            }, _callee2);
          }))()]));
        case 6:
          if (colorMode === 'auto' && darkModeMql) {
            colorMode = darkModeMql.matches ? 'dark' : 'light';
            // Add an event listener for changes to the system theme.
            // If the function exists, it will not be added again.
            unbindThemeChangeListener = bind(darkModeMql, {
              type: 'change',
              listener: checkNativeListener
            });
          } else {
            unbindThemeChangeListener();
          }
          themeAttributes = getThemeHtmlAttrs({
            colorMode: colorMode,
            dark: dark,
            light: light,
            shape: shape,
            spacing: spacing,
            typography: typography,
            UNSAFE_themeOptions: themeLoader ? undefined : UNSAFE_themeOptions
          });
          Object.entries(themeAttributes).forEach(function (_ref5) {
            var _ref6 = _slicedToArray(_ref5, 2),
              key = _ref6[0],
              value = _ref6[1];
            document.documentElement.setAttribute(key, value);
          });
          return _context3.abrupt("return", unbindThemeChangeListener);
        case 10:
        case "end":
          return _context3.stop();
      }
    }, _callee3);
  }));
  return function setGlobalTheme() {
    return _ref.apply(this, arguments);
  };
}();
/**
 * Takes an object containing theme preferences, and returns an array of objects for use in applying styles to the document head.
 * Only supplies the color themes necessary for initial render, based on the current themeState. I.e. if in light mode, dark mode themes are not returned.
 *
 * @param {Object<string, string>} themeState The themes and color mode that should be applied.
 * @param {string} themeState.colorMode Determines which color theme is applied. If set to `auto`, the theme applied will be determined by the OS setting.
 * @param {string} themeState.dark The color theme to be applied when the color mode resolves to 'dark'.
 * @param {string} themeState.light The color theme to be applied when the color mode resolves to 'light'.
 * @param {string} themeState.shape The shape theme to be applied.
 * @param {string} themeState.spacing The spacing theme to be applied.
 * @param {string} themeState.typography The typography theme to be applied.
 * @param {Object} themeState.UNSAFE_themeOptions The custom branding options to be used for custom theme generation
 *
 * @returns A Promise of an object array, containing theme IDs, data-attributes to attach to the theme, and the theme CSS.
 * If an error is encountered while loading themes, the themes array will be empty.
 */
export var getThemeStyles = /*#__PURE__*/function () {
  var _ref7 = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee6(preferences) {
    var themePreferences, results;
    return _regeneratorRuntime.wrap(function _callee6$(_context6) {
      while (1) switch (_context6.prev = _context6.next) {
        case 0:
          if (preferences === 'all') {
            themePreferences = themeIdsWithOverrides;
          } else {
            themePreferences = getThemePreferences({
              colorMode: (preferences === null || preferences === void 0 ? void 0 : preferences.colorMode) || themeStateDefaults['colorMode'],
              dark: (preferences === null || preferences === void 0 ? void 0 : preferences.dark) || themeStateDefaults['dark'],
              light: (preferences === null || preferences === void 0 ? void 0 : preferences.light) || themeStateDefaults['light'],
              shape: (preferences === null || preferences === void 0 ? void 0 : preferences.shape) || themeStateDefaults['shape'],
              spacing: (preferences === null || preferences === void 0 ? void 0 : preferences.spacing) || themeStateDefaults['spacing'],
              typography: (preferences === null || preferences === void 0 ? void 0 : preferences.typography) || themeStateDefaults['typography']
            });
          }
          _context6.next = 3;
          return Promise.all([].concat(_toConsumableArray(themePreferences.map( /*#__PURE__*/function () {
            var _ref8 = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee4(themeId) {
              var css;
              return _regeneratorRuntime.wrap(function _callee4$(_context4) {
                while (1) switch (_context4.prev = _context4.next) {
                  case 0:
                    _context4.prev = 0;
                    _context4.next = 3;
                    return loadThemeCss(themeId);
                  case 3:
                    css = _context4.sent;
                    return _context4.abrupt("return", {
                      id: themeId,
                      attrs: {
                        'data-theme': themeId
                      },
                      css: css
                    });
                  case 7:
                    _context4.prev = 7;
                    _context4.t0 = _context4["catch"](0);
                    return _context4.abrupt("return", undefined);
                  case 10:
                  case "end":
                    return _context4.stop();
                }
              }, _callee4, null, [[0, 7]]);
            }));
            return function (_x3) {
              return _ref8.apply(this, arguments);
            };
          }())), [
          // Add custom themes if they're present
          _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee5() {
            var _preferences$UNSAFE_t;
            var _yield$import2, getCustomThemeStyles, customThemeStyles;
            return _regeneratorRuntime.wrap(function _callee5$(_context5) {
              while (1) switch (_context5.prev = _context5.next) {
                case 0:
                  if (!(preferences !== 'all' && preferences !== null && preferences !== void 0 && preferences.UNSAFE_themeOptions && isValidBrandHex(preferences === null || preferences === void 0 ? void 0 : (_preferences$UNSAFE_t = preferences.UNSAFE_themeOptions) === null || _preferences$UNSAFE_t === void 0 ? void 0 : _preferences$UNSAFE_t.brandColor))) {
                    _context5.next = 15;
                    break;
                  }
                  _context5.prev = 1;
                  _context5.next = 4;
                  return import( /* webpackChunkName: "@atlaskit-internal_atlassian-custom-theme" */
                  './custom-theme');
                case 4:
                  _yield$import2 = _context5.sent;
                  getCustomThemeStyles = _yield$import2.getCustomThemeStyles;
                  _context5.next = 8;
                  return getCustomThemeStyles({
                    colorMode: (preferences === null || preferences === void 0 ? void 0 : preferences.colorMode) || themeStateDefaults['colorMode'],
                    UNSAFE_themeOptions: preferences === null || preferences === void 0 ? void 0 : preferences.UNSAFE_themeOptions
                  });
                case 8:
                  customThemeStyles = _context5.sent;
                  return _context5.abrupt("return", customThemeStyles);
                case 12:
                  _context5.prev = 12;
                  _context5.t0 = _context5["catch"](1);
                  return _context5.abrupt("return", undefined);
                case 15:
                case "end":
                  return _context5.stop();
              }
            }, _callee5, null, [[1, 12]]);
          }))()]));
        case 3:
          results = _context6.sent;
          return _context6.abrupt("return", results.flat().filter(function (theme) {
            return theme !== undefined;
          }));
        case 5:
        case "end":
          return _context6.stop();
      }
    }, _callee6);
  }));
  return function getThemeStyles(_x2) {
    return _ref7.apply(this, arguments);
  };
}();

/**
 * Server-side rendering utility. Generates the valid HTML attributes for a given theme.
 * Note: this utility does not handle automatic theme switching.
 *
 * @param {Object<string, string>} themeOptions - Theme options object
 * @param {string} themeState.colorMode Determines which color theme is applied. If set to `auto`, the theme applied will be determined by the OS setting.
 * @param {string} themeState.dark The color theme to be applied when the color mode resolves to 'dark'.
 * @param {string} themeState.light The color theme to be applied when the color mode resolves to 'light'.
 * @param {string} themeState.spacing The spacing theme to be applied.
 * @param {string} themeState.typography The typography theme to be applied.
 * @param {Object} themeState.UNSAFE_themeOptions The custom branding options to be used for custom theme generation
 *
 * @returns {Object} Object of HTML attributes to be applied to the document root
 */
export var getThemeHtmlAttrs = function getThemeHtmlAttrs() {
  var _result;
  var _ref10 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
    _ref10$colorMode = _ref10.colorMode,
    colorMode = _ref10$colorMode === void 0 ? themeStateDefaults['colorMode'] : _ref10$colorMode,
    _ref10$dark = _ref10.dark,
    dark = _ref10$dark === void 0 ? themeStateDefaults['dark'] : _ref10$dark,
    _ref10$light = _ref10.light,
    light = _ref10$light === void 0 ? themeStateDefaults['light'] : _ref10$light,
    _ref10$shape = _ref10.shape,
    shape = _ref10$shape === void 0 ? themeStateDefaults['shape'] : _ref10$shape,
    _ref10$spacing = _ref10.spacing,
    spacing = _ref10$spacing === void 0 ? themeStateDefaults['spacing'] : _ref10$spacing,
    _ref10$typography = _ref10.typography,
    typography = _ref10$typography === void 0 ? themeStateDefaults['typography'] : _ref10$typography,
    _ref10$UNSAFE_themeOp = _ref10.UNSAFE_themeOptions,
    UNSAFE_themeOptions = _ref10$UNSAFE_themeOp === void 0 ? themeStateDefaults['UNSAFE_themeOptions'] : _ref10$UNSAFE_themeOp;
  var themePreferences = {
    dark: dark,
    light: light,
    shape: shape,
    spacing: spacing,
    typography: typography
  };

  // Load spacing by default, currently behind a feature flag
  if (getBooleanFF('platform.design-system-team.space-and-shape-tokens_q5me6')) {
    themePreferences = {
      dark: dark,
      light: light,
      shape: shape,
      spacing: 'spacing',
      typography: typography
    };
  }
  var themeAttribute = themeObjectToString(themePreferences);
  var result = (_result = {}, _defineProperty(_result, THEME_DATA_ATTRIBUTE, themeAttribute), _defineProperty(_result, COLOR_MODE_ATTRIBUTE, colorMode === 'auto' ? defaultColorMode : colorMode), _result);
  if (UNSAFE_themeOptions && isValidBrandHex(UNSAFE_themeOptions.brandColor)) {
    var optionString = JSON.stringify(UNSAFE_themeOptions);
    var uniqueId = hash(optionString);
    result[CUSTOM_THEME_ATTRIBUTE] = uniqueId;
  }
  return result;
};

/**
 * Provides a script that, when executed before paint, sets the `data-color-mode` attribute based on the current system theme,
 * to enable SSR support for automatic theme switching, avoid a flash of un-themed content on first paint.
 *
 * @param {string} colorMode Determines which color theme is applied. If set to `auto`, the theme applied will be determined by the OS setting.
 *
 * @returns {string} A string to be added to the innerHTML of a script tag in the document head
 */
export var getSSRAutoScript = function getSSRAutoScript(colorMode) {
  return colorMode === 'auto' ? "(\n  () => {\n    try {\n      const mql = window.matchMedia('".concat(darkModeMediaQuery, "');\n      const colorMode = mql.matches ? 'dark' : 'light';\n      document.documentElement.setAttribute('").concat(COLOR_MODE_ATTRIBUTE, "', colorMode);\n    } catch (e) {}\n  }\n)()") : undefined;
};
export default setGlobalTheme;