/**
 * This file contains the source of truth for themes and all associated meta data.
 */
/**
 * Themes: The internal identifier of a theme.
 * These ids are what the actual theme files/folders are called.
 * style-dictionary will attempt to locate these in the file-system.
 */
export type Themes = 'atlassian-light' | 'atlassian-light-future' | 'atlassian-light-increased-contrast' | 'atlassian-dark' | 'atlassian-dark-future' | 'atlassian-dark-increased-contrast' | 'atlassian-legacy-light' | 'atlassian-legacy-dark' | 'atlassian-shape' | 'atlassian-spacing' | 'atlassian-typography-minor3' | 'atlassian-typography-adg3';
export type ThemeFileNames = Themes;
/**
 * ThemeOverrides: The internal identifier of a theme override. Which are themes that contain
 * a subset of tokens intended to override an existing theme. These ids are what the actual
 * theme files/folders are called. style-dictionary will attempt to locate these in the file-system.
 * Theme overrides are temporary and there may not be any defined at times.
 */
export type ThemeOverrides = 'atlassian-light-new-input-border' | 'atlassian-dark-new-input-border';
/**
 * Theme kinds: The type of theme.
 * Some themes are entirely focused on Color, whilst others are purely focused on spacing.
 * In the future other types may be introduced such as typography.
 */
export type ThemeKinds = 'color' | 'spacing' | 'typography' | 'shape';
/**
 * Theme modes: The general purpose of a theme.
 * This attr is used to apply the appropriate system-preference option
 * It may also be used as a selector for mode-specific overrides such as light/dark images.
 * The idea is there may exist many color themes, but every theme must either fit into light or dark.
 */
export declare const themeColorModes: readonly ["light", "dark", "auto"];
export type ThemeColorModes = (typeof themeColorModes)[number];
export type DataColorModes = Exclude<ThemeColorModes, 'auto'>;
/**
 * Contrast preferences: The system contrast preference
 */
export declare const themeContrastModes: readonly ["more", "no-preference", "auto"];
export type ThemeContrastModes = (typeof themeContrastModes)[number];
export type DataContrastModes = 'more' | 'no-preference' | 'auto';
/**
 * Theme ids: The value that will be mounted to the DOM as a data attr
 * For example: `data-theme="light:light dark:dark spacing:spacing"
 *
 * These ids must be kebab case
 */
export declare const themeIds: readonly ["light-increased-contrast", "light", "light-future", "dark", "dark-future", "dark-increased-contrast", "legacy-light", "legacy-dark", "spacing", "shape", "typography-adg3", "typography-minor3"];
export type ThemeIds = (typeof themeIds)[number];
/**
 * Theme override ids: the equivalent of themeIds for theme overrides.
 * Theme overrides are temporary and there may not be any defined at times.
 */
export declare const themeOverrideIds: readonly ["light-new-input-border", "dark-new-input-border"];
export type ThemeOverrideIds = (typeof themeOverrideIds)[number];
export declare const themeIdsWithOverrides: readonly ["light-increased-contrast", "light", "light-future", "dark", "dark-future", "dark-increased-contrast", "legacy-light", "legacy-dark", "spacing", "shape", "typography-adg3", "typography-minor3", "light-new-input-border", "dark-new-input-border"];
export type ThemeIdsWithOverrides = (typeof themeIdsWithOverrides)[number];
/**
 * Theme to use a base. This will create the theme as
 * an extension with all token values marked as optional
 * to allow tokens to be overridden as required.
 */
export type ExtensionThemeId = ThemeIds;
/**
 * Palettes: The set of base tokens a given theme may be populated with.
 * For example: legacy light & dark themes use the "legacyPalette" containing colors from our
 * previous color set.
 */
export type Palettes = 'defaultPalette' | 'legacyPalette' | 'spacingScale' | 'shapePalette' | 'typographyPalette';
/**
 * ThemeConfig: the source of truth for all theme meta-data.
 * This object should be used whenever interfacing with themes.
 */
interface ThemeConfig {
    id: ThemeIds | ThemeOverrideIds;
    displayName: string;
    palette: Palettes;
    attributes: ({
        type: 'color';
        mode: DataColorModes;
    } | {
        type: Extract<ThemeKinds, 'spacing' | 'typography' | 'shape'>;
    }) & {
        /**
         * @deprecated Use top-level `extends` property instead
         */
        extends?: ExtensionThemeId;
    };
    /**
     * Theme to use a base. This will create the theme as
     * an extension with all token values marked as optional
     * to allow tokens to be overridden as required.
     */
    extends?: ThemeIds;
    /**
     * Theme to override. This will cause the theme to only
     * output css variables which can be imported to temporarily
     * override existing themes for testing purposes.
     */
    override?: ThemeIds;
    /**
     * Use when the theme provides an increased contrast
     * version of another theme. This will cause the theme to be loaded
     * under the media query `@media (prefers-contrast: more)`.
     */
    increasesContrastFor?: ThemeIds;
}
declare const themeConfig: Record<Themes | ThemeOverrides, ThemeConfig>;
type HEX = `#${string}`;
export type CSSColor = HEX;
/**
 * ThemeOptionsSchema: additional configuration options used to customize Atlassian's themes
 */
export interface ThemeOptionsSchema {
    brandColor: CSSColor;
}
/**
 * ThemeState: the standard representation of an app's current theme and preferences
 */
export interface ThemeState {
    light: Extract<ThemeIds, 'light' | 'light-future' | 'dark' | 'dark-future' | 'legacy-dark' | 'legacy-light' | 'light-increased-contrast' | 'dark-increased-contrast'>;
    dark: Extract<ThemeIds, 'light' | 'light-future' | 'dark' | 'dark-future' | 'legacy-dark' | 'legacy-light' | 'light-increased-contrast' | 'dark-increased-contrast'>;
    colorMode: ThemeColorModes;
    contrastMode: ThemeContrastModes;
    shape?: Extract<ThemeIds, 'shape'>;
    spacing: Extract<ThemeIds, 'spacing'>;
    typography?: Extract<ThemeIds, 'typography' | 'typography-adg3' | 'typography-minor3'>;
    UNSAFE_themeOptions?: ThemeOptionsSchema;
}
/**
 * themeStateDefaults: the default values for ThemeState used by theming utilities
 */
export declare const themeStateDefaults: ThemeState;
/**
 * Represents theme state once mounted to the page
 * (the page doesn't have an "auto" color mode, it's either light or dark)
 */
export interface ActiveThemeState extends ThemeState {
    colorMode: DataColorModes;
}
export default themeConfig;
