import _extends from "@babel/runtime/helpers/extends";
/** @jsx jsx */

import React, { useCallback, useEffect, useRef, useState } from 'react';
import { jsx } from '@emotion/react';
import { bind } from 'bind-event-listener';
import { usePlatformLeafSyntheticEventHandler } from '@atlaskit/analytics-next';
import noop from '@atlaskit/ds-lib/noop';
import useCloseOnEscapePress from '@atlaskit/ds-lib/use-close-on-escape-press';
import { ExitingPersistence, FadeIn } from '@atlaskit/motion';
import { mediumDurationMs } from '@atlaskit/motion/durations';
import { Popper } from '@atlaskit/popper';
import Portal from '@atlaskit/portal';
import { layers } from '@atlaskit/theme/constants';
import { show } from './internal/tooltip-manager';
import useUniqueId from './internal/use-unique-id';
import TooltipContainer from './TooltipContainer';
import { getMousePosition } from './utilities';
const tooltipZIndex = layers.tooltip();
const analyticsAttributes = {
  componentName: 'tooltip',
  packageName: "@atlaskit/tooltip",
  packageVersion: "17.8.4"
};

// Inverts motion direction
const invertedDirection = {
  top: 'bottom',
  bottom: 'top',
  left: 'right',
  right: 'left'
};

/**
 * Converts a Popper placement to it's general direction.
 *
 * @param position - Popper Placement value, e.g. 'top-start'
 * @returns Popper Direction, e.g. 'top'
 */
const getDirectionFromPlacement = placement => placement.split('-')[0];
function Tooltip({
  children,
  position = 'bottom',
  mousePosition = 'bottom',
  content,
  truncate = false,
  component: Container = TooltipContainer,
  tag: TargetContainer = 'div',
  testId,
  delay = 300,
  onShow = noop,
  onHide = noop,
  hideTooltipOnClick = false,
  hideTooltipOnMouseDown = false,
  analyticsContext,
  strategy = 'fixed'
}) {
  const tooltipPosition = position === 'mouse' ? mousePosition : position;
  const onShowHandler = usePlatformLeafSyntheticEventHandler({
    fn: onShow,
    action: 'displayed',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const onHideHandler = usePlatformLeafSyntheticEventHandler({
    fn: onHide,
    action: 'hidden',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const apiRef = useRef(null);
  const [state, setState] = useState('hide');
  const targetRef = useRef(null);
  const containerRef = useRef(null);
  const setRef = useCallback(node => {
    if (!node) {
      return;
    }
    if (typeof children === 'function') {
      // @ts-ignore - React Ref typing is too strict for this use case
      targetRef.current = node;
    } else {
      // @ts-ignore - React Ref typing is too strict for this use case
      containerRef.current = node;
      // @ts-ignore - React Ref typing is too strict for this use case
      targetRef.current = node.firstChild;
    }
  }, [children]);

  // Putting a few things into refs so that we don't have to break memoization
  const lastState = useRef(state);
  const lastDelay = useRef(delay);
  const lastHandlers = useRef({
    onShowHandler,
    onHideHandler
  });
  const hasCalledShowHandler = useRef(false);
  useEffect(() => {
    lastState.current = state;
    lastDelay.current = delay;
    lastHandlers.current = {
      onShowHandler,
      onHideHandler
    };
  }, [delay, onHideHandler, onShowHandler, state]);
  const start = useCallback(api => {
    // @ts-ignore
    apiRef.current = api;
    hasCalledShowHandler.current = false;
  }, []);
  const done = useCallback(() => {
    if (!apiRef.current) {
      return;
    }
    // Only call onHideHandler if we have called onShowHandler
    if (hasCalledShowHandler.current) {
      lastHandlers.current.onHideHandler();
    }
    // @ts-ignore
    apiRef.current = null;
    // @ts-ignore
    hasCalledShowHandler.current = false;
    // just in case
    setState('hide');
  }, []);
  const abort = useCallback(() => {
    if (!apiRef.current) {
      return;
    }
    apiRef.current.abort();
    // Only call onHideHandler if we have called onShowHandler
    if (hasCalledShowHandler.current) {
      lastHandlers.current.onHideHandler();
    }
    // @ts-ignore
    apiRef.current = null;
  }, []);
  useEffect(function mount() {
    return function unmount() {
      if (apiRef.current) {
        abort();
      }
    };
  }, [abort]);
  const showTooltip = useCallback(source => {
    if (apiRef.current && !apiRef.current.isActive()) {
      abort();
    }

    // Tell the tooltip to keep showing
    if (apiRef.current && apiRef.current.isActive()) {
      apiRef.current.keep();
      return;
    }
    const entry = {
      source,
      delay: lastDelay.current,
      show: ({
        isImmediate
      }) => {
        // Call the onShow handler if it hasn't been called yet
        if (!hasCalledShowHandler.current) {
          hasCalledShowHandler.current = true;
          lastHandlers.current.onShowHandler();
        }
        setState(isImmediate ? 'show-immediate' : 'fade-in');
      },
      hide: ({
        isImmediate
      }) => {
        if (isImmediate) {
          setState('hide');
        } else {
          setState('before-fade-out');
        }
      },
      done: done
    };
    const api = show(entry);
    start(api);
  }, [abort, done, start]);
  const hideTooltipOnEsc = useCallback(() => {
    var _apiRef$current;
    (_apiRef$current = apiRef.current) === null || _apiRef$current === void 0 ? void 0 : _apiRef$current.requestHide({
      isImmediate: true
    });
  }, [apiRef]);
  useCloseOnEscapePress({
    onClose: hideTooltipOnEsc,
    isDisabled: state === 'hide' || state === 'fade-out'
  });
  useEffect(() => {
    if (state === 'hide') {
      return noop;
    }
    if (state === 'before-fade-out') {
      setState('fade-out');
    }
    const unbind = bind(window, {
      type: 'scroll',
      listener: () => {
        if (apiRef.current) {
          apiRef.current.requestHide({
            isImmediate: true
          });
        }
      },
      options: {
        capture: true,
        passive: true,
        once: true
      }
    });
    return unbind;
  }, [state]);
  const onMouseDown = useCallback(() => {
    if (hideTooltipOnMouseDown && apiRef.current) {
      apiRef.current.requestHide({
        isImmediate: true
      });
    }
  }, [hideTooltipOnMouseDown]);
  const onClick = useCallback(() => {
    if (hideTooltipOnClick && apiRef.current) {
      apiRef.current.requestHide({
        isImmediate: true
      });
    }
  }, [hideTooltipOnClick]);

  // Ideally we would be using onMouseEnter here, but
  // because we are binding the event to the target parent
  // we need to listen for the mouseover of all sub elements
  // This means when moving along a tooltip we are quickly toggling
  // between api.requestHide and api.keep. This it not ideal
  const onMouseOver = useCallback(event => {
    // Ignoring events from the container ref
    if (containerRef.current && event.target === containerRef.current) {
      return;
    }

    // Using prevent default as a signal that parent tooltips
    if (event.defaultPrevented) {
      return;
    }
    event.preventDefault();
    const source = position === 'mouse' ? {
      type: 'mouse',
      // TODO: ideally not recalculating this object each time
      mouse: getMousePosition({
        left: event.clientX,
        top: event.clientY
      })
    } : {
      type: 'keyboard'
    };
    showTooltip(source);
  }, [position, showTooltip]);

  // Ideally we would be using onMouseEnter here, but
  // because we are binding the event to the target parent
  // we need to listen for the mouseout of all sub elements
  // This means when moving along a tooltip we are quickly toggling
  // between api.requestHide and api.keep. This it not ideal
  const onMouseOut = useCallback(event => {
    // Ignoring events from the container ref
    if (containerRef.current && event.target === containerRef.current) {
      return;
    }

    // Using prevent default as a signal that parent tooltips
    if (event.defaultPrevented) {
      return;
    }
    event.preventDefault();
    if (apiRef.current) {
      apiRef.current.requestHide({
        isImmediate: false
      });
    }
  }, []);
  const onMouseMove = position === 'mouse' ? event => {
    var _apiRef$current2;
    if ((_apiRef$current2 = apiRef.current) !== null && _apiRef$current2 !== void 0 && _apiRef$current2.isActive()) {
      apiRef.current.mousePosition = getMousePosition({
        left: event.clientX,
        top: event.clientY
      });
    }
  } : undefined;
  const onMouseOverTooltip = useCallback(() => {
    if (apiRef.current && apiRef.current.isActive()) {
      apiRef.current.keep();
      return;
    }
  }, []);
  const onFocus = useCallback(() => {
    showTooltip({
      type: 'keyboard'
    });
  }, [showTooltip]);
  const onBlur = useCallback(() => {
    if (apiRef.current) {
      apiRef.current.requestHide({
        isImmediate: false
      });
    }
  }, []);
  const onAnimationFinished = useCallback(transition => {
    // Using lastState here because motion is not picking up the latest value
    if (transition === 'exiting' && lastState.current === 'fade-out' && apiRef.current) {
      // @ts-ignore: refs are writeable
      apiRef.current.finishHideAnimation();
    }
  }, []);

  // Doing a cast because typescript is struggling to narrow the type
  const CastTargetContainer = TargetContainer;
  const shouldRenderTooltipContainer = state !== 'hide' && Boolean(content);
  const shouldRenderTooltipChildren = state !== 'hide' && state !== 'fade-out';
  const getReferenceElement = () => {
    var _apiRef$current3;
    if (position === 'mouse' && (_apiRef$current3 = apiRef.current) !== null && _apiRef$current3 !== void 0 && _apiRef$current3.mousePosition) {
      var _apiRef$current4;
      return (_apiRef$current4 = apiRef.current) === null || _apiRef$current4 === void 0 ? void 0 : _apiRef$current4.mousePosition;
    }
    return targetRef.current || undefined;
  };
  const tooltipId = useUniqueId('tooltip', shouldRenderTooltipContainer);
  const tooltipTriggerProps = {
    onMouseOver,
    onMouseOut,
    onMouseMove,
    onMouseDown,
    onClick,
    onFocus,
    onBlur,
    ref: setRef
  };

  // Don't set `data-testid` unless it's defined, as it's not in the interface.
  if (testId) {
    // @ts-expect-error - Adding `data-testid` to the TriggerProps interface breaks Buttons.
    tooltipTriggerProps['data-testid'] = `${testId}--container`;
  }

  // This useEffect is purely for managing the aria attribute when using the
  // wrapped children approach.
  useEffect(() => {
    // If there is no container element, we should exit early, because that
    // means they are using the render prop API, and that is implemented in a
    // different way. If there is no target element yet or tooltipId, we also
    // shouldn't do anything because there is nothing to operate on or with.
    if (!containerRef.current || !targetRef.current || !tooltipId) {
      return;
    }

    // Necessary for TS to know that it has the attribute methods
    const target = targetRef.current;
    if (shouldRenderTooltipContainer) {
      target.setAttribute('aria-describedby', tooltipId);
    } else {
      target.removeAttribute('aria-describedby');
    }
  }, [shouldRenderTooltipContainer, tooltipId]);
  return jsx(React.Fragment, null, typeof children === 'function' ?
  // once we deprecate the wrapped approach, we can put the aria
  // attribute back into the tooltipTriggerProps and make it required
  // instead of optional in `types`
  children({
    ...tooltipTriggerProps,
    'aria-describedby': tooltipId
  }) : jsx(CastTargetContainer, _extends({}, tooltipTriggerProps, {
    role: "presentation"
  }), children), shouldRenderTooltipContainer ? jsx(Portal, {
    zIndex: tooltipZIndex
  }, jsx(Popper, {
    placement: tooltipPosition,
    referenceElement: getReferenceElement(),
    strategy: strategy
  }, ({
    ref,
    style,
    update,
    placement
  }) => {
    // Invert the entrance and exit directions.
    // E.g. a tooltip's position is on the 'right', it should enter from and exit to the 'left'
    // This gives the effect the tooltip is appearing from the target
    const direction = position === 'mouse' ? undefined : invertedDirection[getDirectionFromPlacement(placement)];
    return jsx(ExitingPersistence, {
      appear: true
    }, shouldRenderTooltipChildren && jsx(FadeIn, {
      distance: "constant",
      entranceDirection: direction,
      exitDirection: direction,
      onFinish: onAnimationFinished,
      duration: state === 'show-immediate' ? 0 : mediumDurationMs
    }, ({
      className
    }) =>
    // eslint-disable-next-line jsx-a11y/mouse-events-have-key-events
    jsx(Container, {
      ref: ref
      /**
       * "Tooltip" classname is a hook used by tests to manipulate
       * and hide tooltips, including in VR snapshots
       **/,
      className: `Tooltip ${className}`,
      style: style,
      truncate: truncate,
      placement: tooltipPosition,
      testId: testId,
      onMouseOut: onMouseOut,
      onMouseOver: onMouseOverTooltip,
      id: tooltipId
    }, typeof content === 'function' ? content({
      update
    }) : content)));
  })) : null);
}
Tooltip.displayName = 'Tooltip';
export default Tooltip;