import globalize from './internal/globalize';
import ProgressiveDataSet from './progressive-data-set';

var TruncatingProgressiveDataSet = ProgressiveDataSet.extend({
    /**
     * This is a subclass of ProgressiveDataSet. It differs from the superclass
     * in that it works on large data sets where the server truncates results.
     *
     * Rather than determining whether to request more information based on its cache,
     * it uses the size of the response.
     *
     * @example
     * var source = new TruncatingProgressiveDataSet([], {
     *     model: Backbone.Model.extend({ idAttribute: "username" }),
     *     queryEndpoint: "/jira/rest/latest/users",
     *     queryParamKey: "username",
     *     matcher: function(model, query) {
     *         return _.startsWith(model.get('username'), query);
     *     },
     *     maxResponseSize: 20
     * });
     * source.on('respond', doStuffWithMatchingResults);
     * source.query('john');
     */
    initialize: function (models, options) {
        this._maxResponseSize = options.maxResponseSize;
        ProgressiveDataSet.prototype.initialize.call(this, models, options);
    },

    shouldGetMoreResults: function () {
        var response = this.findQueryResponse(this.value);
        return !response || response.length === this._maxResponseSize;
    },

    /**
     * Returns the response for the given query.
     *
     * The default implementation assumes that the endpoint's search algorithm is a prefix
     * matcher.
     *
     * @param query the value to find existing responses
     * @return {Object[]} an array of values representing the IDs of the models provided by the response for the given query.
     * Null is returned if no response is found.
     */
    findQueryResponse: function (query) {
        while (query) {
            var response = this.findQueryCache(query);

            if (response) {
                return response;
            }

            query = query.substr(0, query.length - 1);
        }

        return null;
    }
});

globalize('TruncatingProgressiveDataSet', TruncatingProgressiveDataSet);

export default TruncatingProgressiveDataSet;
