'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RelativeTime = undefined;

var _i18n = require('./i18n');

var _relative = require('./defaultTranslations/relative.time');

var _aureliaEventAggregator = require('aurelia-event-aggregator');



var RelativeTime = exports.RelativeTime = function () {
  RelativeTime.inject = function inject() {
    return [_i18n.I18N, _aureliaEventAggregator.EventAggregator];
  };

  function RelativeTime(i18n, ea) {
    var _this = this;

    

    this.service = i18n;
    this.ea = ea;

    this.service.i18nextReady().then(function () {
      _this.setup();
    });
    this.ea.subscribe('i18n:locale:changed', function (locales) {
      _this.setup(locales);
    });
  }

  RelativeTime.prototype.setup = function setup(locales) {
    var trans = _relative.translations.default || _relative.translations;
    var key = locales && locales.newValue ? locales.newValue : this.service.getLocale();
    var fallbackLng = this.service.i18next.fallbackLng;
    var index = 0;

    if ((index = key.indexOf('-')) >= 0) {
      var baseLocale = key.substring(0, index);

      if (trans[baseLocale]) {
        this.addTranslationResource(baseLocale, trans[baseLocale].translation);
      }
    }

    if (trans[key]) {
      this.addTranslationResource(key, trans[key].translation);
    }
    if (trans[fallbackLng]) {
      this.addTranslationResource(key, trans[fallbackLng].translation);
    }
  };

  RelativeTime.prototype.addTranslationResource = function addTranslationResource(key, translation) {
    var options = this.service.i18next.options;

    if (options.interpolation && options.interpolation.prefix !== '__' || options.interpolation.suffix !== '__') {
      for (var subkey in translation) {
        translation[subkey] = translation[subkey].replace('__count__', options.interpolation.prefix + 'count' + options.interpolation.suffix);
      }
    }

    this.service.i18next.addResources(key, 'translation', translation);
  };

  RelativeTime.prototype.getRelativeTime = function getRelativeTime(time) {
    var now = new Date();
    var diff = now.getTime() - time.getTime();

    var timeDiff = this.getTimeDiffDescription(diff, 'year', 31104000000);
    if (!timeDiff) {
      timeDiff = this.getTimeDiffDescription(diff, 'month', 2592000000);
      if (!timeDiff) {
        timeDiff = this.getTimeDiffDescription(diff, 'day', 86400000);
        if (!timeDiff) {
          timeDiff = this.getTimeDiffDescription(diff, 'hour', 3600000);
          if (!timeDiff) {
            timeDiff = this.getTimeDiffDescription(diff, 'minute', 60000);
            if (!timeDiff) {
              timeDiff = this.getTimeDiffDescription(diff, 'second', 1000);
              if (!timeDiff) {
                timeDiff = this.service.tr('now');
              }
            }
          }
        }
      }
    }

    return timeDiff;
  };

  RelativeTime.prototype.getTimeDiffDescription = function getTimeDiffDescription(diff, unit, timeDivisor) {
    var unitAmount = (diff / timeDivisor).toFixed(0);
    if (unitAmount > 0) {
      return this.service.tr(unit, { count: parseInt(unitAmount, 10), context: 'ago' });
    } else if (unitAmount < 0) {
      var abs = Math.abs(unitAmount);
      return this.service.tr(unit, { count: abs, context: 'in' });
    }

    return null;
  };

  return RelativeTime;
}();