"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMasterProfileName = exports.parseKnownFiles = exports.fromIni = exports.ENV_PROFILE = void 0;
const property_provider_1 = require("@aws-sdk/property-provider");
const shared_ini_file_loader_1 = require("@aws-sdk/shared-ini-file-loader");
const DEFAULT_PROFILE = "default";
exports.ENV_PROFILE = "AWS_PROFILE";
const isStaticCredsProfile = (arg) => Boolean(arg) &&
    typeof arg === "object" &&
    typeof arg.aws_access_key_id === "string" &&
    typeof arg.aws_secret_access_key === "string" &&
    ["undefined", "string"].indexOf(typeof arg.aws_session_token) > -1;
const isAssumeRoleProfile = (arg) => Boolean(arg) &&
    typeof arg === "object" &&
    typeof arg.role_arn === "string" &&
    typeof arg.source_profile === "string" &&
    ["undefined", "string"].indexOf(typeof arg.role_session_name) > -1 &&
    ["undefined", "string"].indexOf(typeof arg.external_id) > -1 &&
    ["undefined", "string"].indexOf(typeof arg.mfa_serial) > -1;
/**
 * Creates a credential provider that will read from ini files and supports
 * role assumption and multi-factor authentication.
 */
const fromIni = (init = {}) => async () => {
    const profiles = await exports.parseKnownFiles(init);
    return resolveProfileData(exports.getMasterProfileName(init), profiles, init);
};
exports.fromIni = fromIni;
/**
 * Load profiles from credentials and config INI files and normalize them into a
 * single profile list.
 *
 * @internal
 */
const parseKnownFiles = async (init) => {
    const { loadedConfig = shared_ini_file_loader_1.loadSharedConfigFiles(init) } = init;
    const parsedFiles = await loadedConfig;
    return {
        ...parsedFiles.configFile,
        ...parsedFiles.credentialsFile,
    };
};
exports.parseKnownFiles = parseKnownFiles;
/**
 * @internal
 */
const getMasterProfileName = (init) => init.profile || process.env[exports.ENV_PROFILE] || DEFAULT_PROFILE;
exports.getMasterProfileName = getMasterProfileName;
const resolveProfileData = async (profileName, profiles, options, visitedProfiles = {}) => {
    const data = profiles[profileName];
    // If this is not the first profile visited, static credentials should be
    // preferred over role assumption metadata. This special treatment of
    // second and subsequent hops is to ensure compatibility with the AWS CLI.
    if (Object.keys(visitedProfiles).length > 0 && isStaticCredsProfile(data)) {
        return resolveStaticCredentials(data);
    }
    // If this is the first profile visited, role assumption keys should be
    // given precedence over static credentials.
    if (isAssumeRoleProfile(data)) {
        const { external_id: ExternalId, mfa_serial, role_arn: RoleArn, role_session_name: RoleSessionName = "aws-sdk-js-" + Date.now(), source_profile, } = data;
        if (!options.roleAssumer) {
            throw new property_provider_1.ProviderError(`Profile ${profileName} requires a role to be assumed, but no` + ` role assumption callback was provided.`, false);
        }
        if (source_profile in visitedProfiles) {
            throw new property_provider_1.ProviderError(`Detected a cycle attempting to resolve credentials for profile` +
                ` ${exports.getMasterProfileName(options)}. Profiles visited: ` +
                Object.keys(visitedProfiles).join(", "), false);
        }
        const sourceCreds = resolveProfileData(source_profile, profiles, options, {
            ...visitedProfiles,
            [source_profile]: true,
        });
        const params = { RoleArn, RoleSessionName, ExternalId };
        if (mfa_serial) {
            if (!options.mfaCodeProvider) {
                throw new property_provider_1.ProviderError(`Profile ${profileName} requires multi-factor authentication,` + ` but no MFA code callback was provided.`, false);
            }
            params.SerialNumber = mfa_serial;
            params.TokenCode = await options.mfaCodeProvider(mfa_serial);
        }
        return options.roleAssumer(await sourceCreds, params);
    }
    // If no role assumption metadata is present, attempt to load static
    // credentials from the selected profile.
    if (isStaticCredsProfile(data)) {
        return resolveStaticCredentials(data);
    }
    // If the profile cannot be parsed or contains neither static credentials
    // nor role assumption metadata, throw an error. This should be considered a
    // terminal resolution error if a profile has been specified by the user
    // (whether via a parameter, an environment variable, or another profile's
    // `source_profile` key).
    throw new property_provider_1.ProviderError(`Profile ${profileName} could not be found or parsed in shared` + ` credentials file.`);
};
const resolveStaticCredentials = (profile) => Promise.resolve({
    accessKeyId: profile.aws_access_key_id,
    secretAccessKey: profile.aws_secret_access_key,
    sessionToken: profile.aws_session_token,
});
//# sourceMappingURL=data:application/json;base64,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