"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventStreamMarshaller = void 0;
const crc32_1 = require("@aws-crypto/crc32");
const HeaderMarshaller_1 = require("./HeaderMarshaller");
const splitMessage_1 = require("./splitMessage");
/**
 * A marshaller that can convert binary-packed event stream messages into
 * JavaScript objects and back again into their binary format.
 */
class EventStreamMarshaller {
    constructor(toUtf8, fromUtf8) {
        this.headerMarshaller = new HeaderMarshaller_1.HeaderMarshaller(toUtf8, fromUtf8);
    }
    /**
     * Convert a structured JavaScript object with tagged headers into a binary
     * event stream message.
     */
    marshall({ headers: rawHeaders, body }) {
        const headers = this.headerMarshaller.format(rawHeaders);
        const length = headers.byteLength + body.byteLength + 16;
        const out = new Uint8Array(length);
        const view = new DataView(out.buffer, out.byteOffset, out.byteLength);
        const checksum = new crc32_1.Crc32();
        // Format message
        view.setUint32(0, length, false);
        view.setUint32(4, headers.byteLength, false);
        view.setUint32(8, checksum.update(out.subarray(0, 8)).digest(), false);
        out.set(headers, 12);
        out.set(body, headers.byteLength + 12);
        // Write trailing message checksum
        view.setUint32(length - 4, checksum.update(out.subarray(8, length - 4)).digest(), false);
        return out;
    }
    /**
     * Convert a binary event stream message into a JavaScript object with an
     * opaque, binary body and tagged, parsed headers.
     */
    unmarshall(message) {
        const { headers, body } = splitMessage_1.splitMessage(message);
        return { headers: this.headerMarshaller.parse(headers), body };
    }
    /**
     * Convert a structured JavaScript object with tagged headers into a binary
     * event stream message header.
     */
    formatHeaders(rawHeaders) {
        return this.headerMarshaller.format(rawHeaders);
    }
}
exports.EventStreamMarshaller = EventStreamMarshaller;
//# sourceMappingURL=data:application/json;base64,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