"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HeaderMarshaller = void 0;
const util_hex_encoding_1 = require("@aws-sdk/util-hex-encoding");
const Int64_1 = require("./Int64");
/**
 * @internal
 */
class HeaderMarshaller {
    constructor(toUtf8, fromUtf8) {
        this.toUtf8 = toUtf8;
        this.fromUtf8 = fromUtf8;
    }
    format(headers) {
        const chunks = [];
        for (const headerName of Object.keys(headers)) {
            const bytes = this.fromUtf8(headerName);
            chunks.push(Uint8Array.from([bytes.byteLength]), bytes, this.formatHeaderValue(headers[headerName]));
        }
        const out = new Uint8Array(chunks.reduce((carry, bytes) => carry + bytes.byteLength, 0));
        let position = 0;
        for (const chunk of chunks) {
            out.set(chunk, position);
            position += chunk.byteLength;
        }
        return out;
    }
    formatHeaderValue(header) {
        switch (header.type) {
            case "boolean":
                return Uint8Array.from([header.value ? 0 /* boolTrue */ : 1 /* boolFalse */]);
            case "byte":
                return Uint8Array.from([2 /* byte */, header.value]);
            case "short":
                const shortView = new DataView(new ArrayBuffer(3));
                shortView.setUint8(0, 3 /* short */);
                shortView.setInt16(1, header.value, false);
                return new Uint8Array(shortView.buffer);
            case "integer":
                const intView = new DataView(new ArrayBuffer(5));
                intView.setUint8(0, 4 /* integer */);
                intView.setInt32(1, header.value, false);
                return new Uint8Array(intView.buffer);
            case "long":
                const longBytes = new Uint8Array(9);
                longBytes[0] = 5 /* long */;
                longBytes.set(header.value.bytes, 1);
                return longBytes;
            case "binary":
                const binView = new DataView(new ArrayBuffer(3 + header.value.byteLength));
                binView.setUint8(0, 6 /* byteArray */);
                binView.setUint16(1, header.value.byteLength, false);
                const binBytes = new Uint8Array(binView.buffer);
                binBytes.set(header.value, 3);
                return binBytes;
            case "string":
                const utf8Bytes = this.fromUtf8(header.value);
                const strView = new DataView(new ArrayBuffer(3 + utf8Bytes.byteLength));
                strView.setUint8(0, 7 /* string */);
                strView.setUint16(1, utf8Bytes.byteLength, false);
                const strBytes = new Uint8Array(strView.buffer);
                strBytes.set(utf8Bytes, 3);
                return strBytes;
            case "timestamp":
                const tsBytes = new Uint8Array(9);
                tsBytes[0] = 8 /* timestamp */;
                tsBytes.set(Int64_1.Int64.fromNumber(header.value.valueOf()).bytes, 1);
                return tsBytes;
            case "uuid":
                if (!UUID_PATTERN.test(header.value)) {
                    throw new Error(`Invalid UUID received: ${header.value}`);
                }
                const uuidBytes = new Uint8Array(17);
                uuidBytes[0] = 9 /* uuid */;
                uuidBytes.set(util_hex_encoding_1.fromHex(header.value.replace(/\-/g, "")), 1);
                return uuidBytes;
        }
    }
    parse(headers) {
        const out = {};
        let position = 0;
        while (position < headers.byteLength) {
            const nameLength = headers.getUint8(position++);
            const name = this.toUtf8(new Uint8Array(headers.buffer, headers.byteOffset + position, nameLength));
            position += nameLength;
            switch (headers.getUint8(position++)) {
                case 0 /* boolTrue */:
                    out[name] = {
                        type: BOOLEAN_TAG,
                        value: true,
                    };
                    break;
                case 1 /* boolFalse */:
                    out[name] = {
                        type: BOOLEAN_TAG,
                        value: false,
                    };
                    break;
                case 2 /* byte */:
                    out[name] = {
                        type: BYTE_TAG,
                        value: headers.getInt8(position++),
                    };
                    break;
                case 3 /* short */:
                    out[name] = {
                        type: SHORT_TAG,
                        value: headers.getInt16(position, false),
                    };
                    position += 2;
                    break;
                case 4 /* integer */:
                    out[name] = {
                        type: INT_TAG,
                        value: headers.getInt32(position, false),
                    };
                    position += 4;
                    break;
                case 5 /* long */:
                    out[name] = {
                        type: LONG_TAG,
                        value: new Int64_1.Int64(new Uint8Array(headers.buffer, headers.byteOffset + position, 8)),
                    };
                    position += 8;
                    break;
                case 6 /* byteArray */:
                    const binaryLength = headers.getUint16(position, false);
                    position += 2;
                    out[name] = {
                        type: BINARY_TAG,
                        value: new Uint8Array(headers.buffer, headers.byteOffset + position, binaryLength),
                    };
                    position += binaryLength;
                    break;
                case 7 /* string */:
                    const stringLength = headers.getUint16(position, false);
                    position += 2;
                    out[name] = {
                        type: STRING_TAG,
                        value: this.toUtf8(new Uint8Array(headers.buffer, headers.byteOffset + position, stringLength)),
                    };
                    position += stringLength;
                    break;
                case 8 /* timestamp */:
                    out[name] = {
                        type: TIMESTAMP_TAG,
                        value: new Date(new Int64_1.Int64(new Uint8Array(headers.buffer, headers.byteOffset + position, 8)).valueOf()),
                    };
                    position += 8;
                    break;
                case 9 /* uuid */:
                    const uuidBytes = new Uint8Array(headers.buffer, headers.byteOffset + position, 16);
                    position += 16;
                    out[name] = {
                        type: UUID_TAG,
                        value: `${util_hex_encoding_1.toHex(uuidBytes.subarray(0, 4))}-${util_hex_encoding_1.toHex(uuidBytes.subarray(4, 6))}-${util_hex_encoding_1.toHex(uuidBytes.subarray(6, 8))}-${util_hex_encoding_1.toHex(uuidBytes.subarray(8, 10))}-${util_hex_encoding_1.toHex(uuidBytes.subarray(10))}`,
                    };
                    break;
                default:
                    throw new Error(`Unrecognized header type tag`);
            }
        }
        return out;
    }
}
exports.HeaderMarshaller = HeaderMarshaller;
var HEADER_VALUE_TYPE;
(function (HEADER_VALUE_TYPE) {
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["boolTrue"] = 0] = "boolTrue";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["boolFalse"] = 1] = "boolFalse";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["byte"] = 2] = "byte";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["short"] = 3] = "short";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["integer"] = 4] = "integer";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["long"] = 5] = "long";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["byteArray"] = 6] = "byteArray";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["string"] = 7] = "string";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["timestamp"] = 8] = "timestamp";
    HEADER_VALUE_TYPE[HEADER_VALUE_TYPE["uuid"] = 9] = "uuid";
})(HEADER_VALUE_TYPE || (HEADER_VALUE_TYPE = {}));
const BOOLEAN_TAG = "boolean";
const BYTE_TAG = "byte";
const SHORT_TAG = "short";
const INT_TAG = "integer";
const LONG_TAG = "long";
const BINARY_TAG = "binary";
const STRING_TAG = "string";
const TIMESTAMP_TAG = "timestamp";
const UUID_TAG = "uuid";
const UUID_PATTERN = /^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/;
//# sourceMappingURL=data:application/json;base64,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