"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.splitMessage = void 0;
const crc32_1 = require("@aws-crypto/crc32");
// All prelude components are unsigned, 32-bit integers
const PRELUDE_MEMBER_LENGTH = 4;
// The prelude consists of two components
const PRELUDE_LENGTH = PRELUDE_MEMBER_LENGTH * 2;
// Checksums are always CRC32 hashes.
const CHECKSUM_LENGTH = 4;
// Messages must include a full prelude, a prelude checksum, and a message checksum
const MINIMUM_MESSAGE_LENGTH = PRELUDE_LENGTH + CHECKSUM_LENGTH * 2;
/**
 * @internal
 */
function splitMessage({ byteLength, byteOffset, buffer }) {
    if (byteLength < MINIMUM_MESSAGE_LENGTH) {
        throw new Error("Provided message too short to accommodate event stream message overhead");
    }
    const view = new DataView(buffer, byteOffset, byteLength);
    const messageLength = view.getUint32(0, false);
    if (byteLength !== messageLength) {
        throw new Error("Reported message length does not match received message length");
    }
    const headerLength = view.getUint32(PRELUDE_MEMBER_LENGTH, false);
    const expectedPreludeChecksum = view.getUint32(PRELUDE_LENGTH, false);
    const expectedMessageChecksum = view.getUint32(byteLength - CHECKSUM_LENGTH, false);
    const checksummer = new crc32_1.Crc32().update(new Uint8Array(buffer, byteOffset, PRELUDE_LENGTH));
    if (expectedPreludeChecksum !== checksummer.digest()) {
        throw new Error(`The prelude checksum specified in the message (${expectedPreludeChecksum}) does not match the calculated CRC32 checksum (${checksummer.digest()})`);
    }
    checksummer.update(new Uint8Array(buffer, byteOffset + PRELUDE_LENGTH, byteLength - (PRELUDE_LENGTH + CHECKSUM_LENGTH)));
    if (expectedMessageChecksum !== checksummer.digest()) {
        throw new Error(`The message checksum (${checksummer.digest()}) did not match the expected value of ${expectedMessageChecksum}`);
    }
    return {
        headers: new DataView(buffer, byteOffset + PRELUDE_LENGTH + CHECKSUM_LENGTH, headerLength),
        body: new Uint8Array(buffer, byteOffset + PRELUDE_LENGTH + CHECKSUM_LENGTH + headerLength, messageLength - headerLength - (PRELUDE_LENGTH + CHECKSUM_LENGTH + CHECKSUM_LENGTH)),
    };
}
exports.splitMessage = splitMessage;
//# sourceMappingURL=data:application/json;base64,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