import { Crc32 } from "@aws-crypto/crc32";
import { HeaderMarshaller } from "./HeaderMarshaller";
import { splitMessage } from "./splitMessage";
/**
 * A marshaller that can convert binary-packed event stream messages into
 * JavaScript objects and back again into their binary format.
 */
var EventStreamMarshaller = /** @class */ (function () {
    function EventStreamMarshaller(toUtf8, fromUtf8) {
        this.headerMarshaller = new HeaderMarshaller(toUtf8, fromUtf8);
    }
    /**
     * Convert a structured JavaScript object with tagged headers into a binary
     * event stream message.
     */
    EventStreamMarshaller.prototype.marshall = function (_a) {
        var rawHeaders = _a.headers, body = _a.body;
        var headers = this.headerMarshaller.format(rawHeaders);
        var length = headers.byteLength + body.byteLength + 16;
        var out = new Uint8Array(length);
        var view = new DataView(out.buffer, out.byteOffset, out.byteLength);
        var checksum = new Crc32();
        // Format message
        view.setUint32(0, length, false);
        view.setUint32(4, headers.byteLength, false);
        view.setUint32(8, checksum.update(out.subarray(0, 8)).digest(), false);
        out.set(headers, 12);
        out.set(body, headers.byteLength + 12);
        // Write trailing message checksum
        view.setUint32(length - 4, checksum.update(out.subarray(8, length - 4)).digest(), false);
        return out;
    };
    /**
     * Convert a binary event stream message into a JavaScript object with an
     * opaque, binary body and tagged, parsed headers.
     */
    EventStreamMarshaller.prototype.unmarshall = function (message) {
        var _a = splitMessage(message), headers = _a.headers, body = _a.body;
        return { headers: this.headerMarshaller.parse(headers), body: body };
    };
    /**
     * Convert a structured JavaScript object with tagged headers into a binary
     * event stream message header.
     */
    EventStreamMarshaller.prototype.formatHeaders = function (rawHeaders) {
        return this.headerMarshaller.format(rawHeaders);
    };
    return EventStreamMarshaller;
}());
export { EventStreamMarshaller };
//# sourceMappingURL=data:application/json;base64,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