import { toHex } from "@aws-sdk/util-hex-encoding";
/**
 * A lossless representation of a signed, 64-bit integer. Instances of this
 * class may be used in arithmetic expressions as if they were numeric
 * primitives, but the binary representation will be preserved unchanged as the
 * `bytes` property of the object. The bytes should be encoded as big-endian,
 * two's complement integers.
 */
var Int64 = /** @class */ (function () {
    function Int64(bytes) {
        this.bytes = bytes;
        if (bytes.byteLength !== 8) {
            throw new Error("Int64 buffers must be exactly 8 bytes");
        }
    }
    Int64.fromNumber = function (number) {
        if (number > 9223372036854775807 || number < -9223372036854775808) {
            throw new Error(number + " is too large (or, if negative, too small) to represent as an Int64");
        }
        var bytes = new Uint8Array(8);
        for (var i = 7, remaining = Math.abs(Math.round(number)); i > -1 && remaining > 0; i--, remaining /= 256) {
            bytes[i] = remaining;
        }
        if (number < 0) {
            negate(bytes);
        }
        return new Int64(bytes);
    };
    /**
     * Called implicitly by infix arithmetic operators.
     */
    Int64.prototype.valueOf = function () {
        var bytes = this.bytes.slice(0);
        var negative = bytes[0] & 128;
        if (negative) {
            negate(bytes);
        }
        return parseInt(toHex(bytes), 16) * (negative ? -1 : 1);
    };
    Int64.prototype.toString = function () {
        return String(this.valueOf());
    };
    return Int64;
}());
export { Int64 };
function negate(bytes) {
    for (var i = 0; i < 8; i++) {
        bytes[i] ^= 0xff;
    }
    for (var i = 7; i > -1; i--) {
        bytes[i]++;
        if (bytes[i] !== 0)
            break;
    }
}
//# sourceMappingURL=data:application/json;base64,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