"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getBucketEndpointPlugin = exports.bucketEndpointMiddlewareOptions = exports.bucketEndpointMiddleware = void 0;
const protocol_http_1 = require("@aws-sdk/protocol-http");
const util_arn_parser_1 = require("@aws-sdk/util-arn-parser");
const bucketHostname_1 = require("./bucketHostname");
const bucketHostnameUtils_1 = require("./bucketHostnameUtils");
const bucketEndpointMiddleware = (options) => (next, context) => async (args) => {
    const { Bucket: bucketName } = args.input;
    let replaceBucketInPath = options.bucketEndpoint;
    const request = args.request;
    if (protocol_http_1.HttpRequest.isInstance(request)) {
        if (options.bucketEndpoint) {
            request.hostname = bucketName;
        }
        else if (util_arn_parser_1.validate(bucketName)) {
            const bucketArn = util_arn_parser_1.parse(bucketName);
            const clientRegion = bucketHostnameUtils_1.getPseudoRegion(await options.region());
            const { partition, signingRegion = clientRegion } = (await options.regionInfoProvider(clientRegion)) || {};
            const useArnRegion = await options.useArnRegion();
            const { hostname, bucketEndpoint, signingRegion: modifiedSigningRegion, signingService } = bucketHostname_1.bucketHostname({
                bucketName: bucketArn,
                baseHostname: request.hostname,
                accelerateEndpoint: options.useAccelerateEndpoint,
                dualstackEndpoint: options.useDualstackEndpoint,
                pathStyleEndpoint: options.forcePathStyle,
                tlsCompatible: request.protocol === "https:",
                useArnRegion,
                clientPartition: partition,
                clientSigningRegion: signingRegion,
                clientRegion: clientRegion,
                isCustomEndpoint: options.isCustomEndpoint,
            });
            // If the request needs to use a region or service name inferred from ARN that different from client region, we
            // need to set them in the handler context so the signer will use them
            if (modifiedSigningRegion && modifiedSigningRegion !== signingRegion) {
                context["signing_region"] = modifiedSigningRegion;
            }
            if (signingService && signingService !== "s3") {
                context["signing_service"] = signingService;
            }
            request.hostname = hostname;
            replaceBucketInPath = bucketEndpoint;
        }
        else {
            const clientRegion = bucketHostnameUtils_1.getPseudoRegion(await options.region());
            const { hostname, bucketEndpoint } = bucketHostname_1.bucketHostname({
                bucketName,
                clientRegion,
                baseHostname: request.hostname,
                accelerateEndpoint: options.useAccelerateEndpoint,
                dualstackEndpoint: options.useDualstackEndpoint,
                pathStyleEndpoint: options.forcePathStyle,
                tlsCompatible: request.protocol === "https:",
                isCustomEndpoint: options.isCustomEndpoint,
            });
            request.hostname = hostname;
            replaceBucketInPath = bucketEndpoint;
        }
        if (replaceBucketInPath) {
            request.path = request.path.replace(/^(\/)?[^\/]+/, "");
            if (request.path === "") {
                request.path = "/";
            }
        }
    }
    return next({ ...args, request });
};
exports.bucketEndpointMiddleware = bucketEndpointMiddleware;
exports.bucketEndpointMiddlewareOptions = {
    tags: ["BUCKET_ENDPOINT"],
    name: "bucketEndpointMiddleware",
    relation: "before",
    toMiddleware: "hostHeaderMiddleware",
    override: true,
};
const getBucketEndpointPlugin = (options) => ({
    applyToStack: (clientStack) => {
        clientStack.addRelativeTo(exports.bucketEndpointMiddleware(options), exports.bucketEndpointMiddlewareOptions);
    },
});
exports.getBucketEndpointPlugin = getBucketEndpointPlugin;
//# sourceMappingURL=data:application/json;base64,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