"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bucketHostname = void 0;
const bucketHostnameUtils_1 = require("./bucketHostnameUtils");
const bucketHostname = (options) => {
    const { isCustomEndpoint, baseHostname, dualstackEndpoint, accelerateEndpoint } = options;
    if (isCustomEndpoint) {
        if (dualstackEndpoint)
            throw new Error("Dualstack endpoint is not supported with custom endpoint");
        if (accelerateEndpoint)
            throw new Error("Accelerate endpoint is not supported with custom endpoint");
    }
    return bucketHostnameUtils_1.isBucketNameOptions(options)
        ? // Construct endpoint when bucketName is a string referring to a bucket name
            getEndpointFromBucketName({ ...options, isCustomEndpoint })
        : // Construct endpoint when bucketName is an ARN referring to an S3 resource like Access Point
            getEndpointFromArn({ ...options, isCustomEndpoint });
};
exports.bucketHostname = bucketHostname;
const getEndpointFromArn = (options) => {
    const { isCustomEndpoint, baseHostname } = options;
    const [clientRegion, hostnameSuffix] = isCustomEndpoint
        ? [options.clientRegion, baseHostname]
        : // Infer client region and hostname suffix from hostname from endpoints.json, like `s3.us-west-2.amazonaws.com`
            bucketHostnameUtils_1.getSuffixForArnEndpoint(baseHostname);
    const { pathStyleEndpoint, dualstackEndpoint = false, accelerateEndpoint = false, tlsCompatible = true, useArnRegion, bucketName, clientPartition = "aws", clientSigningRegion = clientRegion, } = options;
    bucketHostnameUtils_1.validateArnEndpointOptions({ pathStyleEndpoint, accelerateEndpoint, tlsCompatible });
    // Validate and parse the ARN supplied as a bucket name
    const { service, partition, accountId, region, resource } = bucketName;
    bucketHostnameUtils_1.validateService(service);
    bucketHostnameUtils_1.validatePartition(partition, { clientPartition });
    bucketHostnameUtils_1.validateAccountId(accountId);
    bucketHostnameUtils_1.validateRegion(region, { useArnRegion, clientRegion, clientSigningRegion });
    const { accesspointName, outpostId } = bucketHostnameUtils_1.getArnResources(resource);
    bucketHostnameUtils_1.validateDNSHostLabel(`${accesspointName}-${accountId}`, { tlsCompatible });
    const endpointRegion = useArnRegion ? region : clientRegion;
    const signingRegion = useArnRegion ? region : clientSigningRegion;
    if (outpostId) {
        // if this is an Outpost ARN
        bucketHostnameUtils_1.validateOutpostService(service);
        bucketHostnameUtils_1.validateDNSHostLabel(outpostId, { tlsCompatible });
        bucketHostnameUtils_1.validateNoDualstack(dualstackEndpoint);
        bucketHostnameUtils_1.validateNoFIPS(endpointRegion);
        const hostnamePrefix = `${accesspointName}-${accountId}.${outpostId}`;
        return {
            bucketEndpoint: true,
            hostname: `${hostnamePrefix}${isCustomEndpoint ? "" : `.s3-outposts.${endpointRegion}`}.${hostnameSuffix}`,
            signingRegion,
            signingService: "s3-outposts",
        };
    }
    // construct endpoint from Accesspoint ARN
    bucketHostnameUtils_1.validateS3Service(service);
    const hostnamePrefix = `${accesspointName}-${accountId}`;
    return {
        bucketEndpoint: true,
        hostname: `${hostnamePrefix}${isCustomEndpoint ? "" : `.s3-accesspoint${dualstackEndpoint ? ".dualstack" : ""}.${endpointRegion}`}.${hostnameSuffix}`,
        signingRegion,
    };
};
const getEndpointFromBucketName = ({ accelerateEndpoint = false, clientRegion: region, baseHostname, bucketName, dualstackEndpoint = false, pathStyleEndpoint = false, tlsCompatible = true, isCustomEndpoint = false, }) => {
    const [clientRegion, hostnameSuffix] = isCustomEndpoint ? [region, baseHostname] : bucketHostnameUtils_1.getSuffix(baseHostname);
    if (pathStyleEndpoint || !bucketHostnameUtils_1.isDnsCompatibleBucketName(bucketName) || (tlsCompatible && bucketHostnameUtils_1.DOT_PATTERN.test(bucketName))) {
        return {
            bucketEndpoint: false,
            hostname: dualstackEndpoint ? `s3.dualstack.${clientRegion}.${hostnameSuffix}` : baseHostname,
        };
    }
    if (accelerateEndpoint) {
        baseHostname = `s3-accelerate${dualstackEndpoint ? ".dualstack" : ""}.${hostnameSuffix}`;
    }
    else if (dualstackEndpoint) {
        baseHostname = `s3.dualstack.${clientRegion}.${hostnameSuffix}`;
    }
    return {
        bucketEndpoint: true,
        hostname: `${bucketName}.${baseHostname}`,
    };
};
//# sourceMappingURL=data:application/json;base64,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