"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNoFIPS = exports.validateNoDualstack = exports.getArnResources = exports.validateDNSHostLabel = exports.validateAccountId = exports.validateRegion = exports.validatePartition = exports.validateOutpostService = exports.validateS3Service = exports.validateService = exports.validateArnEndpointOptions = exports.getSuffixForArnEndpoint = exports.getSuffix = exports.isDnsCompatibleBucketName = exports.getPseudoRegion = exports.isBucketNameOptions = exports.S3_HOSTNAME_PATTERN = exports.DOT_PATTERN = void 0;
const DOMAIN_PATTERN = /^[a-z0-9][a-z0-9\.\-]{1,61}[a-z0-9]$/;
const IP_ADDRESS_PATTERN = /(\d+\.){3}\d+/;
const DOTS_PATTERN = /\.\./;
exports.DOT_PATTERN = /\./;
exports.S3_HOSTNAME_PATTERN = /^(.+\.)?s3[.-]([a-z0-9-]+)\./;
const S3_US_EAST_1_ALTNAME_PATTERN = /^s3(-external-1)?\.amazonaws\.com$/;
const AWS_PARTITION_SUFFIX = "amazonaws.com";
const isBucketNameOptions = (options) => typeof options.bucketName === "string";
exports.isBucketNameOptions = isBucketNameOptions;
/**
 * Get pseudo region from supplied region. For example, if supplied with `fips-us-west-2`, it returns `us-west-2`.
 * @internal
 */
const getPseudoRegion = (region) => (isFipsRegion(region) ? region.replace(/fips-|-fips/, "") : region);
exports.getPseudoRegion = getPseudoRegion;
/**
 * Determines whether a given string is DNS compliant per the rules outlined by
 * S3. Length, capitaization, and leading dot restrictions are enforced by the
 * DOMAIN_PATTERN regular expression.
 * @internal
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
 */
const isDnsCompatibleBucketName = (bucketName) => DOMAIN_PATTERN.test(bucketName) && !IP_ADDRESS_PATTERN.test(bucketName) && !DOTS_PATTERN.test(bucketName);
exports.isDnsCompatibleBucketName = isDnsCompatibleBucketName;
const getRegionalSuffix = (hostname) => {
    const parts = hostname.match(exports.S3_HOSTNAME_PATTERN);
    return [parts[2], hostname.replace(new RegExp(`^${parts[0]}`), "")];
};
const getSuffix = (hostname) => S3_US_EAST_1_ALTNAME_PATTERN.test(hostname) ? ["us-east-1", AWS_PARTITION_SUFFIX] : getRegionalSuffix(hostname);
exports.getSuffix = getSuffix;
/**
 * Infer region and hostname suffix from a complete hostname
 * @internal
 * @param hostname - Hostname
 * @returns [Region, Hostname suffix]
 */
const getSuffixForArnEndpoint = (hostname) => S3_US_EAST_1_ALTNAME_PATTERN.test(hostname)
    ? [hostname.replace(`.${AWS_PARTITION_SUFFIX}`, ""), AWS_PARTITION_SUFFIX]
    : getRegionalSuffix(hostname);
exports.getSuffixForArnEndpoint = getSuffixForArnEndpoint;
const validateArnEndpointOptions = (options) => {
    if (options.pathStyleEndpoint) {
        throw new Error("Path-style S3 endpoint is not supported when bucket is an ARN");
    }
    if (options.accelerateEndpoint) {
        throw new Error("Accelerate endpoint is not supported when bucket is an ARN");
    }
    if (!options.tlsCompatible) {
        throw new Error("HTTPS is required when bucket is an ARN");
    }
};
exports.validateArnEndpointOptions = validateArnEndpointOptions;
const validateService = (service) => {
    if (service !== "s3" && service !== "s3-outposts") {
        throw new Error("Expect 's3' or 's3-outposts' in ARN service component");
    }
};
exports.validateService = validateService;
const validateS3Service = (service) => {
    if (service !== "s3") {
        throw new Error("Expect 's3' in Accesspoint ARN service component");
    }
};
exports.validateS3Service = validateS3Service;
const validateOutpostService = (service) => {
    if (service !== "s3-outposts") {
        throw new Error("Expect 's3-posts' in Outpost ARN service component");
    }
};
exports.validateOutpostService = validateOutpostService;
/**
 * Validate partition inferred from ARN is the same to `options.clientPartition`.
 * @internal
 */
const validatePartition = (partition, options) => {
    if (partition !== options.clientPartition) {
        throw new Error(`Partition in ARN is incompatible, got "${partition}" but expected "${options.clientPartition}"`);
    }
};
exports.validatePartition = validatePartition;
/**
 * validate region value inferred from ARN. If `options.useArnRegion` is set, it validates the region is not a FIPS
 * region. If `options.useArnRegion` is unset, it validates the region is equal to `options.clientRegion` or
 * `options.clientSigningRegion`.
 * @internal
 */
const validateRegion = (region, options) => {
    if (region === "") {
        throw new Error("ARN region is empty");
    }
    if (!options.useArnRegion &&
        !isEqualRegions(region, options.clientRegion) &&
        !isEqualRegions(region, options.clientSigningRegion)) {
        throw new Error(`Region in ARN is incompatible, got ${region} but expected ${options.clientRegion}`);
    }
    if (options.useArnRegion && isFipsRegion(region)) {
        throw new Error("Endpoint does not support FIPS region");
    }
};
exports.validateRegion = validateRegion;
const isFipsRegion = (region) => region.startsWith("fips-") || region.endsWith("-fips");
const isEqualRegions = (regionA, regionB) => regionA === regionB || exports.getPseudoRegion(regionA) === regionB || regionA === exports.getPseudoRegion(regionB);
/**
 * Validate an account ID
 * @internal
 */
const validateAccountId = (accountId) => {
    if (!/[0-9]{12}/.exec(accountId)) {
        throw new Error("Access point ARN accountID does not match regex '[0-9]{12}'");
    }
};
exports.validateAccountId = validateAccountId;
/**
 * Validate a host label according to https://tools.ietf.org/html/rfc3986#section-3.2.2
 * @internal
 */
const validateDNSHostLabel = (label, options = { tlsCompatible: true }) => {
    // reference: https://tools.ietf.org/html/rfc3986#section-3.2.2
    if (label.length >= 64 ||
        !/^[a-z0-9][a-z0-9.-]+[a-z0-9]$/.test(label) ||
        /(\d+\.){3}\d+/.test(label) ||
        /[.-]{2}/.test(label) ||
        ((options === null || options === void 0 ? void 0 : options.tlsCompatible) && exports.DOT_PATTERN.test(label))) {
        throw new Error(`Invalid DNS label ${label}`);
    }
};
exports.validateDNSHostLabel = validateDNSHostLabel;
/**
 * Validate and parse an Access Point ARN or Outposts ARN
 * @internal
 *
 * @param resource - The resource section of an ARN
 * @returns Access Point Name and optional Outpost ID.
 */
const getArnResources = (resource) => {
    const delimiter = resource.includes(":") ? ":" : "/";
    const [resourceType, ...rest] = resource.split(delimiter);
    if (resourceType === "accesspoint") {
        // Parse accesspoint ARN
        if (rest.length !== 1 || rest[0] === "") {
            throw new Error(`Access Point ARN should have one resource accesspoint${delimiter}{accesspointname}`);
        }
        return { accesspointName: rest[0] };
    }
    else if (resourceType === "outpost") {
        // Parse outpost ARN
        if (!rest[0] || rest[1] !== "accesspoint" || !rest[2] || rest.length !== 3) {
            throw new Error(`Outpost ARN should have resource outpost${delimiter}{outpostId}${delimiter}accesspoint${delimiter}{accesspointName}`);
        }
        const [outpostId, _, accesspointName] = rest;
        return { outpostId, accesspointName };
    }
    else {
        throw new Error(`ARN resource should begin with 'accesspoint${delimiter}' or 'outpost${delimiter}'`);
    }
};
exports.getArnResources = getArnResources;
/**
 * Throw if dual stack configuration is set to true.
 * @internal
 */
const validateNoDualstack = (dualstackEndpoint) => {
    if (dualstackEndpoint)
        throw new Error("Dualstack endpoint is not supported with Outpost");
};
exports.validateNoDualstack = validateNoDualstack;
/**
 * Validate region is not appended or prepended with a `fips-`
 * @internal
 */
const validateNoFIPS = (region) => {
    if (isFipsRegion(region !== null && region !== void 0 ? region : ""))
        throw new Error(`FIPS region is not supported with Outpost, got ${region}`);
};
exports.validateNoFIPS = validateNoFIPS;
//# sourceMappingURL=data:application/json;base64,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