import { __assign, __read } from "tslib";
import { DOT_PATTERN, getArnResources, getSuffix, getSuffixForArnEndpoint, isBucketNameOptions, isDnsCompatibleBucketName, validateAccountId, validateArnEndpointOptions, validateDNSHostLabel, validateNoDualstack, validateNoFIPS, validateOutpostService, validatePartition, validateRegion, validateS3Service, validateService, } from "./bucketHostnameUtils";
export var bucketHostname = function (options) {
    var isCustomEndpoint = options.isCustomEndpoint, baseHostname = options.baseHostname, dualstackEndpoint = options.dualstackEndpoint, accelerateEndpoint = options.accelerateEndpoint;
    if (isCustomEndpoint) {
        if (dualstackEndpoint)
            throw new Error("Dualstack endpoint is not supported with custom endpoint");
        if (accelerateEndpoint)
            throw new Error("Accelerate endpoint is not supported with custom endpoint");
    }
    return isBucketNameOptions(options)
        ? // Construct endpoint when bucketName is a string referring to a bucket name
            getEndpointFromBucketName(__assign(__assign({}, options), { isCustomEndpoint: isCustomEndpoint }))
        : // Construct endpoint when bucketName is an ARN referring to an S3 resource like Access Point
            getEndpointFromArn(__assign(__assign({}, options), { isCustomEndpoint: isCustomEndpoint }));
};
var getEndpointFromArn = function (options) {
    var isCustomEndpoint = options.isCustomEndpoint, baseHostname = options.baseHostname;
    var _a = __read(isCustomEndpoint
        ? [options.clientRegion, baseHostname]
        : // Infer client region and hostname suffix from hostname from endpoints.json, like `s3.us-west-2.amazonaws.com`
            getSuffixForArnEndpoint(baseHostname), 2), clientRegion = _a[0], hostnameSuffix = _a[1];
    var pathStyleEndpoint = options.pathStyleEndpoint, _b = options.dualstackEndpoint, dualstackEndpoint = _b === void 0 ? false : _b, _c = options.accelerateEndpoint, accelerateEndpoint = _c === void 0 ? false : _c, _d = options.tlsCompatible, tlsCompatible = _d === void 0 ? true : _d, useArnRegion = options.useArnRegion, bucketName = options.bucketName, _e = options.clientPartition, clientPartition = _e === void 0 ? "aws" : _e, _f = options.clientSigningRegion, clientSigningRegion = _f === void 0 ? clientRegion : _f;
    validateArnEndpointOptions({ pathStyleEndpoint: pathStyleEndpoint, accelerateEndpoint: accelerateEndpoint, tlsCompatible: tlsCompatible });
    // Validate and parse the ARN supplied as a bucket name
    var service = bucketName.service, partition = bucketName.partition, accountId = bucketName.accountId, region = bucketName.region, resource = bucketName.resource;
    validateService(service);
    validatePartition(partition, { clientPartition: clientPartition });
    validateAccountId(accountId);
    validateRegion(region, { useArnRegion: useArnRegion, clientRegion: clientRegion, clientSigningRegion: clientSigningRegion });
    var _g = getArnResources(resource), accesspointName = _g.accesspointName, outpostId = _g.outpostId;
    validateDNSHostLabel(accesspointName + "-" + accountId, { tlsCompatible: tlsCompatible });
    var endpointRegion = useArnRegion ? region : clientRegion;
    var signingRegion = useArnRegion ? region : clientSigningRegion;
    if (outpostId) {
        // if this is an Outpost ARN
        validateOutpostService(service);
        validateDNSHostLabel(outpostId, { tlsCompatible: tlsCompatible });
        validateNoDualstack(dualstackEndpoint);
        validateNoFIPS(endpointRegion);
        var hostnamePrefix_1 = accesspointName + "-" + accountId + "." + outpostId;
        return {
            bucketEndpoint: true,
            hostname: "" + hostnamePrefix_1 + (isCustomEndpoint ? "" : ".s3-outposts." + endpointRegion) + "." + hostnameSuffix,
            signingRegion: signingRegion,
            signingService: "s3-outposts",
        };
    }
    // construct endpoint from Accesspoint ARN
    validateS3Service(service);
    var hostnamePrefix = accesspointName + "-" + accountId;
    return {
        bucketEndpoint: true,
        hostname: "" + hostnamePrefix + (isCustomEndpoint ? "" : ".s3-accesspoint" + (dualstackEndpoint ? ".dualstack" : "") + "." + endpointRegion) + "." + hostnameSuffix,
        signingRegion: signingRegion,
    };
};
var getEndpointFromBucketName = function (_a) {
    var _b = _a.accelerateEndpoint, accelerateEndpoint = _b === void 0 ? false : _b, region = _a.clientRegion, baseHostname = _a.baseHostname, bucketName = _a.bucketName, _c = _a.dualstackEndpoint, dualstackEndpoint = _c === void 0 ? false : _c, _d = _a.pathStyleEndpoint, pathStyleEndpoint = _d === void 0 ? false : _d, _e = _a.tlsCompatible, tlsCompatible = _e === void 0 ? true : _e, _f = _a.isCustomEndpoint, isCustomEndpoint = _f === void 0 ? false : _f;
    var _g = __read(isCustomEndpoint ? [region, baseHostname] : getSuffix(baseHostname), 2), clientRegion = _g[0], hostnameSuffix = _g[1];
    if (pathStyleEndpoint || !isDnsCompatibleBucketName(bucketName) || (tlsCompatible && DOT_PATTERN.test(bucketName))) {
        return {
            bucketEndpoint: false,
            hostname: dualstackEndpoint ? "s3.dualstack." + clientRegion + "." + hostnameSuffix : baseHostname,
        };
    }
    if (accelerateEndpoint) {
        baseHostname = "s3-accelerate" + (dualstackEndpoint ? ".dualstack" : "") + "." + hostnameSuffix;
    }
    else if (dualstackEndpoint) {
        baseHostname = "s3.dualstack." + clientRegion + "." + hostnameSuffix;
    }
    return {
        bucketEndpoint: true,
        hostname: bucketName + "." + baseHostname,
    };
};
//# sourceMappingURL=data:application/json;base64,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