"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const loggerMiddleware_1 = require("./loggerMiddleware");
describe("getLoggerPlugin", () => {
    const mockClientStack = {
        add: jest.fn(),
    };
    afterEach(() => {
        jest.clearAllMocks();
    });
    it("adds loggerMiddleware", () => {
        loggerMiddleware_1.getLoggerPlugin({}).applyToStack(mockClientStack);
        expect(mockClientStack.add).toHaveBeenCalledTimes(1);
        expect(mockClientStack.add.mock.calls[0][1]).toEqual(loggerMiddleware_1.loggerMiddlewareOptions);
    });
});
describe("loggerMiddleware", () => {
    const mockNext = jest.fn();
    const mockArgs = {
        input: {
            inputKey: "inputValue",
        },
        request: {
            method: "GET",
            headers: {},
        },
    };
    const mockOutput = {
        $metadata: {
            statusCode: 200,
            requestId: "requestId",
            attempts: 2,
            totalRetryDelay: 350,
        },
        outputKey: "outputValue",
    };
    const mockResponse = {
        response: {
            statusCode: 200,
            headers: {
                "x-amzn-requestid": "requestId",
                "x-amz-id-2": "extendedRequestId",
                "x-amz-cf-id": "cfId",
            },
        },
        output: mockOutput,
    };
    afterEach(() => {
        jest.clearAllMocks();
    });
    it("returns without logging if context.logger is not defined", async () => {
        mockNext.mockResolvedValueOnce(mockResponse);
        const response = await loggerMiddleware_1.loggerMiddleware()(mockNext, {})(mockArgs);
        expect(mockNext).toHaveBeenCalledTimes(1);
        expect(response).toStrictEqual(mockResponse);
    });
    it("returns without logging if context.logger doesn't have info function", async () => {
        mockNext.mockResolvedValueOnce(mockResponse);
        const logger = {};
        const response = await loggerMiddleware_1.loggerMiddleware()(mockNext, { logger })(mockArgs);
        expect(mockNext).toHaveBeenCalledTimes(1);
        expect(response).toStrictEqual(mockResponse);
    });
    describe("logs if context.logger has info function", () => {
        it("success case with clientName, commandName, input, metadata", async () => {
            mockNext.mockResolvedValueOnce(mockResponse);
            const logger = { info: jest.fn() };
            const clientName = "mockClientName";
            const commandName = "mockCommandName";
            const mockInputLog = { inputKey: "inputKey", inputSensitiveKey: "SENSITIVE_VALUE" };
            const inputFilterSensitiveLog = jest.fn().mockReturnValueOnce(mockInputLog);
            const mockOutputLog = { outputKey: "outputKey", outputSensitiveKey: "SENSITIVE_VALUE" };
            const outputFilterSensitiveLog = jest.fn().mockReturnValueOnce(mockOutputLog);
            const context = {
                logger,
                clientName,
                commandName,
                inputFilterSensitiveLog,
                outputFilterSensitiveLog,
            };
            const response = await loggerMiddleware_1.loggerMiddleware()(mockNext, context)(mockArgs);
            expect(mockNext).toHaveBeenCalledTimes(1);
            expect(response).toStrictEqual(mockResponse);
            expect(inputFilterSensitiveLog).toHaveBeenCalledTimes(1);
            expect(inputFilterSensitiveLog).toHaveBeenCalledWith(mockArgs.input);
            const { $metadata, ...outputWithoutMetadata } = mockOutput;
            expect(outputFilterSensitiveLog).toHaveBeenCalledTimes(1);
            expect(outputFilterSensitiveLog).toHaveBeenCalledWith(outputWithoutMetadata);
            expect(logger.info).toHaveBeenCalledTimes(1);
            expect(logger.info).toHaveBeenCalledWith({
                clientName,
                commandName,
                input: mockInputLog,
                output: mockOutputLog,
                metadata: $metadata,
            });
        });
        it("header x-amzn-request-id as requestId if x-amzn-requestid is not present", async () => {
            const requestIdBackup = "requestIdBackup";
            const customResponse = {
                ...mockResponse,
                response: {
                    ...mockResponse.response,
                    headers: {
                        "x-amzn-request-id": requestIdBackup,
                    },
                },
            };
            mockNext.mockResolvedValueOnce(customResponse);
            const logger = { info: jest.fn() };
            const inputFilterSensitiveLog = jest.fn().mockImplementationOnce((input) => input);
            const outputFilterSensitiveLog = jest.fn().mockImplementationOnce((output) => output);
            const context = {
                logger,
                inputFilterSensitiveLog,
                outputFilterSensitiveLog,
            };
            const response = await loggerMiddleware_1.loggerMiddleware()(mockNext, context)(mockArgs);
            expect(mockNext).toHaveBeenCalledTimes(1);
            expect(response).toStrictEqual(customResponse);
            const { $metadata, ...outputWithoutMetadata } = mockOutput;
            expect(logger.info).toHaveBeenCalledTimes(1);
            expect(logger.info).toHaveBeenCalledWith({
                input: mockArgs.input,
                output: outputWithoutMetadata,
                metadata: $metadata,
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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