"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3RequestPresigner = void 0;
const signature_v4_1 = require("@aws-sdk/signature-v4");
const constants_1 = require("./constants");
class S3RequestPresigner {
    constructor(options) {
        const resolvedOptions = {
            // Allow `signingName` because we want to support usecase of supply client's resolved config
            // directly. Where service equals signingName.
            service: options.signingName || options.service || "s3",
            uriEscapePath: options.uriEscapePath || false,
            ...options,
        };
        this.signer = new signature_v4_1.SignatureV4(resolvedOptions);
    }
    async presign(requestToSign, { unsignableHeaders = new Set(), unhoistableHeaders = new Set(), ...options } = {}) {
        unsignableHeaders.add("content-type");
        // S3 requires SSE headers to be signed in headers instead of query
        // See: https://github.com/aws/aws-sdk-js-v3/issues/1576
        Object.keys(requestToSign.headers)
            .map((header) => header.toLowerCase())
            .filter((header) => header.startsWith("x-amz-server-side-encryption"))
            .forEach((header) => {
            unhoistableHeaders.add(header);
        });
        requestToSign.headers[constants_1.SHA256_HEADER] = constants_1.UNSIGNED_PAYLOAD;
        if (!requestToSign.headers["host"]) {
            requestToSign.headers.host = requestToSign.hostname;
        }
        return this.signer.presign(requestToSign, {
            expiresIn: 900,
            unsignableHeaders,
            unhoistableHeaders,
            ...options,
        });
    }
}
exports.S3RequestPresigner = S3RequestPresigner;
//# sourceMappingURL=data:application/json;base64,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