"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadSharedConfigFiles = exports.ENV_CONFIG_PATH = exports.ENV_CREDENTIALS_PATH = void 0;
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
exports.ENV_CREDENTIALS_PATH = "AWS_SHARED_CREDENTIALS_FILE";
exports.ENV_CONFIG_PATH = "AWS_CONFIG_FILE";
const swallowError = () => ({});
const loadSharedConfigFiles = (init = {}) => {
    const { filepath = process.env[exports.ENV_CREDENTIALS_PATH] || path_1.join(getHomeDir(), ".aws", "credentials"), configFilepath = process.env[exports.ENV_CONFIG_PATH] || path_1.join(getHomeDir(), ".aws", "config"), } = init;
    return Promise.all([
        slurpFile(configFilepath).then(parseIni).then(normalizeConfigFile).catch(swallowError),
        slurpFile(filepath).then(parseIni).catch(swallowError),
    ]).then((parsedFiles) => {
        const [configFile, credentialsFile] = parsedFiles;
        return {
            configFile,
            credentialsFile,
        };
    });
};
exports.loadSharedConfigFiles = loadSharedConfigFiles;
const profileKeyRegex = /^profile\s(["'])?([^\1]+)\1$/;
const normalizeConfigFile = (data) => {
    const map = {};
    for (const key of Object.keys(data)) {
        let matches;
        if (key === "default") {
            map.default = data.default;
        }
        else if ((matches = profileKeyRegex.exec(key))) {
            // eslint-disable-next-line @typescript-eslint/no-unused-vars
            const [_1, _2, normalizedKey] = matches;
            if (normalizedKey) {
                map[normalizedKey] = data[key];
            }
        }
    }
    return map;
};
const profileNameBlockList = ["__proto__", "profile __proto__"];
const parseIni = (iniData) => {
    const map = {};
    let currentSection;
    for (let line of iniData.split(/\r?\n/)) {
        line = line.split(/(^|\s)[;#]/)[0]; // remove comments
        const section = line.match(/^\s*\[([^\[\]]+)]\s*$/);
        if (section) {
            currentSection = section[1];
            if (profileNameBlockList.includes(currentSection)) {
                throw new Error(`Found invalid profile name "${currentSection}"`);
            }
        }
        else if (currentSection) {
            const item = line.match(/^\s*(.+?)\s*=\s*(.+?)\s*$/);
            if (item) {
                map[currentSection] = map[currentSection] || {};
                map[currentSection][item[1]] = item[2];
            }
        }
    }
    return map;
};
const slurpFile = (path) => new Promise((resolve, reject) => {
    fs_1.readFile(path, "utf8", (err, data) => {
        if (err) {
            reject(err);
        }
        else {
            resolve(data);
        }
    });
});
const getHomeDir = () => {
    const { HOME, USERPROFILE, HOMEPATH, HOMEDRIVE = `C:${path_1.sep}` } = process.env;
    if (HOME)
        return HOME;
    if (USERPROFILE)
        return USERPROFILE;
    if (HOMEPATH)
        return `${HOMEDRIVE}${HOMEPATH}`;
    return os_1.homedir();
};
//# sourceMappingURL=data:application/json;base64,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