import { isNode } from "@azure/ms-rest-js";
/**
 * An aborter instance implements AbortSignal interface, can abort HTTP requests.
 *
 * - Call Aborter.none to create a new Aborter instance without timeout.
 * - Call Aborter.timeout() to create a new Aborter instance with timeout.
 *
 * For an existing instance aborter:
 * - Call aborter.withTimeout() to create and return a child Aborter instance with timeout.
 * - Call aborter.withValue(key, value) to create and return a child Aborter instance with key/value pair.
 * - Call aborter.abort() to abort current instance and all children instances.
 * - Call aborter.getValue(key) to search and get value with corresponding key from current aborter to all parents.
 *
 * @example
 * // Abort without timeout
 * await blockBlobURL.upload(Aborter.none, buf, buf.length);
 *
 * @example
 * // Abort container create in 1000ms
 * await blockBlobURL.upload(Aborter.timeout(1000), buf, buf.length);
 *
 * @example
 * // Share aborter cross multiple operations in 30s
 * // Upload the same data to 2 different data centers at the same time, abort another when any of them is finished
 * const aborter = Aborter.timeout(30 * 1000);
 * blockBlobURL1.upload(aborter, buf, buf.length).then(aborter.abort);
 * blockBlobURL2.upload(aborter, buf, buf.length).then(aborter.abort);
 *
 * @example
 * // Cascaded aborting
 * // All operations can't take more than 30 seconds
 * const aborter = Aborter.timeout(30 * 1000);
 *
 * // Following 2 operations can't take more than 25 seconds
 * await blockBlobURL.upload(aborter.withTimeout(25 * 1000), buf, buf.length);
 * await blockBlobURL.upload(aborter.withTimeout(25 * 1000), buf, buf.length);
 *
 * @export
 * @class Aborter
 * @implements {AbortSignalLike}
 */
var Aborter = /** @class */ (function () {
    // private disposed: boolean = false;
    /**
     * Private constructor for internal usage, creates an instance of Aborter.
     *
     * @param {Aborter} [parent] Optional. Parent aborter.
     * @param {number} [timeout=0] Optional. Timeout before abort in millisecond, 0 means no timeout.
     * @param {string} [key] Optional. Immutable key in string.
     * @param {(string | number | boolean | null)} [value] Optional. Immutable value.
     * @memberof Aborter
     */
    function Aborter(parent, timeout, key, value) {
        var _this = this;
        if (timeout === void 0) { timeout = 0; }
        /**
         * onabort event listener.
         *
         * @memberof Aborter
         */
        this.onabort = null;
        // tslint:disable-next-line:variable-name
        this._aborted = false;
        this.children = []; // When child object calls dispose(), remove child from here
        this.abortEventListeners = [];
        this.parent = parent;
        this.key = key;
        this.value = value;
        if (timeout > 0) {
            this.timer = setTimeout(function () {
                _this.abort.call(_this);
            }, timeout);
            // When called, the active Timeout object will not require the Node.js event loop
            // to remain active. If there is no other activity keeping the event loop running,
            // the process may exit before the Timeout object's callback is invoked.
            if (this.timer && isNode) {
                this.timer.unref();
            }
        }
    }
    Object.defineProperty(Aborter.prototype, "aborted", {
        /**
         * Status of whether aborted or not.
         *
         * @readonly
         * @type {boolean}
         * @memberof Aborter
         */
        get: function () {
            return this._aborted;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Aborter, "none", {
        /**
         * Creates a new Aborter instance without timeout.
         *
         * @readonly
         * @static
         * @type {Aborter}
         * @memberof Aborter
         */
        get: function () {
            return new Aborter(undefined, 0);
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Creates a new Aborter instance with timeout in milliseconds.
     * Set parameter timeout to 0 will not create a timer.
     *
     * @static
     * @param {number} {timeout} in milliseconds
     * @returns {Aborter}
     * @memberof Aborter
     */
    Aborter.timeout = function (timeout) {
        return new Aborter(undefined, timeout);
    };
    /**
     * Create and return a new Aborter instance, which will be appended as a child node of current Aborter.
     * Current Aborter instance becomes father node of the new instance. When current or father Aborter node
     * triggers timeout event, all children nodes abort event will be triggered too.
     *
     * When timeout parameter (in millisecond) is larger than 0, the abort event will be triggered when timeout.
     * Otherwise, call abort() method to manually abort.
     *
     * @param {number} {timeout} Timeout in millisecond.
     * @returns {Aborter} The new Aborter instance created.
     * @memberof Aborter
     */
    Aborter.prototype.withTimeout = function (timeout) {
        var childCancelContext = new Aborter(this, timeout);
        this.children.push(childCancelContext);
        return childCancelContext;
    };
    /**
     * Create and return a new Aborter instance, which will be appended as a child node of current Aborter.
     * Current Aborter instance becomes father node of the new instance. When current or father Aborter node
     * triggers timeout event, all children nodes abort event will be triggered too.
     *
     * Immutable key value pair will be set into the new created Aborter instance.
     * Call getValue() to find out latest value with corresponding key in the chain of
     * [current node] -> [parent node] and [grand parent node]....
     *
     * @param {string} key
     * @param {(string | number | boolean | null)} [value]
     * @returns {Aborter}
     * @memberof Aborter
     */
    Aborter.prototype.withValue = function (key, value) {
        var childCancelContext = new Aborter(this, 0, key, value);
        this.children.push(childCancelContext);
        return childCancelContext;
    };
    /**
     * Find out latest value with corresponding key in the chain of
     * [current node] -> [parent node] -> [grand parent node] -> ... -> [root node].
     *
     * If key is not found, undefined will be returned.
     *
     * @param {string} key
     * @returns {(string | number | boolean | null | undefined)}
     * @memberof Aborter
     */
    Aborter.prototype.getValue = function (key) {
        for (var parent_1 = this; parent_1; parent_1 = parent_1.parent) {
            if (parent_1.key === key) {
                return parent_1.value;
            }
        }
        return undefined;
    };
    /**
     * Trigger abort event immediately, the onabort and all abort event listeners will be triggered.
     * Will try to trigger abort event for all children Aborter nodes.
     *
     * - If there is a timeout, the timer will be cancelled.
     * - If aborted is true, nothing will happen.
     *
     * @returns
     * @memberof Aborter
     */
    Aborter.prototype.abort = function () {
        var _this = this;
        if (this.aborted) {
            return;
        }
        this.cancelTimer();
        if (this.onabort) {
            this.onabort.call(this, { type: "abort" });
        }
        this.abortEventListeners.forEach(function (listener) {
            listener.call(_this, { type: "abort" });
        });
        this.children.forEach(function (child) { return child.cancelByParent(); });
        this._aborted = true;
    };
    // public dispose() {
    //   if (this.disposed || this.aborted) {
    //     return;
    //   }
    //   this.cancelTimer();
    //   // (parent)A <- B <- C(child), if B disposes, when A abort, C will not abort
    //   if (this.parent) {
    //     const index = this.parent.children.indexOf(this);
    //     if (index > -1) {
    //       this.parent.children.splice(index, 1);
    //     }
    //   }
    //   this.disposed = true;
    // }
    /**
     * Added new "abort" event listener, only support "abort" event.
     *
     * @param {"abort"} _type Only support "abort" event
     * @param {(this: AbortSignalLike, ev: any) => any} listener
     * @memberof Aborter
     */
    Aborter.prototype.addEventListener = function (
    // tslint:disable-next-line:variable-name
    _type, listener) {
        this.abortEventListeners.push(listener);
    };
    /**
     * Remove "abort" event listener, only support "abort" event.
     *
     * @param {"abort"} _type Only support "abort" event
     * @param {(this: AbortSignalLike, ev: any) => any} listener
     * @memberof Aborter
     */
    Aborter.prototype.removeEventListener = function (
    // tslint:disable-next-line:variable-name
    _type, listener) {
        var index = this.abortEventListeners.indexOf(listener);
        if (index > -1) {
            this.abortEventListeners.splice(index, 1);
        }
    };
    Aborter.prototype.dispatchEvent = function () {
        throw new Error("Method not implemented.");
    };
    Aborter.prototype.cancelByParent = function () {
        // if (!this.disposed) {
        this.abort();
        // }
    };
    Aborter.prototype.cancelTimer = function () {
        if (this.timer) {
            clearTimeout(this.timer);
        }
    };
    return Aborter;
}());
export { Aborter };
//# sourceMappingURL=Aborter.js.map