import * as tslib_1 from "tslib";
import { BlobURL } from "./BlobURL";
import { AppendBlob } from "./generated/src/operations";
import { rangeToString } from "./IRange";
import { ensureCpkIfSpecified } from "./models";
import { URLConstants } from "./utils/constants";
import { appendToURLPath, setURLParameter } from "./utils/utils.common";
/**
 * AppendBlobURL defines a set of operations applicable to append blobs.
 *
 * @export
 * @class AppendBlobURL
 * @extends {StorageURL}
 */
var AppendBlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(AppendBlobURL, _super);
    /**
     * Creates an instance of AppendBlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to an append blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage append blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/appendblob". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/appendblob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof AppendBlobURL
     */
    function AppendBlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.appendBlobContext = new AppendBlob(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a AppendBlobURL object from ContainerURL instance.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName An append blob name
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.fromContainerURL = function (containerURL, blobName) {
        return new AppendBlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a AppendBlobURL object from BlobURL instance.
     *
     * @static
     * @param {BlobURL} blobURL
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.fromBlobURL = function (blobURL) {
        return new AppendBlobURL(blobURL.url, blobURL.pipeline);
    };
    /**
     * Creates a new AppendBlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.withPipeline = function (pipeline) {
        return new AppendBlobURL(this.url, pipeline);
    };
    /**
     * Creates a new AppendBlobURL object identical to the source but with the
     * specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.withSnapshot = function (snapshot) {
        return new AppendBlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Creates a 0-length append blob. Call AppendBlock to append data to an append blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IAppendBlobCreateOptions} [options]
     * @returns {Promise<Models.AppendBlobsCreateResponse>}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.create = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.appendBlobContext.create(0, {
                        abortSignal: aborter,
                        blobHTTPHeaders: options.blobHTTPHeaders,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Commits a new block of data to the end of the existing append blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/append-block
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {HttpRequestBody} body
     * @param {number} contentLength Length of the body in bytes
     * @param {IAppendBlobAppendBlockOptions} [options]
     * @returns {Promise<Models.AppendBlobsAppendBlockResponse>}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.appendBlock = function (aborter, body, contentLength, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.appendBlobContext.appendBlock(body, contentLength, {
                        abortSignal: aborter,
                        appendPositionAccessConditions: options.accessConditions.appendPositionAccessConditions,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        onUploadProgress: options.progress,
                        transactionalContentMD5: options.transactionalContentMD5,
                        transactionalContentCrc64: options.transactionalContentCrc64,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * The Append Block operation commits a new block of data to the end of an existing append blob
     * where the contents are read from a source url.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/append-block-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} sourceURL
     *                 The url to the blob that will be the source of the copy. A source blob in the same storage account can
     *                 be authenticated via Shared Key. However, if the source is a blob in another account, the source blob
     *                 must either be public or must be authenticated via a shared access signature. If the source blob is
     *                 public, no authentication is required to perform the operation.
     * @param {number} sourceOffset Offset in source to be appended
     * @param {number} count Number of bytes to be appended as a block
     * @param {IAppendBlobAppendBlockFromURLOptions} [options={}]
     * @returns {Promise<Models.AppendBlobAppendBlockFromUrlResponse>}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.appendBlockFromURL = function (aborter, sourceURL, sourceOffset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.appendBlobContext.appendBlockFromUrl(sourceURL, 0, {
                        abortSignal: aborter,
                        sourceRange: rangeToString({ offset: sourceOffset, count: count }),
                        sourceContentMD5: options.sourceContentMD5,
                        sourceContentCrc64: options.sourceContentCrc64,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        appendPositionAccessConditions: options.accessConditions.appendPositionAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        },
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    return AppendBlobURL;
}(BlobURL));
export { AppendBlobURL };
//# sourceMappingURL=AppendBlobURL.js.map