import * as tslib_1 from "tslib";
import { isNode } from "@azure/ms-rest-js";
import { BlobDownloadResponse } from "./BlobDownloadResponse";
import { Blob } from "./generated/src/operations";
import { rangeToString } from "./IRange";
import { ensureCpkIfSpecified, toAccessTier } from "./models";
import { StorageURL } from "./StorageURL";
import { DEFAULT_MAX_DOWNLOAD_RETRY_REQUESTS, URLConstants } from "./utils/constants";
import { appendToURLPath, setURLParameter } from "./utils/utils.common";
/**
 * A BlobURL represents a URL to an Azure Storage blob; the blob may be a block blob,
 * append blob, or page blob.
 *
 * @export
 * @class BlobURL
 * @extends {StorageURL}
 */
var BlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(BlobURL, _super);
    /**
     * Creates an instance of BlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to a blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blob".
     *                     You can append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof BlobURL
     */
    function BlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.blobContext = new Blob(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a BlobURL object from an ContainerURL object.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName A blob name
     * @returns
     * @memberof BlobURL
     */
    BlobURL.fromContainerURL = function (containerURL, blobName) {
        return new BlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a new BlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {BlobURL}
     * @memberof BlobURL
     */
    BlobURL.prototype.withPipeline = function (pipeline) {
        return new BlobURL(this.url, pipeline);
    };
    /**
     * Creates a new BlobURL object identical to the source but with the specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {BlobURL} A new BlobURL object identical to the source but with the specified snapshot timestamp
     * @memberof BlobURL
     */
    BlobURL.prototype.withSnapshot = function (snapshot) {
        return new BlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Reads or downloads a blob from the system, including its metadata and properties.
     * You can also call Get Blob to read a snapshot.
     *
     * * In Node.js, data returns in a Readable stream readableStreamBody
     * * In browsers, data returns in a promise blobBody
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset From which position of the blob to download, >= 0
     * @param {number} [count] How much data to be downloaded, > 0. Will download to the end when undefined
     * @param {IBlobDownloadOptions} [options]
     * @returns {Promise<Models.BlobDownloadResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.download = function (aborter, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var res;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        options.blobAccessConditions = options.blobAccessConditions || {};
                        options.blobAccessConditions.modifiedAccessConditions =
                            options.blobAccessConditions.modifiedAccessConditions || {};
                        ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                        return [4 /*yield*/, this.blobContext.download({
                                abortSignal: aborter,
                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                                modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                                onDownloadProgress: isNode ? undefined : options.progress,
                                range: offset === 0 && !count ? undefined : rangeToString({ offset: offset, count: count }),
                                rangeGetContentMD5: options.rangeGetContentMD5,
                                rangeGetContentCRC64: options.rangeGetContentCrc64,
                                snapshot: options.snapshot,
                                cpkInfo: options.customerProvidedKey
                            })];
                    case 1:
                        res = _a.sent();
                        // Return browser response immediately
                        if (!isNode) {
                            return [2 /*return*/, res];
                        }
                        // We support retrying when download stream unexpected ends in Node.js runtime
                        // Following code shouldn't be bundled into browser build, however some
                        // bundlers may try to bundle following code and "FileReadResponse.ts".
                        // In this case, "FileDownloadResponse.browser.ts" will be used as a shim of "FileDownloadResponse.ts"
                        // The config is in package.json "browser" field
                        if (options.maxRetryRequests === undefined || options.maxRetryRequests < 0) {
                            // TODO: Default value or make it a required parameter?
                            options.maxRetryRequests = DEFAULT_MAX_DOWNLOAD_RETRY_REQUESTS;
                        }
                        if (res.contentLength === undefined) {
                            throw new RangeError("File download response doesn't contain valid content length header");
                        }
                        if (!res.eTag) {
                            throw new RangeError("File download response doesn't contain valid etag header");
                        }
                        return [2 /*return*/, new BlobDownloadResponse(aborter, res, function (start) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                var updatedOptions;
                                return tslib_1.__generator(this, function (_a) {
                                    switch (_a.label) {
                                        case 0:
                                            updatedOptions = {
                                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                                                modifiedAccessConditions: {
                                                    ifMatch: options.blobAccessConditions.modifiedAccessConditions.ifMatch || res.eTag,
                                                    ifModifiedSince: options.blobAccessConditions.modifiedAccessConditions
                                                        .ifModifiedSince,
                                                    ifNoneMatch: options.blobAccessConditions.modifiedAccessConditions.ifNoneMatch,
                                                    ifUnmodifiedSince: options.blobAccessConditions.modifiedAccessConditions
                                                        .ifUnmodifiedSince
                                                },
                                                range: rangeToString({
                                                    count: offset + res.contentLength - start,
                                                    offset: start
                                                }),
                                                rangeGetContentMD5: options.rangeGetContentMD5,
                                                rangeGetContentCRC64: options.rangeGetContentCrc64,
                                                snapshot: options.snapshot,
                                                cpkInfo: options.customerProvidedKey
                                            };
                                            return [4 /*yield*/, this.blobContext.download(tslib_1.__assign({ abortSignal: aborter }, updatedOptions))];
                                        case 1: 
                                        // Debug purpose only
                                        // console.log(
                                        //   `Read from internal stream, range: ${
                                        //     updatedOptions.range
                                        //   }, options: ${JSON.stringify(updatedOptions)}`
                                        // );
                                        return [2 /*return*/, (_a.sent()).readableStreamBody];
                                    }
                                });
                            }); }, offset, res.contentLength, {
                                maxRetryRequests: options.maxRetryRequests,
                                progress: options.progress
                            })];
                }
            });
        });
    };
    /**
     * Returns all user-defined metadata, standard HTTP properties, and system properties
     * for the blob. It does not return the content of the blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobGetPropertiesOptions} [options]
     * @returns {Promise<Models.BlobGetPropertiesResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.getProperties = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.getProperties({
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Marks the specified blob or snapshot for deletion. The blob is later deleted
     * during garbage collection. Note that in order to delete a blob, you must delete
     * all of its snapshots. You can delete both at the same time with the Delete
     * Blob operation.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/delete-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobDeleteOptions} [options]
     * @returns {Promise<Models.BlobDeleteResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.delete = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                return [2 /*return*/, this.blobContext.deleteMethod({
                        abortSignal: aborter,
                        deleteSnapshots: options.deleteSnapshots,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Restores the contents and metadata of soft deleted blob and any associated
     * soft deleted snapshots. Undelete Blob is supported only on version 2017-07-29
     * or later.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/undelete-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.BlobUndeleteResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.undelete = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.undelete({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Sets system properties on the blob.
     *
     * If no value provided, or no value provided for the specificed blob HTTP headers,
     * these blob HTTP headers without a value will be cleared.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.BlobHTTPHeaders} [blobHTTPHeaders] If no value provided, or no value provided for
     *                                                   the specificed blob HTTP headers, these blob HTTP
     *                                                   headers without a value will be cleared.
     * @param {IBlobSetHTTPHeadersOptions} [options]
     * @returns {Promise<Models.BlobSetHTTPHeadersResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.setHTTPHeaders = function (aborter, blobHTTPHeaders, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.setHTTPHeaders({
                        abortSignal: aborter,
                        blobHTTPHeaders: blobHTTPHeaders,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Sets user-defined metadata for the specified blob as one or more name-value pairs.
     *
     * If no option provided, or no metadata defined in the parameter, the blob
     * metadata will be removed.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-metadata
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IMetadata} [metadata] Replace existing metadata with this value.
     *                               If no value provided the existing metadata will be removed.
     * @param {IBlobSetMetadataOptions} [options]
     * @returns {Promise<Models.BlobSetMetadataResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.setMetadata = function (aborter, metadata, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.setMetadata({
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        metadata: metadata,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Establishes and manages a lock on a blob for write and delete operations.
     * The lock duration can be 15 to 60 seconds, or can be infinite.
     * In versions prior to 2012-02-12, the lock duration is 60 seconds.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} proposedLeaseId Can be specified in any valid GUID string format
     * @param {number} duration The lock duration can be 15 to 60 seconds, or can be infinite
     * @param {IBlobAcquireLeaseOptions} [options]
     * @returns {Promise<Models.BlobAcquireLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.acquireLease = function (aborter, proposedLeaseId, duration, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.acquireLease({
                        abortSignal: aborter,
                        duration: duration,
                        modifiedAccessConditions: options.modifiedAccessConditions,
                        proposedLeaseId: proposedLeaseId
                    })];
            });
        });
    };
    /**
     * To free the lease if it is no longer needed so that another client may immediately
     * acquire a lease against the blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IBlobReleaseLeaseOptions} [options]
     * @returns {Promise<Models.BlobReleaseLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.releaseLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.releaseLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To renew an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IBlobRenewLeaseOptions} [options]
     * @returns {Promise<Models.BlobRenewLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.renewLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.renewLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To change the ID of an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {string} proposedLeaseId
     * @param {IBlobChangeLeaseOptions} [options]
     * @returns {Promise<Models.BlobChangeLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.changeLease = function (aborter, leaseId, proposedLeaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.changeLease(leaseId, proposedLeaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To end the lease but ensure that another client cannot acquire a new lease
     * until the current lease period has expired.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} [breakPeriod]
     * @param {IBlobBreakLeaseOptions} [options]
     * @returns {Promise<Models.BlobBreakLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.breakLease = function (aborter, breakPeriod, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.breakLease({
                        abortSignal: aborter,
                        breakPeriod: breakPeriod,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Creates a read-only snapshot of a blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/snapshot-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobCreateSnapshotOptions} [options]
     * @returns {Promise<Models.BlobCreateSnapshotResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.createSnapshot = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.createSnapshot({
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Asynchronously copies a blob to a destination within the storage account.
     * In version 2012-02-12 and later, the source for a Copy Blob operation can be
     * a committed blob in any Azure storage account.
     * Beginning with version 2015-02-21, the source for a Copy Blob operation can be
     * an Azure file in any Azure storage account.
     * Only storage accounts created on or after June 7th, 2012 allow the Copy Blob
     * operation to copy from another storage account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/copy-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource
     * @param {IBlobStartCopyFromURLOptions} [options]
     * @returns {Promise<Models.BlobStartCopyFromURLResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.startCopyFromURL = function (aborter, copySource, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                return [2 /*return*/, this.blobContext.startCopyFromURL(copySource, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        },
                        rehydratePriority: options.rehydratePriority,
                        tier: toAccessTier(options.tier)
                    })];
            });
        });
    };
    /**
     * Aborts a pending asynchronous Copy Blob operation, and leaves a destination blob with zero
     * length and full metadata. Version 2012-02-12 and newer.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/abort-copy-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copyId
     * @param {IBlobAbortCopyFromURLOptions} [options]
     * @returns {Promise<Models.BlobAbortCopyFromURLResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.abortCopyFromURL = function (aborter, copyId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.abortCopyFromURL(copyId, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.leaseAccessConditions
                    })];
            });
        });
    };
    /**
     * The synchronous Copy From URL operation copies a blob or an internet resource to a new blob. It will not
     * return a response until the copy is complete.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/copy-blob-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource The source URL to copy from, Shared Access Signature(SAS) maybe needed for authentication
     * @param {IBlobSyncCopyFromURLOptions} [options={}]
     * @returns {Promise<Models.BlobCopyFromURLResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.syncCopyFromURL = function (aborter, copySource, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                return [2 /*return*/, this.blobContext.copyFromURL(copySource, {
                        abortSignal: aborter,
                        metadata: options.metadata,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        }
                    })];
            });
        });
    };
    /**
     * Sets the tier on a blob. The operation is allowed on a page blob in a premium
     * storage account and on a block blob in a blob storage account (locally redundant
     * storage only). A premium page blob's tier determines the allowed size, IOPS,
     * and bandwidth of the blob. A block blob's tier determines Hot/Cool/Archive
     * storage type. This operation does not update the blob's ETag.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-tier
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {BlockBlobTier | PremiumPageBlobTier | string} tier
     * @param {IBlobSetTierOptions} [options]
     * @returns {Promise<Models.BlobsSetTierResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.setTier = function (aborter, tier, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.blobContext.setTier(toAccessTier(tier), {
                            abortSignal: aborter,
                            leaseAccessConditions: options.leaseAccessConditions,
                            rehydratePriority: options.rehydratePriority
                        })];
                    case 1: return [2 /*return*/, _a.sent()];
                }
            });
        });
    };
    return BlobURL;
}(StorageURL));
export { BlobURL };
//# sourceMappingURL=BlobURL.js.map