import * as tslib_1 from "tslib";
import { Container } from "./generated/src/operations";
import { StorageURL } from "./StorageURL";
import { ETagNone } from "./utils/constants";
import { appendToURLPath, truncatedISO8061Date } from "./utils/utils.common";
/**
 * A ContainerURL represents a URL to the Azure Storage container allowing you to manipulate its blobs.
 *
 * @export
 * @class ContainerURL
 * @extends {StorageURL}
 */
var ContainerURL = /** @class */ (function (_super) {
    tslib_1.__extends(ContainerURL, _super);
    /**
     * Creates an instance of ContainerURL.
     * @param {string} url A URL string pointing to Azure Storage blob container, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer?sasString".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof ContainerURL
     */
    function ContainerURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.containerContext = new Container(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a ContainerURL object from ServiceURL
     *
     * @param serviceURL A ServiceURL object
     * @param containerName A container name
     */
    ContainerURL.fromServiceURL = function (serviceURL, containerName) {
        return new ContainerURL(appendToURLPath(serviceURL.url, encodeURIComponent(containerName)), serviceURL.pipeline);
    };
    /**
     * Creates a new ContainerURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {ContainerURL}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.withPipeline = function (pipeline) {
        return new ContainerURL(this.url, pipeline);
    };
    /**
     * Creates a new container under the specified account. If the container with
     * the same name already exists, the operation fails.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/create-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IContainerCreateOptions} [options]
     * @returns {Promise<Models.ContainerCreateResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.create = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                // Spread operator in destructuring assignments,
                // this will filter out unwanted properties from the response object into result object
                return [2 /*return*/, this.containerContext.create(tslib_1.__assign({}, options, { abortSignal: aborter }))];
            });
        });
    };
    /**
     * Returns all user-defined metadata and system properties for the specified
     * container. The data returned does not include the container's list of blobs.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-container-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IContainersGetPropertiesOptions} [options]
     * @returns {Promise<Models.ContainerGetPropertiesResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.getProperties = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!options.leaseAccessConditions) {
                    options.leaseAccessConditions = {};
                }
                return [2 /*return*/, this.containerContext.getProperties(tslib_1.__assign({ abortSignal: aborter }, options.leaseAccessConditions))];
            });
        });
    };
    /**
     * Marks the specified container for deletion. The container and any blobs
     * contained within it are later deleted during garbage collection.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/delete-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.ContainersDeleteMethodOptionalParams} [options]
     * @returns {Promise<Models.ContainerDeleteResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.delete = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!options.containerAccessConditions) {
                    options.containerAccessConditions = {};
                }
                if (!options.containerAccessConditions.modifiedAccessConditions) {
                    options.containerAccessConditions.modifiedAccessConditions = {};
                }
                if (!options.containerAccessConditions.leaseAccessConditions) {
                    options.containerAccessConditions.leaseAccessConditions = {};
                }
                if ((options.containerAccessConditions.modifiedAccessConditions.ifMatch &&
                    options.containerAccessConditions.modifiedAccessConditions.ifMatch !== ETagNone) ||
                    (options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch &&
                        options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch !== ETagNone)) {
                    throw new RangeError("the IfMatch and IfNoneMatch access conditions must have their default\
        values because they are ignored by the service");
                }
                return [2 /*return*/, this.containerContext.deleteMethod({
                        abortSignal: aborter,
                        leaseAccessConditions: options.containerAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.containerAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Sets one or more user-defined name-value pairs for the specified container.
     *
     * If no option provided, or no metadata defined in the parameter, the container
     * metadata will be removed.
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-container-metadata
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IMetadata} [metadata] Replace existing metadata with this value.
     *                               If no value provided the existing metadata will be removed.
     * @param {IContainerSetMetadataOptions} [options]
     * @returns {Promise<Models.ContainerSetMetadataResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.setMetadata = function (aborter, metadata, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!options.containerAccessConditions) {
                    options.containerAccessConditions = {};
                }
                if (!options.containerAccessConditions.modifiedAccessConditions) {
                    options.containerAccessConditions.modifiedAccessConditions = {};
                }
                if (!options.containerAccessConditions.leaseAccessConditions) {
                    options.containerAccessConditions.leaseAccessConditions = {};
                }
                if (options.containerAccessConditions.modifiedAccessConditions.ifUnmodifiedSince ||
                    (options.containerAccessConditions.modifiedAccessConditions.ifMatch &&
                        options.containerAccessConditions.modifiedAccessConditions.ifMatch !== ETagNone) ||
                    (options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch &&
                        options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch !== ETagNone)) {
                    throw new RangeError("the IfUnmodifiedSince, IfMatch, and IfNoneMatch must have their default values\
        because they are ignored by the blob service");
                }
                return [2 /*return*/, this.containerContext.setMetadata({
                        abortSignal: aborter,
                        leaseAccessConditions: options.containerAccessConditions.leaseAccessConditions,
                        metadata: metadata,
                        modifiedAccessConditions: options.containerAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Gets the permissions for the specified container. The permissions indicate
     * whether container data may be accessed publicly.
     *
     * WARNING: JavaScript Date will potential lost precision when parsing start and expiry string.
     * For example, new Date("2018-12-31T03:44:23.8827891Z").toISOString() will get "2018-12-31T03:44:23.882Z".
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-container-acl
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IContainerGetAccessPolicyOptions} [options]
     * @returns {Promise<ContainerGetAccessPolicyResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.getAccessPolicy = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var response, res, _i, response_1, identifier, accessPolicy;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!options.leaseAccessConditions) {
                            options.leaseAccessConditions = {};
                        }
                        return [4 /*yield*/, this.containerContext.getAccessPolicy({
                                abortSignal: aborter,
                                leaseAccessConditions: options.leaseAccessConditions
                            })];
                    case 1:
                        response = _a.sent();
                        res = {
                            _response: response._response,
                            blobPublicAccess: response.blobPublicAccess,
                            date: response.date,
                            eTag: response.eTag,
                            errorCode: response.errorCode,
                            lastModified: response.lastModified,
                            requestId: response.requestId,
                            clientRequestId: response.clientRequestId,
                            signedIdentifiers: [],
                            version: response.version
                        };
                        for (_i = 0, response_1 = response; _i < response_1.length; _i++) {
                            identifier = response_1[_i];
                            accessPolicy = {
                                permission: identifier.accessPolicy.permission,
                            };
                            if (identifier.accessPolicy.expiry) {
                                accessPolicy.expiry = new Date(identifier.accessPolicy.expiry);
                            }
                            if (identifier.accessPolicy.start) {
                                accessPolicy.start = new Date(identifier.accessPolicy.start);
                            }
                            res.signedIdentifiers.push({
                                accessPolicy: accessPolicy,
                                id: identifier.id
                            });
                        }
                        return [2 /*return*/, res];
                }
            });
        });
    };
    /**
     * Sets the permissions for the specified container. The permissions indicate
     * whether blobs in a container may be accessed publicly.
     *
     * When you set permissions for a container, the existing permissions are replaced.
     * If no access or containerAcl provided, the existing container ACL will be
     * removed.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-container-acl
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {PublicAccessType} [access]
     * @param {ISignedIdentifier[]} [containerAcl]
     * @param {IContainerSetAccessPolicyOptions} [options]
     * @returns {Promise<Models.ContainerSetAccessPolicyResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.setAccessPolicy = function (aborter, access, containerAcl, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var acl, _i, _a, identifier;
            return tslib_1.__generator(this, function (_b) {
                options.containerAccessConditions = options.containerAccessConditions || {};
                acl = [];
                for (_i = 0, _a = containerAcl || []; _i < _a.length; _i++) {
                    identifier = _a[_i];
                    acl.push({
                        accessPolicy: {
                            expiry: identifier.accessPolicy.expiry ? truncatedISO8061Date(identifier.accessPolicy.expiry) : "",
                            permission: identifier.accessPolicy.permission,
                            start: identifier.accessPolicy.start ? truncatedISO8061Date(identifier.accessPolicy.start) : ""
                        },
                        id: identifier.id
                    });
                }
                return [2 /*return*/, this.containerContext.setAccessPolicy({
                        abortSignal: aborter,
                        access: access,
                        containerAcl: acl,
                        leaseAccessConditions: options.containerAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.containerAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Establishes and manages a lock on a container for delete operations.
     * The lock duration can be 15 to 60 seconds, or can be infinite.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} proposedLeaseId Can be specified in any valid GUID string format
     * @param {number} duration Must be between 15 to 60 seconds, or infinite (-1)
     * @param {IContainerAcquireLeaseOptions} [options]
     * @returns {Promise<Models.ContainerAcquireLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.acquireLease = function (aborter, proposedLeaseId, duration, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.acquireLease({
                        abortSignal: aborter,
                        duration: duration,
                        modifiedAccessConditions: options.modifiedAccessConditions,
                        proposedLeaseId: proposedLeaseId
                    })];
            });
        });
    };
    /**
     * To free the lease if it is no longer needed so that another client may
     * immediately acquire a lease against the container.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IContainerReleaseLeaseOptions} [options]
     * @returns {Promise<Models.ContainerReleaseLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.releaseLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.releaseLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To renew an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IContainerRenewLeaseOptions} [options]
     * @returns {Promise<Models.ContainerRenewLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.renewLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.renewLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To end the lease but ensure that another client cannot acquire a new lease
     * until the current lease period has expired.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} period break period
     * @param {IContainerBreakLeaseOptions} [options]
     * @returns {Promise<Models.ContainerBreakLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.breakLease = function (aborter, period, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.breakLease({
                        abortSignal: aborter,
                        breakPeriod: period,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To change the ID of an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {string} proposedLeaseId
     * @param {IContainerChangeLeaseOptions} [options]
     * @returns {Promise<Models.ContainerChangeLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.changeLease = function (aborter, leaseId, proposedLeaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.changeLease(leaseId, proposedLeaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * listBlobFlatSegment returns a single segment of blobs starting from the
     * specified Marker. Use an empty Marker to start enumeration from the beginning.
     * After getting a segment, process it, and then call ListBlobsFlatSegment again
     * (passing the the previously-returned Marker) to get the next segment.
     * @see https://docs.microsoft.com/rest/api/storageservices/list-blobs
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} [marker]
     * @param {IContainerListBlobsSegmentOptions} [options]
     * @returns {Promise<Models.ContainerListBlobFlatSegmentResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.listBlobFlatSegment = function (aborter, marker, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.listBlobFlatSegment(tslib_1.__assign({ abortSignal: aborter, marker: marker }, options))];
            });
        });
    };
    /**
     * listBlobHierarchySegment returns a single segment of blobs starting from
     * the specified Marker. Use an empty Marker to start enumeration from the
     * beginning. After getting a segment, process it, and then call ListBlobsHierarchicalSegment
     * again (passing the the previously-returned Marker) to get the next segment.
     * @see https://docs.microsoft.com/rest/api/storageservices/list-blobs
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} delimiter
     * @param {IContainerListBlobsSegmentOptions} [options]
     * @returns {Promise<Models.ContainerListBlobHierarchySegmentResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.listBlobHierarchySegment = function (aborter, delimiter, marker, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.listBlobHierarchySegment(delimiter, tslib_1.__assign({ abortSignal: aborter, marker: marker }, options))];
            });
        });
    };
    return ContainerURL;
}(StorageURL));
export { ContainerURL };
//# sourceMappingURL=ContainerURL.js.map