import * as tslib_1 from "tslib";
import { BlobURL } from "./BlobURL";
import { PageBlob } from "./generated/src/operations";
import { rangeToString } from "./IRange";
import { ensureCpkIfSpecified, toAccessTier } from "./models";
import { URLConstants } from "./utils/constants";
import { appendToURLPath, setURLParameter } from "./utils/utils.common";
/**
 * PageBlobURL defines a set of operations applicable to page blobs.
 *
 * @export
 * @class PageBlobURL
 * @extends {StorageURL}
 */
var PageBlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(PageBlobURL, _super);
    /**
     * Creates an instance of PageBlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to a page blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage page blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/pageblob". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/pageblob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof PageBlobURL
     */
    function PageBlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.pageBlobContext = new PageBlob(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a PageBlobURL object from ContainerURL instance.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName A page blob name
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.fromContainerURL = function (containerURL, blobName) {
        return new PageBlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a PageBlobURL object from BlobURL instance.
     *
     * @static
     * @param {BlobURL} blobURL
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.fromBlobURL = function (blobURL) {
        return new PageBlobURL(blobURL.url, blobURL.pipeline);
    };
    /**
     * Creates a new PageBlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.withPipeline = function (pipeline) {
        return new PageBlobURL(this.url, pipeline);
    };
    /**
     * Creates a new PageBlobURL object identical to the source but with the
     * specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.withSnapshot = function (snapshot) {
        return new PageBlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Creates a page blob of the specified length. Call uploadPages to upload data
     * data to a page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} size
     * @param {IPageBlobCreateOptions} [options]
     * @returns {Promise<Models.PageBlobCreateResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.create = function (aborter, size, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.pageBlobContext.create(0, size, {
                        abortSignal: aborter,
                        blobHTTPHeaders: options.blobHTTPHeaders,
                        blobSequenceNumber: options.blobSequenceNumber,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey,
                        tier: toAccessTier(options.tier)
                    })];
            });
        });
    };
    /**
     * Writes 1 or more pages to the page blob. The start and end offsets must be a multiple of 512.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-page
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {HttpRequestBody} body
     * @param {number} offset Offset of destination page blob
     * @param {number} count Content length of the body, also number of bytes to be uploaded
     * @param {IPageBlobUploadPagesOptions} [options]
     * @returns {Promise<Models.PageBlobsUploadPagesResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.uploadPages = function (aborter, body, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.pageBlobContext.uploadPages(body, count, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        onUploadProgress: options.progress,
                        range: rangeToString({ offset: offset, count: count }),
                        sequenceNumberAccessConditions: options.accessConditions.sequenceNumberAccessConditions,
                        transactionalContentMD5: options.transactionalContentMD5,
                        transactionalContentCrc64: options.transactionalContentCrc64,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * The Upload Pages operation writes a range of pages to a page blob where the
     * contents are read from a URL.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/put-page-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} sourceURL Specify a URL to the copy source, Shared Access Signature(SAS) maybe needed for authentication
     * @param {number} sourceOffset The source offset to copy from. Pass 0 to copy from the beginning of source page blob
     * @param {number} destOffset Offset of destination page blob
     * @param {number} count Number of bytes to be uploaded from source page blob
     * @param {IPageBlobUploadPagesFromURLOptions} [options={}]
     * @returns {Promise<Models.PageBlobUploadPagesFromURLResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.uploadPagesFromURL = function (aborter, sourceURL, sourceOffset, destOffset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.pageBlobContext.uploadPagesFromURL(sourceURL, rangeToString({ offset: sourceOffset, count: count }), 0, rangeToString({ offset: destOffset, count: count }), {
                        abortSignal: aborter,
                        sourceContentMD5: options.sourceContentMD5,
                        sourceContentCrc64: options.sourceContentCrc64,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        sequenceNumberAccessConditions: options.accessConditions.sequenceNumberAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        },
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Frees the specified pages from the page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-page
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset
     * @param {number} count
     * @param {IPageBlobClearPagesOptions} [options]
     * @returns {Promise<Models.PageBlobClearPagesResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.clearPages = function (aborter, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.clearPages(0, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        range: rangeToString({ offset: offset, count: count }),
                        sequenceNumberAccessConditions: options.accessConditions.sequenceNumberAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Returns the list of valid page ranges for a page blob or snapshot of a page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/get-page-ranges
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset
     * @param {number} count
     * @param {IPageBlobGetPageRangesOptions} [options]
     * @returns {Promise<Models.PageBlobGetPageRangesResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.getPageRanges = function (aborter, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.getPageRanges({
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        range: rangeToString({ offset: offset, count: count })
                    })];
            });
        });
    };
    /**
     * Gets the collection of page ranges that differ between a specified snapshot and this page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/get-page-ranges
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset
     * @param {number} count
     * @param {string} prevSnapshot
     * @param {IPageBlobGetPageRangesDiffOptions} [options]
     * @returns {Promise<Models.PageBlobGetPageRangesDiffResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.getPageRangesDiff = function (aborter, offset, count, prevSnapshot, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.getPageRangesDiff({
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        prevsnapshot: prevSnapshot,
                        range: rangeToString({ offset: offset, count: count })
                    })];
            });
        });
    };
    /**
     * Resizes the page blob to the specified size (which must be a multiple of 512).
     * @see https://docs.microsoft.com/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} size
     * @param {IPageBlobResizeOptions} [options]
     * @returns {Promise<Models.PageBlobResizeResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.resize = function (aborter, size, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.resize(size, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Sets a page blob's sequence number.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.SequenceNumberActionType} sequenceNumberAction
     * @param {number} [sequenceNumber] Required if sequenceNumberAction is max or update
     * @param {IPageBlobUpdateSequenceNumberOptions} [options]
     * @returns {Promise<Models.PageBlobUpdateSequenceNumberResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.updateSequenceNumber = function (aborter, sequenceNumberAction, sequenceNumber, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.updateSequenceNumber(sequenceNumberAction, {
                        abortSignal: aborter,
                        blobSequenceNumber: sequenceNumber,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Begins an operation to start an incremental copy from one page blob's snapshot to this page blob.
     * The snapshot is copied such that only the differential changes between the previously
     * copied snapshot are transferred to the destination.
     * The copied snapshots are complete copies of the original snapshot and can be read or copied from as usual.
     * @see https://docs.microsoft.com/rest/api/storageservices/incremental-copy-blob
     * @see https://docs.microsoft.com/en-us/azure/virtual-machines/windows/incremental-snapshots
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource Specifies the name of the source page blob snapshot. For example,
     *                            https://myaccount.blob.core.windows.net/mycontainer/myblob?snapshot=<DateTime>
     * @param {IPageBlobStartCopyIncrementalOptions} [options]
     * @returns {Promise<Models.PageBlobCopyIncrementalResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.startCopyIncremental = function (aborter, copySource, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.pageBlobContext.copyIncremental(copySource, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    return PageBlobURL;
}(BlobURL));
export { PageBlobURL };
//# sourceMappingURL=PageBlobURL.js.map