import { ipRangeToString } from "./IIPRange";
import { truncatedISO8061Date } from "./utils/utils.common";
/**
 * Protocols for generated SAS.
 *
 * @export
 * @enum {number}
 */
export var SASProtocol;
(function (SASProtocol) {
    /**
     * Protocol that allows HTTPS only
     */
    SASProtocol["HTTPS"] = "https";
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    SASProtocol["HTTPSandHTTP"] = "https,http";
})(SASProtocol || (SASProtocol = {}));
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link IAccountSASSignatureValues} and {@link IBlobSASSignatureValues}
 * types. Once generated, it can be encoded into a {@code String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 *
 * @export
 * @class SASQueryParameters
 */
var SASQueryParameters = /** @class */ (function () {
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param {string} version Representing the storage version
     * @param {string} signature Representing the signature for the SAS token
     * @param {string} [permissions] Representing the storage permissions
     * @param {string} [services] Representing the storage services being accessed (only for Account SAS)
     * @param {string} [resourceTypes] Representing the storage resource types being accessed (only for Account SAS)
     * @param {SASProtocol} [protocol] Representing the allowed HTTP protocol(s)
     * @param {Date} [startTime] Representing the start time for this SAS token
     * @param {Date} [expiryTime] Representing the expiry time for this SAS token
     * @param {IIPRange} [ipRange] Representing the range of valid IP addresses for this SAS token
     * @param {string} [identifier] Representing the signed identifier (only for Service SAS)
     * @param {string} [resource] Representing the storage container or blob (only for Service SAS)
     * @param {string} [cacheControl] Representing the cache-control header (only for Blob/File Service SAS)
     * @param {string} [contentDisposition] Representing the content-disposition header (only for Blob/File Service SAS)
     * @param {string} [contentEncoding] Representing the content-encoding header (only for Blob/File Service SAS)
     * @param {string} [contentLanguage] Representing the content-language header (only for Blob/File Service SAS)
     * @param {string} [contentType] Representing the content-type header (only for Blob/File Service SAS)
     * @param {userDelegationKey} [userDelegationKey] Representing the user delegation key properties
     * @memberof SASQueryParameters
     */
    function SASQueryParameters(version, signature, permissions, services, resourceTypes, protocol, startTime, expiryTime, ipRange, identifier, resource, cacheControl, contentDisposition, contentEncoding, contentLanguage, contentType, userDelegationKey) {
        this.version = version;
        this.services = services;
        this.resourceTypes = resourceTypes;
        this.expiryTime = expiryTime;
        this.permissions = permissions;
        this.protocol = protocol;
        this.startTime = startTime;
        this.ipRangeInner = ipRange;
        this.identifier = identifier;
        this.resource = resource;
        this.signature = signature;
        this.cacheControl = cacheControl;
        this.contentDisposition = contentDisposition;
        this.contentEncoding = contentEncoding;
        this.contentLanguage = contentLanguage;
        this.contentType = contentType;
        if (userDelegationKey) {
            this.signedOid = userDelegationKey.signedOid;
            this.signedTid = userDelegationKey.signedTid;
            this.signedStart = userDelegationKey.signedStart;
            this.signedExpiry = userDelegationKey.signedExpiry;
            this.signedService = userDelegationKey.signedService;
            this.signedVersion = userDelegationKey.signedVersion;
        }
    }
    Object.defineProperty(SASQueryParameters.prototype, "ipRange", {
        /**
         * Optional. IP range allowed for this SAS.
         *
         * @readonly
         * @type {(IIPRange | undefined)}
         * @memberof SASQueryParameters
         */
        get: function () {
            if (this.ipRangeInner) {
                return {
                    end: this.ipRangeInner.end,
                    start: this.ipRangeInner.start
                };
            }
            return undefined;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     * @returns {string}
     * @memberof SASQueryParameters
     */
    SASQueryParameters.prototype.toString = function () {
        var params = [
            "sv",
            "ss",
            "srt",
            "spr",
            "st",
            "se",
            "sip",
            "si",
            "skoid",
            "sktid",
            "skt",
            "ske",
            "sks",
            "skv",
            "sr",
            "sp",
            "sig",
            "rscc",
            "rscd",
            "rsce",
            "rscl",
            "rsct"
        ];
        var queries = [];
        for (var _i = 0, params_1 = params; _i < params_1.length; _i++) {
            var param = params_1[_i];
            switch (param) {
                case "sv":
                    this.tryAppendQueryParameter(queries, param, this.version);
                    break;
                case "ss":
                    this.tryAppendQueryParameter(queries, param, this.services);
                    break;
                case "srt":
                    this.tryAppendQueryParameter(queries, param, this.resourceTypes);
                    break;
                case "spr":
                    this.tryAppendQueryParameter(queries, param, this.protocol);
                    break;
                case "st":
                    this.tryAppendQueryParameter(queries, param, this.startTime ? truncatedISO8061Date(this.startTime, false) : undefined);
                    break;
                case "se":
                    this.tryAppendQueryParameter(queries, param, this.expiryTime ? truncatedISO8061Date(this.expiryTime, false) : undefined);
                    break;
                case "sip":
                    this.tryAppendQueryParameter(queries, param, this.ipRange ? ipRangeToString(this.ipRange) : undefined);
                    break;
                case "si":
                    this.tryAppendQueryParameter(queries, param, this.identifier);
                    break;
                case "skoid": // Signed object ID
                    this.tryAppendQueryParameter(queries, param, this.signedOid);
                    break;
                case "sktid": // Signed tenant ID
                    this.tryAppendQueryParameter(queries, param, this.signedTid);
                    break;
                case "skt": // Signed key start time
                    this.tryAppendQueryParameter(queries, param, this.signedStart ? truncatedISO8061Date(this.signedStart, false) : undefined);
                    break;
                case "ske": // Signed key expiry time
                    this.tryAppendQueryParameter(queries, param, this.signedExpiry ? truncatedISO8061Date(this.signedExpiry, false) : undefined);
                    break;
                case "sks": // Signed key service
                    this.tryAppendQueryParameter(queries, param, this.signedService);
                    break;
                case "skv": // Signed key version
                    this.tryAppendQueryParameter(queries, param, this.signedVersion);
                    break;
                case "sr":
                    this.tryAppendQueryParameter(queries, param, this.resource);
                    break;
                case "sp":
                    this.tryAppendQueryParameter(queries, param, this.permissions);
                    break;
                case "sig":
                    this.tryAppendQueryParameter(queries, param, this.signature);
                    break;
                case "rscc":
                    this.tryAppendQueryParameter(queries, param, this.cacheControl);
                    break;
                case "rscd":
                    this.tryAppendQueryParameter(queries, param, this.contentDisposition);
                    break;
                case "rsce":
                    this.tryAppendQueryParameter(queries, param, this.contentEncoding);
                    break;
                case "rscl":
                    this.tryAppendQueryParameter(queries, param, this.contentLanguage);
                    break;
                case "rsct":
                    this.tryAppendQueryParameter(queries, param, this.contentType);
                    break;
            }
        }
        return queries.join("&");
    };
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @private
     * @param {string[]} queries
     * @param {string} key
     * @param {string} [value]
     * @returns {void}
     * @memberof SASQueryParameters
     */
    SASQueryParameters.prototype.tryAppendQueryParameter = function (queries, key, value) {
        if (!value) {
            return;
        }
        key = encodeURIComponent(key);
        value = encodeURIComponent(value);
        if (key.length > 0 && value.length > 0) {
            queries.push(key + "=" + value);
        }
    };
    return SASQueryParameters;
}());
export { SASQueryParameters };
//# sourceMappingURL=SASQueryParameters.js.map