import * as tslib_1 from "tslib";
import { BatchResponseParser } from "./BatchResponseParser";
import { utf8ByteLength } from "./BatchUtils";
import { Service } from "./generated/src/operations";
import { StorageURL } from "./StorageURL";
import { truncatedISO8061Date } from "./utils/utils.common";
/**
 * A ServiceURL represents a URL to the Azure Storage Blob service allowing you
 * to manipulate blob containers.
 *
 * @export
 * @class ServiceURL
 * @extends {StorageURL}
 */
var ServiceURL = /** @class */ (function (_super) {
    tslib_1.__extends(ServiceURL, _super);
    /**
     * Creates an instance of ServiceURL.
     *
     * @param {string} url A URL string pointing to Azure Storage blob service, such as
     *                     "https://myaccount.blob.core.windows.net". You can append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.blob.core.windows.net?sasString".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof ServiceURL
     */
    function ServiceURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.serviceContext = new Service(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a new ServiceURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {ServiceURL}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.withPipeline = function (pipeline) {
        return new ServiceURL(this.url, pipeline);
    };
    /**
     * Gets the properties of a storage account’s Blob service, including properties
     * for Storage Analytics and CORS (Cross-Origin Resource Sharing) rules.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-service-properties}
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.ServiceGetPropertiesResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getProperties = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.getProperties({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Sets properties for a storage account’s Blob service endpoint, including properties
     * for Storage Analytics, CORS (Cross-Origin Resource Sharing) rules and soft delete settings.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-service-properties}
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.StorageServiceProperties} properties
     * @returns {Promise<Models.ServiceSetPropertiesResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.setProperties = function (aborter, properties) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.setProperties(properties, {
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Retrieves statistics related to replication for the Blob service. It is only
     * available on the secondary location endpoint when read-access geo-redundant
     * replication is enabled for the storage account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-service-stats}
     *
     *  @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.ServiceGetStatisticsResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getStatistics = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.getStatistics({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * The Get Account Information operation returns the sku name and account kind
     * for the specified account.
     * The Get Account Information operation is available on service versions beginning
     * with version 2018-03-28.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-account-information
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.ServiceGetAccountInfoResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getAccountInfo = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.getAccountInfo({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Returns a list of the containers under the specified account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/list-containers2
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} [marker] A string value that identifies the portion of
     *                          the list of containers to be returned with the next listing operation. The
     *                          operation returns the NextMarker value within the response body if the
     *                          listing operation did not return all containers remaining to be listed
     *                          with the current page. The NextMarker value can be used as the value for
     *                          the marker parameter in a subsequent call to request the next page of list
     *                          items. The marker value is opaque to the client.
     * @param {IServiceListContainersSegmentOptions} [options]
     * @returns {Promise<Models.ServiceListContainersSegmentResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.listContainersSegment = function (aborter, marker, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.listContainersSegment(tslib_1.__assign({ abortSignal: aborter, marker: marker }, options))];
            });
        });
    };
    /**
     * ONLY AVAILABLE WHEN USING BEARER TOKEN AUTHENTICATION (TokenCredential).
     *
     * Retrieves a user delegation key for the Blob service. This is only a valid operation when using
     * bearer token authentication.
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-user-delegation-key
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Date} start      The start time for the user delegation SAS. Must be within 7 days of the current time
     * @param {Date} expiry     The end time for the user delegation SAS. Must be within 7 days of the current time
     * @returns {Promise<ServiceGetUserDelegationKeyResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getUserDelegationKey = function (aborter, start, expiry) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var response, userDelegationKey, res;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.serviceContext.getUserDelegationKey({
                            start: truncatedISO8061Date(start, false),
                            expiry: truncatedISO8061Date(expiry, false)
                        }, {
                            abortSignal: aborter
                        })];
                    case 1:
                        response = _a.sent();
                        userDelegationKey = {
                            signedOid: response.signedOid,
                            signedTid: response.signedTid,
                            signedStart: new Date(response.signedStart),
                            signedExpiry: new Date(response.signedExpiry),
                            signedService: response.signedService,
                            signedVersion: response.signedVersion,
                            value: response.value
                        };
                        res = tslib_1.__assign({ _response: response._response, requestId: response.requestId, clientRequestId: response.clientRequestId, version: response.version, date: response.date, errorCode: response.errorCode }, userDelegationKey);
                        return [2 /*return*/, res];
                }
            });
        });
    };
    /**
     * Submit batch request which consists of multiple subrequests.
     *
     * @example
     * let batchDeleteRequest = new BatchDeleteRequest();
     * await batchDeleteRequest.addSubRequest(urlInString0, credential0);
     * await batchDeleteRequest.addSubRequest(urlInString1, credential1, {
     *  deleteSnapshots: "include"
     * });
     * const deleteBatchResp = await serviceURL.submitBatch(Aborter.none, batchDeleteRequest);
     * console.log(deleteBatchResp.subResponsesSucceededCount);
     *
     * @example
     * let batchSetTierRequest = new BatchSetTierRequest();
     * await batchSetTierRequest.addSubRequest(blockBlobURL0, "Cool");
     * await batchSetTierRequest.addSubRequest(blockBlobURL1, "Cool", {
     *  leaseAccessConditions: { leaseId: leaseId }
     * });
     * const setTierBatchResp = await serviceURL.submitBatch(Aborter.none, batchSetTierRequest);
     * console.log(setTierBatchResp.subResponsesSucceededCount);
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/blob-batch
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation.
     * @param {BatchRequest} batchRequest Supported batch request: BatchDeleteRequest or BatchSetTierRequest.
     * @param {Models.ServiceSubmitBatchOptionalParams} [options]
     * @returns {Promise<ServiceSubmitBatchResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.submitBatch = function (aborter, batchRequest, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var batchRequestBody, rawBatchResponse, batchResponseParser, responseSummary, res;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!batchRequest || batchRequest.getSubRequests().size == 0) {
                            throw new RangeError("Batch request should contain one or more sub requests.");
                        }
                        batchRequestBody = batchRequest.getHttpRequestBody();
                        return [4 /*yield*/, this.serviceContext.submitBatch(batchRequestBody, utf8ByteLength(batchRequestBody), batchRequest.getMultiPartContentType(), tslib_1.__assign({ abortSignal: aborter }, options))];
                    case 1:
                        rawBatchResponse = _a.sent();
                        batchResponseParser = new BatchResponseParser(rawBatchResponse, batchRequest.getSubRequests());
                        return [4 /*yield*/, batchResponseParser.parseBatchResponse()];
                    case 2:
                        responseSummary = _a.sent();
                        res = {
                            _response: rawBatchResponse._response,
                            contentType: rawBatchResponse.contentType,
                            errorCode: rawBatchResponse.errorCode,
                            requestId: rawBatchResponse.requestId,
                            clientRequestId: rawBatchResponse.clientRequestId,
                            version: rawBatchResponse.version,
                            subResponses: responseSummary.subResponses,
                            subResponsesSucceededCount: responseSummary.subResponsesSucceededCount,
                            subResponsesFailedCount: responseSummary.subResponsesFailedCount
                        };
                        return [2 /*return*/, res];
                }
            });
        });
    };
    return ServiceURL;
}(StorageURL));
export { ServiceURL };
//# sourceMappingURL=ServiceURL.js.map