import { deserializationPolicy } from "@azure/ms-rest-js";
import { BrowserPolicyFactory } from "./BrowserPolicyFactory";
import { Credential } from "./credentials/Credential";
import { StorageClientContext } from "./generated/src/storageClientContext";
import { KeepAlivePolicyFactory } from "./KeepAlivePolicyFactory";
import { LoggingPolicyFactory } from "./LoggingPolicyFactory";
import { Pipeline } from "./Pipeline";
import { RetryPolicyFactory } from "./RetryPolicyFactory";
import { TelemetryPolicyFactory } from "./TelemetryPolicyFactory";
import { UniqueRequestIDPolicyFactory } from "./UniqueRequestIDPolicyFactory";
import { escapeURLPath, getURLScheme, iEqual } from "./utils/utils.common";
import { AnonymousCredential } from './credentials/AnonymousCredential';
export { deserializationPolicy };
/**
 * A ServiceURL represents a based URL class for ServiceURL, ContainerURL and etc.
 *
 * @export
 * @class StorageURL
 */
var StorageURL = /** @class */ (function () {
    /**
     * Creates an instance of StorageURL.
     * @param {string} url
     * @param {Pipeline} pipeline
     * @memberof StorageURL
     */
    function StorageURL(url, pipeline) {
        // URL should be encoded and only once, protocol layer shouldn't encode URL again
        this.url = escapeURLPath(url);
        this.pipeline = pipeline;
        this.storageClientContext = new StorageClientContext(this.url, pipeline.toServiceClientOptions());
        this.isHttps = iEqual(getURLScheme(this.url) || "", "https");
        this.credential = new AnonymousCredential();
        for (var _i = 0, _a = this.pipeline.factories; _i < _a.length; _i++) {
            var factory = _a[_i];
            if (factory instanceof Credential) {
                this.credential = factory;
            }
        }
        // Override protocol layer's default content-type
        var storageClientContext = this.storageClientContext;
        storageClientContext.requestContentType = undefined;
    }
    /**
     * A static method used to create a new Pipeline object with Credential provided.
     *
     * @static
     * @param {Credential} credential Such as AnonymousCredential, SharedKeyCredential or TokenCredential.
     * @param {INewPipelineOptions} [pipelineOptions] Optional. Options.
     * @returns {Pipeline} A new Pipeline object.
     * @memberof Pipeline
     */
    StorageURL.newPipeline = function (credential, pipelineOptions) {
        if (pipelineOptions === void 0) { pipelineOptions = {}; }
        // Order is important. Closer to the API at the top & closer to the network at the bottom.
        // The credential's policy factory must appear close to the wire so it can sign any
        // changes made by other factories (like UniqueRequestIDPolicyFactory)
        var factories = [
            new KeepAlivePolicyFactory(pipelineOptions.keepAliveOptions),
            new TelemetryPolicyFactory(pipelineOptions.telemetry),
            new UniqueRequestIDPolicyFactory(),
            new BrowserPolicyFactory(),
            deserializationPolicy(),
            new RetryPolicyFactory(pipelineOptions.retryOptions),
            new LoggingPolicyFactory(),
            credential
        ];
        return new Pipeline(factories, {
            HTTPClient: pipelineOptions.httpClient,
            logger: pipelineOptions.logger
        });
    };
    return StorageURL;
}());
export { StorageURL };
//# sourceMappingURL=StorageURL.js.map