import * as tslib_1 from "tslib";
import { generateUuid } from "@azure/ms-rest-js";
import { Batch } from "./utils/Batch";
import { BLOCK_BLOB_MAX_BLOCKS, BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES, BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES, DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES } from "./utils/constants";
import { generateBlockID } from "./utils/utils.common";
/**
 * ONLY AVAILABLE IN BROWSERS.
 *
 * Uploads a browser Blob/File/ArrayBuffer/ArrayBufferView object to block blob.
 *
 * When buffer length <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Blob | ArrayBuffer | ArrayBufferView} browserData Blob, File, ArrayBuffer or ArrayBufferView
 * @param {BlockBlobURL} blockBlobURL
 * @param {IUploadToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
export function uploadBrowserDataToBlockBlob(aborter, browserData, blockBlobURL, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var browserBlob;
        return tslib_1.__generator(this, function (_a) {
            browserBlob = new Blob([browserData]);
            return [2 /*return*/, UploadSeekableBlobToBlockBlob(aborter, function (offset, size) {
                    return browserBlob.slice(offset, offset + size);
                }, browserBlob.size, blockBlobURL, options)];
        });
    });
}
/**
 * ONLY AVAILABLE IN BROWSERS.
 *
 * Uploads a browser Blob object to block blob. Requires a blobFactory as the data source,
 * which need to return a Blob object with the offset and size provided.
 *
 * When buffer length <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {(offset: number, size: number) => Blob} blobFactory
 * @param {number} size
 * @param {BlockBlobURL} blockBlobURL
 * @param {IUploadToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
function UploadSeekableBlobToBlockBlob(aborter, blobFactory, size, blockBlobURL, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var numBlocks, blockList, blockIDPrefix, transferProgress, batch, _loop_1, i;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blockSize) {
                        options.blockSize = 0;
                    }
                    if (options.blockSize < 0 || options.blockSize > BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES) {
                        throw new RangeError("blockSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES);
                    }
                    if (options.maxSingleShotSize !== 0 && !options.maxSingleShotSize) {
                        options.maxSingleShotSize = BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES;
                    }
                    if (options.maxSingleShotSize < 0 ||
                        options.maxSingleShotSize > BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES) {
                        throw new RangeError("maxSingleShotSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES);
                    }
                    if (options.blockSize === 0) {
                        if (size > BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES * BLOCK_BLOB_MAX_BLOCKS) {
                            throw new RangeError(size + " is too larger to upload to a block blob.");
                        }
                        if (size > options.maxSingleShotSize) {
                            options.blockSize = Math.ceil(size / BLOCK_BLOB_MAX_BLOCKS);
                            if (options.blockSize < DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES) {
                                options.blockSize = DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES;
                            }
                        }
                    }
                    if (!options.blobHTTPHeaders) {
                        options.blobHTTPHeaders = {};
                    }
                    if (!options.blobAccessConditions) {
                        options.blobAccessConditions = {};
                    }
                    if (size <= options.maxSingleShotSize) {
                        return [2 /*return*/, blockBlobURL.upload(aborter, blobFactory(0, size), size, options)];
                    }
                    numBlocks = Math.floor((size - 1) / options.blockSize) + 1;
                    if (numBlocks > BLOCK_BLOB_MAX_BLOCKS) {
                        throw new RangeError("The buffer's size is too big or the BlockSize is too small;" +
                            ("the number of blocks must be <= " + BLOCK_BLOB_MAX_BLOCKS));
                    }
                    blockList = [];
                    blockIDPrefix = generateUuid();
                    transferProgress = 0;
                    batch = new Batch(options.parallelism);
                    _loop_1 = function (i) {
                        batch.addOperation(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                            var blockID, start, end, contentLength;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        blockID = generateBlockID(blockIDPrefix, i);
                                        start = options.blockSize * i;
                                        end = i === numBlocks - 1 ? size : start + options.blockSize;
                                        contentLength = end - start;
                                        blockList.push(blockID);
                                        return [4 /*yield*/, blockBlobURL.stageBlock(aborter, blockID, blobFactory(start, contentLength), contentLength, {
                                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions
                                            })];
                                    case 1:
                                        _a.sent();
                                        // Update progress after block is successfully uploaded to server, in case of block trying
                                        // TODO: Hook with convenience layer progress event in finer level
                                        transferProgress += contentLength;
                                        if (options.progress) {
                                            options.progress({
                                                loadedBytes: transferProgress
                                            });
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); });
                    };
                    for (i = 0; i < numBlocks; i++) {
                        _loop_1(i);
                    }
                    return [4 /*yield*/, batch.do()];
                case 1:
                    _a.sent();
                    return [2 /*return*/, blockBlobURL.commitBlockList(aborter, blockList, options)];
            }
        });
    });
}
//# sourceMappingURL=highlevel.browser.js.map