import * as tslib_1 from "tslib";
import { generateUuid } from "@azure/ms-rest-js";
import * as fs from "fs";
import { Batch } from "./utils/Batch";
import { BufferScheduler } from "./utils/BufferScheduler";
import { BLOCK_BLOB_MAX_BLOCKS, BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES, BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES, DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES, } from "./utils/constants";
import { generateBlockID } from "./utils/utils.common";
import { streamToBuffer } from "./utils/utils.node";
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Uploads a local file in blocks to a block blob.
 *
 * When file size <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {string} filePath Full path of local file
 * @param {BlockBlobURL} blockBlobURL BlockBlobURL
 * @param {IUploadToBlockBlobOptions} [options] IUploadToBlockBlobOptions
 * @returns {(Promise<BlobUploadCommonResponse>)} ICommonResponse
 */
export function uploadFileToBlockBlob(aborter, filePath, blockBlobURL, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var size;
        return tslib_1.__generator(this, function (_a) {
            size = fs.statSync(filePath).size;
            return [2 /*return*/, uploadResetableStreamToBlockBlob(aborter, function (offset, count) {
                    return fs.createReadStream(filePath, {
                        autoClose: true,
                        end: count ? offset + count - 1 : Infinity,
                        start: offset
                    });
                }, size, blockBlobURL, options)];
        });
    });
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Accepts a Node.js Readable stream factory, and uploads in blocks to a block blob.
 * The Readable stream factory must returns a Node.js Readable stream starting from the offset defined. The offset
 * is the offset in the block blob to be uploaded.
 *
 * When buffer length <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {(offset: number) => NodeJS.ReadableStream} streamFactory Returns a Node.js Readable stream starting
 *                                                                  from the offset defined
 * @param {number} size Size of the block blob
 * @param {BlockBlobURL} blockBlobURL BlockBlobURL
 * @param {IUploadToBlockBlobOptions} [options] IUploadToBlockBlobOptions
 * @returns {(Promise<BlobUploadCommonResponse>)} ICommonResponse
 */
function uploadResetableStreamToBlockBlob(aborter, streamFactory, size, blockBlobURL, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var numBlocks, blockList, blockIDPrefix, transferProgress, batch, _loop_1, i;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blockSize) {
                        options.blockSize = 0;
                    }
                    if (options.blockSize < 0 || options.blockSize > BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES) {
                        throw new RangeError("blockSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES);
                    }
                    if (options.maxSingleShotSize !== 0 && !options.maxSingleShotSize) {
                        options.maxSingleShotSize = BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES;
                    }
                    if (options.maxSingleShotSize < 0 ||
                        options.maxSingleShotSize > BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES) {
                        throw new RangeError("maxSingleShotSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES);
                    }
                    if (options.blockSize === 0) {
                        if (size > BLOCK_BLOB_MAX_BLOCKS * BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES) {
                            throw new RangeError(size + " is too larger to upload to a block blob.");
                        }
                        if (size > options.maxSingleShotSize) {
                            options.blockSize = Math.ceil(size / BLOCK_BLOB_MAX_BLOCKS);
                            if (options.blockSize < DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES) {
                                options.blockSize = DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES;
                            }
                        }
                    }
                    if (!options.blobHTTPHeaders) {
                        options.blobHTTPHeaders = {};
                    }
                    if (!options.blobAccessConditions) {
                        options.blobAccessConditions = {};
                    }
                    if (size <= options.maxSingleShotSize) {
                        return [2 /*return*/, blockBlobURL.upload(aborter, function () { return streamFactory(0); }, size, options)];
                    }
                    numBlocks = Math.floor((size - 1) / options.blockSize) + 1;
                    if (numBlocks > BLOCK_BLOB_MAX_BLOCKS) {
                        throw new RangeError("The buffer's size is too big or the BlockSize is too small;" +
                            ("the number of blocks must be <= " + BLOCK_BLOB_MAX_BLOCKS));
                    }
                    blockList = [];
                    blockIDPrefix = generateUuid();
                    transferProgress = 0;
                    batch = new Batch(options.parallelism);
                    _loop_1 = function (i) {
                        batch.addOperation(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                            var blockID, start, end, contentLength;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        blockID = generateBlockID(blockIDPrefix, i);
                                        start = options.blockSize * i;
                                        end = i === numBlocks - 1 ? size : start + options.blockSize;
                                        contentLength = end - start;
                                        blockList.push(blockID);
                                        return [4 /*yield*/, blockBlobURL.stageBlock(aborter, blockID, function () { return streamFactory(start, contentLength); }, contentLength, {
                                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions
                                            })];
                                    case 1:
                                        _a.sent();
                                        // Update progress after block is successfully uploaded to server, in case of block trying
                                        transferProgress += contentLength;
                                        if (options.progress) {
                                            options.progress({ loadedBytes: transferProgress });
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); });
                    };
                    for (i = 0; i < numBlocks; i++) {
                        _loop_1(i);
                    }
                    return [4 /*yield*/, batch.do()];
                case 1:
                    _a.sent();
                    return [2 /*return*/, blockBlobURL.commitBlockList(aborter, blockList, options)];
            }
        });
    });
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Downloads an Azure Blob in parallel to a buffer.
 * Offset and count are optional, pass 0 for both to download the entire blob.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Buffer} buffer Buffer to be fill, must have length larger than count
 * @param {BlobURL} blobURL A BlobURL object
 * @param {number} offset From which position of the block blob to download, in bytes
 * @param {number} [count] How much data in bytes to be downloaded. Will download to the end when passing undefined
 * @param {IDownloadFromBlobOptions} [options] IDownloadFromBlobOptions
 * @returns {Promise<void>}
 */
export function downloadBlobToBuffer(aborter, buffer, blobURL, offset, count, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var response, transferProgress, batch, _loop_2, off;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blockSize) {
                        options.blockSize = 0;
                    }
                    if (options.blockSize < 0) {
                        throw new RangeError("blockSize option must be >= 0");
                    }
                    if (options.blockSize === 0) {
                        options.blockSize = DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES;
                    }
                    if (offset < 0) {
                        throw new RangeError("offset option must be >= 0");
                    }
                    if (count && count <= 0) {
                        throw new RangeError("count option must be > 0");
                    }
                    if (!options.blobAccessConditions) {
                        options.blobAccessConditions = {};
                    }
                    if (!!count) return [3 /*break*/, 2];
                    return [4 /*yield*/, blobURL.getProperties(aborter, options)];
                case 1:
                    response = _a.sent();
                    count = response.contentLength - offset;
                    if (count < 0) {
                        throw new RangeError("offset " + offset + " shouldn't be larger than blob size " + response.contentLength);
                    }
                    _a.label = 2;
                case 2:
                    if (buffer.length < count) {
                        throw new RangeError("The buffer's size should be equal to or larger than the request count of bytes: " + count);
                    }
                    transferProgress = 0;
                    batch = new Batch(options.parallelism);
                    _loop_2 = function (off) {
                        batch.addOperation(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                            var chunkEnd, response, stream;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        chunkEnd = offset + count;
                                        if (off + options.blockSize < chunkEnd) {
                                            chunkEnd = off + options.blockSize;
                                        }
                                        return [4 /*yield*/, blobURL.download(aborter, off, chunkEnd - off, {
                                                blobAccessConditions: options.blobAccessConditions,
                                                maxRetryRequests: options.maxRetryRequestsPerBlock
                                            })];
                                    case 1:
                                        response = _a.sent();
                                        stream = response.readableStreamBody;
                                        return [4 /*yield*/, streamToBuffer(stream, buffer, off - offset, chunkEnd - offset)];
                                    case 2:
                                        _a.sent();
                                        // Update progress after block is downloaded, in case of block trying
                                        // Could provide finer grained progress updating inside HTTP requests,
                                        // only if convenience layer download try is enabled
                                        transferProgress += chunkEnd - off;
                                        if (options.progress) {
                                            options.progress({ loadedBytes: transferProgress });
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); });
                    };
                    for (off = offset; off < offset + count; off = off + options.blockSize) {
                        _loop_2(off);
                    }
                    return [4 /*yield*/, batch.do()];
                case 3:
                    _a.sent();
                    return [2 /*return*/];
            }
        });
    });
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Uploads a Node.js Readable stream into block blob.
 *
 * PERFORMANCE IMPROVEMENT TIPS:
 * * Input stream highWaterMark is better to set a same value with bufferSize
 *    parameter, which will avoid Buffer.concat() operations.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Readable} stream Node.js Readable stream
 * @param {BlockBlobURL} blockBlobURL A BlockBlobURL instance
 * @param {number} bufferSize Size of every buffer allocated, also the block size in the uploaded block blob
 * @param {number} maxBuffers Max buffers will allocate during uploading, positive correlation
 *                            with max uploading concurrency
 * @param {IUploadStreamToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
export function uploadStreamToBlockBlob(aborter, stream, blockBlobURL, bufferSize, maxBuffers, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var blockNum, blockIDPrefix, transferProgress, blockList, scheduler;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blobHTTPHeaders) {
                        options.blobHTTPHeaders = {};
                    }
                    if (!options.accessConditions) {
                        options.accessConditions = {};
                    }
                    blockNum = 0;
                    blockIDPrefix = generateUuid();
                    transferProgress = 0;
                    blockList = [];
                    scheduler = new BufferScheduler(stream, bufferSize, maxBuffers, function (buffer) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                        var blockID;
                        return tslib_1.__generator(this, function (_a) {
                            switch (_a.label) {
                                case 0:
                                    blockID = generateBlockID(blockIDPrefix, blockNum);
                                    blockList.push(blockID);
                                    blockNum++;
                                    return [4 /*yield*/, blockBlobURL.stageBlock(aborter, blockID, buffer, buffer.length, {
                                            leaseAccessConditions: options.accessConditions.leaseAccessConditions
                                        })];
                                case 1:
                                    _a.sent();
                                    // Update progress after block is successfully uploaded to server, in case of block trying
                                    transferProgress += buffer.length;
                                    if (options.progress) {
                                        options.progress({ loadedBytes: transferProgress });
                                    }
                                    return [2 /*return*/];
                            }
                        });
                    }); }, 
                    // Parallelism should set a smaller value than maxBuffers, which is helpful to
                    // reduce the possibility when a outgoing handler waits for stream data, in
                    // this situation, outgoing handlers are blocked.
                    // Outgoing queue shouldn't be empty.
                    Math.ceil((maxBuffers / 4) * 3));
                    return [4 /*yield*/, scheduler.do()];
                case 1:
                    _a.sent();
                    return [2 /*return*/, blockBlobURL.commitBlockList(aborter, blockList, options)];
            }
        });
    });
}
//# sourceMappingURL=highlevel.node.js.map