'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var msRest = require('@azure/ms-rest-js');
var tslib_1 = require('tslib');
var stream = require('stream');
var os = require('os');
var Crypto = require('crypto');
var events = require('events');
var fs = require('fs');

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var index = /*#__PURE__*/Object.freeze({

});

/**
 * An aborter instance implements AbortSignal interface, can abort HTTP requests.
 *
 * - Call Aborter.none to create a new Aborter instance without timeout.
 * - Call Aborter.timeout() to create a new Aborter instance with timeout.
 *
 * For an existing instance aborter:
 * - Call aborter.withTimeout() to create and return a child Aborter instance with timeout.
 * - Call aborter.withValue(key, value) to create and return a child Aborter instance with key/value pair.
 * - Call aborter.abort() to abort current instance and all children instances.
 * - Call aborter.getValue(key) to search and get value with corresponding key from current aborter to all parents.
 *
 * @example
 * // Abort without timeout
 * await blockBlobURL.upload(Aborter.none, buf, buf.length);
 *
 * @example
 * // Abort container create in 1000ms
 * await blockBlobURL.upload(Aborter.timeout(1000), buf, buf.length);
 *
 * @example
 * // Share aborter cross multiple operations in 30s
 * // Upload the same data to 2 different data centers at the same time, abort another when any of them is finished
 * const aborter = Aborter.timeout(30 * 1000);
 * blockBlobURL1.upload(aborter, buf, buf.length).then(aborter.abort);
 * blockBlobURL2.upload(aborter, buf, buf.length).then(aborter.abort);
 *
 * @example
 * // Cascaded aborting
 * // All operations can't take more than 30 seconds
 * const aborter = Aborter.timeout(30 * 1000);
 *
 * // Following 2 operations can't take more than 25 seconds
 * await blockBlobURL.upload(aborter.withTimeout(25 * 1000), buf, buf.length);
 * await blockBlobURL.upload(aborter.withTimeout(25 * 1000), buf, buf.length);
 *
 * @export
 * @class Aborter
 * @implements {AbortSignalLike}
 */
var Aborter = /** @class */ (function () {
    // private disposed: boolean = false;
    /**
     * Private constructor for internal usage, creates an instance of Aborter.
     *
     * @param {Aborter} [parent] Optional. Parent aborter.
     * @param {number} [timeout=0] Optional. Timeout before abort in millisecond, 0 means no timeout.
     * @param {string} [key] Optional. Immutable key in string.
     * @param {(string | number | boolean | null)} [value] Optional. Immutable value.
     * @memberof Aborter
     */
    function Aborter(parent, timeout, key, value) {
        var _this = this;
        if (timeout === void 0) { timeout = 0; }
        /**
         * onabort event listener.
         *
         * @memberof Aborter
         */
        this.onabort = null;
        // tslint:disable-next-line:variable-name
        this._aborted = false;
        this.children = []; // When child object calls dispose(), remove child from here
        this.abortEventListeners = [];
        this.parent = parent;
        this.key = key;
        this.value = value;
        if (timeout > 0) {
            this.timer = setTimeout(function () {
                _this.abort.call(_this);
            }, timeout);
            // When called, the active Timeout object will not require the Node.js event loop
            // to remain active. If there is no other activity keeping the event loop running,
            // the process may exit before the Timeout object's callback is invoked.
            if (this.timer && msRest.isNode) {
                this.timer.unref();
            }
        }
    }
    Object.defineProperty(Aborter.prototype, "aborted", {
        /**
         * Status of whether aborted or not.
         *
         * @readonly
         * @type {boolean}
         * @memberof Aborter
         */
        get: function () {
            return this._aborted;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Aborter, "none", {
        /**
         * Creates a new Aborter instance without timeout.
         *
         * @readonly
         * @static
         * @type {Aborter}
         * @memberof Aborter
         */
        get: function () {
            return new Aborter(undefined, 0);
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Creates a new Aborter instance with timeout in milliseconds.
     * Set parameter timeout to 0 will not create a timer.
     *
     * @static
     * @param {number} {timeout} in milliseconds
     * @returns {Aborter}
     * @memberof Aborter
     */
    Aborter.timeout = function (timeout) {
        return new Aborter(undefined, timeout);
    };
    /**
     * Create and return a new Aborter instance, which will be appended as a child node of current Aborter.
     * Current Aborter instance becomes father node of the new instance. When current or father Aborter node
     * triggers timeout event, all children nodes abort event will be triggered too.
     *
     * When timeout parameter (in millisecond) is larger than 0, the abort event will be triggered when timeout.
     * Otherwise, call abort() method to manually abort.
     *
     * @param {number} {timeout} Timeout in millisecond.
     * @returns {Aborter} The new Aborter instance created.
     * @memberof Aborter
     */
    Aborter.prototype.withTimeout = function (timeout) {
        var childCancelContext = new Aborter(this, timeout);
        this.children.push(childCancelContext);
        return childCancelContext;
    };
    /**
     * Create and return a new Aborter instance, which will be appended as a child node of current Aborter.
     * Current Aborter instance becomes father node of the new instance. When current or father Aborter node
     * triggers timeout event, all children nodes abort event will be triggered too.
     *
     * Immutable key value pair will be set into the new created Aborter instance.
     * Call getValue() to find out latest value with corresponding key in the chain of
     * [current node] -> [parent node] and [grand parent node]....
     *
     * @param {string} key
     * @param {(string | number | boolean | null)} [value]
     * @returns {Aborter}
     * @memberof Aborter
     */
    Aborter.prototype.withValue = function (key, value) {
        var childCancelContext = new Aborter(this, 0, key, value);
        this.children.push(childCancelContext);
        return childCancelContext;
    };
    /**
     * Find out latest value with corresponding key in the chain of
     * [current node] -> [parent node] -> [grand parent node] -> ... -> [root node].
     *
     * If key is not found, undefined will be returned.
     *
     * @param {string} key
     * @returns {(string | number | boolean | null | undefined)}
     * @memberof Aborter
     */
    Aborter.prototype.getValue = function (key) {
        for (var parent_1 = this; parent_1; parent_1 = parent_1.parent) {
            if (parent_1.key === key) {
                return parent_1.value;
            }
        }
        return undefined;
    };
    /**
     * Trigger abort event immediately, the onabort and all abort event listeners will be triggered.
     * Will try to trigger abort event for all children Aborter nodes.
     *
     * - If there is a timeout, the timer will be cancelled.
     * - If aborted is true, nothing will happen.
     *
     * @returns
     * @memberof Aborter
     */
    Aborter.prototype.abort = function () {
        var _this = this;
        if (this.aborted) {
            return;
        }
        this.cancelTimer();
        if (this.onabort) {
            this.onabort.call(this, { type: "abort" });
        }
        this.abortEventListeners.forEach(function (listener) {
            listener.call(_this, { type: "abort" });
        });
        this.children.forEach(function (child) { return child.cancelByParent(); });
        this._aborted = true;
    };
    // public dispose() {
    //   if (this.disposed || this.aborted) {
    //     return;
    //   }
    //   this.cancelTimer();
    //   // (parent)A <- B <- C(child), if B disposes, when A abort, C will not abort
    //   if (this.parent) {
    //     const index = this.parent.children.indexOf(this);
    //     if (index > -1) {
    //       this.parent.children.splice(index, 1);
    //     }
    //   }
    //   this.disposed = true;
    // }
    /**
     * Added new "abort" event listener, only support "abort" event.
     *
     * @param {"abort"} _type Only support "abort" event
     * @param {(this: AbortSignalLike, ev: any) => any} listener
     * @memberof Aborter
     */
    Aborter.prototype.addEventListener = function (
    // tslint:disable-next-line:variable-name
    _type, listener) {
        this.abortEventListeners.push(listener);
    };
    /**
     * Remove "abort" event listener, only support "abort" event.
     *
     * @param {"abort"} _type Only support "abort" event
     * @param {(this: AbortSignalLike, ev: any) => any} listener
     * @memberof Aborter
     */
    Aborter.prototype.removeEventListener = function (
    // tslint:disable-next-line:variable-name
    _type, listener) {
        var index = this.abortEventListeners.indexOf(listener);
        if (index > -1) {
            this.abortEventListeners.splice(index, 1);
        }
    };
    Aborter.prototype.dispatchEvent = function () {
        throw new Error("Method not implemented.");
    };
    Aborter.prototype.cancelByParent = function () {
        // if (!this.disposed) {
        this.abort();
        // }
    };
    Aborter.prototype.cancelTimer = function () {
        if (this.timer) {
            clearTimeout(this.timer);
        }
    };
    return Aborter;
}());

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the permissions granted by an AccountSAS. Setting a value
 * to true means that any SAS which uses these permissions will grant permissions for that operation. Once all the
 * values are set, this should be serialized with toString and set as the permissions field on an
 * {@link IAccountSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 *
 * @export
 * @class AccountSASPermissions
 */
var AccountSASPermissions = /** @class */ (function () {
    function AccountSASPermissions() {
        /**
         * Permission to read resources and list queues and tables granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.read = false;
        /**
         * Permission to write resources granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.write = false;
        /**
         * Permission to create blobs and files granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.delete = false;
        /**
         * Permission to list blob containers, blobs, shares, directories, and files granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.list = false;
        /**
         * Permission to add messages, table entities, and append to blobs granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.add = false;
        /**
         * Permission to create blobs and files granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.create = false;
        /**
         * Permissions to update messages and table entities granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.update = false;
        /**
         * Permission to get and delete messages granted.
         *
         * @type {boolean}
         * @memberof AccountSASPermissions
         */
        this.process = false;
    }
    /**
     * Parse initializes the AccountSASPermissions fields from a string.
     *
     * @static
     * @param {string} permissions
     * @returns {AccountSASPermissions}
     * @memberof AccountSASPermissions
     */
    AccountSASPermissions.parse = function (permissions) {
        var accountSASPermissions = new AccountSASPermissions();
        for (var _i = 0, permissions_1 = permissions; _i < permissions_1.length; _i++) {
            var c = permissions_1[_i];
            switch (c) {
                case "r":
                    accountSASPermissions.read = true;
                    break;
                case "w":
                    accountSASPermissions.write = true;
                    break;
                case "d":
                    accountSASPermissions.delete = true;
                    break;
                case "l":
                    accountSASPermissions.list = true;
                    break;
                case "a":
                    accountSASPermissions.add = true;
                    break;
                case "c":
                    accountSASPermissions.create = true;
                    break;
                case "u":
                    accountSASPermissions.update = true;
                    break;
                case "p":
                    accountSASPermissions.process = true;
                    break;
                default:
                    throw new RangeError("Invalid permission character: " + c);
            }
        }
        return accountSASPermissions;
    };
    /**
     * Produces the SAS permissions string for an Azure Storage account.
     * Call this method to set IAccountSASSignatureValues Permissions field.
     *
     * Using this method will guarantee the resource types are in
     * an order accepted by the service.
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
     *
     * @returns {string}
     * @memberof AccountSASPermissions
     */
    AccountSASPermissions.prototype.toString = function () {
        // The order of the characters should be as specified here to ensure correctness:
        // https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
        // Use a string array instead of string concatenating += operator for performance
        var permissions = [];
        if (this.read) {
            permissions.push("r");
        }
        if (this.write) {
            permissions.push("w");
        }
        if (this.delete) {
            permissions.push("d");
        }
        if (this.list) {
            permissions.push("l");
        }
        if (this.add) {
            permissions.push("a");
        }
        if (this.create) {
            permissions.push("c");
        }
        if (this.update) {
            permissions.push("u");
        }
        if (this.process) {
            permissions.push("p");
        }
        return permissions.join("");
    };
    return AccountSASPermissions;
}());

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the resources accessible by an AccountSAS. Setting a value
 * to true means that any SAS which uses these permissions will grant access to that resource type. Once all the
 * values are set, this should be serialized with toString and set as the resources field on an
 * {@link IAccountSASSignatureValues} object. It is possible to construct the resources string without this class, but
 * the order of the resources is particular and this class guarantees correctness.
 *
 * @export
 * @class AccountSASResourceTypes
 */
var AccountSASResourceTypes = /** @class */ (function () {
    function AccountSASResourceTypes() {
        /**
         * Permission to access service level APIs granted.
         *
         * @type {boolean}
         * @memberof AccountSASResourceTypes
         */
        this.service = false;
        /**
         * Permission to access container level APIs (Blob Containers, Tables, Queues, File Shares) granted.
         *
         * @type {boolean}
         * @memberof AccountSASResourceTypes
         */
        this.container = false;
        /**
         * Permission to access object level APIs (Blobs, Table Entities, Queue Messages, Files) granted.
         *
         * @type {boolean}
         * @memberof AccountSASResourceTypes
         */
        this.object = false;
    }
    /**
     * Creates an {@link AccountSASResourceTypes} from the specified resource types string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid resource type.
     *
     * @static
     * @param {string} resourceTypes
     * @returns {AccountSASResourceTypes}
     * @memberof AccountSASResourceTypes
     */
    AccountSASResourceTypes.parse = function (resourceTypes) {
        var accountSASResourceTypes = new AccountSASResourceTypes();
        for (var _i = 0, resourceTypes_1 = resourceTypes; _i < resourceTypes_1.length; _i++) {
            var c = resourceTypes_1[_i];
            switch (c) {
                case "s":
                    accountSASResourceTypes.service = true;
                    break;
                case "c":
                    accountSASResourceTypes.container = true;
                    break;
                case "o":
                    accountSASResourceTypes.object = true;
                    break;
                default:
                    throw new RangeError("Invalid resource type: " + c);
            }
        }
        return accountSASResourceTypes;
    };
    /**
     * Converts the given resource types to a string.
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
     *
     * @returns {string}
     * @memberof AccountSASResourceTypes
     */
    AccountSASResourceTypes.prototype.toString = function () {
        var resourceTypes = [];
        if (this.service) {
            resourceTypes.push("s");
        }
        if (this.container) {
            resourceTypes.push("c");
        }
        if (this.object) {
            resourceTypes.push("o");
        }
        return resourceTypes.join("");
    };
    return AccountSASResourceTypes;
}());

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the services accessible by an AccountSAS. Setting a value
 * to true means that any SAS which uses these permissions will grant access to that service. Once all the
 * values are set, this should be serialized with toString and set as the services field on an
 * {@link IAccountSASSignatureValues} object. It is possible to construct the services string without this class, but
 * the order of the services is particular and this class guarantees correctness.
 *
 * @export
 * @class AccountSASServices
 */
var AccountSASServices = /** @class */ (function () {
    function AccountSASServices() {
        /**
         * Permission to access blob resources granted.
         *
         * @type {boolean}
         * @memberof AccountSASServices
         */
        this.blob = false;
        /**
         * Permission to access file resources granted.
         *
         * @type {boolean}
         * @memberof AccountSASServices
         */
        this.file = false;
        /**
         * Permission to access queue resources granted.
         *
         * @type {boolean}
         * @memberof AccountSASServices
         */
        this.queue = false;
        /**
         * Permission to access table resources granted.
         *
         * @type {boolean}
         * @memberof AccountSASServices
         */
        this.table = false;
    }
    /**
     * Creates an {@link AccountSASServices} from the specified services string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid service.
     *
     * @static
     * @param {string} services
     * @returns {AccountSASServices}
     * @memberof AccountSASServices
     */
    AccountSASServices.parse = function (services) {
        var accountSASServices = new AccountSASServices();
        for (var _i = 0, services_1 = services; _i < services_1.length; _i++) {
            var c = services_1[_i];
            switch (c) {
                case "b":
                    accountSASServices.blob = true;
                    break;
                case "f":
                    accountSASServices.file = true;
                    break;
                case "q":
                    accountSASServices.queue = true;
                    break;
                case "t":
                    accountSASServices.table = true;
                    break;
                default:
                    throw new RangeError("Invalid service character: " + c);
            }
        }
        return accountSASServices;
    };
    /**
     * Converts the given services to a string.
     *
     * @returns {string}
     * @memberof AccountSASServices
     */
    AccountSASServices.prototype.toString = function () {
        var services = [];
        if (this.blob) {
            services.push("b");
        }
        if (this.table) {
            services.push("t");
        }
        if (this.queue) {
            services.push("q");
        }
        if (this.file) {
            services.push("f");
        }
        return services.join("");
    };
    return AccountSASServices;
}());

/**
 * Generate IPRange format string. For example:
 *
 * "8.8.8.8" or "1.1.1.1-255.255.255.255"
 *
 * @export
 * @param {IIPRange} ipRange
 * @returns {string}
 */
function ipRangeToString(ipRange) {
    return ipRange.end ? ipRange.start + "-" + ipRange.end : ipRange.start;
}

/**
 * Reserved URL characters must be properly escaped for Storage services like Blob or File.
 *
 * ## URL encode and escape strategy for JSv10 SDKs
 *
 * When customers pass a URL string into XXXURL classes constrcutor, the URL string may already be URL encoded or not.
 * But before sending to Azure Storage server, the URL must be encoded. However, it's hard for a SDK to guess whether the URL
 * string has been encoded or not. We have 2 potential strategies, and chose strategy two for the XXXURL constructors.
 *
 * ### Strategy One: Assume the customer URL string is not encoded, and always encode URL string in SDK.
 *
 * This is what legacy V2 SDK does, simple and works for most of the cases.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b:",
 *   SDK will encode it to "http://account.blob.core.windows.net/con/b%3A" and send to server. A blob named "b:" will be created.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b%3A",
 *   SDK will encode it to "http://account.blob.core.windows.net/con/b%253A" and send to server. A blob named "b%3A" will be created.
 *
 * But this strategy will make it not possible to create a blob with "?" in it's name. Because when customer URL string is
 * "http://account.blob.core.windows.net/con/blob?name", the "?name" will be treated as URL paramter instead of blob name.
 * If customer URL string is "http://account.blob.core.windows.net/con/blob%3Fname", a blob named "blob%3Fname" will be created.
 * V2 SDK doesn't have this issue because it doesn't allow customer pass in a full URL, it accepts a separate blob name and encodeURIComponent for it.
 * We cannot accept a SDK cannot create a blob name with "?". So we implement strategy two:
 *
 * ### Strategy Two: SDK doesn't assume the URL has been encoded or not. It will just escape the special characters.
 *
 * This is what V10 Blob Go SDK does. It accepts a URL type in Go, and call url.EscapedPath() to escape the special chars unescaped.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b:",
 *   SDK will escape ":" like "http://account.blob.core.windows.net/con/b%3A" and send to server. A blob named "b:" will be created.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b%3A",
 *   There is no special characters, so send "http://account.blob.core.windows.net/con/b%3A" to server. A blob named "b:" will be created.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b%253A",
 *   There is no special characters, so send "http://account.blob.core.windows.net/con/b%253A" to server. A blob named "b%3A" will be created.
 *
 * This strategy gives us flexibility to create with any special characters. But "%" will be treated as a special characters, if the URL string
 * is not encoded, there shouldn't a "%" in the URL string, otherwise the URL is not a valid URL.
 * If customer needs to create a blob with "%" in it's blob name, use "%25" insead of "%". Just like above 3rd sample.
 * And following URL strings are invalid:
 * - "http://account.blob.core.windows.net/con/b%"
 * - "http://account.blob.core.windows.net/con/b%2"
 * - "http://account.blob.core.windows.net/con/b%G"
 *
 * Another special character is "?", use "%2F" to represent a blob name with "?" in a URL string.
 *
 * ### Strategy for containerName, blobName or other specific XXXName parameters in methods such as `BlobURL.fromContainerURL(containerURL, blobName)`
 *
 * We will apply strategy one, and call encodeURIComponent for these parameters like blobName. Because what customers passes in is a plain name instead of a URL.
 *
 * @see https://docs.microsoft.com/en-us/rest/api/storageservices/naming-and-referencing-containers--blobs--and-metadata
 * @see https://docs.microsoft.com/en-us/rest/api/storageservices/naming-and-referencing-shares--directories--files--and-metadata
 *
 * @export
 * @param {string} url
 * @returns {string}
 */
function escapeURLPath(url) {
    var urlParsed = msRest.URLBuilder.parse(url);
    var path = urlParsed.getPath();
    path = path || "/";
    path = escape(path);
    urlParsed.setPath(path);
    return urlParsed.toString();
}
/**
 * Internal escape method implmented Strategy Two mentioned in escapeURL() description.
 *
 * @param {string} text
 * @returns {string}
 */
function escape(text) {
    return encodeURIComponent(text)
        .replace(/%2F/g, "/") // Don't escape for "/"
        .replace(/'/g, "%27") // Escape for "'"
        .replace(/\+/g, "%20")
        .replace(/%25/g, "%"); // Revert encoded "%"
}
/**
 * Append a string to URL path. Will remove duplicated "/" in front of the string
 * when URL path ends with a "/".
 *
 * @export
 * @param {string} url Source URL string
 * @param {string} name String to be appended to URL
 * @returns {string} An updated URL string
 */
function appendToURLPath(url, name) {
    var urlParsed = msRest.URLBuilder.parse(url);
    var path = urlParsed.getPath();
    path = path ? (path.endsWith("/") ? "" + path + name : path + "/" + name) : name;
    urlParsed.setPath(path);
    return urlParsed.toString();
}
/**
 * Set URL parameter name and value. If name exists in URL parameters, old value
 * will be replaced by name key. If not provide value, the parameter will be deleted.
 *
 * @export
 * @param {string} url Source URL string
 * @param {string} name Parameter name
 * @param {string} [value] Parameter value
 * @returns {string} An updated URL string
 */
function setURLParameter(url, name, value) {
    var urlParsed = msRest.URLBuilder.parse(url);
    urlParsed.setQueryParameter(name, value);
    return urlParsed.toString();
}
/**
 * Get URL parameter by name.
 *
 * @export
 * @param {string} url
 * @param {string} name
 * @returns {(string | string[] | undefined)}
 */
function getURLParameter(url, name) {
    var urlParsed = msRest.URLBuilder.parse(url);
    return urlParsed.getQueryParameterValue(name);
}
/**
 * Set URL host.
 *
 * @export
 * @param {string} url Source URL string
 * @param {string} host New host string
 * @returns An updated URL string
 */
function setURLHost(url, host) {
    var urlParsed = msRest.URLBuilder.parse(url);
    urlParsed.setHost(host);
    return urlParsed.toString();
}
/**
 * Get URL path from an URL string.
 *
 * @export
 * @param {string} url Source URL string
 * @returns {(string | undefined)}
 */
function getURLPath(url) {
    var urlParsed = msRest.URLBuilder.parse(url);
    return urlParsed.getPath();
}
/**
 * Get URL scheme from an URL string.
 *
 * @export
 * @param {string} url Source URL string
 * @returns {(string | undefined)}
 */
function getURLScheme(url) {
    var urlParsed = msRest.URLBuilder.parse(url);
    return urlParsed.getScheme();
}
/**
 * Get URL path and query from an URL string.
 *
 * @export
 * @param {string} url Source URL string
 * @returns {(string | undefined)}
 */
function getURLPathAndQuery(url) {
    var urlParsed = msRest.URLBuilder.parse(url);
    var pathString = urlParsed.getPath();
    if (!pathString) {
        throw new RangeError("Invalid url without valid path.");
    }
    var queryString = urlParsed.getQuery() || "";
    queryString = queryString.trim();
    if (queryString != "") {
        queryString = queryString.startsWith("?") ? queryString : "?" + queryString; // Ensure query string start with '?'
    }
    return "" + pathString + queryString;
}
/**
 * Get URL query key value pairs from an URL string.
 *
 * @export
 * @param {string} url
 * @returns {{[key: string]: string}}
 */
function getURLQueries(url) {
    var queryString = msRest.URLBuilder.parse(url).getQuery();
    if (!queryString) {
        return {};
    }
    queryString = queryString.trim();
    queryString = queryString.startsWith("?") ? queryString.substr(1) : queryString;
    var querySubStrings = queryString.split("&");
    querySubStrings = querySubStrings.filter(function (value) {
        var indexOfEqual = value.indexOf("=");
        var lastIndexOfEqual = value.lastIndexOf("=");
        return (indexOfEqual > 0 && indexOfEqual === lastIndexOfEqual && lastIndexOfEqual < value.length - 1);
    });
    var queries = {};
    for (var _i = 0, querySubStrings_1 = querySubStrings; _i < querySubStrings_1.length; _i++) {
        var querySubString = querySubStrings_1[_i];
        var splitResults = querySubString.split("=");
        var key = splitResults[0];
        var value = splitResults[1];
        queries[key] = value;
    }
    return queries;
}
/**
 * Rounds a date off to seconds.
 *
 * @export
 * @param {Date} date
 * @param {boolean} [withMilliseconds=true] If true, YYYY-MM-DDThh:mm:ss.fffffffZ will be returned;
 *                                          If false, YYYY-MM-DDThh:mm:ssZ will be returned.
 * @returns {string} Date string in ISO8061 format, with or without 7 milliseconds component
 */
function truncatedISO8061Date(date, withMilliseconds) {
    if (withMilliseconds === void 0) { withMilliseconds = true; }
    // Date.toISOString() will return like "2018-10-29T06:34:36.139Z"
    var dateString = date.toISOString();
    return withMilliseconds
        ? dateString.substring(0, dateString.length - 1) + "0000" + "Z"
        : dateString.substring(0, dateString.length - 5) + "Z";
}
/**
 * Base64 encode.
 *
 * @export
 * @param {string} content
 * @returns {string}
 */
function base64encode(content) {
    return !msRest.isNode ? btoa(content) : Buffer.from(content).toString("base64");
}
/**
 * Generate a 64 bytes base64 block ID string.
 *
 * @export
 * @param {number} blockIndex
 * @returns {string}
 */
function generateBlockID(blockIDPrefix, blockIndex) {
    // To generate a 64 bytes base64 string, source string should be 48
    var maxSourceStringLength = 48;
    // A blob can have a maximum of 100,000 uncommitted blocks at any given time
    var maxBlockIndexLength = 6;
    var maxAllowedBlockIDPrefixLength = maxSourceStringLength - maxBlockIndexLength;
    if (blockIDPrefix.length > maxAllowedBlockIDPrefixLength) {
        blockIDPrefix = blockIDPrefix.slice(0, maxAllowedBlockIDPrefixLength);
    }
    var res = blockIDPrefix +
        padStart(blockIndex.toString(), maxSourceStringLength - blockIDPrefix.length, "0");
    return base64encode(res);
}
/**
 * Delay specified time interval.
 *
 * @export
 * @param {number} timeInMs
 * @param {AbortSignalLike} [aborter]
 * @param {Error} [abortError]
 */
function delay(timeInMs, aborter, abortError) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        return tslib_1.__generator(this, function (_a) {
            return [2 /*return*/, new Promise(function (resolve, reject) {
                    var timeout;
                    var abortHandler = function () {
                        if (timeout !== undefined) {
                            clearTimeout(timeout);
                        }
                        reject(abortError);
                    };
                    var resolveHandler = function () {
                        if (aborter !== undefined) {
                            aborter.removeEventListener("abort", abortHandler);
                        }
                        resolve();
                    };
                    timeout = setTimeout(resolveHandler, timeInMs);
                    if (aborter !== undefined) {
                        aborter.addEventListener("abort", abortHandler);
                    }
                })];
        });
    });
}
/**
 * String.prototype.padStart()
 *
 * @export
 * @param {string} currentString
 * @param {number} targetLength
 * @param {string} [padString=" "]
 * @returns {string}
 */
function padStart(currentString, targetLength, padString) {
    if (padString === void 0) { padString = " "; }
    if (String.prototype.padStart) {
        return currentString.padStart(targetLength, padString);
    }
    padString = padString || " ";
    if (currentString.length > targetLength) {
        return currentString;
    }
    else {
        targetLength = targetLength - currentString.length;
        if (targetLength > padString.length) {
            padString += padString.repeat(targetLength / padString.length);
        }
        return padString.slice(0, targetLength) + currentString;
    }
}
/**
 * If two strings are equal when compared case insensitive.
 *
 * @export
 * @param {string} str1
 * @param {string} str2
 * @returns {boolean}
 */
function iEqual(str1, str2) {
    return str1.toLocaleLowerCase() === str2.toLocaleLowerCase();
}

(function (SASProtocol) {
    /**
     * Protocol that allows HTTPS only
     */
    SASProtocol["HTTPS"] = "https";
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    SASProtocol["HTTPSandHTTP"] = "https,http";
})(exports.SASProtocol || (exports.SASProtocol = {}));
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link IAccountSASSignatureValues} and {@link IBlobSASSignatureValues}
 * types. Once generated, it can be encoded into a {@code String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 *
 * @export
 * @class SASQueryParameters
 */
var SASQueryParameters = /** @class */ (function () {
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param {string} version Representing the storage version
     * @param {string} signature Representing the signature for the SAS token
     * @param {string} [permissions] Representing the storage permissions
     * @param {string} [services] Representing the storage services being accessed (only for Account SAS)
     * @param {string} [resourceTypes] Representing the storage resource types being accessed (only for Account SAS)
     * @param {SASProtocol} [protocol] Representing the allowed HTTP protocol(s)
     * @param {Date} [startTime] Representing the start time for this SAS token
     * @param {Date} [expiryTime] Representing the expiry time for this SAS token
     * @param {IIPRange} [ipRange] Representing the range of valid IP addresses for this SAS token
     * @param {string} [identifier] Representing the signed identifier (only for Service SAS)
     * @param {string} [resource] Representing the storage container or blob (only for Service SAS)
     * @param {string} [cacheControl] Representing the cache-control header (only for Blob/File Service SAS)
     * @param {string} [contentDisposition] Representing the content-disposition header (only for Blob/File Service SAS)
     * @param {string} [contentEncoding] Representing the content-encoding header (only for Blob/File Service SAS)
     * @param {string} [contentLanguage] Representing the content-language header (only for Blob/File Service SAS)
     * @param {string} [contentType] Representing the content-type header (only for Blob/File Service SAS)
     * @param {userDelegationKey} [userDelegationKey] Representing the user delegation key properties
     * @memberof SASQueryParameters
     */
    function SASQueryParameters(version, signature, permissions, services, resourceTypes, protocol, startTime, expiryTime, ipRange, identifier, resource, cacheControl, contentDisposition, contentEncoding, contentLanguage, contentType, userDelegationKey) {
        this.version = version;
        this.services = services;
        this.resourceTypes = resourceTypes;
        this.expiryTime = expiryTime;
        this.permissions = permissions;
        this.protocol = protocol;
        this.startTime = startTime;
        this.ipRangeInner = ipRange;
        this.identifier = identifier;
        this.resource = resource;
        this.signature = signature;
        this.cacheControl = cacheControl;
        this.contentDisposition = contentDisposition;
        this.contentEncoding = contentEncoding;
        this.contentLanguage = contentLanguage;
        this.contentType = contentType;
        if (userDelegationKey) {
            this.signedOid = userDelegationKey.signedOid;
            this.signedTid = userDelegationKey.signedTid;
            this.signedStart = userDelegationKey.signedStart;
            this.signedExpiry = userDelegationKey.signedExpiry;
            this.signedService = userDelegationKey.signedService;
            this.signedVersion = userDelegationKey.signedVersion;
        }
    }
    Object.defineProperty(SASQueryParameters.prototype, "ipRange", {
        /**
         * Optional. IP range allowed for this SAS.
         *
         * @readonly
         * @type {(IIPRange | undefined)}
         * @memberof SASQueryParameters
         */
        get: function () {
            if (this.ipRangeInner) {
                return {
                    end: this.ipRangeInner.end,
                    start: this.ipRangeInner.start
                };
            }
            return undefined;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     * @returns {string}
     * @memberof SASQueryParameters
     */
    SASQueryParameters.prototype.toString = function () {
        var params = [
            "sv",
            "ss",
            "srt",
            "spr",
            "st",
            "se",
            "sip",
            "si",
            "skoid",
            "sktid",
            "skt",
            "ske",
            "sks",
            "skv",
            "sr",
            "sp",
            "sig",
            "rscc",
            "rscd",
            "rsce",
            "rscl",
            "rsct"
        ];
        var queries = [];
        for (var _i = 0, params_1 = params; _i < params_1.length; _i++) {
            var param = params_1[_i];
            switch (param) {
                case "sv":
                    this.tryAppendQueryParameter(queries, param, this.version);
                    break;
                case "ss":
                    this.tryAppendQueryParameter(queries, param, this.services);
                    break;
                case "srt":
                    this.tryAppendQueryParameter(queries, param, this.resourceTypes);
                    break;
                case "spr":
                    this.tryAppendQueryParameter(queries, param, this.protocol);
                    break;
                case "st":
                    this.tryAppendQueryParameter(queries, param, this.startTime ? truncatedISO8061Date(this.startTime, false) : undefined);
                    break;
                case "se":
                    this.tryAppendQueryParameter(queries, param, this.expiryTime ? truncatedISO8061Date(this.expiryTime, false) : undefined);
                    break;
                case "sip":
                    this.tryAppendQueryParameter(queries, param, this.ipRange ? ipRangeToString(this.ipRange) : undefined);
                    break;
                case "si":
                    this.tryAppendQueryParameter(queries, param, this.identifier);
                    break;
                case "skoid": // Signed object ID
                    this.tryAppendQueryParameter(queries, param, this.signedOid);
                    break;
                case "sktid": // Signed tenant ID
                    this.tryAppendQueryParameter(queries, param, this.signedTid);
                    break;
                case "skt": // Signed key start time
                    this.tryAppendQueryParameter(queries, param, this.signedStart ? truncatedISO8061Date(this.signedStart, false) : undefined);
                    break;
                case "ske": // Signed key expiry time
                    this.tryAppendQueryParameter(queries, param, this.signedExpiry ? truncatedISO8061Date(this.signedExpiry, false) : undefined);
                    break;
                case "sks": // Signed key service
                    this.tryAppendQueryParameter(queries, param, this.signedService);
                    break;
                case "skv": // Signed key version
                    this.tryAppendQueryParameter(queries, param, this.signedVersion);
                    break;
                case "sr":
                    this.tryAppendQueryParameter(queries, param, this.resource);
                    break;
                case "sp":
                    this.tryAppendQueryParameter(queries, param, this.permissions);
                    break;
                case "sig":
                    this.tryAppendQueryParameter(queries, param, this.signature);
                    break;
                case "rscc":
                    this.tryAppendQueryParameter(queries, param, this.cacheControl);
                    break;
                case "rscd":
                    this.tryAppendQueryParameter(queries, param, this.contentDisposition);
                    break;
                case "rsce":
                    this.tryAppendQueryParameter(queries, param, this.contentEncoding);
                    break;
                case "rscl":
                    this.tryAppendQueryParameter(queries, param, this.contentLanguage);
                    break;
                case "rsct":
                    this.tryAppendQueryParameter(queries, param, this.contentType);
                    break;
            }
        }
        return queries.join("&");
    };
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @private
     * @param {string[]} queries
     * @param {string} key
     * @param {string} [value]
     * @returns {void}
     * @memberof SASQueryParameters
     */
    SASQueryParameters.prototype.tryAppendQueryParameter = function (queries, key, value) {
        if (!value) {
            return;
        }
        key = encodeURIComponent(key);
        value = encodeURIComponent(value);
        if (key.length > 0 && value.length > 0) {
            queries.push(key + "=" + value);
        }
    };
    return SASQueryParameters;
}());

var SDK_VERSION = "10.5.0";
var SERVICE_VERSION = "2019-02-02";
var BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES = 256 * 1024 * 1024; // 256MB
var BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES = 100 * 1024 * 1024; // 100MB
var BLOCK_BLOB_MAX_BLOCKS = 50000;
var DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES = 4 * 1024 * 1024; // 4MB
var DEFAULT_MAX_DOWNLOAD_RETRY_REQUESTS = 5;
var URLConstants = {
    Parameters: {
        FORCE_BROWSER_NO_CACHE: "_",
        SIGNATURE: "sig",
        SNAPSHOT: "snapshot",
        TIMEOUT: "timeout"
    }
};
var HTTPURLConnection = {
    HTTP_ACCEPTED: 202,
    HTTP_CONFLICT: 409,
    HTTP_NOT_FOUND: 404,
    HTTP_PRECON_FAILED: 412,
    HTTP_RANGE_NOT_SATISFIABLE: 416
};
var HeaderConstants = {
    AUTHORIZATION: "Authorization",
    AUTHORIZATION_SCHEME: "Bearer",
    CONTENT_ENCODING: "Content-Encoding",
    CONTENT_ID: "Content-ID",
    CONTENT_LANGUAGE: "Content-Language",
    CONTENT_LENGTH: "Content-Length",
    CONTENT_MD5: "Content-Md5",
    CONTENT_TRANSFER_ENCODING: "Content-Transfer-Encoding",
    CONTENT_TYPE: "Content-Type",
    COOKIE: "Cookie",
    DATE: "date",
    IF_MATCH: "if-match",
    IF_MODIFIED_SINCE: "if-modified-since",
    IF_NONE_MATCH: "if-none-match",
    IF_UNMODIFIED_SINCE: "if-unmodified-since",
    PREFIX_FOR_STORAGE: "x-ms-",
    RANGE: "Range",
    USER_AGENT: "User-Agent",
    X_MS_CLIENT_REQUEST_ID: "x-ms-client-request-id",
    X_MS_DATE: "x-ms-date",
    X_MS_ERROR_CODE: "x-ms-error-code",
    X_MS_VERSION: "x-ms-version"
};
var ETagNone = "";
var SIZE_1_MB = 1 * 1024 * 1024;
var BATCH_MAX_REQUEST = 256;
var BATCH_MAX_PAYLOAD_IN_BYTES = 4 * SIZE_1_MB;
var HTTP_LINE_ENDING = "\r\n";
var HTTP_VERSION_1_1 = "HTTP/1.1";
var EncryptionAlgorithmAES25 = "AES256";

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Generates a {@link SASQueryParameters} object which contains all SAS query parameters needed to make an actual
 * REST request.
 *
 * @see https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
 *
 * @param {SharedKeyCredential} sharedKeyCredential
 * @returns {SASQueryParameters}
 * @memberof IAccountSASSignatureValues
 */
function generateAccountSASQueryParameters(accountSASSignatureValues, sharedKeyCredential) {
    var version = accountSASSignatureValues.version
        ? accountSASSignatureValues.version
        : SERVICE_VERSION;
    var parsedPermissions = AccountSASPermissions.parse(accountSASSignatureValues.permissions).toString();
    var parsedServices = AccountSASServices.parse(accountSASSignatureValues.services).toString();
    var parsedResourceTypes = AccountSASResourceTypes.parse(accountSASSignatureValues.resourceTypes).toString();
    var stringToSign = [
        sharedKeyCredential.accountName,
        parsedPermissions,
        parsedServices,
        parsedResourceTypes,
        accountSASSignatureValues.startTime
            ? truncatedISO8061Date(accountSASSignatureValues.startTime, false)
            : "",
        truncatedISO8061Date(accountSASSignatureValues.expiryTime, false),
        accountSASSignatureValues.ipRange ? ipRangeToString(accountSASSignatureValues.ipRange) : "",
        accountSASSignatureValues.protocol ? accountSASSignatureValues.protocol : "",
        version,
        "" // Account SAS requires an additional newline character
    ].join("\n");
    var signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return new SASQueryParameters(version, signature, parsedPermissions, parsedServices, parsedResourceTypes, accountSASSignatureValues.protocol, accountSASSignatureValues.startTime, accountSASSignatureValues.expiryTime, accountSASSignatureValues.ipRange);
}

var ABORT_ERROR = new msRest.RestError("The request was aborted", msRest.RestError.REQUEST_ABORTED_ERROR);
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * A Node.js ReadableStream will internally retry when internal ReadableStream unexpected ends.
 *
 * @class RetriableReadableStream
 * @extends {Readable}
 */
var RetriableReadableStream = /** @class */ (function (_super) {
    tslib_1.__extends(RetriableReadableStream, _super);
    /**
     * Creates an instance of RetriableReadableStream.
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {NodeJS.ReadableStream} source The current ReadableStream returned from getter
     * @param {ReadableStreamGetter} getter A method calling downloading request returning
     *                                      a new ReadableStream from specified offset
     * @param {number} offset Offset position in original data source to read
     * @param {number} count How much data in original data source to read
     * @param {IRetriableReadableStreamOptions} [options={}]
     * @memberof RetriableReadableStream
     */
    function RetriableReadableStream(aborter, source, getter, offset, count, options) {
        if (options === void 0) { options = {}; }
        var _this = _super.call(this) || this;
        _this.retries = 0;
        _this.abortHandler = function () {
            _this.source.pause();
            _this.emit("error", ABORT_ERROR);
        };
        _this.aborter = aborter;
        _this.getter = getter;
        _this.source = source;
        _this.start = offset;
        _this.offset = offset;
        _this.end = offset + count - 1;
        _this.maxRetryRequests =
            options.maxRetryRequests && options.maxRetryRequests >= 0 ? options.maxRetryRequests : 0;
        _this.progress = options.progress;
        _this.options = options;
        aborter.addEventListener("abort", _this.abortHandler);
        _this.setSourceDataHandler();
        _this.setSourceEndHandler();
        _this.setSourceErrorHandler();
        return _this;
    }
    RetriableReadableStream.prototype._read = function () {
        if (!this.aborter.aborted) {
            this.source.resume();
        }
    };
    RetriableReadableStream.prototype.setSourceDataHandler = function () {
        var _this = this;
        this.source.on("data", function (data) {
            if (_this.options.doInjectErrorOnce) {
                _this.options.doInjectErrorOnce = undefined;
                _this.source.pause();
                _this.source.removeAllListeners("data");
                _this.source.emit("end");
                return;
            }
            // console.log(
            //   `Offset: ${this.offset}, Received ${data.length} from internal stream`
            // );
            _this.offset += data.length;
            if (_this.progress) {
                _this.progress({ loadedBytes: _this.offset - _this.start });
            }
            if (!_this.push(data)) {
                _this.source.pause();
            }
        });
    };
    RetriableReadableStream.prototype.setSourceEndHandler = function () {
        var _this = this;
        this.source.on("end", function () {
            // console.log(
            //   `Source stream emits end, offset: ${
            //     this.offset
            //   }, dest end : ${this.end}`
            // );
            if (_this.offset - 1 === _this.end) {
                _this.aborter.removeEventListener("abort", _this.abortHandler);
                _this.push(null);
            }
            else if (_this.offset <= _this.end) {
                // console.log(
                //   `retries: ${this.retries}, max retries: ${this.maxRetries}`
                // );
                if (_this.retries < _this.maxRetryRequests) {
                    _this.retries += 1;
                    _this.getter(_this.offset)
                        .then(function (newSource) {
                        _this.source = newSource;
                        _this.setSourceDataHandler();
                        _this.setSourceEndHandler();
                        _this.setSourceErrorHandler();
                    })
                        .catch(function (error) {
                        _this.emit("error", error);
                    });
                }
                else {
                    _this.emit("error", new Error(
                    // tslint:disable-next-line:max-line-length
                    "Data corruption failure: received less data than required and reached maxRetires limitation. Received data offset: " + (_this
                        .offset - 1) + ", data needed offset: " + _this.end + ", retries: " + _this.retries + ", max retries: " + _this.maxRetryRequests));
                }
            }
            else {
                _this.emit("error", new Error("Data corruption failure: Received more data than original request, data needed offset is " + _this.end + ", received offset: " + (_this.offset - 1)));
            }
        });
    };
    RetriableReadableStream.prototype.setSourceErrorHandler = function () {
        var _this = this;
        this.source.on("error", function (error) {
            _this.emit("error", error);
        });
    };
    return RetriableReadableStream;
}(stream.Readable));

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * BlobDownloadResponse implements Models.BlobDownloadResponse interface, and in Node.js runtime it will
 * automatically retry when internal read stream unexpected ends. (This kind of unexpected ends cannot
 * trigger retries defined in pipeline retry policy.)
 *
 * The readableStreamBody stream will retry underlayer, you can just use it as a normal Node.js
 * Readable stream.
 *
 * @export
 * @class BlobDownloadResponse
 * @implements {Models.BlobDownloadResponse}
 */
var BlobDownloadResponse = /** @class */ (function () {
    /**
     * Creates an instance of BlobDownloadResponse.
     *
     * @param {Aborter} aborter
     * @param {Models.BlobDownloadResponse} originalResponse
     * @param {ReadableStreamGetter} getter
     * @param {number} offset
     * @param {number} count
     * @param {IRetriableReadableStreamOptions} [options={}]
     * @memberof BlobDownloadResponse
     */
    function BlobDownloadResponse(aborter, originalResponse, getter, offset, count, options) {
        if (options === void 0) { options = {}; }
        this.originalResponse = originalResponse;
        this.blobDownloadStream = new RetriableReadableStream(aborter, this.originalResponse.readableStreamBody, getter, offset, count, options);
    }
    Object.defineProperty(BlobDownloadResponse.prototype, "acceptRanges", {
        /**
         * Indicates that the service supports
         * requests for partial file content.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.acceptRanges;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "cacheControl", {
        /**
         * Returns if it was previously specified
         * for the file.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.cacheControl;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentDisposition", {
        /**
         * Returns the value that was specified
         * for the 'x-ms-content-disposition' header and specifies how to process the
         * response.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentDisposition;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentEncoding", {
        /**
         * Returns the value that was specified
         * for the Content-Encoding request header.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentEncoding;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentLanguage", {
        /**
         * Returns the value that was specified
         * for the Content-Language request header.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentLanguage;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "blobSequenceNumber", {
        /**
         * The current sequence number for a
         * page blob. This header is not returned for block blobs or append blobs.
         *
         * @readonly
         * @type {(number | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.blobSequenceNumber;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "blobType", {
        /**
         * The blob's type. Possible values include:
         * 'BlockBlob', 'PageBlob', 'AppendBlob'.
         *
         * @readonly
         * @type {(Models.BlobType | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.blobType;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentLength", {
        /**
         * The number of bytes present in the
         * response body.
         *
         * @readonly
         * @type {(number | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentLength;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentMD5", {
        /**
         * If the file has an MD5 hash and the
         * request is to read the full file, this response header is returned so that
         * the client can check for message content integrity. If the request is to
         * read a specified range and the 'x-ms-range-get-content-md5' is set to
         * true, then the request returns an MD5 hash for the range, as long as the
         * range size is less than or equal to 4 MB. If neither of these sets of
         * conditions is true, then no value is returned for the 'Content-MD5'
         * header.
         *
         * @readonly
         * @type {(Uint8Array | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentMD5;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentRange", {
        /**
         * Indicates the range of bytes returned if
         * the client requested a subset of the file by setting the Range request
         * header.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentRange;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentType", {
        /**
         * The content type specified for the file.
         * The default content type is 'application/octet-stream'
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.contentType;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "copyCompletionTime", {
        /**
         * Conclusion time of the last attempted
         * Copy File operation where this file was the destination file. This value
         * can specify the time of a completed, aborted, or failed copy attempt.
         *
         * @readonly
         * @type {(Date | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.copyCompletionTime;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "copyId", {
        /**
         * String identifier for the last attempted Copy
         * File operation where this file was the destination file.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.copyId;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "copyProgress", {
        /**
         * Contains the number of bytes copied and
         * the total bytes in the source in the last attempted Copy File operation
         * where this file was the destination file. Can show between 0 and
         * Content-Length bytes copied.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.copyProgress;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "copySource", {
        /**
         * URL up to 2KB in length that specifies the
         * source file used in the last attempted Copy File operation where this file
         * was the destination file.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.copySource;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "copyStatus", {
        /**
         * State of the copy operation
         * identified by 'x-ms-copy-id'. Possible values include: 'pending',
         * 'success', 'aborted', 'failed'
         *
         * @readonly
         * @type {(Models.CopyStatusType | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.copyStatus;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "copyStatusDescription", {
        /**
         * Only appears when
         * x-ms-copy-status is failed or pending. Describes cause of fatal or
         * non-fatal copy operation failure.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.copyStatusDescription;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "leaseDuration", {
        /**
         * When a blob is leased,
         * specifies whether the lease is of infinite or fixed duration. Possible
         * values include: 'infinite', 'fixed'.
         *
         * @readonly
         * @type {(Models.LeaseDurationType | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.leaseDuration;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "leaseState", {
        /**
         * Lease state of the blob. Possible
         * values include: 'available', 'leased', 'expired', 'breaking', 'broken'.
         *
         * @readonly
         * @type {(Models.LeaseStateType | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.leaseState;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "leaseStatus", {
        /**
         * The current lease status of the
         * blob. Possible values include: 'locked', 'unlocked'.
         *
         * @readonly
         * @type {(Models.LeaseStatusType | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.leaseStatus;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "date", {
        /**
         * A UTC date/time value generated by the service that
         * indicates the time at which the response was initiated.
         *
         * @readonly
         * @type {(Date | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.date;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "blobCommittedBlockCount", {
        /**
         * The number of committed blocks
         * present in the blob. This header is returned only for append blobs.
         *
         * @readonly
         * @type {(number | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.blobCommittedBlockCount;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "eTag", {
        /**
         * The ETag contains a value that you can use to
         * perform operations conditionally, in quotes.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.eTag;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "errorCode", {
        get: function () {
            return this.originalResponse.errorCode;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "isServerEncrypted", {
        /**
         * The value of this header is set to
         * true if the file data and application metadata are completely encrypted
         * using the specified algorithm. Otherwise, the value is set to false (when
         * the file is unencrypted, or if only parts of the file/application metadata
         * are encrypted).
         *
         * @readonly
         * @type {(boolean | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.isServerEncrypted;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "blobContentMD5", {
        /**
         * If the blob has a MD5 hash, and if
         * request contains range header (Range or x-ms-range), this response header
         * is returned with the value of the whole blob's MD5 value. This value may
         * or may not be equal to the value returned in Content-MD5 header, with the
         * latter calculated from the requested range.
         *
         * @readonly
         * @type {(Uint8Array | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.blobContentMD5;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "lastModified", {
        /**
         * Returns the date and time the file was last
         * modified. Any operation that modifies the file or its properties updates
         * the last modified time.
         *
         * @readonly
         * @type {(Date | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.lastModified;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "metadata", {
        /**
         * A name-value pair
         * to associate with a file storage object.
         *
         * @readonly
         * @type {(IMetadata | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.metadata;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "requestId", {
        /**
         * This header uniquely identifies the request
         * that was made and can be used for troubleshooting the request.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.requestId;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "clientRequestId", {
        /**
         * If a client request id header is sent in the request, this header will be present in the
         * response with the same value.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.clientRequestId;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "version", {
        /**
         * Indicates the version of the File service used
         * to execute the request.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.version;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "encryptionKeySha256", {
        /**
         * The SHA-256 hash of the encryption key used to encrypt the blob. This value is only returned
         * when the blob was encrypted with a customer-provided key.
         *
         * @readonly
         * @type {(string | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.encryptionKeySha256;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "contentCrc64", {
        /**
         * If the request is to read a specified range and the x-ms-range-get-content-crc64 is set to
         * true, then the request returns a crc64 for the range, as long as the range size is less than
         * or equal to 4 MB. If both x-ms-range-get-content-crc64 & x-ms-range-get-content-md5 is
         * specified in the same request, it will fail with 400(Bad Request)
         */
        get: function () {
            return this.originalResponse.contentCrc64;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "blobBody", {
        /**
         * The response body as a browser Blob.
         * Always undefined in node.js.
         *
         * @readonly
         * @type {(Promise<Blob> | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return this.originalResponse.blobBody;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "readableStreamBody", {
        /**
         * The response body as a node.js Readable stream.
         * Always undefined in the browser.
         *
         * It will automatically retry when internal read stream unexpected ends.
         *
         * @readonly
         * @type {(NodeJS.ReadableStream | undefined)}
         * @memberof BlobDownloadResponse
         */
        get: function () {
            return msRest.isNode ? this.blobDownloadStream : undefined;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BlobDownloadResponse.prototype, "_response", {
        get: function () {
            return this.originalResponse._response;
        },
        enumerable: true,
        configurable: true
    });
    return BlobDownloadResponse;
}());

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */
var KeyInfo = {
    serializedName: "KeyInfo",
    type: {
        name: "Composite",
        className: "KeyInfo",
        modelProperties: {
            start: {
                xmlName: "Start",
                required: true,
                serializedName: "Start",
                type: {
                    name: "String"
                }
            },
            expiry: {
                xmlName: "Expiry",
                required: true,
                serializedName: "Expiry",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var UserDelegationKey = {
    serializedName: "UserDelegationKey",
    type: {
        name: "Composite",
        className: "UserDelegationKey",
        modelProperties: {
            signedOid: {
                xmlName: "SignedOid",
                required: true,
                serializedName: "SignedOid",
                type: {
                    name: "String"
                }
            },
            signedTid: {
                xmlName: "SignedTid",
                required: true,
                serializedName: "SignedTid",
                type: {
                    name: "String"
                }
            },
            signedStart: {
                xmlName: "SignedStart",
                required: true,
                serializedName: "SignedStart",
                type: {
                    name: "String"
                }
            },
            signedExpiry: {
                xmlName: "SignedExpiry",
                required: true,
                serializedName: "SignedExpiry",
                type: {
                    name: "String"
                }
            },
            signedService: {
                xmlName: "SignedService",
                required: true,
                serializedName: "SignedService",
                type: {
                    name: "String"
                }
            },
            signedVersion: {
                xmlName: "SignedVersion",
                required: true,
                serializedName: "SignedVersion",
                type: {
                    name: "String"
                }
            },
            value: {
                xmlName: "Value",
                required: true,
                serializedName: "Value",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var StorageError = {
    serializedName: "StorageError",
    type: {
        name: "Composite",
        className: "StorageError",
        modelProperties: {
            message: {
                xmlName: "Message",
                serializedName: "Message",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var DataLakeStorageErrorError = {
    serializedName: "DataLakeStorageError_error",
    type: {
        name: "Composite",
        className: "DataLakeStorageErrorError",
        modelProperties: {
            code: {
                xmlName: "Code",
                serializedName: "Code",
                type: {
                    name: "String"
                }
            },
            message: {
                xmlName: "Message",
                serializedName: "Message",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var DataLakeStorageError = {
    serializedName: "DataLakeStorageError",
    type: {
        name: "Composite",
        className: "DataLakeStorageError",
        modelProperties: {
            error: {
                xmlName: "error",
                serializedName: "error",
                type: {
                    name: "Composite",
                    className: "DataLakeStorageErrorError"
                }
            }
        }
    }
};
var AccessPolicy = {
    serializedName: "AccessPolicy",
    type: {
        name: "Composite",
        className: "AccessPolicy",
        modelProperties: {
            start: {
                xmlName: "Start",
                required: true,
                serializedName: "Start",
                type: {
                    name: "String"
                }
            },
            expiry: {
                xmlName: "Expiry",
                required: true,
                serializedName: "Expiry",
                type: {
                    name: "String"
                }
            },
            permission: {
                xmlName: "Permission",
                required: true,
                serializedName: "Permission",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobProperties = {
    xmlName: "Properties",
    serializedName: "BlobProperties",
    type: {
        name: "Composite",
        className: "BlobProperties",
        modelProperties: {
            creationTime: {
                xmlName: "Creation-Time",
                serializedName: "Creation-Time",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            lastModified: {
                xmlName: "Last-Modified",
                required: true,
                serializedName: "Last-Modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            etag: {
                xmlName: "Etag",
                required: true,
                serializedName: "Etag",
                type: {
                    name: "String"
                }
            },
            contentLength: {
                xmlName: "Content-Length",
                serializedName: "Content-Length",
                type: {
                    name: "Number"
                }
            },
            contentType: {
                xmlName: "Content-Type",
                serializedName: "Content-Type",
                type: {
                    name: "String"
                }
            },
            contentEncoding: {
                xmlName: "Content-Encoding",
                serializedName: "Content-Encoding",
                type: {
                    name: "String"
                }
            },
            contentLanguage: {
                xmlName: "Content-Language",
                serializedName: "Content-Language",
                type: {
                    name: "String"
                }
            },
            contentMD5: {
                xmlName: "Content-MD5",
                serializedName: "Content-MD5",
                type: {
                    name: "ByteArray"
                }
            },
            contentDisposition: {
                xmlName: "Content-Disposition",
                serializedName: "Content-Disposition",
                type: {
                    name: "String"
                }
            },
            cacheControl: {
                xmlName: "Cache-Control",
                serializedName: "Cache-Control",
                type: {
                    name: "String"
                }
            },
            blobSequenceNumber: {
                xmlName: "x-ms-blob-sequence-number",
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            blobType: {
                xmlName: "BlobType",
                serializedName: "BlobType",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "BlockBlob",
                        "PageBlob",
                        "AppendBlob"
                    ]
                }
            },
            leaseStatus: {
                xmlName: "LeaseStatus",
                serializedName: "LeaseStatus",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "locked",
                        "unlocked"
                    ]
                }
            },
            leaseState: {
                xmlName: "LeaseState",
                serializedName: "LeaseState",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "available",
                        "leased",
                        "expired",
                        "breaking",
                        "broken"
                    ]
                }
            },
            leaseDuration: {
                xmlName: "LeaseDuration",
                serializedName: "LeaseDuration",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "infinite",
                        "fixed"
                    ]
                }
            },
            copyId: {
                xmlName: "CopyId",
                serializedName: "CopyId",
                type: {
                    name: "String"
                }
            },
            copyStatus: {
                xmlName: "CopyStatus",
                serializedName: "CopyStatus",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "pending",
                        "success",
                        "aborted",
                        "failed"
                    ]
                }
            },
            copySource: {
                xmlName: "CopySource",
                serializedName: "CopySource",
                type: {
                    name: "String"
                }
            },
            copyProgress: {
                xmlName: "CopyProgress",
                serializedName: "CopyProgress",
                type: {
                    name: "String"
                }
            },
            copyCompletionTime: {
                xmlName: "CopyCompletionTime",
                serializedName: "CopyCompletionTime",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            copyStatusDescription: {
                xmlName: "CopyStatusDescription",
                serializedName: "CopyStatusDescription",
                type: {
                    name: "String"
                }
            },
            serverEncrypted: {
                xmlName: "ServerEncrypted",
                serializedName: "ServerEncrypted",
                type: {
                    name: "Boolean"
                }
            },
            incrementalCopy: {
                xmlName: "IncrementalCopy",
                serializedName: "IncrementalCopy",
                type: {
                    name: "Boolean"
                }
            },
            destinationSnapshot: {
                xmlName: "DestinationSnapshot",
                serializedName: "DestinationSnapshot",
                type: {
                    name: "String"
                }
            },
            deletedTime: {
                xmlName: "DeletedTime",
                serializedName: "DeletedTime",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            remainingRetentionDays: {
                xmlName: "RemainingRetentionDays",
                serializedName: "RemainingRetentionDays",
                type: {
                    name: "Number"
                }
            },
            accessTier: {
                xmlName: "AccessTier",
                serializedName: "AccessTier",
                type: {
                    name: "String"
                }
            },
            accessTierInferred: {
                xmlName: "AccessTierInferred",
                serializedName: "AccessTierInferred",
                type: {
                    name: "Boolean"
                }
            },
            archiveStatus: {
                xmlName: "ArchiveStatus",
                serializedName: "ArchiveStatus",
                type: {
                    name: "String"
                }
            },
            customerProvidedKeySha256: {
                xmlName: "CustomerProvidedKeySha256",
                serializedName: "CustomerProvidedKeySha256",
                type: {
                    name: "String"
                }
            },
            accessTierChangeTime: {
                xmlName: "AccessTierChangeTime",
                serializedName: "AccessTierChangeTime",
                type: {
                    name: "DateTimeRfc1123"
                }
            }
        }
    }
};
var BlobMetadata = {
    xmlName: "Metadata",
    serializedName: "BlobMetadata",
    type: {
        name: "Composite",
        className: "BlobMetadata",
        modelProperties: {
            encrypted: {
                xmlIsAttribute: true,
                xmlName: "Encrypted",
                serializedName: "Encrypted",
                type: {
                    name: "String"
                }
            }
        },
        additionalProperties: {
            type: {
                name: "String"
            }
        }
    }
};
var BlobItem = {
    xmlName: "Blob",
    serializedName: "BlobItem",
    type: {
        name: "Composite",
        className: "BlobItem",
        modelProperties: {
            name: {
                xmlName: "Name",
                required: true,
                serializedName: "Name",
                type: {
                    name: "String"
                }
            },
            deleted: {
                xmlName: "Deleted",
                required: true,
                serializedName: "Deleted",
                type: {
                    name: "Boolean"
                }
            },
            snapshot: {
                xmlName: "Snapshot",
                required: true,
                serializedName: "Snapshot",
                type: {
                    name: "String"
                }
            },
            properties: {
                xmlName: "Properties",
                required: true,
                serializedName: "Properties",
                type: {
                    name: "Composite",
                    className: "BlobProperties"
                }
            },
            metadata: {
                xmlName: "Metadata",
                serializedName: "Metadata",
                type: {
                    name: "Composite",
                    className: "BlobMetadata",
                    additionalProperties: {
                        type: {
                            name: "String"
                        }
                    }
                }
            }
        }
    }
};
var BlobFlatListSegment = {
    xmlName: "Blobs",
    serializedName: "BlobFlatListSegment",
    type: {
        name: "Composite",
        className: "BlobFlatListSegment",
        modelProperties: {
            blobItems: {
                xmlName: "BlobItems",
                xmlElementName: "Blob",
                required: true,
                serializedName: "BlobItems",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "BlobItem"
                        }
                    }
                }
            }
        }
    }
};
var ListBlobsFlatSegmentResponse = {
    xmlName: "EnumerationResults",
    serializedName: "ListBlobsFlatSegmentResponse",
    type: {
        name: "Composite",
        className: "ListBlobsFlatSegmentResponse",
        modelProperties: {
            serviceEndpoint: {
                xmlIsAttribute: true,
                xmlName: "ServiceEndpoint",
                required: true,
                serializedName: "ServiceEndpoint",
                type: {
                    name: "String"
                }
            },
            containerName: {
                xmlIsAttribute: true,
                xmlName: "ContainerName",
                required: true,
                serializedName: "ContainerName",
                type: {
                    name: "String"
                }
            },
            prefix: {
                xmlName: "Prefix",
                serializedName: "Prefix",
                type: {
                    name: "String"
                }
            },
            marker: {
                xmlName: "Marker",
                serializedName: "Marker",
                type: {
                    name: "String"
                }
            },
            maxResults: {
                xmlName: "MaxResults",
                serializedName: "MaxResults",
                type: {
                    name: "Number"
                }
            },
            delimiter: {
                xmlName: "Delimiter",
                serializedName: "Delimiter",
                type: {
                    name: "String"
                }
            },
            segment: {
                xmlName: "Blobs",
                required: true,
                serializedName: "Segment",
                type: {
                    name: "Composite",
                    className: "BlobFlatListSegment"
                }
            },
            nextMarker: {
                xmlName: "NextMarker",
                serializedName: "NextMarker",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobPrefix = {
    serializedName: "BlobPrefix",
    type: {
        name: "Composite",
        className: "BlobPrefix",
        modelProperties: {
            name: {
                xmlName: "Name",
                required: true,
                serializedName: "Name",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobHierarchyListSegment = {
    xmlName: "Blobs",
    serializedName: "BlobHierarchyListSegment",
    type: {
        name: "Composite",
        className: "BlobHierarchyListSegment",
        modelProperties: {
            blobPrefixes: {
                xmlName: "BlobPrefixes",
                xmlElementName: "BlobPrefix",
                serializedName: "BlobPrefixes",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "BlobPrefix"
                        }
                    }
                }
            },
            blobItems: {
                xmlName: "BlobItems",
                xmlElementName: "Blob",
                required: true,
                serializedName: "BlobItems",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "BlobItem"
                        }
                    }
                }
            }
        }
    }
};
var ListBlobsHierarchySegmentResponse = {
    xmlName: "EnumerationResults",
    serializedName: "ListBlobsHierarchySegmentResponse",
    type: {
        name: "Composite",
        className: "ListBlobsHierarchySegmentResponse",
        modelProperties: {
            serviceEndpoint: {
                xmlIsAttribute: true,
                xmlName: "ServiceEndpoint",
                required: true,
                serializedName: "ServiceEndpoint",
                type: {
                    name: "String"
                }
            },
            containerName: {
                xmlIsAttribute: true,
                xmlName: "ContainerName",
                required: true,
                serializedName: "ContainerName",
                type: {
                    name: "String"
                }
            },
            prefix: {
                xmlName: "Prefix",
                serializedName: "Prefix",
                type: {
                    name: "String"
                }
            },
            marker: {
                xmlName: "Marker",
                serializedName: "Marker",
                type: {
                    name: "String"
                }
            },
            maxResults: {
                xmlName: "MaxResults",
                serializedName: "MaxResults",
                type: {
                    name: "Number"
                }
            },
            delimiter: {
                xmlName: "Delimiter",
                serializedName: "Delimiter",
                type: {
                    name: "String"
                }
            },
            segment: {
                xmlName: "Blobs",
                required: true,
                serializedName: "Segment",
                type: {
                    name: "Composite",
                    className: "BlobHierarchyListSegment"
                }
            },
            nextMarker: {
                xmlName: "NextMarker",
                serializedName: "NextMarker",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var Block = {
    serializedName: "Block",
    type: {
        name: "Composite",
        className: "Block",
        modelProperties: {
            name: {
                xmlName: "Name",
                required: true,
                serializedName: "Name",
                type: {
                    name: "String"
                }
            },
            size: {
                xmlName: "Size",
                required: true,
                serializedName: "Size",
                type: {
                    name: "Number"
                }
            }
        }
    }
};
var BlockList = {
    serializedName: "BlockList",
    type: {
        name: "Composite",
        className: "BlockList",
        modelProperties: {
            committedBlocks: {
                xmlIsWrapped: true,
                xmlName: "CommittedBlocks",
                xmlElementName: "Block",
                serializedName: "CommittedBlocks",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "Block"
                        }
                    }
                }
            },
            uncommittedBlocks: {
                xmlIsWrapped: true,
                xmlName: "UncommittedBlocks",
                xmlElementName: "Block",
                serializedName: "UncommittedBlocks",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "Block"
                        }
                    }
                }
            }
        }
    }
};
var BlockLookupList = {
    xmlName: "BlockList",
    serializedName: "BlockLookupList",
    type: {
        name: "Composite",
        className: "BlockLookupList",
        modelProperties: {
            committed: {
                xmlName: "Committed",
                xmlElementName: "Committed",
                serializedName: "Committed",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "String"
                        }
                    }
                }
            },
            uncommitted: {
                xmlName: "Uncommitted",
                xmlElementName: "Uncommitted",
                serializedName: "Uncommitted",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "String"
                        }
                    }
                }
            },
            latest: {
                xmlName: "Latest",
                xmlElementName: "Latest",
                serializedName: "Latest",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "String"
                        }
                    }
                }
            }
        }
    }
};
var ContainerProperties = {
    serializedName: "ContainerProperties",
    type: {
        name: "Composite",
        className: "ContainerProperties",
        modelProperties: {
            lastModified: {
                xmlName: "Last-Modified",
                required: true,
                serializedName: "Last-Modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            etag: {
                xmlName: "Etag",
                required: true,
                serializedName: "Etag",
                type: {
                    name: "String"
                }
            },
            leaseStatus: {
                xmlName: "LeaseStatus",
                serializedName: "LeaseStatus",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "locked",
                        "unlocked"
                    ]
                }
            },
            leaseState: {
                xmlName: "LeaseState",
                serializedName: "LeaseState",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "available",
                        "leased",
                        "expired",
                        "breaking",
                        "broken"
                    ]
                }
            },
            leaseDuration: {
                xmlName: "LeaseDuration",
                serializedName: "LeaseDuration",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "infinite",
                        "fixed"
                    ]
                }
            },
            publicAccess: {
                xmlName: "PublicAccess",
                serializedName: "PublicAccess",
                type: {
                    name: "String"
                }
            },
            hasImmutabilityPolicy: {
                xmlName: "HasImmutabilityPolicy",
                serializedName: "HasImmutabilityPolicy",
                type: {
                    name: "Boolean"
                }
            },
            hasLegalHold: {
                xmlName: "HasLegalHold",
                serializedName: "HasLegalHold",
                type: {
                    name: "Boolean"
                }
            }
        }
    }
};
var ContainerItem = {
    xmlName: "Container",
    serializedName: "ContainerItem",
    type: {
        name: "Composite",
        className: "ContainerItem",
        modelProperties: {
            name: {
                xmlName: "Name",
                required: true,
                serializedName: "Name",
                type: {
                    name: "String"
                }
            },
            properties: {
                xmlName: "Properties",
                required: true,
                serializedName: "Properties",
                type: {
                    name: "Composite",
                    className: "ContainerProperties"
                }
            },
            metadata: {
                xmlName: "Metadata",
                serializedName: "Metadata",
                type: {
                    name: "Dictionary",
                    value: {
                        type: {
                            name: "String"
                        }
                    }
                }
            }
        }
    }
};
var ListContainersSegmentResponse = {
    xmlName: "EnumerationResults",
    serializedName: "ListContainersSegmentResponse",
    type: {
        name: "Composite",
        className: "ListContainersSegmentResponse",
        modelProperties: {
            serviceEndpoint: {
                xmlIsAttribute: true,
                xmlName: "ServiceEndpoint",
                required: true,
                serializedName: "ServiceEndpoint",
                type: {
                    name: "String"
                }
            },
            prefix: {
                xmlName: "Prefix",
                serializedName: "Prefix",
                type: {
                    name: "String"
                }
            },
            marker: {
                xmlName: "Marker",
                serializedName: "Marker",
                type: {
                    name: "String"
                }
            },
            maxResults: {
                xmlName: "MaxResults",
                serializedName: "MaxResults",
                type: {
                    name: "Number"
                }
            },
            containerItems: {
                xmlIsWrapped: true,
                xmlName: "Containers",
                xmlElementName: "Container",
                required: true,
                serializedName: "ContainerItems",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "ContainerItem"
                        }
                    }
                }
            },
            nextMarker: {
                xmlName: "NextMarker",
                serializedName: "NextMarker",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var CorsRule = {
    serializedName: "CorsRule",
    type: {
        name: "Composite",
        className: "CorsRule",
        modelProperties: {
            allowedOrigins: {
                xmlName: "AllowedOrigins",
                required: true,
                serializedName: "AllowedOrigins",
                type: {
                    name: "String"
                }
            },
            allowedMethods: {
                xmlName: "AllowedMethods",
                required: true,
                serializedName: "AllowedMethods",
                type: {
                    name: "String"
                }
            },
            allowedHeaders: {
                xmlName: "AllowedHeaders",
                required: true,
                serializedName: "AllowedHeaders",
                type: {
                    name: "String"
                }
            },
            exposedHeaders: {
                xmlName: "ExposedHeaders",
                required: true,
                serializedName: "ExposedHeaders",
                type: {
                    name: "String"
                }
            },
            maxAgeInSeconds: {
                xmlName: "MaxAgeInSeconds",
                required: true,
                serializedName: "MaxAgeInSeconds",
                constraints: {
                    InclusiveMinimum: 0
                },
                type: {
                    name: "Number"
                }
            }
        }
    }
};
var GeoReplication = {
    serializedName: "GeoReplication",
    type: {
        name: "Composite",
        className: "GeoReplication",
        modelProperties: {
            status: {
                xmlName: "Status",
                required: true,
                serializedName: "Status",
                type: {
                    name: "String"
                }
            },
            lastSyncTime: {
                xmlName: "LastSyncTime",
                required: true,
                serializedName: "LastSyncTime",
                type: {
                    name: "DateTimeRfc1123"
                }
            }
        }
    }
};
var RetentionPolicy = {
    serializedName: "RetentionPolicy",
    type: {
        name: "Composite",
        className: "RetentionPolicy",
        modelProperties: {
            enabled: {
                xmlName: "Enabled",
                required: true,
                serializedName: "Enabled",
                type: {
                    name: "Boolean"
                }
            },
            days: {
                xmlName: "Days",
                serializedName: "Days",
                constraints: {
                    InclusiveMinimum: 1
                },
                type: {
                    name: "Number"
                }
            }
        }
    }
};
var Logging = {
    serializedName: "Logging",
    type: {
        name: "Composite",
        className: "Logging",
        modelProperties: {
            version: {
                xmlName: "Version",
                required: true,
                serializedName: "Version",
                type: {
                    name: "String"
                }
            },
            deleteProperty: {
                xmlName: "Delete",
                required: true,
                serializedName: "Delete",
                type: {
                    name: "Boolean"
                }
            },
            read: {
                xmlName: "Read",
                required: true,
                serializedName: "Read",
                type: {
                    name: "Boolean"
                }
            },
            write: {
                xmlName: "Write",
                required: true,
                serializedName: "Write",
                type: {
                    name: "Boolean"
                }
            },
            retentionPolicy: {
                xmlName: "RetentionPolicy",
                required: true,
                serializedName: "RetentionPolicy",
                type: {
                    name: "Composite",
                    className: "RetentionPolicy"
                }
            }
        }
    }
};
var Metrics = {
    serializedName: "Metrics",
    type: {
        name: "Composite",
        className: "Metrics",
        modelProperties: {
            version: {
                xmlName: "Version",
                serializedName: "Version",
                type: {
                    name: "String"
                }
            },
            enabled: {
                xmlName: "Enabled",
                required: true,
                serializedName: "Enabled",
                type: {
                    name: "Boolean"
                }
            },
            includeAPIs: {
                xmlName: "IncludeAPIs",
                serializedName: "IncludeAPIs",
                type: {
                    name: "Boolean"
                }
            },
            retentionPolicy: {
                xmlName: "RetentionPolicy",
                serializedName: "RetentionPolicy",
                type: {
                    name: "Composite",
                    className: "RetentionPolicy"
                }
            }
        }
    }
};
var PageRange = {
    serializedName: "PageRange",
    type: {
        name: "Composite",
        className: "PageRange",
        modelProperties: {
            start: {
                xmlName: "Start",
                required: true,
                serializedName: "Start",
                type: {
                    name: "Number"
                }
            },
            end: {
                xmlName: "End",
                required: true,
                serializedName: "End",
                type: {
                    name: "Number"
                }
            }
        }
    }
};
var ClearRange = {
    serializedName: "ClearRange",
    type: {
        name: "Composite",
        className: "ClearRange",
        modelProperties: {
            start: {
                xmlName: "Start",
                required: true,
                serializedName: "Start",
                type: {
                    name: "Number"
                }
            },
            end: {
                xmlName: "End",
                required: true,
                serializedName: "End",
                type: {
                    name: "Number"
                }
            }
        }
    }
};
var PageList = {
    serializedName: "PageList",
    type: {
        name: "Composite",
        className: "PageList",
        modelProperties: {
            pageRange: {
                xmlName: "PageRange",
                xmlElementName: "PageRange",
                serializedName: "PageRange",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "PageRange"
                        }
                    }
                }
            },
            clearRange: {
                xmlName: "ClearRange",
                xmlElementName: "ClearRange",
                serializedName: "ClearRange",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "ClearRange"
                        }
                    }
                }
            }
        }
    }
};
var SignedIdentifier = {
    serializedName: "SignedIdentifier",
    type: {
        name: "Composite",
        className: "SignedIdentifier",
        modelProperties: {
            id: {
                xmlName: "Id",
                required: true,
                serializedName: "Id",
                type: {
                    name: "String"
                }
            },
            accessPolicy: {
                xmlName: "AccessPolicy",
                required: true,
                serializedName: "AccessPolicy",
                type: {
                    name: "Composite",
                    className: "AccessPolicy"
                }
            }
        }
    }
};
var StaticWebsite = {
    serializedName: "StaticWebsite",
    type: {
        name: "Composite",
        className: "StaticWebsite",
        modelProperties: {
            enabled: {
                xmlName: "Enabled",
                required: true,
                serializedName: "Enabled",
                type: {
                    name: "Boolean"
                }
            },
            indexDocument: {
                xmlName: "IndexDocument",
                serializedName: "IndexDocument",
                type: {
                    name: "String"
                }
            },
            errorDocument404Path: {
                xmlName: "ErrorDocument404Path",
                serializedName: "ErrorDocument404Path",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var StorageServiceProperties = {
    serializedName: "StorageServiceProperties",
    type: {
        name: "Composite",
        className: "StorageServiceProperties",
        modelProperties: {
            logging: {
                xmlName: "Logging",
                serializedName: "Logging",
                type: {
                    name: "Composite",
                    className: "Logging"
                }
            },
            hourMetrics: {
                xmlName: "HourMetrics",
                serializedName: "HourMetrics",
                type: {
                    name: "Composite",
                    className: "Metrics"
                }
            },
            minuteMetrics: {
                xmlName: "MinuteMetrics",
                serializedName: "MinuteMetrics",
                type: {
                    name: "Composite",
                    className: "Metrics"
                }
            },
            cors: {
                xmlIsWrapped: true,
                xmlName: "Cors",
                xmlElementName: "CorsRule",
                serializedName: "Cors",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "CorsRule"
                        }
                    }
                }
            },
            defaultServiceVersion: {
                xmlName: "DefaultServiceVersion",
                serializedName: "DefaultServiceVersion",
                type: {
                    name: "String"
                }
            },
            deleteRetentionPolicy: {
                xmlName: "DeleteRetentionPolicy",
                serializedName: "DeleteRetentionPolicy",
                type: {
                    name: "Composite",
                    className: "RetentionPolicy"
                }
            },
            staticWebsite: {
                xmlName: "StaticWebsite",
                serializedName: "StaticWebsite",
                type: {
                    name: "Composite",
                    className: "StaticWebsite"
                }
            }
        }
    }
};
var StorageServiceStats = {
    serializedName: "StorageServiceStats",
    type: {
        name: "Composite",
        className: "StorageServiceStats",
        modelProperties: {
            geoReplication: {
                xmlName: "GeoReplication",
                serializedName: "GeoReplication",
                type: {
                    name: "Composite",
                    className: "GeoReplication"
                }
            }
        }
    }
};
var ServiceSetPropertiesHeaders = {
    serializedName: "service-setproperties-headers",
    type: {
        name: "Composite",
        className: "ServiceSetPropertiesHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ServiceGetPropertiesHeaders = {
    serializedName: "service-getproperties-headers",
    type: {
        name: "Composite",
        className: "ServiceGetPropertiesHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ServiceGetStatisticsHeaders = {
    serializedName: "service-getstatistics-headers",
    type: {
        name: "Composite",
        className: "ServiceGetStatisticsHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ServiceListContainersSegmentHeaders = {
    serializedName: "service-listcontainerssegment-headers",
    type: {
        name: "Composite",
        className: "ServiceListContainersSegmentHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ServiceGetUserDelegationKeyHeaders = {
    serializedName: "service-getuserdelegationkey-headers",
    type: {
        name: "Composite",
        className: "ServiceGetUserDelegationKeyHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ServiceGetAccountInfoHeaders = {
    serializedName: "service-getaccountinfo-headers",
    type: {
        name: "Composite",
        className: "ServiceGetAccountInfoHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            skuName: {
                serializedName: "x-ms-sku-name",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "Standard_LRS",
                        "Standard_GRS",
                        "Standard_RAGRS",
                        "Standard_ZRS",
                        "Premium_LRS"
                    ]
                }
            },
            accountKind: {
                serializedName: "x-ms-account-kind",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "Storage",
                        "BlobStorage",
                        "StorageV2"
                    ]
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ServiceSubmitBatchHeaders = {
    serializedName: "service-submitbatch-headers",
    type: {
        name: "Composite",
        className: "ServiceSubmitBatchHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            contentType: {
                serializedName: "content-type",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerCreateHeaders = {
    serializedName: "container-create-headers",
    type: {
        name: "Composite",
        className: "ContainerCreateHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerGetPropertiesHeaders = {
    serializedName: "container-getproperties-headers",
    type: {
        name: "Composite",
        className: "ContainerGetPropertiesHeaders",
        modelProperties: {
            metadata: {
                serializedName: "x-ms-meta",
                type: {
                    name: "Dictionary",
                    value: {
                        type: {
                            name: "String"
                        }
                    }
                },
                headerCollectionPrefix: "x-ms-meta-"
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseDuration: {
                serializedName: "x-ms-lease-duration",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "infinite",
                        "fixed"
                    ]
                }
            },
            leaseState: {
                serializedName: "x-ms-lease-state",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "available",
                        "leased",
                        "expired",
                        "breaking",
                        "broken"
                    ]
                }
            },
            leaseStatus: {
                serializedName: "x-ms-lease-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "locked",
                        "unlocked"
                    ]
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobPublicAccess: {
                serializedName: "x-ms-blob-public-access",
                type: {
                    name: "String"
                }
            },
            hasImmutabilityPolicy: {
                serializedName: "x-ms-has-immutability-policy",
                type: {
                    name: "Boolean"
                }
            },
            hasLegalHold: {
                serializedName: "x-ms-has-legal-hold",
                type: {
                    name: "Boolean"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerDeleteHeaders = {
    serializedName: "container-delete-headers",
    type: {
        name: "Composite",
        className: "ContainerDeleteHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerSetMetadataHeaders = {
    serializedName: "container-setmetadata-headers",
    type: {
        name: "Composite",
        className: "ContainerSetMetadataHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerGetAccessPolicyHeaders = {
    serializedName: "container-getaccesspolicy-headers",
    type: {
        name: "Composite",
        className: "ContainerGetAccessPolicyHeaders",
        modelProperties: {
            blobPublicAccess: {
                serializedName: "x-ms-blob-public-access",
                type: {
                    name: "String"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerSetAccessPolicyHeaders = {
    serializedName: "container-setaccesspolicy-headers",
    type: {
        name: "Composite",
        className: "ContainerSetAccessPolicyHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerAcquireLeaseHeaders = {
    serializedName: "container-acquirelease-headers",
    type: {
        name: "Composite",
        className: "ContainerAcquireLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseId: {
                serializedName: "x-ms-lease-id",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerReleaseLeaseHeaders = {
    serializedName: "container-releaselease-headers",
    type: {
        name: "Composite",
        className: "ContainerReleaseLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerRenewLeaseHeaders = {
    serializedName: "container-renewlease-headers",
    type: {
        name: "Composite",
        className: "ContainerRenewLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseId: {
                serializedName: "x-ms-lease-id",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerBreakLeaseHeaders = {
    serializedName: "container-breaklease-headers",
    type: {
        name: "Composite",
        className: "ContainerBreakLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseTime: {
                serializedName: "x-ms-lease-time",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerChangeLeaseHeaders = {
    serializedName: "container-changelease-headers",
    type: {
        name: "Composite",
        className: "ContainerChangeLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseId: {
                serializedName: "x-ms-lease-id",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerListBlobFlatSegmentHeaders = {
    serializedName: "container-listblobflatsegment-headers",
    type: {
        name: "Composite",
        className: "ContainerListBlobFlatSegmentHeaders",
        modelProperties: {
            contentType: {
                serializedName: "content-type",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerListBlobHierarchySegmentHeaders = {
    serializedName: "container-listblobhierarchysegment-headers",
    type: {
        name: "Composite",
        className: "ContainerListBlobHierarchySegmentHeaders",
        modelProperties: {
            contentType: {
                serializedName: "content-type",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var ContainerGetAccountInfoHeaders = {
    serializedName: "container-getaccountinfo-headers",
    type: {
        name: "Composite",
        className: "ContainerGetAccountInfoHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            skuName: {
                serializedName: "x-ms-sku-name",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "Standard_LRS",
                        "Standard_GRS",
                        "Standard_RAGRS",
                        "Standard_ZRS",
                        "Premium_LRS"
                    ]
                }
            },
            accountKind: {
                serializedName: "x-ms-account-kind",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "Storage",
                        "BlobStorage",
                        "StorageV2"
                    ]
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobDownloadHeaders = {
    serializedName: "blob-download-headers",
    type: {
        name: "Composite",
        className: "BlobDownloadHeaders",
        modelProperties: {
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            metadata: {
                serializedName: "x-ms-meta",
                type: {
                    name: "Dictionary",
                    value: {
                        type: {
                            name: "String"
                        }
                    }
                },
                headerCollectionPrefix: "x-ms-meta-"
            },
            contentLength: {
                serializedName: "content-length",
                type: {
                    name: "Number"
                }
            },
            contentType: {
                serializedName: "content-type",
                type: {
                    name: "String"
                }
            },
            contentRange: {
                serializedName: "content-range",
                type: {
                    name: "String"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            contentEncoding: {
                serializedName: "content-encoding",
                type: {
                    name: "String"
                }
            },
            cacheControl: {
                serializedName: "cache-control",
                type: {
                    name: "String"
                }
            },
            contentDisposition: {
                serializedName: "content-disposition",
                type: {
                    name: "String"
                }
            },
            contentLanguage: {
                serializedName: "content-language",
                type: {
                    name: "String"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            blobType: {
                serializedName: "x-ms-blob-type",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "BlockBlob",
                        "PageBlob",
                        "AppendBlob"
                    ]
                }
            },
            copyCompletionTime: {
                serializedName: "x-ms-copy-completion-time",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            copyStatusDescription: {
                serializedName: "x-ms-copy-status-description",
                type: {
                    name: "String"
                }
            },
            copyId: {
                serializedName: "x-ms-copy-id",
                type: {
                    name: "String"
                }
            },
            copyProgress: {
                serializedName: "x-ms-copy-progress",
                type: {
                    name: "String"
                }
            },
            copySource: {
                serializedName: "x-ms-copy-source",
                type: {
                    name: "String"
                }
            },
            copyStatus: {
                serializedName: "x-ms-copy-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "pending",
                        "success",
                        "aborted",
                        "failed"
                    ]
                }
            },
            leaseDuration: {
                serializedName: "x-ms-lease-duration",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "infinite",
                        "fixed"
                    ]
                }
            },
            leaseState: {
                serializedName: "x-ms-lease-state",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "available",
                        "leased",
                        "expired",
                        "breaking",
                        "broken"
                    ]
                }
            },
            leaseStatus: {
                serializedName: "x-ms-lease-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "locked",
                        "unlocked"
                    ]
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            acceptRanges: {
                serializedName: "accept-ranges",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobCommittedBlockCount: {
                serializedName: "x-ms-blob-committed-block-count",
                type: {
                    name: "Number"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            blobContentMD5: {
                serializedName: "x-ms-blob-content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            contentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobGetPropertiesHeaders = {
    serializedName: "blob-getproperties-headers",
    type: {
        name: "Composite",
        className: "BlobGetPropertiesHeaders",
        modelProperties: {
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            creationTime: {
                serializedName: "x-ms-creation-time",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            metadata: {
                serializedName: "x-ms-meta",
                type: {
                    name: "Dictionary",
                    value: {
                        type: {
                            name: "String"
                        }
                    }
                },
                headerCollectionPrefix: "x-ms-meta-"
            },
            blobType: {
                serializedName: "x-ms-blob-type",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "BlockBlob",
                        "PageBlob",
                        "AppendBlob"
                    ]
                }
            },
            copyCompletionTime: {
                serializedName: "x-ms-copy-completion-time",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            copyStatusDescription: {
                serializedName: "x-ms-copy-status-description",
                type: {
                    name: "String"
                }
            },
            copyId: {
                serializedName: "x-ms-copy-id",
                type: {
                    name: "String"
                }
            },
            copyProgress: {
                serializedName: "x-ms-copy-progress",
                type: {
                    name: "String"
                }
            },
            copySource: {
                serializedName: "x-ms-copy-source",
                type: {
                    name: "String"
                }
            },
            copyStatus: {
                serializedName: "x-ms-copy-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "pending",
                        "success",
                        "aborted",
                        "failed"
                    ]
                }
            },
            isIncrementalCopy: {
                serializedName: "x-ms-incremental-copy",
                type: {
                    name: "Boolean"
                }
            },
            destinationSnapshot: {
                serializedName: "x-ms-copy-destination-snapshot",
                type: {
                    name: "String"
                }
            },
            leaseDuration: {
                serializedName: "x-ms-lease-duration",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "infinite",
                        "fixed"
                    ]
                }
            },
            leaseState: {
                serializedName: "x-ms-lease-state",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "available",
                        "leased",
                        "expired",
                        "breaking",
                        "broken"
                    ]
                }
            },
            leaseStatus: {
                serializedName: "x-ms-lease-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "locked",
                        "unlocked"
                    ]
                }
            },
            contentLength: {
                serializedName: "content-length",
                type: {
                    name: "Number"
                }
            },
            contentType: {
                serializedName: "content-type",
                type: {
                    name: "String"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            contentEncoding: {
                serializedName: "content-encoding",
                type: {
                    name: "String"
                }
            },
            contentDisposition: {
                serializedName: "content-disposition",
                type: {
                    name: "String"
                }
            },
            contentLanguage: {
                serializedName: "content-language",
                type: {
                    name: "String"
                }
            },
            cacheControl: {
                serializedName: "cache-control",
                type: {
                    name: "String"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            acceptRanges: {
                serializedName: "accept-ranges",
                type: {
                    name: "String"
                }
            },
            blobCommittedBlockCount: {
                serializedName: "x-ms-blob-committed-block-count",
                type: {
                    name: "Number"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            accessTier: {
                serializedName: "x-ms-access-tier",
                type: {
                    name: "String"
                }
            },
            accessTierInferred: {
                serializedName: "x-ms-access-tier-inferred",
                type: {
                    name: "Boolean"
                }
            },
            archiveStatus: {
                serializedName: "x-ms-archive-status",
                type: {
                    name: "String"
                }
            },
            accessTierChangeTime: {
                serializedName: "x-ms-access-tier-change-time",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobDeleteHeaders = {
    serializedName: "blob-delete-headers",
    type: {
        name: "Composite",
        className: "BlobDeleteHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobSetAccessControlHeaders = {
    serializedName: "blob-setaccesscontrol-headers",
    type: {
        name: "Composite",
        className: "BlobSetAccessControlHeaders",
        modelProperties: {
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobGetAccessControlHeaders = {
    serializedName: "blob-getaccesscontrol-headers",
    type: {
        name: "Composite",
        className: "BlobGetAccessControlHeaders",
        modelProperties: {
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            xMsOwner: {
                serializedName: "x-ms-owner",
                type: {
                    name: "String"
                }
            },
            xMsGroup: {
                serializedName: "x-ms-group",
                type: {
                    name: "String"
                }
            },
            xMsPermissions: {
                serializedName: "x-ms-permissions",
                type: {
                    name: "String"
                }
            },
            xMsAcl: {
                serializedName: "x-ms-acl",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobRenameHeaders = {
    serializedName: "blob-rename-headers",
    type: {
        name: "Composite",
        className: "BlobRenameHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            contentLength: {
                serializedName: "content-length",
                type: {
                    name: "Number"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            }
        }
    }
};
var PageBlobCreateHeaders = {
    serializedName: "pageblob-create-headers",
    type: {
        name: "Composite",
        className: "PageBlobCreateHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var AppendBlobCreateHeaders = {
    serializedName: "appendblob-create-headers",
    type: {
        name: "Composite",
        className: "AppendBlobCreateHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlockBlobUploadHeaders = {
    serializedName: "blockblob-upload-headers",
    type: {
        name: "Composite",
        className: "BlockBlobUploadHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobUndeleteHeaders = {
    serializedName: "blob-undelete-headers",
    type: {
        name: "Composite",
        className: "BlobUndeleteHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobSetHTTPHeadersHeaders = {
    serializedName: "blob-sethttpheaders-headers",
    type: {
        name: "Composite",
        className: "BlobSetHTTPHeadersHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobSetMetadataHeaders = {
    serializedName: "blob-setmetadata-headers",
    type: {
        name: "Composite",
        className: "BlobSetMetadataHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobAcquireLeaseHeaders = {
    serializedName: "blob-acquirelease-headers",
    type: {
        name: "Composite",
        className: "BlobAcquireLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseId: {
                serializedName: "x-ms-lease-id",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobReleaseLeaseHeaders = {
    serializedName: "blob-releaselease-headers",
    type: {
        name: "Composite",
        className: "BlobReleaseLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobRenewLeaseHeaders = {
    serializedName: "blob-renewlease-headers",
    type: {
        name: "Composite",
        className: "BlobRenewLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseId: {
                serializedName: "x-ms-lease-id",
                type: {
                    name: "String"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobChangeLeaseHeaders = {
    serializedName: "blob-changelease-headers",
    type: {
        name: "Composite",
        className: "BlobChangeLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            leaseId: {
                serializedName: "x-ms-lease-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobBreakLeaseHeaders = {
    serializedName: "blob-breaklease-headers",
    type: {
        name: "Composite",
        className: "BlobBreakLeaseHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            leaseTime: {
                serializedName: "x-ms-lease-time",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobCreateSnapshotHeaders = {
    serializedName: "blob-createsnapshot-headers",
    type: {
        name: "Composite",
        className: "BlobCreateSnapshotHeaders",
        modelProperties: {
            snapshot: {
                serializedName: "x-ms-snapshot",
                type: {
                    name: "String"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobStartCopyFromURLHeaders = {
    serializedName: "blob-startcopyfromurl-headers",
    type: {
        name: "Composite",
        className: "BlobStartCopyFromURLHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            copyId: {
                serializedName: "x-ms-copy-id",
                type: {
                    name: "String"
                }
            },
            copyStatus: {
                serializedName: "x-ms-copy-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "pending",
                        "success",
                        "aborted",
                        "failed"
                    ]
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobCopyFromURLHeaders = {
    serializedName: "blob-copyfromurl-headers",
    type: {
        name: "Composite",
        className: "BlobCopyFromURLHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            copyId: {
                serializedName: "x-ms-copy-id",
                type: {
                    name: "String"
                }
            },
            copyStatus: {
                serializedName: "x-ms-copy-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "success"
                    ]
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobAbortCopyFromURLHeaders = {
    serializedName: "blob-abortcopyfromurl-headers",
    type: {
        name: "Composite",
        className: "BlobAbortCopyFromURLHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobSetTierHeaders = {
    serializedName: "blob-settier-headers",
    type: {
        name: "Composite",
        className: "BlobSetTierHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlobGetAccountInfoHeaders = {
    serializedName: "blob-getaccountinfo-headers",
    type: {
        name: "Composite",
        className: "BlobGetAccountInfoHeaders",
        modelProperties: {
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            skuName: {
                serializedName: "x-ms-sku-name",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "Standard_LRS",
                        "Standard_GRS",
                        "Standard_RAGRS",
                        "Standard_ZRS",
                        "Premium_LRS"
                    ]
                }
            },
            accountKind: {
                serializedName: "x-ms-account-kind",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "Storage",
                        "BlobStorage",
                        "StorageV2"
                    ]
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlockBlobStageBlockHeaders = {
    serializedName: "blockblob-stageblock-headers",
    type: {
        name: "Composite",
        className: "BlockBlobStageBlockHeaders",
        modelProperties: {
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlockBlobStageBlockFromURLHeaders = {
    serializedName: "blockblob-stageblockfromurl-headers",
    type: {
        name: "Composite",
        className: "BlockBlobStageBlockFromURLHeaders",
        modelProperties: {
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlockBlobCommitBlockListHeaders = {
    serializedName: "blockblob-commitblocklist-headers",
    type: {
        name: "Composite",
        className: "BlockBlobCommitBlockListHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var BlockBlobGetBlockListHeaders = {
    serializedName: "blockblob-getblocklist-headers",
    type: {
        name: "Composite",
        className: "BlockBlobGetBlockListHeaders",
        modelProperties: {
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            contentType: {
                serializedName: "content-type",
                type: {
                    name: "String"
                }
            },
            blobContentLength: {
                serializedName: "x-ms-blob-content-length",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobUploadPagesHeaders = {
    serializedName: "pageblob-uploadpages-headers",
    type: {
        name: "Composite",
        className: "PageBlobUploadPagesHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobClearPagesHeaders = {
    serializedName: "pageblob-clearpages-headers",
    type: {
        name: "Composite",
        className: "PageBlobClearPagesHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobUploadPagesFromURLHeaders = {
    serializedName: "pageblob-uploadpagesfromurl-headers",
    type: {
        name: "Composite",
        className: "PageBlobUploadPagesFromURLHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobGetPageRangesHeaders = {
    serializedName: "pageblob-getpageranges-headers",
    type: {
        name: "Composite",
        className: "PageBlobGetPageRangesHeaders",
        modelProperties: {
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            blobContentLength: {
                serializedName: "x-ms-blob-content-length",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobGetPageRangesDiffHeaders = {
    serializedName: "pageblob-getpagerangesdiff-headers",
    type: {
        name: "Composite",
        className: "PageBlobGetPageRangesDiffHeaders",
        modelProperties: {
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            blobContentLength: {
                serializedName: "x-ms-blob-content-length",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobResizeHeaders = {
    serializedName: "pageblob-resize-headers",
    type: {
        name: "Composite",
        className: "PageBlobResizeHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobUpdateSequenceNumberHeaders = {
    serializedName: "pageblob-updatesequencenumber-headers",
    type: {
        name: "Composite",
        className: "PageBlobUpdateSequenceNumberHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobSequenceNumber: {
                serializedName: "x-ms-blob-sequence-number",
                type: {
                    name: "Number"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var PageBlobCopyIncrementalHeaders = {
    serializedName: "pageblob-copyincremental-headers",
    type: {
        name: "Composite",
        className: "PageBlobCopyIncrementalHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            copyId: {
                serializedName: "x-ms-copy-id",
                type: {
                    name: "String"
                }
            },
            copyStatus: {
                serializedName: "x-ms-copy-status",
                type: {
                    name: "Enum",
                    allowedValues: [
                        "pending",
                        "success",
                        "aborted",
                        "failed"
                    ]
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var AppendBlobAppendBlockHeaders = {
    serializedName: "appendblob-appendblock-headers",
    type: {
        name: "Composite",
        className: "AppendBlobAppendBlockHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            clientRequestId: {
                serializedName: "x-ms-client-request-id",
                type: {
                    name: "String"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobAppendOffset: {
                serializedName: "x-ms-blob-append-offset",
                type: {
                    name: "String"
                }
            },
            blobCommittedBlockCount: {
                serializedName: "x-ms-blob-committed-block-count",
                type: {
                    name: "Number"
                }
            },
            isServerEncrypted: {
                serializedName: "x-ms-request-server-encrypted",
                type: {
                    name: "Boolean"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};
var AppendBlobAppendBlockFromUrlHeaders = {
    serializedName: "appendblob-appendblockfromurl-headers",
    type: {
        name: "Composite",
        className: "AppendBlobAppendBlockFromUrlHeaders",
        modelProperties: {
            eTag: {
                serializedName: "etag",
                type: {
                    name: "String"
                }
            },
            lastModified: {
                serializedName: "last-modified",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            contentMD5: {
                serializedName: "content-md5",
                type: {
                    name: "ByteArray"
                }
            },
            xMsContentCrc64: {
                serializedName: "x-ms-content-crc64",
                type: {
                    name: "ByteArray"
                }
            },
            requestId: {
                serializedName: "x-ms-request-id",
                type: {
                    name: "String"
                }
            },
            version: {
                serializedName: "x-ms-version",
                type: {
                    name: "String"
                }
            },
            date: {
                serializedName: "date",
                type: {
                    name: "DateTimeRfc1123"
                }
            },
            blobAppendOffset: {
                serializedName: "x-ms-blob-append-offset",
                type: {
                    name: "String"
                }
            },
            blobCommittedBlockCount: {
                serializedName: "x-ms-blob-committed-block-count",
                type: {
                    name: "Number"
                }
            },
            encryptionKeySha256: {
                serializedName: "x-ms-encryption-key-sha256",
                type: {
                    name: "String"
                }
            },
            errorCode: {
                serializedName: "x-ms-error-code",
                type: {
                    name: "String"
                }
            }
        }
    }
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var Mappers = /*#__PURE__*/Object.freeze({
    ContainerItem: ContainerItem,
    ContainerProperties: ContainerProperties,
    CorsRule: CorsRule,
    GeoReplication: GeoReplication,
    KeyInfo: KeyInfo,
    ListContainersSegmentResponse: ListContainersSegmentResponse,
    Logging: Logging,
    Metrics: Metrics,
    RetentionPolicy: RetentionPolicy,
    ServiceGetAccountInfoHeaders: ServiceGetAccountInfoHeaders,
    ServiceGetPropertiesHeaders: ServiceGetPropertiesHeaders,
    ServiceGetStatisticsHeaders: ServiceGetStatisticsHeaders,
    ServiceGetUserDelegationKeyHeaders: ServiceGetUserDelegationKeyHeaders,
    ServiceListContainersSegmentHeaders: ServiceListContainersSegmentHeaders,
    ServiceSetPropertiesHeaders: ServiceSetPropertiesHeaders,
    ServiceSubmitBatchHeaders: ServiceSubmitBatchHeaders,
    StaticWebsite: StaticWebsite,
    StorageError: StorageError,
    StorageServiceProperties: StorageServiceProperties,
    StorageServiceStats: StorageServiceStats,
    UserDelegationKey: UserDelegationKey
});

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
var access = {
    parameterPath: [
        "options",
        "access"
    ],
    mapper: {
        serializedName: "x-ms-blob-public-access",
        type: {
            name: "String"
        }
    }
};
var action0 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-lease-action",
        defaultValue: 'acquire',
        type: {
            name: "String"
        }
    }
};
var action1 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-lease-action",
        defaultValue: 'release',
        type: {
            name: "String"
        }
    }
};
var action2 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-lease-action",
        defaultValue: 'renew',
        type: {
            name: "String"
        }
    }
};
var action3 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-lease-action",
        defaultValue: 'break',
        type: {
            name: "String"
        }
    }
};
var action4 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-lease-action",
        defaultValue: 'change',
        type: {
            name: "String"
        }
    }
};
var action5 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "action",
        defaultValue: 'setAccessControl',
        type: {
            name: "String"
        }
    }
};
var action6 = {
    parameterPath: "action",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "action",
        defaultValue: 'getAccessControl',
        type: {
            name: "String"
        }
    }
};
var appendPosition = {
    parameterPath: [
        "options",
        "appendPositionAccessConditions",
        "appendPosition"
    ],
    mapper: {
        serializedName: "x-ms-blob-condition-appendpos",
        type: {
            name: "Number"
        }
    }
};
var blobCacheControl = {
    parameterPath: [
        "options",
        "blobHTTPHeaders",
        "blobCacheControl"
    ],
    mapper: {
        serializedName: "x-ms-blob-cache-control",
        type: {
            name: "String"
        }
    }
};
var blobContentDisposition = {
    parameterPath: [
        "options",
        "blobHTTPHeaders",
        "blobContentDisposition"
    ],
    mapper: {
        serializedName: "x-ms-blob-content-disposition",
        type: {
            name: "String"
        }
    }
};
var blobContentEncoding = {
    parameterPath: [
        "options",
        "blobHTTPHeaders",
        "blobContentEncoding"
    ],
    mapper: {
        serializedName: "x-ms-blob-content-encoding",
        type: {
            name: "String"
        }
    }
};
var blobContentLanguage = {
    parameterPath: [
        "options",
        "blobHTTPHeaders",
        "blobContentLanguage"
    ],
    mapper: {
        serializedName: "x-ms-blob-content-language",
        type: {
            name: "String"
        }
    }
};
var blobContentLength = {
    parameterPath: "blobContentLength",
    mapper: {
        required: true,
        serializedName: "x-ms-blob-content-length",
        type: {
            name: "Number"
        }
    }
};
var blobContentMD5 = {
    parameterPath: [
        "options",
        "blobHTTPHeaders",
        "blobContentMD5"
    ],
    mapper: {
        serializedName: "x-ms-blob-content-md5",
        type: {
            name: "ByteArray"
        }
    }
};
var blobContentType = {
    parameterPath: [
        "options",
        "blobHTTPHeaders",
        "blobContentType"
    ],
    mapper: {
        serializedName: "x-ms-blob-content-type",
        type: {
            name: "String"
        }
    }
};
var blobSequenceNumber = {
    parameterPath: [
        "options",
        "blobSequenceNumber"
    ],
    mapper: {
        serializedName: "x-ms-blob-sequence-number",
        defaultValue: 0,
        type: {
            name: "Number"
        }
    }
};
var blobType0 = {
    parameterPath: "blobType",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-blob-type",
        defaultValue: 'PageBlob',
        type: {
            name: "String"
        }
    }
};
var blobType1 = {
    parameterPath: "blobType",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-blob-type",
        defaultValue: 'AppendBlob',
        type: {
            name: "String"
        }
    }
};
var blobType2 = {
    parameterPath: "blobType",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-blob-type",
        defaultValue: 'BlockBlob',
        type: {
            name: "String"
        }
    }
};
var blockId = {
    parameterPath: "blockId",
    mapper: {
        required: true,
        serializedName: "blockid",
        type: {
            name: "String"
        }
    }
};
var breakPeriod = {
    parameterPath: [
        "options",
        "breakPeriod"
    ],
    mapper: {
        serializedName: "x-ms-lease-break-period",
        type: {
            name: "Number"
        }
    }
};
var cacheControl = {
    parameterPath: [
        "options",
        "directoryHttpHeaders",
        "cacheControl"
    ],
    mapper: {
        serializedName: "x-ms-cache-control",
        type: {
            name: "String"
        }
    }
};
var comp0 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'properties',
        type: {
            name: "String"
        }
    }
};
var comp1 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'stats',
        type: {
            name: "String"
        }
    }
};
var comp10 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'copy',
        type: {
            name: "String"
        }
    }
};
var comp11 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'tier',
        type: {
            name: "String"
        }
    }
};
var comp12 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'page',
        type: {
            name: "String"
        }
    }
};
var comp13 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'pagelist',
        type: {
            name: "String"
        }
    }
};
var comp14 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'incrementalcopy',
        type: {
            name: "String"
        }
    }
};
var comp15 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'appendblock',
        type: {
            name: "String"
        }
    }
};
var comp16 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'block',
        type: {
            name: "String"
        }
    }
};
var comp17 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'blocklist',
        type: {
            name: "String"
        }
    }
};
var comp2 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'list',
        type: {
            name: "String"
        }
    }
};
var comp3 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'userdelegationkey',
        type: {
            name: "String"
        }
    }
};
var comp4 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'batch',
        type: {
            name: "String"
        }
    }
};
var comp5 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'metadata',
        type: {
            name: "String"
        }
    }
};
var comp6 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'acl',
        type: {
            name: "String"
        }
    }
};
var comp7 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'lease',
        type: {
            name: "String"
        }
    }
};
var comp8 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'undelete',
        type: {
            name: "String"
        }
    }
};
var comp9 = {
    parameterPath: "comp",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "comp",
        defaultValue: 'snapshot',
        type: {
            name: "String"
        }
    }
};
var contentDisposition = {
    parameterPath: [
        "options",
        "directoryHttpHeaders",
        "contentDisposition"
    ],
    mapper: {
        serializedName: "x-ms-content-disposition",
        type: {
            name: "String"
        }
    }
};
var contentEncoding = {
    parameterPath: [
        "options",
        "directoryHttpHeaders",
        "contentEncoding"
    ],
    mapper: {
        serializedName: "x-ms-content-encoding",
        type: {
            name: "String"
        }
    }
};
var contentLanguage = {
    parameterPath: [
        "options",
        "directoryHttpHeaders",
        "contentLanguage"
    ],
    mapper: {
        serializedName: "x-ms-content-language",
        type: {
            name: "String"
        }
    }
};
var contentLength = {
    parameterPath: "contentLength",
    mapper: {
        required: true,
        serializedName: "Content-Length",
        type: {
            name: "Number"
        }
    }
};
var contentType = {
    parameterPath: [
        "options",
        "directoryHttpHeaders",
        "contentType"
    ],
    mapper: {
        serializedName: "x-ms-content-type",
        type: {
            name: "String"
        }
    }
};
var copyActionAbortConstant = {
    parameterPath: "copyActionAbortConstant",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-copy-action",
        defaultValue: 'abort',
        type: {
            name: "String"
        }
    }
};
var copyId = {
    parameterPath: "copyId",
    mapper: {
        required: true,
        serializedName: "copyid",
        type: {
            name: "String"
        }
    }
};
var copySource = {
    parameterPath: "copySource",
    mapper: {
        required: true,
        serializedName: "x-ms-copy-source",
        type: {
            name: "String"
        }
    }
};
var deleteSnapshots = {
    parameterPath: [
        "options",
        "deleteSnapshots"
    ],
    mapper: {
        serializedName: "x-ms-delete-snapshots",
        type: {
            name: "Enum",
            allowedValues: [
                "include",
                "only"
            ]
        }
    }
};
var delimiter = {
    parameterPath: "delimiter",
    mapper: {
        required: true,
        serializedName: "delimiter",
        type: {
            name: "String"
        }
    }
};
var directoryProperties = {
    parameterPath: [
        "options",
        "directoryProperties"
    ],
    mapper: {
        serializedName: "x-ms-properties",
        type: {
            name: "String"
        }
    }
};
var duration = {
    parameterPath: [
        "options",
        "duration"
    ],
    mapper: {
        serializedName: "x-ms-lease-duration",
        type: {
            name: "Number"
        }
    }
};
var encryptionAlgorithm = {
    parameterPath: [
        "options",
        "cpkInfo",
        "encryptionAlgorithm"
    ],
    mapper: {
        serializedName: "x-ms-encryption-algorithm",
        type: {
            name: "Enum",
            allowedValues: [
                "AES256"
            ]
        }
    }
};
var encryptionKey = {
    parameterPath: [
        "options",
        "cpkInfo",
        "encryptionKey"
    ],
    mapper: {
        serializedName: "x-ms-encryption-key",
        type: {
            name: "String"
        }
    }
};
var encryptionKeySha256 = {
    parameterPath: [
        "options",
        "cpkInfo",
        "encryptionKeySha256"
    ],
    mapper: {
        serializedName: "x-ms-encryption-key-sha256",
        type: {
            name: "String"
        }
    }
};
var group = {
    parameterPath: [
        "options",
        "group"
    ],
    mapper: {
        serializedName: "x-ms-group",
        type: {
            name: "String"
        }
    }
};
var ifMatch = {
    parameterPath: [
        "options",
        "modifiedAccessConditions",
        "ifMatch"
    ],
    mapper: {
        serializedName: "If-Match",
        type: {
            name: "String"
        }
    }
};
var ifModifiedSince = {
    parameterPath: [
        "options",
        "modifiedAccessConditions",
        "ifModifiedSince"
    ],
    mapper: {
        serializedName: "If-Modified-Since",
        type: {
            name: "DateTimeRfc1123"
        }
    }
};
var ifNoneMatch = {
    parameterPath: [
        "options",
        "modifiedAccessConditions",
        "ifNoneMatch"
    ],
    mapper: {
        serializedName: "If-None-Match",
        type: {
            name: "String"
        }
    }
};
var ifSequenceNumberEqualTo = {
    parameterPath: [
        "options",
        "sequenceNumberAccessConditions",
        "ifSequenceNumberEqualTo"
    ],
    mapper: {
        serializedName: "x-ms-if-sequence-number-eq",
        type: {
            name: "Number"
        }
    }
};
var ifSequenceNumberLessThan = {
    parameterPath: [
        "options",
        "sequenceNumberAccessConditions",
        "ifSequenceNumberLessThan"
    ],
    mapper: {
        serializedName: "x-ms-if-sequence-number-lt",
        type: {
            name: "Number"
        }
    }
};
var ifSequenceNumberLessThanOrEqualTo = {
    parameterPath: [
        "options",
        "sequenceNumberAccessConditions",
        "ifSequenceNumberLessThanOrEqualTo"
    ],
    mapper: {
        serializedName: "x-ms-if-sequence-number-le",
        type: {
            name: "Number"
        }
    }
};
var ifUnmodifiedSince = {
    parameterPath: [
        "options",
        "modifiedAccessConditions",
        "ifUnmodifiedSince"
    ],
    mapper: {
        serializedName: "If-Unmodified-Since",
        type: {
            name: "DateTimeRfc1123"
        }
    }
};
var include0 = {
    parameterPath: [
        "options",
        "include"
    ],
    mapper: {
        serializedName: "include",
        type: {
            name: "Enum",
            allowedValues: [
                "metadata"
            ]
        }
    }
};
var include1 = {
    parameterPath: [
        "options",
        "include"
    ],
    mapper: {
        serializedName: "include",
        type: {
            name: "Sequence",
            element: {
                type: {
                    name: "Enum",
                    allowedValues: [
                        "copy",
                        "deleted",
                        "metadata",
                        "snapshots",
                        "uncommittedblobs"
                    ]
                }
            }
        }
    },
    collectionFormat: msRest.QueryCollectionFormat.Csv
};
var leaseId0 = {
    parameterPath: [
        "options",
        "leaseAccessConditions",
        "leaseId"
    ],
    mapper: {
        serializedName: "x-ms-lease-id",
        type: {
            name: "String"
        }
    }
};
var leaseId1 = {
    parameterPath: "leaseId",
    mapper: {
        required: true,
        serializedName: "x-ms-lease-id",
        type: {
            name: "String"
        }
    }
};
var listType = {
    parameterPath: "listType",
    mapper: {
        required: true,
        serializedName: "blocklisttype",
        defaultValue: 'committed',
        type: {
            name: "Enum",
            allowedValues: [
                "committed",
                "uncommitted",
                "all"
            ]
        }
    }
};
var marker0 = {
    parameterPath: [
        "options",
        "marker"
    ],
    mapper: {
        serializedName: "marker",
        type: {
            name: "String"
        }
    }
};
var maxresults = {
    parameterPath: [
        "options",
        "maxresults"
    ],
    mapper: {
        serializedName: "maxresults",
        constraints: {
            InclusiveMinimum: 1
        },
        type: {
            name: "Number"
        }
    }
};
var maxSize = {
    parameterPath: [
        "options",
        "appendPositionAccessConditions",
        "maxSize"
    ],
    mapper: {
        serializedName: "x-ms-blob-condition-maxsize",
        type: {
            name: "Number"
        }
    }
};
var metadata = {
    parameterPath: [
        "options",
        "metadata"
    ],
    mapper: {
        serializedName: "x-ms-meta",
        type: {
            name: "Dictionary",
            value: {
                type: {
                    name: "String"
                }
            }
        },
        headerCollectionPrefix: "x-ms-meta-"
    }
};
var multipartContentType = {
    parameterPath: "multipartContentType",
    mapper: {
        required: true,
        serializedName: "Content-Type",
        type: {
            name: "String"
        }
    }
};
var owner = {
    parameterPath: [
        "options",
        "owner"
    ],
    mapper: {
        serializedName: "x-ms-owner",
        type: {
            name: "String"
        }
    }
};
var pageWrite0 = {
    parameterPath: "pageWrite",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-page-write",
        defaultValue: 'update',
        type: {
            name: "String"
        }
    }
};
var pageWrite1 = {
    parameterPath: "pageWrite",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-page-write",
        defaultValue: 'clear',
        type: {
            name: "String"
        }
    }
};
var pathRenameMode = {
    parameterPath: "pathRenameMode",
    mapper: {
        serializedName: "mode",
        type: {
            name: "Enum",
            allowedValues: [
                "legacy",
                "posix"
            ]
        }
    }
};
var posixAcl = {
    parameterPath: [
        "options",
        "posixAcl"
    ],
    mapper: {
        serializedName: "x-ms-acl",
        type: {
            name: "String"
        }
    }
};
var posixPermissions = {
    parameterPath: [
        "options",
        "posixPermissions"
    ],
    mapper: {
        serializedName: "x-ms-permissions",
        type: {
            name: "String"
        }
    }
};
var posixUmask = {
    parameterPath: [
        "options",
        "posixUmask"
    ],
    mapper: {
        serializedName: "x-ms-umask",
        type: {
            name: "String"
        }
    }
};
var prefix = {
    parameterPath: [
        "options",
        "prefix"
    ],
    mapper: {
        serializedName: "prefix",
        type: {
            name: "String"
        }
    }
};
var prevsnapshot = {
    parameterPath: [
        "options",
        "prevsnapshot"
    ],
    mapper: {
        serializedName: "prevsnapshot",
        type: {
            name: "String"
        }
    }
};
var proposedLeaseId0 = {
    parameterPath: [
        "options",
        "proposedLeaseId"
    ],
    mapper: {
        serializedName: "x-ms-proposed-lease-id",
        type: {
            name: "String"
        }
    }
};
var proposedLeaseId1 = {
    parameterPath: "proposedLeaseId",
    mapper: {
        required: true,
        serializedName: "x-ms-proposed-lease-id",
        type: {
            name: "String"
        }
    }
};
var range0 = {
    parameterPath: [
        "options",
        "range"
    ],
    mapper: {
        serializedName: "x-ms-range",
        type: {
            name: "String"
        }
    }
};
var range1 = {
    parameterPath: "range",
    mapper: {
        required: true,
        serializedName: "x-ms-range",
        type: {
            name: "String"
        }
    }
};
var rangeGetContentCRC64 = {
    parameterPath: [
        "options",
        "rangeGetContentCRC64"
    ],
    mapper: {
        serializedName: "x-ms-range-get-content-crc64",
        type: {
            name: "Boolean"
        }
    }
};
var rangeGetContentMD5 = {
    parameterPath: [
        "options",
        "rangeGetContentMD5"
    ],
    mapper: {
        serializedName: "x-ms-range-get-content-md5",
        type: {
            name: "Boolean"
        }
    }
};
var rehydratePriority = {
    parameterPath: [
        "options",
        "rehydratePriority"
    ],
    mapper: {
        serializedName: "x-ms-rehydrate-priority",
        type: {
            name: "String"
        }
    }
};
var renameSource = {
    parameterPath: "renameSource",
    mapper: {
        required: true,
        serializedName: "x-ms-rename-source",
        type: {
            name: "String"
        }
    }
};
var requestId = {
    parameterPath: [
        "options",
        "requestId"
    ],
    mapper: {
        serializedName: "x-ms-client-request-id",
        type: {
            name: "String"
        }
    }
};
var restype0 = {
    parameterPath: "restype",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "restype",
        defaultValue: 'service',
        type: {
            name: "String"
        }
    }
};
var restype1 = {
    parameterPath: "restype",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "restype",
        defaultValue: 'account',
        type: {
            name: "String"
        }
    }
};
var restype2 = {
    parameterPath: "restype",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "restype",
        defaultValue: 'container',
        type: {
            name: "String"
        }
    }
};
var sequenceNumberAction = {
    parameterPath: "sequenceNumberAction",
    mapper: {
        required: true,
        serializedName: "x-ms-sequence-number-action",
        type: {
            name: "Enum",
            allowedValues: [
                "max",
                "update",
                "increment"
            ]
        }
    }
};
var snapshot = {
    parameterPath: [
        "options",
        "snapshot"
    ],
    mapper: {
        serializedName: "snapshot",
        type: {
            name: "String"
        }
    }
};
var sourceContentCrc64 = {
    parameterPath: [
        "options",
        "sourceContentCrc64"
    ],
    mapper: {
        serializedName: "x-ms-source-content-crc64",
        type: {
            name: "ByteArray"
        }
    }
};
var sourceContentMD5 = {
    parameterPath: [
        "options",
        "sourceContentMD5"
    ],
    mapper: {
        serializedName: "x-ms-source-content-md5",
        type: {
            name: "ByteArray"
        }
    }
};
var sourceIfMatch = {
    parameterPath: [
        "options",
        "sourceModifiedAccessConditions",
        "sourceIfMatch"
    ],
    mapper: {
        serializedName: "x-ms-source-if-match",
        type: {
            name: "String"
        }
    }
};
var sourceIfModifiedSince = {
    parameterPath: [
        "options",
        "sourceModifiedAccessConditions",
        "sourceIfModifiedSince"
    ],
    mapper: {
        serializedName: "x-ms-source-if-modified-since",
        type: {
            name: "DateTimeRfc1123"
        }
    }
};
var sourceIfNoneMatch = {
    parameterPath: [
        "options",
        "sourceModifiedAccessConditions",
        "sourceIfNoneMatch"
    ],
    mapper: {
        serializedName: "x-ms-source-if-none-match",
        type: {
            name: "String"
        }
    }
};
var sourceIfUnmodifiedSince = {
    parameterPath: [
        "options",
        "sourceModifiedAccessConditions",
        "sourceIfUnmodifiedSince"
    ],
    mapper: {
        serializedName: "x-ms-source-if-unmodified-since",
        type: {
            name: "DateTimeRfc1123"
        }
    }
};
var sourceLeaseId = {
    parameterPath: [
        "options",
        "sourceLeaseId"
    ],
    mapper: {
        serializedName: "x-ms-source-lease-id",
        type: {
            name: "String"
        }
    }
};
var sourceRange0 = {
    parameterPath: "sourceRange",
    mapper: {
        required: true,
        serializedName: "x-ms-source-range",
        type: {
            name: "String"
        }
    }
};
var sourceRange1 = {
    parameterPath: [
        "options",
        "sourceRange"
    ],
    mapper: {
        serializedName: "x-ms-source-range",
        type: {
            name: "String"
        }
    }
};
var sourceUrl = {
    parameterPath: "sourceUrl",
    mapper: {
        required: true,
        serializedName: "x-ms-copy-source",
        type: {
            name: "String"
        }
    }
};
var tier0 = {
    parameterPath: [
        "options",
        "tier"
    ],
    mapper: {
        serializedName: "x-ms-access-tier",
        type: {
            name: "String"
        }
    }
};
var tier1 = {
    parameterPath: "tier",
    mapper: {
        required: true,
        serializedName: "x-ms-access-tier",
        type: {
            name: "String"
        }
    }
};
var timeout = {
    parameterPath: [
        "options",
        "timeout"
    ],
    mapper: {
        serializedName: "timeout",
        constraints: {
            InclusiveMinimum: 0
        },
        type: {
            name: "Number"
        }
    }
};
var transactionalContentCrc64 = {
    parameterPath: [
        "options",
        "transactionalContentCrc64"
    ],
    mapper: {
        serializedName: "x-ms-content-crc64",
        type: {
            name: "ByteArray"
        }
    }
};
var transactionalContentMD5 = {
    parameterPath: [
        "options",
        "transactionalContentMD5"
    ],
    mapper: {
        serializedName: "Content-MD5",
        type: {
            name: "ByteArray"
        }
    }
};
var upn = {
    parameterPath: [
        "options",
        "upn"
    ],
    mapper: {
        serializedName: "upn",
        type: {
            name: "Boolean"
        }
    }
};
var url = {
    parameterPath: "url",
    mapper: {
        required: true,
        serializedName: "url",
        defaultValue: '',
        type: {
            name: "String"
        }
    },
    skipEncoding: true
};
var version = {
    parameterPath: "version",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-version",
        defaultValue: '2019-02-02',
        type: {
            name: "String"
        }
    }
};
var xMsRequiresSync = {
    parameterPath: "xMsRequiresSync",
    mapper: {
        required: true,
        isConstant: true,
        serializedName: "x-ms-requires-sync",
        defaultValue: 'true',
        type: {
            name: "String"
        }
    }
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
/** Class representing a Service. */
var Service = /** @class */ (function () {
    /**
     * Create a Service.
     * @param {StorageClientContext} client Reference to the service client.
     */
    function Service(client) {
        this.client = client;
    }
    Service.prototype.setProperties = function (storageServiceProperties, options, callback) {
        return this.client.sendOperationRequest({
            storageServiceProperties: storageServiceProperties,
            options: options
        }, setPropertiesOperationSpec, callback);
    };
    Service.prototype.getProperties = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getPropertiesOperationSpec, callback);
    };
    Service.prototype.getStatistics = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getStatisticsOperationSpec, callback);
    };
    Service.prototype.listContainersSegment = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, listContainersSegmentOperationSpec, callback);
    };
    Service.prototype.getUserDelegationKey = function (keyInfo, options, callback) {
        return this.client.sendOperationRequest({
            keyInfo: keyInfo,
            options: options
        }, getUserDelegationKeyOperationSpec, callback);
    };
    Service.prototype.getAccountInfo = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getAccountInfoOperationSpec, callback);
    };
    Service.prototype.submitBatch = function (body, contentLength, multipartContentType, options, callback) {
        return this.client.sendOperationRequest({
            body: body,
            contentLength: contentLength,
            multipartContentType: multipartContentType,
            options: options
        }, submitBatchOperationSpec, callback);
    };
    return Service;
}());
// Operation Specifications
var serializer = new msRest.Serializer(Mappers, true);
var setPropertiesOperationSpec = {
    httpMethod: "PUT",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype0,
        comp0
    ],
    headerParameters: [
        version,
        requestId
    ],
    requestBody: {
        parameterPath: "storageServiceProperties",
        mapper: tslib_1.__assign({}, StorageServiceProperties, { required: true })
    },
    contentType: "application/xml; charset=utf-8",
    responses: {
        202: {
            headersMapper: ServiceSetPropertiesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};
var getPropertiesOperationSpec = {
    httpMethod: "GET",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype0,
        comp0
    ],
    headerParameters: [
        version,
        requestId
    ],
    responses: {
        200: {
            bodyMapper: StorageServiceProperties,
            headersMapper: ServiceGetPropertiesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};
var getStatisticsOperationSpec = {
    httpMethod: "GET",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype0,
        comp1
    ],
    headerParameters: [
        version,
        requestId
    ],
    responses: {
        200: {
            bodyMapper: StorageServiceStats,
            headersMapper: ServiceGetStatisticsHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};
var listContainersSegmentOperationSpec = {
    httpMethod: "GET",
    urlParameters: [
        url
    ],
    queryParameters: [
        prefix,
        marker0,
        maxresults,
        include0,
        timeout,
        comp2
    ],
    headerParameters: [
        version,
        requestId
    ],
    responses: {
        200: {
            bodyMapper: ListContainersSegmentResponse,
            headersMapper: ServiceListContainersSegmentHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};
var getUserDelegationKeyOperationSpec = {
    httpMethod: "POST",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype0,
        comp3
    ],
    headerParameters: [
        version,
        requestId
    ],
    requestBody: {
        parameterPath: "keyInfo",
        mapper: tslib_1.__assign({}, KeyInfo, { required: true })
    },
    contentType: "application/xml; charset=utf-8",
    responses: {
        200: {
            bodyMapper: UserDelegationKey,
            headersMapper: ServiceGetUserDelegationKeyHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};
var getAccountInfoOperationSpec = {
    httpMethod: "GET",
    urlParameters: [
        url
    ],
    queryParameters: [
        restype1,
        comp0
    ],
    headerParameters: [
        version
    ],
    responses: {
        200: {
            headersMapper: ServiceGetAccountInfoHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};
var submitBatchOperationSpec = {
    httpMethod: "POST",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp4
    ],
    headerParameters: [
        contentLength,
        multipartContentType,
        version,
        requestId
    ],
    requestBody: {
        parameterPath: "body",
        mapper: {
            required: true,
            serializedName: "body",
            type: {
                name: "Stream"
            }
        }
    },
    contentType: "application/xml; charset=utf-8",
    responses: {
        202: {
            bodyMapper: {
                serializedName: "parsedResponse",
                type: {
                    name: "Stream"
                }
            },
            headersMapper: ServiceSubmitBatchHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var Mappers$1 = /*#__PURE__*/Object.freeze({
    AccessPolicy: AccessPolicy,
    BlobFlatListSegment: BlobFlatListSegment,
    BlobHierarchyListSegment: BlobHierarchyListSegment,
    BlobItem: BlobItem,
    BlobMetadata: BlobMetadata,
    BlobPrefix: BlobPrefix,
    BlobProperties: BlobProperties,
    ContainerAcquireLeaseHeaders: ContainerAcquireLeaseHeaders,
    ContainerBreakLeaseHeaders: ContainerBreakLeaseHeaders,
    ContainerChangeLeaseHeaders: ContainerChangeLeaseHeaders,
    ContainerCreateHeaders: ContainerCreateHeaders,
    ContainerDeleteHeaders: ContainerDeleteHeaders,
    ContainerGetAccessPolicyHeaders: ContainerGetAccessPolicyHeaders,
    ContainerGetAccountInfoHeaders: ContainerGetAccountInfoHeaders,
    ContainerGetPropertiesHeaders: ContainerGetPropertiesHeaders,
    ContainerListBlobFlatSegmentHeaders: ContainerListBlobFlatSegmentHeaders,
    ContainerListBlobHierarchySegmentHeaders: ContainerListBlobHierarchySegmentHeaders,
    ContainerReleaseLeaseHeaders: ContainerReleaseLeaseHeaders,
    ContainerRenewLeaseHeaders: ContainerRenewLeaseHeaders,
    ContainerSetAccessPolicyHeaders: ContainerSetAccessPolicyHeaders,
    ContainerSetMetadataHeaders: ContainerSetMetadataHeaders,
    ListBlobsFlatSegmentResponse: ListBlobsFlatSegmentResponse,
    ListBlobsHierarchySegmentResponse: ListBlobsHierarchySegmentResponse,
    SignedIdentifier: SignedIdentifier,
    StorageError: StorageError
});

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
/** Class representing a Container. */
var Container = /** @class */ (function () {
    /**
     * Create a Container.
     * @param {StorageClientContext} client Reference to the service client.
     */
    function Container(client) {
        this.client = client;
    }
    Container.prototype.create = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, createOperationSpec, callback);
    };
    Container.prototype.getProperties = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getPropertiesOperationSpec$1, callback);
    };
    Container.prototype.deleteMethod = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, deleteMethodOperationSpec, callback);
    };
    Container.prototype.setMetadata = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, setMetadataOperationSpec, callback);
    };
    Container.prototype.getAccessPolicy = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getAccessPolicyOperationSpec, callback);
    };
    Container.prototype.setAccessPolicy = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, setAccessPolicyOperationSpec, callback);
    };
    Container.prototype.acquireLease = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, acquireLeaseOperationSpec, callback);
    };
    Container.prototype.releaseLease = function (leaseId, options, callback) {
        return this.client.sendOperationRequest({
            leaseId: leaseId,
            options: options
        }, releaseLeaseOperationSpec, callback);
    };
    Container.prototype.renewLease = function (leaseId, options, callback) {
        return this.client.sendOperationRequest({
            leaseId: leaseId,
            options: options
        }, renewLeaseOperationSpec, callback);
    };
    Container.prototype.breakLease = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, breakLeaseOperationSpec, callback);
    };
    Container.prototype.changeLease = function (leaseId, proposedLeaseId, options, callback) {
        return this.client.sendOperationRequest({
            leaseId: leaseId,
            proposedLeaseId: proposedLeaseId,
            options: options
        }, changeLeaseOperationSpec, callback);
    };
    Container.prototype.listBlobFlatSegment = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, listBlobFlatSegmentOperationSpec, callback);
    };
    Container.prototype.listBlobHierarchySegment = function (delimiter, options, callback) {
        return this.client.sendOperationRequest({
            delimiter: delimiter,
            options: options
        }, listBlobHierarchySegmentOperationSpec, callback);
    };
    Container.prototype.getAccountInfo = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getAccountInfoOperationSpec$1, callback);
    };
    return Container;
}());
// Operation Specifications
var serializer$1 = new msRest.Serializer(Mappers$1, true);
var createOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype2
    ],
    headerParameters: [
        metadata,
        access,
        version,
        requestId
    ],
    responses: {
        201: {
            headersMapper: ContainerCreateHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var getPropertiesOperationSpec$1 = {
    httpMethod: "GET",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype2
    ],
    headerParameters: [
        version,
        requestId,
        leaseId0
    ],
    responses: {
        200: {
            headersMapper: ContainerGetPropertiesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var deleteMethodOperationSpec = {
    httpMethod: "DELETE",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype2
    ],
    headerParameters: [
        version,
        requestId,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        202: {
            headersMapper: ContainerDeleteHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var setMetadataOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype2,
        comp5
    ],
    headerParameters: [
        metadata,
        version,
        requestId,
        leaseId0,
        ifModifiedSince
    ],
    responses: {
        200: {
            headersMapper: ContainerSetMetadataHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var getAccessPolicyOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype2,
        comp6
    ],
    headerParameters: [
        version,
        requestId,
        leaseId0
    ],
    responses: {
        200: {
            bodyMapper: {
                xmlElementName: "SignedIdentifier",
                serializedName: "parsedResponse",
                type: {
                    name: "Sequence",
                    element: {
                        type: {
                            name: "Composite",
                            className: "SignedIdentifier"
                        }
                    }
                }
            },
            headersMapper: ContainerGetAccessPolicyHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var setAccessPolicyOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        restype2,
        comp6
    ],
    headerParameters: [
        access,
        version,
        requestId,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    requestBody: {
        parameterPath: [
            "options",
            "containerAcl"
        ],
        mapper: {
            xmlName: "SignedIdentifiers",
            xmlElementName: "SignedIdentifier",
            serializedName: "containerAcl",
            type: {
                name: "Sequence",
                element: {
                    type: {
                        name: "Composite",
                        className: "SignedIdentifier"
                    }
                }
            }
        }
    },
    contentType: "application/xml; charset=utf-8",
    responses: {
        200: {
            headersMapper: ContainerSetAccessPolicyHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var acquireLeaseOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7,
        restype2
    ],
    headerParameters: [
        duration,
        proposedLeaseId0,
        version,
        requestId,
        action0,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        201: {
            headersMapper: ContainerAcquireLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var releaseLeaseOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7,
        restype2
    ],
    headerParameters: [
        leaseId1,
        version,
        requestId,
        action1,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        200: {
            headersMapper: ContainerReleaseLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var renewLeaseOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7,
        restype2
    ],
    headerParameters: [
        leaseId1,
        version,
        requestId,
        action2,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        200: {
            headersMapper: ContainerRenewLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var breakLeaseOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7,
        restype2
    ],
    headerParameters: [
        breakPeriod,
        version,
        requestId,
        action3,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        202: {
            headersMapper: ContainerBreakLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var changeLeaseOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7,
        restype2
    ],
    headerParameters: [
        leaseId1,
        proposedLeaseId1,
        version,
        requestId,
        action4,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        200: {
            headersMapper: ContainerChangeLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var listBlobFlatSegmentOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        prefix,
        marker0,
        maxresults,
        include1,
        timeout,
        restype2,
        comp2
    ],
    headerParameters: [
        version,
        requestId
    ],
    responses: {
        200: {
            bodyMapper: ListBlobsFlatSegmentResponse,
            headersMapper: ContainerListBlobFlatSegmentHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var listBlobHierarchySegmentOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        prefix,
        delimiter,
        marker0,
        maxresults,
        include1,
        timeout,
        restype2,
        comp2
    ],
    headerParameters: [
        version,
        requestId
    ],
    responses: {
        200: {
            bodyMapper: ListBlobsHierarchySegmentResponse,
            headersMapper: ContainerListBlobHierarchySegmentHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};
var getAccountInfoOperationSpec$1 = {
    httpMethod: "GET",
    path: "{containerName}",
    urlParameters: [
        url
    ],
    queryParameters: [
        restype1,
        comp0
    ],
    headerParameters: [
        version
    ],
    responses: {
        200: {
            headersMapper: ContainerGetAccountInfoHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$1
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var Mappers$2 = /*#__PURE__*/Object.freeze({
    BlobAbortCopyFromURLHeaders: BlobAbortCopyFromURLHeaders,
    BlobAcquireLeaseHeaders: BlobAcquireLeaseHeaders,
    BlobBreakLeaseHeaders: BlobBreakLeaseHeaders,
    BlobChangeLeaseHeaders: BlobChangeLeaseHeaders,
    BlobCopyFromURLHeaders: BlobCopyFromURLHeaders,
    BlobCreateSnapshotHeaders: BlobCreateSnapshotHeaders,
    BlobDeleteHeaders: BlobDeleteHeaders,
    BlobDownloadHeaders: BlobDownloadHeaders,
    BlobGetAccessControlHeaders: BlobGetAccessControlHeaders,
    BlobGetAccountInfoHeaders: BlobGetAccountInfoHeaders,
    BlobGetPropertiesHeaders: BlobGetPropertiesHeaders,
    BlobReleaseLeaseHeaders: BlobReleaseLeaseHeaders,
    BlobRenameHeaders: BlobRenameHeaders,
    BlobRenewLeaseHeaders: BlobRenewLeaseHeaders,
    BlobSetAccessControlHeaders: BlobSetAccessControlHeaders,
    BlobSetHTTPHeadersHeaders: BlobSetHTTPHeadersHeaders,
    BlobSetMetadataHeaders: BlobSetMetadataHeaders,
    BlobSetTierHeaders: BlobSetTierHeaders,
    BlobStartCopyFromURLHeaders: BlobStartCopyFromURLHeaders,
    BlobUndeleteHeaders: BlobUndeleteHeaders,
    DataLakeStorageError: DataLakeStorageError,
    DataLakeStorageErrorError: DataLakeStorageErrorError,
    StorageError: StorageError
});

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
/** Class representing a Blob. */
var Blob$1 = /** @class */ (function () {
    /**
     * Create a Blob.
     * @param {StorageClientContext} client Reference to the service client.
     */
    function Blob(client) {
        this.client = client;
    }
    Blob.prototype.download = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, downloadOperationSpec, callback);
    };
    Blob.prototype.getProperties = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getPropertiesOperationSpec$2, callback);
    };
    Blob.prototype.deleteMethod = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, deleteMethodOperationSpec$1, callback);
    };
    Blob.prototype.setAccessControl = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, setAccessControlOperationSpec, callback);
    };
    Blob.prototype.getAccessControl = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getAccessControlOperationSpec, callback);
    };
    Blob.prototype.rename = function (renameSource, options, callback) {
        return this.client.sendOperationRequest({
            renameSource: renameSource,
            options: options
        }, renameOperationSpec, callback);
    };
    Blob.prototype.undelete = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, undeleteOperationSpec, callback);
    };
    Blob.prototype.setHTTPHeaders = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, setHTTPHeadersOperationSpec, callback);
    };
    Blob.prototype.setMetadata = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, setMetadataOperationSpec$1, callback);
    };
    Blob.prototype.acquireLease = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, acquireLeaseOperationSpec$1, callback);
    };
    Blob.prototype.releaseLease = function (leaseId, options, callback) {
        return this.client.sendOperationRequest({
            leaseId: leaseId,
            options: options
        }, releaseLeaseOperationSpec$1, callback);
    };
    Blob.prototype.renewLease = function (leaseId, options, callback) {
        return this.client.sendOperationRequest({
            leaseId: leaseId,
            options: options
        }, renewLeaseOperationSpec$1, callback);
    };
    Blob.prototype.changeLease = function (leaseId, proposedLeaseId, options, callback) {
        return this.client.sendOperationRequest({
            leaseId: leaseId,
            proposedLeaseId: proposedLeaseId,
            options: options
        }, changeLeaseOperationSpec$1, callback);
    };
    Blob.prototype.breakLease = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, breakLeaseOperationSpec$1, callback);
    };
    Blob.prototype.createSnapshot = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, createSnapshotOperationSpec, callback);
    };
    Blob.prototype.startCopyFromURL = function (copySource, options, callback) {
        return this.client.sendOperationRequest({
            copySource: copySource,
            options: options
        }, startCopyFromURLOperationSpec, callback);
    };
    Blob.prototype.copyFromURL = function (copySource, options, callback) {
        return this.client.sendOperationRequest({
            copySource: copySource,
            options: options
        }, copyFromURLOperationSpec, callback);
    };
    Blob.prototype.abortCopyFromURL = function (copyId, options, callback) {
        return this.client.sendOperationRequest({
            copyId: copyId,
            options: options
        }, abortCopyFromURLOperationSpec, callback);
    };
    Blob.prototype.setTier = function (tier, options, callback) {
        return this.client.sendOperationRequest({
            tier: tier,
            options: options
        }, setTierOperationSpec, callback);
    };
    Blob.prototype.getAccountInfo = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getAccountInfoOperationSpec$2, callback);
    };
    return Blob;
}());
// Operation Specifications
var serializer$2 = new msRest.Serializer(Mappers$2, true);
var downloadOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        snapshot,
        timeout
    ],
    headerParameters: [
        range0,
        rangeGetContentMD5,
        rangeGetContentCRC64,
        version,
        requestId,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            bodyMapper: {
                serializedName: "parsedResponse",
                type: {
                    name: "Stream"
                }
            },
            headersMapper: BlobDownloadHeaders
        },
        206: {
            bodyMapper: {
                serializedName: "parsedResponse",
                type: {
                    name: "Stream"
                }
            },
            headersMapper: BlobDownloadHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var getPropertiesOperationSpec$2 = {
    httpMethod: "HEAD",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        snapshot,
        timeout
    ],
    headerParameters: [
        version,
        requestId,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: BlobGetPropertiesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var deleteMethodOperationSpec$1 = {
    httpMethod: "DELETE",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        snapshot,
        timeout
    ],
    headerParameters: [
        deleteSnapshots,
        version,
        requestId,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        202: {
            headersMapper: BlobDeleteHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var setAccessControlOperationSpec = {
    httpMethod: "PATCH",
    path: "{filesystem}/{path}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        action5
    ],
    headerParameters: [
        owner,
        group,
        posixPermissions,
        posixAcl,
        requestId,
        version,
        leaseId0,
        ifMatch,
        ifNoneMatch,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        200: {
            headersMapper: BlobSetAccessControlHeaders
        },
        default: {
            bodyMapper: DataLakeStorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var getAccessControlOperationSpec = {
    httpMethod: "HEAD",
    path: "{filesystem}/{path}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        upn,
        action6
    ],
    headerParameters: [
        requestId,
        version,
        leaseId0,
        ifMatch,
        ifNoneMatch,
        ifModifiedSince,
        ifUnmodifiedSince
    ],
    responses: {
        200: {
            headersMapper: BlobGetAccessControlHeaders
        },
        default: {
            bodyMapper: DataLakeStorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var renameOperationSpec = {
    httpMethod: "PUT",
    path: "{filesystem}/{path}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        pathRenameMode
    ],
    headerParameters: [
        renameSource,
        directoryProperties,
        posixPermissions,
        posixUmask,
        sourceLeaseId,
        version,
        requestId,
        cacheControl,
        contentType,
        contentEncoding,
        contentLanguage,
        contentDisposition,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch,
        sourceIfModifiedSince,
        sourceIfUnmodifiedSince,
        sourceIfMatch,
        sourceIfNoneMatch
    ],
    responses: {
        201: {
            headersMapper: BlobRenameHeaders
        },
        default: {
            bodyMapper: DataLakeStorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var undeleteOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp8
    ],
    headerParameters: [
        version,
        requestId
    ],
    responses: {
        200: {
            headersMapper: BlobUndeleteHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var setHTTPHeadersOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp0
    ],
    headerParameters: [
        version,
        requestId,
        blobCacheControl,
        blobContentType,
        blobContentMD5,
        blobContentEncoding,
        blobContentLanguage,
        blobContentDisposition,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: BlobSetHTTPHeadersHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var setMetadataOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp5
    ],
    headerParameters: [
        metadata,
        version,
        requestId,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: BlobSetMetadataHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var acquireLeaseOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7
    ],
    headerParameters: [
        duration,
        proposedLeaseId0,
        version,
        requestId,
        action0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        201: {
            headersMapper: BlobAcquireLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var releaseLeaseOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7
    ],
    headerParameters: [
        leaseId1,
        version,
        requestId,
        action1,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: BlobReleaseLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var renewLeaseOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7
    ],
    headerParameters: [
        leaseId1,
        version,
        requestId,
        action2,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: BlobRenewLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var changeLeaseOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7
    ],
    headerParameters: [
        leaseId1,
        proposedLeaseId1,
        version,
        requestId,
        action4,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: BlobChangeLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var breakLeaseOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp7
    ],
    headerParameters: [
        breakPeriod,
        version,
        requestId,
        action3,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        202: {
            headersMapper: BlobBreakLeaseHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var createSnapshotOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp9
    ],
    headerParameters: [
        metadata,
        version,
        requestId,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch,
        leaseId0
    ],
    responses: {
        201: {
            headersMapper: BlobCreateSnapshotHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var startCopyFromURLOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout
    ],
    headerParameters: [
        metadata,
        tier0,
        rehydratePriority,
        copySource,
        version,
        requestId,
        sourceIfModifiedSince,
        sourceIfUnmodifiedSince,
        sourceIfMatch,
        sourceIfNoneMatch,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch,
        leaseId0
    ],
    responses: {
        202: {
            headersMapper: BlobStartCopyFromURLHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var copyFromURLOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout
    ],
    headerParameters: [
        metadata,
        tier0,
        copySource,
        version,
        requestId,
        xMsRequiresSync,
        sourceIfModifiedSince,
        sourceIfUnmodifiedSince,
        sourceIfMatch,
        sourceIfNoneMatch,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch,
        leaseId0
    ],
    responses: {
        202: {
            headersMapper: BlobCopyFromURLHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var abortCopyFromURLOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        copyId,
        timeout,
        comp10
    ],
    headerParameters: [
        version,
        requestId,
        copyActionAbortConstant,
        leaseId0
    ],
    responses: {
        204: {
            headersMapper: BlobAbortCopyFromURLHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var setTierOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp11
    ],
    headerParameters: [
        tier1,
        rehydratePriority,
        version,
        requestId,
        leaseId0
    ],
    responses: {
        200: {
            headersMapper: BlobSetTierHeaders
        },
        202: {
            headersMapper: BlobSetTierHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};
var getAccountInfoOperationSpec$2 = {
    httpMethod: "GET",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        restype1,
        comp0
    ],
    headerParameters: [
        version
    ],
    responses: {
        200: {
            headersMapper: BlobGetAccountInfoHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$2
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var Mappers$3 = /*#__PURE__*/Object.freeze({
    ClearRange: ClearRange,
    PageBlobClearPagesHeaders: PageBlobClearPagesHeaders,
    PageBlobCopyIncrementalHeaders: PageBlobCopyIncrementalHeaders,
    PageBlobCreateHeaders: PageBlobCreateHeaders,
    PageBlobGetPageRangesDiffHeaders: PageBlobGetPageRangesDiffHeaders,
    PageBlobGetPageRangesHeaders: PageBlobGetPageRangesHeaders,
    PageBlobResizeHeaders: PageBlobResizeHeaders,
    PageBlobUpdateSequenceNumberHeaders: PageBlobUpdateSequenceNumberHeaders,
    PageBlobUploadPagesFromURLHeaders: PageBlobUploadPagesFromURLHeaders,
    PageBlobUploadPagesHeaders: PageBlobUploadPagesHeaders,
    PageList: PageList,
    PageRange: PageRange,
    StorageError: StorageError
});

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
/** Class representing a PageBlob. */
var PageBlob = /** @class */ (function () {
    /**
     * Create a PageBlob.
     * @param {StorageClientContext} client Reference to the service client.
     */
    function PageBlob(client) {
        this.client = client;
    }
    PageBlob.prototype.create = function (contentLength, blobContentLength, options, callback) {
        return this.client.sendOperationRequest({
            contentLength: contentLength,
            blobContentLength: blobContentLength,
            options: options
        }, createOperationSpec$1, callback);
    };
    PageBlob.prototype.uploadPages = function (body, contentLength, options, callback) {
        return this.client.sendOperationRequest({
            body: body,
            contentLength: contentLength,
            options: options
        }, uploadPagesOperationSpec, callback);
    };
    PageBlob.prototype.clearPages = function (contentLength, options, callback) {
        return this.client.sendOperationRequest({
            contentLength: contentLength,
            options: options
        }, clearPagesOperationSpec, callback);
    };
    PageBlob.prototype.uploadPagesFromURL = function (sourceUrl, sourceRange, contentLength, range, options, callback) {
        return this.client.sendOperationRequest({
            sourceUrl: sourceUrl,
            sourceRange: sourceRange,
            contentLength: contentLength,
            range: range,
            options: options
        }, uploadPagesFromURLOperationSpec, callback);
    };
    PageBlob.prototype.getPageRanges = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getPageRangesOperationSpec, callback);
    };
    PageBlob.prototype.getPageRangesDiff = function (options, callback) {
        return this.client.sendOperationRequest({
            options: options
        }, getPageRangesDiffOperationSpec, callback);
    };
    PageBlob.prototype.resize = function (blobContentLength, options, callback) {
        return this.client.sendOperationRequest({
            blobContentLength: blobContentLength,
            options: options
        }, resizeOperationSpec, callback);
    };
    PageBlob.prototype.updateSequenceNumber = function (sequenceNumberAction, options, callback) {
        return this.client.sendOperationRequest({
            sequenceNumberAction: sequenceNumberAction,
            options: options
        }, updateSequenceNumberOperationSpec, callback);
    };
    PageBlob.prototype.copyIncremental = function (copySource, options, callback) {
        return this.client.sendOperationRequest({
            copySource: copySource,
            options: options
        }, copyIncrementalOperationSpec, callback);
    };
    return PageBlob;
}());
// Operation Specifications
var serializer$3 = new msRest.Serializer(Mappers$3, true);
var createOperationSpec$1 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout
    ],
    headerParameters: [
        contentLength,
        metadata,
        blobContentLength,
        blobSequenceNumber,
        version,
        requestId,
        tier0,
        blobType0,
        blobContentType,
        blobContentEncoding,
        blobContentLanguage,
        blobContentMD5,
        blobCacheControl,
        blobContentDisposition,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        201: {
            headersMapper: PageBlobCreateHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var uploadPagesOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp12
    ],
    headerParameters: [
        contentLength,
        transactionalContentMD5,
        transactionalContentCrc64,
        range0,
        version,
        requestId,
        pageWrite0,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifSequenceNumberLessThanOrEqualTo,
        ifSequenceNumberLessThan,
        ifSequenceNumberEqualTo,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    requestBody: {
        parameterPath: "body",
        mapper: {
            required: true,
            serializedName: "body",
            type: {
                name: "Stream"
            }
        }
    },
    contentType: "application/octet-stream",
    responses: {
        201: {
            headersMapper: PageBlobUploadPagesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var clearPagesOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp12
    ],
    headerParameters: [
        contentLength,
        range0,
        version,
        requestId,
        pageWrite1,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifSequenceNumberLessThanOrEqualTo,
        ifSequenceNumberLessThan,
        ifSequenceNumberEqualTo,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        201: {
            headersMapper: PageBlobClearPagesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var uploadPagesFromURLOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp12
    ],
    headerParameters: [
        sourceUrl,
        sourceRange0,
        sourceContentMD5,
        sourceContentCrc64,
        contentLength,
        range1,
        version,
        requestId,
        pageWrite0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        leaseId0,
        ifSequenceNumberLessThanOrEqualTo,
        ifSequenceNumberLessThan,
        ifSequenceNumberEqualTo,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch,
        sourceIfModifiedSince,
        sourceIfUnmodifiedSince,
        sourceIfMatch,
        sourceIfNoneMatch
    ],
    responses: {
        201: {
            headersMapper: PageBlobUploadPagesFromURLHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var getPageRangesOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        snapshot,
        timeout,
        comp13
    ],
    headerParameters: [
        range0,
        version,
        requestId,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            bodyMapper: PageList,
            headersMapper: PageBlobGetPageRangesHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var getPageRangesDiffOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        snapshot,
        timeout,
        prevsnapshot,
        comp13
    ],
    headerParameters: [
        range0,
        version,
        requestId,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            bodyMapper: PageList,
            headersMapper: PageBlobGetPageRangesDiffHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var resizeOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp0
    ],
    headerParameters: [
        blobContentLength,
        version,
        requestId,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: PageBlobResizeHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var updateSequenceNumberOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp0
    ],
    headerParameters: [
        sequenceNumberAction,
        blobSequenceNumber,
        version,
        requestId,
        leaseId0,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        200: {
            headersMapper: PageBlobUpdateSequenceNumberHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};
var copyIncrementalOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp14
    ],
    headerParameters: [
        copySource,
        version,
        requestId,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        202: {
            headersMapper: PageBlobCopyIncrementalHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$3
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var Mappers$4 = /*#__PURE__*/Object.freeze({
    AppendBlobAppendBlockFromUrlHeaders: AppendBlobAppendBlockFromUrlHeaders,
    AppendBlobAppendBlockHeaders: AppendBlobAppendBlockHeaders,
    AppendBlobCreateHeaders: AppendBlobCreateHeaders,
    StorageError: StorageError
});

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
/** Class representing a AppendBlob. */
var AppendBlob = /** @class */ (function () {
    /**
     * Create a AppendBlob.
     * @param {StorageClientContext} client Reference to the service client.
     */
    function AppendBlob(client) {
        this.client = client;
    }
    AppendBlob.prototype.create = function (contentLength, options, callback) {
        return this.client.sendOperationRequest({
            contentLength: contentLength,
            options: options
        }, createOperationSpec$2, callback);
    };
    AppendBlob.prototype.appendBlock = function (body, contentLength, options, callback) {
        return this.client.sendOperationRequest({
            body: body,
            contentLength: contentLength,
            options: options
        }, appendBlockOperationSpec, callback);
    };
    AppendBlob.prototype.appendBlockFromUrl = function (sourceUrl, contentLength, options, callback) {
        return this.client.sendOperationRequest({
            sourceUrl: sourceUrl,
            contentLength: contentLength,
            options: options
        }, appendBlockFromUrlOperationSpec, callback);
    };
    return AppendBlob;
}());
// Operation Specifications
var serializer$4 = new msRest.Serializer(Mappers$4, true);
var createOperationSpec$2 = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout
    ],
    headerParameters: [
        contentLength,
        metadata,
        version,
        requestId,
        blobType1,
        blobContentType,
        blobContentEncoding,
        blobContentLanguage,
        blobContentMD5,
        blobCacheControl,
        blobContentDisposition,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    responses: {
        201: {
            headersMapper: AppendBlobCreateHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$4
};
var appendBlockOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp15
    ],
    headerParameters: [
        contentLength,
        transactionalContentMD5,
        transactionalContentCrc64,
        version,
        requestId,
        leaseId0,
        maxSize,
        appendPosition,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    requestBody: {
        parameterPath: "body",
        mapper: {
            required: true,
            serializedName: "body",
            type: {
                name: "Stream"
            }
        }
    },
    contentType: "application/octet-stream",
    responses: {
        201: {
            headersMapper: AppendBlobAppendBlockHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$4
};
var appendBlockFromUrlOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp15
    ],
    headerParameters: [
        sourceUrl,
        sourceRange1,
        sourceContentMD5,
        sourceContentCrc64,
        contentLength,
        transactionalContentMD5,
        version,
        requestId,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        leaseId0,
        maxSize,
        appendPosition,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch,
        sourceIfModifiedSince,
        sourceIfUnmodifiedSince,
        sourceIfMatch,
        sourceIfNoneMatch
    ],
    responses: {
        201: {
            headersMapper: AppendBlobAppendBlockFromUrlHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$4
};

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is regenerated.
 */

var Mappers$5 = /*#__PURE__*/Object.freeze({
    Block: Block,
    BlockBlobCommitBlockListHeaders: BlockBlobCommitBlockListHeaders,
    BlockBlobGetBlockListHeaders: BlockBlobGetBlockListHeaders,
    BlockBlobStageBlockFromURLHeaders: BlockBlobStageBlockFromURLHeaders,
    BlockBlobStageBlockHeaders: BlockBlobStageBlockHeaders,
    BlockBlobUploadHeaders: BlockBlobUploadHeaders,
    BlockList: BlockList,
    BlockLookupList: BlockLookupList,
    StorageError: StorageError
});

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
/** Class representing a BlockBlob. */
var BlockBlob = /** @class */ (function () {
    /**
     * Create a BlockBlob.
     * @param {StorageClientContext} client Reference to the service client.
     */
    function BlockBlob(client) {
        this.client = client;
    }
    BlockBlob.prototype.upload = function (body, contentLength, options, callback) {
        return this.client.sendOperationRequest({
            body: body,
            contentLength: contentLength,
            options: options
        }, uploadOperationSpec, callback);
    };
    BlockBlob.prototype.stageBlock = function (blockId, contentLength, body, options, callback) {
        return this.client.sendOperationRequest({
            blockId: blockId,
            contentLength: contentLength,
            body: body,
            options: options
        }, stageBlockOperationSpec, callback);
    };
    BlockBlob.prototype.stageBlockFromURL = function (blockId, contentLength, sourceUrl, options, callback) {
        return this.client.sendOperationRequest({
            blockId: blockId,
            contentLength: contentLength,
            sourceUrl: sourceUrl,
            options: options
        }, stageBlockFromURLOperationSpec, callback);
    };
    BlockBlob.prototype.commitBlockList = function (blocks, options, callback) {
        return this.client.sendOperationRequest({
            blocks: blocks,
            options: options
        }, commitBlockListOperationSpec, callback);
    };
    BlockBlob.prototype.getBlockList = function (listType, options, callback) {
        return this.client.sendOperationRequest({
            listType: listType,
            options: options
        }, getBlockListOperationSpec, callback);
    };
    return BlockBlob;
}());
// Operation Specifications
var serializer$5 = new msRest.Serializer(Mappers$5, true);
var uploadOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout
    ],
    headerParameters: [
        contentLength,
        metadata,
        tier0,
        version,
        requestId,
        blobType2,
        blobContentType,
        blobContentEncoding,
        blobContentLanguage,
        blobContentMD5,
        blobCacheControl,
        blobContentDisposition,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    requestBody: {
        parameterPath: "body",
        mapper: {
            required: true,
            serializedName: "body",
            type: {
                name: "Stream"
            }
        }
    },
    contentType: "application/octet-stream",
    responses: {
        201: {
            headersMapper: BlockBlobUploadHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$5
};
var stageBlockOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        blockId,
        timeout,
        comp16
    ],
    headerParameters: [
        contentLength,
        transactionalContentMD5,
        transactionalContentCrc64,
        version,
        requestId,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm
    ],
    requestBody: {
        parameterPath: "body",
        mapper: {
            required: true,
            serializedName: "body",
            type: {
                name: "Stream"
            }
        }
    },
    contentType: "application/octet-stream",
    responses: {
        201: {
            headersMapper: BlockBlobStageBlockHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$5
};
var stageBlockFromURLOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        blockId,
        timeout,
        comp16
    ],
    headerParameters: [
        contentLength,
        sourceUrl,
        sourceRange1,
        sourceContentMD5,
        sourceContentCrc64,
        version,
        requestId,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        leaseId0,
        sourceIfModifiedSince,
        sourceIfUnmodifiedSince,
        sourceIfMatch,
        sourceIfNoneMatch
    ],
    responses: {
        201: {
            headersMapper: BlockBlobStageBlockFromURLHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$5
};
var commitBlockListOperationSpec = {
    httpMethod: "PUT",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        timeout,
        comp17
    ],
    headerParameters: [
        transactionalContentMD5,
        transactionalContentCrc64,
        metadata,
        tier0,
        version,
        requestId,
        blobCacheControl,
        blobContentType,
        blobContentEncoding,
        blobContentLanguage,
        blobContentMD5,
        blobContentDisposition,
        leaseId0,
        encryptionKey,
        encryptionKeySha256,
        encryptionAlgorithm,
        ifModifiedSince,
        ifUnmodifiedSince,
        ifMatch,
        ifNoneMatch
    ],
    requestBody: {
        parameterPath: "blocks",
        mapper: tslib_1.__assign({}, BlockLookupList, { required: true })
    },
    contentType: "application/xml; charset=utf-8",
    responses: {
        201: {
            headersMapper: BlockBlobCommitBlockListHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$5
};
var getBlockListOperationSpec = {
    httpMethod: "GET",
    path: "{containerName}/{blob}",
    urlParameters: [
        url
    ],
    queryParameters: [
        snapshot,
        listType,
        timeout,
        comp17
    ],
    headerParameters: [
        version,
        requestId,
        leaseId0
    ],
    responses: {
        200: {
            bodyMapper: BlockList,
            headersMapper: BlockBlobGetBlockListHeaders
        },
        default: {
            bodyMapper: StorageError
        }
    },
    isXML: true,
    serializer: serializer$5
};

/**
 * Generate a range string. For example:
 *
 * "bytes=255-" or "bytes=0-511"
 *
 * @export
 * @param {IRange} iRange
 * @returns {string}
 */
function rangeToString(iRange) {
    if (iRange.offset < 0) {
        throw new RangeError("IRange.offset cannot be smaller than 0.");
    }
    if (iRange.count && iRange.count <= 0) {
        throw new RangeError("IRange.count must be larger than 0. Leave it undefined if you want a range from offset to the end.");
    }
    return iRange.count
        ? "bytes=" + iRange.offset + "-" + (iRange.offset + iRange.count - 1)
        : "bytes=" + iRange.offset + "-";
}

(function (BlockBlobTier) {
    BlockBlobTier["Hot"] = "Hot";
    BlockBlobTier["Cool"] = "Cool";
    BlockBlobTier["Archive"] = "Archive";
})(exports.BlockBlobTier || (exports.BlockBlobTier = {}));
(function (PremiumPageBlobTier) {
    PremiumPageBlobTier["P4"] = "P4";
    PremiumPageBlobTier["P6"] = "P6";
    PremiumPageBlobTier["P10"] = "P10";
    PremiumPageBlobTier["P15"] = "P15";
    PremiumPageBlobTier["P20"] = "P20";
    PremiumPageBlobTier["P30"] = "P30";
    PremiumPageBlobTier["P40"] = "P40";
    PremiumPageBlobTier["P50"] = "P50";
    PremiumPageBlobTier["P60"] = "P60";
    PremiumPageBlobTier["P70"] = "P70";
    PremiumPageBlobTier["P80"] = "P80";
})(exports.PremiumPageBlobTier || (exports.PremiumPageBlobTier = {}));
function toAccessTier(tier) {
    if (tier == undefined) {
        return undefined;
    }
    return tier; // No more check if string is a valid AccessTier, and left this to underlay logic to decide(service).
}
function ensureCpkIfSpecified(cpk, isHttps) {
    if (cpk && !isHttps) {
        throw new RangeError("Customer-provided encryption key must be used over HTTPS.");
    }
    if (cpk && !cpk.encryptionAlgorithm) {
        cpk.encryptionAlgorithm = EncryptionAlgorithmAES25;
    }
}

/**
 * BrowserPolicy will handle differences between Node.js and browser runtime, including:
 *
 * 1. Browsers cache GET/HEAD requests by adding conditional headers such as 'IF_MODIFIED_SINCE'.
 * BrowserPolicy is a policy used to add a timestamp query to GET/HEAD request URL
 * thus avoid the browser cache.
 *
 * 2. Remove cookie header for security
 *
 * 3. Remove content-length header to avoid browsers warning
 *
 * @class BrowserPolicy
 * @extends {BaseRequestPolicy}
 */
var BrowserPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(BrowserPolicy, _super);
    /**
     * Creates an instance of BrowserPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @memberof BrowserPolicy
     */
    function BrowserPolicy(nextPolicy, options) {
        return _super.call(this, nextPolicy, options) || this;
    }
    /**
     * Sends out request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof BrowserPolicy
     */
    BrowserPolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                {
                    return [2 /*return*/, this._nextPolicy.sendRequest(request)];
                }
                if (request.method.toUpperCase() === "GET" || request.method.toUpperCase() === "HEAD") {
                    request.url = setURLParameter(request.url, URLConstants.Parameters.FORCE_BROWSER_NO_CACHE, new Date().getTime().toString());
                }
                request.headers.remove(HeaderConstants.COOKIE);
                // According to XHR standards, content-length should be fully controlled by browsers
                request.headers.remove(HeaderConstants.CONTENT_LENGTH);
                return [2 /*return*/, this._nextPolicy.sendRequest(request)];
            });
        });
    };
    return BrowserPolicy;
}(msRest.BaseRequestPolicy));

/**
 * BrowserPolicyFactory is a factory class helping generating BrowserPolicy objects.
 *
 * @export
 * @class BrowserPolicyFactory
 * @implements {RequestPolicyFactory}
 */
var BrowserPolicyFactory = /** @class */ (function () {
    function BrowserPolicyFactory() {
    }
    BrowserPolicyFactory.prototype.create = function (nextPolicy, options) {
        return new BrowserPolicy(nextPolicy, options);
    };
    return BrowserPolicyFactory;
}());

/**
 * Credential is an abstract class for Azure Storage HTTP requests signing. This
 * class will host an credentialPolicyCreator factory which generates CredentialPolicy.
 *
 * @export
 * @abstract
 * @class Credential
 */
var Credential = /** @class */ (function () {
    function Credential() {
    }
    /**
     * Creates a RequestPolicy object.
     *
     * @param {RequestPolicy} _nextPolicy
     * @param {RequestPolicyOptions} _options
     * @returns {RequestPolicy}
     * @memberof Credential
     */
    Credential.prototype.create = function (
    // tslint:disable-next-line:variable-name
    _nextPolicy, 
    // tslint:disable-next-line:variable-name
    _options) {
        throw new Error("Method should be implemented in children classes.");
    };
    return Credential;
}());

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 *
 * Code generated by Microsoft (R) AutoRest Code Generator.
 * Changes may cause incorrect behavior and will be lost if the code is
 * regenerated.
 */
var packageName = "azure-storage-blob";
var packageVersion = "1.0.0";
var StorageClientContext = /** @class */ (function (_super) {
    tslib_1.__extends(StorageClientContext, _super);
    /**
     * Initializes a new instance of the StorageClientContext class.
     * @param url The URL of the service account, container, or blob that is the targe of the desired
     * operation.
     * @param [options] The parameter options
     */
    function StorageClientContext(url, options) {
        var _this = this;
        if (url == undefined) {
            throw new Error("'url' cannot be null.");
        }
        if (!options) {
            options = {};
        }
        if (!options.userAgent) {
            var defaultUserAgent = msRest.getDefaultUserAgentValue();
            options.userAgent = packageName + "/" + packageVersion + " " + defaultUserAgent;
        }
        _this = _super.call(this, undefined, options) || this;
        _this.version = '2019-02-02';
        _this.baseUri = "{url}";
        _this.requestContentType = "application/json; charset=utf-8";
        _this.url = url;
        if (options.pathRenameMode !== null && options.pathRenameMode !== undefined) {
            _this.pathRenameMode = options.pathRenameMode;
        }
        return _this;
    }
    return StorageClientContext;
}(msRest.ServiceClient));

/**
 * KeepAlivePolicy is a policy used to control keep alive settings for every request.
 *
 * @class KeepAlivePolicy
 * @extends {BaseRequestPolicy}
 */
var KeepAlivePolicy = /** @class */ (function (_super) {
    tslib_1.__extends(KeepAlivePolicy, _super);
    /**
     * Creates an instance of KeepAlivePolicy.
     *
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @param {IKeepAliveOptions} [keepAliveOptions]
     * @memberof KeepAlivePolicy
     */
    function KeepAlivePolicy(nextPolicy, options, keepAliveOptions) {
        var _this = _super.call(this, nextPolicy, options) || this;
        _this.keepAliveOptions = keepAliveOptions;
        return _this;
    }
    /**
     * Sends out request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof KeepAlivePolicy
     */
    KeepAlivePolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                request.keepAlive = this.keepAliveOptions.enable;
                return [2 /*return*/, this._nextPolicy.sendRequest(request)];
            });
        });
    };
    return KeepAlivePolicy;
}(msRest.BaseRequestPolicy));

/**
 * KeepAlivePolicyFactory is a factory class helping generating KeepAlivePolicy objects.
 *
 * @export
 * @class KeepAlivePolicyFactory
 * @implements {RequestPolicyFactory}
 */
var KeepAlivePolicyFactory = /** @class */ (function () {
    /**
     * Creates an instance of KeepAlivePolicyFactory.
     *
     * @param {IKeepAliveOptions} [telemetry]
     * @memberof KeepAlivePolicyFactory
     */
    function KeepAlivePolicyFactory(keepAliveOptions) {
        if (keepAliveOptions === void 0) { keepAliveOptions = { enable: true }; }
        this.keepAliveOptions = keepAliveOptions;
    }
    KeepAlivePolicyFactory.prototype.create = function (nextPolicy, options) {
        return new KeepAlivePolicy(nextPolicy, options, this.keepAliveOptions);
    };
    return KeepAlivePolicyFactory;
}());

// Default values of IRetryOptions
var DEFAULT_REQUEST_LOG_OPTIONS = {
    logWarningIfTryOverThreshold: 3000
};
/**
 * LoggingPolicy is a policy used to log requests.
 *
 * @class LoggingPolicy
 * @extends {BaseRequestPolicy}
 */
var LoggingPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(LoggingPolicy, _super);
    /**
     * Creates an instance of LoggingPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @param {IRequestLogOptions} [loggingOptions=DEFAULT_REQUEST_LOG_OPTIONS]
     * @memberof LoggingPolicy
     */
    function LoggingPolicy(nextPolicy, options, loggingOptions) {
        if (loggingOptions === void 0) { loggingOptions = DEFAULT_REQUEST_LOG_OPTIONS; }
        var _this = _super.call(this, nextPolicy, options) || this;
        _this.tryCount = 0;
        _this.operationStartTime = new Date();
        _this.requestStartTime = new Date();
        _this.loggingOptions = loggingOptions;
        return _this;
    }
    /**
     * Sends out request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof LoggingPolicy
     */
    LoggingPolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var safeURL, response, requestEndTime, requestCompletionTime, operationDuration, currentLevel, logMessage, errorString, messageInfo, err_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.tryCount++;
                        this.requestStartTime = new Date();
                        if (this.tryCount === 1) {
                            this.operationStartTime = this.requestStartTime;
                        }
                        safeURL = request.url;
                        if (getURLParameter(safeURL, URLConstants.Parameters.SIGNATURE)) {
                            safeURL = setURLParameter(safeURL, URLConstants.Parameters.SIGNATURE, "*****");
                        }
                        this.log(msRest.HttpPipelineLogLevel.INFO, "'" + safeURL + "'==> OUTGOING REQUEST (Try number=" + this.tryCount + ").");
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 3, , 4]);
                        return [4 /*yield*/, this._nextPolicy.sendRequest(request)];
                    case 2:
                        response = _a.sent();
                        requestEndTime = new Date();
                        requestCompletionTime = requestEndTime.getTime() - this.requestStartTime.getTime();
                        operationDuration = requestEndTime.getTime() - this.operationStartTime.getTime();
                        currentLevel = msRest.HttpPipelineLogLevel.INFO;
                        logMessage = "";
                        if (this.shouldLog(msRest.HttpPipelineLogLevel.INFO)) {
                            // Assume success and default to informational logging.
                            logMessage = "Successfully Received Response. ";
                        }
                        // If the response took too long, we'll upgrade to warning.
                        if (requestCompletionTime >= this.loggingOptions.logWarningIfTryOverThreshold) {
                            // Log a warning if the try duration exceeded the specified threshold.
                            if (this.shouldLog(msRest.HttpPipelineLogLevel.WARNING)) {
                                currentLevel = msRest.HttpPipelineLogLevel.WARNING;
                                logMessage = "SLOW OPERATION. Duration > " + this.loggingOptions.logWarningIfTryOverThreshold + " ms. ";
                            }
                        }
                        if ((response.status >= 400 &&
                            response.status <= 499 &&
                            (response.status !== HTTPURLConnection.HTTP_NOT_FOUND &&
                                response.status !== HTTPURLConnection.HTTP_CONFLICT &&
                                response.status !== HTTPURLConnection.HTTP_PRECON_FAILED &&
                                response.status !== HTTPURLConnection.HTTP_RANGE_NOT_SATISFIABLE)) ||
                            (response.status >= 500 && response.status <= 509)) {
                            errorString = "REQUEST ERROR: HTTP request failed with status code: " + response.status + ". ";
                            logMessage = errorString;
                            currentLevel = msRest.HttpPipelineLogLevel.ERROR;
                        }
                        messageInfo = "Request try:" + this.tryCount + ", status:" + response.status + " request duration:" + requestCompletionTime + " ms, operation duration:" + operationDuration + " ms\n";
                        this.log(currentLevel, logMessage + messageInfo);
                        return [2 /*return*/, response];
                    case 3:
                        err_1 = _a.sent();
                        this.log(msRest.HttpPipelineLogLevel.ERROR, "Unexpected failure attempting to make request. Error message: " + err_1.message);
                        throw err_1;
                    case 4: return [2 /*return*/];
                }
            });
        });
    };
    return LoggingPolicy;
}(msRest.BaseRequestPolicy));

/**
 * LoggingPolicyFactory is a factory class helping generating LoggingPolicy objects.
 *
 * @export
 * @class LoggingPolicyFactory
 * @implements {RequestPolicyFactory}
 */
var LoggingPolicyFactory = /** @class */ (function () {
    function LoggingPolicyFactory(loggingOptions) {
        this.loggingOptions = loggingOptions;
    }
    LoggingPolicyFactory.prototype.create = function (nextPolicy, options) {
        return new LoggingPolicy(nextPolicy, options, this.loggingOptions);
    };
    return LoggingPolicyFactory;
}());

/**
 * A Pipeline class containing HTTP request policies.
 * You can create a default Pipeline by calling StorageURL.newPipeline().
 * Or you can create a Pipeline with your own policies by the constructor of Pipeline.
 * Refer to StorageURL.newPipeline() and provided policies as reference before
 * implementing your customized Pipeline.
 *
 * @export
 * @class Pipeline
 */
var Pipeline = /** @class */ (function () {
    /**
     * Creates an instance of Pipeline. Customize HTTPClient by implementing IHttpClient interface.
     *
     * @param {RequestPolicyFactory[]} factories
     * @param {IPipelineOptions} [options={}]
     * @memberof Pipeline
     */
    function Pipeline(factories, options) {
        if (options === void 0) { options = {}; }
        this.factories = factories;
        this.options = options;
    }
    /**
     * Transfer Pipeline object to ServiceClientOptions object which required by
     * ServiceClient constructor.
     *
     * @returns {ServiceClientOptions}
     * @memberof Pipeline
     */
    Pipeline.prototype.toServiceClientOptions = function () {
        return {
            httpClient: this.options.HTTPClient,
            httpPipelineLogger: this.options.logger,
            requestPolicyFactories: this.factories
        };
    };
    return Pipeline;
}());

(function (RetryPolicyType) {
    /**
     * Exponential retry. Retry time delay grows exponentially.
     */
    RetryPolicyType[RetryPolicyType["EXPONENTIAL"] = 0] = "EXPONENTIAL";
    /**
     * Linear retry. Retry time delay grows linearly.
     */
    RetryPolicyType[RetryPolicyType["FIXED"] = 1] = "FIXED";
})(exports.RetryPolicyType || (exports.RetryPolicyType = {}));
// Default values of IRetryOptions
var DEFAULT_RETRY_OPTIONS = {
    maxRetryDelayInMs: 120 * 1000,
    maxTries: 4,
    retryDelayInMs: 4 * 1000,
    retryPolicyType: exports.RetryPolicyType.EXPONENTIAL,
    secondaryHost: "",
    tryTimeoutInMs: undefined // Use server side default timeout strategy
};
var RETRY_ABORT_ERROR = new msRest.RestError("The request was aborted", msRest.RestError.REQUEST_ABORTED_ERROR);
/**
 * Retry policy with exponential retry and linear retry implemented.
 *
 * @class RetryPolicy
 * @extends {BaseRequestPolicy}
 */
var RetryPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(RetryPolicy, _super);
    /**
     * Creates an instance of RetryPolicy.
     *
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @param {IRetryOptions} [retryOptions=DEFAULT_RETRY_OPTIONS]
     * @memberof RetryPolicy
     */
    function RetryPolicy(nextPolicy, options, retryOptions) {
        if (retryOptions === void 0) { retryOptions = DEFAULT_RETRY_OPTIONS; }
        var _this = _super.call(this, nextPolicy, options) || this;
        // Initialize retry options
        _this.retryOptions = {
            retryPolicyType: retryOptions.retryPolicyType
                ? retryOptions.retryPolicyType
                : DEFAULT_RETRY_OPTIONS.retryPolicyType,
            maxTries: retryOptions.maxTries && retryOptions.maxTries >= 1
                ? Math.floor(retryOptions.maxTries)
                : DEFAULT_RETRY_OPTIONS.maxTries,
            tryTimeoutInMs: retryOptions.tryTimeoutInMs && retryOptions.tryTimeoutInMs >= 0
                ? retryOptions.tryTimeoutInMs
                : DEFAULT_RETRY_OPTIONS.tryTimeoutInMs,
            retryDelayInMs: retryOptions.retryDelayInMs && retryOptions.retryDelayInMs >= 0
                ? Math.min(retryOptions.retryDelayInMs, retryOptions.maxRetryDelayInMs
                    ? retryOptions.maxRetryDelayInMs
                    : DEFAULT_RETRY_OPTIONS.maxRetryDelayInMs)
                : DEFAULT_RETRY_OPTIONS.retryDelayInMs,
            maxRetryDelayInMs: retryOptions.maxRetryDelayInMs && retryOptions.maxRetryDelayInMs >= 0
                ? retryOptions.maxRetryDelayInMs
                : DEFAULT_RETRY_OPTIONS.maxRetryDelayInMs,
            secondaryHost: retryOptions.secondaryHost
                ? retryOptions.secondaryHost
                : DEFAULT_RETRY_OPTIONS.secondaryHost
        };
        return _this;
    }
    /**
     * Sends request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof RetryPolicy
     */
    RetryPolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.attemptSendRequest(request, false, 1)];
            });
        });
    };
    /**
     * Decide and perform next retry. Won't mutate request parameter.
     *
     * @protected
     * @param {WebResource} request
     * @param {HttpOperationResponse} response
     * @param {boolean} secondaryHas404  If attempt was against the secondary & it returned a StatusNotFound (404), then
     *                                   the resource was not found. This may be due to replication delay. So, in this
     *                                   case, we'll never try the secondary again for this operation.
     * @param {number} attempt           How many retries has been attempted to performed, starting from 1, which includes
     *                                   the attempt will be performed by this method call.
     * @returns {Promise<HttpOperationResponse>}
     * @memberof RetryPolicy
     */
    RetryPolicy.prototype.attemptSendRequest = function (request, secondaryHas404, attempt) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var newRequest, isPrimaryRetry, response, err_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        newRequest = request.clone();
                        isPrimaryRetry = secondaryHas404 ||
                            !this.retryOptions.secondaryHost ||
                            !(request.method === "GET" || request.method === "HEAD" || request.method === "OPTIONS") ||
                            attempt % 2 === 1;
                        if (!isPrimaryRetry) {
                            newRequest.url = setURLHost(newRequest.url, this.retryOptions.secondaryHost);
                        }
                        // Set the server-side timeout query parameter "timeout=[seconds]"
                        if (this.retryOptions.tryTimeoutInMs) {
                            newRequest.url = setURLParameter(newRequest.url, URLConstants.Parameters.TIMEOUT, Math.floor(this.retryOptions.tryTimeoutInMs / 1000).toString());
                        }
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 3, , 4]);
                        this.logf(msRest.HttpPipelineLogLevel.INFO, "RetryPolicy: =====> Try=" + attempt + " " + (isPrimaryRetry ? "Primary" : "Secondary"));
                        return [4 /*yield*/, this._nextPolicy.sendRequest(newRequest)];
                    case 2:
                        response = _a.sent();
                        if (!this.shouldRetry(isPrimaryRetry, attempt, response)) {
                            return [2 /*return*/, response];
                        }
                        secondaryHas404 = secondaryHas404 || (!isPrimaryRetry && response.status === 404);
                        return [3 /*break*/, 4];
                    case 3:
                        err_1 = _a.sent();
                        this.logf(msRest.HttpPipelineLogLevel.ERROR, "RetryPolicy: Caught error, message: " + err_1.message + ", code: " + err_1.code);
                        if (!this.shouldRetry(isPrimaryRetry, attempt, response, err_1)) {
                            throw err_1;
                        }
                        return [3 /*break*/, 4];
                    case 4: return [4 /*yield*/, this.delay(isPrimaryRetry, attempt, request.abortSignal)];
                    case 5:
                        _a.sent();
                        return [4 /*yield*/, this.attemptSendRequest(request, secondaryHas404, ++attempt)];
                    case 6: return [2 /*return*/, _a.sent()];
                }
            });
        });
    };
    /**
     * Decide whether to retry according to last HTTP response and retry counters.
     *
     * @protected
     * @param {boolean} isPrimaryRetry
     * @param {number} attempt
     * @param {HttpOperationResponse} [response]
     * @param {RestError} [err]
     * @returns {boolean}
     * @memberof RetryPolicy
     */
    RetryPolicy.prototype.shouldRetry = function (isPrimaryRetry, attempt, response, err) {
        if (attempt >= this.retryOptions.maxTries) {
            this.logf(msRest.HttpPipelineLogLevel.INFO, "RetryPolicy: Attempt(s) " + attempt + " >= maxTries " + this.retryOptions
                .maxTries + ", no further try.");
            return false;
        }
        // Handle network failures, you may need to customize the list when you implement
        // your own http client
        var retriableErrors = [
            "ETIMEDOUT",
            "ESOCKETTIMEDOUT",
            "ECONNREFUSED",
            "ECONNRESET",
            "ENOENT",
            "ENOTFOUND",
            "TIMEOUT",
            "REQUEST_SEND_ERROR" // For default xhr based http client provided in ms-rest-js
        ];
        if (err) {
            for (var _i = 0, retriableErrors_1 = retriableErrors; _i < retriableErrors_1.length; _i++) {
                var retriableError = retriableErrors_1[_i];
                if (err.name.toUpperCase().includes(retriableError) ||
                    err.message.toUpperCase().includes(retriableError) ||
                    (err.code && err.code.toString().toUpperCase().includes(retriableError))) {
                    this.logf(msRest.HttpPipelineLogLevel.INFO, "RetryPolicy: Network error " + retriableError + " found, will retry.");
                    return true;
                }
            }
        }
        // If attempt was against the secondary & it returned a StatusNotFound (404), then
        // the resource was not found. This may be due to replication delay. So, in this
        // case, we'll never try the secondary again for this operation.
        if (response || err) {
            var statusCode = response ? response.status : err ? err.statusCode : 0;
            if (!isPrimaryRetry && statusCode === 404) {
                this.logf(msRest.HttpPipelineLogLevel.INFO, "RetryPolicy: Secondary access with 404, will retry.");
                return true;
            }
            // Server internal error or server timeout
            if (statusCode === 503 || statusCode === 500) {
                this.logf(msRest.HttpPipelineLogLevel.INFO, "RetryPolicy: Will retry for status code " + statusCode + ".");
                return true;
            }
        }
        return false;
    };
    /**
     * This is to log for debugging purposes only.
     * Comment/uncomment as necessary for releasing/debugging.
     *
     * @private
     * @param {HttpPipelineLogLevel} level
     * @param {string} message
     * @memberof RetryPolicy
     */
    // tslint:disable-next-line:variable-name
    RetryPolicy.prototype.logf = function (_level, _message) {
        // this.log(_level, _message);
    };
    /**
     * Delay a calculated time between retries.
     *
     * @private
     * @param {boolean} isPrimaryRetry
     * @param {number} attempt
     * @param {AbortSignalLike} [abortSignal]
     * @returns
     * @memberof RetryPolicy
     */
    RetryPolicy.prototype.delay = function (isPrimaryRetry, attempt, abortSignal) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var delayTimeInMs;
            return tslib_1.__generator(this, function (_a) {
                delayTimeInMs = 0;
                if (isPrimaryRetry) {
                    switch (this.retryOptions.retryPolicyType) {
                        case exports.RetryPolicyType.EXPONENTIAL:
                            delayTimeInMs = Math.min((Math.pow(2, attempt - 1) - 1) * this.retryOptions.retryDelayInMs, this.retryOptions.maxRetryDelayInMs);
                            break;
                        case exports.RetryPolicyType.FIXED:
                            delayTimeInMs = this.retryOptions.retryDelayInMs;
                            break;
                    }
                }
                else {
                    delayTimeInMs = Math.random() * 1000;
                }
                this.logf(msRest.HttpPipelineLogLevel.INFO, "RetryPolicy: Delay for " + delayTimeInMs + "ms");
                return [2 /*return*/, delay(delayTimeInMs, abortSignal, RETRY_ABORT_ERROR)];
            });
        });
    };
    return RetryPolicy;
}(msRest.BaseRequestPolicy));

/**
 * RetryPolicyFactory is a factory class helping generating RetryPolicy objects.
 *
 * @export
 * @class RetryPolicyFactory
 * @implements {RequestPolicyFactory}
 */
var RetryPolicyFactory = /** @class */ (function () {
    /**
     * Creates an instance of RetryPolicyFactory.
     * @param {IRetryOptions} [retryOptions]
     * @memberof RetryPolicyFactory
     */
    function RetryPolicyFactory(retryOptions) {
        this.retryOptions = retryOptions;
    }
    RetryPolicyFactory.prototype.create = function (nextPolicy, options) {
        return new RetryPolicy(nextPolicy, options, this.retryOptions);
    };
    return RetryPolicyFactory;
}());

/**
 * TelemetryPolicy is a policy used to tag user-agent header for every requests.
 *
 * @class TelemetryPolicy
 * @extends {BaseRequestPolicy}
 */
var TelemetryPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(TelemetryPolicy, _super);
    /**
     * Creates an instance of TelemetryPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @param {ITelemetryOptions} [telemetry]
     * @memberof TelemetryPolicy
     */
    function TelemetryPolicy(nextPolicy, options, telemetry) {
        var _this = _super.call(this, nextPolicy, options) || this;
        _this.telemetry = telemetry;
        return _this;
    }
    /**
     * Sends out request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof TelemetryPolicy
     */
    TelemetryPolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                {
                    if (!request.headers) {
                        request.headers = new msRest.HttpHeaders();
                    }
                    if (!request.headers.get(HeaderConstants.USER_AGENT)) {
                        request.headers.set(HeaderConstants.USER_AGENT, this.telemetry);
                    }
                }
                return [2 /*return*/, this._nextPolicy.sendRequest(request)];
            });
        });
    };
    return TelemetryPolicy;
}(msRest.BaseRequestPolicy));

/**
 * TelemetryPolicyFactory is a factory class helping generating TelemetryPolicy objects.
 *
 * @export
 * @class TelemetryPolicyFactory
 * @implements {RequestPolicyFactory}
 */
var TelemetryPolicyFactory = /** @class */ (function () {
    /**
     * Creates an instance of TelemetryPolicyFactory.
     * @param {ITelemetryOptions} [telemetry]
     * @memberof TelemetryPolicyFactory
     */
    function TelemetryPolicyFactory(telemetry) {
        var userAgentInfo = [];
        {
            if (telemetry) {
                var telemetryString = telemetry.value;
                if (telemetryString.length > 0 && userAgentInfo.indexOf(telemetryString) === -1) {
                    userAgentInfo.push(telemetryString);
                }
            }
            // e.g. Azure-Storage/10.0.0
            var libInfo = "Azure-Storage/" + SDK_VERSION;
            if (userAgentInfo.indexOf(libInfo) === -1) {
                userAgentInfo.push(libInfo);
            }
            // e.g. (NODE-VERSION 4.9.1; Windows_NT 10.0.16299)
            var runtimeInfo = "(NODE-VERSION " + process.version + "; " + os.type() + " " + os.release() + ")";
            if (userAgentInfo.indexOf(runtimeInfo) === -1) {
                userAgentInfo.push(runtimeInfo);
            }
        }
        this.telemetryString = userAgentInfo.join(" ");
    }
    TelemetryPolicyFactory.prototype.create = function (nextPolicy, options) {
        return new TelemetryPolicy(nextPolicy, options, this.telemetryString);
    };
    return TelemetryPolicyFactory;
}());

/**
 * UniqueRequestIDPolicy generates an UUID as x-ms-request-id header value.
 *
 * @class UniqueRequestIDPolicy
 * @extends {BaseRequestPolicy}
 */
var UniqueRequestIDPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(UniqueRequestIDPolicy, _super);
    /**
     * Creates an instance of UniqueRequestIDPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @memberof UniqueRequestIDPolicy
     */
    function UniqueRequestIDPolicy(nextPolicy, options) {
        return _super.call(this, nextPolicy, options) || this;
    }
    /**
     * Sends request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof UniqueRequestIDPolicy
     */
    UniqueRequestIDPolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!request.headers.contains(HeaderConstants.X_MS_CLIENT_REQUEST_ID)) {
                    request.headers.set(HeaderConstants.X_MS_CLIENT_REQUEST_ID, msRest.generateUuid());
                }
                return [2 /*return*/, this._nextPolicy.sendRequest(request)];
            });
        });
    };
    return UniqueRequestIDPolicy;
}(msRest.BaseRequestPolicy));

/**
 * UniqueRequestIDPolicyFactory is a factory class helping generating UniqueRequestIDPolicy objects.
 *
 * @export
 * @class UniqueRequestIDPolicyFactory
 * @implements {RequestPolicyFactory}
 */
var UniqueRequestIDPolicyFactory = /** @class */ (function () {
    function UniqueRequestIDPolicyFactory() {
    }
    UniqueRequestIDPolicyFactory.prototype.create = function (nextPolicy, options) {
        return new UniqueRequestIDPolicy(nextPolicy, options);
    };
    return UniqueRequestIDPolicyFactory;
}());

/**
 * Credential policy used to sign HTTP(S) requests before sending. This is an
 * abstract class.
 *
 * @export
 * @abstract
 * @class CredentialPolicy
 * @extends {BaseRequestPolicy}
 */
var CredentialPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(CredentialPolicy, _super);
    function CredentialPolicy() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Sends out request.
     *
     * @param {WebResource} request
     * @returns {Promise<HttpOperationResponse>}
     * @memberof CredentialPolicy
     */
    CredentialPolicy.prototype.sendRequest = function (request) {
        return this._nextPolicy.sendRequest(this.signRequest(request));
    };
    /**
     * Child classes must implement this method with request signing. This method
     * will be executed in sendRequest().
     *
     * @protected
     * @abstract
     * @param {WebResource} request
     * @returns {WebResource}
     * @memberof CredentialPolicy
     */
    CredentialPolicy.prototype.signRequest = function (request) {
        // Child classes must override this method with request signing. This method
        // will be executed in sendRequest().
        return request;
    };
    return CredentialPolicy;
}(msRest.BaseRequestPolicy));

/**
 * AnonymousCredentialPolicy is used with HTTP(S) requests that read public resources
 * or for use with Shared Access Signatures (SAS).
 *
 * @export
 * @class AnonymousCredentialPolicy
 * @extends {CredentialPolicy}
 */
var AnonymousCredentialPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(AnonymousCredentialPolicy, _super);
    /**
     * Creates an instance of AnonymousCredentialPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @memberof AnonymousCredentialPolicy
     */
    function AnonymousCredentialPolicy(nextPolicy, options) {
        return _super.call(this, nextPolicy, options) || this;
    }
    return AnonymousCredentialPolicy;
}(CredentialPolicy));

/**
 * AnonymousCredential provides a credentialPolicyCreator member used to create
 * AnonymousCredentialPolicy objects. AnonymousCredentialPolicy is used with
 * HTTP(S) requests that read public resources or for use with Shared Access
 * Signatures (SAS).
 *
 * @export
 * @class AnonymousCredential
 * @extends {Credential}
 */
var AnonymousCredential = /** @class */ (function (_super) {
    tslib_1.__extends(AnonymousCredential, _super);
    function AnonymousCredential() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Creates an AnonymousCredentialPolicy object.
     *
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @returns {AnonymousCredentialPolicy}
     * @memberof AnonymousCredential
     */
    AnonymousCredential.prototype.create = function (nextPolicy, options) {
        return new AnonymousCredentialPolicy(nextPolicy, options);
    };
    return AnonymousCredential;
}(Credential));

/**
 * A ServiceURL represents a based URL class for ServiceURL, ContainerURL and etc.
 *
 * @export
 * @class StorageURL
 */
var StorageURL = /** @class */ (function () {
    /**
     * Creates an instance of StorageURL.
     * @param {string} url
     * @param {Pipeline} pipeline
     * @memberof StorageURL
     */
    function StorageURL(url, pipeline) {
        // URL should be encoded and only once, protocol layer shouldn't encode URL again
        this.url = escapeURLPath(url);
        this.pipeline = pipeline;
        this.storageClientContext = new StorageClientContext(this.url, pipeline.toServiceClientOptions());
        this.isHttps = iEqual(getURLScheme(this.url) || "", "https");
        this.credential = new AnonymousCredential();
        for (var _i = 0, _a = this.pipeline.factories; _i < _a.length; _i++) {
            var factory = _a[_i];
            if (factory instanceof Credential) {
                this.credential = factory;
            }
        }
        // Override protocol layer's default content-type
        var storageClientContext = this.storageClientContext;
        storageClientContext.requestContentType = undefined;
    }
    /**
     * A static method used to create a new Pipeline object with Credential provided.
     *
     * @static
     * @param {Credential} credential Such as AnonymousCredential, SharedKeyCredential or TokenCredential.
     * @param {INewPipelineOptions} [pipelineOptions] Optional. Options.
     * @returns {Pipeline} A new Pipeline object.
     * @memberof Pipeline
     */
    StorageURL.newPipeline = function (credential, pipelineOptions) {
        if (pipelineOptions === void 0) { pipelineOptions = {}; }
        // Order is important. Closer to the API at the top & closer to the network at the bottom.
        // The credential's policy factory must appear close to the wire so it can sign any
        // changes made by other factories (like UniqueRequestIDPolicyFactory)
        var factories = [
            new KeepAlivePolicyFactory(pipelineOptions.keepAliveOptions),
            new TelemetryPolicyFactory(pipelineOptions.telemetry),
            new UniqueRequestIDPolicyFactory(),
            new BrowserPolicyFactory(),
            msRest.deserializationPolicy(),
            new RetryPolicyFactory(pipelineOptions.retryOptions),
            new LoggingPolicyFactory(),
            credential
        ];
        return new Pipeline(factories, {
            HTTPClient: pipelineOptions.httpClient,
            logger: pipelineOptions.logger
        });
    };
    return StorageURL;
}());

/**
 * A BlobURL represents a URL to an Azure Storage blob; the blob may be a block blob,
 * append blob, or page blob.
 *
 * @export
 * @class BlobURL
 * @extends {StorageURL}
 */
var BlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(BlobURL, _super);
    /**
     * Creates an instance of BlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to a blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blob".
     *                     You can append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof BlobURL
     */
    function BlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.blobContext = new Blob$1(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a BlobURL object from an ContainerURL object.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName A blob name
     * @returns
     * @memberof BlobURL
     */
    BlobURL.fromContainerURL = function (containerURL, blobName) {
        return new BlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a new BlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {BlobURL}
     * @memberof BlobURL
     */
    BlobURL.prototype.withPipeline = function (pipeline) {
        return new BlobURL(this.url, pipeline);
    };
    /**
     * Creates a new BlobURL object identical to the source but with the specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {BlobURL} A new BlobURL object identical to the source but with the specified snapshot timestamp
     * @memberof BlobURL
     */
    BlobURL.prototype.withSnapshot = function (snapshot) {
        return new BlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Reads or downloads a blob from the system, including its metadata and properties.
     * You can also call Get Blob to read a snapshot.
     *
     * * In Node.js, data returns in a Readable stream readableStreamBody
     * * In browsers, data returns in a promise blobBody
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset From which position of the blob to download, >= 0
     * @param {number} [count] How much data to be downloaded, > 0. Will download to the end when undefined
     * @param {IBlobDownloadOptions} [options]
     * @returns {Promise<Models.BlobDownloadResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.download = function (aborter, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var res;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        options.blobAccessConditions = options.blobAccessConditions || {};
                        options.blobAccessConditions.modifiedAccessConditions =
                            options.blobAccessConditions.modifiedAccessConditions || {};
                        ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                        return [4 /*yield*/, this.blobContext.download({
                                abortSignal: aborter,
                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                                modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                                onDownloadProgress: msRest.isNode ? undefined : options.progress,
                                range: offset === 0 && !count ? undefined : rangeToString({ offset: offset, count: count }),
                                rangeGetContentMD5: options.rangeGetContentMD5,
                                rangeGetContentCRC64: options.rangeGetContentCrc64,
                                snapshot: options.snapshot,
                                cpkInfo: options.customerProvidedKey
                            })];
                    case 1:
                        res = _a.sent();
                        // Return browser response immediately
                        if (!msRest.isNode) {
                            return [2 /*return*/, res];
                        }
                        // We support retrying when download stream unexpected ends in Node.js runtime
                        // Following code shouldn't be bundled into browser build, however some
                        // bundlers may try to bundle following code and "FileReadResponse.ts".
                        // In this case, "FileDownloadResponse.browser.ts" will be used as a shim of "FileDownloadResponse.ts"
                        // The config is in package.json "browser" field
                        if (options.maxRetryRequests === undefined || options.maxRetryRequests < 0) {
                            // TODO: Default value or make it a required parameter?
                            options.maxRetryRequests = DEFAULT_MAX_DOWNLOAD_RETRY_REQUESTS;
                        }
                        if (res.contentLength === undefined) {
                            throw new RangeError("File download response doesn't contain valid content length header");
                        }
                        if (!res.eTag) {
                            throw new RangeError("File download response doesn't contain valid etag header");
                        }
                        return [2 /*return*/, new BlobDownloadResponse(aborter, res, function (start) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                var updatedOptions;
                                return tslib_1.__generator(this, function (_a) {
                                    switch (_a.label) {
                                        case 0:
                                            updatedOptions = {
                                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                                                modifiedAccessConditions: {
                                                    ifMatch: options.blobAccessConditions.modifiedAccessConditions.ifMatch || res.eTag,
                                                    ifModifiedSince: options.blobAccessConditions.modifiedAccessConditions
                                                        .ifModifiedSince,
                                                    ifNoneMatch: options.blobAccessConditions.modifiedAccessConditions.ifNoneMatch,
                                                    ifUnmodifiedSince: options.blobAccessConditions.modifiedAccessConditions
                                                        .ifUnmodifiedSince
                                                },
                                                range: rangeToString({
                                                    count: offset + res.contentLength - start,
                                                    offset: start
                                                }),
                                                rangeGetContentMD5: options.rangeGetContentMD5,
                                                rangeGetContentCRC64: options.rangeGetContentCrc64,
                                                snapshot: options.snapshot,
                                                cpkInfo: options.customerProvidedKey
                                            };
                                            return [4 /*yield*/, this.blobContext.download(tslib_1.__assign({ abortSignal: aborter }, updatedOptions))];
                                        case 1: 
                                        // Debug purpose only
                                        // console.log(
                                        //   `Read from internal stream, range: ${
                                        //     updatedOptions.range
                                        //   }, options: ${JSON.stringify(updatedOptions)}`
                                        // );
                                        return [2 /*return*/, (_a.sent()).readableStreamBody];
                                    }
                                });
                            }); }, offset, res.contentLength, {
                                maxRetryRequests: options.maxRetryRequests,
                                progress: options.progress
                            })];
                }
            });
        });
    };
    /**
     * Returns all user-defined metadata, standard HTTP properties, and system properties
     * for the blob. It does not return the content of the blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobGetPropertiesOptions} [options]
     * @returns {Promise<Models.BlobGetPropertiesResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.getProperties = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.getProperties({
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Marks the specified blob or snapshot for deletion. The blob is later deleted
     * during garbage collection. Note that in order to delete a blob, you must delete
     * all of its snapshots. You can delete both at the same time with the Delete
     * Blob operation.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/delete-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobDeleteOptions} [options]
     * @returns {Promise<Models.BlobDeleteResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.delete = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                return [2 /*return*/, this.blobContext.deleteMethod({
                        abortSignal: aborter,
                        deleteSnapshots: options.deleteSnapshots,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Restores the contents and metadata of soft deleted blob and any associated
     * soft deleted snapshots. Undelete Blob is supported only on version 2017-07-29
     * or later.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/undelete-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.BlobUndeleteResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.undelete = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.undelete({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Sets system properties on the blob.
     *
     * If no value provided, or no value provided for the specificed blob HTTP headers,
     * these blob HTTP headers without a value will be cleared.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.BlobHTTPHeaders} [blobHTTPHeaders] If no value provided, or no value provided for
     *                                                   the specificed blob HTTP headers, these blob HTTP
     *                                                   headers without a value will be cleared.
     * @param {IBlobSetHTTPHeadersOptions} [options]
     * @returns {Promise<Models.BlobSetHTTPHeadersResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.setHTTPHeaders = function (aborter, blobHTTPHeaders, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.setHTTPHeaders({
                        abortSignal: aborter,
                        blobHTTPHeaders: blobHTTPHeaders,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Sets user-defined metadata for the specified blob as one or more name-value pairs.
     *
     * If no option provided, or no metadata defined in the parameter, the blob
     * metadata will be removed.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-metadata
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IMetadata} [metadata] Replace existing metadata with this value.
     *                               If no value provided the existing metadata will be removed.
     * @param {IBlobSetMetadataOptions} [options]
     * @returns {Promise<Models.BlobSetMetadataResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.setMetadata = function (aborter, metadata, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.setMetadata({
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        metadata: metadata,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Establishes and manages a lock on a blob for write and delete operations.
     * The lock duration can be 15 to 60 seconds, or can be infinite.
     * In versions prior to 2012-02-12, the lock duration is 60 seconds.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} proposedLeaseId Can be specified in any valid GUID string format
     * @param {number} duration The lock duration can be 15 to 60 seconds, or can be infinite
     * @param {IBlobAcquireLeaseOptions} [options]
     * @returns {Promise<Models.BlobAcquireLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.acquireLease = function (aborter, proposedLeaseId, duration, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.acquireLease({
                        abortSignal: aborter,
                        duration: duration,
                        modifiedAccessConditions: options.modifiedAccessConditions,
                        proposedLeaseId: proposedLeaseId
                    })];
            });
        });
    };
    /**
     * To free the lease if it is no longer needed so that another client may immediately
     * acquire a lease against the blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IBlobReleaseLeaseOptions} [options]
     * @returns {Promise<Models.BlobReleaseLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.releaseLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.releaseLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To renew an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IBlobRenewLeaseOptions} [options]
     * @returns {Promise<Models.BlobRenewLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.renewLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.renewLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To change the ID of an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {string} proposedLeaseId
     * @param {IBlobChangeLeaseOptions} [options]
     * @returns {Promise<Models.BlobChangeLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.changeLease = function (aborter, leaseId, proposedLeaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.changeLease(leaseId, proposedLeaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To end the lease but ensure that another client cannot acquire a new lease
     * until the current lease period has expired.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} [breakPeriod]
     * @param {IBlobBreakLeaseOptions} [options]
     * @returns {Promise<Models.BlobBreakLeaseResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.breakLease = function (aborter, breakPeriod, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.breakLease({
                        abortSignal: aborter,
                        breakPeriod: breakPeriod,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Creates a read-only snapshot of a blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/snapshot-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobCreateSnapshotOptions} [options]
     * @returns {Promise<Models.BlobCreateSnapshotResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.createSnapshot = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blobContext.createSnapshot({
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Asynchronously copies a blob to a destination within the storage account.
     * In version 2012-02-12 and later, the source for a Copy Blob operation can be
     * a committed blob in any Azure storage account.
     * Beginning with version 2015-02-21, the source for a Copy Blob operation can be
     * an Azure file in any Azure storage account.
     * Only storage accounts created on or after June 7th, 2012 allow the Copy Blob
     * operation to copy from another storage account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/copy-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource
     * @param {IBlobStartCopyFromURLOptions} [options]
     * @returns {Promise<Models.BlobStartCopyFromURLResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.startCopyFromURL = function (aborter, copySource, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                return [2 /*return*/, this.blobContext.startCopyFromURL(copySource, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        },
                        rehydratePriority: options.rehydratePriority,
                        tier: toAccessTier(options.tier)
                    })];
            });
        });
    };
    /**
     * Aborts a pending asynchronous Copy Blob operation, and leaves a destination blob with zero
     * length and full metadata. Version 2012-02-12 and newer.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/abort-copy-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copyId
     * @param {IBlobAbortCopyFromURLOptions} [options]
     * @returns {Promise<Models.BlobAbortCopyFromURLResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.abortCopyFromURL = function (aborter, copyId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.blobContext.abortCopyFromURL(copyId, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.leaseAccessConditions
                    })];
            });
        });
    };
    /**
     * The synchronous Copy From URL operation copies a blob or an internet resource to a new blob. It will not
     * return a response until the copy is complete.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/copy-blob-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource The source URL to copy from, Shared Access Signature(SAS) maybe needed for authentication
     * @param {IBlobSyncCopyFromURLOptions} [options={}]
     * @returns {Promise<Models.BlobCopyFromURLResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.syncCopyFromURL = function (aborter, copySource, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.blobAccessConditions = options.blobAccessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                return [2 /*return*/, this.blobContext.copyFromURL(copySource, {
                        abortSignal: aborter,
                        metadata: options.metadata,
                        leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.blobAccessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        }
                    })];
            });
        });
    };
    /**
     * Sets the tier on a blob. The operation is allowed on a page blob in a premium
     * storage account and on a block blob in a blob storage account (locally redundant
     * storage only). A premium page blob's tier determines the allowed size, IOPS,
     * and bandwidth of the blob. A block blob's tier determines Hot/Cool/Archive
     * storage type. This operation does not update the blob's ETag.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-tier
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {BlockBlobTier | PremiumPageBlobTier | string} tier
     * @param {IBlobSetTierOptions} [options]
     * @returns {Promise<Models.BlobsSetTierResponse>}
     * @memberof BlobURL
     */
    BlobURL.prototype.setTier = function (aborter, tier, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.blobContext.setTier(toAccessTier(tier), {
                            abortSignal: aborter,
                            leaseAccessConditions: options.leaseAccessConditions,
                            rehydratePriority: options.rehydratePriority
                        })];
                    case 1: return [2 /*return*/, _a.sent()];
                }
            });
        });
    };
    return BlobURL;
}(StorageURL));

/**
 * AppendBlobURL defines a set of operations applicable to append blobs.
 *
 * @export
 * @class AppendBlobURL
 * @extends {StorageURL}
 */
var AppendBlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(AppendBlobURL, _super);
    /**
     * Creates an instance of AppendBlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to an append blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage append blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/appendblob". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/appendblob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof AppendBlobURL
     */
    function AppendBlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.appendBlobContext = new AppendBlob(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a AppendBlobURL object from ContainerURL instance.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName An append blob name
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.fromContainerURL = function (containerURL, blobName) {
        return new AppendBlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a AppendBlobURL object from BlobURL instance.
     *
     * @static
     * @param {BlobURL} blobURL
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.fromBlobURL = function (blobURL) {
        return new AppendBlobURL(blobURL.url, blobURL.pipeline);
    };
    /**
     * Creates a new AppendBlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.withPipeline = function (pipeline) {
        return new AppendBlobURL(this.url, pipeline);
    };
    /**
     * Creates a new AppendBlobURL object identical to the source but with the
     * specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.withSnapshot = function (snapshot) {
        return new AppendBlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Creates a 0-length append blob. Call AppendBlock to append data to an append blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IAppendBlobCreateOptions} [options]
     * @returns {Promise<Models.AppendBlobsCreateResponse>}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.create = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.appendBlobContext.create(0, {
                        abortSignal: aborter,
                        blobHTTPHeaders: options.blobHTTPHeaders,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Commits a new block of data to the end of the existing append blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/append-block
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {HttpRequestBody} body
     * @param {number} contentLength Length of the body in bytes
     * @param {IAppendBlobAppendBlockOptions} [options]
     * @returns {Promise<Models.AppendBlobsAppendBlockResponse>}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.appendBlock = function (aborter, body, contentLength, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.appendBlobContext.appendBlock(body, contentLength, {
                        abortSignal: aborter,
                        appendPositionAccessConditions: options.accessConditions.appendPositionAccessConditions,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        onUploadProgress: options.progress,
                        transactionalContentMD5: options.transactionalContentMD5,
                        transactionalContentCrc64: options.transactionalContentCrc64,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * The Append Block operation commits a new block of data to the end of an existing append blob
     * where the contents are read from a source url.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/append-block-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} sourceURL
     *                 The url to the blob that will be the source of the copy. A source blob in the same storage account can
     *                 be authenticated via Shared Key. However, if the source is a blob in another account, the source blob
     *                 must either be public or must be authenticated via a shared access signature. If the source blob is
     *                 public, no authentication is required to perform the operation.
     * @param {number} sourceOffset Offset in source to be appended
     * @param {number} count Number of bytes to be appended as a block
     * @param {IAppendBlobAppendBlockFromURLOptions} [options={}]
     * @returns {Promise<Models.AppendBlobAppendBlockFromUrlResponse>}
     * @memberof AppendBlobURL
     */
    AppendBlobURL.prototype.appendBlockFromURL = function (aborter, sourceURL, sourceOffset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.appendBlobContext.appendBlockFromUrl(sourceURL, 0, {
                        abortSignal: aborter,
                        sourceRange: rangeToString({ offset: sourceOffset, count: count }),
                        sourceContentMD5: options.sourceContentMD5,
                        sourceContentCrc64: options.sourceContentCrc64,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        appendPositionAccessConditions: options.accessConditions.appendPositionAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        },
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    return AppendBlobURL;
}(BlobURL));

var MutexLockStatus;
(function (MutexLockStatus) {
    MutexLockStatus[MutexLockStatus["LOCKED"] = 0] = "LOCKED";
    MutexLockStatus[MutexLockStatus["UNLOCKED"] = 1] = "UNLOCKED";
})(MutexLockStatus || (MutexLockStatus = {}));
/**
 * An async mutex lock.
 *
 * @export
 * @class Mutex
 */
var Mutex = /** @class */ (function () {
    function Mutex() {
    }
    /**
     * Lock for a specific key. If the lock has been acquired by another customer, then
     * will wait until getting the lock.
     *
     * @static
     * @param {string} key lock key
     * @returns {Promise<void>}
     * @memberof Mutex
     */
    Mutex.lock = function (key) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, new Promise(function (resolve) {
                        if (_this.keys[key] === undefined || _this.keys[key] === MutexLockStatus.UNLOCKED) {
                            _this.keys[key] = MutexLockStatus.LOCKED;
                            resolve();
                        }
                        else {
                            _this.onUnlockEvent(key, function () {
                                _this.keys[key] = MutexLockStatus.LOCKED;
                                resolve();
                            });
                        }
                    })];
            });
        });
    };
    /**
     * Unlock a key.
     *
     * @static
     * @param {string} key
     * @returns {Promise<void>}
     * @memberof Mutex
     */
    Mutex.unlock = function (key) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, new Promise(function (resolve) {
                        if (_this.keys[key] === MutexLockStatus.LOCKED) {
                            _this.emitUnlockEvent(key);
                        }
                        delete _this.keys[key];
                        resolve();
                    })];
            });
        });
    };
    Mutex.onUnlockEvent = function (key, handler) {
        if (this.listeners[key] === undefined) {
            this.listeners[key] = [handler];
        }
        else {
            this.listeners[key].push(handler);
        }
    };
    Mutex.emitUnlockEvent = function (key) {
        var _this = this;
        if (this.listeners[key] !== undefined && this.listeners[key].length > 0) {
            var handler_1 = this.listeners[key].shift();
            setImmediate(function () {
                handler_1.call(_this);
            });
        }
    };
    Mutex.keys = {};
    Mutex.listeners = {};
    return Mutex;
}());

/**
 * A BatchRequest represents a based class for BatchDeleteRequest and BatchSetTierRequest.
 *
 * @export
 * @class BatchRequest
 */
var BatchRequest = /** @class */ (function () {
    function BatchRequest() {
        this.batch = "batch";
        this.batchRequest = new InnerBatchRequest();
    }
    /**
     * Get the value of Content-Type for a batch request.
     * The value must be multipart/mixed with a batch boundary.
     * Example: multipart/mixed; boundary=batch_a81786c8-e301-4e42-a729-a32ca24ae252
     */
    BatchRequest.prototype.getMultiPartContentType = function () {
        return this.batchRequest.getMultipartContentType();
    };
    /**
     * Get assembled HTTP request body for sub requests.
     */
    BatchRequest.prototype.getHttpRequestBody = function () {
        return this.batchRequest.getHttpRequestBody();
    };
    /**
     * Get sub requests that are added into the batch request.
     */
    BatchRequest.prototype.getSubRequests = function () {
        return this.batchRequest.getSubRequests();
    };
    BatchRequest.prototype.addSubRequestInternal = function (subRequest, assembleSubRequestFunc) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, Mutex.lock(this.batch)];
                    case 1:
                        _a.sent();
                        _a.label = 2;
                    case 2:
                        _a.trys.push([2, , 4, 6]);
                        this.batchRequest.preAddSubRequest(subRequest);
                        return [4 /*yield*/, assembleSubRequestFunc()];
                    case 3:
                        _a.sent();
                        this.batchRequest.postAddSubRequest(subRequest);
                        return [3 /*break*/, 6];
                    case 4: return [4 /*yield*/, Mutex.unlock(this.batch)];
                    case 5:
                        _a.sent();
                        return [7 /*endfinally*/];
                    case 6: return [2 /*return*/];
                }
            });
        });
    };
    return BatchRequest;
}());
/**
 * A BatchDeleteRequest represents a batch delete request, which consists of one or more delete operations.
 *
 * @export
 * @class BatchDeleteRequest
 * @extends {BatchRequest}
 */
var BatchDeleteRequest = /** @class */ (function (_super) {
    tslib_1.__extends(BatchDeleteRequest, _super);
    function BatchDeleteRequest() {
        return _super.call(this) || this;
    }
    BatchDeleteRequest.prototype.addSubRequest = function (urlOrBlobURL, credentialOrOptions, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var url, credential;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (typeof urlOrBlobURL === 'string' && credentialOrOptions instanceof Credential) {
                            // First overload
                            url = urlOrBlobURL;
                            credential = credentialOrOptions;
                        }
                        else if (urlOrBlobURL instanceof BlobURL) {
                            // Second overload
                            url = urlOrBlobURL.url;
                            credential = urlOrBlobURL.credential;
                            options = credentialOrOptions;
                        }
                        else {
                            throw new RangeError("Invalid arguments. Either url and credential, or BlobURL need be provided.");
                        }
                        if (!options) {
                            options = {};
                        }
                        return [4 /*yield*/, _super.prototype.addSubRequestInternal.call(this, {
                                url: url,
                                credential: credential
                            }, function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                return tslib_1.__generator(this, function (_a) {
                                    switch (_a.label) {
                                        case 0: return [4 /*yield*/, new BlobURL(url, this.batchRequest.createPipeline(credential)).delete(Aborter.none, options)];
                                        case 1:
                                            _a.sent();
                                            return [2 /*return*/];
                                    }
                                });
                            }); })];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    return BatchDeleteRequest;
}(BatchRequest));
/**
 * A BatchSetTierRequest represents a batch set tier request, which consists of one or more set tier operations.
 *
 * @export
 * @class BatchSetTierRequest
 * @extends {BatchRequest}
 */
var BatchSetTierRequest = /** @class */ (function (_super) {
    tslib_1.__extends(BatchSetTierRequest, _super);
    function BatchSetTierRequest() {
        return _super.call(this) || this;
    }
    BatchSetTierRequest.prototype.addSubRequest = function (urlOrBlobURL, credentialOrTier, tierOrOptions, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var url, credential, tier;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (typeof urlOrBlobURL === 'string' && credentialOrTier instanceof Credential) {
                            // First overload
                            url = urlOrBlobURL;
                            credential = credentialOrTier;
                            tier = tierOrOptions;
                        }
                        else if (urlOrBlobURL instanceof BlobURL) {
                            // Second overload
                            url = urlOrBlobURL.url;
                            credential = urlOrBlobURL.credential;
                            tier = credentialOrTier;
                            options = tierOrOptions;
                        }
                        else {
                            throw new RangeError("Invalid arguments. Either url and credential, or BlobURL need be provided.");
                        }
                        if (!options) {
                            options = {};
                        }
                        return [4 /*yield*/, _super.prototype.addSubRequestInternal.call(this, {
                                url: url,
                                credential: credential
                            }, function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                return tslib_1.__generator(this, function (_a) {
                                    switch (_a.label) {
                                        case 0: return [4 /*yield*/, new BlobURL(url, this.batchRequest.createPipeline(credential)).setTier(Aborter.none, tier, options)];
                                        case 1:
                                            _a.sent();
                                            return [2 /*return*/];
                                    }
                                });
                            }); })];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    return BatchSetTierRequest;
}(BatchRequest));
/**
 * Inner batch request class which is responsible for assembling and serializing sub requests.
 * See https://docs.microsoft.com/en-us/rest/api/storageservices/blob-batch#request-body for how request get assembled.
 */
var InnerBatchRequest = /** @class */ (function () {
    function InnerBatchRequest() {
        this.operationCount = 0;
        this.body = "";
        var tempGuid = msRest.generateUuid();
        // batch_{batchid}
        this.boundary = "batch_" + tempGuid;
        // --batch_{batchid}
        // Content-Type: application/http
        // Content-Transfer-Encoding: binary
        this.subRequestPrefix = "--" + this.boundary + HTTP_LINE_ENDING + HeaderConstants.CONTENT_TYPE + ": application/http" + HTTP_LINE_ENDING + HeaderConstants.CONTENT_TRANSFER_ENCODING + ": binary";
        // multipart/mixed; boundary=batch_{batchid}
        this.multipartContentType = "multipart/mixed; boundary=" + this.boundary;
        // --batch_{batchid}--
        this.batchRequestEnding = "--" + this.boundary + "--";
        this.subRequests = new Map();
    }
    /**
     * Create pipeline to assemble sub requests. The idea here is to use exising
     * credential and serialization/deserialization components, with additional policies to
     * filter unnecessary headers, assemble sub requests into request's body
     * and intercept request from going to wire.
     * @param credential
     */
    InnerBatchRequest.prototype.createPipeline = function (credential) {
        var isAnonymousCreds = credential instanceof AnonymousCredential;
        var policyFactoryLength = 3 + (isAnonymousCreds ? 0 : 1); // [deserilizationPolicy, BatchHeaderFilterPolicyFactory, (Optional)Credential, BatchRequestAssemblePolicyFactory]
        var factories = new Array(policyFactoryLength);
        factories[0] = msRest.deserializationPolicy(); // Default deserializationPolicy is provided by protocol layer
        factories[1] = new BatchHeaderFilterPolicyFactory(); // Use batch header filter policy to exclude unnecessary headers
        if (!isAnonymousCreds) {
            factories[2] = credential;
        }
        factories[policyFactoryLength - 1] = new BatchRequestAssemblePolicyFactory(this); // Use batch assemble policy to assemble request and intercept request from going to wire
        return new Pipeline(factories, {});
    };
    InnerBatchRequest.prototype.appendSubRequestToBody = function (request) {
        // Start to assemble sub request
        this.body +=
            [
                this.subRequestPrefix,
                HeaderConstants.CONTENT_ID + ": " + this.operationCount,
                "",
                request.method.toString() + " " + getURLPathAndQuery(request.url) + " " + HTTP_VERSION_1_1 + HTTP_LINE_ENDING // sub request start line with method
            ].join(HTTP_LINE_ENDING);
        for (var _i = 0, _a = request.headers.headersArray(); _i < _a.length; _i++) {
            var header = _a[_i];
            this.body += header.name + ": " + header.value + HTTP_LINE_ENDING;
        }
        this.body += HTTP_LINE_ENDING; // sub request's headers need be ending with an empty line
        // No body to assemble for current batch request support
        // End to assemble sub request
    };
    InnerBatchRequest.prototype.preAddSubRequest = function (subRequest) {
        if (this.operationCount >= BATCH_MAX_REQUEST) {
            throw new RangeError("Cannot exceed " + BATCH_MAX_REQUEST + " sub requests in a single batch");
        }
        // Fast fail if url for sub request is invalid
        var path = getURLPath(subRequest.url);
        if (!path || path == "") {
            throw new RangeError("Invalid url for sub request: '" + subRequest.url + "'");
        }
    };
    InnerBatchRequest.prototype.postAddSubRequest = function (subRequest) {
        this.subRequests.set(this.operationCount, subRequest);
        this.operationCount++;
    };
    // Return the http request body with assembling the ending line to the sub request body.
    InnerBatchRequest.prototype.getHttpRequestBody = function () {
        return "" + this.body + this.batchRequestEnding + HTTP_LINE_ENDING;
    };
    InnerBatchRequest.prototype.getMultipartContentType = function () {
        return this.multipartContentType;
    };
    InnerBatchRequest.prototype.getSubRequests = function () {
        return this.subRequests;
    };
    return InnerBatchRequest;
}());
var BatchRequestAssemblePolicy = /** @class */ (function (_super) {
    tslib_1.__extends(BatchRequestAssemblePolicy, _super);
    function BatchRequestAssemblePolicy(batchRequest, nextPolicy, options) {
        var _this = _super.call(this, nextPolicy, options) || this;
        _this.dummyResponse = {
            request: new msRest.WebResource(),
            status: 200,
            headers: new msRest.HttpHeaders()
        };
        _this.batchRequest = batchRequest;
        return _this;
    }
    BatchRequestAssemblePolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.batchRequest.appendSubRequestToBody(request)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/, this.dummyResponse]; // Intercept request from going to wire
                }
            });
        });
    };
    return BatchRequestAssemblePolicy;
}(msRest.BaseRequestPolicy));
var BatchRequestAssemblePolicyFactory = /** @class */ (function () {
    function BatchRequestAssemblePolicyFactory(batchRequest) {
        this.batchRequest = batchRequest;
    }
    BatchRequestAssemblePolicyFactory.prototype.create = function (nextPolicy, options) {
        return new BatchRequestAssemblePolicy(this.batchRequest, nextPolicy, options);
    };
    return BatchRequestAssemblePolicyFactory;
}());
var BatchHeaderFilterPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(BatchHeaderFilterPolicy, _super);
    function BatchHeaderFilterPolicy(nextPolicy, options) {
        return _super.call(this, nextPolicy, options) || this;
    }
    BatchHeaderFilterPolicy.prototype.sendRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var xMsHeaderName, _i, _a, header;
            return tslib_1.__generator(this, function (_b) {
                xMsHeaderName = "";
                for (_i = 0, _a = request.headers.headersArray(); _i < _a.length; _i++) {
                    header = _a[_i];
                    if (iEqual(header.name, HeaderConstants.X_MS_VERSION)) {
                        xMsHeaderName = header.name;
                    }
                }
                if (xMsHeaderName !== "") {
                    request.headers.remove(xMsHeaderName); // The subrequests should not have the x-ms-version header.
                }
                return [2 /*return*/, this._nextPolicy.sendRequest(request)];
            });
        });
    };
    return BatchHeaderFilterPolicy;
}(msRest.BaseRequestPolicy));
var BatchHeaderFilterPolicyFactory = /** @class */ (function () {
    function BatchHeaderFilterPolicyFactory() {
    }
    BatchHeaderFilterPolicyFactory.prototype.create = function (nextPolicy, options) {
        return new BatchHeaderFilterPolicy(nextPolicy, options);
    };
    return BatchHeaderFilterPolicyFactory;
}());

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS to a blob. Setting
 * a value to true means that any SAS which uses these permissions will grant permissions for that operation. Once all
 * the values are set, this should be serialized with toString and set as the permissions field on a
 * {@link IBlobSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 *
 * @export
 * @class BlobSASPermissions
 */
var BlobSASPermissions = /** @class */ (function () {
    function BlobSASPermissions() {
        /**
         * Specifies Read access granted.
         *
         * @type {boolean}
         * @memberof BlobSASPermissions
         */
        this.read = false;
        /**
         * Specifies Add access granted.
         *
         * @type {boolean}
         * @memberof BlobSASPermissions
         */
        this.add = false;
        /**
         * Specifies Create access granted.
         *
         * @type {boolean}
         * @memberof BlobSASPermissions
         */
        this.create = false;
        /**
         * Specifies Write access granted.
         *
         * @type {boolean}
         * @memberof BlobSASPermissions
         */
        this.write = false;
        /**
         * Specifies Delete access granted.
         *
         * @type {boolean}
         * @memberof BlobSASPermissions
         */
        this.delete = false;
    }
    /**
     * Creates a {@link BlobSASPermissions} from the specified permissions string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid permission.
     *
     * @static
     * @param {string} permissions
     * @returns {BlobSASPermissions}
     * @memberof BlobSASPermissions
     */
    BlobSASPermissions.parse = function (permissions) {
        var blobSASPermissions = new BlobSASPermissions();
        for (var _i = 0, permissions_1 = permissions; _i < permissions_1.length; _i++) {
            var char = permissions_1[_i];
            switch (char) {
                case "r":
                    blobSASPermissions.read = true;
                    break;
                case "a":
                    blobSASPermissions.add = true;
                    break;
                case "c":
                    blobSASPermissions.create = true;
                    break;
                case "w":
                    blobSASPermissions.write = true;
                    break;
                case "d":
                    blobSASPermissions.delete = true;
                    break;
                default:
                    throw new RangeError("Invalid permission: " + char);
            }
        }
        return blobSASPermissions;
    };
    /**
     * Converts the given permissions to a string. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * @returns {string} A string which represents the BlobSASPermissions
     * @memberof BlobSASPermissions
     */
    BlobSASPermissions.prototype.toString = function () {
        var permissions = [];
        if (this.read) {
            permissions.push("r");
        }
        if (this.add) {
            permissions.push("a");
        }
        if (this.create) {
            permissions.push("c");
        }
        if (this.write) {
            permissions.push("w");
        }
        if (this.delete) {
            permissions.push("d");
        }
        return permissions.join("");
    };
    return BlobSASPermissions;
}());

/**
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS to a container.
 * Setting a value to true means that any SAS which uses these permissions will grant permissions for that operation.
 * Once all the values are set, this should be serialized with toString and set as the permissions field on a
 * {@link IBlobSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 *
 * @export
 * @class ContainerSASPermissions
 */
var ContainerSASPermissions = /** @class */ (function () {
    function ContainerSASPermissions() {
        /**
         * Specifies Read access granted.
         *
         * @type {boolean}
         * @memberof ContainerSASPermissions
         */
        this.read = false;
        /**
         * Specifies Add access granted.
         *
         * @type {boolean}
         * @memberof ContainerSASPermissions
         */
        this.add = false;
        /**
         * Specifies Create access granted.
         *
         * @type {boolean}
         * @memberof ContainerSASPermissions
         */
        this.create = false;
        /**
         * Specifies Write access granted.
         *
         * @type {boolean}
         * @memberof ContainerSASPermissions
         */
        this.write = false;
        /**
         * Specifies Delete access granted.
         *
         * @type {boolean}
         * @memberof ContainerSASPermissions
         */
        this.delete = false;
        /**
         * Specifies List access granted.
         *
         * @type {boolean}
         * @memberof ContainerSASPermissions
         */
        this.list = false;
    }
    /**
     * Creates an {@link ContainerSASPermissions} from the specified permissions string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid permission.
     *
     * @static
     * @param {string} permissions
     * @returns
     * @memberof ContainerSASPermissions
     */
    ContainerSASPermissions.parse = function (permissions) {
        var containerSASPermissions = new ContainerSASPermissions();
        for (var _i = 0, permissions_1 = permissions; _i < permissions_1.length; _i++) {
            var char = permissions_1[_i];
            switch (char) {
                case "r":
                    containerSASPermissions.read = true;
                    break;
                case "a":
                    containerSASPermissions.add = true;
                    break;
                case "c":
                    containerSASPermissions.create = true;
                    break;
                case "w":
                    containerSASPermissions.write = true;
                    break;
                case "d":
                    containerSASPermissions.delete = true;
                    break;
                case "l":
                    containerSASPermissions.list = true;
                    break;
                default:
                    throw new RangeError("Invalid permission " + char);
            }
        }
        return containerSASPermissions;
    };
    /**
     * Converts the given permissions to a string. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * The order of the characters should be as specified here to ensure correctness.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @returns {string}
     * @memberof ContainerSASPermissions
     */
    ContainerSASPermissions.prototype.toString = function () {
        var permissions = [];
        if (this.read) {
            permissions.push("r");
        }
        if (this.add) {
            permissions.push("a");
        }
        if (this.create) {
            permissions.push("c");
        }
        if (this.write) {
            permissions.push("w");
        }
        if (this.delete) {
            permissions.push("d");
        }
        if (this.list) {
            permissions.push("l");
        }
        return permissions.join("");
    };
    return ContainerSASPermissions;
}());

/**
 * SharedKeyCredentialPolicy is a policy used to sign HTTP request with a shared key.
 *
 * @export
 * @class SharedKeyCredentialPolicy
 * @extends {CredentialPolicy}
 */
var SharedKeyCredentialPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(SharedKeyCredentialPolicy, _super);
    /**
     * Creates an instance of SharedKeyCredentialPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @param {SharedKeyCredential} factory
     * @memberof SharedKeyCredentialPolicy
     */
    function SharedKeyCredentialPolicy(nextPolicy, options, factory) {
        var _this = _super.call(this, nextPolicy, options) || this;
        _this.factory = factory;
        return _this;
    }
    /**
     * Signs request.
     *
     * @protected
     * @param {WebResource} request
     * @returns {WebResource}
     * @memberof SharedKeyCredentialPolicy
     */
    SharedKeyCredentialPolicy.prototype.signRequest = function (request) {
        request.headers.set(HeaderConstants.X_MS_DATE, new Date().toUTCString());
        if (request.body && typeof request.body === "string" && request.body.length > 0) {
            request.headers.set(HeaderConstants.CONTENT_LENGTH, Buffer.byteLength(request.body));
        }
        var stringToSign = [
            request.method.toUpperCase(),
            this.getHeaderValueToSign(request, HeaderConstants.CONTENT_LANGUAGE),
            this.getHeaderValueToSign(request, HeaderConstants.CONTENT_ENCODING),
            this.getHeaderValueToSign(request, HeaderConstants.CONTENT_LENGTH),
            this.getHeaderValueToSign(request, HeaderConstants.CONTENT_MD5),
            this.getHeaderValueToSign(request, HeaderConstants.CONTENT_TYPE),
            this.getHeaderValueToSign(request, HeaderConstants.DATE),
            this.getHeaderValueToSign(request, HeaderConstants.IF_MODIFIED_SINCE),
            this.getHeaderValueToSign(request, HeaderConstants.IF_MATCH),
            this.getHeaderValueToSign(request, HeaderConstants.IF_NONE_MATCH),
            this.getHeaderValueToSign(request, HeaderConstants.IF_UNMODIFIED_SINCE),
            this.getHeaderValueToSign(request, HeaderConstants.RANGE)
        ].join("\n") +
            "\n" +
            this.getCanonicalizedHeadersString(request) +
            this.getCanonicalizedResourceString(request);
        var signature = this.factory.computeHMACSHA256(stringToSign);
        request.headers.set(HeaderConstants.AUTHORIZATION, "SharedKey " + this.factory.accountName + ":" + signature);
        // Workaround for https://github.com/axios/axios/issues/2107
        // We should always keep the 'content-length' header once the issue is solved
        // For a better explanation about this workaround, look here: https://github.com/Azure/azure-sdk-for-js/pull/3273
        if (typeof request.body !== "function" && !(request.body && request.onUploadProgress)) {
            request.headers.remove(HeaderConstants.CONTENT_LENGTH);
        }
        // console.log(`[URL]:${request.url}`);
        // console.log(`[HEADERS]:${request.headers.toString()}`);
        // console.log(`[STRING TO SIGN]:${JSON.stringify(stringToSign)}`);
        // console.log(`[KEY]: ${request.headers.get(HeaderConstants.AUTHORIZATION)}`);
        return request;
    };
    /**
     * Retrieve header value according to shared key sign rules.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/authenticate-with-shared-key
     *
     * @private
     * @param {WebResource} request
     * @param {string} headerName
     * @returns {string}
     * @memberof SharedKeyCredentialPolicy
     */
    SharedKeyCredentialPolicy.prototype.getHeaderValueToSign = function (request, headerName) {
        var value = request.headers.get(headerName);
        if (!value) {
            return "";
        }
        // When using version 2015-02-21 or later, if Content-Length is zero, then
        // set the Content-Length part of the StringToSign to an empty string.
        // https://docs.microsoft.com/en-us/rest/api/storageservices/authenticate-with-shared-key
        if (headerName === HeaderConstants.CONTENT_LENGTH && value === "0") {
            return "";
        }
        return value;
    };
    /**
     * To construct the CanonicalizedHeaders portion of the signature string, follow these steps:
     * 1. Retrieve all headers for the resource that begin with x-ms-, including the x-ms-date header.
     * 2. Convert each HTTP header name to lowercase.
     * 3. Sort the headers lexicographically by header name, in ascending order.
     *    Each header may appear only once in the string.
     * 4. Replace any linear whitespace in the header value with a single space.
     * 5. Trim any whitespace around the colon in the header.
     * 6. Finally, append a new-line character to each canonicalized header in the resulting list.
     *    Construct the CanonicalizedHeaders string by concatenating all headers in this list into a single string.
     *
     * @private
     * @param {WebResource} request
     * @returns {string}
     * @memberof SharedKeyCredentialPolicy
     */
    SharedKeyCredentialPolicy.prototype.getCanonicalizedHeadersString = function (request) {
        var headersArray = request.headers.headersArray().filter(function (value) {
            return value.name.toLowerCase().startsWith(HeaderConstants.PREFIX_FOR_STORAGE);
        });
        headersArray.sort(function (a, b) {
            return a.name.toLowerCase().localeCompare(b.name.toLowerCase());
        });
        // Remove duplicate headers
        headersArray = headersArray.filter(function (value, index, array) {
            if (index > 0 && value.name.toLowerCase() === array[index - 1].name.toLowerCase()) {
                return false;
            }
            return true;
        });
        var canonicalizedHeadersStringToSign = "";
        headersArray.forEach(function (header) {
            canonicalizedHeadersStringToSign += header.name
                .toLowerCase()
                .trimRight() + ":" + header.value.trimLeft() + "\n";
        });
        return canonicalizedHeadersStringToSign;
    };
    /**
     * Retrieves the webResource canonicalized resource string.
     *
     * @private
     * @param {WebResource} request
     * @returns {string}
     * @memberof SharedKeyCredentialPolicy
     */
    SharedKeyCredentialPolicy.prototype.getCanonicalizedResourceString = function (request) {
        var path = getURLPath(request.url) || "/";
        var canonicalizedResourceString = "";
        canonicalizedResourceString += "/" + this.factory.accountName + path;
        var queries = getURLQueries(request.url);
        var lowercaseQueries = {};
        if (queries) {
            var queryKeys = [];
            for (var key in queries) {
                if (queries.hasOwnProperty(key)) {
                    var lowercaseKey = key.toLowerCase();
                    lowercaseQueries[lowercaseKey] = queries[key];
                    queryKeys.push(lowercaseKey);
                }
            }
            queryKeys.sort();
            for (var _i = 0, queryKeys_1 = queryKeys; _i < queryKeys_1.length; _i++) {
                var key = queryKeys_1[_i];
                canonicalizedResourceString += "\n" + key + ":" + decodeURIComponent(lowercaseQueries[key]);
            }
        }
        return canonicalizedResourceString;
    };
    return SharedKeyCredentialPolicy;
}(CredentialPolicy));

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * SharedKeyCredential for account key authorization of Azure Storage service.
 *
 * @export
 * @class SharedKeyCredential
 * @extends {Credential}
 */
var SharedKeyCredential = /** @class */ (function (_super) {
    tslib_1.__extends(SharedKeyCredential, _super);
    /**
     * Creates an instance of SharedKeyCredential.
     * @param {string} accountName
     * @param {string} accountKey
     * @memberof SharedKeyCredential
     */
    function SharedKeyCredential(accountName, accountKey) {
        var _this = _super.call(this) || this;
        _this.accountName = accountName;
        _this.accountKey = Buffer.from(accountKey, "base64");
        return _this;
    }
    /**
     * Creates a SharedKeyCredentialPolicy object.
     *
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @returns {SharedKeyCredentialPolicy}
     * @memberof SharedKeyCredential
     */
    SharedKeyCredential.prototype.create = function (nextPolicy, options) {
        return new SharedKeyCredentialPolicy(nextPolicy, options, this);
    };
    /**
     * Generates a hash signature for an HTTP request or for a SAS.
     *
     * @param {string} stringToSign
     * @returns {string}
     * @memberof SharedKeyCredential
     */
    SharedKeyCredential.prototype.computeHMACSHA256 = function (stringToSign) {
        return Crypto.createHmac("sha256", this.accountKey)
            .update(stringToSign, "utf8")
            .digest("base64");
    };
    return SharedKeyCredential;
}(Credential));

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * UserDelegationKeyCredential is only used for generation of user delegation SAS.
 * @see https://docs.microsoft.com/en-us/rest/api/storageservices/create-user-delegation-sas
 *
 * @export
 * @class UserDelegationKeyCredential
 */
var UserDelegationKeyCredential = /** @class */ (function () {
    /**
     * Creates an instance of UserDelegationKeyCredential.
     * @param {string} accountName
     * @param {UserDelegationKey} userDelegationKey
     * @memberof UserDelegationKeyCredential
     */
    function UserDelegationKeyCredential(accountName, userDelegationKey) {
        this.accountName = accountName;
        this.userDelegationKey = userDelegationKey;
        this.key = Buffer.from(userDelegationKey.value, "base64");
    }
    /**
     * Generates a hash signature for an HTTP request or for a SAS.
     *
     * @param {string} stringToSign
     * @returns {string}
     * @memberof UserDelegationKeyCredential
     */
    UserDelegationKeyCredential.prototype.computeHMACSHA256 = function (stringToSign) {
        // console.log(`stringToSign: ${JSON.stringify(stringToSign)}`);
        return Crypto.createHmac("sha256", this.key)
            .update(stringToSign, "utf8")
            .digest("base64");
    };
    return UserDelegationKeyCredential;
}());

function generateBlobSASQueryParameters(blobSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName) {
    var version = blobSASSignatureValues.version ? blobSASSignatureValues.version : SERVICE_VERSION;
    var sharedKeyCredential = sharedKeyCredentialOrUserDelegationKey instanceof SharedKeyCredential
        ? sharedKeyCredentialOrUserDelegationKey
        : undefined;
    var userDelegationKeyCredential;
    if (sharedKeyCredential === undefined &&
        accountName !== undefined) {
        userDelegationKeyCredential = new UserDelegationKeyCredential(accountName, sharedKeyCredentialOrUserDelegationKey);
    }
    if (sharedKeyCredential === undefined && userDelegationKeyCredential === undefined) {
        throw TypeError("Invalid sharedKeyCredential, userDelegationKey or accountName.");
    }
    // Version 2018-11-09 adds support for the signed resource and signed blob snapshot time fields.
    // https://docs.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas#constructing-the-signature-string
    if (version >= "2018-11-09") {
        if (sharedKeyCredential !== undefined) {
            return generateBlobSASQueryParameters20181109(blobSASSignatureValues, sharedKeyCredential);
        }
        else {
            return generateBlobSASQueryParametersUDK20181109(blobSASSignatureValues, userDelegationKeyCredential);
        }
    }
    if (version >= "2015-04-05") {
        if (sharedKeyCredential !== undefined) {
            return generateBlobSASQueryParameters20150405(blobSASSignatureValues, sharedKeyCredential);
        }
        else {
            throw new RangeError("'version' must be >= '2018-11-09' when generating user delegation SAS using user delegation key.");
        }
    }
    throw new RangeError("'version' must be >= '2015-04-05'.");
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2015-04-05 AND BEFORE 2018-11-09.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startTime and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiryTime are required.
 * You MUST assign value to identifier or expiryTime & permissions manually if you initial with
 * this constructor.
 *
 * @param {IBlobSASSignatureValues} blobSASSignatureValues
 * @param {SharedKeyCredential} sharedKeyCredential
 * @returns {SASQueryParameters}
 */
function generateBlobSASQueryParameters20150405(blobSASSignatureValues, sharedKeyCredential) {
    if (!blobSASSignatureValues.identifier &&
        (!blobSASSignatureValues.permissions && !blobSASSignatureValues.expiryTime)) {
        throw new RangeError("Must provide 'permissions' and 'expiryTime' for Blob SAS generation when 'identifier' is not provided.");
    }
    var version = blobSASSignatureValues.version ? blobSASSignatureValues.version : SERVICE_VERSION;
    var resource = "c";
    var verifiedPermissions;
    if (blobSASSignatureValues.snapshotTime) {
        throw RangeError("'version' must be >= '2018-11-09' when provided 'snapshotTime'.");
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    if (blobSASSignatureValues.permissions) {
        if (blobSASSignatureValues.blobName) {
            verifiedPermissions = BlobSASPermissions.parse(blobSASSignatureValues.permissions).toString();
            resource = "b";
        }
        else {
            verifiedPermissions = ContainerSASPermissions.parse(blobSASSignatureValues.permissions).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    var stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        blobSASSignatureValues.startTime
            ? truncatedISO8061Date(blobSASSignatureValues.startTime, false)
            : "",
        blobSASSignatureValues.expiryTime
            ? truncatedISO8061Date(blobSASSignatureValues.expiryTime, false)
            : "",
        getCanonicalName(sharedKeyCredential.accountName, blobSASSignatureValues.containerName, blobSASSignatureValues.blobName),
        blobSASSignatureValues.identifier,
        blobSASSignatureValues.ipRange ? ipRangeToString(blobSASSignatureValues.ipRange) : "",
        blobSASSignatureValues.protocol ? blobSASSignatureValues.protocol : "",
        version,
        blobSASSignatureValues.cacheControl ? blobSASSignatureValues.cacheControl : "",
        blobSASSignatureValues.contentDisposition ? blobSASSignatureValues.contentDisposition : "",
        blobSASSignatureValues.contentEncoding ? blobSASSignatureValues.contentEncoding : "",
        blobSASSignatureValues.contentLanguage ? blobSASSignatureValues.contentLanguage : "",
        blobSASSignatureValues.contentType ? blobSASSignatureValues.contentType : ""
    ].join("\n");
    var signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, blobSASSignatureValues.protocol, blobSASSignatureValues.startTime, blobSASSignatureValues.expiryTime, blobSASSignatureValues.ipRange, blobSASSignatureValues.identifier, resource, blobSASSignatureValues.cacheControl, blobSASSignatureValues.contentDisposition, blobSASSignatureValues.contentEncoding, blobSASSignatureValues.contentLanguage, blobSASSignatureValues.contentType);
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2018-11-09.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startTime and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiryTime are required.
 * You MUST assign value to identifier or expiryTime & permissions manually if you initial with
 * this constructor.
 *
 * @param {IBlobSASSignatureValues} blobSASSignatureValues
 * @param {SharedKeyCredential} sharedKeyCredential
 * @returns {SASQueryParameters}
 */
function generateBlobSASQueryParameters20181109(blobSASSignatureValues, sharedKeyCredential) {
    if (!blobSASSignatureValues.identifier &&
        (!blobSASSignatureValues.permissions && !blobSASSignatureValues.expiryTime)) {
        throw new RangeError("Must provide 'permissions' and 'expiryTime' for Blob SAS generation when 'identifier' is not provided.");
    }
    var version = blobSASSignatureValues.version ? blobSASSignatureValues.version : SERVICE_VERSION;
    var resource = "c";
    var verifiedPermissions;
    if (blobSASSignatureValues.blobName === undefined && blobSASSignatureValues.snapshotTime) {
        throw RangeError("Must provide 'blobName' when provided 'snapshotTime'.");
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    if (blobSASSignatureValues.permissions) {
        if (blobSASSignatureValues.blobName) {
            verifiedPermissions = BlobSASPermissions.parse(blobSASSignatureValues.permissions).toString();
            resource = "b";
            if (blobSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
        else {
            verifiedPermissions = ContainerSASPermissions.parse(blobSASSignatureValues.permissions).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    var stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        blobSASSignatureValues.startTime
            ? truncatedISO8061Date(blobSASSignatureValues.startTime, false)
            : "",
        blobSASSignatureValues.expiryTime
            ? truncatedISO8061Date(blobSASSignatureValues.expiryTime, false)
            : "",
        getCanonicalName(sharedKeyCredential.accountName, blobSASSignatureValues.containerName, blobSASSignatureValues.blobName),
        blobSASSignatureValues.identifier,
        blobSASSignatureValues.ipRange ? ipRangeToString(blobSASSignatureValues.ipRange) : "",
        blobSASSignatureValues.protocol ? blobSASSignatureValues.protocol : "",
        version,
        resource,
        blobSASSignatureValues.snapshotTime,
        blobSASSignatureValues.cacheControl ? blobSASSignatureValues.cacheControl : "",
        blobSASSignatureValues.contentDisposition ? blobSASSignatureValues.contentDisposition : "",
        blobSASSignatureValues.contentEncoding ? blobSASSignatureValues.contentEncoding : "",
        blobSASSignatureValues.contentLanguage ? blobSASSignatureValues.contentLanguage : "",
        blobSASSignatureValues.contentType ? blobSASSignatureValues.contentType : ""
    ].join("\n");
    var signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, blobSASSignatureValues.protocol, blobSASSignatureValues.startTime, blobSASSignatureValues.expiryTime, blobSASSignatureValues.ipRange, blobSASSignatureValues.identifier, resource, blobSASSignatureValues.cacheControl, blobSASSignatureValues.contentDisposition, blobSASSignatureValues.contentEncoding, blobSASSignatureValues.contentLanguage, blobSASSignatureValues.contentType);
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2018-11-09.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startTime and identifier.
 *
 * WARNING: identifier will be ignored, permissions and expiryTime are required.
 *
 * @param {IBlobSASSignatureValues} blobSASSignatureValues
 * @param {UserDelegationKeyCredential} userDelegationKeyCredential
 * @returns {SASQueryParameters}
 */
function generateBlobSASQueryParametersUDK20181109(blobSASSignatureValues, userDelegationKeyCredential) {
    if (!blobSASSignatureValues.permissions || !blobSASSignatureValues.expiryTime) {
        throw new RangeError("Must provide 'permissions' and 'expiryTime' for Blob SAS generation when generating user delegation SAS.");
    }
    var version = blobSASSignatureValues.version ? blobSASSignatureValues.version : SERVICE_VERSION;
    var resource = "c";
    var verifiedPermissions;
    if (blobSASSignatureValues.blobName === undefined && blobSASSignatureValues.snapshotTime) {
        throw RangeError("Must provide 'blobName' when provided 'snapshotTime'.");
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    if (blobSASSignatureValues.permissions) {
        if (blobSASSignatureValues.blobName) {
            verifiedPermissions = BlobSASPermissions.parse(blobSASSignatureValues.permissions).toString();
            resource = "b";
            if (blobSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
        else {
            verifiedPermissions = ContainerSASPermissions.parse(blobSASSignatureValues.permissions).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    var stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        blobSASSignatureValues.startTime
            ? truncatedISO8061Date(blobSASSignatureValues.startTime, false)
            : "",
        blobSASSignatureValues.expiryTime
            ? truncatedISO8061Date(blobSASSignatureValues.expiryTime, false)
            : "",
        getCanonicalName(userDelegationKeyCredential.accountName, blobSASSignatureValues.containerName, blobSASSignatureValues.blobName),
        userDelegationKeyCredential.userDelegationKey.signedOid,
        userDelegationKeyCredential.userDelegationKey.signedTid,
        userDelegationKeyCredential.userDelegationKey.signedStart ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedStart, false) : "",
        userDelegationKeyCredential.userDelegationKey.signedExpiry ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedExpiry, false) : "",
        userDelegationKeyCredential.userDelegationKey.signedService,
        userDelegationKeyCredential.userDelegationKey.signedVersion,
        blobSASSignatureValues.ipRange ? ipRangeToString(blobSASSignatureValues.ipRange) : "",
        blobSASSignatureValues.protocol ? blobSASSignatureValues.protocol : "",
        version,
        resource,
        blobSASSignatureValues.snapshotTime,
        blobSASSignatureValues.cacheControl,
        blobSASSignatureValues.contentDisposition,
        blobSASSignatureValues.contentEncoding,
        blobSASSignatureValues.contentLanguage,
        blobSASSignatureValues.contentType
    ].join("\n");
    var signature = userDelegationKeyCredential.computeHMACSHA256(stringToSign);
    return new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, blobSASSignatureValues.protocol, blobSASSignatureValues.startTime, blobSASSignatureValues.expiryTime, blobSASSignatureValues.ipRange, blobSASSignatureValues.identifier, resource, blobSASSignatureValues.cacheControl, blobSASSignatureValues.contentDisposition, blobSASSignatureValues.contentEncoding, blobSASSignatureValues.contentLanguage, blobSASSignatureValues.contentType, userDelegationKeyCredential.userDelegationKey);
}
function getCanonicalName(accountName, containerName, blobName) {
    // Container: "/blob/account/containerName"
    // Blob:      "/blob/account/containerName/blobName"
    var elements = ["/blob/" + accountName + "/" + containerName];
    if (blobName) {
        elements.push("/" + blobName);
    }
    return elements.join("");
}

/**
 * BlockBlobURL defines a set of operations applicable to block blobs.
 *
 * @export
 * @class BlockBlobURL
 * @extends {StorageURL}
 */
var BlockBlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(BlockBlobURL, _super);
    /**
     * Creates an instance of BlockBlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to a block blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage block blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blockblob". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blockblob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof BlockBlobURL
     */
    function BlockBlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.blockBlobContext = new BlockBlob(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a BlockBlobURL object from ContainerURL instance.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName A block blob name
     * @returns {BlockBlobURL}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.fromContainerURL = function (containerURL, blobName) {
        return new BlockBlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a BlockBlobURL object from BlobURL instance.
     *
     * @static
     * @param {BlobURL} blobURL
     * @returns {BlockBlobURL}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.fromBlobURL = function (blobURL) {
        return new BlockBlobURL(blobURL.url, blobURL.pipeline);
    };
    /**
     * Creates a new BlockBlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {BlockBlobURL}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.withPipeline = function (pipeline) {
        return new BlockBlobURL(this.url, pipeline);
    };
    /**
     * Creates a new BlockBlobURL object identical to the source but with the
     * specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {BlockBlobURL}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.withSnapshot = function (snapshot) {
        return new BlockBlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Creates a new block blob, or updates the content of an existing block blob.
     * Updating an existing block blob overwrites any existing metadata on the blob.
     * Partial updates are not supported; the content of the existing blob is
     * overwritten with the new content. To perform a partial update of a block blob's,
     * use stageBlock and commitBlockList.
     *
     * This is a non-parallel uploading method, please use uploadFileToBlockBlob(),
     * uploadStreamToBlockBlob() or uploadBrowserDataToBlockBlob() for better performance
     * with concurrency uploading.
     *
     * @see https://docs.microsoft.com/rest/api/storageservices/put-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {HttpRequestBody} body Blob, string, ArrayBuffer, ArrayBufferView or a function
     *                               which returns a new Readable stream whose offset is from data source beginning.
     * @param {number} contentLength Length of body in bytes. Use Buffer.byteLength() to calculate body length for a
     *                               string including non non-Base64/Hex-encoded characters.
     * @param {IBlockBlobUploadOptions} [options]
     * @returns {Promise<Models.BlockBlobUploadResponse>}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.upload = function (aborter, body, contentLength, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blockBlobContext.upload(body, contentLength, {
                        abortSignal: aborter,
                        blobHTTPHeaders: options.blobHTTPHeaders,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        onUploadProgress: options.progress,
                        cpkInfo: options.customerProvidedKey,
                        tier: toAccessTier(options.tier)
                    })];
            });
        });
    };
    /**
     * Uploads the specified block to the block blob's "staging area" to be later
     * committed by a call to commitBlockList.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-block
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} blockId A 64-byte value that is base64-encoded
     * @param {HttpRequestBody} body
     * @param {number} contentLength
     * @param {IBlockBlobStageBlockOptions} [options]
     * @returns {Promise<Models.BlockBlobStageBlockResponse>}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.stageBlock = function (aborter, blockId, body, contentLength, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blockBlobContext.stageBlock(blockId, contentLength, body, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.leaseAccessConditions,
                        onUploadProgress: options.progress,
                        transactionalContentMD5: options.transactionalContentMD5,
                        transactionalContentCrc64: options.transactionalContentCrc64,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * The Stage Block From URL operation creates a new block to be committed as part
     * of a blob where the contents are read from a URL.
     * This API is available starting in version 2018-03-28.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/put-block-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} blockId A 64-byte value that is base64-encoded
     * @param {string} sourceURL Specifies the URL of the blob. The value
     *                           may be a URL of up to 2 KB in length that specifies a blob.
     *                           The value should be URL-encoded as it would appear
     *                           in a request URI. The source blob must either be public
     *                           or must be authenticated via a shared access signature.
     *                           If the source blob is public, no authentication is required
     *                           to perform the operation. Here are some examples of source object URLs:
     *                           - https://myaccount.blob.core.windows.net/mycontainer/myblob
     *                           - https://myaccount.blob.core.windows.net/mycontainer/myblob?snapshot=<DateTime>
     * @param {number} offset From which position of the blob to download, >= 0
     * @param {number} [count] How much data to be downloaded, > 0. Will download to the end when undefined
     * @param {IBlockBlobStageBlockFromURLOptions} [options={}]
     * @returns {Promise<Models.BlockBlobStageBlockFromURLResponse>}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.stageBlockFromURL = function (aborter, blockId, sourceURL, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blockBlobContext.stageBlockFromURL(blockId, 0, sourceURL, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.leaseAccessConditions,
                        sourceContentMD5: options.sourceContentMD5,
                        sourceContentCrc64: options.sourceContentCrc64,
                        sourceRange: offset === 0 && !count ? undefined : rangeToString({ offset: offset, count: count }),
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Writes a blob by specifying the list of block IDs that make up the blob.
     * In order to be written as part of a blob, a block must have been successfully written
     * to the server in a prior stageBlock operation. You can call commitBlockList to update a blob
     * by uploading only those blocks that have changed, then committing the new and existing
     * blocks together. Any blocks not specified in the block list and permanently deleted.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-block-list
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string[]} blocks  Array of 64-byte value that is base64-encoded
     * @param {IBlockBlobCommitBlockListOptions} [options]
     * @returns {Promise<Models.BlockBlobCommitBlockListResponse>}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.commitBlockList = function (aborter, blocks, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.blockBlobContext.commitBlockList({ latest: blocks }, {
                        abortSignal: aborter,
                        blobHTTPHeaders: options.blobHTTPHeaders,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey,
                        tier: toAccessTier(options.tier)
                    })];
            });
        });
    };
    /**
     * Returns the list of blocks that have been uploaded as part of a block blob
     * using the specified block list filter.
     * @see https://docs.microsoft.com/rest/api/storageservices/get-block-list
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.BlockListType} listType
     * @param {IBlockBlobGetBlockListOptions} [options]
     * @returns {Promise<Models.BlockBlobGetBlockListResponse>}
     * @memberof BlockBlobURL
     */
    BlockBlobURL.prototype.getBlockList = function (aborter, listType, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var res;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.blockBlobContext.getBlockList(listType, {
                            abortSignal: aborter,
                            leaseAccessConditions: options.leaseAccessConditions
                        })];
                    case 1:
                        res = _a.sent();
                        if (!res.committedBlocks) {
                            res.committedBlocks = [];
                        }
                        if (!res.uncommittedBlocks) {
                            res.uncommittedBlocks = [];
                        }
                        return [2 /*return*/, res];
                }
            });
        });
    };
    return BlockBlobURL;
}(BlobURL));

/**
 * A ContainerURL represents a URL to the Azure Storage container allowing you to manipulate its blobs.
 *
 * @export
 * @class ContainerURL
 * @extends {StorageURL}
 */
var ContainerURL = /** @class */ (function (_super) {
    tslib_1.__extends(ContainerURL, _super);
    /**
     * Creates an instance of ContainerURL.
     * @param {string} url A URL string pointing to Azure Storage blob container, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer?sasString".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof ContainerURL
     */
    function ContainerURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.containerContext = new Container(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a ContainerURL object from ServiceURL
     *
     * @param serviceURL A ServiceURL object
     * @param containerName A container name
     */
    ContainerURL.fromServiceURL = function (serviceURL, containerName) {
        return new ContainerURL(appendToURLPath(serviceURL.url, encodeURIComponent(containerName)), serviceURL.pipeline);
    };
    /**
     * Creates a new ContainerURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {ContainerURL}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.withPipeline = function (pipeline) {
        return new ContainerURL(this.url, pipeline);
    };
    /**
     * Creates a new container under the specified account. If the container with
     * the same name already exists, the operation fails.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/create-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IContainerCreateOptions} [options]
     * @returns {Promise<Models.ContainerCreateResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.create = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                // Spread operator in destructuring assignments,
                // this will filter out unwanted properties from the response object into result object
                return [2 /*return*/, this.containerContext.create(tslib_1.__assign({}, options, { abortSignal: aborter }))];
            });
        });
    };
    /**
     * Returns all user-defined metadata and system properties for the specified
     * container. The data returned does not include the container's list of blobs.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-container-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IContainersGetPropertiesOptions} [options]
     * @returns {Promise<Models.ContainerGetPropertiesResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.getProperties = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!options.leaseAccessConditions) {
                    options.leaseAccessConditions = {};
                }
                return [2 /*return*/, this.containerContext.getProperties(tslib_1.__assign({ abortSignal: aborter }, options.leaseAccessConditions))];
            });
        });
    };
    /**
     * Marks the specified container for deletion. The container and any blobs
     * contained within it are later deleted during garbage collection.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/delete-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.ContainersDeleteMethodOptionalParams} [options]
     * @returns {Promise<Models.ContainerDeleteResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.delete = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!options.containerAccessConditions) {
                    options.containerAccessConditions = {};
                }
                if (!options.containerAccessConditions.modifiedAccessConditions) {
                    options.containerAccessConditions.modifiedAccessConditions = {};
                }
                if (!options.containerAccessConditions.leaseAccessConditions) {
                    options.containerAccessConditions.leaseAccessConditions = {};
                }
                if ((options.containerAccessConditions.modifiedAccessConditions.ifMatch &&
                    options.containerAccessConditions.modifiedAccessConditions.ifMatch !== ETagNone) ||
                    (options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch &&
                        options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch !== ETagNone)) {
                    throw new RangeError("the IfMatch and IfNoneMatch access conditions must have their default\
        values because they are ignored by the service");
                }
                return [2 /*return*/, this.containerContext.deleteMethod({
                        abortSignal: aborter,
                        leaseAccessConditions: options.containerAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.containerAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Sets one or more user-defined name-value pairs for the specified container.
     *
     * If no option provided, or no metadata defined in the parameter, the container
     * metadata will be removed.
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-container-metadata
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IMetadata} [metadata] Replace existing metadata with this value.
     *                               If no value provided the existing metadata will be removed.
     * @param {IContainerSetMetadataOptions} [options]
     * @returns {Promise<Models.ContainerSetMetadataResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.setMetadata = function (aborter, metadata, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!options.containerAccessConditions) {
                    options.containerAccessConditions = {};
                }
                if (!options.containerAccessConditions.modifiedAccessConditions) {
                    options.containerAccessConditions.modifiedAccessConditions = {};
                }
                if (!options.containerAccessConditions.leaseAccessConditions) {
                    options.containerAccessConditions.leaseAccessConditions = {};
                }
                if (options.containerAccessConditions.modifiedAccessConditions.ifUnmodifiedSince ||
                    (options.containerAccessConditions.modifiedAccessConditions.ifMatch &&
                        options.containerAccessConditions.modifiedAccessConditions.ifMatch !== ETagNone) ||
                    (options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch &&
                        options.containerAccessConditions.modifiedAccessConditions.ifNoneMatch !== ETagNone)) {
                    throw new RangeError("the IfUnmodifiedSince, IfMatch, and IfNoneMatch must have their default values\
        because they are ignored by the blob service");
                }
                return [2 /*return*/, this.containerContext.setMetadata({
                        abortSignal: aborter,
                        leaseAccessConditions: options.containerAccessConditions.leaseAccessConditions,
                        metadata: metadata,
                        modifiedAccessConditions: options.containerAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Gets the permissions for the specified container. The permissions indicate
     * whether container data may be accessed publicly.
     *
     * WARNING: JavaScript Date will potential lost precision when parsing start and expiry string.
     * For example, new Date("2018-12-31T03:44:23.8827891Z").toISOString() will get "2018-12-31T03:44:23.882Z".
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-container-acl
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IContainerGetAccessPolicyOptions} [options]
     * @returns {Promise<ContainerGetAccessPolicyResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.getAccessPolicy = function (aborter, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var response, res, _i, response_1, identifier, accessPolicy;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!options.leaseAccessConditions) {
                            options.leaseAccessConditions = {};
                        }
                        return [4 /*yield*/, this.containerContext.getAccessPolicy({
                                abortSignal: aborter,
                                leaseAccessConditions: options.leaseAccessConditions
                            })];
                    case 1:
                        response = _a.sent();
                        res = {
                            _response: response._response,
                            blobPublicAccess: response.blobPublicAccess,
                            date: response.date,
                            eTag: response.eTag,
                            errorCode: response.errorCode,
                            lastModified: response.lastModified,
                            requestId: response.requestId,
                            clientRequestId: response.clientRequestId,
                            signedIdentifiers: [],
                            version: response.version
                        };
                        for (_i = 0, response_1 = response; _i < response_1.length; _i++) {
                            identifier = response_1[_i];
                            accessPolicy = {
                                permission: identifier.accessPolicy.permission,
                            };
                            if (identifier.accessPolicy.expiry) {
                                accessPolicy.expiry = new Date(identifier.accessPolicy.expiry);
                            }
                            if (identifier.accessPolicy.start) {
                                accessPolicy.start = new Date(identifier.accessPolicy.start);
                            }
                            res.signedIdentifiers.push({
                                accessPolicy: accessPolicy,
                                id: identifier.id
                            });
                        }
                        return [2 /*return*/, res];
                }
            });
        });
    };
    /**
     * Sets the permissions for the specified container. The permissions indicate
     * whether blobs in a container may be accessed publicly.
     *
     * When you set permissions for a container, the existing permissions are replaced.
     * If no access or containerAcl provided, the existing container ACL will be
     * removed.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-container-acl
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {PublicAccessType} [access]
     * @param {ISignedIdentifier[]} [containerAcl]
     * @param {IContainerSetAccessPolicyOptions} [options]
     * @returns {Promise<Models.ContainerSetAccessPolicyResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.setAccessPolicy = function (aborter, access, containerAcl, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var acl, _i, _a, identifier;
            return tslib_1.__generator(this, function (_b) {
                options.containerAccessConditions = options.containerAccessConditions || {};
                acl = [];
                for (_i = 0, _a = containerAcl || []; _i < _a.length; _i++) {
                    identifier = _a[_i];
                    acl.push({
                        accessPolicy: {
                            expiry: identifier.accessPolicy.expiry ? truncatedISO8061Date(identifier.accessPolicy.expiry) : "",
                            permission: identifier.accessPolicy.permission,
                            start: identifier.accessPolicy.start ? truncatedISO8061Date(identifier.accessPolicy.start) : ""
                        },
                        id: identifier.id
                    });
                }
                return [2 /*return*/, this.containerContext.setAccessPolicy({
                        abortSignal: aborter,
                        access: access,
                        containerAcl: acl,
                        leaseAccessConditions: options.containerAccessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.containerAccessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Establishes and manages a lock on a container for delete operations.
     * The lock duration can be 15 to 60 seconds, or can be infinite.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} proposedLeaseId Can be specified in any valid GUID string format
     * @param {number} duration Must be between 15 to 60 seconds, or infinite (-1)
     * @param {IContainerAcquireLeaseOptions} [options]
     * @returns {Promise<Models.ContainerAcquireLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.acquireLease = function (aborter, proposedLeaseId, duration, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.acquireLease({
                        abortSignal: aborter,
                        duration: duration,
                        modifiedAccessConditions: options.modifiedAccessConditions,
                        proposedLeaseId: proposedLeaseId
                    })];
            });
        });
    };
    /**
     * To free the lease if it is no longer needed so that another client may
     * immediately acquire a lease against the container.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IContainerReleaseLeaseOptions} [options]
     * @returns {Promise<Models.ContainerReleaseLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.releaseLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.releaseLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To renew an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IContainerRenewLeaseOptions} [options]
     * @returns {Promise<Models.ContainerRenewLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.renewLease = function (aborter, leaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.renewLease(leaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To end the lease but ensure that another client cannot acquire a new lease
     * until the current lease period has expired.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} period break period
     * @param {IContainerBreakLeaseOptions} [options]
     * @returns {Promise<Models.ContainerBreakLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.breakLease = function (aborter, period, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.breakLease({
                        abortSignal: aborter,
                        breakPeriod: period,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * To change the ID of an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-container
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {string} proposedLeaseId
     * @param {IContainerChangeLeaseOptions} [options]
     * @returns {Promise<Models.ContainerChangeLeaseResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.changeLease = function (aborter, leaseId, proposedLeaseId, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.changeLease(leaseId, proposedLeaseId, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * listBlobFlatSegment returns a single segment of blobs starting from the
     * specified Marker. Use an empty Marker to start enumeration from the beginning.
     * After getting a segment, process it, and then call ListBlobsFlatSegment again
     * (passing the the previously-returned Marker) to get the next segment.
     * @see https://docs.microsoft.com/rest/api/storageservices/list-blobs
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} [marker]
     * @param {IContainerListBlobsSegmentOptions} [options]
     * @returns {Promise<Models.ContainerListBlobFlatSegmentResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.listBlobFlatSegment = function (aborter, marker, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.listBlobFlatSegment(tslib_1.__assign({ abortSignal: aborter, marker: marker }, options))];
            });
        });
    };
    /**
     * listBlobHierarchySegment returns a single segment of blobs starting from
     * the specified Marker. Use an empty Marker to start enumeration from the
     * beginning. After getting a segment, process it, and then call ListBlobsHierarchicalSegment
     * again (passing the the previously-returned Marker) to get the next segment.
     * @see https://docs.microsoft.com/rest/api/storageservices/list-blobs
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} delimiter
     * @param {IContainerListBlobsSegmentOptions} [options]
     * @returns {Promise<Models.ContainerListBlobHierarchySegmentResponse>}
     * @memberof ContainerURL
     */
    ContainerURL.prototype.listBlobHierarchySegment = function (aborter, delimiter, marker, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.containerContext.listBlobHierarchySegment(delimiter, tslib_1.__assign({ abortSignal: aborter, marker: marker }, options))];
            });
        });
    };
    return ContainerURL;
}(StorageURL));

/**
 * TokenCredentialPolicy is a policy used to sign HTTP request with a token.
 * Such as an OAuth bearer token.
 *
 * @export
 * @class TokenCredentialPolicy
 * @extends {CredentialPolicy}
 */
var TokenCredentialPolicy = /** @class */ (function (_super) {
    tslib_1.__extends(TokenCredentialPolicy, _super);
    /**
     * Creates an instance of TokenCredentialPolicy.
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @param {TokenCredential} tokenCredential
     * @memberof TokenCredentialPolicy
     */
    function TokenCredentialPolicy(nextPolicy, options, tokenCredential) {
        var _this = _super.call(this, nextPolicy, options) || this;
        _this.tokenCredential = tokenCredential;
        _this.authorizationScheme = HeaderConstants.AUTHORIZATION_SCHEME;
        return _this;
    }
    /**
     * Sign request with token.
     *
     * @protected
     * @param {WebResource} request
     * @returns {WebResource}
     * @memberof TokenCredentialPolicy
     */
    TokenCredentialPolicy.prototype.signRequest = function (request) {
        if (!request.headers) {
            request.headers = new msRest.HttpHeaders();
        }
        request.headers.set(HeaderConstants.AUTHORIZATION, this.authorizationScheme + " " + this.tokenCredential.token);
        return request;
    };
    return TokenCredentialPolicy;
}(CredentialPolicy));

/**
 * TokenCredential is a Credential used to generate a TokenCredentialPolicy.
 * Renew token by setting a new token string value to token property.
 *
 * @example
 *  const tokenCredential = new TokenCredential("token");
 *  const pipeline = StorageURL.newPipeline(tokenCredential);
 *
 *  // List containers
 *  const serviceURL = new ServiceURL("https://mystorageaccount.blob.core.windows.net", pipeline);
 *
 *  // Set up a timer to refresh the token
 *  const timerID = setInterval(() => {
 *    // Update token by accessing to public tokenCredential.token
 *    tokenCredential.token = "updatedToken";
 *    // WARNING: Timer must be manually stopped! It will forbid GC of tokenCredential
 *    if (shouldStop()) {
 *      clearInterval(timerID);
 *    }
 *  }, 60 * 60 * 1000); // Set an interval time before your token expired
 * @export
 * @class TokenCredential
 * @extends {Credential}
 *
 */
var TokenCredential = /** @class */ (function (_super) {
    tslib_1.__extends(TokenCredential, _super);
    /**
     * Creates an instance of TokenCredential.
     * @param {string} token
     * @memberof TokenCredential
     */
    function TokenCredential(token) {
        var _this = _super.call(this) || this;
        _this.token = token;
        return _this;
    }
    /**
     * Creates a TokenCredentialPolicy object.
     *
     * @param {RequestPolicy} nextPolicy
     * @param {RequestPolicyOptions} options
     * @returns {TokenCredentialPolicy}
     * @memberof TokenCredential
     */
    TokenCredential.prototype.create = function (nextPolicy, options) {
        return new TokenCredentialPolicy(nextPolicy, options, this);
    };
    return TokenCredential;
}(Credential));

/**
 * States for Batch.
 *
 * @enum {number}
 */
var BatchStates;
(function (BatchStates) {
    BatchStates[BatchStates["Good"] = 0] = "Good";
    BatchStates[BatchStates["Error"] = 1] = "Error";
})(BatchStates || (BatchStates = {}));
/**
 * Batch provides basic parallel execution with concurrency limits.
 * Will stop execute left operations when one of the executed operation throws an error.
 * But Batch cannot cancel ongoing operations, you need to cancel them by yourself.
 *
 * @export
 * @class Batch
 */
var Batch = /** @class */ (function () {
    /**
     * Creates an instance of Batch.
     * @param {number} [concurrency=5]
     * @memberof Batch
     */
    function Batch(concurrency) {
        if (concurrency === void 0) { concurrency = 5; }
        /**
         * Number of active operations under execution.
         *
         * @private
         * @type {number}
         * @memberof Batch
         */
        this.actives = 0;
        /**
         * Number of completed operations under execution.
         *
         * @private
         * @type {number}
         * @memberof Batch
         */
        this.completed = 0;
        /**
         * Offset of next operation to be executed.
         *
         * @private
         * @type {number}
         * @memberof Batch
         */
        this.offset = 0;
        /**
         * Operation array to be executed.
         *
         * @private
         * @type {Operation[]}
         * @memberof Batch
         */
        this.operations = [];
        /**
         * States of Batch. When an error happens, state will turn into error.
         * Batch will stop execute left operations.
         *
         * @private
         * @type {BatchStates}
         * @memberof Batch
         */
        this.state = BatchStates.Good;
        if (concurrency < 1) {
            throw new RangeError("concurrency must be larger than 0");
        }
        this.concurrency = concurrency;
        this.emitter = new events.EventEmitter();
    }
    /**
     * Add a operation into queue.
     *
     * @param {Operation} operation
     * @memberof Batch
     */
    Batch.prototype.addOperation = function (operation) {
        var _this = this;
        this.operations.push(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var error_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        _a.trys.push([0, 2, , 3]);
                        this.actives++;
                        return [4 /*yield*/, operation()];
                    case 1:
                        _a.sent();
                        this.actives--;
                        this.completed++;
                        this.parallelExecute();
                        return [3 /*break*/, 3];
                    case 2:
                        error_1 = _a.sent();
                        this.emitter.emit("error", error_1);
                        return [3 /*break*/, 3];
                    case 3: return [2 /*return*/];
                }
            });
        }); });
    };
    /**
     * Start execute operations in the queue.
     *
     * @returns {Promise<void>}
     * @memberof Batch
     */
    Batch.prototype.do = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                this.parallelExecute();
                return [2 /*return*/, new Promise(function (resolve, reject) {
                        _this.emitter.on("finish", resolve);
                        _this.emitter.on("error", function (error) {
                            _this.state = BatchStates.Error;
                            reject(error);
                        });
                    })];
            });
        });
    };
    /**
     * Get next operation to be executed. Return null when reaching ends.
     *
     * @private
     * @returns {(Operation | null)}
     * @memberof Batch
     */
    Batch.prototype.nextOperation = function () {
        if (this.offset < this.operations.length) {
            return this.operations[this.offset++];
        }
        return null;
    };
    /**
     * Start execute operations. One one the most important difference between
     * this method with do() is that do() wraps as an sync method.
     *
     * @private
     * @returns {void}
     * @memberof Batch
     */
    Batch.prototype.parallelExecute = function () {
        if (this.state === BatchStates.Error) {
            return;
        }
        if (this.completed >= this.operations.length) {
            this.emitter.emit("finish");
            return;
        }
        while (this.actives < this.concurrency) {
            var operation = this.nextOperation();
            if (operation) {
                operation();
            }
            else {
                return;
            }
        }
    };
    return Batch;
}());

/**
 * ONLY AVAILABLE IN BROWSERS.
 *
 * Uploads a browser Blob/File/ArrayBuffer/ArrayBufferView object to block blob.
 *
 * When buffer length <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Blob | ArrayBuffer | ArrayBufferView} browserData Blob, File, ArrayBuffer or ArrayBufferView
 * @param {BlockBlobURL} blockBlobURL
 * @param {IUploadToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
function uploadBrowserDataToBlockBlob(aborter, browserData, blockBlobURL, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var browserBlob;
        return tslib_1.__generator(this, function (_a) {
            browserBlob = new Blob([browserData]);
            return [2 /*return*/, UploadSeekableBlobToBlockBlob(aborter, function (offset, size) {
                    return browserBlob.slice(offset, offset + size);
                }, browserBlob.size, blockBlobURL, options)];
        });
    });
}
/**
 * ONLY AVAILABLE IN BROWSERS.
 *
 * Uploads a browser Blob object to block blob. Requires a blobFactory as the data source,
 * which need to return a Blob object with the offset and size provided.
 *
 * When buffer length <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {(offset: number, size: number) => Blob} blobFactory
 * @param {number} size
 * @param {BlockBlobURL} blockBlobURL
 * @param {IUploadToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
function UploadSeekableBlobToBlockBlob(aborter, blobFactory, size, blockBlobURL, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var numBlocks, blockList, blockIDPrefix, transferProgress, batch, _loop_1, i;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blockSize) {
                        options.blockSize = 0;
                    }
                    if (options.blockSize < 0 || options.blockSize > BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES) {
                        throw new RangeError("blockSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES);
                    }
                    if (options.maxSingleShotSize !== 0 && !options.maxSingleShotSize) {
                        options.maxSingleShotSize = BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES;
                    }
                    if (options.maxSingleShotSize < 0 ||
                        options.maxSingleShotSize > BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES) {
                        throw new RangeError("maxSingleShotSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES);
                    }
                    if (options.blockSize === 0) {
                        if (size > BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES * BLOCK_BLOB_MAX_BLOCKS) {
                            throw new RangeError(size + " is too larger to upload to a block blob.");
                        }
                        if (size > options.maxSingleShotSize) {
                            options.blockSize = Math.ceil(size / BLOCK_BLOB_MAX_BLOCKS);
                            if (options.blockSize < DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES) {
                                options.blockSize = DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES;
                            }
                        }
                    }
                    if (!options.blobHTTPHeaders) {
                        options.blobHTTPHeaders = {};
                    }
                    if (!options.blobAccessConditions) {
                        options.blobAccessConditions = {};
                    }
                    if (size <= options.maxSingleShotSize) {
                        return [2 /*return*/, blockBlobURL.upload(aborter, blobFactory(0, size), size, options)];
                    }
                    numBlocks = Math.floor((size - 1) / options.blockSize) + 1;
                    if (numBlocks > BLOCK_BLOB_MAX_BLOCKS) {
                        throw new RangeError("The buffer's size is too big or the BlockSize is too small;" +
                            ("the number of blocks must be <= " + BLOCK_BLOB_MAX_BLOCKS));
                    }
                    blockList = [];
                    blockIDPrefix = msRest.generateUuid();
                    transferProgress = 0;
                    batch = new Batch(options.parallelism);
                    _loop_1 = function (i) {
                        batch.addOperation(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                            var blockID, start, end, contentLength;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        blockID = generateBlockID(blockIDPrefix, i);
                                        start = options.blockSize * i;
                                        end = i === numBlocks - 1 ? size : start + options.blockSize;
                                        contentLength = end - start;
                                        blockList.push(blockID);
                                        return [4 /*yield*/, blockBlobURL.stageBlock(aborter, blockID, blobFactory(start, contentLength), contentLength, {
                                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions
                                            })];
                                    case 1:
                                        _a.sent();
                                        // Update progress after block is successfully uploaded to server, in case of block trying
                                        // TODO: Hook with convenience layer progress event in finer level
                                        transferProgress += contentLength;
                                        if (options.progress) {
                                            options.progress({
                                                loadedBytes: transferProgress
                                            });
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); });
                    };
                    for (i = 0; i < numBlocks; i++) {
                        _loop_1(i);
                    }
                    return [4 /*yield*/, batch.do()];
                case 1:
                    _a.sent();
                    return [2 /*return*/, blockBlobURL.commitBlockList(aborter, blockList, options)];
            }
        });
    });
}

/**
 * This class accepts a Node.js Readable stream as input, and keeps reading data
 * from the stream into the internal buffer structure, until it reaches maxBuffers.
 * Every available buffer will try to trigger outgoingHandler.
 *
 * The internal buffer structure includes an incoming buffer array, and a outgoing
 * buffer array. The incoming buffer array includes the "empty" buffers can be filled
 * with new incoming data. The outgoing array includes the filled buffers to be
 * handled by outgoingHandler. Every above buffer size is defined by parameter bufferSize.
 *
 * NUM_OF_ALL_BUFFERS = BUFFERS_IN_INCOMING + BUFFERS_IN_OUTGOING + BUFFERS_UNDER_HANDLING
 *
 * NUM_OF_ALL_BUFFERS <= maxBuffers
 *
 * PERFORMANCE IMPROVEMENT TIPS:
 * 1. Input stream highWaterMark is better to set a same value with bufferSize
 *    parameter, which will avoid Buffer.concat() operations.
 * 2. Parallelism should set a smaller value than maxBuffers, which is helpful to
 *    reduce the possibility when a outgoing handler waits for the stream data.
 *    in this situation, outgoing handlers are blocked.
 *    Outgoing queue shouldn't be empty.
 * @export
 * @class BufferScheduler
 */
var BufferScheduler = /** @class */ (function () {
    /**
     * Creates an instance of BufferScheduler.
     *
     * @param {Readable} readable A Node.js Readable stream
     * @param {number} bufferSize Buffer size of every maintained buffer
     * @param {number} maxBuffers How many buffers can be allocated
     * @param {OutgoingHandler} outgoingHandler An async function scheduled to be
     *                                          triggered when a buffer fully filled
     *                                          with stream data
     * @param {number} parallelism Concurrency of executing outgoingHandlers (>0)
     * @param {string} [encoding] [Optional] Encoding of Readable stream when it's a string stream
     * @memberof BufferScheduler
     */
    function BufferScheduler(readable, bufferSize, maxBuffers, outgoingHandler, parallelism, encoding) {
        /**
         * An internal event emitter.
         *
         * @private
         * @type {EventEmitter}
         * @memberof BufferScheduler
         */
        this.emitter = new events.EventEmitter();
        /**
         * An internal offset marker to track data offset in bytes of next outgoingHandler.
         *
         * @private
         * @type {number}
         * @memberof BufferScheduler
         */
        this.offset = 0;
        /**
         * An internal marker to track whether stream is end.
         *
         * @private
         * @type {boolean}
         * @memberof BufferScheduler
         */
        this.isStreamEnd = false;
        /**
         * An internal marker to track whether stream or outgoingHandler returns error.
         *
         * @private
         * @type {boolean}
         * @memberof BufferScheduler
         */
        this.isError = false;
        /**
         * How many handlers are executing.
         *
         * @private
         * @type {number}
         * @memberof BufferScheduler
         */
        this.executingOutgoingHandlers = 0;
        /**
         * How many buffers have been allocated.
         *
         * @private
         * @type {number}
         * @memberof BufferScheduler
         */
        this.numBuffers = 0;
        /**
         * Because this class doesn't know how much data every time stream pops, which
         * is defined by highWaterMarker of the stream. So BufferScheduler will cache
         * data received from the stream, when data in unresolvedDataArray exceeds the
         * blockSize defined, it will try to concat a blockSize of buffer, fill into available
         * buffers from incoming and push to outgoing array.
         *
         * @private
         * @type {Buffer[]}
         * @memberof BufferScheduler
         */
        this.unresolvedDataArray = [];
        /**
         * How much data consisted in unresolvedDataArray.
         *
         * @private
         * @type {number}
         * @memberof BufferScheduler
         */
        this.unresolvedLength = 0;
        /**
         * The array includes all the available buffers can be used to fill data from stream.
         *
         * @private
         * @type {Buffer[]}
         * @memberof BufferScheduler
         */
        this.incoming = [];
        /**
         * The array (queue) includes all the buffers filled from stream data.
         *
         * @private
         * @type {Buffer[]}
         * @memberof BufferScheduler
         */
        this.outgoing = [];
        if (bufferSize <= 0) {
            throw new RangeError("bufferSize must be larger than 0, current is " + bufferSize);
        }
        if (maxBuffers <= 0) {
            throw new RangeError("maxBuffers must be larger than 0, current is " + maxBuffers);
        }
        if (parallelism <= 0) {
            throw new RangeError("parallelism must be larger than 0, current is " + parallelism);
        }
        this.bufferSize = bufferSize;
        this.maxBuffers = maxBuffers;
        this.readable = readable;
        this.outgoingHandler = outgoingHandler;
        this.parallelism = parallelism;
        this.encoding = encoding;
    }
    /**
     * Start the scheduler, will return error when stream of any of the outgoingHandlers
     * returns error.
     *
     * @returns {Promise<void>}
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.do = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, new Promise(function (resolve, reject) {
                        _this.readable.on("data", function (data) {
                            data = typeof data === "string" ? Buffer.from(data, _this.encoding) : data;
                            _this.appendUnresolvedData(data);
                            if (!_this.resolveData()) {
                                _this.readable.pause();
                            }
                        });
                        _this.readable.on("error", function (err) {
                            _this.emitter.emit("error", err);
                        });
                        _this.readable.on("end", function () {
                            _this.isStreamEnd = true;
                            _this.emitter.emit("checkEnd");
                        });
                        _this.emitter.on("error", function (err) {
                            _this.isError = true;
                            _this.readable.pause();
                            reject(err);
                        });
                        _this.emitter.on("checkEnd", function () {
                            if (_this.outgoing.length > 0) {
                                _this.triggerOutgoingHandlers();
                                return;
                            }
                            if (_this.isStreamEnd && _this.executingOutgoingHandlers === 0) {
                                if (_this.unresolvedLength > 0 && _this.unresolvedLength < _this.bufferSize) {
                                    _this.outgoingHandler(_this.shiftBufferFromUnresolvedDataArray(), _this.offset)
                                        .then(resolve)
                                        .catch(reject);
                                }
                                else if (_this.unresolvedLength >= _this.bufferSize) {
                                    return;
                                }
                                else {
                                    resolve();
                                }
                            }
                        });
                    })];
            });
        });
    };
    /**
     * Insert a new data into unresolved array.
     *
     * @private
     * @param {Buffer} data
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.appendUnresolvedData = function (data) {
        this.unresolvedDataArray.push(data);
        this.unresolvedLength += data.length;
    };
    /**
     * Try to shift a buffer with size in blockSize. The buffer returned may be less
     * than blockSize when data in unresolvedDataArray is less than bufferSize.
     *
     * @private
     * @returns {Buffer}
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.shiftBufferFromUnresolvedDataArray = function () {
        if (this.unresolvedLength >= this.bufferSize) {
            if (this.bufferSize === this.unresolvedDataArray[0].length) {
                this.unresolvedLength -= this.bufferSize;
                return this.unresolvedDataArray.shift();
            }
            // Lazy concat because Buffer.concat highly drops performance
            var merged = Buffer.concat(this.unresolvedDataArray, this.unresolvedLength);
            var buffer = merged.slice(0, this.bufferSize);
            merged = merged.slice(this.bufferSize);
            this.unresolvedDataArray = [merged];
            this.unresolvedLength -= buffer.length;
            return buffer;
        }
        else if (this.unresolvedLength > 0) {
            var merged = Buffer.concat(this.unresolvedDataArray, this.unresolvedLength);
            this.unresolvedDataArray = [];
            this.unresolvedLength = 0;
            return merged;
        }
        else {
            return Buffer.allocUnsafe(0);
        }
    };
    /**
     * Resolve data in unresolvedDataArray. For every buffer with size in blockSize
     * shifted, it will try to get (or allocate a buffer) from incoming, and fill it,
     * then push it into outgoing to be handled by outgoing handler.
     *
     * Return false when available buffers in incoming are not enough, else true.
     *
     * @private
     * @returns {boolean} Return false when buffers in incoming are not enough, else true.
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.resolveData = function () {
        while (this.unresolvedLength >= this.bufferSize) {
            var buffer = void 0;
            if (this.incoming.length > 0) {
                buffer = this.incoming.shift();
            }
            else {
                if (this.numBuffers < this.maxBuffers) {
                    buffer = Buffer.allocUnsafe(this.bufferSize);
                    this.numBuffers++;
                }
                else {
                    // No available buffer, wait for buffer returned
                    return false;
                }
            }
            buffer.fill(this.shiftBufferFromUnresolvedDataArray());
            this.outgoing.push(buffer);
            this.triggerOutgoingHandlers();
        }
        return true;
    };
    /**
     * Try to trigger a outgoing handler for every buffer in outgoing. Stop when
     * parallelism reaches.
     *
     * @private
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.triggerOutgoingHandlers = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var buffer;
            return tslib_1.__generator(this, function (_a) {
                do {
                    if (this.executingOutgoingHandlers >= this.parallelism) {
                        return [2 /*return*/];
                    }
                    buffer = this.outgoing.shift();
                    if (buffer) {
                        this.triggerOutgoingHandler(buffer);
                    }
                } while (buffer);
                return [2 /*return*/];
            });
        });
    };
    /**
     * Trigger a outgoing handler for a buffer shifted from outgoing.
     *
     * @private
     * @param {Buffer} buffer
     * @returns {Promise<any>}
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.triggerOutgoingHandler = function (buffer) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var bufferLength, err_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        bufferLength = buffer.length;
                        this.executingOutgoingHandlers++;
                        this.offset += bufferLength;
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 3, , 4]);
                        return [4 /*yield*/, this.outgoingHandler(buffer, this.offset - bufferLength)];
                    case 2:
                        _a.sent();
                        return [3 /*break*/, 4];
                    case 3:
                        err_1 = _a.sent();
                        this.emitter.emit("error", err_1);
                        return [2 /*return*/];
                    case 4:
                        this.executingOutgoingHandlers--;
                        this.reuseBuffer(buffer);
                        this.emitter.emit("checkEnd");
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Return buffer used by outgoing handler into incoming.
     *
     * @private
     * @param {Buffer} buffer
     * @memberof BufferScheduler
     */
    BufferScheduler.prototype.reuseBuffer = function (buffer) {
        this.incoming.push(buffer);
        if (!this.isError && this.resolveData() && !this.isStreamEnd) {
            this.readable.resume();
        }
    };
    return BufferScheduler;
}());

/**
 * Reads a readable stream into buffer. Fill the buffer from offset to end.
 *
 * @export
 * @param {NodeJS.ReadableStream} stream A Node.js Readable stream
 * @param {Buffer} buffer Buffer to be filled, length must >= offset
 * @param {number} offset From which position in the buffer to be filled, inclusive
 * @param {number} end To which position in the buffer to be filled, exclusive
 * @param {string} [encoding] Encoding of the Readable stream
 * @returns {Promise<void>}
 */
function streamToBuffer(stream, buffer, offset, end, encoding) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var pos, count;
        return tslib_1.__generator(this, function (_a) {
            pos = 0;
            count = end - offset;
            return [2 /*return*/, new Promise(function (resolve, reject) {
                    stream.on("readable", function () {
                        if (pos >= count) {
                            resolve();
                            return;
                        }
                        var chunk = stream.read();
                        if (!chunk) {
                            return;
                        }
                        if (typeof chunk === "string") {
                            chunk = Buffer.from(chunk, encoding);
                        }
                        // How much data needed in this chunk
                        var chunkLength = pos + chunk.length > count ? count - pos : chunk.length;
                        buffer.fill(chunk.slice(0, chunkLength), offset + pos, offset + pos + chunkLength);
                        pos += chunkLength;
                    });
                    stream.on("end", function () {
                        if (pos < count) {
                            reject(new Error("Stream drains before getting enough data needed. Data read: " + pos + ", data need: " + count));
                        }
                        resolve();
                    });
                    stream.on("error", reject);
                })];
        });
    });
}
/**
 * Reads a readable stream into buffer entirely.
 *
 * @export
 * @param {NodeJS.ReadableStream} stream A Node.js Readable stream
 * @param {Buffer} buffer Buffer to be filled, length must >= offset
 * @param {string} [encoding] Encoding of the Readable stream
 * @returns {Promise<number>} with the count of bytes read.
 * @throws {RangeError} If buffer size is not big enough.
 */
function streamToBuffer2(stream, buffer, encoding) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var pos, bufferSize;
        return tslib_1.__generator(this, function (_a) {
            pos = 0;
            bufferSize = buffer.length;
            return [2 /*return*/, new Promise(function (resolve, reject) {
                    stream.on("readable", function () {
                        if (pos >= bufferSize) {
                            reject(new Error("Stream exceeds buffer size. Buffer size: " + bufferSize));
                            return;
                        }
                        var chunk = stream.read();
                        if (!chunk) {
                            return;
                        }
                        if (typeof chunk === "string") {
                            chunk = Buffer.from(chunk, encoding);
                        }
                        // How much data needed in this chunk
                        var chunkLength = pos + chunk.length > bufferSize ? bufferSize - pos : chunk.length;
                        buffer.fill(chunk.slice(0, chunkLength), pos, pos + chunkLength);
                        pos += chunkLength;
                    });
                    stream.on("end", function () {
                        resolve(pos);
                    });
                    stream.on("error", reject);
                })];
        });
    });
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Uploads a local file in blocks to a block blob.
 *
 * When file size <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {string} filePath Full path of local file
 * @param {BlockBlobURL} blockBlobURL BlockBlobURL
 * @param {IUploadToBlockBlobOptions} [options] IUploadToBlockBlobOptions
 * @returns {(Promise<BlobUploadCommonResponse>)} ICommonResponse
 */
function uploadFileToBlockBlob(aborter, filePath, blockBlobURL, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var size;
        return tslib_1.__generator(this, function (_a) {
            size = fs.statSync(filePath).size;
            return [2 /*return*/, uploadResetableStreamToBlockBlob(aborter, function (offset, count) {
                    return fs.createReadStream(filePath, {
                        autoClose: true,
                        end: count ? offset + count - 1 : Infinity,
                        start: offset
                    });
                }, size, blockBlobURL, options)];
        });
    });
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Accepts a Node.js Readable stream factory, and uploads in blocks to a block blob.
 * The Readable stream factory must returns a Node.js Readable stream starting from the offset defined. The offset
 * is the offset in the block blob to be uploaded.
 *
 * When buffer length <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {(offset: number) => NodeJS.ReadableStream} streamFactory Returns a Node.js Readable stream starting
 *                                                                  from the offset defined
 * @param {number} size Size of the block blob
 * @param {BlockBlobURL} blockBlobURL BlockBlobURL
 * @param {IUploadToBlockBlobOptions} [options] IUploadToBlockBlobOptions
 * @returns {(Promise<BlobUploadCommonResponse>)} ICommonResponse
 */
function uploadResetableStreamToBlockBlob(aborter, streamFactory, size, blockBlobURL, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var numBlocks, blockList, blockIDPrefix, transferProgress, batch, _loop_1, i;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blockSize) {
                        options.blockSize = 0;
                    }
                    if (options.blockSize < 0 || options.blockSize > BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES) {
                        throw new RangeError("blockSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES);
                    }
                    if (options.maxSingleShotSize !== 0 && !options.maxSingleShotSize) {
                        options.maxSingleShotSize = BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES;
                    }
                    if (options.maxSingleShotSize < 0 ||
                        options.maxSingleShotSize > BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES) {
                        throw new RangeError("maxSingleShotSize option must be >= 0 and <= " + BLOCK_BLOB_MAX_UPLOAD_BLOB_BYTES);
                    }
                    if (options.blockSize === 0) {
                        if (size > BLOCK_BLOB_MAX_BLOCKS * BLOCK_BLOB_MAX_STAGE_BLOCK_BYTES) {
                            throw new RangeError(size + " is too larger to upload to a block blob.");
                        }
                        if (size > options.maxSingleShotSize) {
                            options.blockSize = Math.ceil(size / BLOCK_BLOB_MAX_BLOCKS);
                            if (options.blockSize < DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES) {
                                options.blockSize = DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES;
                            }
                        }
                    }
                    if (!options.blobHTTPHeaders) {
                        options.blobHTTPHeaders = {};
                    }
                    if (!options.blobAccessConditions) {
                        options.blobAccessConditions = {};
                    }
                    if (size <= options.maxSingleShotSize) {
                        return [2 /*return*/, blockBlobURL.upload(aborter, function () { return streamFactory(0); }, size, options)];
                    }
                    numBlocks = Math.floor((size - 1) / options.blockSize) + 1;
                    if (numBlocks > BLOCK_BLOB_MAX_BLOCKS) {
                        throw new RangeError("The buffer's size is too big or the BlockSize is too small;" +
                            ("the number of blocks must be <= " + BLOCK_BLOB_MAX_BLOCKS));
                    }
                    blockList = [];
                    blockIDPrefix = msRest.generateUuid();
                    transferProgress = 0;
                    batch = new Batch(options.parallelism);
                    _loop_1 = function (i) {
                        batch.addOperation(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                            var blockID, start, end, contentLength;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        blockID = generateBlockID(blockIDPrefix, i);
                                        start = options.blockSize * i;
                                        end = i === numBlocks - 1 ? size : start + options.blockSize;
                                        contentLength = end - start;
                                        blockList.push(blockID);
                                        return [4 /*yield*/, blockBlobURL.stageBlock(aborter, blockID, function () { return streamFactory(start, contentLength); }, contentLength, {
                                                leaseAccessConditions: options.blobAccessConditions.leaseAccessConditions
                                            })];
                                    case 1:
                                        _a.sent();
                                        // Update progress after block is successfully uploaded to server, in case of block trying
                                        transferProgress += contentLength;
                                        if (options.progress) {
                                            options.progress({ loadedBytes: transferProgress });
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); });
                    };
                    for (i = 0; i < numBlocks; i++) {
                        _loop_1(i);
                    }
                    return [4 /*yield*/, batch.do()];
                case 1:
                    _a.sent();
                    return [2 /*return*/, blockBlobURL.commitBlockList(aborter, blockList, options)];
            }
        });
    });
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Downloads an Azure Blob in parallel to a buffer.
 * Offset and count are optional, pass 0 for both to download the entire blob.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Buffer} buffer Buffer to be fill, must have length larger than count
 * @param {BlobURL} blobURL A BlobURL object
 * @param {number} offset From which position of the block blob to download, in bytes
 * @param {number} [count] How much data in bytes to be downloaded. Will download to the end when passing undefined
 * @param {IDownloadFromBlobOptions} [options] IDownloadFromBlobOptions
 * @returns {Promise<void>}
 */
function downloadBlobToBuffer(aborter, buffer, blobURL, offset, count, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var response, transferProgress, batch, _loop_2, off;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blockSize) {
                        options.blockSize = 0;
                    }
                    if (options.blockSize < 0) {
                        throw new RangeError("blockSize option must be >= 0");
                    }
                    if (options.blockSize === 0) {
                        options.blockSize = DEFAULT_BLOB_DOWNLOAD_BLOCK_BYTES;
                    }
                    if (offset < 0) {
                        throw new RangeError("offset option must be >= 0");
                    }
                    if (count && count <= 0) {
                        throw new RangeError("count option must be > 0");
                    }
                    if (!options.blobAccessConditions) {
                        options.blobAccessConditions = {};
                    }
                    if (!!count) return [3 /*break*/, 2];
                    return [4 /*yield*/, blobURL.getProperties(aborter, options)];
                case 1:
                    response = _a.sent();
                    count = response.contentLength - offset;
                    if (count < 0) {
                        throw new RangeError("offset " + offset + " shouldn't be larger than blob size " + response.contentLength);
                    }
                    _a.label = 2;
                case 2:
                    if (buffer.length < count) {
                        throw new RangeError("The buffer's size should be equal to or larger than the request count of bytes: " + count);
                    }
                    transferProgress = 0;
                    batch = new Batch(options.parallelism);
                    _loop_2 = function (off) {
                        batch.addOperation(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                            var chunkEnd, response, stream;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        chunkEnd = offset + count;
                                        if (off + options.blockSize < chunkEnd) {
                                            chunkEnd = off + options.blockSize;
                                        }
                                        return [4 /*yield*/, blobURL.download(aborter, off, chunkEnd - off, {
                                                blobAccessConditions: options.blobAccessConditions,
                                                maxRetryRequests: options.maxRetryRequestsPerBlock
                                            })];
                                    case 1:
                                        response = _a.sent();
                                        stream = response.readableStreamBody;
                                        return [4 /*yield*/, streamToBuffer(stream, buffer, off - offset, chunkEnd - offset)];
                                    case 2:
                                        _a.sent();
                                        // Update progress after block is downloaded, in case of block trying
                                        // Could provide finer grained progress updating inside HTTP requests,
                                        // only if convenience layer download try is enabled
                                        transferProgress += chunkEnd - off;
                                        if (options.progress) {
                                            options.progress({ loadedBytes: transferProgress });
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); });
                    };
                    for (off = offset; off < offset + count; off = off + options.blockSize) {
                        _loop_2(off);
                    }
                    return [4 /*yield*/, batch.do()];
                case 3:
                    _a.sent();
                    return [2 /*return*/];
            }
        });
    });
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Uploads a Node.js Readable stream into block blob.
 *
 * PERFORMANCE IMPROVEMENT TIPS:
 * * Input stream highWaterMark is better to set a same value with bufferSize
 *    parameter, which will avoid Buffer.concat() operations.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Readable} stream Node.js Readable stream
 * @param {BlockBlobURL} blockBlobURL A BlockBlobURL instance
 * @param {number} bufferSize Size of every buffer allocated, also the block size in the uploaded block blob
 * @param {number} maxBuffers Max buffers will allocate during uploading, positive correlation
 *                            with max uploading concurrency
 * @param {IUploadStreamToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
function uploadStreamToBlockBlob(aborter, stream, blockBlobURL, bufferSize, maxBuffers, options) {
    if (options === void 0) { options = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var blockNum, blockIDPrefix, transferProgress, blockList, scheduler;
        var _this = this;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    if (!options.blobHTTPHeaders) {
                        options.blobHTTPHeaders = {};
                    }
                    if (!options.accessConditions) {
                        options.accessConditions = {};
                    }
                    blockNum = 0;
                    blockIDPrefix = msRest.generateUuid();
                    transferProgress = 0;
                    blockList = [];
                    scheduler = new BufferScheduler(stream, bufferSize, maxBuffers, function (buffer) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                        var blockID;
                        return tslib_1.__generator(this, function (_a) {
                            switch (_a.label) {
                                case 0:
                                    blockID = generateBlockID(blockIDPrefix, blockNum);
                                    blockList.push(blockID);
                                    blockNum++;
                                    return [4 /*yield*/, blockBlobURL.stageBlock(aborter, blockID, buffer, buffer.length, {
                                            leaseAccessConditions: options.accessConditions.leaseAccessConditions
                                        })];
                                case 1:
                                    _a.sent();
                                    // Update progress after block is successfully uploaded to server, in case of block trying
                                    transferProgress += buffer.length;
                                    if (options.progress) {
                                        options.progress({ loadedBytes: transferProgress });
                                    }
                                    return [2 /*return*/];
                            }
                        });
                    }); }, 
                    // Parallelism should set a smaller value than maxBuffers, which is helpful to
                    // reduce the possibility when a outgoing handler waits for stream data, in
                    // this situation, outgoing handlers are blocked.
                    // Outgoing queue shouldn't be empty.
                    Math.ceil((maxBuffers / 4) * 3));
                    return [4 /*yield*/, scheduler.do()];
                case 1:
                    _a.sent();
                    return [2 /*return*/, blockBlobURL.commitBlockList(aborter, blockList, options)];
            }
        });
    });
}

/**
 * PageBlobURL defines a set of operations applicable to page blobs.
 *
 * @export
 * @class PageBlobURL
 * @extends {StorageURL}
 */
var PageBlobURL = /** @class */ (function (_super) {
    tslib_1.__extends(PageBlobURL, _super);
    /**
     * Creates an instance of PageBlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to a page blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage page blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/pageblob". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/pageblob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof PageBlobURL
     */
    function PageBlobURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.pageBlobContext = new PageBlob(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a PageBlobURL object from ContainerURL instance.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName A page blob name
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.fromContainerURL = function (containerURL, blobName) {
        return new PageBlobURL(appendToURLPath(containerURL.url, encodeURIComponent(blobName)), containerURL.pipeline);
    };
    /**
     * Creates a PageBlobURL object from BlobURL instance.
     *
     * @static
     * @param {BlobURL} blobURL
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.fromBlobURL = function (blobURL) {
        return new PageBlobURL(blobURL.url, blobURL.pipeline);
    };
    /**
     * Creates a new PageBlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.withPipeline = function (pipeline) {
        return new PageBlobURL(this.url, pipeline);
    };
    /**
     * Creates a new PageBlobURL object identical to the source but with the
     * specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {PageBlobURL}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.withSnapshot = function (snapshot) {
        return new PageBlobURL(setURLParameter(this.url, URLConstants.Parameters.SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline);
    };
    /**
     * Creates a page blob of the specified length. Call uploadPages to upload data
     * data to a page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} size
     * @param {IPageBlobCreateOptions} [options]
     * @returns {Promise<Models.PageBlobCreateResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.create = function (aborter, size, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.pageBlobContext.create(0, size, {
                        abortSignal: aborter,
                        blobHTTPHeaders: options.blobHTTPHeaders,
                        blobSequenceNumber: options.blobSequenceNumber,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        metadata: options.metadata,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        cpkInfo: options.customerProvidedKey,
                        tier: toAccessTier(options.tier)
                    })];
            });
        });
    };
    /**
     * Writes 1 or more pages to the page blob. The start and end offsets must be a multiple of 512.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-page
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {HttpRequestBody} body
     * @param {number} offset Offset of destination page blob
     * @param {number} count Content length of the body, also number of bytes to be uploaded
     * @param {IPageBlobUploadPagesOptions} [options]
     * @returns {Promise<Models.PageBlobsUploadPagesResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.uploadPages = function (aborter, body, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.pageBlobContext.uploadPages(body, count, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        onUploadProgress: options.progress,
                        range: rangeToString({ offset: offset, count: count }),
                        sequenceNumberAccessConditions: options.accessConditions.sequenceNumberAccessConditions,
                        transactionalContentMD5: options.transactionalContentMD5,
                        transactionalContentCrc64: options.transactionalContentCrc64,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * The Upload Pages operation writes a range of pages to a page blob where the
     * contents are read from a URL.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/put-page-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} sourceURL Specify a URL to the copy source, Shared Access Signature(SAS) maybe needed for authentication
     * @param {number} sourceOffset The source offset to copy from. Pass 0 to copy from the beginning of source page blob
     * @param {number} destOffset Offset of destination page blob
     * @param {number} count Number of bytes to be uploaded from source page blob
     * @param {IPageBlobUploadPagesFromURLOptions} [options={}]
     * @returns {Promise<Models.PageBlobUploadPagesFromURLResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.uploadPagesFromURL = function (aborter, sourceURL, sourceOffset, destOffset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                options.sourceModifiedAccessConditions = options.sourceModifiedAccessConditions || {};
                ensureCpkIfSpecified(options.customerProvidedKey, this.isHttps);
                return [2 /*return*/, this.pageBlobContext.uploadPagesFromURL(sourceURL, rangeToString({ offset: sourceOffset, count: count }), 0, rangeToString({ offset: destOffset, count: count }), {
                        abortSignal: aborter,
                        sourceContentMD5: options.sourceContentMD5,
                        sourceContentCrc64: options.sourceContentCrc64,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        sequenceNumberAccessConditions: options.accessConditions.sequenceNumberAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        sourceModifiedAccessConditions: {
                            sourceIfMatch: options.sourceModifiedAccessConditions.ifMatch,
                            sourceIfModifiedSince: options.sourceModifiedAccessConditions.ifModifiedSince,
                            sourceIfNoneMatch: options.sourceModifiedAccessConditions.ifNoneMatch,
                            sourceIfUnmodifiedSince: options.sourceModifiedAccessConditions.ifUnmodifiedSince
                        },
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Frees the specified pages from the page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-page
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset
     * @param {number} count
     * @param {IPageBlobClearPagesOptions} [options]
     * @returns {Promise<Models.PageBlobClearPagesResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.clearPages = function (aborter, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.clearPages(0, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        range: rangeToString({ offset: offset, count: count }),
                        sequenceNumberAccessConditions: options.accessConditions.sequenceNumberAccessConditions,
                        cpkInfo: options.customerProvidedKey
                    })];
            });
        });
    };
    /**
     * Returns the list of valid page ranges for a page blob or snapshot of a page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/get-page-ranges
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset
     * @param {number} count
     * @param {IPageBlobGetPageRangesOptions} [options]
     * @returns {Promise<Models.PageBlobGetPageRangesResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.getPageRanges = function (aborter, offset, count, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.getPageRanges({
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        range: rangeToString({ offset: offset, count: count })
                    })];
            });
        });
    };
    /**
     * Gets the collection of page ranges that differ between a specified snapshot and this page blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/get-page-ranges
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset
     * @param {number} count
     * @param {string} prevSnapshot
     * @param {IPageBlobGetPageRangesDiffOptions} [options]
     * @returns {Promise<Models.PageBlobGetPageRangesDiffResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.getPageRangesDiff = function (aborter, offset, count, prevSnapshot, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.getPageRangesDiff({
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions,
                        prevsnapshot: prevSnapshot,
                        range: rangeToString({ offset: offset, count: count })
                    })];
            });
        });
    };
    /**
     * Resizes the page blob to the specified size (which must be a multiple of 512).
     * @see https://docs.microsoft.com/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} size
     * @param {IPageBlobResizeOptions} [options]
     * @returns {Promise<Models.PageBlobResizeResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.resize = function (aborter, size, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.resize(size, {
                        abortSignal: aborter,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Sets a page blob's sequence number.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.SequenceNumberActionType} sequenceNumberAction
     * @param {number} [sequenceNumber] Required if sequenceNumberAction is max or update
     * @param {IPageBlobUpdateSequenceNumberOptions} [options]
     * @returns {Promise<Models.PageBlobUpdateSequenceNumberResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.updateSequenceNumber = function (aborter, sequenceNumberAction, sequenceNumber, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                options.accessConditions = options.accessConditions || {};
                return [2 /*return*/, this.pageBlobContext.updateSequenceNumber(sequenceNumberAction, {
                        abortSignal: aborter,
                        blobSequenceNumber: sequenceNumber,
                        leaseAccessConditions: options.accessConditions.leaseAccessConditions,
                        modifiedAccessConditions: options.accessConditions.modifiedAccessConditions
                    })];
            });
        });
    };
    /**
     * Begins an operation to start an incremental copy from one page blob's snapshot to this page blob.
     * The snapshot is copied such that only the differential changes between the previously
     * copied snapshot are transferred to the destination.
     * The copied snapshots are complete copies of the original snapshot and can be read or copied from as usual.
     * @see https://docs.microsoft.com/rest/api/storageservices/incremental-copy-blob
     * @see https://docs.microsoft.com/en-us/azure/virtual-machines/windows/incremental-snapshots
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource Specifies the name of the source page blob snapshot. For example,
     *                            https://myaccount.blob.core.windows.net/mycontainer/myblob?snapshot=<DateTime>
     * @param {IPageBlobStartCopyIncrementalOptions} [options]
     * @returns {Promise<Models.PageBlobCopyIncrementalResponse>}
     * @memberof PageBlobURL
     */
    PageBlobURL.prototype.startCopyIncremental = function (aborter, copySource, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.pageBlobContext.copyIncremental(copySource, {
                        abortSignal: aborter,
                        modifiedAccessConditions: options.modifiedAccessConditions
                    })];
            });
        });
    };
    return PageBlobURL;
}(BlobURL));

function getBodyAsText(batchResponse) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var buffer, responseLength;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    buffer = Buffer.alloc(BATCH_MAX_PAYLOAD_IN_BYTES);
                    return [4 /*yield*/, streamToBuffer2(batchResponse.readableStreamBody, buffer)];
                case 1:
                    responseLength = _a.sent();
                    // Slice the buffer to trim the empty ending.
                    buffer = buffer.slice(0, responseLength);
                    return [2 /*return*/, buffer.toString()];
            }
        });
    });
}
function utf8ByteLength(str) {
    return Buffer.byteLength(str);
}

var HTTP_HEADER_DELIMITER = ": ";
var SPACE_DELIMITER = " ";
var NOT_FOUND = -1;
/**
 * Util class for parsing batch response.
 */
var BatchResponseParser = /** @class */ (function () {
    function BatchResponseParser(batchResponse, subRequests) {
        if (!batchResponse || !batchResponse.contentType) {
            // In special case(reported), server may return invalid content-type which could not be parsed.
            throw new RangeError("batchResponse is malformed or doesn't contain valid content-type.");
        }
        if (!subRequests || subRequests.size === 0) {
            // This should be prevent during coding.
            throw new RangeError("Invalid state: subRequests is not provided or size is 0.");
        }
        this.batchResponse = batchResponse;
        this.subRequests = subRequests;
        this.responseBatchBoundary = this.batchResponse.contentType.split("=")[1];
        this.perResponsePrefix = "--" + this.responseBatchBoundary + HTTP_LINE_ENDING;
        this.batchResponseEnding = "--" + this.responseBatchBoundary + "--";
    }
    // For example of response, please refer to https://docs.microsoft.com/en-us/rest/api/storageservices/blob-batch#response
    BatchResponseParser.prototype.parseBatchResponse = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var responseBodyAsText, subResponses, subResponseCount, deserializedSubResponses, subResponsesSucceededCount, subResponsesFailedCount, index, subResponse, deserializedSubResponse, responseLines, subRespHeaderStartFound, subRespHeaderEndFound, subRespFailed, contentId, _i, responseLines_1, responseLine, tokens, tokens;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        // When logic reach here, suppose batch request has already succeeded with 202, so we can further parse
                        // sub request's response.
                        if (this.batchResponse._response.status != HTTPURLConnection.HTTP_ACCEPTED) {
                            throw new Error("Invalid state: batch request failed with status: '" + this.batchResponse._response.status + "'.");
                        }
                        return [4 /*yield*/, getBodyAsText(this.batchResponse)];
                    case 1:
                        responseBodyAsText = _a.sent();
                        subResponses = responseBodyAsText
                            .split(this.batchResponseEnding)[0] // string after ending is useless
                            .split(this.perResponsePrefix)
                            .slice(1);
                        subResponseCount = subResponses.length;
                        // Defensive coding in case of potential error parsing.
                        // Note: subResponseCount == 1 is special case where sub request is invalid.
                        // We try to prevent such cases through early validation, e.g. validate sub request count >= 1.
                        // While in unexpected sub request invalid case, we allow sub response to be parsed and return to user.
                        if (subResponseCount != this.subRequests.size && subResponseCount != 1) {
                            throw new Error("Invalid state: sub responses' count is not equal to sub requests' count.");
                        }
                        deserializedSubResponses = new Array(subResponseCount);
                        subResponsesSucceededCount = 0;
                        subResponsesFailedCount = 0;
                        // Parse sub subResponses.
                        for (index = 0; index < subResponseCount; index++) {
                            subResponse = subResponses[index];
                            deserializedSubResponses[index] = {};
                            deserializedSubResponse = deserializedSubResponses[index];
                            deserializedSubResponse.headers = new msRest.HttpHeaders();
                            responseLines = subResponse.split("" + HTTP_LINE_ENDING);
                            subRespHeaderStartFound = false;
                            subRespHeaderEndFound = false;
                            subRespFailed = false;
                            contentId = NOT_FOUND;
                            for (_i = 0, responseLines_1 = responseLines; _i < responseLines_1.length; _i++) {
                                responseLine = responseLines_1[_i];
                                if (!subRespHeaderStartFound) {
                                    // Convention line to indicate content ID
                                    if (responseLine.startsWith(HeaderConstants.CONTENT_ID)) {
                                        contentId = parseInt(responseLine.split(HTTP_HEADER_DELIMITER)[1]);
                                    }
                                    // Http version line with status code indicates the start of sub request's response.
                                    // Example: HTTP/1.1 202 Accepted 
                                    if (responseLine.startsWith(HTTP_VERSION_1_1)) {
                                        subRespHeaderStartFound = true;
                                        tokens = responseLine.split(SPACE_DELIMITER);
                                        deserializedSubResponse.status = parseInt(tokens[1]);
                                        deserializedSubResponse.statusMessage = tokens.slice(2).join(SPACE_DELIMITER);
                                    }
                                    continue; // Skip convention headers not specifically for sub request i.e. Content-Type: application/http and Content-ID: *
                                }
                                if (responseLine.trim() === "") {
                                    // Sub response's header start line already found, and the first empty line indicates header end line found. 
                                    if (!subRespHeaderEndFound) {
                                        subRespHeaderEndFound = true;
                                    }
                                    continue; // Skip empty line
                                }
                                // Note: when code reach here, it indicates subRespHeaderStartFound == true
                                if (!subRespHeaderEndFound) {
                                    if (responseLine.indexOf(HTTP_HEADER_DELIMITER) === -1) {
                                        // Defensive coding to prevent from missing valuable lines.
                                        throw new Error("Invalid state: find non-empty line '" + responseLine + "' without HTTP header delimiter '" + HTTP_HEADER_DELIMITER + "'.");
                                    }
                                    tokens = responseLine.split(HTTP_HEADER_DELIMITER);
                                    deserializedSubResponse.headers.set(tokens[0], tokens[1]);
                                    if (tokens[0] === HeaderConstants.X_MS_ERROR_CODE) {
                                        deserializedSubResponse.errorCode = tokens[1];
                                        subRespFailed = true;
                                    }
                                }
                                else {
                                    // Assemble body of sub response.
                                    if (!deserializedSubResponse.bodyAsText) {
                                        deserializedSubResponse.bodyAsText = "";
                                    }
                                    deserializedSubResponse.bodyAsText += responseLine;
                                }
                            } // Inner for end
                            if (contentId != NOT_FOUND) {
                                deserializedSubResponse._request = this.subRequests.get(contentId);
                            }
                            if (subRespFailed) {
                                subResponsesFailedCount++;
                            }
                            else {
                                subResponsesSucceededCount++;
                            }
                        }
                        return [2 /*return*/, {
                                subResponses: deserializedSubResponses,
                                subResponsesSucceededCount: subResponsesSucceededCount,
                                subResponsesFailedCount: subResponsesFailedCount
                            }];
                }
            });
        });
    };
    return BatchResponseParser;
}());

/**
 * A ServiceURL represents a URL to the Azure Storage Blob service allowing you
 * to manipulate blob containers.
 *
 * @export
 * @class ServiceURL
 * @extends {StorageURL}
 */
var ServiceURL = /** @class */ (function (_super) {
    tslib_1.__extends(ServiceURL, _super);
    /**
     * Creates an instance of ServiceURL.
     *
     * @param {string} url A URL string pointing to Azure Storage blob service, such as
     *                     "https://myaccount.blob.core.windows.net". You can append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.blob.core.windows.net?sasString".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof ServiceURL
     */
    function ServiceURL(url, pipeline) {
        var _this = _super.call(this, url, pipeline) || this;
        _this.serviceContext = new Service(_this.storageClientContext);
        return _this;
    }
    /**
     * Creates a new ServiceURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {ServiceURL}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.withPipeline = function (pipeline) {
        return new ServiceURL(this.url, pipeline);
    };
    /**
     * Gets the properties of a storage account’s Blob service, including properties
     * for Storage Analytics and CORS (Cross-Origin Resource Sharing) rules.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-service-properties}
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.ServiceGetPropertiesResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getProperties = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.getProperties({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Sets properties for a storage account’s Blob service endpoint, including properties
     * for Storage Analytics, CORS (Cross-Origin Resource Sharing) rules and soft delete settings.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-service-properties}
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.StorageServiceProperties} properties
     * @returns {Promise<Models.ServiceSetPropertiesResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.setProperties = function (aborter, properties) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.setProperties(properties, {
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Retrieves statistics related to replication for the Blob service. It is only
     * available on the secondary location endpoint when read-access geo-redundant
     * replication is enabled for the storage account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-service-stats}
     *
     *  @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.ServiceGetStatisticsResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getStatistics = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.getStatistics({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * The Get Account Information operation returns the sku name and account kind
     * for the specified account.
     * The Get Account Information operation is available on service versions beginning
     * with version 2018-03-28.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-account-information
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.ServiceGetAccountInfoResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getAccountInfo = function (aborter) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.getAccountInfo({
                        abortSignal: aborter
                    })];
            });
        });
    };
    /**
     * Returns a list of the containers under the specified account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/list-containers2
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} [marker] A string value that identifies the portion of
     *                          the list of containers to be returned with the next listing operation. The
     *                          operation returns the NextMarker value within the response body if the
     *                          listing operation did not return all containers remaining to be listed
     *                          with the current page. The NextMarker value can be used as the value for
     *                          the marker parameter in a subsequent call to request the next page of list
     *                          items. The marker value is opaque to the client.
     * @param {IServiceListContainersSegmentOptions} [options]
     * @returns {Promise<Models.ServiceListContainersSegmentResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.listContainersSegment = function (aborter, marker, options) {
        if (options === void 0) { options = {}; }
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.serviceContext.listContainersSegment(tslib_1.__assign({ abortSignal: aborter, marker: marker }, options))];
            });
        });
    };
    /**
     * ONLY AVAILABLE WHEN USING BEARER TOKEN AUTHENTICATION (TokenCredential).
     *
     * Retrieves a user delegation key for the Blob service. This is only a valid operation when using
     * bearer token authentication.
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-user-delegation-key
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Date} start      The start time for the user delegation SAS. Must be within 7 days of the current time
     * @param {Date} expiry     The end time for the user delegation SAS. Must be within 7 days of the current time
     * @returns {Promise<ServiceGetUserDelegationKeyResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.getUserDelegationKey = function (aborter, start, expiry) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var response, userDelegationKey, res;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.serviceContext.getUserDelegationKey({
                            start: truncatedISO8061Date(start, false),
                            expiry: truncatedISO8061Date(expiry, false)
                        }, {
                            abortSignal: aborter
                        })];
                    case 1:
                        response = _a.sent();
                        userDelegationKey = {
                            signedOid: response.signedOid,
                            signedTid: response.signedTid,
                            signedStart: new Date(response.signedStart),
                            signedExpiry: new Date(response.signedExpiry),
                            signedService: response.signedService,
                            signedVersion: response.signedVersion,
                            value: response.value
                        };
                        res = tslib_1.__assign({ _response: response._response, requestId: response.requestId, clientRequestId: response.clientRequestId, version: response.version, date: response.date, errorCode: response.errorCode }, userDelegationKey);
                        return [2 /*return*/, res];
                }
            });
        });
    };
    /**
     * Submit batch request which consists of multiple subrequests.
     *
     * @example
     * let batchDeleteRequest = new BatchDeleteRequest();
     * await batchDeleteRequest.addSubRequest(urlInString0, credential0);
     * await batchDeleteRequest.addSubRequest(urlInString1, credential1, {
     *  deleteSnapshots: "include"
     * });
     * const deleteBatchResp = await serviceURL.submitBatch(Aborter.none, batchDeleteRequest);
     * console.log(deleteBatchResp.subResponsesSucceededCount);
     *
     * @example
     * let batchSetTierRequest = new BatchSetTierRequest();
     * await batchSetTierRequest.addSubRequest(blockBlobURL0, "Cool");
     * await batchSetTierRequest.addSubRequest(blockBlobURL1, "Cool", {
     *  leaseAccessConditions: { leaseId: leaseId }
     * });
     * const setTierBatchResp = await serviceURL.submitBatch(Aborter.none, batchSetTierRequest);
     * console.log(setTierBatchResp.subResponsesSucceededCount);
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/blob-batch
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation.
     * @param {BatchRequest} batchRequest Supported batch request: BatchDeleteRequest or BatchSetTierRequest.
     * @param {Models.ServiceSubmitBatchOptionalParams} [options]
     * @returns {Promise<ServiceSubmitBatchResponse>}
     * @memberof ServiceURL
     */
    ServiceURL.prototype.submitBatch = function (aborter, batchRequest, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var batchRequestBody, rawBatchResponse, batchResponseParser, responseSummary, res;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!batchRequest || batchRequest.getSubRequests().size == 0) {
                            throw new RangeError("Batch request should contain one or more sub requests.");
                        }
                        batchRequestBody = batchRequest.getHttpRequestBody();
                        return [4 /*yield*/, this.serviceContext.submitBatch(batchRequestBody, utf8ByteLength(batchRequestBody), batchRequest.getMultiPartContentType(), tslib_1.__assign({ abortSignal: aborter }, options))];
                    case 1:
                        rawBatchResponse = _a.sent();
                        batchResponseParser = new BatchResponseParser(rawBatchResponse, batchRequest.getSubRequests());
                        return [4 /*yield*/, batchResponseParser.parseBatchResponse()];
                    case 2:
                        responseSummary = _a.sent();
                        res = {
                            _response: rawBatchResponse._response,
                            contentType: rawBatchResponse.contentType,
                            errorCode: rawBatchResponse.errorCode,
                            requestId: rawBatchResponse.requestId,
                            clientRequestId: rawBatchResponse.clientRequestId,
                            version: rawBatchResponse.version,
                            subResponses: responseSummary.subResponses,
                            subResponsesSucceededCount: responseSummary.subResponsesSucceededCount,
                            subResponsesFailedCount: responseSummary.subResponsesFailedCount
                        };
                        return [2 /*return*/, res];
                }
            });
        });
    };
    return ServiceURL;
}(StorageURL));

Object.defineProperty(exports, 'BaseRequestPolicy', {
    enumerable: true,
    get: function () {
        return msRest.BaseRequestPolicy;
    }
});
Object.defineProperty(exports, 'HttpHeaders', {
    enumerable: true,
    get: function () {
        return msRest.HttpHeaders;
    }
});
Object.defineProperty(exports, 'HttpPipelineLogLevel', {
    enumerable: true,
    get: function () {
        return msRest.HttpPipelineLogLevel;
    }
});
Object.defineProperty(exports, 'RequestPolicyOptions', {
    enumerable: true,
    get: function () {
        return msRest.RequestPolicyOptions;
    }
});
Object.defineProperty(exports, 'RestError', {
    enumerable: true,
    get: function () {
        return msRest.RestError;
    }
});
Object.defineProperty(exports, 'WebResource', {
    enumerable: true,
    get: function () {
        return msRest.WebResource;
    }
});
Object.defineProperty(exports, 'deserializationPolicy', {
    enumerable: true,
    get: function () {
        return msRest.deserializationPolicy;
    }
});
exports.Aborter = Aborter;
exports.AccountSASPermissions = AccountSASPermissions;
exports.AccountSASResourceTypes = AccountSASResourceTypes;
exports.AccountSASServices = AccountSASServices;
exports.AnonymousCredential = AnonymousCredential;
exports.AnonymousCredentialPolicy = AnonymousCredentialPolicy;
exports.AppendBlobURL = AppendBlobURL;
exports.BatchDeleteRequest = BatchDeleteRequest;
exports.BatchRequest = BatchRequest;
exports.BatchSetTierRequest = BatchSetTierRequest;
exports.BlobSASPermissions = BlobSASPermissions;
exports.BlobURL = BlobURL;
exports.BlockBlobURL = BlockBlobURL;
exports.BrowserPolicyFactory = BrowserPolicyFactory;
exports.ContainerSASPermissions = ContainerSASPermissions;
exports.ContainerURL = ContainerURL;
exports.Credential = Credential;
exports.CredentialPolicy = CredentialPolicy;
exports.LoggingPolicyFactory = LoggingPolicyFactory;
exports.Models = index;
exports.PageBlobURL = PageBlobURL;
exports.Pipeline = Pipeline;
exports.RetryPolicyFactory = RetryPolicyFactory;
exports.SASQueryParameters = SASQueryParameters;
exports.ServiceURL = ServiceURL;
exports.SharedKeyCredential = SharedKeyCredential;
exports.SharedKeyCredentialPolicy = SharedKeyCredentialPolicy;
exports.StorageURL = StorageURL;
exports.TelemetryPolicyFactory = TelemetryPolicyFactory;
exports.TokenCredential = TokenCredential;
exports.TokenCredentialPolicy = TokenCredentialPolicy;
exports.UniqueRequestIDPolicyFactory = UniqueRequestIDPolicyFactory;
exports.downloadBlobToBuffer = downloadBlobToBuffer;
exports.generateAccountSASQueryParameters = generateAccountSASQueryParameters;
exports.generateBlobSASQueryParameters = generateBlobSASQueryParameters;
exports.uploadBrowserDataToBlockBlob = uploadBrowserDataToBlockBlob;
exports.uploadFileToBlockBlob = uploadFileToBlockBlob;
exports.uploadStreamToBlockBlob = uploadStreamToBlockBlob;
//# sourceMappingURL=index.js.map
