import { HttpRequestBody, TransferProgressEvent } from "@azure/ms-rest-js";
import { Aborter } from "./Aborter";
import { BlobURL } from "./BlobURL";
import { ContainerURL } from "./ContainerURL";
import * as Models from "./generated/src/models";
import { IAppendBlobAccessConditions, IBlobAccessConditions, IMetadata } from "./models";
import { Pipeline } from "./Pipeline";
export interface IAppendBlobCreateOptions {
    accessConditions?: IBlobAccessConditions;
    blobHTTPHeaders?: Models.BlobHTTPHeaders;
    metadata?: IMetadata;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IAppendBlobAppendBlockOptions {
    accessConditions?: IAppendBlobAccessConditions;
    progress?: (progress: TransferProgressEvent) => void;
    /**
     * An MD5 hash of the block content. This hash is used to verify the integrity of the block during transport.
     * When this is specified, the storage service compares the hash of the content that has arrived with this value.
     *
     * transactionalContentMD5 and transactionalContentCrc64 cannot be set at same time.
     */
    transactionalContentMD5?: Uint8Array;
    /**
     * A CRC64 hash of the append block content. This hash is used to verify the integrity of the append block during transport.
     * When this is specified, the storage service compares the hash of the content that has arrived with this value.
     *
     * transactionalContentMD5 and transactionalContentCrc64 cannot be set at same time.
     */
    transactionalContentCrc64?: Uint8Array;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IAppendBlobAppendBlockFromURLOptions {
    accessConditions?: IAppendBlobAccessConditions;
    sourceModifiedAccessConditions?: Models.ModifiedAccessConditions;
    /**
     * An MD5 hash of the append block content from the URI.
     * This hash is used to verify the integrity of the append block during transport of the data from the URI.
     * When this is specified, the storage service compares the hash of the content that has arrived from the copy-source with this value.
     *
     * sourceContentMD5 and sourceContentCrc64 cannot be set at same time.
     */
    sourceContentMD5?: Uint8Array;
    /**
     * A CRC64 hash of the append block content from the URI.
     * This hash is used to verify the integrity of the append block during transport of the data from the URI.
     * When this is specified, the storage service compares the hash of the content that has arrived from the copy-source with this value.
     *
     * sourceContentMD5 and sourceContentCrc64 cannot be set at same time.
     */
    sourceContentCrc64?: Uint8Array;
    customerProvidedKey?: Models.CpkInfo;
}
/**
 * AppendBlobURL defines a set of operations applicable to append blobs.
 *
 * @export
 * @class AppendBlobURL
 * @extends {StorageURL}
 */
export declare class AppendBlobURL extends BlobURL {
    /**
     * Creates a AppendBlobURL object from ContainerURL instance.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName An append blob name
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    static fromContainerURL(containerURL: ContainerURL, blobName: string): AppendBlobURL;
    /**
     * Creates a AppendBlobURL object from BlobURL instance.
     *
     * @static
     * @param {BlobURL} blobURL
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    static fromBlobURL(blobURL: BlobURL): AppendBlobURL;
    /**
     * appendBlobsContext provided by protocol layer.
     *
     * @private
     * @type {AppendBlobs}
     * @memberof AppendBlobURL
     */
    private appendBlobContext;
    /**
     * Creates an instance of AppendBlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to an append blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage append blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/appendblob". You can
     *                     append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/appendblob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof AppendBlobURL
     */
    constructor(url: string, pipeline: Pipeline);
    /**
     * Creates a new AppendBlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    withPipeline(pipeline: Pipeline): AppendBlobURL;
    /**
     * Creates a new AppendBlobURL object identical to the source but with the
     * specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {AppendBlobURL}
     * @memberof AppendBlobURL
     */
    withSnapshot(snapshot: string): AppendBlobURL;
    /**
     * Creates a 0-length append blob. Call AppendBlock to append data to an append blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/put-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IAppendBlobCreateOptions} [options]
     * @returns {Promise<Models.AppendBlobsCreateResponse>}
     * @memberof AppendBlobURL
     */
    create(aborter: Aborter, options?: IAppendBlobCreateOptions): Promise<Models.AppendBlobCreateResponse>;
    /**
     * Commits a new block of data to the end of the existing append blob.
     * @see https://docs.microsoft.com/rest/api/storageservices/append-block
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {HttpRequestBody} body
     * @param {number} contentLength Length of the body in bytes
     * @param {IAppendBlobAppendBlockOptions} [options]
     * @returns {Promise<Models.AppendBlobsAppendBlockResponse>}
     * @memberof AppendBlobURL
     */
    appendBlock(aborter: Aborter, body: HttpRequestBody, contentLength: number, options?: IAppendBlobAppendBlockOptions): Promise<Models.AppendBlobAppendBlockResponse>;
    /**
     * The Append Block operation commits a new block of data to the end of an existing append blob
     * where the contents are read from a source url.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/append-block-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} sourceURL
     *                 The url to the blob that will be the source of the copy. A source blob in the same storage account can
     *                 be authenticated via Shared Key. However, if the source is a blob in another account, the source blob
     *                 must either be public or must be authenticated via a shared access signature. If the source blob is
     *                 public, no authentication is required to perform the operation.
     * @param {number} sourceOffset Offset in source to be appended
     * @param {number} count Number of bytes to be appended as a block
     * @param {IAppendBlobAppendBlockFromURLOptions} [options={}]
     * @returns {Promise<Models.AppendBlobAppendBlockFromUrlResponse>}
     * @memberof AppendBlobURL
     */
    appendBlockFromURL(aborter: Aborter, sourceURL: string, sourceOffset: number, count: number, options?: IAppendBlobAppendBlockFromURLOptions): Promise<Models.AppendBlobAppendBlockFromUrlResponse>;
}
//# sourceMappingURL=AppendBlobURL.d.ts.map