import { TransferProgressEvent } from "@azure/ms-rest-js";
import { Aborter } from "./Aborter";
import { ContainerURL } from "./ContainerURL";
import * as Models from "./generated/src/models";
import { IBlobAccessConditions, IMetadata, BlockBlobTier, PremiumPageBlobTier } from "./models";
import { Pipeline } from "./Pipeline";
import { StorageURL } from "./StorageURL";
export interface IBlobDownloadOptions {
    /**
     * An opaque DateTime string value that, when present, specifies the blob snapshot to retrieve.
     */
    snapshot?: string;
    /**
     * When this is set to true and download range of blob, the service returns the MD5 hash for the range,
     * as long as the range is less than or equal to 4 MB in size.
     *
     * rangeGetContentCrc64 and rangeGetContentMD5 cannot be set at same time.
     */
    rangeGetContentMD5?: boolean;
    /**
     * When this is set to true and download range of blob, the service returns the CRC64 hash for the range,
     * as long as the range is less than or equal to 4 MB in size.
     *
     * rangeGetContentCrc64 and rangeGetContentMD5 cannot be set at same time.
     */
    rangeGetContentCrc64?: boolean;
    blobAccessConditions?: IBlobAccessConditions;
    progress?: (progress: TransferProgressEvent) => void;
    /**
     * Optional. ONLY AVAILABLE IN NODE.JS.
     *
     * How many retries will perform when original body download stream unexpected ends.
     * Above kind of ends will not trigger retry policy defined in a pipeline,
     * because they doesn't emit network errors.
     *
     * With this option, every additional retry means an additional FileURL.download() request will be made
     * from the broken point, until the requested range has been successfully downloaded or maxRetryRequests is reached.
     *
     * Default value is 5, please set a larger value when loading large files in poor network.
     *
     * @type {number}
     * @memberof IBlobDownloadOptions
     */
    maxRetryRequests?: number;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IBlobGetPropertiesOptions {
    blobAccessConditions?: IBlobAccessConditions;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IBlobDeleteOptions {
    blobAccessConditions?: IBlobAccessConditions;
    deleteSnapshots?: Models.DeleteSnapshotsOptionType;
}
export interface IBlobSetHTTPHeadersOptions {
    blobAccessConditions?: IBlobAccessConditions;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IBlobSetMetadataOptions {
    blobAccessConditions?: IBlobAccessConditions;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IBlobAcquireLeaseOptions {
    modifiedAccessConditions?: Models.ModifiedAccessConditions;
}
export interface IBlobReleaseLeaseOptions {
    modifiedAccessConditions?: Models.ModifiedAccessConditions;
}
export interface IBlobRenewLeaseOptions {
    modifiedAccessConditions?: Models.ModifiedAccessConditions;
}
export interface IBlobChangeLeaseOptions {
    modifiedAccessConditions?: Models.ModifiedAccessConditions;
}
export interface IBlobBreakLeaseOptions {
    modifiedAccessConditions?: Models.ModifiedAccessConditions;
}
export interface IBlobCreateSnapshotOptions {
    metadata?: IMetadata;
    blobAccessConditions?: IBlobAccessConditions;
    customerProvidedKey?: Models.CpkInfo;
}
export interface IBlobStartCopyFromURLOptions {
    metadata?: IMetadata;
    blobAccessConditions?: IBlobAccessConditions;
    sourceModifiedAccessConditions?: Models.ModifiedAccessConditions;
    tier?: BlockBlobTier | PremiumPageBlobTier | string;
    rehydratePriority?: Models.RehydratePriority;
}
export interface IBlobAbortCopyFromURLOptions {
    leaseAccessConditions?: Models.LeaseAccessConditions;
}
export interface IBlobSyncCopyFromURLOptions {
    metadata?: IMetadata;
    blobAccessConditions?: IBlobAccessConditions;
    sourceModifiedAccessConditions?: Models.ModifiedAccessConditions;
}
export interface IBlobSetTierOptions {
    leaseAccessConditions?: Models.LeaseAccessConditions;
    rehydratePriority?: Models.RehydratePriority;
}
/**
 * A BlobURL represents a URL to an Azure Storage blob; the blob may be a block blob,
 * append blob, or page blob.
 *
 * @export
 * @class BlobURL
 * @extends {StorageURL}
 */
export declare class BlobURL extends StorageURL {
    /**
     * Creates a BlobURL object from an ContainerURL object.
     *
     * @static
     * @param {ContainerURL} containerURL A ContainerURL object
     * @param {string} blobName A blob name
     * @returns
     * @memberof BlobURL
     */
    static fromContainerURL(containerURL: ContainerURL, blobName: string): BlobURL;
    /**
     * blobContext provided by protocol layer.
     *
     * @private
     * @type {Blobs}
     * @memberof BlobURL
     */
    private blobContext;
    /**
     * Creates an instance of BlobURL.
     * This method accepts an encoded URL or non-encoded URL pointing to a blob.
     * Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     * If a blob name includes ? or %, blob name must be encoded in the URL.
     *
     * @param {string} url A URL string pointing to Azure Storage blob, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blob".
     *                     You can append a SAS if using AnonymousCredential, such as
     *                     "https://myaccount.blob.core.windows.net/mycontainer/blob?sasString".
     *                     This method accepts an encoded URL or non-encoded URL pointing to a blob.
     *                     Encoded URL string will NOT be escaped twice, only special characters in URL path will be escaped.
     *                     However, if a blob name includes ? or %, blob name must be encoded in the URL.
     *                     Such as a blob named "my?blob%", the URL should be "https://myaccount.blob.core.windows.net/mycontainer/my%3Fblob%25".
     * @param {Pipeline} pipeline Call StorageURL.newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @memberof BlobURL
     */
    constructor(url: string, pipeline: Pipeline);
    /**
     * Creates a new BlobURL object identical to the source but with the
     * specified request policy pipeline.
     *
     * @param {Pipeline} pipeline
     * @returns {BlobURL}
     * @memberof BlobURL
     */
    withPipeline(pipeline: Pipeline): BlobURL;
    /**
     * Creates a new BlobURL object identical to the source but with the specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base blob.
     *
     * @param {string} snapshot
     * @returns {BlobURL} A new BlobURL object identical to the source but with the specified snapshot timestamp
     * @memberof BlobURL
     */
    withSnapshot(snapshot: string): BlobURL;
    /**
     * Reads or downloads a blob from the system, including its metadata and properties.
     * You can also call Get Blob to read a snapshot.
     *
     * * In Node.js, data returns in a Readable stream readableStreamBody
     * * In browsers, data returns in a promise blobBody
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} offset From which position of the blob to download, >= 0
     * @param {number} [count] How much data to be downloaded, > 0. Will download to the end when undefined
     * @param {IBlobDownloadOptions} [options]
     * @returns {Promise<Models.BlobDownloadResponse>}
     * @memberof BlobURL
     */
    download(aborter: Aborter, offset: number, count?: number, options?: IBlobDownloadOptions): Promise<Models.BlobDownloadResponse>;
    /**
     * Returns all user-defined metadata, standard HTTP properties, and system properties
     * for the blob. It does not return the content of the blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/get-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobGetPropertiesOptions} [options]
     * @returns {Promise<Models.BlobGetPropertiesResponse>}
     * @memberof BlobURL
     */
    getProperties(aborter: Aborter, options?: IBlobGetPropertiesOptions): Promise<Models.BlobGetPropertiesResponse>;
    /**
     * Marks the specified blob or snapshot for deletion. The blob is later deleted
     * during garbage collection. Note that in order to delete a blob, you must delete
     * all of its snapshots. You can delete both at the same time with the Delete
     * Blob operation.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/delete-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobDeleteOptions} [options]
     * @returns {Promise<Models.BlobDeleteResponse>}
     * @memberof BlobURL
     */
    delete(aborter: Aborter, options?: IBlobDeleteOptions): Promise<Models.BlobDeleteResponse>;
    /**
     * Restores the contents and metadata of soft deleted blob and any associated
     * soft deleted snapshots. Undelete Blob is supported only on version 2017-07-29
     * or later.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/undelete-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @returns {Promise<Models.BlobUndeleteResponse>}
     * @memberof BlobURL
     */
    undelete(aborter: Aborter): Promise<Models.BlobUndeleteResponse>;
    /**
     * Sets system properties on the blob.
     *
     * If no value provided, or no value provided for the specificed blob HTTP headers,
     * these blob HTTP headers without a value will be cleared.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-properties
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {Models.BlobHTTPHeaders} [blobHTTPHeaders] If no value provided, or no value provided for
     *                                                   the specificed blob HTTP headers, these blob HTTP
     *                                                   headers without a value will be cleared.
     * @param {IBlobSetHTTPHeadersOptions} [options]
     * @returns {Promise<Models.BlobSetHTTPHeadersResponse>}
     * @memberof BlobURL
     */
    setHTTPHeaders(aborter: Aborter, blobHTTPHeaders?: Models.BlobHTTPHeaders, options?: IBlobSetHTTPHeadersOptions): Promise<Models.BlobSetHTTPHeadersResponse>;
    /**
     * Sets user-defined metadata for the specified blob as one or more name-value pairs.
     *
     * If no option provided, or no metadata defined in the parameter, the blob
     * metadata will be removed.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-metadata
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IMetadata} [metadata] Replace existing metadata with this value.
     *                               If no value provided the existing metadata will be removed.
     * @param {IBlobSetMetadataOptions} [options]
     * @returns {Promise<Models.BlobSetMetadataResponse>}
     * @memberof BlobURL
     */
    setMetadata(aborter: Aborter, metadata?: IMetadata, options?: IBlobSetMetadataOptions): Promise<Models.BlobSetMetadataResponse>;
    /**
     * Establishes and manages a lock on a blob for write and delete operations.
     * The lock duration can be 15 to 60 seconds, or can be infinite.
     * In versions prior to 2012-02-12, the lock duration is 60 seconds.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} proposedLeaseId Can be specified in any valid GUID string format
     * @param {number} duration The lock duration can be 15 to 60 seconds, or can be infinite
     * @param {IBlobAcquireLeaseOptions} [options]
     * @returns {Promise<Models.BlobAcquireLeaseResponse>}
     * @memberof BlobURL
     */
    acquireLease(aborter: Aborter, proposedLeaseId: string, duration: number, options?: IBlobAcquireLeaseOptions): Promise<Models.BlobAcquireLeaseResponse>;
    /**
     * To free the lease if it is no longer needed so that another client may immediately
     * acquire a lease against the blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IBlobReleaseLeaseOptions} [options]
     * @returns {Promise<Models.BlobReleaseLeaseResponse>}
     * @memberof BlobURL
     */
    releaseLease(aborter: Aborter, leaseId: string, options?: IBlobReleaseLeaseOptions): Promise<Models.BlobReleaseLeaseResponse>;
    /**
     * To renew an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {IBlobRenewLeaseOptions} [options]
     * @returns {Promise<Models.BlobRenewLeaseResponse>}
     * @memberof BlobURL
     */
    renewLease(aborter: Aborter, leaseId: string, options?: IBlobRenewLeaseOptions): Promise<Models.BlobRenewLeaseResponse>;
    /**
     * To change the ID of an existing lease.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} leaseId
     * @param {string} proposedLeaseId
     * @param {IBlobChangeLeaseOptions} [options]
     * @returns {Promise<Models.BlobChangeLeaseResponse>}
     * @memberof BlobURL
     */
    changeLease(aborter: Aborter, leaseId: string, proposedLeaseId: string, options?: IBlobChangeLeaseOptions): Promise<Models.BlobChangeLeaseResponse>;
    /**
     * To end the lease but ensure that another client cannot acquire a new lease
     * until the current lease period has expired.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/lease-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {number} [breakPeriod]
     * @param {IBlobBreakLeaseOptions} [options]
     * @returns {Promise<Models.BlobBreakLeaseResponse>}
     * @memberof BlobURL
     */
    breakLease(aborter: Aborter, breakPeriod?: number, options?: IBlobBreakLeaseOptions): Promise<Models.BlobBreakLeaseResponse>;
    /**
     * Creates a read-only snapshot of a blob.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/snapshot-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {IBlobCreateSnapshotOptions} [options]
     * @returns {Promise<Models.BlobCreateSnapshotResponse>}
     * @memberof BlobURL
     */
    createSnapshot(aborter: Aborter, options?: IBlobCreateSnapshotOptions): Promise<Models.BlobCreateSnapshotResponse>;
    /**
     * Asynchronously copies a blob to a destination within the storage account.
     * In version 2012-02-12 and later, the source for a Copy Blob operation can be
     * a committed blob in any Azure storage account.
     * Beginning with version 2015-02-21, the source for a Copy Blob operation can be
     * an Azure file in any Azure storage account.
     * Only storage accounts created on or after June 7th, 2012 allow the Copy Blob
     * operation to copy from another storage account.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/copy-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource
     * @param {IBlobStartCopyFromURLOptions} [options]
     * @returns {Promise<Models.BlobStartCopyFromURLResponse>}
     * @memberof BlobURL
     */
    startCopyFromURL(aborter: Aborter, copySource: string, options?: IBlobStartCopyFromURLOptions): Promise<Models.BlobStartCopyFromURLResponse>;
    /**
     * Aborts a pending asynchronous Copy Blob operation, and leaves a destination blob with zero
     * length and full metadata. Version 2012-02-12 and newer.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/abort-copy-blob
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copyId
     * @param {IBlobAbortCopyFromURLOptions} [options]
     * @returns {Promise<Models.BlobAbortCopyFromURLResponse>}
     * @memberof BlobURL
     */
    abortCopyFromURL(aborter: Aborter, copyId: string, options?: IBlobAbortCopyFromURLOptions): Promise<Models.BlobAbortCopyFromURLResponse>;
    /**
     * The synchronous Copy From URL operation copies a blob or an internet resource to a new blob. It will not
     * return a response until the copy is complete.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/copy-blob-from-url
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {string} copySource The source URL to copy from, Shared Access Signature(SAS) maybe needed for authentication
     * @param {IBlobSyncCopyFromURLOptions} [options={}]
     * @returns {Promise<Models.BlobCopyFromURLResponse>}
     * @memberof BlobURL
     */
    syncCopyFromURL(aborter: Aborter, copySource: string, options?: IBlobSyncCopyFromURLOptions): Promise<Models.BlobCopyFromURLResponse>;
    /**
     * Sets the tier on a blob. The operation is allowed on a page blob in a premium
     * storage account and on a block blob in a blob storage account (locally redundant
     * storage only). A premium page blob's tier determines the allowed size, IOPS,
     * and bandwidth of the blob. A block blob's tier determines Hot/Cool/Archive
     * storage type. This operation does not update the blob's ETag.
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/set-blob-tier
     *
     * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param {BlockBlobTier | PremiumPageBlobTier | string} tier
     * @param {IBlobSetTierOptions} [options]
     * @returns {Promise<Models.BlobsSetTierResponse>}
     * @memberof BlobURL
     */
    setTier(aborter: Aborter, tier: BlockBlobTier | PremiumPageBlobTier | string, options?: IBlobSetTierOptions): Promise<Models.BlobSetTierResponse>;
}
//# sourceMappingURL=BlobURL.d.ts.map