import { IIPRange } from "./IIPRange";
import { UserDelegationKey } from "./ServiceURL";
/**
 * Protocols for generated SAS.
 *
 * @export
 * @enum {number}
 */
export declare enum SASProtocol {
    /**
     * Protocol that allows HTTPS only
     */
    HTTPS = "https",
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    HTTPSandHTTP = "https,http"
}
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link IAccountSASSignatureValues} and {@link IBlobSASSignatureValues}
 * types. Once generated, it can be encoded into a {@code String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 *
 * @export
 * @class SASQueryParameters
 */
export declare class SASQueryParameters {
    /**
     * The storage API version.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly version: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     *
     * @type {SASProtocol}
     * @memberof SASQueryParameters
     */
    readonly protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     *
     * @type {Date}
     * @memberof SASQueryParameters
     */
    readonly startTime?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     *
     * @type {Date}
     * @memberof SASQueryParameters
     */
    readonly expiryTime?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link BlobSASPermissions}, or {@link ContainerSASPermissions} for
     * more details.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly resourceTypes?: string;
    /**
     * Optional. The signed identifier (only for {@link IBlobSASSignatureValues}).
     *
     * @see https://docs.microsoft.com/en-us/rest/api/storageservices/establishing-a-stored-access-policy
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly identifier?: string;
    /**
     * Optional. The storage container or blob (only for {@link IBlobSASSignatureValues}).
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly resource?: string;
    /**
     * The signature for the SAS token.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly signature: string;
    /**
     * Value for cache-control header in Blob/File Service SAS.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly cacheControl?: string;
    /**
     * Value for content-disposition header in Blob/File Service SAS.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly contentDisposition?: string;
    /**
     * Value for content-encoding header in Blob/File Service SAS.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly contentEncoding?: string;
    /**
     * Value for content-length header in Blob/File Service SAS.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly contentLanguage?: string;
    /**
     * Value for content-type header in Blob/File Service SAS.
     *
     * @type {string}
     * @memberof SASQueryParameters
     */
    readonly contentType?: string;
    /**
     * Inner value of getter ipRange.
     *
     * @private
     * @type {IIPRange}
     * @memberof SASQueryParameters
     */
    private readonly ipRangeInner?;
    /**
     * The Azure Active Directory object ID in GUID format.
     * Property of user delegation key.
     *
     * @private
     * @type {string}
     * @memberof SASQueryParameters
     */
    private readonly signedOid?;
    /**
     * The Azure Active Directory tenant ID in GUID format.
     * Property of user delegation key.
     *
     * @private
     * @type {string}
     * @memberof SASQueryParameters
     */
    private readonly signedTid?;
    /**
     * The date-time the key is active.
     * Property of user delegation key.
     *
     * @private
     * @type {Date}
     * @memberof SASQueryParameters
     */
    private readonly signedStart?;
    /**
     * The date-time the key expires.
     * Property of user delegation key.
     *
     * @private
     * @type {Date}
     * @memberof SASQueryParameters
     */
    private readonly signedExpiry?;
    /**
     * Abbreviation of the Azure Storage service that accepts the user delegation key.
     * Property of user delegation key.
     *
     * @private
     * @type {string}
     * @memberof SASQueryParameters
     */
    private readonly signedService?;
    /**
     * The service version that created the user delegation key.
     * Property of user delegation key.
     *
     * @private
     * @type {string}
     * @memberof SASQueryParameters
     */
    private readonly signedVersion?;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     * @type {(IIPRange | undefined)}
     * @memberof SASQueryParameters
     */
    readonly ipRange: IIPRange | undefined;
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param {string} version Representing the storage version
     * @param {string} signature Representing the signature for the SAS token
     * @param {string} [permissions] Representing the storage permissions
     * @param {string} [services] Representing the storage services being accessed (only for Account SAS)
     * @param {string} [resourceTypes] Representing the storage resource types being accessed (only for Account SAS)
     * @param {SASProtocol} [protocol] Representing the allowed HTTP protocol(s)
     * @param {Date} [startTime] Representing the start time for this SAS token
     * @param {Date} [expiryTime] Representing the expiry time for this SAS token
     * @param {IIPRange} [ipRange] Representing the range of valid IP addresses for this SAS token
     * @param {string} [identifier] Representing the signed identifier (only for Service SAS)
     * @param {string} [resource] Representing the storage container or blob (only for Service SAS)
     * @param {string} [cacheControl] Representing the cache-control header (only for Blob/File Service SAS)
     * @param {string} [contentDisposition] Representing the content-disposition header (only for Blob/File Service SAS)
     * @param {string} [contentEncoding] Representing the content-encoding header (only for Blob/File Service SAS)
     * @param {string} [contentLanguage] Representing the content-language header (only for Blob/File Service SAS)
     * @param {string} [contentType] Representing the content-type header (only for Blob/File Service SAS)
     * @param {userDelegationKey} [userDelegationKey] Representing the user delegation key properties
     * @memberof SASQueryParameters
     */
    constructor(version: string, signature: string, permissions?: string, services?: string, resourceTypes?: string, protocol?: SASProtocol, startTime?: Date, expiryTime?: Date, ipRange?: IIPRange, identifier?: string, resource?: string, cacheControl?: string, contentDisposition?: string, contentEncoding?: string, contentLanguage?: string, contentType?: string, userDelegationKey?: UserDelegationKey);
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     * @returns {string}
     * @memberof SASQueryParameters
     */
    toString(): string;
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @private
     * @param {string[]} queries
     * @param {string} key
     * @param {string} [value]
     * @returns {void}
     * @memberof SASQueryParameters
     */
    private tryAppendQueryParameter;
}
//# sourceMappingURL=SASQueryParameters.d.ts.map