/// <reference types="node" />
import { TransferProgressEvent } from "@azure/ms-rest-js";
import { Readable } from "stream";
import { Aborter } from "./Aborter";
import { BlobURL } from "./BlobURL";
import { BlockBlobURL } from "./BlockBlobURL";
import { BlobHTTPHeaders } from "./generated/src/models";
import { BlobUploadCommonResponse, IDownloadFromBlobOptions, IUploadToBlockBlobOptions } from "./highlevel.common";
import { IBlobAccessConditions } from "./models";
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Uploads a local file in blocks to a block blob.
 *
 * When file size <= 256MB, this method will use 1 upload call to finish the upload.
 * Otherwise, this method will call stageBlock to upload blocks, and finally call commitBlockList
 * to commit the block list.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {string} filePath Full path of local file
 * @param {BlockBlobURL} blockBlobURL BlockBlobURL
 * @param {IUploadToBlockBlobOptions} [options] IUploadToBlockBlobOptions
 * @returns {(Promise<BlobUploadCommonResponse>)} ICommonResponse
 */
export declare function uploadFileToBlockBlob(aborter: Aborter, filePath: string, blockBlobURL: BlockBlobURL, options?: IUploadToBlockBlobOptions): Promise<BlobUploadCommonResponse>;
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Downloads an Azure Blob in parallel to a buffer.
 * Offset and count are optional, pass 0 for both to download the entire blob.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Buffer} buffer Buffer to be fill, must have length larger than count
 * @param {BlobURL} blobURL A BlobURL object
 * @param {number} offset From which position of the block blob to download, in bytes
 * @param {number} [count] How much data in bytes to be downloaded. Will download to the end when passing undefined
 * @param {IDownloadFromBlobOptions} [options] IDownloadFromBlobOptions
 * @returns {Promise<void>}
 */
export declare function downloadBlobToBuffer(aborter: Aborter, buffer: Buffer, blobURL: BlobURL, offset: number, count?: number, options?: IDownloadFromBlobOptions): Promise<void>;
/**
 * Option interface for uploadStreamToBlockBlob.
 *
 * @export
 * @interface IUploadStreamToBlockBlobOptions
 */
export interface IUploadStreamToBlockBlobOptions {
    /**
     * Blob HTTP Headers.
     *
     * @type {BlobHTTPHeaders}
     * @memberof IUploadStreamToBlockBlobOptions
     */
    blobHTTPHeaders?: BlobHTTPHeaders;
    /**
     * Metadata of block blob.
     *
     * @type {{ [propertyName: string]: string }}
     * @memberof IUploadStreamToBlockBlobOptions
     */
    metadata?: {
        [propertyName: string]: string;
    };
    /**
     * Access conditions headers.
     *
     * @type {IBlobAccessConditions}
     * @memberof IUploadStreamToBlockBlobOptions
     */
    accessConditions?: IBlobAccessConditions;
    /**
     * Progress updater.
     *
     * @memberof IUploadStreamToBlockBlobOptions
     */
    progress?: (progress: TransferProgressEvent) => void;
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Uploads a Node.js Readable stream into block blob.
 *
 * PERFORMANCE IMPROVEMENT TIPS:
 * * Input stream highWaterMark is better to set a same value with bufferSize
 *    parameter, which will avoid Buffer.concat() operations.
 *
 * @export
 * @param {Aborter} aborter Create a new Aborter instance with Aborter.none or Aborter.timeout(),
 *                          goto documents of Aborter for more examples about request cancellation
 * @param {Readable} stream Node.js Readable stream
 * @param {BlockBlobURL} blockBlobURL A BlockBlobURL instance
 * @param {number} bufferSize Size of every buffer allocated, also the block size in the uploaded block blob
 * @param {number} maxBuffers Max buffers will allocate during uploading, positive correlation
 *                            with max uploading concurrency
 * @param {IUploadStreamToBlockBlobOptions} [options]
 * @returns {Promise<BlobUploadCommonResponse>}
 */
export declare function uploadStreamToBlockBlob(aborter: Aborter, stream: Readable, blockBlobURL: BlockBlobURL, bufferSize: number, maxBuffers: number, options?: IUploadStreamToBlockBlobOptions): Promise<BlobUploadCommonResponse>;
//# sourceMappingURL=highlevel.node.d.ts.map