'use strict';

var p = require('path');
var fs = require('fs');
// const printAST = require('ast-pretty-print')

var _require = require('./helpers'),
    getReplacement = _require.getReplacement,
    requireFromString = _require.requireFromString;

module.exports = prevalPlugin;

function prevalPlugin(babel) {
  var t = babel.types,
      template = babel.template,
      transformFromAst = babel.transformFromAst;

  var assignmentBuilder = template('const NAME = VALUE');
  return {
    name: 'preval',
    visitor: {
      Program(path, _ref) {
        var fileOpts = _ref.file.opts;

        var firstNode = path.node.body[0] || {};
        var comments = firstNode.leadingComments || [];
        var isPreval = comments.some(isPrevalComment);

        if (!isPreval) {
          return;
        }

        comments.find(isPrevalComment).value = ' this file was prevaled';

        var _transformFromAst = transformFromAst(path.node, null,
        /* istanbul ignore next (babel 6 vs babel 7 check) */
        /^6\./.test(babel.version) ? {} : {
          babelrc: false,
          configFile: false
        }),
            string = _transformFromAst.code;

        var replacement = getReplacement({ string, fileOpts, babel });

        var moduleExports = Object.assign({}, t.expressionStatement(t.assignmentExpression('=', t.memberExpression(t.identifier('module'), t.identifier('exports')), replacement)), { leadingComments: comments });

        path.replaceWith(t.program([moduleExports]));
      },
      TaggedTemplateExpression(path, _ref2) {
        var fileOpts = _ref2.file.opts;

        var isPreval = path.node.tag.name === 'preval';
        if (!isPreval) {
          return;
        }
        var string = path.get('quasi').evaluate().value;
        if (!string) {
          throw new Error('Unable to determine the value of your preval string');
        }
        var replacement = getReplacement({ string, fileOpts, babel });
        path.replaceWith(replacement);
      },
      ImportDeclaration(path, _ref3) {
        var fileOpts = _ref3.file.opts;

        var isPreval = looksLike(path, {
          node: {
            source: {
              leadingComments(comments) {
                return comments && comments.some(isPrevalComment);
              }
            }
          }
        });
        if (!isPreval) {
          return;
        }
        var prevalComment = path.node.source.leadingComments.find(isPrevalComment).value.trim();
        var argValues = void 0;
        if (prevalComment !== 'preval') {
          var arg = prevalComment.replace(/preval\((.*)\)/, '$1').trim();
          var argValue = requireFromString({
            string: `module.exports = ${arg}`,
            fileOpts
          });
          argValues = [argValue];
        }

        var absolutePath = p.resolve(p.dirname(fileOpts.filename), path.node.source.value);
        var code = fs.readFileSync(require.resolve(absolutePath));

        var replacement = getReplacement({
          string: code,
          fileOpts,
          args: argValues,
          babel
        });
        path.replaceWith(assignmentBuilder({
          NAME: t.identifier(path.node.specifiers[0].local.name),
          VALUE: replacement
        }));
      },
      CallExpression(path, _ref4) {
        var fileOpts = _ref4.file.opts;

        var isPreval = looksLike(path, {
          node: {
            callee: {
              type: 'MemberExpression',
              object: { name: 'preval' },
              property: { name: 'require' }
            }
          }
        });
        if (!isPreval) {
          return;
        }

        var _path$get = path.get('arguments'),
            source = _path$get[0],
            args = _path$get.slice(1);

        var argValues = args.map(function (a) {
          var result = a.evaluate();
          if (!result.confident) {
            throw new Error('preval cannot determine the value of an argument in preval.require');
          }
          return result.value;
        });
        var absolutePath = p.resolve(p.dirname(fileOpts.filename), source.node.value);
        var code = fs.readFileSync(require.resolve(absolutePath));

        var replacement = getReplacement({
          string: code,
          fileOpts,
          args: argValues,
          babel
        });

        path.replaceWith(replacement);
      }
    }
  };
}

function isPrevalComment(comment) {
  var normalisedComment = comment.value.trim().split(' ')[0].trim();
  return normalisedComment.startsWith('preval') || normalisedComment.startsWith('@preval');
}

function looksLike(a, b) {
  return a && b && Object.keys(b).every(function (bKey) {
    var bVal = b[bKey];
    var aVal = a[bKey];
    if (typeof bVal === 'function') {
      return bVal(aVal);
    }
    return isPrimitive(bVal) ? bVal === aVal : looksLike(aVal, bVal);
  });
}

function isPrimitive(val) {
  // eslint-disable-next-line
  return val == null || /^[sbn]/.test(typeof val);
}

/*
eslint
  import/no-unassigned-import:0
  import/no-dynamic-require:0
*/