/*
 * Copyright 2015, Yahoo Inc.
 * Copyrights licensed under the New BSD License.
 * See the accompanying LICENSE file for terms.
 */
import parser from 'intl-messageformat-parser';
var ESCAPED_CHARS = {
    '\\': '\\\\',
    '\\#': '\\#',
    '{': '\\{',
    '}': '\\}'
};
var ESAPE_CHARS_REGEXP = /\\#|[{}\\]/g;
export default function printICUMessage(message) {
    var ast = parser.parse(message);
    return _printICUMessage(ast);
}
function isPluralFormat(format) {
    return format.type === 'pluralFormat';
}
function isSimpleFormat(format) {
    return (format.type === 'numberFormat' ||
        format.type === 'dateFormat' ||
        format.type === 'timeFormat');
}
function _printICUMessage(ast) {
    var printedNodes = ast.elements.map(function (node) {
        if (node.type === 'messageTextElement') {
            return printMessageTextASTNode(node);
        }
        if (!node.format) {
            return "{" + node.id + "}";
        }
        switch (getArgumentType(node.format)) {
            case 'number':
            case 'date':
            case 'time':
                return printSimpleFormatASTNode(node);
            case 'plural':
            case 'selectordinal':
            case 'select':
                return printOptionalFormatASTNode(node);
        }
    });
    return printedNodes.join('');
}
function getArgumentType(format) {
    var type = format.type;
    // Special-case ordinal plurals to use `selectordinal` instead of `plural`.
    if (isPluralFormat(format) && format.ordinal) {
        return 'selectordinal';
    }
    return type.replace(/Format$/, '').toLowerCase();
}
function printMessageTextASTNode(_a) {
    var value = _a.value;
    return value.replace(ESAPE_CHARS_REGEXP, function (char) { return ESCAPED_CHARS[char]; });
}
function printSimpleFormatASTNode(_a) {
    var id = _a.id, format = _a.format;
    var argumentType = getArgumentType(format);
    var style = isSimpleFormat(format) && format.style ? ", " + format.style : '';
    return "{" + id + ", " + argumentType + style + "}";
}
function printOptionalFormatASTNode(_a) {
    var id = _a.id, format = _a.format;
    var argumentType = getArgumentType(format);
    var offset = format.offset
        ? ", offset:" + format.offset
        : '';
    var options = format.options.map(function (option) {
        var optionValue = _printICUMessage(option.value);
        return " " + option.selector + " {" + optionValue + "}";
    });
    return "{" + id + ", " + argumentType + offset + "," + options.join('') + "}";
}
//# sourceMappingURL=print-icu-message.js.map