import jsTokens from 'js-tokens';
import { isKeyword, isStrictReservedWord } from '@babel/helper-validator-identifier';
import chalk, { Chalk } from 'chalk';

const sometimesKeywords = new Set(["as", "async", "from", "get", "of", "set"]);
function getDefs(chalk) {
  return {
    keyword: chalk.cyan,
    capitalized: chalk.yellow,
    jsxIdentifier: chalk.yellow,
    punctuator: chalk.yellow,
    number: chalk.magenta,
    string: chalk.green,
    regex: chalk.magenta,
    comment: chalk.grey,
    invalid: chalk.white.bgRed.bold
  };
}
const NEWLINE = /\r\n|[\n\r\u2028\u2029]/;
const BRACKET = /^[()[\]{}]$/;
let tokenize;
{
  const getTokenType = function (token) {
    if (token.type === "IdentifierName") {
      if (isKeyword(token.value) || isStrictReservedWord(token.value, true) || sometimesKeywords.has(token.value)) {
        return "keyword";
      }
      if (token.value[0] !== token.value[0].toLowerCase()) {
        return "capitalized";
      }
    }
    if (token.type === "Punctuator" && BRACKET.test(token.value)) {
      return "uncolored";
    }
    if (token.type === "Invalid" && token.value === "@") {
      return "punctuator";
    }
    switch (token.type) {
      case "NumericLiteral":
        return "number";
      case "StringLiteral":
      case "JSXString":
      case "NoSubstitutionTemplate":
        return "string";
      case "RegularExpressionLiteral":
        return "regex";
      case "Punctuator":
      case "JSXPunctuator":
        return "punctuator";
      case "MultiLineComment":
      case "SingleLineComment":
        return "comment";
      case "Invalid":
      case "JSXInvalid":
        return "invalid";
      case "JSXIdentifier":
        return "jsxIdentifier";
      default:
        return "uncolored";
    }
  };
  tokenize = function* (text) {
    for (const token of jsTokens(text, {
      jsx: true
    })) {
      switch (token.type) {
        case "TemplateHead":
          yield {
            type: "string",
            value: token.value.slice(0, -2)
          };
          yield {
            type: "punctuator",
            value: "${"
          };
          break;
        case "TemplateMiddle":
          yield {
            type: "punctuator",
            value: "}"
          };
          yield {
            type: "string",
            value: token.value.slice(1, -2)
          };
          yield {
            type: "punctuator",
            value: "${"
          };
          break;
        case "TemplateTail":
          yield {
            type: "punctuator",
            value: "}"
          };
          yield {
            type: "string",
            value: token.value.slice(1)
          };
          break;
        default:
          yield {
            type: getTokenType(token),
            value: token.value
          };
      }
    }
  };
}
function highlightTokens(defs, text) {
  let highlighted = "";
  for (const {
    type,
    value
  } of tokenize(text)) {
    const colorize = defs[type];
    if (colorize) {
      highlighted += value.split(NEWLINE).map(str => colorize(str)).join("\n");
    } else {
      highlighted += value;
    }
  }
  return highlighted;
}
function shouldHighlight(options) {
  return chalk.level > 0 || options.forceColor;
}
let chalkWithForcedColor = undefined;
function getChalk(forceColor) {
  if (forceColor) {
    chalkWithForcedColor ??= new Chalk({
      level: 1
    });
    return chalkWithForcedColor;
  }
  return chalk;
}
function highlight(code, options = {}) {
  if (code !== "" && shouldHighlight(options)) {
    const defs = getDefs(getChalk(options.forceColor));
    return highlightTokens(defs, code);
  } else {
    return code;
  }
}

export { highlight as default, shouldHighlight };
//# sourceMappingURL=index.js.map
