import * as t from '@babel/types';
import { Node, RemovePropertiesOptions } from '@babel/types';

type BindingKind = "var" | "let" | "const" | "module" | "hoisted" | "param" | "local" | "unknown";
/**
 * This class is responsible for a binding inside of a scope.
 *
 * It tracks the following:
 *
 *  * Node path.
 *  * Amount of times referenced by other nodes.
 *  * Paths to nodes that reassign or modify this binding.
 *  * The kind of binding. (Is it a parameter, declaration etc)
 */
declare class Binding {
    identifier: t.Identifier;
    scope: Scope;
    path: NodePath_Final;
    kind: BindingKind;
    constructor({ identifier, scope, path, kind, }: {
        identifier: t.Identifier;
        scope: Scope;
        path: NodePath_Final;
        kind: BindingKind;
    });
    constantViolations: Array<NodePath_Final>;
    constant: boolean;
    referencePaths: Array<NodePath_Final>;
    referenced: boolean;
    references: number;
    hasDeoptedValue: boolean;
    hasValue: boolean;
    value: any;
    deoptValue(): void;
    setValue(value: any): void;
    clearValue(): void;
    /**
     * Register a constant violation with the provided `path`.
     */
    reassign(path: NodePath_Final): void;
    /**
     * Increment the amount of references to this binding.
     */
    reference(path: NodePath_Final): void;
    /**
     * Decrement the amount of references to this binding.
     */
    dereference(): void;
}

type _Binding = Binding;
declare class Scope {
    uid: number;
    path: NodePath_Final;
    block: t.Pattern | t.Scopable;
    labels: Map<any, any>;
    inited: boolean;
    bindings: {
        [name: string]: Binding;
    };
    references: {
        [name: string]: true;
    };
    globals: {
        [name: string]: t.Identifier | t.JSXIdentifier;
    };
    uids: {
        [name: string]: boolean;
    };
    data: {
        [key: string | symbol]: unknown;
    };
    crawling: boolean;
    /**
     * This searches the current "scope" and collects all references/bindings
     * within.
     */
    constructor(path: NodePath_Final<t.Pattern | t.Scopable>);
    /**
     * Globals.
     */
    static globals: string[];
    /**
     * Variables available in current context.
     */
    static contextVariables: string[];
    get parent(): Scope;
    get parentBlock(): t.Node;
    get hub(): HubInterface;
    traverse<S>(node: t.Node | t.Node[], opts: TraverseOptions<S>, state: S): void;
    traverse(node: t.Node | t.Node[], opts?: TraverseOptions, state?: any): void;
    /**
     * Generate a unique identifier and add it to the current scope.
     */
    generateDeclaredUidIdentifier(name?: string): t.Identifier;
    /**
     * Generate a unique identifier.
     */
    generateUidIdentifier(name?: string): t.Identifier;
    /**
     * Generate a unique `_id1` binding.
     */
    generateUid(name?: string): string;
    /**
     * Generate an `_id1`.
     */
    _generateUid(name: string, i: number): string;
    generateUidBasedOnNode(node: t.Node, defaultName?: string): string;
    /**
     * Generate a unique identifier based on a node.
     */
    generateUidIdentifierBasedOnNode(node: t.Node, defaultName?: string): t.Identifier;
    /**
     * Determine whether evaluating the specific input `node` is a consequenceless reference. ie.
     * evaluating it won't result in potentially arbitrary code from being ran. The following are
     * allowed and determined not to cause side effects:
     *
     *  - `this` expressions
     *  - `super` expressions
     *  - Bound identifiers
     */
    isStatic(node: t.Node): boolean;
    /**
     * Possibly generate a memoised identifier if it is not static and has consequences.
     */
    maybeGenerateMemoised(node: t.Node, dontPush?: boolean): t.Identifier;
    checkBlockScopedCollisions(local: Binding, kind: BindingKind, name: string, id: any): void;
    rename(oldName: string, newName?: string): void;
    /** @deprecated Not used in our codebase */
    _renameFromMap(map: Record<string | symbol, unknown>, oldName: string | symbol, newName: string | symbol, value: unknown): void;
    dump(): void;
    toArray(node: t.Node, i?: number | boolean, arrayLikeIsIterable?: boolean | void): t.ArrayExpression | t.CallExpression | t.Identifier;
    hasLabel(name: string): boolean;
    getLabel(name: string): any;
    registerLabel(path: NodePath_Final<t.LabeledStatement>): void;
    registerDeclaration(path: NodePath_Final): void;
    buildUndefinedNode(): t.UnaryExpression;
    registerConstantViolation(path: NodePath_Final): void;
    registerBinding(kind: Binding["kind"], path: NodePath_Final, bindingPath?: NodePath_Final): void;
    addGlobal(node: t.Identifier | t.JSXIdentifier): void;
    hasUid(name: string): boolean;
    hasGlobal(name: string): boolean;
    hasReference(name: string): boolean;
    isPure(node: t.Node, constantsOnly?: boolean): boolean;
    /**
     * Set some arbitrary data on the current scope.
     */
    setData(key: string | symbol, val: any): any;
    /**
     * Recursively walk up scope tree looking for the data `key`.
     */
    getData(key: string | symbol): any;
    /**
     * Recursively walk up scope tree looking for the data `key` and if it exists,
     * remove it.
     */
    removeData(key: string): void;
    init(): void;
    crawl(): void;
    push(opts: {
        id: t.LVal;
        init?: t.Expression;
        unique?: boolean;
        _blockHoist?: number | undefined;
        kind?: "var" | "let" | "const";
    }): void;
    /**
     * Walk up to the top of the scope tree and get the `Program`.
     */
    getProgramParent(): Scope;
    /**
     * Walk up the scope tree until we hit either a Function or return null.
     */
    getFunctionParent(): Scope | null;
    /**
     * Walk up the scope tree until we hit either a BlockStatement/Loop/Program/Function/Switch or reach the
     * very top and hit Program.
     */
    getBlockParent(): Scope;
    /**
     * Walk up from a pattern scope (function param initializer) until we hit a non-pattern scope,
     * then returns its block parent
     * @returns An ancestry scope whose path is a block parent
     */
    getPatternParent(): Scope;
    /**
     * Walks the scope tree and gathers **all** bindings.
     */
    getAllBindings(): Record<string, Binding>;
    /**
     * Walks the scope tree and gathers all declarations of `kind`.
     */
    getAllBindingsOfKind(...kinds: string[]): Record<string, Binding>;
    bindingIdentifierEquals(name: string, node: t.Node): boolean;
    getBinding(name: string): Binding | undefined;
    getOwnBinding(name: string): Binding | undefined;
    getBindingIdentifier(name: string): t.Identifier;
    getOwnBindingIdentifier(name: string): t.Identifier;
    hasOwnBinding(name: string): boolean;
    hasBinding(name: string, opts?: boolean | {
        noGlobals?: boolean;
        noUids?: boolean;
    }): boolean;
    parentHasBinding(name: string, opts?: {
        noGlobals?: boolean;
        noUids?: boolean;
    }): boolean;
    /**
     * Move a binding of `name` to another `scope`.
     */
    moveBindingTo(name: string, scope: Scope): void;
    removeOwnBinding(name: string): void;
    removeBinding(name: string): void;
    /**
     * Hoist all the `var` variable to the beginning of the function/program
     * scope where their binding will be actually defined. For exmaple,
     *     { var x = 2 }
     * will be transformed to
     *     var x; { x = 2 }
     *
     * @param emit A custom function to emit `var` declarations, for example to
     *   emit them in a different scope.
     */
    hoistVariables(emit?: (id: t.Identifier, hasInit: boolean) => void): void;
}
declare namespace Scope {
    type Binding = _Binding;
}
//# sourceMappingURL=index.d.ts.map

interface HubInterface {
    getCode(): string | void;
    getScope(): Scope | void;
    addHelper(name: string): any;
    buildError(node: Node, msg: string, Error: new () => Error): Error;
}
declare class Hub implements HubInterface {
    getCode(): void;
    getScope(): void;
    addHelper(): void;
    buildError(node: Node, msg: string, Error?: TypeErrorConstructor): Error;
}

declare class TraversalContext<S = unknown> {
    constructor(scope: Scope, opts: ExplodedTraverseOptions<S>, state: S, parentPath: NodePath_Final);
    parentPath: NodePath_Final;
    scope: Scope;
    state: S;
    opts: ExplodedTraverseOptions<S>;
    queue: Array<NodePath_Final> | null;
    priorityQueue: Array<NodePath_Final> | null;
    /**
     * This method does a simple check to determine whether or not we really need to attempt
     * visit a node. This will prevent us from constructing a NodePath.
     */
    shouldVisit(node: t.Node): boolean;
    create(node: t.Node, container: t.Node | t.Node[], key: string | number, listKey?: string): NodePath_Final;
    maybeQueue(path: NodePath_Final, notPriority?: boolean): void;
    visitMultiple(container: t.Node[], parent: t.Node, listKey: string): boolean;
    visitSingle(node: t.Node, key: string): boolean;
    visitQueue(queue: Array<NodePath_Final>): boolean;
    visit(node: t.Node, key: string): boolean;
}

interface VirtualTypeAliases {
    BindingIdentifier: t.Identifier;
    BlockScoped: t.Node;
    ExistentialTypeParam: t.ExistsTypeAnnotation;
    Expression: t.Expression;
    Flow: t.Flow | t.ImportDeclaration | t.ExportDeclaration | t.ImportSpecifier;
    ForAwaitStatement: t.ForOfStatement;
    Generated: t.Node;
    NumericLiteralTypeAnnotation: t.NumberLiteralTypeAnnotation;
    Pure: t.Node;
    Referenced: t.Node;
    ReferencedIdentifier: t.Identifier | t.JSXIdentifier;
    ReferencedMemberExpression: t.MemberExpression;
    RestProperty: t.RestElement;
    Scope: t.Scopable | t.Pattern;
    SpreadProperty: t.RestElement;
    Statement: t.Statement;
    User: t.Node;
    Var: t.VariableDeclaration;
}

/*
 * This file is auto-generated! Do not modify it directly.
 * To re-generate run 'make build'
 */


interface ExplVisitorBase<S> {
  AnyTypeAnnotation?: ExplVisitNode<S, t.AnyTypeAnnotation>;
  ArgumentPlaceholder?: ExplVisitNode<S, t.ArgumentPlaceholder>;
  ArrayExpression?: ExplVisitNode<S, t.ArrayExpression>;
  ArrayPattern?: ExplVisitNode<S, t.ArrayPattern>;
  ArrayTypeAnnotation?: ExplVisitNode<S, t.ArrayTypeAnnotation>;
  ArrowFunctionExpression?: ExplVisitNode<S, t.ArrowFunctionExpression>;
  AssignmentExpression?: ExplVisitNode<S, t.AssignmentExpression>;
  AssignmentPattern?: ExplVisitNode<S, t.AssignmentPattern>;
  AwaitExpression?: ExplVisitNode<S, t.AwaitExpression>;
  BigIntLiteral?: ExplVisitNode<S, t.BigIntLiteral>;
  BinaryExpression?: ExplVisitNode<S, t.BinaryExpression>;
  BindExpression?: ExplVisitNode<S, t.BindExpression>;
  BlockStatement?: ExplVisitNode<S, t.BlockStatement>;
  BooleanLiteral?: ExplVisitNode<S, t.BooleanLiteral>;
  BooleanLiteralTypeAnnotation?: ExplVisitNode<
    S,
    t.BooleanLiteralTypeAnnotation
  >;
  BooleanTypeAnnotation?: ExplVisitNode<S, t.BooleanTypeAnnotation>;
  BreakStatement?: ExplVisitNode<S, t.BreakStatement>;
  CallExpression?: ExplVisitNode<S, t.CallExpression>;
  CatchClause?: ExplVisitNode<S, t.CatchClause>;
  ClassAccessorProperty?: ExplVisitNode<S, t.ClassAccessorProperty>;
  ClassBody?: ExplVisitNode<S, t.ClassBody>;
  ClassDeclaration?: ExplVisitNode<S, t.ClassDeclaration>;
  ClassExpression?: ExplVisitNode<S, t.ClassExpression>;
  ClassImplements?: ExplVisitNode<S, t.ClassImplements>;
  ClassMethod?: ExplVisitNode<S, t.ClassMethod>;
  ClassPrivateMethod?: ExplVisitNode<S, t.ClassPrivateMethod>;
  ClassPrivateProperty?: ExplVisitNode<S, t.ClassPrivateProperty>;
  ClassProperty?: ExplVisitNode<S, t.ClassProperty>;
  ConditionalExpression?: ExplVisitNode<S, t.ConditionalExpression>;
  ContinueStatement?: ExplVisitNode<S, t.ContinueStatement>;
  DebuggerStatement?: ExplVisitNode<S, t.DebuggerStatement>;
  DecimalLiteral?: ExplVisitNode<S, t.DecimalLiteral>;
  DeclareClass?: ExplVisitNode<S, t.DeclareClass>;
  DeclareExportAllDeclaration?: ExplVisitNode<S, t.DeclareExportAllDeclaration>;
  DeclareExportDeclaration?: ExplVisitNode<S, t.DeclareExportDeclaration>;
  DeclareFunction?: ExplVisitNode<S, t.DeclareFunction>;
  DeclareInterface?: ExplVisitNode<S, t.DeclareInterface>;
  DeclareModule?: ExplVisitNode<S, t.DeclareModule>;
  DeclareModuleExports?: ExplVisitNode<S, t.DeclareModuleExports>;
  DeclareOpaqueType?: ExplVisitNode<S, t.DeclareOpaqueType>;
  DeclareTypeAlias?: ExplVisitNode<S, t.DeclareTypeAlias>;
  DeclareVariable?: ExplVisitNode<S, t.DeclareVariable>;
  DeclaredPredicate?: ExplVisitNode<S, t.DeclaredPredicate>;
  Decorator?: ExplVisitNode<S, t.Decorator>;
  Directive?: ExplVisitNode<S, t.Directive>;
  DirectiveLiteral?: ExplVisitNode<S, t.DirectiveLiteral>;
  DoExpression?: ExplVisitNode<S, t.DoExpression>;
  DoWhileStatement?: ExplVisitNode<S, t.DoWhileStatement>;
  EmptyStatement?: ExplVisitNode<S, t.EmptyStatement>;
  EmptyTypeAnnotation?: ExplVisitNode<S, t.EmptyTypeAnnotation>;
  EnumBooleanBody?: ExplVisitNode<S, t.EnumBooleanBody>;
  EnumBooleanMember?: ExplVisitNode<S, t.EnumBooleanMember>;
  EnumDeclaration?: ExplVisitNode<S, t.EnumDeclaration>;
  EnumDefaultedMember?: ExplVisitNode<S, t.EnumDefaultedMember>;
  EnumNumberBody?: ExplVisitNode<S, t.EnumNumberBody>;
  EnumNumberMember?: ExplVisitNode<S, t.EnumNumberMember>;
  EnumStringBody?: ExplVisitNode<S, t.EnumStringBody>;
  EnumStringMember?: ExplVisitNode<S, t.EnumStringMember>;
  EnumSymbolBody?: ExplVisitNode<S, t.EnumSymbolBody>;
  ExistsTypeAnnotation?: ExplVisitNode<S, t.ExistsTypeAnnotation>;
  ExportAllDeclaration?: ExplVisitNode<S, t.ExportAllDeclaration>;
  ExportDefaultDeclaration?: ExplVisitNode<S, t.ExportDefaultDeclaration>;
  ExportDefaultSpecifier?: ExplVisitNode<S, t.ExportDefaultSpecifier>;
  ExportNamedDeclaration?: ExplVisitNode<S, t.ExportNamedDeclaration>;
  ExportNamespaceSpecifier?: ExplVisitNode<S, t.ExportNamespaceSpecifier>;
  ExportSpecifier?: ExplVisitNode<S, t.ExportSpecifier>;
  ExpressionStatement?: ExplVisitNode<S, t.ExpressionStatement>;
  File?: ExplVisitNode<S, t.File>;
  ForInStatement?: ExplVisitNode<S, t.ForInStatement>;
  ForOfStatement?: ExplVisitNode<S, t.ForOfStatement>;
  ForStatement?: ExplVisitNode<S, t.ForStatement>;
  FunctionDeclaration?: ExplVisitNode<S, t.FunctionDeclaration>;
  FunctionExpression?: ExplVisitNode<S, t.FunctionExpression>;
  FunctionTypeAnnotation?: ExplVisitNode<S, t.FunctionTypeAnnotation>;
  FunctionTypeParam?: ExplVisitNode<S, t.FunctionTypeParam>;
  GenericTypeAnnotation?: ExplVisitNode<S, t.GenericTypeAnnotation>;
  Identifier?: ExplVisitNode<S, t.Identifier>;
  IfStatement?: ExplVisitNode<S, t.IfStatement>;
  Import?: ExplVisitNode<S, t.Import>;
  ImportAttribute?: ExplVisitNode<S, t.ImportAttribute>;
  ImportDeclaration?: ExplVisitNode<S, t.ImportDeclaration>;
  ImportDefaultSpecifier?: ExplVisitNode<S, t.ImportDefaultSpecifier>;
  ImportExpression?: ExplVisitNode<S, t.ImportExpression>;
  ImportNamespaceSpecifier?: ExplVisitNode<S, t.ImportNamespaceSpecifier>;
  ImportSpecifier?: ExplVisitNode<S, t.ImportSpecifier>;
  IndexedAccessType?: ExplVisitNode<S, t.IndexedAccessType>;
  InferredPredicate?: ExplVisitNode<S, t.InferredPredicate>;
  InterfaceDeclaration?: ExplVisitNode<S, t.InterfaceDeclaration>;
  InterfaceExtends?: ExplVisitNode<S, t.InterfaceExtends>;
  InterfaceTypeAnnotation?: ExplVisitNode<S, t.InterfaceTypeAnnotation>;
  InterpreterDirective?: ExplVisitNode<S, t.InterpreterDirective>;
  IntersectionTypeAnnotation?: ExplVisitNode<S, t.IntersectionTypeAnnotation>;
  JSXAttribute?: ExplVisitNode<S, t.JSXAttribute>;
  JSXClosingElement?: ExplVisitNode<S, t.JSXClosingElement>;
  JSXClosingFragment?: ExplVisitNode<S, t.JSXClosingFragment>;
  JSXElement?: ExplVisitNode<S, t.JSXElement>;
  JSXEmptyExpression?: ExplVisitNode<S, t.JSXEmptyExpression>;
  JSXExpressionContainer?: ExplVisitNode<S, t.JSXExpressionContainer>;
  JSXFragment?: ExplVisitNode<S, t.JSXFragment>;
  JSXIdentifier?: ExplVisitNode<S, t.JSXIdentifier>;
  JSXMemberExpression?: ExplVisitNode<S, t.JSXMemberExpression>;
  JSXNamespacedName?: ExplVisitNode<S, t.JSXNamespacedName>;
  JSXOpeningElement?: ExplVisitNode<S, t.JSXOpeningElement>;
  JSXOpeningFragment?: ExplVisitNode<S, t.JSXOpeningFragment>;
  JSXSpreadAttribute?: ExplVisitNode<S, t.JSXSpreadAttribute>;
  JSXSpreadChild?: ExplVisitNode<S, t.JSXSpreadChild>;
  JSXText?: ExplVisitNode<S, t.JSXText>;
  LabeledStatement?: ExplVisitNode<S, t.LabeledStatement>;
  LogicalExpression?: ExplVisitNode<S, t.LogicalExpression>;
  MemberExpression?: ExplVisitNode<S, t.MemberExpression>;
  MetaProperty?: ExplVisitNode<S, t.MetaProperty>;
  MixedTypeAnnotation?: ExplVisitNode<S, t.MixedTypeAnnotation>;
  ModuleExpression?: ExplVisitNode<S, t.ModuleExpression>;
  NewExpression?: ExplVisitNode<S, t.NewExpression>;
  NullLiteral?: ExplVisitNode<S, t.NullLiteral>;
  NullLiteralTypeAnnotation?: ExplVisitNode<S, t.NullLiteralTypeAnnotation>;
  NullableTypeAnnotation?: ExplVisitNode<S, t.NullableTypeAnnotation>;
  NumberLiteral?: ExplVisitNode<S, t.NumberLiteral>;
  NumberLiteralTypeAnnotation?: ExplVisitNode<S, t.NumberLiteralTypeAnnotation>;
  NumberTypeAnnotation?: ExplVisitNode<S, t.NumberTypeAnnotation>;
  NumericLiteral?: ExplVisitNode<S, t.NumericLiteral>;
  ObjectExpression?: ExplVisitNode<S, t.ObjectExpression>;
  ObjectMethod?: ExplVisitNode<S, t.ObjectMethod>;
  ObjectPattern?: ExplVisitNode<S, t.ObjectPattern>;
  ObjectProperty?: ExplVisitNode<S, t.ObjectProperty>;
  ObjectTypeAnnotation?: ExplVisitNode<S, t.ObjectTypeAnnotation>;
  ObjectTypeCallProperty?: ExplVisitNode<S, t.ObjectTypeCallProperty>;
  ObjectTypeIndexer?: ExplVisitNode<S, t.ObjectTypeIndexer>;
  ObjectTypeInternalSlot?: ExplVisitNode<S, t.ObjectTypeInternalSlot>;
  ObjectTypeProperty?: ExplVisitNode<S, t.ObjectTypeProperty>;
  ObjectTypeSpreadProperty?: ExplVisitNode<S, t.ObjectTypeSpreadProperty>;
  OpaqueType?: ExplVisitNode<S, t.OpaqueType>;
  OptionalCallExpression?: ExplVisitNode<S, t.OptionalCallExpression>;
  OptionalIndexedAccessType?: ExplVisitNode<S, t.OptionalIndexedAccessType>;
  OptionalMemberExpression?: ExplVisitNode<S, t.OptionalMemberExpression>;
  ParenthesizedExpression?: ExplVisitNode<S, t.ParenthesizedExpression>;
  PipelineBareFunction?: ExplVisitNode<S, t.PipelineBareFunction>;
  PipelinePrimaryTopicReference?: ExplVisitNode<
    S,
    t.PipelinePrimaryTopicReference
  >;
  PipelineTopicExpression?: ExplVisitNode<S, t.PipelineTopicExpression>;
  Placeholder?: ExplVisitNode<S, t.Placeholder>;
  PrivateName?: ExplVisitNode<S, t.PrivateName>;
  Program?: ExplVisitNode<S, t.Program>;
  QualifiedTypeIdentifier?: ExplVisitNode<S, t.QualifiedTypeIdentifier>;
  RecordExpression?: ExplVisitNode<S, t.RecordExpression>;
  RegExpLiteral?: ExplVisitNode<S, t.RegExpLiteral>;
  RegexLiteral?: ExplVisitNode<S, t.RegexLiteral>;
  RestElement?: ExplVisitNode<S, t.RestElement>;
  RestProperty?: ExplVisitNode<S, t.RestProperty>;
  ReturnStatement?: ExplVisitNode<S, t.ReturnStatement>;
  SequenceExpression?: ExplVisitNode<S, t.SequenceExpression>;
  SpreadElement?: ExplVisitNode<S, t.SpreadElement>;
  SpreadProperty?: ExplVisitNode<S, t.SpreadProperty>;
  StaticBlock?: ExplVisitNode<S, t.StaticBlock>;
  StringLiteral?: ExplVisitNode<S, t.StringLiteral>;
  StringLiteralTypeAnnotation?: ExplVisitNode<S, t.StringLiteralTypeAnnotation>;
  StringTypeAnnotation?: ExplVisitNode<S, t.StringTypeAnnotation>;
  Super?: ExplVisitNode<S, t.Super>;
  SwitchCase?: ExplVisitNode<S, t.SwitchCase>;
  SwitchStatement?: ExplVisitNode<S, t.SwitchStatement>;
  SymbolTypeAnnotation?: ExplVisitNode<S, t.SymbolTypeAnnotation>;
  TSAnyKeyword?: ExplVisitNode<S, t.TSAnyKeyword>;
  TSArrayType?: ExplVisitNode<S, t.TSArrayType>;
  TSAsExpression?: ExplVisitNode<S, t.TSAsExpression>;
  TSBigIntKeyword?: ExplVisitNode<S, t.TSBigIntKeyword>;
  TSBooleanKeyword?: ExplVisitNode<S, t.TSBooleanKeyword>;
  TSCallSignatureDeclaration?: ExplVisitNode<S, t.TSCallSignatureDeclaration>;
  TSConditionalType?: ExplVisitNode<S, t.TSConditionalType>;
  TSConstructSignatureDeclaration?: ExplVisitNode<
    S,
    t.TSConstructSignatureDeclaration
  >;
  TSConstructorType?: ExplVisitNode<S, t.TSConstructorType>;
  TSDeclareFunction?: ExplVisitNode<S, t.TSDeclareFunction>;
  TSDeclareMethod?: ExplVisitNode<S, t.TSDeclareMethod>;
  TSEnumDeclaration?: ExplVisitNode<S, t.TSEnumDeclaration>;
  TSEnumMember?: ExplVisitNode<S, t.TSEnumMember>;
  TSExportAssignment?: ExplVisitNode<S, t.TSExportAssignment>;
  TSExpressionWithTypeArguments?: ExplVisitNode<
    S,
    t.TSExpressionWithTypeArguments
  >;
  TSExternalModuleReference?: ExplVisitNode<S, t.TSExternalModuleReference>;
  TSFunctionType?: ExplVisitNode<S, t.TSFunctionType>;
  TSImportEqualsDeclaration?: ExplVisitNode<S, t.TSImportEqualsDeclaration>;
  TSImportType?: ExplVisitNode<S, t.TSImportType>;
  TSIndexSignature?: ExplVisitNode<S, t.TSIndexSignature>;
  TSIndexedAccessType?: ExplVisitNode<S, t.TSIndexedAccessType>;
  TSInferType?: ExplVisitNode<S, t.TSInferType>;
  TSInstantiationExpression?: ExplVisitNode<S, t.TSInstantiationExpression>;
  TSInterfaceBody?: ExplVisitNode<S, t.TSInterfaceBody>;
  TSInterfaceDeclaration?: ExplVisitNode<S, t.TSInterfaceDeclaration>;
  TSIntersectionType?: ExplVisitNode<S, t.TSIntersectionType>;
  TSIntrinsicKeyword?: ExplVisitNode<S, t.TSIntrinsicKeyword>;
  TSLiteralType?: ExplVisitNode<S, t.TSLiteralType>;
  TSMappedType?: ExplVisitNode<S, t.TSMappedType>;
  TSMethodSignature?: ExplVisitNode<S, t.TSMethodSignature>;
  TSModuleBlock?: ExplVisitNode<S, t.TSModuleBlock>;
  TSModuleDeclaration?: ExplVisitNode<S, t.TSModuleDeclaration>;
  TSNamedTupleMember?: ExplVisitNode<S, t.TSNamedTupleMember>;
  TSNamespaceExportDeclaration?: ExplVisitNode<
    S,
    t.TSNamespaceExportDeclaration
  >;
  TSNeverKeyword?: ExplVisitNode<S, t.TSNeverKeyword>;
  TSNonNullExpression?: ExplVisitNode<S, t.TSNonNullExpression>;
  TSNullKeyword?: ExplVisitNode<S, t.TSNullKeyword>;
  TSNumberKeyword?: ExplVisitNode<S, t.TSNumberKeyword>;
  TSObjectKeyword?: ExplVisitNode<S, t.TSObjectKeyword>;
  TSOptionalType?: ExplVisitNode<S, t.TSOptionalType>;
  TSParameterProperty?: ExplVisitNode<S, t.TSParameterProperty>;
  TSParenthesizedType?: ExplVisitNode<S, t.TSParenthesizedType>;
  TSPropertySignature?: ExplVisitNode<S, t.TSPropertySignature>;
  TSQualifiedName?: ExplVisitNode<S, t.TSQualifiedName>;
  TSRestType?: ExplVisitNode<S, t.TSRestType>;
  TSSatisfiesExpression?: ExplVisitNode<S, t.TSSatisfiesExpression>;
  TSStringKeyword?: ExplVisitNode<S, t.TSStringKeyword>;
  TSSymbolKeyword?: ExplVisitNode<S, t.TSSymbolKeyword>;
  TSThisType?: ExplVisitNode<S, t.TSThisType>;
  TSTupleType?: ExplVisitNode<S, t.TSTupleType>;
  TSTypeAliasDeclaration?: ExplVisitNode<S, t.TSTypeAliasDeclaration>;
  TSTypeAnnotation?: ExplVisitNode<S, t.TSTypeAnnotation>;
  TSTypeAssertion?: ExplVisitNode<S, t.TSTypeAssertion>;
  TSTypeLiteral?: ExplVisitNode<S, t.TSTypeLiteral>;
  TSTypeOperator?: ExplVisitNode<S, t.TSTypeOperator>;
  TSTypeParameter?: ExplVisitNode<S, t.TSTypeParameter>;
  TSTypeParameterDeclaration?: ExplVisitNode<S, t.TSTypeParameterDeclaration>;
  TSTypeParameterInstantiation?: ExplVisitNode<
    S,
    t.TSTypeParameterInstantiation
  >;
  TSTypePredicate?: ExplVisitNode<S, t.TSTypePredicate>;
  TSTypeQuery?: ExplVisitNode<S, t.TSTypeQuery>;
  TSTypeReference?: ExplVisitNode<S, t.TSTypeReference>;
  TSUndefinedKeyword?: ExplVisitNode<S, t.TSUndefinedKeyword>;
  TSUnionType?: ExplVisitNode<S, t.TSUnionType>;
  TSUnknownKeyword?: ExplVisitNode<S, t.TSUnknownKeyword>;
  TSVoidKeyword?: ExplVisitNode<S, t.TSVoidKeyword>;
  TaggedTemplateExpression?: ExplVisitNode<S, t.TaggedTemplateExpression>;
  TemplateElement?: ExplVisitNode<S, t.TemplateElement>;
  TemplateLiteral?: ExplVisitNode<S, t.TemplateLiteral>;
  ThisExpression?: ExplVisitNode<S, t.ThisExpression>;
  ThisTypeAnnotation?: ExplVisitNode<S, t.ThisTypeAnnotation>;
  ThrowStatement?: ExplVisitNode<S, t.ThrowStatement>;
  TopicReference?: ExplVisitNode<S, t.TopicReference>;
  TryStatement?: ExplVisitNode<S, t.TryStatement>;
  TupleExpression?: ExplVisitNode<S, t.TupleExpression>;
  TupleTypeAnnotation?: ExplVisitNode<S, t.TupleTypeAnnotation>;
  TypeAlias?: ExplVisitNode<S, t.TypeAlias>;
  TypeAnnotation?: ExplVisitNode<S, t.TypeAnnotation>;
  TypeCastExpression?: ExplVisitNode<S, t.TypeCastExpression>;
  TypeParameter?: ExplVisitNode<S, t.TypeParameter>;
  TypeParameterDeclaration?: ExplVisitNode<S, t.TypeParameterDeclaration>;
  TypeParameterInstantiation?: ExplVisitNode<S, t.TypeParameterInstantiation>;
  TypeofTypeAnnotation?: ExplVisitNode<S, t.TypeofTypeAnnotation>;
  UnaryExpression?: ExplVisitNode<S, t.UnaryExpression>;
  UnionTypeAnnotation?: ExplVisitNode<S, t.UnionTypeAnnotation>;
  UpdateExpression?: ExplVisitNode<S, t.UpdateExpression>;
  V8IntrinsicIdentifier?: ExplVisitNode<S, t.V8IntrinsicIdentifier>;
  VariableDeclaration?: ExplVisitNode<S, t.VariableDeclaration>;
  VariableDeclarator?: ExplVisitNode<S, t.VariableDeclarator>;
  Variance?: ExplVisitNode<S, t.Variance>;
  VoidTypeAnnotation?: ExplVisitNode<S, t.VoidTypeAnnotation>;
  WhileStatement?: ExplVisitNode<S, t.WhileStatement>;
  WithStatement?: ExplVisitNode<S, t.WithStatement>;
  YieldExpression?: ExplVisitNode<S, t.YieldExpression>;
}

interface VisitorBaseNodes<S> {
  AnyTypeAnnotation?: VisitNode<S, t.AnyTypeAnnotation>;
  ArgumentPlaceholder?: VisitNode<S, t.ArgumentPlaceholder>;
  ArrayExpression?: VisitNode<S, t.ArrayExpression>;
  ArrayPattern?: VisitNode<S, t.ArrayPattern>;
  ArrayTypeAnnotation?: VisitNode<S, t.ArrayTypeAnnotation>;
  ArrowFunctionExpression?: VisitNode<S, t.ArrowFunctionExpression>;
  AssignmentExpression?: VisitNode<S, t.AssignmentExpression>;
  AssignmentPattern?: VisitNode<S, t.AssignmentPattern>;
  AwaitExpression?: VisitNode<S, t.AwaitExpression>;
  BigIntLiteral?: VisitNode<S, t.BigIntLiteral>;
  BinaryExpression?: VisitNode<S, t.BinaryExpression>;
  BindExpression?: VisitNode<S, t.BindExpression>;
  BlockStatement?: VisitNode<S, t.BlockStatement>;
  BooleanLiteral?: VisitNode<S, t.BooleanLiteral>;
  BooleanLiteralTypeAnnotation?: VisitNode<S, t.BooleanLiteralTypeAnnotation>;
  BooleanTypeAnnotation?: VisitNode<S, t.BooleanTypeAnnotation>;
  BreakStatement?: VisitNode<S, t.BreakStatement>;
  CallExpression?: VisitNode<S, t.CallExpression>;
  CatchClause?: VisitNode<S, t.CatchClause>;
  ClassAccessorProperty?: VisitNode<S, t.ClassAccessorProperty>;
  ClassBody?: VisitNode<S, t.ClassBody>;
  ClassDeclaration?: VisitNode<S, t.ClassDeclaration>;
  ClassExpression?: VisitNode<S, t.ClassExpression>;
  ClassImplements?: VisitNode<S, t.ClassImplements>;
  ClassMethod?: VisitNode<S, t.ClassMethod>;
  ClassPrivateMethod?: VisitNode<S, t.ClassPrivateMethod>;
  ClassPrivateProperty?: VisitNode<S, t.ClassPrivateProperty>;
  ClassProperty?: VisitNode<S, t.ClassProperty>;
  ConditionalExpression?: VisitNode<S, t.ConditionalExpression>;
  ContinueStatement?: VisitNode<S, t.ContinueStatement>;
  DebuggerStatement?: VisitNode<S, t.DebuggerStatement>;
  DecimalLiteral?: VisitNode<S, t.DecimalLiteral>;
  DeclareClass?: VisitNode<S, t.DeclareClass>;
  DeclareExportAllDeclaration?: VisitNode<S, t.DeclareExportAllDeclaration>;
  DeclareExportDeclaration?: VisitNode<S, t.DeclareExportDeclaration>;
  DeclareFunction?: VisitNode<S, t.DeclareFunction>;
  DeclareInterface?: VisitNode<S, t.DeclareInterface>;
  DeclareModule?: VisitNode<S, t.DeclareModule>;
  DeclareModuleExports?: VisitNode<S, t.DeclareModuleExports>;
  DeclareOpaqueType?: VisitNode<S, t.DeclareOpaqueType>;
  DeclareTypeAlias?: VisitNode<S, t.DeclareTypeAlias>;
  DeclareVariable?: VisitNode<S, t.DeclareVariable>;
  DeclaredPredicate?: VisitNode<S, t.DeclaredPredicate>;
  Decorator?: VisitNode<S, t.Decorator>;
  Directive?: VisitNode<S, t.Directive>;
  DirectiveLiteral?: VisitNode<S, t.DirectiveLiteral>;
  DoExpression?: VisitNode<S, t.DoExpression>;
  DoWhileStatement?: VisitNode<S, t.DoWhileStatement>;
  EmptyStatement?: VisitNode<S, t.EmptyStatement>;
  EmptyTypeAnnotation?: VisitNode<S, t.EmptyTypeAnnotation>;
  EnumBooleanBody?: VisitNode<S, t.EnumBooleanBody>;
  EnumBooleanMember?: VisitNode<S, t.EnumBooleanMember>;
  EnumDeclaration?: VisitNode<S, t.EnumDeclaration>;
  EnumDefaultedMember?: VisitNode<S, t.EnumDefaultedMember>;
  EnumNumberBody?: VisitNode<S, t.EnumNumberBody>;
  EnumNumberMember?: VisitNode<S, t.EnumNumberMember>;
  EnumStringBody?: VisitNode<S, t.EnumStringBody>;
  EnumStringMember?: VisitNode<S, t.EnumStringMember>;
  EnumSymbolBody?: VisitNode<S, t.EnumSymbolBody>;
  ExistsTypeAnnotation?: VisitNode<S, t.ExistsTypeAnnotation>;
  ExportAllDeclaration?: VisitNode<S, t.ExportAllDeclaration>;
  ExportDefaultDeclaration?: VisitNode<S, t.ExportDefaultDeclaration>;
  ExportDefaultSpecifier?: VisitNode<S, t.ExportDefaultSpecifier>;
  ExportNamedDeclaration?: VisitNode<S, t.ExportNamedDeclaration>;
  ExportNamespaceSpecifier?: VisitNode<S, t.ExportNamespaceSpecifier>;
  ExportSpecifier?: VisitNode<S, t.ExportSpecifier>;
  ExpressionStatement?: VisitNode<S, t.ExpressionStatement>;
  File?: VisitNode<S, t.File>;
  ForInStatement?: VisitNode<S, t.ForInStatement>;
  ForOfStatement?: VisitNode<S, t.ForOfStatement>;
  ForStatement?: VisitNode<S, t.ForStatement>;
  FunctionDeclaration?: VisitNode<S, t.FunctionDeclaration>;
  FunctionExpression?: VisitNode<S, t.FunctionExpression>;
  FunctionTypeAnnotation?: VisitNode<S, t.FunctionTypeAnnotation>;
  FunctionTypeParam?: VisitNode<S, t.FunctionTypeParam>;
  GenericTypeAnnotation?: VisitNode<S, t.GenericTypeAnnotation>;
  Identifier?: VisitNode<S, t.Identifier>;
  IfStatement?: VisitNode<S, t.IfStatement>;
  Import?: VisitNode<S, t.Import>;
  ImportAttribute?: VisitNode<S, t.ImportAttribute>;
  ImportDeclaration?: VisitNode<S, t.ImportDeclaration>;
  ImportDefaultSpecifier?: VisitNode<S, t.ImportDefaultSpecifier>;
  ImportExpression?: VisitNode<S, t.ImportExpression>;
  ImportNamespaceSpecifier?: VisitNode<S, t.ImportNamespaceSpecifier>;
  ImportSpecifier?: VisitNode<S, t.ImportSpecifier>;
  IndexedAccessType?: VisitNode<S, t.IndexedAccessType>;
  InferredPredicate?: VisitNode<S, t.InferredPredicate>;
  InterfaceDeclaration?: VisitNode<S, t.InterfaceDeclaration>;
  InterfaceExtends?: VisitNode<S, t.InterfaceExtends>;
  InterfaceTypeAnnotation?: VisitNode<S, t.InterfaceTypeAnnotation>;
  InterpreterDirective?: VisitNode<S, t.InterpreterDirective>;
  IntersectionTypeAnnotation?: VisitNode<S, t.IntersectionTypeAnnotation>;
  JSXAttribute?: VisitNode<S, t.JSXAttribute>;
  JSXClosingElement?: VisitNode<S, t.JSXClosingElement>;
  JSXClosingFragment?: VisitNode<S, t.JSXClosingFragment>;
  JSXElement?: VisitNode<S, t.JSXElement>;
  JSXEmptyExpression?: VisitNode<S, t.JSXEmptyExpression>;
  JSXExpressionContainer?: VisitNode<S, t.JSXExpressionContainer>;
  JSXFragment?: VisitNode<S, t.JSXFragment>;
  JSXIdentifier?: VisitNode<S, t.JSXIdentifier>;
  JSXMemberExpression?: VisitNode<S, t.JSXMemberExpression>;
  JSXNamespacedName?: VisitNode<S, t.JSXNamespacedName>;
  JSXOpeningElement?: VisitNode<S, t.JSXOpeningElement>;
  JSXOpeningFragment?: VisitNode<S, t.JSXOpeningFragment>;
  JSXSpreadAttribute?: VisitNode<S, t.JSXSpreadAttribute>;
  JSXSpreadChild?: VisitNode<S, t.JSXSpreadChild>;
  JSXText?: VisitNode<S, t.JSXText>;
  LabeledStatement?: VisitNode<S, t.LabeledStatement>;
  LogicalExpression?: VisitNode<S, t.LogicalExpression>;
  MemberExpression?: VisitNode<S, t.MemberExpression>;
  MetaProperty?: VisitNode<S, t.MetaProperty>;
  MixedTypeAnnotation?: VisitNode<S, t.MixedTypeAnnotation>;
  ModuleExpression?: VisitNode<S, t.ModuleExpression>;
  NewExpression?: VisitNode<S, t.NewExpression>;
  NullLiteral?: VisitNode<S, t.NullLiteral>;
  NullLiteralTypeAnnotation?: VisitNode<S, t.NullLiteralTypeAnnotation>;
  NullableTypeAnnotation?: VisitNode<S, t.NullableTypeAnnotation>;
  NumberLiteral?: VisitNode<S, t.NumberLiteral>;
  NumberLiteralTypeAnnotation?: VisitNode<S, t.NumberLiteralTypeAnnotation>;
  NumberTypeAnnotation?: VisitNode<S, t.NumberTypeAnnotation>;
  NumericLiteral?: VisitNode<S, t.NumericLiteral>;
  ObjectExpression?: VisitNode<S, t.ObjectExpression>;
  ObjectMethod?: VisitNode<S, t.ObjectMethod>;
  ObjectPattern?: VisitNode<S, t.ObjectPattern>;
  ObjectProperty?: VisitNode<S, t.ObjectProperty>;
  ObjectTypeAnnotation?: VisitNode<S, t.ObjectTypeAnnotation>;
  ObjectTypeCallProperty?: VisitNode<S, t.ObjectTypeCallProperty>;
  ObjectTypeIndexer?: VisitNode<S, t.ObjectTypeIndexer>;
  ObjectTypeInternalSlot?: VisitNode<S, t.ObjectTypeInternalSlot>;
  ObjectTypeProperty?: VisitNode<S, t.ObjectTypeProperty>;
  ObjectTypeSpreadProperty?: VisitNode<S, t.ObjectTypeSpreadProperty>;
  OpaqueType?: VisitNode<S, t.OpaqueType>;
  OptionalCallExpression?: VisitNode<S, t.OptionalCallExpression>;
  OptionalIndexedAccessType?: VisitNode<S, t.OptionalIndexedAccessType>;
  OptionalMemberExpression?: VisitNode<S, t.OptionalMemberExpression>;
  ParenthesizedExpression?: VisitNode<S, t.ParenthesizedExpression>;
  PipelineBareFunction?: VisitNode<S, t.PipelineBareFunction>;
  PipelinePrimaryTopicReference?: VisitNode<S, t.PipelinePrimaryTopicReference>;
  PipelineTopicExpression?: VisitNode<S, t.PipelineTopicExpression>;
  Placeholder?: VisitNode<S, t.Placeholder>;
  PrivateName?: VisitNode<S, t.PrivateName>;
  Program?: VisitNode<S, t.Program>;
  QualifiedTypeIdentifier?: VisitNode<S, t.QualifiedTypeIdentifier>;
  RecordExpression?: VisitNode<S, t.RecordExpression>;
  RegExpLiteral?: VisitNode<S, t.RegExpLiteral>;
  RegexLiteral?: VisitNode<S, t.RegexLiteral>;
  RestElement?: VisitNode<S, t.RestElement>;
  RestProperty?: VisitNode<S, t.RestProperty>;
  ReturnStatement?: VisitNode<S, t.ReturnStatement>;
  SequenceExpression?: VisitNode<S, t.SequenceExpression>;
  SpreadElement?: VisitNode<S, t.SpreadElement>;
  SpreadProperty?: VisitNode<S, t.SpreadProperty>;
  StaticBlock?: VisitNode<S, t.StaticBlock>;
  StringLiteral?: VisitNode<S, t.StringLiteral>;
  StringLiteralTypeAnnotation?: VisitNode<S, t.StringLiteralTypeAnnotation>;
  StringTypeAnnotation?: VisitNode<S, t.StringTypeAnnotation>;
  Super?: VisitNode<S, t.Super>;
  SwitchCase?: VisitNode<S, t.SwitchCase>;
  SwitchStatement?: VisitNode<S, t.SwitchStatement>;
  SymbolTypeAnnotation?: VisitNode<S, t.SymbolTypeAnnotation>;
  TSAnyKeyword?: VisitNode<S, t.TSAnyKeyword>;
  TSArrayType?: VisitNode<S, t.TSArrayType>;
  TSAsExpression?: VisitNode<S, t.TSAsExpression>;
  TSBigIntKeyword?: VisitNode<S, t.TSBigIntKeyword>;
  TSBooleanKeyword?: VisitNode<S, t.TSBooleanKeyword>;
  TSCallSignatureDeclaration?: VisitNode<S, t.TSCallSignatureDeclaration>;
  TSConditionalType?: VisitNode<S, t.TSConditionalType>;
  TSConstructSignatureDeclaration?: VisitNode<
    S,
    t.TSConstructSignatureDeclaration
  >;
  TSConstructorType?: VisitNode<S, t.TSConstructorType>;
  TSDeclareFunction?: VisitNode<S, t.TSDeclareFunction>;
  TSDeclareMethod?: VisitNode<S, t.TSDeclareMethod>;
  TSEnumDeclaration?: VisitNode<S, t.TSEnumDeclaration>;
  TSEnumMember?: VisitNode<S, t.TSEnumMember>;
  TSExportAssignment?: VisitNode<S, t.TSExportAssignment>;
  TSExpressionWithTypeArguments?: VisitNode<S, t.TSExpressionWithTypeArguments>;
  TSExternalModuleReference?: VisitNode<S, t.TSExternalModuleReference>;
  TSFunctionType?: VisitNode<S, t.TSFunctionType>;
  TSImportEqualsDeclaration?: VisitNode<S, t.TSImportEqualsDeclaration>;
  TSImportType?: VisitNode<S, t.TSImportType>;
  TSIndexSignature?: VisitNode<S, t.TSIndexSignature>;
  TSIndexedAccessType?: VisitNode<S, t.TSIndexedAccessType>;
  TSInferType?: VisitNode<S, t.TSInferType>;
  TSInstantiationExpression?: VisitNode<S, t.TSInstantiationExpression>;
  TSInterfaceBody?: VisitNode<S, t.TSInterfaceBody>;
  TSInterfaceDeclaration?: VisitNode<S, t.TSInterfaceDeclaration>;
  TSIntersectionType?: VisitNode<S, t.TSIntersectionType>;
  TSIntrinsicKeyword?: VisitNode<S, t.TSIntrinsicKeyword>;
  TSLiteralType?: VisitNode<S, t.TSLiteralType>;
  TSMappedType?: VisitNode<S, t.TSMappedType>;
  TSMethodSignature?: VisitNode<S, t.TSMethodSignature>;
  TSModuleBlock?: VisitNode<S, t.TSModuleBlock>;
  TSModuleDeclaration?: VisitNode<S, t.TSModuleDeclaration>;
  TSNamedTupleMember?: VisitNode<S, t.TSNamedTupleMember>;
  TSNamespaceExportDeclaration?: VisitNode<S, t.TSNamespaceExportDeclaration>;
  TSNeverKeyword?: VisitNode<S, t.TSNeverKeyword>;
  TSNonNullExpression?: VisitNode<S, t.TSNonNullExpression>;
  TSNullKeyword?: VisitNode<S, t.TSNullKeyword>;
  TSNumberKeyword?: VisitNode<S, t.TSNumberKeyword>;
  TSObjectKeyword?: VisitNode<S, t.TSObjectKeyword>;
  TSOptionalType?: VisitNode<S, t.TSOptionalType>;
  TSParameterProperty?: VisitNode<S, t.TSParameterProperty>;
  TSParenthesizedType?: VisitNode<S, t.TSParenthesizedType>;
  TSPropertySignature?: VisitNode<S, t.TSPropertySignature>;
  TSQualifiedName?: VisitNode<S, t.TSQualifiedName>;
  TSRestType?: VisitNode<S, t.TSRestType>;
  TSSatisfiesExpression?: VisitNode<S, t.TSSatisfiesExpression>;
  TSStringKeyword?: VisitNode<S, t.TSStringKeyword>;
  TSSymbolKeyword?: VisitNode<S, t.TSSymbolKeyword>;
  TSThisType?: VisitNode<S, t.TSThisType>;
  TSTupleType?: VisitNode<S, t.TSTupleType>;
  TSTypeAliasDeclaration?: VisitNode<S, t.TSTypeAliasDeclaration>;
  TSTypeAnnotation?: VisitNode<S, t.TSTypeAnnotation>;
  TSTypeAssertion?: VisitNode<S, t.TSTypeAssertion>;
  TSTypeLiteral?: VisitNode<S, t.TSTypeLiteral>;
  TSTypeOperator?: VisitNode<S, t.TSTypeOperator>;
  TSTypeParameter?: VisitNode<S, t.TSTypeParameter>;
  TSTypeParameterDeclaration?: VisitNode<S, t.TSTypeParameterDeclaration>;
  TSTypeParameterInstantiation?: VisitNode<S, t.TSTypeParameterInstantiation>;
  TSTypePredicate?: VisitNode<S, t.TSTypePredicate>;
  TSTypeQuery?: VisitNode<S, t.TSTypeQuery>;
  TSTypeReference?: VisitNode<S, t.TSTypeReference>;
  TSUndefinedKeyword?: VisitNode<S, t.TSUndefinedKeyword>;
  TSUnionType?: VisitNode<S, t.TSUnionType>;
  TSUnknownKeyword?: VisitNode<S, t.TSUnknownKeyword>;
  TSVoidKeyword?: VisitNode<S, t.TSVoidKeyword>;
  TaggedTemplateExpression?: VisitNode<S, t.TaggedTemplateExpression>;
  TemplateElement?: VisitNode<S, t.TemplateElement>;
  TemplateLiteral?: VisitNode<S, t.TemplateLiteral>;
  ThisExpression?: VisitNode<S, t.ThisExpression>;
  ThisTypeAnnotation?: VisitNode<S, t.ThisTypeAnnotation>;
  ThrowStatement?: VisitNode<S, t.ThrowStatement>;
  TopicReference?: VisitNode<S, t.TopicReference>;
  TryStatement?: VisitNode<S, t.TryStatement>;
  TupleExpression?: VisitNode<S, t.TupleExpression>;
  TupleTypeAnnotation?: VisitNode<S, t.TupleTypeAnnotation>;
  TypeAlias?: VisitNode<S, t.TypeAlias>;
  TypeAnnotation?: VisitNode<S, t.TypeAnnotation>;
  TypeCastExpression?: VisitNode<S, t.TypeCastExpression>;
  TypeParameter?: VisitNode<S, t.TypeParameter>;
  TypeParameterDeclaration?: VisitNode<S, t.TypeParameterDeclaration>;
  TypeParameterInstantiation?: VisitNode<S, t.TypeParameterInstantiation>;
  TypeofTypeAnnotation?: VisitNode<S, t.TypeofTypeAnnotation>;
  UnaryExpression?: VisitNode<S, t.UnaryExpression>;
  UnionTypeAnnotation?: VisitNode<S, t.UnionTypeAnnotation>;
  UpdateExpression?: VisitNode<S, t.UpdateExpression>;
  V8IntrinsicIdentifier?: VisitNode<S, t.V8IntrinsicIdentifier>;
  VariableDeclaration?: VisitNode<S, t.VariableDeclaration>;
  VariableDeclarator?: VisitNode<S, t.VariableDeclarator>;
  Variance?: VisitNode<S, t.Variance>;
  VoidTypeAnnotation?: VisitNode<S, t.VoidTypeAnnotation>;
  WhileStatement?: VisitNode<S, t.WhileStatement>;
  WithStatement?: VisitNode<S, t.WithStatement>;
  YieldExpression?: VisitNode<S, t.YieldExpression>;
}

interface VisitorBaseAliases<S> {
  Accessor?: VisitNode<S, t.Accessor>;
  Binary?: VisitNode<S, t.Binary>;
  Block?: VisitNode<S, t.Block>;
  BlockParent?: VisitNode<S, t.BlockParent>;
  Class?: VisitNode<S, t.Class>;
  CompletionStatement?: VisitNode<S, t.CompletionStatement>;
  Conditional?: VisitNode<S, t.Conditional>;
  Declaration?: VisitNode<S, t.Declaration>;
  EnumBody?: VisitNode<S, t.EnumBody>;
  EnumMember?: VisitNode<S, t.EnumMember>;
  ExportDeclaration?: VisitNode<S, t.ExportDeclaration>;
  Expression?: VisitNode<S, t.Expression>;
  ExpressionWrapper?: VisitNode<S, t.ExpressionWrapper>;
  Flow?: VisitNode<S, t.Flow>;
  FlowBaseAnnotation?: VisitNode<S, t.FlowBaseAnnotation>;
  FlowDeclaration?: VisitNode<S, t.FlowDeclaration>;
  FlowPredicate?: VisitNode<S, t.FlowPredicate>;
  FlowType?: VisitNode<S, t.FlowType>;
  For?: VisitNode<S, t.For>;
  ForXStatement?: VisitNode<S, t.ForXStatement>;
  Function?: VisitNode<S, t.Function>;
  FunctionParent?: VisitNode<S, t.FunctionParent>;
  Immutable?: VisitNode<S, t.Immutable>;
  ImportOrExportDeclaration?: VisitNode<S, t.ImportOrExportDeclaration>;
  JSX?: VisitNode<S, t.JSX>;
  LVal?: VisitNode<S, t.LVal>;
  Literal?: VisitNode<S, t.Literal>;
  Loop?: VisitNode<S, t.Loop>;
  Method?: VisitNode<S, t.Method>;
  Miscellaneous?: VisitNode<S, t.Miscellaneous>;
  ModuleDeclaration?: VisitNode<S, t.ModuleDeclaration>;
  ModuleSpecifier?: VisitNode<S, t.ModuleSpecifier>;
  ObjectMember?: VisitNode<S, t.ObjectMember>;
  Pattern?: VisitNode<S, t.Pattern>;
  PatternLike?: VisitNode<S, t.PatternLike>;
  Private?: VisitNode<S, t.Private>;
  Property?: VisitNode<S, t.Property>;
  Pureish?: VisitNode<S, t.Pureish>;
  Scopable?: VisitNode<S, t.Scopable>;
  Standardized?: VisitNode<S, t.Standardized>;
  Statement?: VisitNode<S, t.Statement>;
  TSBaseType?: VisitNode<S, t.TSBaseType>;
  TSEntityName?: VisitNode<S, t.TSEntityName>;
  TSType?: VisitNode<S, t.TSType>;
  TSTypeElement?: VisitNode<S, t.TSTypeElement>;
  Terminatorless?: VisitNode<S, t.Terminatorless>;
  TypeScript?: VisitNode<S, t.TypeScript>;
  UnaryLike?: VisitNode<S, t.UnaryLike>;
  UserWhitespacable?: VisitNode<S, t.UserWhitespacable>;
  While?: VisitNode<S, t.While>;
}

type VisitPhase = "enter" | "exit";
interface VisitNodeObject<S, P extends t.Node> {
    enter?: VisitNodeFunction<S, P>;
    exit?: VisitNodeFunction<S, P>;
}
interface ExplVisitNode<S, P extends t.Node> {
    enter?: VisitNodeFunction<S, P>[];
    exit?: VisitNodeFunction<S, P>[];
}
interface ExplodedVisitor<S = unknown> extends ExplVisitorBase<S>, ExplVisitNode<S, t.Node> {
    _exploded: true;
    _verified: true;
}
interface VisitorVirtualAliases<S> {
    BindingIdentifier?: VisitNode<S, VirtualTypeAliases["BindingIdentifier"]>;
    BlockScoped?: VisitNode<S, VirtualTypeAliases["BlockScoped"]>;
    ExistentialTypeParam?: VisitNode<S, VirtualTypeAliases["ExistentialTypeParam"]>;
    Expression?: VisitNode<S, VirtualTypeAliases["Expression"]>;
    ForAwaitStatement?: VisitNode<S, VirtualTypeAliases["ForAwaitStatement"]>;
    Generated?: VisitNode<S, VirtualTypeAliases["Generated"]>;
    NumericLiteralTypeAnnotation?: VisitNode<S, VirtualTypeAliases["NumericLiteralTypeAnnotation"]>;
    Pure?: VisitNode<S, VirtualTypeAliases["Pure"]>;
    Referenced?: VisitNode<S, VirtualTypeAliases["Referenced"]>;
    ReferencedIdentifier?: VisitNode<S, VirtualTypeAliases["ReferencedIdentifier"]>;
    ReferencedMemberExpression?: VisitNode<S, VirtualTypeAliases["ReferencedMemberExpression"]>;
    Scope?: VisitNode<S, VirtualTypeAliases["Scope"]>;
    Statement?: VisitNode<S, VirtualTypeAliases["Statement"]>;
    User?: VisitNode<S, VirtualTypeAliases["User"]>;
    Var?: VisitNode<S, VirtualTypeAliases["Var"]>;
}
interface VisitorBase<S> extends VisitNodeObject<S, t.Node>, VisitorBaseNodes<S>, VisitorBaseAliases<S>, VisitorVirtualAliases<S> {
    [k: `${string}|${string}`]: VisitNode<S, t.Node>;
}
type Visitor<S = unknown> = VisitorBase<S> | ExplodedVisitor<S>;
type VisitNode<S, P extends t.Node> = VisitNodeFunction<S, P> | VisitNodeObject<S, P>;
type VisitNodeFunction<S, P extends t.Node> = (this: S, path: NodePath_Final<P>, state: S) => void;

/**
 * Starting at the parent path of the current `NodePath` and going up the
 * tree, return the first `NodePath` that causes the provided `callback`
 * to return a truthy value, or `null` if the `callback` never returns a
 * truthy value.
 */
declare function findParent(this: NodePath_Final, callback: (path: NodePath_Final) => boolean): NodePath_Final | null;
/**
 * Starting at current `NodePath` and going up the tree, return the first
 * `NodePath` that causes the provided `callback` to return a truthy value,
 * or `null` if the `callback` never returns a truthy value.
 */
declare function find(this: NodePath_Final, callback: (path: NodePath_Final) => boolean): NodePath_Final | null;
/**
 * Get the parent function of the current path.
 */
declare function getFunctionParent(this: NodePath_Final): NodePath_Final<t.Function> | null;
/**
 * Walk up the tree until we hit a parent node path in a list.
 */
declare function getStatementParent(this: NodePath_Final): NodePath_Final<t.Statement>;
/**
 * Get the deepest common ancestor and then from it, get the earliest relationship path
 * to that ancestor.
 *
 * Earliest is defined as being "before" all the other nodes in terms of list container
 * position and visiting key.
 */
declare function getEarliestCommonAncestorFrom(this: NodePath_Final, paths: Array<NodePath_Final>): NodePath_Final;
/**
 * Get the earliest path in the tree where the provided `paths` intersect.
 *
 * TODO: Possible optimisation target.
 */
declare function getDeepestCommonAncestorFrom(this: NodePath_Final, paths: Array<NodePath_Final>, filter?: (deepest: NodePath_Final, i: number, ancestries: NodePath_Final[][]) => NodePath_Final): NodePath_Final;
/**
 * Build an array of node paths containing the entire ancestry of the current node path.
 *
 * NOTE: The current node path is included in this.
 */
declare function getAncestry(this: NodePath_Final): Array<NodePath_Final>;
/**
 * A helper to find if `this` path is an ancestor of @param maybeDescendant
 */
declare function isAncestor(this: NodePath_Final, maybeDescendant: NodePath_Final): boolean;
/**
 * A helper to find if `this` path is a descendant of @param maybeAncestor
 */
declare function isDescendant(this: NodePath_Final, maybeAncestor: NodePath_Final): boolean;
declare function inType(this: NodePath_Final, ...candidateTypes: string[]): boolean;

/**
 * Infer the type of the current `NodePath`.
 */
declare function getTypeAnnotation(this: NodePath_Final): t.FlowType | t.TSType;
declare function isBaseType(this: NodePath_Final, baseName: string, soft?: boolean): boolean;
declare function couldBeBaseType(this: NodePath_Final, name: string): boolean;
declare function baseTypeStrictlyMatches(this: NodePath_Final, rightArg: NodePath_Final): boolean;
declare function isGenericType(this: NodePath_Final, genericName: string): boolean;

/**
 * Replace a node with an array of multiple. This method performs the following steps:
 *
 *  - Inherit the comments of first provided node with that of the current node.
 *  - Insert the provided nodes after the current node.
 *  - Remove the current node.
 */
declare function replaceWithMultiple(this: NodePath_Final, nodes: t.Node | t.Node[]): NodePath_Final[];
/**
 * Parse a string as an expression and replace the current node with the result.
 *
 * NOTE: This is typically not a good idea to use. Building source strings when
 * transforming ASTs is an antipattern and SHOULD NOT be encouraged. Even if it's
 * easier to use, your transforms will be extremely brittle.
 */
declare function replaceWithSourceString(this: NodePath_Final, replacement: string): [NodePath<t.ArrayExpression> | NodePath<t.ArrowFunctionExpression> | NodePath<t.AssignmentExpression> | NodePath<t.AwaitExpression> | NodePath<t.BigIntLiteral> | NodePath<t.BinaryExpression> | NodePath<t.BindExpression> | NodePath<t.BooleanLiteral> | NodePath<t.CallExpression> | NodePath<t.ClassExpression> | NodePath<t.ConditionalExpression> | NodePath<t.DecimalLiteral> | NodePath<t.DoExpression> | NodePath<t.FunctionExpression> | NodePath<t.Identifier> | NodePath<t.Import> | NodePath<t.ImportExpression> | NodePath<t.JSXElement> | NodePath<t.JSXFragment> | NodePath<t.LogicalExpression> | NodePath<t.MemberExpression> | NodePath<t.MetaProperty> | NodePath<t.ModuleExpression> | NodePath<t.NewExpression> | NodePath<t.NullLiteral> | NodePath<t.NumericLiteral> | NodePath<t.ObjectExpression> | NodePath<t.OptionalCallExpression> | NodePath<t.OptionalMemberExpression> | NodePath<t.ParenthesizedExpression> | NodePath<t.PipelineBareFunction> | NodePath<t.PipelinePrimaryTopicReference> | NodePath<t.PipelineTopicExpression> | NodePath<t.RecordExpression> | NodePath<t.RegExpLiteral> | NodePath<t.SequenceExpression> | NodePath<t.StringLiteral> | NodePath<t.TSAsExpression> | NodePath<t.TSInstantiationExpression> | NodePath<t.TSNonNullExpression> | NodePath<t.TSSatisfiesExpression> | NodePath<t.TSTypeAssertion> | NodePath<t.TaggedTemplateExpression> | NodePath<t.TemplateLiteral> | NodePath<t.ThisExpression> | NodePath<t.TopicReference> | NodePath<t.TupleExpression> | NodePath<t.TypeCastExpression> | NodePath<t.UnaryExpression> | NodePath<t.UpdateExpression> | NodePath<t.YieldExpression>];
/**
 * Replace the current node with another.
 */
declare function replaceWith<R extends t.Node>(this: NodePath_Final, replacementPath: R): [NodePath_Final<R>];
declare function replaceWith<R extends NodePath_Final>(this: NodePath_Final, replacementPath: R): [R];
/**
 * This method takes an array of statements nodes and then explodes it
 * into expressions. This method retains completion records which is
 * extremely important to retain original semantics.
 */
declare function replaceExpressionWithStatements(this: NodePath_Final, nodes: Array<t.Statement>): (NodePath<t.BlockStatement> | NodePath<t.BreakStatement> | NodePath<t.ClassDeclaration> | NodePath<t.ContinueStatement> | NodePath<t.DebuggerStatement> | NodePath<t.DeclareClass> | NodePath<t.DeclareExportAllDeclaration> | NodePath<t.DeclareExportDeclaration> | NodePath<t.DeclareFunction> | NodePath<t.DeclareInterface> | NodePath<t.DeclareModule> | NodePath<t.DeclareModuleExports> | NodePath<t.DeclareOpaqueType> | NodePath<t.DeclareTypeAlias> | NodePath<t.DeclareVariable> | NodePath<t.DoWhileStatement> | NodePath<t.EmptyStatement> | NodePath<t.EnumDeclaration> | NodePath<t.ExportAllDeclaration> | NodePath<t.ExportDefaultDeclaration> | NodePath<t.ExportNamedDeclaration> | NodePath<t.ExpressionStatement> | NodePath<t.ForInStatement> | NodePath<t.ForOfStatement> | NodePath<t.ForStatement> | NodePath<t.FunctionDeclaration> | NodePath<t.IfStatement> | NodePath<t.ImportDeclaration> | NodePath<t.InterfaceDeclaration> | NodePath<t.LabeledStatement> | NodePath<t.OpaqueType> | NodePath<t.ReturnStatement> | NodePath<t.SwitchStatement> | NodePath<t.TSDeclareFunction> | NodePath<t.TSEnumDeclaration> | NodePath<t.TSExportAssignment> | NodePath<t.TSImportEqualsDeclaration> | NodePath<t.TSInterfaceDeclaration> | NodePath<t.TSModuleDeclaration> | NodePath<t.TSNamespaceExportDeclaration> | NodePath<t.TSTypeAliasDeclaration> | NodePath<t.ThrowStatement> | NodePath<t.TryStatement> | NodePath<t.TypeAlias> | NodePath<t.VariableDeclaration> | NodePath<t.WhileStatement> | NodePath<t.WithStatement>)[] | (NodePath<t.ArrayExpression> | NodePath<t.ArrowFunctionExpression> | NodePath<t.AssignmentExpression> | NodePath<t.AwaitExpression> | NodePath<t.BigIntLiteral> | NodePath<t.BinaryExpression> | NodePath<t.BindExpression> | NodePath<t.BooleanLiteral> | NodePath<t.CallExpression> | NodePath<t.ClassExpression> | NodePath<t.ConditionalExpression> | NodePath<t.DecimalLiteral> | NodePath<t.DoExpression> | NodePath<t.FunctionExpression> | NodePath<t.Identifier> | NodePath<t.Import> | NodePath<t.ImportExpression> | NodePath<t.JSXElement> | NodePath<t.JSXFragment> | NodePath<t.LogicalExpression> | NodePath<t.MemberExpression> | NodePath<t.MetaProperty> | NodePath<t.ModuleExpression> | NodePath<t.NewExpression> | NodePath<t.NullLiteral> | NodePath<t.NumericLiteral> | NodePath<t.ObjectExpression> | NodePath<t.OptionalCallExpression> | NodePath<t.OptionalMemberExpression> | NodePath<t.ParenthesizedExpression> | NodePath<t.PipelineBareFunction> | NodePath<t.PipelinePrimaryTopicReference> | NodePath<t.PipelineTopicExpression> | NodePath<t.RecordExpression> | NodePath<t.RegExpLiteral> | NodePath<t.SequenceExpression> | NodePath<t.StringLiteral> | NodePath<t.TSAnyKeyword> | NodePath<t.TSArrayType> | NodePath<t.TSAsExpression> | NodePath<t.TSBigIntKeyword> | NodePath<t.TSBooleanKeyword> | NodePath<t.TSConditionalType> | NodePath<t.TSConstructorType> | NodePath<t.TSExpressionWithTypeArguments> | NodePath<t.TSFunctionType> | NodePath<t.TSImportType> | NodePath<t.TSIndexedAccessType> | NodePath<t.TSInferType> | NodePath<t.TSInstantiationExpression> | NodePath<t.TSIntersectionType> | NodePath<t.TSIntrinsicKeyword> | NodePath<t.TSLiteralType> | NodePath<t.TSMappedType> | NodePath<t.TSNeverKeyword> | NodePath<t.TSNonNullExpression> | NodePath<t.TSNullKeyword> | NodePath<t.TSNumberKeyword> | NodePath<t.TSObjectKeyword> | NodePath<t.TSOptionalType> | NodePath<t.TSParenthesizedType> | NodePath<t.TSRestType> | NodePath<t.TSSatisfiesExpression> | NodePath<t.TSStringKeyword> | NodePath<t.TSSymbolKeyword> | NodePath<t.TSThisType> | NodePath<t.TSTupleType> | NodePath<t.TSTypeAssertion> | NodePath<t.TSTypeLiteral> | NodePath<t.TSTypeOperator> | NodePath<t.TSTypePredicate> | NodePath<t.TSTypeQuery> | NodePath<t.TSTypeReference> | NodePath<t.TSUndefinedKeyword> | NodePath<t.TSUnionType> | NodePath<t.TSUnknownKeyword> | NodePath<t.TSVoidKeyword> | NodePath<t.TaggedTemplateExpression> | NodePath<t.TemplateLiteral> | NodePath<t.ThisExpression> | NodePath<t.TopicReference> | NodePath<t.TupleExpression> | NodePath<t.TypeCastExpression> | NodePath<t.UnaryExpression> | NodePath<t.UpdateExpression> | NodePath<t.YieldExpression>)[];
declare function replaceInline(this: NodePath_Final, nodes: t.Node | Array<t.Node>): NodePath_Final[];

/**
 * Walk the input `node` and statically evaluate if it's truthy.
 *
 * Returning `true` when we're sure that the expression will evaluate to a
 * truthy value, `false` if we're sure that it will evaluate to a falsy
 * value and `undefined` if we aren't sure. Because of this please do not
 * rely on coercion when using this method and check with === if it's false.
 *
 * For example do:
 *
 *   if (t.evaluateTruthy(node) === false) falsyLogic();
 *
 * **AND NOT**
 *
 *   if (!t.evaluateTruthy(node)) falsyLogic();
 *
 */
declare function evaluateTruthy(this: NodePath_Final): boolean;
/**
 * Walk the input `node` and statically evaluate it.
 *
 * Returns an object in the form `{ confident, value, deopt }`. `confident`
 * indicates whether or not we had to drop out of evaluating the expression
 * because of hitting an unknown node that we couldn't confidently find the
 * value of, in which case `deopt` is the path of said node.
 *
 * Example:
 *
 *   t.evaluate(parse("5 + 5")) // { confident: true, value: 10 }
 *   t.evaluate(parse("!true")) // { confident: true, value: false }
 *   t.evaluate(parse("foo + foo")) // { confident: false, value: undefined, deopt: NodePath }
 *
 */
declare function evaluate(this: NodePath_Final): {
    confident: boolean;
    value: any;
    deopt?: NodePath_Final;
};

declare function toComputedKey(this: NodePath_Final): t.PrivateName | t.Expression;
declare function ensureBlock(this: NodePath_Final<t.Loop | t.WithStatement | t.Function | t.LabeledStatement | t.CatchClause>): void;
/**
 * Given an arbitrary function, process its content as if it were an arrow function, moving references
 * to "this", "arguments", "super", and such into the function's parent scope. This method is useful if
 * you have wrapped some set of items in an IIFE or other function, but want "this", "arguments", and super"
 * to continue behaving as expected.
 */
declare function unwrapFunctionEnvironment(this: NodePath_Final): void;
/**
 * Convert a given arrow function into a normal ES5 function expression.
 */
declare function arrowFunctionToExpression(this: NodePath_Final<t.ArrowFunctionExpression>, { allowInsertArrow, allowInsertArrowWithRest, noNewArrows, }?: {
    allowInsertArrow?: boolean | void;
    allowInsertArrowWithRest?: boolean | void;
    noNewArrows?: boolean;
}): NodePath_Final<Exclude<t.Function, t.Method | t.ArrowFunctionExpression> | t.CallExpression>;
declare function splitExportDeclaration(this: NodePath_Final<t.ExportDefaultDeclaration | t.ExportNamedDeclaration>): NodePath_Final<t.Declaration>;
declare function ensureFunctionName<N extends t.FunctionExpression | t.ClassExpression>(this: NodePath_Final<N>, supportUnicodeId: boolean): null | NodePath_Final<N>;

/**
 * Match the current node if it matches the provided `pattern`.
 *
 * For example, given the match `React.createClass` it would match the
 * parsed nodes of `React.createClass` and `React["createClass"]`.
 */
declare function matchesPattern(this: NodePath_Final, pattern: string, allowPartial?: boolean): boolean;
/**
 * Check whether we have the input `key`. If the `key` references an array then we check
 * if the array has any items, otherwise we just check if it's falsy.
 */
declare function has<N extends t.Node>(this: NodePath_Final<N>, key: keyof N): boolean;
/**
 * Description
 */
declare function isStatic(this: NodePath_Final): boolean;
/**
 * Opposite of `has`.
 */
declare function isnt<N extends t.Node>(this: NodePath_Final<N>, key: keyof N): boolean;
/**
 * Check whether the path node `key` strict equals `value`.
 */
declare function equals<N extends t.Node>(this: NodePath_Final<N>, key: keyof N, value: any): boolean;
/**
 * Check the type against our stored internal type of the node. This is handy when a node has
 * been removed yet we still internally know the type and need it to calculate node replacement.
 */
declare function isNodeType(this: NodePath_Final, type: string): boolean;
/**
 * This checks whether or not we're in one of the following positions:
 *
 *   for (KEY in right);
 *   for (KEY;;);
 *
 * This is because these spots allow VariableDeclarations AND normal expressions so we need
 * to tell the path replacement that it's ok to replace this with an expression.
 */
declare function canHaveVariableDeclarationOrExpression(this: NodePath_Final): boolean;
/**
 * This checks whether we are swapping an arrow function's body between an
 * expression and a block statement (or vice versa).
 *
 * This is because arrow functions may implicitly return an expression, which
 * is the same as containing a block statement.
 */
declare function canSwapBetweenExpressionAndStatement(this: NodePath_Final, replacement: t.Node): boolean;
/**
 * Check whether the current path references a completion record
 */
declare function isCompletionRecord(this: NodePath_Final, allowInsideFunction?: boolean): boolean;
/**
 * Check whether or not the current `key` allows either a single statement or block statement
 * so we can explode it if necessary.
 */
declare function isStatementOrBlock(this: NodePath_Final): boolean;
/**
 * Check if the currently assigned path references the `importName` of `moduleSource`.
 */
declare function referencesImport(this: NodePath_Final, moduleSource: string, importName: string): boolean;
/**
 * Get the source code associated with this node.
 */
declare function getSource(this: NodePath_Final): string;
declare function willIMaybeExecuteBefore(this: NodePath_Final, target: NodePath_Final): boolean;
type RelativeExecutionStatus = "before" | "after" | "unknown";
/**
 * Given a `target` check the execution status of it relative to the current path.
 *
 * "Execution status" simply refers to where or not we **think** this will execute
 * before or after the input `target` element.
 */
declare function _guessExecutionStatusRelativeTo(this: NodePath_Final, target: NodePath_Final): RelativeExecutionStatus;
/**
 * Resolve a "pointer" `NodePath` to it's absolute path.
 */
declare function resolve(this: NodePath_Final, dangerous?: boolean, resolved?: NodePath_Final[]): NodePath_Final;
declare function isConstantExpression(this: NodePath_Final): boolean;
declare function isInStrictMode(this: NodePath_Final): boolean;

declare function call(this: NodePath_Final, key: VisitPhase): boolean;
declare function isDenylisted(this: NodePath_Final): boolean;

declare function visit(this: NodePath_Final): boolean;
declare function skip(this: NodePath_Final): void;
declare function skipKey(this: NodePath_Final, key: string): void;
declare function stop(this: NodePath_Final): void;
declare function setScope(this: NodePath_Final): void;
declare function setContext<S = unknown>(this: NodePath_Final, context?: TraversalContext<S>): NodePath_Final;
/**
 * Here we resync the node paths `key` and `container`. If they've changed according
 * to what we have stored internally then we attempt to resync by crawling and looking
 * for the new values.
 */
declare function resync(this: NodePath_Final): void;
declare function popContext(this: NodePath_Final): void;
declare function pushContext(this: NodePath_Final, context: TraversalContext): void;
declare function setup(this: NodePath_Final, parentPath: NodePath_Final | undefined, container: t.Node | t.Node[], listKey: string, key: string | number): void;
declare function setKey(this: NodePath_Final, key: string | number): void;
declare function requeue(this: NodePath_Final, pathToQueue?: NodePath_Final): void;
declare function requeueComputedKeyAndDecorators(this: NodePath_Final<t.Method | t.Property>): void;

declare function remove(this: NodePath_Final): void;

/**
 * Insert the provided nodes before the current one.
 */
declare function insertBefore(this: NodePath_Final, nodes_: t.Node | t.Node[]): NodePath_Final[];
/**
 * Insert the provided nodes after the current one. When inserting nodes after an
 * expression, ensure that the completion record is correct by pushing the current node.
 */
declare function insertAfter(this: NodePath_Final, nodes_: t.Node | t.Node[]): NodePath_Final[];
/**
 * Update all sibling node paths after `fromIndex` by `incrementBy`.
 */
declare function updateSiblingKeys(this: NodePath_Final, fromIndex: number, incrementBy: number): void;
declare function unshiftContainer<N extends t.Node, K extends keyof N & string>(this: NodePath_Final<N>, listKey: K, nodes: N[K] extends (infer E)[] ? E | E[] : never): (NodePath<t.AnyTypeAnnotation> | NodePath<t.ArgumentPlaceholder> | NodePath<t.ArrayExpression> | NodePath<t.ArrayPattern> | NodePath<t.ArrayTypeAnnotation> | NodePath<t.ArrowFunctionExpression> | NodePath<t.AssignmentExpression> | NodePath<t.AssignmentPattern> | NodePath<t.AwaitExpression> | NodePath<t.BigIntLiteral> | NodePath<t.BinaryExpression> | NodePath<t.BindExpression> | NodePath<t.BlockStatement> | NodePath<t.BooleanLiteral> | NodePath<t.BooleanLiteralTypeAnnotation> | NodePath<t.BooleanTypeAnnotation> | NodePath<t.BreakStatement> | NodePath<t.CallExpression> | NodePath<t.CatchClause> | NodePath<t.ClassAccessorProperty> | NodePath<t.ClassBody> | NodePath<t.ClassDeclaration> | NodePath<t.ClassExpression> | NodePath<t.ClassImplements> | NodePath<t.ClassMethod> | NodePath<t.ClassPrivateMethod> | NodePath<t.ClassPrivateProperty> | NodePath<t.ClassProperty> | NodePath<t.ConditionalExpression> | NodePath<t.ContinueStatement> | NodePath<t.DebuggerStatement> | NodePath<t.DecimalLiteral> | NodePath<t.DeclareClass> | NodePath<t.DeclareExportAllDeclaration> | NodePath<t.DeclareExportDeclaration> | NodePath<t.DeclareFunction> | NodePath<t.DeclareInterface> | NodePath<t.DeclareModule> | NodePath<t.DeclareModuleExports> | NodePath<t.DeclareOpaqueType> | NodePath<t.DeclareTypeAlias> | NodePath<t.DeclareVariable> | NodePath<t.DeclaredPredicate> | NodePath<t.Decorator> | NodePath<t.Directive> | NodePath<t.DirectiveLiteral> | NodePath<t.DoExpression> | NodePath<t.DoWhileStatement> | NodePath<t.EmptyStatement> | NodePath<t.EmptyTypeAnnotation> | NodePath<t.EnumBooleanBody> | NodePath<t.EnumBooleanMember> | NodePath<t.EnumDeclaration> | NodePath<t.EnumDefaultedMember> | NodePath<t.EnumNumberBody> | NodePath<t.EnumNumberMember> | NodePath<t.EnumStringBody> | NodePath<t.EnumStringMember> | NodePath<t.EnumSymbolBody> | NodePath<t.ExistsTypeAnnotation> | NodePath<t.ExportAllDeclaration> | NodePath<t.ExportDefaultDeclaration> | NodePath<t.ExportDefaultSpecifier> | NodePath<t.ExportNamedDeclaration> | NodePath<t.ExportNamespaceSpecifier> | NodePath<t.ExportSpecifier> | NodePath<t.ExpressionStatement> | NodePath<t.File> | NodePath<t.ForInStatement> | NodePath<t.ForOfStatement> | NodePath<t.ForStatement> | NodePath<t.FunctionDeclaration> | NodePath<t.FunctionExpression> | NodePath<t.FunctionTypeAnnotation> | NodePath<t.FunctionTypeParam> | NodePath<t.GenericTypeAnnotation> | NodePath<t.Identifier> | NodePath<t.IfStatement> | NodePath<t.Import> | NodePath<t.ImportAttribute> | NodePath<t.ImportDeclaration> | NodePath<t.ImportDefaultSpecifier> | NodePath<t.ImportExpression> | NodePath<t.ImportNamespaceSpecifier> | NodePath<t.ImportSpecifier> | NodePath<t.IndexedAccessType> | NodePath<t.InferredPredicate> | NodePath<t.InterfaceDeclaration> | NodePath<t.InterfaceExtends> | NodePath<t.InterfaceTypeAnnotation> | NodePath<t.InterpreterDirective> | NodePath<t.IntersectionTypeAnnotation> | NodePath<t.JSXAttribute> | NodePath<t.JSXClosingElement> | NodePath<t.JSXClosingFragment> | NodePath<t.JSXElement> | NodePath<t.JSXEmptyExpression> | NodePath<t.JSXExpressionContainer> | NodePath<t.JSXFragment> | NodePath<t.JSXIdentifier> | NodePath<t.JSXMemberExpression> | NodePath<t.JSXNamespacedName> | NodePath<t.JSXOpeningElement> | NodePath<t.JSXOpeningFragment> | NodePath<t.JSXSpreadAttribute> | NodePath<t.JSXSpreadChild> | NodePath<t.JSXText> | NodePath<t.LabeledStatement> | NodePath<t.LogicalExpression> | NodePath<t.MemberExpression> | NodePath<t.MetaProperty> | NodePath<t.MixedTypeAnnotation> | NodePath<t.ModuleExpression> | NodePath<t.NewExpression> | NodePath<t.NullLiteral> | NodePath<t.NullLiteralTypeAnnotation> | NodePath<t.NullableTypeAnnotation> | NodePath<t.NumberLiteral> | NodePath<t.NumberLiteralTypeAnnotation> | NodePath<t.NumberTypeAnnotation> | NodePath<t.NumericLiteral> | NodePath<t.ObjectExpression> | NodePath<t.ObjectMethod> | NodePath<t.ObjectPattern> | NodePath<t.ObjectProperty> | NodePath<t.ObjectTypeAnnotation> | NodePath<t.ObjectTypeCallProperty> | NodePath<t.ObjectTypeIndexer> | NodePath<t.ObjectTypeInternalSlot> | NodePath<t.ObjectTypeProperty> | NodePath<t.ObjectTypeSpreadProperty> | NodePath<t.OpaqueType> | NodePath<t.OptionalCallExpression> | NodePath<t.OptionalIndexedAccessType> | NodePath<t.OptionalMemberExpression> | NodePath<t.ParenthesizedExpression> | NodePath<t.PipelineBareFunction> | NodePath<t.PipelinePrimaryTopicReference> | NodePath<t.PipelineTopicExpression> | NodePath<t.Placeholder> | NodePath<t.PrivateName> | NodePath<t.Program> | NodePath<t.QualifiedTypeIdentifier> | NodePath<t.RecordExpression> | NodePath<t.RegExpLiteral> | NodePath<t.RegexLiteral> | NodePath<t.RestElement> | NodePath<t.RestProperty> | NodePath<t.ReturnStatement> | NodePath<t.SequenceExpression> | NodePath<t.SpreadElement> | NodePath<t.SpreadProperty> | NodePath<t.StaticBlock> | NodePath<t.StringLiteral> | NodePath<t.StringLiteralTypeAnnotation> | NodePath<t.StringTypeAnnotation> | NodePath<t.Super> | NodePath<t.SwitchCase> | NodePath<t.SwitchStatement> | NodePath<t.SymbolTypeAnnotation> | NodePath<t.TSAnyKeyword> | NodePath<t.TSArrayType> | NodePath<t.TSAsExpression> | NodePath<t.TSBigIntKeyword> | NodePath<t.TSBooleanKeyword> | NodePath<t.TSCallSignatureDeclaration> | NodePath<t.TSConditionalType> | NodePath<t.TSConstructSignatureDeclaration> | NodePath<t.TSConstructorType> | NodePath<t.TSDeclareFunction> | NodePath<t.TSDeclareMethod> | NodePath<t.TSEnumDeclaration> | NodePath<t.TSEnumMember> | NodePath<t.TSExportAssignment> | NodePath<t.TSExpressionWithTypeArguments> | NodePath<t.TSExternalModuleReference> | NodePath<t.TSFunctionType> | NodePath<t.TSImportEqualsDeclaration> | NodePath<t.TSImportType> | NodePath<t.TSIndexSignature> | NodePath<t.TSIndexedAccessType> | NodePath<t.TSInferType> | NodePath<t.TSInstantiationExpression> | NodePath<t.TSInterfaceBody> | NodePath<t.TSInterfaceDeclaration> | NodePath<t.TSIntersectionType> | NodePath<t.TSIntrinsicKeyword> | NodePath<t.TSLiteralType> | NodePath<t.TSMappedType> | NodePath<t.TSMethodSignature> | NodePath<t.TSModuleBlock> | NodePath<t.TSModuleDeclaration> | NodePath<t.TSNamedTupleMember> | NodePath<t.TSNamespaceExportDeclaration> | NodePath<t.TSNeverKeyword> | NodePath<t.TSNonNullExpression> | NodePath<t.TSNullKeyword> | NodePath<t.TSNumberKeyword> | NodePath<t.TSObjectKeyword> | NodePath<t.TSOptionalType> | NodePath<t.TSParameterProperty> | NodePath<t.TSParenthesizedType> | NodePath<t.TSPropertySignature> | NodePath<t.TSQualifiedName> | NodePath<t.TSRestType> | NodePath<t.TSSatisfiesExpression> | NodePath<t.TSStringKeyword> | NodePath<t.TSSymbolKeyword> | NodePath<t.TSThisType> | NodePath<t.TSTupleType> | NodePath<t.TSTypeAliasDeclaration> | NodePath<t.TSTypeAnnotation> | NodePath<t.TSTypeAssertion> | NodePath<t.TSTypeLiteral> | NodePath<t.TSTypeOperator> | NodePath<t.TSTypeParameter> | NodePath<t.TSTypeParameterDeclaration> | NodePath<t.TSTypeParameterInstantiation> | NodePath<t.TSTypePredicate> | NodePath<t.TSTypeQuery> | NodePath<t.TSTypeReference> | NodePath<t.TSUndefinedKeyword> | NodePath<t.TSUnionType> | NodePath<t.TSUnknownKeyword> | NodePath<t.TSVoidKeyword> | NodePath<t.TaggedTemplateExpression> | NodePath<t.TemplateElement> | NodePath<t.TemplateLiteral> | NodePath<t.ThisExpression> | NodePath<t.ThisTypeAnnotation> | NodePath<t.ThrowStatement> | NodePath<t.TopicReference> | NodePath<t.TryStatement> | NodePath<t.TupleExpression> | NodePath<t.TupleTypeAnnotation> | NodePath<t.TypeAlias> | NodePath<t.TypeAnnotation> | NodePath<t.TypeCastExpression> | NodePath<t.TypeParameter> | NodePath<t.TypeParameterDeclaration> | NodePath<t.TypeParameterInstantiation> | NodePath<t.TypeofTypeAnnotation> | NodePath<t.UnaryExpression> | NodePath<t.UnionTypeAnnotation> | NodePath<t.UpdateExpression> | NodePath<t.V8IntrinsicIdentifier> | NodePath<t.VariableDeclaration> | NodePath<t.VariableDeclarator> | NodePath<t.Variance> | NodePath<t.VoidTypeAnnotation> | NodePath<t.WhileStatement> | NodePath<t.WithStatement> | NodePath<t.YieldExpression>)[];
declare function pushContainer<P extends NodePath_Final, K extends string & keyof P["node"]>(this: P, listKey: K, nodes: P["node"][K] extends (infer E)[] ? E | E[] : never): NodePath_Final[];
/**
 * Hoist the current node to the highest scope possible and return a UID
 * referencing it.
 */
declare function hoist<T extends t.Node>(this: NodePath_Final<T>, scope?: Scope): NodePath_Final<t.Expression>;

declare function getOpposite(this: NodePath_Final): NodePath_Final | null;
/**
 * Retrieve the completion records of a given path.
 * Note: to ensure proper support on `break` statement, this method
 * will manipulate the AST around the break statement. Do not call the method
 * twice for the same path.
 *
 * @export
 * @param {NodePath} this
 * @returns {NodePath[]} Completion records
 */
declare function getCompletionRecords(this: NodePath_Final): NodePath_Final[];
declare function getSibling(this: NodePath_Final, key: string | number): NodePath_Final;
declare function getPrevSibling(this: NodePath_Final): NodePath_Final;
declare function getNextSibling(this: NodePath_Final): NodePath_Final;
declare function getAllNextSiblings(this: NodePath_Final): NodePath_Final[];
declare function getAllPrevSiblings(this: NodePath_Final): NodePath_Final[];
type MaybeToIndex<T extends string> = T extends `${bigint}` ? number : T;
type Pattern<Obj extends string, Prop extends string> = `${Obj}.${Prop}`;
type Split<P extends string> = P extends Pattern<infer O, infer U> ? [MaybeToIndex<O>, ...Split<U>] : [MaybeToIndex<P>];
type Trav<Node extends t.Node | t.Node[], Path extends unknown[]> = Path extends [infer K, ...infer R] ? K extends keyof Node ? Node[K] extends t.Node | t.Node[] ? R extends [] ? Node[K] : Trav<Node[K], R> : never : never : never;
type ToNodePath<T> = T extends Array<t.Node | null | undefined> ? Array<NodePath_Final<T[number]>> : T extends t.Node | null | undefined ? NodePath_Final<T> : never;
declare function get<T extends NodePath_Final, K extends keyof T["node"]>(this: T, key: K, context?: boolean | TraversalContext): T extends any ? T["node"][K] extends Array<t.Node | null | undefined> ? Array<NodePath_Final<T["node"][K][number]>> : T["node"][K] extends t.Node | null | undefined ? NodePath_Final<T["node"][K]> : never : never;
declare function get<T extends NodePath_Final, K extends string>(this: T, key: K, context?: boolean | TraversalContext): T extends any ? ToNodePath<Trav<T["node"], Split<K>>> : never;
declare function get(this: NodePath_Final, key: string, context?: true | TraversalContext): NodePath_Final | NodePath_Final[];

declare function getAssignmentIdentifiers(this: NodePath_Final): Record<string, t.Identifier>;
declare function getBindingIdentifiers(duplicates: true): Record<string, t.Identifier[]>;
declare function getBindingIdentifiers(duplicates?: false): Record<string, t.Identifier>;
declare function getBindingIdentifiers(duplicates: boolean): Record<string, t.Identifier[] | t.Identifier>;

declare function getOuterBindingIdentifiers(duplicates: true): Record<string, t.Identifier[]>;
declare function getOuterBindingIdentifiers(duplicates?: false): Record<string, t.Identifier>;
declare function getOuterBindingIdentifiers(duplicates: boolean): Record<string, t.Identifier[] | t.Identifier>;

declare function getBindingIdentifierPaths(duplicates: true, outerOnly?: boolean): Record<string, NodePath_Final<t.Identifier>[]>;
declare function getBindingIdentifierPaths(duplicates: false, outerOnly?: boolean): Record<string, NodePath_Final<t.Identifier>>;
declare function getBindingIdentifierPaths(duplicates?: boolean, outerOnly?: boolean): Record<string, NodePath_Final<t.Identifier> | NodePath_Final<t.Identifier>[]>;

declare function getOuterBindingIdentifierPaths(duplicates: true): Record<string, NodePath_Final<t.Identifier>[]>;
declare function getOuterBindingIdentifierPaths(duplicates?: false): Record<string, NodePath_Final<t.Identifier>>;
declare function getOuterBindingIdentifierPaths(duplicates?: boolean): Record<string, NodePath_Final<t.Identifier> | NodePath_Final<t.Identifier>[]>;

/**
 * Share comments amongst siblings.
 */
declare function shareCommentsWithSiblings(this: NodePath_Final): void;
declare function addComment(this: NodePath_Final, type: t.CommentTypeShorthand, content: string, line?: boolean): void;
/**
 * Give node `comments` of the specified `type`.
 */
declare function addComments(this: NodePath_Final, type: t.CommentTypeShorthand, comments: t.Comment[]): void;

/*
 * This file is auto-generated! Do not modify it directly.
 * To re-generate run 'make build'
 */


type Opts$2<Obj> = Partial<{
  [Prop in keyof Obj]: Obj[Prop] extends t.Node
    ? t.Node
    : Obj[Prop] extends t.Node[]
      ? t.Node[]
      : Obj[Prop];
}>;

interface NodePathAssertions {
  assertAccessor(opts?: Opts$2<t.Accessor>): asserts this is NodePath_Final<t.Accessor>;
  assertAnyTypeAnnotation(
    opts?: Opts$2<t.AnyTypeAnnotation>,
  ): asserts this is NodePath_Final<t.AnyTypeAnnotation>;
  assertArgumentPlaceholder(
    opts?: Opts$2<t.ArgumentPlaceholder>,
  ): asserts this is NodePath_Final<t.ArgumentPlaceholder>;
  assertArrayExpression(
    opts?: Opts$2<t.ArrayExpression>,
  ): asserts this is NodePath_Final<t.ArrayExpression>;
  assertArrayPattern(
    opts?: Opts$2<t.ArrayPattern>,
  ): asserts this is NodePath_Final<t.ArrayPattern>;
  assertArrayTypeAnnotation(
    opts?: Opts$2<t.ArrayTypeAnnotation>,
  ): asserts this is NodePath_Final<t.ArrayTypeAnnotation>;
  assertArrowFunctionExpression(
    opts?: Opts$2<t.ArrowFunctionExpression>,
  ): asserts this is NodePath_Final<t.ArrowFunctionExpression>;
  assertAssignmentExpression(
    opts?: Opts$2<t.AssignmentExpression>,
  ): asserts this is NodePath_Final<t.AssignmentExpression>;
  assertAssignmentPattern(
    opts?: Opts$2<t.AssignmentPattern>,
  ): asserts this is NodePath_Final<t.AssignmentPattern>;
  assertAwaitExpression(
    opts?: Opts$2<t.AwaitExpression>,
  ): asserts this is NodePath_Final<t.AwaitExpression>;
  assertBigIntLiteral(
    opts?: Opts$2<t.BigIntLiteral>,
  ): asserts this is NodePath_Final<t.BigIntLiteral>;
  assertBinary(opts?: Opts$2<t.Binary>): asserts this is NodePath_Final<t.Binary>;
  assertBinaryExpression(
    opts?: Opts$2<t.BinaryExpression>,
  ): asserts this is NodePath_Final<t.BinaryExpression>;
  assertBindExpression(
    opts?: Opts$2<t.BindExpression>,
  ): asserts this is NodePath_Final<t.BindExpression>;
  assertBlock(opts?: Opts$2<t.Block>): asserts this is NodePath_Final<t.Block>;
  assertBlockParent(
    opts?: Opts$2<t.BlockParent>,
  ): asserts this is NodePath_Final<t.BlockParent>;
  assertBlockStatement(
    opts?: Opts$2<t.BlockStatement>,
  ): asserts this is NodePath_Final<t.BlockStatement>;
  assertBooleanLiteral(
    opts?: Opts$2<t.BooleanLiteral>,
  ): asserts this is NodePath_Final<t.BooleanLiteral>;
  assertBooleanLiteralTypeAnnotation(
    opts?: Opts$2<t.BooleanLiteralTypeAnnotation>,
  ): asserts this is NodePath_Final<t.BooleanLiteralTypeAnnotation>;
  assertBooleanTypeAnnotation(
    opts?: Opts$2<t.BooleanTypeAnnotation>,
  ): asserts this is NodePath_Final<t.BooleanTypeAnnotation>;
  assertBreakStatement(
    opts?: Opts$2<t.BreakStatement>,
  ): asserts this is NodePath_Final<t.BreakStatement>;
  assertCallExpression(
    opts?: Opts$2<t.CallExpression>,
  ): asserts this is NodePath_Final<t.CallExpression>;
  assertCatchClause(
    opts?: Opts$2<t.CatchClause>,
  ): asserts this is NodePath_Final<t.CatchClause>;
  assertClass(opts?: Opts$2<t.Class>): asserts this is NodePath_Final<t.Class>;
  assertClassAccessorProperty(
    opts?: Opts$2<t.ClassAccessorProperty>,
  ): asserts this is NodePath_Final<t.ClassAccessorProperty>;
  assertClassBody(
    opts?: Opts$2<t.ClassBody>,
  ): asserts this is NodePath_Final<t.ClassBody>;
  assertClassDeclaration(
    opts?: Opts$2<t.ClassDeclaration>,
  ): asserts this is NodePath_Final<t.ClassDeclaration>;
  assertClassExpression(
    opts?: Opts$2<t.ClassExpression>,
  ): asserts this is NodePath_Final<t.ClassExpression>;
  assertClassImplements(
    opts?: Opts$2<t.ClassImplements>,
  ): asserts this is NodePath_Final<t.ClassImplements>;
  assertClassMethod(
    opts?: Opts$2<t.ClassMethod>,
  ): asserts this is NodePath_Final<t.ClassMethod>;
  assertClassPrivateMethod(
    opts?: Opts$2<t.ClassPrivateMethod>,
  ): asserts this is NodePath_Final<t.ClassPrivateMethod>;
  assertClassPrivateProperty(
    opts?: Opts$2<t.ClassPrivateProperty>,
  ): asserts this is NodePath_Final<t.ClassPrivateProperty>;
  assertClassProperty(
    opts?: Opts$2<t.ClassProperty>,
  ): asserts this is NodePath_Final<t.ClassProperty>;
  assertCompletionStatement(
    opts?: Opts$2<t.CompletionStatement>,
  ): asserts this is NodePath_Final<t.CompletionStatement>;
  assertConditional(
    opts?: Opts$2<t.Conditional>,
  ): asserts this is NodePath_Final<t.Conditional>;
  assertConditionalExpression(
    opts?: Opts$2<t.ConditionalExpression>,
  ): asserts this is NodePath_Final<t.ConditionalExpression>;
  assertContinueStatement(
    opts?: Opts$2<t.ContinueStatement>,
  ): asserts this is NodePath_Final<t.ContinueStatement>;
  assertDebuggerStatement(
    opts?: Opts$2<t.DebuggerStatement>,
  ): asserts this is NodePath_Final<t.DebuggerStatement>;
  assertDecimalLiteral(
    opts?: Opts$2<t.DecimalLiteral>,
  ): asserts this is NodePath_Final<t.DecimalLiteral>;
  assertDeclaration(
    opts?: Opts$2<t.Declaration>,
  ): asserts this is NodePath_Final<t.Declaration>;
  assertDeclareClass(
    opts?: Opts$2<t.DeclareClass>,
  ): asserts this is NodePath_Final<t.DeclareClass>;
  assertDeclareExportAllDeclaration(
    opts?: Opts$2<t.DeclareExportAllDeclaration>,
  ): asserts this is NodePath_Final<t.DeclareExportAllDeclaration>;
  assertDeclareExportDeclaration(
    opts?: Opts$2<t.DeclareExportDeclaration>,
  ): asserts this is NodePath_Final<t.DeclareExportDeclaration>;
  assertDeclareFunction(
    opts?: Opts$2<t.DeclareFunction>,
  ): asserts this is NodePath_Final<t.DeclareFunction>;
  assertDeclareInterface(
    opts?: Opts$2<t.DeclareInterface>,
  ): asserts this is NodePath_Final<t.DeclareInterface>;
  assertDeclareModule(
    opts?: Opts$2<t.DeclareModule>,
  ): asserts this is NodePath_Final<t.DeclareModule>;
  assertDeclareModuleExports(
    opts?: Opts$2<t.DeclareModuleExports>,
  ): asserts this is NodePath_Final<t.DeclareModuleExports>;
  assertDeclareOpaqueType(
    opts?: Opts$2<t.DeclareOpaqueType>,
  ): asserts this is NodePath_Final<t.DeclareOpaqueType>;
  assertDeclareTypeAlias(
    opts?: Opts$2<t.DeclareTypeAlias>,
  ): asserts this is NodePath_Final<t.DeclareTypeAlias>;
  assertDeclareVariable(
    opts?: Opts$2<t.DeclareVariable>,
  ): asserts this is NodePath_Final<t.DeclareVariable>;
  assertDeclaredPredicate(
    opts?: Opts$2<t.DeclaredPredicate>,
  ): asserts this is NodePath_Final<t.DeclaredPredicate>;
  assertDecorator(
    opts?: Opts$2<t.Decorator>,
  ): asserts this is NodePath_Final<t.Decorator>;
  assertDirective(
    opts?: Opts$2<t.Directive>,
  ): asserts this is NodePath_Final<t.Directive>;
  assertDirectiveLiteral(
    opts?: Opts$2<t.DirectiveLiteral>,
  ): asserts this is NodePath_Final<t.DirectiveLiteral>;
  assertDoExpression(
    opts?: Opts$2<t.DoExpression>,
  ): asserts this is NodePath_Final<t.DoExpression>;
  assertDoWhileStatement(
    opts?: Opts$2<t.DoWhileStatement>,
  ): asserts this is NodePath_Final<t.DoWhileStatement>;
  assertEmptyStatement(
    opts?: Opts$2<t.EmptyStatement>,
  ): asserts this is NodePath_Final<t.EmptyStatement>;
  assertEmptyTypeAnnotation(
    opts?: Opts$2<t.EmptyTypeAnnotation>,
  ): asserts this is NodePath_Final<t.EmptyTypeAnnotation>;
  assertEnumBody(opts?: Opts$2<t.EnumBody>): asserts this is NodePath_Final<t.EnumBody>;
  assertEnumBooleanBody(
    opts?: Opts$2<t.EnumBooleanBody>,
  ): asserts this is NodePath_Final<t.EnumBooleanBody>;
  assertEnumBooleanMember(
    opts?: Opts$2<t.EnumBooleanMember>,
  ): asserts this is NodePath_Final<t.EnumBooleanMember>;
  assertEnumDeclaration(
    opts?: Opts$2<t.EnumDeclaration>,
  ): asserts this is NodePath_Final<t.EnumDeclaration>;
  assertEnumDefaultedMember(
    opts?: Opts$2<t.EnumDefaultedMember>,
  ): asserts this is NodePath_Final<t.EnumDefaultedMember>;
  assertEnumMember(
    opts?: Opts$2<t.EnumMember>,
  ): asserts this is NodePath_Final<t.EnumMember>;
  assertEnumNumberBody(
    opts?: Opts$2<t.EnumNumberBody>,
  ): asserts this is NodePath_Final<t.EnumNumberBody>;
  assertEnumNumberMember(
    opts?: Opts$2<t.EnumNumberMember>,
  ): asserts this is NodePath_Final<t.EnumNumberMember>;
  assertEnumStringBody(
    opts?: Opts$2<t.EnumStringBody>,
  ): asserts this is NodePath_Final<t.EnumStringBody>;
  assertEnumStringMember(
    opts?: Opts$2<t.EnumStringMember>,
  ): asserts this is NodePath_Final<t.EnumStringMember>;
  assertEnumSymbolBody(
    opts?: Opts$2<t.EnumSymbolBody>,
  ): asserts this is NodePath_Final<t.EnumSymbolBody>;
  assertExistsTypeAnnotation(
    opts?: Opts$2<t.ExistsTypeAnnotation>,
  ): asserts this is NodePath_Final<t.ExistsTypeAnnotation>;
  assertExportAllDeclaration(
    opts?: Opts$2<t.ExportAllDeclaration>,
  ): asserts this is NodePath_Final<t.ExportAllDeclaration>;
  assertExportDeclaration(
    opts?: Opts$2<t.ExportDeclaration>,
  ): asserts this is NodePath_Final<t.ExportDeclaration>;
  assertExportDefaultDeclaration(
    opts?: Opts$2<t.ExportDefaultDeclaration>,
  ): asserts this is NodePath_Final<t.ExportDefaultDeclaration>;
  assertExportDefaultSpecifier(
    opts?: Opts$2<t.ExportDefaultSpecifier>,
  ): asserts this is NodePath_Final<t.ExportDefaultSpecifier>;
  assertExportNamedDeclaration(
    opts?: Opts$2<t.ExportNamedDeclaration>,
  ): asserts this is NodePath_Final<t.ExportNamedDeclaration>;
  assertExportNamespaceSpecifier(
    opts?: Opts$2<t.ExportNamespaceSpecifier>,
  ): asserts this is NodePath_Final<t.ExportNamespaceSpecifier>;
  assertExportSpecifier(
    opts?: Opts$2<t.ExportSpecifier>,
  ): asserts this is NodePath_Final<t.ExportSpecifier>;
  assertExpression(
    opts?: Opts$2<t.Expression>,
  ): asserts this is NodePath_Final<t.Expression>;
  assertExpressionStatement(
    opts?: Opts$2<t.ExpressionStatement>,
  ): asserts this is NodePath_Final<t.ExpressionStatement>;
  assertExpressionWrapper(
    opts?: Opts$2<t.ExpressionWrapper>,
  ): asserts this is NodePath_Final<t.ExpressionWrapper>;
  assertFile(opts?: Opts$2<t.File>): asserts this is NodePath_Final<t.File>;
  assertFlow(opts?: Opts$2<t.Flow>): asserts this is NodePath_Final<t.Flow>;
  assertFlowBaseAnnotation(
    opts?: Opts$2<t.FlowBaseAnnotation>,
  ): asserts this is NodePath_Final<t.FlowBaseAnnotation>;
  assertFlowDeclaration(
    opts?: Opts$2<t.FlowDeclaration>,
  ): asserts this is NodePath_Final<t.FlowDeclaration>;
  assertFlowPredicate(
    opts?: Opts$2<t.FlowPredicate>,
  ): asserts this is NodePath_Final<t.FlowPredicate>;
  assertFlowType(opts?: Opts$2<t.FlowType>): asserts this is NodePath_Final<t.FlowType>;
  assertFor(opts?: Opts$2<t.For>): asserts this is NodePath_Final<t.For>;
  assertForInStatement(
    opts?: Opts$2<t.ForInStatement>,
  ): asserts this is NodePath_Final<t.ForInStatement>;
  assertForOfStatement(
    opts?: Opts$2<t.ForOfStatement>,
  ): asserts this is NodePath_Final<t.ForOfStatement>;
  assertForStatement(
    opts?: Opts$2<t.ForStatement>,
  ): asserts this is NodePath_Final<t.ForStatement>;
  assertForXStatement(
    opts?: Opts$2<t.ForXStatement>,
  ): asserts this is NodePath_Final<t.ForXStatement>;
  assertFunction(opts?: Opts$2<t.Function>): asserts this is NodePath_Final<t.Function>;
  assertFunctionDeclaration(
    opts?: Opts$2<t.FunctionDeclaration>,
  ): asserts this is NodePath_Final<t.FunctionDeclaration>;
  assertFunctionExpression(
    opts?: Opts$2<t.FunctionExpression>,
  ): asserts this is NodePath_Final<t.FunctionExpression>;
  assertFunctionParent(
    opts?: Opts$2<t.FunctionParent>,
  ): asserts this is NodePath_Final<t.FunctionParent>;
  assertFunctionTypeAnnotation(
    opts?: Opts$2<t.FunctionTypeAnnotation>,
  ): asserts this is NodePath_Final<t.FunctionTypeAnnotation>;
  assertFunctionTypeParam(
    opts?: Opts$2<t.FunctionTypeParam>,
  ): asserts this is NodePath_Final<t.FunctionTypeParam>;
  assertGenericTypeAnnotation(
    opts?: Opts$2<t.GenericTypeAnnotation>,
  ): asserts this is NodePath_Final<t.GenericTypeAnnotation>;
  assertIdentifier(
    opts?: Opts$2<t.Identifier>,
  ): asserts this is NodePath_Final<t.Identifier>;
  assertIfStatement(
    opts?: Opts$2<t.IfStatement>,
  ): asserts this is NodePath_Final<t.IfStatement>;
  assertImmutable(
    opts?: Opts$2<t.Immutable>,
  ): asserts this is NodePath_Final<t.Immutable>;
  assertImport(opts?: Opts$2<t.Import>): asserts this is NodePath_Final<t.Import>;
  assertImportAttribute(
    opts?: Opts$2<t.ImportAttribute>,
  ): asserts this is NodePath_Final<t.ImportAttribute>;
  assertImportDeclaration(
    opts?: Opts$2<t.ImportDeclaration>,
  ): asserts this is NodePath_Final<t.ImportDeclaration>;
  assertImportDefaultSpecifier(
    opts?: Opts$2<t.ImportDefaultSpecifier>,
  ): asserts this is NodePath_Final<t.ImportDefaultSpecifier>;
  assertImportExpression(
    opts?: Opts$2<t.ImportExpression>,
  ): asserts this is NodePath_Final<t.ImportExpression>;
  assertImportNamespaceSpecifier(
    opts?: Opts$2<t.ImportNamespaceSpecifier>,
  ): asserts this is NodePath_Final<t.ImportNamespaceSpecifier>;
  assertImportOrExportDeclaration(
    opts?: Opts$2<t.ImportOrExportDeclaration>,
  ): asserts this is NodePath_Final<t.ImportOrExportDeclaration>;
  assertImportSpecifier(
    opts?: Opts$2<t.ImportSpecifier>,
  ): asserts this is NodePath_Final<t.ImportSpecifier>;
  assertIndexedAccessType(
    opts?: Opts$2<t.IndexedAccessType>,
  ): asserts this is NodePath_Final<t.IndexedAccessType>;
  assertInferredPredicate(
    opts?: Opts$2<t.InferredPredicate>,
  ): asserts this is NodePath_Final<t.InferredPredicate>;
  assertInterfaceDeclaration(
    opts?: Opts$2<t.InterfaceDeclaration>,
  ): asserts this is NodePath_Final<t.InterfaceDeclaration>;
  assertInterfaceExtends(
    opts?: Opts$2<t.InterfaceExtends>,
  ): asserts this is NodePath_Final<t.InterfaceExtends>;
  assertInterfaceTypeAnnotation(
    opts?: Opts$2<t.InterfaceTypeAnnotation>,
  ): asserts this is NodePath_Final<t.InterfaceTypeAnnotation>;
  assertInterpreterDirective(
    opts?: Opts$2<t.InterpreterDirective>,
  ): asserts this is NodePath_Final<t.InterpreterDirective>;
  assertIntersectionTypeAnnotation(
    opts?: Opts$2<t.IntersectionTypeAnnotation>,
  ): asserts this is NodePath_Final<t.IntersectionTypeAnnotation>;
  assertJSX(opts?: Opts$2<t.JSX>): asserts this is NodePath_Final<t.JSX>;
  assertJSXAttribute(
    opts?: Opts$2<t.JSXAttribute>,
  ): asserts this is NodePath_Final<t.JSXAttribute>;
  assertJSXClosingElement(
    opts?: Opts$2<t.JSXClosingElement>,
  ): asserts this is NodePath_Final<t.JSXClosingElement>;
  assertJSXClosingFragment(
    opts?: Opts$2<t.JSXClosingFragment>,
  ): asserts this is NodePath_Final<t.JSXClosingFragment>;
  assertJSXElement(
    opts?: Opts$2<t.JSXElement>,
  ): asserts this is NodePath_Final<t.JSXElement>;
  assertJSXEmptyExpression(
    opts?: Opts$2<t.JSXEmptyExpression>,
  ): asserts this is NodePath_Final<t.JSXEmptyExpression>;
  assertJSXExpressionContainer(
    opts?: Opts$2<t.JSXExpressionContainer>,
  ): asserts this is NodePath_Final<t.JSXExpressionContainer>;
  assertJSXFragment(
    opts?: Opts$2<t.JSXFragment>,
  ): asserts this is NodePath_Final<t.JSXFragment>;
  assertJSXIdentifier(
    opts?: Opts$2<t.JSXIdentifier>,
  ): asserts this is NodePath_Final<t.JSXIdentifier>;
  assertJSXMemberExpression(
    opts?: Opts$2<t.JSXMemberExpression>,
  ): asserts this is NodePath_Final<t.JSXMemberExpression>;
  assertJSXNamespacedName(
    opts?: Opts$2<t.JSXNamespacedName>,
  ): asserts this is NodePath_Final<t.JSXNamespacedName>;
  assertJSXOpeningElement(
    opts?: Opts$2<t.JSXOpeningElement>,
  ): asserts this is NodePath_Final<t.JSXOpeningElement>;
  assertJSXOpeningFragment(
    opts?: Opts$2<t.JSXOpeningFragment>,
  ): asserts this is NodePath_Final<t.JSXOpeningFragment>;
  assertJSXSpreadAttribute(
    opts?: Opts$2<t.JSXSpreadAttribute>,
  ): asserts this is NodePath_Final<t.JSXSpreadAttribute>;
  assertJSXSpreadChild(
    opts?: Opts$2<t.JSXSpreadChild>,
  ): asserts this is NodePath_Final<t.JSXSpreadChild>;
  assertJSXText(opts?: Opts$2<t.JSXText>): asserts this is NodePath_Final<t.JSXText>;
  assertLVal(opts?: Opts$2<t.LVal>): asserts this is NodePath_Final<t.LVal>;
  assertLabeledStatement(
    opts?: Opts$2<t.LabeledStatement>,
  ): asserts this is NodePath_Final<t.LabeledStatement>;
  assertLiteral(opts?: Opts$2<t.Literal>): asserts this is NodePath_Final<t.Literal>;
  assertLogicalExpression(
    opts?: Opts$2<t.LogicalExpression>,
  ): asserts this is NodePath_Final<t.LogicalExpression>;
  assertLoop(opts?: Opts$2<t.Loop>): asserts this is NodePath_Final<t.Loop>;
  assertMemberExpression(
    opts?: Opts$2<t.MemberExpression>,
  ): asserts this is NodePath_Final<t.MemberExpression>;
  assertMetaProperty(
    opts?: Opts$2<t.MetaProperty>,
  ): asserts this is NodePath_Final<t.MetaProperty>;
  assertMethod(opts?: Opts$2<t.Method>): asserts this is NodePath_Final<t.Method>;
  assertMiscellaneous(
    opts?: Opts$2<t.Miscellaneous>,
  ): asserts this is NodePath_Final<t.Miscellaneous>;
  assertMixedTypeAnnotation(
    opts?: Opts$2<t.MixedTypeAnnotation>,
  ): asserts this is NodePath_Final<t.MixedTypeAnnotation>;
  assertModuleDeclaration(
    opts?: Opts$2<t.ModuleDeclaration>,
  ): asserts this is NodePath_Final<t.ModuleDeclaration>;
  assertModuleExpression(
    opts?: Opts$2<t.ModuleExpression>,
  ): asserts this is NodePath_Final<t.ModuleExpression>;
  assertModuleSpecifier(
    opts?: Opts$2<t.ModuleSpecifier>,
  ): asserts this is NodePath_Final<t.ModuleSpecifier>;
  assertNewExpression(
    opts?: Opts$2<t.NewExpression>,
  ): asserts this is NodePath_Final<t.NewExpression>;
  assertNullLiteral(
    opts?: Opts$2<t.NullLiteral>,
  ): asserts this is NodePath_Final<t.NullLiteral>;
  assertNullLiteralTypeAnnotation(
    opts?: Opts$2<t.NullLiteralTypeAnnotation>,
  ): asserts this is NodePath_Final<t.NullLiteralTypeAnnotation>;
  assertNullableTypeAnnotation(
    opts?: Opts$2<t.NullableTypeAnnotation>,
  ): asserts this is NodePath_Final<t.NullableTypeAnnotation>;
  assertNumberLiteral(
    opts?: Opts$2<t.NumberLiteral>,
  ): asserts this is NodePath_Final<t.NumberLiteral>;
  assertNumberLiteralTypeAnnotation(
    opts?: Opts$2<t.NumberLiteralTypeAnnotation>,
  ): asserts this is NodePath_Final<t.NumberLiteralTypeAnnotation>;
  assertNumberTypeAnnotation(
    opts?: Opts$2<t.NumberTypeAnnotation>,
  ): asserts this is NodePath_Final<t.NumberTypeAnnotation>;
  assertNumericLiteral(
    opts?: Opts$2<t.NumericLiteral>,
  ): asserts this is NodePath_Final<t.NumericLiteral>;
  assertObjectExpression(
    opts?: Opts$2<t.ObjectExpression>,
  ): asserts this is NodePath_Final<t.ObjectExpression>;
  assertObjectMember(
    opts?: Opts$2<t.ObjectMember>,
  ): asserts this is NodePath_Final<t.ObjectMember>;
  assertObjectMethod(
    opts?: Opts$2<t.ObjectMethod>,
  ): asserts this is NodePath_Final<t.ObjectMethod>;
  assertObjectPattern(
    opts?: Opts$2<t.ObjectPattern>,
  ): asserts this is NodePath_Final<t.ObjectPattern>;
  assertObjectProperty(
    opts?: Opts$2<t.ObjectProperty>,
  ): asserts this is NodePath_Final<t.ObjectProperty>;
  assertObjectTypeAnnotation(
    opts?: Opts$2<t.ObjectTypeAnnotation>,
  ): asserts this is NodePath_Final<t.ObjectTypeAnnotation>;
  assertObjectTypeCallProperty(
    opts?: Opts$2<t.ObjectTypeCallProperty>,
  ): asserts this is NodePath_Final<t.ObjectTypeCallProperty>;
  assertObjectTypeIndexer(
    opts?: Opts$2<t.ObjectTypeIndexer>,
  ): asserts this is NodePath_Final<t.ObjectTypeIndexer>;
  assertObjectTypeInternalSlot(
    opts?: Opts$2<t.ObjectTypeInternalSlot>,
  ): asserts this is NodePath_Final<t.ObjectTypeInternalSlot>;
  assertObjectTypeProperty(
    opts?: Opts$2<t.ObjectTypeProperty>,
  ): asserts this is NodePath_Final<t.ObjectTypeProperty>;
  assertObjectTypeSpreadProperty(
    opts?: Opts$2<t.ObjectTypeSpreadProperty>,
  ): asserts this is NodePath_Final<t.ObjectTypeSpreadProperty>;
  assertOpaqueType(
    opts?: Opts$2<t.OpaqueType>,
  ): asserts this is NodePath_Final<t.OpaqueType>;
  assertOptionalCallExpression(
    opts?: Opts$2<t.OptionalCallExpression>,
  ): asserts this is NodePath_Final<t.OptionalCallExpression>;
  assertOptionalIndexedAccessType(
    opts?: Opts$2<t.OptionalIndexedAccessType>,
  ): asserts this is NodePath_Final<t.OptionalIndexedAccessType>;
  assertOptionalMemberExpression(
    opts?: Opts$2<t.OptionalMemberExpression>,
  ): asserts this is NodePath_Final<t.OptionalMemberExpression>;
  assertParenthesizedExpression(
    opts?: Opts$2<t.ParenthesizedExpression>,
  ): asserts this is NodePath_Final<t.ParenthesizedExpression>;
  assertPattern(opts?: Opts$2<t.Pattern>): asserts this is NodePath_Final<t.Pattern>;
  assertPatternLike(
    opts?: Opts$2<t.PatternLike>,
  ): asserts this is NodePath_Final<t.PatternLike>;
  assertPipelineBareFunction(
    opts?: Opts$2<t.PipelineBareFunction>,
  ): asserts this is NodePath_Final<t.PipelineBareFunction>;
  assertPipelinePrimaryTopicReference(
    opts?: Opts$2<t.PipelinePrimaryTopicReference>,
  ): asserts this is NodePath_Final<t.PipelinePrimaryTopicReference>;
  assertPipelineTopicExpression(
    opts?: Opts$2<t.PipelineTopicExpression>,
  ): asserts this is NodePath_Final<t.PipelineTopicExpression>;
  assertPlaceholder(
    opts?: Opts$2<t.Placeholder>,
  ): asserts this is NodePath_Final<t.Placeholder>;
  assertPrivate(opts?: Opts$2<t.Private>): asserts this is NodePath_Final<t.Private>;
  assertPrivateName(
    opts?: Opts$2<t.PrivateName>,
  ): asserts this is NodePath_Final<t.PrivateName>;
  assertProgram(opts?: Opts$2<t.Program>): asserts this is NodePath_Final<t.Program>;
  assertProperty(opts?: Opts$2<t.Property>): asserts this is NodePath_Final<t.Property>;
  assertPureish(opts?: Opts$2<t.Pureish>): asserts this is NodePath_Final<t.Pureish>;
  assertQualifiedTypeIdentifier(
    opts?: Opts$2<t.QualifiedTypeIdentifier>,
  ): asserts this is NodePath_Final<t.QualifiedTypeIdentifier>;
  assertRecordExpression(
    opts?: Opts$2<t.RecordExpression>,
  ): asserts this is NodePath_Final<t.RecordExpression>;
  assertRegExpLiteral(
    opts?: Opts$2<t.RegExpLiteral>,
  ): asserts this is NodePath_Final<t.RegExpLiteral>;
  assertRegexLiteral(
    opts?: Opts$2<t.RegexLiteral>,
  ): asserts this is NodePath_Final<t.RegexLiteral>;
  assertRestElement(
    opts?: Opts$2<t.RestElement>,
  ): asserts this is NodePath_Final<t.RestElement>;
  assertRestProperty(
    opts?: Opts$2<t.RestProperty>,
  ): asserts this is NodePath_Final<t.RestProperty>;
  assertReturnStatement(
    opts?: Opts$2<t.ReturnStatement>,
  ): asserts this is NodePath_Final<t.ReturnStatement>;
  assertScopable(opts?: Opts$2<t.Scopable>): asserts this is NodePath_Final<t.Scopable>;
  assertSequenceExpression(
    opts?: Opts$2<t.SequenceExpression>,
  ): asserts this is NodePath_Final<t.SequenceExpression>;
  assertSpreadElement(
    opts?: Opts$2<t.SpreadElement>,
  ): asserts this is NodePath_Final<t.SpreadElement>;
  assertSpreadProperty(
    opts?: Opts$2<t.SpreadProperty>,
  ): asserts this is NodePath_Final<t.SpreadProperty>;
  assertStandardized(
    opts?: Opts$2<t.Standardized>,
  ): asserts this is NodePath_Final<t.Standardized>;
  assertStatement(
    opts?: Opts$2<t.Statement>,
  ): asserts this is NodePath_Final<t.Statement>;
  assertStaticBlock(
    opts?: Opts$2<t.StaticBlock>,
  ): asserts this is NodePath_Final<t.StaticBlock>;
  assertStringLiteral(
    opts?: Opts$2<t.StringLiteral>,
  ): asserts this is NodePath_Final<t.StringLiteral>;
  assertStringLiteralTypeAnnotation(
    opts?: Opts$2<t.StringLiteralTypeAnnotation>,
  ): asserts this is NodePath_Final<t.StringLiteralTypeAnnotation>;
  assertStringTypeAnnotation(
    opts?: Opts$2<t.StringTypeAnnotation>,
  ): asserts this is NodePath_Final<t.StringTypeAnnotation>;
  assertSuper(opts?: Opts$2<t.Super>): asserts this is NodePath_Final<t.Super>;
  assertSwitchCase(
    opts?: Opts$2<t.SwitchCase>,
  ): asserts this is NodePath_Final<t.SwitchCase>;
  assertSwitchStatement(
    opts?: Opts$2<t.SwitchStatement>,
  ): asserts this is NodePath_Final<t.SwitchStatement>;
  assertSymbolTypeAnnotation(
    opts?: Opts$2<t.SymbolTypeAnnotation>,
  ): asserts this is NodePath_Final<t.SymbolTypeAnnotation>;
  assertTSAnyKeyword(
    opts?: Opts$2<t.TSAnyKeyword>,
  ): asserts this is NodePath_Final<t.TSAnyKeyword>;
  assertTSArrayType(
    opts?: Opts$2<t.TSArrayType>,
  ): asserts this is NodePath_Final<t.TSArrayType>;
  assertTSAsExpression(
    opts?: Opts$2<t.TSAsExpression>,
  ): asserts this is NodePath_Final<t.TSAsExpression>;
  assertTSBaseType(
    opts?: Opts$2<t.TSBaseType>,
  ): asserts this is NodePath_Final<t.TSBaseType>;
  assertTSBigIntKeyword(
    opts?: Opts$2<t.TSBigIntKeyword>,
  ): asserts this is NodePath_Final<t.TSBigIntKeyword>;
  assertTSBooleanKeyword(
    opts?: Opts$2<t.TSBooleanKeyword>,
  ): asserts this is NodePath_Final<t.TSBooleanKeyword>;
  assertTSCallSignatureDeclaration(
    opts?: Opts$2<t.TSCallSignatureDeclaration>,
  ): asserts this is NodePath_Final<t.TSCallSignatureDeclaration>;
  assertTSConditionalType(
    opts?: Opts$2<t.TSConditionalType>,
  ): asserts this is NodePath_Final<t.TSConditionalType>;
  assertTSConstructSignatureDeclaration(
    opts?: Opts$2<t.TSConstructSignatureDeclaration>,
  ): asserts this is NodePath_Final<t.TSConstructSignatureDeclaration>;
  assertTSConstructorType(
    opts?: Opts$2<t.TSConstructorType>,
  ): asserts this is NodePath_Final<t.TSConstructorType>;
  assertTSDeclareFunction(
    opts?: Opts$2<t.TSDeclareFunction>,
  ): asserts this is NodePath_Final<t.TSDeclareFunction>;
  assertTSDeclareMethod(
    opts?: Opts$2<t.TSDeclareMethod>,
  ): asserts this is NodePath_Final<t.TSDeclareMethod>;
  assertTSEntityName(
    opts?: Opts$2<t.TSEntityName>,
  ): asserts this is NodePath_Final<t.TSEntityName>;
  assertTSEnumDeclaration(
    opts?: Opts$2<t.TSEnumDeclaration>,
  ): asserts this is NodePath_Final<t.TSEnumDeclaration>;
  assertTSEnumMember(
    opts?: Opts$2<t.TSEnumMember>,
  ): asserts this is NodePath_Final<t.TSEnumMember>;
  assertTSExportAssignment(
    opts?: Opts$2<t.TSExportAssignment>,
  ): asserts this is NodePath_Final<t.TSExportAssignment>;
  assertTSExpressionWithTypeArguments(
    opts?: Opts$2<t.TSExpressionWithTypeArguments>,
  ): asserts this is NodePath_Final<t.TSExpressionWithTypeArguments>;
  assertTSExternalModuleReference(
    opts?: Opts$2<t.TSExternalModuleReference>,
  ): asserts this is NodePath_Final<t.TSExternalModuleReference>;
  assertTSFunctionType(
    opts?: Opts$2<t.TSFunctionType>,
  ): asserts this is NodePath_Final<t.TSFunctionType>;
  assertTSImportEqualsDeclaration(
    opts?: Opts$2<t.TSImportEqualsDeclaration>,
  ): asserts this is NodePath_Final<t.TSImportEqualsDeclaration>;
  assertTSImportType(
    opts?: Opts$2<t.TSImportType>,
  ): asserts this is NodePath_Final<t.TSImportType>;
  assertTSIndexSignature(
    opts?: Opts$2<t.TSIndexSignature>,
  ): asserts this is NodePath_Final<t.TSIndexSignature>;
  assertTSIndexedAccessType(
    opts?: Opts$2<t.TSIndexedAccessType>,
  ): asserts this is NodePath_Final<t.TSIndexedAccessType>;
  assertTSInferType(
    opts?: Opts$2<t.TSInferType>,
  ): asserts this is NodePath_Final<t.TSInferType>;
  assertTSInstantiationExpression(
    opts?: Opts$2<t.TSInstantiationExpression>,
  ): asserts this is NodePath_Final<t.TSInstantiationExpression>;
  assertTSInterfaceBody(
    opts?: Opts$2<t.TSInterfaceBody>,
  ): asserts this is NodePath_Final<t.TSInterfaceBody>;
  assertTSInterfaceDeclaration(
    opts?: Opts$2<t.TSInterfaceDeclaration>,
  ): asserts this is NodePath_Final<t.TSInterfaceDeclaration>;
  assertTSIntersectionType(
    opts?: Opts$2<t.TSIntersectionType>,
  ): asserts this is NodePath_Final<t.TSIntersectionType>;
  assertTSIntrinsicKeyword(
    opts?: Opts$2<t.TSIntrinsicKeyword>,
  ): asserts this is NodePath_Final<t.TSIntrinsicKeyword>;
  assertTSLiteralType(
    opts?: Opts$2<t.TSLiteralType>,
  ): asserts this is NodePath_Final<t.TSLiteralType>;
  assertTSMappedType(
    opts?: Opts$2<t.TSMappedType>,
  ): asserts this is NodePath_Final<t.TSMappedType>;
  assertTSMethodSignature(
    opts?: Opts$2<t.TSMethodSignature>,
  ): asserts this is NodePath_Final<t.TSMethodSignature>;
  assertTSModuleBlock(
    opts?: Opts$2<t.TSModuleBlock>,
  ): asserts this is NodePath_Final<t.TSModuleBlock>;
  assertTSModuleDeclaration(
    opts?: Opts$2<t.TSModuleDeclaration>,
  ): asserts this is NodePath_Final<t.TSModuleDeclaration>;
  assertTSNamedTupleMember(
    opts?: Opts$2<t.TSNamedTupleMember>,
  ): asserts this is NodePath_Final<t.TSNamedTupleMember>;
  assertTSNamespaceExportDeclaration(
    opts?: Opts$2<t.TSNamespaceExportDeclaration>,
  ): asserts this is NodePath_Final<t.TSNamespaceExportDeclaration>;
  assertTSNeverKeyword(
    opts?: Opts$2<t.TSNeverKeyword>,
  ): asserts this is NodePath_Final<t.TSNeverKeyword>;
  assertTSNonNullExpression(
    opts?: Opts$2<t.TSNonNullExpression>,
  ): asserts this is NodePath_Final<t.TSNonNullExpression>;
  assertTSNullKeyword(
    opts?: Opts$2<t.TSNullKeyword>,
  ): asserts this is NodePath_Final<t.TSNullKeyword>;
  assertTSNumberKeyword(
    opts?: Opts$2<t.TSNumberKeyword>,
  ): asserts this is NodePath_Final<t.TSNumberKeyword>;
  assertTSObjectKeyword(
    opts?: Opts$2<t.TSObjectKeyword>,
  ): asserts this is NodePath_Final<t.TSObjectKeyword>;
  assertTSOptionalType(
    opts?: Opts$2<t.TSOptionalType>,
  ): asserts this is NodePath_Final<t.TSOptionalType>;
  assertTSParameterProperty(
    opts?: Opts$2<t.TSParameterProperty>,
  ): asserts this is NodePath_Final<t.TSParameterProperty>;
  assertTSParenthesizedType(
    opts?: Opts$2<t.TSParenthesizedType>,
  ): asserts this is NodePath_Final<t.TSParenthesizedType>;
  assertTSPropertySignature(
    opts?: Opts$2<t.TSPropertySignature>,
  ): asserts this is NodePath_Final<t.TSPropertySignature>;
  assertTSQualifiedName(
    opts?: Opts$2<t.TSQualifiedName>,
  ): asserts this is NodePath_Final<t.TSQualifiedName>;
  assertTSRestType(
    opts?: Opts$2<t.TSRestType>,
  ): asserts this is NodePath_Final<t.TSRestType>;
  assertTSSatisfiesExpression(
    opts?: Opts$2<t.TSSatisfiesExpression>,
  ): asserts this is NodePath_Final<t.TSSatisfiesExpression>;
  assertTSStringKeyword(
    opts?: Opts$2<t.TSStringKeyword>,
  ): asserts this is NodePath_Final<t.TSStringKeyword>;
  assertTSSymbolKeyword(
    opts?: Opts$2<t.TSSymbolKeyword>,
  ): asserts this is NodePath_Final<t.TSSymbolKeyword>;
  assertTSThisType(
    opts?: Opts$2<t.TSThisType>,
  ): asserts this is NodePath_Final<t.TSThisType>;
  assertTSTupleType(
    opts?: Opts$2<t.TSTupleType>,
  ): asserts this is NodePath_Final<t.TSTupleType>;
  assertTSType(opts?: Opts$2<t.TSType>): asserts this is NodePath_Final<t.TSType>;
  assertTSTypeAliasDeclaration(
    opts?: Opts$2<t.TSTypeAliasDeclaration>,
  ): asserts this is NodePath_Final<t.TSTypeAliasDeclaration>;
  assertTSTypeAnnotation(
    opts?: Opts$2<t.TSTypeAnnotation>,
  ): asserts this is NodePath_Final<t.TSTypeAnnotation>;
  assertTSTypeAssertion(
    opts?: Opts$2<t.TSTypeAssertion>,
  ): asserts this is NodePath_Final<t.TSTypeAssertion>;
  assertTSTypeElement(
    opts?: Opts$2<t.TSTypeElement>,
  ): asserts this is NodePath_Final<t.TSTypeElement>;
  assertTSTypeLiteral(
    opts?: Opts$2<t.TSTypeLiteral>,
  ): asserts this is NodePath_Final<t.TSTypeLiteral>;
  assertTSTypeOperator(
    opts?: Opts$2<t.TSTypeOperator>,
  ): asserts this is NodePath_Final<t.TSTypeOperator>;
  assertTSTypeParameter(
    opts?: Opts$2<t.TSTypeParameter>,
  ): asserts this is NodePath_Final<t.TSTypeParameter>;
  assertTSTypeParameterDeclaration(
    opts?: Opts$2<t.TSTypeParameterDeclaration>,
  ): asserts this is NodePath_Final<t.TSTypeParameterDeclaration>;
  assertTSTypeParameterInstantiation(
    opts?: Opts$2<t.TSTypeParameterInstantiation>,
  ): asserts this is NodePath_Final<t.TSTypeParameterInstantiation>;
  assertTSTypePredicate(
    opts?: Opts$2<t.TSTypePredicate>,
  ): asserts this is NodePath_Final<t.TSTypePredicate>;
  assertTSTypeQuery(
    opts?: Opts$2<t.TSTypeQuery>,
  ): asserts this is NodePath_Final<t.TSTypeQuery>;
  assertTSTypeReference(
    opts?: Opts$2<t.TSTypeReference>,
  ): asserts this is NodePath_Final<t.TSTypeReference>;
  assertTSUndefinedKeyword(
    opts?: Opts$2<t.TSUndefinedKeyword>,
  ): asserts this is NodePath_Final<t.TSUndefinedKeyword>;
  assertTSUnionType(
    opts?: Opts$2<t.TSUnionType>,
  ): asserts this is NodePath_Final<t.TSUnionType>;
  assertTSUnknownKeyword(
    opts?: Opts$2<t.TSUnknownKeyword>,
  ): asserts this is NodePath_Final<t.TSUnknownKeyword>;
  assertTSVoidKeyword(
    opts?: Opts$2<t.TSVoidKeyword>,
  ): asserts this is NodePath_Final<t.TSVoidKeyword>;
  assertTaggedTemplateExpression(
    opts?: Opts$2<t.TaggedTemplateExpression>,
  ): asserts this is NodePath_Final<t.TaggedTemplateExpression>;
  assertTemplateElement(
    opts?: Opts$2<t.TemplateElement>,
  ): asserts this is NodePath_Final<t.TemplateElement>;
  assertTemplateLiteral(
    opts?: Opts$2<t.TemplateLiteral>,
  ): asserts this is NodePath_Final<t.TemplateLiteral>;
  assertTerminatorless(
    opts?: Opts$2<t.Terminatorless>,
  ): asserts this is NodePath_Final<t.Terminatorless>;
  assertThisExpression(
    opts?: Opts$2<t.ThisExpression>,
  ): asserts this is NodePath_Final<t.ThisExpression>;
  assertThisTypeAnnotation(
    opts?: Opts$2<t.ThisTypeAnnotation>,
  ): asserts this is NodePath_Final<t.ThisTypeAnnotation>;
  assertThrowStatement(
    opts?: Opts$2<t.ThrowStatement>,
  ): asserts this is NodePath_Final<t.ThrowStatement>;
  assertTopicReference(
    opts?: Opts$2<t.TopicReference>,
  ): asserts this is NodePath_Final<t.TopicReference>;
  assertTryStatement(
    opts?: Opts$2<t.TryStatement>,
  ): asserts this is NodePath_Final<t.TryStatement>;
  assertTupleExpression(
    opts?: Opts$2<t.TupleExpression>,
  ): asserts this is NodePath_Final<t.TupleExpression>;
  assertTupleTypeAnnotation(
    opts?: Opts$2<t.TupleTypeAnnotation>,
  ): asserts this is NodePath_Final<t.TupleTypeAnnotation>;
  assertTypeAlias(
    opts?: Opts$2<t.TypeAlias>,
  ): asserts this is NodePath_Final<t.TypeAlias>;
  assertTypeAnnotation(
    opts?: Opts$2<t.TypeAnnotation>,
  ): asserts this is NodePath_Final<t.TypeAnnotation>;
  assertTypeCastExpression(
    opts?: Opts$2<t.TypeCastExpression>,
  ): asserts this is NodePath_Final<t.TypeCastExpression>;
  assertTypeParameter(
    opts?: Opts$2<t.TypeParameter>,
  ): asserts this is NodePath_Final<t.TypeParameter>;
  assertTypeParameterDeclaration(
    opts?: Opts$2<t.TypeParameterDeclaration>,
  ): asserts this is NodePath_Final<t.TypeParameterDeclaration>;
  assertTypeParameterInstantiation(
    opts?: Opts$2<t.TypeParameterInstantiation>,
  ): asserts this is NodePath_Final<t.TypeParameterInstantiation>;
  assertTypeScript(
    opts?: Opts$2<t.TypeScript>,
  ): asserts this is NodePath_Final<t.TypeScript>;
  assertTypeofTypeAnnotation(
    opts?: Opts$2<t.TypeofTypeAnnotation>,
  ): asserts this is NodePath_Final<t.TypeofTypeAnnotation>;
  assertUnaryExpression(
    opts?: Opts$2<t.UnaryExpression>,
  ): asserts this is NodePath_Final<t.UnaryExpression>;
  assertUnaryLike(
    opts?: Opts$2<t.UnaryLike>,
  ): asserts this is NodePath_Final<t.UnaryLike>;
  assertUnionTypeAnnotation(
    opts?: Opts$2<t.UnionTypeAnnotation>,
  ): asserts this is NodePath_Final<t.UnionTypeAnnotation>;
  assertUpdateExpression(
    opts?: Opts$2<t.UpdateExpression>,
  ): asserts this is NodePath_Final<t.UpdateExpression>;
  assertUserWhitespacable(
    opts?: Opts$2<t.UserWhitespacable>,
  ): asserts this is NodePath_Final<t.UserWhitespacable>;
  assertV8IntrinsicIdentifier(
    opts?: Opts$2<t.V8IntrinsicIdentifier>,
  ): asserts this is NodePath_Final<t.V8IntrinsicIdentifier>;
  assertVariableDeclaration(
    opts?: Opts$2<t.VariableDeclaration>,
  ): asserts this is NodePath_Final<t.VariableDeclaration>;
  assertVariableDeclarator(
    opts?: Opts$2<t.VariableDeclarator>,
  ): asserts this is NodePath_Final<t.VariableDeclarator>;
  assertVariance(opts?: Opts$2<t.Variance>): asserts this is NodePath_Final<t.Variance>;
  assertVoidTypeAnnotation(
    opts?: Opts$2<t.VoidTypeAnnotation>,
  ): asserts this is NodePath_Final<t.VoidTypeAnnotation>;
  assertWhile(opts?: Opts$2<t.While>): asserts this is NodePath_Final<t.While>;
  assertWhileStatement(
    opts?: Opts$2<t.WhileStatement>,
  ): asserts this is NodePath_Final<t.WhileStatement>;
  assertWithStatement(
    opts?: Opts$2<t.WithStatement>,
  ): asserts this is NodePath_Final<t.WithStatement>;
  assertYieldExpression(
    opts?: Opts$2<t.YieldExpression>,
  ): asserts this is NodePath_Final<t.YieldExpression>;
}

type Opts$1<Obj> = Partial<{
    [Prop in keyof Obj]: Obj[Prop] extends t.Node ? t.Node : Obj[Prop] extends t.Node[] ? t.Node[] : Obj[Prop];
}>;
interface VirtualTypeNodePathValidators {
    isBindingIdentifier(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["BindingIdentifier"]>): this is NodePath_Final<VirtualTypeAliases["BindingIdentifier"]>;
    isBlockScoped(opts?: Opts$1<VirtualTypeAliases["BlockScoped"]>): boolean;
    /**
     * @deprecated
     */
    isExistentialTypeParam(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["ExistentialTypeParam"]>): this is NodePath_Final<VirtualTypeAliases["ExistentialTypeParam"]>;
    isExpression(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["Expression"]>): this is NodePath_Final<t.Expression>;
    isFlow(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["Flow"]>): this is NodePath_Final<t.Flow>;
    isForAwaitStatement(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["ForAwaitStatement"]>): this is NodePath_Final<VirtualTypeAliases["ForAwaitStatement"]>;
    isGenerated(opts?: VirtualTypeAliases["Generated"]): boolean;
    /**
     * @deprecated
     */
    isNumericLiteralTypeAnnotation(opts?: VirtualTypeAliases["NumericLiteralTypeAnnotation"]): void;
    isPure(opts?: VirtualTypeAliases["Pure"]): boolean;
    isReferenced(opts?: VirtualTypeAliases["Referenced"]): boolean;
    isReferencedIdentifier(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["ReferencedIdentifier"]>): this is NodePath_Final<VirtualTypeAliases["ReferencedIdentifier"]>;
    isReferencedMemberExpression(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["ReferencedMemberExpression"]>): this is NodePath_Final<VirtualTypeAliases["ReferencedMemberExpression"]>;
    isRestProperty(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["RestProperty"]>): this is NodePath_Final<t.RestProperty>;
    isScope(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["Scope"]>): this is NodePath_Final<VirtualTypeAliases["Scope"]>;
    isSpreadProperty(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["SpreadProperty"]>): this is NodePath_Final<t.SpreadProperty>;
    isStatement(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["Statement"]>): this is NodePath_Final<t.Statement>;
    isUser(opts?: VirtualTypeAliases["User"]): boolean;
    isVar(this: NodePath_Final, opts?: Opts$1<VirtualTypeAliases["Var"]>): this is NodePath_Final<VirtualTypeAliases["Var"]>;
}

/*
 * This file is auto-generated! Do not modify it directly.
 * To re-generate run 'make build'
 */


type Opts<Obj> = Partial<{
  [Prop in keyof Obj]: Obj[Prop] extends t.Node
    ? t.Node
    : Obj[Prop] extends t.Node[]
      ? t.Node[]
      : Obj[Prop];
}>;

interface BaseNodePathValidators {
  isAccessor(
    this: NodePath_Final,
    opts?: Opts<t.Accessor>,
  ): this is NodePath_Final<t.Accessor>;
  isAnyTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.AnyTypeAnnotation>,
  ): this is NodePath_Final<t.AnyTypeAnnotation>;
  isArgumentPlaceholder(
    this: NodePath_Final,
    opts?: Opts<t.ArgumentPlaceholder>,
  ): this is NodePath_Final<t.ArgumentPlaceholder>;
  isArrayExpression(
    this: NodePath_Final,
    opts?: Opts<t.ArrayExpression>,
  ): this is NodePath_Final<t.ArrayExpression>;
  isArrayPattern(
    this: NodePath_Final,
    opts?: Opts<t.ArrayPattern>,
  ): this is NodePath_Final<t.ArrayPattern>;
  isArrayTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.ArrayTypeAnnotation>,
  ): this is NodePath_Final<t.ArrayTypeAnnotation>;
  isArrowFunctionExpression(
    this: NodePath_Final,
    opts?: Opts<t.ArrowFunctionExpression>,
  ): this is NodePath_Final<t.ArrowFunctionExpression>;
  isAssignmentExpression(
    this: NodePath_Final,
    opts?: Opts<t.AssignmentExpression>,
  ): this is NodePath_Final<t.AssignmentExpression>;
  isAssignmentPattern(
    this: NodePath_Final,
    opts?: Opts<t.AssignmentPattern>,
  ): this is NodePath_Final<t.AssignmentPattern>;
  isAwaitExpression(
    this: NodePath_Final,
    opts?: Opts<t.AwaitExpression>,
  ): this is NodePath_Final<t.AwaitExpression>;
  isBigIntLiteral(
    this: NodePath_Final,
    opts?: Opts<t.BigIntLiteral>,
  ): this is NodePath_Final<t.BigIntLiteral>;
  isBinary(this: NodePath_Final, opts?: Opts<t.Binary>): this is NodePath_Final<t.Binary>;
  isBinaryExpression(
    this: NodePath_Final,
    opts?: Opts<t.BinaryExpression>,
  ): this is NodePath_Final<t.BinaryExpression>;
  isBindExpression(
    this: NodePath_Final,
    opts?: Opts<t.BindExpression>,
  ): this is NodePath_Final<t.BindExpression>;
  isBlock(this: NodePath_Final, opts?: Opts<t.Block>): this is NodePath_Final<t.Block>;
  isBlockParent(
    this: NodePath_Final,
    opts?: Opts<t.BlockParent>,
  ): this is NodePath_Final<t.BlockParent>;
  isBlockStatement(
    this: NodePath_Final,
    opts?: Opts<t.BlockStatement>,
  ): this is NodePath_Final<t.BlockStatement>;
  isBooleanLiteral(
    this: NodePath_Final,
    opts?: Opts<t.BooleanLiteral>,
  ): this is NodePath_Final<t.BooleanLiteral>;
  isBooleanLiteralTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.BooleanLiteralTypeAnnotation>,
  ): this is NodePath_Final<t.BooleanLiteralTypeAnnotation>;
  isBooleanTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.BooleanTypeAnnotation>,
  ): this is NodePath_Final<t.BooleanTypeAnnotation>;
  isBreakStatement(
    this: NodePath_Final,
    opts?: Opts<t.BreakStatement>,
  ): this is NodePath_Final<t.BreakStatement>;
  isCallExpression(
    this: NodePath_Final,
    opts?: Opts<t.CallExpression>,
  ): this is NodePath_Final<t.CallExpression>;
  isCatchClause(
    this: NodePath_Final,
    opts?: Opts<t.CatchClause>,
  ): this is NodePath_Final<t.CatchClause>;
  isClass(this: NodePath_Final, opts?: Opts<t.Class>): this is NodePath_Final<t.Class>;
  isClassAccessorProperty(
    this: NodePath_Final,
    opts?: Opts<t.ClassAccessorProperty>,
  ): this is NodePath_Final<t.ClassAccessorProperty>;
  isClassBody(
    this: NodePath_Final,
    opts?: Opts<t.ClassBody>,
  ): this is NodePath_Final<t.ClassBody>;
  isClassDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ClassDeclaration>,
  ): this is NodePath_Final<t.ClassDeclaration>;
  isClassExpression(
    this: NodePath_Final,
    opts?: Opts<t.ClassExpression>,
  ): this is NodePath_Final<t.ClassExpression>;
  isClassImplements(
    this: NodePath_Final,
    opts?: Opts<t.ClassImplements>,
  ): this is NodePath_Final<t.ClassImplements>;
  isClassMethod(
    this: NodePath_Final,
    opts?: Opts<t.ClassMethod>,
  ): this is NodePath_Final<t.ClassMethod>;
  isClassPrivateMethod(
    this: NodePath_Final,
    opts?: Opts<t.ClassPrivateMethod>,
  ): this is NodePath_Final<t.ClassPrivateMethod>;
  isClassPrivateProperty(
    this: NodePath_Final,
    opts?: Opts<t.ClassPrivateProperty>,
  ): this is NodePath_Final<t.ClassPrivateProperty>;
  isClassProperty(
    this: NodePath_Final,
    opts?: Opts<t.ClassProperty>,
  ): this is NodePath_Final<t.ClassProperty>;
  isCompletionStatement(
    this: NodePath_Final,
    opts?: Opts<t.CompletionStatement>,
  ): this is NodePath_Final<t.CompletionStatement>;
  isConditional(
    this: NodePath_Final,
    opts?: Opts<t.Conditional>,
  ): this is NodePath_Final<t.Conditional>;
  isConditionalExpression(
    this: NodePath_Final,
    opts?: Opts<t.ConditionalExpression>,
  ): this is NodePath_Final<t.ConditionalExpression>;
  isContinueStatement(
    this: NodePath_Final,
    opts?: Opts<t.ContinueStatement>,
  ): this is NodePath_Final<t.ContinueStatement>;
  isDebuggerStatement(
    this: NodePath_Final,
    opts?: Opts<t.DebuggerStatement>,
  ): this is NodePath_Final<t.DebuggerStatement>;
  isDecimalLiteral(
    this: NodePath_Final,
    opts?: Opts<t.DecimalLiteral>,
  ): this is NodePath_Final<t.DecimalLiteral>;
  isDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.Declaration>,
  ): this is NodePath_Final<t.Declaration>;
  isDeclareClass(
    this: NodePath_Final,
    opts?: Opts<t.DeclareClass>,
  ): this is NodePath_Final<t.DeclareClass>;
  isDeclareExportAllDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.DeclareExportAllDeclaration>,
  ): this is NodePath_Final<t.DeclareExportAllDeclaration>;
  isDeclareExportDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.DeclareExportDeclaration>,
  ): this is NodePath_Final<t.DeclareExportDeclaration>;
  isDeclareFunction(
    this: NodePath_Final,
    opts?: Opts<t.DeclareFunction>,
  ): this is NodePath_Final<t.DeclareFunction>;
  isDeclareInterface(
    this: NodePath_Final,
    opts?: Opts<t.DeclareInterface>,
  ): this is NodePath_Final<t.DeclareInterface>;
  isDeclareModule(
    this: NodePath_Final,
    opts?: Opts<t.DeclareModule>,
  ): this is NodePath_Final<t.DeclareModule>;
  isDeclareModuleExports(
    this: NodePath_Final,
    opts?: Opts<t.DeclareModuleExports>,
  ): this is NodePath_Final<t.DeclareModuleExports>;
  isDeclareOpaqueType(
    this: NodePath_Final,
    opts?: Opts<t.DeclareOpaqueType>,
  ): this is NodePath_Final<t.DeclareOpaqueType>;
  isDeclareTypeAlias(
    this: NodePath_Final,
    opts?: Opts<t.DeclareTypeAlias>,
  ): this is NodePath_Final<t.DeclareTypeAlias>;
  isDeclareVariable(
    this: NodePath_Final,
    opts?: Opts<t.DeclareVariable>,
  ): this is NodePath_Final<t.DeclareVariable>;
  isDeclaredPredicate(
    this: NodePath_Final,
    opts?: Opts<t.DeclaredPredicate>,
  ): this is NodePath_Final<t.DeclaredPredicate>;
  isDecorator(
    this: NodePath_Final,
    opts?: Opts<t.Decorator>,
  ): this is NodePath_Final<t.Decorator>;
  isDirective(
    this: NodePath_Final,
    opts?: Opts<t.Directive>,
  ): this is NodePath_Final<t.Directive>;
  isDirectiveLiteral(
    this: NodePath_Final,
    opts?: Opts<t.DirectiveLiteral>,
  ): this is NodePath_Final<t.DirectiveLiteral>;
  isDoExpression(
    this: NodePath_Final,
    opts?: Opts<t.DoExpression>,
  ): this is NodePath_Final<t.DoExpression>;
  isDoWhileStatement(
    this: NodePath_Final,
    opts?: Opts<t.DoWhileStatement>,
  ): this is NodePath_Final<t.DoWhileStatement>;
  isEmptyStatement(
    this: NodePath_Final,
    opts?: Opts<t.EmptyStatement>,
  ): this is NodePath_Final<t.EmptyStatement>;
  isEmptyTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.EmptyTypeAnnotation>,
  ): this is NodePath_Final<t.EmptyTypeAnnotation>;
  isEnumBody(
    this: NodePath_Final,
    opts?: Opts<t.EnumBody>,
  ): this is NodePath_Final<t.EnumBody>;
  isEnumBooleanBody(
    this: NodePath_Final,
    opts?: Opts<t.EnumBooleanBody>,
  ): this is NodePath_Final<t.EnumBooleanBody>;
  isEnumBooleanMember(
    this: NodePath_Final,
    opts?: Opts<t.EnumBooleanMember>,
  ): this is NodePath_Final<t.EnumBooleanMember>;
  isEnumDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.EnumDeclaration>,
  ): this is NodePath_Final<t.EnumDeclaration>;
  isEnumDefaultedMember(
    this: NodePath_Final,
    opts?: Opts<t.EnumDefaultedMember>,
  ): this is NodePath_Final<t.EnumDefaultedMember>;
  isEnumMember(
    this: NodePath_Final,
    opts?: Opts<t.EnumMember>,
  ): this is NodePath_Final<t.EnumMember>;
  isEnumNumberBody(
    this: NodePath_Final,
    opts?: Opts<t.EnumNumberBody>,
  ): this is NodePath_Final<t.EnumNumberBody>;
  isEnumNumberMember(
    this: NodePath_Final,
    opts?: Opts<t.EnumNumberMember>,
  ): this is NodePath_Final<t.EnumNumberMember>;
  isEnumStringBody(
    this: NodePath_Final,
    opts?: Opts<t.EnumStringBody>,
  ): this is NodePath_Final<t.EnumStringBody>;
  isEnumStringMember(
    this: NodePath_Final,
    opts?: Opts<t.EnumStringMember>,
  ): this is NodePath_Final<t.EnumStringMember>;
  isEnumSymbolBody(
    this: NodePath_Final,
    opts?: Opts<t.EnumSymbolBody>,
  ): this is NodePath_Final<t.EnumSymbolBody>;
  isExistsTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.ExistsTypeAnnotation>,
  ): this is NodePath_Final<t.ExistsTypeAnnotation>;
  isExportAllDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ExportAllDeclaration>,
  ): this is NodePath_Final<t.ExportAllDeclaration>;
  isExportDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ExportDeclaration>,
  ): this is NodePath_Final<t.ExportDeclaration>;
  isExportDefaultDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ExportDefaultDeclaration>,
  ): this is NodePath_Final<t.ExportDefaultDeclaration>;
  isExportDefaultSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ExportDefaultSpecifier>,
  ): this is NodePath_Final<t.ExportDefaultSpecifier>;
  isExportNamedDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ExportNamedDeclaration>,
  ): this is NodePath_Final<t.ExportNamedDeclaration>;
  isExportNamespaceSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ExportNamespaceSpecifier>,
  ): this is NodePath_Final<t.ExportNamespaceSpecifier>;
  isExportSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ExportSpecifier>,
  ): this is NodePath_Final<t.ExportSpecifier>;
  isExpression(
    this: NodePath_Final,
    opts?: Opts<t.Expression>,
  ): this is NodePath_Final<t.Expression>;
  isExpressionStatement(
    this: NodePath_Final,
    opts?: Opts<t.ExpressionStatement>,
  ): this is NodePath_Final<t.ExpressionStatement>;
  isExpressionWrapper(
    this: NodePath_Final,
    opts?: Opts<t.ExpressionWrapper>,
  ): this is NodePath_Final<t.ExpressionWrapper>;
  isFile(this: NodePath_Final, opts?: Opts<t.File>): this is NodePath_Final<t.File>;
  isFlow(this: NodePath_Final, opts?: Opts<t.Flow>): this is NodePath_Final<t.Flow>;
  isFlowBaseAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.FlowBaseAnnotation>,
  ): this is NodePath_Final<t.FlowBaseAnnotation>;
  isFlowDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.FlowDeclaration>,
  ): this is NodePath_Final<t.FlowDeclaration>;
  isFlowPredicate(
    this: NodePath_Final,
    opts?: Opts<t.FlowPredicate>,
  ): this is NodePath_Final<t.FlowPredicate>;
  isFlowType(
    this: NodePath_Final,
    opts?: Opts<t.FlowType>,
  ): this is NodePath_Final<t.FlowType>;
  isFor(this: NodePath_Final, opts?: Opts<t.For>): this is NodePath_Final<t.For>;
  isForInStatement(
    this: NodePath_Final,
    opts?: Opts<t.ForInStatement>,
  ): this is NodePath_Final<t.ForInStatement>;
  isForOfStatement(
    this: NodePath_Final,
    opts?: Opts<t.ForOfStatement>,
  ): this is NodePath_Final<t.ForOfStatement>;
  isForStatement(
    this: NodePath_Final,
    opts?: Opts<t.ForStatement>,
  ): this is NodePath_Final<t.ForStatement>;
  isForXStatement(
    this: NodePath_Final,
    opts?: Opts<t.ForXStatement>,
  ): this is NodePath_Final<t.ForXStatement>;
  isFunction(
    this: NodePath_Final,
    opts?: Opts<t.Function>,
  ): this is NodePath_Final<t.Function>;
  isFunctionDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.FunctionDeclaration>,
  ): this is NodePath_Final<t.FunctionDeclaration>;
  isFunctionExpression(
    this: NodePath_Final,
    opts?: Opts<t.FunctionExpression>,
  ): this is NodePath_Final<t.FunctionExpression>;
  isFunctionParent(
    this: NodePath_Final,
    opts?: Opts<t.FunctionParent>,
  ): this is NodePath_Final<t.FunctionParent>;
  isFunctionTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.FunctionTypeAnnotation>,
  ): this is NodePath_Final<t.FunctionTypeAnnotation>;
  isFunctionTypeParam(
    this: NodePath_Final,
    opts?: Opts<t.FunctionTypeParam>,
  ): this is NodePath_Final<t.FunctionTypeParam>;
  isGenericTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.GenericTypeAnnotation>,
  ): this is NodePath_Final<t.GenericTypeAnnotation>;
  isIdentifier(
    this: NodePath_Final,
    opts?: Opts<t.Identifier>,
  ): this is NodePath_Final<t.Identifier>;
  isIfStatement(
    this: NodePath_Final,
    opts?: Opts<t.IfStatement>,
  ): this is NodePath_Final<t.IfStatement>;
  isImmutable(
    this: NodePath_Final,
    opts?: Opts<t.Immutable>,
  ): this is NodePath_Final<t.Immutable>;
  isImport(this: NodePath_Final, opts?: Opts<t.Import>): this is NodePath_Final<t.Import>;
  isImportAttribute(
    this: NodePath_Final,
    opts?: Opts<t.ImportAttribute>,
  ): this is NodePath_Final<t.ImportAttribute>;
  isImportDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ImportDeclaration>,
  ): this is NodePath_Final<t.ImportDeclaration>;
  isImportDefaultSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ImportDefaultSpecifier>,
  ): this is NodePath_Final<t.ImportDefaultSpecifier>;
  isImportExpression(
    this: NodePath_Final,
    opts?: Opts<t.ImportExpression>,
  ): this is NodePath_Final<t.ImportExpression>;
  isImportNamespaceSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ImportNamespaceSpecifier>,
  ): this is NodePath_Final<t.ImportNamespaceSpecifier>;
  isImportOrExportDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ImportOrExportDeclaration>,
  ): this is NodePath_Final<t.ImportOrExportDeclaration>;
  isImportSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ImportSpecifier>,
  ): this is NodePath_Final<t.ImportSpecifier>;
  isIndexedAccessType(
    this: NodePath_Final,
    opts?: Opts<t.IndexedAccessType>,
  ): this is NodePath_Final<t.IndexedAccessType>;
  isInferredPredicate(
    this: NodePath_Final,
    opts?: Opts<t.InferredPredicate>,
  ): this is NodePath_Final<t.InferredPredicate>;
  isInterfaceDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.InterfaceDeclaration>,
  ): this is NodePath_Final<t.InterfaceDeclaration>;
  isInterfaceExtends(
    this: NodePath_Final,
    opts?: Opts<t.InterfaceExtends>,
  ): this is NodePath_Final<t.InterfaceExtends>;
  isInterfaceTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.InterfaceTypeAnnotation>,
  ): this is NodePath_Final<t.InterfaceTypeAnnotation>;
  isInterpreterDirective(
    this: NodePath_Final,
    opts?: Opts<t.InterpreterDirective>,
  ): this is NodePath_Final<t.InterpreterDirective>;
  isIntersectionTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.IntersectionTypeAnnotation>,
  ): this is NodePath_Final<t.IntersectionTypeAnnotation>;
  isJSX(this: NodePath_Final, opts?: Opts<t.JSX>): this is NodePath_Final<t.JSX>;
  isJSXAttribute(
    this: NodePath_Final,
    opts?: Opts<t.JSXAttribute>,
  ): this is NodePath_Final<t.JSXAttribute>;
  isJSXClosingElement(
    this: NodePath_Final,
    opts?: Opts<t.JSXClosingElement>,
  ): this is NodePath_Final<t.JSXClosingElement>;
  isJSXClosingFragment(
    this: NodePath_Final,
    opts?: Opts<t.JSXClosingFragment>,
  ): this is NodePath_Final<t.JSXClosingFragment>;
  isJSXElement(
    this: NodePath_Final,
    opts?: Opts<t.JSXElement>,
  ): this is NodePath_Final<t.JSXElement>;
  isJSXEmptyExpression(
    this: NodePath_Final,
    opts?: Opts<t.JSXEmptyExpression>,
  ): this is NodePath_Final<t.JSXEmptyExpression>;
  isJSXExpressionContainer(
    this: NodePath_Final,
    opts?: Opts<t.JSXExpressionContainer>,
  ): this is NodePath_Final<t.JSXExpressionContainer>;
  isJSXFragment(
    this: NodePath_Final,
    opts?: Opts<t.JSXFragment>,
  ): this is NodePath_Final<t.JSXFragment>;
  isJSXIdentifier(
    this: NodePath_Final,
    opts?: Opts<t.JSXIdentifier>,
  ): this is NodePath_Final<t.JSXIdentifier>;
  isJSXMemberExpression(
    this: NodePath_Final,
    opts?: Opts<t.JSXMemberExpression>,
  ): this is NodePath_Final<t.JSXMemberExpression>;
  isJSXNamespacedName(
    this: NodePath_Final,
    opts?: Opts<t.JSXNamespacedName>,
  ): this is NodePath_Final<t.JSXNamespacedName>;
  isJSXOpeningElement(
    this: NodePath_Final,
    opts?: Opts<t.JSXOpeningElement>,
  ): this is NodePath_Final<t.JSXOpeningElement>;
  isJSXOpeningFragment(
    this: NodePath_Final,
    opts?: Opts<t.JSXOpeningFragment>,
  ): this is NodePath_Final<t.JSXOpeningFragment>;
  isJSXSpreadAttribute(
    this: NodePath_Final,
    opts?: Opts<t.JSXSpreadAttribute>,
  ): this is NodePath_Final<t.JSXSpreadAttribute>;
  isJSXSpreadChild(
    this: NodePath_Final,
    opts?: Opts<t.JSXSpreadChild>,
  ): this is NodePath_Final<t.JSXSpreadChild>;
  isJSXText(
    this: NodePath_Final,
    opts?: Opts<t.JSXText>,
  ): this is NodePath_Final<t.JSXText>;
  isLVal(this: NodePath_Final, opts?: Opts<t.LVal>): this is NodePath_Final<t.LVal>;
  isLabeledStatement(
    this: NodePath_Final,
    opts?: Opts<t.LabeledStatement>,
  ): this is NodePath_Final<t.LabeledStatement>;
  isLiteral(
    this: NodePath_Final,
    opts?: Opts<t.Literal>,
  ): this is NodePath_Final<t.Literal>;
  isLogicalExpression(
    this: NodePath_Final,
    opts?: Opts<t.LogicalExpression>,
  ): this is NodePath_Final<t.LogicalExpression>;
  isLoop(this: NodePath_Final, opts?: Opts<t.Loop>): this is NodePath_Final<t.Loop>;
  isMemberExpression(
    this: NodePath_Final,
    opts?: Opts<t.MemberExpression>,
  ): this is NodePath_Final<t.MemberExpression>;
  isMetaProperty(
    this: NodePath_Final,
    opts?: Opts<t.MetaProperty>,
  ): this is NodePath_Final<t.MetaProperty>;
  isMethod(this: NodePath_Final, opts?: Opts<t.Method>): this is NodePath_Final<t.Method>;
  isMiscellaneous(
    this: NodePath_Final,
    opts?: Opts<t.Miscellaneous>,
  ): this is NodePath_Final<t.Miscellaneous>;
  isMixedTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.MixedTypeAnnotation>,
  ): this is NodePath_Final<t.MixedTypeAnnotation>;
  isModuleDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.ModuleDeclaration>,
  ): this is NodePath_Final<t.ModuleDeclaration>;
  isModuleExpression(
    this: NodePath_Final,
    opts?: Opts<t.ModuleExpression>,
  ): this is NodePath_Final<t.ModuleExpression>;
  isModuleSpecifier(
    this: NodePath_Final,
    opts?: Opts<t.ModuleSpecifier>,
  ): this is NodePath_Final<t.ModuleSpecifier>;
  isNewExpression(
    this: NodePath_Final,
    opts?: Opts<t.NewExpression>,
  ): this is NodePath_Final<t.NewExpression>;
  isNullLiteral(
    this: NodePath_Final,
    opts?: Opts<t.NullLiteral>,
  ): this is NodePath_Final<t.NullLiteral>;
  isNullLiteralTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.NullLiteralTypeAnnotation>,
  ): this is NodePath_Final<t.NullLiteralTypeAnnotation>;
  isNullableTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.NullableTypeAnnotation>,
  ): this is NodePath_Final<t.NullableTypeAnnotation>;
  isNumberLiteral(
    this: NodePath_Final,
    opts?: Opts<t.NumberLiteral>,
  ): this is NodePath_Final<t.NumberLiteral>;
  isNumberLiteralTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.NumberLiteralTypeAnnotation>,
  ): this is NodePath_Final<t.NumberLiteralTypeAnnotation>;
  isNumberTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.NumberTypeAnnotation>,
  ): this is NodePath_Final<t.NumberTypeAnnotation>;
  isNumericLiteral(
    this: NodePath_Final,
    opts?: Opts<t.NumericLiteral>,
  ): this is NodePath_Final<t.NumericLiteral>;
  isObjectExpression(
    this: NodePath_Final,
    opts?: Opts<t.ObjectExpression>,
  ): this is NodePath_Final<t.ObjectExpression>;
  isObjectMember(
    this: NodePath_Final,
    opts?: Opts<t.ObjectMember>,
  ): this is NodePath_Final<t.ObjectMember>;
  isObjectMethod(
    this: NodePath_Final,
    opts?: Opts<t.ObjectMethod>,
  ): this is NodePath_Final<t.ObjectMethod>;
  isObjectPattern(
    this: NodePath_Final,
    opts?: Opts<t.ObjectPattern>,
  ): this is NodePath_Final<t.ObjectPattern>;
  isObjectProperty(
    this: NodePath_Final,
    opts?: Opts<t.ObjectProperty>,
  ): this is NodePath_Final<t.ObjectProperty>;
  isObjectTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.ObjectTypeAnnotation>,
  ): this is NodePath_Final<t.ObjectTypeAnnotation>;
  isObjectTypeCallProperty(
    this: NodePath_Final,
    opts?: Opts<t.ObjectTypeCallProperty>,
  ): this is NodePath_Final<t.ObjectTypeCallProperty>;
  isObjectTypeIndexer(
    this: NodePath_Final,
    opts?: Opts<t.ObjectTypeIndexer>,
  ): this is NodePath_Final<t.ObjectTypeIndexer>;
  isObjectTypeInternalSlot(
    this: NodePath_Final,
    opts?: Opts<t.ObjectTypeInternalSlot>,
  ): this is NodePath_Final<t.ObjectTypeInternalSlot>;
  isObjectTypeProperty(
    this: NodePath_Final,
    opts?: Opts<t.ObjectTypeProperty>,
  ): this is NodePath_Final<t.ObjectTypeProperty>;
  isObjectTypeSpreadProperty(
    this: NodePath_Final,
    opts?: Opts<t.ObjectTypeSpreadProperty>,
  ): this is NodePath_Final<t.ObjectTypeSpreadProperty>;
  isOpaqueType(
    this: NodePath_Final,
    opts?: Opts<t.OpaqueType>,
  ): this is NodePath_Final<t.OpaqueType>;
  isOptionalCallExpression(
    this: NodePath_Final,
    opts?: Opts<t.OptionalCallExpression>,
  ): this is NodePath_Final<t.OptionalCallExpression>;
  isOptionalIndexedAccessType(
    this: NodePath_Final,
    opts?: Opts<t.OptionalIndexedAccessType>,
  ): this is NodePath_Final<t.OptionalIndexedAccessType>;
  isOptionalMemberExpression(
    this: NodePath_Final,
    opts?: Opts<t.OptionalMemberExpression>,
  ): this is NodePath_Final<t.OptionalMemberExpression>;
  isParenthesizedExpression(
    this: NodePath_Final,
    opts?: Opts<t.ParenthesizedExpression>,
  ): this is NodePath_Final<t.ParenthesizedExpression>;
  isPattern(
    this: NodePath_Final,
    opts?: Opts<t.Pattern>,
  ): this is NodePath_Final<t.Pattern>;
  isPatternLike(
    this: NodePath_Final,
    opts?: Opts<t.PatternLike>,
  ): this is NodePath_Final<t.PatternLike>;
  isPipelineBareFunction(
    this: NodePath_Final,
    opts?: Opts<t.PipelineBareFunction>,
  ): this is NodePath_Final<t.PipelineBareFunction>;
  isPipelinePrimaryTopicReference(
    this: NodePath_Final,
    opts?: Opts<t.PipelinePrimaryTopicReference>,
  ): this is NodePath_Final<t.PipelinePrimaryTopicReference>;
  isPipelineTopicExpression(
    this: NodePath_Final,
    opts?: Opts<t.PipelineTopicExpression>,
  ): this is NodePath_Final<t.PipelineTopicExpression>;
  isPlaceholder(
    this: NodePath_Final,
    opts?: Opts<t.Placeholder>,
  ): this is NodePath_Final<t.Placeholder>;
  isPrivate(
    this: NodePath_Final,
    opts?: Opts<t.Private>,
  ): this is NodePath_Final<t.Private>;
  isPrivateName(
    this: NodePath_Final,
    opts?: Opts<t.PrivateName>,
  ): this is NodePath_Final<t.PrivateName>;
  isProgram(
    this: NodePath_Final,
    opts?: Opts<t.Program>,
  ): this is NodePath_Final<t.Program>;
  isProperty(
    this: NodePath_Final,
    opts?: Opts<t.Property>,
  ): this is NodePath_Final<t.Property>;
  isPureish(
    this: NodePath_Final,
    opts?: Opts<t.Pureish>,
  ): this is NodePath_Final<t.Pureish>;
  isQualifiedTypeIdentifier(
    this: NodePath_Final,
    opts?: Opts<t.QualifiedTypeIdentifier>,
  ): this is NodePath_Final<t.QualifiedTypeIdentifier>;
  isRecordExpression(
    this: NodePath_Final,
    opts?: Opts<t.RecordExpression>,
  ): this is NodePath_Final<t.RecordExpression>;
  isRegExpLiteral(
    this: NodePath_Final,
    opts?: Opts<t.RegExpLiteral>,
  ): this is NodePath_Final<t.RegExpLiteral>;
  isRegexLiteral(
    this: NodePath_Final,
    opts?: Opts<t.RegexLiteral>,
  ): this is NodePath_Final<t.RegexLiteral>;
  isRestElement(
    this: NodePath_Final,
    opts?: Opts<t.RestElement>,
  ): this is NodePath_Final<t.RestElement>;
  isRestProperty(
    this: NodePath_Final,
    opts?: Opts<t.RestProperty>,
  ): this is NodePath_Final<t.RestProperty>;
  isReturnStatement(
    this: NodePath_Final,
    opts?: Opts<t.ReturnStatement>,
  ): this is NodePath_Final<t.ReturnStatement>;
  isScopable(
    this: NodePath_Final,
    opts?: Opts<t.Scopable>,
  ): this is NodePath_Final<t.Scopable>;
  isSequenceExpression(
    this: NodePath_Final,
    opts?: Opts<t.SequenceExpression>,
  ): this is NodePath_Final<t.SequenceExpression>;
  isSpreadElement(
    this: NodePath_Final,
    opts?: Opts<t.SpreadElement>,
  ): this is NodePath_Final<t.SpreadElement>;
  isSpreadProperty(
    this: NodePath_Final,
    opts?: Opts<t.SpreadProperty>,
  ): this is NodePath_Final<t.SpreadProperty>;
  isStandardized(
    this: NodePath_Final,
    opts?: Opts<t.Standardized>,
  ): this is NodePath_Final<t.Standardized>;
  isStatement(
    this: NodePath_Final,
    opts?: Opts<t.Statement>,
  ): this is NodePath_Final<t.Statement>;
  isStaticBlock(
    this: NodePath_Final,
    opts?: Opts<t.StaticBlock>,
  ): this is NodePath_Final<t.StaticBlock>;
  isStringLiteral(
    this: NodePath_Final,
    opts?: Opts<t.StringLiteral>,
  ): this is NodePath_Final<t.StringLiteral>;
  isStringLiteralTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.StringLiteralTypeAnnotation>,
  ): this is NodePath_Final<t.StringLiteralTypeAnnotation>;
  isStringTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.StringTypeAnnotation>,
  ): this is NodePath_Final<t.StringTypeAnnotation>;
  isSuper(this: NodePath_Final, opts?: Opts<t.Super>): this is NodePath_Final<t.Super>;
  isSwitchCase(
    this: NodePath_Final,
    opts?: Opts<t.SwitchCase>,
  ): this is NodePath_Final<t.SwitchCase>;
  isSwitchStatement(
    this: NodePath_Final,
    opts?: Opts<t.SwitchStatement>,
  ): this is NodePath_Final<t.SwitchStatement>;
  isSymbolTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.SymbolTypeAnnotation>,
  ): this is NodePath_Final<t.SymbolTypeAnnotation>;
  isTSAnyKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSAnyKeyword>,
  ): this is NodePath_Final<t.TSAnyKeyword>;
  isTSArrayType(
    this: NodePath_Final,
    opts?: Opts<t.TSArrayType>,
  ): this is NodePath_Final<t.TSArrayType>;
  isTSAsExpression(
    this: NodePath_Final,
    opts?: Opts<t.TSAsExpression>,
  ): this is NodePath_Final<t.TSAsExpression>;
  isTSBaseType(
    this: NodePath_Final,
    opts?: Opts<t.TSBaseType>,
  ): this is NodePath_Final<t.TSBaseType>;
  isTSBigIntKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSBigIntKeyword>,
  ): this is NodePath_Final<t.TSBigIntKeyword>;
  isTSBooleanKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSBooleanKeyword>,
  ): this is NodePath_Final<t.TSBooleanKeyword>;
  isTSCallSignatureDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSCallSignatureDeclaration>,
  ): this is NodePath_Final<t.TSCallSignatureDeclaration>;
  isTSConditionalType(
    this: NodePath_Final,
    opts?: Opts<t.TSConditionalType>,
  ): this is NodePath_Final<t.TSConditionalType>;
  isTSConstructSignatureDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSConstructSignatureDeclaration>,
  ): this is NodePath_Final<t.TSConstructSignatureDeclaration>;
  isTSConstructorType(
    this: NodePath_Final,
    opts?: Opts<t.TSConstructorType>,
  ): this is NodePath_Final<t.TSConstructorType>;
  isTSDeclareFunction(
    this: NodePath_Final,
    opts?: Opts<t.TSDeclareFunction>,
  ): this is NodePath_Final<t.TSDeclareFunction>;
  isTSDeclareMethod(
    this: NodePath_Final,
    opts?: Opts<t.TSDeclareMethod>,
  ): this is NodePath_Final<t.TSDeclareMethod>;
  isTSEntityName(
    this: NodePath_Final,
    opts?: Opts<t.TSEntityName>,
  ): this is NodePath_Final<t.TSEntityName>;
  isTSEnumDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSEnumDeclaration>,
  ): this is NodePath_Final<t.TSEnumDeclaration>;
  isTSEnumMember(
    this: NodePath_Final,
    opts?: Opts<t.TSEnumMember>,
  ): this is NodePath_Final<t.TSEnumMember>;
  isTSExportAssignment(
    this: NodePath_Final,
    opts?: Opts<t.TSExportAssignment>,
  ): this is NodePath_Final<t.TSExportAssignment>;
  isTSExpressionWithTypeArguments(
    this: NodePath_Final,
    opts?: Opts<t.TSExpressionWithTypeArguments>,
  ): this is NodePath_Final<t.TSExpressionWithTypeArguments>;
  isTSExternalModuleReference(
    this: NodePath_Final,
    opts?: Opts<t.TSExternalModuleReference>,
  ): this is NodePath_Final<t.TSExternalModuleReference>;
  isTSFunctionType(
    this: NodePath_Final,
    opts?: Opts<t.TSFunctionType>,
  ): this is NodePath_Final<t.TSFunctionType>;
  isTSImportEqualsDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSImportEqualsDeclaration>,
  ): this is NodePath_Final<t.TSImportEqualsDeclaration>;
  isTSImportType(
    this: NodePath_Final,
    opts?: Opts<t.TSImportType>,
  ): this is NodePath_Final<t.TSImportType>;
  isTSIndexSignature(
    this: NodePath_Final,
    opts?: Opts<t.TSIndexSignature>,
  ): this is NodePath_Final<t.TSIndexSignature>;
  isTSIndexedAccessType(
    this: NodePath_Final,
    opts?: Opts<t.TSIndexedAccessType>,
  ): this is NodePath_Final<t.TSIndexedAccessType>;
  isTSInferType(
    this: NodePath_Final,
    opts?: Opts<t.TSInferType>,
  ): this is NodePath_Final<t.TSInferType>;
  isTSInstantiationExpression(
    this: NodePath_Final,
    opts?: Opts<t.TSInstantiationExpression>,
  ): this is NodePath_Final<t.TSInstantiationExpression>;
  isTSInterfaceBody(
    this: NodePath_Final,
    opts?: Opts<t.TSInterfaceBody>,
  ): this is NodePath_Final<t.TSInterfaceBody>;
  isTSInterfaceDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSInterfaceDeclaration>,
  ): this is NodePath_Final<t.TSInterfaceDeclaration>;
  isTSIntersectionType(
    this: NodePath_Final,
    opts?: Opts<t.TSIntersectionType>,
  ): this is NodePath_Final<t.TSIntersectionType>;
  isTSIntrinsicKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSIntrinsicKeyword>,
  ): this is NodePath_Final<t.TSIntrinsicKeyword>;
  isTSLiteralType(
    this: NodePath_Final,
    opts?: Opts<t.TSLiteralType>,
  ): this is NodePath_Final<t.TSLiteralType>;
  isTSMappedType(
    this: NodePath_Final,
    opts?: Opts<t.TSMappedType>,
  ): this is NodePath_Final<t.TSMappedType>;
  isTSMethodSignature(
    this: NodePath_Final,
    opts?: Opts<t.TSMethodSignature>,
  ): this is NodePath_Final<t.TSMethodSignature>;
  isTSModuleBlock(
    this: NodePath_Final,
    opts?: Opts<t.TSModuleBlock>,
  ): this is NodePath_Final<t.TSModuleBlock>;
  isTSModuleDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSModuleDeclaration>,
  ): this is NodePath_Final<t.TSModuleDeclaration>;
  isTSNamedTupleMember(
    this: NodePath_Final,
    opts?: Opts<t.TSNamedTupleMember>,
  ): this is NodePath_Final<t.TSNamedTupleMember>;
  isTSNamespaceExportDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSNamespaceExportDeclaration>,
  ): this is NodePath_Final<t.TSNamespaceExportDeclaration>;
  isTSNeverKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSNeverKeyword>,
  ): this is NodePath_Final<t.TSNeverKeyword>;
  isTSNonNullExpression(
    this: NodePath_Final,
    opts?: Opts<t.TSNonNullExpression>,
  ): this is NodePath_Final<t.TSNonNullExpression>;
  isTSNullKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSNullKeyword>,
  ): this is NodePath_Final<t.TSNullKeyword>;
  isTSNumberKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSNumberKeyword>,
  ): this is NodePath_Final<t.TSNumberKeyword>;
  isTSObjectKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSObjectKeyword>,
  ): this is NodePath_Final<t.TSObjectKeyword>;
  isTSOptionalType(
    this: NodePath_Final,
    opts?: Opts<t.TSOptionalType>,
  ): this is NodePath_Final<t.TSOptionalType>;
  isTSParameterProperty(
    this: NodePath_Final,
    opts?: Opts<t.TSParameterProperty>,
  ): this is NodePath_Final<t.TSParameterProperty>;
  isTSParenthesizedType(
    this: NodePath_Final,
    opts?: Opts<t.TSParenthesizedType>,
  ): this is NodePath_Final<t.TSParenthesizedType>;
  isTSPropertySignature(
    this: NodePath_Final,
    opts?: Opts<t.TSPropertySignature>,
  ): this is NodePath_Final<t.TSPropertySignature>;
  isTSQualifiedName(
    this: NodePath_Final,
    opts?: Opts<t.TSQualifiedName>,
  ): this is NodePath_Final<t.TSQualifiedName>;
  isTSRestType(
    this: NodePath_Final,
    opts?: Opts<t.TSRestType>,
  ): this is NodePath_Final<t.TSRestType>;
  isTSSatisfiesExpression(
    this: NodePath_Final,
    opts?: Opts<t.TSSatisfiesExpression>,
  ): this is NodePath_Final<t.TSSatisfiesExpression>;
  isTSStringKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSStringKeyword>,
  ): this is NodePath_Final<t.TSStringKeyword>;
  isTSSymbolKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSSymbolKeyword>,
  ): this is NodePath_Final<t.TSSymbolKeyword>;
  isTSThisType(
    this: NodePath_Final,
    opts?: Opts<t.TSThisType>,
  ): this is NodePath_Final<t.TSThisType>;
  isTSTupleType(
    this: NodePath_Final,
    opts?: Opts<t.TSTupleType>,
  ): this is NodePath_Final<t.TSTupleType>;
  isTSType(this: NodePath_Final, opts?: Opts<t.TSType>): this is NodePath_Final<t.TSType>;
  isTSTypeAliasDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeAliasDeclaration>,
  ): this is NodePath_Final<t.TSTypeAliasDeclaration>;
  isTSTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeAnnotation>,
  ): this is NodePath_Final<t.TSTypeAnnotation>;
  isTSTypeAssertion(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeAssertion>,
  ): this is NodePath_Final<t.TSTypeAssertion>;
  isTSTypeElement(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeElement>,
  ): this is NodePath_Final<t.TSTypeElement>;
  isTSTypeLiteral(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeLiteral>,
  ): this is NodePath_Final<t.TSTypeLiteral>;
  isTSTypeOperator(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeOperator>,
  ): this is NodePath_Final<t.TSTypeOperator>;
  isTSTypeParameter(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeParameter>,
  ): this is NodePath_Final<t.TSTypeParameter>;
  isTSTypeParameterDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeParameterDeclaration>,
  ): this is NodePath_Final<t.TSTypeParameterDeclaration>;
  isTSTypeParameterInstantiation(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeParameterInstantiation>,
  ): this is NodePath_Final<t.TSTypeParameterInstantiation>;
  isTSTypePredicate(
    this: NodePath_Final,
    opts?: Opts<t.TSTypePredicate>,
  ): this is NodePath_Final<t.TSTypePredicate>;
  isTSTypeQuery(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeQuery>,
  ): this is NodePath_Final<t.TSTypeQuery>;
  isTSTypeReference(
    this: NodePath_Final,
    opts?: Opts<t.TSTypeReference>,
  ): this is NodePath_Final<t.TSTypeReference>;
  isTSUndefinedKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSUndefinedKeyword>,
  ): this is NodePath_Final<t.TSUndefinedKeyword>;
  isTSUnionType(
    this: NodePath_Final,
    opts?: Opts<t.TSUnionType>,
  ): this is NodePath_Final<t.TSUnionType>;
  isTSUnknownKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSUnknownKeyword>,
  ): this is NodePath_Final<t.TSUnknownKeyword>;
  isTSVoidKeyword(
    this: NodePath_Final,
    opts?: Opts<t.TSVoidKeyword>,
  ): this is NodePath_Final<t.TSVoidKeyword>;
  isTaggedTemplateExpression(
    this: NodePath_Final,
    opts?: Opts<t.TaggedTemplateExpression>,
  ): this is NodePath_Final<t.TaggedTemplateExpression>;
  isTemplateElement(
    this: NodePath_Final,
    opts?: Opts<t.TemplateElement>,
  ): this is NodePath_Final<t.TemplateElement>;
  isTemplateLiteral(
    this: NodePath_Final,
    opts?: Opts<t.TemplateLiteral>,
  ): this is NodePath_Final<t.TemplateLiteral>;
  isTerminatorless(
    this: NodePath_Final,
    opts?: Opts<t.Terminatorless>,
  ): this is NodePath_Final<t.Terminatorless>;
  isThisExpression(
    this: NodePath_Final,
    opts?: Opts<t.ThisExpression>,
  ): this is NodePath_Final<t.ThisExpression>;
  isThisTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.ThisTypeAnnotation>,
  ): this is NodePath_Final<t.ThisTypeAnnotation>;
  isThrowStatement(
    this: NodePath_Final,
    opts?: Opts<t.ThrowStatement>,
  ): this is NodePath_Final<t.ThrowStatement>;
  isTopicReference(
    this: NodePath_Final,
    opts?: Opts<t.TopicReference>,
  ): this is NodePath_Final<t.TopicReference>;
  isTryStatement(
    this: NodePath_Final,
    opts?: Opts<t.TryStatement>,
  ): this is NodePath_Final<t.TryStatement>;
  isTupleExpression(
    this: NodePath_Final,
    opts?: Opts<t.TupleExpression>,
  ): this is NodePath_Final<t.TupleExpression>;
  isTupleTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.TupleTypeAnnotation>,
  ): this is NodePath_Final<t.TupleTypeAnnotation>;
  isTypeAlias(
    this: NodePath_Final,
    opts?: Opts<t.TypeAlias>,
  ): this is NodePath_Final<t.TypeAlias>;
  isTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.TypeAnnotation>,
  ): this is NodePath_Final<t.TypeAnnotation>;
  isTypeCastExpression(
    this: NodePath_Final,
    opts?: Opts<t.TypeCastExpression>,
  ): this is NodePath_Final<t.TypeCastExpression>;
  isTypeParameter(
    this: NodePath_Final,
    opts?: Opts<t.TypeParameter>,
  ): this is NodePath_Final<t.TypeParameter>;
  isTypeParameterDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.TypeParameterDeclaration>,
  ): this is NodePath_Final<t.TypeParameterDeclaration>;
  isTypeParameterInstantiation(
    this: NodePath_Final,
    opts?: Opts<t.TypeParameterInstantiation>,
  ): this is NodePath_Final<t.TypeParameterInstantiation>;
  isTypeScript(
    this: NodePath_Final,
    opts?: Opts<t.TypeScript>,
  ): this is NodePath_Final<t.TypeScript>;
  isTypeofTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.TypeofTypeAnnotation>,
  ): this is NodePath_Final<t.TypeofTypeAnnotation>;
  isUnaryExpression(
    this: NodePath_Final,
    opts?: Opts<t.UnaryExpression>,
  ): this is NodePath_Final<t.UnaryExpression>;
  isUnaryLike(
    this: NodePath_Final,
    opts?: Opts<t.UnaryLike>,
  ): this is NodePath_Final<t.UnaryLike>;
  isUnionTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.UnionTypeAnnotation>,
  ): this is NodePath_Final<t.UnionTypeAnnotation>;
  isUpdateExpression(
    this: NodePath_Final,
    opts?: Opts<t.UpdateExpression>,
  ): this is NodePath_Final<t.UpdateExpression>;
  isUserWhitespacable(
    this: NodePath_Final,
    opts?: Opts<t.UserWhitespacable>,
  ): this is NodePath_Final<t.UserWhitespacable>;
  isV8IntrinsicIdentifier(
    this: NodePath_Final,
    opts?: Opts<t.V8IntrinsicIdentifier>,
  ): this is NodePath_Final<t.V8IntrinsicIdentifier>;
  isVariableDeclaration(
    this: NodePath_Final,
    opts?: Opts<t.VariableDeclaration>,
  ): this is NodePath_Final<t.VariableDeclaration>;
  isVariableDeclarator(
    this: NodePath_Final,
    opts?: Opts<t.VariableDeclarator>,
  ): this is NodePath_Final<t.VariableDeclarator>;
  isVariance(
    this: NodePath_Final,
    opts?: Opts<t.Variance>,
  ): this is NodePath_Final<t.Variance>;
  isVoidTypeAnnotation(
    this: NodePath_Final,
    opts?: Opts<t.VoidTypeAnnotation>,
  ): this is NodePath_Final<t.VoidTypeAnnotation>;
  isWhile(this: NodePath_Final, opts?: Opts<t.While>): this is NodePath_Final<t.While>;
  isWhileStatement(
    this: NodePath_Final,
    opts?: Opts<t.WhileStatement>,
  ): this is NodePath_Final<t.WhileStatement>;
  isWithStatement(
    this: NodePath_Final,
    opts?: Opts<t.WithStatement>,
  ): this is NodePath_Final<t.WithStatement>;
  isYieldExpression(
    this: NodePath_Final,
    opts?: Opts<t.YieldExpression>,
  ): this is NodePath_Final<t.YieldExpression>;
}

interface NodePathValidators
  extends Omit<BaseNodePathValidators, keyof VirtualTypeNodePathValidators>,
    VirtualTypeNodePathValidators {}

declare const methods: {
    findParent: typeof findParent;
    find: typeof find;
    getFunctionParent: typeof getFunctionParent;
    getStatementParent: typeof getStatementParent;
    getEarliestCommonAncestorFrom: typeof getEarliestCommonAncestorFrom;
    getDeepestCommonAncestorFrom: typeof getDeepestCommonAncestorFrom;
    getAncestry: typeof getAncestry;
    isAncestor: typeof isAncestor;
    isDescendant: typeof isDescendant;
    inType: typeof inType;
    getTypeAnnotation: typeof getTypeAnnotation;
    isBaseType: typeof isBaseType;
    couldBeBaseType: typeof couldBeBaseType;
    baseTypeStrictlyMatches: typeof baseTypeStrictlyMatches;
    isGenericType: typeof isGenericType;
    replaceWithMultiple: typeof replaceWithMultiple;
    replaceWithSourceString: typeof replaceWithSourceString;
    replaceWith: typeof replaceWith;
    replaceExpressionWithStatements: typeof replaceExpressionWithStatements;
    replaceInline: typeof replaceInline;
    evaluateTruthy: typeof evaluateTruthy;
    evaluate: typeof evaluate;
    toComputedKey: typeof toComputedKey;
    ensureBlock: typeof ensureBlock;
    unwrapFunctionEnvironment: typeof unwrapFunctionEnvironment;
    arrowFunctionToExpression: typeof arrowFunctionToExpression;
    splitExportDeclaration: typeof splitExportDeclaration;
    ensureFunctionName: typeof ensureFunctionName;
    matchesPattern: typeof matchesPattern;
    has: typeof has;
    isStatic: typeof isStatic;
    is: typeof has;
    isnt: typeof isnt;
    equals: typeof equals;
    isNodeType: typeof isNodeType;
    canHaveVariableDeclarationOrExpression: typeof canHaveVariableDeclarationOrExpression;
    canSwapBetweenExpressionAndStatement: typeof canSwapBetweenExpressionAndStatement;
    isCompletionRecord: typeof isCompletionRecord;
    isStatementOrBlock: typeof isStatementOrBlock;
    referencesImport: typeof referencesImport;
    getSource: typeof getSource;
    willIMaybeExecuteBefore: typeof willIMaybeExecuteBefore;
    _guessExecutionStatusRelativeTo: typeof _guessExecutionStatusRelativeTo;
    resolve: typeof resolve;
    isConstantExpression: typeof isConstantExpression;
    isInStrictMode: typeof isInStrictMode;
    call: typeof call;
    isDenylisted: typeof isDenylisted;
    isBlacklisted: typeof isDenylisted;
    visit: typeof visit;
    skip: typeof skip;
    skipKey: typeof skipKey;
    stop: typeof stop;
    setScope: typeof setScope;
    setContext: typeof setContext;
    resync: typeof resync;
    popContext: typeof popContext;
    pushContext: typeof pushContext;
    setup: typeof setup;
    setKey: typeof setKey;
    requeue: typeof requeue;
    requeueComputedKeyAndDecorators: typeof requeueComputedKeyAndDecorators;
    remove: typeof remove;
    insertBefore: typeof insertBefore;
    insertAfter: typeof insertAfter;
    updateSiblingKeys: typeof updateSiblingKeys;
    unshiftContainer: typeof unshiftContainer;
    pushContainer: typeof pushContainer;
    hoist: typeof hoist;
    getOpposite: typeof getOpposite;
    getCompletionRecords: typeof getCompletionRecords;
    getSibling: typeof getSibling;
    getPrevSibling: typeof getPrevSibling;
    getNextSibling: typeof getNextSibling;
    getAllNextSiblings: typeof getAllNextSiblings;
    getAllPrevSiblings: typeof getAllPrevSiblings;
    get: typeof get;
    getAssignmentIdentifiers: typeof getAssignmentIdentifiers;
    getBindingIdentifiers: typeof getBindingIdentifiers;
    getOuterBindingIdentifiers: typeof getOuterBindingIdentifiers;
    getBindingIdentifierPaths: typeof getBindingIdentifierPaths;
    getOuterBindingIdentifierPaths: typeof getOuterBindingIdentifierPaths;
    shareCommentsWithSiblings: typeof shareCommentsWithSiblings;
    addComment: typeof addComment;
    addComments: typeof addComments;
};
interface NodePathOverwrites {
    /**
     * NOTE: This assertion doesn't narrow the type on unions of
     * NodePaths, due to https://github.com/microsoft/TypeScript/issues/44212
     *
     * @see ./conversion.ts for implementation.
     */
    ensureBlock(this: NodePath_Final): asserts this is NodePath_Final<(t.Loop | t.WithStatement | t.Function | t.LabeledStatement | t.CatchClause) & {
        body: t.BlockStatement;
    }>;
    /**
     * @see ./introspection.ts for implementation.
     */
    isStatementOrBlock(this: NodePath_Final): this is NodePath_Final<t.Statement | t.Block>;
}
type NodePathMixins = Omit<typeof methods, keyof NodePathOverwrites>;
interface NodePath<T extends t.Node> extends InstanceType<typeof NodePath_Final>, NodePathAssertions, NodePathValidators, NodePathMixins, NodePathOverwrites {
    type: T["type"] | null;
    node: T;
    parent: t.ParentMaps[T["type"]];
    parentPath: t.ParentMaps[T["type"]] extends null ? null : NodePath_Final<t.ParentMaps[T["type"]]> | null;
}
declare const NodePath_Final: {
    new (hub: HubInterface, parent: t.Node | null): {
        parent: t.Node;
        hub: HubInterface;
        data: Record<string | symbol, unknown>;
        context: TraversalContext;
        scope: Scope;
        contexts: Array<TraversalContext>;
        state: any;
        opts: ExplodedTraverseOptions | null;
        _traverseFlags: number;
        removed: boolean;
        shouldStop: boolean;
        shouldSkip: boolean;
        skipKeys: Record<string, boolean> | null;
        parentPath: NodePath_Final | null;
        container: t.Node | Array<t.Node> | null;
        listKey: string | null;
        key: string | number | null;
        node: t.Node | null;
        type: t.Node["type"] | null;
        getScope(this: NodePath_Final, scope: Scope): Scope;
        setData(key: string | symbol, val: any): any;
        getData(key: string | symbol, def?: any): any;
        hasNode(): boolean;
        buildCodeFrameError(msg: string, Error?: new () => Error): Error;
        traverse<T>(this: NodePath_Final, visitor: Visitor<T>, state: T): void;
        traverse(this: NodePath_Final, visitor: Visitor): void;
        set(key: string, node: any): void;
        getPathLocation(this: NodePath_Final): string;
        debug(this: NodePath_Final, message: string): void;
        toString(): string;
        inList: boolean;
        readonly parentKey: string;
    };
    get({ hub, parentPath, parent, container, listKey, key, }: {
        hub?: HubInterface;
        parentPath: NodePath_Final | null;
        parent: t.Node;
        container: t.Node | t.Node[];
        listKey?: string;
        key: string | number;
    }): NodePath_Final;
};
type NodePath_Final<T extends t.Node = t.Node> = T extends any ? NodePath<T> : never;

declare let pathsCache: WeakMap<HubInterface | typeof nullHub, WeakMap<Node, Map<Node, NodePath_Final>>>;

declare let scope: WeakMap<Node, Scope>;
declare function clear(): void;
declare function clearPath(): void;
declare function clearScope(): void;
declare const nullHub: Readonly<{}>;
declare function getCachedPaths(hub: HubInterface | null, parent: Node): Map<Node, NodePath_Final>;
declare function getOrCreateCachedPaths(hub: HubInterface | null, parent: Node): Map<Node, NodePath_Final>;

declare const __cache_ts_clear: typeof clear;
declare const __cache_ts_clearPath: typeof clearPath;
declare const __cache_ts_clearScope: typeof clearScope;
declare const __cache_ts_getCachedPaths: typeof getCachedPaths;
declare const __cache_ts_getOrCreateCachedPaths: typeof getOrCreateCachedPaths;
declare const __cache_ts_scope: typeof scope;
declare namespace __cache_ts {
  export { __cache_ts_clear as clear, __cache_ts_clearPath as clearPath, __cache_ts_clearScope as clearScope, __cache_ts_getCachedPaths as getCachedPaths, __cache_ts_getOrCreateCachedPaths as getOrCreateCachedPaths, pathsCache as path, __cache_ts_scope as scope };
}

type VisitWrapper<S = any> = (stateName: string | undefined, visitorType: VisitPhase, callback: VisitNodeFunction<S, Node>) => VisitNodeFunction<S, Node>;
declare function isExplodedVisitor(visitor: Visitor): visitor is ExplodedVisitor;

/**
 * explode() will take a visitor object with all of the various shorthands
 * that we support, and validates & normalizes it into a common format, ready
 * to be used in traversal
 *
 * The various shorthands are:
 * * `Identifier() { ... }` -> `Identifier: { enter() { ... } }`
 * * `"Identifier|NumericLiteral": { ... }` -> `Identifier: { ... }, NumericLiteral: { ... }`
 * * Aliases in `@babel/types`: e.g. `Property: { ... }` -> `ObjectProperty: { ... }, ClassProperty: { ... }`
 * Other normalizations are:
 * * Visitors of virtual types are wrapped, so that they are only visited when
 *   their dynamic check passes
 * * `enter` and `exit` functions are wrapped in arrays, to ease merging of
 *   visitors
 */
declare function explode$1<S>(visitor: Visitor<S>): ExplodedVisitor<S>;

declare function verify$1(visitor: Visitor): void;
declare function merge<State>(visitors: Visitor<State>[]): ExplodedVisitor<State>;
declare function merge(visitors: Visitor<unknown>[], states?: any[], wrapper?: Function | null): ExplodedVisitor<unknown>;
declare function environmentVisitor<S>(visitor: Visitor<S>): Visitor<S>;

type __visitors_ts_VisitWrapper<S = any> = VisitWrapper<S>;
declare const __visitors_ts_environmentVisitor: typeof environmentVisitor;
declare const __visitors_ts_isExplodedVisitor: typeof isExplodedVisitor;
declare const __visitors_ts_merge: typeof merge;
declare namespace __visitors_ts {
  export { type __visitors_ts_VisitWrapper as VisitWrapper, __visitors_ts_environmentVisitor as environmentVisitor, explode$1 as explode, __visitors_ts_isExplodedVisitor as isExplodedVisitor, __visitors_ts_merge as merge, verify$1 as verify };
}

type TraverseOptions<S = t.Node> = {
    scope?: Scope;
    noScope?: boolean;
    denylist?: string[];
    shouldSkip?: (node: NodePath_Final) => boolean;
} & Visitor<S>;
type ExplodedTraverseOptions<S = t.Node> = TraverseOptions<S> & ExplodedVisitor<S>;
declare function traverse<S>(parent: t.Node, opts: TraverseOptions<S>, scope: Scope | undefined, state: S, parentPath?: NodePath_Final, visitSelf?: boolean): void;
declare function traverse(parent: t.Node, opts: TraverseOptions, scope?: Scope, state?: any, parentPath?: NodePath_Final, visitSelf?: boolean): void;
declare namespace traverse {
    var visitors: typeof __visitors_ts;
    var verify: typeof verify$1;
    var explode: typeof explode$1;
    var cheap: (node: t.Node, enter: (node: t.Node) => void) => void;
    var node: (node: t.Node, opts: ExplodedTraverseOptions, scope?: Scope, state?: any, path?: NodePath_Final, skipKeys?: Record<string, boolean>) => void;
    var clearNode: (node: t.Node, opts?: RemovePropertiesOptions) => void;
    var removeProperties: (tree: t.Node, opts?: RemovePropertiesOptions) => t.Node;
    var hasType: (tree: t.Node, type: t.Node["type"], denylistTypes?: Array<string>) => boolean;
    var cache: typeof __cache_ts;
}

export { Binding, type ExplodedTraverseOptions, type ExplodedVisitor, Hub, type HubInterface, NodePath_Final as NodePath, Scope, type TraverseOptions, type Visitor, type VisitorBase, traverse as default, __visitors_ts as visitors };
