

declare module 'babylonjs' { 
    export = BABYLON; 
}
declare module BABYLON {
    /**
     * Set of assets to keep when moving a scene into an asset container.
     */
    class KeepAssets {
        /**
         * Cameras to keep.
         */
        cameras: Camera[];
        /**
         * Lights to keep.
         */
        lights: Light[];
        /**
         * Meshes to keep.
         */
        meshes: AbstractMesh[];
        /**
         * Skeletons to keep.
         */
        skeletons: Skeleton[];
        /**
         * ParticleSystems to keep.
         */
        particleSystems: IParticleSystem[];
        /**
         * Animations to keep.
         */
        animations: Animation[];
        /**
         * AnimationGroups to keep.
         */
        animationGroups: AnimationGroup[];
        /**
         * MultiMaterials to keep.
         */
        multiMaterials: MultiMaterial[];
        /**
         * Materials to keep.
         */
        materials: Material[];
        /**
         * MorphTargetManagers to keep.
         */
        morphTargetManagers: MorphTargetManager[];
        /**
         * Geometries to keep.
         */
        geometries: Geometry[];
        /**
         * TransformNodes to keep.
         */
        transformNodes: TransformNode[];
        /**
         * LensFlareSystems to keep.
         */
        lensFlareSystems: LensFlareSystem[];
        /**
         * ShadowGenerators to keep.
         */
        shadowGenerators: ShadowGenerator[];
        /**
         * ActionManagers to keep.
         */
        actionManagers: ActionManager[];
        /**
         * Sounds to keep.
         */
        sounds: Sound[];
        /**
         * Textures to keep.
         */
        textures: Texture[];
        /**
         * Effect layers to keep.
         */
        effectLayers: EffectLayer[];
    }
    /**
     * Container with a set of assets that can be added or removed from a scene.
     */
    class AssetContainer {
        /**
         * The scene the AssetContainer belongs to.
         */
        scene: Scene;
        /**
         * Cameras populated in the container.
         */
        cameras: Camera[];
        /**
         * Lights populated in the container.
         */
        lights: Light[];
        /**
         * Meshes populated in the container.
         */
        meshes: AbstractMesh[];
        /**
         * Skeletons populated in the container.
         */
        skeletons: Skeleton[];
        /**
         * ParticleSystems populated in the container.
         */
        particleSystems: IParticleSystem[];
        /**
         * Animations populated in the container.
         */
        animations: Animation[];
        /**
         * AnimationGroups populated in the container.
         */
        animationGroups: AnimationGroup[];
        /**
         * MultiMaterials populated in the container.
         */
        multiMaterials: MultiMaterial[];
        /**
         * Materials populated in the container.
         */
        materials: Material[];
        /**
         * MorphTargetManagers populated in the container.
         */
        morphTargetManagers: MorphTargetManager[];
        /**
         * Geometries populated in the container.
         */
        geometries: Geometry[];
        /**
         * TransformNodes populated in the container.
         */
        transformNodes: TransformNode[];
        /**
         * LensFlareSystems populated in the container.
         */
        lensFlareSystems: LensFlareSystem[];
        /**
         * ShadowGenerators populated in the container.
         */
        shadowGenerators: ShadowGenerator[];
        /**
         * ActionManagers populated in the container.
         */
        actionManagers: ActionManager[];
        /**
         * Sounds populated in the container.
         */
        sounds: Sound[];
        /**
         * Textures populated in the container.
         */
        textures: Texture[];
        /**
         * Effect layers populated in the container.
         */
        effectLayers: EffectLayer[];
        /**
         * Instantiates an AssetContainer.
         * @param scene The scene the AssetContainer belongs to.
         */
        constructor(scene: Scene);
        /**
         * Adds all the assets from the container to the scene.
         */
        addAllToScene(): void;
        /**
         * Removes all the assets in the container from the scene
         */
        removeAllFromScene(): void;
        private _moveAssets<T>(sourceAssets, targetAssets, keepAssets);
        /**
         * Removes all the assets contained in the scene and adds them to the container.
         * @param keepAssets Set of assets to keep in the scene. (default: empty)
         */
        moveAllFromScene(keepAssets?: KeepAssets): void;
    }
}

interface Window {
    mozIndexedDB: IDBFactory;
    webkitIndexedDB: IDBFactory;
    msIndexedDB: IDBFactory;
    webkitURL: typeof URL;
    mozRequestAnimationFrame(callback: FrameRequestCallback): number;
    oRequestAnimationFrame(callback: FrameRequestCallback): number;
    WebGLRenderingContext: WebGLRenderingContext;
    MSGesture: MSGesture;
    CANNON: any;
    AudioContext: AudioContext;
    webkitAudioContext: AudioContext;
    PointerEvent: any;
    Math: Math;
    Uint8Array: Uint8ArrayConstructor;
    Float32Array: Float32ArrayConstructor;
    mozURL: typeof URL;
    msURL: typeof URL;
    VRFrameData: any;
    DracoDecoderModule: any;
}
interface WebGLRenderingContext {
    drawArraysInstanced(mode: number, first: number, count: number, primcount: number): void;
    drawElementsInstanced(mode: number, count: number, type: number, offset: number, primcount: number): void;
    vertexAttribDivisor(index: number, divisor: number): void;
    createVertexArray(): any;
    bindVertexArray(vao?: WebGLVertexArrayObject | null): void;
    deleteVertexArray(vao: WebGLVertexArrayObject): void;
    blitFramebuffer(srcX0: number, srcY0: number, srcX1: number, srcY1: number, dstX0: number, dstY0: number, dstX1: number, dstY1: number, mask: number, filter: number): void;
    renderbufferStorageMultisample(target: number, samples: number, internalformat: number, width: number, height: number): void;
    bindBufferBase(target: number, index: number, buffer: WebGLBuffer | null): void;
    getUniformBlockIndex(program: WebGLProgram, uniformBlockName: string): number;
    uniformBlockBinding(program: WebGLProgram, uniformBlockIndex: number, uniformBlockBinding: number): void;
    createQuery(): WebGLQuery;
    deleteQuery(query: WebGLQuery): void;
    beginQuery(target: number, query: WebGLQuery): void;
    endQuery(target: number): void;
    getQueryParameter(query: WebGLQuery, pname: number): any;
    getQuery(target: number, pname: number): any;
    MAX_SAMPLES: number;
    RGBA8: number;
    READ_FRAMEBUFFER: number;
    DRAW_FRAMEBUFFER: number;
    UNIFORM_BUFFER: number;
    HALF_FLOAT_OES: number;
    RGBA16F: number;
    RGBA32F: number;
    R32F: number;
    RG32F: number;
    RGB32F: number;
    RED: number;
    RG: number;
    UNSIGNED_INT_24_8: number;
    DEPTH24_STENCIL8: number;
    drawBuffers(buffers: number[]): void;
    readBuffer(src: number): void;
    readonly COLOR_ATTACHMENT0: number;
    readonly COLOR_ATTACHMENT1: number;
    readonly COLOR_ATTACHMENT2: number;
    readonly COLOR_ATTACHMENT3: number;
    ANY_SAMPLES_PASSED_CONSERVATIVE: number;
    ANY_SAMPLES_PASSED: number;
    QUERY_RESULT_AVAILABLE: number;
    QUERY_RESULT: number;
}
interface Document {
    mozCancelFullScreen(): void;
    msCancelFullScreen(): void;
    mozFullScreen: boolean;
    msIsFullScreen: boolean;
    fullscreen: boolean;
    mozPointerLockElement: HTMLElement;
    msPointerLockElement: HTMLElement;
    webkitPointerLockElement: HTMLElement;
}
interface HTMLCanvasElement {
    msRequestPointerLock?(): void;
    mozRequestPointerLock?(): void;
    webkitRequestPointerLock?(): void;
}
interface CanvasRenderingContext2D {
    msImageSmoothingEnabled: boolean;
}
interface WebGLBuffer {
    references: number;
    capacity: number;
    is32Bits: boolean;
}
interface WebGLProgram {
    transformFeedback?: WebGLTransformFeedback | null;
    __SPECTOR_rebuildProgram?: ((vertexSourceCode: string, fragmentSourceCode: string, onCompiled: (program: WebGLProgram) => void, onError: (message: string) => void) => void) | null;
}
interface MouseEvent {
    mozMovementX: number;
    mozMovementY: number;
    webkitMovementX: number;
    webkitMovementY: number;
    msMovementX: number;
    msMovementY: number;
}
interface Navigator {
    mozGetVRDevices: (any: any) => any;
    webkitGetUserMedia(constraints: MediaStreamConstraints, successCallback: NavigatorUserMediaSuccessCallback, errorCallback: NavigatorUserMediaErrorCallback): void;
    mozGetUserMedia(constraints: MediaStreamConstraints, successCallback: NavigatorUserMediaSuccessCallback, errorCallback: NavigatorUserMediaErrorCallback): void;
    msGetUserMedia(constraints: MediaStreamConstraints, successCallback: NavigatorUserMediaSuccessCallback, errorCallback: NavigatorUserMediaErrorCallback): void;
    webkitGetGamepads(): Gamepad[];
    msGetGamepads(): Gamepad[];
    webkitGamepads(): Gamepad[];
}
interface HTMLVideoElement {
    mozSrcObject: any;
}
interface Screen {
    orientation: string;
    mozOrientation: string;
}
interface Math {
    fround(x: number): number;
    imul(a: number, b: number): number;
}
interface EXT_disjoint_timer_query {
    QUERY_COUNTER_BITS_EXT: number;
    TIME_ELAPSED_EXT: number;
    TIMESTAMP_EXT: number;
    GPU_DISJOINT_EXT: number;
    QUERY_RESULT_EXT: number;
    QUERY_RESULT_AVAILABLE_EXT: number;
    queryCounterEXT(query: WebGLQuery, target: number): void;
    createQueryEXT(): WebGLQuery;
    beginQueryEXT(target: number, query: WebGLQuery): void;
    endQueryEXT(target: number): void;
    getQueryObjectEXT(query: WebGLQuery, target: number): any;
    deleteQueryEXT(query: WebGLQuery): void;
}
interface WebGLUniformLocation {
    _currentState: any;
}

declare module BABYLON {
    /**
     * Node is the basic class for all scene objects (Mesh, Light Camera).
     */
    class Node {
        /**
         * Gets or sets the name of the node
         */
        name: string;
        /**
         * Gets or sets the id of the node
         */
        id: string;
        /**
         * Gets or sets the unique id of the node
         */
        uniqueId: number;
        /**
         * Gets or sets a string used to store user defined state for the node
         */
        state: string;
        /**
         * Gets or sets an object used to store user defined information for the node
         */
        metadata: any;
        /**
         * Gets or sets a boolean used to define if the node must be serialized
         */
        doNotSerialize: boolean;
        /** @hidden */
        _isDisposed: boolean;
        /**
         * Gets a list of Animations associated with the node
         */
        animations: Animation[];
        private _ranges;
        /**
         * Callback raised when the node is ready to be used
         */
        onReady: (node: Node) => void;
        private _isEnabled;
        private _isReady;
        /** @hidden */
        _currentRenderId: number;
        private _parentRenderId;
        protected _childRenderId: number;
        /** @hidden */
        _waitingParentId: Nullable<string>;
        private _scene;
        /** @hidden */
        _cache: any;
        private _parentNode;
        private _children;
        /**
         * Gets a boolean indicating if the node has been disposed
         * @returns true if the node was disposed
         */
        isDisposed(): boolean;
        /**
         * Gets or sets the parent of the node
         */
        parent: Nullable<Node>;
        private _animationPropertiesOverride;
        /**
         * Gets or sets the animation properties override
         */
        animationPropertiesOverride: Nullable<AnimationPropertiesOverride>;
        /**
         * Gets a string idenfifying the name of the class
         * @returns "Node" string
         */
        getClassName(): string;
        /**
        * An event triggered when the mesh is disposed
        */
        onDisposeObservable: Observable<Node>;
        private _onDisposeObserver;
        /**
         * Sets a callback that will be raised when the node will be disposed
         */
        onDispose: () => void;
        /**
         * Creates a new Node
         * @param {string} name - the name and id to be given to this node
         * @param {BABYLON.Scene} the scene this node will be added to
         */
        constructor(name: string, scene?: Nullable<Scene>);
        /**
         * Gets the scene of the node
         * @returns a {BABYLON.Scene}
         */
        getScene(): Scene;
        /**
         * Gets the engine of the node
         * @returns a {BABYLON.Engine}
         */
        getEngine(): Engine;
        private _behaviors;
        /**
         * Attach a behavior to the node
         * @see http://doc.babylonjs.com/features/behaviour
         * @param behavior defines the behavior to attach
         * @returns the current Node
         */
        addBehavior(behavior: Behavior<Node>): Node;
        /**
         * Remove an attached behavior
         * @see http://doc.babylonjs.com/features/behaviour
         * @param behavior defines the behavior to attach
         * @returns the current Node
         */
        removeBehavior(behavior: Behavior<Node>): Node;
        /**
         * Gets the list of attached behaviors
         * @see http://doc.babylonjs.com/features/behaviour
         */
        readonly behaviors: Behavior<Node>[];
        /**
         * Gets an attached behavior by name
         * @param name defines the name of the behavior to look for
         * @see http://doc.babylonjs.com/features/behaviour
         * @returns null if behavior was not found else the requested behavior
         */
        getBehaviorByName(name: string): Nullable<Behavior<Node>>;
        /**
         * Returns the world matrix of the node
         * @returns a matrix containing the node's world matrix
         */
        getWorldMatrix(): Matrix;
        /** @hidden */
        _getWorldMatrixDeterminant(): number;
        /** @hidden */
        _initCache(): void;
        /** @hidden */
        updateCache(force?: boolean): void;
        /** @hidden */
        _updateCache(ignoreParentClass?: boolean): void;
        /** @hidden */
        _isSynchronized(): boolean;
        /** @hidden */
        _markSyncedWithParent(): void;
        /** @hidden */
        isSynchronizedWithParent(): boolean;
        /** @hidden */
        isSynchronized(updateCache?: boolean): boolean;
        /** @hidden */
        hasNewParent(update?: boolean): boolean;
        /**
         * Is this node ready to be used/rendered
         * @param completeCheck defines if a complete check (including materials and lights) has to be done (false by default)
         * @return true if the node is ready
         */
        isReady(completeCheck?: boolean): boolean;
        /**
         * Is this node enabled?
         * If the node has a parent, all ancestors will be checked and false will be returned if any are false (not enabled), otherwise will return true
         * @param checkAncestors indicates if this method should check the ancestors. The default is to check the ancestors. If set to false, the method will return the value of this node without checking ancestors
         * @return whether this node (and its parent) is enabled
         */
        isEnabled(checkAncestors?: boolean): boolean;
        /**
         * Set the enabled state of this node
         * @param value defines the new enabled state
         */
        setEnabled(value: boolean): void;
        /**
         * Is this node a descendant of the given node?
         * The function will iterate up the hierarchy until the ancestor was found or no more parents defined
         * @param ancestor defines the parent node to inspect
         * @returns a boolean indicating if this node is a descendant of the given node
         */
        isDescendantOf(ancestor: Node): boolean;
        /** @hidden */
        _getDescendants(results: Node[], directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): void;
        /**
         * Will return all nodes that have this node as ascendant
         * @param directDescendantsOnly defines if true only direct descendants of 'this' will be considered, if false direct and also indirect (children of children, an so on in a recursive manner) descendants of 'this' will be considered
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @return all children nodes of all types
         */
        getDescendants(directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): Node[];
        /**
         * Get all child-meshes of this node
         * @param directDescendantsOnly defines if true only direct descendants of 'this' will be considered, if false direct and also indirect (children of children, an so on in a recursive manner) descendants of 'this' will be considered
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @returns an array of {BABYLON.AbstractMesh}
         */
        getChildMeshes(directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): AbstractMesh[];
        /**
         * Get all child-transformNodes of this node
         * @param directDescendantsOnly defines if true only direct descendants of 'this' will be considered, if false direct and also indirect (children of children, an so on in a recursive manner) descendants of 'this' will be considered
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @returns an array of {BABYLON.TransformNode}
         */
        getChildTransformNodes(directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): TransformNode[];
        /**
         * Get all direct children of this node
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @returns an array of {BABYLON.Node}
         */
        getChildren(predicate?: (node: Node) => boolean): Node[];
        /** @hidden */
        _setReady(state: boolean): void;
        /**
         * Get an animation by name
         * @param name defines the name of the animation to look for
         * @returns null if not found else the requested animation
         */
        getAnimationByName(name: string): Nullable<Animation>;
        /**
         * Creates an animation range for this node
         * @param name defines the name of the range
         * @param from defines the starting key
         * @param to defines the end key
         */
        createAnimationRange(name: string, from: number, to: number): void;
        /**
         * Delete a specific animation range
         * @param name defines the name of the range to delete
         * @param deleteFrames defines if animation frames from the range must be deleted as well
         */
        deleteAnimationRange(name: string, deleteFrames?: boolean): void;
        /**
         * Get an animation range by name
         * @param name defines the name of the animation range to look for
         * @returns null if not found else the requested animation range
         */
        getAnimationRange(name: string): Nullable<AnimationRange>;
        /**
         * Will start the animation sequence
         * @param name defines the range frames for animation sequence
         * @param loop defines if the animation should loop (false by default)
         * @param speedRatio defines the speed factor in which to run the animation (1 by default)
         * @param onAnimationEnd defines a function to be executed when the animation ended (undefined by default)
         * @returns the object created for this animation. If range does not exist, it will return null
         */
        beginAnimation(name: string, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Serialize animation ranges into a JSON compatible object
         * @returns serialization object
         */
        serializeAnimationRanges(): any;
        /**
         * Computes the world matrix of the node
         * @param force defines if the cache version should be invalidated forcing the world matrix to be created from scratch
         * @returns the world matrix
         */
        computeWorldMatrix(force?: boolean): Matrix;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Parse animation range data from a serialization object and store them into a given node
         * @param node defines where to store the animation ranges
         * @param parsedNode defines the serialization object to read data from
         * @param scene defines the hosting scene
         */
        static ParseAnimationRanges(node: Node, parsedNode: any, scene: Scene): void;
    }
}

declare module BABYLON {
    /**
     * Define an interface for all classes that will hold resources
     */
    interface IDisposable {
        /**
         * Releases all held resources
         */
        dispose(): void;
    }
    /**
     * Interface used to let developers provide their own mesh selection mechanism
     */
    interface IActiveMeshCandidateProvider {
        /**
         * Return the list of active meshes
         * @param scene defines the current scene
         * @returns the list of active meshes
         */
        getMeshes(scene: Scene): AbstractMesh[];
        /**
         * Indicates if the meshes have been checked to make sure they are isEnabled()
         */
        readonly checksIsEnabled: boolean;
    }
    /**
     * This class is used by the onRenderingGroupObservable
     */
    class RenderingGroupInfo {
        /**
         * The Scene that being rendered
         */
        scene: Scene;
        /**
         * The camera currently used for the rendering pass
         */
        camera: Nullable<Camera>;
        /**
         * The ID of the renderingGroup being processed
         */
        renderingGroupId: number;
        /**
         * The rendering stage, can be either STAGE_PRECLEAR, STAGE_PREOPAQUE, STAGE_PRETRANSPARENT, STAGE_POSTTRANSPARENT
         */
        renderStage: number;
        /**
         * Stage corresponding to the very first hook in the renderingGroup phase: before the render buffer may be cleared
         * This stage will be fired no matter what
         */
        static STAGE_PRECLEAR: number;
        /**
         * Called before opaque object are rendered.
         * This stage will be fired only if there's 3D Opaque content to render
         */
        static STAGE_PREOPAQUE: number;
        /**
         * Called after the opaque objects are rendered and before the transparent ones
         * This stage will be fired only if there's 3D transparent content to render
         */
        static STAGE_PRETRANSPARENT: number;
        /**
         * Called after the transparent object are rendered, last hook of the renderingGroup phase
         * This stage will be fired no matter what
         */
        static STAGE_POSTTRANSPARENT: number;
    }
    /**
     * Represents a scene to be rendered by the engine.
     * @see http://doc.babylonjs.com/features/scene
     */
    class Scene implements IAnimatable {
        private static _FOGMODE_NONE;
        private static _FOGMODE_EXP;
        private static _FOGMODE_EXP2;
        private static _FOGMODE_LINEAR;
        private static _uniqueIdCounter;
        /**
         * Gets or sets the minimum deltatime when deterministic lock step is enabled
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         */
        static MinDeltaTime: number;
        /**
         * Gets or sets the maximum deltatime when deterministic lock step is enabled
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         */
        static MaxDeltaTime: number;
        /** The fog is deactivated */
        static readonly FOGMODE_NONE: number;
        /** The fog density is following an exponential function */
        static readonly FOGMODE_EXP: number;
        /** The fog density is following an exponential function faster than FOGMODE_EXP */
        static readonly FOGMODE_EXP2: number;
        /** The fog density is following a linear function. */
        static readonly FOGMODE_LINEAR: number;
        /**
         * Gets or sets a boolean that indicates if the scene must clear the render buffer before rendering a frame
         */
        autoClear: boolean;
        /**
         * Gets or sets a boolean that indicates if the scene must clear the depth and stencil buffers before rendering a frame
         */
        autoClearDepthAndStencil: boolean;
        /**
         * Defines the color used to clear the render buffer (Default is (0.2, 0.2, 0.3, 1.0))
         */
        clearColor: Color4;
        /**
         * Defines the color used to simulate the ambient color (Default is (0, 0, 0))
         */
        ambientColor: Color3;
        /** @hidden */
        _environmentBRDFTexture: BaseTexture;
        /** @hidden */
        protected _environmentTexture: BaseTexture;
        /**
         * Texture used in all pbr material as the reflection texture.
         * As in the majority of the scene they are the same (exception for multi room and so on),
         * this is easier to reference from here than from all the materials.
         */
        /**
         * Texture used in all pbr material as the reflection texture.
         * As in the majority of the scene they are the same (exception for multi room and so on),
         * this is easier to set here than in all the materials.
         */
        environmentTexture: BaseTexture;
        /** @hidden */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Default image processing configuration used either in the rendering
         * Forward main pass or through the imageProcessingPostProcess if present.
         * As in the majority of the scene they are the same (exception for multi camera),
         * this is easier to reference from here than from all the materials and post process.
         *
         * No setter as we it is a shared configuration, you can set the values instead.
         */
        readonly imageProcessingConfiguration: ImageProcessingConfiguration;
        private _forceWireframe;
        /**
         * Gets or sets a boolean indicating if all rendering must be done in wireframe
         */
        forceWireframe: boolean;
        private _forcePointsCloud;
        /**
         * Gets or sets a boolean indicating if all rendering must be done in point cloud
         */
        forcePointsCloud: boolean;
        /**
         * Gets or sets a boolean indicating if all bounding boxes must be rendered
         */
        forceShowBoundingBoxes: boolean;
        /**
         * Gets or sets the active clipplane
         */
        clipPlane: Nullable<Plane>;
        /**
         * Gets or sets a boolean indicating if animations are enabled
         */
        animationsEnabled: boolean;
        private _animationPropertiesOverride;
        /**
         * Gets or sets the animation properties override
         */
        animationPropertiesOverride: Nullable<AnimationPropertiesOverride>;
        /**
         * Gets or sets a boolean indicating if a constant deltatime has to be used
         * This is mostly useful for testing purposes when you do not want the animations to scale with the framerate
         */
        useConstantAnimationDeltaTime: boolean;
        /**
         * Gets or sets a boolean indicating if the scene must keep the meshUnderPointer property updated
         * Please note that it requires to run a ray cast through the scene on every frame
         */
        constantlyUpdateMeshUnderPointer: boolean;
        /**
         * Defines the HTML cursor to use when hovering over interactive elements
         */
        hoverCursor: string;
        /**
         * Defines the HTML default cursor to use (empty by default)
         */
        defaultCursor: string;
        /**
         * This is used to call preventDefault() on pointer down
         * in order to block unwanted artifacts like system double clicks
         */
        preventDefaultOnPointerDown: boolean;
        /**
         * Gets or sets user defined metadata
         */
        metadata: any;
        /**
         * Gets the name of the plugin used to load this scene (null by default)
         */
        loadingPluginName: string;
        /**
         * Use this array to add regular expressions used to disable offline support for specific urls
         */
        disableOfflineSupportExceptionRules: RegExp[];
        private _spritePredicate;
        /**
        * An event triggered when the scene is disposed.
        */
        onDisposeObservable: Observable<Scene>;
        private _onDisposeObserver;
        /** Sets a function to be executed when this scene is disposed. */
        onDispose: () => void;
        /**
        * An event triggered before rendering the scene (right after animations and physics)
        */
        onBeforeRenderObservable: Observable<Scene>;
        private _onBeforeRenderObserver;
        /** Sets a function to be executed before rendering this scene */
        beforeRender: Nullable<() => void>;
        /**
        * An event triggered after rendering the scene
        */
        onAfterRenderObservable: Observable<Scene>;
        private _onAfterRenderObserver;
        /** Sets a function to be executed after rendering this scene */
        afterRender: Nullable<() => void>;
        /**
        * An event triggered before animating the scene
        */
        onBeforeAnimationsObservable: Observable<Scene>;
        /**
        * An event triggered after animations processing
        */
        onAfterAnimationsObservable: Observable<Scene>;
        /**
        * An event triggered before draw calls are ready to be sent
        */
        onBeforeDrawPhaseObservable: Observable<Scene>;
        /**
        * An event triggered after draw calls have been sent
        */
        onAfterDrawPhaseObservable: Observable<Scene>;
        /**
        * An event triggered when physic simulation is about to be run
        */
        onBeforePhysicsObservable: Observable<Scene>;
        /**
        * An event triggered when physic simulation has been done
        */
        onAfterPhysicsObservable: Observable<Scene>;
        /**
        * An event triggered when the scene is ready
        */
        onReadyObservable: Observable<Scene>;
        /**
        * An event triggered before rendering a camera
        */
        onBeforeCameraRenderObservable: Observable<Camera>;
        private _onBeforeCameraRenderObserver;
        /** Sets a function to be executed before rendering a camera*/
        beforeCameraRender: () => void;
        /**
        * An event triggered after rendering a camera
        */
        onAfterCameraRenderObservable: Observable<Camera>;
        private _onAfterCameraRenderObserver;
        /** Sets a function to be executed after rendering a camera*/
        afterCameraRender: () => void;
        /**
        * An event triggered when active meshes evaluation is about to start
        */
        onBeforeActiveMeshesEvaluationObservable: Observable<Scene>;
        /**
        * An event triggered when active meshes evaluation is done
        */
        onAfterActiveMeshesEvaluationObservable: Observable<Scene>;
        /**
        * An event triggered when particles rendering is about to start
        * Note: This event can be trigger more than once per frame (because particles can be rendered by render target textures as well)
        */
        onBeforeParticlesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when particles rendering is done
        * Note: This event can be trigger more than once per frame (because particles can be rendered by render target textures as well)
        */
        onAfterParticlesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when sprites rendering is about to start
        * Note: This event can be trigger more than once per frame (because sprites can be rendered by render target textures as well)
        */
        onBeforeSpritesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when sprites rendering is done
        * Note: This event can be trigger more than once per frame (because sprites can be rendered by render target textures as well)
        */
        onAfterSpritesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when SceneLoader.Append or SceneLoader.Load or SceneLoader.ImportMesh were successfully executed
        */
        onDataLoadedObservable: Observable<Scene>;
        /**
        * An event triggered when a camera is created
        */
        onNewCameraAddedObservable: Observable<Camera>;
        /**
        * An event triggered when a camera is removed
        */
        onCameraRemovedObservable: Observable<Camera>;
        /**
        * An event triggered when a light is created
        */
        onNewLightAddedObservable: Observable<Light>;
        /**
        * An event triggered when a light is removed
        */
        onLightRemovedObservable: Observable<Light>;
        /**
        * An event triggered when a geometry is created
        */
        onNewGeometryAddedObservable: Observable<Geometry>;
        /**
        * An event triggered when a geometry is removed
        */
        onGeometryRemovedObservable: Observable<Geometry>;
        /**
        * An event triggered when a transform node is created
        */
        onNewTransformNodeAddedObservable: Observable<TransformNode>;
        /**
        * An event triggered when a transform node is removed
        */
        onTransformNodeRemovedObservable: Observable<TransformNode>;
        /**
        * An event triggered when a mesh is created
        */
        onNewMeshAddedObservable: Observable<AbstractMesh>;
        /**
        * An event triggered when a mesh is removed
        */
        onMeshRemovedObservable: Observable<AbstractMesh>;
        /**
        * An event triggered when render targets are about to be rendered
        * Can happen multiple times per frame.
        */
        onBeforeRenderTargetsRenderObservable: Observable<Scene>;
        /**
        * An event triggered when render targets were rendered.
        * Can happen multiple times per frame.
        */
        onAfterRenderTargetsRenderObservable: Observable<Scene>;
        /**
        * An event triggered before calculating deterministic simulation step
        */
        onBeforeStepObservable: Observable<Scene>;
        /**
        * An event triggered after calculating deterministic simulation step
        */
        onAfterStepObservable: Observable<Scene>;
        /**
         * This Observable will be triggered for each stage of each renderingGroup of each rendered camera.
         * The RenderinGroupInfo class contains all the information about the context in which the observable is called
         * If you wish to register an Observer only for a given set of renderingGroup, use the mask with a combination of the renderingGroup index elevated to the power of two (1 for renderingGroup 0, 2 for renderingrOup1, 4 for 2 and 8 for 3)
         */
        onRenderingGroupObservable: Observable<RenderingGroupInfo>;
        /**
         * Gets a list of Animations associated with the scene
         */
        animations: Animation[];
        private _registeredForLateAnimationBindings;
        /**
         * Gets or sets a predicate used to select candidate meshes for a pointer down event
         */
        pointerDownPredicate: (Mesh: AbstractMesh) => boolean;
        /**
         * Gets or sets a predicate used to select candidate meshes for a pointer up event
         */
        pointerUpPredicate: (Mesh: AbstractMesh) => boolean;
        /**
         * Gets or sets a predicate used to select candidate meshes for a pointer move event
         */
        pointerMovePredicate: (Mesh: AbstractMesh) => boolean;
        private _onPointerMove;
        private _onPointerDown;
        private _onPointerUp;
        /** Deprecated. Use onPointerObservable instead */
        onPointerMove: (evt: PointerEvent, pickInfo: PickingInfo, type: PointerEventTypes) => void;
        /** Deprecated. Use onPointerObservable instead */
        onPointerDown: (evt: PointerEvent, pickInfo: PickingInfo, type: PointerEventTypes) => void;
        /** Deprecated. Use onPointerObservable instead */
        onPointerUp: (evt: PointerEvent, pickInfo: Nullable<PickingInfo>, type: PointerEventTypes) => void;
        /** Deprecated. Use onPointerObservable instead */
        onPointerPick: (evt: PointerEvent, pickInfo: PickingInfo) => void;
        private _gamepadManager;
        /**
         * Gets the gamepad manager associated with the scene
         * @see http://doc.babylonjs.com/how_to/how_to_use_gamepads
         */
        readonly gamepadManager: GamepadManager;
        /**
         * This observable event is triggered when any ponter event is triggered. It is registered during Scene.attachControl() and it is called BEFORE the 3D engine process anything (mesh/sprite picking for instance).
         * You have the possibility to skip the process and the call to onPointerObservable by setting PointerInfoPre.skipOnPointerObservable to true
         */
        onPrePointerObservable: Observable<PointerInfoPre>;
        /**
         * Observable event triggered each time an input event is received from the rendering canvas
         */
        onPointerObservable: Observable<PointerInfo>;
        /**
         * Gets the pointer coordinates without any translation (ie. straight out of the pointer event)
         */
        readonly unTranslatedPointer: Vector2;
        /** The distance in pixel that you have to move to prevent some events */
        static DragMovementThreshold: number;
        /** Time in milliseconds to wait to raise long press events if button is still pressed */
        static LongPressDelay: number;
        /** Time in milliseconds with two consecutive clicks will be considered as a double click */
        static DoubleClickDelay: number;
        /** If you need to check double click without raising a single click at first click, enable this flag */
        static ExclusiveDoubleClickMode: boolean;
        private _initClickEvent;
        private _initActionManager;
        private _delayedSimpleClick;
        private _delayedSimpleClickTimeout;
        private _previousDelayedSimpleClickTimeout;
        private _meshPickProceed;
        private _previousButtonPressed;
        private _currentPickResult;
        private _previousPickResult;
        private _totalPointersPressed;
        private _doubleClickOccured;
        /** Define this parameter if you are using multiple cameras and you want to specify which one should be used for pointer position */
        cameraToUseForPointers: Nullable<Camera>;
        private _pointerX;
        private _pointerY;
        private _unTranslatedPointerX;
        private _unTranslatedPointerY;
        private _startingPointerPosition;
        private _previousStartingPointerPosition;
        private _startingPointerTime;
        private _previousStartingPointerTime;
        private _pointerCaptures;
        private _timeAccumulator;
        private _currentStepId;
        private _currentInternalStep;
        /** @hidden */
        _mirroredCameraPosition: Nullable<Vector3>;
        /**
         * This observable event is triggered when any keyboard event si raised and registered during Scene.attachControl()
         * You have the possibility to skip the process and the call to onKeyboardObservable by setting KeyboardInfoPre.skipOnPointerObservable to true
         */
        onPreKeyboardObservable: Observable<KeyboardInfoPre>;
        /**
         * Observable event triggered each time an keyboard event is received from the hosting window
         */
        onKeyboardObservable: Observable<KeyboardInfo>;
        private _onKeyDown;
        private _onKeyUp;
        private _onCanvasFocusObserver;
        private _onCanvasBlurObserver;
        private _useRightHandedSystem;
        /**
        * Gets or sets a boolean indicating if the scene must use right-handed coordinates system
        */
        useRightHandedSystem: boolean;
        /**
         * Sets the step Id used by deterministic lock step
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         * @param newStepId defines the step Id
         */
        setStepId(newStepId: number): void;
        /**
         * Gets the step Id used by deterministic lock step
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         * @returns the step Id
         */
        getStepId(): number;
        /**
         * Gets the internal step used by deterministic lock step
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         * @returns the internal step
         */
        getInternalStep(): number;
        private _fogEnabled;
        /**
        * Gets or sets a boolean indicating if fog is enabled on this scene
        * @see http://doc.babylonjs.com/babylon101/environment#fog
        */
        fogEnabled: boolean;
        private _fogMode;
        /**
        * Gets or sets the fog mode to use
        * @see http://doc.babylonjs.com/babylon101/environment#fog
        */
        fogMode: number;
        /**
        * Gets or sets the fog color to use
        * @see http://doc.babylonjs.com/babylon101/environment#fog
        */
        fogColor: Color3;
        /**
        * Gets or sets the fog density to use
        * @see http://doc.babylonjs.com/babylon101/environment#fog
        */
        fogDensity: number;
        /**
        * Gets or sets the fog start distance to use
        * @see http://doc.babylonjs.com/babylon101/environment#fog
        */
        fogStart: number;
        /**
        * Gets or sets the fog end distance to use
        * @see http://doc.babylonjs.com/babylon101/environment#fog
        */
        fogEnd: number;
        private _shadowsEnabled;
        /**
        * Gets or sets a boolean indicating if shadows are enabled on this scene
        */
        shadowsEnabled: boolean;
        private _lightsEnabled;
        /**
        * Gets or sets a boolean indicating if lights are enabled on this scene
        */
        lightsEnabled: boolean;
        /**
        * All of the lights added to this scene
        * @see http://doc.babylonjs.com/babylon101/lights
        */
        lights: Light[];
        /** All of the cameras added to this scene.
         * @see http://doc.babylonjs.com/babylon101/cameras
         */
        cameras: Camera[];
        /** All of the active cameras added to this scene. */
        activeCameras: Camera[];
        /** The current active camera */
        activeCamera: Nullable<Camera>;
        /**
        * All of the tranform nodes added to this scene
        * @see http://doc.babylonjs.com/how_to/transformnode
        */
        transformNodes: TransformNode[];
        /**
        * All of the (abstract) meshes added to this scene
        */
        meshes: AbstractMesh[];
        /**
        * All of the animation groups added to this scene
        * @see http://doc.babylonjs.com/how_to/group
        */
        animationGroups: AnimationGroup[];
        private _geometries;
        /**
        * All of the materials added to this scene
        * @see http://doc.babylonjs.com/babylon101/materials
        */
        materials: Material[];
        /**
        * All of the multi-materials added to this scene
        * @see http://doc.babylonjs.com/how_to/multi_materials
        */
        multiMaterials: MultiMaterial[];
        private _defaultMaterial;
        /** The default material used on meshes when no material is affected */
        /** The default material used on meshes when no material is affected */
        defaultMaterial: Material;
        private _texturesEnabled;
        /**
        * Gets or sets a boolean indicating if textures are enabled on this scene
        */
        texturesEnabled: boolean;
        /**
        * All of the textures added to this scene
        */
        textures: BaseTexture[];
        /**
        * Gets or sets a boolean indicating if particles are enabled on this scene
        */
        particlesEnabled: boolean;
        /**
        * All of the particle systems added to this scene
        * @see http://doc.babylonjs.com/babylon101/particles
        */
        particleSystems: IParticleSystem[];
        /**
        * Gets or sets a boolean indicating if sprites are enabled on this scene
        */
        spritesEnabled: boolean;
        /**
        * All of the sprite managers added to this scene
        * @see http://doc.babylonjs.com/babylon101/sprites
        */
        spriteManagers: SpriteManager[];
        /**
         * The list of layers (background and foreground) of the scene
         */
        layers: Layer[];
        /**
         * The list of effect layers (highlights/glow) added to the scene
         * @see http://doc.babylonjs.com/how_to/highlight_layer
         * @see http://doc.babylonjs.com/how_to/glow_layer
         */
        effectLayers: EffectLayer[];
        private _skeletonsEnabled;
        /**
        * Gets or sets a boolean indicating if skeletons are enabled on this scene
        */
        skeletonsEnabled: boolean;
        /**
         * The list of skeletons added to the scene
         * @see http://doc.babylonjs.com/how_to/how_to_use_bones_and_skeletons
         */
        skeletons: Skeleton[];
        /**
         * The list of morph target managers added to the scene
         * @see http://doc.babylonjs.com/how_to/how_to_dynamically_morph_a_mesh
         */
        morphTargetManagers: MorphTargetManager[];
        /**
        * Gets or sets a boolean indicating if lens flares are enabled on this scene
        */
        lensFlaresEnabled: boolean;
        /**
         * The list of lens flare system added to the scene
         * @see http://doc.babylonjs.com/how_to/how_to_use_lens_flares
         */
        lensFlareSystems: LensFlareSystem[];
        /**
        * Gets or sets a boolean indicating if collisions are enabled on this scene
        * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
        */
        collisionsEnabled: boolean;
        private _workerCollisions;
        /** @hidden */
        collisionCoordinator: ICollisionCoordinator;
        /**
         * Defines the gravity applied to this scene (used only for collisions)
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
         */
        gravity: Vector3;
        /**
        * Gets or sets a boolean indicating if postprocesses are enabled on this scene
        */
        postProcessesEnabled: boolean;
        /**
         * The list of postprocesses added to the scene
         */
        postProcesses: PostProcess[];
        /**
         * Gets the current postprocess manager
         */
        postProcessManager: PostProcessManager;
        private _postProcessRenderPipelineManager;
        /**
         * Gets the postprocess render pipeline manager
         * @see http://doc.babylonjs.com/how_to/how_to_use_postprocessrenderpipeline
         * @see http://doc.babylonjs.com/how_to/using_default_rendering_pipeline
         */
        readonly postProcessRenderPipelineManager: PostProcessRenderPipelineManager;
        /**
        * Gets or sets a boolean indicating if render targets are enabled on this scene
        */
        renderTargetsEnabled: boolean;
        /**
        * Gets or sets a boolean indicating if next render targets must be dumped as image for debugging purposes
        * We recommend not using it and instead rely on Spector.js: http://spector.babylonjs.com
        */
        dumpNextRenderTargets: boolean;
        /**
         * The list of user defined render targets added to the scene
         */
        customRenderTargets: RenderTargetTexture[];
        /**
         * Defines if texture loading must be delayed
         * If true, textures will only be loaded when they need to be rendered
         */
        useDelayedTextureLoading: boolean;
        /**
         * Gets the list of meshes imported to the scene through SceneLoader
         */
        importedMeshesFiles: String[];
        /**
        * Gets or sets a boolean indicating if probes are enabled on this scene
        */
        probesEnabled: boolean;
        /**
         * The list of reflection probes added to the scene
         * @see http://doc.babylonjs.com/how_to/how_to_use_reflection_probes
         */
        reflectionProbes: ReflectionProbe[];
        /**
         * @hidden
         */
        database: Database;
        /**
         * Gets or sets the action manager associated with the scene
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions
        */
        actionManager: ActionManager;
        /** @hidden */
        _actionManagers: ActionManager[];
        private _meshesForIntersections;
        /**
        * Gets or sets a boolean indicating if procedural textures are enabled on this scene
        */
        proceduralTexturesEnabled: boolean;
        /**
         * The list of procedural textures added to the scene
         * @see http://doc.babylonjs.com/how_to/how_to_use_procedural_textures
         */
        proceduralTextures: ProceduralTexture[];
        private _mainSoundTrack;
        /**
         * The list of sound tracks added to the scene
         * @see http://doc.babylonjs.com/how_to/playing_sounds_and_music
         */
        soundTracks: SoundTrack[];
        private _audioEnabled;
        private _headphone;
        /**
         * Gets the main soundtrack associated with the scene
         */
        readonly mainSoundTrack: SoundTrack;
        /**
         * Gets or sets the VRExperienceHelper attached to the scene
         * @see http://doc.babylonjs.com/how_to/webvr_helper
         * @ignorenaming
         */
        VRHelper: VRExperienceHelper;
        /**
         * Gets or sets the simplification queue attached to the scene
         * @see http://doc.babylonjs.com/how_to/in-browser_mesh_simplification
         */
        simplificationQueue: SimplificationQueue;
        private _engine;
        private _totalVertices;
        /** @hidden */
        _activeIndices: PerfCounter;
        /** @hidden */
        _activeParticles: PerfCounter;
        /** @hidden */
        _activeBones: PerfCounter;
        private _animationRatio;
        private _animationTimeLast;
        private _animationTime;
        /**
         * Gets or sets a general scale for animation speed
         * @see https://www.babylonjs-playground.com/#IBU2W7#3
         */
        animationTimeScale: number;
        /** @hidden */
        _cachedMaterial: Nullable<Material>;
        /** @hidden */
        _cachedEffect: Nullable<Effect>;
        /** @hidden */
        _cachedVisibility: Nullable<number>;
        private _renderId;
        private _executeWhenReadyTimeoutId;
        private _intermediateRendering;
        private _viewUpdateFlag;
        private _projectionUpdateFlag;
        private _alternateViewUpdateFlag;
        private _alternateProjectionUpdateFlag;
        /** @hidden */
        _toBeDisposed: SmartArray<Nullable<IDisposable>>;
        private _activeRequests;
        private _pendingData;
        private _isDisposed;
        /**
         * Gets or sets a boolean indicating that all submeshes of active meshes must be rendered
         * Use this boolean to avoid computing frustum clipping on submeshes (This could help when you are CPU bound)
         */
        dispatchAllSubMeshesOfActiveMeshes: boolean;
        private _activeMeshes;
        private _processedMaterials;
        private _renderTargets;
        /** @hidden */
        _activeParticleSystems: SmartArray<IParticleSystem>;
        private _activeSkeletons;
        private _softwareSkinnedMeshes;
        private _renderingManager;
        private _physicsEngine;
        /** @hidden */
        _activeAnimatables: Animatable[];
        private _transformMatrix;
        private _sceneUbo;
        private _alternateSceneUbo;
        private _pickWithRayInverseMatrix;
        private _boundingBoxRenderer;
        private _outlineRenderer;
        private _viewMatrix;
        private _projectionMatrix;
        private _alternateViewMatrix;
        private _alternateProjectionMatrix;
        private _alternateTransformMatrix;
        private _useAlternateCameraConfiguration;
        private _alternateRendering;
        /** @hidden */
        _forcedViewPosition: Nullable<Vector3>;
        /** @hidden */
        readonly _isAlternateRenderingEnabled: boolean;
        private _frustumPlanes;
        /**
         * Gets the list of frustum planes (built from the active camera)
         */
        readonly frustumPlanes: Plane[];
        /**
         * Gets or sets a boolean indicating if lights must be sorted by priority (off by default)
         * This is useful if there are more lights that the maximum simulteanous authorized
         */
        requireLightSorting: boolean;
        private _selectionOctree;
        private _pointerOverMesh;
        private _pointerOverSprite;
        private _debugLayer;
        private _depthRenderer;
        private _geometryBufferRenderer;
        /**
         * Gets the current geometry buffer associated to the scene.
         */
        /**
         * Sets the current geometry buffer for the scene.
         */
        geometryBufferRenderer: Nullable<GeometryBufferRenderer>;
        private _pickedDownMesh;
        private _pickedUpMesh;
        private _pickedDownSprite;
        private _externalData;
        private _uid;
        /**
         * Creates a new Scene
         * @param engine defines the engine to use to render this scene
         */
        constructor(engine: Engine);
        /**
         * Gets the debug layer associated with the scene
         * @see http://doc.babylonjs.com/features/playground_debuglayer
         */
        readonly debugLayer: DebugLayer;
        /**
         * Gets a boolean indicating if collisions are processed on a web worker
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity#web-worker-based-collision-system-since-21
         */
        workerCollisions: boolean;
        /**
         * Gets the octree used to boost mesh selection (picking)
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene_with_octrees
         */
        readonly selectionOctree: Octree<AbstractMesh>;
        /**
         * Gets the mesh that is currently under the pointer
         */
        readonly meshUnderPointer: Nullable<AbstractMesh>;
        /**
         * Gets the current on-screen X position of the pointer
         */
        readonly pointerX: number;
        /**
         * Gets the current on-screen Y position of the pointer
         */
        readonly pointerY: number;
        /**
         * Gets the cached material (ie. the latest rendered one)
         * @returns the cached material
         */
        getCachedMaterial(): Nullable<Material>;
        /**
         * Gets the cached effect (ie. the latest rendered one)
         * @returns the cached effect
         */
        getCachedEffect(): Nullable<Effect>;
        /**
         * Gets the cached visibility state (ie. the latest rendered one)
         * @returns the cached visibility state
         */
        getCachedVisibility(): Nullable<number>;
        /**
         * Gets a boolean indicating if the current material / effect / visibility must be bind again
         * @param material defines the current material
         * @param effect defines the current effect
         * @param visibility defines the current visibility state
         * @returns true if one parameter is not cached
         */
        isCachedMaterialInvalid(material: Material, effect: Effect, visibility?: number): boolean;
        /**
         * Gets the bounding box renderer associated with the scene
         * @returns a BoundingBoxRenderer
         */
        getBoundingBoxRenderer(): BoundingBoxRenderer;
        /**
         * Gets the outline renderer associated with the scene
         * @returns a OutlineRenderer
         */
        getOutlineRenderer(): OutlineRenderer;
        /**
         * Gets the engine associated with the scene
         * @returns an Engine
         */
        getEngine(): Engine;
        /**
         * Gets the total number of vertices rendered per frame
         * @returns the total number of vertices rendered per frame
         */
        getTotalVertices(): number;
        /**
         * Gets the performance counter for total vertices
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene#instrumentation
         */
        readonly totalVerticesPerfCounter: PerfCounter;
        /**
         * Gets the total number of active indices rendered per frame (You can deduce the number of rendered triangles by dividing this number by 3)
         * @returns the total number of active indices rendered per frame
         */
        getActiveIndices(): number;
        /**
         * Gets the performance counter for active indices
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene#instrumentation
         */
        readonly totalActiveIndicesPerfCounter: PerfCounter;
        /**
         * Gets the total number of active particles rendered per frame
         * @returns the total number of active particles rendered per frame
         */
        getActiveParticles(): number;
        /**
         * Gets the performance counter for active particles
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene#instrumentation
         */
        readonly activeParticlesPerfCounter: PerfCounter;
        /**
         * Gets the total number of active bones rendered per frame
         * @returns the total number of active bones rendered per frame
         */
        getActiveBones(): number;
        /**
         * Gets the performance counter for active bones
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene#instrumentation
         */
        readonly activeBonesPerfCounter: PerfCounter;
        /** @hidden */
        getInterFramePerfCounter(): number;
        /** @hidden */
        readonly interFramePerfCounter: Nullable<PerfCounter>;
        /** @hidden */
        getLastFrameDuration(): number;
        /** @hidden */
        readonly lastFramePerfCounter: Nullable<PerfCounter>;
        /** @hidden */
        getEvaluateActiveMeshesDuration(): number;
        /** @hidden */
        readonly evaluateActiveMeshesDurationPerfCounter: Nullable<PerfCounter>;
        /**
         * Gets the array of active meshes
         * @returns an array of AbstractMesh
         */
        getActiveMeshes(): SmartArray<AbstractMesh>;
        /** @hidden */
        getRenderTargetsDuration(): number;
        /** @hidden */
        getRenderDuration(): number;
        /** @hidden */
        readonly renderDurationPerfCounter: Nullable<PerfCounter>;
        /** @hidden */
        getParticlesDuration(): number;
        /** @hidden */
        readonly particlesDurationPerfCounter: Nullable<PerfCounter>;
        /** @hidden */
        getSpritesDuration(): number;
        /** @hidden */
        readonly spriteDuractionPerfCounter: Nullable<PerfCounter>;
        /**
         * Gets the animation ratio (which is 1.0 is the scene renders at 60fps and 2 if the scene renders at 30fps, etc.)
         * @returns a number
         */
        getAnimationRatio(): number;
        /**
         * Gets an unique Id for the current frame
         * @returns a number
         */
        getRenderId(): number;
        /** Call this function if you want to manually increment the render Id*/
        incrementRenderId(): void;
        private _updatePointerPosition(evt);
        private _createUbo();
        private _createAlternateUbo();
        /**
         * Use this method to simulate a pointer move on a mesh
         * The pickResult parameter can be obtained from a scene.pick or scene.pickWithRay
         * @param pickResult pickingInfo of the object wished to simulate pointer event on
         * @param pointerEventInit pointer event state to be used when simulating the pointer event (eg. pointer id for multitouch)
         * @returns the current scene
         */
        simulatePointerMove(pickResult: PickingInfo, pointerEventInit?: PointerEventInit): Scene;
        private _processPointerMove(pickResult, evt);
        /**
         * Use this method to simulate a pointer down on a mesh
         * The pickResult parameter can be obtained from a scene.pick or scene.pickWithRay
         * @param pickResult pickingInfo of the object wished to simulate pointer event on
         * @param pointerEventInit pointer event state to be used when simulating the pointer event (eg. pointer id for multitouch)
         * @returns the current scene
         */
        simulatePointerDown(pickResult: PickingInfo, pointerEventInit?: PointerEventInit): Scene;
        private _processPointerDown(pickResult, evt);
        /**
         * Use this method to simulate a pointer up on a mesh
         * The pickResult parameter can be obtained from a scene.pick or scene.pickWithRay
         * @param pickResult pickingInfo of the object wished to simulate pointer event on
         * @param pointerEventInit pointer event state to be used when simulating the pointer event (eg. pointer id for multitouch)
         * @returns the current scene
         */
        simulatePointerUp(pickResult: PickingInfo, pointerEventInit?: PointerEventInit): Scene;
        private _processPointerUp(pickResult, evt, clickInfo);
        /**
        * Attach events to the canvas (To handle actionManagers triggers and raise onPointerMove, onPointerDown and onPointerUp
        * @param attachUp defines if you want to attach events to pointerup
        * @param attachDown defines if you want to attach events to pointerdown
        * @param attachMove defines if you want to attach events to pointermove
        */
        attachControl(attachUp?: boolean, attachDown?: boolean, attachMove?: boolean): void;
        /** Detaches all event handlers*/
        detachControl(): void;
        /**
         * This function will check if the scene can be rendered (textures are loaded, shaders are compiled)
         * Delay loaded resources are not taking in account
         * @return true if all required resources are ready
         */
        isReady(): boolean;
        /** Resets all cached information relative to material (including effect and visibility) */
        resetCachedMaterial(): void;
        /**
         * Registers a function to be called before every frame render
         * @param func defines the function to register
         */
        registerBeforeRender(func: () => void): void;
        /**
         * Unregisters a function called before every frame render
         * @param func defines the function to unregister
         */
        unregisterBeforeRender(func: () => void): void;
        /**
         * Registers a function to be called after every frame render
         * @param func defines the function to register
         */
        registerAfterRender(func: () => void): void;
        /**
         * Unregisters a function called after every frame render
         * @param func defines the function to unregister
         */
        unregisterAfterRender(func: () => void): void;
        private _executeOnceBeforeRender(func);
        /**
         * The provided function will run before render once and will be disposed afterwards.
         * A timeout delay can be provided so that the function will be executed in N ms.
         * The timeout is using the browser's native setTimeout so time percision cannot be guaranteed.
         * @param func The function to be executed.
         * @param timeout optional delay in ms
         */
        executeOnceBeforeRender(func: () => void, timeout?: number): void;
        /** @hidden */
        _addPendingData(data: any): void;
        /** @hidden */
        _removePendingData(data: any): void;
        /**
         * Returns the number of items waiting to be loaded
         * @returns the number of items waiting to be loaded
         */
        getWaitingItemsCount(): number;
        /**
         * Returns a boolean indicating if the scene is still loading data
         */
        readonly isLoading: boolean;
        /**
         * Registers a function to be executed when the scene is ready
         * @param {Function} func - the function to be executed
         */
        executeWhenReady(func: () => void): void;
        /**
         * Returns a promise that resolves when the scene is ready
         * @returns A promise that resolves when the scene is ready
         */
        whenReadyAsync(): Promise<void>;
        /** @hidden */
        _checkIsReady(): void;
        /**
         * Will start the animation sequence of a given target
         * @param target defines the target
         * @param from defines from which frame should animation start
         * @param to defines until which frame should animation run.
         * @param weight defines the weight to apply to the animation (1.0 by default)
         * @param loop defines if the animation loops
         * @param speedRatio defines the speed in which to run the animation (1.0 by default)
         * @param onAnimationEnd defines the function to be executed when the animation ends
         * @param animatable defines an animatable object. If not provided a new one will be created from the given params
         * @returns the animatable object created for this animation
         */
        beginWeightedAnimation(target: any, from: number, to: number, weight?: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable): Animatable;
        /**
         * Will start the animation sequence of a given target
         * @param target defines the target
         * @param from defines from which frame should animation start
         * @param to defines until which frame should animation run.
         * @param loop defines if the animation loops
         * @param speedRatio defines the speed in which to run the animation (1.0 by default)
         * @param onAnimationEnd defines the function to be executed when the animation ends
         * @param animatable defines an animatable object. If not provided a new one will be created from the given params
         * @param stopCurrent defines if the current animations must be stopped first (true by default)
         * @returns the animatable object created for this animation
         */
        beginAnimation(target: any, from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable, stopCurrent?: boolean): Animatable;
        /**
         * Begin a new animation on a given node
         * @param target defines the target where the animation will take place
         * @param animations defines the list of animations to start
         * @param from defines the initial value
         * @param to defines the final value
         * @param loop defines if you want animation to loop (off by default)
         * @param speedRatio defines the speed ratio to apply to all animations
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of created animatables
         */
        beginDirectAnimation(target: any, animations: Animation[], from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Animatable;
        /**
         * Begin a new animation on a given node and its hierarchy
         * @param target defines the root node where the animation will take place
         * @param directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used.
         * @param animations defines the list of animations to start
         * @param from defines the initial value
         * @param to defines the final value
         * @param loop defines if you want animation to loop (off by default)
         * @param speedRatio defines the speed ratio to apply to all animations
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of animatables created for all nodes
         */
        beginDirectHierarchyAnimation(target: Node, directDescendantsOnly: boolean, animations: Animation[], from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Animatable[];
        /**
         * Gets the animatable associated with a specific target
         * @param target defines the target of the animatable
         * @returns the required animatable if found
         */
        getAnimatableByTarget(target: any): Nullable<Animatable>;
        /**
         * Gets all animatables associated with a given target
         * @param target defines the target to look animatables for
         * @returns an array of Animatables
         */
        getAllAnimatablesByTarget(target: any): Array<Animatable>;
        /**
         * Gets all animatable attached to the scene
         */
        readonly animatables: Animatable[];
        /**
         * Will stop the animation of the given target
         * @param target - the target
         * @param animationName - the name of the animation to stop (all animations will be stopped if empty)
         */
        stopAnimation(target: any, animationName?: string): void;
        /**
         * Stops and removes all animations that have been applied to the scene
         */
        stopAllAnimations(): void;
        private _animate();
        /** @hidden */
        _registerTargetForLateAnimationBinding(runtimeAnimation: RuntimeAnimation, originalValue: any): void;
        private _processLateAnimationBindingsForMatrices(holder);
        private _processLateAnimationBindingsForQuaternions(holder);
        private _processLateAnimationBindings();
        /** @hidden */
        _switchToAlternateCameraConfiguration(active: boolean): void;
        /**
         * Gets the current view matrix
         * @returns a Matrix
         */
        getViewMatrix(): Matrix;
        /**
         * Gets the current projection matrix
         * @returns a Matrix
         */
        getProjectionMatrix(): Matrix;
        /**
         * Gets the current transform matrix
         * @returns a Matrix made of View * Projection
         */
        getTransformMatrix(): Matrix;
        /**
         * Sets the current transform matrix
         * @param view defines the View matrix to use
         * @param projection defines the Projection matrix to use
         */
        setTransformMatrix(view: Matrix, projection: Matrix): void;
        /** @hidden */
        _setAlternateTransformMatrix(view: Matrix, projection: Matrix): void;
        /**
         * Gets the uniform buffer used to store scene data
         * @returns a UniformBuffer
         */
        getSceneUniformBuffer(): UniformBuffer;
        /**
         * Gets an unique (relatively to the current scene) Id
         * @returns an unique number for the scene
         */
        getUniqueId(): number;
        /**
         * Add a mesh to the list of scene's meshes
         * @param newMesh defines the mesh to add
         */
        addMesh(newMesh: AbstractMesh): void;
        /**
           * Remove a mesh for the list of scene's meshes
           * @param toRemove defines the mesh to remove
           * @param recursive if all child meshes should also be removed from the scene
           * @returns the index where the mesh was in the mesh list
           */
        removeMesh(toRemove: AbstractMesh, recursive?: boolean): number;
        /**
         * Add a transform node to the list of scene's transform nodes
         * @param newTransformNode defines the transform node to add
         */
        addTransformNode(newTransformNode: TransformNode): void;
        /**
         * Remove a transform node for the list of scene's transform nodes
         * @param toRemove defines the transform node to remove
         * @returns the index where the transform node was in the transform node list
         */
        removeTransformNode(toRemove: TransformNode): number;
        /**
         * Remove a skeleton for the list of scene's skeletons
         * @param toRemove defines the skeleton to remove
         * @returns the index where the skeleton was in the skeleton list
         */
        removeSkeleton(toRemove: Skeleton): number;
        /**
         * Remove a morph target for the list of scene's morph targets
         * @param toRemove defines the morph target to remove
         * @returns the index where the morph target was in the morph target list
         */
        removeMorphTargetManager(toRemove: MorphTargetManager): number;
        /**
         * Remove a light for the list of scene's lights
         * @param toRemove defines the light to remove
         * @returns the index where the light was in the light list
         */
        removeLight(toRemove: Light): number;
        /**
         * Remove a camera for the list of scene's cameras
         * @param toRemove defines the camera to remove
         * @returns the index where the camera was in the camera list
         */
        removeCamera(toRemove: Camera): number;
        /**
         * Remove a particle system for the list of scene's particle systems
         * @param toRemove defines the particle system to remove
         * @returns the index where the particle system was in the particle system list
         */
        removeParticleSystem(toRemove: IParticleSystem): number;
        /**
         * Remove a animation for the list of scene's animations
         * @param toRemove defines the animation to remove
         * @returns the index where the animation was in the animation list
         */
        removeAnimation(toRemove: Animation): number;
        /**
         * Removes the given animation group from this scene.
         * @param toRemove The animation group to remove
         * @returns The index of the removed animation group
         */
        removeAnimationGroup(toRemove: AnimationGroup): number;
        /**
         * Removes the given multi-material from this scene.
         * @param toRemove The multi-material to remove
         * @returns The index of the removed multi-material
         */
        removeMultiMaterial(toRemove: MultiMaterial): number;
        /**
         * Removes the given material from this scene.
         * @param toRemove The material to remove
         * @returns The index of the removed material
         */
        removeMaterial(toRemove: Material): number;
        /**
         * Removes the given lens flare system from this scene.
         * @param toRemove The lens flare system to remove
         * @returns The index of the removed lens flare system
         */
        removeLensFlareSystem(toRemove: LensFlareSystem): number;
        /**
         * Removes the given action manager from this scene.
         * @param toRemove The action manager to remove
         * @returns The index of the removed action manager
         */
        removeActionManager(toRemove: ActionManager): number;
        /**
         * Removes the given effect layer from this scene.
         * @param toRemove defines the effect layer to remove
         * @returns the index of the removed effect layer
         */
        removeEffectLayer(toRemove: EffectLayer): number;
        /**
         * Removes the given texture from this scene.
         * @param toRemove The texture to remove
         * @returns The index of the removed texture
         */
        removeTexture(toRemove: BaseTexture): number;
        /**
         * Adds the given light to this scene
         * @param newLight The light to add
         */
        addLight(newLight: Light): void;
        /**
         * Sorts the list list based on light priorities
         */
        sortLightsByPriority(): void;
        /**
         * Adds the given camera to this scene
         * @param newCamera The camera to add
         */
        addCamera(newCamera: Camera): void;
        /**
         * Adds the given skeleton to this scene
         * @param newSkeleton The skeleton to add
         */
        addSkeleton(newSkeleton: Skeleton): void;
        /**
         * Adds the given particle system to this scene
         * @param newParticleSystem The particle system to add
         */
        addParticleSystem(newParticleSystem: IParticleSystem): void;
        /**
         * Adds the given animation to this scene
         * @param newAnimation The animation to add
         */
        addAnimation(newAnimation: Animation): void;
        /**
         * Adds the given animation group to this scene.
         * @param newAnimationGroup The animation group to add
         */
        addAnimationGroup(newAnimationGroup: AnimationGroup): void;
        /**
         * Adds the given multi-material to this scene
         * @param newMultiMaterial The multi-material to add
         */
        addMultiMaterial(newMultiMaterial: MultiMaterial): void;
        /**
         * Adds the given material to this scene
         * @param newMaterial The material to add
         */
        addMaterial(newMaterial: Material): void;
        /**
         * Adds the given morph target to this scene
         * @param newMorphTargetManager The morph target to add
         */
        addMorphTargetManager(newMorphTargetManager: MorphTargetManager): void;
        /**
         * Adds the given geometry to this scene
         * @param newGeometry The geometry to add
         */
        addGeometry(newGeometry: Geometry): void;
        /**
         * Adds the given lens flare system to this scene
         * @param newLensFlareSystem The lens flare system to add
         */
        addLensFlareSystem(newLensFlareSystem: LensFlareSystem): void;
        /**
         * Adds the given effect layer to this scene
         * @param newEffectLayer defines the effect layer to add
         */
        addEffectLayer(newEffectLayer: EffectLayer): void;
        /**
         * Adds the given action manager to this scene
         * @param newActionManager The action manager to add
         */
        addActionManager(newActionManager: ActionManager): void;
        /**
         * Adds the given texture to this scene.
         * @param newTexture The texture to add
         */
        addTexture(newTexture: BaseTexture): void;
        /**
         * Switch active camera
         * @param newCamera defines the new active camera
         * @param attachControl defines if attachControl must be called for the new active camera (default: true)
         */
        switchActiveCamera(newCamera: Camera, attachControl?: boolean): void;
        /**
         * sets the active camera of the scene using its ID
         * @param id defines the camera's ID
         * @return the new active camera or null if none found.
         */
        setActiveCameraByID(id: string): Nullable<Camera>;
        /**
         * sets the active camera of the scene using its name
         * @param name defines the camera's name
         * @returns the new active camera or null if none found.
         */
        setActiveCameraByName(name: string): Nullable<Camera>;
        /**
         * get an animation group using its name
         * @param name defines the material's name
         * @return the animation group or null if none found.
         */
        getAnimationGroupByName(name: string): Nullable<AnimationGroup>;
        /**
         * get a material using its id
         * @param id defines the material's ID
         * @return the material or null if none found.
         */
        getMaterialByID(id: string): Nullable<Material>;
        /**
         * Gets a material using its name
         * @param name defines the material's name
         * @return the material or null if none found.
         */
        getMaterialByName(name: string): Nullable<Material>;
        /**
         * Gets a lens flare system using its name
         * @param name defines the name to look for
         * @returns the lens flare system or null if not found
         */
        getLensFlareSystemByName(name: string): Nullable<LensFlareSystem>;
        /**
         * Gets a lens flare system using its id
         * @param id defines the id to look for
         * @returns the lens flare system or null if not found
         */
        getLensFlareSystemByID(id: string): Nullable<LensFlareSystem>;
        /**
         * Gets a camera using its id
         * @param id defines the id to look for
         * @returns the camera or null if not found
         */
        getCameraByID(id: string): Nullable<Camera>;
        /**
         * Gets a camera using its unique id
         * @param uniqueId defines the unique id to look for
         * @returns the camera or null if not found
         */
        getCameraByUniqueID(uniqueId: number): Nullable<Camera>;
        /**
         * Gets a camera using its name
         * @param name defines the camera's name
         * @return the camera or null if none found.
         */
        getCameraByName(name: string): Nullable<Camera>;
        /**
         * Gets a bone using its id
         * @param id defines the bone's id
         * @return the bone or null if not found
         */
        getBoneByID(id: string): Nullable<Bone>;
        /**
        * Gets a bone using its id
        * @param name defines the bone's name
        * @return the bone or null if not found
        */
        getBoneByName(name: string): Nullable<Bone>;
        /**
         * Gets a light node using its name
         * @param name defines the the light's name
         * @return the light or null if none found.
         */
        getLightByName(name: string): Nullable<Light>;
        /**
         * Gets a light node using its id
         * @param id defines the light's id
         * @return the light or null if none found.
         */
        getLightByID(id: string): Nullable<Light>;
        /**
         * Gets a light node using its scene-generated unique ID
         * @param uniqueId defines the light's unique id
         * @return the light or null if none found.
         */
        getLightByUniqueID(uniqueId: number): Nullable<Light>;
        /**
         * Gets a particle system by id
         * @param id defines the particle system id
         * @return the corresponding system or null if none found
         */
        getParticleSystemByID(id: string): Nullable<IParticleSystem>;
        /**
         * Gets a geometry using its ID
         * @param id defines the geometry's id
         * @return the geometry or null if none found.
         */
        getGeometryByID(id: string): Nullable<Geometry>;
        /**
         * Add a new geometry to this scene
         * @param geometry defines the geometry to be added to the scene.
         * @param force defines if the geometry must be pushed even if a geometry with this id already exists
         * @return a boolean defining if the geometry was added or not
         */
        pushGeometry(geometry: Geometry, force?: boolean): boolean;
        /**
         * Removes an existing geometry
         * @param geometry defines the geometry to be removed from the scene
         * @return a boolean defining if the geometry was removed or not
         */
        removeGeometry(geometry: Geometry): boolean;
        /**
         * Gets the list of geometries attached to the scene
         * @returns an array of Geometry
         */
        getGeometries(): Geometry[];
        /**
         * Gets the first added mesh found of a given ID
         * @param id defines the id to search for
         * @return the mesh found or null if not found at all
         */
        getMeshByID(id: string): Nullable<AbstractMesh>;
        /**
         * Gets a list of meshes using their id
         * @param id defines the id to search for
         * @returns a list of meshes
         */
        getMeshesByID(id: string): Array<AbstractMesh>;
        /**
         * Gets the first added transform node found of a given ID
         * @param id defines the id to search for
         * @return the found transform node or null if not found at all.
         */
        getTransformNodeByID(id: string): Nullable<TransformNode>;
        /**
         * Gets a list of transform nodes using their id
         * @param id defines the id to search for
         * @returns a list of transform nodes
         */
        getTransformNodesByID(id: string): Array<TransformNode>;
        /**
         * Gets a mesh with its auto-generated unique id
         * @param uniqueId defines the unique id to search for
         * @return the found mesh or null if not found at all.
         */
        getMeshByUniqueID(uniqueId: number): Nullable<AbstractMesh>;
        /**
         * Gets a the last added mesh using a given id
         * @param id defines the id to search for
         * @return the found mesh or null if not found at all.
         */
        getLastMeshByID(id: string): Nullable<AbstractMesh>;
        /**
         * Gets a the last added node (Mesh, Camera, Light) using a given id
         * @param id defines the id to search for
         * @return the found node or null if not found at all
         */
        getLastEntryByID(id: string): Nullable<Node>;
        /**
         * Gets a node (Mesh, Camera, Light) using a given id
         * @param id defines the id to search for
         * @return the found node or null if not found at all
         */
        getNodeByID(id: string): Nullable<Node>;
        /**
         * Gets a node (Mesh, Camera, Light) using a given name
         * @param name defines the name to search for
         * @return the found node or null if not found at all.
         */
        getNodeByName(name: string): Nullable<Node>;
        /**
         * Gets a mesh using a given name
         * @param name defines the name to search for
         * @return the found mesh or null if not found at all.
         */
        getMeshByName(name: string): Nullable<AbstractMesh>;
        /**
         * Gets a transform node using a given name
         * @param name defines the name to search for
         * @return the found transform node or null if not found at all.
         */
        getTransformNodeByName(name: string): Nullable<TransformNode>;
        /**
         * Gets a sound using a given name
         * @param name defines the name to search for
         * @return the found sound or null if not found at all.
         */
        getSoundByName(name: string): Nullable<Sound>;
        /**
         * Gets a skeleton using a given id (if many are found, this function will pick the last one)
         * @param id defines the id to search for
         * @return the found skeleton or null if not found at all.
         */
        getLastSkeletonByID(id: string): Nullable<Skeleton>;
        /**
         * Gets a skeleton using a given id (if many are found, this function will pick the first one)
         * @param id defines the id to search for
         * @return the found skeleton or null if not found at all.
         */
        getSkeletonById(id: string): Nullable<Skeleton>;
        /**
         * Gets a skeleton using a given name
         * @param name defines the name to search for
         * @return the found skeleton or null if not found at all.
         */
        getSkeletonByName(name: string): Nullable<Skeleton>;
        /**
         * Gets a morph target manager  using a given id (if many are found, this function will pick the last one)
         * @param id defines the id to search for
         * @return the found morph target manager or null if not found at all.
         */
        getMorphTargetManagerById(id: number): Nullable<MorphTargetManager>;
        /**
         * Gets a boolean indicating if the given mesh is active
         * @param mesh defines the mesh to look for
         * @returns true if the mesh is in the active list
         */
        isActiveMesh(mesh: AbstractMesh): boolean;
        /**
         * Return a the first highlight layer of the scene with a given name.
         * @param name The name of the highlight layer to look for.
         * @return The highlight layer if found otherwise null.
         */
        getHighlightLayerByName(name: string): Nullable<HighlightLayer>;
        /**
         * Return a the first highlight layer of the scene with a given name.
         * @param name The name of the highlight layer to look for.
         * @return The highlight layer if found otherwise null.
         */
        getGlowLayerByName(name: string): Nullable<GlowLayer>;
        /**
         * Return a unique id as a string which can serve as an identifier for the scene
         */
        readonly uid: string;
        /**
         * Add an externaly attached data from its key.
         * This method call will fail and return false, if such key already exists.
         * If you don't care and just want to get the data no matter what, use the more convenient getOrAddExternalDataWithFactory() method.
         * @param key the unique key that identifies the data
         * @param data the data object to associate to the key for this Engine instance
         * @return true if no such key were already present and the data was added successfully, false otherwise
         */
        addExternalData<T>(key: string, data: T): boolean;
        /**
         * Get an externaly attached data from its key
         * @param key the unique key that identifies the data
         * @return the associated data, if present (can be null), or undefined if not present
         */
        getExternalData<T>(key: string): Nullable<T>;
        /**
         * Get an externaly attached data from its key, create it using a factory if it's not already present
         * @param key the unique key that identifies the data
         * @param factory the factory that will be called to create the instance if and only if it doesn't exists
         * @return the associated data, can be null if the factory returned null.
         */
        getOrAddExternalDataWithFactory<T>(key: string, factory: (k: string) => T): T;
        /**
         * Remove an externaly attached data from the Engine instance
         * @param key the unique key that identifies the data
         * @return true if the data was successfully removed, false if it doesn't exist
         */
        removeExternalData(key: string): boolean;
        private _evaluateSubMesh(subMesh, mesh);
        /**
         * Clear the processed materials smart array preventing retention point in material dispose.
         */
        freeProcessedMaterials(): void;
        /**
         * Clear the active meshes smart array preventing retention point in mesh dispose.
         */
        freeActiveMeshes(): void;
        /**
         * Clear the info related to rendering groups preventing retention points during dispose.
         */
        freeRenderingGroups(): void;
        /** @hidden */
        _isInIntermediateRendering(): boolean;
        private _activeMeshCandidateProvider;
        /**
         * Defines the current active mesh candidate provider
         * @param provider defines the provider to use
         */
        setActiveMeshCandidateProvider(provider: IActiveMeshCandidateProvider): void;
        /**
         * Gets the current active mesh candidate provider
         * @returns the current active mesh candidate provider
         */
        getActiveMeshCandidateProvider(): IActiveMeshCandidateProvider;
        private _activeMeshesFrozen;
        /**
         * Use this function to stop evaluating active meshes. The current list will be keep alive between frames
         * @returns the current scene
         */
        freezeActiveMeshes(): Scene;
        /**
         * Use this function to restart evaluating active meshes on every frame
         * @returns the current scene
         */
        unfreezeActiveMeshes(): Scene;
        private _evaluateActiveMeshes();
        private _activeMesh(sourceMesh, mesh);
        /**
         * Update the transform matrix to update from the current active camera
         * @param force defines a boolean used to force the update even if cache is up to date
         */
        updateTransformMatrix(force?: boolean): void;
        /**
         * Defines an alternate camera (used mostly in VR-like scenario where two cameras can render the same scene from a slightly different point of view)
         * @param alternateCamera defines the camera to use
         */
        updateAlternateTransformMatrix(alternateCamera: Camera): void;
        private _renderForCamera(camera, rigParent?);
        private _processSubCameras(camera);
        private _checkIntersections();
        /**
         * Render the scene
         */
        render(): void;
        private _updateAudioParameters();
        /**
         * Gets or sets if audio support is enabled
         * @see http://doc.babylonjs.com/how_to/playing_sounds_and_music
         */
        audioEnabled: boolean;
        private _disableAudio();
        private _enableAudio();
        /**
         * Gets or sets if audio will be output to headphones
         * @see http://doc.babylonjs.com/how_to/playing_sounds_and_music
         */
        headphone: boolean;
        private _switchAudioModeForHeadphones();
        private _switchAudioModeForNormalSpeakers();
        /**
         * Creates a depth renderer a given camera which contains a depth map which can be used for post processing.
         * @param camera The camera to create the depth renderer on (default: scene's active camera)
         * @returns the created depth renderer
         */
        enableDepthRenderer(camera?: Nullable<Camera>): DepthRenderer;
        /**
         * Disables a depth renderer for a given camera
         * @param camera The camera to disable the depth renderer on (default: scene's active camera)
         */
        disableDepthRenderer(camera?: Nullable<Camera>): void;
        /**
         * Enables a GeometryBufferRender and associates it with the scene
         * @param ratio defines the scaling ratio to apply to the renderer (1 by default which means same resolution)
         * @returns the GeometryBufferRenderer
         */
        enableGeometryBufferRenderer(ratio?: number): Nullable<GeometryBufferRenderer>;
        /**
         * Disables the GeometryBufferRender associated with the scene
         */
        disableGeometryBufferRenderer(): void;
        /**
         * Freeze all materials
         * A frozen material will not be updatable but should be faster to render
         */
        freezeMaterials(): void;
        /**
         * Unfreeze all materials
         * A frozen material will not be updatable but should be faster to render
         */
        unfreezeMaterials(): void;
        /**
         * Releases all held ressources
         */
        dispose(): void;
        /**
         * Gets if the scene is already disposed
         */
        readonly isDisposed: boolean;
        /**
         *  Releases sounds & soundtracks
         */
        disposeSounds(): void;
        /**
         * Get the world extend vectors with an optional filter
         *
         * @param filterPredicate the predicate - which meshes should be included when calculating the world size
         * @returns {{ min: Vector3; max: Vector3 }} min and max vectors
         */
        getWorldExtends(filterPredicate?: (mesh: AbstractMesh) => boolean): {
            min: Vector3;
            max: Vector3;
        };
        /**
         * Creates or updates the octree used to boost selection (picking)
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene_with_octrees
         * @param maxCapacity defines the maximum capacity per leaf
         * @param maxDepth defines the maximum depth of the octree
         * @returns an octree of AbstractMesh
         */
        createOrUpdateSelectionOctree(maxCapacity?: number, maxDepth?: number): Octree<AbstractMesh>;
        /**
         * Creates a ray that can be used to pick in the scene
         * @param x defines the x coordinate of the origin (on-screen)
         * @param y defines the y coordinate of the origin (on-screen)
         * @param world defines the world matrix to use if you want to pick in object space (instead of world space)
         * @param camera defines the camera to use for the picking
         * @param cameraViewSpace defines if picking will be done in view space (false by default)
         * @returns a Ray
         */
        createPickingRay(x: number, y: number, world: Matrix, camera: Nullable<Camera>, cameraViewSpace?: boolean): Ray;
        /**
         * Creates a ray that can be used to pick in the scene
         * @param x defines the x coordinate of the origin (on-screen)
         * @param y defines the y coordinate of the origin (on-screen)
         * @param world defines the world matrix to use if you want to pick in object space (instead of world space)
         * @param result defines the ray where to store the picking ray
         * @param camera defines the camera to use for the picking
         * @param cameraViewSpace defines if picking will be done in view space (false by default)
         * @returns the current scene
         */
        createPickingRayToRef(x: number, y: number, world: Matrix, result: Ray, camera: Nullable<Camera>, cameraViewSpace?: boolean): Scene;
        /**
         * Creates a ray that can be used to pick in the scene
         * @param x defines the x coordinate of the origin (on-screen)
         * @param y defines the y coordinate of the origin (on-screen)
         * @param camera defines the camera to use for the picking
         * @returns a Ray
         */
        createPickingRayInCameraSpace(x: number, y: number, camera?: Camera): Ray;
        /**
         * Creates a ray that can be used to pick in the scene
         * @param x defines the x coordinate of the origin (on-screen)
         * @param y defines the y coordinate of the origin (on-screen)
         * @param result defines the ray where to store the picking ray
         * @param camera defines the camera to use for the picking
         * @returns the current scene
         */
        createPickingRayInCameraSpaceToRef(x: number, y: number, result: Ray, camera?: Camera): Scene;
        private _internalPick(rayFunction, predicate?, fastCheck?);
        private _internalMultiPick(rayFunction, predicate?);
        private _internalPickSprites(ray, predicate?, fastCheck?, camera?);
        private _tempPickingRay;
        /** Launch a ray to try to pick a mesh in the scene
         * @param x position on screen
         * @param y position on screen
         * @param predicate Predicate function used to determine eligible meshes. Can be set to null. In this case, a mesh must be enabled, visible and with isPickable set to true
         * @param fastCheck Launch a fast check only using the bounding boxes. Can be set to null.
         * @param camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
         * @returns a PickingInfo
         */
        pick(x: number, y: number, predicate?: (mesh: AbstractMesh) => boolean, fastCheck?: boolean, camera?: Nullable<Camera>): Nullable<PickingInfo>;
        /** Launch a ray to try to pick a sprite in the scene
         * @param x position on screen
         * @param y position on screen
         * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
         * @param fastCheck Launch a fast check only using the bounding boxes. Can be set to null.
         * @param camera camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
         * @returns a PickingInfo
         */
        pickSprite(x: number, y: number, predicate?: (sprite: Sprite) => boolean, fastCheck?: boolean, camera?: Camera): Nullable<PickingInfo>;
        private _cachedRayForTransform;
        /** Use the given ray to pick a mesh in the scene
         * @param ray The ray to use to pick meshes
         * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
         * @param fastCheck Launch a fast check only using the bounding boxes. Can be set to null
         * @returns a PickingInfo
         */
        pickWithRay(ray: Ray, predicate: (mesh: AbstractMesh) => boolean, fastCheck?: boolean): Nullable<PickingInfo>;
        /**
         * Launch a ray to try to pick a mesh in the scene
         * @param x X position on screen
         * @param y Y position on screen
         * @param predicate Predicate function used to determine eligible meshes. Can be set to null. In this case, a mesh must be enabled, visible and with isPickable set to true
         * @param camera camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
         * @returns an array of PickingInfo
         */
        multiPick(x: number, y: number, predicate?: (mesh: AbstractMesh) => boolean, camera?: Camera): Nullable<PickingInfo[]>;
        /**
         * Launch a ray to try to pick a mesh in the scene
         * @param ray Ray to use
         * @param predicate Predicate function used to determine eligible meshes. Can be set to null. In this case, a mesh must be enabled, visible and with isPickable set to true
         * @returns an array of PickingInfo
         */
        multiPickWithRay(ray: Ray, predicate: (mesh: AbstractMesh) => boolean): Nullable<PickingInfo[]>;
        /**
         * Force the value of meshUnderPointer
         * @param mesh defines the mesh to use
         */
        setPointerOverMesh(mesh: Nullable<AbstractMesh>): void;
        /**
         * Gets the mesh under the pointer
         * @returns a Mesh or null if no mesh is under the pointer
         */
        getPointerOverMesh(): Nullable<AbstractMesh>;
        /**
         * Force the sprite under the pointer
         * @param sprite defines the sprite to use
         */
        setPointerOverSprite(sprite: Nullable<Sprite>): void;
        /**
         * Gets the sprite under the pointer
         * @returns a Sprite or null if no sprite is under the pointer
         */
        getPointerOverSprite(): Nullable<Sprite>;
        /**
         * Gets the current physics engine
         * @returns a PhysicsEngine or null if none attached
         */
        getPhysicsEngine(): Nullable<PhysicsEngine>;
        /**
         * Enables physics to the current scene
         * @param gravity defines the scene's gravity for the physics engine
         * @param plugin defines the physics engine to be used. defaults to OimoJS.
         * @return a boolean indicating if the physics engine was initialized
         */
        enablePhysics(gravity?: Nullable<Vector3>, plugin?: IPhysicsEnginePlugin): boolean;
        /**
         * Disables and disposes the physics engine associated with the scene
         */
        disablePhysicsEngine(): void;
        /**
         * Gets a boolean indicating if there is an active physics engine
         * @returns a boolean indicating if there is an active physics engine
         */
        isPhysicsEnabled(): boolean;
        /**
         * Deletes a physics compound impostor
         * @param compound defines the compound to delete
         */
        deleteCompoundImpostor(compound: any): void;
        /** @hidden */
        _rebuildGeometries(): void;
        /** @hidden */
        _rebuildTextures(): void;
        /**
         * Creates a default light for the scene.
         * @param replace Whether to replace the existing lights in the scene.
         */
        createDefaultLight(replace?: boolean): void;
        /**
         * Creates a default camera for the scene.
         * @param createArcRotateCamera Whether to create an arc rotate or a free camera.
         * @param replace Whether to replace the existing active camera in the scene.
         * @param attachCameraControls Whether to attach camera controls to the canvas.
         */
        createDefaultCamera(createArcRotateCamera?: boolean, replace?: boolean, attachCameraControls?: boolean): void;
        /**
         * Creates a default camera and a default light
         * @param createArcRotateCamera defines that the camera will be an ArcRotateCamera
         * @param replace defines if the camera and/or light will replace the existing ones
         * @param attachCameraControls defines if attachControl will be called on the new camera
         */
        createDefaultCameraOrLight(createArcRotateCamera?: boolean, replace?: boolean, attachCameraControls?: boolean): void;
        /**
         * Creates a new sky box
         * @see http://doc.babylonjs.com/babylon101/environment#skybox
         * @param environmentTexture defines the texture to use as environment texture
         * @param pbr defines if PBRMaterial must be used instead of StandardMaterial
         * @param scale defines the overall scale of the skybox
         * @param blur defines if blurring must be applied to the environment texture (works only with pbr === true)
         * @param setGlobalEnvTexture defines a boolean indicating that scene.environmentTexture must match the current skybox texture (true by default)
         * @returns a new mesh holding the sky box
         */
        createDefaultSkybox(environmentTexture?: BaseTexture, pbr?: boolean, scale?: number, blur?: number, setGlobalEnvTexture?: boolean): Nullable<Mesh>;
        /**
         * Creates a new environment
         * @see http://doc.babylonjs.com/babylon101/environment#skybox
         * @param options defines the options you can use to configure the environment
         * @returns the new EnvironmentHelper
         */
        createDefaultEnvironment(options: Partial<IEnvironmentHelperOptions>): Nullable<EnvironmentHelper>;
        /**
         * Creates a new VREXperienceHelper
         * @see http://doc.babylonjs.com/how_to/webvr_helper
         * @param webVROptions defines the options used to create the new VREXperienceHelper
         * @returns a new VREXperienceHelper
         */
        createDefaultVRExperience(webVROptions?: VRExperienceHelperOptions): VRExperienceHelper;
        private _getByTags(list, tagsQuery, forEach?);
        /**
         * Get a list of meshes by tags
         * @param tagsQuery defines the tags query to use
         * @param forEach defines a predicate used to filter results
         * @returns an array of Mesh
         */
        getMeshesByTags(tagsQuery: string, forEach?: (mesh: AbstractMesh) => void): Mesh[];
        /**
         * Get a list of cameras by tags
         * @param tagsQuery defines the tags query to use
         * @param forEach defines a predicate used to filter results
         * @returns an array of Camera
         */
        getCamerasByTags(tagsQuery: string, forEach?: (camera: Camera) => void): Camera[];
        /**
         * Get a list of lights by tags
         * @param tagsQuery defines the tags query to use
         * @param forEach defines a predicate used to filter results
         * @returns an array of Light
         */
        getLightsByTags(tagsQuery: string, forEach?: (light: Light) => void): Light[];
        /**
         * Get a list of materials by tags
         * @param tagsQuery defines the tags query to use
         * @param forEach defines a predicate used to filter results
         * @returns an array of Material
         */
        getMaterialByTags(tagsQuery: string, forEach?: (material: Material) => void): Material[];
        /**
         * Overrides the default sort function applied in the renderging group to prepare the meshes.
         * This allowed control for front to back rendering or reversly depending of the special needs.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param opaqueSortCompareFn The opaque queue comparison function use to sort.
         * @param alphaTestSortCompareFn The alpha test queue comparison function use to sort.
         * @param transparentSortCompareFn The transparent queue comparison function use to sort.
         */
        setRenderingOrder(renderingGroupId: number, opaqueSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, alphaTestSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, transparentSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>): void;
        /**
         * Specifies whether or not the stencil and depth buffer are cleared between two rendering groups.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param autoClearDepthStencil Automatically clears depth and stencil between groups if true.
         * @param depth Automatically clears depth between groups if true and autoClear is true.
         * @param stencil Automatically clears stencil between groups if true and autoClear is true.
         */
        setRenderingAutoClearDepthStencil(renderingGroupId: number, autoClearDepthStencil: boolean, depth?: boolean, stencil?: boolean): void;
        /**
         * Will flag all materials as dirty to trigger new shader compilation
         * @param flag defines the flag used to specify which material part must be marked as dirty
         * @param predicate If not null, it will be used to specifiy if a material has to be marked as dirty
         */
        markAllMaterialsAsDirty(flag: number, predicate?: (mat: Material) => boolean): void;
        /** @hidden */
        _loadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (data: any) => void, useDatabase?: boolean, useArrayBuffer?: boolean, onError?: (request?: XMLHttpRequest, exception?: any) => void): IFileRequest;
        /** @hidden */
        _loadFileAsync(url: string, useDatabase?: boolean, useArrayBuffer?: boolean): Promise<string | ArrayBuffer>;
    }
}

declare module BABYLON {
    type Nullable<T> = T | null;
    type float = number;
    type double = number;
    type int = number;
    type FloatArray = number[] | Float32Array;
    type IndicesArray = number[] | Int32Array | Uint32Array | Uint16Array;
    /**
     * Alias for types that can be used by a Buffer or VertexBuffer.
     */
    type DataArray = number[] | ArrayBuffer | ArrayBufferView;
}

declare module BABYLON {
    /**
     * The action to be carried out following a trigger
     * @see http://doc.babylonjs.com/how_to/how_to_use_actions#available-actions
     */
    class Action {
        /** the trigger, with or without parameters, for the action */
        triggerOptions: any;
        /**
         * Trigger for the action
         */
        trigger: number;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        private _nextActiveAction;
        private _child;
        private _condition?;
        private _triggerParameter;
        /**
        * An event triggered prior to action being executed.
        */
        onBeforeExecuteObservable: Observable<Action>;
        /**
         * Creates a new Action
         * @param triggerOptions the trigger, with or without parameters, for the action
         * @param condition an optional determinant of action
         */
        constructor(
            /** the trigger, with or without parameters, for the action */
            triggerOptions: any, condition?: Condition);
        /**
         * Internal only
         * @hidden
         */
        _prepare(): void;
        /**
         * Gets the trigger parameters
         * @returns the trigger parameters
         */
        getTriggerParameter(): any;
        /**
         * Internal only - executes current action event
         * @hidden
         */
        _executeCurrent(evt?: ActionEvent): void;
        /**
         * Execute placeholder for child classes
         * @param evt optional action event
         */
        execute(evt?: ActionEvent): void;
        /**
         * Skips to next active action
         */
        skipToNextActiveAction(): void;
        /**
         * Adds action to chain of actions, may be a DoNothingAction
         * @param action defines the next action to execute
         * @returns The action passed in
         * @see https://www.babylonjs-playground.com/#1T30HR#0
         */
        then(action: Action): Action;
        /**
         * Internal only
         * @hidden
         */
        _getProperty(propertyPath: string): string;
        /**
         * Internal only
         * @hidden
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @param parent of child
         * @returns the serialized object
         */
        serialize(parent: any): any;
        /**
         * Internal only called by serialize
         * @hidden
         */
        protected _serialize(serializedAction: any, parent?: any): any;
        /**
         * Internal only
         * @hidden
         */
        static _SerializeValueAsString: (value: any) => string;
        /**
         * Internal only
         * @hidden
         */
        static _GetTargetProperty: (target: Scene | Node) => {
            name: string;
            targetType: string;
            value: string;
        };
    }
}

declare module BABYLON {
    /**
     * ActionEvent is the event being sent when an action is triggered.
     */
    class ActionEvent {
        /** The mesh or sprite that triggered the action */
        source: any;
        /** The X mouse cursor position at the time of the event */
        pointerX: number;
        /** The Y mouse cursor position at the time of the event */
        pointerY: number;
        /** The mesh that is currently pointed at (can be null) */
        meshUnderPointer: Nullable<AbstractMesh>;
        /** the original (browser) event that triggered the ActionEvent */
        sourceEvent: any;
        /** additional data for the event */
        additionalData: any;
        /**
         * Creates a new ActionEvent
         * @param source The mesh or sprite that triggered the action
         * @param pointerX The X mouse cursor position at the time of the event
         * @param pointerY The Y mouse cursor position at the time of the event
         * @param meshUnderPointer The mesh that is currently pointed at (can be null)
         * @param sourceEvent the original (browser) event that triggered the ActionEvent
         * @param additionalData additional data for the event
         */
        constructor(
            /** The mesh or sprite that triggered the action */
            source: any, 
            /** The X mouse cursor position at the time of the event */
            pointerX: number, 
            /** The Y mouse cursor position at the time of the event */
            pointerY: number, 
            /** The mesh that is currently pointed at (can be null) */
            meshUnderPointer: Nullable<AbstractMesh>, 
            /** the original (browser) event that triggered the ActionEvent */
            sourceEvent?: any, 
            /** additional data for the event */
            additionalData?: any);
        /**
         * Helper function to auto-create an ActionEvent from a source mesh.
         * @param source The source mesh that triggered the event
         * @param evt The original (browser) event
         * @param additionalData additional data for the event
         * @returns the new ActionEvent
         */
        static CreateNew(source: AbstractMesh, evt?: Event, additionalData?: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a source sprite
         * @param source The source sprite that triggered the event
         * @param scene Scene associated with the sprite
         * @param evt The original (browser) event
         * @param additionalData additional data for the event
         * @returns the new ActionEvent
         */
        static CreateNewFromSprite(source: Sprite, scene: Scene, evt?: Event, additionalData?: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a scene. If triggered by a mesh use ActionEvent.CreateNew
         * @param scene the scene where the event occurred
         * @param evt The original (browser) event
         * @returns the new ActionEvent
         */
        static CreateNewFromScene(scene: Scene, evt: Event): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a primitive
         * @param prim defines the target primitive
         * @param pointerPos defines the pointer position
         * @param evt The original (browser) event
         * @param additionalData additional data for the event
         * @returns the new ActionEvent
         */
        static CreateNewFromPrimitive(prim: any, pointerPos: Vector2, evt?: Event, additionalData?: any): ActionEvent;
    }
    /**
     * Action Manager manages all events to be triggered on a given mesh or the global scene.
     * A single scene can have many Action Managers to handle predefined actions on specific meshes.
     * @see http://doc.babylonjs.com/how_to/how_to_use_actions
     */
    class ActionManager {
        private static _NothingTrigger;
        private static _OnPickTrigger;
        private static _OnLeftPickTrigger;
        private static _OnRightPickTrigger;
        private static _OnCenterPickTrigger;
        private static _OnPickDownTrigger;
        private static _OnDoublePickTrigger;
        private static _OnPickUpTrigger;
        private static _OnLongPressTrigger;
        private static _OnPointerOverTrigger;
        private static _OnPointerOutTrigger;
        private static _OnEveryFrameTrigger;
        private static _OnIntersectionEnterTrigger;
        private static _OnIntersectionExitTrigger;
        private static _OnKeyDownTrigger;
        private static _OnKeyUpTrigger;
        private static _OnPickOutTrigger;
        /**
         * Nothing
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly NothingTrigger: number;
        /**
         * On pick
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickTrigger: number;
        /**
         * On left pick
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnLeftPickTrigger: number;
        /**
         * On right pick
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnRightPickTrigger: number;
        /**
         * On center pick
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnCenterPickTrigger: number;
        /**
         * On pick down
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickDownTrigger: number;
        /**
         * On double pick
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnDoublePickTrigger: number;
        /**
         * On pick up
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickUpTrigger: number;
        /**
         * On pick out.
         * This trigger will only be raised if you also declared a OnPickDown
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickOutTrigger: number;
        /**
         * On long press
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnLongPressTrigger: number;
        /**
         * On pointer over
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPointerOverTrigger: number;
        /**
         * On pointer out
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPointerOutTrigger: number;
        /**
         * On every frame
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnEveryFrameTrigger: number;
        /**
         * On intersection enter
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnIntersectionEnterTrigger: number;
        /**
         * On intersection exit
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnIntersectionExitTrigger: number;
        /**
         * On key down
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnKeyDownTrigger: number;
        /**
         * On key up
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnKeyUpTrigger: number;
        /** Gets the list of active triggers */
        static Triggers: {
            [key: string]: number;
        };
        /** Gets the list of actions */
        actions: Action[];
        /** Gets the cursor to use when hovering items */
        hoverCursor: string;
        private _scene;
        /**
         * Creates a new action manager
         * @param scene defines the hosting scene
         */
        constructor(scene: Scene);
        /**
         * Releases all associated resources
         */
        dispose(): void;
        /**
         * Gets hosting scene
         * @returns the hosting scene
         */
        getScene(): Scene;
        /**
         * Does this action manager handles actions of any of the given triggers
         * @param triggers defines the triggers to be tested
         * @return a boolean indicating whether one (or more) of the triggers is handled
         */
        hasSpecificTriggers(triggers: number[]): boolean;
        /**
         * Does this action manager handles actions of a given trigger
         * @param trigger defines the trigger to be tested
         * @param parameterPredicate defines an optional predicate to filter triggers by parameter
         * @return whether the trigger is handled
         */
        hasSpecificTrigger(trigger: number, parameterPredicate?: (parameter: any) => boolean): boolean;
        /**
         * Does this action manager has pointer triggers
         */
        readonly hasPointerTriggers: boolean;
        /**
         * Does this action manager has pick triggers
         */
        readonly hasPickTriggers: boolean;
        /**
         * Does exist one action manager with at least one trigger
         **/
        static readonly HasTriggers: boolean;
        /**
         * Does exist one action manager with at least one pick trigger
         **/
        static readonly HasPickTriggers: boolean;
        /**
         * Does exist one action manager that handles actions of a given trigger
         * @param trigger defines the trigger to be tested
         * @return a boolean indicating whether the trigger is handeled by at least one action manager
        **/
        static HasSpecificTrigger(trigger: number): boolean;
        /**
         * Registers an action to this action manager
         * @param action defines the action to be registered
         * @return the action amended (prepared) after registration
         */
        registerAction(action: Action): Nullable<Action>;
        /**
         * Unregisters an action to this action manager
         * @param action defines the action to be unregistered
         * @return a boolean indicating whether the action has been unregistered
         */
        unregisterAction(action: Action): Boolean;
        /**
         * Process a specific trigger
         * @param trigger defines the trigger to process
         * @param evt defines the event details to be processed
         */
        processTrigger(trigger: number, evt?: ActionEvent): void;
        /** @hidden */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /** @hidden */
        _getProperty(propertyPath: string): string;
        /**
         * Serialize this manager to a JSON object
         * @param name defines the property name to store this manager
         * @returns a JSON representation of this manager
         */
        serialize(name: string): any;
        /**
         * Creates a new ActionManager from a JSON data
         * @param parsedActions defines the JSON data to read from
         * @param object defines the hosting mesh
         * @param scene defines the hosting scene
         */
        static Parse(parsedActions: any, object: Nullable<AbstractMesh>, scene: Scene): void;
        /**
         * Get a trigger name by index
         * @param trigger defines the trigger index
         * @returns a trigger name
         */
        static GetTriggerName(trigger: number): string;
    }
}

declare module BABYLON {
    /**
     * A Condition applied to an Action
     */
    class Condition {
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @hidden
         */
        _evaluationId: number;
        /**
         * Internal only
         * @hidden
         */
        _currentResult: boolean;
        /**
         * Creates a new Condition
         * @param actionManager the manager of the action the condition is applied to
         */
        constructor(actionManager: ActionManager);
        /**
         * Check if the current condition is valid
         * @returns a boolean
         */
        isValid(): boolean;
        /**
         * Internal only
         * @hidden
         */
        _getProperty(propertyPath: string): string;
        /**
         * Internal only
         * @hidden
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @returns the serialized object
         */
        serialize(): any;
        /**
         * Internal only
         * @hidden
         */
        protected _serialize(serializedCondition: any): any;
    }
    /**
     * Defines specific conditional operators as extensions of Condition
     */
    class ValueCondition extends Condition {
        /** path to specify the property of the target the conditional operator uses  */
        propertyPath: string;
        /** the value compared by the conditional operator against the current value of the property */
        value: any;
        /** the conditional operator, default ValueCondition.IsEqual */
        operator: number;
        /**
         * Internal only
         * @hidden
         */
        private static _IsEqual;
        /**
         * Internal only
         * @hidden
         */
        private static _IsDifferent;
        /**
         * Internal only
         * @hidden
         */
        private static _IsGreater;
        /**
         * Internal only
         * @hidden
         */
        private static _IsLesser;
        /**
         * returns the number for IsEqual
         */
        static readonly IsEqual: number;
        /**
         * Returns the number for IsDifferent
         */
        static readonly IsDifferent: number;
        /**
         * Returns the number for IsGreater
         */
        static readonly IsGreater: number;
        /**
         * Returns the number for IsLesser
         */
        static readonly IsLesser: number;
        /**
         * Internal only The action manager for the condition
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @hidden
         */
        private _target;
        /**
         * Internal only
         * @hidden
         */
        private _effectiveTarget;
        /**
         * Internal only
         * @hidden
         */
        private _property;
        /**
         * Creates a new ValueCondition
         * @param actionManager manager for the action the condition applies to
         * @param target for the action
         * @param propertyPath path to specify the property of the target the conditional operator uses
         * @param value the value compared by the conditional operator against the current value of the property
         * @param operator the conditional operator, default ValueCondition.IsEqual
         */
        constructor(actionManager: ActionManager, target: any, 
            /** path to specify the property of the target the conditional operator uses  */
            propertyPath: string, 
            /** the value compared by the conditional operator against the current value of the property */
            value: any, 
            /** the conditional operator, default ValueCondition.IsEqual */
            operator?: number);
        /**
         * Compares the given value with the property value for the specified conditional operator
         * @returns the result of the comparison
         */
        isValid(): boolean;
        /**
         * Serialize the ValueCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
        /**
         * Gets the name of the conditional operator for the ValueCondition
         * @param operator the conditional operator
         * @returns the name
         */
        static GetOperatorName(operator: number): string;
    }
    /**
     * Defines a predicate condition as an extension of Condition
     */
    class PredicateCondition extends Condition {
        /** defines the predicate function used to validate the condition */
        predicate: () => boolean;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Creates a new PredicateCondition
         * @param actionManager manager for the action the condition applies to
         * @param predicate defines the predicate function used to validate the condition
         */
        constructor(actionManager: ActionManager, 
            /** defines the predicate function used to validate the condition */
            predicate: () => boolean);
        /**
         * @returns the validity of the predicate condition
         */
        isValid(): boolean;
    }
    /**
     * Defines a state condition as an extension of Condition
     */
    class StateCondition extends Condition {
        value: string;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @hidden
         */
        private _target;
        /**
         * Creates a new StateCondition
         * @param actionManager manager for the action the condition applies to
         * @param target of the condition
         * @param value to compare with target state
         */
        constructor(actionManager: ActionManager, target: any, value: string);
        /**
         * @returns the validity of the state
         */
        isValid(): boolean;
        /**
         * Serialize the StateCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
    }
}

declare module BABYLON {
    class SwitchBooleanAction extends Action {
        propertyPath: string;
        private _target;
        private _effectiveTarget;
        private _property;
        constructor(triggerOptions: any, target: any, propertyPath: string, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class SetStateAction extends Action {
        value: string;
        private _target;
        constructor(triggerOptions: any, target: any, value: string, condition?: Condition);
        execute(): void;
        serialize(parent: any): any;
    }
    class SetValueAction extends Action {
        propertyPath: string;
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class IncrementValueAction extends Action {
        propertyPath: string;
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class PlayAnimationAction extends Action {
        from: number;
        to: number;
        loop: boolean | undefined;
        private _target;
        constructor(triggerOptions: any, target: any, from: number, to: number, loop?: boolean | undefined, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class StopAnimationAction extends Action {
        private _target;
        constructor(triggerOptions: any, target: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class DoNothingAction extends Action {
        constructor(triggerOptions?: any, condition?: Condition);
        execute(): void;
        serialize(parent: any): any;
    }
    class CombineAction extends Action {
        children: Action[];
        constructor(triggerOptions: any, children: Action[], condition?: Condition);
        _prepare(): void;
        execute(evt: ActionEvent): void;
        serialize(parent: any): any;
    }
    class ExecuteCodeAction extends Action {
        func: (evt: ActionEvent) => void;
        constructor(triggerOptions: any, func: (evt: ActionEvent) => void, condition?: Condition);
        execute(evt: ActionEvent): void;
    }
    class SetParentAction extends Action {
        private _parent;
        private _target;
        constructor(triggerOptions: any, target: any, parent: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class PlaySoundAction extends Action {
        private _sound;
        constructor(triggerOptions: any, sound: Sound, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class StopSoundAction extends Action {
        private _sound;
        constructor(triggerOptions: any, sound: Sound, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
}

declare module BABYLON {
    class InterpolateValueAction extends Action {
        propertyPath: string;
        value: any;
        duration: number;
        stopOtherAnimations: boolean | undefined;
        onInterpolationDone: (() => void) | undefined;
        private _target;
        private _effectiveTarget;
        private _property;
        onInterpolationDoneObservable: Observable<InterpolateValueAction>;
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, duration?: number, condition?: Condition, stopOtherAnimations?: boolean | undefined, onInterpolationDone?: (() => void) | undefined);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
}

declare module BABYLON {
    class Animatable {
        target: any;
        fromFrame: number;
        toFrame: number;
        loopAnimation: boolean;
        onAnimationEnd: (() => void) | null | undefined;
        private _localDelayOffset;
        private _pausedDelay;
        private _runtimeAnimations;
        private _paused;
        private _scene;
        private _speedRatio;
        private _weight;
        private _syncRoot;
        animationStarted: boolean;
        /**
         * Gets the root Animatable used to synchronize and normalize animations
         */
        readonly syncRoot: Animatable;
        /**
         * Gets the current frame of the first RuntimeAnimation
         * Used to synchronize Animatables
         */
        readonly masterFrame: number;
        /**
         * Gets or sets the animatable weight (-1.0 by default meaning not weighted)
         */
        weight: number;
        /**
         * Gets or sets the speed ratio to apply to the animatable (1.0 by default)
         */
        speedRatio: number;
        constructor(scene: Scene, target: any, fromFrame?: number, toFrame?: number, loopAnimation?: boolean, speedRatio?: number, onAnimationEnd?: (() => void) | null | undefined, animations?: any);
        /**
         * Synchronize and normalize current Animatable with a source Animatable
         * This is useful when using animation weights and when animations are not of the same length
         * @param root defines the root Animatable to synchronize with
         * @returns the current Animatable
         */
        syncWith(root: Animatable): Animatable;
        getAnimations(): RuntimeAnimation[];
        appendAnimations(target: any, animations: Animation[]): void;
        getAnimationByTargetProperty(property: string): Nullable<Animation>;
        getRuntimeAnimationByTargetProperty(property: string): Nullable<RuntimeAnimation>;
        reset(): void;
        enableBlending(blendingSpeed: number): void;
        disableBlending(): void;
        goToFrame(frame: number): void;
        pause(): void;
        restart(): void;
        stop(animationName?: string): void;
        _animate(delay: number): boolean;
    }
}

declare module BABYLON {
    /**
     * Represents the range of an animation
     */
    class AnimationRange {
        /**The name of the animation range**/
        name: string;
        /**The starting frame of the animation */
        from: number;
        /**The ending frame of the animation*/
        to: number;
        /**
         * Initializes the range of an animation
         * @param name The name of the animation range
         * @param from The starting frame of the animation
         * @param to The ending frame of the animation
         */
        constructor(
            /**The name of the animation range**/
            name: string, 
            /**The starting frame of the animation */
            from: number, 
            /**The ending frame of the animation*/
            to: number);
        /**
         * Makes a copy of the animation range
         * @returns A copy of the animation range
         */
        clone(): AnimationRange;
    }
    /**
     * Composed of a frame, and an action function
     */
    class AnimationEvent {
        /** The frame for which the event is triggered **/
        frame: number;
        /** The event to perform when triggered **/
        action: () => void;
        /** Specifies if the event should be triggered only once**/
        onlyOnce: boolean | undefined;
        /**
         * Specifies if the animation event is done
         */
        isDone: boolean;
        /**
         * Initializes the animation event
         * @param frame The frame for which the event is triggered
         * @param action The event to perform when triggered
         * @param onlyOnce Specifies if the event should be triggered only once
         */
        constructor(
            /** The frame for which the event is triggered **/
            frame: number, 
            /** The event to perform when triggered **/
            action: () => void, 
            /** Specifies if the event should be triggered only once**/
            onlyOnce?: boolean | undefined);
    }
    /**
     * A cursor which tracks a point on a path
     */
    class PathCursor {
        private path;
        /**
         * Stores path cursor callbacks for when an onchange event is triggered
         */
        private _onchange;
        /**
         * The value of the path cursor
         */
        value: number;
        /**
         * The animation array of the path cursor
         */
        animations: Animation[];
        /**
         * Initializes the path cursor
         * @param path The path to track
         */
        constructor(path: Path2);
        /**
         * Gets the cursor point on the path
         * @returns A point on the path cursor at the cursor location
         */
        getPoint(): Vector3;
        /**
         * Moves the cursor ahead by the step amount
         * @param step The amount to move the cursor forward
         * @returns This path cursor
         */
        moveAhead(step?: number): PathCursor;
        /**
         * Moves the cursor behind by the step amount
         * @param step The amount to move the cursor back
         * @returns This path cursor
         */
        moveBack(step?: number): PathCursor;
        /**
         * Moves the cursor by the step amount
         * If the step amount is greater than one, an exception is thrown
         * @param step The amount to move the cursor
         * @returns This path cursor
         */
        move(step: number): PathCursor;
        /**
         * Ensures that the value is limited between zero and one
         * @returns This path cursor
         */
        private ensureLimits();
        /**
         * Runs onchange callbacks on change (used by the animation engine)
         * @returns This path cursor
         */
        private raiseOnChange();
        /**
         * Executes a function on change
         * @param f A path cursor onchange callback
         * @returns This path cursor
         */
        onchange(f: (cursor: PathCursor) => void): PathCursor;
    }
    /**
     * Defines an interface which represents an animation key frame
     */
    interface IAnimationKey {
        /**
         * Frame of the key frame
         */
        frame: number;
        /**
         * Value at the specifies key frame
         */
        value: any;
        /**
         * The input tangent for the cubic hermite spline
         */
        inTangent?: any;
        /**
         * The output tangent for the cubic hermite spline
         */
        outTangent?: any;
        /**
         * The animation interpolation type
         */
        interpolation?: AnimationKeyInterpolation;
    }
    /**
     * Enum for the animation key frame interpolation type
     */
    enum AnimationKeyInterpolation {
        /**
         * Do not interpolate between keys and use the start key value only. Tangents are ignored
         */
        STEP = 1,
    }
    /**
     * Class used to store any kind of animation
     */
    class Animation {
        /**Name of the animation */
        name: string;
        /**Property to animate */
        targetProperty: string;
        /**The frames per second of the animation */
        framePerSecond: number;
        /**The data type of the animation */
        dataType: number;
        /**The loop mode of the animation */
        loopMode: number | undefined;
        /**Specifies if blending should be enabled */
        enableBlending: boolean | undefined;
        /**
         * Use matrix interpolation instead of using direct key value when animating matrices
         */
        static AllowMatricesInterpolation: boolean;
        /**
         * When matrix interpolation is enabled, this boolean forces the system to use Matrix.DecomposeLerp instead of Matrix.Lerp. Interpolation is more precise but slower
         */
        static AllowMatrixDecomposeForInterpolation: boolean;
        /**
         * Stores the key frames of the animation
         */
        private _keys;
        /**
         * Stores the easing function of the animation
         */
        private _easingFunction;
        /**
         * @hidden Internal use only
         */
        _runtimeAnimations: RuntimeAnimation[];
        /**
         * The set of event that will be linked to this animation
         */
        private _events;
        /**
         * Stores an array of target property paths
         */
        targetPropertyPath: string[];
        /**
         * Stores the blending speed of the animation
         */
        blendingSpeed: number;
        /**
         * Stores the animation ranges for the animation
         */
        private _ranges;
        /**
         * @hidden Internal use
         */
        static _PrepareAnimation(name: string, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction): Nullable<Animation>;
        /**
         * Sets up an animation
         * @param property The property to animate
         * @param animationType The animation type to apply
         * @param framePerSecond The frames per second of the animation
         * @param easingFunction The easing function used in the animation
         * @returns The created animation
         */
        static CreateAnimation(property: string, animationType: number, framePerSecond: number, easingFunction: EasingFunction): Animation;
        /**
         * Create and start an animation on a node
         * @param name defines the name of the global animation that will be run on all nodes
         * @param node defines the root node where the animation will take place
         * @param targetProperty defines property to animate
         * @param framePerSecond defines the number of frame per second yo use
         * @param totalFrame defines the number of frames in total
         * @param from defines the initial value
         * @param to defines the final value
         * @param loopMode defines which loop mode you want to use (off by default)
         * @param easingFunction defines the easing function to use (linear by default)
         * @param onAnimationEnd defines the callback to call when animation end
         * @returns the animatable created for this animation
         */
        static CreateAndStartAnimation(name: string, node: Node, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Create and start an animation on a node and its descendants
         * @param name defines the name of the global animation that will be run on all nodes
         * @param node defines the root node where the animation will take place
         * @param directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used
         * @param targetProperty defines property to animate
         * @param framePerSecond defines the number of frame per second to use
         * @param totalFrame defines the number of frames in total
         * @param from defines the initial value
         * @param to defines the final value
         * @param loopMode defines which loop mode you want to use (off by default)
         * @param easingFunction defines the easing function to use (linear by default)
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of animatables created for all nodes
         * @example https://www.babylonjs-playground.com/#MH0VLI
         */
        static CreateAndStartHierarchyAnimation(name: string, node: Node, directDescendantsOnly: boolean, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable[]>;
        /**
         * Creates a new animation, merges it with the existing animations and starts it
         * @param name Name of the animation
         * @param node Node which contains the scene that begins the animations
         * @param targetProperty Specifies which property to animate
         * @param framePerSecond The frames per second of the animation
         * @param totalFrame The total number of frames
         * @param from The frame at the beginning of the animation
         * @param to The frame at the end of the animation
         * @param loopMode Specifies the loop mode of the animation
         * @param easingFunction (Optional) The easing function of the animation, which allow custom mathematical formulas for animations
         * @param onAnimationEnd Callback to run once the animation is complete
         * @returns Nullable animation
         */
        static CreateMergeAndStartAnimation(name: string, node: Node, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Transition property of the Camera to the target Value
         * @param property The property to transition
         * @param targetValue The target Value of the property
         * @param host The object where the property to animate belongs
         * @param scene Scene used to run the animation
         * @param frameRate Framerate (in frame/s) to use
         * @param transition The transition type we want to use
         * @param duration The duration of the animation, in milliseconds
         * @param onAnimationEnd Callback trigger at the end of the animation
         * @returns Nullable animation
         */
        static TransitionTo(property: string, targetValue: any, host: any, scene: Scene, frameRate: number, transition: Animation, duration: number, onAnimationEnd?: Nullable<() => void>): Nullable<Animatable>;
        /**
         * Return the array of runtime animations currently using this animation
         */
        readonly runtimeAnimations: RuntimeAnimation[];
        /**
         * Specifies if any of the runtime animations are currently running
         */
        readonly hasRunningRuntimeAnimations: boolean;
        /**
         * Initializes the animation
         * @param name Name of the animation
         * @param targetProperty Property to animate
         * @param framePerSecond The frames per second of the animation
         * @param dataType The data type of the animation
         * @param loopMode The loop mode of the animation
         * @param enableBlendings Specifies if blending should be enabled
         */
        constructor(
            /**Name of the animation */
            name: string, 
            /**Property to animate */
            targetProperty: string, 
            /**The frames per second of the animation */
            framePerSecond: number, 
            /**The data type of the animation */
            dataType: number, 
            /**The loop mode of the animation */
            loopMode?: number | undefined, 
            /**Specifies if blending should be enabled */
            enableBlending?: boolean | undefined);
        /**
         * Converts the animation to a string
         * @param fullDetails support for multiple levels of logging within scene loading
         * @returns String form of the animation
         */
        toString(fullDetails?: boolean): string;
        /**
         * Add an event to this animation
         * @param event Event to add
         */
        addEvent(event: AnimationEvent): void;
        /**
         * Remove all events found at the given frame
         * @param frame The frame to remove events from
         */
        removeEvents(frame: number): void;
        /**
         * Retrieves all the events from the animation
         * @returns Events from the animation
         */
        getEvents(): AnimationEvent[];
        /**
         * Creates an animation range
         * @param name Name of the animation range
         * @param from Starting frame of the animation range
         * @param to Ending frame of the animation
         */
        createRange(name: string, from: number, to: number): void;
        /**
         * Deletes an animation range by name
         * @param name Name of the animation range to delete
         * @param deleteFrames Specifies if the key frames for the range should also be deleted (true) or not (false)
         */
        deleteRange(name: string, deleteFrames?: boolean): void;
        /**
         * Gets the animation range by name, or null if not defined
         * @param name Name of the animation range
         * @returns Nullable animation range
         */
        getRange(name: string): Nullable<AnimationRange>;
        /**
         * Gets the key frames from the animation
         * @returns The key frames of the animation
         */
        getKeys(): Array<IAnimationKey>;
        /**
         * Gets the highest frame rate of the animation
         * @returns Highest frame rate of the animation
         */
        getHighestFrame(): number;
        /**
         * Gets the easing function of the animation
         * @returns Easing function of the animation
         */
        getEasingFunction(): IEasingFunction;
        /**
         * Sets the easing function of the animation
         * @param easingFunction A custom mathematical formula for animation
         */
        setEasingFunction(easingFunction: EasingFunction): void;
        /**
         * Interpolates a scalar linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated scalar value
         */
        floatInterpolateFunction(startValue: number, endValue: number, gradient: number): number;
        /**
         * Interpolates a scalar cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated scalar value
         */
        floatInterpolateFunctionWithTangents(startValue: number, outTangent: number, endValue: number, inTangent: number, gradient: number): number;
        /**
         * Interpolates a quaternion using a spherical linear interpolation
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated quaternion value
         */
        quaternionInterpolateFunction(startValue: Quaternion, endValue: Quaternion, gradient: number): Quaternion;
        /**
         * Interpolates a quaternion cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation curve
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated quaternion value
         */
        quaternionInterpolateFunctionWithTangents(startValue: Quaternion, outTangent: Quaternion, endValue: Quaternion, inTangent: Quaternion, gradient: number): Quaternion;
        /**
         * Interpolates a Vector3 linearl
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated scalar value
         */
        vector3InterpolateFunction(startValue: Vector3, endValue: Vector3, gradient: number): Vector3;
        /**
         * Interpolates a Vector3 cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns InterpolatedVector3 value
         */
        vector3InterpolateFunctionWithTangents(startValue: Vector3, outTangent: Vector3, endValue: Vector3, inTangent: Vector3, gradient: number): Vector3;
        /**
         * Interpolates a Vector2 linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Vector2 value
         */
        vector2InterpolateFunction(startValue: Vector2, endValue: Vector2, gradient: number): Vector2;
        /**
         * Interpolates a Vector2 cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Vector2 value
         */
        vector2InterpolateFunctionWithTangents(startValue: Vector2, outTangent: Vector2, endValue: Vector2, inTangent: Vector2, gradient: number): Vector2;
        /**
         * Interpolates a size linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Size value
         */
        sizeInterpolateFunction(startValue: Size, endValue: Size, gradient: number): Size;
        /**
         * Interpolates a Color3 linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Color3 value
         */
        color3InterpolateFunction(startValue: Color3, endValue: Color3, gradient: number): Color3;
        /**
         * @hidden Internal use only
         */
        _getKeyValue(value: any): any;
        /**
         * @hidden Internal use only
         */
        _interpolate(currentFrame: number, repeatCount: number, workValue?: any, loopMode?: number, offsetValue?: any, highLimitValue?: any): any;
        /**
         * Defines the function to use to interpolate matrices
         * @param startValue defines the start matrix
         * @param endValue defines the end matrix
         * @param gradient defines the gradient between both matrices
         * @param result defines an optional target matrix where to store the interpolation
         * @returns the interpolated matrix
         */
        matrixInterpolateFunction(startValue: Matrix, endValue: Matrix, gradient: number, result?: Matrix): Matrix;
        /**
         * Makes a copy of the animation
         * @returns Cloned animation
         */
        clone(): Animation;
        /**
         * Sets the key frames of the animation
         * @param values The animation key frames to set
         */
        setKeys(values: Array<IAnimationKey>): void;
        /**
         * Serializes the animation to an object
         * @returns Serialized object
         */
        serialize(): any;
        /**
         * Float animation type
         */
        private static _ANIMATIONTYPE_FLOAT;
        /**
         * Vector3 animation type
         */
        private static _ANIMATIONTYPE_VECTOR3;
        /**
         * Quaternion animation type
         */
        private static _ANIMATIONTYPE_QUATERNION;
        /**
         * Matrix animation type
         */
        private static _ANIMATIONTYPE_MATRIX;
        /**
         * Color3 animation type
         */
        private static _ANIMATIONTYPE_COLOR3;
        /**
         * Vector2 animation type
         */
        private static _ANIMATIONTYPE_VECTOR2;
        /**
         * Size animation type
         */
        private static _ANIMATIONTYPE_SIZE;
        /**
         * Relative Loop Mode
         */
        private static _ANIMATIONLOOPMODE_RELATIVE;
        /**
         * Cycle Loop Mode
         */
        private static _ANIMATIONLOOPMODE_CYCLE;
        /**
         * Constant Loop Mode
         */
        private static _ANIMATIONLOOPMODE_CONSTANT;
        /**
         * Get the float animation type
         */
        static readonly ANIMATIONTYPE_FLOAT: number;
        /**
         * Get the Vector3 animation type
         */
        static readonly ANIMATIONTYPE_VECTOR3: number;
        /**
         * Get the Vectpr2 animation type
         */
        static readonly ANIMATIONTYPE_VECTOR2: number;
        /**
         * Get the Size animation type
         */
        static readonly ANIMATIONTYPE_SIZE: number;
        /**
         * Get the Quaternion animation type
         */
        static readonly ANIMATIONTYPE_QUATERNION: number;
        /**
         * Get the Matrix animation type
         */
        static readonly ANIMATIONTYPE_MATRIX: number;
        /**
         * Get the Color3 animation type
         */
        static readonly ANIMATIONTYPE_COLOR3: number;
        /**
         * Get the Relative Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_RELATIVE: number;
        /**
         * Get the Cycle Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_CYCLE: number;
        /**
         * Get the Constant Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_CONSTANT: number;
        /** @hidden */
        static _UniversalLerp(left: any, right: any, amount: number): any;
        /**
         * Parses an animation object and creates an animation
         * @param parsedAnimation Parsed animation object
         * @returns Animation object
         */
        static Parse(parsedAnimation: any): Animation;
        /**
         * Appends the serialized animations from the source animations
         * @param source Source containing the animations
         * @param destination Target to store the animations
         */
        static AppendSerializedAnimations(source: IAnimatable, destination: any): void;
    }
}

declare module BABYLON {
    /**
     * This class defines the direct association between an animation and a target
     */
    class TargetedAnimation {
        animation: Animation;
        target: any;
    }
    /**
     * Use this class to create coordinated animations on multiple targets
     */
    class AnimationGroup implements IDisposable {
        name: string;
        private _scene;
        private _targetedAnimations;
        private _animatables;
        private _from;
        private _to;
        private _isStarted;
        private _speedRatio;
        onAnimationEndObservable: Observable<TargetedAnimation>;
        /**
         * Gets the first frame
         */
        readonly from: number;
        /**
         * Gets the last frame
         */
        readonly to: number;
        /**
         * Define if the animations are started
         */
        readonly isStarted: boolean;
        /**
         * Gets or sets the speed ratio to use for all animations
         */
        /**
         * Gets or sets the speed ratio to use for all animations
         */
        speedRatio: number;
        /**
         * Gets the targeted animations for this animation group
         */
        readonly targetedAnimations: Array<TargetedAnimation>;
        /**
         * returning the list of animatables controlled by this animation group.
         */
        readonly animatables: Array<Animatable>;
        constructor(name: string, scene?: Nullable<Scene>);
        /**
         * Add an animation (with its target) in the group
         * @param animation defines the animation we want to add
         * @param target defines the target of the animation
         * @returns the {BABYLON.TargetedAnimation} object
         */
        addTargetedAnimation(animation: Animation, target: any): TargetedAnimation;
        /**
         * This function will normalize every animation in the group to make sure they all go from beginFrame to endFrame
         * It can add constant keys at begin or end
         * @param beginFrame defines the new begin frame for all animations or the smallest begin frame of all animations if null (defaults to null)
         * @param endFrame defines the new end frame for all animations or the largest end frame of all animations if null (defaults to null)
         */
        normalize(beginFrame?: Nullable<number>, endFrame?: Nullable<number>): AnimationGroup;
        /**
         * Start all animations on given targets
         * @param loop defines if animations must loop
         * @param speedRatio defines the ratio to apply to animation speed (1 by default)
         * @param from defines the from key (optional)
         * @param to defines the to key (optional)
         * @returns the current animation group
         */
        start(loop?: boolean, speedRatio?: number, from?: number, to?: number): AnimationGroup;
        /**
         * Pause all animations
         */
        pause(): AnimationGroup;
        /**
         * Play all animations to initial state
         * This function will start() the animations if they were not started or will restart() them if they were paused
         * @param loop defines if animations must loop
         */
        play(loop?: boolean): AnimationGroup;
        /**
         * Reset all animations to initial state
         */
        reset(): AnimationGroup;
        /**
         * Restart animations from key 0
         */
        restart(): AnimationGroup;
        /**
         * Stop all animations
         */
        stop(): AnimationGroup;
        /**
         * Set animation weight for all animatables
         * @param weight defines the weight to use
         * @return the animationGroup
         * @see http://doc.babylonjs.com/babylon101/animations#animation-weights
         */
        setWeightForAllAnimatables(weight: number): AnimationGroup;
        /**
         * Synchronize and normalize all animatables with a source animatable
         * @param root defines the root animatable to synchronize with
         * @return the animationGroup
         * @see http://doc.babylonjs.com/babylon101/animations#animation-weights
         */
        syncAllAnimationsWith(root: Animatable): AnimationGroup;
        /**
         * Goes to a specific frame in this animation group
         * @param frame the frame number to go to
         * @return the animationGroup
         */
        goToFrame(frame: number): AnimationGroup;
        /**
         * Dispose all associated resources
         */
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Class used to override all child animations of a given target
     */
    class AnimationPropertiesOverride {
        /**
         * Gets or sets a value indicating if animation blending must be used
         */
        enableBlending: boolean;
        /**
         * Gets or sets the blending speed to use when enableBlending is true
         */
        blendingSpeed: number;
        /**
         * Gets or sets the default loop mode to use
         */
        loopMode: number;
    }
}

declare module BABYLON {
    interface IEasingFunction {
        ease(gradient: number): number;
    }
    class EasingFunction implements IEasingFunction {
        private static _EASINGMODE_EASEIN;
        private static _EASINGMODE_EASEOUT;
        private static _EASINGMODE_EASEINOUT;
        static readonly EASINGMODE_EASEIN: number;
        static readonly EASINGMODE_EASEOUT: number;
        static readonly EASINGMODE_EASEINOUT: number;
        private _easingMode;
        setEasingMode(easingMode: number): void;
        getEasingMode(): number;
        easeInCore(gradient: number): number;
        ease(gradient: number): number;
    }
    class CircleEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class BackEase extends EasingFunction implements IEasingFunction {
        amplitude: number;
        constructor(amplitude?: number);
        easeInCore(gradient: number): number;
    }
    class BounceEase extends EasingFunction implements IEasingFunction {
        bounces: number;
        bounciness: number;
        constructor(bounces?: number, bounciness?: number);
        easeInCore(gradient: number): number;
    }
    class CubicEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class ElasticEase extends EasingFunction implements IEasingFunction {
        oscillations: number;
        springiness: number;
        constructor(oscillations?: number, springiness?: number);
        easeInCore(gradient: number): number;
    }
    class ExponentialEase extends EasingFunction implements IEasingFunction {
        exponent: number;
        constructor(exponent?: number);
        easeInCore(gradient: number): number;
    }
    class PowerEase extends EasingFunction implements IEasingFunction {
        power: number;
        constructor(power?: number);
        easeInCore(gradient: number): number;
    }
    class QuadraticEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class QuarticEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class QuinticEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class SineEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class BezierCurveEase extends EasingFunction implements IEasingFunction {
        x1: number;
        y1: number;
        x2: number;
        y2: number;
        constructor(x1?: number, y1?: number, x2?: number, y2?: number);
        easeInCore(gradient: number): number;
    }
}

declare module BABYLON {
    /**
     * Defines a runtime animation
     */
    class RuntimeAnimation {
        /**
         * The current frame of the runtime animation
         */
        private _currentFrame;
        /**
         * The animation used by the runtime animation
         */
        private _animation;
        /**
         * The target of the runtime animation
         */
        private _target;
        /**
         * The initiating animatable
         */
        private _host;
        /**
         * The original value of the runtime animation
         */
        private _originalValue;
        /**
         * The original blend value of the runtime animation
         */
        private _originalBlendValue;
        /**
         * The offsets cache of the runtime animation
         */
        private _offsetsCache;
        /**
         * The high limits cache of the runtime animation
         */
        private _highLimitsCache;
        /**
         * Specifies if the runtime animation has been stopped
         */
        private _stopped;
        /**
         * The blending factor of the runtime animation
         */
        private _blendingFactor;
        /**
         * The BabylonJS scene
         */
        private _scene;
        /**
         * The current value of the runtime animation
         */
        private _currentValue;
        /** @hidden */
        _workValue: any;
        /**
         * The active target of the runtime animation
         */
        private _activeTarget;
        /**
         * The target path of the runtime animation
         */
        private _targetPath;
        /**
         * The weight of the runtime animation
         */
        private _weight;
        /**
         * The ratio offset of the runtime animation
         */
        private _ratioOffset;
        /**
         * The previous delay of the runtime animation
         */
        private _previousDelay;
        /**
         * The previous ratio of the runtime animation
         */
        private _previousRatio;
        /**
         * Gets the current frame of the runtime animation
         */
        readonly currentFrame: number;
        /**
         * Gets the weight of the runtime animation
         */
        readonly weight: number;
        /**
         * Gets the current value of the runtime animation
         */
        readonly currentValue: any;
        /**
         * Gets the target path of the runtime animation
         */
        readonly targetPath: string;
        /**
         * Gets the actual target of the runtime animation
         */
        readonly target: any;
        /**
         * Create a new RuntimeAnimation object
         * @param target defines the target of the animation
         * @param animation defines the source animation object
         * @param scene defines the hosting scene
         * @param host defines the initiating Animatable
         */
        constructor(target: any, animation: Animation, scene: Scene, host: Animatable);
        /**
         * Gets the animation from the runtime animation
         */
        readonly animation: Animation;
        /**
         * Resets the runtime animation to the beginning
         * @param restoreOriginal defines whether to restore the target property to the original value
         */
        reset(restoreOriginal?: boolean): void;
        /**
         * Specifies if the runtime animation is stopped
         * @returns Boolean specifying if the runtime animation is stopped
         */
        isStopped(): boolean;
        /**
         * Disposes of the runtime animation
         */
        dispose(): void;
        /**
         * Interpolates the animation from the current frame
         * @param currentFrame The frame to interpolate the animation to
         * @param repeatCount The number of times that the animation should loop
         * @param loopMode The type of looping mode to use
         * @param offsetValue Animation offset value
         * @param highLimitValue The high limit value
         * @returns The interpolated value
         */
        private _interpolate(currentFrame, repeatCount, loopMode?, offsetValue?, highLimitValue?);
        /**
         * Apply the interpolated value to the target
         * @param currentValue defines the value computed by the animation
         * @param weight defines the weight to apply to this value (Defaults to 1.0)
         */
        setValue(currentValue: any, weight?: number): void;
        private _setValue(target, currentValue, weight, targetIndex?);
        /**
         * Gets the loop pmode of the runtime animation
         * @returns Loop Mode
         */
        private _getCorrectLoopMode();
        /**
         * Move the current animation to a given frame
         * @param frame defines the frame to move to
         */
        goToFrame(frame: number): void;
        /**
         * @hidden Internal use only
         */
        _prepareForSpeedRatioChange(newSpeedRatio: number): void;
        /**
         * Execute the current animation
         * @param delay defines the delay to add to the current frame
         * @param from defines the lower bound of the animation range
         * @param to defines the upper bound of the animation range
         * @param loop defines if the current animation must loop
         * @param speedRatio defines the current speed ratio
         * @param weight defines the weight of the animation (default is -1 so no weight)
         * @returns a boolean indicating if the animation has ended
         */
        animate(delay: number, from: number, to: number, loop: boolean, speedRatio: number, weight?: number): boolean;
    }
}

declare module BABYLON {
    /**
     * Class used to work with sound analyzer using fast fourier transform (FFT)
     * @see http://doc.babylonjs.com/how_to/playing_sounds_and_music
     */
    class Analyser {
        /**
         * Gets or sets the smoothing
         * @ignorenaming
         */
        SMOOTHING: number;
        /**
         * Gets or sets the FFT table size
         * @ignorenaming
         */
        FFT_SIZE: number;
        /**
         * Gets or sets the bar graph amplitude
         * @ignorenaming
         */
        BARGRAPHAMPLITUDE: number;
        /**
         * Gets or sets the position of the debug canvas
         * @ignorenaming
         */
        DEBUGCANVASPOS: {
            x: number;
            y: number;
        };
        /**
         * Gets or sets the debug canvas size
         * @ignorenaming
         */
        DEBUGCANVASSIZE: {
            width: number;
            height: number;
        };
        private _byteFreqs;
        private _byteTime;
        private _floatFreqs;
        private _webAudioAnalyser;
        private _debugCanvas;
        private _debugCanvasContext;
        private _scene;
        private _registerFunc;
        private _audioEngine;
        /**
         * Creates a new analyser
         * @param scene defines hosting scene
         */
        constructor(scene: Scene);
        /**
         * Get the number of data values you will have to play with for the visualization
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/frequencyBinCount
         * @returns a number
         */
        getFrequencyBinCount(): number;
        /**
         * Gets the current frequency data as a byte array
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/getByteFrequencyData
         * @returns a Uint8Array
         */
        getByteFrequencyData(): Uint8Array;
        /**
         * Gets the current waveform as a byte array
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/getByteTimeDomainData
         * @returns a Uint8Array
         */
        getByteTimeDomainData(): Uint8Array;
        /**
         * Gets the current frequency data as a float array
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/getByteFrequencyData
         * @returns a Float32Array
         */
        getFloatFrequencyData(): Float32Array;
        /**
         * Renders the debug canvas
         */
        drawDebugCanvas(): void;
        /**
         * Stops rendering the debug canvas and removes it
         */
        stopDebugCanvas(): void;
        /**
         * Connects two audio nodes
         * @param inputAudioNode defines first node to connect
         * @param outputAudioNode defines second node to connect
         */
        connectAudioNodes(inputAudioNode: AudioNode, outputAudioNode: AudioNode): void;
        /**
         * Releases all associated resources
         */
        dispose(): void;
    }
}

declare module BABYLON {
    class AudioEngine {
        private _audioContext;
        private _audioContextInitialized;
        canUseWebAudio: boolean;
        masterGain: GainNode;
        private _connectedAnalyser;
        WarnedWebAudioUnsupported: boolean;
        unlocked: boolean;
        onAudioUnlocked: () => any;
        isMP3supported: boolean;
        isOGGsupported: boolean;
        readonly audioContext: Nullable<AudioContext>;
        constructor();
        private _unlockiOSaudio();
        private _initializeAudioContext();
        dispose(): void;
        getGlobalVolume(): number;
        setGlobalVolume(newVolume: number): void;
        connectToAnalyser(analyser: Analyser): void;
    }
}

declare module BABYLON {
    class Sound {
        name: string;
        autoplay: boolean;
        loop: boolean;
        useCustomAttenuation: boolean;
        soundTrackId: number;
        spatialSound: boolean;
        refDistance: number;
        rolloffFactor: number;
        maxDistance: number;
        distanceModel: string;
        private _panningModel;
        onended: () => any;
        private _playbackRate;
        private _streaming;
        private _startTime;
        private _startOffset;
        private _position;
        private _localDirection;
        private _volume;
        private _isReadyToPlay;
        isPlaying: boolean;
        isPaused: boolean;
        private _isDirectional;
        private _readyToPlayCallback;
        private _audioBuffer;
        private _soundSource;
        private _streamingSource;
        private _soundPanner;
        private _soundGain;
        private _inputAudioNode;
        private _ouputAudioNode;
        private _coneInnerAngle;
        private _coneOuterAngle;
        private _coneOuterGain;
        private _scene;
        private _connectedMesh;
        private _customAttenuationFunction;
        private _registerFunc;
        private _isOutputConnected;
        private _htmlAudioElement;
        private _urlType;
        /**
        * Create a sound and attach it to a scene
        * @param name Name of your sound
        * @param urlOrArrayBuffer Url to the sound to load async or ArrayBuffer
        * @param readyToPlayCallback Provide a callback function if you'd like to load your code once the sound is ready to be played
        * @param options Objects to provide with the current available options: autoplay, loop, volume, spatialSound, maxDistance, rolloffFactor, refDistance, distanceModel, panningModel, streaming
        */
        constructor(name: string, urlOrArrayBuffer: any, scene: Scene, readyToPlayCallback?: Nullable<() => void>, options?: any);
        dispose(): void;
        isReady(): boolean;
        private _soundLoaded(audioData);
        setAudioBuffer(audioBuffer: AudioBuffer): void;
        updateOptions(options: any): void;
        private _createSpatialParameters();
        private _updateSpatialParameters();
        switchPanningModelToHRTF(): void;
        switchPanningModelToEqualPower(): void;
        private _switchPanningModel();
        connectToSoundTrackAudioNode(soundTrackAudioNode: AudioNode): void;
        /**
        * Transform this sound into a directional source
        * @param coneInnerAngle Size of the inner cone in degree
        * @param coneOuterAngle Size of the outer cone in degree
        * @param coneOuterGain Volume of the sound outside the outer cone (between 0.0 and 1.0)
        */
        setDirectionalCone(coneInnerAngle: number, coneOuterAngle: number, coneOuterGain: number): void;
        setPosition(newPosition: Vector3): void;
        setLocalDirectionToMesh(newLocalDirection: Vector3): void;
        private _updateDirection();
        updateDistanceFromListener(): void;
        setAttenuationFunction(callback: (currentVolume: number, currentDistance: number, maxDistance: number, refDistance: number, rolloffFactor: number) => number): void;
        /**
        * Play the sound
        * @param time (optional) Start the sound after X seconds. Start immediately (0) by default.
        * @param offset (optional) Start the sound setting it at a specific time
        */
        play(time?: number, offset?: number): void;
        private _onended();
        /**
        * Stop the sound
        * @param time (optional) Stop the sound after X seconds. Stop immediately (0) by default.
        */
        stop(time?: number): void;
        pause(): void;
        setVolume(newVolume: number, time?: number): void;
        setPlaybackRate(newPlaybackRate: number): void;
        getVolume(): number;
        attachToMesh(meshToConnectTo: AbstractMesh): void;
        detachFromMesh(): void;
        private _onRegisterAfterWorldMatrixUpdate(node);
        clone(): Nullable<Sound>;
        getAudioBuffer(): AudioBuffer | null;
        serialize(): any;
        static Parse(parsedSound: any, scene: Scene, rootUrl: string, sourceSound?: Sound): Sound;
    }
}

declare module BABYLON {
    class SoundTrack {
        private _outputAudioNode;
        private _scene;
        id: number;
        soundCollection: Array<Sound>;
        private _isMainTrack;
        private _connectedAnalyser;
        private _options;
        private _isInitialized;
        constructor(scene: Scene, options?: any);
        private _initializeSoundTrackAudioGraph();
        dispose(): void;
        AddSound(sound: Sound): void;
        RemoveSound(sound: Sound): void;
        setVolume(newVolume: number): void;
        switchPanningModelToHRTF(): void;
        switchPanningModelToEqualPower(): void;
        connectToAnalyser(analyser: Analyser): void;
    }
}

declare module BABYLON {
    interface Behavior<T extends Node> {
        name: string;
        init(): void;
        attach(node: T): void;
        detach(): void;
    }
}

declare module BABYLON {
    /**
     * Class used to store bone information
     * @see http://doc.babylonjs.com/how_to/how_to_use_bones_and_skeletons
     */
    class Bone extends Node {
        /**
         * defines the bone name
         */
        name: string;
        private static _tmpVecs;
        private static _tmpQuat;
        private static _tmpMats;
        /**
         * Gets the list of child bones
         */
        children: Bone[];
        /** Gets the animations associated with this bone */
        animations: Animation[];
        /**
         * Gets or sets bone length
         */
        length: number;
        /**
         * @hidden Internal only
         * Set this value to map this bone to a different index in the transform matrices
         * Set this value to -1 to exclude the bone from the transform matrices
         */
        _index: Nullable<number>;
        private _skeleton;
        private _localMatrix;
        private _restPose;
        private _baseMatrix;
        private _absoluteTransform;
        private _invertedAbsoluteTransform;
        private _parent;
        private _scalingDeterminant;
        private _worldTransform;
        private _localScaling;
        private _localRotation;
        private _localPosition;
        private _needToDecompose;
        private _needToCompose;
        /** @hidden */
        /** @hidden */
        _matrix: Matrix;
        /**
         * Create a new bone
         * @param name defines the bone name
         * @param skeleton defines the parent skeleton
         * @param parentBone defines the parent (can be null if the bone is the root)
         * @param localMatrix defines the local matrix
         * @param restPose defines the rest pose matrix
         * @param baseMatrix defines the base matrix
         * @param index defines index of the bone in the hiearchy
         */
        constructor(
            /**
             * defines the bone name
             */
            name: string, skeleton: Skeleton, parentBone?: Nullable<Bone>, localMatrix?: Nullable<Matrix>, restPose?: Nullable<Matrix>, baseMatrix?: Nullable<Matrix>, index?: Nullable<number>);
        /**
         * Gets the parent skeleton
         * @returns a skeleton
         */
        getSkeleton(): Skeleton;
        /**
         * Gets parent bone
         * @returns a bone or null if the bone is the root of the bone hierarchy
         */
        getParent(): Nullable<Bone>;
        /**
         * Sets the parent bone
         * @param parent defines the parent (can be null if the bone is the root)
         * @param updateDifferenceMatrix defines if the difference matrix must be updated
         */
        setParent(parent: Nullable<Bone>, updateDifferenceMatrix?: boolean): void;
        /**
         * Gets the local matrix
         * @returns a matrix
         */
        getLocalMatrix(): Matrix;
        /**
         * Gets the base matrix (initial matrix which remains unchanged)
         * @returns a matrix
         */
        getBaseMatrix(): Matrix;
        /**
         * Gets the rest pose matrix
         * @returns a matrix
         */
        getRestPose(): Matrix;
        /**
         * Gets a matrix used to store world matrix (ie. the matrix sent to shaders)
         */
        getWorldMatrix(): Matrix;
        /**
         * Sets the local matrix to rest pose matrix
         */
        returnToRest(): void;
        /**
         * Gets the inverse of the absolute transform matrix.
         * This matrix will be multiplied by local matrix to get the difference matrix (ie. the difference between original state and current state)
         * @returns a matrix
         */
        getInvertedAbsoluteTransform(): Matrix;
        /**
         * Gets the absolute transform matrix (ie base matrix * parent world matrix)
         * @returns a matrix
         */
        getAbsoluteTransform(): Matrix;
        /** Gets or sets current position (in local space) */
        position: Vector3;
        /** Gets or sets current rotation (in local space) */
        rotation: Vector3;
        /** Gets or sets current rotation quaternion (in local space) */
        rotationQuaternion: Quaternion;
        /** Gets or sets current scaling (in local space) */
        scaling: Vector3;
        /**
         * Gets the animation properties override
         */
        readonly animationPropertiesOverride: Nullable<AnimationPropertiesOverride>;
        private _decompose();
        private _compose();
        /**
         * Update the base and local matrices
         * @param matrix defines the new base or local matrix
         * @param updateDifferenceMatrix defines if the difference matrix must be updated
         * @param updateLocalMatrix defines if the local matrix should be updated
         */
        updateMatrix(matrix: Matrix, updateDifferenceMatrix?: boolean, updateLocalMatrix?: boolean): void;
        /** @hidden */
        _updateDifferenceMatrix(rootMatrix?: Matrix, updateChildren?: boolean): void;
        /**
         * Flag the bone as dirty (Forcing it to update everything)
         */
        markAsDirty(): void;
        private _markAsDirtyAndCompose();
        private _markAsDirtyAndDecompose();
        /**
         * Copy an animation range from another bone
         * @param source defines the source bone
         * @param rangeName defines the range name to copy
         * @param frameOffset defines the frame offset
         * @param rescaleAsRequired defines if rescaling must be applied if required
         * @param skelDimensionsRatio defines the scaling ratio
         * @returns true if operation was successful
         */
        copyAnimationRange(source: Bone, rangeName: string, frameOffset: number, rescaleAsRequired?: boolean, skelDimensionsRatio?: Nullable<Vector3>): boolean;
        /**
         * Translate the bone in local or world space
         * @param vec The amount to translate the bone
         * @param space The space that the translation is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         */
        translate(vec: Vector3, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the postion of the bone in local or world space
         * @param position The position to set the bone
         * @param space The space that the position is in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         */
        setPosition(position: Vector3, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the absolute position of the bone (world space)
         * @param position The position to set the bone
         * @param mesh The mesh that this bone is attached to
         */
        setAbsolutePosition(position: Vector3, mesh?: AbstractMesh): void;
        /**
         * Scale the bone on the x, y and z axes (in local space)
         * @param x The amount to scale the bone on the x axis
         * @param y The amount to scale the bone on the y axis
         * @param z The amount to scale the bone on the z axis
         * @param scaleChildren sets this to true if children of the bone should be scaled as well (false by default)
         */
        scale(x: number, y: number, z: number, scaleChildren?: boolean): void;
        /**
         * Set the bone scaling in local space
         * @param scale defines the scaling vector
         */
        setScale(scale: Vector3): void;
        /**
         * Gets the current scaling in local space
         * @returns the current scaling vector
         */
        getScale(): Vector3;
        /**
         * Gets the current scaling in local space and stores it in a target vector
         * @param result defines the target vector
         */
        getScaleToRef(result: Vector3): void;
        /**
         * Set the yaw, pitch, and roll of the bone in local or world space
         * @param yaw The rotation of the bone on the y axis
         * @param pitch The rotation of the bone on the x axis
         * @param roll The rotation of the bone on the z axis
         * @param space The space that the axes of rotation are in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         */
        setYawPitchRoll(yaw: number, pitch: number, roll: number, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Add a rotation to the bone on an axis in local or world space
         * @param axis The axis to rotate the bone on
         * @param amount The amount to rotate the bone
         * @param space The space that the axis is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         */
        rotate(axis: Vector3, amount: number, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the rotation of the bone to a particular axis angle in local or world space
         * @param axis The axis to rotate the bone on
         * @param angle The angle that the bone should be rotated to
         * @param space The space that the axis is in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         */
        setAxisAngle(axis: Vector3, angle: number, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the euler rotation of the bone in local of world space
         * @param rotation The euler rotation that the bone should be set to
         * @param space The space that the rotation is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         */
        setRotation(rotation: Vector3, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the quaternion rotation of the bone in local of world space
         * @param quat The quaternion rotation that the bone should be set to
         * @param space The space that the rotation is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         */
        setRotationQuaternion(quat: Quaternion, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the rotation matrix of the bone in local of world space
         * @param rotMat The rotation matrix that the bone should be set to
         * @param space The space that the rotation is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         */
        setRotationMatrix(rotMat: Matrix, space?: Space, mesh?: AbstractMesh): void;
        private _rotateWithMatrix(rmat, space?, mesh?);
        private _getNegativeRotationToRef(rotMatInv, mesh?);
        /**
         * Get the position of the bone in local or world space
         * @param space The space that the returned position is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         * @returns The position of the bone
         */
        getPosition(space?: Space, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the position of the bone to a vector3 in local or world space
         * @param space The space that the returned position is in
         * @param mesh The mesh that this bone is attached to. This is only used in world space
         * @param result The vector3 to copy the position to
         */
        getPositionToRef(space: Space | undefined, mesh: Nullable<AbstractMesh>, result: Vector3): void;
        /**
         * Get the absolute position of the bone (world space)
         * @param mesh The mesh that this bone is attached to
         * @returns The absolute position of the bone
         */
        getAbsolutePosition(mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the absolute position of the bone (world space) to the result param
         * @param mesh The mesh that this bone is attached to
         * @param result The vector3 to copy the absolute position to
         */
        getAbsolutePositionToRef(mesh: AbstractMesh, result: Vector3): void;
        /**
         * Compute the absolute transforms of this bone and its children
         */
        computeAbsoluteTransforms(): void;
        /**
         * Get the world direction from an axis that is in the local space of the bone
         * @param localAxis The local direction that is used to compute the world direction
         * @param mesh The mesh that this bone is attached to
         * @returns The world direction
         */
        getDirection(localAxis: Vector3, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the world direction to a vector3 from an axis that is in the local space of the bone
         * @param localAxis The local direction that is used to compute the world direction
         * @param mesh The mesh that this bone is attached to
         * @param result The vector3 that the world direction will be copied to
         */
        getDirectionToRef(localAxis: Vector3, mesh: AbstractMesh | null | undefined, result: Vector3): void;
        /**
         * Get the euler rotation of the bone in local or world space
         * @param space The space that the rotation should be in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         * @returns The euler rotation
         */
        getRotation(space?: Space, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the euler rotation of the bone to a vector3.  The rotation can be in either local or world space
         * @param space The space that the rotation should be in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         * @param result The vector3 that the rotation should be copied to
         */
        getRotationToRef(space: Space | undefined, mesh: AbstractMesh | null | undefined, result: Vector3): void;
        /**
         * Get the quaternion rotation of the bone in either local or world space
         * @param space The space that the rotation should be in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         * @returns The quaternion rotation
         */
        getRotationQuaternion(space?: Space, mesh?: Nullable<AbstractMesh>): Quaternion;
        /**
         * Copy the quaternion rotation of the bone to a quaternion.  The rotation can be in either local or world space
         * @param space The space that the rotation should be in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         * @param result The quaternion that the rotation should be copied to
         */
        getRotationQuaternionToRef(space: Space | undefined, mesh: AbstractMesh | null | undefined, result: Quaternion): void;
        /**
         * Get the rotation matrix of the bone in local or world space
         * @param space The space that the rotation should be in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         * @returns The rotation matrix
         */
        getRotationMatrix(space: Space | undefined, mesh: AbstractMesh): Matrix;
        /**
         * Copy the rotation matrix of the bone to a matrix.  The rotation can be in either local or world space
         * @param space The space that the rotation should be in
         * @param mesh The mesh that this bone is attached to.  This is only used in world space
         * @param result The quaternion that the rotation should be copied to
         */
        getRotationMatrixToRef(space: Space | undefined, mesh: AbstractMesh, result: Matrix): void;
        /**
         * Get the world position of a point that is in the local space of the bone
         * @param position The local position
         * @param mesh The mesh that this bone is attached to
         * @returns The world position
         */
        getAbsolutePositionFromLocal(position: Vector3, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Get the world position of a point that is in the local space of the bone and copy it to the result param
         * @param position The local position
         * @param mesh The mesh that this bone is attached to
         * @param result The vector3 that the world position should be copied to
         */
        getAbsolutePositionFromLocalToRef(position: Vector3, mesh: AbstractMesh | null | undefined, result: Vector3): void;
        /**
         * Get the local position of a point that is in world space
         * @param position The world position
         * @param mesh The mesh that this bone is attached to
         * @returns The local position
         */
        getLocalPositionFromAbsolute(position: Vector3, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Get the local position of a point that is in world space and copy it to the result param
         * @param position The world position
         * @param mesh The mesh that this bone is attached to
         * @param result The vector3 that the local position should be copied to
         */
        getLocalPositionFromAbsoluteToRef(position: Vector3, mesh: AbstractMesh | null | undefined, result: Vector3): void;
    }
}

declare module BABYLON {
    /**
     * Class used to apply inverse kinematics to bones
     * @see http://doc.babylonjs.com/how_to/how_to_use_bones_and_skeletons#boneikcontroller
     */
    class BoneIKController {
        private static _tmpVecs;
        private static _tmpQuat;
        private static _tmpMats;
        /**
         * Gets or sets the target mesh
         */
        targetMesh: AbstractMesh;
        /** Gets or sets the mesh used as pole */
        poleTargetMesh: AbstractMesh;
        /**
         * Gets or sets the bone used as pole
         */
        poleTargetBone: Nullable<Bone>;
        /**
         * Gets or sets the target position
         */
        targetPosition: Vector3;
        /**
         * Gets or sets the pole target position
         */
        poleTargetPosition: Vector3;
        /**
         * Gets or sets the pole target local offset
         */
        poleTargetLocalOffset: Vector3;
        /**
         * Gets or sets the pole angle
         */
        poleAngle: number;
        /**
         * Gets or sets the mesh associated with the controller
         */
        mesh: AbstractMesh;
        /**
         * The amount to slerp (spherical linear interpolation) to the target.  Set this to a value between 0 and 1 (a value of 1 disables slerp)
         */
        slerpAmount: number;
        private _bone1Quat;
        private _bone1Mat;
        private _bone2Ang;
        private _bone1;
        private _bone2;
        private _bone1Length;
        private _bone2Length;
        private _maxAngle;
        private _maxReach;
        private _rightHandedSystem;
        private _bendAxis;
        private _slerping;
        private _adjustRoll;
        /**
         * Gets or sets maximum allowed angle
         */
        maxAngle: number;
        /**
         * Creates a new BoneIKController
         * @param mesh defines the mesh to control
         * @param bone defines the bone to control
         * @param options defines options to set up the controller
         */
        constructor(mesh: AbstractMesh, bone: Bone, options?: {
            targetMesh?: AbstractMesh;
            poleTargetMesh?: AbstractMesh;
            poleTargetBone?: Bone;
            poleTargetLocalOffset?: Vector3;
            poleAngle?: number;
            bendAxis?: Vector3;
            maxAngle?: number;
            slerpAmount?: number;
        });
        private _setMaxAngle(ang);
        /**
         * Force the controller to update the bones
         */
        update(): void;
    }
}

declare module BABYLON {
    /**
     * Class used to make a bone look toward a point in space
     * @see http://doc.babylonjs.com/how_to/how_to_use_bones_and_skeletons#bonelookcontroller
     */
    class BoneLookController {
        private static _tmpVecs;
        private static _tmpQuat;
        private static _tmpMats;
        /**
         * The target Vector3 that the bone will look at
         */
        target: Vector3;
        /**
         * The mesh that the bone is attached to
         */
        mesh: AbstractMesh;
        /**
         * The bone that will be looking to the target
         */
        bone: Bone;
        /**
         * The up axis of the coordinate system that is used when the bone is rotated
         */
        upAxis: Vector3;
        /**
         * The space that the up axis is in - BABYLON.Space.BONE, BABYLON.Space.LOCAL (default), or BABYLON.Space.WORLD
         */
        upAxisSpace: Space;
        /**
         * Used to make an adjustment to the yaw of the bone
         */
        adjustYaw: number;
        /**
         * Used to make an adjustment to the pitch of the bone
         */
        adjustPitch: number;
        /**
         * Used to make an adjustment to the roll of the bone
         */
        adjustRoll: number;
        /**
         * The amount to slerp (spherical linear interpolation) to the target.  Set this to a value between 0 and 1 (a value of 1 disables slerp)
         */
        slerpAmount: number;
        private _minYaw;
        private _maxYaw;
        private _minPitch;
        private _maxPitch;
        private _minYawSin;
        private _minYawCos;
        private _maxYawSin;
        private _maxYawCos;
        private _midYawConstraint;
        private _minPitchTan;
        private _maxPitchTan;
        private _boneQuat;
        private _slerping;
        private _transformYawPitch;
        private _transformYawPitchInv;
        private _firstFrameSkipped;
        private _yawRange;
        private _fowardAxis;
        /**
         * Gets or sets the minimum yaw angle that the bone can look to
         */
        minYaw: number;
        /**
         * Gets or sets the maximum yaw angle that the bone can look to
         */
        maxYaw: number;
        /**
         * Gets or sets the minimum pitch angle that the bone can look to
         */
        minPitch: number;
        /**
         * Gets or sets the maximum pitch angle that the bone can look to
         */
        maxPitch: number;
        /**
         * Create a BoneLookController
         * @param mesh the mesh that the bone belongs to
         * @param bone the bone that will be looking to the target
         * @param target the target Vector3 to look at
         * @param settings optional settings:
         * * maxYaw: the maximum angle the bone will yaw to
         * * minYaw: the minimum angle the bone will yaw to
         * * maxPitch: the maximum angle the bone will pitch to
         * * minPitch: the minimum angle the bone will yaw to
         * * slerpAmount: set the between 0 and 1 to make the bone slerp to the target.
         * * upAxis: the up axis of the coordinate system
         * * upAxisSpace: the space that the up axis is in - BABYLON.Space.BONE, BABYLON.Space.LOCAL (default), or BABYLON.Space.WORLD.
         * * yawAxis: set yawAxis if the bone does not yaw on the y axis
         * * pitchAxis: set pitchAxis if the bone does not pitch on the x axis
         * * adjustYaw: used to make an adjustment to the yaw of the bone
         * * adjustPitch: used to make an adjustment to the pitch of the bone
         * * adjustRoll: used to make an adjustment to the roll of the bone
         **/
        constructor(mesh: AbstractMesh, bone: Bone, target: Vector3, options?: {
            maxYaw?: number;
            minYaw?: number;
            maxPitch?: number;
            minPitch?: number;
            slerpAmount?: number;
            upAxis?: Vector3;
            upAxisSpace?: Space;
            yawAxis?: Vector3;
            pitchAxis?: Vector3;
            adjustYaw?: number;
            adjustPitch?: number;
            adjustRoll?: number;
        });
        /**
         * Update the bone to look at the target.  This should be called before the scene is rendered (use scene.registerBeforeRender())
         */
        update(): void;
        private _getAngleDiff(ang1, ang2);
        private _getAngleBetween(ang1, ang2);
        private _isAngleBetween(ang, ang1, ang2);
    }
}

declare module BABYLON {
    /**
     * Class used to handle skinning animations
     * @see http://doc.babylonjs.com/how_to/how_to_use_bones_and_skeletons
     */
    class Skeleton implements IAnimatable {
        /** defines the skeleton name */
        name: string;
        /** defines the skeleton Id */
        id: string;
        /**
         * Gets the list of child bones
         */
        bones: Bone[];
        /**
         * Gets an estimate of the dimension of the skeleton at rest
         */
        dimensionsAtRest: Vector3;
        /**
         * Gets a boolean indicating if the root matrix is provided by meshes or by the current skeleton (this is the default value)
         */
        needInitialSkinMatrix: boolean;
        /**
         * Gets the list of animations attached to this skeleton
         */
        animations: Array<Animation>;
        private _scene;
        private _isDirty;
        private _transformMatrices;
        private _meshesWithPoseMatrix;
        private _animatables;
        private _identity;
        private _synchronizedWithMesh;
        private _ranges;
        private _lastAbsoluteTransformsUpdateId;
        /**
         * Specifies if the skeleton should be serialized
         */
        doNotSerialize: boolean;
        private _animationPropertiesOverride;
        /**
         * Gets or sets the animation properties override
         */
        animationPropertiesOverride: Nullable<AnimationPropertiesOverride>;
        /**
         * An observable triggered before computing the skeleton's matrices
         */
        onBeforeComputeObservable: Observable<Skeleton>;
        /**
         * Creates a new skeleton
         * @param name defines the skeleton name
         * @param id defines the skeleton Id
         * @param scene defines the hosting scene
         */
        constructor(
            /** defines the skeleton name */
            name: string, 
            /** defines the skeleton Id */
            id: string, scene: Scene);
        /**
         * Gets the list of transform matrices to send to shaders (one matrix per bone)
         * @param mesh defines the mesh to use to get the root matrix (if needInitialSkinMatrix === true)
         * @returns a Float32Array containing matrices data
         */
        getTransformMatrices(mesh: AbstractMesh): Float32Array;
        /**
         * Gets the current hosting scene
         * @returns a scene object
         */
        getScene(): Scene;
        /**
         * Gets a string representing the current skeleton data
         * @param fullDetails defines a boolean indicating if we want a verbose version
         * @returns a string representing the current skeleton data
         */
        toString(fullDetails?: boolean): string;
        /**
        * Get bone's index searching by name
        * @param name defines bone's name to search for
        * @return the indice of the bone. Returns -1 if not found
        */
        getBoneIndexByName(name: string): number;
        /**
         * Creater a new animation range
         * @param name defines the name of the range
         * @param from defines the start key
         * @param to defines the end key
         */
        createAnimationRange(name: string, from: number, to: number): void;
        /**
         * Delete a specific animation range
         * @param name defines the name of the range
         * @param deleteFrames defines if frames must be removed as well
         */
        deleteAnimationRange(name: string, deleteFrames?: boolean): void;
        /**
         * Gets a specific animation range
         * @param name defines the name of the range to look for
         * @returns the requested animation range or null if not found
         */
        getAnimationRange(name: string): Nullable<AnimationRange>;
        /**
         * Gets the list of all animation ranges defined on this skeleton
         * @returns an array
         */
        getAnimationRanges(): Nullable<AnimationRange>[];
        /**
         * Copy animation range from a source skeleton.
         * This is not for a complete retargeting, only between very similar skeleton's with only possible bone length differences
         * @param source defines the source skeleton
         * @param name defines the name of the range to copy
         * @param rescaleAsRequired defines if rescaling must be applied if required
         * @returns true if operation was successful
         */
        copyAnimationRange(source: Skeleton, name: string, rescaleAsRequired?: boolean): boolean;
        /**
         * Forces the skeleton to go to rest pose
         */
        returnToRest(): void;
        private _getHighestAnimationFrame();
        /**
         * Begin a specific animation range
         * @param name defines the name of the range to start
         * @param loop defines if looping must be turned on (false by default)
         * @param speedRatio defines the speed ratio to apply (1 by default)
         * @param onAnimationEnd defines a callback which will be called when animation will end
         * @returns a new animatable
         */
        beginAnimation(name: string, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Nullable<Animatable>;
        /** @hidden */
        _markAsDirty(): void;
        /** @hidden */
        _registerMeshWithPoseMatrix(mesh: AbstractMesh): void;
        /** @hidden */
        _unregisterMeshWithPoseMatrix(mesh: AbstractMesh): void;
        /** @hidden */
        _computeTransformMatrices(targetMatrix: Float32Array, initialSkinMatrix: Nullable<Matrix>): void;
        /**
         * Build all resources required to render a skeleton
         */
        prepare(): void;
        /**
         * Gets the list of animatables currently running for this skeleton
         * @returns an array of animatables
         */
        getAnimatables(): IAnimatable[];
        /**
         * Clone the current skeleton
         * @param name defines the name of the new skeleton
         * @param id defines the id of the enw skeleton
         * @returns the new skeleton
         */
        clone(name: string, id: string): Skeleton;
        /**
         * Enable animation blending for this skeleton
         * @param blendingSpeed defines the blending speed to apply
         * @see http://doc.babylonjs.com/babylon101/animations#animation-blending
         */
        enableBlending(blendingSpeed?: number): void;
        /**
         * Releases all resources associated with the current skeleton
         */
        dispose(): void;
        /**
         * Serialize the skeleton in a JSON object
         * @returns a JSON object
         */
        serialize(): any;
        /**
         * Creates a new skeleton from serialized data
         * @param parsedSkeleton defines the serialized data
         * @param scene defines the hosting scene
         * @returns a new skeleton
         */
        static Parse(parsedSkeleton: any, scene: Scene): Skeleton;
        /**
         * Compute all node absolute transforms
         * @param forceUpdate defines if computation must be done even if cache is up to date
         */
        computeAbsoluteTransforms(forceUpdate?: boolean): void;
        /**
         * Gets the root pose matrix
         * @returns a matrix
         */
        getPoseMatrix(): Nullable<Matrix>;
        /**
         * Sorts bones per internal index
         */
        sortBones(): void;
        private _sortBones(index, bones, visited);
    }
}

declare module BABYLON {
    class Collider {
        /** Define if a collision was found */
        collisionFound: boolean;
        /**
         * Define last intersection point in local space
         */
        intersectionPoint: Vector3;
        /**
         * Define last collided mesh
         */
        collidedMesh: Nullable<AbstractMesh>;
        private _collisionPoint;
        private _planeIntersectionPoint;
        private _tempVector;
        private _tempVector2;
        private _tempVector3;
        private _tempVector4;
        private _edge;
        private _baseToVertex;
        private _destinationPoint;
        private _slidePlaneNormal;
        private _displacementVector;
        _radius: Vector3;
        _retry: number;
        private _velocity;
        private _basePoint;
        private _epsilon;
        _velocityWorldLength: number;
        _basePointWorld: Vector3;
        private _velocityWorld;
        private _normalizedVelocity;
        _initialVelocity: Vector3;
        _initialPosition: Vector3;
        private _nearestDistance;
        private _collisionMask;
        collisionMask: number;
        /**
         * Gets the plane normal used to compute the sliding response (in local space)
         */
        readonly slidePlaneNormal: Vector3;
        _initialize(source: Vector3, dir: Vector3, e: number): void;
        _checkPointInTriangle(point: Vector3, pa: Vector3, pb: Vector3, pc: Vector3, n: Vector3): boolean;
        _canDoCollision(sphereCenter: Vector3, sphereRadius: number, vecMin: Vector3, vecMax: Vector3): boolean;
        _testTriangle(faceIndex: number, trianglePlaneArray: Array<Plane>, p1: Vector3, p2: Vector3, p3: Vector3, hasMaterial: boolean): void;
        _collide(trianglePlaneArray: Array<Plane>, pts: Vector3[], indices: IndicesArray, indexStart: number, indexEnd: number, decal: number, hasMaterial: boolean): void;
        _getResponse(pos: Vector3, vel: Vector3): void;
    }
}

declare module BABYLON {
    var CollisionWorker: string;
    interface ICollisionCoordinator {
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: Nullable<AbstractMesh>, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
        destroy(): void;
        onMeshAdded(mesh: AbstractMesh): void;
        onMeshUpdated(mesh: AbstractMesh): void;
        onMeshRemoved(mesh: AbstractMesh): void;
        onGeometryAdded(geometry: Geometry): void;
        onGeometryUpdated(geometry: Geometry): void;
        onGeometryDeleted(geometry: Geometry): void;
    }
    interface SerializedMesh {
        id: string;
        name: string;
        uniqueId: number;
        geometryId: Nullable<string>;
        sphereCenter: Array<number>;
        sphereRadius: number;
        boxMinimum: Array<number>;
        boxMaximum: Array<number>;
        worldMatrixFromCache: any;
        subMeshes: Array<SerializedSubMesh>;
        checkCollisions: boolean;
    }
    interface SerializedSubMesh {
        position: number;
        verticesStart: number;
        verticesCount: number;
        indexStart: number;
        indexCount: number;
        hasMaterial: boolean;
        sphereCenter: Array<number>;
        sphereRadius: number;
        boxMinimum: Array<number>;
        boxMaximum: Array<number>;
    }
    /**
     * Interface describing the value associated with a geometry
     */
    interface SerializedGeometry {
        /**
         * Defines the unique ID of the geometry
         */
        id: string;
        /**
         * Defines the array containing the positions
         */
        positions: Float32Array;
        /**
         * Defines the array containing the indices
         */
        indices: Uint32Array;
        /**
         * Defines the array containing the normals
         */
        normals: Float32Array;
    }
    interface BabylonMessage {
        taskType: WorkerTaskType;
        payload: InitPayload | CollidePayload | UpdatePayload;
    }
    interface SerializedColliderToWorker {
        position: Array<number>;
        velocity: Array<number>;
        radius: Array<number>;
    }
    /** Defines supported task for worker process */
    enum WorkerTaskType {
        /** Initialization */
        INIT = 0,
        /** Update of geometry */
        UPDATE = 1,
        /** Evaluate collision */
        COLLIDE = 2,
    }
    interface WorkerReply {
        error: WorkerReplyType;
        taskType: WorkerTaskType;
        payload?: any;
    }
    interface CollisionReplyPayload {
        newPosition: Array<number>;
        collisionId: number;
        collidedMeshUniqueId: number;
    }
    interface InitPayload {
    }
    interface CollidePayload {
        collisionId: number;
        collider: SerializedColliderToWorker;
        maximumRetry: number;
        excludedMeshUniqueId: Nullable<number>;
    }
    interface UpdatePayload {
        updatedMeshes: {
            [n: number]: SerializedMesh;
        };
        updatedGeometries: {
            [s: string]: SerializedGeometry;
        };
        removedMeshes: Array<number>;
        removedGeometries: Array<string>;
    }
    /** Defines kind of replies returned by worker */
    enum WorkerReplyType {
        /** Success */
        SUCCESS = 0,
        /** Unkown error */
        UNKNOWN_ERROR = 1,
    }
    class CollisionCoordinatorWorker implements ICollisionCoordinator {
        private _scene;
        private _scaledPosition;
        private _scaledVelocity;
        private _collisionsCallbackArray;
        private _init;
        private _runningUpdated;
        private _worker;
        private _addUpdateMeshesList;
        private _addUpdateGeometriesList;
        private _toRemoveMeshesArray;
        private _toRemoveGeometryArray;
        constructor();
        static SerializeMesh: (mesh: AbstractMesh) => SerializedMesh;
        static SerializeGeometry: (geometry: Geometry) => SerializedGeometry;
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: AbstractMesh, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
        destroy(): void;
        onMeshAdded(mesh: AbstractMesh): void;
        onMeshUpdated: (transformNode: TransformNode) => void;
        onMeshRemoved(mesh: AbstractMesh): void;
        onGeometryAdded(geometry: Geometry): void;
        onGeometryUpdated: (geometry: Geometry) => void;
        onGeometryDeleted(geometry: Geometry): void;
        private _afterRender;
        private _onMessageFromWorker;
    }
    class CollisionCoordinatorLegacy implements ICollisionCoordinator {
        private _scene;
        private _scaledPosition;
        private _scaledVelocity;
        private _finalPosition;
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: AbstractMesh, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
        destroy(): void;
        onMeshAdded(mesh: AbstractMesh): void;
        onMeshUpdated(mesh: AbstractMesh): void;
        onMeshRemoved(mesh: AbstractMesh): void;
        onGeometryAdded(geometry: Geometry): void;
        onGeometryUpdated(geometry: Geometry): void;
        onGeometryDeleted(geometry: Geometry): void;
        private _collideWithWorld(position, velocity, collider, maximumRetry, finalPosition, excludedMesh?);
    }
}

declare function importScripts(...urls: string[]): void;
declare const safePostMessage: any;
declare module BABYLON {
    var WorkerIncluded: boolean;
    class CollisionCache {
        private _meshes;
        private _geometries;
        getMeshes(): {
            [n: number]: SerializedMesh;
        };
        getGeometries(): {
            [s: number]: SerializedGeometry;
        };
        getMesh(id: any): SerializedMesh;
        addMesh(mesh: SerializedMesh): void;
        removeMesh(uniqueId: number): void;
        getGeometry(id: string): SerializedGeometry;
        addGeometry(geometry: SerializedGeometry): void;
        removeGeometry(id: string): void;
    }
    class CollideWorker {
        collider: Collider;
        private _collisionCache;
        private finalPosition;
        private collisionsScalingMatrix;
        private collisionTranformationMatrix;
        constructor(collider: Collider, _collisionCache: CollisionCache, finalPosition: Vector3);
        collideWithWorld(position: Vector3, velocity: Vector3, maximumRetry: number, excludedMeshUniqueId: Nullable<number>): void;
        private checkCollision(mesh);
        private processCollisionsForSubMeshes(transformMatrix, mesh);
        private collideForSubMesh(subMesh, transformMatrix, meshGeometry);
        private checkSubmeshCollision(subMesh);
    }
    interface ICollisionDetector {
        onInit(payload: InitPayload): void;
        onUpdate(payload: UpdatePayload): void;
        onCollision(payload: CollidePayload): void;
    }
    class CollisionDetectorTransferable implements ICollisionDetector {
        private _collisionCache;
        onInit(payload: InitPayload): void;
        onUpdate(payload: UpdatePayload): void;
        onCollision(payload: CollidePayload): void;
    }
}

declare module BABYLON {
    class IntersectionInfo {
        bu: Nullable<number>;
        bv: Nullable<number>;
        distance: number;
        faceId: number;
        subMeshId: number;
        constructor(bu: Nullable<number>, bv: Nullable<number>, distance: number);
    }
    class PickingInfo {
        hit: boolean;
        distance: number;
        pickedPoint: Nullable<Vector3>;
        pickedMesh: Nullable<AbstractMesh>;
        bu: number;
        bv: number;
        faceId: number;
        subMeshId: number;
        pickedSprite: Nullable<Sprite>;
        getNormal(useWorldCoordinates?: boolean, useVerticesNormals?: boolean): Nullable<Vector3>;
        getTextureCoordinates(): Nullable<Vector2>;
    }
}

declare module BABYLON {
    class ArcRotateCamera extends TargetCamera {
        alpha: number;
        beta: number;
        radius: number;
        protected _target: Vector3;
        protected _targetHost: Nullable<AbstractMesh>;
        target: Vector3;
        inertialAlphaOffset: number;
        inertialBetaOffset: number;
        inertialRadiusOffset: number;
        lowerAlphaLimit: Nullable<number>;
        upperAlphaLimit: Nullable<number>;
        lowerBetaLimit: number;
        upperBetaLimit: number;
        lowerRadiusLimit: Nullable<number>;
        upperRadiusLimit: Nullable<number>;
        inertialPanningX: number;
        inertialPanningY: number;
        pinchToPanMaxDistance: number;
        panningDistanceLimit: Nullable<number>;
        panningOriginTarget: Vector3;
        panningInertia: number;
        angularSensibilityX: number;
        angularSensibilityY: number;
        pinchPrecision: number;
        pinchDeltaPercentage: number;
        panningSensibility: number;
        keysUp: number[];
        keysDown: number[];
        keysLeft: number[];
        keysRight: number[];
        wheelPrecision: number;
        wheelDeltaPercentage: number;
        zoomOnFactor: number;
        targetScreenOffset: Vector2;
        allowUpsideDown: boolean;
        _viewMatrix: Matrix;
        _useCtrlForPanning: boolean;
        _panningMouseButton: number;
        inputs: ArcRotateCameraInputsManager;
        _reset: () => void;
        panningAxis: Vector3;
        protected _localDirection: Vector3;
        protected _transformedDirection: Vector3;
        private _bouncingBehavior;
        readonly bouncingBehavior: Nullable<BouncingBehavior>;
        useBouncingBehavior: boolean;
        private _framingBehavior;
        readonly framingBehavior: Nullable<FramingBehavior>;
        useFramingBehavior: boolean;
        private _autoRotationBehavior;
        readonly autoRotationBehavior: Nullable<AutoRotationBehavior>;
        useAutoRotationBehavior: boolean;
        onMeshTargetChangedObservable: Observable<Nullable<AbstractMesh>>;
        onCollide: (collidedMesh: AbstractMesh) => void;
        checkCollisions: boolean;
        collisionRadius: Vector3;
        protected _collider: Collider;
        protected _previousPosition: Vector3;
        protected _collisionVelocity: Vector3;
        protected _newPosition: Vector3;
        protected _previousAlpha: number;
        protected _previousBeta: number;
        protected _previousRadius: number;
        protected _collisionTriggered: boolean;
        protected _targetBoundingCenter: Nullable<Vector3>;
        private _computationVector;
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, scene: Scene, setActiveOnSceneIfNoneActive?: boolean);
        _initCache(): void;
        _updateCache(ignoreParentClass?: boolean): void;
        protected _getTargetPosition(): Vector3;
        /**
         * Store current camera state (fov, position, etc..)
         */
        private _storedAlpha;
        private _storedBeta;
        private _storedRadius;
        private _storedTarget;
        storeState(): Camera;
        /**
         * Restored camera state. You must call storeState() first
         */
        _restoreStateValues(): boolean;
        _isSynchronizedViewMatrix(): boolean;
        attachControl(element: HTMLElement, noPreventDefault?: boolean, useCtrlForPanning?: boolean, panningMouseButton?: number): void;
        detachControl(element: HTMLElement): void;
        _checkInputs(): void;
        protected _checkLimits(): void;
        rebuildAnglesAndRadius(): void;
        setPosition(position: Vector3): void;
        setTarget(target: AbstractMesh | Vector3, toBoundingCenter?: boolean, allowSamePosition?: boolean): void;
        _getViewMatrix(): Matrix;
        protected _onCollisionPositionChange: (collisionId: number, newPosition: Vector3, collidedMesh?: Nullable<AbstractMesh>) => void;
        zoomOn(meshes?: AbstractMesh[], doNotUpdateMaxZ?: boolean): void;
        focusOn(meshesOrMinMaxVectorAndDistance: AbstractMesh[] | {
            min: Vector3;
            max: Vector3;
            distance: number;
        }, doNotUpdateMaxZ?: boolean): void;
        /**
         * @override
         * Override Camera.createRigCamera
         */
        createRigCamera(name: string, cameraIndex: number): Camera;
        /**
         * @override
         * Override Camera._updateRigCameras
         */
        _updateRigCameras(): void;
        dispose(): void;
        getClassName(): string;
    }
}

declare module BABYLON {
    class ArcRotateCameraInputsManager extends CameraInputsManager<ArcRotateCamera> {
        constructor(camera: ArcRotateCamera);
        addMouseWheel(): ArcRotateCameraInputsManager;
        addPointers(): ArcRotateCameraInputsManager;
        addKeyboard(): ArcRotateCameraInputsManager;
        addGamepad(): ArcRotateCameraInputsManager;
        addVRDeviceOrientation(): ArcRotateCameraInputsManager;
    }
}

declare module BABYLON {
    class Camera extends Node {
        inputs: CameraInputsManager<Camera>;
        private static _PERSPECTIVE_CAMERA;
        private static _ORTHOGRAPHIC_CAMERA;
        private static _FOVMODE_VERTICAL_FIXED;
        private static _FOVMODE_HORIZONTAL_FIXED;
        private static _RIG_MODE_NONE;
        private static _RIG_MODE_STEREOSCOPIC_ANAGLYPH;
        private static _RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_PARALLEL;
        private static _RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_CROSSEYED;
        private static _RIG_MODE_STEREOSCOPIC_OVERUNDER;
        private static _RIG_MODE_VR;
        private static _RIG_MODE_WEBVR;
        static readonly PERSPECTIVE_CAMERA: number;
        static readonly ORTHOGRAPHIC_CAMERA: number;
        /**
         * This is the default FOV mode for perspective cameras.
         * This setting aligns the upper and lower bounds of the viewport to the upper and lower bounds of the camera frustum.
         *
         */
        static readonly FOVMODE_VERTICAL_FIXED: number;
        /**
         * This setting aligns the left and right bounds of the viewport to the left and right bounds of the camera frustum.
         *
         */
        static readonly FOVMODE_HORIZONTAL_FIXED: number;
        static readonly RIG_MODE_NONE: number;
        static readonly RIG_MODE_STEREOSCOPIC_ANAGLYPH: number;
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_PARALLEL: number;
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_CROSSEYED: number;
        static readonly RIG_MODE_STEREOSCOPIC_OVERUNDER: number;
        static readonly RIG_MODE_VR: number;
        static readonly RIG_MODE_WEBVR: number;
        static ForceAttachControlToAlwaysPreventDefault: boolean;
        static UseAlternateWebVRRendering: boolean;
        position: Vector3;
        /**
         * The vector the camera should consider as up.
         * (default is Vector3(0, 1, 0) aka Vector3.Up())
         */
        upVector: Vector3;
        orthoLeft: Nullable<number>;
        orthoRight: Nullable<number>;
        orthoBottom: Nullable<number>;
        orthoTop: Nullable<number>;
        /**
         * FOV is set in Radians. (default is 0.8)
         */
        fov: number;
        minZ: number;
        maxZ: number;
        inertia: number;
        mode: number;
        isIntermediate: boolean;
        viewport: Viewport;
        /**
         * Restricts the camera to viewing objects with the same layerMask.
         * A camera with a layerMask of 1 will render mesh.layerMask & camera.layerMask!== 0
         */
        layerMask: number;
        /**
         * fovMode sets the camera frustum bounds to the viewport bounds. (default is FOVMODE_VERTICAL_FIXED)
         */
        fovMode: number;
        cameraRigMode: number;
        interaxialDistance: number;
        isStereoscopicSideBySide: boolean;
        _cameraRigParams: any;
        _rigCameras: Camera[];
        _rigPostProcess: Nullable<PostProcess>;
        protected _webvrViewMatrix: Matrix;
        _skipRendering: boolean;
        _alternateCamera: Camera;
        customRenderTargets: RenderTargetTexture[];
        onViewMatrixChangedObservable: Observable<Camera>;
        onProjectionMatrixChangedObservable: Observable<Camera>;
        onAfterCheckInputsObservable: Observable<Camera>;
        onRestoreStateObservable: Observable<Camera>;
        private _computedViewMatrix;
        _projectionMatrix: Matrix;
        private _doNotComputeProjectionMatrix;
        private _worldMatrix;
        _postProcesses: Nullable<PostProcess>[];
        private _transformMatrix;
        _activeMeshes: SmartArray<AbstractMesh>;
        protected _globalPosition: Vector3;
        private _frustumPlanes;
        private _refreshFrustumPlanes;
        constructor(name: string, position: Vector3, scene: Scene, setActiveOnSceneIfNoneActive?: boolean);
        private _storedFov;
        private _stateStored;
        /**
         * Store current camera state (fov, position, etc..)
         */
        storeState(): Camera;
        /**
         * Restores the camera state values if it has been stored. You must call storeState() first
         */
        protected _restoreStateValues(): boolean;
        /**
         * Restored camera state. You must call storeState() first
         */
        restoreState(): boolean;
        getClassName(): string;
        /**
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        readonly globalPosition: Vector3;
        getActiveMeshes(): SmartArray<AbstractMesh>;
        isActiveMesh(mesh: Mesh): boolean;
        /**
         * Is this camera ready to be used/rendered
         * @param completeCheck defines if a complete check (including post processes) has to be done (false by default)
         * @return true if the camera is ready
         */
        isReady(completeCheck?: boolean): boolean;
        _initCache(): void;
        _updateCache(ignoreParentClass?: boolean): void;
        _isSynchronized(): boolean;
        _isSynchronizedViewMatrix(): boolean;
        _isSynchronizedProjectionMatrix(): boolean;
        attachControl(element: HTMLElement, noPreventDefault?: boolean): void;
        detachControl(element: HTMLElement): void;
        update(): void;
        _checkInputs(): void;
        readonly rigCameras: Camera[];
        readonly rigPostProcess: Nullable<PostProcess>;
        /**
         * Internal, gets the first post proces.
         * @returns the first post process to be run on this camera.
         */
        _getFirstPostProcess(): Nullable<PostProcess>;
        private _cascadePostProcessesToRigCams();
        attachPostProcess(postProcess: PostProcess, insertAt?: Nullable<number>): number;
        detachPostProcess(postProcess: PostProcess): void;
        getWorldMatrix(): Matrix;
        _getViewMatrix(): Matrix;
        getViewMatrix(force?: boolean): Matrix;
        freezeProjectionMatrix(projection?: Matrix): void;
        unfreezeProjectionMatrix(): void;
        getProjectionMatrix(force?: boolean): Matrix;
        getTranformationMatrix(): Matrix;
        private updateFrustumPlanes();
        isInFrustum(target: ICullable): boolean;
        isCompletelyInFrustum(target: ICullable): boolean;
        getForwardRay(length?: number, transform?: Matrix, origin?: Vector3): Ray;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        readonly leftCamera: Nullable<FreeCamera>;
        readonly rightCamera: Nullable<FreeCamera>;
        getLeftTarget(): Nullable<Vector3>;
        getRightTarget(): Nullable<Vector3>;
        setCameraRigMode(mode: number, rigParams: any): void;
        private _getVRProjectionMatrix();
        protected _updateCameraRotationMatrix(): void;
        protected _updateWebVRCameraRotationMatrix(): void;
        /**
         * This function MUST be overwritten by the different WebVR cameras available.
         * The context in which it is running is the RIG camera. So 'this' is the TargetCamera, left or right.
         */
        protected _getWebVRProjectionMatrix(): Matrix;
        /**
         * This function MUST be overwritten by the different WebVR cameras available.
         * The context in which it is running is the RIG camera. So 'this' is the TargetCamera, left or right.
         */
        protected _getWebVRViewMatrix(): Matrix;
        setCameraRigParameter(name: string, value: any): void;
        /**
         * needs to be overridden by children so sub has required properties to be copied
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * May need to be overridden by children
         */
        _updateRigCameras(): void;
        _setupInputs(): void;
        serialize(): any;
        clone(name: string): Camera;
        getDirection(localAxis: Vector3): Vector3;
        getDirectionToRef(localAxis: Vector3, result: Vector3): void;
        static GetConstructorFromName(type: string, name: string, scene: Scene, interaxial_distance?: number, isStereoscopicSideBySide?: boolean): () => Camera;
        computeWorldMatrix(): Matrix;
        static Parse(parsedCamera: any, scene: Scene): Camera;
    }
}

declare module BABYLON {
    var CameraInputTypes: {};
    interface ICameraInput<TCamera extends Camera> {
        camera: Nullable<TCamera>;
        getClassName(): string;
        getSimpleName(): string;
        attachControl: (element: HTMLElement, noPreventDefault?: boolean) => void;
        detachControl: (element: Nullable<HTMLElement>) => void;
        checkInputs?: () => void;
    }
    interface CameraInputsMap<TCamera extends Camera> {
        [name: string]: ICameraInput<TCamera>;
        [idx: number]: ICameraInput<TCamera>;
    }
    class CameraInputsManager<TCamera extends Camera> {
        attached: CameraInputsMap<TCamera>;
        attachedElement: Nullable<HTMLElement>;
        noPreventDefault: boolean;
        camera: TCamera;
        checkInputs: () => void;
        constructor(camera: TCamera);
        /**
         * Add an input method to a camera.
         * builtin inputs example: camera.inputs.addGamepad();
         * custom inputs example: camera.inputs.add(new BABYLON.FreeCameraGamepadInput());
         * @param input camera input method
         */
        add(input: ICameraInput<TCamera>): void;
        /**
         * Remove a specific input method from a camera
         * example: camera.inputs.remove(camera.inputs.attached.mouse);
         * @param inputToRemove camera input method
         */
        remove(inputToRemove: ICameraInput<TCamera>): void;
        removeByType(inputType: string): void;
        private _addCheckInputs(fn);
        attachInput(input: ICameraInput<TCamera>): void;
        attachElement(element: HTMLElement, noPreventDefault?: boolean): void;
        detachElement(element: HTMLElement, disconnect?: boolean): void;
        rebuildInputCheck(): void;
        /**
         * Remove all attached input methods from a camera
         */
        clear(): void;
        serialize(serializedCamera: any): void;
        parse(parsedCamera: any): void;
    }
}

declare module BABYLON {
    /**
     * This is a camera specifically designed to react to device orientation events such as a modern mobile device
     * being tilted forward or back and left or right.
     */
    class DeviceOrientationCamera extends FreeCamera {
        private _initialQuaternion;
        private _quaternionCache;
        /**
         * Creates a new device orientation camera
         * @param name The name of the camera
         * @param position The start position camera
         * @param scene The scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene: Scene);
        /**
         * Gets the current instance class name ("DeviceOrientationCamera").
         * This helps avoiding instanceof at run time.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Checks and applies the current values of the inputs to the camera. (Internal use only)
         */
        _checkInputs(): void;
        /**
         * Reset the camera to its default orientation on the specified axis only.
         * @param axis The axis to reset
         */
        resetToCurrentRotation(axis?: Axis): void;
    }
}

declare module BABYLON {
    class FollowCamera extends TargetCamera {
        radius: number;
        rotationOffset: number;
        heightOffset: number;
        cameraAcceleration: number;
        maxCameraSpeed: number;
        lockedTarget: Nullable<AbstractMesh>;
        constructor(name: string, position: Vector3, scene: Scene, lockedTarget?: Nullable<AbstractMesh>);
        private getRadians(degrees);
        private follow(cameraTarget);
        _checkInputs(): void;
        getClassName(): string;
    }
    class ArcFollowCamera extends TargetCamera {
        alpha: number;
        beta: number;
        radius: number;
        target: Nullable<AbstractMesh>;
        private _cartesianCoordinates;
        constructor(name: string, alpha: number, beta: number, radius: number, target: Nullable<AbstractMesh>, scene: Scene);
        private follow();
        _checkInputs(): void;
        getClassName(): string;
    }
}

declare module BABYLON {
    class FreeCamera extends TargetCamera {
        ellipsoid: Vector3;
        ellipsoidOffset: Vector3;
        checkCollisions: boolean;
        applyGravity: boolean;
        inputs: FreeCameraInputsManager;
        /**
         * Gets the input sensibility for a mouse input. (default is 2000.0)
         * Higher values reduce sensitivity.
         */
        /**
         * Sets the input sensibility for a mouse input. (default is 2000.0)
         * Higher values reduce sensitivity.
         */
        angularSensibility: number;
        keysUp: number[];
        keysDown: number[];
        keysLeft: number[];
        keysRight: number[];
        onCollide: (collidedMesh: AbstractMesh) => void;
        private _collider;
        private _needMoveForGravity;
        private _oldPosition;
        private _diffPosition;
        private _newPosition;
        _localDirection: Vector3;
        _transformedDirection: Vector3;
        constructor(name: string, position: Vector3, scene: Scene, setActiveOnSceneIfNoneActive?: boolean);
        attachControl(element: HTMLElement, noPreventDefault?: boolean): void;
        detachControl(element: HTMLElement): void;
        private _collisionMask;
        collisionMask: number;
        _collideWithWorld(displacement: Vector3): void;
        private _onCollisionPositionChange;
        _checkInputs(): void;
        _decideIfNeedsToMove(): boolean;
        _updatePosition(): void;
        dispose(): void;
        getClassName(): string;
    }
}

declare module BABYLON {
    class FreeCameraInputsManager extends CameraInputsManager<FreeCamera> {
        constructor(camera: FreeCamera);
        addKeyboard(): FreeCameraInputsManager;
        addMouse(touchEnabled?: boolean): FreeCameraInputsManager;
        addGamepad(): FreeCameraInputsManager;
        addDeviceOrientation(): FreeCameraInputsManager;
        addTouch(): FreeCameraInputsManager;
        addVirtualJoystick(): FreeCameraInputsManager;
    }
}

declare module BABYLON {
    class GamepadCamera extends UniversalCamera {
        gamepadAngularSensibility: number;
        gamepadMoveSensibility: number;
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
    }
}

declare module BABYLON {
    /**
     * Camera used to simulate anaglyphic rendering (based on FreeCamera)
     */
    class AnaglyphFreeCamera extends FreeCamera {
        /**
         * Creates a new AnaglyphFreeCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, scene: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphFreeCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate anaglyphic rendering (based on ArcRotateCamera)
     */
    class AnaglyphArcRotateCamera extends ArcRotateCamera {
        /**
         * Creates a new AnaglyphArcRotateCamera
         * @param name defines camera name
         * @param alpha defines alpha angle (in radians)
         * @param beta defines beta angle (in radians)
         * @param radius defines radius
         * @param target defines camera target
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, interaxialDistance: number, scene: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphArcRotateCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate anaglyphic rendering (based on GamepadCamera)
     */
    class AnaglyphGamepadCamera extends GamepadCamera {
        /**
         * Creates a new AnaglyphGamepadCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, scene: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphGamepadCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate anaglyphic rendering (based on UniversalCamera)
     */
    class AnaglyphUniversalCamera extends UniversalCamera {
        /**
         * Creates a new AnaglyphUniversalCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, scene: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphUniversalCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate stereoscopic rendering (based on FreeCamera)
     */
    class StereoscopicFreeCamera extends FreeCamera {
        /**
         * Creates a new StereoscopicFreeCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicFreeCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate stereoscopic rendering (based on ArcRotateCamera)
     */
    class StereoscopicArcRotateCamera extends ArcRotateCamera {
        /**
         * Creates a new StereoscopicArcRotateCamera
         * @param name defines camera name
         * @param alpha defines alpha angle (in radians)
         * @param beta defines beta angle (in radians)
         * @param radius defines radius
         * @param target defines camera target
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicArcRotateCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate stereoscopic rendering (based on GamepadCamera)
     */
    class StereoscopicGamepadCamera extends GamepadCamera {
        /**
         * Creates a new StereoscopicGamepadCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicGamepadCamera
         */
        getClassName(): string;
    }
    /**
     * Camera used to simulate stereoscopic rendering (based on UniversalCamera)
     */
    class StereoscopicUniversalCamera extends UniversalCamera {
        /**
         * Creates a new StereoscopicUniversalCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicUniversalCamera
         */
        getClassName(): string;
    }
}

declare module BABYLON {
    class TargetCamera extends Camera {
        cameraDirection: Vector3;
        cameraRotation: Vector2;
        rotation: Vector3;
        rotationQuaternion: Quaternion;
        speed: number;
        noRotationConstraint: boolean;
        lockedTarget: any;
        _currentTarget: Vector3;
        _viewMatrix: Matrix;
        _camMatrix: Matrix;
        _cameraTransformMatrix: Matrix;
        _cameraRotationMatrix: Matrix;
        private _rigCamTransformMatrix;
        _referencePoint: Vector3;
        private _currentUpVector;
        _transformedReferencePoint: Vector3;
        protected _globalCurrentTarget: Vector3;
        protected _globalCurrentUpVector: Vector3;
        _reset: () => void;
        constructor(name: string, position: Vector3, scene: Scene, setActiveOnSceneIfNoneActive?: boolean);
        getFrontPosition(distance: number): Vector3;
        _getLockedTargetPosition(): Nullable<Vector3>;
        /**
         * Store current camera state (fov, position, etc..)
         */
        private _storedPosition;
        private _storedRotation;
        private _storedRotationQuaternion;
        storeState(): Camera;
        /**
         * Restored camera state. You must call storeState() first
         */
        _restoreStateValues(): boolean;
        _initCache(): void;
        _updateCache(ignoreParentClass?: boolean): void;
        _isSynchronizedViewMatrix(): boolean;
        _computeLocalCameraSpeed(): number;
        setTarget(target: Vector3): void;
        /**
         * Return the current target position of the camera. This value is expressed in local space.
         */
        getTarget(): Vector3;
        _decideIfNeedsToMove(): boolean;
        _updatePosition(): void;
        _checkInputs(): void;
        protected _updateCameraRotationMatrix(): void;
        _getViewMatrix(): Matrix;
        protected _computeViewMatrix(position: Vector3, target: Vector3, up: Vector3): void;
        /**
         * @override
         * Override Camera.createRigCamera
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * @override
         * Override Camera._updateRigCameras
         */
        _updateRigCameras(): void;
        private _getRigCamPosition(halfSpace, result);
        getClassName(): string;
    }
}

declare module BABYLON {
    class TouchCamera extends FreeCamera {
        touchAngularSensibility: number;
        touchMoveSensibility: number;
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
        _setupInputs(): void;
    }
}

declare module BABYLON {
    class UniversalCamera extends TouchCamera {
        gamepadAngularSensibility: number;
        gamepadMoveSensibility: number;
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
    }
}

declare module BABYLON {
    class VirtualJoysticksCamera extends FreeCamera {
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
    }
}

interface VRDisplay extends EventTarget {
    /**
     * Dictionary of capabilities describing the VRDisplay.
     */
    readonly capabilities: VRDisplayCapabilities;
    /**
     * z-depth defining the far plane of the eye view frustum
     * enables mapping of values in the render target depth
     * attachment to scene coordinates. Initially set to 10000.0.
     */
    depthFar: number;
    /**
     * z-depth defining the near plane of the eye view frustum
     * enables mapping of values in the render target depth
     * attachment to scene coordinates. Initially set to 0.01.
     */
    depthNear: number;
    /**
     * An identifier for this distinct VRDisplay. Used as an
     * association point in the Gamepad API.
     */
    readonly displayId: number;
    /**
     * A display name, a user-readable name identifying it.
     */
    readonly displayName: string;
    readonly isConnected: boolean;
    readonly isPresenting: boolean;
    /**
     * If this VRDisplay supports room-scale experiences, the optional
     * stage attribute contains details on the room-scale parameters.
     */
    readonly stageParameters: VRStageParameters | null;
    /**
     * Passing the value returned by `requestAnimationFrame` to
     * `cancelAnimationFrame` will unregister the callback.
     */
    cancelAnimationFrame(handle: number): void;
    /**
     * Stops presenting to the VRDisplay.
     */
    exitPresent(): Promise<void>;
    getEyeParameters(whichEye: string): VREyeParameters;
    /**
     * Populates the passed VRFrameData with the information required to render
     * the current frame.
     */
    getFrameData(frameData: VRFrameData): boolean;
    /**
     * Get the layers currently being presented.
     */
    getLayers(): VRLayer[];
    /**
     * Return a VRPose containing the future predicted pose of the VRDisplay
     * when the current frame will be presented. The value returned will not
     * change until JavaScript has returned control to the browser.
     *
     * The VRPose will contain the position, orientation, velocity,
     * and acceleration of each of these properties.
     */
    getPose(): VRPose;
    /**
     * Return the current instantaneous pose of the VRDisplay, with no
     * prediction applied.
     */
    getImmediatePose(): VRPose;
    /**
     * The callback passed to `requestAnimationFrame` will be called
     * any time a new frame should be rendered. When the VRDisplay is
     * presenting the callback will be called at the native refresh
     * rate of the HMD. When not presenting this function acts
     * identically to how window.requestAnimationFrame acts. Content should
     * make no assumptions of frame rate or vsync behavior as the HMD runs
     * asynchronously from other displays and at differing refresh rates.
     */
    requestAnimationFrame(callback: FrameRequestCallback): number;
    /**
     * Begin presenting to the VRDisplay. Must be called in response to a user gesture.
     * Repeat calls while already presenting will update the VRLayers being displayed.
     */
    requestPresent(layers: VRLayer[]): Promise<void>;
    /**
     * Reset the pose for this display, treating its current position and
     * orientation as the "origin/zero" values. VRPose.position,
     * VRPose.orientation, and VRStageParameters.sittingToStandingTransform may be
     * updated when calling resetPose(). This should be called in only
     * sitting-space experiences.
     */
    resetPose(): void;
    /**
     * The VRLayer provided to the VRDisplay will be captured and presented
     * in the HMD. Calling this function has the same effect on the source
     * canvas as any other operation that uses its source image, and canvases
     * created without preserveDrawingBuffer set to true will be cleared.
     */
    submitFrame(pose?: VRPose): void;
}
declare var VRDisplay: {
    prototype: VRDisplay;
    new (): VRDisplay;
};
interface VRLayer {
    leftBounds?: number[] | null;
    rightBounds?: number[] | null;
    source?: HTMLCanvasElement | null;
}
interface VRDisplayCapabilities {
    readonly canPresent: boolean;
    readonly hasExternalDisplay: boolean;
    readonly hasOrientation: boolean;
    readonly hasPosition: boolean;
    readonly maxLayers: number;
}
interface VREyeParameters {
    /** @deprecated */
    readonly fieldOfView: VRFieldOfView;
    readonly offset: Float32Array;
    readonly renderHeight: number;
    readonly renderWidth: number;
}
interface VRFieldOfView {
    readonly downDegrees: number;
    readonly leftDegrees: number;
    readonly rightDegrees: number;
    readonly upDegrees: number;
}
interface VRFrameData {
    readonly leftProjectionMatrix: Float32Array;
    readonly leftViewMatrix: Float32Array;
    readonly pose: VRPose;
    readonly rightProjectionMatrix: Float32Array;
    readonly rightViewMatrix: Float32Array;
    readonly timestamp: number;
}
interface VRPose {
    readonly angularAcceleration: Float32Array | null;
    readonly angularVelocity: Float32Array | null;
    readonly linearAcceleration: Float32Array | null;
    readonly linearVelocity: Float32Array | null;
    readonly orientation: Float32Array | null;
    readonly position: Float32Array | null;
    readonly timestamp: number;
}
interface VRStageParameters {
    sittingToStandingTransform?: Float32Array;
    sizeX?: number;
    sizeY?: number;
}
interface Navigator {
    getVRDisplays(): Promise<VRDisplay[]>;
    readonly activeVRDisplays: ReadonlyArray<VRDisplay>;
}
interface Window {
    onvrdisplayconnected: ((this: Window, ev: Event) => any) | null;
    onvrdisplaydisconnected: ((this: Window, ev: Event) => any) | null;
    onvrdisplaypresentchange: ((this: Window, ev: Event) => any) | null;
    addEventListener(type: "vrdisplayconnected", listener: (ev: Event) => any, useCapture?: boolean): void;
    addEventListener(type: "vrdisplaydisconnected", listener: (ev: Event) => any, useCapture?: boolean): void;
    addEventListener(type: "vrdisplaypresentchange", listener: (ev: Event) => any, useCapture?: boolean): void;
}
interface Gamepad {
    readonly displayId: number;
}

/**
 * Module Debug contains the (visual) components to debug a scene correctly
 */
declare module BABYLON.Debug {
    /**
     * The Axes viewer will show 3 axes in a specific point in space
     */
    class AxesViewer {
        private _xline;
        private _yline;
        private _zline;
        private _xmesh;
        private _ymesh;
        private _zmesh;
        scene: Nullable<Scene>;
        scaleLines: number;
        constructor(scene: Scene, scaleLines?: number);
        update(position: Vector3, xaxis: Vector3, yaxis: Vector3, zaxis: Vector3): void;
        dispose(): void;
    }
}

declare module BABYLON.Debug {
    /**
     * The BoneAxesViewer will attach 3 axes to a specific bone of a specific mesh
     */
    class BoneAxesViewer extends AxesViewer {
        mesh: Nullable<Mesh>;
        bone: Nullable<Bone>;
        pos: Vector3;
        xaxis: Vector3;
        yaxis: Vector3;
        zaxis: Vector3;
        constructor(scene: Scene, bone: Bone, mesh: Mesh, scaleLines?: number);
        update(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class DebugLayer {
        private _scene;
        static InspectorURL: string;
        private _inspector;
        private BJSINSPECTOR;
        constructor(scene: Scene);
        /** Creates the inspector window. */
        private _createInspector(config?);
        isVisible(): boolean;
        hide(): void;
        show(config?: {
            popup?: boolean;
            initialTab?: number;
            parentElement?: HTMLElement;
            newColors?: {
                backgroundColor?: string;
                backgroundColorLighter?: string;
                backgroundColorLighter2?: string;
                backgroundColorLighter3?: string;
                color?: string;
                colorTop?: string;
                colorBot?: string;
            };
        }): void;
    }
}

declare module BABYLON.Debug {
    /**
     * Used to show the physics impostor around the specific mesh.
     */
    class PhysicsViewer {
        protected _impostors: Array<Nullable<PhysicsImpostor>>;
        protected _meshes: Array<Nullable<AbstractMesh>>;
        protected _scene: Nullable<Scene>;
        protected _numMeshes: number;
        protected _physicsEnginePlugin: Nullable<IPhysicsEnginePlugin>;
        private _renderFunction;
        private _debugBoxMesh;
        private _debugSphereMesh;
        private _debugMaterial;
        constructor(scene: Scene);
        protected _updateDebugMeshes(): void;
        showImpostor(impostor: PhysicsImpostor): void;
        hideImpostor(impostor: Nullable<PhysicsImpostor>): void;
        private _getDebugMaterial(scene);
        private _getDebugBoxMesh(scene);
        private _getDebugSphereMesh(scene);
        private _getDebugMesh(impostor, scene);
        dispose(): void;
    }
}

declare module BABYLON {
    class RayHelper {
        ray: Nullable<Ray>;
        private _renderPoints;
        private _renderLine;
        private _renderFunction;
        private _scene;
        private _updateToMeshFunction;
        private _attachedToMesh;
        private _meshSpaceDirection;
        private _meshSpaceOrigin;
        static CreateAndShow(ray: Ray, scene: Scene, color: Color3): RayHelper;
        constructor(ray: Ray);
        show(scene: Scene, color: Color3): void;
        hide(): void;
        private _render();
        attachToMesh(mesh: AbstractMesh, meshSpaceDirection?: Vector3, meshSpaceOrigin?: Vector3, length?: number): void;
        detachFromMesh(): void;
        private _updateToMesh();
        dispose(): void;
    }
}

declare module BABYLON.Debug {
    /**
    * Demo available here: http://www.babylonjs-playground.com/#1BZJVJ#8
    */
    class SkeletonViewer {
        skeleton: Skeleton;
        mesh: AbstractMesh;
        autoUpdateBonesMatrices: boolean;
        renderingGroupId: number;
        color: Color3;
        private _scene;
        private _debugLines;
        private _debugMesh;
        private _isEnabled;
        private _renderFunction;
        constructor(skeleton: Skeleton, mesh: AbstractMesh, scene: Scene, autoUpdateBonesMatrices?: boolean, renderingGroupId?: number);
        isEnabled: boolean;
        private _getBonePosition(position, bone, meshMat, x?, y?, z?);
        private _getLinesForBonesWithLength(bones, meshMat);
        private _getLinesForBonesNoLength(bones, meshMat);
        update(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Interface for attribute information associated with buffer instanciation
     */
    class InstancingAttributeInfo {
        /**
         * Index/offset of the attribute in the vertex shader
         */
        index: number;
        /**
         * size of the attribute, 1, 2, 3 or 4
         */
        attributeSize: number;
        /**
         * type of the attribute, gl.BYTE, gl.UNSIGNED_BYTE, gl.SHORT, gl.UNSIGNED_SHORT, gl.FIXED, gl.FLOAT.
         * default is FLOAT
         */
        attribyteType: number;
        /**
         * normalization of fixed-point data. behavior unclear, use FALSE, default is FALSE
         */
        normalized: boolean;
        /**
         * Offset of the data in the Vertex Buffer acting as the instancing buffer
         */
        offset: number;
        /**
         * Name of the GLSL attribute, for debugging purpose only
         */
        attributeName: string;
    }
    /**
     * Define options used to create a render target texture
     */
    class RenderTargetCreationOptions {
        /**
         * Specifies is mipmaps must be generated
         */
        generateMipMaps?: boolean;
        /** Specifies whether or not a depth should be allocated in the texture (true by default) */
        generateDepthBuffer?: boolean;
        /** Specifies whether or not a stencil should be allocated in the texture (false by default)*/
        generateStencilBuffer?: boolean;
        /** Defines texture type (int by default) */
        type?: number;
        /** Defines sampling mode (trilinear by default) */
        samplingMode?: number;
        /** Defines format (RGBA by default) */
        format?: number;
    }
    /**
     * Define options used to create a depth texture
     */
    class DepthTextureCreationOptions {
        /** Specifies whether or not a stencil should be allocated in the texture */
        generateStencil?: boolean;
        /** Specifies whether or not bilinear filtering is enable on the texture */
        bilinearFiltering?: boolean;
        /** Specifies the comparison function to set on the texture. If 0 or undefined, the texture is not in comparison mode */
        comparisonFunction?: number;
        /** Specifies if the created texture is a cube texture */
        isCube?: boolean;
    }
    /**
     * Class used to describe the capabilities of the engine relatively to the current browser
     */
    class EngineCapabilities {
        /** Maximum textures units per fragment shader */
        maxTexturesImageUnits: number;
        /** Maximum texture units per vertex shader */
        maxVertexTextureImageUnits: number;
        /** Maximum textures units in the entire pipeline */
        maxCombinedTexturesImageUnits: number;
        /** Maximum texture size */
        maxTextureSize: number;
        /** Maximum cube texture size */
        maxCubemapTextureSize: number;
        /** Maximum render texture size */
        maxRenderTextureSize: number;
        /** Maximum number of vertex attributes */
        maxVertexAttribs: number;
        /** Maximum number of varyings */
        maxVaryingVectors: number;
        /** Maximum number of uniforms per vertex shader */
        maxVertexUniformVectors: number;
        /** Maximum number of uniforms per fragment shader */
        maxFragmentUniformVectors: number;
        /** Defines if standard derivates (dx/dy) are supported */
        standardDerivatives: boolean;
        /** Defines if s3tc texture compression is supported */
        s3tc: Nullable<WEBGL_compressed_texture_s3tc>;
        /** Defines if pvrtc texture compression is supported */
        pvrtc: any;
        /** Defines if etc1 texture compression is supported */
        etc1: any;
        /** Defines if etc2 texture compression is supported */
        etc2: any;
        /** Defines if astc texture compression is supported */
        astc: any;
        /** Defines if float textures are supported */
        textureFloat: boolean;
        /** Defines if vertex array objects are supported */
        vertexArrayObject: boolean;
        /** Gets the webgl extension for anisotropic filtering (null if not supported) */
        textureAnisotropicFilterExtension: Nullable<EXT_texture_filter_anisotropic>;
        /** Gets the maximum level of anisotropy supported */
        maxAnisotropy: number;
        /** Defines if instancing is supported */
        instancedArrays: boolean;
        /** Defines if 32 bits indices are supported */
        uintIndices: boolean;
        /** Defines if high precision shaders are supported */
        highPrecisionShaderSupported: boolean;
        /** Defines if depth reading in the fragment shader is supported */
        fragmentDepthSupported: boolean;
        /** Defines if float texture linear filtering is supported*/
        textureFloatLinearFiltering: boolean;
        /** Defines if rendering to float textures is supported */
        textureFloatRender: boolean;
        /** Defines if half float textures are supported*/
        textureHalfFloat: boolean;
        /** Defines if half float texture linear filtering is supported*/
        textureHalfFloatLinearFiltering: boolean;
        /** Defines if rendering to half float textures is supported */
        textureHalfFloatRender: boolean;
        /** Defines if textureLOD shader command is supported */
        textureLOD: boolean;
        /** Defines if draw buffers extension is supported */
        drawBuffersExtension: boolean;
        /** Defines if depth textures are supported */
        depthTextureExtension: boolean;
        /** Defines if float color buffer are supported */
        colorBufferFloat: boolean;
        /** Gets disjoint timer query extension (null if not supported) */
        timerQuery: EXT_disjoint_timer_query;
        /** Defines if timestamp can be used with timer query */
        canUseTimestampForTimerQuery: boolean;
    }
    /** Interface defining initialization parameters for Engine class */
    interface EngineOptions extends WebGLContextAttributes {
        /**
         * Defines if the engine should no exceed a specified device ratio
         * @see https://developer.mozilla.org/en-US/docs/Web/API/Window/devicePixelRatio
         */
        limitDeviceRatio?: number;
        /**
         * Defines if webvr should be enabled automatically
         * @see http://doc.babylonjs.com/how_to/webvr_camera
         */
        autoEnableWebVR?: boolean;
        /**
         * Defines if webgl2 should be turned off even if supported
         * @see http://doc.babylonjs.com/features/webgl2
         */
        disableWebGL2Support?: boolean;
        /**
         * Defines if webaudio should be initialized as well
         * @see http://doc.babylonjs.com/how_to/playing_sounds_and_music
         */
        audioEngine?: boolean;
        /**
         * Defines if animations should run using a deterministic lock step
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         */
        deterministicLockstep?: boolean;
        /** Defines the maximum steps to use with deterministic lock step mode */
        lockstepMaxSteps?: number;
        /**
         * Defines that engine should ignore context lost events
         * If this event happens when this parameter is true, you will have to reload the page to restore rendering
         */
        doNotHandleContextLost?: boolean;
    }
    /**
     * Defines the interface used by display changed events
     */
    interface IDisplayChangedEventArgs {
        /** Gets the vrDisplay object (if any) */
        vrDisplay: Nullable<any>;
        /** Gets a boolean indicating if webVR is supported */
        vrSupported: boolean;
    }
    /**
     * The engine class is responsible for interfacing with all lower-level APIs such as WebGL and Audio
     */
    class Engine {
        /** Use this array to turn off some WebGL2 features on known buggy browsers version */
        static ExceptionList: ({
            key: string;
            capture: string;
            captureConstraint: number;
            targets: string[];
        } | {
            key: string;
            capture: null;
            captureConstraint: null;
            targets: string[];
        })[];
        /** Gets the list of created engines */
        static Instances: Engine[];
        /**
         * Gets the latest created engine
         */
        static readonly LastCreatedEngine: Nullable<Engine>;
        /**
         * Gets the latest created scene
         */
        static readonly LastCreatedScene: Nullable<Scene>;
        /**
         * Will flag all materials in all scenes in all engines as dirty to trigger new shader compilation
         * @param flag defines which part of the materials must be marked as dirty
         * @param predicate defines a predicate used to filter which materials should be affected
         */
        static MarkAllMaterialsAsDirty(flag: number, predicate?: (mat: Material) => boolean): void;
        private static _ALPHA_DISABLE;
        private static _ALPHA_ADD;
        private static _ALPHA_COMBINE;
        private static _ALPHA_SUBTRACT;
        private static _ALPHA_MULTIPLY;
        private static _ALPHA_MAXIMIZED;
        private static _ALPHA_ONEONE;
        private static _ALPHA_PREMULTIPLIED;
        private static _ALPHA_PREMULTIPLIED_PORTERDUFF;
        private static _ALPHA_INTERPOLATE;
        private static _ALPHA_SCREENMODE;
        private static _DELAYLOADSTATE_NONE;
        private static _DELAYLOADSTATE_LOADED;
        private static _DELAYLOADSTATE_LOADING;
        private static _DELAYLOADSTATE_NOTLOADED;
        private static _TEXTUREFORMAT_ALPHA;
        private static _TEXTUREFORMAT_LUMINANCE;
        private static _TEXTUREFORMAT_LUMINANCE_ALPHA;
        private static _TEXTUREFORMAT_RGB;
        private static _TEXTUREFORMAT_RGBA;
        private static _TEXTUREFORMAT_R32F;
        private static _TEXTUREFORMAT_RG32F;
        private static _TEXTUREFORMAT_RGB32F;
        private static _TEXTUREFORMAT_RGBA32F;
        private static _TEXTURETYPE_UNSIGNED_INT;
        private static _TEXTURETYPE_FLOAT;
        private static _TEXTURETYPE_HALF_FLOAT;
        private static _NEVER;
        private static _ALWAYS;
        private static _LESS;
        private static _EQUAL;
        private static _LEQUAL;
        private static _GREATER;
        private static _GEQUAL;
        private static _NOTEQUAL;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will never pass. i.e. Nothing will be drawn */
        static readonly NEVER: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will always pass. i.e. Pixels will be drawn in the order they are drawn */
        static readonly ALWAYS: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than the stored value */
        static readonly LESS: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is equals to the stored value */
        static readonly EQUAL: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than or equal to the stored value */
        static readonly LEQUAL: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than the stored value */
        static readonly GREATER: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than or equal to the stored value */
        static readonly GEQUAL: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is not equal to the stored value */
        static readonly NOTEQUAL: number;
        private static _KEEP;
        private static _REPLACE;
        private static _INCR;
        private static _DECR;
        private static _INVERT;
        private static _INCR_WRAP;
        private static _DECR_WRAP;
        /** Passed to stencilOperation to specify that stencil value must be kept */
        static readonly KEEP: number;
        /** Passed to stencilOperation to specify that stencil value must be replaced */
        static readonly REPLACE: number;
        /** Passed to stencilOperation to specify that stencil value must be incremented */
        static readonly INCR: number;
        /** Passed to stencilOperation to specify that stencil value must be decremented */
        static readonly DECR: number;
        /** Passed to stencilOperation to specify that stencil value must be inverted */
        static readonly INVERT: number;
        /** Passed to stencilOperation to specify that stencil value must be incremented with wrapping */
        static readonly INCR_WRAP: number;
        /** Passed to stencilOperation to specify that stencil value must be decremented with wrapping */
        static readonly DECR_WRAP: number;
        /** Defines that alpha blending is disabled */
        static readonly ALPHA_DISABLE: number;
        /** Defines that alpha blending to SRC + DEST */
        static readonly ALPHA_ONEONE: number;
        /** Defines that alpha blending to SRC ALPHA * SRC + DEST */
        static readonly ALPHA_ADD: number;
        /** Defines that alpha blending to SRC ALPHA * SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_COMBINE: number;
        /** Defines that alpha blending to DEST - SRC * DEST */
        static readonly ALPHA_SUBTRACT: number;
        /** Defines that alpha blending to SRC * DEST */
        static readonly ALPHA_MULTIPLY: number;
        /** Defines that alpha blending to SRC ALPHA * SRC + (1 - SRC) * DEST */
        static readonly ALPHA_MAXIMIZED: number;
        /** Defines that alpha blending to SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_PREMULTIPLIED: number;
        /**
         * Defines that alpha blending to SRC + (1 - SRC ALPHA) * DEST
         * Alpha will be set to (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_PREMULTIPLIED_PORTERDUFF: number;
        /** Defines that alpha blending to CST * SRC + (1 - CST) * DEST */
        static readonly ALPHA_INTERPOLATE: number;
        /**
         * Defines that alpha blending to SRC + (1 - SRC) * DEST
         * Alpha will be set to SRC ALPHA + (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_SCREENMODE: number;
        /** Defines that the ressource is not delayed*/
        static readonly DELAYLOADSTATE_NONE: number;
        /** Defines that the ressource was successfully delay loaded */
        static readonly DELAYLOADSTATE_LOADED: number;
        /** Defines that the ressource is currently delay loading */
        static readonly DELAYLOADSTATE_LOADING: number;
        /** Defines that the ressource is delayed and has not started loading */
        static readonly DELAYLOADSTATE_NOTLOADED: number;
        /** ALPHA */
        static readonly TEXTUREFORMAT_ALPHA: number;
        /** LUMINANCE */
        static readonly TEXTUREFORMAT_LUMINANCE: number;
        /**
         * R32F
         */
        static readonly TEXTUREFORMAT_R32F: number;
        /**
         * RG32F
         */
        static readonly TEXTUREFORMAT_RG32F: number;
        /**
         * RGB32F
         */
        static readonly TEXTUREFORMAT_RGB32F: number;
        /**
         * RGBA32F
         */
        static readonly TEXTUREFORMAT_RGBA32F: number;
        /** LUMINANCE_ALPHA */
        static readonly TEXTUREFORMAT_LUMINANCE_ALPHA: number;
        /** RGB */
        static readonly TEXTUREFORMAT_RGB: number;
        /** RGBA */
        static readonly TEXTUREFORMAT_RGBA: number;
        /** UNSIGNED_INT */
        static readonly TEXTURETYPE_UNSIGNED_INT: number;
        /** FLOAT */
        static readonly TEXTURETYPE_FLOAT: number;
        /** HALF_FLOAT */
        static readonly TEXTURETYPE_HALF_FLOAT: number;
        private static _SCALEMODE_FLOOR;
        private static _SCALEMODE_NEAREST;
        private static _SCALEMODE_CEILING;
        /** Defines that texture rescaling will use a floor to find the closer power of 2 size */
        static readonly SCALEMODE_FLOOR: number;
        /** Defines that texture rescaling will look for the nearest power of 2 size */
        static readonly SCALEMODE_NEAREST: number;
        /** Defines that texture rescaling will use a ceil to find the closer power of 2 size */
        static readonly SCALEMODE_CEILING: number;
        /**
         * Returns the current version of the framework
         */
        static readonly Version: string;
        /**
         * Gets or sets the epsilon value used by collision engine
         */
        static CollisionsEpsilon: number;
        /**
         * Gets or sets the relative url used to load code if using the engine in non-minified mode
         */
        static CodeRepository: string;
        /**
         * Gets or sets the relative url used to load shaders if using the engine in non-minified mode
         */
        static ShadersRepository: string;
        /**
         * Gets or sets a boolean that indicates if textures must be forced to power of 2 size even if not required
         */
        forcePOTTextures: boolean;
        /**
         * Gets a boolean indicating if the engine is currently rendering in fullscreen mode
         */
        isFullscreen: boolean;
        /**
         * Gets a boolean indicating if the pointer is currently locked
         */
        isPointerLock: boolean;
        /**
         * Gets or sets a boolean indicating if back faces must be culled (true by default)
         */
        cullBackFaces: boolean;
        /**
         * Gets or sets a boolean indicating if the engine must keep rendering even if the window is not in foregroun
         */
        renderEvenInBackground: boolean;
        /**
         * Gets or sets a boolean indicating that cache can be kept between frames
         */
        preventCacheWipeBetweenFrames: boolean;
        /**
         * Gets or sets a boolean to enable/disable IndexedDB support and avoid XHR on .manifest
         **/
        enableOfflineSupport: boolean;
        /**
         * Gets or sets a boolean to enable/disable checking manifest if IndexedDB support is enabled (Babylon.js will always consider the database is up to date)
         **/
        disableManifestCheck: boolean;
        /**
         * Gets the list of created scenes
         */
        scenes: Scene[];
        /**
         * Gets the list of created postprocesses
         */
        postProcesses: PostProcess[];
        /**
         * Observable event triggered each time the rendering canvas is resized
         */
        onResizeObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas loses focus
         */
        onCanvasBlurObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas gains focus
         */
        onCanvasFocusObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas receives pointerout event
         */
        onCanvasPointerOutObservable: Observable<PointerEvent>;
        /**
         * Observable event triggered before each texture is initialized
         */
        onBeforeTextureInitObservable: Observable<Texture>;
        private _vrDisplay;
        private _vrSupported;
        private _oldSize;
        private _oldHardwareScaleFactor;
        private _vrExclusivePointerMode;
        private _webVRInitPromise;
        /**
         * Gets a boolean indicating that the engine is currently in VR exclusive mode for the pointers
         * @see https://docs.microsoft.com/en-us/microsoft-edge/webvr/essentials#mouse-input
         */
        readonly isInVRExclusivePointerMode: boolean;
        /**
         * Gets or sets a boolean indicating that uniform buffers must be disabled even if they are supported
         */
        disableUniformBuffers: boolean;
        /** @hidden */
        _uniformBuffers: UniformBuffer[];
        /**
         * Gets a boolean indicating that the engine supports uniform buffers
         * @see http://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
         */
        readonly supportsUniformBuffers: boolean;
        /**
         * Observable raised when the engine begins a new frame
         */
        onBeginFrameObservable: Observable<Engine>;
        /**
         * Observable raised when the engine ends the current frame
         */
        onEndFrameObservable: Observable<Engine>;
        /**
         * Observable raised when the engine is about to compile a shader
         */
        onBeforeShaderCompilationObservable: Observable<Engine>;
        /**
         * Observable raised when the engine has jsut compiled a shader
         */
        onAfterShaderCompilationObservable: Observable<Engine>;
        private _gl;
        private _renderingCanvas;
        private _windowIsBackground;
        private _webGLVersion;
        /**
         * Gets a boolean indicating that only power of 2 textures are supported
         * Please note that you can still use non power of 2 textures but in this case the engine will forcefully convert them
         */
        readonly needPOTTextures: boolean;
        /** @hidden */
        _badOS: boolean;
        /** @hidden */
        _badDesktopOS: boolean;
        /**
         * Gets or sets a value indicating if we want to disable texture binding optmization.
         * This could be required on some buggy drivers which wants to have textures bound in a progressive order.
         * By default Babylon.js will try to let textures bound where they are and only update the samplers to point where the texture is
         */
        disableTextureBindingOptimization: boolean;
        /**
         * Gets the audio engine
         * @see http://doc.babylonjs.com/how_to/playing_sounds_and_music
         * @ignorenaming
         */
        static audioEngine: AudioEngine;
        private _onFocus;
        private _onBlur;
        private _onCanvasPointerOut;
        private _onCanvasBlur;
        private _onCanvasFocus;
        private _onFullscreenChange;
        private _onPointerLockChange;
        private _onVRDisplayPointerRestricted;
        private _onVRDisplayPointerUnrestricted;
        private _onVrDisplayConnect;
        private _onVrDisplayDisconnect;
        private _onVrDisplayPresentChange;
        /**
         * Observable signaled when VR display mode changes
         */
        onVRDisplayChangedObservable: Observable<IDisplayChangedEventArgs>;
        /**
         * Observable signaled when VR request present is complete
         */
        onVRRequestPresentComplete: Observable<boolean>;
        /**
         * Observable signaled when VR request present starts
         */
        onVRRequestPresentStart: Observable<Engine>;
        private _hardwareScalingLevel;
        /** @hidden */
        protected _caps: EngineCapabilities;
        private _pointerLockRequested;
        private _isStencilEnable;
        private _colorWrite;
        private _loadingScreen;
        /** @hidden */
        _drawCalls: PerfCounter;
        /** @hidden */
        _textureCollisions: PerfCounter;
        private _glVersion;
        private _glRenderer;
        private _glVendor;
        private _videoTextureSupported;
        private _renderingQueueLaunched;
        private _activeRenderLoops;
        private _deterministicLockstep;
        private _lockstepMaxSteps;
        /**
         * Observable signaled when a context lost event is raised
         */
        onContextLostObservable: Observable<Engine>;
        /**
         * Observable signaled when a context restored event is raised
         */
        onContextRestoredObservable: Observable<Engine>;
        private _onContextLost;
        private _onContextRestored;
        private _contextWasLost;
        private _doNotHandleContextLost;
        private _performanceMonitor;
        private _fps;
        private _deltaTime;
        /**
         * Turn this value on if you want to pause FPS computation when in background
         */
        disablePerformanceMonitorInBackground: boolean;
        /**
         * Gets the performance monitor attached to this engine
         * @see http://doc.babylonjs.com/how_to/optimizing_your_scene#engineinstrumentation
         */
        readonly performanceMonitor: PerformanceMonitor;
        /** @hidden */
        protected _depthCullingState: _DepthCullingState;
        /** @hidden */
        protected _stencilState: _StencilState;
        /** @hidden */
        protected _alphaState: _AlphaState;
        /** @hidden */
        protected _alphaMode: number;
        private _internalTexturesCache;
        /** @hidden */
        protected _activeChannel: number;
        private _currentTextureChannel;
        /** @hidden */
        protected _boundTexturesCache: {
            [key: string]: Nullable<InternalTexture>;
        };
        /** @hidden */
        protected _currentEffect: Nullable<Effect>;
        /** @hidden */
        protected _currentProgram: Nullable<WebGLProgram>;
        private _compiledEffects;
        private _vertexAttribArraysEnabled;
        /** @hidden */
        protected _cachedViewport: Nullable<Viewport>;
        private _cachedVertexArrayObject;
        /** @hidden */
        protected _cachedVertexBuffers: any;
        /** @hidden */
        protected _cachedIndexBuffer: Nullable<WebGLBuffer>;
        /** @hidden */
        protected _cachedEffectForVertexBuffers: Nullable<Effect>;
        /** @hidden */
        protected _currentRenderTarget: Nullable<InternalTexture>;
        private _uintIndicesCurrentlySet;
        private _currentBoundBuffer;
        /** @hidden */
        protected _currentFramebuffer: Nullable<WebGLFramebuffer>;
        private _currentBufferPointers;
        private _currentInstanceLocations;
        private _currentInstanceBuffers;
        private _textureUnits;
        private _firstBoundInternalTextureTracker;
        private _lastBoundInternalTextureTracker;
        private _workingCanvas;
        private _workingContext;
        private _rescalePostProcess;
        private _dummyFramebuffer;
        private _externalData;
        private _bindedRenderFunction;
        private _vaoRecordInProgress;
        private _mustWipeVertexAttributes;
        private _emptyTexture;
        private _emptyCubeTexture;
        private _emptyTexture3D;
        private _frameHandler;
        private _nextFreeTextureSlots;
        private _maxSimultaneousTextures;
        private _activeRequests;
        private _texturesSupported;
        private _textureFormatInUse;
        /**
         * Gets the list of texture formats supported
         */
        readonly texturesSupported: Array<string>;
        /**
         * Gets the list of texture formats in use
         */
        readonly textureFormatInUse: Nullable<string>;
        /**
         * Gets the current viewport
         */
        readonly currentViewport: Nullable<Viewport>;
        /**
         * Gets the default empty texture
         */
        readonly emptyTexture: InternalTexture;
        /**
         * Gets the default empty 3D texture
         */
        readonly emptyTexture3D: InternalTexture;
        /**
         * Gets the default empty cube texture
         */
        readonly emptyCubeTexture: InternalTexture;
        /**
         * Creates a new engine
         * @param canvasOrContext defines the canvas or WebGL context to use for rendering
         * @param antialias defines enable antialiasing (default: false)
         * @param options defines further options to be sent to the getContext() function
         * @param adaptToDeviceRatio defines whether to adapt to the device's viewport characteristics (default: false)
         */
        constructor(canvasOrContext: Nullable<HTMLCanvasElement | WebGLRenderingContext>, antialias?: boolean, options?: EngineOptions, adaptToDeviceRatio?: boolean);
        private _rebuildInternalTextures();
        private _rebuildEffects();
        private _rebuildBuffers();
        private _initGLContext();
        /**
         * Gets version of the current webGL context
         */
        readonly webGLVersion: number;
        /**
         * Returns true if the stencil buffer has been enabled through the creation option of the context.
         */
        readonly isStencilEnable: boolean;
        private _prepareWorkingCanvas();
        /**
         * Reset the texture cache to empty state
         */
        resetTextureCache(): void;
        /**
         * Gets a boolean indicating that the engine is running in deterministic lock step mode
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         * @returns true if engine is in deterministic lock step mode
         */
        isDeterministicLockStep(): boolean;
        /**
         * Gets the max steps when engine is running in deterministic lock step
         * @see http://doc.babylonjs.com/babylon101/animations#deterministic-lockstep
         * @returns the max steps
         */
        getLockstepMaxSteps(): number;
        /**
         * Gets an object containing information about the current webGL context
         * @returns an object containing the vender, the renderer and the version of the current webGL context
         */
        getGlInfo(): {
            vendor: string;
            renderer: string;
            version: string;
        };
        /**
         * Gets current aspect ratio
         * @param camera defines the camera to use to get the aspect ratio
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the aspect ratio
         */
        getAspectRatio(camera: Camera, useScreen?: boolean): number;
        /**
         * Gets the current render width
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the current render width
         */
        getRenderWidth(useScreen?: boolean): number;
        /**
         * Gets the current render height
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the current render height
         */
        getRenderHeight(useScreen?: boolean): number;
        /**
         * Gets the HTML canvas attached with the current webGL context
         * @returns a HTML canvas
         */
        getRenderingCanvas(): Nullable<HTMLCanvasElement>;
        /**
         * Gets the client rect of the HTML canvas attached with the current webGL context
         * @returns a client rectanglee
         */
        getRenderingCanvasClientRect(): Nullable<ClientRect>;
        /**
         * Defines the hardware scaling level.
         * By default the hardware scaling level is computed from the window device ratio.
         * if level = 1 then the engine will render at the exact resolution of the canvas. If level = 0.5 then the engine will render at twice the size of the canvas.
         * @param level defines the level to use
         */
        setHardwareScalingLevel(level: number): void;
        /**
         * Gets the current hardware scaling level.
         * By default the hardware scaling level is computed from the window device ratio.
         * if level = 1 then the engine will render at the exact resolution of the canvas. If level = 0.5 then the engine will render at twice the size of the canvas.
         * @returns a number indicating the current hardware scaling level
         */
        getHardwareScalingLevel(): number;
        /**
         * Gets the list of loaded textures
         * @returns an array containing all loaded textures
         */
        getLoadedTexturesCache(): InternalTexture[];
        /**
         * Gets the object containing all engine capabilities
         * @returns the EngineCapabilities object
         */
        getCaps(): EngineCapabilities;
        /** @hidden */
        readonly drawCalls: number;
        /** @hidden */
        readonly drawCallsPerfCounter: Nullable<PerfCounter>;
        /**
         * Gets the current depth function
         * @returns a number defining the depth function
         */
        getDepthFunction(): Nullable<number>;
        /**
         * Sets the current depth function
         * @param depthFunc defines the function to use
         */
        setDepthFunction(depthFunc: number): void;
        /**
         * Sets the current depth function to GREATER
         */
        setDepthFunctionToGreater(): void;
        /**
         * Sets the current depth function to GEQUAL
         */
        setDepthFunctionToGreaterOrEqual(): void;
        /**
         * Sets the current depth function to LESS
         */
        setDepthFunctionToLess(): void;
        /**
         * Sets the current depth function to LEQUAL
         */
        setDepthFunctionToLessOrEqual(): void;
        /**
         * Gets a boolean indicating if stencil buffer is enabled
         * @returns the current stencil buffer state
         */
        getStencilBuffer(): boolean;
        /**
         * Enable or disable the stencil buffer
         * @param enable defines if the stencil buffer must be enabled or disabled
         */
        setStencilBuffer(enable: boolean): void;
        /**
         * Gets the current stencil mask
         * @returns a number defining the new stencil mask to use
         */
        getStencilMask(): number;
        /**
         * Sets the current stencil mask
         * @param mask defines the new stencil mask to use
         */
        setStencilMask(mask: number): void;
        /**
         * Gets the current stencil function
         * @returns a number defining the stencil function to use
         */
        getStencilFunction(): number;
        /**
         * Gets the current stencil reference value
         * @returns a number defining the stencil reference value to use
         */
        getStencilFunctionReference(): number;
        /**
         * Gets the current stencil mask
         * @returns a number defining the stencil mask to use
         */
        getStencilFunctionMask(): number;
        /**
         * Sets the current stencil function
         * @param stencilFunc defines the new stencil function to use
         */
        setStencilFunction(stencilFunc: number): void;
        /**
         * Sets the current stencil reference
         * @param reference defines the new stencil reference to use
         */
        setStencilFunctionReference(reference: number): void;
        /**
         * Sets the current stencil mask
         * @param mask defines the new stencil mask to use
         */
        setStencilFunctionMask(mask: number): void;
        /**
         * Gets the current stencil operation when stencil fails
         * @returns a number defining stencil operation to use when stencil fails
         */
        getStencilOperationFail(): number;
        /**
         * Gets the current stencil operation when depth fails
         * @returns a number defining stencil operation to use when depth fails
         */
        getStencilOperationDepthFail(): number;
        /**
         * Gets the current stencil operation when stencil passes
         * @returns a number defining stencil operation to use when stencil passes
         */
        getStencilOperationPass(): number;
        /**
         * Sets the stencil operation to use when stencil fails
         * @param operation defines the stencil operation to use when stencil fails
         */
        setStencilOperationFail(operation: number): void;
        /**
         * Sets the stencil operation to use when depth fails
         * @param operation defines the stencil operation to use when depth fails
         */
        setStencilOperationDepthFail(operation: number): void;
        /**
         * Sets the stencil operation to use when stencil passes
         * @param operation defines the stencil operation to use when stencil passes
         */
        setStencilOperationPass(operation: number): void;
        /**
         * Sets a boolean indicating if the dithering state is enabled or disabled
         * @param value defines the dithering state
         */
        setDitheringState(value: boolean): void;
        /**
         * Sets a boolean indicating if the rasterizer state is enabled or disabled
         * @param value defines the rasterizer state
         */
        setRasterizerState(value: boolean): void;
        /**
         * stop executing a render loop function and remove it from the execution array
         * @param renderFunction defines the function to be removed. If not provided all functions will be removed.
         */
        stopRenderLoop(renderFunction?: () => void): void;
        /** @hidden */
        _renderLoop(): void;
        /**
         * Register and execute a render loop. The engine can have more than one render function
         * @param renderFunction defines the function to continuously execute
         */
        runRenderLoop(renderFunction: () => void): void;
        /**
         * Toggle full screen mode
         * @param requestPointerLock defines if a pointer lock should be requested from the user
         * @param options defines an option object to be sent to the requestFullscreen function
         */
        switchFullscreen(requestPointerLock: boolean): void;
        /**
         * Clear the current render buffer or the current render target (if any is set up)
         * @param color defines the color to use
         * @param backBuffer defines if the back buffer must be cleared
         * @param depth defines if the depth buffer must be cleared
         * @param stencil defines if the stencil buffer must be cleared
         */
        clear(color: Nullable<Color4>, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        /**
         * Executes a scissor clear (ie. a clear on a specific portion of the screen)
         * @param x defines the x-coordinate of the top left corner of the clear rectangle
         * @param y defines the y-coordinate of the corner of the clear rectangle
         * @param width defines the width of the clear rectangle
         * @param height defines the height of the clear rectangle
         * @param clearColor defines the clear color
         */
        scissorClear(x: number, y: number, width: number, height: number, clearColor: Color4): void;
        /**
         * Set the WebGL's viewport
         * @param viewport defines the viewport element to be used
         * @param requiredWidth defines the width required for rendering. If not provided the rendering canvas' width is used
         * @param requiredHeight defines the height required for rendering. If not provided the rendering canvas' height is used
         */
        setViewport(viewport: Viewport, requiredWidth?: number, requiredHeight?: number): void;
        /**
         * Directly set the WebGL Viewport
         * @param x defines the x coordinate of the viewport (in screen space)
         * @param y defines the y coordinate of the viewport (in screen space)
         * @param width defines the width of the viewport (in screen space)
         * @param height defines the height of the viewport (in screen space)
         * @return the current viewport Object (if any) that is being replaced by this call. You can restore this viewport later on to go back to the original state
         */
        setDirectViewport(x: number, y: number, width: number, height: number): Nullable<Viewport>;
        /**
         * Begin a new frame
         */
        beginFrame(): void;
        /**
         * Enf the current frame
         */
        endFrame(): void;
        /**
         * Resize the view according to the canvas' size
         */
        resize(): void;
        /**
         * Force a specific size of the canvas
         * @param width defines the new canvas' width
         * @param height defines the new canvas' height
         */
        setSize(width: number, height: number): void;
        /**
         * Gets a boolean indicating if a webVR device was detected
         * @returns true if a webVR device was detected
         */
        isVRDevicePresent(): boolean;
        /**
         * Gets the current webVR device
         * @returns the current webVR device (or null)
         */
        getVRDevice(): any;
        /**
         * Initializes a webVR display and starts listening to display change events
         * The onVRDisplayChangedObservable will be notified upon these changes
         * @returns The onVRDisplayChangedObservable
         */
        initWebVR(): Observable<IDisplayChangedEventArgs>;
        /**
         * Initializes a webVR display and starts listening to display change events
         * The onVRDisplayChangedObservable will be notified upon these changes
         * @returns A promise containing a VRDisplay and if vr is supported
         */
        initWebVRAsync(): Promise<IDisplayChangedEventArgs>;
        /**
         * Call this function to switch to webVR mode
         * Will do nothing if webVR is not supported or if there is no webVR device
         * @see http://doc.babylonjs.com/how_to/webvr_camera
         */
        enableVR(): void;
        /**
         * Call this function to leave webVR mode
         * Will do nothing if webVR is not supported or if there is no webVR device
         * @see http://doc.babylonjs.com/how_to/webvr_camera
         */
        disableVR(): void;
        private _onVRFullScreenTriggered;
        private _getVRDisplaysAsync();
        /**
         * Binds the frame buffer to the specified texture.
         * @param texture The texture to render to or null for the default canvas
         * @param faceIndex The face of the texture to render to in case of cube texture
         * @param requiredWidth The width of the target to render to
         * @param requiredHeight The height of the target to render to
         * @param forceFullscreenViewport Forces the viewport to be the entire texture/screen if true
         * @param depthStencilTexture The depth stencil texture to use to render
         */
        bindFramebuffer(texture: InternalTexture, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean, depthStencilTexture?: InternalTexture): void;
        private bindUnboundFramebuffer(framebuffer);
        /**
         * Unbind the current render target texture from the webGL context
         * @param texture defines the render target texture to unbind
         * @param disableGenerateMipMaps defines a boolean indicating that mipmaps must not be generated
         * @param onBeforeUnbind defines a function which will be called before the effective unbind
         */
        unBindFramebuffer(texture: InternalTexture, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        /**
         * Unbind a list of render target textures from the webGL context
         * This is used only when drawBuffer extension or webGL2 are active
         * @param textures defines the render target textures to unbind
         * @param disableGenerateMipMaps defines a boolean indicating that mipmaps must not be generated
         * @param onBeforeUnbind defines a function which will be called before the effective unbind
         */
        unBindMultiColorAttachmentFramebuffer(textures: InternalTexture[], disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        /**
         * Force the mipmap generation for the given render target texture
         * @param texture defines the render target texture to use
         */
        generateMipMapsForCubemap(texture: InternalTexture): void;
        /**
         * Force a webGL flush (ie. a flush of all waiting webGL commands)
         */
        flushFramebuffer(): void;
        /**
         * Unbind the current render target and bind the default framebuffer
         */
        restoreDefaultFramebuffer(): void;
        /**
         * Create an uniform buffer
         * @see http://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
         * @param elements defines the content of the uniform buffer
         * @returns the webGL uniform buffer
         */
        createUniformBuffer(elements: FloatArray): WebGLBuffer;
        /**
         * Create a dynamic uniform buffer
         * @see http://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
         * @param elements defines the content of the uniform buffer
         * @returns the webGL uniform buffer
         */
        createDynamicUniformBuffer(elements: FloatArray): WebGLBuffer;
        /**
         * Update an existing uniform buffer
         * @see http://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
         * @param uniformBuffer defines the target uniform buffer
         * @param elements defines the content to update
         * @param offset defines the offset in the uniform buffer where update should start
         * @param count defines the size of the data to update
         */
        updateUniformBuffer(uniformBuffer: WebGLBuffer, elements: FloatArray, offset?: number, count?: number): void;
        private _resetVertexBufferBinding();
        /**
         * Creates a vertex buffer
         * @param data the data for the vertex buffer
         * @returns the new WebGL static buffer
         */
        createVertexBuffer(data: DataArray): WebGLBuffer;
        /**
         * Creates a dynamic vertex buffer
         * @param data the data for the dynamic vertex buffer
         * @returns the new WebGL dynamic buffer
         */
        createDynamicVertexBuffer(data: DataArray): WebGLBuffer;
        /**
         * Update a dynamic index buffer
         * @param indexBuffer defines the target index buffer
         * @param indices defines the data to update
         * @param offset defines the offset in the target index buffer where update should start
         */
        updateDynamicIndexBuffer(indexBuffer: WebGLBuffer, indices: IndicesArray, offset?: number): void;
        /**
         * Updates a dynamic vertex buffer.
         * @param vertexBuffer the vertex buffer to update
         * @param data the data used to update the vertex buffer
         * @param byteOffset the byte offset of the data
         * @param byteLength the byte length of the data
         */
        updateDynamicVertexBuffer(vertexBuffer: WebGLBuffer, data: DataArray, byteOffset?: number, byteLength?: number): void;
        private _resetIndexBufferBinding();
        /**
         * Creates a new index buffer
         * @param indices defines the content of the index buffer
         * @param updatable defines if the index buffer must be updatable
         * @returns a new webGL buffer
         */
        createIndexBuffer(indices: IndicesArray, updatable?: boolean): WebGLBuffer;
        /**
         * Bind a webGL buffer to the webGL context
         * @param buffer defines the buffer to bind
         */
        bindArrayBuffer(buffer: Nullable<WebGLBuffer>): void;
        /**
         * Bind an uniform buffer to the current webGL context
         * @param buffer defines the buffer to bind
         */
        bindUniformBuffer(buffer: Nullable<WebGLBuffer>): void;
        /**
         * Bind a buffer to the current webGL context at a given location
         * @param buffer defines the buffer to bind
         * @param location defines the index where to bind the buffer
         */
        bindUniformBufferBase(buffer: WebGLBuffer, location: number): void;
        /**
         * Bind a specific block at a given index in a specific shader program
         * @param shaderProgram defines the shader program
         * @param blockName defines the block name
         * @param index defines the index where to bind the block
         */
        bindUniformBlock(shaderProgram: WebGLProgram, blockName: string, index: number): void;
        private bindIndexBuffer(buffer);
        private bindBuffer(buffer, target);
        /**
         * update the bound buffer with the given data
         * @param data defines the data to update
         */
        updateArrayBuffer(data: Float32Array): void;
        private _vertexAttribPointer(buffer, indx, size, type, normalized, stride, offset);
        private _bindIndexBufferWithCache(indexBuffer);
        private _bindVertexBuffersAttributes(vertexBuffers, effect);
        /**
         * Records a vertex array object
         * @see http://doc.babylonjs.com/features/webgl2#vertex-array-objects
         * @param vertexBuffers defines the list of vertex buffers to store
         * @param indexBuffer defines the index buffer to store
         * @param effect defines the effect to store
         * @returns the new vertex array object
         */
        recordVertexArrayObject(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: Nullable<WebGLBuffer>, effect: Effect): WebGLVertexArrayObject;
        /**
         * Bind a specific vertex array object
         * @see http://doc.babylonjs.com/features/webgl2#vertex-array-objects
         * @param vertexArrayObject defines the vertex array object to bind
         * @param indexBuffer defines the index buffer to bind
         */
        bindVertexArrayObject(vertexArrayObject: WebGLVertexArrayObject, indexBuffer: Nullable<WebGLBuffer>): void;
        /**
         * Bind webGl buffers directly to the webGL context
         * @param vertexBuffer defines the vertex buffer to bind
         * @param indexBuffer defines the index buffer to bind
         * @param vertexDeclaration defines the vertex declaration to use with the vertex buffer
         * @param vertexStrideSize defines the vertex stride of the vertex buffer
         * @param effect defines the effect associated with the vertex buffer
         */
        bindBuffersDirectly(vertexBuffer: WebGLBuffer, indexBuffer: WebGLBuffer, vertexDeclaration: number[], vertexStrideSize: number, effect: Effect): void;
        private _unbindVertexArrayObject();
        /**
         * Bind a list of vertex buffers to the webGL context
         * @param vertexBuffers defines the list of vertex buffers to bind
         * @param indexBuffer defines the index buffer to bind
         * @param effect defines the effect associated with the vertex buffers
         */
        bindBuffers(vertexBuffers: {
            [key: string]: Nullable<VertexBuffer>;
        }, indexBuffer: Nullable<WebGLBuffer>, effect: Effect): void;
        /**
         * Unbind all instance attributes
         */
        unbindInstanceAttributes(): void;
        /**
         * Release and free the memory of a vertex array object
         * @param vao defines the vertex array object to delete
         */
        releaseVertexArrayObject(vao: WebGLVertexArrayObject): void;
        /** @hidden */
        _releaseBuffer(buffer: WebGLBuffer): boolean;
        /**
         * Creates a webGL buffer to use with instanciation
         * @param capacity defines the size of the buffer
         * @returns the webGL buffer
         */
        createInstancesBuffer(capacity: number): WebGLBuffer;
        /**
         * Delete a webGL buffer used with instanciation
         * @param buffer defines the webGL buffer to delete
         */
        deleteInstancesBuffer(buffer: WebGLBuffer): void;
        /**
         * Update the content of a webGL buffer used with instanciation and bind it to the webGL context
         * @param instancesBuffer defines the webGL buffer to update and bind
         * @param data defines the data to store in the buffer
         * @param offsetLocations defines the offsets or attributes information used to determine where data must be stored in the buffer
         */
        updateAndBindInstancesBuffer(instancesBuffer: WebGLBuffer, data: Float32Array, offsetLocations: number[] | InstancingAttributeInfo[]): void;
        /**
         * Apply all cached states (depth, culling, stencil and alpha)
         */
        applyStates(): void;
        /**
         * Send a draw order
         * @param useTriangles defines if triangles must be used to draw (else wireframe will be used)
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instanciation is enabled)
         */
        draw(useTriangles: boolean, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of points
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instanciation is enabled)
         */
        drawPointClouds(verticesStart: number, verticesCount: number, instancesCount?: number): void;
        /**
         * Draw a list of unindexed primitives
         * @param useTriangles defines if triangles must be used to draw (else wireframe will be used)
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instanciation is enabled)
         */
        drawUnIndexed(useTriangles: boolean, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        /**
         * Draw a list of indexed primitives
         * @param fillMode defines the primitive to use
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instanciation is enabled)
         */
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of unindexed primitives
         * @param fillMode defines the primitive to use
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instanciation is enabled)
         */
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        private _drawMode(fillMode);
        /** @hidden */
        _releaseEffect(effect: Effect): void;
        /** @hidden */
        _deleteProgram(program: WebGLProgram): void;
        /**
         * Create a new effect (used to store vertex/fragment shaders)
         * @param baseName defines the base name of the effect (The name of file without .fragment.fx or .vertex.fx)
         * @param attributesNamesOrOptions defines either a list of attribute names or an EffectCreationOptions object
         * @param uniformsNamesOrEngine defines either a list of uniform names or the engine to use
         * @param samplers defines an array of string used to represent textures
         * @param defines defines the string containing the defines to use to compile the shaders
         * @param fallbacks defines the list of potential fallbacks to use if shader conmpilation fails
         * @param onCompiled defines a function to call when the effect creation is successful
         * @param onError defines a function to call when the effect creation has failed
         * @param indexParameters defines an object containing the index values to use to compile shaders (like the maximum number of simultaneous lights)
         * @returns the new Effect
         */
        createEffect(baseName: any, attributesNamesOrOptions: string[] | EffectCreationOptions, uniformsNamesOrEngine: string[] | Engine, samplers?: string[], defines?: string, fallbacks?: EffectFallbacks, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void, indexParameters?: any): Effect;
        /**
         * Create an effect to use with particle systems
         * @param fragmentName defines the base name of the effect (The name of file without .fragment.fx)
         * @param uniformsNames defines a list of attribute names
         * @param samplers defines an array of string used to represent textures
         * @param defines defines the string containing the defines to use to compile the shaders
         * @param fallbacks defines the list of potential fallbacks to use if shader conmpilation fails
         * @param onCompiled defines a function to call when the effect creation is successful
         * @param onError defines a function to call when the effect creation has failed
         * @returns the new Effect
         */
        createEffectForParticles(fragmentName: string, uniformsNames?: string[], samplers?: string[], defines?: string, fallbacks?: EffectFallbacks, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): Effect;
        /**
         * Directly creates a webGL program
         * @param vertexCode defines the vertex shader code to use
         * @param fragmentCode defines the fragment shader code to use
         * @param context defines the webGL context to use (if not set, the current one will be used)
         * @param transformFeedbackVaryings defines the list of transform feedback varyings to use
         * @returns the new webGL program
         */
        createRawShaderProgram(vertexCode: string, fragmentCode: string, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        /**
         * Creates a webGL program
         * @param vertexCode  defines the vertex shader code to use
         * @param fragmentCode defines the fragment shader code to use
         * @param defines defines the string containing the defines to use to compile the shaders
         * @param context defines the webGL context to use (if not set, the current one will be used)
         * @param transformFeedbackVaryings defines the list of transform feedback varyings to use
         * @returns the new webGL program
         */
        createShaderProgram(vertexCode: string, fragmentCode: string, defines: Nullable<string>, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        private _createShaderProgram(vertexShader, fragmentShader, context, transformFeedbackVaryings?);
        /**
         * Gets the list of webGL uniform locations associated with a specific program based on a list of uniform names
         * @param shaderProgram defines the webGL program to use
         * @param uniformsNames defines the list of uniform names
         * @returns an array of webGL uniform locations
         */
        getUniforms(shaderProgram: WebGLProgram, uniformsNames: string[]): Nullable<WebGLUniformLocation>[];
        /**
         * Gets the lsit of active attributes for a given webGL program
         * @param shaderProgram defines the webGL program to use
         * @param attributesNames defines the list of attribute names to get
         * @returns an array of indices indicating the offset of each attribute
         */
        getAttributes(shaderProgram: WebGLProgram, attributesNames: string[]): number[];
        /**
         * Activates an effect, mkaing it the current one (ie. the one used for rendering)
         * @param effect defines the effect to activate
         */
        enableEffect(effect: Nullable<Effect>): void;
        /**
         * Set the value of an uniform to an array of int32
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         */
        setIntArray(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         */
        setIntArray2(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         */
        setIntArray3(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         */
        setIntArray4(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        /**
         * Set the value of an uniform to an array of float32
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         */
        setFloatArray(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        /**
         * Set the value of an uniform to an array of float32 (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         */
        setFloatArray2(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        /**
         * Set the value of an uniform to an array of float32 (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         */
        setFloatArray3(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        /**
         * Set the value of an uniform to an array of float32 (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         */
        setFloatArray4(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        /**
         * Set the value of an uniform to an array of number
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         */
        setArray(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        /**
         * Set the value of an uniform to an array of number (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         */
        setArray2(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        /**
         * Set the value of an uniform to an array of number (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         */
        setArray3(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        /**
         * Set the value of an uniform to an array of number (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         */
        setArray4(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        /**
         * Set the value of an uniform to an array of float32 (stored as matrices)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrices defines the array of float32 to store
         */
        setMatrices(uniform: Nullable<WebGLUniformLocation>, matrices: Float32Array): void;
        /**
         * Set the value of an uniform to a matrix
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the matrix to store
         */
        setMatrix(uniform: Nullable<WebGLUniformLocation>, matrix: Matrix): void;
        /**
         * Set the value of an uniform to a matrix (3x3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the Float32Array representing the 3x3 matrix to store
         */
        setMatrix3x3(uniform: Nullable<WebGLUniformLocation>, matrix: Float32Array): void;
        /**
         * Set the value of an uniform to a matrix (2x2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the Float32Array representing the 2x2 matrix to store
         */
        setMatrix2x2(uniform: Nullable<WebGLUniformLocation>, matrix: Float32Array): void;
        /**
         * Set the value of an uniform to a number (int)
         * @param uniform defines the webGL uniform location where to store the value
         * @param value defines the int number to store
         */
        setInt(uniform: Nullable<WebGLUniformLocation>, value: number): void;
        /**
         * Set the value of an uniform to a number (float)
         * @param uniform defines the webGL uniform location where to store the value
         * @param value defines the float number to store
         */
        setFloat(uniform: Nullable<WebGLUniformLocation>, value: number): void;
        /**
         * Set the value of an uniform to a vec2
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         */
        setFloat2(uniform: Nullable<WebGLUniformLocation>, x: number, y: number): void;
        /**
         * Set the value of an uniform to a vec3
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         */
        setFloat3(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number): void;
        /**
         * Set the value of an uniform to a boolean
         * @param uniform defines the webGL uniform location where to store the value
         * @param bool defines the boolean to store
         */
        setBool(uniform: Nullable<WebGLUniformLocation>, bool: number): void;
        /**
         * Set the value of an uniform to a vec4
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @param w defines the 4th component of the value
         */
        setFloat4(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number, w: number): void;
        /**
         * Set the value of an uniform to a Color3
         * @param uniform defines the webGL uniform location where to store the value
         * @param color3 defines the color to store
         */
        setColor3(uniform: Nullable<WebGLUniformLocation>, color3: Color3): void;
        /**
         * Set the value of an uniform to a Color3 and an alpha value
         * @param uniform defines the webGL uniform location where to store the value
         * @param color3 defines the color to store
         * @param alpha defines the alpha component to store
         */
        setColor4(uniform: Nullable<WebGLUniformLocation>, color3: Color3, alpha: number): void;
        /**
         * Sets a Color4 on a uniform variable
         * @param uniform defines the uniform location
         * @param color4 defines the value to be set
         */
        setDirectColor4(uniform: Nullable<WebGLUniformLocation>, color4: Color4): void;
        /**
         * Set various states to the webGL context
         * @param culling defines backface culling state
         * @param zOffset defines the value to apply to zOffset (0 by default)
         * @param force defines if states must be applied even if cache is up to date
         * @param reverseSide defines if culling must be reversed (CCW instead of CW and CW instead of CCW)
         */
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean): void;
        /**
         * Set the z offset to apply to current rendering
         * @param value defines the offset to apply
         */
        setZOffset(value: number): void;
        /**
         * Gets the current value of the zOffset
         * @returns the current zOffset state
         */
        getZOffset(): number;
        /**
         * Enable or disable depth buffering
         * @param enable defines the state to set
         */
        setDepthBuffer(enable: boolean): void;
        /**
         * Gets a boolean indicating if depth writing is enabled
         * @returns the current depth writing state
         */
        getDepthWrite(): boolean;
        /**
         * Enable or disable depth writing
         * @param enable defines the state to set
         */
        setDepthWrite(enable: boolean): void;
        /**
         * Enable or disable color writing
         * @param enable defines the state to set
         */
        setColorWrite(enable: boolean): void;
        /**
         * Gets a boolean indicating if color writing is enabled
         * @returns the current color writing state
         */
        getColorWrite(): boolean;
        /**
         * Sets alpha constants used by some alpha blending modes
         * @param r defines the red component
         * @param g defines the green component
         * @param b defines the blue component
         * @param a defines the alpha component
         */
        setAlphaConstants(r: number, g: number, b: number, a: number): void;
        /**
         * Sets the current alpha mode
         * @param mode defines the mode to use (one of the BABYLON.Engine.ALPHA_XXX)
         * @param noDepthWriteChange defines if depth writing state should remains unchanged (false by default)
         * @see http://doc.babylonjs.com/resources/transparency_and_how_meshes_are_rendered
         */
        setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
        /**
         * Gets the current alpha mode
         * @see http://doc.babylonjs.com/resources/transparency_and_how_meshes_are_rendered
         * @returns the current alpha mode
         */
        getAlphaMode(): number;
        /**
         * Force the entire cache to be cleared
         * You should not have to use this function unless your engine needs to share the webGL context with another engine
         * @param bruteForce defines a boolean to force clearing ALL caches (including stencil, detoh and alpha states)
         */
        wipeCaches(bruteForce?: boolean): void;
        /**
         * Set the compressed texture format to use, based on the formats you have, and the formats
         * supported by the hardware / browser.
         *
         * Khronos Texture Container (.ktx) files are used to support this.  This format has the
         * advantage of being specifically designed for OpenGL.  Header elements directly correspond
         * to API arguments needed to compressed textures.  This puts the burden on the container
         * generator to house the arcane code for determining these for current & future formats.
         *
         * for description see https://www.khronos.org/opengles/sdk/tools/KTX/
         * for file layout see https://www.khronos.org/opengles/sdk/tools/KTX/file_format_spec/
         *
         * Note: The result of this call is not taken into account when a texture is base64.
         *
         * @param formatsAvailable defines the list of those format families you have created
         * on your server.  Syntax: '-' + format family + '.ktx'.  (Case and order do not matter.)
         *
         * Current families are astc, dxt, pvrtc, etc2, & etc1.
         * @returns The extension selected.
         */
        setTextureFormatToUse(formatsAvailable: Array<string>): Nullable<string>;
        /** @hidden */
        _createTexture(): WebGLTexture;
        /**
         * Usually called from BABYLON.Texture.ts.
         * Passed information to create a WebGLTexture
         * @param urlArg defines a value which contains one of the following:
         * * A conventional http URL, e.g. 'http://...' or 'file://...'
         * * A base64 string of in-line texture data, e.g. 'data:image/jpg;base64,/...'
         * * An indicator that data being passed using the buffer parameter, e.g. 'data:mytexture.jpg'
         * @param noMipmap defines a boolean indicating that no mipmaps shall be generated.  Ignored for compressed textures.  They must be in the file
         * @param invertY when true, image is flipped when loaded.  You probably want true. Ignored for compressed textures.  Must be flipped in the file
         * @param scene needed for loading to the correct scene
         * @param samplingMode mode with should be used sample / access the texture (Default: BABYLON.Texture.TRILINEAR_SAMPLINGMODE)
         * @param onLoad optional callback to be called upon successful completion
         * @param onError optional callback to be called upon failure
         * @param buffer a source of a file previously fetched as either an ArrayBuffer (compressed or image format) or HTMLImageElement (image format)
         * @param fallback an internal argument in case the function must be called again, due to etc1 not having alpha capabilities
         * @param format internal format.  Default: RGB when extension is '.jpg' else RGBA.  Ignored for compressed textures
         * @returns a InternalTexture for assignment back into BABYLON.Texture
         */
        createTexture(urlArg: Nullable<string>, noMipmap: boolean, invertY: boolean, scene: Nullable<Scene>, samplingMode?: number, onLoad?: Nullable<() => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<ArrayBuffer | HTMLImageElement>, fallback?: Nullable<InternalTexture>, format?: Nullable<number>): InternalTexture;
        private _rescaleTexture(source, destination, scene, internalFormat, onComplete);
        /**
         * Update a raw texture
         * @param texture defines the texture to update
         * @param data defines the data to store in the texture
         * @param format defines the format of the data
         * @param invertY defines if data must be stored with Y axis inverted
         * @param compression defines the compression used (null by default)
         * @param type defines the type fo the data (BABYLON.Engine.TEXTURETYPE_UNSIGNED_INT by default)
         */
        updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression?: Nullable<string>, type?: number): void;
        /**
         * Creates a raw texture
         * @param data defines the data to store in the texture
         * @param width defines the width of the texture
         * @param height defines the height of the texture
         * @param format defines the format of the data
         * @param generateMipMaps defines if the engine should generate the mip levels
         * @param invertY defines if data must be stored with Y axis inverted
         * @param samplingMode defines the required sampling mode (BABYLON.Texture.NEAREST_SAMPLINGMODE by default)
         * @param compression defines the compression used (null by default)
         * @param type defines the type fo the data (BABYLON.Engine.TEXTURETYPE_UNSIGNED_INT by default)
         * @returns the raw texture inside an InternalTexture
         */
        createRawTexture(data: Nullable<ArrayBufferView>, width: number, height: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>, type?: number): InternalTexture;
        /**
         * Creates a dynamic texture
         * @param width defines the width of the texture
         * @param height defines the height of the texture
         * @param generateMipMaps defines if the engine should generate the mip levels
         * @param samplingMode defines the required sampling mode (BABYLON.Texture.NEAREST_SAMPLINGMODE by default)
         * @returns the dynamic texture inside an InternalTexture
         */
        createDynamicTexture(width: number, height: number, generateMipMaps: boolean, samplingMode: number): InternalTexture;
        /**
         * Update the sampling mode of a given texture
         * @param samplingMode defines the required sampling mode
         * @param texture defines the texture to update
         */
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture): void;
        /**
         * Update the content of a dynamic texture
         * @param texture defines the texture to update
         * @param canvas defines the canvas containing the source
         * @param invertY defines if data must be stored with Y axis inverted
         * @param premulAlpha defines if alpha is stored as premultiplied
         * @param format defines the format of the data
         */
        updateDynamicTexture(texture: Nullable<InternalTexture>, canvas: HTMLCanvasElement, invertY: boolean, premulAlpha?: boolean, format?: number): void;
        /**
         * Update a video texture
         * @param texture defines the texture to update
         * @param video defines the video element to use
         * @param invertY defines if data must be stored with Y axis inverted
         */
        updateVideoTexture(texture: Nullable<InternalTexture>, video: HTMLVideoElement, invertY: boolean): void;
        /**
         * Updates a depth texture Comparison Mode and Function.
         * If the comparison Function is equal to 0, the mode will be set to none.
         * Otherwise, this only works in webgl 2 and requires a shadow sampler in the shader.
         * @param texture The texture to set the comparison function for
         * @param comparisonFunction The comparison function to set, 0 if no comparison required
         */
        updateTextureComparisonFunction(texture: InternalTexture, comparisonFunction: number): void;
        private _setupDepthStencilTexture(internalTexture, size, generateStencil, bilinearFiltering, comparisonFunction);
        /**
         * Creates a depth stencil texture.
         * This is only available in WebGL 2 or with the depth texture extension available.
         * @param size The size of face edge in the texture.
         * @param options The options defining the texture.
         * @returns The texture
         */
        createDepthStencilTexture(size: number | {
            width: number;
            height: number;
        }, options: DepthTextureCreationOptions): InternalTexture;
        /**
         * Creates a depth stencil texture.
         * This is only available in WebGL 2 or with the depth texture extension available.
         * @param size The size of face edge in the texture.
         * @param options The options defining the texture.
         * @returns The texture
         */
        private _createDepthStencilTexture(size, options);
        /**
         * Creates a depth stencil cube texture.
         * This is only available in WebGL 2.
         * @param size The size of face edge in the cube texture.
         * @param options The options defining the cube texture.
         * @returns The cube texture
         */
        private _createDepthStencilCubeTexture(size, options);
        /**
         * Sets the frame buffer Depth / Stencil attachement of the render target to the defined depth stencil texture.
         * @param renderTarget The render target to set the frame buffer for
         */
        setFrameBufferDepthStencilTexture(renderTarget: RenderTargetTexture): void;
        /**
         * Creates a new render target texture
         * @param size defines the size of the texture
         * @param options defines the options used to create the texture
         * @returns a new render target texture stored in an InternalTexture
         */
        createRenderTargetTexture(size: number | {
            width: number;
            height: number;
        }, options: boolean | RenderTargetCreationOptions): InternalTexture;
        /**
         * Create a multi render target texture
         * @see http://doc.babylonjs.com/features/webgl2#multiple-render-target
         * @param size defines the size of the texture
         * @param options defines the creation options
         * @returns the cube texture as an InternalTexture
         */
        createMultipleRenderTarget(size: any, options: IMultiRenderTargetOptions): InternalTexture[];
        private _setupFramebufferDepthAttachments(generateStencilBuffer, generateDepthBuffer, width, height, samples?);
        /**
         * Updates the sample count of a render target texture
         * @see http://doc.babylonjs.com/features/webgl2#multisample-render-targets
         * @param texture defines the texture to update
         * @param samples defines the sample count to set
         * @returns the effective sample count (could be 0 if multisample render targets are not supported)
         */
        updateRenderTargetTextureSampleCount(texture: Nullable<InternalTexture>, samples: number): number;
        /**
         * Update the sample count for a given multiple render target texture
         * @see http://doc.babylonjs.com/features/webgl2#multisample-render-targets
         * @param textures defines the textures to update
         * @param samples defines the sample count to set
         * @returns the effective sample count (could be 0 if multisample render targets are not supported)
         */
        updateMultipleRenderTargetTextureSampleCount(textures: Nullable<InternalTexture[]>, samples: number): number;
        /** @hidden */
        _uploadDataToTexture(target: number, lod: number, internalFormat: number, width: number, height: number, format: number, type: number, data: ArrayBufferView): void;
        /** @hidden */
        _uploadCompressedDataToTexture(target: number, lod: number, internalFormat: number, width: number, height: number, data: ArrayBufferView): void;
        /**
         * Creates a new render target cube texture
         * @param size defines the size of the texture
         * @param options defines the options used to create the texture
         * @returns a new render target cube texture stored in an InternalTexture
         */
        createRenderTargetCubeTexture(size: number, options?: Partial<RenderTargetCreationOptions>): InternalTexture;
        /**
         * Create a cube texture from prefiltered data (ie. the mipmaps contain ready to use data for PBR reflection)
         * @param rootUrl defines the url where the file to load is located
         * @param scene defines the current scene
         * @param scale defines scale to apply to the mip map selection
         * @param offset defines offset to apply to the mip map selection
         * @param onLoad defines an optional callback raised when the texture is loaded
         * @param onError defines an optional callback raised if there is an issue to load the texture
         * @param format defines the format of the data
         * @param forcedExtension defines the extension to use to pick the right loader
         * @returns the cube texture as an InternalTexture
         */
        createPrefilteredCubeTexture(rootUrl: string, scene: Nullable<Scene>, scale: number, offset: number, onLoad?: Nullable<(internalTexture: Nullable<InternalTexture>) => void>, onError?: Nullable<(message?: string, exception?: any) => void>, format?: number, forcedExtension?: any): InternalTexture;
        /**
         * Creates a cube texture
         * @param rootUrl defines the url where the files to load is located
         * @param scene defines the current scene
         * @param files defines the list of files to load (1 per face)
         * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
         * @param onLoad defines an optional callback raised when the texture is loaded
         * @param onError defines an optional callback raised if there is an issue to load the texture
         * @param format defines the format of the data
         * @param forcedExtension defines the extension to use to pick the right loader
         * @param createPolynomials if a polynomial sphere should be created for the cube texture
         * @returns the cube texture as an InternalTexture
         */
        createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap?: boolean, onLoad?: Nullable<(data?: any) => void>, onError?: Nullable<(message?: string, exception?: any) => void>, format?: number, forcedExtension?: any, createPolynomials?: boolean): InternalTexture;
        private setCubeMapTextureParams(gl, loadMipmap);
        /**
         * Update a raw cube texture
         * @param texture defines the texture to udpdate
         * @param data defines the data to store
         * @param format defines the data format
         * @param type defines the type fo the data (BABYLON.Engine.TEXTURETYPE_UNSIGNED_INT by default)
         * @param invertY defines if data must be stored with Y axis inverted
         * @param compression defines the compression used (null by default)
         * @param level defines which level of the texture to update
         */
        updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean, compression?: Nullable<string>, level?: number): void;
        /**
         * Creates a new raw cube texture
         * @param data defines the array of data to use to create each face
         * @param size defines the size of the textures
         * @param format defines the format of the data
         * @param type defines the type fo the data (like BABYLON.Engine.TEXTURETYPE_UNSIGNED_INT)
         * @param generateMipMaps  defines if the engine should generate the mip levels
         * @param invertY defines if data must be stored with Y axis inverted
         * @param samplingMode defines the required sampling mode (like BABYLON.Texture.NEAREST_SAMPLINGMODE)
         * @param compression defines the compression used (null by default)
         * @returns the cube texture as an InternalTexture
         */
        createRawCubeTexture(data: Nullable<ArrayBufferView[]>, size: number, format: number, type: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>): InternalTexture;
        /**
         * Creates a new raw cube texture from a specified url
         * @param url defines the url where the data is located
         * @param scene defines the current scene
         * @param size defines the size of the textures
         * @param format defines the format of the data
         * @param type defines the type fo the data (like BABYLON.Engine.TEXTURETYPE_UNSIGNED_INT)
         * @param noMipmap defines if the engine should avoid generating the mip levels
         * @param callback defines a callback used to extract texture data from loaded data
         * @param mipmapGenerator defines to provide an optional tool to generate mip levels
         * @param onLoad defines a callback called when texture is loaded
         * @param onError defines a callback called if there is an error
         * @param samplingMode defines the required sampling mode (like BABYLON.Texture.NEAREST_SAMPLINGMODE)
         * @param invertY defines if data must be stored with Y axis inverted
         * @returns the cube texture as an InternalTexture
         */
        createRawCubeTextureFromUrl(url: string, scene: Scene, size: number, format: number, type: number, noMipmap: boolean, callback: (ArrayBuffer: ArrayBuffer) => Nullable<ArrayBufferView[]>, mipmapGenerator: Nullable<((faces: ArrayBufferView[]) => ArrayBufferView[][])>, onLoad?: Nullable<() => void>, onError?: Nullable<(message?: string, exception?: any) => void>, samplingMode?: number, invertY?: boolean): InternalTexture;
        /**
         * Update a raw 3D texture
         * @param texture defines the texture to update
         * @param data defines the data to store
         * @param format defines the data format
         * @param invertY defines if data must be stored with Y axis inverted
         * @param compression defines the used compression (can be null)
         */
        updateRawTexture3D(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression?: Nullable<string>): void;
        /**
         * Creates a new raw 3D texture
         * @param data defines the data used to create the texture
         * @param width defines the width of the texture
         * @param height defines the height of the texture
         * @param depth defines the depth of the texture
         * @param format defines the format of the texture
         * @param generateMipMaps defines if the engine must generate mip levels
         * @param invertY defines if data must be stored with Y axis inverted
         * @param samplingMode defines the required sampling mode (like BABYLON.Texture.NEAREST_SAMPLINGMODE)
         * @param compression defines the compressed used (can be null)
         * @returns a new raw 3D texture (stored in an InternalTexture)
         */
        createRawTexture3D(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>): InternalTexture;
        private _prepareWebGLTextureContinuation(texture, scene, noMipmap, isCompressed, samplingMode);
        private _prepareWebGLTexture(texture, scene, width, height, invertY, noMipmap, isCompressed, processFunction, samplingMode?);
        private _convertRGBtoRGBATextureData(rgbData, width, height, textureType);
        /** @hidden */
        _releaseFramebufferObjects(texture: InternalTexture): void;
        /** @hidden */
        _releaseTexture(texture: InternalTexture): void;
        private setProgram(program);
        private _boundUniforms;
        /**
         * Binds an effect to the webGL context
         * @param effect defines the effect to bind
         */
        bindSamplers(effect: Effect): void;
        private _moveBoundTextureOnTop(internalTexture);
        private _getCorrectTextureChannel(channel, internalTexture);
        private _linkTrackers(previous, next);
        private _removeDesignatedSlot(internalTexture);
        private _activateCurrentTexture();
        /** @hidden */
        protected _bindTextureDirectly(target: number, texture: Nullable<InternalTexture>, forTextureDataUpdate?: boolean, force?: boolean): void;
        /** @hidden */
        _bindTexture(channel: number, texture: Nullable<InternalTexture>): void;
        /**
         * Sets a texture to the webGL context from a postprocess
         * @param channel defines the channel to use
         * @param postProcess defines the source postprocess
         */
        setTextureFromPostProcess(channel: number, postProcess: Nullable<PostProcess>): void;
        /**
         * Binds the output of the passed in post process to the texture channel specified
         * @param channel The channel the texture should be bound to
         * @param postProcess The post process which's output should be bound
         */
        setTextureFromPostProcessOutput(channel: number, postProcess: Nullable<PostProcess>): void;
        /**
         * Unbind all textures from the webGL context
         */
        unbindAllTextures(): void;
        /**
         * Sets a texture to the according uniform.
         * @param channel The texture channel
         * @param uniform The uniform to set
         * @param texture The texture to apply
         */
        setTexture(channel: number, uniform: Nullable<WebGLUniformLocation>, texture: Nullable<BaseTexture>): void;
        /**
         * Sets a depth stencil texture from a render target to the according uniform.
         * @param channel The texture channel
         * @param uniform The uniform to set
         * @param texture The render target texture containing the depth stencil texture to apply
         */
        setDepthStencilTexture(channel: number, uniform: Nullable<WebGLUniformLocation>, texture: Nullable<RenderTargetTexture>): void;
        private _bindSamplerUniformToChannel(sourceSlot, destination);
        private _getTextureWrapMode(mode);
        private _setTexture(channel, texture, isPartOfTextureArray?, depthStencilTexture?);
        /**
         * Sets an array of texture to the webGL context
         * @param channel defines the channel where the texture array must be set
         * @param uniform defines the associated uniform location
         * @param textures defines the array of textures to bind
         */
        setTextureArray(channel: number, uniform: Nullable<WebGLUniformLocation>, textures: BaseTexture[]): void;
        /** @hidden */
        _setAnisotropicLevel(target: number, texture: BaseTexture): void;
        private _setTextureParameterFloat(target, parameter, value, texture);
        private _setTextureParameterInteger(target, parameter, value, texture?);
        /**
         * Reads pixels from the current frame buffer. Please note that this function can be slow
         * @param x defines the x coordinate of the rectangle where pixels must be read
         * @param y defines the y coordinate of the rectangle where pixels must be read
         * @param width defines the width of the rectangle where pixels must be read
         * @param height defines the height of the rectangle where pixels must be read
         * @returns a Uint8Array containing RGBA colors
         */
        readPixels(x: number, y: number, width: number, height: number): Uint8Array;
        /**
         * Add an externaly attached data from its key.
         * This method call will fail and return false, if such key already exists.
         * If you don't care and just want to get the data no matter what, use the more convenient getOrAddExternalDataWithFactory() method.
         * @param key the unique key that identifies the data
         * @param data the data object to associate to the key for this Engine instance
         * @return true if no such key were already present and the data was added successfully, false otherwise
         */
        addExternalData<T>(key: string, data: T): boolean;
        /**
         * Get an externaly attached data from its key
         * @param key the unique key that identifies the data
         * @return the associated data, if present (can be null), or undefined if not present
         */
        getExternalData<T>(key: string): T;
        /**
         * Get an externaly attached data from its key, create it using a factory if it's not already present
         * @param key the unique key that identifies the data
         * @param factory the factory that will be called to create the instance if and only if it doesn't exists
         * @return the associated data, can be null if the factory returned null.
         */
        getOrAddExternalDataWithFactory<T>(key: string, factory: (k: string) => T): T;
        /**
         * Remove an externaly attached data from the Engine instance
         * @param key the unique key that identifies the data
         * @return true if the data was successfully removed, false if it doesn't exist
         */
        removeExternalData(key: string): boolean;
        /**
         * Unbind all vertex attributes from the webGL context
         */
        unbindAllAttributes(): void;
        /**
         * Force the engine to release all cached effects. This means that next effect compilation will have to be done completely even if a similar effect was already compiled
         */
        releaseEffects(): void;
        /**
         * Dispose and release all associated resources
         */
        dispose(): void;
        /**
         * Display the loading screen
         * @see http://doc.babylonjs.com/how_to/creating_a_custom_loading_screen
         */
        displayLoadingUI(): void;
        /**
         * Hide the loading screen
         * @see http://doc.babylonjs.com/how_to/creating_a_custom_loading_screen
         */
        hideLoadingUI(): void;
        /**
         * Gets the current loading screen object
         * @see http://doc.babylonjs.com/how_to/creating_a_custom_loading_screen
         */
        /**
         * Sets the current loading screen object
         * @see http://doc.babylonjs.com/how_to/creating_a_custom_loading_screen
         */
        loadingScreen: ILoadingScreen;
        /**
         * Sets the current loading screen text
         * @see http://doc.babylonjs.com/how_to/creating_a_custom_loading_screen
         */
        loadingUIText: string;
        /**
         * Sets the current loading screen background color
         * @see http://doc.babylonjs.com/how_to/creating_a_custom_loading_screen
         */
        loadingUIBackgroundColor: string;
        /**
         * Attach a new callback raised when context lost event is fired
         * @param callback defines the callback to call
         */
        attachContextLostEvent(callback: ((event: WebGLContextEvent) => void)): void;
        /**
         * Attach a new callback raised when context restored event is fired
         * @param callback defines the callback to call
         */
        attachContextRestoredEvent(callback: ((event: WebGLContextEvent) => void)): void;
        /**
         * Gets the source code of the vertex shader associated with a specific webGL program
         * @param program defines the program to use
         * @returns a string containing the source code of the vertex shader associated with the program
         */
        getVertexShaderSource(program: WebGLProgram): Nullable<string>;
        /**
         * Gets the source code of the fragment shader associated with a specific webGL program
         * @param program defines the program to use
         * @returns a string containing the source code of the fragment shader associated with the program
         */
        getFragmentShaderSource(program: WebGLProgram): Nullable<string>;
        /**
         * Get the current error code of the webGL context
         * @returns the error code
         * @see https://developer.mozilla.org/en-US/docs/Web/API/WebGLRenderingContext/getError
         */
        getError(): number;
        /**
         * Gets the current framerate
         * @returns a number representing the framerate
         */
        getFps(): number;
        /**
         * Gets the time spent between current and previous frame
         * @returns a number representing the delta time in ms
         */
        getDeltaTime(): number;
        private _measureFps();
        /** @hidden */
        _readTexturePixels(texture: InternalTexture, width: number, height: number, faceIndex?: number): ArrayBufferView;
        private _canRenderToFloatFramebuffer();
        private _canRenderToHalfFloatFramebuffer();
        private _canRenderToFramebuffer(type);
        /** @hidden */
        _getWebGLTextureType(type: number): number;
        private _getInternalFormat(format);
        /** @hidden */
        _getRGBABufferInternalSizedFormat(type: number, format?: number): number;
        /** @hidden */
        _getRGBAMultiSampleBufferFormat(type: number): number;
        /**
         * Create a new webGL query (you must be sure that queries are supported by checking getCaps() function)
         * @return the new query
         */
        createQuery(): WebGLQuery;
        /**
         * Delete and release a webGL query
         * @param query defines the query to delete
         * @return the current engine
         */
        deleteQuery(query: WebGLQuery): Engine;
        /**
         * Check if a given query has resolved and got its value
         * @param query defines the query to check
         * @returns true if the query got its value
         */
        isQueryResultAvailable(query: WebGLQuery): boolean;
        /**
         * Gets the value of a given query
         * @param query defines the query to check
         * @returns the value of the query
         */
        getQueryResult(query: WebGLQuery): number;
        /**
         * Initiates an occlusion query
         * @param algorithmType defines the algorithm to use
         * @param query defines the query to use
         * @returns the current engine
         * @see http://doc.babylonjs.com/features/occlusionquery
         */
        beginOcclusionQuery(algorithmType: number, query: WebGLQuery): Engine;
        /**
         * Ends an occlusion query
         * @see http://doc.babylonjs.com/features/occlusionquery
         * @param algorithmType defines the algorithm to use
         * @returns the current engine
         */
        endOcclusionQuery(algorithmType: number): Engine;
        private _createTimeQuery();
        private _deleteTimeQuery(query);
        private _getTimeQueryResult(query);
        private _getTimeQueryAvailability(query);
        private _currentNonTimestampToken;
        /**
         * Starts a time query (used to measure time spent by the GPU on a specific frame)
         * Please note that only one query can be issued at a time
         * @returns a time token used to track the time span
         */
        startTimeQuery(): Nullable<_TimeToken>;
        /**
         * Ends a time query
         * @param token defines the token used to measure the time span
         * @returns the time spent (in ns)
         */
        endTimeQuery(token: _TimeToken): int;
        private getGlAlgorithmType(algorithmType);
        /**
         * Creates a webGL transform feedback object
         * Please makes sure to check webGLVersion property to check if you are running webGL 2+
         * @returns the webGL transform feedback object
         */
        createTransformFeedback(): WebGLTransformFeedback;
        /**
         * Delete a webGL transform feedback object
         * @param value defines the webGL transform feedback object to delete
         */
        deleteTransformFeedback(value: WebGLTransformFeedback): void;
        /**
         * Bind a webGL transform feedback object to the webgl context
         * @param value defines the webGL transform feedback object to bind
         */
        bindTransformFeedback(value: Nullable<WebGLTransformFeedback>): void;
        /**
         * Begins a transform feedback operation
         * @param usePoints defines if points or triangles must be used
         */
        beginTransformFeedback(usePoints?: boolean): void;
        /**
         * Ends a transform feedback operation
         */
        endTransformFeedback(): void;
        /**
         * Specify the varyings to use with transform feedback
         * @param program defines the associated webGL program
         * @param value defines the list of strings representing the varying names
         */
        setTranformFeedbackVaryings(program: WebGLProgram, value: string[]): void;
        /**
         * Bind a webGL buffer for a transform feedback operation
         * @param value defines the webGL buffer to bind
         */
        bindTransformFeedbackBuffer(value: Nullable<WebGLBuffer>): void;
        /** @hidden */
        _loadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (data: any) => void, database?: Database, useArrayBuffer?: boolean, onError?: (request?: XMLHttpRequest, exception?: any) => void): IFileRequest;
        /** @hidden */
        _loadFileAsync(url: string, database?: Database, useArrayBuffer?: boolean): Promise<string | ArrayBuffer>;
        private _partialLoadFile(url, index, loadedFiles, scene, onfinish, onErrorCallBack?);
        private _cascadeLoadFiles(scene, onfinish, files, onError?);
        /**
         * Gets a boolean indicating if the engine can be instanciated (ie. if a webGL context can be found)
         * @returns true if the engine can be created
         * @ignorenaming
         */
        static isSupported(): boolean;
    }
}

declare module BABYLON {
    class NullEngineOptions {
        renderWidth: number;
        renderHeight: number;
        textureSize: number;
        deterministicLockstep: boolean;
        lockstepMaxSteps: number;
    }
    /**
     * The null engine class provides support for headless version of babylon.js.
     * This can be used in server side scenario or for testing purposes
     */
    class NullEngine extends Engine {
        private _options;
        isDeterministicLockStep(): boolean;
        getLockstepMaxSteps(): number;
        getHardwareScalingLevel(): number;
        constructor(options?: NullEngineOptions);
        createVertexBuffer(vertices: FloatArray): WebGLBuffer;
        createIndexBuffer(indices: IndicesArray): WebGLBuffer;
        clear(color: Color4, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        getRenderWidth(useScreen?: boolean): number;
        getRenderHeight(useScreen?: boolean): number;
        setViewport(viewport: Viewport, requiredWidth?: number, requiredHeight?: number): void;
        createShaderProgram(vertexCode: string, fragmentCode: string, defines: string, context?: WebGLRenderingContext): WebGLProgram;
        getUniforms(shaderProgram: WebGLProgram, uniformsNames: string[]): WebGLUniformLocation[];
        getAttributes(shaderProgram: WebGLProgram, attributesNames: string[]): number[];
        bindSamplers(effect: Effect): void;
        enableEffect(effect: Effect): void;
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean): void;
        setIntArray(uniform: WebGLUniformLocation, array: Int32Array): void;
        setIntArray2(uniform: WebGLUniformLocation, array: Int32Array): void;
        setIntArray3(uniform: WebGLUniformLocation, array: Int32Array): void;
        setIntArray4(uniform: WebGLUniformLocation, array: Int32Array): void;
        setFloatArray(uniform: WebGLUniformLocation, array: Float32Array): void;
        setFloatArray2(uniform: WebGLUniformLocation, array: Float32Array): void;
        setFloatArray3(uniform: WebGLUniformLocation, array: Float32Array): void;
        setFloatArray4(uniform: WebGLUniformLocation, array: Float32Array): void;
        setArray(uniform: WebGLUniformLocation, array: number[]): void;
        setArray2(uniform: WebGLUniformLocation, array: number[]): void;
        setArray3(uniform: WebGLUniformLocation, array: number[]): void;
        setArray4(uniform: WebGLUniformLocation, array: number[]): void;
        setMatrices(uniform: WebGLUniformLocation, matrices: Float32Array): void;
        setMatrix(uniform: WebGLUniformLocation, matrix: Matrix): void;
        setMatrix3x3(uniform: WebGLUniformLocation, matrix: Float32Array): void;
        setMatrix2x2(uniform: WebGLUniformLocation, matrix: Float32Array): void;
        setFloat(uniform: WebGLUniformLocation, value: number): void;
        setFloat2(uniform: WebGLUniformLocation, x: number, y: number): void;
        setFloat3(uniform: WebGLUniformLocation, x: number, y: number, z: number): void;
        setBool(uniform: WebGLUniformLocation, bool: number): void;
        setFloat4(uniform: WebGLUniformLocation, x: number, y: number, z: number, w: number): void;
        setColor3(uniform: WebGLUniformLocation, color3: Color3): void;
        setColor4(uniform: WebGLUniformLocation, color3: Color3, alpha: number): void;
        setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
        bindBuffers(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: WebGLBuffer, effect: Effect): void;
        wipeCaches(bruteForce?: boolean): void;
        draw(useTriangles: boolean, indexStart: number, indexCount: number, instancesCount?: number): void;
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        _createTexture(): WebGLTexture;
        _releaseTexture(texture: InternalTexture): void;
        createTexture(urlArg: string, noMipmap: boolean, invertY: boolean, scene: Scene, samplingMode?: number, onLoad?: Nullable<() => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<ArrayBuffer | HTMLImageElement>, fallBack?: InternalTexture, format?: number): InternalTexture;
        createRenderTargetTexture(size: any, options: boolean | RenderTargetCreationOptions): InternalTexture;
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture): void;
        bindFramebuffer(texture: InternalTexture, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean): void;
        unBindFramebuffer(texture: InternalTexture, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        createDynamicVertexBuffer(vertices: FloatArray): WebGLBuffer;
        updateDynamicIndexBuffer(indexBuffer: WebGLBuffer, indices: IndicesArray, offset?: number): void;
        /**
         * Updates a dynamic vertex buffer.
         * @param vertexBuffer the vertex buffer to update
         * @param data the data used to update the vertex buffer
         * @param byteOffset the byte offset of the data (optional)
         * @param byteLength the byte length of the data (optional)
         */
        updateDynamicVertexBuffer(vertexBuffer: WebGLBuffer, vertices: FloatArray, byteOffset?: number, byteLength?: number): void;
        protected _bindTextureDirectly(target: number, texture: InternalTexture): void;
        _bindTexture(channel: number, texture: InternalTexture): void;
        _releaseBuffer(buffer: WebGLBuffer): boolean;
        releaseEffects(): void;
    }
}

interface WebGLRenderingContext {
    readonly RASTERIZER_DISCARD: number;
    readonly DEPTH_COMPONENT24: number;
    readonly TEXTURE_3D: number;
    readonly TEXTURE_2D_ARRAY: number;
    readonly TEXTURE_COMPARE_FUNC: number;
    readonly TEXTURE_COMPARE_MODE: number;
    readonly COMPARE_REF_TO_TEXTURE: number;
    readonly TEXTURE_WRAP_R: number;
    texImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, format: number, type: number, pixels: ArrayBufferView | null): void;
    texImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, format: number, type: number, pixels: ArrayBufferView, offset: number): void;
    texImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, format: number, type: number, pixels: ImageBitmap | ImageData | HTMLVideoElement | HTMLImageElement | HTMLCanvasElement): void;
    compressedTexImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, data: ArrayBufferView, offset?: number, length?: number): void;
    readonly TRANSFORM_FEEDBACK: number;
    readonly INTERLEAVED_ATTRIBS: number;
    readonly TRANSFORM_FEEDBACK_BUFFER: number;
    createTransformFeedback(): WebGLTransformFeedback;
    deleteTransformFeedback(transformFeedbac: WebGLTransformFeedback): void;
    bindTransformFeedback(target: number, transformFeedback: WebGLTransformFeedback | null): void;
    beginTransformFeedback(primitiveMode: number): void;
    endTransformFeedback(): void;
    transformFeedbackVaryings(program: WebGLProgram, varyings: string[], bufferMode: number): void;
}
interface ImageBitmap {
    readonly width: number;
    readonly height: number;
    close(): void;
}
interface WebGLQuery extends WebGLObject {
}
declare var WebGLQuery: {
    prototype: WebGLQuery;
    new (): WebGLQuery;
};
interface WebGLSampler extends WebGLObject {
}
declare var WebGLSampler: {
    prototype: WebGLSampler;
    new (): WebGLSampler;
};
interface WebGLSync extends WebGLObject {
}
declare var WebGLSync: {
    prototype: WebGLSync;
    new (): WebGLSync;
};
interface WebGLTransformFeedback extends WebGLObject {
}
declare var WebGLTransformFeedback: {
    prototype: WebGLTransformFeedback;
    new (): WebGLTransformFeedback;
};
interface WebGLVertexArrayObject extends WebGLObject {
}
declare var WebGLVertexArrayObject: {
    prototype: WebGLVertexArrayObject;
    new (): WebGLVertexArrayObject;
};

declare module BABYLON {
    class KeyboardEventTypes {
        static _KEYDOWN: number;
        static _KEYUP: number;
        static readonly KEYDOWN: number;
        static readonly KEYUP: number;
    }
    class KeyboardInfo {
        type: number;
        event: KeyboardEvent;
        constructor(type: number, event: KeyboardEvent);
    }
    /**
     * This class is used to store keyboard related info for the onPreKeyboardObservable event.
     * Set the skipOnKeyboardObservable property to true if you want the engine to stop any process after this event is triggered, even not calling onKeyboardObservable
     */
    class KeyboardInfoPre extends KeyboardInfo {
        constructor(type: number, event: KeyboardEvent);
        skipOnPointerObservable: boolean;
    }
}

declare module BABYLON {
    class PointerEventTypes {
        static _POINTERDOWN: number;
        static _POINTERUP: number;
        static _POINTERMOVE: number;
        static _POINTERWHEEL: number;
        static _POINTERPICK: number;
        static _POINTERTAP: number;
        static _POINTERDOUBLETAP: number;
        static readonly POINTERDOWN: number;
        static readonly POINTERUP: number;
        static readonly POINTERMOVE: number;
        static readonly POINTERWHEEL: number;
        static readonly POINTERPICK: number;
        static readonly POINTERTAP: number;
        static readonly POINTERDOUBLETAP: number;
    }
    class PointerInfoBase {
        type: number;
        event: PointerEvent | MouseWheelEvent;
        constructor(type: number, event: PointerEvent | MouseWheelEvent);
    }
    /**
     * This class is used to store pointer related info for the onPrePointerObservable event.
     * Set the skipOnPointerObservable property to true if you want the engine to stop any process after this event is triggered, even not calling onPointerObservable
     */
    class PointerInfoPre extends PointerInfoBase {
        constructor(type: number, event: PointerEvent | MouseWheelEvent, localX: number, localY: number);
        localPosition: Vector2;
        skipOnPointerObservable: boolean;
    }
    /**
     * This type contains all the data related to a pointer event in Babylon.js.
     * The event member is an instance of PointerEvent for all types except PointerWheel and is of type MouseWheelEvent when type equals PointerWheel. The different event types can be found in the PointerEventTypes class.
     */
    class PointerInfo extends PointerInfoBase {
        pickInfo: Nullable<PickingInfo>;
        constructor(type: number, event: PointerEvent | MouseWheelEvent, pickInfo: Nullable<PickingInfo>);
    }
}

declare module BABYLON {
    class BoundingBox implements ICullable {
        vectors: Vector3[];
        center: Vector3;
        centerWorld: Vector3;
        extendSize: Vector3;
        extendSizeWorld: Vector3;
        directions: Vector3[];
        vectorsWorld: Vector3[];
        minimumWorld: Vector3;
        maximumWorld: Vector3;
        minimum: Vector3;
        maximum: Vector3;
        private _worldMatrix;
        /**
         * Creates a new bounding box
         * @param min defines the minimum vector (in local space)
         * @param max defines the maximum vector (in local space)
         */
        constructor(min: Vector3, max: Vector3);
        /**
         * Recreates the entire bounding box from scratch
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         */
        reConstruct(min: Vector3, max: Vector3): void;
        getWorldMatrix(): Matrix;
        setWorldMatrix(matrix: Matrix): BoundingBox;
        _update(world: Matrix): void;
        isInFrustum(frustumPlanes: Plane[]): boolean;
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        intersectsPoint(point: Vector3): boolean;
        intersectsSphere(sphere: BoundingSphere): boolean;
        intersectsMinMax(min: Vector3, max: Vector3): boolean;
        static Intersects(box0: BoundingBox, box1: BoundingBox): boolean;
        static IntersectsSphere(minPoint: Vector3, maxPoint: Vector3, sphereCenter: Vector3, sphereRadius: number): boolean;
        static IsCompletelyInFrustum(boundingVectors: Vector3[], frustumPlanes: Plane[]): boolean;
        static IsInFrustum(boundingVectors: Vector3[], frustumPlanes: Plane[]): boolean;
    }
}

declare module BABYLON {
    interface ICullable {
        isInFrustum(frustumPlanes: Plane[]): boolean;
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
    }
    class BoundingInfo implements ICullable {
        minimum: Vector3;
        maximum: Vector3;
        boundingBox: BoundingBox;
        boundingSphere: BoundingSphere;
        private _isLocked;
        constructor(minimum: Vector3, maximum: Vector3);
        isLocked: boolean;
        update(world: Matrix): void;
        /**
         * Recreate the bounding info to be centered around a specific point given a specific extend.
         * @param center New center of the bounding info
         * @param extend New extend of the bounding info
         */
        centerOn(center: Vector3, extend: Vector3): BoundingInfo;
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Gets the world distance between the min and max points of the bounding box
         */
        readonly diagonalLength: number;
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        _checkCollision(collider: Collider): boolean;
        intersectsPoint(point: Vector3): boolean;
        intersects(boundingInfo: BoundingInfo, precise: boolean): boolean;
    }
}

declare module BABYLON {
    class BoundingSphere {
        center: Vector3;
        radius: number;
        centerWorld: Vector3;
        radiusWorld: number;
        minimum: Vector3;
        maximum: Vector3;
        private _tempRadiusVector;
        /**
         * Creates a new bounding sphere
         * @param min defines the minimum vector (in local space)
         * @param max defines the maximum vector (in local space)
         */
        constructor(min: Vector3, max: Vector3);
        /**
         * Recreates the entire bounding sphere from scratch
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         */
        reConstruct(min: Vector3, max: Vector3): void;
        _update(world: Matrix): void;
        isInFrustum(frustumPlanes: Plane[]): boolean;
        intersectsPoint(point: Vector3): boolean;
        static Intersects(sphere0: BoundingSphere, sphere1: BoundingSphere): boolean;
    }
}

declare module BABYLON {
    class Ray {
        origin: Vector3;
        direction: Vector3;
        length: number;
        private _edge1;
        private _edge2;
        private _pvec;
        private _tvec;
        private _qvec;
        private _tmpRay;
        constructor(origin: Vector3, direction: Vector3, length?: number);
        intersectsBoxMinMax(minimum: Vector3, maximum: Vector3): boolean;
        intersectsBox(box: BoundingBox): boolean;
        intersectsSphere(sphere: BoundingSphere): boolean;
        intersectsTriangle(vertex0: Vector3, vertex1: Vector3, vertex2: Vector3): Nullable<IntersectionInfo>;
        intersectsPlane(plane: Plane): Nullable<number>;
        intersectsMesh(mesh: AbstractMesh, fastCheck?: boolean): PickingInfo;
        intersectsMeshes(meshes: Array<AbstractMesh>, fastCheck?: boolean, results?: Array<PickingInfo>): Array<PickingInfo>;
        private _comparePickingInfo(pickingInfoA, pickingInfoB);
        private static smallnum;
        private static rayl;
        /**
         * Intersection test between the ray and a given segment whithin a given tolerance (threshold)
         * @param sega the first point of the segment to test the intersection against
         * @param segb the second point of the segment to test the intersection against
         * @param threshold the tolerance margin, if the ray doesn't intersect the segment but is close to the given threshold, the intersection is successful
         * @return the distance from the ray origin to the intersection point if there's intersection, or -1 if there's no intersection
         */
        intersectionSegment(sega: Vector3, segb: Vector3, threshold: number): number;
        update(x: number, y: number, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix): Ray;
        static Zero(): Ray;
        static CreateNew(x: number, y: number, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix): Ray;
        /**
        * Function will create a new transformed ray starting from origin and ending at the end point. Ray's length will be set, and ray will be
        * transformed to the given world matrix.
        * @param origin The origin point
        * @param end The end point
        * @param world a matrix to transform the ray to. Default is the identity matrix.
        */
        static CreateNewFromTo(origin: Vector3, end: Vector3, world?: Matrix): Ray;
        static Transform(ray: Ray, matrix: Matrix): Ray;
        static TransformToRef(ray: Ray, matrix: Matrix, result: Ray): void;
    }
}

declare module BABYLON {
    class StickValues {
        x: number;
        y: number;
        constructor(x: number, y: number);
    }
    interface GamepadButtonChanges {
        changed: boolean;
        pressChanged: boolean;
        touchChanged: boolean;
        valueChanged: boolean;
    }
    class Gamepad {
        id: string;
        index: number;
        browserGamepad: any;
        type: number;
        private _leftStick;
        private _rightStick;
        _isConnected: boolean;
        private _leftStickAxisX;
        private _leftStickAxisY;
        private _rightStickAxisX;
        private _rightStickAxisY;
        private _onleftstickchanged;
        private _onrightstickchanged;
        static GAMEPAD: number;
        static GENERIC: number;
        static XBOX: number;
        static POSE_ENABLED: number;
        protected _invertLeftStickY: boolean;
        readonly isConnected: boolean;
        constructor(id: string, index: number, browserGamepad: any, leftStickX?: number, leftStickY?: number, rightStickX?: number, rightStickY?: number);
        onleftstickchanged(callback: (values: StickValues) => void): void;
        onrightstickchanged(callback: (values: StickValues) => void): void;
        leftStick: StickValues;
        rightStick: StickValues;
        update(): void;
        dispose(): void;
    }
    class GenericPad extends Gamepad {
        private _buttons;
        private _onbuttondown;
        private _onbuttonup;
        onButtonDownObservable: Observable<number>;
        onButtonUpObservable: Observable<number>;
        onbuttondown(callback: (buttonPressed: number) => void): void;
        onbuttonup(callback: (buttonReleased: number) => void): void;
        constructor(id: string, index: number, browserGamepad: any);
        private _setButtonValue(newValue, currentValue, buttonIndex);
        update(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class GamepadManager {
        private _scene;
        private _babylonGamepads;
        private _oneGamepadConnected;
        _isMonitoring: boolean;
        private _gamepadEventSupported;
        private _gamepadSupport;
        onGamepadConnectedObservable: Observable<Gamepad>;
        onGamepadDisconnectedObservable: Observable<Gamepad>;
        private _onGamepadConnectedEvent;
        private _onGamepadDisconnectedEvent;
        constructor(_scene?: Scene | undefined);
        readonly gamepads: Gamepad[];
        getGamepadByType(type?: number): Nullable<Gamepad>;
        dispose(): void;
        private _addNewGamepad(gamepad);
        private _startMonitoringGamepads();
        private _stopMonitoringGamepads();
        _checkGamepadsStatus(): void;
        private _updateGamepadObjects();
    }
}

declare module BABYLON {
    /**
     * Defines supported buttons for XBox360 compatible gamepads
     */
    enum Xbox360Button {
        /** A */
        A = 0,
        /** B */
        B = 1,
        /** X */
        X = 2,
        /** Y */
        Y = 3,
        /** Start */
        Start = 4,
        /** Back */
        Back = 5,
        /** Left button */
        LB = 6,
        /** Right button */
        RB = 7,
        /** Left stick */
        LeftStick = 8,
        /** Right stick */
        RightStick = 9,
    }
    /** Defines values for XBox360 DPad  */
    enum Xbox360Dpad {
        /** Up */
        Up = 0,
        /** Down */
        Down = 1,
        /** Left */
        Left = 2,
        /** Right */
        Right = 3,
    }
    /**
     * Defines a XBox360 gamepad
     */
    class Xbox360Pad extends Gamepad {
        private _leftTrigger;
        private _rightTrigger;
        private _onlefttriggerchanged;
        private _onrighttriggerchanged;
        private _onbuttondown;
        private _onbuttonup;
        private _ondpaddown;
        private _ondpadup;
        /** Observable raised when a button is pressed */
        onButtonDownObservable: Observable<Xbox360Button>;
        /** Observable raised when a button is released */
        onButtonUpObservable: Observable<Xbox360Button>;
        /** Observable raised when a pad is pressed */
        onPadDownObservable: Observable<Xbox360Dpad>;
        /** Observable raised when a pad is released */
        onPadUpObservable: Observable<Xbox360Dpad>;
        private _buttonA;
        private _buttonB;
        private _buttonX;
        private _buttonY;
        private _buttonBack;
        private _buttonStart;
        private _buttonLB;
        private _buttonRB;
        private _buttonLeftStick;
        private _buttonRightStick;
        private _dPadUp;
        private _dPadDown;
        private _dPadLeft;
        private _dPadRight;
        private _isXboxOnePad;
        /**
         * Creates a new XBox360 gamepad object
         * @param id defines the id of this gamepad
         * @param index defines its index
         * @param gamepad defines the internal HTML gamepad object
         * @param xboxOne defines if it is a XBox One gamepad
         */
        constructor(id: string, index: number, gamepad: any, xboxOne?: boolean);
        /**
         * Defines the callback to call when left trigger is pressed
         * @param callback defines the callback to use
         */
        onlefttriggerchanged(callback: (value: number) => void): void;
        /**
         * Defines the callback to call when right trigger is pressed
         * @param callback defines the callback to use
         */
        onrighttriggerchanged(callback: (value: number) => void): void;
        /**
         * Gets or sets left trigger value
         */
        leftTrigger: number;
        /**
         * Gets or sets right trigger value
         */
        rightTrigger: number;
        /**
         * Defines the callback to call when a button is pressed
         * @param callback defines the callback to use
         */
        onbuttondown(callback: (buttonPressed: Xbox360Button) => void): void;
        /**
         * Defines the callback to call when a button is released
         * @param callback defines the callback to use
         */
        onbuttonup(callback: (buttonReleased: Xbox360Button) => void): void;
        /**
         * Defines the callback to call when a pad is pressed
         * @param callback defines the callback to use
         */
        ondpaddown(callback: (dPadPressed: Xbox360Dpad) => void): void;
        /**
         * Defines the callback to call when a pad is released
         * @param callback defines the callback to use
         */
        ondpadup(callback: (dPadReleased: Xbox360Dpad) => void): void;
        private _setButtonValue(newValue, currentValue, buttonType);
        private _setDPadValue(newValue, currentValue, buttonType);
        /** Gets or sets value of A button */
        buttonA: number;
        /** Gets or sets value of B button */
        buttonB: number;
        /** Gets or sets value of X button */
        buttonX: number;
        /** Gets or sets value of Y button */
        buttonY: number;
        /** Gets or sets value of Start button  */
        buttonStart: number;
        /** Gets or sets value of Back button  */
        buttonBack: number;
        /** Gets or sets value of Left button  */
        buttonLB: number;
        /** Gets or sets value of Right button  */
        buttonRB: number;
        /** Gets or sets value of left stick */
        buttonLeftStick: number;
        /** Gets or sets value of right stick */
        buttonRightStick: number;
        /** Gets or sets value of DPad up */
        dPadUp: number;
        /** Gets or sets value of DPad down */
        dPadDown: number;
        /** Gets or sets value of DPad left */
        dPadLeft: number;
        /** Gets or sets value of DPad right */
        dPadRight: number;
        /**
         * Force the gamepad to synchronize with device values
         */
        update(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Represents the different options available during the creation of
     * a Environment helper.
     *
     * This can control the default ground, skybox and image processing setup of your scene.
     */
    interface IEnvironmentHelperOptions {
        /**
         * Specifies wether or not to create a ground.
         * True by default.
         */
        createGround: boolean;
        /**
         * Specifies the ground size.
         * 15 by default.
         */
        groundSize: number;
        /**
         * The texture used on the ground for the main color.
         * Comes from the BabylonJS CDN by default.
         *
         * Remarks: Can be either a texture or a url.
         */
        groundTexture: string | BaseTexture;
        /**
         * The color mixed in the ground texture by default.
         * BabylonJS clearColor by default.
         */
        groundColor: Color3;
        /**
         * Specifies the ground opacity.
         * 1 by default.
         */
        groundOpacity: number;
        /**
         * Enables the ground to receive shadows.
         * True by default.
         */
        enableGroundShadow: boolean;
        /**
         * Helps preventing the shadow to be fully black on the ground.
         * 0.5 by default.
         */
        groundShadowLevel: number;
        /**
         * Creates a mirror texture attach to the ground.
         * false by default.
         */
        enableGroundMirror: boolean;
        /**
         * Specifies the ground mirror size ratio.
         * 0.3 by default as the default kernel is 64.
         */
        groundMirrorSizeRatio: number;
        /**
         * Specifies the ground mirror blur kernel size.
         * 64 by default.
         */
        groundMirrorBlurKernel: number;
        /**
         * Specifies the ground mirror visibility amount.
         * 1 by default
         */
        groundMirrorAmount: number;
        /**
         * Specifies the ground mirror reflectance weight.
         * This uses the standard weight of the background material to setup the fresnel effect
         * of the mirror.
         * 1 by default.
         */
        groundMirrorFresnelWeight: number;
        /**
         * Specifies the ground mirror Falloff distance.
         * This can helps reducing the size of the reflection.
         * 0 by Default.
         */
        groundMirrorFallOffDistance: number;
        /**
         * Specifies the ground mirror texture type.
         * Unsigned Int by Default.
         */
        groundMirrorTextureType: number;
        /**
         * Specifies a bias applied to the ground vertical position to prevent z-fighyting with
         * the shown objects.
         */
        groundYBias: number;
        /**
         * Specifies wether or not to create a skybox.
         * True by default.
         */
        createSkybox: boolean;
        /**
         * Specifies the skybox size.
         * 20 by default.
         */
        skyboxSize: number;
        /**
         * The texture used on the skybox for the main color.
         * Comes from the BabylonJS CDN by default.
         *
         * Remarks: Can be either a texture or a url.
         */
        skyboxTexture: string | BaseTexture;
        /**
         * The color mixed in the skybox texture by default.
         * BabylonJS clearColor by default.
         */
        skyboxColor: Color3;
        /**
         * The background rotation around the Y axis of the scene.
         * This helps aligning the key lights of your scene with the background.
         * 0 by default.
         */
        backgroundYRotation: number;
        /**
         * Compute automatically the size of the elements to best fit with the scene.
         */
        sizeAuto: boolean;
        /**
         * Default position of the rootMesh if autoSize is not true.
         */
        rootPosition: Vector3;
        /**
         * Sets up the image processing in the scene.
         * true by default.
         */
        setupImageProcessing: boolean;
        /**
         * The texture used as your environment texture in the scene.
         * Comes from the BabylonJS CDN by default and in use if setupImageProcessing is true.
         *
         * Remarks: Can be either a texture or a url.
         */
        environmentTexture: string | BaseTexture;
        /**
         * The value of the exposure to apply to the scene.
         * 0.6 by default if setupImageProcessing is true.
         */
        cameraExposure: number;
        /**
         * The value of the contrast to apply to the scene.
         * 1.6 by default if setupImageProcessing is true.
         */
        cameraContrast: number;
        /**
         * Specifies wether or not tonemapping should be enabled in the scene.
         * true by default if setupImageProcessing is true.
         */
        toneMappingEnabled: boolean;
    }
    /**
     * The Environment helper class can be used to add a fully featuread none expensive background to your scene.
     * It includes by default a skybox and a ground relying on the BackgroundMaterial.
     * It also helps with the default setup of your imageProcessing configuration.
     */
    class EnvironmentHelper {
        /**
         * Default ground texture URL.
         */
        private static _groundTextureCDNUrl;
        /**
         * Default skybox texture URL.
         */
        private static _skyboxTextureCDNUrl;
        /**
         * Default environment texture URL.
         */
        private static _environmentTextureCDNUrl;
        /**
         * Creates the default options for the helper.
         */
        private static _getDefaultOptions();
        private _rootMesh;
        /**
         * Gets the root mesh created by the helper.
         */
        readonly rootMesh: Mesh;
        private _skybox;
        /**
         * Gets the skybox created by the helper.
         */
        readonly skybox: Nullable<Mesh>;
        private _skyboxTexture;
        /**
         * Gets the skybox texture created by the helper.
         */
        readonly skyboxTexture: Nullable<BaseTexture>;
        private _skyboxMaterial;
        /**
         * Gets the skybox material created by the helper.
         */
        readonly skyboxMaterial: Nullable<BackgroundMaterial>;
        private _ground;
        /**
         * Gets the ground mesh created by the helper.
         */
        readonly ground: Nullable<Mesh>;
        private _groundTexture;
        /**
         * Gets the ground texture created by the helper.
         */
        readonly groundTexture: Nullable<BaseTexture>;
        private _groundMirror;
        /**
         * Gets the ground mirror created by the helper.
         */
        readonly groundMirror: Nullable<MirrorTexture>;
        /**
         * Gets the ground mirror render list to helps pushing the meshes
         * you wish in the ground reflection.
         */
        readonly groundMirrorRenderList: Nullable<AbstractMesh[]>;
        private _groundMaterial;
        /**
         * Gets the ground material created by the helper.
         */
        readonly groundMaterial: Nullable<BackgroundMaterial>;
        /**
         * Stores the creation options.
         */
        private readonly _scene;
        private _options;
        /**
         * This observable will be notified with any error during the creation of the environment,
         * mainly texture creation errors.
         */
        onErrorObservable: Observable<{
            message?: string;
            exception?: any;
        }>;
        /**
         * constructor
         * @param options
         * @param scene The scene to add the material to
         */
        constructor(options: Partial<IEnvironmentHelperOptions>, scene: Scene);
        /**
         * Updates the background according to the new options
         * @param options
         */
        updateOptions(options: Partial<IEnvironmentHelperOptions>): void;
        /**
         * Sets the primary color of all the available elements.
         * @param color the main color to affect to the ground and the background
         */
        setMainColor(color: Color3): void;
        /**
         * Setup the image processing according to the specified options.
         */
        private _setupImageProcessing();
        /**
         * Setup the environment texture according to the specified options.
         */
        private _setupEnvironmentTexture();
        /**
         * Setup the background according to the specified options.
         */
        private _setupBackground();
        /**
         * Get the scene sizes according to the setup.
         */
        private _getSceneSize();
        /**
         * Setup the ground according to the specified options.
         */
        private _setupGround(sceneSize);
        /**
         * Setup the ground material according to the specified options.
         */
        private _setupGroundMaterial();
        /**
         * Setup the ground diffuse texture according to the specified options.
         */
        private _setupGroundDiffuseTexture();
        /**
         * Setup the ground mirror texture according to the specified options.
         */
        private _setupGroundMirrorTexture(sceneSize);
        /**
         * Setup the ground to receive the mirror texture.
         */
        private _setupMirrorInGroundMaterial();
        /**
         * Setup the skybox according to the specified options.
         */
        private _setupSkybox(sceneSize);
        /**
         * Setup the skybox material according to the specified options.
         */
        private _setupSkyboxMaterial();
        /**
         * Setup the skybox reflection texture according to the specified options.
         */
        private _setupSkyboxReflectionTexture();
        private _errorHandler;
        /**
         * Dispose all the elements created by the Helper.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Display a 360 degree video on an approximately spherical surface, useful for VR applications or skyboxes.
     * As a subclass of Node, this allow parenting to the camera or multiple videos with different locations in the scene.
     * This class achieves its effect with a VideoTexture and a correctly configured BackgroundMaterial on an inverted sphere.
     * Potential additions to this helper include zoom and and non-infinite distance rendering effects.
     */
    class VideoDome extends Node {
        /**
         * The video texture being displayed on the sphere
         */
        protected _videoTexture: VideoTexture;
        /**
         * The skybox material
         */
        protected _material: BackgroundMaterial;
        /**
         * The surface used for the skybox
         */
        protected _mesh: Mesh;
        /**
         * The current fov(field of view) multiplier, 0.0 - 2.0. Defaults to 1.0. Lower values "zoom in" and higher values "zoom out".
         * Also see the options.resolution property.
         */
        fovMultiplier: number;
        /**
         * Create an instance of this class and pass through the parameters to the relevant classes, VideoTexture, StandardMaterial, and Mesh.
         * @param name Element's name, child elements will append suffixes for their own names.
         * @param urlsOrVideo
         * @param options An object containing optional or exposed sub element properties:
         * @param options **resolution=12** Integer, lower resolutions have more artifacts at extreme fovs
         * @param options **clickToPlay=false** Add a click to play listener to the video, does not prevent autoplay.
         * @param options **autoPlay=true** Automatically attempt to being playing the video.
         * @param options **loop=true** Automatically loop video on end.
         * @param options **size=1000** Physical radius to create the dome at, defaults to approximately half the far clip plane.
         */
        constructor(name: string, urlsOrVideo: string | string[] | HTMLVideoElement, options: {
            resolution?: number;
            clickToPlay?: boolean;
            autoPlay?: boolean;
            loop?: boolean;
            size?: number;
        }, scene: Scene);
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
    }
}

declare module BABYLON {
    /**
     * This class can be used to get instrumentation data from a Babylon engine
     */
    class EngineInstrumentation implements IDisposable {
        engine: Engine;
        private _captureGPUFrameTime;
        private _gpuFrameTimeToken;
        private _gpuFrameTime;
        private _captureShaderCompilationTime;
        private _shaderCompilationTime;
        private _onBeginFrameObserver;
        private _onEndFrameObserver;
        private _onBeforeShaderCompilationObserver;
        private _onAfterShaderCompilationObserver;
        /**
         * Gets the perf counter used for GPU frame time
         */
        readonly gpuFrameTimeCounter: PerfCounter;
        /**
         * Gets the GPU frame time capture status
         */
        /**
         * Enable or disable the GPU frame time capture
         */
        captureGPUFrameTime: boolean;
        /**
         * Gets the perf counter used for shader compilation time
         */
        readonly shaderCompilationTimeCounter: PerfCounter;
        /**
         * Gets the shader compilation time capture status
         */
        /**
         * Enable or disable the shader compilation time capture
         */
        captureShaderCompilationTime: boolean;
        constructor(engine: Engine);
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * This class can be used to get instrumentation data from a Babylon engine
     */
    class SceneInstrumentation implements IDisposable {
        scene: Scene;
        private _captureActiveMeshesEvaluationTime;
        private _activeMeshesEvaluationTime;
        private _captureRenderTargetsRenderTime;
        private _renderTargetsRenderTime;
        private _captureFrameTime;
        private _frameTime;
        private _captureRenderTime;
        private _renderTime;
        private _captureInterFrameTime;
        private _interFrameTime;
        private _captureParticlesRenderTime;
        private _particlesRenderTime;
        private _captureSpritesRenderTime;
        private _spritesRenderTime;
        private _capturePhysicsTime;
        private _physicsTime;
        private _captureAnimationsTime;
        private _animationsTime;
        private _onBeforeActiveMeshesEvaluationObserver;
        private _onAfterActiveMeshesEvaluationObserver;
        private _onBeforeRenderTargetsRenderObserver;
        private _onAfterRenderTargetsRenderObserver;
        private _onAfterRenderObserver;
        private _onBeforeDrawPhaseObserver;
        private _onAfterDrawPhaseObserver;
        private _onBeforeAnimationsObserver;
        private _onBeforeParticlesRenderingObserver;
        private _onAfterParticlesRenderingObserver;
        private _onBeforeSpritesRenderingObserver;
        private _onAfterSpritesRenderingObserver;
        private _onBeforePhysicsObserver;
        private _onAfterPhysicsObserver;
        private _onAfterAnimationsObserver;
        /**
         * Gets the perf counter used for active meshes evaluation time
         */
        readonly activeMeshesEvaluationTimeCounter: PerfCounter;
        /**
         * Gets the active meshes evaluation time capture status
         */
        /**
         * Enable or disable the active meshes evaluation time capture
         */
        captureActiveMeshesEvaluationTime: boolean;
        /**
         * Gets the perf counter used for render targets render time
         */
        readonly renderTargetsRenderTimeCounter: PerfCounter;
        /**
         * Gets the render targets render time capture status
         */
        /**
         * Enable or disable the render targets render time capture
         */
        captureRenderTargetsRenderTime: boolean;
        /**
         * Gets the perf counter used for particles render time
         */
        readonly particlesRenderTimeCounter: PerfCounter;
        /**
         * Gets the particles render time capture status
         */
        /**
         * Enable or disable the particles render time capture
         */
        captureParticlesRenderTime: boolean;
        /**
         * Gets the perf counter used for sprites render time
         */
        readonly spritesRenderTimeCounter: PerfCounter;
        /**
         * Gets the sprites render time capture status
         */
        /**
         * Enable or disable the sprites render time capture
         */
        captureSpritesRenderTime: boolean;
        /**
         * Gets the perf counter used for physics time
         */
        readonly physicsTimeCounter: PerfCounter;
        /**
         * Gets the physics time capture status
         */
        /**
         * Enable or disable the physics time capture
         */
        capturePhysicsTime: boolean;
        /**
         * Gets the perf counter used for animations time
         */
        readonly animationsTimeCounter: PerfCounter;
        /**
         * Gets the animations time capture status
         */
        /**
         * Enable or disable the animations time capture
         */
        captureAnimationsTime: boolean;
        /**
         * Gets the perf counter used for frame time capture
         */
        readonly frameTimeCounter: PerfCounter;
        /**
         * Gets the frame time capture status
         */
        /**
         * Enable or disable the frame time capture
         */
        captureFrameTime: boolean;
        /**
         * Gets the perf counter used for inter-frames time capture
         */
        readonly interFrameTimeCounter: PerfCounter;
        /**
         * Gets the inter-frames time capture status
         */
        /**
         * Enable or disable the inter-frames time capture
         */
        captureInterFrameTime: boolean;
        /**
         * Gets the perf counter used for render time capture
         */
        readonly renderTimeCounter: PerfCounter;
        /**
         * Gets the render time capture status
         */
        /**
         * Enable or disable the render time capture
         */
        captureRenderTime: boolean;
        /**
         * Gets the perf counter used for draw calls
         */
        readonly drawCallsCounter: PerfCounter;
        /**
         * Gets the perf counter used for texture collisions
         */
        readonly textureCollisionsCounter: PerfCounter;
        constructor(scene: Scene);
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * @hidden
     **/
    class _TimeToken {
        _startTimeQuery: Nullable<WebGLQuery>;
        _endTimeQuery: Nullable<WebGLQuery>;
        _timeElapsedQuery: Nullable<WebGLQuery>;
        _timeElapsedQueryEnded: boolean;
    }
}

declare module BABYLON {
    /**
     * Effect layer options. This helps customizing the behaviour
     * of the effect layer.
     */
    interface IEffectLayerOptions {
        /**
         * Multiplication factor apply to the canvas size to compute the render target size
         * used to generated the objects (the smaller the faster).
         */
        mainTextureRatio: number;
        /**
         * Enforces a fixed size texture to ensure effect stability across devices.
         */
        mainTextureFixedSize?: number;
        /**
         * Alpha blending mode used to apply the blur. Default depends of the implementation.
         */
        alphaBlendingMode: number;
        /**
         * The camera attached to the layer.
         */
        camera: Nullable<Camera>;
    }
    /**
     * The effect layer Helps adding post process effect blended with the main pass.
     *
     * This can be for instance use to generate glow or higlight effects on the scene.
     *
     * The effect layer class can not be used directly and is intented to inherited from to be
     * customized per effects.
     */
    abstract class EffectLayer {
        private _vertexBuffers;
        private _indexBuffer;
        private _cachedDefines;
        private _effectLayerMapGenerationEffect;
        private _effectLayerOptions;
        private _mergeEffect;
        protected _scene: Scene;
        protected _engine: Engine;
        protected _maxSize: number;
        protected _mainTextureDesiredSize: ISize;
        protected _mainTexture: RenderTargetTexture;
        protected _shouldRender: boolean;
        protected _postProcesses: PostProcess[];
        protected _textures: BaseTexture[];
        protected _emissiveTextureAndColor: {
            texture: Nullable<BaseTexture>;
            color: Color4;
        };
        /**
         * The name of the layer
         */
        name: string;
        /**
         * The clear color of the texture used to generate the glow map.
         */
        neutralColor: Color4;
        /**
         * Specifies wether the highlight layer is enabled or not.
         */
        isEnabled: boolean;
        /**
         * Gets the camera attached to the layer.
         */
        readonly camera: Nullable<Camera>;
        /**
         * An event triggered when the effect layer has been disposed.
         */
        onDisposeObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the effect layer is about rendering the main texture with the glowy parts.
         */
        onBeforeRenderMainTextureObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the generated texture is being merged in the scene.
         */
        onBeforeComposeObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the generated texture has been merged in the scene.
         */
        onAfterComposeObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the efffect layer changes its size.
         */
        onSizeChangedObservable: Observable<EffectLayer>;
        /**
         * Instantiates a new effect Layer and references it in the scene.
         * @param name The name of the layer
         * @param scene The scene to use the layer in
         */
        constructor(
            /** The Friendly of the effect in the scene */
            name: string, scene: Scene);
        /**
         * Get the effect name of the layer.
         * @return The effect name
         */
        abstract getEffectName(): string;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @return true if ready otherwise, false
         */
        abstract isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Returns wether or nood the layer needs stencil enabled during the mesh rendering.
         * @returns true if the effect requires stencil during the main canvas render pass.
         */
        abstract needStencil(): boolean;
        /**
         * Create the merge effect. This is the shader use to blit the information back
         * to the main canvas at the end of the scene rendering.
         * @returns The effect containing the shader used to merge the effect on the  main canvas
         */
        protected abstract _createMergeEffect(): Effect;
        /**
         * Creates the render target textures and post processes used in the effect layer.
         */
        protected abstract _createTextureAndPostProcesses(): void;
        /**
         * Implementation specific of rendering the generating effect on the main canvas.
         * @param effect The effect used to render through
         */
        protected abstract _internalRender(effect: Effect): void;
        /**
         * Sets the required values for both the emissive texture and and the main color.
         */
        protected abstract _setEmissiveTextureAndColor(mesh: Mesh, subMesh: SubMesh, material: Material): void;
        /**
         * Free any resources and references associated to a mesh.
         * Internal use
         * @param mesh The mesh to free.
         */
        abstract _disposeMesh(mesh: Mesh): void;
        /**
         * Serializes this layer (Glow or Highlight for example)
         * @returns a serialized layer object
         */
        abstract serialize?(): any;
        /**
         * Initializes the effect layer with the required options.
         * @param options Sets of none mandatory options to use with the layer (see IEffectLayerOptions for more information)
         */
        protected _init(options: Partial<IEffectLayerOptions>): void;
        /**
         * Generates the index buffer of the full screen quad blending to the main canvas.
         */
        private _generateIndexBuffer();
        /**
         * Generates the vertex buffer of the full screen quad blending to the main canvas.
         */
        private _genrateVertexBuffer();
        /**
         * Sets the main texture desired size which is the closest power of two
         * of the engine canvas size.
         */
        private _setMainTextureSize();
        /**
         * Creates the main texture for the effect layer.
         */
        protected _createMainTexture(): void;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @param emissiveTexture the associated emissive texture used to generate the glow
         * @return true if ready otherwise, false
         */
        protected _isReady(subMesh: SubMesh, useInstances: boolean, emissiveTexture: Nullable<BaseTexture>): boolean;
        /**
         * Renders the glowing part of the scene by blending the blurred glowing meshes on top of the rendered scene.
         */
        render(): void;
        /**
         * Determine if a given mesh will be used in the current effect.
         * @param mesh mesh to test
         * @returns true if the mesh will be used
         */
        hasMesh(mesh: AbstractMesh): boolean;
        /**
         * Returns true if the layer contains information to display, otherwise false.
         * @returns true if the glow layer should be rendered
         */
        shouldRender(): boolean;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderMesh(mesh: Mesh): boolean;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderEmissiveTextureForMesh(mesh: Mesh): boolean;
        /**
         * Renders the submesh passed in parameter to the generation map.
         */
        protected _renderSubMesh(subMesh: SubMesh): void;
        /**
         * Rebuild the required buffers.
         * @hidden Internal use only.
         */
        _rebuild(): void;
        /**
         * Dispose only the render target textures and post process.
         */
        private _disposeTextureAndPostProcesses();
        /**
         * Dispose the highlight layer and free resources.
         */
        dispose(): void;
        /**
          * Gets the class name of the effect layer
          * @returns the string with the class name of the effect layer
          */
        getClassName(): string;
        /**
         * Creates an effect layer from parsed effect layer data
         * @param parsedEffectLayer defines effect layer data
         * @param scene defines the current scene
         * @param rootUrl defines the root URL containing the effect layer information
         * @returns a parsed effect Layer
         */
        static Parse(parsedEffectLayer: any, scene: Scene, rootUrl: string): EffectLayer;
    }
}

declare module BABYLON {
    /**
     * Glow layer options. This helps customizing the behaviour
     * of the glow layer.
     */
    interface IGlowLayerOptions {
        /**
         * Multiplication factor apply to the canvas size to compute the render target size
         * used to generated the glowing objects (the smaller the faster).
         */
        mainTextureRatio: number;
        /**
         * Enforces a fixed size texture to ensure resize independant blur.
         */
        mainTextureFixedSize?: number;
        /**
         * How big is the kernel of the blur texture.
         */
        blurKernelSize: number;
        /**
         * The camera attached to the layer.
         */
        camera: Nullable<Camera>;
        /**
         * Enable MSAA by chosing the number of samples.
         */
        mainTextureSamples?: number;
    }
    /**
     * The glow layer Helps adding a glow effect around the emissive parts of a mesh.
     *
     * Once instantiated in a scene, simply use the pushMesh or removeMesh method to add or remove
     * glowy meshes to your scene.
     *
     * Documentation: https://doc.babylonjs.com/how_to/glow_layer
     */
    class GlowLayer extends EffectLayer {
        /**
         * Effect Name of the layer.
         */
        static readonly EffectName: string;
        /**
         * The default blur kernel size used for the glow.
         */
        static DefaultBlurKernelSize: number;
        /**
         * The default texture size ratio used for the glow.
         */
        static DefaultTextureRatio: number;
        /**
         * Gets the kernel size of the blur.
         */
        /**
         * Sets the kernel size of the blur.
         */
        blurKernelSize: number;
        /**
         * Gets the glow intensity.
         */
        /**
         * Sets the glow intensity.
         */
        intensity: number;
        private _options;
        private _intensity;
        private _horizontalBlurPostprocess1;
        private _verticalBlurPostprocess1;
        private _horizontalBlurPostprocess2;
        private _verticalBlurPostprocess2;
        private _blurTexture1;
        private _blurTexture2;
        private _postProcesses1;
        private _postProcesses2;
        private _includedOnlyMeshes;
        private _excludedMeshes;
        /**
         * Callback used to let the user override the color selection on a per mesh basis
         */
        customEmissiveColorSelector: (mesh: Mesh, subMesh: SubMesh, material: Material, result: Color4) => void;
        /**
         * Callback used to let the user override the texture selection on a per mesh basis
         */
        customEmissiveTextureSelector: (mesh: Mesh, subMesh: SubMesh, material: Material) => Texture;
        /**
         * Instantiates a new glow Layer and references it to the scene.
         * @param name The name of the layer
         * @param scene The scene to use the layer in
         * @param options Sets of none mandatory options to use with the layer (see IGlowLayerOptions for more information)
         */
        constructor(name: string, scene: Scene, options?: Partial<IGlowLayerOptions>);
        /**
         * Get the effect name of the layer.
         * @return The effect name
         */
        getEffectName(): string;
        /**
         * Create the merge effect. This is the shader use to blit the information back
         * to the main canvas at the end of the scene rendering.
         */
        protected _createMergeEffect(): Effect;
        /**
         * Creates the render target textures and post processes used in the glow layer.
         */
        protected _createTextureAndPostProcesses(): void;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @param emissiveTexture the associated emissive texture used to generate the glow
         * @return true if ready otherwise, false
         */
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Returns wether or nood the layer needs stencil enabled during the mesh rendering.
         */
        needStencil(): boolean;
        /**
         * Implementation specific of rendering the generating effect on the main canvas.
         * @param effect The effect used to render through
         */
        protected _internalRender(effect: Effect): void;
        /**
         * Sets the required values for both the emissive texture and and the main color.
         */
        protected _setEmissiveTextureAndColor(mesh: Mesh, subMesh: SubMesh, material: Material): void;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderMesh(mesh: Mesh): boolean;
        /**
         * Add a mesh in the exclusion list to prevent it to impact or being impacted by the glow layer.
         * @param mesh The mesh to exclude from the glow layer
         */
        addExcludedMesh(mesh: Mesh): void;
        /**
          * Remove a mesh from the exclusion list to let it impact or being impacted by the glow layer.
          * @param mesh The mesh to remove
          */
        removeExcludedMesh(mesh: Mesh): void;
        /**
         * Add a mesh in the inclusion list to impact or being impacted by the glow layer.
         * @param mesh The mesh to include in the glow layer
         */
        addIncludedOnlyMesh(mesh: Mesh): void;
        /**
          * Remove a mesh from the Inclusion list to prevent it to impact or being impacted by the glow layer.
          * @param mesh The mesh to remove
          */
        removeIncludedOnlyMesh(mesh: Mesh): void;
        /**
         * Determine if a given mesh will be used in the glow layer
         * @param mesh The mesh to test
         * @returns true if the mesh will be highlighted by the current glow layer
         */
        hasMesh(mesh: AbstractMesh): boolean;
        /**
         * Free any resources and references associated to a mesh.
         * Internal use
         * @param mesh The mesh to free.
         */
        _disposeMesh(mesh: Mesh): void;
        /**
          * Gets the class name of the effect layer
          * @returns the string with the class name of the effect layer
          */
        getClassName(): string;
        /**
         * Serializes this glow layer
         * @returns a serialized glow layer object
         */
        serialize(): any;
        /**
         * Creates a Glow Layer from parsed glow layer data
         * @param parsedGlowLayer defines glow layer data
         * @param scene defines the current scene
         * @param rootUrl defines the root URL containing the glow layer information
         * @returns a parsed Glow Layer
         */
        static Parse(parsedGlowLayer: any, scene: Scene, rootUrl: string): GlowLayer;
    }
}

declare module BABYLON {
    /**
     * Highlight layer options. This helps customizing the behaviour
     * of the highlight layer.
     */
    interface IHighlightLayerOptions {
        /**
         * Multiplication factor apply to the canvas size to compute the render target size
         * used to generated the glowing objects (the smaller the faster).
         */
        mainTextureRatio: number;
        /**
         * Enforces a fixed size texture to ensure resize independant blur.
         */
        mainTextureFixedSize?: number;
        /**
         * Multiplication factor apply to the main texture size in the first step of the blur to reduce the size
         * of the picture to blur (the smaller the faster).
         */
        blurTextureSizeRatio: number;
        /**
         * How big in texel of the blur texture is the vertical blur.
         */
        blurVerticalSize: number;
        /**
         * How big in texel of the blur texture is the horizontal blur.
         */
        blurHorizontalSize: number;
        /**
         * Alpha blending mode used to apply the blur. Default is combine.
         */
        alphaBlendingMode: number;
        /**
         * The camera attached to the layer.
         */
        camera: Nullable<Camera>;
        /**
         * Should we display highlight as a solid stroke?
         */
        isStroke?: boolean;
    }
    /**
     * The highlight layer Helps adding a glow effect around a mesh.
     *
     * Once instantiated in a scene, simply use the pushMesh or removeMesh method to add or remove
     * glowy meshes to your scene.
     *
     * !!! THIS REQUIRES AN ACTIVE STENCIL BUFFER ON THE CANVAS !!!
     */
    class HighlightLayer extends EffectLayer {
        name: string;
        /**
         * Effect Name of the highlight layer.
         */
        static readonly EffectName: string;
        /**
         * The neutral color used during the preparation of the glow effect.
         * This is black by default as the blend operation is a blend operation.
         */
        static NeutralColor: Color4;
        /**
         * Stencil value used for glowing meshes.
         */
        static GlowingMeshStencilReference: number;
        /**
         * Stencil value used for the other meshes in the scene.
         */
        static NormalMeshStencilReference: number;
        /**
         * Specifies whether or not the inner glow is ACTIVE in the layer.
         */
        innerGlow: boolean;
        /**
         * Specifies whether or not the outer glow is ACTIVE in the layer.
         */
        outerGlow: boolean;
        /**
         * Gets the horizontal size of the blur.
         */
        /**
         * Specifies the horizontal size of the blur.
         */
        blurHorizontalSize: number;
        /**
         * Gets the vertical size of the blur.
         */
        /**
         * Specifies the vertical size of the blur.
         */
        blurVerticalSize: number;
        /**
         * An event triggered when the highlight layer is being blurred.
         */
        onBeforeBlurObservable: Observable<HighlightLayer>;
        /**
         * An event triggered when the highlight layer has been blurred.
         */
        onAfterBlurObservable: Observable<HighlightLayer>;
        private _instanceGlowingMeshStencilReference;
        private _options;
        private _downSamplePostprocess;
        private _horizontalBlurPostprocess;
        private _verticalBlurPostprocess;
        private _blurTexture;
        private _meshes;
        private _excludedMeshes;
        /**
         * Instantiates a new highlight Layer and references it to the scene..
         * @param name The name of the layer
         * @param scene The scene to use the layer in
         * @param options Sets of none mandatory options to use with the layer (see IHighlightLayerOptions for more information)
         */
        constructor(name: string, scene: Scene, options?: Partial<IHighlightLayerOptions>);
        /**
         * Get the effect name of the layer.
         * @return The effect name
         */
        getEffectName(): string;
        /**
         * Create the merge effect. This is the shader use to blit the information back
         * to the main canvas at the end of the scene rendering.
         */
        protected _createMergeEffect(): Effect;
        /**
         * Creates the render target textures and post processes used in the highlight layer.
         */
        protected _createTextureAndPostProcesses(): void;
        /**
         * Returns wether or nood the layer needs stencil enabled during the mesh rendering.
         */
        needStencil(): boolean;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @param emissiveTexture the associated emissive texture used to generate the glow
         * @return true if ready otherwise, false
         */
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Implementation specific of rendering the generating effect on the main canvas.
         * @param effect The effect used to render through
         */
        protected _internalRender(effect: Effect): void;
        /**
         * Returns true if the layer contains information to display, otherwise false.
         */
        shouldRender(): boolean;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderMesh(mesh: Mesh): boolean;
        /**
         * Sets the required values for both the emissive texture and and the main color.
         */
        protected _setEmissiveTextureAndColor(mesh: Mesh, subMesh: SubMesh, material: Material): void;
        /**
         * Add a mesh in the exclusion list to prevent it to impact or being impacted by the highlight layer.
         * @param mesh The mesh to exclude from the highlight layer
         */
        addExcludedMesh(mesh: Mesh): void;
        /**
          * Remove a mesh from the exclusion list to let it impact or being impacted by the highlight layer.
          * @param mesh The mesh to highlight
          */
        removeExcludedMesh(mesh: Mesh): void;
        /**
         * Determine if a given mesh will be highlighted by the current HighlightLayer
         * @param mesh mesh to test
         * @returns true if the mesh will be highlighted by the current HighlightLayer
         */
        hasMesh(mesh: AbstractMesh): boolean;
        /**
         * Add a mesh in the highlight layer in order to make it glow with the chosen color.
         * @param mesh The mesh to highlight
         * @param color The color of the highlight
         * @param glowEmissiveOnly Extract the glow from the emissive texture
         */
        addMesh(mesh: Mesh, color: Color3, glowEmissiveOnly?: boolean): void;
        /**
         * Remove a mesh from the highlight layer in order to make it stop glowing.
         * @param mesh The mesh to highlight
         */
        removeMesh(mesh: Mesh): void;
        /**
         * Force the stencil to the normal expected value for none glowing parts
         */
        private _defaultStencilReference(mesh);
        /**
         * Free any resources and references associated to a mesh.
         * Internal use
         * @param mesh The mesh to free.
         */
        _disposeMesh(mesh: Mesh): void;
        /**
         * Dispose the highlight layer and free resources.
         */
        dispose(): void;
        /**
          * Gets the class name of the effect layer
          * @returns the string with the class name of the effect layer
          */
        getClassName(): string;
        /**
         * Serializes this Highlight layer
         * @returns a serialized Highlight layer object
         */
        serialize(): any;
        /**
         * Creates a Highlight layer from parsed Highlight layer data
         * @param parsedHightlightLayer defines the Highlight layer data
         * @param scene defines the current scene
         * @param rootUrl defines the root URL containing the Highlight layer information
         * @returns a parsed Highlight layer
         */
        static Parse(parsedHightlightLayer: any, scene: Scene, rootUrl: string): HighlightLayer;
    }
}

declare module BABYLON {
    class Layer {
        name: string;
        texture: Nullable<Texture>;
        isBackground: boolean;
        color: Color4;
        scale: Vector2;
        offset: Vector2;
        alphaBlendingMode: number;
        alphaTest: boolean;
        layerMask: number;
        private _scene;
        private _vertexBuffers;
        private _indexBuffer;
        private _effect;
        private _alphaTestEffect;
        /**
        * An event triggered when the layer is disposed.
        */
        onDisposeObservable: Observable<Layer>;
        private _onDisposeObserver;
        onDispose: () => void;
        /**
        * An event triggered before rendering the scene
        */
        onBeforeRenderObservable: Observable<Layer>;
        private _onBeforeRenderObserver;
        onBeforeRender: () => void;
        /**
        * An event triggered after rendering the scene
        */
        onAfterRenderObservable: Observable<Layer>;
        private _onAfterRenderObserver;
        onAfterRender: () => void;
        constructor(name: string, imgUrl: Nullable<string>, scene: Nullable<Scene>, isBackground?: boolean, color?: Color4);
        private _createIndexBuffer();
        _rebuild(): void;
        render(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class LensFlare {
        size: number;
        position: number;
        color: Color3;
        texture: Nullable<Texture>;
        alphaMode: number;
        private _system;
        static AddFlare(size: number, position: number, color: Color3, imgUrl: string, system: LensFlareSystem): LensFlare;
        constructor(size: number, position: number, color: Color3, imgUrl: string, system: LensFlareSystem);
        dispose(): void;
    }
}

declare module BABYLON {
    class LensFlareSystem {
        name: string;
        lensFlares: LensFlare[];
        borderLimit: number;
        viewportBorder: number;
        meshesSelectionPredicate: (mesh: AbstractMesh) => boolean;
        layerMask: number;
        id: string;
        private _scene;
        private _emitter;
        private _vertexBuffers;
        private _indexBuffer;
        private _effect;
        private _positionX;
        private _positionY;
        private _isEnabled;
        constructor(name: string, emitter: any, scene: Scene);
        isEnabled: boolean;
        getScene(): Scene;
        getEmitter(): any;
        setEmitter(newEmitter: any): void;
        getEmitterPosition(): Vector3;
        computeEffectivePosition(globalViewport: Viewport): boolean;
        _isVisible(): boolean;
        render(): boolean;
        dispose(): void;
        static Parse(parsedLensFlareSystem: any, scene: Scene, rootUrl: string): LensFlareSystem;
        serialize(): any;
    }
}

declare module BABYLON {
    /**
     * A directional light is defined by a direction (what a surprise!).
     * The light is emitted from everywhere in the specified direction, and has an infinite range.
     * An example of a directional light is when a distance planet is lit by the apparently parallel lines of light from its sun. Light in a downward direction will light the top of an object.
     * Documentation: https://doc.babylonjs.com/babylon101/lights
     */
    class DirectionalLight extends ShadowLight {
        private _shadowFrustumSize;
        /**
         * Fix frustum size for the shadow generation. This is disabled if the value is 0.
         */
        /**
         * Specifies a fix frustum size for the shadow generation.
         */
        shadowFrustumSize: number;
        private _shadowOrthoScale;
        /**
         * Gets the shadow projection scale against the optimal computed one.
         * 0.1 by default which means that the projection window is increase by 10% from the optimal size.
         * This does not impact in fixed frustum size (shadowFrustumSize being set)
         */
        /**
         * Sets the shadow projection scale against the optimal computed one.
         * 0.1 by default which means that the projection window is increase by 10% from the optimal size.
         * This does not impact in fixed frustum size (shadowFrustumSize being set)
         */
        shadowOrthoScale: number;
        /**
         * Automatically compute the projection matrix to best fit (including all the casters)
         * on each frame.
         */
        autoUpdateExtends: boolean;
        private _orthoLeft;
        private _orthoRight;
        private _orthoTop;
        private _orthoBottom;
        /**
         * Creates a DirectionalLight object in the scene, oriented towards the passed direction (Vector3).
         * The directional light is emitted from everywhere in the given direction.
         * It can cast shawdows.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The friendly name of the light
         * @param direction The direction of the light
         * @param scene The scene the light belongs to
         */
        constructor(name: string, direction: Vector3, scene: Scene);
        /**
         * Returns the string "DirectionalLight".
         * @return The class name
         */
        getClassName(): string;
        /**
         * Returns the integer 1.
         * @return The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Sets the passed matrix "matrix" as projection matrix for the shadows cast by the light according to the passed view matrix.
         * Returns the DirectionalLight Shadow projection matrix.
         */
        protected _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        /**
         * Sets the passed matrix "matrix" as fixed frustum projection matrix for the shadows cast by the light according to the passed view matrix.
         * Returns the DirectionalLight Shadow projection matrix.
         */
        protected _setDefaultFixedFrustumShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix): void;
        /**
         * Sets the passed matrix "matrix" as auto extend projection matrix for the shadows cast by the light according to the passed view matrix.
         * Returns the DirectionalLight Shadow projection matrix.
         */
        protected _setDefaultAutoExtendShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _buildUniformLayout(): void;
        /**
         * Sets the passed Effect object with the DirectionalLight transformed position (or position if not parented) and the passed name.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The directional light
         */
        transferToEffect(effect: Effect, lightIndex: string): DirectionalLight;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         *
         * Values are fixed on directional lights as it relies on an ortho projection hence the need to convert being
         * -1 and 1 to 0 and 1 doing (depth + min) / (min + max) -> (depth + 1) / (1 + 1) -> (depth * 0.5) + 0.5.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         *
         * Values are fixed on directional lights as it relies on an ortho projection hence the need to convert being
         * -1 and 1 to 0 and 1 doing (depth + min) / (min + max) -> (depth + 1) / (1 + 1) -> (depth * 0.5) + 0.5.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
    }
}

declare module BABYLON {
    /**
     * The HemisphericLight simulates the ambient environment light,
     * so the passed direction is the light reflection direction, not the incoming direction.
     */
    class HemisphericLight extends Light {
        /**
         * The groundColor is the light in the opposite direction to the one specified during creation.
         * You can think of the diffuse and specular light as coming from the centre of the object in the given direction and the groundColor light in the opposite direction.
         */
        groundColor: Color3;
        /**
         * The light reflection direction, not the incoming direction.
         */
        direction: Vector3;
        private _worldMatrix;
        /**
         * Creates a HemisphericLight object in the scene according to the passed direction (Vector3).
         * The HemisphericLight simulates the ambient environment light, so the passed direction is the light reflection direction, not the incoming direction.
         * The HemisphericLight can't cast shadows.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The friendly name of the light
         * @param direction The direction of the light reflection
         * @param scene The scene the light belongs to
         */
        constructor(name: string, direction: Vector3, scene: Scene);
        protected _buildUniformLayout(): void;
        /**
         * Returns the string "HemisphericLight".
         * @return The class name
         */
        getClassName(): string;
        /**
         * Sets the HemisphericLight direction towards the passed target (Vector3).
         * Returns the updated direction.
         * @param target The target the direction should point to
         * @return The computed direction
         */
        setDirectionToTarget(target: Vector3): Vector3;
        /**
         * Returns the shadow generator associated to the light.
         * @returns Always null for hemispheric lights because it does not support shadows.
         */
        getShadowGenerator(): Nullable<ShadowGenerator>;
        /**
         * Sets the passed Effect object with the HemisphericLight normalized direction and color and the passed name (string).
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The hemispheric light
         */
        transferToEffect(effect: Effect, lightIndex: string): HemisphericLight;
        /**
         * @hidden internal use only.
         */
        _getWorldMatrix(): Matrix;
        /**
         * Returns the integer 3.
         * @return The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
    }
}

declare module BABYLON {
    /**
     * Base class of all the lights in Babylon. It groups all the generic information about lights.
     * Lights are used, as you would expect, to affect how meshes are seen, in terms of both illumination and colour.
     * All meshes allow light to pass through them unless shadow generation is activated. The default number of lights allowed is four but this can be increased.
     */
    abstract class Light extends Node {
        private static _LIGHTMAP_DEFAULT;
        private static _LIGHTMAP_SPECULAR;
        private static _LIGHTMAP_SHADOWSONLY;
        /**
         * If every light affecting the material is in this lightmapMode,
         * material.lightmapTexture adds or multiplies
         * (depends on material.useLightmapAsShadowmap)
         * after every other light calculations.
         */
        static readonly LIGHTMAP_DEFAULT: number;
        /**
         * material.lightmapTexture as only diffuse lighting from this light
         * adds only specular lighting from this light
         * adds dynamic shadows
         */
        static readonly LIGHTMAP_SPECULAR: number;
        /**
         * material.lightmapTexture as only lighting
         * no light calculation from this light
         * only adds dynamic shadows from this light
         */
        static readonly LIGHTMAP_SHADOWSONLY: number;
        private static _INTENSITYMODE_AUTOMATIC;
        private static _INTENSITYMODE_LUMINOUSPOWER;
        private static _INTENSITYMODE_LUMINOUSINTENSITY;
        private static _INTENSITYMODE_ILLUMINANCE;
        private static _INTENSITYMODE_LUMINANCE;
        /**
         * Each light type uses the default quantity according to its type:
         *      point/spot lights use luminous intensity
         *      directional lights use illuminance
         */
        static readonly INTENSITYMODE_AUTOMATIC: number;
        /**
         * lumen (lm)
         */
        static readonly INTENSITYMODE_LUMINOUSPOWER: number;
        /**
         * candela (lm/sr)
         */
        static readonly INTENSITYMODE_LUMINOUSINTENSITY: number;
        /**
         * lux (lm/m^2)
         */
        static readonly INTENSITYMODE_ILLUMINANCE: number;
        /**
         * nit (cd/m^2)
         */
        static readonly INTENSITYMODE_LUMINANCE: number;
        private static _LIGHTTYPEID_POINTLIGHT;
        private static _LIGHTTYPEID_DIRECTIONALLIGHT;
        private static _LIGHTTYPEID_SPOTLIGHT;
        private static _LIGHTTYPEID_HEMISPHERICLIGHT;
        /**
         * Light type const id of the point light.
         */
        static readonly LIGHTTYPEID_POINTLIGHT: number;
        /**
         * Light type const id of the directional light.
         */
        static readonly LIGHTTYPEID_DIRECTIONALLIGHT: number;
        /**
         * Light type const id of the spot light.
         */
        static readonly LIGHTTYPEID_SPOTLIGHT: number;
        /**
         * Light type const id of the hemispheric light.
         */
        static readonly LIGHTTYPEID_HEMISPHERICLIGHT: number;
        /**
         * Diffuse gives the basic color to an object.
         */
        diffuse: Color3;
        /**
         * Specular produces a highlight color on an object.
         * Note: This is note affecting PBR materials.
         */
        specular: Color3;
        /**
         * Strength of the light.
         * Note: By default it is define in the framework own unit.
         * Note: In PBR materials the intensityMode can be use to chose what unit the intensity is defined in.
         */
        intensity: number;
        /**
         * Defines how far from the source the light is impacting in scene units.
         * Note: Unused in PBR material as the distance light falloff is defined following the inverse squared falloff.
         */
        range: number;
        /**
         * Cached photometric scale default to 1.0 as the automatic intensity mode defaults to 1.0 for every type
         * of light.
         */
        private _photometricScale;
        private _intensityMode;
        /**
         * Gets the photometric scale used to interpret the intensity.
         * This is only relevant with PBR Materials where the light intensity can be defined in a physical way.
         */
        /**
         * Sets the photometric scale used to interpret the intensity.
         * This is only relevant with PBR Materials where the light intensity can be defined in a physical way.
         */
        intensityMode: number;
        private _radius;
        /**
         * Gets the light radius used by PBR Materials to simulate soft area lights.
         */
        /**
         * sets the light radius used by PBR Materials to simulate soft area lights.
         */
        radius: number;
        private _renderPriority;
        /**
         * Defines the rendering priority of the lights. It can help in case of fallback or number of lights
         * exceeding the number allowed of the materials.
         */
        renderPriority: number;
        /**
         * Defines wether or not the shadows are enabled for this light. This can help turning off/on shadow without detaching
         * the current shadow generator.
         */
        shadowEnabled: boolean;
        private _includedOnlyMeshes;
        /**
         * Gets the only meshes impacted by this light.
         */
        /**
         * Sets the only meshes impacted by this light.
         */
        includedOnlyMeshes: AbstractMesh[];
        private _excludedMeshes;
        /**
         * Gets the meshes not impacted by this light.
         */
        /**
         * Sets the meshes not impacted by this light.
         */
        excludedMeshes: AbstractMesh[];
        private _excludeWithLayerMask;
        /**
         * Gets the layer id use to find what meshes are not impacted by the light.
         * Inactive if 0
         */
        /**
         * Sets the layer id use to find what meshes are not impacted by the light.
         * Inactive if 0
         */
        excludeWithLayerMask: number;
        private _includeOnlyWithLayerMask;
        /**
         * Gets the layer id use to find what meshes are impacted by the light.
         * Inactive if 0
         */
        /**
         * Sets the layer id use to find what meshes are impacted by the light.
         * Inactive if 0
         */
        includeOnlyWithLayerMask: number;
        private _lightmapMode;
        /**
         * Gets the lightmap mode of this light (should be one of the constants defined by Light.LIGHTMAP_x)
         */
        /**
         * Sets the lightmap mode of this light (should be one of the constants defined by Light.LIGHTMAP_x)
         */
        lightmapMode: number;
        private _parentedWorldMatrix;
        /**
         * Shadow generator associted to the light.
         * Internal use only.
         */
        _shadowGenerator: Nullable<IShadowGenerator>;
        /**
         * @hidden Internal use only.
         */
        _excludedMeshesIds: string[];
        /**
         * @hidden Internal use only.
         */
        _includedOnlyMeshesIds: string[];
        /**
         * The current light unifom buffer.
         * @hidden Internal use only.
         */
        _uniformBuffer: UniformBuffer;
        /**
         * Creates a Light object in the scene.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The firendly name of the light
         * @param scene The scene the light belongs too
         */
        constructor(name: string, scene: Scene);
        protected abstract _buildUniformLayout(): void;
        /**
         * Sets the passed Effect "effect" with the Light information.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The light
         */
        abstract transferToEffect(effect: Effect, lightIndex: string): Light;
        /**
         * @hidden internal use only.
         */
        abstract _getWorldMatrix(): Matrix;
        /**
         * Returns the string "Light".
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Converts the light information to a readable string for debug purpose.
         * @param fullDetails Supports for multiple levels of logging within scene loading
         * @returns the human readable light info
         */
        toString(fullDetails?: boolean): string;
        /**
         * Set the enabled state of this node.
         * @param value - the new enabled state
         */
        setEnabled(value: boolean): void;
        /**
         * Returns the Light associated shadow generator if any.
         * @return the associated shadow generator.
         */
        getShadowGenerator(): Nullable<IShadowGenerator>;
        /**
         * Returns a Vector3, the absolute light position in the World.
         * @returns the world space position of the light
         */
        getAbsolutePosition(): Vector3;
        /**
         * Specifies if the light will affect the passed mesh.
         * @param mesh The mesh to test against the light
         * @return true the mesh is affected otherwise, false.
         */
        canAffectMesh(mesh: AbstractMesh): boolean;
        /**
         * Computes and Returns the light World matrix.
         * @returns the world matrix
         */
        getWorldMatrix(): Matrix;
        /**
         * Sort function to order lights for rendering.
         * @param a First Light object to compare to second.
         * @param b Second Light object to compare first.
         * @return -1 to reduce's a's index relative to be, 0 for no change, 1 to increase a's index relative to b.
         */
        static CompareLightsPriority(a: Light, b: Light): number;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Returns the light type ID (integer).
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Returns the intensity scaled by the Photometric Scale according to the light type and intensity mode.
         * @returns the scaled intensity in intensity mode unit
         */
        getScaledIntensity(): number;
        /**
         * Returns a new Light object, named "name", from the current one.
         * @param name The name of the cloned light
         * @returns the new created light
         */
        clone(name: string): Nullable<Light>;
        /**
         * Serializes the current light into a Serialization object.
         * @returns the serialized object.
         */
        serialize(): any;
        /**
         * Creates a new typed light from the passed type (integer) : point light = 0, directional light = 1, spot light = 2, hemispheric light = 3.
         * This new light is named "name" and added to the passed scene.
         * @param type Type according to the types available in Light.LIGHTTYPEID_x
         * @param name The friendly name of the light
         * @param scene The scene the new light will belong to
         * @returns the constructor function
         */
        static GetConstructorFromName(type: number, name: string, scene: Scene): Nullable<() => Light>;
        /**
         * Parses the passed "parsedLight" and returns a new instanced Light from this parsing.
         * @param parsedLight The JSON representation of the light
         * @param scene The scene to create the parsed light in
         * @returns the created light after parsing
         */
        static Parse(parsedLight: any, scene: Scene): Nullable<Light>;
        private _hookArrayForExcluded(array);
        private _hookArrayForIncludedOnly(array);
        private _resyncMeshes();
        /**
         * Forces the meshes to update their light related information in their rendering used effects
         * @hidden Internal Use Only
         */
        _markMeshesAsLightDirty(): void;
        /**
         * Recomputes the cached photometric scale if needed.
         */
        private _computePhotometricScale();
        /**
         * Returns the Photometric Scale according to the light type and intensity mode.
         */
        private _getPhotometricScale();
        /**
         * Reorder the light in the scene according to their defined priority.
         * @hidden Internal Use Only
         */
        _reorderLightsInScene(): void;
    }
}

declare module BABYLON {
    /**
     * A point light is a light defined by an unique point in world space.
     * The light is emitted in every direction from this point.
     * A good example of a point light is a standard light bulb.
     * Documentation: https://doc.babylonjs.com/babylon101/lights
     */
    class PointLight extends ShadowLight {
        private _shadowAngle;
        /**
         * Getter: In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         * This specifies what angle the shadow will use to be created.
         *
         * It default to 90 degrees to work nicely with the cube texture generation for point lights shadow maps.
         */
        /**
         * Setter: In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         * This specifies what angle the shadow will use to be created.
         *
         * It default to 90 degrees to work nicely with the cube texture generation for point lights shadow maps.
         */
        shadowAngle: number;
        /**
         * Gets the direction if it has been set.
         * In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         */
        /**
         * In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         */
        direction: Vector3;
        /**
         * Creates a PointLight object from the passed name and position (Vector3) and adds it in the scene.
         * A PointLight emits the light in every direction.
         * It can cast shadows.
         * If the scene camera is already defined and you want to set your PointLight at the camera position, just set it :
         * ```javascript
         * var pointLight = new BABYLON.PointLight("pl", camera.position, scene);
         * ```
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The light friendly name
         * @param position The position of the point light in the scene
         * @param scene The scene the lights belongs to
         */
        constructor(name: string, position: Vector3, scene: Scene);
        /**
         * Returns the string "PointLight"
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Returns the integer 0.
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Specifies wether or not the shadowmap should be a cube texture.
         * @returns true if the shadowmap needs to be a cube texture.
         */
        needCube(): boolean;
        /**
         * Returns a new Vector3 aligned with the PointLight cube system according to the passed cube face index (integer).
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Sets the passed matrix "matrix" as a left-handed perspective projection matrix with the following settings :
         * - fov = PI / 2
         * - aspect ratio : 1.0
         * - z-near and far equal to the active camera minZ and maxZ.
         * Returns the PointLight.
         */
        protected _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _buildUniformLayout(): void;
        /**
         * Sets the passed Effect "effect" with the PointLight transformed position (or position, if none) and passed name (string).
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The point light
         */
        transferToEffect(effect: Effect, lightIndex: string): PointLight;
    }
}

declare module BABYLON {
    /**
     * Interface describing all the common properties and methods a shadow light needs to implement.
     * This helps both the shadow generator and materials to genrate the corresponding shadow maps
     * as well as binding the different shadow properties to the effects.
     */
    interface IShadowLight extends Light {
        /**
         * The light id in the scene (used in scene.findLighById for instance)
         */
        id: string;
        /**
         * The position the shdow will be casted from.
         */
        position: Vector3;
        /**
         * In 2d mode (needCube being false), the direction used to cast the shadow.
         */
        direction: Vector3;
        /**
         * The transformed position. Position of the light in world space taking parenting in account.
         */
        transformedPosition: Vector3;
        /**
         * The transformed direction. Direction of the light in world space taking parenting in account.
         */
        transformedDirection: Vector3;
        /**
         * The friendly name of the light in the scene.
         */
        name: string;
        /**
         * Defines the shadow projection clipping minimum z value.
         */
        shadowMinZ: number;
        /**
         * Defines the shadow projection clipping maximum z value.
         */
        shadowMaxZ: number;
        /**
         * Computes the transformed information (transformedPosition and transformedDirection in World space) of the current light
         * @returns true if the information has been computed, false if it does not need to (no parenting)
         */
        computeTransformedInformation(): boolean;
        /**
         * Gets the scene the light belongs to.
         * @returns The scene
         */
        getScene(): Scene;
        /**
         * Callback defining a custom Projection Matrix Builder.
         * This can be used to override the default projection matrix computation.
         */
        customProjectionMatrixBuilder: (viewMatrix: Matrix, renderList: Array<AbstractMesh>, result: Matrix) => void;
        /**
         * Sets the shadow projection matrix in parameter to the generated projection matrix.
         * @param matrix The materix to updated with the projection information
         * @param viewMatrix The transform matrix of the light
         * @param renderList The list of mesh to render in the map
         * @returns The current light
         */
        setShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): IShadowLight;
        /**
         * Gets the current depth scale used in ESM.
         * @returns The scale
         */
        getDepthScale(): number;
        /**
         * Returns whether or not the shadow generation require a cube texture or a 2d texture.
         * @returns true if a cube texture needs to be use
         */
        needCube(): boolean;
        /**
         * Detects if the projection matrix requires to be recomputed this frame.
         * @returns true if it requires to be recomputed otherwise, false.
         */
        needProjectionMatrixCompute(): boolean;
        /**
         * Forces the shadow generator to recompute the projection matrix even if position and direction did not changed.
         */
        forceProjectionMatrixCompute(): void;
        /**
         * Get the direction to use to render the shadow map. In case of cube texture, the face index can be passed.
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
    }
    /**
     * Base implementation IShadowLight
     * It groups all the common behaviour in order to reduce dupplication and better follow the DRY pattern.
     */
    abstract class ShadowLight extends Light implements IShadowLight {
        protected abstract _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _position: Vector3;
        protected _setPosition(value: Vector3): void;
        /**
         * Sets the position the shadow will be casted from. Also use as the light position for both
         * point and spot lights.
         */
        /**
         * Sets the position the shadow will be casted from. Also use as the light position for both
         * point and spot lights.
         */
        position: Vector3;
        protected _direction: Vector3;
        protected _setDirection(value: Vector3): void;
        /**
         * In 2d mode (needCube being false), gets the direction used to cast the shadow.
         * Also use as the light direction on spot and directional lights.
         */
        /**
         * In 2d mode (needCube being false), sets the direction used to cast the shadow.
         * Also use as the light direction on spot and directional lights.
         */
        direction: Vector3;
        private _shadowMinZ;
        /**
         * Gets the shadow projection clipping minimum z value.
         */
        /**
         * Sets the shadow projection clipping minimum z value.
         */
        shadowMinZ: number;
        private _shadowMaxZ;
        /**
         * Sets the shadow projection clipping maximum z value.
         */
        /**
         * Gets the shadow projection clipping maximum z value.
         */
        shadowMaxZ: number;
        /**
         * Callback defining a custom Projection Matrix Builder.
         * This can be used to override the default projection matrix computation.
         */
        customProjectionMatrixBuilder: (viewMatrix: Matrix, renderList: Array<AbstractMesh>, result: Matrix) => void;
        /**
         * The transformed position. Position of the light in world space taking parenting in account.
         */
        transformedPosition: Vector3;
        /**
         * The transformed direction. Direction of the light in world space taking parenting in account.
         */
        transformedDirection: Vector3;
        private _worldMatrix;
        private _needProjectionMatrixCompute;
        /**
         * Computes the transformed information (transformedPosition and transformedDirection in World space) of the current light
         * @returns true if the information has been computed, false if it does not need to (no parenting)
         */
        computeTransformedInformation(): boolean;
        /**
         * Return the depth scale used for the shadow map.
         * @returns the depth scale.
         */
        getDepthScale(): number;
        /**
         * Get the direction to use to render the shadow map. In case of cube texture, the face index can be passed.
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Returns the ShadowLight absolute position in the World.
         * @returns the position vector in world space
         */
        getAbsolutePosition(): Vector3;
        /**
         * Sets the ShadowLight direction toward the passed target.
         * @param target The point tot target in local space
         * @returns the updated ShadowLight direction
         */
        setDirectionToTarget(target: Vector3): Vector3;
        /**
         * Returns the light rotation in euler definition.
         * @returns the x y z rotation in local space.
         */
        getRotation(): Vector3;
        /**
         * Returns whether or not the shadow generation require a cube texture or a 2d texture.
         * @returns true if a cube texture needs to be use
         */
        needCube(): boolean;
        /**
         * Detects if the projection matrix requires to be recomputed this frame.
         * @returns true if it requires to be recomputed otherwise, false.
         */
        needProjectionMatrixCompute(): boolean;
        /**
         * Forces the shadow generator to recompute the projection matrix even if position and direction did not changed.
         */
        forceProjectionMatrixCompute(): void;
        /**
         * Get the world matrix of the sahdow lights.
         * @hidden Internal Use Only
         */
        _getWorldMatrix(): Matrix;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
        /**
         * Sets the shadow projection matrix in parameter to the generated projection matrix.
         * @param matrix The materix to updated with the projection information
         * @param viewMatrix The transform matrix of the light
         * @param renderList The list of mesh to render in the map
         * @returns The current light
         */
        setShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): IShadowLight;
    }
}

declare module BABYLON {
    /**
     * A spot light is defined by a position, a direction, an angle, and an exponent.
     * These values define a cone of light starting from the position, emitting toward the direction.
     * The angle, in radians, defines the size (field of illumination) of the spotlight's conical beam,
     * and the exponent defines the speed of the decay of the light with distance (reach).
     * Documentation: https://doc.babylonjs.com/babylon101/lights
     */
    class SpotLight extends ShadowLight {
        private _angle;
        /**
         * Gets the cone angle of the spot light in Radians.
         */
        /**
         * Sets the cone angle of the spot light in Radians.
         */
        angle: number;
        private _shadowAngleScale;
        /**
         * Allows scaling the angle of the light for shadow generation only.
         */
        /**
         * Allows scaling the angle of the light for shadow generation only.
         */
        shadowAngleScale: number;
        /**
         * The light decay speed with the distance from the emission spot.
         */
        exponent: number;
        private _projectionTextureMatrix;
        /**
        * Allows reading the projecton texture
        */
        readonly projectionTextureMatrix: Matrix;
        protected _projectionTextureLightNear: number;
        /**
         * Gets the near clip of the Spotlight for texture projection.
         */
        /**
         * Sets the near clip of the Spotlight for texture projection.
         */
        projectionTextureLightNear: number;
        protected _projectionTextureLightFar: number;
        /**
         * Gets the far clip of the Spotlight for texture projection.
         */
        /**
         * Sets the far clip of the Spotlight for texture projection.
         */
        projectionTextureLightFar: number;
        protected _projectionTextureUpDirection: Vector3;
        /**
         * Gets the Up vector of the Spotlight for texture projection.
         */
        /**
         * Sets the Up vector of the Spotlight for texture projection.
         */
        projectionTextureUpDirection: Vector3;
        private _projectionTexture;
        /**
         * Gets the projection texture of the light.
        */
        /**
        * Sets the projection texture of the light.
        */
        projectionTexture: Nullable<BaseTexture>;
        private _projectionTextureViewLightDirty;
        private _projectionTextureProjectionLightDirty;
        private _projectionTextureDirty;
        private _projectionTextureViewTargetVector;
        private _projectionTextureViewLightMatrix;
        private _projectionTextureProjectionLightMatrix;
        private _projectionTextureScalingMatrix;
        /**
         * Creates a SpotLight object in the scene. A spot light is a simply light oriented cone.
         * It can cast shadows.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The light friendly name
         * @param position The position of the spot light in the scene
         * @param direction The direction of the light in the scene
         * @param angle The cone angle of the light in Radians
         * @param exponent The light decay speed with the distance from the emission spot
         * @param scene The scene the lights belongs to
         */
        constructor(name: string, position: Vector3, direction: Vector3, angle: number, exponent: number, scene: Scene);
        /**
         * Returns the string "SpotLight".
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Returns the integer 2.
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Overrides the direction setter to recompute the projection texture view light Matrix.
         */
        protected _setDirection(value: Vector3): void;
        /**
         * Overrides the position setter to recompute the projection texture view light Matrix.
         */
        protected _setPosition(value: Vector3): void;
        /**
         * Sets the passed matrix "matrix" as perspective projection matrix for the shadows and the passed view matrix with the fov equal to the SpotLight angle and and aspect ratio of 1.0.
         * Returns the SpotLight.
         */
        protected _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _computeProjectionTextureViewLightMatrix(): void;
        protected _computeProjectionTextureProjectionLightMatrix(): void;
        /**
         * Main function for light texture projection matrix computing.
         */
        protected _computeProjectionTextureMatrix(): void;
        protected _buildUniformLayout(): void;
        /**
         * Sets the passed Effect object with the SpotLight transfomed position (or position if not parented) and normalized direction.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The spot light
         */
        transferToEffect(effect: Effect, lightIndex: string): SpotLight;
        /**
         * Disposes the light and the associated resources.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    interface ILoadingScreen {
        displayLoadingUI: () => void;
        hideLoadingUI: () => void;
        loadingUIBackgroundColor: string;
        loadingUIText: string;
    }
    class DefaultLoadingScreen implements ILoadingScreen {
        private _renderingCanvas;
        private _loadingText;
        private _loadingDivBackgroundColor;
        private _loadingDiv;
        private _loadingTextDiv;
        constructor(_renderingCanvas: HTMLCanvasElement, _loadingText?: string, _loadingDivBackgroundColor?: string);
        displayLoadingUI(): void;
        hideLoadingUI(): void;
        loadingUIText: string;
        loadingUIBackgroundColor: string;
        private _resizeLoadingUI;
    }
}

declare module BABYLON {
    class SceneLoaderProgressEvent {
        readonly lengthComputable: boolean;
        readonly loaded: number;
        readonly total: number;
        constructor(lengthComputable: boolean, loaded: number, total: number);
        static FromProgressEvent(event: ProgressEvent): SceneLoaderProgressEvent;
    }
    interface ISceneLoaderPluginExtensions {
        [extension: string]: {
            isBinary: boolean;
        };
    }
    interface ISceneLoaderPluginFactory {
        name: string;
        createPlugin(): ISceneLoaderPlugin | ISceneLoaderPluginAsync;
        canDirectLoad?: (data: string) => boolean;
    }
    interface ISceneLoaderPlugin {
        /**
         * The friendly name of this plugin.
         */
        name: string;
        /**
         * The file extensions supported by this plugin.
         */
        extensions: string | ISceneLoaderPluginExtensions;
        /**
         * Import meshes into a scene.
         * @param meshesNames An array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
         * @param scene The scene to import into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param meshes The meshes array to import into
         * @param particleSystems The particle systems array to import into
         * @param skeletons The skeletons array to import into
         * @param onError The callback when import fails
         * @returns True if successful or false otherwise
         */
        importMesh(meshesNames: any, scene: Scene, data: any, rootUrl: string, meshes: AbstractMesh[], particleSystems: ParticleSystem[], skeletons: Skeleton[], onError?: (message: string, exception?: any) => void): boolean;
        /**
         * Load into a scene.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onError The callback when import fails
         * @returns true if successful or false otherwise
         */
        load(scene: Scene, data: string, rootUrl: string, onError?: (message: string, exception?: any) => void): boolean;
        /**
         * The callback that returns true if the data can be directly loaded.
         */
        canDirectLoad?: (data: string) => boolean;
        /**
         * The callback that allows custom handling of the root url based on the response url.
         */
        rewriteRootURL?: (rootUrl: string, responseURL?: string) => string;
        /**
         * Load into an asset container.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onError The callback when import fails
         * @returns The loaded asset container
         */
        loadAssetContainer(scene: Scene, data: string, rootUrl: string, onError?: (message: string, exception?: any) => void): AssetContainer;
    }
    interface ISceneLoaderPluginAsync {
        /**
         * The friendly name of this plugin.
         */
        name: string;
        /**
         * The file extensions supported by this plugin.
         */
        extensions: string | ISceneLoaderPluginExtensions;
        /**
         * Import meshes into a scene.
         * @param meshesNames An array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
         * @param scene The scene to import into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onProgress The callback when the load progresses
         * @returns The loaded meshes, particle systems, skeletons, and animation groups
         */
        importMeshAsync(meshesNames: any, scene: Scene, data: any, rootUrl: string, onProgress?: (event: SceneLoaderProgressEvent) => void): Promise<{
            meshes: AbstractMesh[];
            particleSystems: ParticleSystem[];
            skeletons: Skeleton[];
            animationGroups: AnimationGroup[];
        }>;
        /**
         * Load into a scene.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onProgress The callback when the load progresses
         * @returns Nothing
         */
        loadAsync(scene: Scene, data: string, rootUrl: string, onProgress?: (event: SceneLoaderProgressEvent) => void): Promise<void>;
        /**
         * The callback that returns true if the data can be directly loaded.
         */
        canDirectLoad?: (data: string) => boolean;
        /**
         * The callback that allows custom handling of the root url based on the response url.
         */
        rewriteRootURL?: (rootUrl: string, responseURL?: string) => string;
        /**
         * Load into an asset container.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onProgress The callback when the load progresses
         * @returns The loaded asset container
         */
        loadAssetContainerAsync(scene: Scene, data: string, rootUrl: string, onProgress?: (event: SceneLoaderProgressEvent) => void): Promise<AssetContainer>;
    }
    class SceneLoader {
        private static _ForceFullSceneLoadingForIncremental;
        private static _ShowLoadingScreen;
        private static _CleanBoneMatrixWeights;
        static readonly NO_LOGGING: number;
        static readonly MINIMAL_LOGGING: number;
        static readonly SUMMARY_LOGGING: number;
        static readonly DETAILED_LOGGING: number;
        private static _loggingLevel;
        static ForceFullSceneLoadingForIncremental: boolean;
        static ShowLoadingScreen: boolean;
        static loggingLevel: number;
        static CleanBoneMatrixWeights: boolean;
        static OnPluginActivatedObservable: Observable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        private static _registeredPlugins;
        private static _getDefaultPlugin();
        private static _getPluginForExtension(extension);
        private static _getPluginForDirectLoad(data);
        private static _getPluginForFilename(sceneFilename);
        private static _getDirectLoad(sceneFilename);
        private static _loadData(rootUrl, sceneFilename, scene, onSuccess, onProgress, onError, onDispose, pluginExtension);
        static GetPluginForExtension(extension: string): ISceneLoaderPlugin | ISceneLoaderPluginAsync | ISceneLoaderPluginFactory;
        static IsPluginForExtensionAvailable(extension: string): boolean;
        static RegisterPlugin(plugin: ISceneLoaderPlugin | ISceneLoaderPluginAsync): void;
        /**
         * Import meshes into a scene
         * @param meshNames an array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
         * @param rootUrl a string that defines the root url for scene and resources
         * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
         * @param scene the instance of BABYLON.Scene to append to
         * @param onSuccess a callback with a list of imported meshes, particleSystems, and skeletons when import succeeds
         * @param onProgress a callback with a progress event for each file being loaded
         * @param onError a callback with the scene, a message, and possibly an exception when import fails
         * @param pluginExtension the extension used to determine the plugin
         * @returns The loaded plugin
         */
        static ImportMesh(meshNames: any, rootUrl: string, sceneFilename: string, scene: Scene, onSuccess?: Nullable<(meshes: AbstractMesh[], particleSystems: ParticleSystem[], skeletons: Skeleton[], animationGroups: AnimationGroup[]) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Import meshes into a scene
        * @param meshNames an array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene the instance of BABYLON.Scene to append to
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded list of imported meshes, particle systems, skeletons, and animation groups
        */
        static ImportMeshAsync(meshNames: any, rootUrl: string, sceneFilename: string, scene: Scene, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<{
            meshes: AbstractMesh[];
            particleSystems: ParticleSystem[];
            skeletons: Skeleton[];
            animationGroups: AnimationGroup[];
        }>;
        /**
        * Load a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param engine is the instance of BABYLON.Engine to use to create the scene
        * @param onSuccess a callback with the scene when import succeeds
        * @param onProgress a callback with a progress event for each file being loaded
        * @param onError a callback with the scene, a message, and possibly an exception when import fails
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded plugin
        */
        static Load(rootUrl: string, sceneFilename: any, engine: Engine, onSuccess?: Nullable<(scene: Scene) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Load a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param engine is the instance of BABYLON.Engine to use to create the scene
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded scene
        */
        static LoadAsync(rootUrl: string, sceneFilename: any, engine: Engine, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<Scene>;
        /**
        * Append a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onSuccess a callback with the scene when import succeeds
        * @param onProgress a callback with a progress event for each file being loaded
        * @param onError a callback with the scene, a message, and possibly an exception when import fails
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded plugin
        */
        static Append(rootUrl: string, sceneFilename: any, scene: Scene, onSuccess?: Nullable<(scene: Scene) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Append a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The given scene
        */
        static AppendAsync(rootUrl: string, sceneFilename: any, scene: Scene, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<Scene>;
        /**
        * Load a scene into an asset container
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onSuccess a callback with the scene when import succeeds
        * @param onProgress a callback with a progress event for each file being loaded
        * @param onError a callback with the scene, a message, and possibly an exception when import fails
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded plugin
        */
        static LoadAssetContainer(rootUrl: string, sceneFilename: any, scene: Scene, onSuccess?: Nullable<(assets: AssetContainer) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Load a scene into an asset container
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded asset container
        */
        static LoadAssetContainerAsync(rootUrl: string, sceneFilename: any, scene: Scene, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<AssetContainer>;
    }
}

declare module BABYLON {
    /**
     * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
     * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
     * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
     * corresponding to low luminance, medium luminance, and high luminance areas respectively.
     */
    class ColorCurves {
        private _dirty;
        private _tempColor;
        private _globalCurve;
        private _highlightsCurve;
        private _midtonesCurve;
        private _shadowsCurve;
        private _positiveCurve;
        private _negativeCurve;
        private _globalHue;
        private _globalDensity;
        private _globalSaturation;
        private _globalExposure;
        /**
         * Gets the global Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the global Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        globalHue: number;
        /**
         * Gets the global Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the global Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        globalDensity: number;
        /**
         * Gets the global Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the global Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        globalSaturation: number;
        private _highlightsHue;
        private _highlightsDensity;
        private _highlightsSaturation;
        private _highlightsExposure;
        /**
         * Gets the highlights Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the highlights Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        highlightsHue: number;
        /**
         * Gets the highlights Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the highlights Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        highlightsDensity: number;
        /**
         * Gets the highlights Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the highlights Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        highlightsSaturation: number;
        /**
         * Gets the highlights Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        /**
         * Sets the highlights Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        highlightsExposure: number;
        private _midtonesHue;
        private _midtonesDensity;
        private _midtonesSaturation;
        private _midtonesExposure;
        /**
         * Gets the midtones Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the midtones Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        midtonesHue: number;
        /**
         * Gets the midtones Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the midtones Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        midtonesDensity: number;
        /**
         * Gets the midtones Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the midtones Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        midtonesSaturation: number;
        /**
         * Gets the midtones Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        /**
         * Sets the midtones Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        midtonesExposure: number;
        private _shadowsHue;
        private _shadowsDensity;
        private _shadowsSaturation;
        private _shadowsExposure;
        /**
         * Gets the shadows Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the shadows Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        shadowsHue: number;
        /**
         * Gets the shadows Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the shadows Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        shadowsDensity: number;
        /**
         * Gets the shadows Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the shadows Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        shadowsSaturation: number;
        /**
         * Gets the shadows Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        /**
         * Sets the shadows Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        shadowsExposure: number;
        getClassName(): string;
        /**
         * Binds the color curves to the shader.
         * @param colorCurves The color curve to bind
         * @param effect The effect to bind to
         */
        static Bind(colorCurves: ColorCurves, effect: Effect, positiveUniform?: string, neutralUniform?: string, negativeUniform?: string): void;
        /**
         * Prepare the list of uniforms associated with the ColorCurves effects.
         * @param uniformsList The list of uniforms used in the effect
         */
        static PrepareUniforms(uniformsList: string[]): void;
        /**
         * Returns color grading data based on a hue, density, saturation and exposure value.
         * @param filterHue The hue of the color filter.
         * @param filterDensity The density of the color filter.
         * @param saturation The saturation.
         * @param exposure The exposure.
         * @param result The result data container.
         */
        private getColorGradingDataToRef(hue, density, saturation, exposure, result);
        /**
         * Takes an input slider value and returns an adjusted value that provides extra control near the centre.
         * @param value The input slider value in range [-100,100].
         * @returns Adjusted value.
         */
        private static applyColorGradingSliderNonlinear(value);
        /**
         * Returns an RGBA Color4 based on Hue, Saturation and Brightness (also referred to as value, HSV).
         * @param hue The hue (H) input.
         * @param saturation The saturation (S) input.
         * @param brightness The brightness (B) input.
         * @result An RGBA color represented as Vector4.
         */
        private static fromHSBToRef(hue, saturation, brightness, result);
        /**
         * Returns a value clamped between min and max
         * @param value The value to clamp
         * @param min The minimum of value
         * @param max The maximum of value
         * @returns The clamped value.
         */
        private static clamp(value, min, max);
        /**
         * Clones the current color curve instance.
         * @return The cloned curves
         */
        clone(): ColorCurves;
        /**
         * Serializes the current color curve instance to a json representation.
         * @return a JSON representation
         */
        serialize(): any;
        /**
         * Parses the color curve from a json representation.
         * @param source the JSON source to parse
         * @return The parsed curves
         */
        static Parse(source: any): ColorCurves;
    }
}

declare module BABYLON {
    /**
     * EffectFallbacks can be used to add fallbacks (properties to disable) to certain properties when desired to improve performance.
     * (Eg. Start at high quality with reflection and fog, if fps is low, remove reflection, if still low remove fog)
     */
    class EffectFallbacks {
        private _defines;
        private _currentRank;
        private _maxRank;
        private _mesh;
        /**
         * Removes the fallback from the bound mesh.
         */
        unBindMesh(): void;
        /**
         * Adds a fallback on the specified property.
         * @param rank The rank of the fallback (Lower ranks will be fallbacked to first)
         * @param define The name of the define in the shader
         */
        addFallback(rank: number, define: string): void;
        /**
         * Sets the mesh to use CPU skinning when needing to fallback.
         * @param rank The rank of the fallback (Lower ranks will be fallbacked to first)
         * @param mesh The mesh to use the fallbacks.
         */
        addCPUSkinningFallback(rank: number, mesh: AbstractMesh): void;
        /**
         * Checks to see if more fallbacks are still availible.
         */
        readonly isMoreFallbacks: boolean;
        /**
         * Removes the defines that shoould be removed when falling back.
         * @param currentDefines defines the current define statements for the shader.
         * @param effect defines the current effect we try to compile
         * @returns The resulting defines with defines of the current rank removed.
         */
        reduce(currentDefines: string, effect: Effect): string;
    }
    /**
     * Options to be used when creating an effect.
     */
    class EffectCreationOptions {
        /**
         * Atrributes that will be used in the shader.
         */
        attributes: string[];
        /**
         * Uniform varible names that will be set in the shader.
         */
        uniformsNames: string[];
        /**
         * Uniform buffer varible names that will be set in the shader.
         */
        uniformBuffersNames: string[];
        /**
         * Sampler texture variable names that will be set in the shader.
         */
        samplers: string[];
        /**
         * Define statements that will be set in the shader.
         */
        defines: any;
        /**
         * Possible fallbacks for this effect to improve performance when needed.
         */
        fallbacks: Nullable<EffectFallbacks>;
        /**
         * Callback that will be called when the shader is compiled.
         */
        onCompiled: Nullable<(effect: Effect) => void>;
        /**
         * Callback that will be called if an error occurs during shader compilation.
         */
        onError: Nullable<(effect: Effect, errors: string) => void>;
        /**
         * Parameters to be used with Babylons include syntax to iterate over an array (eg. {lights: 10})
         */
        indexParameters: any;
        /**
         * Max number of lights that can be used in the shader.
         */
        maxSimultaneousLights: number;
        /**
         * See https://developer.mozilla.org/en-US/docs/Web/API/WebGL2RenderingContext/transformFeedbackVaryings
         */
        transformFeedbackVaryings: Nullable<string[]>;
    }
    /**
     * Effect containing vertex and fragment shader that can be executed on an object.
     */
    class Effect {
        /**
         * Name of the effect.
         */
        name: any;
        /**
         * String container all the define statements that should be set on the shader.
         */
        defines: string;
        /**
         * Callback that will be called when the shader is compiled.
         */
        onCompiled: Nullable<(effect: Effect) => void>;
        /**
         * Callback that will be called if an error occurs during shader compilation.
         */
        onError: Nullable<(effect: Effect, errors: string) => void>;
        /**
         * Callback that will be called when effect is bound.
         */
        onBind: Nullable<(effect: Effect) => void>;
        /**
         * Unique ID of the effect.
         */
        uniqueId: number;
        /**
         * Observable that will be called when the shader is compiled.
         */
        onCompileObservable: Observable<Effect>;
        /**
         * Observable that will be called if an error occurs during shader compilation.
         */
        onErrorObservable: Observable<Effect>;
        /**
         * Observable that will be called when effect is bound.
         */
        onBindObservable: Observable<Effect>;
        /** @hidden */
        _bonesComputationForcedToCPU: boolean;
        private static _uniqueIdSeed;
        private _engine;
        private _uniformBuffersNames;
        private _uniformsNames;
        private _samplers;
        private _isReady;
        private _compilationError;
        private _attributesNames;
        private _attributes;
        private _uniforms;
        /**
         * Key for the effect.
         */
        _key: string;
        private _indexParameters;
        private _fallbacks;
        private _vertexSourceCode;
        private _fragmentSourceCode;
        private _vertexSourceCodeOverride;
        private _fragmentSourceCodeOverride;
        private _transformFeedbackVaryings;
        /**
         * Compiled shader to webGL program.
         */
        _program: WebGLProgram;
        private _valueCache;
        private static _baseCache;
        /**
         * Instantiates an effect.
         * An effect can be used to create/manage/execute vertex and fragment shaders.
         * @param baseName Name of the effect.
         * @param attributesNamesOrOptions List of attribute names that will be passed to the shader or set of all options to create the effect.
         * @param uniformsNamesOrEngine List of uniform variable names that will be passed to the shader or the engine that will be used to render effect.
         * @param samplers List of sampler variables that will be passed to the shader.
         * @param engine Engine to be used to render the effect
         * @param defines Define statements to be added to the shader.
         * @param fallbacks Possible fallbacks for this effect to improve performance when needed.
         * @param onCompiled Callback that will be called when the shader is compiled.
         * @param onError Callback that will be called if an error occurs during shader compilation.
         * @param indexParameters Parameters to be used with Babylons include syntax to iterate over an array (eg. {lights: 10})
         */
        constructor(baseName: any, attributesNamesOrOptions: string[] | EffectCreationOptions, uniformsNamesOrEngine: string[] | Engine, samplers?: Nullable<string[]>, engine?: Engine, defines?: Nullable<string>, fallbacks?: Nullable<EffectFallbacks>, onCompiled?: Nullable<(effect: Effect) => void>, onError?: Nullable<(effect: Effect, errors: string) => void>, indexParameters?: any);
        /**
         * Unique key for this effect
         */
        readonly key: string;
        /**
         * If the effect has been compiled and prepared.
         * @returns if the effect is compiled and prepared.
         */
        isReady(): boolean;
        /**
         * The engine the effect was initialized with.
         * @returns the engine.
         */
        getEngine(): Engine;
        /**
         * The compiled webGL program for the effect
         * @returns the webGL program.
         */
        getProgram(): WebGLProgram;
        /**
         * The set of names of attribute variables for the shader.
         * @returns An array of attribute names.
         */
        getAttributesNames(): string[];
        /**
         * Returns the attribute at the given index.
         * @param index The index of the attribute.
         * @returns The location of the attribute.
         */
        getAttributeLocation(index: number): number;
        /**
         * Returns the attribute based on the name of the variable.
         * @param name of the attribute to look up.
         * @returns the attribute location.
         */
        getAttributeLocationByName(name: string): number;
        /**
         * The number of attributes.
         * @returns the numnber of attributes.
         */
        getAttributesCount(): number;
        /**
         * Gets the index of a uniform variable.
         * @param uniformName of the uniform to look up.
         * @returns the index.
         */
        getUniformIndex(uniformName: string): number;
        /**
         * Returns the attribute based on the name of the variable.
         * @param uniformName of the uniform to look up.
         * @returns the location of the uniform.
         */
        getUniform(uniformName: string): Nullable<WebGLUniformLocation>;
        /**
         * Returns an array of sampler variable names
         * @returns The array of sampler variable neames.
         */
        getSamplers(): string[];
        /**
         * The error from the last compilation.
         * @returns the error string.
         */
        getCompilationError(): string;
        /**
         * Adds a callback to the onCompiled observable and call the callback imediatly if already ready.
         * @param func The callback to be used.
         */
        executeWhenCompiled(func: (effect: Effect) => void): void;
        /** @hidden */
        _loadVertexShader(vertex: any, callback: (data: any) => void): void;
        /** @hidden */
        _loadFragmentShader(fragment: any, callback: (data: any) => void): void;
        private _dumpShadersSource(vertexCode, fragmentCode, defines);
        private _processShaderConversion(sourceCode, isFragment, callback);
        private _processIncludes(sourceCode, callback);
        private _processPrecision(source);
        /**
         * Recompiles the webGL program
         * @param vertexSourceCode The source code for the vertex shader.
         * @param fragmentSourceCode The source code for the fragment shader.
         * @param onCompiled Callback called when completed.
         * @param onError Callback called on error.
         */
        _rebuildProgram(vertexSourceCode: string, fragmentSourceCode: string, onCompiled: (program: WebGLProgram) => void, onError: (message: string) => void): void;
        /**
         * Gets the uniform locations of the the specified variable names
         * @param names THe names of the variables to lookup.
         * @returns Array of locations in the same order as variable names.
         */
        getSpecificUniformLocations(names: string[]): Nullable<WebGLUniformLocation>[];
        /**
         * Prepares the effect
         */
        _prepareEffect(): void;
        /**
         * Checks if the effect is supported. (Must be called after compilation)
         */
        readonly isSupported: boolean;
        /**
         * Binds a texture to the engine to be used as output of the shader.
         * @param channel Name of the output variable.
         * @param texture Texture to bind.
         */
        _bindTexture(channel: string, texture: InternalTexture): void;
        /**
         * Sets a texture on the engine to be used in the shader.
         * @param channel Name of the sampler variable.
         * @param texture Texture to set.
         */
        setTexture(channel: string, texture: Nullable<BaseTexture>): void;
        /**
         * Sets a depth stencil texture from a render target on the engine to be used in the shader.
         * @param channel Name of the sampler variable.
         * @param texture Texture to set.
         */
        setDepthStencilTexture(channel: string, texture: Nullable<RenderTargetTexture>): void;
        /**
         * Sets an array of textures on the engine to be used in the shader.
         * @param channel Name of the variable.
         * @param textures Textures to set.
         */
        setTextureArray(channel: string, textures: BaseTexture[]): void;
        /**
         * Sets a texture to be the input of the specified post process. (To use the output, pass in the next post process in the pipeline)
         * @param channel Name of the sampler variable.
         * @param postProcess Post process to get the input texture from.
         */
        setTextureFromPostProcess(channel: string, postProcess: Nullable<PostProcess>): void;
        /**
         * (Warning! setTextureFromPostProcessOutput may be desired instead)
         * Sets the input texture of the passed in post process to be input of this effect. (To use the output of the passed in post process use setTextureFromPostProcessOutput)
         * @param channel Name of the sampler variable.
         * @param postProcess Post process to get the output texture from.
         */
        setTextureFromPostProcessOutput(channel: string, postProcess: Nullable<PostProcess>): void;
        /** @hidden */
        _cacheMatrix(uniformName: string, matrix: Matrix): boolean;
        /** @hidden */
        _cacheFloat2(uniformName: string, x: number, y: number): boolean;
        /** @hidden */
        _cacheFloat3(uniformName: string, x: number, y: number, z: number): boolean;
        /** @hidden */
        _cacheFloat4(uniformName: string, x: number, y: number, z: number, w: number): boolean;
        /**
         * Binds a buffer to a uniform.
         * @param buffer Buffer to bind.
         * @param name Name of the uniform variable to bind to.
         */
        bindUniformBuffer(buffer: WebGLBuffer, name: string): void;
        /**
         * Binds block to a uniform.
         * @param blockName Name of the block to bind.
         * @param index Index to bind.
         */
        bindUniformBlock(blockName: string, index: number): void;
        /**
         * Sets an interger value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value Value to be set.
         * @returns this effect.
         */
        setInt(uniformName: string, value: number): Effect;
        /**
         * Sets an int array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an int array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray2(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an int array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray3(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an int array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray4(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an float array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an float array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray2(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an float array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray3(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an float array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray4(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray(uniformName: string, array: number[]): Effect;
        /**
         * Sets an array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray2(uniformName: string, array: number[]): Effect;
        /**
         * Sets an array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray3(uniformName: string, array: number[]): Effect;
        /**
         * Sets an array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray4(uniformName: string, array: number[]): Effect;
        /**
         * Sets matrices on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrices matrices to be set.
         * @returns this effect.
         */
        setMatrices(uniformName: string, matrices: Float32Array): Effect;
        /**
         * Sets matrix on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         * @returns this effect.
         */
        setMatrix(uniformName: string, matrix: Matrix): Effect;
        /**
         * Sets a 3x3 matrix on a uniform variable. (Speicified as [1,2,3,4,5,6,7,8,9] will result in [1,2,3][4,5,6][7,8,9] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         * @returns this effect.
         */
        setMatrix3x3(uniformName: string, matrix: Float32Array): Effect;
        /**
         * Sets a 2x2 matrix on a uniform variable. (Speicified as [1,2,3,4] will result in [1,2][3,4] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         * @returns this effect.
         */
        setMatrix2x2(uniformName: string, matrix: Float32Array): Effect;
        /**
         * Sets a float on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value value to be set.
         * @returns this effect.
         */
        setFloat(uniformName: string, value: number): Effect;
        /**
         * Sets a boolean on a uniform variable.
         * @param uniformName Name of the variable.
         * @param bool value to be set.
         * @returns this effect.
         */
        setBool(uniformName: string, bool: boolean): Effect;
        /**
         * Sets a Vector2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector2 vector2 to be set.
         * @returns this effect.
         */
        setVector2(uniformName: string, vector2: Vector2): Effect;
        /**
         * Sets a float2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float2.
         * @param y Second float in float2.
         * @returns this effect.
         */
        setFloat2(uniformName: string, x: number, y: number): Effect;
        /**
         * Sets a Vector3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector3 Value to be set.
         * @returns this effect.
         */
        setVector3(uniformName: string, vector3: Vector3): Effect;
        /**
         * Sets a float3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float3.
         * @param y Second float in float3.
         * @param z Third float in float3.
         * @returns this effect.
         */
        setFloat3(uniformName: string, x: number, y: number, z: number): Effect;
        /**
         * Sets a Vector4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector4 Value to be set.
         * @returns this effect.
         */
        setVector4(uniformName: string, vector4: Vector4): Effect;
        /**
         * Sets a float4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float4.
         * @param y Second float in float4.
         * @param z Third float in float4.
         * @param w Fourth float in float4.
         * @returns this effect.
         */
        setFloat4(uniformName: string, x: number, y: number, z: number, w: number): Effect;
        /**
         * Sets a Color3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         * @returns this effect.
         */
        setColor3(uniformName: string, color3: Color3): Effect;
        /**
         * Sets a Color4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         * @param alpha Alpha value to be set.
         * @returns this effect.
         */
        setColor4(uniformName: string, color3: Color3, alpha: number): Effect;
        /**
         * Sets a Color4 on a uniform variable
         * @param uniformName defines the name of the variable
         * @param color4 defines the value to be set
         * @returns this effect.
         */
        setDirectColor4(uniformName: string, color4: Color4): Effect;
        /**
         * Store of each shader (The can be looked up using effect.key)
         */
        static ShadersStore: {
            [key: string]: string;
        };
        /**
         * Store of each included file for a shader (The can be looked up using effect.key)
         */
        static IncludesShadersStore: {
            [key: string]: string;
        };
        /**
         * Resets the cache of effects.
         */
        static ResetCache(): void;
    }
}

declare module BABYLON {
    class FresnelParameters {
        private _isEnabled;
        isEnabled: boolean;
        leftColor: Color3;
        rightColor: Color3;
        bias: number;
        power: number;
        clone(): FresnelParameters;
        serialize(): any;
        static Parse(parsedFresnelParameters: any): FresnelParameters;
    }
}

declare module BABYLON {
    /**
     * Interface to follow in your material defines to integrate easily the
     * Image proccessing functions.
     * @hidden
     */
    interface IImageProcessingConfigurationDefines {
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        CONTRAST: boolean;
        EXPOSURE: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
    }
    /**
     * This groups together the common properties used for image processing either in direct forward pass
     * or through post processing effect depending on the use of the image processing pipeline in your scene
     * or not.
     */
    class ImageProcessingConfiguration {
        /**
         * Color curves setup used in the effect if colorCurvesEnabled is set to true
         */
        colorCurves: Nullable<ColorCurves>;
        private _colorCurvesEnabled;
        /**
         * Gets wether the color curves effect is enabled.
         */
        /**
         * Sets wether the color curves effect is enabled.
         */
        colorCurvesEnabled: boolean;
        /**
         * Color grading LUT texture used in the effect if colorGradingEnabled is set to true
         */
        colorGradingTexture: Nullable<BaseTexture>;
        private _colorGradingEnabled;
        /**
         * Gets wether the color grading effect is enabled.
         */
        /**
         * Sets wether the color grading effect is enabled.
         */
        colorGradingEnabled: boolean;
        private _colorGradingWithGreenDepth;
        /**
         * Gets wether the color grading effect is using a green depth for the 3d Texture.
         */
        /**
         * Sets wether the color grading effect is using a green depth for the 3d Texture.
         */
        colorGradingWithGreenDepth: boolean;
        private _colorGradingBGR;
        /**
         * Gets wether the color grading texture contains BGR values.
         */
        /**
         * Sets wether the color grading texture contains BGR values.
         */
        colorGradingBGR: boolean;
        _exposure: number;
        /**
         * Gets the Exposure used in the effect.
         */
        /**
         * Sets the Exposure used in the effect.
         */
        exposure: number;
        private _toneMappingEnabled;
        /**
         * Gets wether the tone mapping effect is enabled.
         */
        /**
         * Sets wether the tone mapping effect is enabled.
         */
        toneMappingEnabled: boolean;
        protected _contrast: number;
        /**
         * Gets the contrast used in the effect.
         */
        /**
         * Sets the contrast used in the effect.
         */
        contrast: number;
        /**
         * Vignette stretch size.
         */
        vignetteStretch: number;
        /**
         * Vignette centre X Offset.
         */
        vignetteCentreX: number;
        /**
         * Vignette centre Y Offset.
         */
        vignetteCentreY: number;
        /**
         * Vignette weight or intensity of the vignette effect.
         */
        vignetteWeight: number;
        /**
         * Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        vignetteColor: Color4;
        /**
         * Camera field of view used by the Vignette effect.
         */
        vignetteCameraFov: number;
        private _vignetteBlendMode;
        /**
         * Gets the vignette blend mode allowing different kind of effect.
         */
        /**
         * Sets the vignette blend mode allowing different kind of effect.
         */
        vignetteBlendMode: number;
        private _vignetteEnabled;
        /**
         * Gets wether the vignette effect is enabled.
         */
        /**
         * Sets wether the vignette effect is enabled.
         */
        vignetteEnabled: boolean;
        private _applyByPostProcess;
        /**
         * Gets wether the image processing is applied through a post process or not.
         */
        /**
         * Sets wether the image processing is applied through a post process or not.
         */
        applyByPostProcess: boolean;
        private _isEnabled;
        /**
         * Gets wether the image processing is enabled or not.
         */
        /**
         * Sets wether the image processing is enabled or not.
         */
        isEnabled: boolean;
        /**
        * An event triggered when the configuration changes and requires Shader to Update some parameters.
        */
        onUpdateParameters: Observable<ImageProcessingConfiguration>;
        /**
         * Method called each time the image processing information changes requires to recompile the effect.
         */
        protected _updateParameters(): void;
        getClassName(): string;
        /**
         * Prepare the list of uniforms associated with the Image Processing effects.
         * @param uniformsList The list of uniforms used in the effect
         * @param defines the list of defines currently in use
         */
        static PrepareUniforms(uniforms: string[], defines: IImageProcessingConfigurationDefines): void;
        /**
         * Prepare the list of samplers associated with the Image Processing effects.
         * @param uniformsList The list of uniforms used in the effect
         * @param defines the list of defines currently in use
         */
        static PrepareSamplers(samplersList: string[], defines: IImageProcessingConfigurationDefines): void;
        /**
         * Prepare the list of defines associated to the shader.
         * @param defines the list of defines to complete
         */
        prepareDefines(defines: IImageProcessingConfigurationDefines, forPostProcess?: boolean): void;
        /**
         * Returns true if all the image processing information are ready.
         */
        isReady(): boolean;
        /**
         * Binds the image processing to the shader.
         * @param effect The effect to bind to
         */
        bind(effect: Effect, aspectRatio?: number): void;
        /**
         * Clones the current image processing instance.
         * @return The cloned image processing
         */
        clone(): ImageProcessingConfiguration;
        /**
         * Serializes the current image processing instance to a json representation.
         * @return a JSON representation
         */
        serialize(): any;
        /**
         * Parses the image processing from a json representation.
         * @param source the JSON source to parse
         * @return The parsed image processing
         */
        static Parse(source: any): ImageProcessingConfiguration;
        private static _VIGNETTEMODE_MULTIPLY;
        private static _VIGNETTEMODE_OPAQUE;
        /**
         * Used to apply the vignette as a mix with the pixel color.
         */
        static readonly VIGNETTEMODE_MULTIPLY: number;
        /**
         * Used to apply the vignette as a replacement of the pixel color.
         */
        static readonly VIGNETTEMODE_OPAQUE: number;
    }
}

declare module BABYLON {
    /**
     * Manages the defines for the Material
     */
    class MaterialDefines {
        private _keys;
        private _isDirty;
        /** @hidden */
        _renderId: number;
        /** @hidden */
        _areLightsDirty: boolean;
        /** @hidden */
        _areAttributesDirty: boolean;
        /** @hidden */
        _areTexturesDirty: boolean;
        /** @hidden */
        _areFresnelDirty: boolean;
        /** @hidden */
        _areMiscDirty: boolean;
        /** @hidden */
        _areImageProcessingDirty: boolean;
        /** @hidden */
        _normals: boolean;
        /** @hidden */
        _uvs: boolean;
        /** @hidden */
        _needNormals: boolean;
        /** @hidden */
        _needUVs: boolean;
        /**
         * Specifies if the material needs to be re-calculated
         */
        readonly isDirty: boolean;
        /**
         * Marks the material to indicate that it has been re-calculated
         */
        markAsProcessed(): void;
        /**
         * Marks the material to indicate that it needs to be re-calculated
         */
        markAsUnprocessed(): void;
        /**
         * Marks the material to indicate all of its defines need to be re-calculated
         */
        markAllAsDirty(): void;
        /**
         * Marks the material to indicate that image processing needs to be re-calculated
         */
        markAsImageProcessingDirty(): void;
        /**
         * Marks the material to indicate the lights need to be re-calculated
         */
        markAsLightDirty(): void;
        /**
         * Marks the attribute state as changed
         */
        markAsAttributesDirty(): void;
        /**
         * Marks the texture state as changed
         */
        markAsTexturesDirty(): void;
        /**
         * Marks the fresnel state as changed
         */
        markAsFresnelDirty(): void;
        /**
         * Marks the misc state as changed
         */
        markAsMiscDirty(): void;
        /**
         * Rebuilds the material defines
         */
        rebuild(): void;
        /**
         * Specifies if two material defines are equal
         * @param other - A material define instance to compare to
         * @returns - Boolean indicating if the material defines are equal (true) or not (false)
         */
        isEqual(other: MaterialDefines): boolean;
        /**
         * Clones this instance's defines to another instance
         * @param other - material defines to clone values to
         */
        cloneTo(other: MaterialDefines): void;
        /**
         * Resets the material define values
         */
        reset(): void;
        /**
         * Converts the material define values to a string
         * @returns - String of material define information
         */
        toString(): string;
    }
    /**
     * Base class for the main features of a material in Babylon.js
     */
    class Material implements IAnimatable {
        private static _TriangleFillMode;
        private static _WireFrameFillMode;
        private static _PointFillMode;
        private static _PointListDrawMode;
        private static _LineListDrawMode;
        private static _LineLoopDrawMode;
        private static _LineStripDrawMode;
        private static _TriangleStripDrawMode;
        private static _TriangleFanDrawMode;
        /**
         * Returns the triangle fill mode
         */
        static readonly TriangleFillMode: number;
        /**
         * Returns the wireframe mode
         */
        static readonly WireFrameFillMode: number;
        /**
         * Returns the point fill mode
         */
        static readonly PointFillMode: number;
        /**
         * Returns the point list draw mode
         */
        static readonly PointListDrawMode: number;
        /**
         * Returns the line list draw mode
         */
        static readonly LineListDrawMode: number;
        /**
         * Returns the line loop draw mode
         */
        static readonly LineLoopDrawMode: number;
        /**
         * Returns the line strip draw mode
         */
        static readonly LineStripDrawMode: number;
        /**
         * Returns the triangle strip draw mode
         */
        static readonly TriangleStripDrawMode: number;
        /**
         * Returns the triangle fan draw mode
         */
        static readonly TriangleFanDrawMode: number;
        /**
         * Stores the clock-wise side orientation
         */
        private static _ClockWiseSideOrientation;
        /**
         * Stores the counter clock-wise side orientation
         */
        private static _CounterClockWiseSideOrientation;
        /**
         * Returns the clock-wise side orientation
         */
        static readonly ClockWiseSideOrientation: number;
        /**
         * Returns the counter clock-wise side orientation
         */
        static readonly CounterClockWiseSideOrientation: number;
        /**
         * The dirty texture flag value
         */
        private static _TextureDirtyFlag;
        /**
         * The dirty light flag value
         */
        private static _LightDirtyFlag;
        /**
         * The dirty fresnel flag value
         */
        private static _FresnelDirtyFlag;
        /**
         * The dirty attribute flag value
         */
        private static _AttributesDirtyFlag;
        /**
         * The dirty misc flag value
         */
        private static _MiscDirtyFlag;
        /**
         * Returns the dirty texture flag value
         */
        static readonly TextureDirtyFlag: number;
        /**
         * Returns the dirty light flag value
         */
        static readonly LightDirtyFlag: number;
        /**
         * Returns the dirty fresnel flag value
         */
        static readonly FresnelDirtyFlag: number;
        /**
         * Returns the dirty attributes flag value
         */
        static readonly AttributesDirtyFlag: number;
        /**
         * Returns the dirty misc flag value
         */
        static readonly MiscDirtyFlag: number;
        /**
         * The ID of the material
         */
        id: string;
        /**
         * The name of the material
         */
        name: string;
        /**
         * Specifies if the ready state should be checked on each call
         */
        checkReadyOnEveryCall: boolean;
        /**
         * Specifies if the ready state should be checked once
         */
        checkReadyOnlyOnce: boolean;
        /**
         * The state of the material
         */
        state: string;
        /**
         * The alpha value of the material
         */
        protected _alpha: number;
        /**
         * Gets the alpha value of the material
         */
        /**
         * Sets the alpha value of the material
         */
        alpha: number;
        /**
         * Specifies if back face culling is enabled
         */
        protected _backFaceCulling: boolean;
        /**
         * Gets the back-face culling state
         */
        /**
         * Sets the back-face culling state
         */
        backFaceCulling: boolean;
        /**
         * Stores the value for side orientation
         */
        sideOrientation: number;
        /**
         * Callback triggered when the material is compiled
         */
        onCompiled: (effect: Effect) => void;
        /**
         * Callback triggered when an error occurs
         */
        onError: (effect: Effect, errors: string) => void;
        /**
         * Callback triggered to get the render target textures
         */
        getRenderTargetTextures: () => SmartArray<RenderTargetTexture>;
        /**
         * Specifies if the material should be serialized
         */
        doNotSerialize: boolean;
        /**
         * Specifies if the effect should be stored on sub meshes
         */
        storeEffectOnSubMeshes: boolean;
        /**
         * Stores the animations for the material
         */
        animations: Array<Animation>;
        /**
        * An event triggered when the material is disposed
        */
        onDisposeObservable: Observable<Material>;
        /**
         * An observer which watches for dispose events
         */
        private _onDisposeObserver;
        /**
         * Called during a dispose event
         */
        onDispose: () => void;
        /**
        * An event triggered when the material is bound
        */
        onBindObservable: Observable<AbstractMesh>;
        /**
         * An observer which watches for bind events
         */
        private _onBindObserver;
        /**
         * Called during a bind event
         */
        onBind: (Mesh: AbstractMesh) => void;
        /**
        * An event triggered when the material is unbound
        */
        onUnBindObservable: Observable<Material>;
        /**
         * Stores the value of the alpha mode
         */
        private _alphaMode;
        /**
         * Gets the value of the alpha mode
         */
        /**
         * Sets the value of the alpha mode.
         *
         * | Value | Type | Description |
         * | --- | --- | --- |
         * | 0 | ALPHA_DISABLE |   |
         * | 1 | ALPHA_ADD |   |
         * | 2 | ALPHA_COMBINE |   |
         * | 3 | ALPHA_SUBTRACT |   |
         * | 4 | ALPHA_MULTIPLY |   |
         * | 5 | ALPHA_MAXIMIZED |   |
         * | 6 | ALPHA_ONEONE |   |
         * | 7 | ALPHA_PREMULTIPLIED |   |
         * | 8 | ALPHA_PREMULTIPLIED_PORTERDUFF |   |
         * | 9 | ALPHA_INTERPOLATE |   |
         * | 10 | ALPHA_SCREENMODE |   |
         *
         */
        alphaMode: number;
        /**
         * Stores the state of the need depth pre-pass value
         */
        private _needDepthPrePass;
        /**
         * Gets the depth pre-pass value
         */
        /**
         * Sets the need depth pre-pass value
         */
        needDepthPrePass: boolean;
        /**
         * Specifies if depth writing should be disabled
         */
        disableDepthWrite: boolean;
        /**
         * Specifies if depth writing should be forced
         */
        forceDepthWrite: boolean;
        /**
         * Specifies if there should be a separate pass for culling
         */
        separateCullingPass: boolean;
        /**
         * Stores the state specifing if fog should be enabled
         */
        private _fogEnabled;
        /**
         * Gets the value of the fog enabled state
         */
        /**
         * Sets the state for enabling fog
         */
        fogEnabled: boolean;
        /**
         * Stores the size of points
         */
        pointSize: number;
        /**
         * Stores the z offset value
         */
        zOffset: number;
        /**
         * Gets a value specifying if wireframe mode is enabled
         */
        /**
         * Sets the state of wireframe mode
         */
        wireframe: boolean;
        /**
         * Gets the value specifying if point clouds are enabled
         */
        /**
         * Sets the state of point cloud mode
         */
        pointsCloud: boolean;
        /**
         * Gets the material fill mode
         */
        /**
         * Sets the material fill mode
         */
        fillMode: number;
        /**
         * Stores the effects for the material
         */
        _effect: Nullable<Effect>;
        /**
         * Specifies if the material was previously ready
         */
        _wasPreviouslyReady: boolean;
        /**
         * Specifies if uniform buffers should be used
         */
        private _useUBO;
        /**
         * Stores a reference to the scene
         */
        private _scene;
        /**
         * Stores the fill mode state
         */
        private _fillMode;
        /**
         * Specifies if the depth write state should be cached
         */
        private _cachedDepthWriteState;
        /**
         * Stores the uniform buffer
         */
        protected _uniformBuffer: UniformBuffer;
        /**
         * Creates a material instance
         * @param name defines the name of the material
         * @param scene defines the scene to reference
         * @param doNotAdd specifies if the material should be added to the scene
         */
        constructor(name: string, scene: Scene, doNotAdd?: boolean);
        /**
         * Returns a string representation of the current material
         * @param fullDetails defines a boolean indicating which levels of logging is desired
         * @returns a string with material information
         */
        toString(fullDetails?: boolean): string;
        /**
         * Gets the class name of the material
         * @returns a string with the class name of the material
         */
        getClassName(): string;
        /**
         * Specifies if updates for the material been locked
         */
        readonly isFrozen: boolean;
        /**
         * Locks updates for the material
         */
        freeze(): void;
        /**
         * Unlocks updates for the material
         */
        unfreeze(): void;
        /**
         * Specifies if the material is ready to be used
         * @param mesh defines the mesh to check
         * @param useInstances specifies if instances should be used
         * @returns a boolean indicating if the material is ready to be used
         */
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        /**
         * Specifies that the submesh is ready to be used
         * @param mesh defines the mesh to check
         * @param subMesh defines which submesh to check
         * @param useInstances specifies that instances should be used
         * @returns a boolean indicating that the submesh is ready or not
         */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: BaseSubMesh, useInstances?: boolean): boolean;
        /**
         * Returns the material effect
         * @returns the effect associated with the material
         */
        getEffect(): Nullable<Effect>;
        /**
         * Returns the current scene
         * @returns a Scene
         */
        getScene(): Scene;
        /**
         * Specifies if the material will require alpha blending
         * @returns a boolean specifying if alpha blending is needed
         */
        needAlphaBlending(): boolean;
        /**
         * Specifies if the mesh will require alpha blending
         * @param mesh defines the mesh to check
         * @returns a boolean specifying if alpha blending is needed for the mesh
         */
        needAlphaBlendingForMesh(mesh: AbstractMesh): boolean;
        /**
         * Specifies if this material should be rendered in alpha test mode
         * @returns a boolean specifying if an alpha test is needed.
         */
        needAlphaTesting(): boolean;
        /**
         * Gets the texture used for the alpha test
         * @returns the texture to use for alpha testing
         */
        getAlphaTestTexture(): Nullable<BaseTexture>;
        /**
         * Marks the material to indicate that it needs to be re-calculated
         */
        markDirty(): void;
        /** @hidden */
        _preBind(effect?: Effect, overrideOrientation?: Nullable<number>): boolean;
        /**
         * Binds the material to the mesh
         * @param world defines the world transformation matrix
         * @param mesh defines the mesh to bind the material to
         */
        bind(world: Matrix, mesh?: Mesh): void;
        /**
         * Binds the submesh to the material
         * @param world defines the world transformation matrix
         * @param mesh defines the mesh containing the submesh
         * @param subMesh defines the submesh to bind the material to
         */
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        /**
         * Binds the world matrix to the material
         * @param world defines the world transformation matrix
         */
        bindOnlyWorldMatrix(world: Matrix): void;
        /**
         * Binds the scene's uniform buffer to the effect.
         * @param effect defines the effect to bind to the scene uniform buffer
         * @param sceneUbo defines the uniform buffer storing scene data
         */
        bindSceneUniformBuffer(effect: Effect, sceneUbo: UniformBuffer): void;
        /**
         * Binds the view matrix to the effect
         * @param effect defines the effect to bind the view matrix to
         */
        bindView(effect: Effect): void;
        /**
         * Binds the view projection matrix to the effect
         * @param effect defines the effect to bind the view projection matrix to
         */
        bindViewProjection(effect: Effect): void;
        /**
         * Specifies if material alpha testing should be turned on for the mesh
         * @param mesh defines the mesh to check
         */
        protected _shouldTurnAlphaTestOn(mesh: AbstractMesh): boolean;
        /**
         * Processes to execute after binding the material to a mesh
         * @param mesh defines the rendered mesh
         */
        protected _afterBind(mesh?: Mesh): void;
        /**
         * Unbinds the material from the mesh
         */
        unbind(): void;
        /**
         * Gets the active textures from the material
         * @returns an array of textures
         */
        getActiveTextures(): BaseTexture[];
        /**
         * Specifies if the material uses a texture
         * @param texture defines the texture to check against the material
         * @returns a boolean specifying if the material uses the texture
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Makes a duplicate of the material, and gives it a new name
         * @param name defines the new name for the duplicated material
         * @returns the cloned material
         */
        clone(name: string): Nullable<Material>;
        /**
         * Gets the meshes bound to the material
         * @returns an array of meshes bound to the material
         */
        getBindedMeshes(): AbstractMesh[];
        /**
         * Force shader compilation
         * @param mesh defines the mesh associated with this material
         * @param onCompiled defines a function to execute once the material is compiled
         * @param options defines the options to configure the compilation
         */
        forceCompilation(mesh: AbstractMesh, onCompiled?: (material: Material) => void, options?: Partial<{
            clipPlane: boolean;
        }>): void;
        /**
         * Force shader compilation
         * @param mesh defines the mesh that will use this material
         * @param options defines additional options for compiling the shaders
         * @returns a promise that resolves when the compilation completes
         */
        forceCompilationAsync(mesh: AbstractMesh, options?: Partial<{
            clipPlane: boolean;
        }>): Promise<void>;
        /**
         * Marks a define in the material to indicate that it needs to be re-computed
         * @param flag defines a flag used to determine which parts of the material have to be marked as dirty
         */
        markAsDirty(flag: number): void;
        /**
         * Marks all submeshes of a material to indicate that their material defines need to be re-calculated
         * @param func defines a function which checks material defines against the submeshes
         */
        protected _markAllSubMeshesAsDirty(func: (defines: MaterialDefines) => void): void;
        /**
         * Indicates that image processing needs to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsImageProcessingDirty(): void;
        /**
         * Indicates that textures need to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Indicates that fresnel needs to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsFresnelDirty(): void;
        /**
         * Indicates that fresnel and misc need to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsFresnelAndMiscDirty(): void;
        /**
         * Indicates that lights need to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsLightsDirty(): void;
        /**
         * Indicates that attributes need to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsAttributesDirty(): void;
        /**
         * Indicates that misc needs to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsMiscDirty(): void;
        /**
         * Indicates that textures and misc need to be re-calculated for all submeshes
         */
        protected _markAllSubMeshesAsTexturesAndMiscDirty(): void;
        /**
         * Disposes the material
         * @param forceDisposeEffect specifies if effects should be forcefully disposed
         * @param forceDisposeTextures specifies if textures should be forcefully disposed
         */
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        /**
         * Serializes this material
         * @returns the serialized material object
         */
        serialize(): any;
        /**
         * Creates a MultiMaterial from parsed MultiMaterial data.
         * @param parsedMultiMaterial defines parsed MultiMaterial data.
         * @param scene defines the hosting scene
         * @returns a new MultiMaterial
         */
        static ParseMultiMaterial(parsedMultiMaterial: any, scene: Scene): MultiMaterial;
        /**
         * Creates a material from parsed material data
         * @param parsedMaterial defines parsed material data
         * @param scene defines the hosting scene
         * @param rootUrl defines the root URL to use to load textures
         * @returns a new material
         */
        static Parse(parsedMaterial: any, scene: Scene, rootUrl: string): any;
    }
}

declare module BABYLON {
    /**
     * "Static Class" containing the most commonly used helper while dealing with material for
     * rendering purpose.
     *
     * It contains the basic tools to help defining defines, binding uniform for the common part of the materials.
     *
     * This works by convention in BabylonJS but is meant to be use only with shader following the in place naming rules and conventions.
     */
    class MaterialHelper {
        /**
         * Bind the current view position to an effect.
         * @param effect The effect to be bound
         * @param scene The scene the eyes position is used from
         */
        static BindEyePosition(effect: Effect, scene: Scene): void;
        /**
         * Helps preparing the defines values about the UVs in used in the effect.
         * UVs are shared as much as we can accross chanels in the shaders.
         * @param texture The texture we are preparing the UVs for
         * @param defines The defines to update
         * @param key The chanel key "diffuse", "specular"... used in the shader
         */
        static PrepareDefinesForMergedUV(texture: BaseTexture, defines: any, key: string): void;
        /**
         * Binds a texture matrix value to its corrsponding uniform
         * @param texture The texture to bind the matrix for
         * @param uniformBuffer The uniform buffer receivin the data
         * @param key The chanel key "diffuse", "specular"... used in the shader
         */
        static BindTextureMatrix(texture: BaseTexture, uniformBuffer: UniformBuffer, key: string): void;
        /**
         * Helper used to prepare the list of defines associated with misc. values for shader compilation
         * @param mesh defines the current mesh
         * @param scene defines the current scene
         * @param useLogarithmicDepth defines if logarithmic depth has to be turned on
         * @param pointsCloud defines if point cloud rendering has to be turned on
         * @param fogEnabled defines if fog has to be turned on
         * @param alphaTest defines if alpha testing has to be turned on
         * @param defines defines the current list of defines
         */
        static PrepareDefinesForMisc(mesh: AbstractMesh, scene: Scene, useLogarithmicDepth: boolean, pointsCloud: boolean, fogEnabled: boolean, alphaTest: boolean, defines: any): void;
        /**
         * Helper used to prepare the list of defines associated with frame values for shader compilation
         * @param scene defines the current scene
         * @param engine defines the current engine
         * @param defines specifies the list of active defines
         * @param useInstances defines if instances have to be turned on
         * @param useClipPlane defines if clip plane have to be turned on
         */
        static PrepareDefinesForFrameBoundValues(scene: Scene, engine: Engine, defines: any, useInstances: boolean, useClipPlane?: Nullable<boolean>): void;
        /**
         * Prepares the defines used in the shader depending on the attributes data available in the mesh
         * @param mesh The mesh containing the geometry data we will draw
         * @param defines The defines to update
         * @param useVertexColor Precise whether vertex colors should be used or not (override mesh info)
         * @param useBones Precise whether bones should be used or not (override mesh info)
         * @param useMorphTargets Precise whether morph targets should be used or not (override mesh info)
         * @param useVertexAlpha Precise whether vertex alpha should be used or not (override mesh info)
         * @returns false if defines are considered not dirty and have not been checked
         */
        static PrepareDefinesForAttributes(mesh: AbstractMesh, defines: any, useVertexColor: boolean, useBones: boolean, useMorphTargets?: boolean, useVertexAlpha?: boolean): boolean;
        /**
         * Prepares the defines related to the light information passed in parameter
         * @param scene The scene we are intending to draw
         * @param mesh The mesh the effect is compiling for
         * @param defines The defines to update
         * @param specularSupported Specifies whether specular is supported or not (override lights data)
         * @param maxSimultaneousLights Specfies how manuy lights can be added to the effect at max
         * @param disableLighting Specifies whether the lighting is disabled (override scene and light)
         * @returns true if normals will be required for the rest of the effect
         */
        static PrepareDefinesForLights(scene: Scene, mesh: AbstractMesh, defines: any, specularSupported: boolean, maxSimultaneousLights?: number, disableLighting?: boolean): boolean;
        /**
         * Prepares the uniforms and samplers list to be used in the effect. This can automatically remove from the list uniforms
         * that won t be acctive due to defines being turned off.
         * @param uniformsListOrOptions The uniform names to prepare or an EffectCreationOptions containing the liist and extra information
         * @param samplersList The samplers list
         * @param defines The defines helping in the list generation
         * @param maxSimultaneousLights The maximum number of simultanous light allowed in the effect
         */
        static PrepareUniformsAndSamplersList(uniformsListOrOptions: string[] | EffectCreationOptions, samplersList?: string[], defines?: any, maxSimultaneousLights?: number): void;
        /**
         * This helps decreasing rank by rank the shadow quality (0 being the highest rank and quality)
         * @param defines The defines to update while falling back
         * @param fallbacks The authorized effect fallbacks
         * @param maxSimultaneousLights The maximum number of lights allowed
         * @param rank the current rank of the Effect
         * @returns The newly affected rank
         */
        static HandleFallbacksForShadows(defines: any, fallbacks: EffectFallbacks, maxSimultaneousLights?: number, rank?: number): number;
        /**
         * Prepares the list of attributes required for morph targets according to the effect defines.
         * @param attribs The current list of supported attribs
         * @param mesh The mesh to prepare the morph targets attributes for
         * @param defines The current Defines of the effect
         */
        static PrepareAttributesForMorphTargets(attribs: string[], mesh: AbstractMesh, defines: any): void;
        /**
         * Prepares the list of attributes required for bones according to the effect defines.
         * @param attribs The current list of supported attribs
         * @param mesh The mesh to prepare the bones attributes for
         * @param defines The current Defines of the effect
         * @param fallbacks The current efffect fallback strategy
         */
        static PrepareAttributesForBones(attribs: string[], mesh: AbstractMesh, defines: any, fallbacks: EffectFallbacks): void;
        /**
         * Prepares the list of attributes required for instances according to the effect defines.
         * @param attribs The current list of supported attribs
         * @param defines The current Defines of the effect
         */
        static PrepareAttributesForInstances(attribs: string[], defines: any): void;
        /**
         * Binds the light shadow information to the effect for the given mesh.
         * @param light The light containing the generator
         * @param scene The scene the lights belongs to
         * @param mesh The mesh we are binding the information to render
         * @param lightIndex The light index in the effect used to render the mesh
         * @param effect The effect we are binding the data to
         */
        static BindLightShadow(light: Light, scene: Scene, mesh: AbstractMesh, lightIndex: string, effect: Effect): void;
        /**
         * Binds the light information to the effect.
         * @param light The light containing the generator
         * @param effect The effect we are binding the data to
         * @param lightIndex The light index in the effect used to render
         */
        static BindLightProperties(light: Light, effect: Effect, lightIndex: number): void;
        /**
         * Binds the lights information from the scene to the effect for the given mesh.
         * @param scene The scene the lights belongs to
         * @param mesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         * @param defines The generated defines for the effect
         * @param maxSimultaneousLights The maximum number of light that can be bound to the effect
         * @param usePhysicalLightFalloff Specifies whether the light falloff is defined physically or not
         */
        static BindLights(scene: Scene, mesh: AbstractMesh, effect: Effect, defines: any, maxSimultaneousLights?: number, usePhysicalLightFalloff?: boolean): void;
        /**
         * Binds the fog information from the scene to the effect for the given mesh.
         * @param scene The scene the lights belongs to
         * @param mesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         */
        static BindFogParameters(scene: Scene, mesh: AbstractMesh, effect: Effect): void;
        /**
         * Binds the bones information from the mesh to the effect.
         * @param mesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         */
        static BindBonesParameters(mesh?: AbstractMesh, effect?: Effect): void;
        /**
         * Binds the morph targets information from the mesh to the effect.
         * @param abstractMesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         */
        static BindMorphTargetParameters(abstractMesh: AbstractMesh, effect: Effect): void;
        /**
         * Binds the logarithmic depth information from the scene to the effect for the given defines.
         * @param defines The generated defines used in the effect
         * @param effect The effect we are binding the data to
         * @param scene The scene we are willing to render with logarithmic scale for
         */
        static BindLogDepth(defines: any, effect: Effect, scene: Scene): void;
        /**
         * Binds the clip plane information from the scene to the effect.
         * @param scene The scene the clip plane information are extracted from
         * @param effect The effect we are binding the data to
         */
        static BindClipPlane(effect: Effect, scene: Scene): void;
    }
}

declare module BABYLON {
    class MultiMaterial extends Material {
        private _subMaterials;
        subMaterials: Nullable<Material>[];
        constructor(name: string, scene: Scene);
        private _hookArray(array);
        getSubMaterial(index: number): Nullable<Material>;
        getActiveTextures(): BaseTexture[];
        getClassName(): string;
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: BaseSubMesh, useInstances?: boolean): boolean;
        clone(name: string, cloneChildren?: boolean): MultiMaterial;
        serialize(): any;
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
    }
}

declare module BABYLON {
    class PushMaterial extends Material {
        protected _activeEffect: Effect;
        protected _normalMatrix: Matrix;
        constructor(name: string, scene: Scene);
        getEffect(): Effect;
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        /**
        * Binds the given world matrix to the active effect
        *
        * @param world the matrix to bind
        */
        bindOnlyWorldMatrix(world: Matrix): void;
        /**
         * Binds the given normal matrix to the active effect
         *
         * @param normalMatrix the matrix to bind
         */
        bindOnlyNormalMatrix(normalMatrix: Matrix): void;
        bind(world: Matrix, mesh?: Mesh): void;
        protected _afterBind(mesh: Mesh, effect?: Nullable<Effect>): void;
        protected _mustRebind(scene: Scene, effect: Effect, visibility?: number): boolean;
    }
}

declare module BABYLON {
    class ShaderMaterial extends Material {
        private _shaderPath;
        private _options;
        private _textures;
        private _textureArrays;
        private _floats;
        private _ints;
        private _floatsArrays;
        private _colors3;
        private _colors3Arrays;
        private _colors4;
        private _vectors2;
        private _vectors3;
        private _vectors4;
        private _matrices;
        private _matrices3x3;
        private _matrices2x2;
        private _vectors2Arrays;
        private _vectors3Arrays;
        private _cachedWorldViewMatrix;
        private _renderId;
        constructor(name: string, scene: Scene, shaderPath: any, options: any);
        getClassName(): string;
        needAlphaBlending(): boolean;
        needAlphaTesting(): boolean;
        private _checkUniform(uniformName);
        setTexture(name: string, texture: Texture): ShaderMaterial;
        setTextureArray(name: string, textures: Texture[]): ShaderMaterial;
        setFloat(name: string, value: number): ShaderMaterial;
        setInt(name: string, value: number): ShaderMaterial;
        setFloats(name: string, value: number[]): ShaderMaterial;
        setColor3(name: string, value: Color3): ShaderMaterial;
        setColor3Array(name: string, value: Color3[]): ShaderMaterial;
        setColor4(name: string, value: Color4): ShaderMaterial;
        setVector2(name: string, value: Vector2): ShaderMaterial;
        setVector3(name: string, value: Vector3): ShaderMaterial;
        setVector4(name: string, value: Vector4): ShaderMaterial;
        setMatrix(name: string, value: Matrix): ShaderMaterial;
        setMatrix3x3(name: string, value: Float32Array): ShaderMaterial;
        setMatrix2x2(name: string, value: Float32Array): ShaderMaterial;
        setArray2(name: string, value: number[]): ShaderMaterial;
        setArray3(name: string, value: number[]): ShaderMaterial;
        private _checkCache(scene, mesh?, useInstances?);
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        bindOnlyWorldMatrix(world: Matrix): void;
        bind(world: Matrix, mesh?: Mesh): void;
        getActiveTextures(): BaseTexture[];
        hasTexture(texture: BaseTexture): boolean;
        clone(name: string): ShaderMaterial;
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        serialize(): any;
        static Parse(source: any, scene: Scene, rootUrl: string): ShaderMaterial;
    }
}

declare module BABYLON {
    /** @hidden */
    class StandardMaterialDefines extends MaterialDefines implements IImageProcessingConfigurationDefines {
        MAINUV1: boolean;
        MAINUV2: boolean;
        DIFFUSE: boolean;
        DIFFUSEDIRECTUV: number;
        AMBIENT: boolean;
        AMBIENTDIRECTUV: number;
        OPACITY: boolean;
        OPACITYDIRECTUV: number;
        OPACITYRGB: boolean;
        REFLECTION: boolean;
        EMISSIVE: boolean;
        EMISSIVEDIRECTUV: number;
        SPECULAR: boolean;
        SPECULARDIRECTUV: number;
        BUMP: boolean;
        BUMPDIRECTUV: number;
        PARALLAX: boolean;
        PARALLAXOCCLUSION: boolean;
        SPECULAROVERALPHA: boolean;
        CLIPPLANE: boolean;
        ALPHATEST: boolean;
        DEPTHPREPASS: boolean;
        ALPHAFROMDIFFUSE: boolean;
        POINTSIZE: boolean;
        FOG: boolean;
        SPECULARTERM: boolean;
        DIFFUSEFRESNEL: boolean;
        OPACITYFRESNEL: boolean;
        REFLECTIONFRESNEL: boolean;
        REFRACTIONFRESNEL: boolean;
        EMISSIVEFRESNEL: boolean;
        FRESNEL: boolean;
        NORMAL: boolean;
        UV1: boolean;
        UV2: boolean;
        VERTEXCOLOR: boolean;
        VERTEXALPHA: boolean;
        NUM_BONE_INFLUENCERS: number;
        BonesPerMesh: number;
        INSTANCES: boolean;
        GLOSSINESS: boolean;
        ROUGHNESS: boolean;
        EMISSIVEASILLUMINATION: boolean;
        LINKEMISSIVEWITHDIFFUSE: boolean;
        REFLECTIONFRESNELFROMSPECULAR: boolean;
        LIGHTMAP: boolean;
        LIGHTMAPDIRECTUV: number;
        OBJECTSPACE_NORMALMAP: boolean;
        USELIGHTMAPASSHADOWMAP: boolean;
        REFLECTIONMAP_3D: boolean;
        REFLECTIONMAP_SPHERICAL: boolean;
        REFLECTIONMAP_PLANAR: boolean;
        REFLECTIONMAP_CUBIC: boolean;
        USE_LOCAL_REFLECTIONMAP_CUBIC: boolean;
        REFLECTIONMAP_PROJECTION: boolean;
        REFLECTIONMAP_SKYBOX: boolean;
        REFLECTIONMAP_EXPLICIT: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR_FIXED: boolean;
        REFLECTIONMAP_MIRROREDEQUIRECTANGULAR_FIXED: boolean;
        INVERTCUBICMAP: boolean;
        LOGARITHMICDEPTH: boolean;
        REFRACTION: boolean;
        REFRACTIONMAP_3D: boolean;
        REFLECTIONOVERALPHA: boolean;
        TWOSIDEDLIGHTING: boolean;
        SHADOWFLOAT: boolean;
        MORPHTARGETS: boolean;
        MORPHTARGETS_NORMAL: boolean;
        MORPHTARGETS_TANGENT: boolean;
        NUM_MORPH_INFLUENCERS: number;
        NONUNIFORMSCALING: boolean;
        PREMULTIPLYALPHA: boolean;
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        CONTRAST: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
        /**
         * If the reflection texture on this material is in linear color space
         * @hidden
         */
        IS_REFLECTION_LINEAR: boolean;
        /**
         * If the refraction texture on this material is in linear color space
         * @hidden
         */
        IS_REFRACTION_LINEAR: boolean;
        EXPOSURE: boolean;
        constructor();
        setReflectionMode(modeToEnable: string): void;
    }
    class StandardMaterial extends PushMaterial {
        private _diffuseTexture;
        diffuseTexture: Nullable<BaseTexture>;
        private _ambientTexture;
        ambientTexture: Nullable<BaseTexture>;
        private _opacityTexture;
        opacityTexture: Nullable<BaseTexture>;
        private _reflectionTexture;
        reflectionTexture: Nullable<BaseTexture>;
        private _emissiveTexture;
        emissiveTexture: Nullable<BaseTexture>;
        private _specularTexture;
        specularTexture: Nullable<BaseTexture>;
        private _bumpTexture;
        bumpTexture: Nullable<BaseTexture>;
        private _lightmapTexture;
        lightmapTexture: Nullable<BaseTexture>;
        private _refractionTexture;
        refractionTexture: Nullable<BaseTexture>;
        ambientColor: Color3;
        diffuseColor: Color3;
        specularColor: Color3;
        emissiveColor: Color3;
        specularPower: number;
        private _useAlphaFromDiffuseTexture;
        useAlphaFromDiffuseTexture: boolean;
        private _useEmissiveAsIllumination;
        useEmissiveAsIllumination: boolean;
        private _linkEmissiveWithDiffuse;
        linkEmissiveWithDiffuse: boolean;
        private _useSpecularOverAlpha;
        useSpecularOverAlpha: boolean;
        private _useReflectionOverAlpha;
        useReflectionOverAlpha: boolean;
        private _disableLighting;
        disableLighting: boolean;
        private _useObjectSpaceNormalMap;
        /**
         * Allows using an object space normal map (instead of tangent space).
         */
        useObjectSpaceNormalMap: boolean;
        private _useParallax;
        useParallax: boolean;
        private _useParallaxOcclusion;
        useParallaxOcclusion: boolean;
        parallaxScaleBias: number;
        private _roughness;
        roughness: number;
        indexOfRefraction: number;
        invertRefractionY: boolean;
        /**
         * Defines the alpha limits in alpha test mode
         */
        alphaCutOff: number;
        private _useLightmapAsShadowmap;
        useLightmapAsShadowmap: boolean;
        private _diffuseFresnelParameters;
        diffuseFresnelParameters: FresnelParameters;
        private _opacityFresnelParameters;
        opacityFresnelParameters: FresnelParameters;
        private _reflectionFresnelParameters;
        reflectionFresnelParameters: FresnelParameters;
        private _refractionFresnelParameters;
        refractionFresnelParameters: FresnelParameters;
        private _emissiveFresnelParameters;
        emissiveFresnelParameters: FresnelParameters;
        private _useReflectionFresnelFromSpecular;
        useReflectionFresnelFromSpecular: boolean;
        private _useGlossinessFromSpecularMapAlpha;
        useGlossinessFromSpecularMapAlpha: boolean;
        private _maxSimultaneousLights;
        maxSimultaneousLights: number;
        /**
         * If sets to true, x component of normal map value will invert (x = 1.0 - x).
         */
        private _invertNormalMapX;
        invertNormalMapX: boolean;
        /**
         * If sets to true, y component of normal map value will invert (y = 1.0 - y).
         */
        private _invertNormalMapY;
        invertNormalMapY: boolean;
        /**
         * If sets to true and backfaceCulling is false, normals will be flipped on the backside.
         */
        private _twoSidedLighting;
        twoSidedLighting: boolean;
        /**
         * Default configuration related to image processing available in the standard Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Gets the image processing configuration used either in this material.
         */
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the Standard Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>): void;
        /**
         * Gets wether the color curves effect is enabled.
         */
        /**
         * Sets wether the color curves effect is enabled.
         */
        cameraColorCurvesEnabled: boolean;
        /**
         * Gets wether the color grading effect is enabled.
         */
        /**
         * Gets wether the color grading effect is enabled.
         */
        cameraColorGradingEnabled: boolean;
        /**
         * Gets wether tonemapping is enabled or not.
         */
        /**
         * Sets wether tonemapping is enabled or not
         */
        cameraToneMappingEnabled: boolean;
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        cameraExposure: number;
        /**
         * Gets The camera contrast used on this material.
         */
        /**
         * Sets The camera contrast used on this material.
         */
        cameraContrast: number;
        /**
         * Gets the Color Grading 2D Lookup Texture.
         */
        /**
         * Sets the Color Grading 2D Lookup Texture.
         */
        cameraColorGradingTexture: Nullable<BaseTexture>;
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        cameraColorCurves: Nullable<ColorCurves>;
        customShaderNameResolve: (shaderName: string, uniforms: string[], uniformBuffers: string[], samplers: string[], defines: StandardMaterialDefines) => string;
        protected _renderTargets: SmartArray<RenderTargetTexture>;
        protected _worldViewProjectionMatrix: Matrix;
        protected _globalAmbientColor: Color3;
        protected _useLogarithmicDepth: boolean;
        constructor(name: string, scene: Scene);
        getClassName(): string;
        useLogarithmicDepth: boolean;
        needAlphaBlending(): boolean;
        needAlphaTesting(): boolean;
        protected _shouldUseAlphaFromDiffuseTexture(): boolean;
        getAlphaTestTexture(): Nullable<BaseTexture>;
        /**
         * Child classes can use it to update shaders
         */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: SubMesh, useInstances?: boolean): boolean;
        buildUniformLayout(): void;
        unbind(): void;
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        getAnimatables(): IAnimatable[];
        getActiveTextures(): BaseTexture[];
        hasTexture(texture: BaseTexture): boolean;
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        clone(name: string): StandardMaterial;
        serialize(): any;
        static Parse(source: any, scene: Scene, rootUrl: string): StandardMaterial;
        static _DiffuseTextureEnabled: boolean;
        static DiffuseTextureEnabled: boolean;
        static _AmbientTextureEnabled: boolean;
        static AmbientTextureEnabled: boolean;
        static _OpacityTextureEnabled: boolean;
        static OpacityTextureEnabled: boolean;
        static _ReflectionTextureEnabled: boolean;
        static ReflectionTextureEnabled: boolean;
        static _EmissiveTextureEnabled: boolean;
        static EmissiveTextureEnabled: boolean;
        static _SpecularTextureEnabled: boolean;
        static SpecularTextureEnabled: boolean;
        static _BumpTextureEnabled: boolean;
        static BumpTextureEnabled: boolean;
        static _LightmapTextureEnabled: boolean;
        static LightmapTextureEnabled: boolean;
        static _RefractionTextureEnabled: boolean;
        static RefractionTextureEnabled: boolean;
        static _ColorGradingTextureEnabled: boolean;
        static ColorGradingTextureEnabled: boolean;
        static _FresnelEnabled: boolean;
        static FresnelEnabled: boolean;
    }
}

declare module BABYLON {
    class UniformBuffer {
        private _engine;
        private _buffer;
        private _data;
        private _bufferData;
        private _dynamic?;
        private _uniformLocations;
        private _uniformSizes;
        private _uniformLocationPointer;
        private _needSync;
        private _noUBO;
        private _currentEffect;
        private static _MAX_UNIFORM_SIZE;
        private static _tempBuffer;
        /**
         * Wrapper for updateUniform.
         * @method updateMatrix3x3
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Float32Array} matrix
         */
        updateMatrix3x3: (name: string, matrix: Float32Array) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Float32Array} matrix
         */
        updateMatrix2x2: (name: string, matrix: Float32Array) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         */
        updateFloat: (name: string, x: number) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateFloat2: (name: string, x: number, y: number, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {number} z
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateFloat3: (name: string, x: number, y: number, z: number, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {number} z
         * @param {number} w
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateFloat4: (name: string, x: number, y: number, z: number, w: number, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Matrix} A 4x4 matrix.
         */
        updateMatrix: (name: string, mat: Matrix) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Vector3} vector
         */
        updateVector3: (name: string, vector: Vector3) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Vector4} vector
         */
        updateVector4: (name: string, vector: Vector4) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateColor3: (name: string, color: Color3, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         * @param {number} alpha
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateColor4: (name: string, color: Color3, alpha: number, suffix?: string) => void;
        /**
         * Uniform buffer objects.
         *
         * Handles blocks of uniform on the GPU.
         *
         * If WebGL 2 is not available, this class falls back on traditionnal setUniformXXX calls.
         *
         * For more information, please refer to :
         * https://www.khronos.org/opengl/wiki/Uniform_Buffer_Object
         */
        constructor(engine: Engine, data?: number[], dynamic?: boolean);
        /**
         * Indicates if the buffer is using the WebGL2 UBO implementation,
         * or just falling back on setUniformXXX calls.
         */
        readonly useUbo: boolean;
        /**
         * Indicates if the WebGL underlying uniform buffer is in sync
         * with the javascript cache data.
         */
        readonly isSync: boolean;
        /**
         * Indicates if the WebGL underlying uniform buffer is dynamic.
         * Also, a dynamic UniformBuffer will disable cache verification and always
         * update the underlying WebGL uniform buffer to the GPU.
         */
        isDynamic(): boolean;
        /**
         * The data cache on JS side.
         */
        getData(): Float32Array;
        /**
         * The underlying WebGL Uniform buffer.
         */
        getBuffer(): Nullable<WebGLBuffer>;
        /**
         * std140 layout specifies how to align data within an UBO structure.
         * See https://khronos.org/registry/OpenGL/specs/gl/glspec45.core.pdf#page=159
         * for specs.
         */
        private _fillAlignment(size);
        /**
         * Adds an uniform in the buffer.
         * Warning : the subsequents calls of this function must be in the same order as declared in the shader
         * for the layout to be correct !
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number|number[]} size Data size, or data directly.
         */
        addUniform(name: string, size: number | number[]): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Matrix} mat A 4x4 matrix.
         */
        addMatrix(name: string, mat: Matrix): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         */
        addFloat2(name: string, x: number, y: number): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {number} z
         */
        addFloat3(name: string, x: number, y: number, z: number): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         */
        addColor3(name: string, color: Color3): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         * @param {number} alpha
         */
        addColor4(name: string, color: Color3, alpha: number): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Vector3} vector
         */
        addVector3(name: string, vector: Vector3): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         */
        addMatrix3x3(name: string): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         */
        addMatrix2x2(name: string): void;
        /**
         * Effectively creates the WebGL Uniform Buffer, once layout is completed with `addUniform`.
         */
        create(): void;
        _rebuild(): void;
        /**
         * Updates the WebGL Uniform Buffer on the GPU.
         * If the `dynamic` flag is set to true, no cache comparison is done.
         * Otherwise, the buffer will be updated only if the cache differs.
         */
        update(): void;
        /**
         * Updates the value of an uniform. The `update` method must be called afterwards to make it effective in the GPU.
         * @param {string} uniformName Name of the uniform, as used in the uniform block in the shader.
         * @param {number[]|Float32Array} data Flattened data
         * @param {number} size Size of the data.
         */
        updateUniform(uniformName: string, data: FloatArray, size: number): void;
        private _updateMatrix3x3ForUniform(name, matrix);
        private _updateMatrix3x3ForEffect(name, matrix);
        private _updateMatrix2x2ForEffect(name, matrix);
        private _updateMatrix2x2ForUniform(name, matrix);
        private _updateFloatForEffect(name, x);
        private _updateFloatForUniform(name, x);
        private _updateFloat2ForEffect(name, x, y, suffix?);
        private _updateFloat2ForUniform(name, x, y, suffix?);
        private _updateFloat3ForEffect(name, x, y, z, suffix?);
        private _updateFloat3ForUniform(name, x, y, z, suffix?);
        private _updateFloat4ForEffect(name, x, y, z, w, suffix?);
        private _updateFloat4ForUniform(name, x, y, z, w, suffix?);
        private _updateMatrixForEffect(name, mat);
        private _updateMatrixForUniform(name, mat);
        private _updateVector3ForEffect(name, vector);
        private _updateVector3ForUniform(name, vector);
        private _updateVector4ForEffect(name, vector);
        private _updateVector4ForUniform(name, vector);
        private _updateColor3ForEffect(name, color, suffix?);
        private _updateColor3ForUniform(name, color, suffix?);
        private _updateColor4ForEffect(name, color, alpha, suffix?);
        private _updateColor4ForUniform(name, color, alpha, suffix?);
        /**
         * Sets a sampler uniform on the effect.
         * @param {string} name Name of the sampler.
         * @param {Texture} texture
         */
        setTexture(name: string, texture: Nullable<BaseTexture>): void;
        /**
         * Directly updates the value of the uniform in the cache AND on the GPU.
         * @param {string} uniformName Name of the uniform, as used in the uniform block in the shader.
         * @param {number[]|Float32Array} data Flattened data
         */
        updateUniformDirectly(uniformName: string, data: FloatArray): void;
        /**
         * Binds this uniform buffer to an effect.
         * @param {Effect} effect
         * @param {string} name Name of the uniform block in the shader.
         */
        bindToEffect(effect: Effect, name: string): void;
        /**
         * Disposes the uniform buffer.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    class Scalar {
        /**
         * Two pi constants convenient for computation.
         */
        static TwoPi: number;
        /**
         * Boolean : true if the absolute difference between a and b is lower than epsilon (default = 1.401298E-45)
         */
        static WithinEpsilon(a: number, b: number, epsilon?: number): boolean;
        /**
         * Returns a string : the upper case translation of the number i to hexadecimal.
         */
        static ToHex(i: number): string;
        /**
         * Returns -1 if value is negative and +1 is value is positive.
         * Returns the value itself if it's equal to zero.
         */
        static Sign(value: number): number;
        /**
         * Returns the value itself if it's between min and max.
         * Returns min if the value is lower than min.
         * Returns max if the value is greater than max.
         */
        static Clamp(value: number, min?: number, max?: number): number;
        /**
         * Returns the log2 of value.
         */
        static Log2(value: number): number;
        /**
        * Loops the value, so that it is never larger than length and never smaller than 0.
        *
        * This is similar to the modulo operator but it works with floating point numbers.
        * For example, using 3.0 for t and 2.5 for length, the result would be 0.5.
        * With t = 5 and length = 2.5, the result would be 0.0.
        * Note, however, that the behaviour is not defined for negative numbers as it is for the modulo operator
        */
        static Repeat(value: number, length: number): number;
        /**
        * Normalize the value between 0.0 and 1.0 using min and max values
        */
        static Normalize(value: number, min: number, max: number): number;
        /**
        * Denormalize the value from 0.0 and 1.0 using min and max values
        */
        static Denormalize(normalized: number, min: number, max: number): number;
        /**
        * Calculates the shortest difference between two given angles given in degrees.
        */
        static DeltaAngle(current: number, target: number): number;
        /**
        * PingPongs the value t, so that it is never larger than length and never smaller than 0.
        *
        * The returned value will move back and forth between 0 and length
        */
        static PingPong(tx: number, length: number): number;
        /**
        * Interpolates between min and max with smoothing at the limits.
        *
        * This function interpolates between min and max in a similar way to Lerp. However, the interpolation will gradually speed up
        * from the start and slow down toward the end. This is useful for creating natural-looking animation, fading and other transitions.
        */
        static SmoothStep(from: number, to: number, tx: number): number;
        /**
        * Moves a value current towards target.
        *
        * This is essentially the same as Mathf.Lerp but instead the function will ensure that the speed never exceeds maxDelta.
        * Negative values of maxDelta pushes the value away from target.
        */
        static MoveTowards(current: number, target: number, maxDelta: number): number;
        /**
        * Same as MoveTowards but makes sure the values interpolate correctly when they wrap around 360 degrees.
        *
        * Variables current and target are assumed to be in degrees. For optimization reasons, negative values of maxDelta
        *  are not supported and may cause oscillation. To push current away from a target angle, add 180 to that angle instead.
        */
        static MoveTowardsAngle(current: number, target: number, maxDelta: number): number;
        /**
            * Creates a new scalar with values linearly interpolated of "amount" between the start scalar and the end scalar.
            */
        static Lerp(start: number, end: number, amount: number): number;
        /**
        * Same as Lerp but makes sure the values interpolate correctly when they wrap around 360 degrees.
        * The parameter t is clamped to the range [0, 1]. Variables a and b are assumed to be in degrees.
        */
        static LerpAngle(start: number, end: number, amount: number): number;
        /**
        * Calculates the linear parameter t that produces the interpolant value within the range [a, b].
        */
        static InverseLerp(a: number, b: number, value: number): number;
        /**
         * Returns a new scalar located for "amount" (float) on the Hermite spline defined by the scalars "value1", "value3", "tangent1", "tangent2".
         */
        static Hermite(value1: number, tangent1: number, value2: number, tangent2: number, amount: number): number;
        /**
        * Returns a random float number between and min and max values
        */
        static RandomRange(min: number, max: number): number;
        /**
        * This function returns percentage of a number in a given range.
        *
        * RangeToPercent(40,20,60) will return 0.5 (50%)
        * RangeToPercent(34,0,100) will return 0.34 (34%)
        */
        static RangeToPercent(number: number, min: number, max: number): number;
        /**
        * This function returns number that corresponds to the percentage in a given range.
        *
        * PercentToRange(0.34,0,100) will return 34.
        */
        static PercentToRange(percent: number, min: number, max: number): number;
        /**
         * Returns the angle converted to equivalent value between -Math.PI and Math.PI radians.
         * @param angle The angle to normalize in radian.
         * @return The converted angle.
         */
        static NormalizeRadians(angle: number): number;
    }
}

declare module BABYLON {
    const ToGammaSpace: number;
    const ToLinearSpace = 2.2;
    const Epsilon = 0.001;
    /**
     * Class used to hold a RBG color
     */
    class Color3 {
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r: number;
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g: number;
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b: number;
        /**
         * Creates a new Color3 object from red, green, blue values, all between 0 and 1
         * @param r defines the red component (between 0 and 1, default is 0)
         * @param g defines the green component (between 0 and 1, default is 0)
         * @param b defines the blue component (between 0 and 1, default is 0)
         */
        constructor(
            /**
             * Defines the red component (between 0 and 1, default is 0)
             */
            r?: number, 
            /**
             * Defines the green component (between 0 and 1, default is 0)
             */
            g?: number, 
            /**
             * Defines the blue component (between 0 and 1, default is 0)
             */
            b?: number);
        /**
         * Creates a string with the Color3 current values
         * @returns the string representation of the Color3 object
         */
        toString(): string;
        /**
         * Returns the string "Color3"
         * @returns "Color3"
         */
        getClassName(): string;
        /**
         * Compute the Color3 hash code
         * @returns an unique number that can be used to hash Color3 objects
         */
        getHashCode(): number;
        /**
         * Stores in the given array from the given starting index the red, green, blue values as successive elements
         * @param array defines the array where to store the r,g,b components
         * @param index defines an optional index in the target array to define where to start storing values
         * @returns the current Color3 object
         */
        toArray(array: FloatArray, index?: number): Color3;
        /**
         * Returns a new {BABYLON.Color4} object from the current Color3 and the given alpha
         * @param alpha defines the alpha component on the new {BABYLON.Color4} object (default is 1)
         * @returns a new {BABYLON.Color4} object
         */
        toColor4(alpha?: number): Color4;
        /**
         * Returns a new array populated with 3 numeric elements : red, green and blue values
         * @returns the new array
         */
        asArray(): number[];
        /**
         * Returns the luminance value
         * @returns a float value
         */
        toLuminance(): number;
        /**
         * Multiply each Color3 rgb values by the given Color3 rgb values in a new Color3 object
         * @param otherColor defines the second operand
         * @returns the new Color3 object
         */
        multiply(otherColor: Color3): Color3;
        /**
         * Multiply the rgb values of the Color3 and the given Color3 and stores the result in the object "result"
         * @param otherColor defines the second operand
         * @param result defines the Color3 object where to store the result
         * @returns the current Color3
         */
        multiplyToRef(otherColor: Color3, result: Color3): Color3;
        /**
         * Determines equality between Color3 objects
         * @param otherColor defines the second operand
         * @returns true if the rgb values are equal to the given ones
         */
        equals(otherColor: Color3): boolean;
        /**
         * Determines equality between the current Color3 object and a set of r,b,g values
         * @param r defines the red component to check
         * @param g defines the green component to check
         * @param b defines the blue component to check
         * @returns true if the rgb values are equal to the given ones
         */
        equalsFloats(r: number, g: number, b: number): boolean;
        /**
         * Multiplies in place each rgb value by scale
         * @param scale defines the scaling factor
         * @returns the updated Color3
         */
        scale(scale: number): Color3;
        /**
         * Multiplies the rgb values by scale and stores the result into "result"
         * @param scale defines the scaling factor
         * @param result defines the Color3 object where to store the result
         * @returns the unmodified current Color3
         */
        scaleToRef(scale: number, result: Color3): Color3;
        /**
         * Scale the current Color3 values by a factor and add the result to a given Color3
         * @param scale defines the scale factor
         * @param result defines color to store the result into
         * @returns the unmodified current Color3
         */
        scaleAndAddToRef(scale: number, result: Color3): Color3;
        /**
         * Clamps the rgb values by the min and max values and stores the result into "result"
         * @param min defines minimum clamping value (default is 0)
         * @param max defines maximum clamping value (default is 1)
         * @param result defines color to store the result into
         * @returns the original Color3
         */
        clampToRef(min: number | undefined, max: number | undefined, result: Color3): Color3;
        /**
         * Creates a new Color3 set with the added values of the current Color3 and of the given one
         * @param otherColor defines the second operand
         * @returns the new Color3
         */
        add(otherColor: Color3): Color3;
        /**
         * Stores the result of the addition of the current Color3 and given one rgb values into "result"
         * @param otherColor defines the second operand
         * @param result defines Color3 object to store the result into
         * @returns the unmodified current Color3
         */
        addToRef(otherColor: Color3, result: Color3): Color3;
        /**
         * Returns a new Color3 set with the subtracted values of the given one from the current Color3
         * @param otherColor defines the second operand
         * @returns the new Color3
         */
        subtract(otherColor: Color3): Color3;
        /**
         * Stores the result of the subtraction of given one from the current Color3 rgb values into "result"
         * @param otherColor defines the second operand
         * @param result defines Color3 object to store the result into
         * @returns the unmodified current Color3
         */
        subtractToRef(otherColor: Color3, result: Color3): Color3;
        /**
         * Copy the current object
         * @returns a new Color3 copied the current one
         */
        clone(): Color3;
        /**
         * Copies the rgb values from the source in the current Color3
         * @param source defines the source Color3 object
         * @returns the updated Color3 object
         */
        copyFrom(source: Color3): Color3;
        /**
         * Updates the Color3 rgb values from the given floats
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @returns the current Color3 object
         */
        copyFromFloats(r: number, g: number, b: number): Color3;
        /**
         * Updates the Color3 rgb values from the given floats
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @returns the current Color3 object
         */
        set(r: number, g: number, b: number): Color3;
        /**
         * Compute the Color3 hexadecimal code as a string
         * @returns a string containing the hexadecimal representation of the Color3 object
         */
        toHexString(): string;
        /**
         * Computes a new Color3 converted from the current one to linear space
         * @returns a new Color3 object
         */
        toLinearSpace(): Color3;
        /**
         * Converts the Color3 values to linear space and stores the result in "convertedColor"
         * @param convertedColor defines the Color3 object where to store the linear space version
         * @returns the unmodified Color3
         */
        toLinearSpaceToRef(convertedColor: Color3): Color3;
        /**
         * Computes a new Color3 converted from the current one to gamma space
         * @returns a new Color3 object
         */
        toGammaSpace(): Color3;
        /**
         * Converts the Color3 values to gamma space and stores the result in "convertedColor"
         * @param convertedColor defines the Color3 object where to store the gamma space version
         * @returns the unmodified Color3
         */
        toGammaSpaceToRef(convertedColor: Color3): Color3;
        /**
         * Creates a new Color3 from the string containing valid hexadecimal values
         * @param hex defines a string containing valid hexadecimal values
         * @returns a new Color3 object
         */
        static FromHexString(hex: string): Color3;
        /**
         * Creates a new Vector3 from the starting index of the given array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns a new Color3 object
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Color3;
        /**
         * Creates a new Color3 from integer values (< 256)
         * @param r defines the red component to read from (value between 0 and 255)
         * @param g defines the green component to read from (value between 0 and 255)
         * @param b defines the blue component to read from (value between 0 and 255)
         * @returns a new Color3 object
         */
        static FromInts(r: number, g: number, b: number): Color3;
        /**
         * Creates a new Color3 with values linearly interpolated of "amount" between the start Color3 and the end Color3
         * @param start defines the start Color3 value
         * @param end defines the end Color3 value
         * @param amount defines the gradient value between start and end
         * @returns a new Color3 object
         */
        static Lerp(start: Color3, end: Color3, amount: number): Color3;
        /**
         * Returns a Color3 value containing a red color
         * @returns a new Color3 object
         */
        static Red(): Color3;
        /**
         * Returns a Color3 value containing a green color
         * @returns a new Color3 object
         */
        static Green(): Color3;
        /**
         * Returns a Color3 value containing a blue color
         * @returns a new Color3 object
         */
        static Blue(): Color3;
        /**
         * Returns a Color3 value containing a black color
         * @returns a new Color3 object
         */
        static Black(): Color3;
        /**
         * Returns a Color3 value containing a white color
         * @returns a new Color3 object
         */
        static White(): Color3;
        /**
         * Returns a Color3 value containing a purple color
         * @returns a new Color3 object
         */
        static Purple(): Color3;
        /**
         * Returns a Color3 value containing a magenta color
         * @returns a new Color3 object
         */
        static Magenta(): Color3;
        /**
         * Returns a Color3 value containing a yellow color
         * @returns a new Color3 object
         */
        static Yellow(): Color3;
        /**
         * Returns a Color3 value containing a gray color
         * @returns a new Color3 object
         */
        static Gray(): Color3;
        /**
         * Returns a Color3 value containing a teal color
         * @returns a new Color3 object
         */
        static Teal(): Color3;
        /**
         * Returns a Color3 value containing a random color
         * @returns a new Color3 object
         */
        static Random(): Color3;
    }
    /**
     * Class used to hold a RBGA color
     */
    class Color4 {
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r: number;
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g: number;
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b: number;
        /**
         * Defines the alpha component (between 0 and 1, default is 1)
         */
        a: number;
        /**
         * Creates a new Color4 object from red, green, blue values, all between 0 and 1
         * @param r defines the red component (between 0 and 1, default is 0)
         * @param g defines the green component (between 0 and 1, default is 0)
         * @param b defines the blue component (between 0 and 1, default is 0)
         * @param a defines the alpha component (between 0 and 1, default is 1)
         */
        constructor(
            /**
             * Defines the red component (between 0 and 1, default is 0)
             */
            r?: number, 
            /**
             * Defines the green component (between 0 and 1, default is 0)
             */
            g?: number, 
            /**
             * Defines the blue component (between 0 and 1, default is 0)
             */
            b?: number, 
            /**
             * Defines the alpha component (between 0 and 1, default is 1)
             */
            a?: number);
        /**
         * Adds in place the given Color4 values to the current Color4 object
         * @param right defines the second operand
         * @returns the current updated Color4 object
         */
        addInPlace(right: Color4): Color4;
        /**
         * Creates a new array populated with 4 numeric elements : red, green, blue, alpha values
         * @returns the new array
         */
        asArray(): number[];
        /**
         * Stores from the starting index in the given array the Color4 successive values
         * @param array defines the array where to store the r,g,b components
         * @param index defines an optional index in the target array to define where to start storing values
         * @returns the current Color4 object
         */
        toArray(array: number[], index?: number): Color4;
        /**
         * Creates a new Color4 set with the added values of the current Color4 and of the given one
         * @param right defines the second operand
         * @returns a new Color4 object
         */
        add(right: Color4): Color4;
        /**
         * Creates a new Color4 set with the subtracted values of the given one from the current Color4
         * @param right defines the second operand
         * @returns a new Color4 object
         */
        subtract(right: Color4): Color4;
        /**
         * Subtracts the given ones from the current Color4 values and stores the results in "result"
         * @param right defines the second operand
         * @param result defines the Color4 object where to store the result
         * @returns the current Color4 object
         */
        subtractToRef(right: Color4, result: Color4): Color4;
        /**
         * Creates a new Color4 with the current Color4 values multiplied by scale
         * @param scale defines the scaling factor to apply
         * @returns a new Color4 object
         */
        scale(scale: number): Color4;
        /**
         * Multiplies the current Color4 values by scale and stores the result in "result"
         * @param scale defines the scaling factor to apply
         * @param result defines the Color4 object where to store the result
         * @returns the current unmodified Color4
         */
        scaleToRef(scale: number, result: Color4): Color4;
        /**
         * Scale the current Color4 values by a factor and add the result to a given Color4
         * @param scale defines the scale factor
         * @param result defines the Color4 object where to store the result
         * @returns the unmodified current Color4
         */
        scaleAndAddToRef(scale: number, result: Color4): Color4;
        /**
         * Clamps the rgb values by the min and max values and stores the result into "result"
         * @param min defines minimum clamping value (default is 0)
         * @param max defines maximum clamping value (default is 1)
         * @param result defines color to store the result into.
         * @returns the cuurent Color4
         */
        clampToRef(min: number | undefined, max: number | undefined, result: Color4): Color4;
        /**
          * Multipy an Color4 value by another and return a new Color4 object
          * @param color defines the Color4 value to multiply by
          * @returns a new Color4 object
          */
        multiply(color: Color4): Color4;
        /**
         * Multipy a Color4 value by another and push the result in a reference value
         * @param color defines the Color4 value to multiply by
         * @param result defines the Color4 to fill the result in
         * @returns the result Color4
         */
        multiplyToRef(color: Color4, result: Color4): Color4;
        /**
         * Creates a string with the Color4 current values
         * @returns the string representation of the Color4 object
         */
        toString(): string;
        /**
         * Returns the string "Color4"
         * @returns "Color4"
         */
        getClassName(): string;
        /**
         * Compute the Color4 hash code
         * @returns an unique number that can be used to hash Color4 objects
         */
        getHashCode(): number;
        /**
         * Creates a new Color4 copied from the current one
         * @returns a new Color4 object
         */
        clone(): Color4;
        /**
         * Copies the given Color4 values into the current one
         * @param source defines the source Color4 object
         * @returns the current updated Color4 object
         */
        copyFrom(source: Color4): Color4;
        /**
         * Copies the given float values into the current one
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @param a defines the alpha component to read from
         * @returns the current updated Color4 object
         */
        copyFromFloats(r: number, g: number, b: number, a: number): Color4;
        /**
         * Copies the given float values into the current one
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @param a defines the alpha component to read from
         * @returns the current updated Color4 object
         */
        set(r: number, g: number, b: number, a: number): Color4;
        /**
         * Compute the Color4 hexadecimal code as a string
         * @returns a string containing the hexadecimal representation of the Color4 object
         */
        toHexString(): string;
        /**
         * Computes a new Color4 converted from the current one to linear space
         * @returns a new Color4 object
         */
        toLinearSpace(): Color4;
        /**
         * Converts the Color4 values to linear space and stores the result in "convertedColor"
         * @param convertedColor defines the Color4 object where to store the linear space version
         * @returns the unmodified Color4
         */
        toLinearSpaceToRef(convertedColor: Color4): Color4;
        /**
         * Computes a new Color4 converted from the current one to gamma space
         * @returns a new Color4 object
         */
        toGammaSpace(): Color4;
        /**
         * Converts the Color4 values to gamma space and stores the result in "convertedColor"
         * @param convertedColor defines the Color4 object where to store the gamma space version
         * @returns the unmodified Color4
         */
        toGammaSpaceToRef(convertedColor: Color4): Color4;
        /**
         * Creates a new Color4 from the string containing valid hexadecimal values
         * @param hex defines a string containing valid hexadecimal values
         * @returns a new Color4 object
         */
        static FromHexString(hex: string): Color4;
        /**
         * Creates a new Color4 object set with the linearly interpolated values of "amount" between the left Color4 object and the right Color4 object
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @returns a new Color4 object
         */
        static Lerp(left: Color4, right: Color4, amount: number): Color4;
        /**
         * Set the given "result" with the linearly interpolated values of "amount" between the left Color4 object and the right Color4 object
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @param result defines the Color4 object where to store data
         */
        static LerpToRef(left: Color4, right: Color4, amount: number, result: Color4): void;
        /**
         * Creates a new Color4 from the starting index element of the given array
         * @param array defines the source array to read from
         * @param offset defines the offset in the source array
         * @returns a new Color4 object
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Color4;
        /**
         * Creates a new Color3 from integer values (< 256)
         * @param r defines the red component to read from (value between 0 and 255)
         * @param g defines the green component to read from (value between 0 and 255)
         * @param b defines the blue component to read from (value between 0 and 255)
         * @param a defines the alpha component to read from (value between 0 and 255)
         * @returns a new Color3 object
         */
        static FromInts(r: number, g: number, b: number, a: number): Color4;
        /**
         * Check the content of a given array and convert it to an array containing RGBA data
         * If the original array was already containing count * 4 values then it is returned directly
         * @param colors defines the array to check
         * @param count defines the number of RGBA data to expect
         * @returns an array containing count * 4 values (RGBA)
         */
        static CheckColors4(colors: number[], count: number): number[];
    }
    /**
     * Class representing a vector containing 2 coordinates
     */
    class Vector2 {
        /** defines the first coordinate */
        x: number;
        /** defines the second coordinate */
        y: number;
        /**
         * Creates a new Vector2 from the given x and y coordinates
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         */
        constructor(
            /** defines the first coordinate */
            x: number, 
            /** defines the second coordinate */
            y: number);
        /**
         * Gets a string with the Vector2 coordinates
         * @returns a string with the Vector2 coordinates
         */
        toString(): string;
        /**
         * Gets class name
         * @returns the string "Vector2"
         */
        getClassName(): string;
        /**
         * Gets current vector hash code
         * @returns the Vector2 hash code as a number
         */
        getHashCode(): number;
        /**
         * Sets the Vector2 coordinates in the given array or Float32Array from the given index.
         * @param array defines the source array
         * @param index defines the offset in source array
         * @returns the current Vector2
         */
        toArray(array: FloatArray, index?: number): Vector2;
        /**
         * Copy the current vector to an array
         * @returns a new array with 2 elements: the Vector2 coordinates.
         */
        asArray(): number[];
        /**
         * Sets the Vector2 coordinates with the given Vector2 coordinates
         * @param source defines the source Vector2
         * @returns the current updated Vector2
         */
        copyFrom(source: Vector2): Vector2;
        /**
         * Sets the Vector2 coordinates with the given floats
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         * @returns the current updated Vector2
         */
        copyFromFloats(x: number, y: number): Vector2;
        /**
         * Sets the Vector2 coordinates with the given floats
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         * @returns the current updated Vector2
         */
        set(x: number, y: number): Vector2;
        /**
         * Add another vector with the current one
         * @param otherVector defines the other vector
         * @returns a new Vector2 set with the addition of the current Vector2 and the given one coordinates
         */
        add(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the addition of the current Vector2 and the given one coordinates
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        addToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Set the Vector2 coordinates by adding the given Vector2 coordinates
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        addInPlace(otherVector: Vector2): Vector2;
        /**
         * Gets a new Vector2 by adding the current Vector2 coordinates to the given Vector3 x, y coordinates
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        addVector3(otherVector: Vector3): Vector2;
        /**
         * Gets a new Vector2 set with the subtracted coordinates of the given one from the current Vector2
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        subtract(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the subtraction of the given one from the current Vector2 coordinates.
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        subtractToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Sets the current Vector2 coordinates by subtracting from it the given one coordinates
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        subtractInPlace(otherVector: Vector2): Vector2;
        /**
         * Multiplies in place the current Vector2 coordinates by the given ones
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        multiplyInPlace(otherVector: Vector2): Vector2;
        /**
         * Returns a new Vector2 set with the multiplication of the current Vector2 and the given one coordinates
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        multiply(otherVector: Vector2): Vector2;
        /**
         * Sets "result" coordinates with the multiplication of the current Vector2 and the given one coordinates
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        multiplyToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Gets a new Vector2 set with the Vector2 coordinates multiplied by the given floats
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         * @returns a new Vector2
         */
        multiplyByFloats(x: number, y: number): Vector2;
        /**
         * Returns a new Vector2 set with the Vector2 coordinates divided by the given one coordinates
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        divide(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the Vector2 divided by the given one coordinates
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        divideToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Divides the current Vector3 coordinates by the given ones
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        divideInPlace(otherVector: Vector2): Vector2;
        /**
         * Gets a new Vector2 with current Vector2 negated coordinates
         * @returns a new Vector2
         */
        negate(): Vector2;
        /**
         * Multiply the Vector2 coordinates by scale
         * @param scale defines the scaling factor
         * @returns the current updated Vector2
         */
        scaleInPlace(scale: number): Vector2;
        /**
         * Returns a new Vector2 scaled by "scale" from the current Vector2
         * @param scale defines the scaling factor
         * @returns a new Vector2
         */
        scale(scale: number): Vector2;
        /**
         * Scale the current Vector2 values by a factor to a given Vector2
         * @param scale defines the scale factor
         * @param result defines the Vector2 object where to store the result
         * @returns the unmodified current Vector2
         */
        scaleToRef(scale: number, result: Vector2): Vector2;
        /**
         * Scale the current Vector2 values by a factor and add the result to a given Vector2
         * @param scale defines the scale factor
         * @param result defines the Vector2 object where to store the result
         * @returns the unmodified current Vector2
         */
        scaleAndAddToRef(scale: number, result: Vector2): Vector2;
        /**
         * Gets a boolean if two vectors are equals
         * @param otherVector defines the other vector
         * @returns true if the given vector coordinates strictly equal the current Vector2 ones
         */
        equals(otherVector: Vector2): boolean;
        /**
         * Gets a boolean if two vectors are equals (using an epsilon value)
         * @param otherVector defines the other vector
         * @param epsilon defines the minimal distance to consider equality
         * @returns true if the given vector coordinates are close to the current ones by a distance of epsilon.
         */
        equalsWithEpsilon(otherVector: Vector2, epsilon?: number): boolean;
        /**
         * Gets the length of the vector
         * @returns the vector length (float)
         */
        length(): number;
        /**
         * Gets the vector squared length
         * @returns the vector squared length (float)
         */
        lengthSquared(): number;
        /**
         * Normalize the vector
         * @returns the current updated Vector2
         */
        normalize(): Vector2;
        /**
         * Gets a new Vector2 copied from the Vector2
         * @returns a new Vector2
         */
        clone(): Vector2;
        /**
         * Gets a new Vector2(0, 0)
         * @returns a new Vector2
         */
        static Zero(): Vector2;
        /**
         * Gets a new Vector2(1, 1)
         * @returns a new Vector2
         */
        static One(): Vector2;
        /**
         * Gets a new Vector2 set from the given index element of the given array
         * @param array defines the data source
         * @param offset defines the offset in the data source
         * @returns a new Vector2
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Vector2;
        /**
         * Sets "result" from the given index element of the given array
         * @param array defines the data source
         * @param offset defines the offset in the data source
         * @param result defines the target vector
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Vector2): void;
        /**
         * Gets a new Vector2 located for "amount" (float) on the CatmullRom spline defined by the given four Vector2
         * @param value1 defines 1st point of control
         * @param value2 defines 2nd point of control
         * @param value3 defines 3rd point of control
         * @param value4 defines 4th point of control
         * @param amount defines the interpolation factor
         * @returns a new Vector2
         */
        static CatmullRom(value1: Vector2, value2: Vector2, value3: Vector2, value4: Vector2, amount: number): Vector2;
        /**
         * Returns a new Vector2 set with same the coordinates than "value" ones if the vector "value" is in the square defined by "min" and "max".
         * If a coordinate of "value" is lower than "min" coordinates, the returned Vector2 is given this "min" coordinate.
         * If a coordinate of "value" is greater than "max" coordinates, the returned Vector2 is given this "max" coordinate
         * @param value defines the value to clamp
         * @param min defines the lower limit
         * @param max defines the upper limit
         * @returns a new Vector2
         */
        static Clamp(value: Vector2, min: Vector2, max: Vector2): Vector2;
        /**
         * Returns a new Vector2 located for "amount" (float) on the Hermite spline defined by the vectors "value1", "value3", "tangent1", "tangent2"
         * @param value1 defines the 1st control point
         * @param tangent1 defines the outgoing tangent
         * @param value2 defines the 2nd control point
         * @param tangent2 defines the incoming tangent
         * @param amount defines the interpolation factor
         * @returns a new Vector2
         */
        static Hermite(value1: Vector2, tangent1: Vector2, value2: Vector2, tangent2: Vector2, amount: number): Vector2;
        /**
         * Returns a new Vector2 located for "amount" (float) on the linear interpolation between the vector "start" adn the vector "end".
         * @param start defines the start vector
         * @param end defines the end vector
         * @param amount defines the interpolation factor
         * @returns a new Vector2
         */
        static Lerp(start: Vector2, end: Vector2, amount: number): Vector2;
        /**
         * Gets the dot product of the vector "left" and the vector "right"
         * @param left defines first vector
         * @param right defines second vector
         * @returns the dot product (float)
         */
        static Dot(left: Vector2, right: Vector2): number;
        /**
         * Returns a new Vector2 equal to the normalized given vector
         * @param vector defines the vector to normalize
         * @returns a new Vector2
         */
        static Normalize(vector: Vector2): Vector2;
        /**
         * Gets a new Vector2 set with the minimal coordinate values from the "left" and "right" vectors
         * @param left defines 1st vector
         * @param right defines 2nd vector
         * @returns a new Vector2
         */
        static Minimize(left: Vector2, right: Vector2): Vector2;
        /**
         * Gets a new Vecto2 set with the maximal coordinate values from the "left" and "right" vectors
         * @param left defines 1st vector
         * @param right defines 2nd vector
         * @returns a new Vector2
         */
        static Maximize(left: Vector2, right: Vector2): Vector2;
        /**
         * Gets a new Vector2 set with the transformed coordinates of the given vector by the given transformation matrix
         * @param vector defines the vector to transform
         * @param transformation defines the matrix to apply
         * @returns a new Vector2
         */
        static Transform(vector: Vector2, transformation: Matrix): Vector2;
        /**
         * Transforms the given vector coordinates by the given transformation matrix and stores the result in the vector "result" coordinates
         * @param vector defines the vector to transform
         * @param transformation defines the matrix to apply
         * @param result defines the target vector
         */
        static TransformToRef(vector: Vector2, transformation: Matrix, result: Vector2): void;
        /**
         * Determines if a given vector is included in a triangle
         * @param p defines the vector to test
         * @param p0 defines 1st triangle point
         * @param p1 defines 2nd triangle point
         * @param p2 defines 3rd triangle point
         * @returns true if the point "p" is in the triangle defined by the vertors "p0", "p1", "p2"
         */
        static PointInTriangle(p: Vector2, p0: Vector2, p1: Vector2, p2: Vector2): boolean;
        /**
         * Gets the distance between the vectors "value1" and "value2"
         * @param value1 defines first vector
         * @param value2 defines second vector
         * @returns the distance between vectors
         */
        static Distance(value1: Vector2, value2: Vector2): number;
        /**
         * Returns the squared distance between the vectors "value1" and "value2"
         * @param value1 defines first vector
         * @param value2 defines second vector
         * @returns the squared distance between vectors
         */
        static DistanceSquared(value1: Vector2, value2: Vector2): number;
        /**
         * Gets a new Vector2 located at the center of the vectors "value1" and "value2"
         * @param value1 defines first vector
         * @param value2 defines second vector
         * @returns a new Vector2
         */
        static Center(value1: Vector2, value2: Vector2): Vector2;
        /**
         * Gets the shortest distance (float) between the point "p" and the segment defined by the two points "segA" and "segB".
         * @param p defines the middle point
         * @param segA defines one point of the segment
         * @param segB defines the other point of the segment
         * @returns the shortest distance
         */
        static DistanceOfPointFromSegment(p: Vector2, segA: Vector2, segB: Vector2): number;
    }
    /**
     * Classed used to store (x,y,z) vector representation
     * A Vector3 is the main object used in 3D geometry
     * It can represent etiher the coordinates of a point the space, either a direction
     * Reminder: Babylon.js uses a left handed forward facing system
     */
    class Vector3 {
        /**
         * Defines the first coordinates (on X axis)
         */
        x: number;
        /**
         * Defines the second coordinates (on Y axis)
         */
        y: number;
        /**
         * Defines the third coordinates (on Z axis)
         */
        z: number;
        /**
         * Creates a new Vector3 object from the given x, y, z (floats) coordinates.
         * @param x defines the first coordinates (on X axis)
         * @param y defines the second coordinates (on Y axis)
         * @param z defines the third coordinates (on Z axis)
         */
        constructor(
            /**
             * Defines the first coordinates (on X axis)
             */
            x: number, 
            /**
             * Defines the second coordinates (on Y axis)
             */
            y: number, 
            /**
             * Defines the third coordinates (on Z axis)
             */
            z: number);
        /**
         * Creates a string representation of the Vector3
         * @returns a string with the Vector3 coordinates.
         */
        toString(): string;
        /**
         * Gets the class name
         * @returns the string "Vector3"
         */
        getClassName(): string;
        /**
         * Creates the Vector3 hash code
         * @returns a number which tends to be unique between Vector3 instances
         */
        getHashCode(): number;
        /**
         * Creates an array containing three elements : the coordinates of the Vector3
         * @returns a new array of numbers
         */
        asArray(): number[];
        /**
         * Populates the given array or Float32Array from the given index with the successive coordinates of the Vector3
         * @param array defines the destination array
         * @param index defines the offset in the destination array
         * @returns the current Vector3
         */
        toArray(array: FloatArray, index?: number): Vector3;
        /**
         * Converts the current Vector3 into a quaternion (considering that the Vector3 contains Euler angles representation of a rotation)
         * @returns a new Quaternion object, computed from the Vector3 coordinates
         */
        toQuaternion(): Quaternion;
        /**
         * Adds the given vector to the current Vector3
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        addInPlace(otherVector: Vector3): Vector3;
        /**
         * Gets a new Vector3, result of the addition the current Vector3 and the given vector
         * @param otherVector defines the second operand
         * @returns the resulting Vector3
         */
        add(otherVector: Vector3): Vector3;
        /**
         * Adds the current Vector3 to the given one and stores the result in the vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        addToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Subtract the given vector from the current Vector3
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        subtractInPlace(otherVector: Vector3): Vector3;
        /**
         * Returns a new Vector3, result of the subtraction of the given vector from the current Vector3
         * @param otherVector defines the second operand
         * @returns the resulting Vector3
         */
        subtract(otherVector: Vector3): Vector3;
        /**
         * Subtracts the given vector from the current Vector3 and stores the result in the vector "result".
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        subtractToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 set with the subtraction of the given floats from the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the resulting Vector3
         */
        subtractFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Subtracts the given floats from the current Vector3 coordinates and set the given vector "result" with this result
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        subtractFromFloatsToRef(x: number, y: number, z: number, result: Vector3): Vector3;
        /**
         * Gets a new Vector3 set with the current Vector3 negated coordinates
         * @returns a new Vector3
         */
        negate(): Vector3;
        /**
         * Multiplies the Vector3 coordinates by the float "scale"
         * @param scale defines the multiplier factor
         * @returns the current updated Vector3
         */
        scaleInPlace(scale: number): Vector3;
        /**
         * Returns a new Vector3 set with the current Vector3 coordinates multiplied by the float "scale"
         * @param scale defines the multiplier factor
         * @returns a new Vector3
         */
        scale(scale: number): Vector3;
        /**
         * Multiplies the current Vector3 coordinates by the float "scale" and stores the result in the given vector "result" coordinates
         * @param scale defines the multiplier factor
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        scaleToRef(scale: number, result: Vector3): Vector3;
        /**
         * Scale the current Vector3 values by a factor and add the result to a given Vector3
         * @param scale defines the scale factor
         * @param result defines the Vector3 object where to store the result
         * @returns the unmodified current Vector3
         */
        scaleAndAddToRef(scale: number, result: Vector3): Vector3;
        /**
         * Returns true if the current Vector3 and the given vector coordinates are strictly equal
         * @param otherVector defines the second operand
         * @returns true if both vectors are equals
         */
        equals(otherVector: Vector3): boolean;
        /**
         * Returns true if the current Vector3 and the given vector coordinates are distant less than epsilon
         * @param otherVector defines the second operand
         * @param epsilon defines the minimal distance to define values as equals
         * @returns true if both vectors are distant less than epsilon
         */
        equalsWithEpsilon(otherVector: Vector3, epsilon?: number): boolean;
        /**
         * Returns true if the current Vector3 coordinates equals the given floats
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns true if both vectors are equals
         */
        equalsToFloats(x: number, y: number, z: number): boolean;
        /**
         * Multiplies the current Vector3 coordinates by the given ones
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        multiplyInPlace(otherVector: Vector3): Vector3;
        /**
         * Returns a new Vector3, result of the multiplication of the current Vector3 by the given vector
         * @param otherVector defines the second operand
         * @returns the new Vector3
         */
        multiply(otherVector: Vector3): Vector3;
        /**
         * Multiplies the current Vector3 by the given one and stores the result in the given vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        multiplyToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 set with the result of the mulliplication of the current Vector3 coordinates by the given floats
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the new Vector3
         */
        multiplyByFloats(x: number, y: number, z: number): Vector3;
        /**
         * Returns a new Vector3 set with the result of the division of the current Vector3 coordinates by the given ones
         * @param otherVector defines the second operand
         * @returns the new Vector3
         */
        divide(otherVector: Vector3): Vector3;
        /**
         * Divides the current Vector3 coordinates by the given ones and stores the result in the given vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        divideToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Divides the current Vector3 coordinates by the given ones.
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        divideInPlace(otherVector: Vector3): Vector3;
        /**
         * Updates the current Vector3 with the minimal coordinate values between its and the given vector ones
         * @param other defines the second operand
         * @returns the current updated Vector3
         */
        minimizeInPlace(other: Vector3): Vector3;
        /**
         * Updates the current Vector3 with the maximal coordinate values between its and the given vector ones.
         * @param other defines the second operand
         * @returns the current updated Vector3
         */
        maximizeInPlace(other: Vector3): Vector3;
        /**
         * Gets a boolean indicating that the vector is non uniform meaning x, y or z are not all the same
         */
        readonly isNonUniform: boolean;
        /**
         * Gets the length of the Vector3
         * @returns the length of the Vecto3
         */
        length(): number;
        /**
         * Gets the squared length of the Vector3
         * @returns squared length of the Vector3
         */
        lengthSquared(): number;
        /**
         * Normalize the current Vector3.
         * Please note that this is an in place operation.
         * @returns the current updated Vector3
         */
        normalize(): Vector3;
        /**
         * Normalize the current Vector3 to a new vector
         * @returns the new Vector3
         */
        normalizeToNew(): Vector3;
        /**
         * Normalize the current Vector3 to the reference
         * @param reference define the Vector3 to update
         * @returns the updated Vector3
         */
        normalizeToRef(reference: Vector3): Vector3;
        /**
         * Creates a new Vector3 copied from the current Vector3
         * @returns the new Vector3
         */
        clone(): Vector3;
        /**
         * Copies the given vector coordinates to the current Vector3 ones
         * @param source defines the source Vector3
         * @returns the current updated Vector3
         */
        copyFrom(source: Vector3): Vector3;
        /**
         * Copies the given floats to the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        copyFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Copies the given floats to the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        set(x: number, y: number, z: number): Vector3;
        /**
         * Get the clip factor between two vectors
         * @param vector0 defines the first operand
         * @param vector1 defines the second operand
         * @param axis defines the axis to use
         * @param size defines the size along the axis
         * @returns the clip factor
         */
        static GetClipFactor(vector0: Vector3, vector1: Vector3, axis: Vector3, size: number): number;
        /**
         * Get angle between two vectors
         * @param vector0 angle between vector0 and vector1
         * @param vector1 angle between vector0 and vector1
         * @param normal direction of the normal
         * @return the angle between vector0 and vector1
         */
        static GetAngleBetweenVectors(vector0: Vector3, vector1: Vector3, normal: Vector3): number;
        /**
         * Returns a new Vector3 set from the index "offset" of the given array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @returns the new Vector3
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Vector3;
        /**
         * Returns a new Vector3 set from the index "offset" of the given Float32Array
         * This function is deprecated.  Use FromArray instead
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @returns the new Vector3
         */
        static FromFloatArray(array: Float32Array, offset?: number): Vector3;
        /**
         * Sets the given vector "result" with the element values from the index "offset" of the given array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param result defines the Vector3 where to store the result
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Vector3): void;
        /**
         * Sets the given vector "result" with the element values from the index "offset" of the given Float32Array
         * This function is deprecated.  Use FromArrayToRef instead.
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param result defines the Vector3 where to store the result
         */
        static FromFloatArrayToRef(array: Float32Array, offset: number, result: Vector3): void;
        /**
         * Sets the given vector "result" with the given floats.
         * @param x defines the x coordinate of the source
         * @param y defines the y coordinate of the source
         * @param z defines the z coordinate of the source
         * @param result defines the Vector3 where to store the result
         */
        static FromFloatsToRef(x: number, y: number, z: number, result: Vector3): void;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, 0.0)
         * @returns a new empty Vector3
         */
        static Zero(): Vector3;
        /**
         * Returns a new Vector3 set to (1.0, 1.0, 1.0)
         * @returns a new unit Vector3
         */
        static One(): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 1.0, 0.0)
         * @returns a new up Vector3
         */
        static Up(): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, 1.0)
         * @returns a new forward Vector3
         */
        static Forward(): Vector3;
        /**
         * Returns a new Vector3 set to (1.0, 0.0, 0.0)
         * @returns a new right Vector3
         */
        static Right(): Vector3;
        /**
         * Returns a new Vector3 set to (-1.0, 0.0, 0.0)
         * @returns a new left Vector3
         */
        static Left(): Vector3;
        /**
         * Returns a new Vector3 set with the result of the transformation by the given matrix of the given vector.
         * This method computes tranformed coordinates only, not transformed direction vectors (ie. it takes translation in account)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @returns the transformed Vector3
         */
        static TransformCoordinates(vector: Vector3, transformation: Matrix): Vector3;
        /**
         * Sets the given vector "result" coordinates with the result of the transformation by the given matrix of the given vector
         * This method computes tranformed coordinates only, not transformed direction vectors (ie. it takes translation in account)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformCoordinatesToRef(vector: Vector3, transformation: Matrix, result: Vector3): void;
        /**
         * Sets the given vector "result" coordinates with the result of the transformation by the given matrix of the given floats (x, y, z)
         * This method computes tranformed coordinates only, not transformed direction vectors
         * @param x define the x coordinate of the source vector
         * @param y define the y coordinate of the source vector
         * @param z define the z coordinate of the source vector
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformCoordinatesFromFloatsToRef(x: number, y: number, z: number, transformation: Matrix, result: Vector3): void;
        /**
         * Returns a new Vector3 set with the result of the normal transformation by the given matrix of the given vector
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @returns the new Vector3
         */
        static TransformNormal(vector: Vector3, transformation: Matrix): Vector3;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given vector
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformNormalToRef(vector: Vector3, transformation: Matrix, result: Vector3): void;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given floats (x, y, z)
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param x define the x coordinate of the source vector
         * @param y define the y coordinate of the source vector
         * @param z define the z coordinate of the source vector
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformNormalFromFloatsToRef(x: number, y: number, z: number, transformation: Matrix, result: Vector3): void;
        /**
         * Returns a new Vector3 located for "amount" on the CatmullRom interpolation spline defined by the vectors "value1", "value2", "value3", "value4"
         * @param value1 defines the first control point
         * @param value2 defines the second control point
         * @param value3 defines the third control point
         * @param value4 defines the fourth control point
         * @param amount defines the amount on the spline to use
         * @returns the new Vector3
         */
        static CatmullRom(value1: Vector3, value2: Vector3, value3: Vector3, value4: Vector3, amount: number): Vector3;
        /**
         * Returns a new Vector3 set with the coordinates of "value", if the vector "value" is in the cube defined by the vectors "min" and "max"
         * If a coordinate value of "value" is lower than one of the "min" coordinate, then this "value" coordinate is set with the "min" one
         * If a coordinate value of "value" is greater than one of the "max" coordinate, then this "value" coordinate is set with the "max" one
         * @param value defines the current value
         * @param min defines the lower range value
         * @param max defines the upper range value
         * @returns the new Vector3
         */
        static Clamp(value: Vector3, min: Vector3, max: Vector3): Vector3;
        /**
         * Returns a new Vector3 located for "amount" (float) on the Hermite interpolation spline defined by the vectors "value1", "tangent1", "value2", "tangent2"
         * @param value1 defines the first control point
         * @param tangent1 defines the first tangent vector
         * @param value2 defines the second control point
         * @param tangent2 defines the second tangent vector
         * @param amount defines the amount on the interpolation spline (between 0 and 1)
         * @returns the new Vector3
         */
        static Hermite(value1: Vector3, tangent1: Vector3, value2: Vector3, tangent2: Vector3, amount: number): Vector3;
        /**
         * Returns a new Vector3 located for "amount" (float) on the linear interpolation between the vectors "start" and "end"
         * @param start defines the start value
         * @param end defines the end value
         * @param amount max defines amount between both (between 0 and 1)
         * @returns the new Vector3
         */
        static Lerp(start: Vector3, end: Vector3, amount: number): Vector3;
        /**
         * Sets the given vector "result" with the result of the linear interpolation from the vector "start" for "amount" to the vector "end"
         * @param start defines the start value
         * @param end defines the end value
         * @param amount max defines amount between both (between 0 and 1)
         * @param result defines the Vector3 where to store the result
         */
        static LerpToRef(start: Vector3, end: Vector3, amount: number, result: Vector3): void;
        /**
         * Returns the dot product (float) between the vectors "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the dot product
         */
        static Dot(left: Vector3, right: Vector3): number;
        /**
         * Returns a new Vector3 as the cross product of the vectors "left" and "right"
         * The cross product is then orthogonal to both "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the cross product
         */
        static Cross(left: Vector3, right: Vector3): Vector3;
        /**
         * Sets the given vector "result" with the cross product of "left" and "right"
         * The cross product is then orthogonal to both "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @param result defines the Vector3 where to store the result
         */
        static CrossToRef(left: Vector3, right: Vector3, result: Vector3): void;
        /**
         * Returns a new Vector3 as the normalization of the given vector
         * @param vector defines the Vector3 to normalize
         * @returns the new Vector3
         */
        static Normalize(vector: Vector3): Vector3;
        /**
         * Sets the given vector "result" with the normalization of the given first vector
         * @param vector defines the Vector3 to normalize
         * @param result defines the Vector3 where to store the result
         */
        static NormalizeToRef(vector: Vector3, result: Vector3): void;
        private static _viewportMatrixCache;
        /**
         * Project a Vector3 onto screen space
         * @param vector defines the Vector3 to project
         * @param world defines the world matrix to use
         * @param transform defines the transform (view x projection) matrix to use
         * @param viewport defines the screen viewport to use
         * @returns the new Vector3
         */
        static Project(vector: Vector3, world: Matrix, transform: Matrix, viewport: Viewport): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param transform defines the transform (view x projection) matrix to use
         * @returns the new Vector3
         */
        static UnprojectFromTransform(source: Vector3, viewportWidth: number, viewportHeight: number, world: Matrix, transform: Matrix): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @returns the new Vector3
         */
        static Unproject(source: Vector3, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @param result defines the Vector3 where to store the result
         */
        static UnprojectToRef(source: Vector3, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix, result: Vector3): void;
        /**
         * Unproject from screen space to object space
         * @param sourceX defines the screen space x coordinate to use
         * @param sourceY defines the screen space y coordinate to use
         * @param sourceZ defines the screen space z coordinate to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @param result defines the Vector3 where to store the result
         */
        static UnprojectFloatsToRef(sourceX: float, sourceY: float, sourceZ: float, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix, result: Vector3): void;
        /**
         * Gets the minimal coordinate values between two Vector3
         * @param left defines the first operand
         * @param right defines the second operand
         * @returns the new Vector3
         */
        static Minimize(left: Vector3, right: Vector3): Vector3;
        /**
         * Gets the maximal coordinate values between two Vector3
         * @param left defines the first operand
         * @param right defines the second operand
         * @returns the new Vector3
         */
        static Maximize(left: Vector3, right: Vector3): Vector3;
        /**
         * Returns the distance between the vectors "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the distance
         */
        static Distance(value1: Vector3, value2: Vector3): number;
        /**
         * Returns the squared distance between the vectors "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the squared distance
         */
        static DistanceSquared(value1: Vector3, value2: Vector3): number;
        /**
         * Returns a new Vector3 located at the center between "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the new Vector3
         */
        static Center(value1: Vector3, value2: Vector3): Vector3;
        /**
         * Given three orthogonal normalized left-handed oriented Vector3 axis in space (target system),
         * RotationFromAxis() returns the rotation Euler angles (ex : rotation.x, rotation.y, rotation.z) to apply
         * to something in order to rotate it from its local system to the given target system
         * Note: axis1, axis2 and axis3 are normalized during this operation
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @returns a new Vector3
         */
        static RotationFromAxis(axis1: Vector3, axis2: Vector3, axis3: Vector3): Vector3;
        /**
         * The same than RotationFromAxis but updates the given ref Vector3 parameter instead of returning a new Vector3
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @param ref defines the Vector3 where to store the result
         */
        static RotationFromAxisToRef(axis1: Vector3, axis2: Vector3, axis3: Vector3, ref: Vector3): void;
    }
    class Vector4 {
        x: number;
        y: number;
        z: number;
        w: number;
        /**
         * Creates a Vector4 object from the given floats.
         */
        constructor(x: number, y: number, z: number, w: number);
        /**
         * Returns the string with the Vector4 coordinates.
         */
        toString(): string;
        /**
         * Returns the string "Vector4".
         */
        getClassName(): string;
        /**
         * Returns the Vector4 hash code.
         */
        getHashCode(): number;
        /**
         * Returns a new array populated with 4 elements : the Vector4 coordinates.
         */
        asArray(): number[];
        /**
         * Populates the given array from the given index with the Vector4 coordinates.
         * Returns the Vector4.
         */
        toArray(array: FloatArray, index?: number): Vector4;
        /**
         * Adds the given vector to the current Vector4.
         * Returns the updated Vector4.
         */
        addInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 as the result of the addition of the current Vector4 and the given one.
         */
        add(otherVector: Vector4): Vector4;
        /**
         * Updates the given vector "result" with the result of the addition of the current Vector4 and the given one.
         * Returns the current Vector4.
         */
        addToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Subtract in place the given vector from the current Vector4.
         * Returns the updated Vector4.
         */
        subtractInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 with the result of the subtraction of the given vector from the current Vector4.
         */
        subtract(otherVector: Vector4): Vector4;
        /**
         * Sets the given vector "result" with the result of the subtraction of the given vector from the current Vector4.
         * Returns the current Vector4.
         */
        subtractToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the result of the subtraction of the given floats from the current Vector4 coordinates.
         */
        subtractFromFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Sets the given vector "result" set with the result of the subtraction of the given floats from the current Vector4 coordinates.
         * Returns the current Vector4.
         */
        subtractFromFloatsToRef(x: number, y: number, z: number, w: number, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the current Vector4 negated coordinates.
         */
        negate(): Vector4;
        /**
         * Multiplies the current Vector4 coordinates by scale (float).
         * Returns the updated Vector4.
         */
        scaleInPlace(scale: number): Vector4;
        /**
         * Returns a new Vector4 set with the current Vector4 coordinates multiplied by scale (float).
         */
        scale(scale: number): Vector4;
        /**
         * Sets the given vector "result" with the current Vector4 coordinates multiplied by scale (float).
         * Returns the current Vector4.
         */
        scaleToRef(scale: number, result: Vector4): Vector4;
        /**
         * Scale the current Vector4 values by a factor and add the result to a given Vector4
         * @param scale defines the scale factor
         * @param result defines the Vector4 object where to store the result
         * @returns the unmodified current Vector4
         */
        scaleAndAddToRef(scale: number, result: Vector4): Vector4;
        /**
         * Boolean : True if the current Vector4 coordinates are stricly equal to the given ones.
         */
        equals(otherVector: Vector4): boolean;
        /**
         * Boolean : True if the current Vector4 coordinates are each beneath the distance "epsilon" from the given vector ones.
         */
        equalsWithEpsilon(otherVector: Vector4, epsilon?: number): boolean;
        /**
         * Boolean : True if the given floats are strictly equal to the current Vector4 coordinates.
         */
        equalsToFloats(x: number, y: number, z: number, w: number): boolean;
        /**
         * Multiplies in place the current Vector4 by the given one.
         * Returns the updated Vector4.
         */
        multiplyInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the multiplication result of the current Vector4 and the given one.
         */
        multiply(otherVector: Vector4): Vector4;
        /**
         * Updates the given vector "result" with the multiplication result of the current Vector4 and the given one.
         * Returns the current Vector4.
         */
        multiplyToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the multiplication result of the given floats and the current Vector4 coordinates.
         */
        multiplyByFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Returns a new Vector4 set with the division result of the current Vector4 by the given one.
         */
        divide(otherVector: Vector4): Vector4;
        /**
         * Updates the given vector "result" with the division result of the current Vector4 by the given one.
         * Returns the current Vector4.
         */
        divideToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Divides the current Vector3 coordinates by the given ones.
         * @returns the updated Vector3.
         */
        divideInPlace(otherVector: Vector4): Vector4;
        /**
         * Updates the Vector4 coordinates with the minimum values between its own and the given vector ones
         * @param other defines the second operand
         * @returns the current updated Vector4
         */
        minimizeInPlace(other: Vector4): Vector4;
        /**
         * Updates the Vector4 coordinates with the maximum values between its own and the given vector ones
         * @param other defines the second operand
         * @returns the current updated Vector4
         */
        maximizeInPlace(other: Vector4): Vector4;
        /**
         * Returns the Vector4 length (float).
         */
        length(): number;
        /**
         * Returns the Vector4 squared length (float).
         */
        lengthSquared(): number;
        /**
         * Normalizes in place the Vector4.
         * Returns the updated Vector4.
         */
        normalize(): Vector4;
        /**
         * Returns a new Vector3 from the Vector4 (x, y, z) coordinates.
         */
        toVector3(): Vector3;
        /**
         * Returns a new Vector4 copied from the current one.
         */
        clone(): Vector4;
        /**
         * Updates the current Vector4 with the given one coordinates.
         * Returns the updated Vector4.
         */
        copyFrom(source: Vector4): Vector4;
        /**
         * Updates the current Vector4 coordinates with the given floats.
         * Returns the updated Vector4.
         */
        copyFromFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Updates the current Vector4 coordinates with the given floats.
         * Returns the updated Vector4.
         */
        set(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Returns a new Vector4 set from the starting index of the given array.
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Vector4;
        /**
         * Updates the given vector "result" from the starting index of the given array.
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Vector4): void;
        /**
         * Updates the given vector "result" from the starting index of the given Float32Array.
         */
        static FromFloatArrayToRef(array: Float32Array, offset: number, result: Vector4): void;
        /**
         * Updates the given vector "result" coordinates from the given floats.
         */
        static FromFloatsToRef(x: number, y: number, z: number, w: number, result: Vector4): void;
        /**
         * Returns a new Vector4 set to (0.0, 0.0, 0.0, 0.0)
         */
        static Zero(): Vector4;
        /**
         * Returns a new Vector4 set to (1.0, 1.0, 1.0, 1.0)
         */
        static One(): Vector4;
        /**
         * Returns a new normalized Vector4 from the given one.
         */
        static Normalize(vector: Vector4): Vector4;
        /**
         * Updates the given vector "result" from the normalization of the given one.
         */
        static NormalizeToRef(vector: Vector4, result: Vector4): void;
        static Minimize(left: Vector4, right: Vector4): Vector4;
        static Maximize(left: Vector4, right: Vector4): Vector4;
        /**
         * Returns the distance (float) between the vectors "value1" and "value2".
         */
        static Distance(value1: Vector4, value2: Vector4): number;
        /**
         * Returns the squared distance (float) between the vectors "value1" and "value2".
         */
        static DistanceSquared(value1: Vector4, value2: Vector4): number;
        /**
         * Returns a new Vector4 located at the center between the vectors "value1" and "value2".
         */
        static Center(value1: Vector4, value2: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the result of the normal transformation by the given matrix of the given vector.
         * This methods computes transformed normalized direction vectors only.
         */
        static TransformNormal(vector: Vector4, transformation: Matrix): Vector4;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given vector.
         * This methods computes transformed normalized direction vectors only.
         */
        static TransformNormalToRef(vector: Vector4, transformation: Matrix, result: Vector4): void;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given floats (x, y, z, w).
         * This methods computes transformed normalized direction vectors only.
         */
        static TransformNormalFromFloatsToRef(x: number, y: number, z: number, w: number, transformation: Matrix, result: Vector4): void;
    }
    interface ISize {
        width: number;
        height: number;
    }
    class Size implements ISize {
        width: number;
        height: number;
        /**
         * Creates a Size object from the given width and height (floats).
         */
        constructor(width: number, height: number);
        toString(): string;
        /**
         * Returns the string "Size"
         */
        getClassName(): string;
        /**
         * Returns the Size hash code.
         */
        getHashCode(): number;
        /**
         * Updates the current size from the given one.
         * Returns the updated Size.
         */
        copyFrom(src: Size): void;
        /**
         * Updates in place the current Size from the given floats.
         * Returns the updated Size.
         */
        copyFromFloats(width: number, height: number): Size;
        /**
         * Updates in place the current Size from the given floats.
         * Returns the updated Size.
         */
        set(width: number, height: number): Size;
        /**
         * Returns a new Size set with the multiplication result of the current Size and the given floats.
         */
        multiplyByFloats(w: number, h: number): Size;
        /**
         * Returns a new Size copied from the given one.
         */
        clone(): Size;
        /**
         * Boolean : True if the current Size and the given one width and height are strictly equal.
         */
        equals(other: Size): boolean;
        /**
         * Returns the surface of the Size : width * height (float).
         */
        readonly surface: number;
        /**
         * Returns a new Size set to (0.0, 0.0)
         */
        static Zero(): Size;
        /**
         * Returns a new Size set as the addition result of the current Size and the given one.
         */
        add(otherSize: Size): Size;
        /**
         * Returns a new Size set as the subtraction result of  the given one from the current Size.
         */
        subtract(otherSize: Size): Size;
        /**
         * Returns a new Size set at the linear interpolation "amount" between "start" and "end".
         */
        static Lerp(start: Size, end: Size, amount: number): Size;
    }
    /**
     * Class used to store quaternion data
     * @see https://en.wikipedia.org/wiki/Quaternion
     * @see http://doc.babylonjs.com/features/position,_rotation,_scaling
     */
    class Quaternion {
        /** defines the first component (0 by default) */
        x: number;
        /** defines the second component (0 by default) */
        y: number;
        /** defines the third component (0 by default) */
        z: number;
        /** defines the fourth component (1.0 by default) */
        w: number;
        /**
         * Creates a new Quaternion from the given floats
         * @param x defines the first component (0 by default)
         * @param y defines the second component (0 by default)
         * @param z defines the third component (0 by default)
         * @param w defines the fourth component (1.0 by default)
         */
        constructor(
            /** defines the first component (0 by default) */
            x?: number, 
            /** defines the second component (0 by default) */
            y?: number, 
            /** defines the third component (0 by default) */
            z?: number, 
            /** defines the fourth component (1.0 by default) */
            w?: number);
        /**
         * Gets a string representation for the current quaternion
         * @returns a string with the Quaternion coordinates
         */
        toString(): string;
        /**
         * Gets the class name of the quaternion
         * @returns the string "Quaternion"
         */
        getClassName(): string;
        /**
         * Gets a hash code for this quaternion
         * @returns the quaternion hash code
         */
        getHashCode(): number;
        /**
         * Copy the quaternion to an array
         * @returns a new array populated with 4 elements from the quaternion coordinates
         */
        asArray(): number[];
        /**
         * Check if two quaternions are equals
         * @param otherQuaternion defines the second operand
         * @return true if the current quaternion and the given one coordinates are strictly equals
         */
        equals(otherQuaternion: Quaternion): boolean;
        /**
         * Clone the current quaternion
         * @returns a new quaternion copied from the current one
         */
        clone(): Quaternion;
        /**
         * Copy a quaternion to the current one
         * @param other defines the other quaternion
         * @returns the updated current quaternion
         */
        copyFrom(other: Quaternion): Quaternion;
        /**
         * Updates the current quaternion with the given float coordinates
         * @param x defines the x coordinate
         * @param y defines the y coordinate
         * @param z defines the z coordinate
         * @param w defines the w coordinate
         * @returns the updated current quaternion
         */
        copyFromFloats(x: number, y: number, z: number, w: number): Quaternion;
        /**
         * Updates the current quaternion from the given float coordinates
         * @param x defines the x coordinate
         * @param y defines the y coordinate
         * @param z defines the z coordinate
         * @param w defines the w coordinate
         * @returns the updated current quaternion
         */
        set(x: number, y: number, z: number, w: number): Quaternion;
        /**
         * Adds two quaternions
         * @param other defines the second operand
         * @returns a new quaternion as the addition result of the given one and the current quaternion
         */
        add(other: Quaternion): Quaternion;
        /**
         * Add a quaternion to the current one
         * @param other defines the quaternion to add
         * @returns the current quaternion
         */
        addInPlace(other: Quaternion): Quaternion;
        /**
         * Subtract two quaternions
         * @param other defines the second operand
         * @returns a new quaternion as the subtraction result of the given one from the current one
         */
        subtract(other: Quaternion): Quaternion;
        /**
         * Multiplies the current quaternion by a scale factor
         * @param value defines the scale factor
         * @returns a new quaternion set by multiplying the current quaternion coordinates by the float "scale"
         */
        scale(value: number): Quaternion;
        /**
         * Scale the current quaternion values by a factor and stores the result to a given quaternion
         * @param scale defines the scale factor
         * @param result defines the Quaternion object where to store the result
         * @returns the unmodified current quaternion
         */
        scaleToRef(scale: number, result: Quaternion): Quaternion;
        /**
         * Multiplies in place the current quaternion by a scale factor
         * @param value defines the scale factor
         * @returns the current modified quaternion
         */
        scaleInPlace(value: number): Quaternion;
        /**
         * Scale the current quaternion values by a factor and add the result to a given quaternion
         * @param scale defines the scale factor
         * @param result defines the Quaternion object where to store the result
         * @returns the unmodified current quaternion
         */
        scaleAndAddToRef(scale: number, result: Quaternion): Quaternion;
        /**
         * Multiplies two quaternions
         * @param q1 defines the second operand
         * @returns a new quaternion set as the multiplication result of the current one with the given one "q1"
         */
        multiply(q1: Quaternion): Quaternion;
        /**
         * Sets the given "result" as the the multiplication result of the current one with the given one "q1"
         * @param q1 defines the second operand
         * @param result defines the target quaternion
         * @returns the current quaternion
         */
        multiplyToRef(q1: Quaternion, result: Quaternion): Quaternion;
        /**
         * Updates the current quaternion with the multiplication of itself with the given one "q1"
         * @param q1 defines the second operand
         * @returns the currentupdated quaternion
         */
        multiplyInPlace(q1: Quaternion): Quaternion;
        /**
         * Conjugates (1-q) the current quaternion and stores the result in the given quaternion
         * @param ref defines the target quaternion
         * @returns the current quaternion
         */
        conjugateToRef(ref: Quaternion): Quaternion;
        /**
         * Conjugates in place (1-q) the current quaternion
         * @returns the current updated quaternion
         */
        conjugateInPlace(): Quaternion;
        /**
         * Conjugates in place (1-q) the current quaternion
         * @returns a new quaternion
         */
        conjugate(): Quaternion;
        /**
         * Gets length of current quaternion
         * @returns the quaternion length (float)
         */
        length(): number;
        /**
         * Normalize in place the current quaternion
         * @returns the current updated quaternion
         */
        normalize(): Quaternion;
        /**
         * Returns a new Vector3 set with the Euler angles translated from the current quaternion
         * @param order is a reserved parameter and is ignore for now
         * @returns a new Vector3 containing the Euler angles
         */
        toEulerAngles(order?: string): Vector3;
        /**
         * Sets the given vector3 "result" with the Euler angles translated from the current quaternion
         * @param result defines the vector which will be filled with the Euler angles
         * @param order is a reserved parameter and is ignore for now
         * @returns the current unchanged quaternion
         */
        toEulerAnglesToRef(result: Vector3, order?: string): Quaternion;
        /**
         * Updates the given rotation matrix with the current quaternion values
         * @param result defines the target matrix
         * @returns the current unchanged quaternion
         */
        toRotationMatrix(result: Matrix): Quaternion;
        /**
         * Updates the current quaternion from the given rotation matrix values
         * @param matrix defines the source matrix
         * @returns the current updated quaternion
         */
        fromRotationMatrix(matrix: Matrix): Quaternion;
        /**
         * Creates a new quaternion from a rotation matrix
         * @param matrix defines the source matrix
         * @returns a new quaternion created from the given rotation matrix values
         */
        static FromRotationMatrix(matrix: Matrix): Quaternion;
        /**
         * Updates the given quaternion with the given rotation matrix values
         * @param matrix defines the source matrix
         * @param result defines the target quaternion
         */
        static FromRotationMatrixToRef(matrix: Matrix, result: Quaternion): void;
        /**
         * Returns the dot product (float) between the quaternions "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the dot product
         */
        static Dot(left: Quaternion, right: Quaternion): number;
        /**
         * Checks if the two quaternions are close to each other
         * @param quat0 defines the first quaternion to check
         * @param quat1 defines the second quaternion to check
         * @returns true if the two quaternions are close to each other
         */
        static AreClose(quat0: Quaternion, quat1: Quaternion): boolean;
        /**
         * Creates an empty quaternion
         * @returns a new quaternion set to (0.0, 0.0, 0.0)
         */
        static Zero(): Quaternion;
        /**
         * Inverse a given quaternion
         * @param q defines the source quaternion
         * @returns a new quaternion as the inverted current quaternion
         */
        static Inverse(q: Quaternion): Quaternion;
        /**
         * Creates an identity quaternion
         * @returns the identity quaternion
         */
        static Identity(): Quaternion;
        /**
         * Gets a boolean indicating if the given quaternion is identity
         * @param quaternion defines the quaternion to check
         * @returns true if the quaternion is identity
         */
        static IsIdentity(quaternion: Quaternion): boolean;
        /**
         * Creates a quaternion from a rotation around an axis
         * @param axis defines the axis to use
         * @param angle defines the angle to use
         * @returns a new quaternion created from the given axis (Vector3) and angle in radians (float)
         */
        static RotationAxis(axis: Vector3, angle: number): Quaternion;
        /**
         * Creates a rotation around an axis and stores it into the given quaternion
         * @param axis defines the axis to use
         * @param angle defines the angle to use
         * @param result defines the target quaternion
         * @returns the target quaternion
         */
        static RotationAxisToRef(axis: Vector3, angle: number, result: Quaternion): Quaternion;
        /**
         * Creates a new quaternion from data stored into an array
         * @param array defines the data source
         * @param offset defines the offset in the source array where the data starts
         * @returns a new quaternion
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Quaternion;
        /**
         * Creates a new quaternion from the given Euler float angles (y, x, z)
         * @param yaw defines the rotation around Y axis
         * @param pitch defines the rotation around X axis
         * @param roll defines the rotation around Z axis
         * @returns the new quaternion
         */
        static RotationYawPitchRoll(yaw: number, pitch: number, roll: number): Quaternion;
        /**
         * Creates a new rotation from the given Euler float angles (y, x, z) and stores it in the target quaternion
         * @param yaw defines the rotation around Y axis
         * @param pitch defines the rotation around X axis
         * @param roll defines the rotation around Z axis
         * @param result defines the target quaternion
         */
        static RotationYawPitchRollToRef(yaw: number, pitch: number, roll: number, result: Quaternion): void;
        /**
         * Creates a new quaternion from the given Euler float angles expressed in z-x-z orientation
         * @param alpha defines the rotation around first axis
         * @param beta defines the rotation around second axis
         * @param gamma defines the rotation around third axis
         * @returns the new quaternion
         */
        static RotationAlphaBetaGamma(alpha: number, beta: number, gamma: number): Quaternion;
        /**
         * Creates a new quaternion from the given Euler float angles expressed in z-x-z orientation and stores it in the target quaternion
         * @param alpha defines the rotation around first axis
         * @param beta defines the rotation around second axis
         * @param gamma defines the rotation around third axis
         * @param result defines the target quaternion
         */
        static RotationAlphaBetaGammaToRef(alpha: number, beta: number, gamma: number, result: Quaternion): void;
        /**
         * Creates a new quaternion containing the rotation value to reach the target (axis1, axis2, axis3) orientation as a rotated XYZ system (axis1, axis2 and axis3 are normalized during this operation)
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @returns the new quaternion
         */
        static RotationQuaternionFromAxis(axis1: Vector3, axis2: Vector3, axis3: Vector3): Quaternion;
        /**
         * Creates a rotation value to reach the target (axis1, axis2, axis3) orientation as a rotated XYZ system (axis1, axis2 and axis3 are normalized during this operation) and stores it in the target quaternion
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @param ref defines the target quaternion
         */
        static RotationQuaternionFromAxisToRef(axis1: Vector3, axis2: Vector3, axis3: Vector3, ref: Quaternion): void;
        /**
         * Interpolates between two quaternions
         * @param left defines first quaternion
         * @param right defines second quaternion
         * @param amount defines the gradient to use
         * @returns the new interpolated quaternion
         */
        static Slerp(left: Quaternion, right: Quaternion, amount: number): Quaternion;
        /**
         * Interpolates between two quaternions and stores it into a target quaternion
         * @param left defines first quaternion
         * @param right defines second quaternion
         * @param amount defines the gradient to use
         * @param result defines the target quaternion
         */
        static SlerpToRef(left: Quaternion, right: Quaternion, amount: number, result: Quaternion): void;
        /**
         * Interpolate between two quaternions using Hermite interpolation
         * @param value1 defines first quaternion
         * @param tangent1 defines the incoming tangent
         * @param value2 defines second quaternion
         * @param tangent2 defines the outgoing tangent
         * @param amount defines the target quaternion
         * @returns the new interpolated quaternion
         */
        static Hermite(value1: Quaternion, tangent1: Quaternion, value2: Quaternion, tangent2: Quaternion, amount: number): Quaternion;
    }
    /**
     * Class used to store matrix data (4x4)
     */
    class Matrix {
        private static _tempQuaternion;
        private static _xAxis;
        private static _yAxis;
        private static _zAxis;
        private static _updateFlagSeed;
        private static _identityReadOnly;
        private _isIdentity;
        private _isIdentityDirty;
        /**
         * Gets the update flag of the matrix which is an unique number for the matrix.
         * It will be incremented every time the matrix data change.
         * You can use it to speed the comparison between two versions of the same matrix.
         */
        updateFlag: number;
        /**
         * Gets or sets the internal data of the matrix
         */
        m: Float32Array;
        /** @hidden */
        _markAsUpdated(): void;
        /**
         * Creates an empty matrix (filled with zeros)
         */
        constructor();
        /**
         * Check if the current matrix is indentity
         * @param considerAsTextureMatrix defines if the current matrix must be considered as a texture matrix (3x2)
         * @returns true is the matrix is the identity matrix
         */
        isIdentity(considerAsTextureMatrix?: boolean): boolean;
        /**
         * Gets the determinant of the matrix
         * @returns the matrix determinant
         */
        determinant(): number;
        /**
         * Returns the matrix as a Float32Array
         * @returns the matrix underlying array
         */
        toArray(): Float32Array;
        /**
         * Returns the matrix as a Float32Array
        * @returns the matrix underlying array.
        */
        asArray(): Float32Array;
        /**
         * Inverts the current matrix in place
         * @returns the current inverted matrix
         */
        invert(): Matrix;
        /**
         * Sets all the matrix elements to zero
         * @returns the current matrix
         */
        reset(): Matrix;
        /**
         * Adds the current matrix with a second one
         * @param other defines the matrix to add
         * @returns a new matrix as the addition of the current matrix and the given one
         */
        add(other: Matrix): Matrix;
        /**
         * Sets the given matrix "result" to the addition of the current matrix and the given one
         * @param other defines the matrix to add
         * @param result defines the target matrix
         * @returns the current matrix
         */
        addToRef(other: Matrix, result: Matrix): Matrix;
        /**
         * Adds in place the given matrix to the current matrix
         * @param other defines the second operand
         * @returns the current updated matrix
         */
        addToSelf(other: Matrix): Matrix;
        /**
         * Sets the given matrix to the current inverted Matrix
         * @param other defines the target matrix
         * @returns the unmodified current matrix
         */
        invertToRef(other: Matrix): Matrix;
        /**
         * Inserts the translation vector (using 3 floats) in the current matrix
         * @param x defines the 1st component of the translation
         * @param y defines the 2nd component of the translation
         * @param z defines the 3rd component of the translation
         * @returns the current updated matrix
         */
        setTranslationFromFloats(x: number, y: number, z: number): Matrix;
        /**
         * Inserts the translation vector in the current matrix
         * @param vector3 defines the translation to insert
         * @returns the current updated matrix
         */
        setTranslation(vector3: Vector3): Matrix;
        /**
         * Gets the translation value of the current matrix
         * @returns a new Vector3 as the extracted translation from the matrix
         */
        getTranslation(): Vector3;
        /**
         * Fill a Vector3 with the extracted translation from the matrix
         * @param result defines the Vector3 where to store the translation
         * @returns the current matrix
         */
        getTranslationToRef(result: Vector3): Matrix;
        /**
         * Remove rotation and scaling part from the matrix
         * @returns the updated matrix
         */
        removeRotationAndScaling(): Matrix;
        /**
         * Multiply two matrices
         * @param other defines the second operand
         * @returns a new matrix set with the multiplication result of the current Matrix and the given one
         */
        multiply(other: Matrix): Matrix;
        /**
         * Copy the current matrix from the given one
         * @param other defines the source matrix
         * @returns the current updated matrix
         */
        copyFrom(other: Matrix): Matrix;
        /**
         * Populates the given array from the starting index with the current matrix values
         * @param array defines the target array
         * @param offset defines the offset in the target array where to start storing values
         * @returns the current matrix
         */
        copyToArray(array: Float32Array, offset?: number): Matrix;
        /**
         * Sets the given matrix "result" with the multiplication result of the current Matrix and the given one
         * @param other defines the second operand
         * @param result defines the matrix where to store the multiplication
         * @returns the current matrix
         */
        multiplyToRef(other: Matrix, result: Matrix): Matrix;
        /**
         * Sets the Float32Array "result" from the given index "offset" with the multiplication of the current matrix and the given one
         * @param other defines the second operand
         * @param result defines the array where to store the multiplication
         * @param offset defines the offset in the target array where to start storing values
         * @returns the current matrix
         */
        multiplyToArray(other: Matrix, result: Float32Array, offset: number): Matrix;
        /**
         * Check equality between this matrix and a second one
         * @param value defines the second matrix to compare
         * @returns true is the current matrix and the given one values are strictly equal
         */
        equals(value: Matrix): boolean;
        /**
         * Clone the current matrix
         * @returns a new matrix from the current matrix
         */
        clone(): Matrix;
        /**
         * Returns the name of the current matrix class
         * @returns the string "Matrix"
         */
        getClassName(): string;
        /**
         * Gets the hash code of the current matrix
         * @returns the hash code
         */
        getHashCode(): number;
        /**
         * Decomposes the current Matrix into a translation, rotation and scaling components
         * @param scale defines the scale vector3 given as a reference to update
         * @param rotation defines the rotation quaternion given as a reference to update
         * @param translation defines the translation vector3 given as a reference to update
         * @returns true if operation was successful
         */
        decompose(scale?: Vector3, rotation?: Quaternion, translation?: Vector3): boolean;
        /**
         * Gets specific row of the matrix
         * @param index defines the number of the row to get
         * @returns the index-th row of the current matrix as a new Vector4
         */
        getRow(index: number): Nullable<Vector4>;
        /**
         * Sets the index-th row of the current matrix to the vector4 values
         * @param index defines the number of the row to set
         * @param row defines the target vector4
         * @returns the updated current matrix
         */
        setRow(index: number, row: Vector4): Matrix;
        /**
         * Compute the transpose of the matrix
         * @returns the new transposed matrix
         */
        transpose(): Matrix;
        /**
         * Compute the transpose of the matrix and store it in a given matrix
         * @param result defines the target matrix
         * @returns the current matrix
         */
        transposeToRef(result: Matrix): Matrix;
        /**
         * Sets the index-th row of the current matrix with the given 4 x float values
         * @param index defines the row index
         * @param x defines the x component to set
         * @param y defines the y component to set
         * @param z defines the z component to set
         * @param w defines the w component to set
         * @returns the updated current matrix
         */
        setRowFromFloats(index: number, x: number, y: number, z: number, w: number): Matrix;
        /**
         * Compute a new matrix set with the current matrix values multiplied by scale (float)
         * @param scale defines the scale factor
         * @returns a new matrix
         */
        scale(scale: number): Matrix;
        /**
         * Scale the current matrix values by a factor to a given result matrix
         * @param scale defines the scale factor
         * @param result defines the matrix to store the result
         * @returns the current matrix
         */
        scaleToRef(scale: number, result: Matrix): Matrix;
        /**
         * Scale the current matrix values by a factor and add the result to a given matrix
         * @param scale defines the scale factor
         * @param result defines the Matrix to store the result
         * @returns the current matrix
         */
        scaleAndAddToRef(scale: number, result: Matrix): Matrix;
        /**
         * Writes to the given matrix a normal matrix, computed from this one (using values from identity matrix for fourth row and column).
         * @param ref matrix to store the result
         */
        toNormalMatrix(ref: Matrix): void;
        /**
         * Gets only rotation part of the current matrix
         * @returns a new matrix sets to the extracted rotation matrix from the current one
         */
        getRotationMatrix(): Matrix;
        /**
         * Extracts the rotation matrix from the current one and sets it as the given "result"
         * @param result defines the target matrix to store data to
         * @returns the current matrix
         */
        getRotationMatrixToRef(result: Matrix): Matrix;
        /**
         * Creates a matrix from an array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns a new Matrix set from the starting index of the given array
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Matrix;
        /**
         * Copy the content of an array into a given matrix
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @param result defines the target matrix
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Matrix): void;
        /**
         * Stores an array into a matrix after having multiplied each component by a given factor
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param scale defines the scaling factor
         * @param result defines the target matrix
         */
        static FromFloat32ArrayToRefScaled(array: Float32Array, offset: number, scale: number, result: Matrix): void;
        /**
         * Stores a list of values (16) inside a given matrix
         * @param initialM11 defines 1st value of 1st row
         * @param initialM12 defines 2nd value of 1st row
         * @param initialM13 defines 3rd value of 1st row
         * @param initialM14 defines 4th value of 1st row
         * @param initialM21 defines 1st value of 2nd row
         * @param initialM22 defines 2nd value of 2nd row
         * @param initialM23 defines 3rd value of 2nd row
         * @param initialM24 defines 4th value of 2nd row
         * @param initialM31 defines 1st value of 3rd row
         * @param initialM32 defines 2nd value of 3rd row
         * @param initialM33 defines 3rd value of 3rd row
         * @param initialM34 defines 4th value of 3rd row
         * @param initialM41 defines 1st value of 4th row
         * @param initialM42 defines 2nd value of 4th row
         * @param initialM43 defines 3rd value of 4th row
         * @param initialM44 defines 4th value of 4th row
         * @param result defines the target matrix
         */
        static FromValuesToRef(initialM11: number, initialM12: number, initialM13: number, initialM14: number, initialM21: number, initialM22: number, initialM23: number, initialM24: number, initialM31: number, initialM32: number, initialM33: number, initialM34: number, initialM41: number, initialM42: number, initialM43: number, initialM44: number, result: Matrix): void;
        /**
         * Gets an identity matrix that must not be updated
         */
        static readonly IdentityReadOnly: Matrix;
        /**
         * Creates new matrix from a list of values (16)
         * @param initialM11 defines 1st value of 1st row
         * @param initialM12 defines 2nd value of 1st row
         * @param initialM13 defines 3rd value of 1st row
         * @param initialM14 defines 4th value of 1st row
         * @param initialM21 defines 1st value of 2nd row
         * @param initialM22 defines 2nd value of 2nd row
         * @param initialM23 defines 3rd value of 2nd row
         * @param initialM24 defines 4th value of 2nd row
         * @param initialM31 defines 1st value of 3rd row
         * @param initialM32 defines 2nd value of 3rd row
         * @param initialM33 defines 3rd value of 3rd row
         * @param initialM34 defines 4th value of 3rd row
         * @param initialM41 defines 1st value of 4th row
         * @param initialM42 defines 2nd value of 4th row
         * @param initialM43 defines 3rd value of 4th row
         * @param initialM44 defines 4th value of 4th row
         * @returns the new matrix
         */
        static FromValues(initialM11: number, initialM12: number, initialM13: number, initialM14: number, initialM21: number, initialM22: number, initialM23: number, initialM24: number, initialM31: number, initialM32: number, initialM33: number, initialM34: number, initialM41: number, initialM42: number, initialM43: number, initialM44: number): Matrix;
        /**
         * Creates a new matrix composed by merging scale (vector3), rotation (quaternion) and translation (vector3)
         * @param scale defines the scale vector3
         * @param rotation defines the rotation quaternion
         * @param translation defines the translation vector3
         * @returns a new matrix
         */
        static Compose(scale: Vector3, rotation: Quaternion, translation: Vector3): Matrix;
        /**
         * Sets a matrix to a value composed by merging scale (vector3), rotation (quaternion) and translation (vector3)
         * @param scale defines the scale vector3
         * @param rotation defines the rotation quaternion
         * @param translation defines the translation vector3
         * @param result defines the target matrix
         */
        static ComposeToRef(scale: Vector3, rotation: Quaternion, translation: Vector3, result: Matrix): void;
        /**
         * Creates a new identity matrix
         * @returns a new identity matrix
         */
        static Identity(): Matrix;
        /**
         * Creates a new identity matrix and stores the result in a given matrix
         * @param result defines the target matrix
         */
        static IdentityToRef(result: Matrix): void;
        /**
         * Creates a new zero matrix
         * @returns a new zero matrix
         */
        static Zero(): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the X axis
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationX(angle: number): Matrix;
        /**
         * Creates a new matrix as the invert of a given matrix
         * @param source defines the source matrix
         * @returns the new matrix
         */
        static Invert(source: Matrix): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the X axis and stores it in a given matrix
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationXToRef(angle: number, result: Matrix): void;
        /**
         * Creates a new rotation matrix for "angle" radians around the Y axis
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationY(angle: number): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the Y axis and stores it in a given matrix
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationYToRef(angle: number, result: Matrix): void;
        /**
         * Creates a new rotation matrix for "angle" radians around the Z axis
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationZ(angle: number): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the Z axis and stores it in a given matrix
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationZToRef(angle: number, result: Matrix): void;
        /**
         * Creates a new rotation matrix for "angle" radians around the given axis
         * @param axis defines the axis to use
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationAxis(axis: Vector3, angle: number): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the given axis and stores it in a given matrix
         * @param axis defines the axis to use
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationAxisToRef(axis: Vector3, angle: number, result: Matrix): void;
        /**
         * Creates a rotation matrix
         * @param yaw defines the yaw angle in radians (Y axis)
         * @param pitch defines the pitch angle in radians (X axis)
         * @param roll defines the roll angle in radians (X axis)
         * @returns the new rotation matrix
         */
        static RotationYawPitchRoll(yaw: number, pitch: number, roll: number): Matrix;
        /**
         * Creates a rotation matrix and stores it in a given matrix
         * @param yaw defines the yaw angle in radians (Y axis)
         * @param pitch defines the pitch angle in radians (X axis)
         * @param roll defines the roll angle in radians (X axis)
         * @param result defines the target matrix
         */
        static RotationYawPitchRollToRef(yaw: number, pitch: number, roll: number, result: Matrix): void;
        /**
         * Creates a scaling matrix
         * @param x defines the scale factor on X axis
         * @param y defines the scale factor on Y axis
         * @param z defines the scale factor on Z axis
         * @returns the new matrix
         */
        static Scaling(x: number, y: number, z: number): Matrix;
        /**
         * Creates a scaling matrix and stores it in a given matrix
         * @param x defines the scale factor on X axis
         * @param y defines the scale factor on Y axis
         * @param z defines the scale factor on Z axis
         * @param result defines the target matrix
         */
        static ScalingToRef(x: number, y: number, z: number, result: Matrix): void;
        /**
         * Creates a translation matrix
         * @param x defines the translation on X axis
         * @param y defines the translation on Y axis
         * @param z defines the translationon Z axis
         * @returns the new matrix
         */
        static Translation(x: number, y: number, z: number): Matrix;
        /**
         * Creates a translation matrix and stores it in a given matrix
         * @param x defines the translation on X axis
         * @param y defines the translation on Y axis
         * @param z defines the translationon Z axis
         * @param result defines the target matrix
         */
        static TranslationToRef(x: number, y: number, z: number, result: Matrix): void;
        /**
         * Returns a new Matrix whose values are the interpolated values for "gradient" (float) between the ones of the matrices "startValue" and "endValue".
         * @param startValue defines the start value
         * @param endValue defines the end value
         * @param gradient defines the gradient factor
         * @returns the new matrix
         */
        static Lerp(startValue: Matrix, endValue: Matrix, gradient: number): Matrix;
        /**
         * Set the given matrix "result" as the interpolated values for "gradient" (float) between the ones of the matrices "startValue" and "endValue".
         * @param startValue defines the start value
         * @param endValue defines the end value
         * @param gradient defines the gradient factor
         * @param result defines the Matrix object where to store data
         */
        static LerpToRef(startValue: Matrix, endValue: Matrix, gradient: number, result: Matrix): void;
        /**
         * Builds a new matrix whose values are computed by:
         * * decomposing the the "startValue" and "endValue" matrices into their respective scale, rotation and translation matrices
         * * interpolating for "gradient" (float) the values between each of these decomposed matrices between the start and the end
         * * recomposing a new matrix from these 3 interpolated scale, rotation and translation matrices
         * @param startValue defines the first matrix
         * @param endValue defines the second matrix
         * @param gradient defines the gradient between the two matrices
         * @returns the new matrix
         */
        static DecomposeLerp(startValue: Matrix, endValue: Matrix, gradient: number): Matrix;
        /**
         * Update a matrix to values which are computed by:
         * * decomposing the the "startValue" and "endValue" matrices into their respective scale, rotation and translation matrices
         * * interpolating for "gradient" (float) the values between each of these decomposed matrices between the start and the end
         * * recomposing a new matrix from these 3 interpolated scale, rotation and translation matrices
         * @param startValue defines the first matrix
         * @param endValue defines the second matrix
         * @param gradient defines the gradient between the two matrices
         * @param result defines the target matrix
         */
        static DecomposeLerpToRef(startValue: Matrix, endValue: Matrix, gradient: number, result: Matrix): void;
        /**
         * Gets a new rotation matrix used to rotate an entity so as it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up"
         * This function works in left handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @returns the new matrix
         */
        static LookAtLH(eye: Vector3, target: Vector3, up: Vector3): Matrix;
        /**
         * Sets the given "result" Matrix to a rotation matrix used to rotate an entity so that it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up".
         * This function works in left handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @param result defines the target matrix
         */
        static LookAtLHToRef(eye: Vector3, target: Vector3, up: Vector3, result: Matrix): void;
        /**
         * Gets a new rotation matrix used to rotate an entity so as it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up"
         * This function works in right handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @returns the new matrix
         */
        static LookAtRH(eye: Vector3, target: Vector3, up: Vector3): Matrix;
        /**
         * Sets the given "result" Matrix to a rotation matrix used to rotate an entity so that it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up".
         * This function works in right handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @param result defines the target matrix
         */
        static LookAtRHToRef(eye: Vector3, target: Vector3, up: Vector3, result: Matrix): void;
        /**
         * Create a left-handed orthographic projection matrix
         * @param width defines the viewport width
         * @param height defines the viewport height
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed orthographic projection matrix
         */
        static OrthoLH(width: number, height: number, znear: number, zfar: number): Matrix;
        /**
         * Store a left-handed orthographic projection to a given matrix
         * @param width defines the viewport width
         * @param height defines the viewport height
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         */
        static OrthoLHToRef(width: number, height: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Create a left-handed orthographic projection matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed orthographic projection matrix
         */
        static OrthoOffCenterLH(left: number, right: number, bottom: number, top: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a left-handed orthographic projection into a given matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         */
        static OrthoOffCenterLHToRef(left: number, right: number, bottom: number, top: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Creates a right-handed orthographic projection matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a right-handed orthographic projection matrix
         */
        static OrthoOffCenterRH(left: number, right: number, bottom: number, top: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a right-handed orthographic projection into a given matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         */
        static OrthoOffCenterRHToRef(left: number, right: number, bottom: number, top: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Creates a left-handed perspective projection matrix
         * @param width defines the viewport width
         * @param height defines the viewport height
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed perspective projection matrix
         */
        static PerspectiveLH(width: number, height: number, znear: number, zfar: number): Matrix;
        /**
         * Creates a left-handed perspective projection matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed perspective projection matrix
         */
        static PerspectiveFovLH(fov: number, aspect: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a left-handed perspective projection into a given matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         * @param isVerticalFovFixed defines it the fov is vertically fixed (default) or horizontally
         */
        static PerspectiveFovLHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Creates a right-handed perspective projection matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a right-handed perspective projection matrix
         */
        static PerspectiveFovRH(fov: number, aspect: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a right-handed perspective projection into a given matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         * @param isVerticalFovFixed defines it the fov is vertically fixed (default) or horizontally
         */
        static PerspectiveFovRHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Stores a perspective projection for WebVR info a given matrix
         * @param fov defines the field of view
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         * @param rightHanded defines if the matrix must be in right-handed mode (false by default)
         */
        static PerspectiveFovWebVRToRef(fov: {
            upDegrees: number;
            downDegrees: number;
            leftDegrees: number;
            rightDegrees: number;
        }, znear: number, zfar: number, result: Matrix, rightHanded?: boolean): void;
        /**
         * Computes a complete transformation matrix
         * @param viewport defines the viewport to use
         * @param world defines the world matrix
         * @param view defines the view matrix
         * @param projection defines the projection matrix
         * @param zmin defines the near clip plane
         * @param zmax defines the far clip plane
         * @returns the transformation matrix
         */
        static GetFinalMatrix(viewport: Viewport, world: Matrix, view: Matrix, projection: Matrix, zmin: number, zmax: number): Matrix;
        /**
         * Extracts a 2x2 matrix from a given matrix and store the result in a Float32Array
         * @param matrix defines the matrix to use
         * @returns a new Float32Array array with 4 elements : the 2x2 matrix extracted from the given matrix
         */
        static GetAsMatrix2x2(matrix: Matrix): Float32Array;
        /**
         * Extracts a 3x3 matrix from a given matrix and store the result in a Float32Array
         * @param matrix defines the matrix to use
         * @returns a new Float32Array array with 9 elements : the 3x3 matrix extracted from the given matrix
         */
        static GetAsMatrix3x3(matrix: Matrix): Float32Array;
        /**
         * Compute the transpose of a given matrix
         * @param matrix defines the matrix to transpose
         * @returns the new matrix
         */
        static Transpose(matrix: Matrix): Matrix;
        /**
         * Compute the transpose of a matrix and store it in a target matrix
         * @param matrix defines the matrix to transpose
         * @param result defines the target matrix
         */
        static TransposeToRef(matrix: Matrix, result: Matrix): void;
        /**
         * Computes a reflection matrix from a plane
         * @param plane defines the reflection plane
         * @returns a new matrix
         */
        static Reflection(plane: Plane): Matrix;
        /**
         * Computes a reflection matrix from a plane
         * @param plane defines the reflection plane
         * @param result defines the target matrix
         */
        static ReflectionToRef(plane: Plane, result: Matrix): void;
        /**
         * Sets the given matrix as a rotation matrix composed from the 3 left handed axes
         * @param xaxis defines the value of the 1st axis
         * @param yaxis defines the value of the 2nd axis
         * @param zaxis defines the value of the 3rd axis
         * @param result defines the target matrix
         */
        static FromXYZAxesToRef(xaxis: Vector3, yaxis: Vector3, zaxis: Vector3, result: Matrix): void;
        /**
         * Creates a rotation matrix from a quaternion and stores it in a target matrix
         * @param quat defines the quaternion to use
         * @param result defines the target matrix
         */
        static FromQuaternionToRef(quat: Quaternion, result: Matrix): void;
    }
    class Plane {
        normal: Vector3;
        d: number;
        /**
         * Creates a Plane object according to the given floats a, b, c, d and the plane equation : ax + by + cz + d = 0
         */
        constructor(a: number, b: number, c: number, d: number);
        /**
         * Returns the plane coordinates as a new array of 4 elements [a, b, c, d].
         */
        asArray(): number[];
        /**
         * Returns a new plane copied from the current Plane.
         */
        clone(): Plane;
        /**
         * Returns the string "Plane".
         */
        getClassName(): string;
        /**
         * Returns the Plane hash code.
         */
        getHashCode(): number;
        /**
         * Normalize the current Plane in place.
         * Returns the updated Plane.
         */
        normalize(): Plane;
        /**
         * Returns a new Plane as the result of the transformation of the current Plane by the given matrix.
         */
        transform(transformation: Matrix): Plane;
        /**
         * Returns the dot product (float) of the point coordinates and the plane normal.
         */
        dotCoordinate(point: Vector3): number;
        /**
         * Updates the current Plane from the plane defined by the three given points.
         * Returns the updated Plane.
         */
        copyFromPoints(point1: Vector3, point2: Vector3, point3: Vector3): Plane;
        /**
         * Boolean : True is the vector "direction"  is the same side than the plane normal.
         */
        isFrontFacingTo(direction: Vector3, epsilon: number): boolean;
        /**
         * Returns the signed distance (float) from the given point to the Plane.
         */
        signedDistanceTo(point: Vector3): number;
        /**
         * Returns a new Plane from the given array.
         */
        static FromArray(array: ArrayLike<number>): Plane;
        /**
         * Returns a new Plane defined by the three given points.
         */
        static FromPoints(point1: Vector3, point2: Vector3, point3: Vector3): Plane;
        /**
         * Returns a new Plane the normal vector to this plane at the given origin point.
         * Note : the vector "normal" is updated because normalized.
         */
        static FromPositionAndNormal(origin: Vector3, normal: Vector3): Plane;
        /**
         * Returns the signed distance between the plane defined by the normal vector at the "origin"" point and the given other point.
         */
        static SignedDistanceToPlaneFromPositionAndNormal(origin: Vector3, normal: Vector3, point: Vector3): number;
    }
    class Viewport {
        x: number;
        y: number;
        width: number;
        height: number;
        /**
         * Creates a Viewport object located at (x, y) and sized (width, height).
         */
        constructor(x: number, y: number, width: number, height: number);
        toGlobal(renderWidthOrEngine: number | Engine, renderHeight: number): Viewport;
        /**
         * Returns a new Viewport copied from the current one.
         */
        clone(): Viewport;
    }
    class Frustum {
        /**
         * Returns a new array of 6 Frustum planes computed by the given transformation matrix.
         */
        static GetPlanes(transform: Matrix): Plane[];
        static GetNearPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetFarPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetLeftPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetRightPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetTopPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetBottomPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        /**
         * Sets the given array "frustumPlanes" with the 6 Frustum planes computed by the given transformation matrix.
         */
        static GetPlanesToRef(transform: Matrix, frustumPlanes: Plane[]): void;
    }
    /** Defines supported spaces */
    enum Space {
        /** Local (object) space */
        LOCAL = 0,
        /** World space */
        WORLD = 1,
        /** Bone space */
        BONE = 2,
    }
    /** Defines the 3 main axes */
    class Axis {
        /** X axis */
        static X: Vector3;
        /** Y axis */
        static Y: Vector3;
        /** Z axis */
        static Z: Vector3;
    }
    class BezierCurve {
        /**
         * Returns the cubic Bezier interpolated value (float) at "t" (float) from the given x1, y1, x2, y2 floats.
         */
        static interpolate(t: number, x1: number, y1: number, x2: number, y2: number): number;
    }
    /**
     * Defines potential orientation for back face culling
     */
    enum Orientation {
        /**
         * Clockwise
         */
        CW = 0,
        /** Counter clockwise */
        CCW = 1,
    }
    /**
     * Defines angle representation
     */
    class Angle {
        private _radians;
        /**
         * Creates an Angle object of "radians" radians (float).
         */
        constructor(radians: number);
        /**
         * Get value in degrees
         * @returns the Angle value in degrees (float)
         */
        degrees(): number;
        /**
         * Get value in radians
         * @returns the Angle value in radians (float)
         */
        radians(): number;
        /**
         * Gets a new Angle object valued with the angle value in radians between the two given vectors
         * @param a defines first vector
         * @param b defines second vector
         * @returns a new Angle
         */
        static BetweenTwoPoints(a: Vector2, b: Vector2): Angle;
        /**
         * Gets a new Angle object from the given float in radians
         * @param radians defines the angle value in radians
         * @returns a new Angle
         */
        static FromRadians(radians: number): Angle;
        /**
         * Gets a new Angle object from the given float in degrees
         * @param degrees defines the angle value in degrees
         * @returns a new Angle
         */
        static FromDegrees(degrees: number): Angle;
    }
    class Arc2 {
        startPoint: Vector2;
        midPoint: Vector2;
        endPoint: Vector2;
        centerPoint: Vector2;
        radius: number;
        angle: Angle;
        startAngle: Angle;
        orientation: Orientation;
        /**
         * Creates an Arc object from the three given points : start, middle and end.
         */
        constructor(startPoint: Vector2, midPoint: Vector2, endPoint: Vector2);
    }
    class Path2 {
        private _points;
        private _length;
        closed: boolean;
        /**
         * Creates a Path2 object from the starting 2D coordinates x and y.
         */
        constructor(x: number, y: number);
        /**
         * Adds a new segment until the given coordinates (x, y) to the current Path2.
         * Returns the updated Path2.
         */
        addLineTo(x: number, y: number): Path2;
        /**
         * Adds _numberOfSegments_ segments according to the arc definition (middle point coordinates, end point coordinates, the arc start point being the current Path2 last point) to the current Path2.
         * Returns the updated Path2.
         */
        addArcTo(midX: number, midY: number, endX: number, endY: number, numberOfSegments?: number): Path2;
        /**
         * Closes the Path2.
         * Returns the Path2.
         */
        close(): Path2;
        /**
         * Returns the Path2 total length (float).
         */
        length(): number;
        /**
         * Returns the Path2 internal array of points.
         */
        getPoints(): Vector2[];
        /**
         * Returns a new Vector2 located at a percentage of the Path2 total length on this path.
         */
        getPointAtLengthPosition(normalizedLengthPosition: number): Vector2;
        /**
         * Returns a new Path2 starting at the coordinates (x, y).
         */
        static StartingAt(x: number, y: number): Path2;
    }
    class Path3D {
        path: Vector3[];
        private _curve;
        private _distances;
        private _tangents;
        private _normals;
        private _binormals;
        private _raw;
        /**
        * new Path3D(path, normal, raw)
        * Creates a Path3D. A Path3D is a logical math object, so not a mesh.
        * please read the description in the tutorial :  http://doc.babylonjs.com/tutorials/How_to_use_Path3D
        * path : an array of Vector3, the curve axis of the Path3D
        * normal (optional) : Vector3, the first wanted normal to the curve. Ex (0, 1, 0) for a vertical normal.
        * raw (optional, default false) : boolean, if true the returned Path3D isn't normalized. Useful to depict path acceleration or speed.
        */
        constructor(path: Vector3[], firstNormal?: Nullable<Vector3>, raw?: boolean);
        /**
         * Returns the Path3D array of successive Vector3 designing its curve.
         */
        getCurve(): Vector3[];
        /**
         * Returns an array populated with tangent vectors on each Path3D curve point.
         */
        getTangents(): Vector3[];
        /**
         * Returns an array populated with normal vectors on each Path3D curve point.
         */
        getNormals(): Vector3[];
        /**
         * Returns an array populated with binormal vectors on each Path3D curve point.
         */
        getBinormals(): Vector3[];
        /**
         * Returns an array populated with distances (float) of the i-th point from the first curve point.
         */
        getDistances(): number[];
        /**
         * Forces the Path3D tangent, normal, binormal and distance recomputation.
         * Returns the same object updated.
         */
        update(path: Vector3[], firstNormal?: Nullable<Vector3>): Path3D;
        private _compute(firstNormal);
        private _getFirstNonNullVector(index);
        private _getLastNonNullVector(index);
        private _normalVector(v0, vt, va);
    }
    class Curve3 {
        private _points;
        private _length;
        /**
         * Returns a Curve3 object along a Quadratic Bezier curve : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#quadratic-bezier-curve
         * @param v0 (Vector3) the origin point of the Quadratic Bezier
         * @param v1 (Vector3) the control point
         * @param v2 (Vector3) the end point of the Quadratic Bezier
         * @param nbPoints (integer) the wanted number of points in the curve
         */
        static CreateQuadraticBezier(v0: Vector3, v1: Vector3, v2: Vector3, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a Cubic Bezier curve : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#cubic-bezier-curve
         * @param v0 (Vector3) the origin point of the Cubic Bezier
         * @param v1 (Vector3) the first control point
         * @param v2 (Vector3) the second control point
         * @param v3 (Vector3) the end point of the Cubic Bezier
         * @param nbPoints (integer) the wanted number of points in the curve
         */
        static CreateCubicBezier(v0: Vector3, v1: Vector3, v2: Vector3, v3: Vector3, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a Hermite Spline curve : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#hermite-spline
         * @param p1 (Vector3) the origin point of the Hermite Spline
         * @param t1 (Vector3) the tangent vector at the origin point
         * @param p2 (Vector3) the end point of the Hermite Spline
         * @param t2 (Vector3) the tangent vector at the end point
         * @param nbPoints (integer) the wanted number of points in the curve
         */
        static CreateHermiteSpline(p1: Vector3, t1: Vector3, p2: Vector3, t2: Vector3, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a CatmullRom Spline curve :
         * @param points (array of Vector3) the points the spline must pass through. At least, four points required.
         * @param nbPoints (integer) the wanted number of points between each curve control points.
         */
        static CreateCatmullRomSpline(points: Vector3[], nbPoints: number): Curve3;
        /**
         * A Curve3 object is a logical object, so not a mesh, to handle curves in the 3D geometric space.
         * A Curve3 is designed from a series of successive Vector3.
         * Tuto : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#curve3-object
         */
        constructor(points: Vector3[]);
        /**
         * Returns the Curve3 stored array of successive Vector3
         */
        getPoints(): Vector3[];
        /**
         * Returns the computed length (float) of the curve.
         */
        length(): number;
        /**
         * Returns a new instance of Curve3 object : var curve = curveA.continue(curveB);
         * This new Curve3 is built by translating and sticking the curveB at the end of the curveA.
         * curveA and curveB keep unchanged.
         */
        continue(curve: Curve3): Curve3;
        private _computeLength(path);
    }
    class PositionNormalVertex {
        position: Vector3;
        normal: Vector3;
        constructor(position?: Vector3, normal?: Vector3);
        clone(): PositionNormalVertex;
    }
    class PositionNormalTextureVertex {
        position: Vector3;
        normal: Vector3;
        uv: Vector2;
        constructor(position?: Vector3, normal?: Vector3, uv?: Vector2);
        clone(): PositionNormalTextureVertex;
    }
    class Tmp {
        static Color3: Color3[];
        static Vector2: Vector2[];
        static Vector3: Vector3[];
        static Vector4: Vector4[];
        static Quaternion: Quaternion[];
        static Matrix: Matrix[];
    }
}

declare module BABYLON {
    class SphericalPolynomial {
        x: Vector3;
        y: Vector3;
        z: Vector3;
        xx: Vector3;
        yy: Vector3;
        zz: Vector3;
        xy: Vector3;
        yz: Vector3;
        zx: Vector3;
        addAmbient(color: Color3): void;
        static getSphericalPolynomialFromHarmonics(harmonics: SphericalHarmonics): SphericalPolynomial;
        scale(scale: number): void;
    }
    class SphericalHarmonics {
        L00: Vector3;
        L1_1: Vector3;
        L10: Vector3;
        L11: Vector3;
        L2_2: Vector3;
        L2_1: Vector3;
        L20: Vector3;
        L21: Vector3;
        L22: Vector3;
        addLight(direction: Vector3, color: Color3, deltaSolidAngle: number): void;
        scale(scale: number): void;
        convertIncidentRadianceToIrradiance(): void;
        convertIrradianceToLambertianRadiance(): void;
        static getsphericalHarmonicsFromPolynomial(polynomial: SphericalPolynomial): SphericalHarmonics;
    }
}

declare module BABYLON {
    /**
     * Defines a target to use with MorphTargetManager
     * @see http://doc.babylonjs.com/how_to/how_to_use_morphtargets
     */
    class MorphTarget implements IAnimatable {
        /** defines the name of the target */
        name: string;
        /**
         * Gets or sets the list of animations
         */
        animations: Animation[];
        private _scene;
        private _positions;
        private _normals;
        private _tangents;
        private _influence;
        /**
         * Observable raised when the influence changes
         */
        onInfluenceChanged: Observable<boolean>;
        /**
         * Gets or sets the influence of this target (ie. its weight in the overall morphing)
         */
        influence: number;
        private _animationPropertiesOverride;
        /**
         * Gets or sets the animation properties override
         */
        animationPropertiesOverride: Nullable<AnimationPropertiesOverride>;
        /**
         * Creates a new MorphTarget
         * @param name defines the name of the target
         * @param influence defines the influence to use
         */
        constructor(
            /** defines the name of the target */
            name: string, influence?: number, scene?: Nullable<Scene>);
        /**
         * Gets a boolean defining if the target contains position data
         */
        readonly hasPositions: boolean;
        /**
         * Gets a boolean defining if the target contains normal data
         */
        readonly hasNormals: boolean;
        /**
         * Gets a boolean defining if the target contains tangent data
         */
        readonly hasTangents: boolean;
        /**
         * Affects position data to this target
         * @param data defines the position data to use
         */
        setPositions(data: Nullable<FloatArray>): void;
        /**
         * Gets the position data stored in this target
         * @returns a FloatArray containing the position data (or null if not present)
         */
        getPositions(): Nullable<FloatArray>;
        /**
         * Affects normal data to this target
         * @param data defines the normal data to use
         */
        setNormals(data: Nullable<FloatArray>): void;
        /**
         * Gets the normal data stored in this target
         * @returns a FloatArray containing the normal data (or null if not present)
         */
        getNormals(): Nullable<FloatArray>;
        /**
         * Affects tangent data to this target
         * @param data defines the tangent data to use
         */
        setTangents(data: Nullable<FloatArray>): void;
        /**
         * Gets the tangent data stored in this target
         * @returns a FloatArray containing the tangent data (or null if not present)
         */
        getTangents(): Nullable<FloatArray>;
        /**
         * Serializes the current target into a Serialization object
         * @returns the serialized object
         */
        serialize(): any;
        /**
         * Creates a new target from serialized data
         * @param serializationObject defines the serialized data to use
         * @returns a new MorphTarget
         */
        static Parse(serializationObject: any): MorphTarget;
        /**
         * Creates a MorphTarget from mesh data
         * @param mesh defines the source mesh
         * @param name defines the name to use for the new target
         * @param influence defines the influence to attach to the target
         * @returns a new MorphTarget
         */
        static FromMesh(mesh: AbstractMesh, name?: string, influence?: number): MorphTarget;
    }
}

declare module BABYLON {
    /**
     * This class is used to deform meshes using morphing between different targets
     * @see http://doc.babylonjs.com/how_to/how_to_use_morphtargets
     */
    class MorphTargetManager {
        private _targets;
        private _targetObservable;
        private _activeTargets;
        private _scene;
        private _influences;
        private _supportsNormals;
        private _supportsTangents;
        private _vertexCount;
        private _uniqueId;
        private _tempInfluences;
        /**
         * Creates a new MorphTargetManager
         * @param scene defines the current scene
         */
        constructor(scene?: Nullable<Scene>);
        /**
         * Gets the unique ID of this manager
         */
        readonly uniqueId: number;
        /**
         * Gets the number of vertices handled by this manager
         */
        readonly vertexCount: number;
        /**
         * Gets a boolean indicating if this manager supports morphing of normals
         */
        readonly supportsNormals: boolean;
        /**
         * Gets a boolean indicating if this manager supports morphing of tangents
         */
        readonly supportsTangents: boolean;
        /**
         * Gets the number of targets stored in this manager
         */
        readonly numTargets: number;
        /**
         * Gets the number of influencers (ie. the number of targets with influences > 0)
         */
        readonly numInfluencers: number;
        /**
         * Gets the list of influences (one per target)
         */
        readonly influences: Float32Array;
        /**
         * Gets the active target at specified index. An active target is a target with an influence > 0
         * @param index defines the index to check
         * @returns the requested target
         */
        getActiveTarget(index: number): MorphTarget;
        /**
         * Gets the target at specified index
         * @param index defines the index to check
         * @returns the requested target
         */
        getTarget(index: number): MorphTarget;
        /**
         * Add a new target to this manager
         * @param target defines the target to add
         */
        addTarget(target: MorphTarget): void;
        /**
         * Removes a target from the manager
         * @param target defines the target to remove
         */
        removeTarget(target: MorphTarget): void;
        /**
         * Serializes the current manager into a Serialization object
         * @returns the serialized object
         */
        serialize(): any;
        private _syncActiveTargets(needUpdate);
        /**
         * Syncrhonize the targets with all the meshes using this morph target manager
         */
        synchronize(): void;
        /**
         * Creates a new MorphTargetManager from serialized data
         * @param serializationObject defines the serialized data
         * @param scene defines the hosting scene
         * @returns the new MorphTargetManager
         */
        static Parse(serializationObject: any, scene: Scene): MorphTargetManager;
    }
}

declare module BABYLON {
    /**
     * Class used to store all common mesh properties
     */
    class AbstractMesh extends TransformNode implements IDisposable, ICullable, IGetSetVerticesData {
        /** No occlusion */
        static OCCLUSION_TYPE_NONE: number;
        /** Occlusion set to optimisitic */
        static OCCLUSION_TYPE_OPTIMISTIC: number;
        /** Occlusion set to strict */
        static OCCLUSION_TYPE_STRICT: number;
        /** Use an accurante occlusion algorithm */
        static OCCLUSION_ALGORITHM_TYPE_ACCURATE: number;
        /** Use a conservative occlusion algorithm */
        static OCCLUSION_ALGORITHM_TYPE_CONSERVATIVE: number;
        /**
         * No billboard
         */
        static readonly BILLBOARDMODE_NONE: number;
        /** Billboard on X axis */
        static readonly BILLBOARDMODE_X: number;
        /** Billboard on Y axis */
        static readonly BILLBOARDMODE_Y: number;
        /** Billboard on Z axis */
        static readonly BILLBOARDMODE_Z: number;
        /** Billboard on all axes */
        static readonly BILLBOARDMODE_ALL: number;
        private _facetPositions;
        private _facetNormals;
        private _facetPartitioning;
        private _facetNb;
        private _partitioningSubdivisions;
        private _partitioningBBoxRatio;
        private _facetDataEnabled;
        private _facetParameters;
        private _bbSize;
        private _subDiv;
        private _facetDepthSort;
        private _facetDepthSortEnabled;
        private _depthSortedIndices;
        private _depthSortedFacets;
        private _facetDepthSortFunction;
        private _facetDepthSortFrom;
        private _facetDepthSortOrigin;
        private _invertedMatrix;
        /**
         * Gets the number of facets in the mesh
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata#what-is-a-mesh-facet
         */
        readonly facetNb: number;
        /**
         * Gets or set the number (integer) of subdivisions per axis in the partioning space
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata#tweaking-the-partitioning
         */
        partitioningSubdivisions: number;
        /**
         * The ratio (float) to apply to the bouding box size to set to the partioning space.
         * Ex : 1.01 (default) the partioning space is 1% bigger than the bounding box
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata#tweaking-the-partitioning
         */
        partitioningBBoxRatio: number;
        /**
         * Gets or sets a boolean indicating that the facets must be depth sorted on next call to `updateFacetData()`.
         * Works only for updatable meshes.
         * Doesn't work with multi-materials
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata#facet-depth-sort
         */
        mustDepthSortFacets: boolean;
        /**
         * The location (Vector3) where the facet depth sort must be computed from.
         * By default, the active camera position.
         * Used only when facet depth sort is enabled
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata#facet-depth-sort
         */
        facetDepthSortFrom: Vector3;
        /**
         * gets a boolean indicating if facetData is enabled
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata#what-is-a-mesh-facet
         */
        readonly isFacetDataEnabled: boolean;
        /** @hidden */
        _updateNonUniformScalingState(value: boolean): boolean;
        /**
        * An event triggered when this mesh collides with another one
        */
        onCollideObservable: Observable<AbstractMesh>;
        private _onCollideObserver;
        /** Set a function to call when this mesh collides with another one */
        onCollide: () => void;
        /**
        * An event triggered when the collision's position changes
        */
        onCollisionPositionChangeObservable: Observable<Vector3>;
        private _onCollisionPositionChangeObserver;
        /** Set a function to call when the collision's position changes */
        onCollisionPositionChange: () => void;
        /**
        * An event triggered when material is changed
        */
        onMaterialChangedObservable: Observable<AbstractMesh>;
        /**
         * Gets or sets the orientation for POV movement & rotation
         */
        definedFacingForward: boolean;
        /**
        * This property determines the type of occlusion query algorithm to run in WebGl, you can use:
        * * AbstractMesh.OCCLUSION_ALGORITHM_TYPE_ACCURATE which is mapped to GL_ANY_SAMPLES_PASSED.
        * * AbstractMesh.OCCLUSION_ALGORITHM_TYPE_CONSERVATIVE (Default Value) which is mapped to GL_ANY_SAMPLES_PASSED_CONSERVATIVE which is a false positive algorithm that is faster than GL_ANY_SAMPLES_PASSED but less accurate.
        * @see http://doc.babylonjs.com/features/occlusionquery
        */
        occlusionQueryAlgorithmType: number;
        /**
         * This property is responsible for starting the occlusion query within the Mesh or not, this property is also used to determine what should happen when the occlusionRetryCount is reached. It has supports 3 values:
         * * OCCLUSION_TYPE_NONE (Default Value): this option means no occlusion query whith the Mesh.
         * * OCCLUSION_TYPE_OPTIMISTIC: this option is means use occlusion query and if occlusionRetryCount is reached and the query is broken show the mesh.
         * * OCCLUSION_TYPE_STRICT: this option is means use occlusion query and if occlusionRetryCount is reached and the query is broken restore the last state of the mesh occlusion if the mesh was visible then show the mesh if was hidden then hide don't show.
         * @see http://doc.babylonjs.com/features/occlusionquery
         */
        occlusionType: number;
        /**
        * This number indicates the number of allowed retries before stop the occlusion query, this is useful if the occlusion query is taking long time before to the query result is retireved, the query result indicates if the object is visible within the scene or not and based on that Babylon.Js engine decideds to show or hide the object.
        * The default value is -1 which means don't break the query and wait till the result
        * @see http://doc.babylonjs.com/features/occlusionquery
        */
        occlusionRetryCount: number;
        private _occlusionInternalRetryCounter;
        protected _isOccluded: boolean;
        /**
        * Gets or sets whether the mesh is occluded or not, it is used also to set the intial state of the mesh to be occluded or not
        * @see http://doc.babylonjs.com/features/occlusionquery
        */
        isOccluded: boolean;
        private _isOcclusionQueryInProgress;
        /**
         * Flag to check the progress status of the query
         * @see http://doc.babylonjs.com/features/occlusionquery
         */
        readonly isOcclusionQueryInProgress: boolean;
        private _occlusionQuery;
        private _visibility;
        /**
         * Gets or sets mesh visibility between 0 and 1 (default is 1)
         */
        /**
         * Gets or sets mesh visibility between 0 and 1 (default is 1)
         */
        visibility: number;
        /** Gets or sets the alpha index used to sort transparent meshes
         * @see http://doc.babylonjs.com/resources/transparency_and_how_meshes_are_rendered#alpha-index
         */
        alphaIndex: number;
        /**
         * Gets or sets a boolean indicating if the mesh is visible (renderable). Default is true
         */
        isVisible: boolean;
        /**
         * Gets or sets a boolean indicating if the mesh can be picked (by scene.pick for instance or through actions). Default is true
         */
        isPickable: boolean;
        /**
         * Gets or sets a boolean indicating if the bounding box must be rendered as well (false by default)
         */
        showBoundingBox: boolean;
        /** Gets or sets a boolean indicating that bounding boxes of subMeshes must be rendered as well (false by default) */
        showSubMeshesBoundingBox: boolean;
        /** Gets or sets a boolean indicating if the mesh must be considered as a ray blocker for lens flares (false by default)
         * @see http://doc.babylonjs.com/how_to/how_to_use_lens_flares
         */
        isBlocker: boolean;
        /**
         * Gets or sets a boolean indicating that pointer move events must be supported on this mesh (false by default)
         */
        enablePointerMoveEvents: boolean;
        /**
         * Specifies the rendering group id for this mesh (0 by default)
         * @see http://doc.babylonjs.com/resources/transparency_and_how_meshes_are_rendered#rendering-groups
         */
        renderingGroupId: number;
        private _material;
        /** Gets or sets current material */
        material: Nullable<Material>;
        private _receiveShadows;
        /**
         * Gets or sets a boolean indicating that this mesh can receive realtime shadows
         * @see http://doc.babylonjs.com/babylon101/shadows
         */
        receiveShadows: boolean;
        /**
         * Gets or sets a boolean indicating if the outline must be rendered as well
         * @see https://www.babylonjs-playground.com/#10WJ5S#3
         */
        renderOutline: boolean;
        /** Defines color to use when rendering outline */
        outlineColor: Color3;
        /** Define width to use when rendering outline */
        outlineWidth: number;
        /**
         * Gets or sets a boolean indicating if the overlay must be rendered as well
         * @see https://www.babylonjs-playground.com/#10WJ5S#2
         */
        renderOverlay: boolean;
        /** Defines color to use when rendering overlay */
        overlayColor: Color3;
        /** Defines alpha to use when rendering overlay */
        overlayAlpha: number;
        private _hasVertexAlpha;
        /** Gets or sets a boolean indicating that this mesh contains vertex color data with alpha values */
        hasVertexAlpha: boolean;
        private _useVertexColors;
        /** Gets or sets a boolean indicating that this mesh needs to use vertex color data to render (if this kind of vertex data is available in the geometry) */
        useVertexColors: boolean;
        private _computeBonesUsingShaders;
        /**
         * Gets or sets a boolean indicating that bone animations must be computed by the CPU (false by default)
         */
        computeBonesUsingShaders: boolean;
        private _numBoneInfluencers;
        /** Gets or sets the number of allowed bone influences per vertex (4 by default) */
        numBoneInfluencers: number;
        private _applyFog;
        /** Gets or sets a boolean indicating that this mesh will allow fog to be rendered on it (true by default) */
        applyFog: boolean;
        /** Gets or sets a boolean indicating that internal octree (if available) can be used to boost submeshes selection (true by default) */
        useOctreeForRenderingSelection: boolean;
        /** Gets or sets a boolean indicating that internal octree (if available) can be used to boost submeshes picking (true by default) */
        useOctreeForPicking: boolean;
        /** Gets or sets a boolean indicating that internal octree (if available) can be used to boost submeshes collision (true by default) */
        useOctreeForCollisions: boolean;
        private _layerMask;
        /**
         * Gets or sets the current layer mask (default is 0x0FFFFFFF)
         * @see http://doc.babylonjs.com/how_to/layermasks_and_multi-cam_textures
         */
        layerMask: number;
        /**
         * True if the mesh must be rendered in any case (this will shortcut the frustum clipping phase)
         */
        alwaysSelectAsActiveMesh: boolean;
        /**
         * Gets or sets the current action manager
         * @see http://doc.babylonjs.com/how_to/how_to_use_actions
         */
        actionManager: Nullable<ActionManager>;
        /**
         * Gets or sets impostor used for physic simulation
         * @see http://doc.babylonjs.com/features/physics_engine
         */
        physicsImpostor: Nullable<PhysicsImpostor>;
        private _checkCollisions;
        private _collisionMask;
        private _collisionGroup;
        /**
         * Gets or sets the ellipsoid used to impersonate this mesh when using collision engine (default is (0.5, 1, 0.5))
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
         */
        ellipsoid: Vector3;
        /**
         * Gets or sets the ellipsoid offset used to impersonate this mesh when using collision engine (default is (0, 0, 0))
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
         */
        ellipsoidOffset: Vector3;
        private _collider;
        private _oldPositionForCollisions;
        private _diffPositionForCollisions;
        /**
         * Gets or sets a collision mask used to mask collisions (default is -1).
         * A collision between A and B will happen if A.collisionGroup & b.collisionMask !== 0
         */
        collisionMask: number;
        /**
         * Gets or sets the current collision group mask (-1 by default).
         * A collision between A and B will happen if A.collisionGroup & b.collisionMask !== 0
         */
        collisionGroup: number;
        /**
         * Defines edge width used when edgesRenderer is enabled
         * @see https://www.babylonjs-playground.com/#10OJSG#13
         */
        edgesWidth: number;
        /**
         * Defines edge color used when edgesRenderer is enabled
         * @see https://www.babylonjs-playground.com/#10OJSG#13
         */
        edgesColor: Color4;
        /** @hidden */
        _edgesRenderer: Nullable<EdgesRenderer>;
        private _collisionsTransformMatrix;
        private _collisionsScalingMatrix;
        /** @hidden */
        _masterMesh: Nullable<AbstractMesh>;
        /** @hidden */
        _boundingInfo: Nullable<BoundingInfo>;
        /** @hidden */
        _renderId: number;
        /**
         * Gets or sets the list of subMeshes
         * @see http://doc.babylonjs.com/how_to/multi_materials
         */
        subMeshes: SubMesh[];
        /** @hidden */
        _submeshesOctree: Octree<SubMesh>;
        /** @hidden */
        _intersectionsInProgress: AbstractMesh[];
        /** @hidden */
        _unIndexed: boolean;
        /** @hidden */
        _lightSources: Light[];
        /** @hidden */
        readonly _positions: Nullable<Vector3[]>;
        /** @hidden */
        _waitingActions: any;
        /** @hidden */
        _waitingFreezeWorldMatrix: Nullable<boolean>;
        private _skeleton;
        /** @hidden */
        _bonesTransformMatrices: Nullable<Float32Array>;
        /**
         * Gets or sets a skeleton to apply skining transformations
         * @see http://doc.babylonjs.com/how_to/how_to_use_bones_and_skeletons
         */
        skeleton: Nullable<Skeleton>;
        /**
         * Creates a new AbstractMesh
         * @param name defines the name of the mesh
         * @param scene defines the hosting scene
         */
        constructor(name: string, scene?: Nullable<Scene>);
        /**
         * Returns the string "AbstractMesh"
         * @returns "AbstractMesh"
         */
        getClassName(): string;
        /**
         * Gets a string representation of the current mesh
         * @param fullDetails defines a boolean indicating if full details must be included
         * @returns a string representation of the current mesh
         */
        toString(fullDetails?: boolean): string;
        /** @hidden */
        _rebuild(): void;
        /** @hidden */
        _resyncLightSources(): void;
        /** @hidden */
        _resyncLighSource(light: Light): void;
        /** @hidden */
        _unBindEffect(): void;
        /** @hidden */
        _removeLightSource(light: Light): void;
        private _markSubMeshesAsDirty(func);
        /** @hidden */
        _markSubMeshesAsLightDirty(): void;
        /** @hidden */
        _markSubMeshesAsAttributesDirty(): void;
        /** @hidden */
        _markSubMeshesAsMiscDirty(): void;
        /**
        * Gets or sets a Vector3 depicting the mesh scaling along each local axis X, Y, Z.  Default is (1.0, 1.0, 1.0)
        */
        scaling: Vector3;
        /**
         * Disables the mesh edge rendering mode
         * @returns the currentAbstractMesh
         */
        disableEdgesRendering(): AbstractMesh;
        /**
         * Enables the edge rendering mode on the mesh.
         * This mode makes the mesh edges visible
         * @param epsilon defines the maximal distance between two angles to detect a face
         * @param checkVerticesInsteadOfIndices indicates that we should check vertex list directly instead of faces
         * @returns the currentAbstractMesh
         * @see https://www.babylonjs-playground.com/#19O9TU#0
         */
        enableEdgesRendering(epsilon?: number, checkVerticesInsteadOfIndices?: boolean): AbstractMesh;
        /**
         * Returns true if the mesh is blocked. Implemented by child classes
         */
        readonly isBlocked: boolean;
        /**
         * Returns the mesh itself by default. Implemented by child classes
         * @param camera defines the camera to use to pick the right LOD level
         * @returns the currentAbstractMesh
         */
        getLOD(camera: Camera): AbstractMesh;
        /**
         * Returns 0 by default. Implemented by child classes
         * @returns an integer
         */
        getTotalVertices(): number;
        /**
         * Returns null by default. Implemented by child classes
         * @returns null
         */
        getIndices(): Nullable<IndicesArray>;
        /**
         * Returns the array of the requested vertex data kind. Implemented by child classes
         * @param kind defines the vertex data kind to use
         * @returns null
         */
        getVerticesData(kind: string): Nullable<FloatArray>;
        /**
         * Sets the vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, a new Geometry object is set to the mesh and then passed this vertex data.
         * Note that a new underlying VertexBuffer object is created each call.
         * If the `kind` is the `PositionKind`, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         * @param kind defines vertex data kind:
         * * BABYLON.VertexBuffer.PositionKind
         * * BABYLON.VertexBuffer.UVKind
         * * BABYLON.VertexBuffer.UV2Kind
         * * BABYLON.VertexBuffer.UV3Kind
         * * BABYLON.VertexBuffer.UV4Kind
         * * BABYLON.VertexBuffer.UV5Kind
         * * BABYLON.VertexBuffer.UV6Kind
         * * BABYLON.VertexBuffer.ColorKind
         * * BABYLON.VertexBuffer.MatricesIndicesKind
         * * BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * * BABYLON.VertexBuffer.MatricesWeightsKind
         * * BABYLON.VertexBuffer.MatricesWeightsExtraKind
         * @param data defines the data source
         * @param updatable defines if the data must be flagged as updatable (or static)
         * @param stride defines the vertex stride (size of an entire vertex). Can be null and in this case will be deduced from vertex data kind
         * @returns the current mesh
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): AbstractMesh;
        /**
         * Updates the existing vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, it is simply returned as it is.
         * @param kind defines vertex data kind:
         * * BABYLON.VertexBuffer.PositionKind
         * * BABYLON.VertexBuffer.UVKind
         * * BABYLON.VertexBuffer.UV2Kind
         * * BABYLON.VertexBuffer.UV3Kind
         * * BABYLON.VertexBuffer.UV4Kind
         * * BABYLON.VertexBuffer.UV5Kind
         * * BABYLON.VertexBuffer.UV6Kind
         * * BABYLON.VertexBuffer.ColorKind
         * * BABYLON.VertexBuffer.MatricesIndicesKind
         * * BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * * BABYLON.VertexBuffer.MatricesWeightsKind
         * * BABYLON.VertexBuffer.MatricesWeightsExtraKind
         * @param data defines the data source
         * @param updateExtends If `kind` is `PositionKind` and if `updateExtends` is true, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed
         * @param makeItUnique If true, a new global geometry is created from this data and is set to the mesh
         * @returns the current mesh
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): AbstractMesh;
        /**
         * Sets the mesh indices,
         * If the mesh has no geometry, a new Geometry object is created and set to the mesh.
         * @param indices Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array)
         * @param totalVertices Defines the total number of vertices
         * @returns the current mesh
         */
        setIndices(indices: IndicesArray, totalVertices: Nullable<number>): AbstractMesh;
        /**
         * Gets a boolean indicating if specific vertex data is present
         * @param kind defines the vertex data kind to use
         * @returns true is data kind is present
         */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Returns the mesh BoundingInfo object or creates a new one and returns if it was undefined
         * @returns a BoundingInfo
         */
        getBoundingInfo(): BoundingInfo;
        /**
         * Uniformly scales the mesh to fit inside of a unit cube (1 X 1 X 1 units)
         * @param includeDescendants Use the hierarchy's bounding box instead of the mesh's bounding box
         * @returns the current mesh
         */
        normalizeToUnitCube(includeDescendants?: boolean): AbstractMesh;
        /**
         * Overwrite the current bounding info
         * @param boundingInfo defines the new bounding info
         * @returns the current mesh
         */
        setBoundingInfo(boundingInfo: BoundingInfo): AbstractMesh;
        /** Gets a boolean indicating if this mesh has skinning data and an attached skeleton */
        readonly useBones: boolean;
        /** @hidden */
        _preActivate(): void;
        /** @hidden */
        _preActivateForIntermediateRendering(renderId: number): void;
        /** @hidden */
        _activate(renderId: number): void;
        /**
         * Gets the current world matrix
         * @returns a Matrix
         */
        getWorldMatrix(): Matrix;
        /** @hidden */
        _getWorldMatrixDeterminant(): number;
        /**
         * Perform relative position change from the point of view of behind the front of the mesh.
         * This is performed taking into account the meshes current rotation, so you do not have to care.
         * Supports definition of mesh facing forward or backward
         * @param amountRight defines the distance on the right axis
         * @param amountUp defines the distance on the up axis
         * @param amountForward defines the distance on the forward axis
         * @returns the current mesh
         */
        movePOV(amountRight: number, amountUp: number, amountForward: number): AbstractMesh;
        /**
         * Calculate relative position change from the point of view of behind the front of the mesh.
         * This is performed taking into account the meshes current rotation, so you do not have to care.
         * Supports definition of mesh facing forward or backward
         * @param amountRight defines the distance on the right axis
         * @param amountUp defines the distance on the up axis
         * @param amountForward defines the distance on the forward axis
         * @returns the new displacement vector
         */
        calcMovePOV(amountRight: number, amountUp: number, amountForward: number): Vector3;
        /**
         * Perform relative rotation change from the point of view of behind the front of the mesh.
         * Supports definition of mesh facing forward or backward
         * @param flipBack defines the flip
         * @param twirlClockwise defines the twirl
         * @param tiltRight defines the tilt
         * @returns the current mesh
         */
        rotatePOV(flipBack: number, twirlClockwise: number, tiltRight: number): AbstractMesh;
        /**
         * Calculate relative rotation change from the point of view of behind the front of the mesh.
         * Supports definition of mesh facing forward or backward.
         * @param flipBack defines the flip
         * @param twirlClockwise defines the twirl
         * @param tiltRight defines the tilt
         * @returns the new rotation vector
         */
        calcRotatePOV(flipBack: number, twirlClockwise: number, tiltRight: number): Vector3;
        /**
         * Return the minimum and maximum world vectors of the entire hierarchy under current mesh
         * @param includeDescendants Include bounding info from descendants as well (true by default)
         * @returns the new bounding vectors
         */
        getHierarchyBoundingVectors(includeDescendants?: boolean): {
            min: Vector3;
            max: Vector3;
        };
        /** @hidden */
        _updateBoundingInfo(): AbstractMesh;
        /** @hidden */
        _updateSubMeshesBoundingInfo(matrix: Matrix): AbstractMesh;
        /** @hidden */
        protected _afterComputeWorldMatrix(): void;
        /**
         * Returns `true` if the mesh is within the frustum defined by the passed array of planes.
         * A mesh is in the frustum if its bounding box intersects the frustum
         * @param frustumPlanes defines the frustum to test
         * @returns true if the mesh is in the frustum planes
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Returns `true` if the mesh is completely in the frustum defined be the passed array of planes.
         * A mesh is completely in the frustum if its bounding box it completely inside the frustum.
         * @param frustumPlanes defines the frustum to test
         * @returns true if the mesh is completely in the frustum planes
         */
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * True if the mesh intersects another mesh or a SolidParticle object
         * @param mesh defines a target mesh or SolidParticle to test
         * @param precise Unless the parameter `precise` is set to `true` the intersection is computed according to Axis Aligned Bounding Boxes (AABB), else according to OBB (Oriented BBoxes)
         * @param includeDescendants Can be set to true to test if the mesh defined in parameters intersects with the current mesh or any child meshes
         * @returns true if there is an intersection
         */
        intersectsMesh(mesh: AbstractMesh | SolidParticle, precise?: boolean, includeDescendants?: boolean): boolean;
        /**
         * Returns true if the passed point (Vector3) is inside the mesh bounding box
         * @param point defines the point to test
         * @returns true if there is an intersection
         */
        intersectsPoint(point: Vector3): boolean;
        /**
         * Gets the current physics impostor
         * @see http://doc.babylonjs.com/features/physics_engine
         * @returns a physics impostor or null
         */
        getPhysicsImpostor(): Nullable<PhysicsImpostor>;
        /**
         * Gets the position of the current mesh in camera space
         * @param camera defines the camera to use
         * @returns a position
         */
        getPositionInCameraSpace(camera?: Nullable<Camera>): Vector3;
        /**
         * Returns the distance from the mesh to the active camera
         * @param camera defines the camera to use
         * @returns the distance
         */
        getDistanceToCamera(camera?: Nullable<Camera>): number;
        /**
         * Apply a physic impulse to the mesh
         * @param force defines the force to apply
         * @param contactPoint defines where to apply the force
         * @returns the current mesh
         * @see http://doc.babylonjs.com/how_to/using_the_physics_engine
         */
        applyImpulse(force: Vector3, contactPoint: Vector3): AbstractMesh;
        /**
         * Creates a physic joint between two meshes
         * @param otherMesh defines the other mesh to use
         * @param pivot1 defines the pivot to use on this mesh
         * @param pivot2 defines the pivot to use on the other mesh
         * @param options defines additional options (can be plugin dependent)
         * @returns the current mesh
         * @see https://www.babylonjs-playground.com/#0BS5U0#0
         */
        setPhysicsLinkWith(otherMesh: Mesh, pivot1: Vector3, pivot2: Vector3, options?: any): AbstractMesh;
        /**
         * Gets or sets a boolean indicating that this mesh can be used in the collision engine
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
         */
        checkCollisions: boolean;
        /**
         * Gets Collider object used to compute collisions (not physics)
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
         */
        readonly collider: Collider;
        /**
         * Move the mesh using collision engine
         * @see http://doc.babylonjs.com/babylon101/cameras,_mesh_collisions_and_gravity
         * @param displacement defines the requested displacement vector
         * @returns the current mesh
         */
        moveWithCollisions(displacement: Vector3): AbstractMesh;
        private _onCollisionPositionChange;
        /**
        * This function will create an octree to help to select the right submeshes for rendering, picking and collision computations.
        * Please note that you must have a decent number of submeshes to get performance improvements when using an octree
        * @param maxCapacity defines the maximum size of each block (64 by default)
        * @param maxDepth defines the maximum depth to use (no more than 2 levels by default)
        * @returns the new octree
        * @see https://www.babylonjs-playground.com/#NA4OQ#12
        * @see http://doc.babylonjs.com/how_to/optimizing_your_scene_with_octrees
        */
        createOrUpdateSubmeshesOctree(maxCapacity?: number, maxDepth?: number): Octree<SubMesh>;
        /** @hidden */
        _collideForSubMesh(subMesh: SubMesh, transformMatrix: Matrix, collider: Collider): AbstractMesh;
        /** @hidden */
        _processCollisionsForSubMeshes(collider: Collider, transformMatrix: Matrix): AbstractMesh;
        /** @hidden */
        _checkCollision(collider: Collider): AbstractMesh;
        /** @hidden */
        _generatePointsArray(): boolean;
        /**
         * Checks if the passed Ray intersects with the mesh
         * @param ray defines the ray to use
         * @param fastCheck defines if fast mode (but less precise) must be used (false by default)
         * @returns the picking info
         * @see http://doc.babylonjs.com/babylon101/intersect_collisions_-_mesh
         */
        intersects(ray: Ray, fastCheck?: boolean): PickingInfo;
        /**
         * Clones the current mesh
         * @param name defines the mesh name
         * @param newParent defines the new mesh parent
         * @param doNotCloneChildren defines a boolean indicating that children must not be cloned (false by default)
         * @returns the new mesh
         */
        clone(name: string, newParent: Node, doNotCloneChildren?: boolean): Nullable<AbstractMesh>;
        /**
         * Disposes all the submeshes of the current meshnp
         * @returns the current mesh
         */
        releaseSubMeshes(): AbstractMesh;
        /**
         * Releases resources associated with this abstract mesh.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Adds the passed mesh as a child to the current mesh
         * @param mesh defines the child mesh
         * @returns the current mesh
         */
        addChild(mesh: AbstractMesh): AbstractMesh;
        /**
         * Removes the passed mesh from the current mesh children list
         * @param mesh defines the child mesh
         * @returns the current mesh
         */
        removeChild(mesh: AbstractMesh): AbstractMesh;
        /** @hidden */
        private _initFacetData();
        /**
         * Updates the mesh facetData arrays and the internal partitioning when the mesh is morphed or updated.
         * This method can be called within the render loop.
         * You don't need to call this method by yourself in the render loop when you update/morph a mesh with the methods CreateXXX() as they automatically manage this computation
         * @returns the current mesh
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        updateFacetData(): AbstractMesh;
        /**
         * Returns the facetLocalNormals array.
         * The normals are expressed in the mesh local spac
         * @returns an array of Vector3
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetLocalNormals(): Vector3[];
        /**
         * Returns the facetLocalPositions array.
         * The facet positions are expressed in the mesh local space
         * @returns an array of Vector3
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetLocalPositions(): Vector3[];
        /**
         * Returns the facetLocalPartioning array
         * @returns an array of array of numbers
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetLocalPartitioning(): number[][];
        /**
         * Returns the i-th facet position in the world system.
         * This method allocates a new Vector3 per call
         * @param i defines the facet index
         * @returns a new Vector3
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetPosition(i: number): Vector3;
        /**
         * Sets the reference Vector3 with the i-th facet position in the world system
         * @param i defines the facet index
         * @param ref defines the target vector
         * @returns the current mesh
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetPositionToRef(i: number, ref: Vector3): AbstractMesh;
        /**
         * Returns the i-th facet normal in the world system.
         * This method allocates a new Vector3 per call
         * @param i defines the facet index
         * @returns a new Vector3
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetNormal(i: number): Vector3;
        /**
         * Sets the reference Vector3 with the i-th facet normal in the world system
         * @param i defines the facet index
         * @param ref defines the target vector
         * @returns the current mesh
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetNormalToRef(i: number, ref: Vector3): this;
        /**
         * Returns the facets (in an array) in the same partitioning block than the one the passed coordinates are located (expressed in the mesh local system)
         * @param x defines x coordinate
         * @param y defines y coordinate
         * @param z defines z coordinate
         * @returns the array of facet indexes
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetsAtLocalCoordinates(x: number, y: number, z: number): Nullable<number[]>;
        /**
         * Returns the closest mesh facet index at (x,y,z) World coordinates, null if not found
         * @param projected sets as the (x,y,z) world projection on the facet
         * @param checkFace if true (default false), only the facet "facing" to (x,y,z) or only the ones "turning their backs", according to the parameter "facing" are returned
         * @param facing if facing and checkFace are true, only the facet "facing" to (x, y, z) are returned : positive dot (x, y, z) * facet position. If facing si false and checkFace is true, only the facet "turning their backs" to (x, y, z) are returned : negative dot (x, y, z) * facet position
         * @param x defines x coordinate
         * @param y defines y coordinate
         * @param z defines z coordinate
         * @returns the face index if found (or null instead)
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getClosestFacetAtCoordinates(x: number, y: number, z: number, projected?: Vector3, checkFace?: boolean, facing?: boolean): Nullable<number>;
        /**
         * Returns the closest mesh facet index at (x,y,z) local coordinates, null if not found
         * @param projected sets as the (x,y,z) local projection on the facet
         * @param checkFace if true (default false), only the facet "facing" to (x,y,z) or only the ones "turning their backs", according to the parameter "facing" are returned
         * @param facing if facing and checkFace are true, only the facet "facing" to (x, y, z) are returned : positive dot (x, y, z) * facet position. If facing si false and checkFace is true, only the facet "turning their backs" to (x, y, z) are returned : negative dot (x, y, z) * facet position
         * @param x defines x coordinate
         * @param y defines y coordinate
         * @param z defines z coordinate
         * @returns the face index if found (or null instead)
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getClosestFacetAtLocalCoordinates(x: number, y: number, z: number, projected?: Vector3, checkFace?: boolean, facing?: boolean): Nullable<number>;
        /**
         * Returns the object "parameter" set with all the expected parameters for facetData computation by ComputeNormals()
         * @returns the parameters
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        getFacetDataParameters(): any;
        /**
         * Disables the feature FacetData and frees the related memory
         * @returns the current mesh
         * @see http://doc.babylonjs.com/how_to/how_to_use_facetdata
         */
        disableFacetData(): AbstractMesh;
        /**
         * Updates the AbstractMesh indices array
         * @param indices defines the data source
         * @returns the current mesh
         */
        updateIndices(indices: IndicesArray): AbstractMesh;
        /**
         * Creates new normals data for the mesh
         * @param updatable defines if the normal vertex buffer must be flagged as updatable
         * @returns the current mesh
         */
        createNormals(updatable: boolean): AbstractMesh;
        /**
         * Align the mesh with a normal
         * @param normal defines the normal to use
         * @param upDirection can be used to redefined the up vector to use (will use the (0, 1, 0) by default)
         * @returns the current mesh
         */
        alignWithNormal(normal: Vector3, upDirection?: Vector3): AbstractMesh;
        /** @hidden */
        protected _checkOcclusionQuery(): void;
    }
}

declare module BABYLON {
    class Buffer {
        private _engine;
        private _buffer;
        private _data;
        private _updatable;
        private _instanced;
        /**
         * Gets the byte stride.
         */
        readonly byteStride: number;
        /**
         * Constructor
         * @param engine the engine
         * @param data the data to use for this buffer
         * @param updatable whether the data is updatable
         * @param stride the stride (optional)
         * @param postponeInternalCreation whether to postpone creating the internal WebGL buffer (optional)
         * @param instanced whether the buffer is instanced (optional)
         * @param useBytes set to true if the stride in in bytes (optional)
         */
        constructor(engine: any, data: DataArray, updatable: boolean, stride?: number, postponeInternalCreation?: boolean, instanced?: boolean, useBytes?: boolean);
        /**
         * Create a new {BABYLON.VertexBuffer} based on the current buffer
         * @param kind defines the vertex buffer kind (position, normal, etc.)
         * @param offset defines offset in the buffer (0 by default)
         * @param size defines the size in floats of attributes (position is 3 for instance)
         * @param stride defines the stride size in floats in the buffer (the offset to apply to reach next value when data is interleaved)
         * @param instanced defines if the vertex buffer contains indexed data
         * @param useBytes defines if the offset and stride are in bytes
         * @returns the new vertex buffer
         */
        createVertexBuffer(kind: string, offset: number, size: number, stride?: number, instanced?: boolean, useBytes?: boolean): VertexBuffer;
        isUpdatable(): boolean;
        getData(): Nullable<DataArray>;
        getBuffer(): Nullable<WebGLBuffer>;
        /**
         * Gets the stride in float32 units (i.e. byte stride / 4).
         * May not be an integer if the byte stride is not divisible by 4.
         * DEPRECATED. Use byteStride instead.
         * @returns the stride in float32 units
         */
        getStrideSize(): number;
        create(data?: Nullable<DataArray>): void;
        _rebuild(): void;
        update(data: DataArray): void;
        /**
         * Updates the data directly.
         * @param data the new data
         * @param offset the new offset
         * @param vertexCount the vertex count (optional)
         * @param useBytes set to true if the offset is in bytes
         */
        updateDirectly(data: DataArray, offset: number, vertexCount?: number, useBytes?: boolean): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class CSG {
        private polygons;
        matrix: Matrix;
        position: Vector3;
        rotation: Vector3;
        rotationQuaternion: Nullable<Quaternion>;
        scaling: Vector3;
        static FromMesh(mesh: Mesh): CSG;
        private static FromPolygons(polygons);
        clone(): CSG;
        union(csg: CSG): CSG;
        unionInPlace(csg: CSG): void;
        subtract(csg: CSG): CSG;
        subtractInPlace(csg: CSG): void;
        intersect(csg: CSG): CSG;
        intersectInPlace(csg: CSG): void;
        inverse(): CSG;
        inverseInPlace(): void;
        copyTransformAttributes(csg: CSG): CSG;
        buildMeshGeometry(name: string, scene: Scene, keepSubMeshes: boolean): Mesh;
        toMesh(name: string, material: Nullable<Material>, scene: Scene, keepSubMeshes: boolean): Mesh;
    }
}

declare module BABYLON {
    /**
     * Class used to store geometry data (vertex buffers + index buffer)
     */
    class Geometry implements IGetSetVerticesData {
        /**
         * Gets or sets the unique ID of the geometry
         */
        id: string;
        /**
         * Gets the delay loading state of the geometry (none by default which means not delayed)
         */
        delayLoadState: number;
        /**
         * Gets the file containing the data to load when running in delay load state
         */
        delayLoadingFile: Nullable<string>;
        /**
         * Callback called when the geometry is updated
         */
        onGeometryUpdated: (geometry: Geometry, kind?: string) => void;
        private _scene;
        private _engine;
        private _meshes;
        private _totalVertices;
        private _indices;
        private _vertexBuffers;
        private _isDisposed;
        private _extend;
        private _boundingBias;
        /** @hidden */
        _delayInfo: Array<string>;
        private _indexBuffer;
        private _indexBufferIsUpdatable;
        /** @hidden */
        _boundingInfo: Nullable<BoundingInfo>;
        /** @hidden */
        _delayLoadingFunction: Nullable<(any: any, geometry: Geometry) => void>;
        /** @hidden */
        _softwareSkinningRenderId: number;
        private _vertexArrayObjects;
        private _updatable;
        /** @hidden */
        _positions: Nullable<Vector3[]>;
        /**
         *  Gets or sets the Bias Vector to apply on the bounding elements (box/sphere), the max extend is computed as v += v * bias.x + bias.y, the min is computed as v -= v * bias.x + bias.y
         */
        /**
         *  Gets or sets the Bias Vector to apply on the bounding elements (box/sphere), the max extend is computed as v += v * bias.x + bias.y, the min is computed as v -= v * bias.x + bias.y
         */
        boundingBias: Vector2;
        /**
         * Static function used to attach a new empty geometry to a mesh
         * @param mesh defines the mesh to attach the geometry to
         * @returns the new {BABYLON.Geometry}
         */
        static CreateGeometryForMesh(mesh: Mesh): Geometry;
        /**
         * Creates a new geometry
         * @param id defines the unique ID
         * @param scene defines the hosting scene
         * @param vertexData defines the {BABYLON.VertexData} used to get geometry data
         * @param updatable defines if geometry must be updatable (false by default)
         * @param mesh defines the mesh that will be associated with the geometry
         */
        constructor(id: string, scene: Scene, vertexData?: VertexData, updatable?: boolean, mesh?: Nullable<Mesh>);
        /**
         * Gets the current extend of the geometry
         */
        readonly extend: {
            minimum: Vector3;
            maximum: Vector3;
        };
        /**
         * Gets the hosting scene
         * @returns the hosting {BABYLON.Scene}
         */
        getScene(): Scene;
        /**
         * Gets the hosting engine
         * @returns the hosting {BABYLON.Engine}
         */
        getEngine(): Engine;
        /**
         * Defines if the geometry is ready to use
         * @returns true if the geometry is ready to be used
         */
        isReady(): boolean;
        /**
         * Gets a value indicating that the geometry should not be serialized
         */
        readonly doNotSerialize: boolean;
        /** @hidden */
        _rebuild(): void;
        /**
         * Affects all gemetry data in one call
         * @param vertexData defines the geometry data
         * @param updatable defines if the geometry must be flagged as updatable (false as default)
         */
        setAllVerticesData(vertexData: VertexData, updatable?: boolean): void;
        /**
         * Set specific vertex data
         * @param kind defines the data kind (Position, normal, etc...)
         * @param data defines the vertex data to use
         * @param updatable defines if the vertex must be flagged as updatable (false as default)
         * @param stride defines the stride to use (0 by default). This value is deduced from the kind value if not specified
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): void;
        /**
         * Removes a specific vertex data
         * @param kind defines the data kind (Position, normal, etc...)
         */
        removeVerticesData(kind: string): void;
        /**
         * Affect a vertex buffer to the geometry. the vertexBuffer.getKind() function is used to determine where to store the data
         * @param buffer defines the vertex buffer to use
         * @param totalVertices defines the total number of vertices for position kind (could be null)
         */
        setVerticesBuffer(buffer: VertexBuffer, totalVertices?: Nullable<number>): void;
        /**
         * Update a specific vertex buffer
         * This function will directly update the underlying WebGLBuffer according to the passed numeric array or Float32Array
         * It will do nothing if the buffer is not updatable
         * @param kind defines the data kind (Position, normal, etc...)
         * @param data defines the data to use
         * @param offset defines the offset in the target buffer where to store the data
         * @param useBytes set to true if the offset is in bytes
         */
        updateVerticesDataDirectly(kind: string, data: DataArray, offset: number, useBytes?: boolean): void;
        /**
         * Update a specific vertex buffer
         * This function will create a new buffer if the current one is not updatable
         * @param kind defines the data kind (Position, normal, etc...)
         * @param data defines the data to use
         * @param updateExtends defines if the geometry extends must be recomputed (false by default)
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean): void;
        private _updateBoundingInfo(updateExtends, data);
        /** @hidden */
        _bind(effect: Nullable<Effect>, indexToBind?: Nullable<WebGLBuffer>): void;
        /**
         * Gets total number of vertices
         * @returns the total number of vertices
         */
        getTotalVertices(): number;
        /**
         * Gets a specific vertex data attached to this geometry. Float data is constructed if the vertex buffer data cannot be returned directly.
         * @param kind defines the data kind (Position, normal, etc...)
         * @param copyWhenShared defines if the returned array must be cloned upon returning it if the current geometry is shared between multiple meshes
         * @param forceCopy defines a boolean indicating that the returned array must be cloned upon returning it
         * @returns a float array containing vertex data
         */
        getVerticesData(kind: string, copyWhenShared?: boolean, forceCopy?: boolean): Nullable<FloatArray>;
        /**
         * Returns a boolean defining if the vertex data for the requested `kind` is updatable
         * @param kind defines the data kind (Position, normal, etc...)
         * @returns true if the vertex buffer with the specified kind is updatable
         */
        isVertexBufferUpdatable(kind: string): boolean;
        /**
         * Gets a specific vertex buffer
         * @param kind defines the data kind (Position, normal, etc...)
         * @returns a {BABYLON.VertexBuffer}
         */
        getVertexBuffer(kind: string): Nullable<VertexBuffer>;
        /**
         * Returns all vertex buffers
         * @return an object holding all vertex buffers indexed by kind
         */
        getVertexBuffers(): Nullable<{
            [key: string]: VertexBuffer;
        }>;
        /**
         * Gets a boolean indicating if specific vertex buffer is present
         * @param kind defines the data kind (Position, normal, etc...)
         * @returns true if data is present
         */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Gets a list of all attached data kinds (Position, normal, etc...)
         * @returns a list of string containing all kinds
         */
        getVerticesDataKinds(): string[];
        /**
         * Update index buffer
         * @param indices defines the indices to store in the index buffer
         * @param offset defines the offset in the target buffer where to store the data
         */
        updateIndices(indices: IndicesArray, offset?: number): void;
        /**
         * Creates a new index buffer
         * @param indices defines the indices to store in the index buffer
         * @param totalVertices defines the total number of vertices (could be null)
         * @param updatable defines if the index buffer must be flagged as updatable (false by default)
         */
        setIndices(indices: IndicesArray, totalVertices?: Nullable<number>, updatable?: boolean): void;
        /**
         * Return the total number of indices
         * @returns the total number of indices
         */
        getTotalIndices(): number;
        /**
         * Gets the index buffer array
         * @param copyWhenShared defines if the returned array must be cloned upon returning it if the current geometry is shared between multiple meshes
         * @returns the index buffer array
         */
        getIndices(copyWhenShared?: boolean): Nullable<IndicesArray>;
        /**
         * Gets the index buffer
         * @return the index buffer
         */
        getIndexBuffer(): Nullable<WebGLBuffer>;
        /** @hidden */
        _releaseVertexArrayObject(effect?: Nullable<Effect>): void;
        /**
         * Release the associated resources for a specific mesh
         * @param mesh defines the source mesh
         * @param shouldDispose defines if the geometry must be disposed if there is no more mesh pointing to it
         */
        releaseForMesh(mesh: Mesh, shouldDispose?: boolean): void;
        /**
         * Apply current geometry to a given mesh
         * @param mesh defines the mesh to apply geometry to
         */
        applyToMesh(mesh: Mesh): void;
        private _updateExtend(data?);
        private _applyToMesh(mesh);
        private notifyUpdate(kind?);
        /**
         * Load the geometry if it was flagged as delay loaded
         * @param scene defines the hosting scene
         * @param onLoaded defines a callback called when the geometry is loaded
         */
        load(scene: Scene, onLoaded?: () => void): void;
        private _queueLoad(scene, onLoaded?);
        /**
         * Invert the geometry to move from a right handed system to a left handed one.
         */
        toLeftHanded(): void;
        /** @hidden */
        _resetPointsArrayCache(): void;
        /** @hidden */
        _generatePointsArray(): boolean;
        /**
         * Gets a value indicating if the geometry is disposed
         * @returns true if the geometry was disposed
         */
        isDisposed(): boolean;
        private _disposeVertexArrayObjects();
        /**
         * Free all associated resources
         */
        dispose(): void;
        /**
         * Clone the current geometry into a new geometry
         * @param id defines the unique ID of the new geometry
         * @returns a new geometry object
         */
        copy(id: string): Geometry;
        /**
         * Serialize the current geometry info (and not the vertices data) into a JSON object
         * @return a JSON representation of the current geometry data (without the vertices data)
         */
        serialize(): any;
        private toNumberArray(origin);
        /**
         * Serialize all vertices data into a JSON oject
         * @returns a JSON representation of the current geometry data
         */
        serializeVerticeData(): any;
        /**
         * Extracts a clone of a mesh geometry
         * @param mesh defines the source mesh
         * @param id defines the unique ID of the new geometry object
         * @returns the new geometry object
         */
        static ExtractFromMesh(mesh: Mesh, id: string): Nullable<Geometry>;
        /**
         * You should now use Tools.RandomId(), this method is still here for legacy reasons.
         * Implementation from http://stackoverflow.com/questions/105034/how-to-create-a-guid-uuid-in-javascript/2117523#answer-2117523
         * Be aware Math.random() could cause collisions, but:
         * "All but 6 of the 128 bits of the ID are randomly generated, which means that for any two ids, there's a 1 in 2^^122 (or 5.3x10^^36) chance they'll collide"
         * @returns a string containing a new GUID
         */
        static RandomId(): string;
        /** @hidden */
        static _ImportGeometry(parsedGeometry: any, mesh: Mesh): void;
        private static _CleanMatricesWeights(parsedGeometry, mesh);
        /**
         * Create a new geometry from persisted data (Using .babylon file format)
         * @param parsedVertexData defines the persisted data
         * @param scene defines the hosting scene
         * @param rootUrl defines the root url to use to load assets (like delayed data)
         * @returns the new geometry object
         */
        static Parse(parsedVertexData: any, scene: Scene, rootUrl: string): Nullable<Geometry>;
    }
    /**
     * Abstract class used to provide common services for all typed geometries
     * @hidden
     */
    class _PrimitiveGeometry extends Geometry {
        private _canBeRegenerated;
        private _beingRegenerated;
        /**
         * Creates a new typed geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param _canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         */
        constructor(id: string, scene: Scene, _canBeRegenerated?: boolean, mesh?: Nullable<Mesh>);
        /**
         * Gets a value indicating if the geometry supports being regenerated with new parameters (false by default)
         * @returns true if the geometry can be regenerated
         */
        canBeRegenerated(): boolean;
        /**
         * If the geometry supports regeneration, the function will recreates the geometry with updated parameter values
         */
        regenerate(): void;
        /**
         * Clone the geometry
         * @param id defines the unique ID of the new geometry
         * @returns the new geometry
         */
        asNewGeometry(id: string): Geometry;
        setAllVerticesData(vertexData: VertexData, updatable?: boolean): void;
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean): void;
        /** @hidden */
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
    }
    /**
     * Creates a ribbon geometry
     * @description See http://doc.babylonjs.com/how_to/ribbon_tutorial, http://doc.babylonjs.com/resources/maths_make_ribbons
     */
    class RibbonGeometry extends _PrimitiveGeometry {
        /**
         * Defines the array of paths to use
         */
        pathArray: Vector3[][];
        /**
         * Defines if the last and first points of each path in your pathArray must be joined
         */
        closeArray: boolean;
        /**
         * Defines if the last and first points of each path in your pathArray must be joined
         */
        closePath: boolean;
        /**
         * Defines the offset between points
         */
        offset: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a ribbon geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param pathArray defines the array of paths to use
         * @param closeArray defines if the last path and the first path must be  joined
         * @param closePath defines if the last and first points of each path in your pathArray must be joined
         * @param offset defines the offset between points
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the array of paths to use
             */
            pathArray: Vector3[][], 
            /**
             * Defines if the last and first points of each path in your pathArray must be joined
             */
            closeArray: boolean, 
            /**
             * Defines if the last and first points of each path in your pathArray must be joined
             */
            closePath: boolean, 
            /**
             * Defines the offset between points
             */
            offset: number, canBeRegenerated?: boolean, mesh?: Mesh, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        /** @hidden */
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
    }
    /**
     * Creates a box geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#box
     */
    class BoxGeometry extends _PrimitiveGeometry {
        /**
         * Defines the zise of the box (width, height and depth are the same)
         */
        size: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a box geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param size defines the zise of the box (width, height and depth are the same)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the zise of the box (width, height and depth are the same)
             */
            size: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedBox: any, scene: Scene): Nullable<BoxGeometry>;
    }
    /**
     * Creates a sphere geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#sphere
     */
    class SphereGeometry extends _PrimitiveGeometry {
        /**
         * Defines the number of segments to use to create the sphere
         */
        segments: number;
        /**
         * Defines the diameter of the sphere
         */
        diameter: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Create a new sphere geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param segments defines the number of segments to use to create the sphere
         * @param diameter defines the diameter of the sphere
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the number of segments to use to create the sphere
             */
            segments: number, 
            /**
             * Defines the diameter of the sphere
             */
            diameter: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedSphere: any, scene: Scene): Nullable<SphereGeometry>;
    }
    /**
     * Creates a disc geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#disc-or-regular-polygon
     */
    class DiscGeometry extends _PrimitiveGeometry {
        /**
         * Defines the radius of the disc
         */
        radius: number;
        /**
         * Defines the tesselation factor to apply to the disc
         */
        tessellation: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a new disc geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param radius defines the radius of the disc
         * @param tessellation defines the tesselation factor to apply to the disc
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the radius of the disc
             */
            radius: number, 
            /**
             * Defines the tesselation factor to apply to the disc
             */
            tessellation: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
    }
    /**
     * Creates a new cylinder geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#cylinder-or-cone
     */
    class CylinderGeometry extends _PrimitiveGeometry {
        /**
         * Defines the height of the cylinder
         */
        height: number;
        /**
         * Defines the diameter of the cylinder's top cap
         */
        diameterTop: number;
        /**
         * Defines the diameter of the cylinder's bottom cap
         */
        diameterBottom: number;
        /**
         * Defines the tessellation factor to apply to the cylinder
         */
        tessellation: number;
        /**
         * Defines the number of subdivisions to apply to the cylinder (1 by default)
         */
        subdivisions: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a new cylinder geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param height defines the height of the cylinder
         * @param diameterTop defines the diameter of the cylinder's top cap
         * @param diameterBottom defines the diameter of the cylinder's bottom cap
         * @param tessellation defines the tessellation factor to apply to the cylinder (number of radial sides)
         * @param subdivisions defines the number of subdivisions to apply to the cylinder (number of rings) (1 by default)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the height of the cylinder
             */
            height: number, 
            /**
             * Defines the diameter of the cylinder's top cap
             */
            diameterTop: number, 
            /**
             * Defines the diameter of the cylinder's bottom cap
             */
            diameterBottom: number, 
            /**
             * Defines the tessellation factor to apply to the cylinder
             */
            tessellation: number, 
            /**
             * Defines the number of subdivisions to apply to the cylinder (1 by default)
             */
            subdivisions?: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedCylinder: any, scene: Scene): Nullable<CylinderGeometry>;
    }
    /**
     * Creates a new torus geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#torus
     */
    class TorusGeometry extends _PrimitiveGeometry {
        /**
         * Defines the diameter of the torus
         */
        diameter: number;
        /**
         * Defines the thickness of the torus (ie. internal diameter)
         */
        thickness: number;
        /**
         * Defines the tesselation factor to apply to the torus
         */
        tessellation: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a new torus geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param diameter defines the diameter of the torus
         * @param thickness defines the thickness of the torus (ie. internal diameter)
         * @param tessellation defines the tesselation factor to apply to the torus (number of segments along the circle)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the diameter of the torus
             */
            diameter: number, 
            /**
             * Defines the thickness of the torus (ie. internal diameter)
             */
            thickness: number, 
            /**
             * Defines the tesselation factor to apply to the torus
             */
            tessellation: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedTorus: any, scene: Scene): Nullable<TorusGeometry>;
    }
    /**
     * Creates a new ground geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#ground
     */
    class GroundGeometry extends _PrimitiveGeometry {
        /**
         * Defines the width of the ground
         */
        width: number;
        /**
         * Defines the height of the ground
         */
        height: number;
        /**
         * Defines the subdivisions to apply to the ground
         */
        subdivisions: number;
        /**
         * Creates a new ground geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param width defines the width of the ground
         * @param height defines the height of the ground
         * @param subdivisions defines the subdivisions to apply to the ground
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the width of the ground
             */
            width: number, 
            /**
             * Defines the height of the ground
             */
            height: number, 
            /**
             * Defines the subdivisions to apply to the ground
             */
            subdivisions: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedGround: any, scene: Scene): Nullable<GroundGeometry>;
    }
    /**
     * Creates a tiled ground geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#tiled-ground
     */
    class TiledGroundGeometry extends _PrimitiveGeometry {
        /**
         * Defines the minimum value on X axis
         */
        xmin: number;
        /**
         * Defines the minimum value on Z axis
         */
        zmin: number;
        /**
         * Defines the maximum value on X axis
         */
        xmax: number;
        /**
         * Defines the maximum value on Z axis
         */
        zmax: number;
        /**
         * Defines the subdivisions to apply to the ground
         */
        subdivisions: {
            w: number;
            h: number;
        };
        /**
         * Defines the precision to use when computing the tiles
         */
        precision: {
            w: number;
            h: number;
        };
        /**
         * Creates a tiled ground geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param xmin defines the minimum value on X axis
         * @param zmin defines the minimum value on Z axis
         * @param xmax defines the maximum value on X axis
         * @param zmax defines the maximum value on Z axis
         * @param subdivisions defines the subdivisions to apply to the ground (number of subdivisions (tiles) on the height and the width of the map)
         * @param precision defines the precision to use when computing the tiles
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the minimum value on X axis
             */
            xmin: number, 
            /**
             * Defines the minimum value on Z axis
             */
            zmin: number, 
            /**
             * Defines the maximum value on X axis
             */
            xmax: number, 
            /**
             * Defines the maximum value on Z axis
             */
            zmax: number, 
            /**
             * Defines the subdivisions to apply to the ground
             */
            subdivisions: {
            w: number;
            h: number;
        }, 
            /**
             * Defines the precision to use when computing the tiles
             */
            precision: {
            w: number;
            h: number;
        }, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
    }
    /**
     * Creates a plane geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#plane
     */
    class PlaneGeometry extends _PrimitiveGeometry {
        /**
         * Defines the size of the plane (width === height)
         */
        size: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a plane geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param size defines the size of the plane (width === height)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the size of the plane (width === height)
             */
            size: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedPlane: any, scene: Scene): Nullable<PlaneGeometry>;
    }
    /**
     * Creates a torus knot geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#torus-knot
     */
    class TorusKnotGeometry extends _PrimitiveGeometry {
        /**
         * Defines the radius of the torus knot
         */
        radius: number;
        /**
         * Defines the thickness of the torus knot tube
         */
        tube: number;
        /**
         * Defines the number of radial segments
         */
        radialSegments: number;
        /**
         * Defines the number of tubular segments
         */
        tubularSegments: number;
        /**
         * Defines the first number of windings
         */
        p: number;
        /**
         * Defines the second number of windings
         */
        q: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a torus knot geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param radius defines the radius of the torus knot
         * @param tube defines the thickness of the torus knot tube
         * @param radialSegments defines the number of radial segments
         * @param tubularSegments defines the number of tubular segments
         * @param p defines the first number of windings
         * @param q defines the second number of windings
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the radius of the torus knot
             */
            radius: number, 
            /**
             * Defines the thickness of the torus knot tube
             */
            tube: number, 
            /**
             * Defines the number of radial segments
             */
            radialSegments: number, 
            /**
             * Defines the number of tubular segments
             */
            tubularSegments: number, 
            /**
             * Defines the first number of windings
             */
            p: number, 
            /**
             * Defines the second number of windings
             */
            q: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedTorusKnot: any, scene: Scene): Nullable<TorusKnotGeometry>;
    }
}

declare module BABYLON {
    class GroundMesh extends Mesh {
        generateOctree: boolean;
        private _heightQuads;
        _subdivisionsX: number;
        _subdivisionsY: number;
        _width: number;
        _height: number;
        _minX: number;
        _maxX: number;
        _minZ: number;
        _maxZ: number;
        constructor(name: string, scene: Scene);
        getClassName(): string;
        readonly subdivisions: number;
        readonly subdivisionsX: number;
        readonly subdivisionsY: number;
        optimize(chunksCount: number, octreeBlocksSize?: number): void;
        /**
         * Returns a height (y) value in the Worl system :
         * the ground altitude at the coordinates (x, z) expressed in the World system.
         * Returns the ground y position if (x, z) are outside the ground surface.
         */
        getHeightAtCoordinates(x: number, z: number): number;
        /**
         * Returns a normalized vector (Vector3) orthogonal to the ground
         * at the ground coordinates (x, z) expressed in the World system.
         * Returns Vector3(0.0, 1.0, 0.0) if (x, z) are outside the ground surface.
         */
        getNormalAtCoordinates(x: number, z: number): Vector3;
        /**
         * Updates the Vector3 passed a reference with a normalized vector orthogonal to the ground
         * at the ground coordinates (x, z) expressed in the World system.
         * Doesn't uptade the reference Vector3 if (x, z) are outside the ground surface.
         * Returns the GroundMesh.
         */
        getNormalAtCoordinatesToRef(x: number, z: number, ref: Vector3): GroundMesh;
        /**
        * Force the heights to be recomputed for getHeightAtCoordinates() or getNormalAtCoordinates()
        * if the ground has been updated.
        * This can be used in the render loop.
        * Returns the GroundMesh.
        */
        updateCoordinateHeights(): GroundMesh;
        private _getFacetAt(x, z);
        private _initHeightQuads();
        private _computeHeightQuads();
        serialize(serializationObject: any): void;
        static Parse(parsedMesh: any, scene: Scene): GroundMesh;
    }
}

declare module BABYLON {
    /**
     * Creates an instance based on a source mesh.
     */
    class InstancedMesh extends AbstractMesh {
        private _sourceMesh;
        private _currentLOD;
        constructor(name: string, source: Mesh);
        /**
         * Returns the string "InstancedMesh".
         */
        getClassName(): string;
        readonly receiveShadows: boolean;
        readonly material: Nullable<Material>;
        readonly visibility: number;
        readonly skeleton: Nullable<Skeleton>;
        renderingGroupId: number;
        /**
         * Returns the total number of vertices (integer).
         */
        getTotalVertices(): number;
        readonly sourceMesh: Mesh;
        /**
         * Is this node ready to be used/rendered
         * @param completeCheck defines if a complete check (including materials and lights) has to be done (false by default)
         * @return {boolean} is it ready
         */
        isReady(completeCheck?: boolean): boolean;
        /**
         * Returns a float array or a Float32Array of the requested kind of data : positons, normals, uvs, etc.
         */
        getVerticesData(kind: string, copyWhenShared?: boolean): Nullable<FloatArray>;
        /**
         * Sets the vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, a new Geometry object is set to the mesh and then passed this vertex data.
         * The `data` are either a numeric array either a Float32Array.
         * The parameter `updatable` is passed as is to the underlying Geometry object constructor (if initianilly none) or updater.
         * The parameter `stride` is an optional positive integer, it is usually automatically deducted from the `kind` (3 for positions or normals, 2 for UV, etc).
         * Note that a new underlying VertexBuffer object is created each call.
         * If the `kind` is the `PositionKind`, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): Mesh;
        /**
         * Updates the existing vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, it is simply returned as it is.
         * The `data` are either a numeric array either a Float32Array.
         * No new underlying VertexBuffer object is created.
         * If the `kind` is the `PositionKind` and if `updateExtends` is true, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         * If the parameter `makeItUnique` is true, a new global geometry is created from this positions and is set to the mesh.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): Mesh;
        /**
         * Sets the mesh indices.
         * Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array).
         * If the mesh has no geometry, a new Geometry object is created and set to the mesh.
         * This method creates a new index buffer each call.
         * Returns the Mesh.
         */
        setIndices(indices: IndicesArray, totalVertices?: Nullable<number>): Mesh;
        /**
         * Boolean : True if the mesh owns the requested kind of data.
         */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Returns an array of indices (IndicesArray).
         */
        getIndices(): Nullable<IndicesArray>;
        readonly _positions: Nullable<Vector3[]>;
        /**
         * Sets a new updated BoundingInfo to the mesh.
         * Returns the mesh.
         */
        refreshBoundingInfo(): InstancedMesh;
        _preActivate(): InstancedMesh;
        _activate(renderId: number): InstancedMesh;
        /**
         * Returns the current associated LOD AbstractMesh.
         */
        getLOD(camera: Camera): AbstractMesh;
        _syncSubMeshes(): InstancedMesh;
        _generatePointsArray(): boolean;
        /**
         * Creates a new InstancedMesh from the current mesh.
         * - name (string) : the cloned mesh name
         * - newParent (optional Node) : the optional Node to parent the clone to.
         * - doNotCloneChildren (optional boolean, default `false`) : if `true` the model children aren't cloned.
         *
         * Returns the clone.
         */
        clone(name: string, newParent: Node, doNotCloneChildren?: boolean): InstancedMesh;
        /**
         * Disposes the InstancedMesh.
         * Returns nothing.
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
    }
}

declare module BABYLON {
    class LinesMesh extends Mesh {
        useVertexColor: boolean | undefined;
        useVertexAlpha: boolean | undefined;
        color: Color3;
        alpha: number;
        /**
         * The intersection Threshold is the margin applied when intersection a segment of the LinesMesh with a Ray.
         * This margin is expressed in world space coordinates, so its value may vary.
         * Default value is 0.1
         * @returns the intersection Threshold value.
         */
        /**
         * The intersection Threshold is the margin applied when intersection a segment of the LinesMesh with a Ray.
         * This margin is expressed in world space coordinates, so its value may vary.
         * @param value the new threshold to apply
         */
        intersectionThreshold: number;
        private _intersectionThreshold;
        private _colorShader;
        constructor(name: string, scene?: Nullable<Scene>, parent?: Nullable<Node>, source?: LinesMesh, doNotCloneChildren?: boolean, useVertexColor?: boolean | undefined, useVertexAlpha?: boolean | undefined);
        /**
         * Returns the string "LineMesh"
         */
        getClassName(): string;
        material: Material;
        readonly checkCollisions: boolean;
        createInstance(name: string): InstancedMesh;
        _bind(subMesh: SubMesh, effect: Effect, fillMode: number): LinesMesh;
        _draw(subMesh: SubMesh, fillMode: number, instancesCount?: number): LinesMesh;
        dispose(doNotRecurse?: boolean): void;
        /**
         * Returns a new LineMesh object cloned from the current one.
         */
        clone(name: string, newParent?: Node, doNotCloneChildren?: boolean): LinesMesh;
    }
}

declare module BABYLON {
    /**
     * @hidden
     **/
    class _InstancesBatch {
        mustReturn: boolean;
        visibleInstances: Nullable<InstancedMesh[]>[];
        renderSelf: boolean[];
    }
    class Mesh extends AbstractMesh implements IGetSetVerticesData {
        static _FRONTSIDE: number;
        static _BACKSIDE: number;
        static _DOUBLESIDE: number;
        static _DEFAULTSIDE: number;
        static _NO_CAP: number;
        static _CAP_START: number;
        static _CAP_END: number;
        static _CAP_ALL: number;
        /**
         * Mesh side orientation : usually the external or front surface
         */
        static readonly FRONTSIDE: number;
        /**
         * Mesh side orientation : usually the internal or back surface
         */
        static readonly BACKSIDE: number;
        /**
         * Mesh side orientation : both internal and external or front and back surfaces
         */
        static readonly DOUBLESIDE: number;
        /**
         * Mesh side orientation : by default, `FRONTSIDE`
         */
        static readonly DEFAULTSIDE: number;
        /**
         * Mesh cap setting : no cap
         */
        static readonly NO_CAP: number;
        /**
         * Mesh cap setting : one cap at the beginning of the mesh
         */
        static readonly CAP_START: number;
        /**
         * Mesh cap setting : one cap at the end of the mesh
         */
        static readonly CAP_END: number;
        /**
         * Mesh cap setting : two caps, one at the beginning  and one at the end of the mesh
         */
        static readonly CAP_ALL: number;
        /**
         * An event triggered before rendering the mesh
         */
        onBeforeRenderObservable: Observable<Mesh>;
        /**
        * An event triggered after rendering the mesh
        */
        onAfterRenderObservable: Observable<Mesh>;
        /**
        * An event triggered before drawing the mesh
        */
        onBeforeDrawObservable: Observable<Mesh>;
        private _onBeforeDrawObserver;
        onBeforeDraw: () => void;
        delayLoadState: number;
        instances: InstancedMesh[];
        delayLoadingFile: string;
        _binaryInfo: any;
        private _LODLevels;
        onLODLevelSelection: (distance: number, mesh: Mesh, selectedLevel: Mesh) => void;
        private _morphTargetManager;
        morphTargetManager: Nullable<MorphTargetManager>;
        _geometry: Nullable<Geometry>;
        _delayInfo: Array<string>;
        _delayLoadingFunction: (any: any, mesh: Mesh) => void;
        _visibleInstances: any;
        private _renderIdForInstances;
        private _batchCache;
        private _instancesBufferSize;
        private _instancesBuffer;
        private _instancesData;
        private _overridenInstanceCount;
        private _effectiveMaterial;
        _shouldGenerateFlatShading: boolean;
        private _preActivateId;
        _originalBuilderSideOrientation: number;
        overrideMaterialSideOrientation: Nullable<number>;
        private _areNormalsFrozen;
        private _sourcePositions;
        private _sourceNormals;
        private _source;
        readonly source: Nullable<Mesh>;
        isUnIndexed: boolean;
        /**
         * @constructor
         * @param {string} name The value used by scene.getMeshByName() to do a lookup.
         * @param {Scene} scene The scene to add this mesh to.
         * @param {Node} parent The parent of this mesh, if it has one
         * @param {Mesh} source An optional Mesh from which geometry is shared, cloned.
         * @param {boolean} doNotCloneChildren When cloning, skip cloning child meshes of source, default False.
         *                  When false, achieved by calling a clone(), also passing False.
         *                  This will make creation of children, recursive.
         * @param {boolean} clonePhysicsImpostor When cloning, include cloning mesh physics impostor, default True.
         */
        constructor(name: string, scene?: Nullable<Scene>, parent?: Nullable<Node>, source?: Nullable<Mesh>, doNotCloneChildren?: boolean, clonePhysicsImpostor?: boolean);
        /**
         * Returns the string "Mesh".
         */
        getClassName(): string;
        /**
         * Returns a string.
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        _unBindEffect(): void;
        /**
         * True if the mesh has some Levels Of Details (LOD).
         * Returns a boolean.
         */
        readonly hasLODLevels: boolean;
        /**
         * Gets the list of {BABYLON.MeshLODLevel} associated with the current mesh
         * @returns an array of {BABYLON.MeshLODLevel}
         */
        getLODLevels(): MeshLODLevel[];
        private _sortLODLevels();
        /**
         * Add a mesh as LOD level triggered at the given distance.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         * @param {number} distance The distance from the center of the object to show this level
         * @param {Mesh} mesh The mesh to be added as LOD level
         * @return {Mesh} This mesh (for chaining)
         */
        addLODLevel(distance: number, mesh: Mesh): Mesh;
        /**
         * Returns the LOD level mesh at the passed distance or null if not found.
         * It is related to the method `addLODLevel(distance, mesh)`.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         * Returns an object Mesh or `null`.
         */
        getLODLevelAtDistance(distance: number): Nullable<Mesh>;
        /**
         * Remove a mesh from the LOD array
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         * @param {Mesh} mesh The mesh to be removed.
         * @return {Mesh} This mesh (for chaining)
         */
        removeLODLevel(mesh: Mesh): Mesh;
        /**
         * Returns the registered LOD mesh distant from the parameter `camera` position if any, else returns the current mesh.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         */
        getLOD(camera: Camera, boundingSphere?: BoundingSphere): AbstractMesh;
        /**
         * Returns the mesh internal Geometry object.
         */
        readonly geometry: Nullable<Geometry>;
        /**
         * Returns a positive integer : the total number of vertices within the mesh geometry or zero if the mesh has no geometry.
         */
        getTotalVertices(): number;
        /**
         * Returns an array of integers or floats, or a Float32Array, depending on the requested `kind` (positions, indices, normals, etc).
         * If `copywhenShared` is true (default false) and if the mesh geometry is shared among some other meshes, the returned array is a copy of the internal one.
         * You can force the copy with forceCopy === true
         * Returns null if the mesh has no geometry or no vertex buffer.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        getVerticesData(kind: string, copyWhenShared?: boolean, forceCopy?: boolean): Nullable<FloatArray>;
        /**
         * Returns the mesh VertexBuffer object from the requested `kind` : positions, indices, normals, etc.
         * Returns `null` if the mesh has no geometry.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        getVertexBuffer(kind: string): Nullable<VertexBuffer>;
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Returns a boolean defining if the vertex data for the requested `kind` is updatable.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        isVertexBufferUpdatable(kind: string): boolean;
        /**
         * Returns a string : the list of existing `kinds` of Vertex Data for this mesh.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        getVerticesDataKinds(): string[];
        /**
         * Returns a positive integer : the total number of indices in this mesh geometry.
         * Returns zero if the mesh has no geometry.
         */
        getTotalIndices(): number;
        /**
         * Returns an array of integers or a typed array (Int32Array, Uint32Array, Uint16Array) populated with the mesh indices.
         * If the parameter `copyWhenShared` is true (default false) and and if the mesh geometry is shared among some other meshes, the returned array is a copy of the internal one.
         * Returns an empty array if the mesh has no geometry.
         */
        getIndices(copyWhenShared?: boolean): Nullable<IndicesArray>;
        readonly isBlocked: boolean;
        /**
         * Determine if the current mesh is ready to be rendered
         * @param completeCheck defines if a complete check (including materials and lights) has to be done (false by default)
         * @param forceInstanceSupport will check if the mesh will be ready when used with instances (false by default)
         * @returns true if all associated assets are ready (material, textures, shaders)
         */
        isReady(completeCheck?: boolean, forceInstanceSupport?: boolean): boolean;
        /**
         * Boolean : true if the normals aren't to be recomputed on next mesh `positions` array update.
         * This property is pertinent only for updatable parametric shapes.
         */
        readonly areNormalsFrozen: boolean;
        /**
         * This function affects parametric shapes on vertex position update only : ribbons, tubes, etc.
         * It has no effect at all on other shapes.
         * It prevents the mesh normals from being recomputed on next `positions` array update.
         * Returns the Mesh.
         */
        freezeNormals(): Mesh;
        /**
         * This function affects parametric shapes on vertex position update only : ribbons, tubes, etc.
         * It has no effect at all on other shapes.
         * It reactivates the mesh normals computation if it was previously frozen.
         * Returns the Mesh.
         */
        unfreezeNormals(): Mesh;
        /**
         * Overrides instance count. Only applicable when custom instanced InterleavedVertexBuffer are used rather than InstancedMeshs
         */
        overridenInstanceCount: number;
        _preActivate(): Mesh;
        _preActivateForIntermediateRendering(renderId: number): Mesh;
        _registerInstanceForRenderId(instance: InstancedMesh, renderId: number): Mesh;
        /**
         * This method recomputes and sets a new BoundingInfo to the mesh unless it is locked.
         * This means the mesh underlying bounding box and sphere are recomputed.
         * Returns the Mesh.
         */
        refreshBoundingInfo(): Mesh;
        _refreshBoundingInfo(applySkeleton: boolean): Mesh;
        private _getPositionData(applySkeleton);
        _createGlobalSubMesh(force: boolean): Nullable<SubMesh>;
        subdivide(count: number): void;
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): Mesh;
        markVerticesDataAsUpdatable(kind: string, updatable?: boolean): void;
        /**
         * Sets the mesh VertexBuffer.
         * Returns the Mesh.
         */
        setVerticesBuffer(buffer: VertexBuffer): Mesh;
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): Mesh;
        /**
         * This method updates the vertex positions of an updatable mesh according to the `positionFunction` returned values.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#other-shapes-updatemeshpositions
         * The parameter `positionFunction` is a simple JS function what is passed the mesh `positions` array. It doesn't need to return anything.
         * The parameter `computeNormals` is a boolean (default true) to enable/disable the mesh normal recomputation after the vertex position update.
         * Returns the Mesh.
         */
        updateMeshPositions(positionFunction: (data: FloatArray) => void, computeNormals?: boolean): Mesh;
        /**
         * Creates a un-shared specific occurence of the geometry for the mesh.
         * Returns the Mesh.
         */
        makeGeometryUnique(): Mesh;
        setIndices(indices: IndicesArray, totalVertices?: Nullable<number>, updatable?: boolean): Mesh;
        /**
         * Update the current index buffer
         * Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array)
         * Returns the Mesh.
         */
        updateIndices(indices: IndicesArray, offset?: number): Mesh;
        /**
         * Invert the geometry to move from a right handed system to a left handed one.
         * Returns the Mesh.
         */
        toLeftHanded(): Mesh;
        _bind(subMesh: SubMesh, effect: Effect, fillMode: number): Mesh;
        _draw(subMesh: SubMesh, fillMode: number, instancesCount?: number, alternate?: boolean): Mesh;
        /**
         * Registers for this mesh a javascript function called just before the rendering process.
         * This function is passed the current mesh.
         * Return the Mesh.
         */
        registerBeforeRender(func: (mesh: AbstractMesh) => void): Mesh;
        /**
         * Disposes a previously registered javascript function called before the rendering.
         * This function is passed the current mesh.
         * Returns the Mesh.
         */
        unregisterBeforeRender(func: (mesh: AbstractMesh) => void): Mesh;
        /**
         * Registers for this mesh a javascript function called just after the rendering is complete.
         * This function is passed the current mesh.
         * Returns the Mesh.
         */
        registerAfterRender(func: (mesh: AbstractMesh) => void): Mesh;
        /**
         * Disposes a previously registered javascript function called after the rendering.
         * This function is passed the current mesh.
         * Return the Mesh.
         */
        unregisterAfterRender(func: (mesh: AbstractMesh) => void): Mesh;
        _getInstancesRenderList(subMeshId: number): _InstancesBatch;
        _renderWithInstances(subMesh: SubMesh, fillMode: number, batch: _InstancesBatch, effect: Effect, engine: Engine): Mesh;
        _processRendering(subMesh: SubMesh, effect: Effect, fillMode: number, batch: _InstancesBatch, hardwareInstancedRendering: boolean, onBeforeDraw: (isInstance: boolean, world: Matrix, effectiveMaterial?: Material) => void, effectiveMaterial?: Material): Mesh;
        /**
         * Triggers the draw call for the mesh.
         * Usually, you don't need to call this method by your own because the mesh rendering is handled by the scene rendering manager.
         * Returns the Mesh.
         */
        render(subMesh: SubMesh, enableAlphaMode: boolean): Mesh;
        private _onBeforeDraw(isInstance, world, effectiveMaterial?);
        /**
         * Returns an array populated with ParticleSystem objects whose the mesh is the emitter.
         */
        getEmittedParticleSystems(): IParticleSystem[];
        /**
         * Returns an array populated with ParticleSystem objects whose the mesh or its children are the emitter.
         */
        getHierarchyEmittedParticleSystems(): IParticleSystem[];
        /**
         * Normalize matrix weights so that all vertices have a total weight set to 1
         */
        cleanMatrixWeights(): void;
        _checkDelayState(): Mesh;
        private _queueLoad(scene);
        /**
         * Boolean, true is the mesh in the frustum defined by the Plane objects from the `frustumPlanes` array parameter.
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Sets the mesh material by the material or multiMaterial `id` property.
         * The material `id` is a string identifying the material or the multiMaterial.
         * This method returns the Mesh.
         */
        setMaterialByID(id: string): Mesh;
        /**
         * Returns as a new array populated with the mesh material and/or skeleton, if any.
         */
        getAnimatables(): IAnimatable[];
        /**
         * Modifies the mesh geometry according to the passed transformation matrix.
         * This method returns nothing but it really modifies the mesh even if it's originally not set as updatable.
         * The mesh normals are modified accordingly the same transformation.
         * tuto : http://doc.babylonjs.com/resources/baking_transformations
         * Note that, under the hood, this method sets a new VertexBuffer each call.
         * Returns the Mesh.
         */
        bakeTransformIntoVertices(transform: Matrix): Mesh;
        /**
         * Modifies the mesh geometry according to its own current World Matrix.
         * The mesh World Matrix is then reset.
         * This method returns nothing but really modifies the mesh even if it's originally not set as updatable.
         * tuto : tuto : http://doc.babylonjs.com/resources/baking_transformations
         * Note that, under the hood, this method sets a new VertexBuffer each call.
         * Returns the Mesh.
         */
        bakeCurrentTransformIntoVertices(): Mesh;
        readonly _positions: Nullable<Vector3[]>;
        _resetPointsArrayCache(): Mesh;
        _generatePointsArray(): boolean;
        /**
         * Returns a new Mesh object generated from the current mesh properties.
         * This method must not get confused with createInstance().
         * The parameter `name` is a string, the name given to the new mesh.
         * The optional parameter `newParent` can be any Node object (default `null`).
         * The optional parameter `doNotCloneChildren` (default `false`) allows/denies the recursive cloning of the original mesh children if any.
         * The parameter `clonePhysicsImpostor` (default `true`)  allows/denies the cloning in the same time of the original mesh `body` used by the physics engine, if any.
         */
        clone(name: string, newParent?: Node, doNotCloneChildren?: boolean, clonePhysicsImpostor?: boolean): Mesh;
        /**
         * Releases resources associated with this mesh.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Modifies the mesh geometry according to a displacement map.
         * A displacement map is a colored image. Each pixel color value (actually a gradient computed from red, green, blue values) will give the displacement to apply to each mesh vertex.
         * The mesh must be set as updatable. Its internal geometry is directly modified, no new buffer are allocated.
         * This method returns nothing.
         * The parameter `url` is a string, the URL from the image file is to be downloaded.
         * The parameters `minHeight` and `maxHeight` are the lower and upper limits of the displacement.
         * The parameter `onSuccess` is an optional Javascript function to be called just after the mesh is modified. It is passed the modified mesh and must return nothing.
         * The parameter `uvOffset` is an optional vector2 used to offset UV.
         * The parameter `uvScale` is an optional vector2 used to scale UV.
         *
         * Returns the Mesh.
         */
        applyDisplacementMap(url: string, minHeight: number, maxHeight: number, onSuccess?: (mesh: Mesh) => void, uvOffset?: Vector2, uvScale?: Vector2): Mesh;
        /**
         * Modifies the mesh geometry according to a displacementMap buffer.
         * A displacement map is a colored image. Each pixel color value (actually a gradient computed from red, green, blue values) will give the displacement to apply to each mesh vertex.
         * The mesh must be set as updatable. Its internal geometry is directly modified, no new buffer are allocated.
         * This method returns nothing.
         * The parameter `buffer` is a `Uint8Array` buffer containing series of `Uint8` lower than 255, the red, green, blue and alpha values of each successive pixel.
         * The parameters `heightMapWidth` and `heightMapHeight` are positive integers to set the width and height of the buffer image.
         * The parameters `minHeight` and `maxHeight` are the lower and upper limits of the displacement.
         * The parameter `uvOffset` is an optional vector2 used to offset UV.
         * The parameter `uvScale` is an optional vector2 used to scale UV.
         *
         * Returns the Mesh.
         */
        applyDisplacementMapFromBuffer(buffer: Uint8Array, heightMapWidth: number, heightMapHeight: number, minHeight: number, maxHeight: number, uvOffset?: Vector2, uvScale?: Vector2): Mesh;
        /**
         * Modify the mesh to get a flat shading rendering.
         * This means each mesh facet will then have its own normals. Usually new vertices are added in the mesh geometry to get this result.
         * This method returns the Mesh.
         * Warning : the mesh is really modified even if not set originally as updatable and, under the hood, a new VertexBuffer is allocated.
         */
        convertToFlatShadedMesh(): Mesh;
        /**
         * This method removes all the mesh indices and add new vertices (duplication) in order to unfold facets into buffers.
         * In other words, more vertices, no more indices and a single bigger VBO.
         * The mesh is really modified even if not set originally as updatable. Under the hood, a new VertexBuffer is allocated.
         * Returns the Mesh.
         */
        convertToUnIndexedMesh(): Mesh;
        /**
         * Inverses facet orientations and inverts also the normals with `flipNormals` (default `false`) if true.
         * This method returns the Mesh.
         * Warning : the mesh is really modified even if not set originally as updatable. A new VertexBuffer is created under the hood each call.
         */
        flipFaces(flipNormals?: boolean): Mesh;
        /**
         * Creates a new InstancedMesh object from the mesh model.
         * An instance shares the same properties and the same material than its model.
         * Only these properties of each instance can then be set individually :
         * - position
         * - rotation
         * - rotationQuaternion
         * - setPivotMatrix
         * - scaling
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_Instances
         * Warning : this method is not supported for Line mesh and LineSystem
         */
        createInstance(name: string): InstancedMesh;
        /**
         * Synchronises all the mesh instance submeshes to the current mesh submeshes, if any.
         * After this call, all the mesh instances have the same submeshes than the current mesh.
         * This method returns the Mesh.
         */
        synchronizeInstances(): Mesh;
        /**
         * Simplify the mesh according to the given array of settings.
         * Function will return immediately and will simplify async. It returns the Mesh.
         * @param settings a collection of simplification settings.
         * @param parallelProcessing should all levels calculate parallel or one after the other.
         * @param type the type of simplification to run.
         * @param successCallback optional success callback to be called after the simplification finished processing all settings.
         */
        simplify(settings: Array<ISimplificationSettings>, parallelProcessing?: boolean, simplificationType?: SimplificationType, successCallback?: (mesh?: Mesh, submeshIndex?: number) => void): Mesh;
        /**
         * Optimization of the mesh's indices, in case a mesh has duplicated vertices.
         * The function will only reorder the indices and will not remove unused vertices to avoid problems with submeshes.
         * This should be used together with the simplification to avoid disappearing triangles.
         * Returns the Mesh.
         * @param successCallback an optional success callback to be called after the optimization finished.
         */
        optimizeIndices(successCallback?: (mesh?: Mesh) => void): Mesh;
        serialize(serializationObject: any): void;
        _syncGeometryWithMorphTargetManager(): void;
        /**
         * Returns a new Mesh object parsed from the source provided.
         * The parameter `parsedMesh` is the source.
         * The parameter `rootUrl` is a string, it's the root URL to prefix the `delayLoadingFile` property with
         */
        static Parse(parsedMesh: any, scene: Scene, rootUrl: string): Mesh;
        /**
         * Creates a ribbon mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The ribbon is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         *
         * Please read this full tutorial to understand how to design a ribbon : http://doc.babylonjs.com/tutorials/Ribbon_Tutorial
         * The parameter `pathArray` is a required array of paths, what are each an array of successive Vector3. The pathArray parameter depicts the ribbon geometry.
         * The parameter `closeArray` (boolean, default false) creates a seam between the first and the last paths of the path array.
         * The parameter `closePath` (boolean, default false) creates a seam between the first and the last points of each path of the path array.
         * The parameter `offset` (positive integer, default : rounded half size of the pathArray length), is taken in account only if the `pathArray` is containing a single path.
         * It's the offset to join together the points from the same path. Ex : offset = 10 means the point 1 is joined to the point 11.
         * The optional parameter `instance` is an instance of an existing Ribbon object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#ribbon
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateRibbon(name: string, pathArray: Vector3[][], closeArray: boolean | undefined, closePath: boolean, offset: number, scene?: Scene, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates a plane polygonal mesh.  By default, this is a disc.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `radius` sets the radius size (float) of the polygon (default 0.5).
         * The parameter `tessellation` sets the number of polygon sides (positive integer, default 64). So a tessellation valued to 3 will build a triangle, to 4 a square, etc.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateDisc(name: string, radius: number, tessellation: number, scene?: Nullable<Scene>, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a box mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `size` sets the size (float) of each box side (default 1).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateBox(name: string, size: number, scene?: Nullable<Scene>, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a sphere mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `diameter` sets the diameter size (float) of the sphere (default 1).
         * The parameter `segments` sets the sphere number of horizontal stripes (positive integer, default 32).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateSphere(name: string, segments: number, diameter: number, scene?: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a cylinder or a cone mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `height` sets the height size (float) of the cylinder/cone (float, default 2).
         * The parameter `diameter` sets the diameter of the top and bottom cap at once (float, default 1).
         * The parameters `diameterTop` and `diameterBottom` overwrite the parameter `diameter` and set respectively the top cap and bottom cap diameter (floats, default 1). The parameter "diameterBottom" can't be zero.
         * The parameter `tessellation` sets the number of cylinder sides (positive integer, default 24). Set it to 3 to get a prism for instance.
         * The parameter `subdivisions` sets the number of rings along the cylinder height (positive integer, default 1).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateCylinder(name: string, height: number, diameterTop: number, diameterBottom: number, tessellation: number, subdivisions: any, scene?: Scene, updatable?: any, sideOrientation?: number): Mesh;
        /**
         * Creates a torus mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `diameter` sets the diameter size (float) of the torus (default 1).
         * The parameter `thickness` sets the diameter size of the tube of the torus (float, default 0.5).
         * The parameter `tessellation` sets the number of torus sides (postive integer, default 16).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTorus(name: string, diameter: number, thickness: number, tessellation: number, scene?: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a torus knot mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `radius` sets the global radius size (float) of the torus knot (default 2).
         * The parameter `radialSegments` sets the number of sides on each tube segments (positive integer, default 32).
         * The parameter `tubularSegments` sets the number of tubes to decompose the knot into (positive integer, default 32).
         * The parameters `p` and `q` are the number of windings on each axis (positive integers, default 2 and 3).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTorusKnot(name: string, radius: number, tube: number, radialSegments: number, tubularSegments: number, p: number, q: number, scene?: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a line mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * A line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function.
         * The parameter `points` is an array successive Vector3.
         * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * When updating an instance, remember that only point positions can change, not the number of points.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLines(name: string, points: Vector3[], scene?: Nullable<Scene>, updatable?: boolean, instance?: Nullable<LinesMesh>): LinesMesh;
        /**
         * Creates a dashed line mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * A dashed line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function.
         * The parameter `points` is an array successive Vector3.
         * The parameter `dashNb` is the intended total number of dashes (positive integer, default 200).
         * The parameter `dashSize` is the size of the dashes relatively the dash number (positive float, default 3).
         * The parameter `gapSize` is the size of the gap between two successive dashes relatively the dash number (positive float, default 1).
         * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * When updating an instance, remember that only point positions can change, not the number of points.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateDashedLines(name: string, points: Vector3[], dashSize: number, gapSize: number, dashNb: number, scene?: Nullable<Scene>, updatable?: boolean, instance?: LinesMesh): LinesMesh;
        /**
         * Creates a polygon mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The polygon's shape will depend on the input parameters and is constructed parallel to a ground mesh.
         * The parameter `shape` is a required array of successive Vector3 representing the corners of the polygon in th XoZ plane, that is y = 0 for all vectors.
         * You can set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * Remember you can only change the shape positions, not their number when updating a polygon.
         */
        static CreatePolygon(name: string, shape: Vector3[], scene: Scene, holes?: Vector3[][], updatable?: boolean, sideOrientation?: number): Mesh;
        /**
          * Creates an extruded polygon mesh, with depth in the Y direction.
          * Please consider using the same method from the MeshBuilder class instead.
         */
        static ExtrudePolygon(name: string, shape: Vector3[], depth: number, scene: Scene, holes?: Vector3[][], updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates an extruded shape mesh.
         * The extrusion is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * Please consider using the same method from the MeshBuilder class instead.
         *
         * Please read this full tutorial to understand how to design an extruded shape : http://doc.babylonjs.com/how_to/parametric_shapes#extruded-shapes
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be
         * extruded along the Z axis.
         * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * The parameter `rotation` (float, default 0 radians) is the angle value to rotate the shape each step (each path point), from the former step (so rotation added each step) along the curve.
         * The parameter `scale` (float, default 1) is the value to scale the shape.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static ExtrudeShape(name: string, shape: Vector3[], path: Vector3[], scale: number, rotation: number, cap: number, scene?: Nullable<Scene>, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates an custom extruded shape mesh.
         * The custom extrusion is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * Please consider using the same method from the MeshBuilder class instead.
         *
         * Please read this full tutorial to understand how to design a custom extruded shape : http://doc.babylonjs.com/how_to/parametric_shapes#extruded-shapes
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be
         * extruded along the Z axis.
         * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * The parameter `rotationFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path
         * and the distance of this point from the begining of the path :
         * ```javascript
         * var rotationFunction = function(i, distance) {
         *     // do things
         *     return rotationValue; }
         * ```
         * It must returns a float value that will be the rotation in radians applied to the shape on each path point.
         * The parameter `scaleFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path
         * and the distance of this point from the begining of the path :
         * ```javascript
         * var scaleFunction = function(i, distance) {
         *     // do things
         *    return scaleValue;}
         * ```
         * It must returns a float value that will be the scale value applied to the shape on each path point.
         * The parameter `ribbonClosePath` (boolean, default false) forces the extrusion underlying ribbon to close all the paths in its `pathArray`.
         * The parameter `ribbonCloseArray` (boolean, default false) forces the extrusion underlying ribbon to close its `pathArray`.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static ExtrudeShapeCustom(name: string, shape: Vector3[], path: Vector3[], scaleFunction: Function, rotationFunction: Function, ribbonCloseArray: boolean, ribbonClosePath: boolean, cap: number, scene: Scene, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates lathe mesh.
         * The lathe is a shape with a symetry axis : a 2D model shape is rotated around this axis to design the lathe.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be rotated in its local space : the shape must be designed in the xOy plane and will be
         * rotated around the Y axis. It's usually a 2D shape, so the Vector3 z coordinates are often set to zero.
         * The parameter `radius` (positive float, default 1) is the radius value of the lathe.
         * The parameter `tessellation` (positive integer, default 64) is the side number of the lathe.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLathe(name: string, shape: Vector3[], radius: number, tessellation: number, scene: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a plane mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `size` sets the size (float) of both sides of the plane at once (default 1).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreatePlane(name: string, size: number, scene: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a ground mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameters `width` and `height` (floats, default 1) set the width and height sizes of the ground.
         * The parameter `subdivisions` (positive integer) sets the number of subdivisions per side.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateGround(name: string, width: number, height: number, subdivisions: number, scene?: Scene, updatable?: boolean): Mesh;
        /**
         * Creates a tiled ground mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameters `xmin` and `xmax` (floats, default -1 and 1) set the ground minimum and maximum X coordinates.
         * The parameters `zmin` and `zmax` (floats, default -1 and 1) set the ground minimum and maximum Z coordinates.
         * The parameter `subdivisions` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 6, h: 6}`). `w` and `h` are the
         * numbers of subdivisions on the ground width and height. Each subdivision is called a tile.
         * The parameter `precision` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 2, h: 2}`). `w` and `h` are the
         * numbers of subdivisions on the ground width and height of each tile.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTiledGround(name: string, xmin: number, zmin: number, xmax: number, zmax: number, subdivisions: {
            w: number;
            h: number;
        }, precision: {
            w: number;
            h: number;
        }, scene: Scene, updatable?: boolean): Mesh;
        /**
         * Creates a ground mesh from a height map.
         * tuto : http://doc.babylonjs.com/babylon101/height_map
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `url` sets the URL of the height map image resource.
         * The parameters `width` and `height` (positive floats, default 10) set the ground width and height sizes.
         * The parameter `subdivisions` (positive integer, default 1) sets the number of subdivision per side.
         * The parameter `minHeight` (float, default 0) is the minimum altitude on the ground.
         * The parameter `maxHeight` (float, default 1) is the maximum altitude on the ground.
         * The parameter `onReady` is a javascript callback function that will be called  once the mesh is just built (the height map download can last some time).
         * This function is passed the newly built mesh :
         * ```javascript
         * function(mesh) { // do things
         *     return; }
         * ```
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateGroundFromHeightMap(name: string, url: string, width: number, height: number, subdivisions: number, minHeight: number, maxHeight: number, scene: Scene, updatable?: boolean, onReady?: (mesh: GroundMesh) => void): GroundMesh;
        /**
         * Creates a tube mesh.
         * The tube is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `path` is a required array of successive Vector3. It is the curve used as the axis of the tube.
         * The parameter `radius` (positive float, default 1) sets the tube radius size.
         * The parameter `tessellation` (positive float, default 64) is the number of sides on the tubular surface.
         * The parameter `radiusFunction` (javascript function, default null) is a vanilla javascript function. If it is not null, it overwrittes the parameter `radius`.
         * This function is called on each point of the tube path and is passed the index `i` of the i-th point and the distance of this point from the first point of the path.
         * It must return a radius value (positive float) :
         * ```javascript
         * var radiusFunction = function(i, distance) {
         *     // do things
         *     return radius; }
         * ```
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing Tube object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#tube
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTube(name: string, path: Vector3[], radius: number, tessellation: number, radiusFunction: {
            (i: number, distance: number): number;
        }, cap: number, scene: Scene, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates a polyhedron mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `type` (positive integer, max 14, default 0) sets the polyhedron type to build among the 15 embbeded types. Please refer to the type sheet in the tutorial
         *  to choose the wanted type.
         * The parameter `size` (positive float, default 1) sets the polygon size.
         * You can overwrite the `size` on each dimension bu using the parameters `sizeX`, `sizeY` or `sizeZ` (positive floats, default to `size` value).
         * You can build other polyhedron types than the 15 embbeded ones by setting the parameter `custom` (`polyhedronObject`, default null). If you set the parameter `custom`, this overwrittes the parameter `type`.
         * A `polyhedronObject` is a formatted javascript object. You'll find a full file with pre-set polyhedra here : https://github.com/BabylonJS/Extensions/tree/master/Polyhedron
         * You can set the color and the UV of each side of the polyhedron with the parameters `faceColors` (Color4, default `(1, 1, 1, 1)`) and faceUV (Vector4, default `(0, 0, 1, 1)`).
         * To understand how to set `faceUV` or `faceColors`, please read this by considering the right number of faces of your polyhedron, instead of only 6 for the box : http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
         * The parameter `flat` (boolean, default true). If set to false, it gives the polyhedron a single global face, so less vertices and shared normals. In this case, `faceColors` and `faceUV` are ignored.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreatePolyhedron(name: string, options: {
            type?: number;
            size?: number;
            sizeX?: number;
            sizeY?: number;
            sizeZ?: number;
            custom?: any;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            updatable?: boolean;
            sideOrientation?: number;
        }, scene: Scene): Mesh;
        /**
         * Creates a sphere based upon an icosahedron with 20 triangular faces which can be subdivided.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `radius` sets the radius size (float) of the icosphere (default 1).
         * You can set some different icosphere dimensions, for instance to build an ellipsoid, by using the parameters `radiusX`, `radiusY` and `radiusZ` (all by default have the same value than `radius`).
         * The parameter `subdivisions` sets the number of subdivisions (postive integer, default 4). The more subdivisions, the more faces on the icosphere whatever its size.
         * The parameter `flat` (boolean, default true) gives each side its own normals. Set it to false to get a smooth continuous light reflection on the surface.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateIcoSphere(name: string, options: {
            radius?: number;
            flat?: boolean;
            subdivisions?: number;
            sideOrientation?: number;
            updatable?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a decal mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * A decal is a mesh usually applied as a model onto the surface of another mesh. So don't forget the parameter `sourceMesh` depicting the decal.
         * The parameter `position` (Vector3, default `(0, 0, 0)`) sets the position of the decal in World coordinates.
         * The parameter `normal` (Vector3, default Vector3.Up) sets the normal of the mesh where the decal is applied onto in World coordinates.
         * The parameter `size` (Vector3, default `(1, 1, 1)`) sets the decal scaling.
         * The parameter `angle` (float in radian, default 0) sets the angle to rotate the decal.
         */
        static CreateDecal(name: string, sourceMesh: AbstractMesh, position: Vector3, normal: Vector3, size: Vector3, angle: number): Mesh;
        /**
         * @returns original positions used for CPU skinning.  Useful for integrating Morphing with skeletons in same mesh.
         */
        setPositionsForCPUSkinning(): Float32Array;
        /**
         * @returns original normals used for CPU skinning.  Useful for integrating Morphing with skeletons in same mesh.
         */
        setNormalsForCPUSkinning(): Float32Array;
        /**
         * Updates the vertex buffer by applying transformation from the bones.
         * Returns the Mesh.
         *
         * @param {skeleton} skeleton to apply
         */
        applySkeleton(skeleton: Skeleton): Mesh;
        /**
         * Returns an object `{min:` Vector3`, max:` Vector3`}`
         * This min and max Vector3 are the minimum and maximum vectors of each mesh bounding box from the passed array, in the World system
         */
        static MinMax(meshes: AbstractMesh[]): {
            min: Vector3;
            max: Vector3;
        };
        /**
         * Returns a Vector3, the center of the `{min:` Vector3`, max:` Vector3`}` or the center of MinMax vector3 computed from a mesh array.
         */
        static Center(meshesOrMinMaxVector: {
            min: Vector3;
            max: Vector3;
        } | AbstractMesh[]): Vector3;
        /**
         * Merge the array of meshes into a single mesh for performance reasons.
         * @param {Array<Mesh>} meshes - The vertices source.  They should all be of the same material.  Entries can empty
         * @param {boolean} disposeSource - When true (default), dispose of the vertices from the source meshes
         * @param {boolean} allow32BitsIndices - When the sum of the vertices > 64k, this must be set to true.
         * @param {Mesh} meshSubclass - When set, vertices inserted into this Mesh.  Meshes can then be merged into a Mesh sub-class.
         * @param {boolean} subdivideWithSubMeshes - When true (false default), subdivide mesh to his subMesh array with meshes source.
         */
        static MergeMeshes(meshes: Array<Mesh>, disposeSource?: boolean, allow32BitsIndices?: boolean, meshSubclass?: Mesh, subdivideWithSubMeshes?: boolean): Nullable<Mesh>;
    }
}

declare module BABYLON {
    /**
     * Define an interface for all classes that will get and set the data on vertices
     */
    interface IGetSetVerticesData {
        isVerticesDataPresent(kind: string): boolean;
        getVerticesData(kind: string, copyWhenShared?: boolean, forceCopy?: boolean): Nullable<FloatArray>;
        getIndices(copyWhenShared?: boolean): Nullable<IndicesArray>;
        setVerticesData(kind: string, data: FloatArray, updatable: boolean): void;
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): void;
        setIndices(indices: IndicesArray, totalVertices: Nullable<number>, updatable?: boolean): void;
    }
    /**
     * This class contains the various kinds of data on every vertex of a mesh used in determining its shape and appearance
     */
    class VertexData {
        /**
         * An array of the x, y, z position of each vertex  [...., x, y, z, .....]
         */
        positions: Nullable<FloatArray>;
        /**
         * An array of the x, y, z normal vector of each vertex  [...., x, y, z, .....]
         */
        normals: Nullable<FloatArray>;
        /**
         * An array of the x, y, z tangent vector of each vertex  [...., x, y, z, .....]
         */
        tangents: Nullable<FloatArray>;
        /**
         * An array of u,v which maps a texture image onto each vertex  [...., u, v, .....]
         */
        uvs: Nullable<FloatArray>;
        /**
         * A second array of u,v which maps a texture image onto each vertex  [...., u, v, .....]
         */
        uvs2: Nullable<FloatArray>;
        /**
         * A third array of u,v which maps a texture image onto each vertex  [...., u, v, .....]
         */
        uvs3: Nullable<FloatArray>;
        /**
         * A fourth array of u,v which maps a texture image onto each vertex  [...., u, v, .....]
         */
        uvs4: Nullable<FloatArray>;
        /**
         * A fifth array of u,v which maps a texture image onto each vertex  [...., u, v, .....]
         */
        uvs5: Nullable<FloatArray>;
        /**
         * A sixth array of u,v which maps a texture image onto each vertex  [...., u, v, .....]
         */
        uvs6: Nullable<FloatArray>;
        /**
         * An array of the r, g, b, a, color of each vertex  [...., r, g, b, a, .....]
         */
        colors: Nullable<FloatArray>;
        /**
         * An array containing the list of indices to the array of matrices produced by bones, each vertex have up to 4 indices (8 if the matricesIndicesExtra is set).
         */
        matricesIndices: Nullable<FloatArray>;
        /**
         * An array containing the list of weights defining the weight of each indexed matrix in the final computation
         */
        matricesWeights: Nullable<FloatArray>;
        /**
         * An array extending the number of possible indices
         */
        matricesIndicesExtra: Nullable<FloatArray>;
        /**
         * An array extending the number of possible weights when the number of indices is extended
         */
        matricesWeightsExtra: Nullable<FloatArray>;
        /**
         * An array of i, j, k the three vertex indices required for each triangular facet  [...., i, j, k .....]
         */
        indices: Nullable<IndicesArray>;
        /**
         * Uses the passed data array to set the set the values for the specified kind of data
         * @param data a linear array of floating numbers
         * @param kind the type of data that is being set, eg positions, colors etc
         */
        set(data: FloatArray, kind: string): void;
        /**
         * Associates the vertexData to the passed Mesh.
         * Sets it as updatable or not (default `false`)
         * @param mesh the mesh the vertexData is applied to
         * @param updatable when used and having the value true allows new data to update the vertexData
         * @returns the VertexData
         */
        applyToMesh(mesh: Mesh, updatable?: boolean): VertexData;
        /**
         * Associates the vertexData to the passed Geometry.
         * Sets it as updatable or not (default `false`)
         * @param geometry the geometry the vertexData is applied to
         * @param updatable when used and having the value true allows new data to update the vertexData
         * @returns VertexData
         */
        applyToGeometry(geometry: Geometry, updatable?: boolean): VertexData;
        /**
         * Updates the associated mesh
         * @param mesh the mesh to be updated
         * @param updateExtends when true the mesh BoundingInfo will be renewed when and if position kind is updated, optional with default false
         * @param makeItUnique when true, and when and if position kind is updated, a new global geometry will be  created from these positions and set to the mesh, optional with default false
         * @returns VertexData
         */
        updateMesh(mesh: Mesh, updateExtends?: boolean, makeItUnique?: boolean): VertexData;
        /**
         * Updates the associated geometry
         * @param geometry the geometry to be updated
         * @param updateExtends when true BoundingInfo will be renewed when and if position kind is updated, optional with default false
         * @param makeItUnique when true, and when and if position kind is updated, a new global geometry will be created from these positions and set to the mesh, optional with default false
         * @returns VertexData.
         */
        updateGeometry(geometry: Geometry, updateExtends?: boolean, makeItUnique?: boolean): VertexData;
        private _applyTo(meshOrGeometry, updatable?);
        private _update(meshOrGeometry, updateExtends?, makeItUnique?);
        /**
         * Transforms each position and each normal of the vertexData according to the passed Matrix
         * @param matrix the transforming matrix
         * @returns the VertexData
         */
        transform(matrix: Matrix): VertexData;
        /**
         * Merges the passed VertexData into the current one
         * @param other the VertexData to be merged into the current one
         * @returns the modified VertexData
         */
        merge(other: VertexData): VertexData;
        private _mergeElement(source, other);
        private _validate();
        /**
         * Serializes the VertexData
         * @returns a serialized object
         */
        serialize(): any;
        /**
         * Extracts the vertexData from a mesh
         * @param mesh the mesh from which to extract the VertexData
         * @param copyWhenShared defines if the VertexData must be cloned when shared between multiple meshes, optional, default false
         * @param forceCopy indicating that the VertexData must be cloned, optional, default false
         * @returns the object VertexData associated to the passed mesh
         */
        static ExtractFromMesh(mesh: Mesh, copyWhenShared?: boolean, forceCopy?: boolean): VertexData;
        /**
         * Extracts the vertexData from the geometry
         * @param geometry the geometry from which to extract the VertexData
         * @param copyWhenShared defines if the VertexData must be cloned when the geometrty is shared between multiple meshes, optional, default false
         * @param forceCopy indicating that the VertexData must be cloned, optional, default false
         * @returns the object VertexData associated to the passed mesh
         */
        static ExtractFromGeometry(geometry: Geometry, copyWhenShared?: boolean, forceCopy?: boolean): VertexData;
        private static _ExtractFrom(meshOrGeometry, copyWhenShared?, forceCopy?);
        /**
         * Creates the VertexData for a Ribbon
         * @param options an object used to set the following optional parameters for the ribbon, required but can be empty
          * * pathArray array of paths, each of which an array of successive Vector3
          * * closeArray creates a seam between the first and the last paths of the pathArray, optional, default false
          * * closePath creates a seam between the first and the last points of each path of the path array, optional, default false
          * * offset a positive integer, only used when pathArray contains a single path (offset = 10 means the point 1 is joined to the point 11), default rounded half size of the pathArray length
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
          * * invertUV swaps in the U and V coordinates when applying a texture, optional, default false
          * * uvs a linear array, of length 2 * number of vertices, of custom UV values, optional
          * * colors a linear array, of length 4 * number of vertices, of custom color values, optional
         * @returns the VertexData of the ribbon
         */
        static CreateRibbon(options: {
            pathArray: Vector3[][];
            closeArray?: boolean;
            closePath?: boolean;
            offset?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            invertUV?: boolean;
            uvs?: Vector2[];
            colors?: Color4[];
        }): VertexData;
        /**
         * Creates the VertexData for a box
         * @param options an object used to set the following optional parameters for the box, required but can be empty
          * * size sets the width, height and depth of the box to the value of size, optional default 1
          * * width sets the width (x direction) of the box, overwrites the width set by size, optional, default size
          * * height sets the height (y direction) of the box, overwrites the height set by size, optional, default size
          * * depth sets the depth (z direction) of the box, overwrites the depth set by size, optional, default size
          * * faceUV an array of 6 Vector4 elements used to set different images to each box side
          * * faceColors an array of 6 Color3 elements used to set different colors to each box side
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the box
         */
        static CreateBox(options: {
            size?: number;
            width?: number;
            height?: number;
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData for an ellipsoid, defaults to a sphere
         * @param options an object used to set the following optional parameters for the box, required but can be empty
          * * segments sets the number of horizontal strips optional, default 32
          * * diameter sets the axes dimensions, diameterX, diameterY and diameterZ to the value of diameter, optional default 1
          * * diameterX sets the diameterX (x direction) of the ellipsoid, overwrites the diameterX set by diameter, optional, default diameter
          * * diameterY sets the diameterY (y direction) of the ellipsoid, overwrites the diameterY set by diameter, optional, default diameter
          * * diameterZ sets the diameterZ (z direction) of the ellipsoid, overwrites the diameterZ set by diameter, optional, default diameter
          * * arc a number from 0 to 1, to create an unclosed ellipsoid based on the fraction of the circumference (latitude) given by the arc value, optional, default 1
          * * slice a number from 0 to 1, to create an unclosed ellipsoid based on the fraction of the height (latitude) given by the arc value, optional, default 1
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the ellipsoid
         */
        static CreateSphere(options: {
            segments?: number;
            diameter?: number;
            diameterX?: number;
            diameterY?: number;
            diameterZ?: number;
            arc?: number;
            slice?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData for a cylinder, cone or prism
         * @param options an object used to set the following optional parameters for the box, required but can be empty
          * * height sets the height (y direction) of the cylinder, optional, default 2
          * * diameterTop sets the diameter of the top of the cone, overwrites diameter,  optional, default diameter
          * * diameterBottom sets the diameter of the bottom of the cone, overwrites diameter,  optional, default diameter
          * * diameter sets the diameter of the top and bottom of the cone, optional default 1
          * * tessellation the number of prism sides, 3 for a triangular prism, optional, default 24
          * * subdivisions` the number of rings along the cylinder height, optional, default 1
          * * arc a number from 0 to 1, to create an unclosed cylinder based on the fraction of the circumference given by the arc value, optional, default 1
          * * faceColors an array of Color3 elements used to set different colors to the top, rings and bottom respectively
          * * faceUV an array of Vector4 elements used to set different images to the top, rings and bottom respectively
          * * hasRings when true makes each subdivision independantly treated as a face for faceUV and faceColors, optional, default false
          * * enclose when true closes an open cylinder by adding extra flat faces between the height axis and vertical edges, think cut cake
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the cylinder, cone or prism
         */
        static CreateCylinder(options: {
            height?: number;
            diameterTop?: number;
            diameterBottom?: number;
            diameter?: number;
            tessellation?: number;
            subdivisions?: number;
            arc?: number;
            faceColors?: Color4[];
            faceUV?: Vector4[];
            hasRings?: boolean;
            enclose?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData for a torus
         * @param options an object used to set the following optional parameters for the box, required but can be empty
          * * diameter the diameter of the torus, optional default 1
          * * thickness the diameter of the tube forming the torus, optional default 0.5
          * * tessellation the number of prism sides, 3 for a triangular prism, optional, default 24
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the torus
         */
        static CreateTorus(options: {
            diameter?: number;
            thickness?: number;
            tessellation?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the LineSystem
         * @param options an object used to set the following optional parameters for the LineSystem, required but can be empty
         *  - lines an array of lines, each line being an array of successive Vector3
         *  - colors an array of line colors, each of the line colors being an array of successive Color4, one per line point
         * @returns the VertexData of the LineSystem
         */
        static CreateLineSystem(options: {
            lines: Vector3[][];
            colors?: Nullable<Color4[][]>;
        }): VertexData;
        /**
         * Create the VertexData for a DashedLines
         * @param options an object used to set the following optional parameters for the DashedLines, required but can be empty
         *  - points an array successive Vector3
         *  - dashSize the size of the dashes relative to the dash number, optional, default 3
         *  - gapSize the size of the gap between two successive dashes relative to the dash number, optional, default 1
         *  - dashNb the intended total number of dashes, optional, default 200
         * @returns the VertexData for the DashedLines
         */
        static CreateDashedLines(options: {
            points: Vector3[];
            dashSize?: number;
            gapSize?: number;
            dashNb?: number;
        }): VertexData;
        /**
         * Creates the VertexData for a Ground
         * @param options an object used to set the following optional parameters for the Ground, required but can be empty
         *  - width the width (x direction) of the ground, optional, default 1
         *  - height the height (z direction) of the ground, optional, default 1
         *  - subdivisions the number of subdivisions per side, optional, default 1
         * @returns the VertexData of the Ground
         */
        static CreateGround(options: {
            width?: number;
            height?: number;
            subdivisions?: number;
            subdivisionsX?: number;
            subdivisionsY?: number;
        }): VertexData;
        /**
         * Creates the VertexData for a TiledGround by subdividing the ground into tiles
         * @param options an object used to set the following optional parameters for the Ground, required but can be empty
          * * xmin the ground minimum X coordinate, optional, default -1
          * * zmin the ground minimum Z coordinate, optional, default -1
          * * xmax the ground maximum X coordinate, optional, default 1
          * * zmax the ground maximum Z coordinate, optional, default 1
          * * subdivisions a javascript object {w: positive integer, h: positive integer}, `w` and `h` are the numbers of subdivisions on the ground width and height creating 'tiles', default {w: 6, h: 6}
          * * precision a javascript object {w: positive integer, h: positive integer}, `w` and `h` are the numbers of subdivisions on the tile width and height, default {w: 2, h: 2}
         * @returns the VertexData of the TiledGround
         */
        static CreateTiledGround(options: {
            xmin: number;
            zmin: number;
            xmax: number;
            zmax: number;
            subdivisions?: {
                w: number;
                h: number;
            };
            precision?: {
                w: number;
                h: number;
            };
        }): VertexData;
        /**
         * Creates the VertexData of the Ground designed from a heightmap
         * @param options an object used to set the following parameters for the Ground, required and provided by MeshBuilder.CreateGroundFromHeightMap
          * * width the width (x direction) of the ground
          * * height the height (z direction) of the ground
          * * subdivisions the number of subdivisions per side
          * * minHeight the minimum altitude on the ground, optional, default 0
          * * maxHeight the maximum altitude on the ground, optional default 1
          * * colorFilter the filter to apply to the image pixel colors to compute the height, optional Color3, default (0.3, 0.59, 0.11)
          * * buffer the array holding the image color data
          * * bufferWidth the width of image
          * * bufferHeight the height of image
         * @returns the VertexData of the Ground designed from a heightmap
         */
        static CreateGroundFromHeightMap(options: {
            width: number;
            height: number;
            subdivisions: number;
            minHeight: number;
            maxHeight: number;
            colorFilter: Color3;
            buffer: Uint8Array;
            bufferWidth: number;
            bufferHeight: number;
        }): VertexData;
        /**
         * Creates the VertexData for a Plane
         * @param options an object used to set the following optional parameters for the plane, required but can be empty
          * * size sets the width and height of the plane to the value of size, optional default 1
          * * width sets the width (x direction) of the plane, overwrites the width set by size, optional, default size
          * * height sets the height (y direction) of the plane, overwrites the height set by size, optional, default size
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the box
         */
        static CreatePlane(options: {
            size?: number;
            width?: number;
            height?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Disc or regular Polygon
         * @param options an object used to set the following optional parameters for the disc, required but can be empty
          * * radius the radius of the disc, optional default 0.5
          * * tessellation the number of polygon sides, optional, default 64
          * * arc a number from 0 to 1, to create an unclosed polygon based on the fraction of the circumference given by the arc value, optional, default 1
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the box
         */
        static CreateDisc(options: {
            radius?: number;
            tessellation?: number;
            arc?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData for an irregular Polygon in the XoZ plane using a mesh built by polygonTriangulation.build()
         * All parameters are provided by MeshBuilder.CreatePolygon as needed
         * @param polygon a mesh built from polygonTriangulation.build()
         * @param sideOrientation takes the values BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * @param fUV an array of Vector4 elements used to set different images to the top, rings and bottom respectively
         * @param fColors an array of Color3 elements used to set different colors to the top, rings and bottom respectively
         * @param frontUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
         * @param backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the Polygon
         */
        static CreatePolygon(polygon: Mesh, sideOrientation: number, fUV?: Vector4[], fColors?: Color4[], frontUVs?: Vector4, backUVs?: Vector4): VertexData;
        /**
         * Creates the VertexData of the IcoSphere
         * @param options an object used to set the following optional parameters for the IcoSphere, required but can be empty
          * * radius the radius of the IcoSphere, optional default 1
          * * radiusX allows stretching in the x direction, optional, default radius
          * * radiusY allows stretching in the y direction, optional, default radius
          * * radiusZ allows stretching in the z direction, optional, default radius
          * * flat when true creates a flat shaded mesh, optional, default true
          * * subdivisions increasing the subdivisions increases the number of faces, optional, default 4
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the IcoSphere
         */
        static CreateIcoSphere(options: {
            radius?: number;
            radiusX?: number;
            radiusY?: number;
            radiusZ?: number;
            flat?: boolean;
            subdivisions?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData for a Polyhedron
         * @param options an object used to set the following optional parameters for the polyhedron, required but can be empty
         * * type provided types are:
         *  * 0 : Tetrahedron, 1 : Octahedron, 2 : Dodecahedron, 3 : Icosahedron, 4 : Rhombicuboctahedron, 5 : Triangular Prism, 6 : Pentagonal Prism, 7 : Hexagonal Prism, 8 : Square Pyramid (J1)
         *  * 9 : Pentagonal Pyramid (J2), 10 : Triangular Dipyramid (J12), 11 : Pentagonal Dipyramid (J13), 12 : Elongated Square Dipyramid (J15), 13 : Elongated Pentagonal Dipyramid (J16), 14 : Elongated Pentagonal Cupola (J20)
         * * size the size of the IcoSphere, optional default 1
         * * sizeX allows stretching in the x direction, optional, default size
         * * sizeY allows stretching in the y direction, optional, default size
         * * sizeZ allows stretching in the z direction, optional, default size
         * * custom a number that overwrites the type to create from an extended set of polyhedron from https://www.babylonjs-playground.com/#21QRSK#15 with minimised editor
         * * faceUV an array of Vector4 elements used to set different images to the top, rings and bottom respectively
         * * faceColors an array of Color3 elements used to set different colors to the top, rings and bottom respectively
         * * flat when true creates a flat shaded mesh, optional, default true
         * * subdivisions increasing the subdivisions increases the number of faces, optional, default 4
         * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
         * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the Polyhedron
         */
        static CreatePolyhedron(options: {
            type?: number;
            size?: number;
            sizeX?: number;
            sizeY?: number;
            sizeZ?: number;
            custom?: any;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            flat?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData for a TorusKnot
         * @param options an object used to set the following optional parameters for the TorusKnot, required but can be empty
          * * radius the radius of the torus knot, optional, default 2
          * * tube the thickness of the tube, optional, default 0.5
          * * radialSegments the number of sides on each tube segments, optional, default 32
          * * tubularSegments the number of tubes to decompose the knot into, optional, default 32
          * * p the number of windings around the z axis, optional,  default 2
          * * q the number of windings around the x axis, optional,  default 3
          * * sideOrientation optional and takes the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
          * * frontUvs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the front side, optional, default vector4 (0, 0, 1, 1)
          * * backUVs only usable when you create a double-sided mesh, used to choose what parts of the texture image to crop and apply on the back side, optional, default vector4 (0, 0, 1, 1)
         * @returns the VertexData of the Torus Knot
         */
        static CreateTorusKnot(options: {
            radius?: number;
            tube?: number;
            radialSegments?: number;
            tubularSegments?: number;
            p?: number;
            q?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Compute normals for given positions and indices
         * @param positions an array of vertex positions, [...., x, y, z, ......]
         * @param indices an array of indices in groups of three for each triangular facet, [...., i, j, k, ......]
         * @param normals an array of vertex normals, [...., x, y, z, ......]
         * @param options an object used to set the following optional parameters for the TorusKnot, optional
          * * facetNormals : optional array of facet normals (vector3)
          * * facetPositions : optional array of facet positions (vector3)
          * * facetPartitioning : optional partitioning array. facetPositions is required for facetPartitioning computation
          * * ratio : optional partitioning ratio / bounding box, required for facetPartitioning computation
          * * bInfo : optional bounding info, required for facetPartitioning computation
          * * bbSize : optional bounding box size data, required for facetPartitioning computation
          * * subDiv : optional partitioning data about subdivsions on  each axis (int), required for facetPartitioning computation
          * * useRightHandedSystem: optional boolean to for right handed system computation
          * * depthSort : optional boolean to enable the facet depth sort computation
          * * distanceTo : optional Vector3 to compute the facet depth from this location
          * * depthSortedFacets : optional array of depthSortedFacets to store the facet distances from the reference location
         */
        static ComputeNormals(positions: any, indices: any, normals: any, options?: {
            facetNormals?: any;
            facetPositions?: any;
            facetPartitioning?: any;
            ratio?: number;
            bInfo?: any;
            bbSize?: Vector3;
            subDiv?: any;
            useRightHandedSystem?: boolean;
            depthSort?: boolean;
            distanceTo?: Vector3;
            depthSortedFacets?: any;
        }): void;
        private static _ComputeSides(sideOrientation, positions, indices, normals, uvs, frontUVs?, backUVs?);
        /**
         * Applies VertexData created from the imported parameters to the geometry
         * @param parsedVertexData the parsed data from an imported file
         * @param geometry the geometry to apply the VertexData to
         */
        static ImportVertexData(parsedVertexData: any, geometry: Geometry): void;
    }
}

declare module BABYLON {
    /**
     * Class containing static functions to help procedurally build meshes
     */
    class MeshBuilder {
        private static updateSideOrientation(orientation?);
        /**
         * Creates a box mesh
         * * The parameter `size` sets the size (float) of each box side (default 1)
         * * You can set some different box dimensions by using the parameters `width`, `height` and `depth` (all by default have the same value than `size`)
         * * You can set different colors and different images to each box side by using the parameters `faceColors` (an array of 6 Color3 elements) and `faceUV` (an array of 6 Vector4 elements)
         * * Please read this tutorial : http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#box
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the box mesh
         */
        static CreateBox(name: string, options: {
            size?: number;
            width?: number;
            height?: number;
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates a sphere mesh
         * * The parameter `diameter` sets the diameter size (float) of the sphere (default 1)
         * * You can set some different sphere dimensions, for instance to build an ellipsoid, by using the parameters `diameterX`, `diameterY` and `diameterZ` (all by default have the same value than `diameter`)
         * * The parameter `segments` sets the sphere number of horizontal stripes (positive integer, default 32)
         * * You can create an unclosed sphere with the parameter `arc` (positive float, default 1), valued between 0 and 1, what is the ratio of the circumference (latitude) : 2 x PI x ratio
         * * You can create an unclosed sphere on its height with the parameter `slice` (positive float, default1), valued between 0 and 1, what is the height ratio (longitude)
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the sphere mesh
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#sphere
         */
        static CreateSphere(name: string, options: {
            segments?: number;
            diameter?: number;
            diameterX?: number;
            diameterY?: number;
            diameterZ?: number;
            arc?: number;
            slice?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
        }, scene: any): Mesh;
        /**
         * Creates a plane polygonal mesh.  By default, this is a disc
         * * The parameter `radius` sets the radius size (float) of the polygon (default 0.5)
         * * The parameter `tessellation` sets the number of polygon sides (positive integer, default 64). So a tessellation valued to 3 will build a triangle, to 4 a square, etc
         * * You can create an unclosed polygon with the parameter `arc` (positive float, default 1), valued between 0 and 1, what is the ratio of the circumference : 2 x PI x ratio
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the plane polygonal mesh
         * @see http://doc.babylonjs.com/how_to/set_shapes#disc-or-regular-polygon
         */
        static CreateDisc(name: string, options: {
            radius?: number;
            tessellation?: number;
            arc?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates a sphere based upon an icosahedron with 20 triangular faces which can be subdivided
         * * The parameter `radius` sets the radius size (float) of the icosphere (default 1)
         * * You can set some different icosphere dimensions, for instance to build an ellipsoid, by using the parameters `radiusX`, `radiusY` and `radiusZ` (all by default have the same value than `radius`)
         * * The parameter `subdivisions` sets the number of subdivisions (postive integer, default 4). The more subdivisions, the more faces on the icosphere whatever its size
         * * The parameter `flat` (boolean, default true) gives each side its own normals. Set it to false to get a smooth continuous light reflection on the surface
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the icosahedron mesh
         * @see http://doc.babylonjs.com/how_to/polyhedra_shapes#icosphere
         */
        static CreateIcoSphere(name: string, options: {
            radius?: number;
            radiusX?: number;
            radiusY?: number;
            radiusZ?: number;
            flat?: boolean;
            subdivisions?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a ribbon mesh. The ribbon is a parametric shape.  It has no predefined shape. Its final shape will depend on the input parameters
         * * The parameter `pathArray` is a required array of paths, what are each an array of successive Vector3. The pathArray parameter depicts the ribbon geometry
         * * The parameter `closeArray` (boolean, default false) creates a seam between the first and the last paths of the path array
         * * The parameter `closePath` (boolean, default false) creates a seam between the first and the last points of each path of the path array
         * * The parameter `offset` (positive integer, default : rounded half size of the pathArray length), is taken in account only if the `pathArray` is containing a single path
         * * It's the offset to join the points from the same path. Ex : offset = 10 means the point 1 is joined to the point 11
         * * The optional parameter `instance` is an instance of an existing Ribbon object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#ribbon
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture
         * * The parameter `uvs` is an optional flat array of `Vector2` to update/set each ribbon vertex with its own custom UV values instead of the computed ones
         * * The parameters `colors` is an optional flat array of `Color4` to set/update each ribbon vertex with its own custom color values
         * * Note that if you use the parameters `uvs` or `colors`, the passed arrays must be populated with the right number of elements, it is to say the number of ribbon vertices. Remember that if you set `closePath` to `true`, there's one extra vertex per path in the geometry
         * * Moreover, you can use the parameter `color` with `instance` (to update the ribbon), only if you previously used it at creation time
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the ribbon mesh
         * @see http://doc.babylonjs.com/tutorials/Ribbon_Tutorial
         * @see http://doc.babylonjs.com/tutorials/Parametric_Shapes
         */
        static CreateRibbon(name: string, options: {
            pathArray: Vector3[][];
            closeArray?: boolean;
            closePath?: boolean;
            offset?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
            uvs?: Vector2[];
            colors?: Color4[];
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates a cylinder or a cone mesh
         * * The parameter `height` sets the height size (float) of the cylinder/cone (float, default 2).
         * * The parameter `diameter` sets the diameter of the top and bottom cap at once (float, default 1).
         * * The parameters `diameterTop` and `diameterBottom` overwrite the parameter `diameter` and set respectively the top cap and bottom cap diameter (floats, default 1). The parameter "diameterBottom" can't be zero.
         * * The parameter `tessellation` sets the number of cylinder sides (positive integer, default 24). Set it to 3 to get a prism for instance.
         * * The parameter `subdivisions` sets the number of rings along the cylinder height (positive integer, default 1).
         * * The parameter `hasRings` (boolean, default false) makes the subdivisions independent from each other, so they become different faces.
         * * The parameter `enclose`  (boolean, default false) adds two extra faces per subdivision to a sliced cylinder to close it around its height axis.
         * * The parameter `arc` (float, default 1) is the ratio (max 1) to apply to the circumference to slice the cylinder.
         * * You can set different colors and different images to each box side by using the parameters `faceColors` (an array of n Color3 elements) and `faceUV` (an array of n Vector4 elements).
         * * The value of n is the number of cylinder faces. If the cylinder has only 1 subdivisions, n equals : top face + cylinder surface + bottom face = 3
         * * Now, if the cylinder has 5 independent subdivisions (hasRings = true), n equals : top face + 5 stripe surfaces + bottom face = 2 + 5 = 7
         * * Finally, if the cylinder has 5 independent subdivisions and is enclose, n equals : top face + 5 x (stripe surface + 2 closing faces) + bottom face = 2 + 5 * 3 = 17
         * * Each array (color or UVs) is always ordered the same way : the first element is the bottom cap, the last element is the top cap. The other elements are each a ring surface.
         * * If `enclose` is false, a ring surface is one element.
         * * If `enclose` is true, a ring surface is 3 successive elements in the array : the tubular surface, then the two closing faces.
         * * Example how to set colors and textures on a sliced cylinder : http://www.html5gamedevs.com/topic/17945-creating-a-closed-slice-of-a-cylinder/#comment-106379
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the cylinder mesh
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#cylinder-or-cone
         */
        static CreateCylinder(name: string, options: {
            height?: number;
            diameterTop?: number;
            diameterBottom?: number;
            diameter?: number;
            tessellation?: number;
            subdivisions?: number;
            arc?: number;
            faceColors?: Color4[];
            faceUV?: Vector4[];
            updatable?: boolean;
            hasRings?: boolean;
            enclose?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: any): Mesh;
        /**
         * Creates a torus mesh
         * * The parameter `diameter` sets the diameter size (float) of the torus (default 1)
         * * The parameter `thickness` sets the diameter size of the tube of the torus (float, default 0.5)
         * * The parameter `tessellation` sets the number of torus sides (postive integer, default 16)
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the torus mesh
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#torus
         */
        static CreateTorus(name: string, options: {
            diameter?: number;
            thickness?: number;
            tessellation?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: any): Mesh;
        /**
         * Creates a torus knot mesh
         * * The parameter `radius` sets the global radius size (float) of the torus knot (default 2)
         * * The parameter `radialSegments` sets the number of sides on each tube segments (positive integer, default 32)
         * * The parameter `tubularSegments` sets the number of tubes to decompose the knot into (positive integer, default 32)
         * * The parameters `p` and `q` are the number of windings on each axis (positive integers, default 2 and 3)
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the torus knot mesh
         * @see  http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#torus-knot
         */
        static CreateTorusKnot(name: string, options: {
            radius?: number;
            tube?: number;
            radialSegments?: number;
            tubularSegments?: number;
            p?: number;
            q?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: any): Mesh;
        /**
         * Creates a line system mesh. A line system is a pool of many lines gathered in a single mesh
         * * A line system mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of lines as an input parameter
         * * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineSystem to this static function
         * * The parameter `lines` is an array of lines, each line being an array of successive Vector3
         * * The optional parameter `instance` is an instance of an existing LineSystem object to be updated with the passed `lines` parameter
         * * The optional parameter `colors` is an array of line colors, each line colors being an array of successive Color4, one per line point
         * * The optional parameter `useVertexAlpha` is to be set to `false` (default `true`) when you don't need the alpha blending (faster)
         * * Updating a simple Line mesh, you just need to update every line in the `lines` array : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * * When updating an instance, remember that only line point positions can change, not the number of points, neither the number of lines
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#line-system
         * @param name defines the name of the new line system
         * @param options defines the options used to create the line system
         * @param scene defines the hosting scene
         * @returns a new line system mesh
         */
        static CreateLineSystem(name: string, options: {
            lines: Vector3[][];
            updatable?: boolean;
            instance?: Nullable<LinesMesh>;
            colors?: Nullable<Color4[][]>;
            useVertexAlpha?: boolean;
        }, scene: Nullable<Scene>): LinesMesh;
        /**
         * Creates a line mesh
         * A line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter
         * * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function
         * * The parameter `points` is an array successive Vector3
         * * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * * The optional parameter `colors` is an array of successive Color4, one per line point
         * * The optional parameter `useVertexAlpha` is to be set to `false` (default `true`) when you don't need alpha blending (faster)
         * * When updating an instance, remember that only point positions can change, not the number of points
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#lines
         * @param name defines the name of the new line system
         * @param options defines the options used to create the line system
         * @param scene defines the hosting scene
         * @returns a new line mesh
         */
        static CreateLines(name: string, options: {
            points: Vector3[];
            updatable?: boolean;
            instance?: Nullable<LinesMesh>;
            colors?: Color4[];
            useVertexAlpha?: boolean;
        }, scene?: Nullable<Scene>): LinesMesh;
        /**
         * Creates a dashed line mesh
         * * A dashed line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter
         * * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function
         * * The parameter `points` is an array successive Vector3
         * * The parameter `dashNb` is the intended total number of dashes (positive integer, default 200)
         * * The parameter `dashSize` is the size of the dashes relatively the dash number (positive float, default 3)
         * * The parameter `gapSize` is the size of the gap between two successive dashes relatively the dash number (positive float, default 1)
         * * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * * When updating an instance, remember that only point positions can change, not the number of points
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the dashed line mesh
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#dashed-lines
         */
        static CreateDashedLines(name: string, options: {
            points: Vector3[];
            dashSize?: number;
            gapSize?: number;
            dashNb?: number;
            updatable?: boolean;
            instance?: LinesMesh;
        }, scene?: Nullable<Scene>): LinesMesh;
        /**
         * Creates an extruded shape mesh. The extrusion is a parametric shape. It has no predefined shape. Its final shape will depend on the input parameters.
         * * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be extruded along the Z axis.
         * * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * * The parameter `rotation` (float, default 0 radians) is the angle value to rotate the shape each step (each path point), from the former step (so rotation added each step) along the curve.
         * * The parameter `scale` (float, default 1) is the value to scale the shape.
         * * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture.
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the extruded shape mesh
         * @see http://doc.babylonjs.com/tutorials/Parametric_Shapes
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#extruded-shapes
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#extruded-shapes
         */
        static ExtrudeShape(name: string, options: {
            shape: Vector3[];
            path: Vector3[];
            scale?: number;
            rotation?: number;
            cap?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates an custom extruded shape mesh.
         * The custom extrusion is a parametric shape. It has no predefined shape. Its final shape will depend on the input parameters.
         * * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be extruded along the Z axis.
         * * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * * The parameter `rotationFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path and the distance of this point from the begining of the path
         * * It must returns a float value that will be the rotation in radians applied to the shape on each path point.
         * * The parameter `scaleFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path and the distance of this point from the begining of the path
         * * It must returns a float value that will be the scale value applied to the shape on each path point
         * * The parameter `ribbonClosePath` (boolean, default false) forces the extrusion underlying ribbon to close all the paths in its `pathArray`
         * * The parameter `ribbonCloseArray` (boolean, default false) forces the extrusion underlying ribbon to close its `pathArray`
         * * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * * Remember you can only change the shape or path point positions, not their number when updating an extruded shape
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the custom extruded shape mesh
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#custom-extruded-shapes
         * @see http://doc.babylonjs.com/tutorials/Parametric_Shapes
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#extruded-shapes
         */
        static ExtrudeShapeCustom(name: string, options: {
            shape: Vector3[];
            path: Vector3[];
            scaleFunction?: any;
            rotationFunction?: any;
            ribbonCloseArray?: boolean;
            ribbonClosePath?: boolean;
            cap?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates lathe mesh.
         * The lathe is a shape with a symetry axis : a 2D model shape is rotated around this axis to design the lathe
         * * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be rotated in its local space : the shape must be designed in the xOy plane and will be rotated around the Y axis. It's usually a 2D shape, so the Vector3 z coordinates are often set to zero
         * * The parameter `radius` (positive float, default 1) is the radius value of the lathe
         * * The parameter `tessellation` (positive integer, default 64) is the side number of the lathe
         * * The parameter `arc` (positive float, default 1) is the ratio of the lathe. 0.5 builds for instance half a lathe, so an opened shape
         * * The parameter `closed` (boolean, default true) opens/closes the lathe circumference. This should be set to false when used with the parameter "arc"
         * * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the lathe mesh
         * @see http://doc.babylonjs.com/how_to/parametric_shapes#lathe
         */
        static CreateLathe(name: string, options: {
            shape: Vector3[];
            radius?: number;
            tessellation?: number;
            arc?: number;
            closed?: boolean;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            cap?: number;
            invertUV?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a plane mesh
         * * The parameter `size` sets the size (float) of both sides of the plane at once (default 1)
         * * You can set some different plane dimensions by using the parameters `width` and `height` (both by default have the same value than `size`)
         * * The parameter `sourcePlane` is a Plane instance. It builds a mesh plane from a Math plane
         * * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4). Detail here : http://doc.babylonjs.com/babylon101/discover_basic_elements#side-orientation
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the plane mesh
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#plane
         */
        static CreatePlane(name: string, options: {
            size?: number;
            width?: number;
            height?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
            sourcePlane?: Plane;
        }, scene: Scene): Mesh;
        /**
         * Creates a ground mesh
         * * The parameters `width` and `height` (floats, default 1) set the width and height sizes of the ground
         * * The parameter `subdivisions` (positive integer) sets the number of subdivisions per side
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the ground mesh
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#plane
         */
        static CreateGround(name: string, options: {
            width?: number;
            height?: number;
            subdivisions?: number;
            subdivisionsX?: number;
            subdivisionsY?: number;
            updatable?: boolean;
        }, scene: any): Mesh;
        /**
         * Creates a tiled ground mesh
         * * The parameters `xmin` and `xmax` (floats, default -1 and 1) set the ground minimum and maximum X coordinates
         * * The parameters `zmin` and `zmax` (floats, default -1 and 1) set the ground minimum and maximum Z coordinates
         * * The parameter `subdivisions` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 6, h: 6}`). `w` and `h` are the numbers of subdivisions on the ground width and height. Each subdivision is called a tile
         * * The parameter `precision` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 2, h: 2}`). `w` and `h` are the numbers of subdivisions on the ground width and height of each tile
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the tiled ground mesh
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#tiled-ground
         */
        static CreateTiledGround(name: string, options: {
            xmin: number;
            zmin: number;
            xmax: number;
            zmax: number;
            subdivisions?: {
                w: number;
                h: number;
            };
            precision?: {
                w: number;
                h: number;
            };
            updatable?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a ground mesh from a height map
         * * The parameter `url` sets the URL of the height map image resource.
         * * The parameters `width` and `height` (positive floats, default 10) set the ground width and height sizes.
         * * The parameter `subdivisions` (positive integer, default 1) sets the number of subdivision per side.
         * * The parameter `minHeight` (float, default 0) is the minimum altitude on the ground.
         * * The parameter `maxHeight` (float, default 1) is the maximum altitude on the ground.
         * * The parameter `colorFilter` (optional Color3, default (0.3, 0.59, 0.11) ) is the filter to apply to the image pixel colors to compute the height.
         * * The parameter `onReady` is a javascript callback function that will be called  once the mesh is just built (the height map download can last some time).
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * @param name defines the name of the mesh
         * @param url defines the url to the height map
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the ground mesh
         * @see http://doc.babylonjs.com/babylon101/height_map
         * @see http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#ground-from-a-height-map
         */
        static CreateGroundFromHeightMap(name: string, url: string, options: {
            width?: number;
            height?: number;
            subdivisions?: number;
            minHeight?: number;
            maxHeight?: number;
            colorFilter?: Color3;
            updatable?: boolean;
            onReady?: (mesh: GroundMesh) => void;
        }, scene: Scene): GroundMesh;
        /**
         * Creates a polygon mesh
         * The polygon's shape will depend on the input parameters and is constructed parallel to a ground mesh
         * * The parameter `shape` is a required array of successive Vector3 representing the corners of the polygon in th XoZ plane, that is y = 0 for all vectors
         * * You can set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created
         * * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4)
         * * Remember you can only change the shape positions, not their number when updating a polygon
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the polygon mesh
         */
        static CreatePolygon(name: string, options: {
            shape: Vector3[];
            holes?: Vector3[][];
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: Scene): Mesh;
        /**
         * Creates an extruded polygon mesh, with depth in the Y direction.
         * * You can set different colors and different images to the top, bottom and extruded side by using the parameters `faceColors` (an array of 3 Color3 elements) and `faceUV` (an array of 3 Vector4 elements)
         * @see http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
         * @param name defines the name of the mesh
         * @param options defines the options used to create the mesh
         * @param scene defines the hosting scene
         * @returns the polygon mesh
         */
        static ExtrudePolygon(name: string, options: {
            shape: Vector3[];
            holes?: Vector3[][];
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: Scene): Mesh;
        /**
         * Creates a tube mesh.
         * The