declare module "babylonjs/types" {
    /** Alias type for value that can be null */
    export type Nullable<T> = T | null;
    /**
     * Alias type for number that are floats
     * @ignorenaming
     */
    export type float = number;
    /**
     * Alias type for number that are doubles.
     * @ignorenaming
     */
    export type double = number;
    /**
     * Alias type for number that are integer
     * @ignorenaming
     */
    export type int = number;
    /** Alias type for number array or Float32Array */
    export type FloatArray = number[] | Float32Array;
    /** Alias type for number array or Float32Array or Int32Array or Uint32Array or Uint16Array */
    export type IndicesArray = number[] | Int32Array | Uint32Array | Uint16Array;
    /**
     * Alias for types that can be used by a Buffer or VertexBuffer.
     */
    export type DataArray = number[] | ArrayBuffer | ArrayBufferView;
    /**
     * Alias type for primitive types
     * @ignorenaming
     */
    type Primitive = undefined | null | boolean | string | number | Function;
    /**
     * Type modifier to make all the properties of an object Readonly
     */
    export type Immutable<T> = T extends Primitive ? T : T extends Array<infer U> ? ReadonlyArray<U> : DeepImmutable<T>;
    /**
     * Type modifier to make all the properties of an object Readonly recursively
     */
    export type DeepImmutable<T> = T extends Primitive ? T : T extends Array<infer U> ? DeepImmutableArray<U> : DeepImmutableObject<T>;
    /**
     * Type modifier to make object properties readonly.
     */
    export type DeepImmutableObject<T> = {
        readonly [K in keyof T]: DeepImmutable<T[K]>;
    };
    /** @hidden */
    interface DeepImmutableArray<T> extends ReadonlyArray<DeepImmutable<T>> {
    }
}
declare module "babylonjs/Misc/observable" {
    import { Nullable } from "babylonjs/types";
    /**
     * A class serves as a medium between the observable and its observers
     */
    export class EventState {
        /**
         * Create a new EventState
         * @param mask defines the mask associated with this state
         * @param skipNextObservers defines a flag which will instruct the observable to skip following observers when set to true
         * @param target defines the original target of the state
         * @param currentTarget defines the current target of the state
         */
        constructor(mask: number, skipNextObservers?: boolean, target?: any, currentTarget?: any);
        /**
         * Initialize the current event state
         * @param mask defines the mask associated with this state
         * @param skipNextObservers defines a flag which will instruct the observable to skip following observers when set to true
         * @param target defines the original target of the state
         * @param currentTarget defines the current target of the state
         * @returns the current event state
         */
        initalize(mask: number, skipNextObservers?: boolean, target?: any, currentTarget?: any): EventState;
        /**
         * An Observer can set this property to true to prevent subsequent observers of being notified
         */
        skipNextObservers: boolean;
        /**
         * Get the mask value that were used to trigger the event corresponding to this EventState object
         */
        mask: number;
        /**
         * The object that originally notified the event
         */
        target?: any;
        /**
         * The current object in the bubbling phase
         */
        currentTarget?: any;
        /**
         * This will be populated with the return value of the last function that was executed.
         * If it is the first function in the callback chain it will be the event data.
         */
        lastReturnValue?: any;
        /**
        * User defined information that will be sent to observers
        */
        userInfo?: any;
    }
    /**
     * Represent an Observer registered to a given Observable object.
     */
    export class Observer<T> {
        /**
         * Defines the callback to call when the observer is notified
         */
        callback: (eventData: T, eventState: EventState) => void;
        /**
         * Defines the mask of the observer (used to filter notifications)
         */
        mask: number;
        /**
         * Defines the current scope used to restore the JS context
         */
        scope: any;
        /** @hidden */
        _willBeUnregistered: boolean;
        /**
         * Gets or sets a property defining that the observer as to be unregistered after the next notification
         */
        unregisterOnNextCall: boolean;
        /**
         * Creates a new observer
         * @param callback defines the callback to call when the observer is notified
         * @param mask defines the mask of the observer (used to filter notifications)
         * @param scope defines the current scope used to restore the JS context
         */
        constructor(
        /**
         * Defines the callback to call when the observer is notified
         */
        callback: (eventData: T, eventState: EventState) => void, 
        /**
         * Defines the mask of the observer (used to filter notifications)
         */
        mask: number, 
        /**
         * Defines the current scope used to restore the JS context
         */
        scope?: any);
    }
    /**
     * Represent a list of observers registered to multiple Observables object.
     */
    export class MultiObserver<T> {
        private _observers;
        private _observables;
        /**
         * Release associated resources
         */
        dispose(): void;
        /**
         * Raise a callback when one of the observable will notify
         * @param observables defines a list of observables to watch
         * @param callback defines the callback to call on notification
         * @param mask defines the mask used to filter notifications
         * @param scope defines the current scope used to restore the JS context
         * @returns the new MultiObserver
         */
        static Watch<T>(observables: Observable<T>[], callback: (eventData: T, eventState: EventState) => void, mask?: number, scope?: any): MultiObserver<T>;
    }
    /**
     * The Observable class is a simple implementation of the Observable pattern.
     *
     * There's one slight particularity though: a given Observable can notify its observer using a particular mask value, only the Observers registered with this mask value will be notified.
     * This enable a more fine grained execution without having to rely on multiple different Observable objects.
     * For instance you may have a given Observable that have four different types of notifications: Move (mask = 0x01), Stop (mask = 0x02), Turn Right (mask = 0X04), Turn Left (mask = 0X08).
     * A given observer can register itself with only Move and Stop (mask = 0x03), then it will only be notified when one of these two occurs and will never be for Turn Left/Right.
     */
    export class Observable<T> {
        private _observers;
        private _eventState;
        private _onObserverAdded;
        /**
         * Gets the list of observers
         */
        get observers(): Array<Observer<T>>;
        /**
         * Creates a new observable
         * @param onObserverAdded defines a callback to call when a new observer is added
         */
        constructor(onObserverAdded?: (observer: Observer<T>) => void);
        /**
         * Create a new Observer with the specified callback
         * @param callback the callback that will be executed for that Observer
         * @param mask the mask used to filter observers
         * @param insertFirst if true the callback will be inserted at the first position, hence executed before the others ones. If false (default behavior) the callback will be inserted at the last position, executed after all the others already present.
         * @param scope optional scope for the callback to be called from
         * @param unregisterOnFirstCall defines if the observer as to be unregistered after the next notification
         * @returns the new observer created for the callback
         */
        add(callback: (eventData: T, eventState: EventState) => void, mask?: number, insertFirst?: boolean, scope?: any, unregisterOnFirstCall?: boolean): Nullable<Observer<T>>;
        /**
         * Create a new Observer with the specified callback and unregisters after the next notification
         * @param callback the callback that will be executed for that Observer
         * @returns the new observer created for the callback
         */
        addOnce(callback: (eventData: T, eventState: EventState) => void): Nullable<Observer<T>>;
        /**
         * Remove an Observer from the Observable object
         * @param observer the instance of the Observer to remove
         * @returns false if it doesn't belong to this Observable
         */
        remove(observer: Nullable<Observer<T>>): boolean;
        /**
         * Remove a callback from the Observable object
         * @param callback the callback to remove
         * @param scope optional scope. If used only the callbacks with this scope will be removed
         * @returns false if it doesn't belong to this Observable
        */
        removeCallback(callback: (eventData: T, eventState: EventState) => void, scope?: any): boolean;
        private _deferUnregister;
        private _remove;
        /**
         * Moves the observable to the top of the observer list making it get called first when notified
         * @param observer the observer to move
         */
        makeObserverTopPriority(observer: Observer<T>): void;
        /**
         * Moves the observable to the bottom of the observer list making it get called last when notified
         * @param observer the observer to move
         */
        makeObserverBottomPriority(observer: Observer<T>): void;
        /**
         * Notify all Observers by calling their respective callback with the given data
         * Will return true if all observers were executed, false if an observer set skipNextObservers to true, then prevent the subsequent ones to execute
         * @param eventData defines the data to send to all observers
         * @param mask defines the mask of the current notification (observers with incompatible mask (ie mask & observer.mask === 0) will not be notified)
         * @param target defines the original target of the state
         * @param currentTarget defines the current target of the state
         * @param userInfo defines any user info to send to observers
         * @returns false if the complete observer chain was not processed (because one observer set the skipNextObservers to true)
         */
        notifyObservers(eventData: T, mask?: number, target?: any, currentTarget?: any, userInfo?: any): boolean;
        /**
         * Calling this will execute each callback, expecting it to be a promise or return a value.
         * If at any point in the chain one function fails, the promise will fail and the execution will not continue.
         * This is useful when a chain of events (sometimes async events) is needed to initialize a certain object
         * and it is crucial that all callbacks will be executed.
         * The order of the callbacks is kept, callbacks are not executed parallel.
         *
         * @param eventData The data to be sent to each callback
         * @param mask is used to filter observers defaults to -1
         * @param target defines the callback target (see EventState)
         * @param currentTarget defines he current object in the bubbling phase
         * @param userInfo defines any user info to send to observers
         * @returns {Promise<T>} will return a Promise than resolves when all callbacks executed successfully.
         */
        notifyObserversWithPromise(eventData: T, mask?: number, target?: any, currentTarget?: any, userInfo?: any): Promise<T>;
        /**
         * Notify a specific observer
         * @param observer defines the observer to notify
         * @param eventData defines the data to be sent to each callback
         * @param mask is used to filter observers defaults to -1
         */
        notifyObserver(observer: Observer<T>, eventData: T, mask?: number): void;
        /**
         * Gets a boolean indicating if the observable has at least one observer
         * @returns true is the Observable has at least one Observer registered
         */
        hasObservers(): boolean;
        /**
        * Clear the list of observers
        */
        clear(): void;
        /**
         * Clone the current observable
         * @returns a new observable
         */
        clone(): Observable<T>;
        /**
         * Does this observable handles observer registered with a given mask
         * @param mask defines the mask to be tested
         * @return whether or not one observer registered with the given mask is handeled
        **/
        hasSpecificMask(mask?: number): boolean;
    }
}
declare module "babylonjs/Misc/domManagement" {
    /**
     * Sets of helpers dealing with the DOM and some of the recurrent functions needed in
     * Babylon.js
     */
    export class DomManagement {
        /**
         * Checks if the window object exists
         * @returns true if the window object exists
         */
        static IsWindowObjectExist(): boolean;
        /**
         * Checks if the navigator object exists
         * @returns true if the navigator object exists
         */
        static IsNavigatorAvailable(): boolean;
        /**
         * Check if the document object exists
         * @returns true if the document object exists
         */
        static IsDocumentAvailable(): boolean;
        /**
         * Extracts text content from a DOM element hierarchy
         * @param element defines the root element
         * @returns a string
         */
        static GetDOMTextContent(element: HTMLElement): string;
    }
}
declare module "babylonjs/Misc/logger" {
    /**
     * Logger used througouht the application to allow configuration of
     * the log level required for the messages.
     */
    export class Logger {
        /**
         * No log
         */
        static readonly NoneLogLevel: number;
        /**
         * Only message logs
         */
        static readonly MessageLogLevel: number;
        /**
         * Only warning logs
         */
        static readonly WarningLogLevel: number;
        /**
         * Only error logs
         */
        static readonly ErrorLogLevel: number;
        /**
         * All logs
         */
        static readonly AllLogLevel: number;
        private static _LogCache;
        /**
         * Gets a value indicating the number of loading errors
         * @ignorenaming
         */
        static errorsCount: number;
        /**
         * Callback called when a new log is added
         */
        static OnNewCacheEntry: (entry: string) => void;
        private static _AddLogEntry;
        private static _FormatMessage;
        private static _LogDisabled;
        private static _LogEnabled;
        private static _WarnDisabled;
        private static _WarnEnabled;
        private static _ErrorDisabled;
        private static _ErrorEnabled;
        /**
         * Log a message to the console
         */
        static Log: (message: string) => void;
        /**
         * Write a warning message to the console
         */
        static Warn: (message: string) => void;
        /**
         * Write an error message to the console
         */
        static Error: (message: string) => void;
        /**
         * Gets current log cache (list of logs)
         */
        static get LogCache(): string;
        /**
         * Clears the log cache
         */
        static ClearLogCache(): void;
        /**
         * Sets the current log level (MessageLogLevel / WarningLogLevel / ErrorLogLevel)
         */
        static set LogLevels(level: number);
    }
}
declare module "babylonjs/Misc/typeStore" {
    /** @hidden */
    export class _TypeStore {
        /** @hidden */
        static RegisteredTypes: {
            [key: string]: Object;
        };
        /** @hidden */
        static GetClass(fqdn: string): any;
    }
}
declare module "babylonjs/Misc/stringTools" {
    /**
     * Helper to manipulate strings
     */
    export class StringTools {
        /**
         * Checks for a matching suffix at the end of a string (for ES5 and lower)
         * @param str Source string
         * @param suffix Suffix to search for in the source string
         * @returns Boolean indicating whether the suffix was found (true) or not (false)
         */
        static EndsWith(str: string, suffix: string): boolean;
        /**
         * Checks for a matching suffix at the beginning of a string (for ES5 and lower)
         * @param str Source string
         * @param suffix Suffix to search for in the source string
         * @returns Boolean indicating whether the suffix was found (true) or not (false)
         */
        static StartsWith(str: string, suffix: string): boolean;
        /**
         * Decodes a buffer into a string
         * @param buffer The buffer to decode
         * @returns The decoded string
         */
        static Decode(buffer: Uint8Array | Uint16Array): string;
        /**
         * Encode a buffer to a base64 string
         * @param buffer defines the buffer to encode
         * @returns the encoded string
         */
        static EncodeArrayBufferToBase64(buffer: ArrayBuffer | ArrayBufferView): string;
        /**
        * Converts a number to string and pads with preceeding zeroes until it is of specified length.
        * @param num the number to convert and pad
        * @param length the expected length of the string
        * @returns the padded string
        */
        static PadNumber(num: number, length: number): string;
    }
}
declare module "babylonjs/Misc/deepCopier" {
    /**
     * Class containing a set of static utilities functions for deep copy.
     */
    export class DeepCopier {
        /**
         * Tries to copy an object by duplicating every property
         * @param source defines the source object
         * @param destination defines the target object
         * @param doNotCopyList defines a list of properties to avoid
         * @param mustCopyList defines a list of properties to copy (even if they start with _)
         */
        static DeepCopy(source: any, destination: any, doNotCopyList?: string[], mustCopyList?: string[]): void;
    }
}
declare module "babylonjs/Misc/precisionDate" {
    /**
     * Class containing a set of static utilities functions for precision date
     */
    export class PrecisionDate {
        /**
         * Gets either window.performance.now() if supported or Date.now() else
         */
        static get Now(): number;
    }
}
declare module "babylonjs/Misc/devTools" {
    /** @hidden */
    export class _DevTools {
        static WarnImport(name: string): string;
    }
}
declare module "babylonjs/Misc/interfaces/iWebRequest" {
    /**
     * Interface used to define the mechanism to get data from the network
     */
    export interface IWebRequest {
        /**
         * Returns client's response url
         */
        responseURL: string;
        /**
         * Returns client's status
         */
        status: number;
        /**
         * Returns client's status as a text
         */
        statusText: string;
    }
}
declare module "babylonjs/Misc/webRequest" {
    import { IWebRequest } from "babylonjs/Misc/interfaces/iWebRequest";
    import { Nullable } from "babylonjs/types";
    /**
     * Extended version of XMLHttpRequest with support for customizations (headers, ...)
     */
    export class WebRequest implements IWebRequest {
        private readonly _xhr;
        /**
         * Custom HTTP Request Headers to be sent with XMLHttpRequests
         * i.e. when loading files, where the server/service expects an Authorization header
         */
        static CustomRequestHeaders: {
            [key: string]: string;
        };
        /**
         * Add callback functions in this array to update all the requests before they get sent to the network
         */
        static CustomRequestModifiers: ((request: XMLHttpRequest, url: string) => void)[];
        private _injectCustomRequestHeaders;
        /**
         * Gets or sets a function to be called when loading progress changes
         */
        get onprogress(): ((this: XMLHttpRequest, ev: ProgressEvent) => any) | null;
        set onprogress(value: ((this: XMLHttpRequest, ev: ProgressEvent) => any) | null);
        /**
         * Returns client's state
         */
        get readyState(): number;
        /**
         * Returns client's status
         */
        get status(): number;
        /**
         * Returns client's status as a text
         */
        get statusText(): string;
        /**
         * Returns client's response
         */
        get response(): any;
        /**
         * Returns client's response url
         */
        get responseURL(): string;
        /**
         * Returns client's response as text
         */
        get responseText(): string;
        /**
         * Gets or sets the expected response type
         */
        get responseType(): XMLHttpRequestResponseType;
        set responseType(value: XMLHttpRequestResponseType);
        /** @hidden */
        addEventListener<K extends keyof XMLHttpRequestEventMap>(type: K, listener: (this: XMLHttpRequest, ev: XMLHttpRequestEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;
        /** @hidden */
        removeEventListener<K extends keyof XMLHttpRequestEventMap>(type: K, listener: (this: XMLHttpRequest, ev: XMLHttpRequestEventMap[K]) => any, options?: boolean | EventListenerOptions): void;
        /**
         * Cancels any network activity
         */
        abort(): void;
        /**
         * Initiates the request. The optional argument provides the request body. The argument is ignored if request method is GET or HEAD
         * @param body defines an optional request body
         */
        send(body?: Document | BodyInit | null): void;
        /**
         * Sets the request method, request URL
         * @param method defines the method to use (GET, POST, etc..)
         * @param url defines the url to connect with
         */
        open(method: string, url: string): void;
        /**
         * Sets the value of a request header.
         * @param name The name of the header whose value is to be set
         * @param value The value to set as the body of the header
         */
        setRequestHeader(name: string, value: string): void;
        /**
         * Get the string containing the text of a particular header's value.
         * @param name The name of the header
         * @returns The string containing the text of the given header name
         */
        getResponseHeader(name: string): Nullable<string>;
    }
}
declare module "babylonjs/Misc/fileRequest" {
    import { Observable } from "babylonjs/Misc/observable";
    /**
     * File request interface
     */
    export interface IFileRequest {
        /**
         * Raised when the request is complete (success or error).
         */
        onCompleteObservable: Observable<IFileRequest>;
        /**
         * Aborts the request for a file.
         */
        abort: () => void;
    }
}
declare module "babylonjs/Materials/Textures/renderTargetCreationOptions" {
    /**
     * Define options used to create a render target texture
     */
    export class RenderTargetCreationOptions {
        /**
         * Specifies is mipmaps must be generated
         */
        generateMipMaps?: boolean;
        /** Specifies whether or not a depth should be allocated in the texture (true by default) */
        generateDepthBuffer?: boolean;
        /** Specifies whether or not a stencil should be allocated in the texture (false by default)*/
        generateStencilBuffer?: boolean;
        /** Defines texture type (int by default) */
        type?: number;
        /** Defines sampling mode (trilinear by default) */
        samplingMode?: number;
        /** Defines format (RGBA by default) */
        format?: number;
    }
}
declare module "babylonjs/Engines/constants" {
    /** Defines the cross module used constants to avoid circular dependncies */
    export class Constants {
        /** Defines that alpha blending is disabled */
        static readonly ALPHA_DISABLE: number;
        /** Defines that alpha blending is SRC ALPHA * SRC + DEST */
        static readonly ALPHA_ADD: number;
        /** Defines that alpha blending is SRC ALPHA * SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_COMBINE: number;
        /** Defines that alpha blending is DEST - SRC * DEST */
        static readonly ALPHA_SUBTRACT: number;
        /** Defines that alpha blending is SRC * DEST */
        static readonly ALPHA_MULTIPLY: number;
        /** Defines that alpha blending is SRC ALPHA * SRC + (1 - SRC) * DEST */
        static readonly ALPHA_MAXIMIZED: number;
        /** Defines that alpha blending is SRC + DEST */
        static readonly ALPHA_ONEONE: number;
        /** Defines that alpha blending is SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_PREMULTIPLIED: number;
        /**
         * Defines that alpha blending is SRC + (1 - SRC ALPHA) * DEST
         * Alpha will be set to (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_PREMULTIPLIED_PORTERDUFF: number;
        /** Defines that alpha blending is CST * SRC + (1 - CST) * DEST */
        static readonly ALPHA_INTERPOLATE: number;
        /**
         * Defines that alpha blending is SRC + (1 - SRC) * DEST
         * Alpha will be set to SRC ALPHA + (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_SCREENMODE: number;
        /**
         * Defines that alpha blending is SRC + DST
         * Alpha will be set to SRC ALPHA + DST ALPHA
         */
        static readonly ALPHA_ONEONE_ONEONE: number;
        /**
         * Defines that alpha blending is SRC * DST ALPHA + DST
         * Alpha will be set to 0
         */
        static readonly ALPHA_ALPHATOCOLOR: number;
        /**
         * Defines that alpha blending is SRC * (1 - DST) + DST * (1 - SRC)
         */
        static readonly ALPHA_REVERSEONEMINUS: number;
        /**
         * Defines that alpha blending is SRC + DST * (1 - SRC ALPHA)
         * Alpha will be set to SRC ALPHA + DST ALPHA * (1 - SRC ALPHA)
         */
        static readonly ALPHA_SRC_DSTONEMINUSSRCALPHA: number;
        /**
         * Defines that alpha blending is SRC + DST
         * Alpha will be set to SRC ALPHA
         */
        static readonly ALPHA_ONEONE_ONEZERO: number;
        /**
         * Defines that alpha blending is SRC * (1 - DST) + DST * (1 - SRC)
         * Alpha will be set to DST ALPHA
         */
        static readonly ALPHA_EXCLUSION: number;
        /** Defines that alpha blending equation a SUM */
        static readonly ALPHA_EQUATION_ADD: number;
        /** Defines that alpha blending equation a SUBSTRACTION */
        static readonly ALPHA_EQUATION_SUBSTRACT: number;
        /** Defines that alpha blending equation a REVERSE SUBSTRACTION */
        static readonly ALPHA_EQUATION_REVERSE_SUBTRACT: number;
        /** Defines that alpha blending equation a MAX operation */
        static readonly ALPHA_EQUATION_MAX: number;
        /** Defines that alpha blending equation a MIN operation */
        static readonly ALPHA_EQUATION_MIN: number;
        /**
         * Defines that alpha blending equation a DARKEN operation:
         * It takes the min of the src and sums the alpha channels.
         */
        static readonly ALPHA_EQUATION_DARKEN: number;
        /** Defines that the ressource is not delayed*/
        static readonly DELAYLOADSTATE_NONE: number;
        /** Defines that the ressource was successfully delay loaded */
        static readonly DELAYLOADSTATE_LOADED: number;
        /** Defines that the ressource is currently delay loading */
        static readonly DELAYLOADSTATE_LOADING: number;
        /** Defines that the ressource is delayed and has not started loading */
        static readonly DELAYLOADSTATE_NOTLOADED: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will never pass. i.e. Nothing will be drawn */
        static readonly NEVER: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will always pass. i.e. Pixels will be drawn in the order they are drawn */
        static readonly ALWAYS: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than the stored value */
        static readonly LESS: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is equals to the stored value */
        static readonly EQUAL: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than or equal to the stored value */
        static readonly LEQUAL: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than the stored value */
        static readonly GREATER: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than or equal to the stored value */
        static readonly GEQUAL: number;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is not equal to the stored value */
        static readonly NOTEQUAL: number;
        /** Passed to stencilOperation to specify that stencil value must be kept */
        static readonly KEEP: number;
        /** Passed to stencilOperation to specify that stencil value must be replaced */
        static readonly REPLACE: number;
        /** Passed to stencilOperation to specify that stencil value must be incremented */
        static readonly INCR: number;
        /** Passed to stencilOperation to specify that stencil value must be decremented */
        static readonly DECR: number;
        /** Passed to stencilOperation to specify that stencil value must be inverted */
        static readonly INVERT: number;
        /** Passed to stencilOperation to specify that stencil value must be incremented with wrapping */
        static readonly INCR_WRAP: number;
        /** Passed to stencilOperation to specify that stencil value must be decremented with wrapping */
        static readonly DECR_WRAP: number;
        /** Texture is not repeating outside of 0..1 UVs */
        static readonly TEXTURE_CLAMP_ADDRESSMODE: number;
        /** Texture is repeating outside of 0..1 UVs */
        static readonly TEXTURE_WRAP_ADDRESSMODE: number;
        /** Texture is repeating and mirrored */
        static readonly TEXTURE_MIRROR_ADDRESSMODE: number;
        /** ALPHA */
        static readonly TEXTUREFORMAT_ALPHA: number;
        /** LUMINANCE */
        static readonly TEXTUREFORMAT_LUMINANCE: number;
        /** LUMINANCE_ALPHA */
        static readonly TEXTUREFORMAT_LUMINANCE_ALPHA: number;
        /** RGB */
        static readonly TEXTUREFORMAT_RGB: number;
        /** RGBA */
        static readonly TEXTUREFORMAT_RGBA: number;
        /** RED */
        static readonly TEXTUREFORMAT_RED: number;
        /** RED (2nd reference) */
        static readonly TEXTUREFORMAT_R: number;
        /** RG */
        static readonly TEXTUREFORMAT_RG: number;
        /** RED_INTEGER */
        static readonly TEXTUREFORMAT_RED_INTEGER: number;
        /** RED_INTEGER (2nd reference) */
        static readonly TEXTUREFORMAT_R_INTEGER: number;
        /** RG_INTEGER */
        static readonly TEXTUREFORMAT_RG_INTEGER: number;
        /** RGB_INTEGER */
        static readonly TEXTUREFORMAT_RGB_INTEGER: number;
        /** RGBA_INTEGER */
        static readonly TEXTUREFORMAT_RGBA_INTEGER: number;
        /** UNSIGNED_BYTE */
        static readonly TEXTURETYPE_UNSIGNED_BYTE: number;
        /** UNSIGNED_BYTE (2nd reference) */
        static readonly TEXTURETYPE_UNSIGNED_INT: number;
        /** FLOAT */
        static readonly TEXTURETYPE_FLOAT: number;
        /** HALF_FLOAT */
        static readonly TEXTURETYPE_HALF_FLOAT: number;
        /** BYTE */
        static readonly TEXTURETYPE_BYTE: number;
        /** SHORT */
        static readonly TEXTURETYPE_SHORT: number;
        /** UNSIGNED_SHORT */
        static readonly TEXTURETYPE_UNSIGNED_SHORT: number;
        /** INT */
        static readonly TEXTURETYPE_INT: number;
        /** UNSIGNED_INT */
        static readonly TEXTURETYPE_UNSIGNED_INTEGER: number;
        /** UNSIGNED_SHORT_4_4_4_4 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_4_4_4_4: number;
        /** UNSIGNED_SHORT_5_5_5_1 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_5_5_5_1: number;
        /** UNSIGNED_SHORT_5_6_5 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_5_6_5: number;
        /** UNSIGNED_INT_2_10_10_10_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_2_10_10_10_REV: number;
        /** UNSIGNED_INT_24_8 */
        static readonly TEXTURETYPE_UNSIGNED_INT_24_8: number;
        /** UNSIGNED_INT_10F_11F_11F_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_10F_11F_11F_REV: number;
        /** UNSIGNED_INT_5_9_9_9_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_5_9_9_9_REV: number;
        /** FLOAT_32_UNSIGNED_INT_24_8_REV */
        static readonly TEXTURETYPE_FLOAT_32_UNSIGNED_INT_24_8_REV: number;
        /** nearest is mag = nearest and min = nearest and no mip */
        static readonly TEXTURE_NEAREST_SAMPLINGMODE: number;
        /** mag = nearest and min = nearest and mip = none */
        static readonly TEXTURE_NEAREST_NEAREST: number;
        /** Bilinear is mag = linear and min = linear and no mip */
        static readonly TEXTURE_BILINEAR_SAMPLINGMODE: number;
        /** mag = linear and min = linear and mip = none */
        static readonly TEXTURE_LINEAR_LINEAR: number;
        /** Trilinear is mag = linear and min = linear and mip = linear */
        static readonly TEXTURE_TRILINEAR_SAMPLINGMODE: number;
        /** Trilinear is mag = linear and min = linear and mip = linear */
        static readonly TEXTURE_LINEAR_LINEAR_MIPLINEAR: number;
        /** mag = nearest and min = nearest and mip = nearest */
        static readonly TEXTURE_NEAREST_NEAREST_MIPNEAREST: number;
        /** mag = nearest and min = linear and mip = nearest */
        static readonly TEXTURE_NEAREST_LINEAR_MIPNEAREST: number;
        /** mag = nearest and min = linear and mip = linear */
        static readonly TEXTURE_NEAREST_LINEAR_MIPLINEAR: number;
        /** mag = nearest and min = linear and mip = none */
        static readonly TEXTURE_NEAREST_LINEAR: number;
        /** nearest is mag = nearest and min = nearest and mip = linear */
        static readonly TEXTURE_NEAREST_NEAREST_MIPLINEAR: number;
        /** mag = linear and min = nearest and mip = nearest */
        static readonly TEXTURE_LINEAR_NEAREST_MIPNEAREST: number;
        /** mag = linear and min = nearest and mip = linear */
        static readonly TEXTURE_LINEAR_NEAREST_MIPLINEAR: number;
        /** Bilinear is mag = linear and min = linear and mip = nearest */
        static readonly TEXTURE_LINEAR_LINEAR_MIPNEAREST: number;
        /** mag = linear and min = nearest and mip = none */
        static readonly TEXTURE_LINEAR_NEAREST: number;
        /** Explicit coordinates mode */
        static readonly TEXTURE_EXPLICIT_MODE: number;
        /** Spherical coordinates mode */
        static readonly TEXTURE_SPHERICAL_MODE: number;
        /** Planar coordinates mode */
        static readonly TEXTURE_PLANAR_MODE: number;
        /** Cubic coordinates mode */
        static readonly TEXTURE_CUBIC_MODE: number;
        /** Projection coordinates mode */
        static readonly TEXTURE_PROJECTION_MODE: number;
        /** Skybox coordinates mode */
        static readonly TEXTURE_SKYBOX_MODE: number;
        /** Inverse Cubic coordinates mode */
        static readonly TEXTURE_INVCUBIC_MODE: number;
        /** Equirectangular coordinates mode */
        static readonly TEXTURE_EQUIRECTANGULAR_MODE: number;
        /** Equirectangular Fixed coordinates mode */
        static readonly TEXTURE_FIXED_EQUIRECTANGULAR_MODE: number;
        /** Equirectangular Fixed Mirrored coordinates mode */
        static readonly TEXTURE_FIXED_EQUIRECTANGULAR_MIRRORED_MODE: number;
        /** Offline (baking) quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_OFFLINE: number;
        /** High quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_HIGH: number;
        /** Medium quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_MEDIUM: number;
        /** Low quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_LOW: number;
        /** Defines that texture rescaling will use a floor to find the closer power of 2 size */
        static readonly SCALEMODE_FLOOR: number;
        /** Defines that texture rescaling will look for the nearest power of 2 size */
        static readonly SCALEMODE_NEAREST: number;
        /** Defines that texture rescaling will use a ceil to find the closer power of 2 size */
        static readonly SCALEMODE_CEILING: number;
        /**
         * The dirty texture flag value
         */
        static readonly MATERIAL_TextureDirtyFlag: number;
        /**
         * The dirty light flag value
         */
        static readonly MATERIAL_LightDirtyFlag: number;
        /**
         * The dirty fresnel flag value
         */
        static readonly MATERIAL_FresnelDirtyFlag: number;
        /**
         * The dirty attribute flag value
         */
        static readonly MATERIAL_AttributesDirtyFlag: number;
        /**
         * The dirty misc flag value
         */
        static readonly MATERIAL_MiscDirtyFlag: number;
        /**
         * The dirty prepass flag value
         */
        static readonly MATERIAL_PrePassDirtyFlag: number;
        /**
         * The all dirty flag value
         */
        static readonly MATERIAL_AllDirtyFlag: number;
        /**
         * Returns the triangle fill mode
         */
        static readonly MATERIAL_TriangleFillMode: number;
        /**
         * Returns the wireframe mode
         */
        static readonly MATERIAL_WireFrameFillMode: number;
        /**
         * Returns the point fill mode
         */
        static readonly MATERIAL_PointFillMode: number;
        /**
         * Returns the point list draw mode
         */
        static readonly MATERIAL_PointListDrawMode: number;
        /**
         * Returns the line list draw mode
         */
        static readonly MATERIAL_LineListDrawMode: number;
        /**
         * Returns the line loop draw mode
         */
        static readonly MATERIAL_LineLoopDrawMode: number;
        /**
         * Returns the line strip draw mode
         */
        static readonly MATERIAL_LineStripDrawMode: number;
        /**
         * Returns the triangle strip draw mode
         */
        static readonly MATERIAL_TriangleStripDrawMode: number;
        /**
         * Returns the triangle fan draw mode
         */
        static readonly MATERIAL_TriangleFanDrawMode: number;
        /**
         * Stores the clock-wise side orientation
         */
        static readonly MATERIAL_ClockWiseSideOrientation: number;
        /**
         * Stores the counter clock-wise side orientation
         */
        static readonly MATERIAL_CounterClockWiseSideOrientation: number;
        /**
         * Nothing
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_NothingTrigger: number;
        /**
         * On pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnPickTrigger: number;
        /**
         * On left pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnLeftPickTrigger: number;
        /**
         * On right pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnRightPickTrigger: number;
        /**
         * On center pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnCenterPickTrigger: number;
        /**
         * On pick down
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnPickDownTrigger: number;
        /**
         * On double pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnDoublePickTrigger: number;
        /**
         * On pick up
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnPickUpTrigger: number;
        /**
         * On pick out.
         * This trigger will only be raised if you also declared a OnPickDown
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnPickOutTrigger: number;
        /**
         * On long press
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnLongPressTrigger: number;
        /**
         * On pointer over
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnPointerOverTrigger: number;
        /**
         * On pointer out
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnPointerOutTrigger: number;
        /**
         * On every frame
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnEveryFrameTrigger: number;
        /**
         * On intersection enter
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnIntersectionEnterTrigger: number;
        /**
         * On intersection exit
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnIntersectionExitTrigger: number;
        /**
         * On key down
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnKeyDownTrigger: number;
        /**
         * On key up
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly ACTION_OnKeyUpTrigger: number;
        /**
         * Billboard mode will only apply to Y axis
         */
        static readonly PARTICLES_BILLBOARDMODE_Y: number;
        /**
         * Billboard mode will apply to all axes
         */
        static readonly PARTICLES_BILLBOARDMODE_ALL: number;
        /**
         * Special billboard mode where the particle will be biilboard to the camera but rotated to align with direction
         */
        static readonly PARTICLES_BILLBOARDMODE_STRETCHED: number;
        /** Default culling strategy : this is an exclusion test and it's the more accurate.
         *  Test order :
         *  Is the bounding sphere outside the frustum ?
         *  If not, are the bounding box vertices outside the frustum ?
         *  It not, then the cullable object is in the frustum.
         */
        static readonly MESHES_CULLINGSTRATEGY_STANDARD: number;
        /** Culling strategy : Bounding Sphere Only.
         *  This is an exclusion test. It's faster than the standard strategy because the bounding box is not tested.
         *  It's also less accurate than the standard because some not visible objects can still be selected.
         *  Test : is the bounding sphere outside the frustum ?
         *  If not, then the cullable object is in the frustum.
         */
        static readonly MESHES_CULLINGSTRATEGY_BOUNDINGSPHERE_ONLY: number;
        /** Culling strategy : Optimistic Inclusion.
         *  This in an inclusion test first, then the standard exclusion test.
         *  This can be faster when a cullable object is expected to be almost always in the camera frustum.
         *  This could also be a little slower than the standard test when the tested object center is not the frustum but one of its bounding box vertex is still inside.
         *  Anyway, it's as accurate as the standard strategy.
         *  Test :
         *  Is the cullable object bounding sphere center in the frustum ?
         *  If not, apply the default culling strategy.
         */
        static readonly MESHES_CULLINGSTRATEGY_OPTIMISTIC_INCLUSION: number;
        /** Culling strategy : Optimistic Inclusion then Bounding Sphere Only.
         *  This in an inclusion test first, then the bounding sphere only exclusion test.
         *  This can be the fastest test when a cullable object is expected to be almost always in the camera frustum.
         *  This could also be a little slower than the BoundingSphereOnly strategy when the tested object center is not in the frustum but its bounding sphere still intersects it.
         *  It's less accurate than the standard strategy and as accurate as the BoundingSphereOnly strategy.
         *  Test :
         *  Is the cullable object bounding sphere center in the frustum ?
         *  If not, apply the Bounding Sphere Only strategy. No Bounding Box is tested here.
         */
        static readonly MESHES_CULLINGSTRATEGY_OPTIMISTIC_INCLUSION_THEN_BSPHERE_ONLY: number;
        /**
         * No logging while loading
         */
        static readonly SCENELOADER_NO_LOGGING: number;
        /**
         * Minimal logging while loading
         */
        static readonly SCENELOADER_MINIMAL_LOGGING: number;
        /**
         * Summary logging while loading
         */
        static readonly SCENELOADER_SUMMARY_LOGGING: number;
        /**
         * Detailled logging while loading
         */
        static readonly SCENELOADER_DETAILED_LOGGING: number;
        /**
         * Constant used to retrieve the irradiance texture index in the textures array in the prepass
         * using getIndex(Constants.PREPASS_IRRADIANCE_TEXTURE_TYPE)
         */
        static readonly PREPASS_IRRADIANCE_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the position texture index in the textures array in the prepass
         * using getIndex(Constants.PREPASS_POSITION_TEXTURE_INDEX)
         */
        static readonly PREPASS_POSITION_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the velocity texture index in the textures array in the prepass
         * using getIndex(Constants.PREPASS_VELOCITY_TEXTURE_INDEX)
         */
        static readonly PREPASS_VELOCITY_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the reflectivity texture index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_REFLECTIVITY_TEXTURE_TYPE)
         */
        static readonly PREPASS_REFLECTIVITY_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the lit color texture index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_COLOR_TEXTURE_TYPE)
         */
        static readonly PREPASS_COLOR_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve depth + normal index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_DEPTHNORMAL_TEXTURE_TYPE)
         */
        static readonly PREPASS_DEPTHNORMAL_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve albedo index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_ALBEDO_TEXTURE_TYPE)
         */
        static readonly PREPASS_ALBEDO_TEXTURE_TYPE: number;
    }
}
declare module "babylonjs/Materials/Textures/internalTextureLoader" {
    import { Nullable } from "babylonjs/types";
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    /**
     * This represents the required contract to create a new type of texture loader.
     */
    export interface IInternalTextureLoader {
        /**
         * Defines wether the loader supports cascade loading the different faces.
         */
        supportCascades: boolean;
        /**
         * This returns if the loader support the current file information.
         * @param extension defines the file extension of the file being loaded
         * @param mimeType defines the optional mime type of the file being loaded
         * @returns true if the loader can load the specified file
         */
        canLoad(extension: string, mimeType?: string): boolean;
        /**
         * Uploads the cube texture data to the WebGL texture. It has already been bound.
         * @param data contains the texture data
         * @param texture defines the BabylonJS internal texture
         * @param createPolynomials will be true if polynomials have been requested
         * @param onLoad defines the callback to trigger once the texture is ready
         * @param onError defines the callback to trigger in case of error
         * @param options options to be passed to the loader
         */
        loadCubeData(data: ArrayBufferView | ArrayBufferView[], texture: InternalTexture, createPolynomials: boolean, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, options?: any): void;
        /**
         * Uploads the 2D texture data to the WebGL texture. It has already been bound once in the callback.
         * @param data contains the texture data
         * @param texture defines the BabylonJS internal texture
         * @param callback defines the method to call once ready to upload
         * @param options options to be passed to the loader
         */
        loadData(data: ArrayBufferView, texture: InternalTexture, callback: (width: number, height: number, loadMipmap: boolean, isCompressed: boolean, done: () => void, loadFailed?: boolean) => void, options?: any): void;
    }
}
declare module "babylonjs/Engines/IPipelineContext" {
    /**
     * Class used to store and describe the pipeline context associated with an effect
     */
    export interface IPipelineContext {
        /**
         * Gets a boolean indicating that this pipeline context is supporting asynchronous creating
         */
        isAsync: boolean;
        /**
         * Gets a boolean indicating that the context is ready to be used (like shaders / pipelines are compiled and ready for instance)
         */
        isReady: boolean;
        /** @hidden */
        _getVertexShaderCode(): string | null;
        /** @hidden */
        _getFragmentShaderCode(): string | null;
        /** @hidden */
        _handlesSpectorRebuildCallback(onCompiled: (compiledObject: any) => void): void;
    }
}
declare module "babylonjs/Meshes/dataBuffer" {
    /**
     * Class used to store gfx data (like WebGLBuffer)
     */
    export class DataBuffer {
        /**
         * Gets or sets the number of objects referencing this buffer
         */
        references: number;
        /** Gets or sets the size of the underlying buffer */
        capacity: number;
        /**
         * Gets or sets a boolean indicating if the buffer contains 32bits indices
         */
        is32Bits: boolean;
        /**
         * Gets the underlying buffer
         */
        get underlyingResource(): any;
    }
}
declare module "babylonjs/Engines/Processors/iShaderProcessor" {
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    /** @hidden */
    export interface IShaderProcessor {
        attributeProcessor?: (attribute: string) => string;
        varyingProcessor?: (varying: string, isFragment: boolean) => string;
        uniformProcessor?: (uniform: string, isFragment: boolean) => string;
        uniformBufferProcessor?: (uniformBuffer: string, isFragment: boolean) => string;
        endOfUniformBufferProcessor?: (closingBracketLine: string, isFragment: boolean) => string;
        lineProcessor?: (line: string, isFragment: boolean) => string;
        preProcessor?: (code: string, defines: string[], isFragment: boolean) => string;
        postProcessor?: (code: string, defines: string[], isFragment: boolean, engine: ThinEngine) => string;
    }
}
declare module "babylonjs/Engines/Processors/shaderProcessingOptions" {
    import { IShaderProcessor } from "babylonjs/Engines/Processors/iShaderProcessor";
    /** @hidden */
    export interface ProcessingOptions {
        defines: string[];
        indexParameters: any;
        isFragment: boolean;
        shouldUseHighPrecisionShader: boolean;
        supportsUniformBuffers: boolean;
        shadersRepository: string;
        includesShadersStore: {
            [key: string]: string;
        };
        processor?: IShaderProcessor;
        version: string;
        platformName: string;
        lookForClosingBracketForUniformBuffer?: boolean;
    }
}
declare module "babylonjs/Engines/Processors/shaderCodeNode" {
    import { ProcessingOptions } from "babylonjs/Engines/Processors/shaderProcessingOptions";
    /** @hidden */
    export class ShaderCodeNode {
        line: string;
        children: ShaderCodeNode[];
        additionalDefineKey?: string;
        additionalDefineValue?: string;
        isValid(preprocessors: {
            [key: string]: string;
        }): boolean;
        process(preprocessors: {
            [key: string]: string;
        }, options: ProcessingOptions): string;
    }
}
declare module "babylonjs/Engines/Processors/shaderCodeCursor" {
    /** @hidden */
    export class ShaderCodeCursor {
        private _lines;
        lineIndex: number;
        get currentLine(): string;
        get canRead(): boolean;
        set lines(value: string[]);
    }
}
declare module "babylonjs/Engines/Processors/shaderCodeConditionNode" {
    import { ShaderCodeNode } from "babylonjs/Engines/Processors/shaderCodeNode";
    import { ProcessingOptions } from "babylonjs/Engines/Processors/shaderProcessingOptions";
    /** @hidden */
    export class ShaderCodeConditionNode extends ShaderCodeNode {
        process(preprocessors: {
            [key: string]: string;
        }, options: ProcessingOptions): string;
    }
}
declare module "babylonjs/Engines/Processors/Expressions/shaderDefineExpression" {
    /** @hidden */
    export class ShaderDefineExpression {
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
        private static _OperatorPriority;
        private static _Stack;
        static postfixToInfix(postfix: string[]): string;
        static infixToPostfix(infix: string): string[];
    }
}
declare module "babylonjs/Engines/Processors/shaderCodeTestNode" {
    import { ShaderCodeNode } from "babylonjs/Engines/Processors/shaderCodeNode";
    import { ShaderDefineExpression } from "babylonjs/Engines/Processors/Expressions/shaderDefineExpression";
    /** @hidden */
    export class ShaderCodeTestNode extends ShaderCodeNode {
        testExpression: ShaderDefineExpression;
        isValid(preprocessors: {
            [key: string]: string;
        }): boolean;
    }
}
declare module "babylonjs/Engines/Processors/Expressions/Operators/shaderDefineIsDefinedOperator" {
    import { ShaderDefineExpression } from "babylonjs/Engines/Processors/Expressions/shaderDefineExpression";
    /** @hidden */
    export class ShaderDefineIsDefinedOperator extends ShaderDefineExpression {
        define: string;
        not: boolean;
        constructor(define: string, not?: boolean);
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }
}
declare module "babylonjs/Engines/Processors/Expressions/Operators/shaderDefineOrOperator" {
    import { ShaderDefineExpression } from "babylonjs/Engines/Processors/Expressions/shaderDefineExpression";
    /** @hidden */
    export class ShaderDefineOrOperator extends ShaderDefineExpression {
        leftOperand: ShaderDefineExpression;
        rightOperand: ShaderDefineExpression;
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }
}
declare module "babylonjs/Engines/Processors/Expressions/Operators/shaderDefineAndOperator" {
    import { ShaderDefineExpression } from "babylonjs/Engines/Processors/Expressions/shaderDefineExpression";
    /** @hidden */
    export class ShaderDefineAndOperator extends ShaderDefineExpression {
        leftOperand: ShaderDefineExpression;
        rightOperand: ShaderDefineExpression;
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }
}
declare module "babylonjs/Engines/Processors/Expressions/Operators/shaderDefineArithmeticOperator" {
    import { ShaderDefineExpression } from "babylonjs/Engines/Processors/Expressions/shaderDefineExpression";
    /** @hidden */
    export class ShaderDefineArithmeticOperator extends ShaderDefineExpression {
        define: string;
        operand: string;
        testValue: string;
        constructor(define: string, operand: string, testValue: string);
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }
}
declare module "babylonjs/Offline/IOfflineProvider" {
    /**
     * Class used to enable access to offline support
     * @see https://doc.babylonjs.com/how_to/caching_resources_in_indexeddb
     */
    export interface IOfflineProvider {
        /**
         * Gets a boolean indicating if scene must be saved in the database
         */
        enableSceneOffline: boolean;
        /**
         * Gets a boolean indicating if textures must be saved in the database
         */
        enableTexturesOffline: boolean;
        /**
         * Open the offline support and make it available
         * @param successCallback defines the callback to call on success
         * @param errorCallback defines the callback to call on error
         */
        open(successCallback: () => void, errorCallback: () => void): void;
        /**
         * Loads an image from the offline support
         * @param url defines the url to load from
         * @param image defines the target DOM image
         */
        loadImage(url: string, image: HTMLImageElement): void;
        /**
         * Loads a file from offline support
         * @param url defines the URL to load from
         * @param sceneLoaded defines a callback to call on success
         * @param progressCallBack defines a callback to call when progress changed
         * @param errorCallback defines a callback to call on error
         * @param useArrayBuffer defines a boolean to use array buffer instead of text string
         */
        loadFile(url: string, sceneLoaded: (data: any) => void, progressCallBack?: (data: any) => void, errorCallback?: () => void, useArrayBuffer?: boolean): void;
    }
}
declare module "babylonjs/Misc/filesInputStore" {
    /**
     * Class used to help managing file picking and drag'n'drop
     * File Storage
     */
    export class FilesInputStore {
        /**
         * List of files ready to be loaded
         */
        static FilesToLoad: {
            [key: string]: File;
        };
    }
}
declare module "babylonjs/Misc/retryStrategy" {
    import { WebRequest } from "babylonjs/Misc/webRequest";
    /**
     * Class used to define a retry strategy when error happens while loading assets
     */
    export class RetryStrategy {
        /**
         * Function used to defines an exponential back off strategy
         * @param maxRetries defines the maximum number of retries (3 by default)
         * @param baseInterval defines the interval between retries
         * @returns the strategy function to use
         */
        static ExponentialBackoff(maxRetries?: number, baseInterval?: number): (url: string, request: WebRequest, retryIndex: number) => number;
    }
}
declare module "babylonjs/Misc/baseError" {
    /**
     * @ignore
     * Application error to support additional information when loading a file
     */
    export abstract class BaseError extends Error {
        protected static _setPrototypeOf: (o: any, proto: object | null) => any;
    }
}
declare module "babylonjs/Misc/fileTools" {
    import { WebRequest } from "babylonjs/Misc/webRequest";
    import { Nullable } from "babylonjs/types";
    import { IOfflineProvider } from "babylonjs/Offline/IOfflineProvider";
    import { IFileRequest } from "babylonjs/Misc/fileRequest";
    import { BaseError } from "babylonjs/Misc/baseError";
    /** @ignore */
    export class LoadFileError extends BaseError {
        request?: WebRequest;
        file?: File;
        /**
         * Creates a new LoadFileError
         * @param message defines the message of the error
         * @param request defines the optional web request
         * @param file defines the optional file
         */
        constructor(message: string, object?: WebRequest | File);
    }
    /** @ignore */
    export class RequestFileError extends BaseError {
        request: WebRequest;
        /**
         * Creates a new LoadFileError
         * @param message defines the message of the error
         * @param request defines the optional web request
         */
        constructor(message: string, request: WebRequest);
    }
    /** @ignore */
    export class ReadFileError extends BaseError {
        file: File;
        /**
         * Creates a new ReadFileError
         * @param message defines the message of the error
         * @param file defines the optional file
         */
        constructor(message: string, file: File);
    }
    /**
     * @hidden
     */
    export class FileTools {
        /**
         * Gets or sets the retry strategy to apply when an error happens while loading an asset
         */
        static DefaultRetryStrategy: (url: string, request: WebRequest, retryIndex: number) => number;
        /**
         * Gets or sets the base URL to use to load assets
         */
        static BaseUrl: string;
        /**
         * Default behaviour for cors in the application.
         * It can be a string if the expected behavior is identical in the entire app.
         * Or a callback to be able to set it per url or on a group of them (in case of Video source for instance)
         */
        static CorsBehavior: string | ((url: string | string[]) => string);
        /**
         * Gets or sets a function used to pre-process url before using them to load assets
         */
        static PreprocessUrl: (url: string) => string;
        /**
         * Removes unwanted characters from an url
         * @param url defines the url to clean
         * @returns the cleaned url
         */
        private static _CleanUrl;
        /**
         * Sets the cors behavior on a dom element. This will add the required Tools.CorsBehavior to the element.
         * @param url define the url we are trying
         * @param element define the dom element where to configure the cors policy
         */
        static SetCorsBehavior(url: string | string[], element: {
            crossOrigin: string | null;
        }): void;
        /**
         * Loads an image as an HTMLImageElement.
         * @param input url string, ArrayBuffer, or Blob to load
         * @param onLoad callback called when the image successfully loads
         * @param onError callback called when the image fails to load
         * @param offlineProvider offline provider for caching
         * @param mimeType optional mime type
         * @returns the HTMLImageElement of the loaded image
         */
        static LoadImage(input: string | ArrayBuffer | ArrayBufferView | Blob, onLoad: (img: HTMLImageElement | ImageBitmap) => void, onError: (message?: string, exception?: any) => void, offlineProvider: Nullable<IOfflineProvider>, mimeType?: string): Nullable<HTMLImageElement>;
        /**
         * Reads a file from a File object
         * @param file defines the file to load
         * @param onSuccess defines the callback to call when data is loaded
         * @param onProgress defines the callback to call during loading process
         * @param useArrayBuffer defines a boolean indicating that data must be returned as an ArrayBuffer
         * @param onError defines the callback to call when an error occurs
         * @returns a file request object
         */
        static ReadFile(file: File, onSuccess: (data: any) => void, onProgress?: (ev: ProgressEvent) => any, useArrayBuffer?: boolean, onError?: (error: ReadFileError) => void): IFileRequest;
        /**
         * Loads a file from a url
         * @param url url to load
         * @param onSuccess callback called when the file successfully loads
         * @param onProgress callback called while file is loading (if the server supports this mode)
         * @param offlineProvider defines the offline provider for caching
         * @param useArrayBuffer defines a boolean indicating that date must be returned as ArrayBuffer
         * @param onError callback called when the file fails to load
         * @returns a file request object
         */
        static LoadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (ev: ProgressEvent) => void, offlineProvider?: IOfflineProvider, useArrayBuffer?: boolean, onError?: (request?: WebRequest, exception?: LoadFileError) => void): IFileRequest;
        /**
         * Loads a file
         * @param url url to load
         * @param onSuccess callback called when the file successfully loads
         * @param onProgress callback called while file is loading (if the server supports this mode)
         * @param useArrayBuffer defines a boolean indicating that date must be returned as ArrayBuffer
         * @param onError callback called when the file fails to load
         * @param onOpened callback called when the web request is opened
         * @returns a file request object
         */
        static RequestFile(url: string, onSuccess: (data: string | ArrayBuffer, request?: WebRequest) => void, onProgress?: (event: ProgressEvent) => void, offlineProvider?: IOfflineProvider, useArrayBuffer?: boolean, onError?: (error: RequestFileError) => void, onOpened?: (request: WebRequest) => void): IFileRequest;
        /**
         * Checks if the loaded document was accessed via `file:`-Protocol.
         * @returns boolean
         */
        static IsFileURL(): boolean;
    }
}
declare module "babylonjs/Engines/Processors/shaderProcessor" {
    import { ProcessingOptions } from "babylonjs/Engines/Processors/shaderProcessingOptions";
    import { WebRequest } from "babylonjs/Misc/webRequest";
    import { LoadFileError } from "babylonjs/Misc/fileTools";
    import { IOfflineProvider } from "babylonjs/Offline/IOfflineProvider";
    import { IFileRequest } from "babylonjs/Misc/fileRequest";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    /** @hidden */
    export class ShaderProcessor {
        static Process(sourceCode: string, options: ProcessingOptions, callback: (migratedCode: string) => void, engine: ThinEngine): void;
        private static _ProcessPrecision;
        private static _ExtractOperation;
        private static _BuildSubExpression;
        private static _BuildExpression;
        private static _MoveCursorWithinIf;
        private static _MoveCursor;
        private static _EvaluatePreProcessors;
        private static _PreparePreProcessors;
        private static _ProcessShaderConversion;
        private static _ProcessIncludes;
        /**
         * Loads a file from a url
         * @param url url to load
         * @param onSuccess callback called when the file successfully loads
         * @param onProgress callback called while file is loading (if the server supports this mode)
         * @param offlineProvider defines the offline provider for caching
         * @param useArrayBuffer defines a boolean indicating that date must be returned as ArrayBuffer
         * @param onError callback called when the file fails to load
         * @returns a file request object
         * @hidden
         */
        static _FileToolsLoadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (ev: ProgressEvent) => void, offlineProvider?: IOfflineProvider, useArrayBuffer?: boolean, onError?: (request?: WebRequest, exception?: LoadFileError) => void): IFileRequest;
    }
}
declare module "babylonjs/Maths/math.like" {
    import { float, int, DeepImmutable } from "babylonjs/types";
    /**
     * @hidden
     */
    export interface IColor4Like {
        r: float;
        g: float;
        b: float;
        a: float;
    }
    /**
     * @hidden
     */
    export interface IColor3Like {
        r: float;
        g: float;
        b: float;
    }
    /**
     * @hidden
     */
    export interface IVector4Like {
        x: float;
        y: float;
        z: float;
        w: float;
    }
    /**
     * @hidden
     */
    export interface IVector3Like {
        x: float;
        y: float;
        z: float;
    }
    /**
     * @hidden
     */
    export interface IVector2Like {
        x: float;
        y: float;
    }
    /**
     * @hidden
     */
    export interface IMatrixLike {
        toArray(): DeepImmutable<Float32Array | Array<number>>;
        updateFlag: int;
    }
    /**
     * @hidden
     */
    export interface IViewportLike {
        x: float;
        y: float;
        width: float;
        height: float;
    }
    /**
     * @hidden
     */
    export interface IPlaneLike {
        normal: IVector3Like;
        d: float;
        normalize(): void;
    }
}
declare module "babylonjs/Materials/iEffectFallbacks" {
    import { Effect } from "babylonjs/Materials/effect";
    /**
     * Interface used to define common properties for effect fallbacks
     */
    export interface IEffectFallbacks {
        /**
         * Removes the defines that should be removed when falling back.
         * @param currentDefines defines the current define statements for the shader.
         * @param effect defines the current effect we try to compile
         * @returns The resulting defines with defines of the current rank removed.
         */
        reduce(currentDefines: string, effect: Effect): string;
        /**
         * Removes the fallback from the bound mesh.
         */
        unBindMesh(): void;
        /**
         * Checks to see if more fallbacks are still availible.
         */
        hasMoreFallbacks: boolean;
    }
}
declare module "babylonjs/Maths/math.size" {
    /**
     * Interface for the size containing width and height
     */
    export interface ISize {
        /**
         * Width
         */
        width: number;
        /**
         * Heighht
         */
        height: number;
    }
    /**
     * Size containing widht and height
     */
    export class Size implements ISize {
        /**
         * Width
         */
        width: number;
        /**
         * Height
         */
        height: number;
        /**
         * Creates a Size object from the given width and height (floats).
         * @param width width of the new size
         * @param height height of the new size
         */
        constructor(width: number, height: number);
        /**
         * Returns a string with the Size width and height
         * @returns a string with the Size width and height
         */
        toString(): string;
        /**
         * "Size"
         * @returns the string "Size"
         */
        getClassName(): string;
        /**
         * Returns the Size hash code.
         * @returns a hash code for a unique width and height
         */
        getHashCode(): number;
        /**
         * Updates the current size from the given one.
         * @param src the given size
         */
        copyFrom(src: Size): void;
        /**
         * Updates in place the current Size from the given floats.
         * @param width width of the new size
         * @param height height of the new size
         * @returns the updated Size.
         */
        copyFromFloats(width: number, height: number): Size;
        /**
         * Updates in place the current Size from the given floats.
         * @param width width to set
         * @param height height to set
         * @returns the updated Size.
         */
        set(width: number, height: number): Size;
        /**
         * Multiplies the width and height by numbers
         * @param w factor to multiple the width by
         * @param h factor to multiple the height by
         * @returns a new Size set with the multiplication result of the current Size and the given floats.
         */
        multiplyByFloats(w: number, h: number): Size;
        /**
         * Clones the size
         * @returns a new Size copied from the given one.
         */
        clone(): Size;
        /**
         * True if the current Size and the given one width and height are strictly equal.
         * @param other the other size to compare against
         * @returns True if the current Size and the given one width and height are strictly equal.
         */
        equals(other: Size): boolean;
        /**
         * The surface of the Size : width * height (float).
         */
        get surface(): number;
        /**
         * Create a new size of zero
         * @returns a new Size set to (0.0, 0.0)
         */
        static Zero(): Size;
        /**
         * Sums the width and height of two sizes
         * @param otherSize size to add to this size
         * @returns a new Size set as the addition result of the current Size and the given one.
         */
        add(otherSize: Size): Size;
        /**
         * Subtracts the width and height of two
         * @param otherSize size to subtract to this size
         * @returns a new Size set as the subtraction result of  the given one from the current Size.
         */
        subtract(otherSize: Size): Size;
        /**
         * Creates a new Size set at the linear interpolation "amount" between "start" and "end"
         * @param start starting size to lerp between
         * @param end end size to lerp between
         * @param amount amount to lerp between the start and end values
         * @returns a new Size set at the linear interpolation "amount" between "start" and "end"
         */
        static Lerp(start: Size, end: Size, amount: number): Size;
    }
}
declare module "babylonjs/Materials/Textures/thinTexture" {
    import { Nullable } from "babylonjs/types";
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { ISize } from "babylonjs/Maths/math.size";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    /**
     * Base class of all the textures in babylon.
     * It groups all the common properties required to work with Thin Engine.
     */
    export class ThinTexture {
        protected _wrapU: number;
        /**
        * | Value | Type               | Description |
        * | ----- | ------------------ | ----------- |
        * | 0     | CLAMP_ADDRESSMODE  |             |
        * | 1     | WRAP_ADDRESSMODE   |             |
        * | 2     | MIRROR_ADDRESSMODE |             |
        */
        get wrapU(): number;
        set wrapU(value: number);
        protected _wrapV: number;
        /**
        * | Value | Type               | Description |
        * | ----- | ------------------ | ----------- |
        * | 0     | CLAMP_ADDRESSMODE  |             |
        * | 1     | WRAP_ADDRESSMODE   |             |
        * | 2     | MIRROR_ADDRESSMODE |             |
        */
        get wrapV(): number;
        set wrapV(value: number);
        /**
         * | Value | Type               | Description |
         * | ----- | ------------------ | ----------- |
         * | 0     | CLAMP_ADDRESSMODE  |             |
         * | 1     | WRAP_ADDRESSMODE   |             |
         * | 2     | MIRROR_ADDRESSMODE |             |
         */
        wrapR: number;
        /**
         * With compliant hardware and browser (supporting anisotropic filtering)
         * this defines the level of anisotropic filtering in the texture.
         * The higher the better but the slower. This defaults to 4 as it seems to be the best tradeoff.
         */
        anisotropicFilteringLevel: number;
        /**
         * Define the current state of the loading sequence when in delayed load mode.
         */
        delayLoadState: number;
        /**
         * How a texture is mapped.
         * Unused in thin texture mode.
         */
        get coordinatesMode(): number;
        /**
         * Define if the texture is a cube texture or if false a 2d texture.
         */
        get isCube(): boolean;
        set isCube(value: boolean);
        /**
         * Define if the texture is a 3d texture (webgl 2) or if false a 2d texture.
         */
        get is3D(): boolean;
        set is3D(value: boolean);
        /**
         * Define if the texture is a 2d array texture (webgl 2) or if false a 2d texture.
         */
        get is2DArray(): boolean;
        set is2DArray(value: boolean);
        /**
         * Get the class name of the texture.
         * @returns "ThinTexture"
         */
        getClassName(): string;
        /** @hidden */
        _texture: Nullable<InternalTexture>;
        protected _engine: Nullable<ThinEngine>;
        private _cachedSize;
        private _cachedBaseSize;
        /**
         * Instantiates a new ThinTexture.
         * Base class of all the textures in babylon.
         * This can be used as an internal texture wrapper in ThinEngine to benefit from the cache
         * @param internalTexture Define the internalTexture to wrap
         */
        constructor(internalTexture: Nullable<InternalTexture>);
        /**
         * Get if the texture is ready to be used (downloaded, converted, mip mapped...).
         * @returns true if fully ready
         */
        isReady(): boolean;
        /**
         * Triggers the load sequence in delayed load mode.
         */
        delayLoad(): void;
        /**
         * Get the underlying lower level texture from Babylon.
         * @returns the insternal texture
         */
        getInternalTexture(): Nullable<InternalTexture>;
        /**
         * Get the size of the texture.
         * @returns the texture size.
         */
        getSize(): ISize;
        /**
         * Get the base size of the texture.
         * It can be different from the size if the texture has been resized for POT for instance
         * @returns the base size
         */
        getBaseSize(): ISize;
        /**
         * Update the sampling mode of the texture.
         * Default is Trilinear mode.
         *
         * | Value | Type               | Description |
         * | ----- | ------------------ | ----------- |
         * | 1     | NEAREST_SAMPLINGMODE or NEAREST_NEAREST_MIPLINEAR  | Nearest is: mag = nearest, min = nearest, mip = linear |
         * | 2     | BILINEAR_SAMPLINGMODE or LINEAR_LINEAR_MIPNEAREST | Bilinear is: mag = linear, min = linear, mip = nearest |
         * | 3     | TRILINEAR_SAMPLINGMODE or LINEAR_LINEAR_MIPLINEAR | Trilinear is: mag = linear, min = linear, mip = linear |
         * | 4     | NEAREST_NEAREST_MIPNEAREST |             |
         * | 5    | NEAREST_LINEAR_MIPNEAREST |             |
         * | 6    | NEAREST_LINEAR_MIPLINEAR |             |
         * | 7    | NEAREST_LINEAR |             |
         * | 8    | NEAREST_NEAREST |             |
         * | 9   | LINEAR_NEAREST_MIPNEAREST |             |
         * | 10   | LINEAR_NEAREST_MIPLINEAR |             |
         * | 11   | LINEAR_LINEAR |             |
         * | 12   | LINEAR_NEAREST |             |
         *
         *    > _mag_: magnification filter (close to the viewer)
         *    > _min_: minification filter (far from the viewer)
         *    > _mip_: filter used between mip map levels
         *@param samplingMode Define the new sampling mode of the texture
         */
        updateSamplingMode(samplingMode: number): void;
        /**
         * Release and destroy the underlying lower level texture aka internalTexture.
         */
        releaseInternalTexture(): void;
        /**
         * Dispose the texture and release its associated resources.
         */
        dispose(): void;
    }
}
declare module "babylonjs/Misc/smartArray" {
    /**
     * Defines an array and its length.
     * It can be helpfull to group result from both Arrays and smart arrays in one structure.
     */
    export interface ISmartArrayLike<T> {
        /**
         * The data of the array.
         */
        data: Array<T>;
        /**
         * The active length of the array.
         */
        length: number;
    }
    /**
     * Defines an GC Friendly array where the backfield array do not shrink to prevent over allocations.
     */
    export class SmartArray<T> implements ISmartArrayLike<T> {
        /**
         * The full set of data from the array.
         */
        data: Array<T>;
        /**
         * The active length of the array.
         */
        length: number;
        protected _id: number;
        /**
         * Instantiates a Smart Array.
         * @param capacity defines the default capacity of the array.
         */
        constructor(capacity: number);
        /**
         * Pushes a value at the end of the active data.
         * @param value defines the object to push in the array.
         */
        push(value: T): void;
        /**
         * Iterates over the active data and apply the lambda to them.
         * @param func defines the action to apply on each value.
         */
        forEach(func: (content: T) => void): void;
        /**
         * Sorts the full sets of data.
         * @param compareFn defines the comparison function to apply.
         */
        sort(compareFn: (a: T, b: T) => number): void;
        /**
         * Resets the active data to an empty array.
         */
        reset(): void;
        /**
         * Releases all the data from the array as well as the array.
         */
        dispose(): void;
        /**
         * Concats the active data with a given array.
         * @param array defines the data to concatenate with.
         */
        concat(array: any): void;
        /**
         * Returns the position of a value in the active data.
         * @param value defines the value to find the index for
         * @returns the index if found in the active data otherwise -1
         */
        indexOf(value: T): number;
        /**
         * Returns whether an element is part of the active data.
         * @param value defines the value to look for
         * @returns true if found in the active data otherwise false
         */
        contains(value: T): boolean;
        private static _GlobalId;
    }
    /**
     * Defines an GC Friendly array where the backfield array do not shrink to prevent over allocations.
     * The data in this array can only be present once
     */
    export class SmartArrayNoDuplicate<T> extends SmartArray<T> {
        private _duplicateId;
        /**
         * Pushes a value at the end of the active data.
         * THIS DOES NOT PREVENT DUPPLICATE DATA
         * @param value defines the object to push in the array.
         */
        push(value: T): void;
        /**
         * Pushes a value at the end of the active data.
         * If the data is already present, it won t be added again
         * @param value defines the object to push in the array.
         * @returns true if added false if it was already present
         */
        pushNoDuplicate(value: T): boolean;
        /**
         * Resets the active data to an empty array.
         */
        reset(): void;
        /**
         * Concats the active data with a given array.
         * This ensures no dupplicate will be present in the result.
         * @param array defines the data to concatenate with.
         */
        concatWithNoDuplicate(array: any): void;
    }
}
declare module "babylonjs/Misc/andOrNotEvaluator" {
    /**
     * Class used to evalaute queries containing `and` and `or` operators
     */
    export class AndOrNotEvaluator {
        /**
         * Evaluate a query
         * @param query defines the query to evaluate
         * @param evaluateCallback defines the callback used to filter result
         * @returns true if the query matches
         */
        static Eval(query: string, evaluateCallback: (val: any) => boolean): boolean;
        private static _HandleParenthesisContent;
        private static _SimplifyNegation;
    }
}
declare module "babylonjs/Misc/tags" {
    /**
     * Class used to store custom tags
     */
    export class Tags {
        /**
         * Adds support for tags on the given object
         * @param obj defines the object to use
         */
        static EnableFor(obj: any): void;
        /**
         * Removes tags support
         * @param obj defines the object to use
         */
        static DisableFor(obj: any): void;
        /**
         * Gets a boolean indicating if the given object has tags
         * @param obj defines the object to use
         * @returns a boolean
         */
        static HasTags(obj: any): boolean;
        /**
         * Gets the tags available on a given object
         * @param obj defines the object to use
         * @param asString defines if the tags must be returned as a string instead of an array of strings
         * @returns the tags
         */
        static GetTags(obj: any, asString?: boolean): any;
        /**
         * Adds tags to an object
         * @param obj defines the object to use
         * @param tagsString defines the tag string. The tags 'true' and 'false' are reserved and cannot be used as tags.
         * A tag cannot start with '||', '&&', and '!'. It cannot contain whitespaces
         */
        static AddTagsTo(obj: any, tagsString: string): void;
        /**
         * @hidden
         */
        static _AddTagTo(obj: any, tag: string): void;
        /**
         * Removes specific tags from a specific object
         * @param obj defines the object to use
         * @param tagsString defines the tags to remove
         */
        static RemoveTagsFrom(obj: any, tagsString: string): void;
        /**
         * @hidden
         */
        static _RemoveTagFrom(obj: any, tag: string): void;
        /**
         * Defines if tags hosted on an object match a given query
         * @param obj defines the object to use
         * @param tagsQuery defines the tag query
         * @returns a boolean
         */
        static MatchesQuery(obj: any, tagsQuery: string): boolean;
    }
}
declare module "babylonjs/Maths/math.scalar" {
    /**
     * Scalar computation library
     */
    export class Scalar {
        /**
         * Two pi constants convenient for computation.
         */
        static TwoPi: number;
        /**
         * Boolean : true if the absolute difference between a and b is lower than epsilon (default = 1.401298E-45)
         * @param a number
         * @param b number
         * @param epsilon (default = 1.401298E-45)
         * @returns true if the absolute difference between a and b is lower than epsilon (default = 1.401298E-45)
         */
        static WithinEpsilon(a: number, b: number, epsilon?: number): boolean;
        /**
         * Returns a string : the upper case translation of the number i to hexadecimal.
         * @param i number
         * @returns the upper case translation of the number i to hexadecimal.
         */
        static ToHex(i: number): string;
        /**
         * Returns -1 if value is negative and +1 is value is positive.
         * @param value the value
         * @returns the value itself if it's equal to zero.
         */
        static Sign(value: number): number;
        /**
         * Returns the value itself if it's between min and max.
         * Returns min if the value is lower than min.
         * Returns max if the value is greater than max.
         * @param value the value to clmap
         * @param min the min value to clamp to (default: 0)
         * @param max the max value to clamp to (default: 1)
         * @returns the clamped value
         */
        static Clamp(value: number, min?: number, max?: number): number;
        /**
         * the log2 of value.
         * @param value the value to compute log2 of
         * @returns the log2 of value.
         */
        static Log2(value: number): number;
        /**
        * Loops the value, so that it is never larger than length and never smaller than 0.
        *
        * This is similar to the modulo operator but it works with floating point numbers.
        * For example, using 3.0 for t and 2.5 for length, the result would be 0.5.
        * With t = 5 and length = 2.5, the result would be 0.0.
        * Note, however, that the behaviour is not defined for negative numbers as it is for the modulo operator
        * @param value the value
        * @param length the length
        * @returns the looped value
        */
        static Repeat(value: number, length: number): number;
        /**
         * Normalize the value between 0.0 and 1.0 using min and max values
         * @param value value to normalize
         * @param min max to normalize between
         * @param max min to normalize between
         * @returns the normalized value
         */
        static Normalize(value: number, min: number, max: number): number;
        /**
        * Denormalize the value from 0.0 and 1.0 using min and max values
        * @param normalized value to denormalize
        * @param min max to denormalize between
        * @param max min to denormalize between
        * @returns the denormalized value
        */
        static Denormalize(normalized: number, min: number, max: number): number;
        /**
        * Calculates the shortest difference between two given angles given in degrees.
        * @param current current angle in degrees
        * @param target target angle in degrees
        * @returns the delta
        */
        static DeltaAngle(current: number, target: number): number;
        /**
        * PingPongs the value t, so that it is never larger than length and never smaller than 0.
        * @param tx value
        * @param length length
        * @returns The returned value will move back and forth between 0 and length
        */
        static PingPong(tx: number, length: number): number;
        /**
        * Interpolates between min and max with smoothing at the limits.
        *
        * This function interpolates between min and max in a similar way to Lerp. However, the interpolation will gradually speed up
        * from the start and slow down toward the end. This is useful for creating natural-looking animation, fading and other transitions.
        * @param from from
        * @param to to
        * @param tx value
        * @returns the smooth stepped value
        */
        static SmoothStep(from: number, to: number, tx: number): number;
        /**
        * Moves a value current towards target.
        *
        * This is essentially the same as Mathf.Lerp but instead the function will ensure that the speed never exceeds maxDelta.
        * Negative values of maxDelta pushes the value away from target.
        * @param current current value
        * @param target target value
        * @param maxDelta max distance to move
        * @returns resulting value
        */
        static MoveTowards(current: number, target: number, maxDelta: number): number;
        /**
        * Same as MoveTowards but makes sure the values interpolate correctly when they wrap around 360 degrees.
        *
        * Variables current and target are assumed to be in degrees. For optimization reasons, negative values of maxDelta
        *  are not supported and may cause oscillation. To push current away from a target angle, add 180 to that angle instead.
        * @param current current value
        * @param target target value
        * @param maxDelta max distance to move
        * @returns resulting angle
        */
        static MoveTowardsAngle(current: number, target: number, maxDelta: number): number;
        /**
         * Creates a new scalar with values linearly interpolated of "amount" between the start scalar and the end scalar.
         * @param start start value
         * @param end target value
         * @param amount amount to lerp between
         * @returns the lerped value
         */
        static Lerp(start: number, end: number, amount: number): number;
        /**
        * Same as Lerp but makes sure the values interpolate correctly when they wrap around 360 degrees.
        * The parameter t is clamped to the range [0, 1]. Variables a and b are assumed to be in degrees.
        * @param start start value
        * @param end target value
        * @param amount amount to lerp between
        * @returns the lerped value
        */
        static LerpAngle(start: number, end: number, amount: number): number;
        /**
        * Calculates the linear parameter t that produces the interpolant value within the range [a, b].
        * @param a start value
        * @param b target value
        * @param value value between a and b
        * @returns the inverseLerp value
        */
        static InverseLerp(a: number, b: number, value: number): number;
        /**
         * Returns a new scalar located for "amount" (float) on the Hermite spline defined by the scalars "value1", "value3", "tangent1", "tangent2".
         * @see http://mathworld.wolfram.com/HermitePolynomial.html
         * @param value1 spline value
         * @param tangent1 spline value
         * @param value2 spline value
         * @param tangent2 spline value
         * @param amount input value
         * @returns hermite result
         */
        static Hermite(value1: number, tangent1: number, value2: number, tangent2: number, amount: number): number;
        /**
        * Returns a random float number between and min and max values
        * @param min min value of random
        * @param max max value of random
        * @returns random value
        */
        static RandomRange(min: number, max: number): number;
        /**
        * This function returns percentage of a number in a given range.
        *
        * RangeToPercent(40,20,60) will return 0.5 (50%)
        * RangeToPercent(34,0,100) will return 0.34 (34%)
        * @param number to convert to percentage
        * @param min min range
        * @param max max range
        * @returns the percentage
        */
        static RangeToPercent(number: number, min: number, max: number): number;
        /**
        * This function returns number that corresponds to the percentage in a given range.
        *
        * PercentToRange(0.34,0,100) will return 34.
        * @param percent to convert to number
        * @param min min range
        * @param max max range
        * @returns the number
        */
        static PercentToRange(percent: number, min: number, max: number): number;
        /**
         * Returns the angle converted to equivalent value between -Math.PI and Math.PI radians.
         * @param angle The angle to normalize in radian.
         * @return The converted angle.
         */
        static NormalizeRadians(angle: number): number;
    }
}
declare module "babylonjs/Maths/math.constants" {
    /**
     * Constant used to convert a value to gamma space
     * @ignorenaming
     */
    export const ToGammaSpace: number;
    /**
     * Constant used to convert a value to linear space
     * @ignorenaming
     */
    export const ToLinearSpace = 2.2;
    /**
     * Constant used to define the minimal number value in Babylon.js
     * @ignorenaming
     */
    let Epsilon: number;
    export { Epsilon };
}
declare module "babylonjs/Maths/math.viewport" {
    /**
     * Class used to represent a viewport on screen
     */
    export class Viewport {
        /** viewport left coordinate */
        x: number;
        /** viewport top coordinate */
        y: number;
        /**viewport width */
        width: number;
        /** viewport height */
        height: number;
        /**
         * Creates a Viewport object located at (x, y) and sized (width, height)
         * @param x defines viewport left coordinate
         * @param y defines viewport top coordinate
         * @param width defines the viewport width
         * @param height defines the viewport height
         */
        constructor(
        /** viewport left coordinate */
        x: number, 
        /** viewport top coordinate */
        y: number, 
        /**viewport width */
        width: number, 
        /** viewport height */
        height: number);
        /**
         * Creates a new viewport using absolute sizing (from 0-> width, 0-> height instead of 0->1)
         * @param renderWidth defines the rendering width
         * @param renderHeight defines the rendering height
         * @returns a new Viewport
         */
        toGlobal(renderWidth: number, renderHeight: number): Viewport;
        /**
         * Stores absolute viewport value into a target viewport (from 0-> width, 0-> height instead of 0->1)
         * @param renderWidth defines the rendering width
         * @param renderHeight defines the rendering height
         * @param ref defines the target viewport
         * @returns the current viewport
         */
        toGlobalToRef(renderWidth: number, renderHeight: number, ref: Viewport): Viewport;
        /**
         * Returns a new Viewport copied from the current one
         * @returns a new Viewport
         */
        clone(): Viewport;
    }
}
declare module "babylonjs/Misc/arrayTools" {
    /**
     * Class containing a set of static utilities functions for arrays.
     */
    export class ArrayTools {
        /**
         * Returns an array of the given size filled with element built from the given constructor and the paramters
         * @param size the number of element to construct and put in the array
         * @param itemBuilder a callback responsible for creating new instance of item. Called once per array entry.
         * @returns a new array filled with new objects
         */
        static BuildArray<T>(size: number, itemBuilder: () => T): Array<T>;
    }
}
declare module "babylonjs/Maths/math.plane" {
    import { DeepImmutable } from "babylonjs/types";
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    /**
     * Represents a plane by the equation ax + by + cz + d = 0
     */
    export class Plane {
        private static _TmpMatrix;
        /**
         * Normal of the plane (a,b,c)
         */
        normal: Vector3;
        /**
         * d component of the plane
         */
        d: number;
        /**
         * Creates a Plane object according to the given floats a, b, c, d and the plane equation : ax + by + cz + d = 0
         * @param a a component of the plane
         * @param b b component of the plane
         * @param c c component of the plane
         * @param d d component of the plane
         */
        constructor(a: number, b: number, c: number, d: number);
        /**
         * @returns the plane coordinates as a new array of 4 elements [a, b, c, d].
         */
        asArray(): number[];
        /**
         * @returns a new plane copied from the current Plane.
         */
        clone(): Plane;
        /**
         * @returns the string "Plane".
         */
        getClassName(): string;
        /**
         * @returns the Plane hash code.
         */
        getHashCode(): number;
        /**
         * Normalize the current Plane in place.
         * @returns the updated Plane.
         */
        normalize(): Plane;
        /**
         * Applies a transformation the plane and returns the result
         * @param transformation the transformation matrix to be applied to the plane
         * @returns a new Plane as the result of the transformation of the current Plane by the given matrix.
         */
        transform(transformation: DeepImmutable<Matrix>): Plane;
        /**
         * Compute the dot product between the point and the plane normal
         * @param point point to calculate the dot product with
         * @returns the dot product (float) of the point coordinates and the plane normal.
         */
        dotCoordinate(point: DeepImmutable<Vector3>): number;
        /**
         * Updates the current Plane from the plane defined by the three given points.
         * @param point1 one of the points used to contruct the plane
         * @param point2 one of the points used to contruct the plane
         * @param point3 one of the points used to contruct the plane
         * @returns the updated Plane.
         */
        copyFromPoints(point1: DeepImmutable<Vector3>, point2: DeepImmutable<Vector3>, point3: DeepImmutable<Vector3>): Plane;
        /**
         * Checks if the plane is facing a given direction (meaning if the plane's normal is pointing in the opposite direction of the given vector).
         * Note that for this function to work as expected you should make sure that:
         *   - direction and the plane normal are normalized
         *   - epsilon is a number just bigger than -1, something like -0.99 for eg
         * @param direction the direction to check if the plane is facing
         * @param epsilon value the dot product is compared against (returns true if dot <= epsilon)
         * @returns True if the plane is facing the given direction
         */
        isFrontFacingTo(direction: DeepImmutable<Vector3>, epsilon: number): boolean;
        /**
         * Calculates the distance to a point
         * @param point point to calculate distance to
         * @returns the signed distance (float) from the given point to the Plane.
         */
        signedDistanceTo(point: DeepImmutable<Vector3>): number;
        /**
         * Creates a plane from an  array
         * @param array the array to create a plane from
         * @returns a new Plane from the given array.
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>): Plane;
        /**
         * Creates a plane from three points
         * @param point1 point used to create the plane
         * @param point2 point used to create the plane
         * @param point3 point used to create the plane
         * @returns a new Plane defined by the three given points.
         */
        static FromPoints(point1: DeepImmutable<Vector3>, point2: DeepImmutable<Vector3>, point3: DeepImmutable<Vector3>): Plane;
        /**
         * Creates a plane from an origin point and a normal
         * @param origin origin of the plane to be constructed
         * @param normal normal of the plane to be constructed
         * @returns a new Plane the normal vector to this plane at the given origin point.
         * Note : the vector "normal" is updated because normalized.
         */
        static FromPositionAndNormal(origin: DeepImmutable<Vector3>, normal: Vector3): Plane;
        /**
         * Calculates the distance from a plane and a point
         * @param origin origin of the plane to be constructed
         * @param normal normal of the plane to be constructed
         * @param point point to calculate distance to
         * @returns the signed distance between the plane defined by the normal vector at the "origin"" point and the given other point.
         */
        static SignedDistanceToPlaneFromPositionAndNormal(origin: DeepImmutable<Vector3>, normal: DeepImmutable<Vector3>, point: DeepImmutable<Vector3>): number;
    }
}
declare module "babylonjs/Engines/performanceConfigurator" {
    /** @hidden */
    export class PerformanceConfigurator {
        /** @hidden */
        static MatrixUse64Bits: boolean;
        /** @hidden */
        static MatrixTrackPrecisionChange: boolean;
        /** @hidden */
        static MatrixCurrentType: any;
        /** @hidden */
        static MatrixTrackedMatrices: Array<any> | null;
        /** @hidden */
        static SetMatrixPrecision(use64bits: boolean): void;
    }
}
declare module "babylonjs/Maths/math.vector" {
    import { Viewport } from "babylonjs/Maths/math.viewport";
    import { DeepImmutable, Nullable, FloatArray, float } from "babylonjs/types";
    import { IPlaneLike } from "babylonjs/Maths/math.like";
    import { Plane } from "babylonjs/Maths/math.plane";
    /**
     * Class representing a vector containing 2 coordinates
     */
    export class Vector2 {
        /** defines the first coordinate */
        x: number;
        /** defines the second coordinate */
        y: number;
        /**
         * Creates a new Vector2 from the given x and y coordinates
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         */
        constructor(
        /** defines the first coordinate */
        x?: number, 
        /** defines the second coordinate */
        y?: number);
        /**
         * Gets a string with the Vector2 coordinates
         * @returns a string with the Vector2 coordinates
         */
        toString(): string;
        /**
         * Gets class name
         * @returns the string "Vector2"
         */
        getClassName(): string;
        /**
         * Gets current vector hash code
         * @returns the Vector2 hash code as a number
         */
        getHashCode(): number;
        /**
         * Sets the Vector2 coordinates in the given array or Float32Array from the given index.
         * @param array defines the source array
         * @param index defines the offset in source array
         * @returns the current Vector2
         */
        toArray(array: FloatArray, index?: number): Vector2;
        /**
         * Update the current vector from an array
         * @param array defines the destination array
         * @param index defines the offset in the destination array
         * @returns the current Vector3
         */
        fromArray(array: FloatArray, index?: number): Vector2;
        /**
         * Copy the current vector to an array
         * @returns a new array with 2 elements: the Vector2 coordinates.
         */
        asArray(): number[];
        /**
         * Sets the Vector2 coordinates with the given Vector2 coordinates
         * @param source defines the source Vector2
         * @returns the current updated Vector2
         */
        copyFrom(source: DeepImmutable<Vector2>): Vector2;
        /**
         * Sets the Vector2 coordinates with the given floats
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         * @returns the current updated Vector2
         */
        copyFromFloats(x: number, y: number): Vector2;
        /**
         * Sets the Vector2 coordinates with the given floats
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         * @returns the current updated Vector2
         */
        set(x: number, y: number): Vector2;
        /**
         * Add another vector with the current one
         * @param otherVector defines the other vector
         * @returns a new Vector2 set with the addition of the current Vector2 and the given one coordinates
         */
        add(otherVector: DeepImmutable<Vector2>): Vector2;
        /**
         * Sets the "result" coordinates with the addition of the current Vector2 and the given one coordinates
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        addToRef(otherVector: DeepImmutable<Vector2>, result: Vector2): Vector2;
        /**
         * Set the Vector2 coordinates by adding the given Vector2 coordinates
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        addInPlace(otherVector: DeepImmutable<Vector2>): Vector2;
        /**
         * Gets a new Vector2 by adding the current Vector2 coordinates to the given Vector3 x, y coordinates
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        addVector3(otherVector: Vector3): Vector2;
        /**
         * Gets a new Vector2 set with the subtracted coordinates of the given one from the current Vector2
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        subtract(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the subtraction of the given one from the current Vector2 coordinates.
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        subtractToRef(otherVector: DeepImmutable<Vector2>, result: Vector2): Vector2;
        /**
         * Sets the current Vector2 coordinates by subtracting from it the given one coordinates
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        subtractInPlace(otherVector: DeepImmutable<Vector2>): Vector2;
        /**
         * Multiplies in place the current Vector2 coordinates by the given ones
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        multiplyInPlace(otherVector: DeepImmutable<Vector2>): Vector2;
        /**
         * Returns a new Vector2 set with the multiplication of the current Vector2 and the given one coordinates
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        multiply(otherVector: DeepImmutable<Vector2>): Vector2;
        /**
         * Sets "result" coordinates with the multiplication of the current Vector2 and the given one coordinates
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        multiplyToRef(otherVector: DeepImmutable<Vector2>, result: Vector2): Vector2;
        /**
         * Gets a new Vector2 set with the Vector2 coordinates multiplied by the given floats
         * @param x defines the first coordinate
         * @param y defines the second coordinate
         * @returns a new Vector2
         */
        multiplyByFloats(x: number, y: number): Vector2;
        /**
         * Returns a new Vector2 set with the Vector2 coordinates divided by the given one coordinates
         * @param otherVector defines the other vector
         * @returns a new Vector2
         */
        divide(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the Vector2 divided by the given one coordinates
         * @param otherVector defines the other vector
         * @param result defines the target vector
         * @returns the unmodified current Vector2
         */
        divideToRef(otherVector: DeepImmutable<Vector2>, result: Vector2): Vector2;
        /**
         * Divides the current Vector2 coordinates by the given ones
         * @param otherVector defines the other vector
         * @returns the current updated Vector2
         */
        divideInPlace(otherVector: DeepImmutable<Vector2>): Vector2;
        /**
         * Gets a new Vector2 with current Vector2 negated coordinates
         * @returns a new Vector2
         */
        negate(): Vector2;
        /**
         * Negate this vector in place
         * @returns this
         */
        negateInPlace(): Vector2;
        /**
         * Negate the current Vector2 and stores the result in the given vector "result" coordinates
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector2
         */
        negateToRef(result: Vector2): Vector2;
        /**
         * Multiply the Vector2 coordinates by scale
         * @param scale defines the scaling factor
         * @returns the current updated Vector2
         */
        scaleInPlace(scale: number): Vector2;
        /**
         * Returns a new Vector2 scaled by "scale" from the current Vector2
         * @param scale defines the scaling factor
         * @returns a new Vector2
         */
        scale(scale: number): Vector2;
        /**
         * Scale the current Vector2 values by a factor to a given Vector2
         * @param scale defines the scale factor
         * @param result defines the Vector2 object where to store the result
         * @returns the unmodified current Vector2
         */
        scaleToRef(scale: number, result: Vector2): Vector2;
        /**
         * Scale the current Vector2 values by a factor and add the result to a given Vector2
         * @param scale defines the scale factor
         * @param result defines the Vector2 object where to store the result
         * @returns the unmodified current Vector2
         */
        scaleAndAddToRef(scale: number, result: Vector2): Vector2;
        /**
         * Gets a boolean if two vectors are equals
         * @param otherVector defines the other vector
         * @returns true if the given vector coordinates strictly equal the current Vector2 ones
         */
        equals(otherVector: DeepImmutable<Vector2>): boolean;
        /**
         * Gets a boolean if two vectors are equals (using an epsilon value)
         * @param otherVector defines the other vector
         * @param epsilon defines the minimal distance to consider equality
         * @returns true if the given vector coordinates are close to the current ones by a distance of epsilon.
         */
        equalsWithEpsilon(otherVector: DeepImmutable<Vector2>, epsilon?: number): boolean;
        /**
         * Gets a new Vector2 from current Vector2 floored values
         * @returns a new Vector2
         */
        floor(): Vector2;
        /**
         * Gets a new Vector2 from current Vector2 floored values
         * @returns a new Vector2
         */
        fract(): Vector2;
        /**
         * Gets the length of the vector
         * @returns the vector length (float)
         */
        length(): number;
        /**
         * Gets the vector squared length
         * @returns the vector squared length (float)
         */
        lengthSquared(): number;
        /**
         * Normalize the vector
         * @returns the current updated Vector2
         */
        normalize(): Vector2;
        /**
         * Gets a new Vector2 copied from the Vector2
         * @returns a new Vector2
         */
        clone(): Vector2;
        /**
         * Gets a new Vector2(0, 0)
         * @returns a new Vector2
         */
        static Zero(): Vector2;
        /**
         * Gets a new Vector2(1, 1)
         * @returns a new Vector2
         */
        static One(): Vector2;
        /**
         * Gets a new Vector2 set from the given index element of the given array
         * @param array defines the data source
         * @param offset defines the offset in the data source
         * @returns a new Vector2
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Vector2;
        /**
         * Sets "result" from the given index element of the given array
         * @param array defines the data source
         * @param offset defines the offset in the data source
         * @param result defines the target vector
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number, result: Vector2): void;
        /**
         * Gets a new Vector2 located for "amount" (float) on the CatmullRom spline defined by the given four Vector2
         * @param value1 defines 1st point of control
         * @param value2 defines 2nd point of control
         * @param value3 defines 3rd point of control
         * @param value4 defines 4th point of control
         * @param amount defines the interpolation factor
         * @returns a new Vector2
         */
        static CatmullRom(value1: DeepImmutable<Vector2>, value2: DeepImmutable<Vector2>, value3: DeepImmutable<Vector2>, value4: DeepImmutable<Vector2>, amount: number): Vector2;
        /**
         * Returns a new Vector2 set with same the coordinates than "value" ones if the vector "value" is in the square defined by "min" and "max".
         * If a coordinate of "value" is lower than "min" coordinates, the returned Vector2 is given this "min" coordinate.
         * If a coordinate of "value" is greater than "max" coordinates, the returned Vector2 is given this "max" coordinate
         * @param value defines the value to clamp
         * @param min defines the lower limit
         * @param max defines the upper limit
         * @returns a new Vector2
         */
        static Clamp(value: DeepImmutable<Vector2>, min: DeepImmutable<Vector2>, max: DeepImmutable<Vector2>): Vector2;
        /**
         * Returns a new Vector2 located for "amount" (float) on the Hermite spline defined by the vectors "value1", "value3", "tangent1", "tangent2"
         * @param value1 defines the 1st control point
         * @param tangent1 defines the outgoing tangent
         * @param value2 defines the 2nd control point
         * @param tangent2 defines the incoming tangent
         * @param amount defines the interpolation factor
         * @returns a new Vector2
         */
        static Hermite(value1: DeepImmutable<Vector2>, tangent1: DeepImmutable<Vector2>, value2: DeepImmutable<Vector2>, tangent2: DeepImmutable<Vector2>, amount: number): Vector2;
        /**
         * Returns a new Vector2 located for "amount" (float) on the linear interpolation between the vector "start" adn the vector "end".
         * @param start defines the start vector
         * @param end defines the end vector
         * @param amount defines the interpolation factor
         * @returns a new Vector2
         */
        static Lerp(start: DeepImmutable<Vector2>, end: DeepImmutable<Vector2>, amount: number): Vector2;
        /**
         * Gets the dot product of the vector "left" and the vector "right"
         * @param left defines first vector
         * @param right defines second vector
         * @returns the dot product (float)
         */
        static Dot(left: DeepImmutable<Vector2>, right: DeepImmutable<Vector2>): number;
        /**
         * Returns a new Vector2 equal to the normalized given vector
         * @param vector defines the vector to normalize
         * @returns a new Vector2
         */
        static Normalize(vector: DeepImmutable<Vector2>): Vector2;
        /**
         * Gets a new Vector2 set with the minimal coordinate values from the "left" and "right" vectors
         * @param left defines 1st vector
         * @param right defines 2nd vector
         * @returns a new Vector2
         */
        static Minimize(left: DeepImmutable<Vector2>, right: DeepImmutable<Vector2>): Vector2;
        /**
         * Gets a new Vecto2 set with the maximal coordinate values from the "left" and "right" vectors
         * @param left defines 1st vector
         * @param right defines 2nd vector
         * @returns a new Vector2
         */
        static Maximize(left: DeepImmutable<Vector2>, right: DeepImmutable<Vector2>): Vector2;
        /**
         * Gets a new Vector2 set with the transformed coordinates of the given vector by the given transformation matrix
         * @param vector defines the vector to transform
         * @param transformation defines the matrix to apply
         * @returns a new Vector2
         */
        static Transform(vector: DeepImmutable<Vector2>, transformation: DeepImmutable<Matrix>): Vector2;
        /**
         * Transforms the given vector coordinates by the given transformation matrix and stores the result in the vector "result" coordinates
         * @param vector defines the vector to transform
         * @param transformation defines the matrix to apply
         * @param result defines the target vector
         */
        static TransformToRef(vector: DeepImmutable<Vector2>, transformation: DeepImmutable<Matrix>, result: Vector2): void;
        /**
         * Determines if a given vector is included in a triangle
         * @param p defines the vector to test
         * @param p0 defines 1st triangle point
         * @param p1 defines 2nd triangle point
         * @param p2 defines 3rd triangle point
         * @returns true if the point "p" is in the triangle defined by the vertors "p0", "p1", "p2"
         */
        static PointInTriangle(p: DeepImmutable<Vector2>, p0: DeepImmutable<Vector2>, p1: DeepImmutable<Vector2>, p2: DeepImmutable<Vector2>): boolean;
        /**
         * Gets the distance between the vectors "value1" and "value2"
         * @param value1 defines first vector
         * @param value2 defines second vector
         * @returns the distance between vectors
         */
        static Distance(value1: DeepImmutable<Vector2>, value2: DeepImmutable<Vector2>): number;
        /**
         * Returns the squared distance between the vectors "value1" and "value2"
         * @param value1 defines first vector
         * @param value2 defines second vector
         * @returns the squared distance between vectors
         */
        static DistanceSquared(value1: DeepImmutable<Vector2>, value2: DeepImmutable<Vector2>): number;
        /**
         * Gets a new Vector2 located at the center of the vectors "value1" and "value2"
         * @param value1 defines first vector
         * @param value2 defines second vector
         * @returns a new Vector2
         */
        static Center(value1: DeepImmutable<Vector2>, value2: DeepImmutable<Vector2>): Vector2;
        /**
         * Gets the shortest distance (float) between the point "p" and the segment defined by the two points "segA" and "segB".
         * @param p defines the middle point
         * @param segA defines one point of the segment
         * @param segB defines the other point of the segment
         * @returns the shortest distance
         */
        static DistanceOfPointFromSegment(p: DeepImmutable<Vector2>, segA: DeepImmutable<Vector2>, segB: DeepImmutable<Vector2>): number;
    }
    /**
     * Class used to store (x,y,z) vector representation
     * A Vector3 is the main object used in 3D geometry
     * It can represent etiher the coordinates of a point the space, either a direction
     * Reminder: js uses a left handed forward facing system
     */
    export class Vector3 {
        private static _UpReadOnly;
        private static _ZeroReadOnly;
        /** @hidden */
        _x: number;
        /** @hidden */
        _y: number;
        /** @hidden */
        _z: number;
        /** @hidden */
        _isDirty: boolean;
        /** Gets or sets the x coordinate */
        get x(): number;
        set x(value: number);
        /** Gets or sets the y coordinate */
        get y(): number;
        set y(value: number);
        /** Gets or sets the z coordinate */
        get z(): number;
        set z(value: number);
        /**
         * Creates a new Vector3 object from the given x, y, z (floats) coordinates.
         * @param x defines the first coordinates (on X axis)
         * @param y defines the second coordinates (on Y axis)
         * @param z defines the third coordinates (on Z axis)
         */
        constructor(x?: number, y?: number, z?: number);
        /**
         * Creates a string representation of the Vector3
         * @returns a string with the Vector3 coordinates.
         */
        toString(): string;
        /**
         * Gets the class name
         * @returns the string "Vector3"
         */
        getClassName(): string;
        /**
         * Creates the Vector3 hash code
         * @returns a number which tends to be unique between Vector3 instances
         */
        getHashCode(): number;
        /**
         * Creates an array containing three elements : the coordinates of the Vector3
         * @returns a new array of numbers
         */
        asArray(): number[];
        /**
         * Populates the given array or Float32Array from the given index with the successive coordinates of the Vector3
         * @param array defines the destination array
         * @param index defines the offset in the destination array
         * @returns the current Vector3
         */
        toArray(array: FloatArray, index?: number): Vector3;
        /**
         * Update the current vector from an array
         * @param array defines the destination array
         * @param index defines the offset in the destination array
         * @returns the current Vector3
         */
        fromArray(array: FloatArray, index?: number): Vector3;
        /**
         * Converts the current Vector3 into a quaternion (considering that the Vector3 contains Euler angles representation of a rotation)
         * @returns a new Quaternion object, computed from the Vector3 coordinates
         */
        toQuaternion(): Quaternion;
        /**
         * Adds the given vector to the current Vector3
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        addInPlace(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Adds the given coordinates to the current Vector3
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        addInPlaceFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Gets a new Vector3, result of the addition the current Vector3 and the given vector
         * @param otherVector defines the second operand
         * @returns the resulting Vector3
         */
        add(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Adds the current Vector3 to the given one and stores the result in the vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        addToRef(otherVector: DeepImmutable<Vector3>, result: Vector3): Vector3;
        /**
         * Subtract the given vector from the current Vector3
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        subtractInPlace(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Returns a new Vector3, result of the subtraction of the given vector from the current Vector3
         * @param otherVector defines the second operand
         * @returns the resulting Vector3
         */
        subtract(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Subtracts the given vector from the current Vector3 and stores the result in the vector "result".
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        subtractToRef(otherVector: DeepImmutable<Vector3>, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 set with the subtraction of the given floats from the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the resulting Vector3
         */
        subtractFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Subtracts the given floats from the current Vector3 coordinates and set the given vector "result" with this result
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        subtractFromFloatsToRef(x: number, y: number, z: number, result: Vector3): Vector3;
        /**
         * Gets a new Vector3 set with the current Vector3 negated coordinates
         * @returns a new Vector3
         */
        negate(): Vector3;
        /**
         * Negate this vector in place
         * @returns this
         */
        negateInPlace(): Vector3;
        /**
         * Negate the current Vector3 and stores the result in the given vector "result" coordinates
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        negateToRef(result: Vector3): Vector3;
        /**
         * Multiplies the Vector3 coordinates by the float "scale"
         * @param scale defines the multiplier factor
         * @returns the current updated Vector3
         */
        scaleInPlace(scale: number): Vector3;
        /**
         * Returns a new Vector3 set with the current Vector3 coordinates multiplied by the float "scale"
         * @param scale defines the multiplier factor
         * @returns a new Vector3
         */
        scale(scale: number): Vector3;
        /**
         * Multiplies the current Vector3 coordinates by the float "scale" and stores the result in the given vector "result" coordinates
         * @param scale defines the multiplier factor
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        scaleToRef(scale: number, result: Vector3): Vector3;
        /**
         * Scale the current Vector3 values by a factor and add the result to a given Vector3
         * @param scale defines the scale factor
         * @param result defines the Vector3 object where to store the result
         * @returns the unmodified current Vector3
         */
        scaleAndAddToRef(scale: number, result: Vector3): Vector3;
        /**
         * Projects the current vector3 to a plane along a ray starting from a specified origin and directed towards the point.
         * @param origin defines the origin of the projection ray
         * @param plane defines the plane to project to
         * @returns the projected vector3
         */
        projectOnPlane(plane: Plane, origin: Vector3): Vector3;
        /**
         * Projects the current vector3 to a plane along a ray starting from a specified origin and directed towards the point.
         * @param origin defines the origin of the projection ray
         * @param plane defines the plane to project to
         * @param result defines the Vector3 where to store the result
         */
        projectOnPlaneToRef(plane: Plane, origin: Vector3, result: Vector3): void;
        /**
         * Returns true if the current Vector3 and the given vector coordinates are strictly equal
         * @param otherVector defines the second operand
         * @returns true if both vectors are equals
         */
        equals(otherVector: DeepImmutable<Vector3>): boolean;
        /**
         * Returns true if the current Vector3 and the given vector coordinates are distant less than epsilon
         * @param otherVector defines the second operand
         * @param epsilon defines the minimal distance to define values as equals
         * @returns true if both vectors are distant less than epsilon
         */
        equalsWithEpsilon(otherVector: DeepImmutable<Vector3>, epsilon?: number): boolean;
        /**
         * Returns true if the current Vector3 coordinates equals the given floats
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns true if both vectors are equals
         */
        equalsToFloats(x: number, y: number, z: number): boolean;
        /**
         * Multiplies the current Vector3 coordinates by the given ones
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        multiplyInPlace(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Returns a new Vector3, result of the multiplication of the current Vector3 by the given vector
         * @param otherVector defines the second operand
         * @returns the new Vector3
         */
        multiply(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Multiplies the current Vector3 by the given one and stores the result in the given vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        multiplyToRef(otherVector: DeepImmutable<Vector3>, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 set with the result of the mulliplication of the current Vector3 coordinates by the given floats
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the new Vector3
         */
        multiplyByFloats(x: number, y: number, z: number): Vector3;
        /**
         * Returns a new Vector3 set with the result of the division of the current Vector3 coordinates by the given ones
         * @param otherVector defines the second operand
         * @returns the new Vector3
         */
        divide(otherVector: DeepImmutable<Vector3>): Vector3;
        /**
         * Divides the current Vector3 coordinates by the given ones and stores the result in the given vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        divideToRef(otherVector: DeepImmutable<Vector3>, result: Vector3): Vector3;
        /**
         * Divides the current Vector3 coordinates by the given ones.
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        divideInPlace(otherVector: Vector3): Vector3;
        /**
         * Updates the current Vector3 with the minimal coordinate values between its and the given vector ones
         * @param other defines the second operand
         * @returns the current updated Vector3
         */
        minimizeInPlace(other: DeepImmutable<Vector3>): Vector3;
        /**
         * Updates the current Vector3 with the maximal coordinate values between its and the given vector ones.
         * @param other defines the second operand
         * @returns the current updated Vector3
         */
        maximizeInPlace(other: DeepImmutable<Vector3>): Vector3;
        /**
         * Updates the current Vector3 with the minimal coordinate values between its and the given coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        minimizeInPlaceFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Updates the current Vector3 with the maximal coordinate values between its and the given coordinates.
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        maximizeInPlaceFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Due to float precision, scale of a mesh could be uniform but float values are off by a small fraction
         * Check if is non uniform within a certain amount of decimal places to account for this
         * @param epsilon the amount the values can differ
         * @returns if the the vector is non uniform to a certain number of decimal places
         */
        isNonUniformWithinEpsilon(epsilon: number): boolean;
        /**
         * Gets a boolean indicating that the vector is non uniform meaning x, y or z are not all the same
         */
        get isNonUniform(): boolean;
        /**
         * Gets a new Vector3 from current Vector3 floored values
         * @returns a new Vector3
         */
        floor(): Vector3;
        /**
         * Gets a new Vector3 from current Vector3 floored values
         * @returns a new Vector3
         */
        fract(): Vector3;
        /**
         * Gets the length of the Vector3
         * @returns the length of the Vector3
         */
        length(): number;
        /**
         * Gets the squared length of the Vector3
         * @returns squared length of the Vector3
         */
        lengthSquared(): number;
        /**
         * Normalize the current Vector3.
         * Please note that this is an in place operation.
         * @returns the current updated Vector3
         */
        normalize(): Vector3;
        /**
         * Reorders the x y z properties of the vector in place
         * @param order new ordering of the properties (eg. for vector 1,2,3 with "ZYX" will produce 3,2,1)
         * @returns the current updated vector
         */
        reorderInPlace(order: string): this;
        /**
         * Rotates the vector around 0,0,0 by a quaternion
         * @param quaternion the rotation quaternion
         * @param result vector to store the result
         * @returns the resulting vector
         */
        rotateByQuaternionToRef(quaternion: Quaternion, result: Vector3): Vector3;
        /**
         * Rotates a vector around a given point
         * @param quaternion the rotation quaternion
         * @param point the point to rotate around
         * @param result vector to store the result
         * @returns the resulting vector
         */
        rotateByQuaternionAroundPointToRef(quaternion: Quaternion, point: Vector3, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 as the cross product of the current vector and the "other" one
         * The cross product is then orthogonal to both current and "other"
         * @param other defines the right operand
         * @returns the cross product
         */
        cross(other: Vector3): Vector3;
        /**
         * Normalize the current Vector3 with the given input length.
         * Please note that this is an in place operation.
         * @param len the length of the vector
         * @returns the current updated Vector3
         */
        normalizeFromLength(len: number): Vector3;
        /**
         * Normalize the current Vector3 to a new vector
         * @returns the new Vector3
         */
        normalizeToNew(): Vector3;
        /**
         * Normalize the current Vector3 to the reference
         * @param reference define the Vector3 to update
         * @returns the updated Vector3
         */
        normalizeToRef(reference: Vector3): Vector3;
        /**
         * Creates a new Vector3 copied from the current Vector3
         * @returns the new Vector3
         */
        clone(): Vector3;
        /**
         * Copies the given vector coordinates to the current Vector3 ones
         * @param source defines the source Vector3
         * @returns the current updated Vector3
         */
        copyFrom(source: DeepImmutable<Vector3>): Vector3;
        /**
         * Copies the given floats to the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        copyFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Copies the given floats to the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        set(x: number, y: number, z: number): Vector3;
        /**
         * Copies the given float to the current Vector3 coordinates
         * @param v defines the x, y and z coordinates of the operand
         * @returns the current updated Vector3
         */
        setAll(v: number): Vector3;
        /**
         * Get the clip factor between two vectors
         * @param vector0 defines the first operand
         * @param vector1 defines the second operand
         * @param axis defines the axis to use
         * @param size defines the size along the axis
         * @returns the clip factor
         */
        static GetClipFactor(vector0: DeepImmutable<Vector3>, vector1: DeepImmutable<Vector3>, axis: DeepImmutable<Vector3>, size: number): number;
        /**
         * Get angle between two vectors
         * @param vector0 angle between vector0 and vector1
         * @param vector1 angle between vector0 and vector1
         * @param normal direction of the normal
         * @return the angle between vector0 and vector1
         */
        static GetAngleBetweenVectors(vector0: DeepImmutable<Vector3>, vector1: DeepImmutable<Vector3>, normal: DeepImmutable<Vector3>): number;
        /**
         * Returns a new Vector3 set from the index "offset" of the given array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @returns the new Vector3
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Vector3;
        /**
         * Returns a new Vector3 set from the index "offset" of the given Float32Array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @returns the new Vector3
         * @deprecated Please use FromArray instead.
         */
        static FromFloatArray(array: DeepImmutable<Float32Array>, offset?: number): Vector3;
        /**
         * Sets the given vector "result" with the element values from the index "offset" of the given array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param result defines the Vector3 where to store the result
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number, result: Vector3): void;
        /**
         * Sets the given vector "result" with the element values from the index "offset" of the given Float32Array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param result defines the Vector3 where to store the result
         * @deprecated Please use FromArrayToRef instead.
         */
        static FromFloatArrayToRef(array: DeepImmutable<Float32Array>, offset: number, result: Vector3): void;
        /**
         * Sets the given vector "result" with the given floats.
         * @param x defines the x coordinate of the source
         * @param y defines the y coordinate of the source
         * @param z defines the z coordinate of the source
         * @param result defines the Vector3 where to store the result
         */
        static FromFloatsToRef(x: number, y: number, z: number, result: Vector3): void;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, 0.0)
         * @returns a new empty Vector3
         */
        static Zero(): Vector3;
        /**
         * Returns a new Vector3 set to (1.0, 1.0, 1.0)
         * @returns a new unit Vector3
         */
        static One(): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 1.0, 0.0)
         * @returns a new up Vector3
         */
        static Up(): Vector3;
        /**
         * Gets a up Vector3 that must not be updated
         */
        static get UpReadOnly(): DeepImmutable<Vector3>;
        /**
         * Gets a zero Vector3 that must not be updated
         */
        static get ZeroReadOnly(): DeepImmutable<Vector3>;
        /**
         * Returns a new Vector3 set to (0.0, -1.0, 0.0)
         * @returns a new down Vector3
         */
        static Down(): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, 1.0)
         * @param rightHandedSystem is the scene right-handed (negative z)
         * @returns a new forward Vector3
         */
        static Forward(rightHandedSystem?: boolean): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, -1.0)
         * @param rightHandedSystem is the scene right-handed (negative-z)
         * @returns a new forward Vector3
         */
        static Backward(rightHandedSystem?: boolean): Vector3;
        /**
         * Returns a new Vector3 set to (1.0, 0.0, 0.0)
         * @returns a new right Vector3
         */
        static Right(): Vector3;
        /**
         * Returns a new Vector3 set to (-1.0, 0.0, 0.0)
         * @returns a new left Vector3
         */
        static Left(): Vector3;
        /**
         * Returns a new Vector3 set with the result of the transformation by the given matrix of the given vector.
         * This method computes tranformed coordinates only, not transformed direction vectors (ie. it takes translation in account)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @returns the transformed Vector3
         */
        static TransformCoordinates(vector: DeepImmutable<Vector3>, transformation: DeepImmutable<Matrix>): Vector3;
        /**
         * Sets the given vector "result" coordinates with the result of the transformation by the given matrix of the given vector
         * This method computes tranformed coordinates only, not transformed direction vectors (ie. it takes translation in account)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformCoordinatesToRef(vector: DeepImmutable<Vector3>, transformation: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Sets the given vector "result" coordinates with the result of the transformation by the given matrix of the given floats (x, y, z)
         * This method computes tranformed coordinates only, not transformed direction vectors
         * @param x define the x coordinate of the source vector
         * @param y define the y coordinate of the source vector
         * @param z define the z coordinate of the source vector
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformCoordinatesFromFloatsToRef(x: number, y: number, z: number, transformation: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Returns a new Vector3 set with the result of the normal transformation by the given matrix of the given vector
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @returns the new Vector3
         */
        static TransformNormal(vector: DeepImmutable<Vector3>, transformation: DeepImmutable<Matrix>): Vector3;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given vector
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformNormalToRef(vector: DeepImmutable<Vector3>, transformation: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given floats (x, y, z)
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param x define the x coordinate of the source vector
         * @param y define the y coordinate of the source vector
         * @param z define the z coordinate of the source vector
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformNormalFromFloatsToRef(x: number, y: number, z: number, transformation: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Returns a new Vector3 located for "amount" on the CatmullRom interpolation spline defined by the vectors "value1", "value2", "value3", "value4"
         * @param value1 defines the first control point
         * @param value2 defines the second control point
         * @param value3 defines the third control point
         * @param value4 defines the fourth control point
         * @param amount defines the amount on the spline to use
         * @returns the new Vector3
         */
        static CatmullRom(value1: DeepImmutable<Vector3>, value2: DeepImmutable<Vector3>, value3: DeepImmutable<Vector3>, value4: DeepImmutable<Vector3>, amount: number): Vector3;
        /**
         * Returns a new Vector3 set with the coordinates of "value", if the vector "value" is in the cube defined by the vectors "min" and "max"
         * If a coordinate value of "value" is lower than one of the "min" coordinate, then this "value" coordinate is set with the "min" one
         * If a coordinate value of "value" is greater than one of the "max" coordinate, then this "value" coordinate is set with the "max" one
         * @param value defines the current value
         * @param min defines the lower range value
         * @param max defines the upper range value
         * @returns the new Vector3
         */
        static Clamp(value: DeepImmutable<Vector3>, min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>): Vector3;
        /**
         * Sets the given vector "result" with the coordinates of "value", if the vector "value" is in the cube defined by the vectors "min" and "max"
         * If a coordinate value of "value" is lower than one of the "min" coordinate, then this "value" coordinate is set with the "min" one
         * If a coordinate value of "value" is greater than one of the "max" coordinate, then this "value" coordinate is set with the "max" one
         * @param value defines the current value
         * @param min defines the lower range value
         * @param max defines the upper range value
         * @param result defines the Vector3 where to store the result
         */
        static ClampToRef(value: DeepImmutable<Vector3>, min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, result: Vector3): void;
        /**
         * Checks if a given vector is inside a specific range
         * @param v defines the vector to test
         * @param min defines the minimum range
         * @param max defines the maximum range
         */
        static CheckExtends(v: Vector3, min: Vector3, max: Vector3): void;
        /**
         * Returns a new Vector3 located for "amount" (float) on the Hermite interpolation spline defined by the vectors "value1", "tangent1", "value2", "tangent2"
         * @param value1 defines the first control point
         * @param tangent1 defines the first tangent vector
         * @param value2 defines the second control point
         * @param tangent2 defines the second tangent vector
         * @param amount defines the amount on the interpolation spline (between 0 and 1)
         * @returns the new Vector3
         */
        static Hermite(value1: DeepImmutable<Vector3>, tangent1: DeepImmutable<Vector3>, value2: DeepImmutable<Vector3>, tangent2: DeepImmutable<Vector3>, amount: number): Vector3;
        /**
         * Returns a new Vector3 located for "amount" (float) on the linear interpolation between the vectors "start" and "end"
         * @param start defines the start value
         * @param end defines the end value
         * @param amount max defines amount between both (between 0 and 1)
         * @returns the new Vector3
         */
        static Lerp(start: DeepImmutable<Vector3>, end: DeepImmutable<Vector3>, amount: number): Vector3;
        /**
         * Sets the given vector "result" with the result of the linear interpolation from the vector "start" for "amount" to the vector "end"
         * @param start defines the start value
         * @param end defines the end value
         * @param amount max defines amount between both (between 0 and 1)
         * @param result defines the Vector3 where to store the result
         */
        static LerpToRef(start: DeepImmutable<Vector3>, end: DeepImmutable<Vector3>, amount: number, result: Vector3): void;
        /**
         * Returns the dot product (float) between the vectors "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the dot product
         */
        static Dot(left: DeepImmutable<Vector3>, right: DeepImmutable<Vector3>): number;
        /**
         * Returns a new Vector3 as the cross product of the vectors "left" and "right"
         * The cross product is then orthogonal to both "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the cross product
         */
        static Cross(left: DeepImmutable<Vector3>, right: DeepImmutable<Vector3>): Vector3;
        /**
         * Sets the given vector "result" with the cross product of "left" and "right"
         * The cross product is then orthogonal to both "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @param result defines the Vector3 where to store the result
         */
        static CrossToRef(left: DeepImmutable<Vector3>, right: DeepImmutable<Vector3>, result: Vector3): void;
        /**
         * Returns a new Vector3 as the normalization of the given vector
         * @param vector defines the Vector3 to normalize
         * @returns the new Vector3
         */
        static Normalize(vector: DeepImmutable<Vector3>): Vector3;
        /**
         * Sets the given vector "result" with the normalization of the given first vector
         * @param vector defines the Vector3 to normalize
         * @param result defines the Vector3 where to store the result
         */
        static NormalizeToRef(vector: DeepImmutable<Vector3>, result: Vector3): void;
        /**
         * Project a Vector3 onto screen space
         * @param vector defines the Vector3 to project
         * @param world defines the world matrix to use
         * @param transform defines the transform (view x projection) matrix to use
         * @param viewport defines the screen viewport to use
         * @returns the new Vector3
         */
        static Project(vector: DeepImmutable<Vector3>, world: DeepImmutable<Matrix>, transform: DeepImmutable<Matrix>, viewport: DeepImmutable<Viewport>): Vector3;
        /**
         * Project a Vector3 onto screen space to reference
         * @param vector defines the Vector3 to project
         * @param world defines the world matrix to use
         * @param transform defines the transform (view x projection) matrix to use
         * @param viewport defines the screen viewport to use
         * @param result the vector in which the screen space will be stored
         * @returns the new Vector3
         */
        static ProjectToRef(vector: DeepImmutable<Vector3>, world: DeepImmutable<Matrix>, transform: DeepImmutable<Matrix>, viewport: DeepImmutable<Viewport>, result: DeepImmutable<Vector3>): Vector3;
        /** @hidden */
        static _UnprojectFromInvertedMatrixToRef(source: DeepImmutable<Vector3>, matrix: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param transform defines the transform (view x projection) matrix to use
         * @returns the new Vector3
         */
        static UnprojectFromTransform(source: Vector3, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, transform: DeepImmutable<Matrix>): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @returns the new Vector3
         */
        static Unproject(source: DeepImmutable<Vector3>, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @param result defines the Vector3 where to store the result
         */
        static UnprojectToRef(source: DeepImmutable<Vector3>, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Unproject from screen space to object space
         * @param sourceX defines the screen space x coordinate to use
         * @param sourceY defines the screen space y coordinate to use
         * @param sourceZ defines the screen space z coordinate to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @param result defines the Vector3 where to store the result
         */
        static UnprojectFloatsToRef(sourceX: float, sourceY: float, sourceZ: float, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>, result: Vector3): void;
        /**
         * Gets the minimal coordinate values between two Vector3
         * @param left defines the first operand
         * @param right defines the second operand
         * @returns the new Vector3
         */
        static Minimize(left: DeepImmutable<Vector3>, right: DeepImmutable<Vector3>): Vector3;
        /**
         * Gets the maximal coordinate values between two Vector3
         * @param left defines the first operand
         * @param right defines the second operand
         * @returns the new Vector3
         */
        static Maximize(left: DeepImmutable<Vector3>, right: DeepImmutable<Vector3>): Vector3;
        /**
         * Returns the distance between the vectors "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the distance
         */
        static Distance(value1: DeepImmutable<Vector3>, value2: DeepImmutable<Vector3>): number;
        /**
         * Returns the squared distance between the vectors "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the squared distance
         */
        static DistanceSquared(value1: DeepImmutable<Vector3>, value2: DeepImmutable<Vector3>): number;
        /**
         * Returns a new Vector3 located at the center between "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the new Vector3
         */
        static Center(value1: DeepImmutable<Vector3>, value2: DeepImmutable<Vector3>): Vector3;
        /**
         * Given three orthogonal normalized left-handed oriented Vector3 axis in space (target system),
         * RotationFromAxis() returns the rotation Euler angles (ex : rotation.x, rotation.y, rotation.z) to apply
         * to something in order to rotate it from its local system to the given target system
         * Note: axis1, axis2 and axis3 are normalized during this operation
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @returns a new Vector3
         */
        static RotationFromAxis(axis1: DeepImmutable<Vector3>, axis2: DeepImmutable<Vector3>, axis3: DeepImmutable<Vector3>): Vector3;
        /**
         * The same than RotationFromAxis but updates the given ref Vector3 parameter instead of returning a new Vector3
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @param ref defines the Vector3 where to store the result
         */
        static RotationFromAxisToRef(axis1: DeepImmutable<Vector3>, axis2: DeepImmutable<Vector3>, axis3: DeepImmutable<Vector3>, ref: Vector3): void;
    }
    /**
     * Vector4 class created for EulerAngle class conversion to Quaternion
     */
    export class Vector4 {
        /** x value of the vector */
        x: number;
        /** y value of the vector */
        y: number;
        /** z value of the vector */
        z: number;
        /** w value of the vector */
        w: number;
        /**
         * Creates a Vector4 object from the given floats.
         * @param x x value of the vector
         * @param y y value of the vector
         * @param z z value of the vector
         * @param w w value of the vector
         */
        constructor(
        /** x value of the vector */
        x: number, 
        /** y value of the vector */
        y: number, 
        /** z value of the vector */
        z: number, 
        /** w value of the vector */
        w: number);
        /**
         * Returns the string with the Vector4 coordinates.
         * @returns a string containing all the vector values
         */
        toString(): string;
        /**
         * Returns the string "Vector4".
         * @returns "Vector4"
         */
        getClassName(): string;
        /**
         * Returns the Vector4 hash code.
         * @returns a unique hash code
         */
        getHashCode(): number;
        /**
         * Returns a new array populated with 4 elements : the Vector4 coordinates.
         * @returns the resulting array
         */
        asArray(): number[];
        /**
         * Populates the given array from the given index with the Vector4 coordinates.
         * @param array array to populate
         * @param index index of the array to start at (default: 0)
         * @returns the Vector4.
         */
        toArray(array: FloatArray, index?: number): Vector4;
        /**
         * Update the current vector from an array
         * @param array defines the destination array
         * @param index defines the offset in the destination array
         * @returns the current Vector3
         */
        fromArray(array: FloatArray, index?: number): Vector4;
        /**
         * Adds the given vector to the current Vector4.
         * @param otherVector the vector to add
         * @returns the updated Vector4.
         */
        addInPlace(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Returns a new Vector4 as the result of the addition of the current Vector4 and the given one.
         * @param otherVector the vector to add
         * @returns the resulting vector
         */
        add(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the given vector "result" with the result of the addition of the current Vector4 and the given one.
         * @param otherVector the vector to add
         * @param result the vector to store the result
         * @returns the current Vector4.
         */
        addToRef(otherVector: DeepImmutable<Vector4>, result: Vector4): Vector4;
        /**
         * Subtract in place the given vector from the current Vector4.
         * @param otherVector the vector to subtract
         * @returns the updated Vector4.
         */
        subtractInPlace(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Returns a new Vector4 with the result of the subtraction of the given vector from the current Vector4.
         * @param otherVector the vector to add
         * @returns the new vector with the result
         */
        subtract(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Sets the given vector "result" with the result of the subtraction of the given vector from the current Vector4.
         * @param otherVector the vector to subtract
         * @param result the vector to store the result
         * @returns the current Vector4.
         */
        subtractToRef(otherVector: DeepImmutable<Vector4>, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the result of the subtraction of the given floats from the current Vector4 coordinates.
         */
        /**
         * Returns a new Vector4 set with the result of the subtraction of the given floats from the current Vector4 coordinates.
         * @param x value to subtract
         * @param y value to subtract
         * @param z value to subtract
         * @param w value to subtract
         * @returns new vector containing the result
         */
        subtractFromFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Sets the given vector "result" set with the result of the subtraction of the given floats from the current Vector4 coordinates.
         * @param x value to subtract
         * @param y value to subtract
         * @param z value to subtract
         * @param w value to subtract
         * @param result the vector to store the result in
         * @returns the current Vector4.
         */
        subtractFromFloatsToRef(x: number, y: number, z: number, w: number, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the current Vector4 negated coordinates.
         * @returns a new vector with the negated values
         */
        negate(): Vector4;
        /**
         * Negate this vector in place
         * @returns this
         */
        negateInPlace(): Vector4;
        /**
         * Negate the current Vector4 and stores the result in the given vector "result" coordinates
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector4
         */
        negateToRef(result: Vector4): Vector4;
        /**
         * Multiplies the current Vector4 coordinates by scale (float).
         * @param scale the number to scale with
         * @returns the updated Vector4.
         */
        scaleInPlace(scale: number): Vector4;
        /**
         * Returns a new Vector4 set with the current Vector4 coordinates multiplied by scale (float).
         * @param scale the number to scale with
         * @returns a new vector with the result
         */
        scale(scale: number): Vector4;
        /**
         * Sets the given vector "result" with the current Vector4 coordinates multiplied by scale (float).
         * @param scale the number to scale with
         * @param result a vector to store the result in
         * @returns the current Vector4.
         */
        scaleToRef(scale: number, result: Vector4): Vector4;
        /**
         * Scale the current Vector4 values by a factor and add the result to a given Vector4
         * @param scale defines the scale factor
         * @param result defines the Vector4 object where to store the result
         * @returns the unmodified current Vector4
         */
        scaleAndAddToRef(scale: number, result: Vector4): Vector4;
        /**
         * Boolean : True if the current Vector4 coordinates are stricly equal to the given ones.
         * @param otherVector the vector to compare against
         * @returns true if they are equal
         */
        equals(otherVector: DeepImmutable<Vector4>): boolean;
        /**
         * Boolean : True if the current Vector4 coordinates are each beneath the distance "epsilon" from the given vector ones.
         * @param otherVector vector to compare against
         * @param epsilon (Default: very small number)
         * @returns true if they are equal
         */
        equalsWithEpsilon(otherVector: DeepImmutable<Vector4>, epsilon?: number): boolean;
        /**
         * Boolean : True if the given floats are strictly equal to the current Vector4 coordinates.
         * @param x x value to compare against
         * @param y y value to compare against
         * @param z z value to compare against
         * @param w w value to compare against
         * @returns true if equal
         */
        equalsToFloats(x: number, y: number, z: number, w: number): boolean;
        /**
         * Multiplies in place the current Vector4 by the given one.
         * @param otherVector vector to multiple with
         * @returns the updated Vector4.
         */
        multiplyInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the multiplication result of the current Vector4 and the given one.
         * @param otherVector vector to multiple with
         * @returns resulting new vector
         */
        multiply(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the given vector "result" with the multiplication result of the current Vector4 and the given one.
         * @param otherVector vector to multiple with
         * @param result vector to store the result
         * @returns the current Vector4.
         */
        multiplyToRef(otherVector: DeepImmutable<Vector4>, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the multiplication result of the given floats and the current Vector4 coordinates.
         * @param x x value multiply with
         * @param y y value multiply with
         * @param z z value multiply with
         * @param w w value multiply with
         * @returns resulting new vector
         */
        multiplyByFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Returns a new Vector4 set with the division result of the current Vector4 by the given one.
         * @param otherVector vector to devide with
         * @returns resulting new vector
         */
        divide(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the given vector "result" with the division result of the current Vector4 by the given one.
         * @param otherVector vector to devide with
         * @param result vector to store the result
         * @returns the current Vector4.
         */
        divideToRef(otherVector: DeepImmutable<Vector4>, result: Vector4): Vector4;
        /**
         * Divides the current Vector3 coordinates by the given ones.
         * @param otherVector vector to devide with
         * @returns the updated Vector3.
         */
        divideInPlace(otherVector: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the Vector4 coordinates with the minimum values between its own and the given vector ones
         * @param other defines the second operand
         * @returns the current updated Vector4
         */
        minimizeInPlace(other: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the Vector4 coordinates with the maximum values between its own and the given vector ones
         * @param other defines the second operand
         * @returns the current updated Vector4
         */
        maximizeInPlace(other: DeepImmutable<Vector4>): Vector4;
        /**
         * Gets a new Vector4 from current Vector4 floored values
         * @returns a new Vector4
         */
        floor(): Vector4;
        /**
         * Gets a new Vector4 from current Vector3 floored values
         * @returns a new Vector4
         */
        fract(): Vector4;
        /**
         * Returns the Vector4 length (float).
         * @returns the length
         */
        length(): number;
        /**
         * Returns the Vector4 squared length (float).
         * @returns the length squared
         */
        lengthSquared(): number;
        /**
         * Normalizes in place the Vector4.
         * @returns the updated Vector4.
         */
        normalize(): Vector4;
        /**
         * Returns a new Vector3 from the Vector4 (x, y, z) coordinates.
         * @returns this converted to a new vector3
         */
        toVector3(): Vector3;
        /**
         * Returns a new Vector4 copied from the current one.
         * @returns the new cloned vector
         */
        clone(): Vector4;
        /**
         * Updates the current Vector4 with the given one coordinates.
         * @param source the source vector to copy from
         * @returns the updated Vector4.
         */
        copyFrom(source: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the current Vector4 coordinates with the given floats.
         * @param x float to copy from
         * @param y float to copy from
         * @param z float to copy from
         * @param w float to copy from
         * @returns the updated Vector4.
         */
        copyFromFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Updates the current Vector4 coordinates with the given floats.
         * @param x float to set from
         * @param y float to set from
         * @param z float to set from
         * @param w float to set from
         * @returns the updated Vector4.
         */
        set(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Copies the given float to the current Vector3 coordinates
         * @param v defines the x, y, z and w coordinates of the operand
         * @returns the current updated Vector3
         */
        setAll(v: number): Vector4;
        /**
         * Returns a new Vector4 set from the starting index of the given array.
         * @param array the array to pull values from
         * @param offset the offset into the array to start at
         * @returns the new vector
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Vector4;
        /**
         * Updates the given vector "result" from the starting index of the given array.
         * @param array the array to pull values from
         * @param offset the offset into the array to start at
         * @param result the vector to store the result in
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number, result: Vector4): void;
        /**
         * Updates the given vector "result" from the starting index of the given Float32Array.
         * @param array the array to pull values from
         * @param offset the offset into the array to start at
         * @param result the vector to store the result in
         */
        static FromFloatArrayToRef(array: DeepImmutable<Float32Array>, offset: number, result: Vector4): void;
        /**
         * Updates the given vector "result" coordinates from the given floats.
         * @param x float to set from
         * @param y float to set from
         * @param z float to set from
         * @param w float to set from
         * @param result the vector to the floats in
         */
        static FromFloatsToRef(x: number, y: number, z: number, w: number, result: Vector4): void;
        /**
         * Returns a new Vector4 set to (0.0, 0.0, 0.0, 0.0)
         * @returns the new vector
         */
        static Zero(): Vector4;
        /**
         * Returns a new Vector4 set to (1.0, 1.0, 1.0, 1.0)
         * @returns the new vector
         */
        static One(): Vector4;
        /**
         * Returns a new normalized Vector4 from the given one.
         * @param vector the vector to normalize
         * @returns the vector
         */
        static Normalize(vector: DeepImmutable<Vector4>): Vector4;
        /**
         * Updates the given vector "result" from the normalization of the given one.
         * @param vector the vector to normalize
         * @param result the vector to store the result in
         */
        static NormalizeToRef(vector: DeepImmutable<Vector4>, result: Vector4): void;
        /**
         * Returns a vector with the minimum values from the left and right vectors
         * @param left left vector to minimize
         * @param right right vector to minimize
         * @returns a new vector with the minimum of the left and right vector values
         */
        static Minimize(left: DeepImmutable<Vector4>, right: DeepImmutable<Vector4>): Vector4;
        /**
         * Returns a vector with the maximum values from the left and right vectors
         * @param left left vector to maximize
         * @param right right vector to maximize
         * @returns a new vector with the maximum of the left and right vector values
         */
        static Maximize(left: DeepImmutable<Vector4>, right: DeepImmutable<Vector4>): Vector4;
        /**
         * Returns the distance (float) between the vectors "value1" and "value2".
         * @param value1 value to calulate the distance between
         * @param value2 value to calulate the distance between
         * @return the distance between the two vectors
         */
        static Distance(value1: DeepImmutable<Vector4>, value2: DeepImmutable<Vector4>): number;
        /**
         * Returns the squared distance (float) between the vectors "value1" and "value2".
         * @param value1 value to calulate the distance between
         * @param value2 value to calulate the distance between
         * @return the distance between the two vectors squared
         */
        static DistanceSquared(value1: DeepImmutable<Vector4>, value2: DeepImmutable<Vector4>): number;
        /**
         * Returns a new Vector4 located at the center between the vectors "value1" and "value2".
         * @param value1 value to calulate the center between
         * @param value2 value to calulate the center between
         * @return the center between the two vectors
         */
        static Center(value1: DeepImmutable<Vector4>, value2: DeepImmutable<Vector4>): Vector4;
        /**
         * Returns a new Vector4 set with the result of the normal transformation by the given matrix of the given vector.
         * This methods computes transformed normalized direction vectors only.
         * @param vector the vector to transform
         * @param transformation the transformation matrix to apply
         * @returns the new vector
         */
        static TransformNormal(vector: DeepImmutable<Vector4>, transformation: DeepImmutable<Matrix>): Vector4;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given vector.
         * This methods computes transformed normalized direction vectors only.
         * @param vector the vector to transform
         * @param transformation the transformation matrix to apply
         * @param result the vector to store the result in
         */
        static TransformNormalToRef(vector: DeepImmutable<Vector4>, transformation: DeepImmutable<Matrix>, result: Vector4): void;
        /**
         * Sets the given vector "result" with the result of the normal transformation by the given matrix of the given floats (x, y, z, w).
         * This methods computes transformed normalized direction vectors only.
         * @param x value to transform
         * @param y value to transform
         * @param z value to transform
         * @param w value to transform
         * @param transformation the transformation matrix to apply
         * @param result the vector to store the results in
         */
        static TransformNormalFromFloatsToRef(x: number, y: number, z: number, w: number, transformation: DeepImmutable<Matrix>, result: Vector4): void;
        /**
         * Creates a new Vector4 from a Vector3
         * @param source defines the source data
         * @param w defines the 4th component (default is 0)
         * @returns a new Vector4
         */
        static FromVector3(source: Vector3, w?: number): Vector4;
    }
    /**
     * Class used to store quaternion data
     * @see https://en.wikipedia.org/wiki/Quaternion
     * @see https://doc.babylonjs.com/features/position,_rotation,_scaling
     */
    export class Quaternion {
        /** @hidden */
        _x: number;
        /** @hidden */
        _y: number;
        /** @hidden */
        _z: number;
        /** @hidden */
        _w: number;
        /** @hidden */
        _isDirty: boolean;
        /** Gets or sets the x coordinate */
        get x(): number;
        set x(value: number);
        /** Gets or sets the y coordinate */
        get y(): number;
        set y(value: number);
        /** Gets or sets the z coordinate */
        get z(): number;
        set z(value: number);
        /** Gets or sets the w coordinate */
        get w(): number;
        set w(value: number);
        /**
         * Creates a new Quaternion from the given floats
         * @param x defines the first component (0 by default)
         * @param y defines the second component (0 by default)
         * @param z defines the third component (0 by default)
         * @param w defines the fourth component (1.0 by default)
         */
        constructor(x?: number, y?: number, z?: number, w?: number);
        /**
         * Gets a string representation for the current quaternion
         * @returns a string with the Quaternion coordinates
         */
        toString(): string;
        /**
         * Gets the class name of the quaternion
         * @returns the string "Quaternion"
         */
        getClassName(): string;
        /**
         * Gets a hash code for this quaternion
         * @returns the quaternion hash code
         */
        getHashCode(): number;
        /**
         * Copy the quaternion to an array
         * @returns a new array populated with 4 elements from the quaternion coordinates
         */
        asArray(): number[];
        /**
         * Check if two quaternions are equals
         * @param otherQuaternion defines the second operand
         * @return true if the current quaternion and the given one coordinates are strictly equals
         */
        equals(otherQuaternion: DeepImmutable<Quaternion>): boolean;
        /**
         * Gets a boolean if two quaternions are equals (using an epsilon value)
         * @param otherQuaternion defines the other quaternion
         * @param epsilon defines the minimal distance to consider equality
         * @returns true if the given quaternion coordinates are close to the current ones by a distance of epsilon.
         */
        equalsWithEpsilon(otherQuaternion: DeepImmutable<Quaternion>, epsilon?: number): boolean;
        /**
         * Clone the current quaternion
         * @returns a new quaternion copied from the current one
         */
        clone(): Quaternion;
        /**
         * Copy a quaternion to the current one
         * @param other defines the other quaternion
         * @returns the updated current quaternion
         */
        copyFrom(other: DeepImmutable<Quaternion>): Quaternion;
        /**
         * Updates the current quaternion with the given float coordinates
         * @param x defines the x coordinate
         * @param y defines the y coordinate
         * @param z defines the z coordinate
         * @param w defines the w coordinate
         * @returns the updated current quaternion
         */
        copyFromFloats(x: number, y: number, z: number, w: number): Quaternion;
        /**
         * Updates the current quaternion from the given float coordinates
         * @param x defines the x coordinate
         * @param y defines the y coordinate
         * @param z defines the z coordinate
         * @param w defines the w coordinate
         * @returns the updated current quaternion
         */
        set(x: number, y: number, z: number, w: number): Quaternion;
        /**
         * Adds two quaternions
         * @param other defines the second operand
         * @returns a new quaternion as the addition result of the given one and the current quaternion
         */
        add(other: DeepImmutable<Quaternion>): Quaternion;
        /**
         * Add a quaternion to the current one
         * @param other defines the quaternion to add
         * @returns the current quaternion
         */
        addInPlace(other: DeepImmutable<Quaternion>): Quaternion;
        /**
         * Subtract two quaternions
         * @param other defines the second operand
         * @returns a new quaternion as the subtraction result of the given one from the current one
         */
        subtract(other: Quaternion): Quaternion;
        /**
         * Multiplies the current quaternion by a scale factor
         * @param value defines the scale factor
         * @returns a new quaternion set by multiplying the current quaternion coordinates by the float "scale"
         */
        scale(value: number): Quaternion;
        /**
         * Scale the current quaternion values by a factor and stores the result to a given quaternion
         * @param scale defines the scale factor
         * @param result defines the Quaternion object where to store the result
         * @returns the unmodified current quaternion
         */
        scaleToRef(scale: number, result: Quaternion): Quaternion;
        /**
         * Multiplies in place the current quaternion by a scale factor
         * @param value defines the scale factor
         * @returns the current modified quaternion
         */
        scaleInPlace(value: number): Quaternion;
        /**
         * Scale the current quaternion values by a factor and add the result to a given quaternion
         * @param scale defines the scale factor
         * @param result defines the Quaternion object where to store the result
         * @returns the unmodified current quaternion
         */
        scaleAndAddToRef(scale: number, result: Quaternion): Quaternion;
        /**
         * Multiplies two quaternions
         * @param q1 defines the second operand
         * @returns a new quaternion set as the multiplication result of the current one with the given one "q1"
         */
        multiply(q1: DeepImmutable<Quaternion>): Quaternion;
        /**
         * Sets the given "result" as the the multiplication result of the current one with the given one "q1"
         * @param q1 defines the second operand
         * @param result defines the target quaternion
         * @returns the current quaternion
         */
        multiplyToRef(q1: DeepImmutable<Quaternion>, result: Quaternion): Quaternion;
        /**
         * Updates the current quaternion with the multiplication of itself with the given one "q1"
         * @param q1 defines the second operand
         * @returns the currentupdated quaternion
         */
        multiplyInPlace(q1: DeepImmutable<Quaternion>): Quaternion;
        /**
         * Conjugates (1-q) the current quaternion and stores the result in the given quaternion
         * @param ref defines the target quaternion
         * @returns the current quaternion
         */
        conjugateToRef(ref: Quaternion): Quaternion;
        /**
         * Conjugates in place (1-q) the current quaternion
         * @returns the current updated quaternion
         */
        conjugateInPlace(): Quaternion;
        /**
         * Conjugates in place (1-q) the current quaternion
         * @returns a new quaternion
         */
        conjugate(): Quaternion;
        /**
         * Gets length of current quaternion
         * @returns the quaternion length (float)
         */
        length(): number;
        /**
         * Normalize in place the current quaternion
         * @returns the current updated quaternion
         */
        normalize(): Quaternion;
        /**
         * Returns a new Vector3 set with the Euler angles translated from the current quaternion
         * @param order is a reserved parameter and is ignored for now
         * @returns a new Vector3 containing the Euler angles
         */
        toEulerAngles(order?: string): Vector3;
        /**
         * Sets the given vector3 "result" with the Euler angles translated from the current quaternion
         * @param result defines the vector which will be filled with the Euler angles
         * @returns the current unchanged quaternion
         */
        toEulerAnglesToRef(result: Vector3): Quaternion;
        /**
         * Updates the given rotation matrix with the current quaternion values
         * @param result defines the target matrix
         * @returns the current unchanged quaternion
         */
        toRotationMatrix(result: Matrix): Quaternion;
        /**
         * Updates the current quaternion from the given rotation matrix values
         * @param matrix defines the source matrix
         * @returns the current updated quaternion
         */
        fromRotationMatrix(matrix: DeepImmutable<Matrix>): Quaternion;
        /**
         * Creates a new quaternion from a rotation matrix
         * @param matrix defines the source matrix
         * @returns a new quaternion created from the given rotation matrix values
         */
        static FromRotationMatrix(matrix: DeepImmutable<Matrix>): Quaternion;
        /**
         * Updates the given quaternion with the given rotation matrix values
         * @param matrix defines the source matrix
         * @param result defines the target quaternion
         */
        static FromRotationMatrixToRef(matrix: DeepImmutable<Matrix>, result: Quaternion): void;
        /**
         * Returns the dot product (float) between the quaternions "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the dot product
         */
        static Dot(left: DeepImmutable<Quaternion>, right: DeepImmutable<Quaternion>): number;
        /**
         * Checks if the two quaternions are close to each other
         * @param quat0 defines the first quaternion to check
         * @param quat1 defines the second quaternion to check
         * @returns true if the two quaternions are close to each other
         */
        static AreClose(quat0: DeepImmutable<Quaternion>, quat1: DeepImmutable<Quaternion>): boolean;
        /**
         * Creates an empty quaternion
         * @returns a new quaternion set to (0.0, 0.0, 0.0)
         */
        static Zero(): Quaternion;
        /**
         * Inverse a given quaternion
         * @param q defines the source quaternion
         * @returns a new quaternion as the inverted current quaternion
         */
        static Inverse(q: DeepImmutable<Quaternion>): Quaternion;
        /**
         * Inverse a given quaternion
         * @param q defines the source quaternion
         * @param result the quaternion the result will be stored in
         * @returns the result quaternion
         */
        static InverseToRef(q: Quaternion, result: Quaternion): Quaternion;
        /**
         * Creates an identity quaternion
         * @returns the identity quaternion
         */
        static Identity(): Quaternion;
        /**
         * Gets a boolean indicating if the given quaternion is identity
         * @param quaternion defines the quaternion to check
         * @returns true if the quaternion is identity
         */
        static IsIdentity(quaternion: DeepImmutable<Quaternion>): boolean;
        /**
         * Creates a quaternion from a rotation around an axis
         * @param axis defines the axis to use
         * @param angle defines the angle to use
         * @returns a new quaternion created from the given axis (Vector3) and angle in radians (float)
         */
        static RotationAxis(axis: DeepImmutable<Vector3>, angle: number): Quaternion;
        /**
         * Creates a rotation around an axis and stores it into the given quaternion
         * @param axis defines the axis to use
         * @param angle defines the angle to use
         * @param result defines the target quaternion
         * @returns the target quaternion
         */
        static RotationAxisToRef(axis: DeepImmutable<Vector3>, angle: number, result: Quaternion): Quaternion;
        /**
         * Creates a new quaternion from data stored into an array
         * @param array defines the data source
         * @param offset defines the offset in the source array where the data starts
         * @returns a new quaternion
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Quaternion;
        /**
         * Updates the given quaternion "result" from the starting index of the given array.
         * @param array the array to pull values from
         * @param offset the offset into the array to start at
         * @param result the quaternion to store the result in
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number, result: Quaternion): void;
        /**
         * Create a quaternion from Euler rotation angles
         * @param x Pitch
         * @param y Yaw
         * @param z Roll
         * @returns the new Quaternion
         */
        static FromEulerAngles(x: number, y: number, z: number): Quaternion;
        /**
         * Updates a quaternion from Euler rotation angles
         * @param x Pitch
         * @param y Yaw
         * @param z Roll
         * @param result the quaternion to store the result
         * @returns the updated quaternion
         */
        static FromEulerAnglesToRef(x: number, y: number, z: number, result: Quaternion): Quaternion;
        /**
         * Create a quaternion from Euler rotation vector
         * @param vec the Euler vector (x Pitch, y Yaw, z Roll)
         * @returns the new Quaternion
         */
        static FromEulerVector(vec: DeepImmutable<Vector3>): Quaternion;
        /**
         * Updates a quaternion from Euler rotation vector
         * @param vec the Euler vector (x Pitch, y Yaw, z Roll)
         * @param result the quaternion to store the result
         * @returns the updated quaternion
         */
        static FromEulerVectorToRef(vec: DeepImmutable<Vector3>, result: Quaternion): Quaternion;
        /**
         * Creates a new quaternion from the given Euler float angles (y, x, z)
         * @param yaw defines the rotation around Y axis
         * @param pitch defines the rotation around X axis
         * @param roll defines the rotation around Z axis
         * @returns the new quaternion
         */
        static RotationYawPitchRoll(yaw: number, pitch: number, roll: number): Quaternion;
        /**
         * Creates a new rotation from the given Euler float angles (y, x, z) and stores it in the target quaternion
         * @param yaw defines the rotation around Y axis
         * @param pitch defines the rotation around X axis
         * @param roll defines the rotation around Z axis
         * @param result defines the target quaternion
         */
        static RotationYawPitchRollToRef(yaw: number, pitch: number, roll: number, result: Quaternion): void;
        /**
         * Creates a new quaternion from the given Euler float angles expressed in z-x-z orientation
         * @param alpha defines the rotation around first axis
         * @param beta defines the rotation around second axis
         * @param gamma defines the rotation around third axis
         * @returns the new quaternion
         */
        static RotationAlphaBetaGamma(alpha: number, beta: number, gamma: number): Quaternion;
        /**
         * Creates a new quaternion from the given Euler float angles expressed in z-x-z orientation and stores it in the target quaternion
         * @param alpha defines the rotation around first axis
         * @param beta defines the rotation around second axis
         * @param gamma defines the rotation around third axis
         * @param result defines the target quaternion
         */
        static RotationAlphaBetaGammaToRef(alpha: number, beta: number, gamma: number, result: Quaternion): void;
        /**
         * Creates a new quaternion containing the rotation value to reach the target (axis1, axis2, axis3) orientation as a rotated XYZ system (axis1, axis2 and axis3 are normalized during this operation)
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @returns the new quaternion
         */
        static RotationQuaternionFromAxis(axis1: DeepImmutable<Vector3>, axis2: DeepImmutable<Vector3>, axis3: DeepImmutable<Vector3>): Quaternion;
        /**
         * Creates a rotation value to reach the target (axis1, axis2, axis3) orientation as a rotated XYZ system (axis1, axis2 and axis3 are normalized during this operation) and stores it in the target quaternion
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @param ref defines the target quaternion
         */
        static RotationQuaternionFromAxisToRef(axis1: DeepImmutable<Vector3>, axis2: DeepImmutable<Vector3>, axis3: DeepImmutable<Vector3>, ref: Quaternion): void;
        /**
         * Interpolates between two quaternions
         * @param left defines first quaternion
         * @param right defines second quaternion
         * @param amount defines the gradient to use
         * @returns the new interpolated quaternion
         */
        static Slerp(left: DeepImmutable<Quaternion>, right: DeepImmutable<Quaternion>, amount: number): Quaternion;
        /**
         * Interpolates between two quaternions and stores it into a target quaternion
         * @param left defines first quaternion
         * @param right defines second quaternion
         * @param amount defines the gradient to use
         * @param result defines the target quaternion
         */
        static SlerpToRef(left: DeepImmutable<Quaternion>, right: DeepImmutable<Quaternion>, amount: number, result: Quaternion): void;
        /**
         * Interpolate between two quaternions using Hermite interpolation
         * @param value1 defines first quaternion
         * @param tangent1 defines the incoming tangent
         * @param value2 defines second quaternion
         * @param tangent2 defines the outgoing tangent
         * @param amount defines the target quaternion
         * @returns the new interpolated quaternion
         */
        static Hermite(value1: DeepImmutable<Quaternion>, tangent1: DeepImmutable<Quaternion>, value2: DeepImmutable<Quaternion>, tangent2: DeepImmutable<Quaternion>, amount: number): Quaternion;
    }
    /**
     * Class used to store matrix data (4x4)
     */
    export class Matrix {
        /**
         * Gets the precision of matrix computations
         */
        static get Use64Bits(): boolean;
        private static _updateFlagSeed;
        private static _identityReadOnly;
        private _isIdentity;
        private _isIdentityDirty;
        private _isIdentity3x2;
        private _isIdentity3x2Dirty;
        /**
         * Gets the update flag of the matrix which is an unique number for the matrix.
         * It will be incremented every time the matrix data change.
         * You can use it to speed the comparison between two versions of the same matrix.
         */
        updateFlag: number;
        private readonly _m;
        /**
         * Gets the internal data of the matrix
         */
        get m(): DeepImmutable<Float32Array | Array<number>>;
        /** @hidden */
        _markAsUpdated(): void;
        /** @hidden */
        private _updateIdentityStatus;
        /**
         * Creates an empty matrix (filled with zeros)
         */
        constructor();
        /**
         * Check if the current matrix is identity
         * @returns true is the matrix is the identity matrix
         */
        isIdentity(): boolean;
        /**
         * Check if the current matrix is identity as a texture matrix (3x2 store in 4x4)
         * @returns true is the matrix is the identity matrix
         */
        isIdentityAs3x2(): boolean;
        /**
         * Gets the determinant of the matrix
         * @returns the matrix determinant
         */
        determinant(): number;
        /**
         * Returns the matrix as a Float32Array or Array<number>
         * @returns the matrix underlying array
         */
        toArray(): DeepImmutable<Float32Array | Array<number>>;
        /**
         * Returns the matrix as a Float32Array or Array<number>
        * @returns the matrix underlying array.
        */
        asArray(): DeepImmutable<Float32Array | Array<number>>;
        /**
         * Inverts the current matrix in place
         * @returns the current inverted matrix
         */
        invert(): Matrix;
        /**
         * Sets all the matrix elements to zero
         * @returns the current matrix
         */
        reset(): Matrix;
        /**
         * Adds the current matrix with a second one
         * @param other defines the matrix to add
         * @returns a new matrix as the addition of the current matrix and the given one
         */
        add(other: DeepImmutable<Matrix>): Matrix;
        /**
         * Sets the given matrix "result" to the addition of the current matrix and the given one
         * @param other defines the matrix to add
         * @param result defines the target matrix
         * @returns the current matrix
         */
        addToRef(other: DeepImmutable<Matrix>, result: Matrix): Matrix;
        /**
         * Adds in place the given matrix to the current matrix
         * @param other defines the second operand
         * @returns the current updated matrix
         */
        addToSelf(other: DeepImmutable<Matrix>): Matrix;
        /**
         * Sets the given matrix to the current inverted Matrix
         * @param other defines the target matrix
         * @returns the unmodified current matrix
         */
        invertToRef(other: Matrix): Matrix;
        /**
         * add a value at the specified position in the current Matrix
         * @param index the index of the value within the matrix. between 0 and 15.
         * @param value the value to be added
         * @returns the current updated matrix
         */
        addAtIndex(index: number, value: number): Matrix;
        /**
         * mutiply the specified position in the current Matrix by a value
         * @param index the index of the value within the matrix. between 0 and 15.
         * @param value the value to be added
         * @returns the current updated matrix
         */
        multiplyAtIndex(index: number, value: number): Matrix;
        /**
         * Inserts the translation vector (using 3 floats) in the current matrix
         * @param x defines the 1st component of the translation
         * @param y defines the 2nd component of the translation
         * @param z defines the 3rd component of the translation
         * @returns the current updated matrix
         */
        setTranslationFromFloats(x: number, y: number, z: number): Matrix;
        /**
         * Adds the translation vector (using 3 floats) in the current matrix
         * @param x defines the 1st component of the translation
         * @param y defines the 2nd component of the translation
         * @param z defines the 3rd component of the translation
         * @returns the current updated matrix
         */
        addTranslationFromFloats(x: number, y: number, z: number): Matrix;
        /**
         * Inserts the translation vector in the current matrix
         * @param vector3 defines the translation to insert
         * @returns the current updated matrix
         */
        setTranslation(vector3: DeepImmutable<Vector3>): Matrix;
        /**
         * Gets the translation value of the current matrix
         * @returns a new Vector3 as the extracted translation from the matrix
         */
        getTranslation(): Vector3;
        /**
         * Fill a Vector3 with the extracted translation from the matrix
         * @param result defines the Vector3 where to store the translation
         * @returns the current matrix
         */
        getTranslationToRef(result: Vector3): Matrix;
        /**
         * Remove rotation and scaling part from the matrix
         * @returns the updated matrix
         */
        removeRotationAndScaling(): Matrix;
        /**
         * Multiply two matrices
         * @param other defines the second operand
         * @returns a new matrix set with the multiplication result of the current Matrix and the given one
         */
        multiply(other: DeepImmutable<Matrix>): Matrix;
        /**
         * Copy the current matrix from the given one
         * @param other defines the source matrix
         * @returns the current updated matrix
         */
        copyFrom(other: DeepImmutable<Matrix>): Matrix;
        /**
         * Populates the given array from the starting index with the current matrix values
         * @param array defines the target array
         * @param offset defines the offset in the target array where to start storing values
         * @returns the current matrix
         */
        copyToArray(array: Float32Array | Array<number>, offset?: number): Matrix;
        /**
         * Sets the given matrix "result" with the multiplication result of the current Matrix and the given one
         * @param other defines the second operand
         * @param result defines the matrix where to store the multiplication
         * @returns the current matrix
         */
        multiplyToRef(other: DeepImmutable<Matrix>, result: Matrix): Matrix;
        /**
         * Sets the Float32Array "result" from the given index "offset" with the multiplication of the current matrix and the given one
         * @param other defines the second operand
         * @param result defines the array where to store the multiplication
         * @param offset defines the offset in the target array where to start storing values
         * @returns the current matrix
         */
        multiplyToArray(other: DeepImmutable<Matrix>, result: Float32Array | Array<number>, offset: number): Matrix;
        /**
         * Check equality between this matrix and a second one
         * @param value defines the second matrix to compare
         * @returns true is the current matrix and the given one values are strictly equal
         */
        equals(value: DeepImmutable<Matrix>): boolean;
        /**
         * Clone the current matrix
         * @returns a new matrix from the current matrix
         */
        clone(): Matrix;
        /**
         * Returns the name of the current matrix class
         * @returns the string "Matrix"
         */
        getClassName(): string;
        /**
         * Gets the hash code of the current matrix
         * @returns the hash code
         */
        getHashCode(): number;
        /**
         * Decomposes the current Matrix into a translation, rotation and scaling components
         * @param scale defines the scale vector3 given as a reference to update
         * @param rotation defines the rotation quaternion given as a reference to update
         * @param translation defines the translation vector3 given as a reference to update
         * @returns true if operation was successful
         */
        decompose(scale?: Vector3, rotation?: Quaternion, translation?: Vector3): boolean;
        /**
         * Gets specific row of the matrix
         * @param index defines the number of the row to get
         * @returns the index-th row of the current matrix as a new Vector4
         */
        getRow(index: number): Nullable<Vector4>;
        /**
         * Sets the index-th row of the current matrix to the vector4 values
         * @param index defines the number of the row to set
         * @param row defines the target vector4
         * @returns the updated current matrix
         */
        setRow(index: number, row: Vector4): Matrix;
        /**
         * Compute the transpose of the matrix
         * @returns the new transposed matrix
         */
        transpose(): Matrix;
        /**
         * Compute the transpose of the matrix and store it in a given matrix
         * @param result defines the target matrix
         * @returns the current matrix
         */
        transposeToRef(result: Matrix): Matrix;
        /**
         * Sets the index-th row of the current matrix with the given 4 x float values
         * @param index defines the row index
         * @param x defines the x component to set
         * @param y defines the y component to set
         * @param z defines the z component to set
         * @param w defines the w component to set
         * @returns the updated current matrix
         */
        setRowFromFloats(index: number, x: number, y: number, z: number, w: number): Matrix;
        /**
         * Compute a new matrix set with the current matrix values multiplied by scale (float)
         * @param scale defines the scale factor
         * @returns a new matrix
         */
        scale(scale: number): Matrix;
        /**
         * Scale the current matrix values by a factor to a given result matrix
         * @param scale defines the scale factor
         * @param result defines the matrix to store the result
         * @returns the current matrix
         */
        scaleToRef(scale: number, result: Matrix): Matrix;
        /**
         * Scale the current matrix values by a factor and add the result to a given matrix
         * @param scale defines the scale factor
         * @param result defines the Matrix to store the result
         * @returns the current matrix
         */
        scaleAndAddToRef(scale: number, result: Matrix): Matrix;
        /**
         * Writes to the given matrix a normal matrix, computed from this one (using values from identity matrix for fourth row and column).
         * @param ref matrix to store the result
         */
        toNormalMatrix(ref: Matrix): void;
        /**
         * Gets only rotation part of the current matrix
         * @returns a new matrix sets to the extracted rotation matrix from the current one
         */
        getRotationMatrix(): Matrix;
        /**
         * Extracts the rotation matrix from the current one and sets it as the given "result"
         * @param result defines the target matrix to store data to
         * @returns the current matrix
         */
        getRotationMatrixToRef(result: Matrix): Matrix;
        /**
         * Toggles model matrix from being right handed to left handed in place and vice versa
         */
        toggleModelMatrixHandInPlace(): void;
        /**
         * Toggles projection matrix from being right handed to left handed in place and vice versa
         */
        toggleProjectionMatrixHandInPlace(): void;
        /**
         * Creates a matrix from an array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns a new Matrix set from the starting index of the given array
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Matrix;
        /**
         * Copy the content of an array into a given matrix
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @param result defines the target matrix
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number, result: Matrix): void;
        /**
         * Stores an array into a matrix after having multiplied each component by a given factor
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param scale defines the scaling factor
         * @param result defines the target matrix
         */
        static FromFloat32ArrayToRefScaled(array: DeepImmutable<Float32Array | Array<number>>, offset: number, scale: number, result: Matrix): void;
        /**
         * Gets an identity matrix that must not be updated
         */
        static get IdentityReadOnly(): DeepImmutable<Matrix>;
        /**
         * Stores a list of values (16) inside a given matrix
         * @param initialM11 defines 1st value of 1st row
         * @param initialM12 defines 2nd value of 1st row
         * @param initialM13 defines 3rd value of 1st row
         * @param initialM14 defines 4th value of 1st row
         * @param initialM21 defines 1st value of 2nd row
         * @param initialM22 defines 2nd value of 2nd row
         * @param initialM23 defines 3rd value of 2nd row
         * @param initialM24 defines 4th value of 2nd row
         * @param initialM31 defines 1st value of 3rd row
         * @param initialM32 defines 2nd value of 3rd row
         * @param initialM33 defines 3rd value of 3rd row
         * @param initialM34 defines 4th value of 3rd row
         * @param initialM41 defines 1st value of 4th row
         * @param initialM42 defines 2nd value of 4th row
         * @param initialM43 defines 3rd value of 4th row
         * @param initialM44 defines 4th value of 4th row
         * @param result defines the target matrix
         */
        static FromValuesToRef(initialM11: number, initialM12: number, initialM13: number, initialM14: number, initialM21: number, initialM22: number, initialM23: number, initialM24: number, initialM31: number, initialM32: number, initialM33: number, initialM34: number, initialM41: number, initialM42: number, initialM43: number, initialM44: number, result: Matrix): void;
        /**
         * Creates new matrix from a list of values (16)
         * @param initialM11 defines 1st value of 1st row
         * @param initialM12 defines 2nd value of 1st row
         * @param initialM13 defines 3rd value of 1st row
         * @param initialM14 defines 4th value of 1st row
         * @param initialM21 defines 1st value of 2nd row
         * @param initialM22 defines 2nd value of 2nd row
         * @param initialM23 defines 3rd value of 2nd row
         * @param initialM24 defines 4th value of 2nd row
         * @param initialM31 defines 1st value of 3rd row
         * @param initialM32 defines 2nd value of 3rd row
         * @param initialM33 defines 3rd value of 3rd row
         * @param initialM34 defines 4th value of 3rd row
         * @param initialM41 defines 1st value of 4th row
         * @param initialM42 defines 2nd value of 4th row
         * @param initialM43 defines 3rd value of 4th row
         * @param initialM44 defines 4th value of 4th row
         * @returns the new matrix
         */
        static FromValues(initialM11: number, initialM12: number, initialM13: number, initialM14: number, initialM21: number, initialM22: number, initialM23: number, initialM24: number, initialM31: number, initialM32: number, initialM33: number, initialM34: number, initialM41: number, initialM42: number, initialM43: number, initialM44: number): Matrix;
        /**
         * Creates a new matrix composed by merging scale (vector3), rotation (quaternion) and translation (vector3)
         * @param scale defines the scale vector3
         * @param rotation defines the rotation quaternion
         * @param translation defines the translation vector3
         * @returns a new matrix
         */
        static Compose(scale: DeepImmutable<Vector3>, rotation: DeepImmutable<Quaternion>, translation: DeepImmutable<Vector3>): Matrix;
        /**
         * Sets a matrix to a value composed by merging scale (vector3), rotation (quaternion) and translation (vector3)
         * @param scale defines the scale vector3
         * @param rotation defines the rotation quaternion
         * @param translation defines the translation vector3
         * @param result defines the target matrix
         */
        static ComposeToRef(scale: DeepImmutable<Vector3>, rotation: DeepImmutable<Quaternion>, translation: DeepImmutable<Vector3>, result: Matrix): void;
        /**
         * Creates a new identity matrix
         * @returns a new identity matrix
         */
        static Identity(): Matrix;
        /**
         * Creates a new identity matrix and stores the result in a given matrix
         * @param result defines the target matrix
         */
        static IdentityToRef(result: Matrix): void;
        /**
         * Creates a new zero matrix
         * @returns a new zero matrix
         */
        static Zero(): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the X axis
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationX(angle: number): Matrix;
        /**
         * Creates a new matrix as the invert of a given matrix
         * @param source defines the source matrix
         * @returns the new matrix
         */
        static Invert(source: DeepImmutable<Matrix>): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the X axis and stores it in a given matrix
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationXToRef(angle: number, result: Matrix): void;
        /**
         * Creates a new rotation matrix for "angle" radians around the Y axis
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationY(angle: number): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the Y axis and stores it in a given matrix
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationYToRef(angle: number, result: Matrix): void;
        /**
         * Creates a new rotation matrix for "angle" radians around the Z axis
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationZ(angle: number): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the Z axis and stores it in a given matrix
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationZToRef(angle: number, result: Matrix): void;
        /**
         * Creates a new rotation matrix for "angle" radians around the given axis
         * @param axis defines the axis to use
         * @param angle defines the angle (in radians) to use
         * @return the new matrix
         */
        static RotationAxis(axis: DeepImmutable<Vector3>, angle: number): Matrix;
        /**
         * Creates a new rotation matrix for "angle" radians around the given axis and stores it in a given matrix
         * @param axis defines the axis to use
         * @param angle defines the angle (in radians) to use
         * @param result defines the target matrix
         */
        static RotationAxisToRef(axis: DeepImmutable<Vector3>, angle: number, result: Matrix): void;
        /**
         * Takes normalised vectors and returns a rotation matrix to align "from" with "to".
         * Taken from http://www.iquilezles.org/www/articles/noacos/noacos.htm
         * @param from defines the vector to align
         * @param to defines the vector to align to
         * @param result defines the target matrix
         */
        static RotationAlignToRef(from: DeepImmutable<Vector3>, to: DeepImmutable<Vector3>, result: Matrix): void;
        /**
         * Creates a rotation matrix
         * @param yaw defines the yaw angle in radians (Y axis)
         * @param pitch defines the pitch angle in radians (X axis)
         * @param roll defines the roll angle in radians (Z axis)
         * @returns the new rotation matrix
         */
        static RotationYawPitchRoll(yaw: number, pitch: number, roll: number): Matrix;
        /**
         * Creates a rotation matrix and stores it in a given matrix
         * @param yaw defines the yaw angle in radians (Y axis)
         * @param pitch defines the pitch angle in radians (X axis)
         * @param roll defines the roll angle in radians (Z axis)
         * @param result defines the target matrix
         */
        static RotationYawPitchRollToRef(yaw: number, pitch: number, roll: number, result: Matrix): void;
        /**
         * Creates a scaling matrix
         * @param x defines the scale factor on X axis
         * @param y defines the scale factor on Y axis
         * @param z defines the scale factor on Z axis
         * @returns the new matrix
         */
        static Scaling(x: number, y: number, z: number): Matrix;
        /**
         * Creates a scaling matrix and stores it in a given matrix
         * @param x defines the scale factor on X axis
         * @param y defines the scale factor on Y axis
         * @param z defines the scale factor on Z axis
         * @param result defines the target matrix
         */
        static ScalingToRef(x: number, y: number, z: number, result: Matrix): void;
        /**
         * Creates a translation matrix
         * @param x defines the translation on X axis
         * @param y defines the translation on Y axis
         * @param z defines the translationon Z axis
         * @returns the new matrix
         */
        static Translation(x: number, y: number, z: number): Matrix;
        /**
         * Creates a translation matrix and stores it in a given matrix
         * @param x defines the translation on X axis
         * @param y defines the translation on Y axis
         * @param z defines the translationon Z axis
         * @param result defines the target matrix
         */
        static TranslationToRef(x: number, y: number, z: number, result: Matrix): void;
        /**
         * Returns a new Matrix whose values are the interpolated values for "gradient" (float) between the ones of the matrices "startValue" and "endValue".
         * @param startValue defines the start value
         * @param endValue defines the end value
         * @param gradient defines the gradient factor
         * @returns the new matrix
         */
        static Lerp(startValue: DeepImmutable<Matrix>, endValue: DeepImmutable<Matrix>, gradient: number): Matrix;
        /**
         * Set the given matrix "result" as the interpolated values for "gradient" (float) between the ones of the matrices "startValue" and "endValue".
         * @param startValue defines the start value
         * @param endValue defines the end value
         * @param gradient defines the gradient factor
         * @param result defines the Matrix object where to store data
         */
        static LerpToRef(startValue: DeepImmutable<Matrix>, endValue: DeepImmutable<Matrix>, gradient: number, result: Matrix): void;
        /**
         * Builds a new matrix whose values are computed by:
         * * decomposing the the "startValue" and "endValue" matrices into their respective scale, rotation and translation matrices
         * * interpolating for "gradient" (float) the values between each of these decomposed matrices between the start and the end
         * * recomposing a new matrix from these 3 interpolated scale, rotation and translation matrices
         * @param startValue defines the first matrix
         * @param endValue defines the second matrix
         * @param gradient defines the gradient between the two matrices
         * @returns the new matrix
         */
        static DecomposeLerp(startValue: DeepImmutable<Matrix>, endValue: DeepImmutable<Matrix>, gradient: number): Matrix;
        /**
         * Update a matrix to values which are computed by:
         * * decomposing the the "startValue" and "endValue" matrices into their respective scale, rotation and translation matrices
         * * interpolating for "gradient" (float) the values between each of these decomposed matrices between the start and the end
         * * recomposing a new matrix from these 3 interpolated scale, rotation and translation matrices
         * @param startValue defines the first matrix
         * @param endValue defines the second matrix
         * @param gradient defines the gradient between the two matrices
         * @param result defines the target matrix
         */
        static DecomposeLerpToRef(startValue: DeepImmutable<Matrix>, endValue: DeepImmutable<Matrix>, gradient: number, result: Matrix): void;
        /**
         * Gets a new rotation matrix used to rotate an entity so as it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up"
         * This function works in left handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @returns the new matrix
         */
        static LookAtLH(eye: DeepImmutable<Vector3>, target: DeepImmutable<Vector3>, up: DeepImmutable<Vector3>): Matrix;
        /**
         * Sets the given "result" Matrix to a rotation matrix used to rotate an entity so that it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up".
         * This function works in left handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @param result defines the target matrix
         */
        static LookAtLHToRef(eye: DeepImmutable<Vector3>, target: DeepImmutable<Vector3>, up: DeepImmutable<Vector3>, result: Matrix): void;
        /**
         * Gets a new rotation matrix used to rotate an entity so as it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up"
         * This function works in right handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @returns the new matrix
         */
        static LookAtRH(eye: DeepImmutable<Vector3>, target: DeepImmutable<Vector3>, up: DeepImmutable<Vector3>): Matrix;
        /**
         * Sets the given "result" Matrix to a rotation matrix used to rotate an entity so that it looks at the target vector3, from the eye vector3 position, the up vector3 being oriented like "up".
         * This function works in right handed mode
         * @param eye defines the final position of the entity
         * @param target defines where the entity should look at
         * @param up defines the up vector for the entity
         * @param result defines the target matrix
         */
        static LookAtRHToRef(eye: DeepImmutable<Vector3>, target: DeepImmutable<Vector3>, up: DeepImmutable<Vector3>, result: Matrix): void;
        /**
         * Create a left-handed orthographic projection matrix
         * @param width defines the viewport width
         * @param height defines the viewport height
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed orthographic projection matrix
         */
        static OrthoLH(width: number, height: number, znear: number, zfar: number): Matrix;
        /**
         * Store a left-handed orthographic projection to a given matrix
         * @param width defines the viewport width
         * @param height defines the viewport height
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         */
        static OrthoLHToRef(width: number, height: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Create a left-handed orthographic projection matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed orthographic projection matrix
         */
        static OrthoOffCenterLH(left: number, right: number, bottom: number, top: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a left-handed orthographic projection into a given matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         */
        static OrthoOffCenterLHToRef(left: number, right: number, bottom: number, top: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Creates a right-handed orthographic projection matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a right-handed orthographic projection matrix
         */
        static OrthoOffCenterRH(left: number, right: number, bottom: number, top: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a right-handed orthographic projection into a given matrix
         * @param left defines the viewport left coordinate
         * @param right defines the viewport right coordinate
         * @param bottom defines the viewport bottom coordinate
         * @param top defines the viewport top coordinate
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         */
        static OrthoOffCenterRHToRef(left: number, right: number, bottom: number, top: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Creates a left-handed perspective projection matrix
         * @param width defines the viewport width
         * @param height defines the viewport height
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed perspective projection matrix
         */
        static PerspectiveLH(width: number, height: number, znear: number, zfar: number): Matrix;
        /**
         * Creates a left-handed perspective projection matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a left-handed perspective projection matrix
         */
        static PerspectiveFovLH(fov: number, aspect: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a left-handed perspective projection into a given matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         * @param isVerticalFovFixed defines it the fov is vertically fixed (default) or horizontally
         */
        static PerspectiveFovLHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Stores a left-handed perspective projection into a given matrix with depth reversed
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar not used as infinity is used as far clip
         * @param result defines the target matrix
         * @param isVerticalFovFixed defines it the fov is vertically fixed (default) or horizontally
         */
        static PerspectiveFovReverseLHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Creates a right-handed perspective projection matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @returns a new matrix as a right-handed perspective projection matrix
         */
        static PerspectiveFovRH(fov: number, aspect: number, znear: number, zfar: number): Matrix;
        /**
         * Stores a right-handed perspective projection into a given matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         * @param isVerticalFovFixed defines it the fov is vertically fixed (default) or horizontally
         */
        static PerspectiveFovRHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Stores a right-handed perspective projection into a given matrix
         * @param fov defines the horizontal field of view
         * @param aspect defines the aspect ratio
         * @param znear defines the near clip plane
         * @param zfar not used as infinity is used as far clip
         * @param result defines the target matrix
         * @param isVerticalFovFixed defines it the fov is vertically fixed (default) or horizontally
         */
        static PerspectiveFovReverseRHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Stores a perspective projection for WebVR info a given matrix
         * @param fov defines the field of view
         * @param znear defines the near clip plane
         * @param zfar defines the far clip plane
         * @param result defines the target matrix
         * @param rightHanded defines if the matrix must be in right-handed mode (false by default)
         */
        static PerspectiveFovWebVRToRef(fov: {
            upDegrees: number;
            downDegrees: number;
            leftDegrees: number;
            rightDegrees: number;
        }, znear: number, zfar: number, result: Matrix, rightHanded?: boolean): void;
        /**
         * Computes a complete transformation matrix
         * @param viewport defines the viewport to use
         * @param world defines the world matrix
         * @param view defines the view matrix
         * @param projection defines the projection matrix
         * @param zmin defines the near clip plane
         * @param zmax defines the far clip plane
         * @returns the transformation matrix
         */
        static GetFinalMatrix(viewport: DeepImmutable<Viewport>, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>, zmin: number, zmax: number): Matrix;
        /**
         * Extracts a 2x2 matrix from a given matrix and store the result in a Float32Array
         * @param matrix defines the matrix to use
         * @returns a new Float32Array array with 4 elements : the 2x2 matrix extracted from the given matrix
         */
        static GetAsMatrix2x2(matrix: DeepImmutable<Matrix>): Float32Array | Array<number>;
        /**
         * Extracts a 3x3 matrix from a given matrix and store the result in a Float32Array
         * @param matrix defines the matrix to use
         * @returns a new Float32Array array with 9 elements : the 3x3 matrix extracted from the given matrix
         */
        static GetAsMatrix3x3(matrix: DeepImmutable<Matrix>): Float32Array | Array<number>;
        /**
         * Compute the transpose of a given matrix
         * @param matrix defines the matrix to transpose
         * @returns the new matrix
         */
        static Transpose(matrix: DeepImmutable<Matrix>): Matrix;
        /**
         * Compute the transpose of a matrix and store it in a target matrix
         * @param matrix defines the matrix to transpose
         * @param result defines the target matrix
         */
        static TransposeToRef(matrix: DeepImmutable<Matrix>, result: Matrix): void;
        /**
         * Computes a reflection matrix from a plane
         * @param plane defines the reflection plane
         * @returns a new matrix
         */
        static Reflection(plane: DeepImmutable<IPlaneLike>): Matrix;
        /**
         * Computes a reflection matrix from a plane
         * @param plane defines the reflection plane
         * @param result defines the target matrix
         */
        static ReflectionToRef(plane: DeepImmutable<IPlaneLike>, result: Matrix): void;
        /**
         * Sets the given matrix as a rotation matrix composed from the 3 left handed axes
         * @param xaxis defines the value of the 1st axis
         * @param yaxis defines the value of the 2nd axis
         * @param zaxis defines the value of the 3rd axis
         * @param result defines the target matrix
         */
        static FromXYZAxesToRef(xaxis: DeepImmutable<Vector3>, yaxis: DeepImmutable<Vector3>, zaxis: DeepImmutable<Vector3>, result: Matrix): void;
        /**
         * Creates a rotation matrix from a quaternion and stores it in a target matrix
         * @param quat defines the quaternion to use
         * @param result defines the target matrix
         */
        static FromQuaternionToRef(quat: DeepImmutable<Quaternion>, result: Matrix): void;
    }
    /**
     * @hidden
     */
    export class TmpVectors {
        static Vector2: Vector2[];
        static Vector3: Vector3[];
        static Vector4: Vector4[];
        static Quaternion: Quaternion[];
        static Matrix: Matrix[];
    }
}
declare module "babylonjs/Maths/math.path" {
    import { DeepImmutable, Nullable } from "babylonjs/types";
    import { Vector2, Vector3 } from "babylonjs/Maths/math.vector";
    /**
     * Defines potential orientation for back face culling
     */
    export enum Orientation {
        /**
         * Clockwise
         */
        CW = 0,
        /** Counter clockwise */
        CCW = 1
    }
    /** Class used to represent a Bezier curve */
    export class BezierCurve {
        /**
         * Returns the cubic Bezier interpolated value (float) at "t" (float) from the given x1, y1, x2, y2 floats
         * @param t defines the time
         * @param x1 defines the left coordinate on X axis
         * @param y1 defines the left coordinate on Y axis
         * @param x2 defines the right coordinate on X axis
         * @param y2 defines the right coordinate on Y axis
         * @returns the interpolated value
         */
        static Interpolate(t: number, x1: number, y1: number, x2: number, y2: number): number;
    }
    /**
     * Defines angle representation
     */
    export class Angle {
        private _radians;
        /**
         * Creates an Angle object of "radians" radians (float).
         * @param radians the angle in radians
         */
        constructor(radians: number);
        /**
         * Get value in degrees
         * @returns the Angle value in degrees (float)
         */
        degrees(): number;
        /**
         * Get value in radians
         * @returns the Angle value in radians (float)
         */
        radians(): number;
        /**
         * Gets a new Angle object valued with the gradient angle, in radians, of the line joining two points
         * @param a defines first point as the origin
         * @param b defines point
         * @returns a new Angle
         */
        static BetweenTwoPoints(a: DeepImmutable<Vector2>, b: DeepImmutable<Vector2>): Angle;
        /**
         * Gets a new Angle object from the given float in radians
         * @param radians defines the angle value in radians
         * @returns a new Angle
         */
        static FromRadians(radians: number): Angle;
        /**
         * Gets a new Angle object from the given float in degrees
         * @param degrees defines the angle value in degrees
         * @returns a new Angle
         */
        static FromDegrees(degrees: number): Angle;
    }
    /**
     * This represents an arc in a 2d space.
     */
    export class Arc2 {
        /** Defines the start point of the arc */
        startPoint: Vector2;
        /** Defines the mid point of the arc */
        midPoint: Vector2;
        /** Defines the end point of the arc */
        endPoint: Vector2;
        /**
         * Defines the center point of the arc.
         */
        centerPoint: Vector2;
        /**
         * Defines the radius of the arc.
         */
        radius: number;
        /**
         * Defines the angle of the arc (from mid point to end point).
         */
        angle: Angle;
        /**
         * Defines the start angle of the arc (from start point to middle point).
         */
        startAngle: Angle;
        /**
         * Defines the orientation of the arc (clock wise/counter clock wise).
         */
        orientation: Orientation;
        /**
         * Creates an Arc object from the three given points : start, middle and end.
         * @param startPoint Defines the start point of the arc
         * @param midPoint Defines the midlle point of the arc
         * @param endPoint Defines the end point of the arc
         */
        constructor(
        /** Defines the start point of the arc */
        startPoint: Vector2, 
        /** Defines the mid point of the arc */
        midPoint: Vector2, 
        /** Defines the end point of the arc */
        endPoint: Vector2);
    }
    /**
     * Represents a 2D path made up of multiple 2D points
     */
    export class Path2 {
        private _points;
        private _length;
        /**
         * If the path start and end point are the same
         */
        closed: boolean;
        /**
         * Creates a Path2 object from the starting 2D coordinates x and y.
         * @param x the starting points x value
         * @param y the starting points y value
         */
        constructor(x: number, y: number);
        /**
         * Adds a new segment until the given coordinates (x, y) to the current Path2.
         * @param x the added points x value
         * @param y the added points y value
         * @returns the updated Path2.
         */
        addLineTo(x: number, y: number): Path2;
        /**
         * Adds _numberOfSegments_ segments according to the arc definition (middle point coordinates, end point coordinates, the arc start point being the current Path2 last point) to the current Path2.
         * @param midX middle point x value
         * @param midY middle point y value
         * @param endX end point x value
         * @param endY end point y value
         * @param numberOfSegments (default: 36)
         * @returns the updated Path2.
         */
        addArcTo(midX: number, midY: number, endX: number, endY: number, numberOfSegments?: number): Path2;
        /**
         * Closes the Path2.
         * @returns the Path2.
         */
        close(): Path2;
        /**
         * Gets the sum of the distance between each sequential point in the path
         * @returns the Path2 total length (float).
         */
        length(): number;
        /**
         * Gets the points which construct the path
         * @returns the Path2 internal array of points.
         */
        getPoints(): Vector2[];
        /**
         * Retreives the point at the distance aways from the starting point
         * @param normalizedLengthPosition the length along the path to retreive the point from
         * @returns a new Vector2 located at a percentage of the Path2 total length on this path.
         */
        getPointAtLengthPosition(normalizedLengthPosition: number): Vector2;
        /**
         * Creates a new path starting from an x and y position
         * @param x starting x value
         * @param y starting y value
         * @returns a new Path2 starting at the coordinates (x, y).
         */
        static StartingAt(x: number, y: number): Path2;
    }
    /**
     * Represents a 3D path made up of multiple 3D points
     */
    export class Path3D {
        /**
         * an array of Vector3, the curve axis of the Path3D
         */
        path: Vector3[];
        private _curve;
        private _distances;
        private _tangents;
        private _normals;
        private _binormals;
        private _raw;
        private _alignTangentsWithPath;
        private readonly _pointAtData;
        /**
         * new Path3D(path, normal, raw)
         * Creates a Path3D. A Path3D is a logical math object, so not a mesh.
         * please read the description in the tutorial : https://doc.babylonjs.com/how_to/how_to_use_path3d
         * @param path an array of Vector3, the curve axis of the Path3D
         * @param firstNormal (options) Vector3, the first wanted normal to the curve. Ex (0, 1, 0) for a vertical normal.
         * @param raw (optional, default false) : boolean, if true the returned Path3D isn't normalized. Useful to depict path acceleration or speed.
         * @param alignTangentsWithPath (optional, default false) : boolean, if true the tangents will be aligned with the path.
         */
        constructor(
        /**
         * an array of Vector3, the curve axis of the Path3D
         */
        path: Vector3[], firstNormal?: Nullable<Vector3>, raw?: boolean, alignTangentsWithPath?: boolean);
        /**
         * Returns the Path3D array of successive Vector3 designing its curve.
         * @returns the Path3D array of successive Vector3 designing its curve.
         */
        getCurve(): Vector3[];
        /**
         * Returns the Path3D array of successive Vector3 designing its curve.
         * @returns the Path3D array of successive Vector3 designing its curve.
         */
        getPoints(): Vector3[];
        /**
         * @returns the computed length (float) of the path.
         */
        length(): number;
        /**
         * Returns an array populated with tangent vectors on each Path3D curve point.
         * @returns an array populated with tangent vectors on each Path3D curve point.
         */
        getTangents(): Vector3[];
        /**
         * Returns an array populated with normal vectors on each Path3D curve point.
         * @returns an array populated with normal vectors on each Path3D curve point.
         */
        getNormals(): Vector3[];
        /**
         * Returns an array populated with binormal vectors on each Path3D curve point.
         * @returns an array populated with binormal vectors on each Path3D curve point.
         */
        getBinormals(): Vector3[];
        /**
         * Returns an array populated with distances (float) of the i-th point from the first curve point.
         * @returns an array populated with distances (float) of the i-th point from the first curve point.
         */
        getDistances(): number[];
        /**
         * Returns an interpolated point along this path
         * @param position the position of the point along this path, from 0.0 to 1.0
         * @returns a new Vector3 as the point
         */
        getPointAt(position: number): Vector3;
        /**
         * Returns the tangent vector of an interpolated Path3D curve point at the specified position along this path.
         * @param position the position of the point along this path, from 0.0 to 1.0
         * @param interpolated (optional, default false) : boolean, if true returns an interpolated tangent instead of the tangent of the previous path point.
         * @returns a tangent vector corresponding to the interpolated Path3D curve point, if not interpolated, the tangent is taken from the precomputed tangents array.
         */
        getTangentAt(position: number, interpolated?: boolean): Vector3;
        /**
         * Returns the tangent vector of an interpolated Path3D curve point at the specified position along this path.
         * @param position the position of the point along this path, from 0.0 to 1.0
         * @param interpolated (optional, default false) : boolean, if true returns an interpolated normal instead of the normal of the previous path point.
         * @returns a normal vector corresponding to the interpolated Path3D curve point, if not interpolated, the normal is taken from the precomputed normals array.
         */
        getNormalAt(position: number, interpolated?: boolean): Vector3;
        /**
         * Returns the binormal vector of an interpolated Path3D curve point at the specified position along this path.
         * @param position the position of the point along this path, from 0.0 to 1.0
         * @param interpolated (optional, default false) : boolean, if true returns an interpolated binormal instead of the binormal of the previous path point.
         * @returns a binormal vector corresponding to the interpolated Path3D curve point, if not interpolated, the binormal is taken from the precomputed binormals array.
         */
        getBinormalAt(position: number, interpolated?: boolean): Vector3;
        /**
         * Returns the distance (float) of an interpolated Path3D curve point at the specified position along this path.
         * @param position the position of the point along this path, from 0.0 to 1.0
         * @returns the distance of the interpolated Path3D curve point at the specified position along this path.
         */
        getDistanceAt(position: number): number;
        /**
         * Returns the array index of the previous point of an interpolated point along this path
         * @param position the position of the point to interpolate along this path, from 0.0 to 1.0
         * @returns the array index
         */
        getPreviousPointIndexAt(position: number): number;
        /**
         * Returns the position of an interpolated point relative to the two path points it lies between, from 0.0 (point A) to 1.0 (point B)
         * @param position the position of the point to interpolate along this path, from 0.0 to 1.0
         * @returns the sub position
         */
        getSubPositionAt(position: number): number;
        /**
         * Returns the position of the closest virtual point on this path to an arbitrary Vector3, from 0.0 to 1.0
         * @param target the vector of which to get the closest position to
         * @returns the position of the closest virtual point on this path to the target vector
         */
        getClosestPositionTo(target: Vector3): number;
        /**
         * Returns a sub path (slice) of this path
         * @param start the position of the fist path point, from 0.0 to 1.0, or a negative value, which will get wrapped around from the end of the path to 0.0 to 1.0 values
         * @param end the position of the last path point, from 0.0 to 1.0, or a negative value, which will get wrapped around from the end of the path to 0.0 to 1.0 values
         * @returns a sub path (slice) of this path
         */
        slice(start?: number, end?: number): Path3D;
        /**
         * Forces the Path3D tangent, normal, binormal and distance recomputation.
         * @param path path which all values are copied into the curves points
         * @param firstNormal which should be projected onto the curve
         * @param alignTangentsWithPath (optional, default false) : boolean, if true the tangents will be aligned with the path
         * @returns the same object updated.
         */
        update(path: Vector3[], firstNormal?: Nullable<Vector3>, alignTangentsWithPath?: boolean): Path3D;
        private _compute;
        private _getFirstNonNullVector;
        private _getLastNonNullVector;
        private _normalVector;
        /**
         * Updates the point at data for an interpolated point along this curve
         * @param position the position of the point along this curve, from 0.0 to 1.0
         * @interpolateTNB wether to compute the interpolated tangent, normal and binormal
         * @returns the (updated) point at data
         */
        private _updatePointAtData;
        /**
         * Updates the point at data from the specified parameters
         * @param position where along the path the interpolated point is, from 0.0 to 1.0
         * @param point the interpolated point
         * @param parentIndex the index of an existing curve point that is on, or else positionally the first behind, the interpolated point
         */
        private _setPointAtData;
        /**
         * Updates the point at interpolation matrix for the tangents, normals and binormals
         */
        private _updateInterpolationMatrix;
    }
    /**
     * A Curve3 object is a logical object, so not a mesh, to handle curves in the 3D geometric space.
     * A Curve3 is designed from a series of successive Vector3.
     * @see https://doc.babylonjs.com/how_to/how_to_use_curve3
     */
    export class Curve3 {
        private _points;
        private _length;
        /**
         * Returns a Curve3 object along a Quadratic Bezier curve : https://doc.babylonjs.com/how_to/how_to_use_curve3#quadratic-bezier-curve
         * @param v0 (Vector3) the origin point of the Quadratic Bezier
         * @param v1 (Vector3) the control point
         * @param v2 (Vector3) the end point of the Quadratic Bezier
         * @param nbPoints (integer) the wanted number of points in the curve
         * @returns the created Curve3
         */
        static CreateQuadraticBezier(v0: DeepImmutable<Vector3>, v1: DeepImmutable<Vector3>, v2: DeepImmutable<Vector3>, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a Cubic Bezier curve : https://doc.babylonjs.com/how_to/how_to_use_curve3#cubic-bezier-curve
         * @param v0 (Vector3) the origin point of the Cubic Bezier
         * @param v1 (Vector3) the first control point
         * @param v2 (Vector3) the second control point
         * @param v3 (Vector3) the end point of the Cubic Bezier
         * @param nbPoints (integer) the wanted number of points in the curve
         * @returns the created Curve3
         */
        static CreateCubicBezier(v0: DeepImmutable<Vector3>, v1: DeepImmutable<Vector3>, v2: DeepImmutable<Vector3>, v3: DeepImmutable<Vector3>, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a Hermite Spline curve : https://doc.babylonjs.com/how_to/how_to_use_curve3#hermite-spline
         * @param p1 (Vector3) the origin point of the Hermite Spline
         * @param t1 (Vector3) the tangent vector at the origin point
         * @param p2 (Vector3) the end point of the Hermite Spline
         * @param t2 (Vector3) the tangent vector at the end point
         * @param nbPoints (integer) the wanted number of points in the curve
         * @returns the created Curve3
         */
        static CreateHermiteSpline(p1: DeepImmutable<Vector3>, t1: DeepImmutable<Vector3>, p2: DeepImmutable<Vector3>, t2: DeepImmutable<Vector3>, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a CatmullRom Spline curve :
         * @param points (array of Vector3) the points the spline must pass through. At least, four points required
         * @param nbPoints (integer) the wanted number of points between each curve control points
         * @param closed (boolean) optional with default false, when true forms a closed loop from the points
         * @returns the created Curve3
         */
        static CreateCatmullRomSpline(points: DeepImmutable<Vector3[]>, nbPoints: number, closed?: boolean): Curve3;
        /**
         * A Curve3 object is a logical object, so not a mesh, to handle curves in the 3D geometric space.
         * A Curve3 is designed from a series of successive Vector3.
         * Tuto : https://doc.babylonjs.com/how_to/how_to_use_curve3#curve3-object
         * @param points points which make up the curve
         */
        constructor(points: Vector3[]);
        /**
         * @returns the Curve3 stored array of successive Vector3
         */
        getPoints(): Vector3[];
        /**
         * @returns the computed length (float) of the curve.
         */
        length(): number;
        /**
         * Returns a new instance of Curve3 object : var curve = curveA.continue(curveB);
         * This new Curve3 is built by translating and sticking the curveB at the end of the curveA.
         * curveA and curveB keep unchanged.
         * @param curve the curve to continue from this curve
         * @returns the newly constructed curve
         */
        continue(curve: DeepImmutable<Curve3>): Curve3;
        private _computeLength;
    }
}
declare module "babylonjs/Animations/easing" {
    /**
     * This represents the main contract an easing function should follow.
     * Easing functions are used throughout the animation system.
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export interface IEasingFunction {
        /**
         * Given an input gradient between 0 and 1, this returns the corrseponding value
         * of the easing function.
         * The link below provides some of the most common examples of easing functions.
         * @see https://easings.net/
         * @param gradient Defines the value between 0 and 1 we want the easing value for
         * @returns the corresponding value on the curve defined by the easing function
         */
        ease(gradient: number): number;
    }
    /**
     * Base class used for every default easing function.
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class EasingFunction implements IEasingFunction {
        /**
         * Interpolation follows the mathematical formula associated with the easing function.
         */
        static readonly EASINGMODE_EASEIN: number;
        /**
         * Interpolation follows 100% interpolation minus the output of the formula associated with the easing function.
         */
        static readonly EASINGMODE_EASEOUT: number;
        /**
         * Interpolation uses EaseIn for the first half of the animation and EaseOut for the second half.
         */
        static readonly EASINGMODE_EASEINOUT: number;
        private _easingMode;
        /**
         * Sets the easing mode of the current function.
         * @param easingMode Defines the willing mode (EASINGMODE_EASEIN, EASINGMODE_EASEOUT or EASINGMODE_EASEINOUT)
         */
        setEasingMode(easingMode: number): void;
        /**
         * Gets the current easing mode.
         * @returns the easing mode
         */
        getEasingMode(): number;
        /**
         * @hidden
         */
        easeInCore(gradient: number): number;
        /**
         * Given an input gradient between 0 and 1, this returns the corresponding value
         * of the easing function.
         * @param gradient Defines the value between 0 and 1 we want the easing value for
         * @returns the corresponding value on the curve defined by the easing function
         */
        ease(gradient: number): number;
    }
    /**
     * Easing function with a circle shape (see link below).
     * @see https://easings.net/#easeInCirc
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class CircleEase extends EasingFunction implements IEasingFunction {
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a ease back shape (see link below).
     * @see https://easings.net/#easeInBack
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class BackEase extends EasingFunction implements IEasingFunction {
        /** Defines the amplitude of the function */
        amplitude: number;
        /**
         * Instantiates a back ease easing
         * @see https://easings.net/#easeInBack
         * @param amplitude Defines the amplitude of the function
         */
        constructor(
        /** Defines the amplitude of the function */
        amplitude?: number);
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a bouncing shape (see link below).
     * @see https://easings.net/#easeInBounce
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class BounceEase extends EasingFunction implements IEasingFunction {
        /** Defines the number of bounces */
        bounces: number;
        /** Defines the amplitude of the bounce */
        bounciness: number;
        /**
         * Instantiates a bounce easing
         * @see https://easings.net/#easeInBounce
         * @param bounces Defines the number of bounces
         * @param bounciness Defines the amplitude of the bounce
         */
        constructor(
        /** Defines the number of bounces */
        bounces?: number, 
        /** Defines the amplitude of the bounce */
        bounciness?: number);
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 3 shape (see link below).
     * @see https://easings.net/#easeInCubic
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class CubicEase extends EasingFunction implements IEasingFunction {
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with an elastic shape (see link below).
     * @see https://easings.net/#easeInElastic
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class ElasticEase extends EasingFunction implements IEasingFunction {
        /** Defines the number of oscillations*/
        oscillations: number;
        /** Defines the amplitude of the oscillations*/
        springiness: number;
        /**
         * Instantiates an elastic easing function
         * @see https://easings.net/#easeInElastic
         * @param oscillations Defines the number of oscillations
         * @param springiness Defines the amplitude of the oscillations
         */
        constructor(
        /** Defines the number of oscillations*/
        oscillations?: number, 
        /** Defines the amplitude of the oscillations*/
        springiness?: number);
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with an exponential shape (see link below).
     * @see https://easings.net/#easeInExpo
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class ExponentialEase extends EasingFunction implements IEasingFunction {
        /** Defines the exponent of the function */
        exponent: number;
        /**
         * Instantiates an exponential easing function
         * @see https://easings.net/#easeInExpo
         * @param exponent Defines the exponent of the function
         */
        constructor(
        /** Defines the exponent of the function */
        exponent?: number);
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power shape (see link below).
     * @see https://easings.net/#easeInQuad
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class PowerEase extends EasingFunction implements IEasingFunction {
        /** Defines the power of the function */
        power: number;
        /**
         * Instantiates an power base easing function
         * @see https://easings.net/#easeInQuad
         * @param power Defines the power of the function
         */
        constructor(
        /** Defines the power of the function */
        power?: number);
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 2 shape (see link below).
     * @see https://easings.net/#easeInQuad
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class QuadraticEase extends EasingFunction implements IEasingFunction {
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 4 shape (see link below).
     * @see https://easings.net/#easeInQuart
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class QuarticEase extends EasingFunction implements IEasingFunction {
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 5 shape (see link below).
     * @see https://easings.net/#easeInQuint
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class QuinticEase extends EasingFunction implements IEasingFunction {
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a sin shape (see link below).
     * @see https://easings.net/#easeInSine
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class SineEase extends EasingFunction implements IEasingFunction {
        /** @hidden */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a bezier shape (see link below).
     * @see http://cubic-bezier.com/#.17,.67,.83,.67
     * @see https://doc.babylonjs.com/babylon101/animations#easing-functions
     */
    export class BezierCurveEase extends EasingFunction implements IEasingFunction {
        /** Defines the x component of the start tangent in the bezier curve */
        x1: number;
        /** Defines the y component of the start tangent in the bezier curve */
        y1: number;
        /** Defines the x component of the end tangent in the bezier curve */
        x2: number;
        /** Defines the y component of the end tangent in the bezier curve */
        y2: number;
        /**
         * Instantiates a bezier function
         * @see http://cubic-bezier.com/#.17,.67,.83,.67
         * @param x1 Defines the x component of the start tangent in the bezier curve
         * @param y1 Defines the y component of the start tangent in the bezier curve
         * @param x2 Defines the x component of the end tangent in the bezier curve
         * @param y2 Defines the y component of the end tangent in the bezier curve
         */
        constructor(
        /** Defines the x component of the start tangent in the bezier curve */
        x1?: number, 
        /** Defines the y component of the start tangent in the bezier curve */
        y1?: number, 
        /** Defines the x component of the end tangent in the bezier curve */
        x2?: number, 
        /** Defines the y component of the end tangent in the bezier curve */
        y2?: number);
        /** @hidden */
        easeInCore(gradient: number): number;
    }
}
declare module "babylonjs/Maths/math.color" {
    import { DeepImmutable, FloatArray } from "babylonjs/types";
    /**
     * Class used to hold a RBG color
     */
    export class Color3 {
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r: number;
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g: number;
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b: number;
        /**
         * Creates a new Color3 object from red, green, blue values, all between 0 and 1
         * @param r defines the red component (between 0 and 1, default is 0)
         * @param g defines the green component (between 0 and 1, default is 0)
         * @param b defines the blue component (between 0 and 1, default is 0)
         */
        constructor(
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r?: number, 
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g?: number, 
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b?: number);
        /**
         * Creates a string with the Color3 current values
         * @returns the string representation of the Color3 object
         */
        toString(): string;
        /**
         * Returns the string "Color3"
         * @returns "Color3"
         */
        getClassName(): string;
        /**
         * Compute the Color3 hash code
         * @returns an unique number that can be used to hash Color3 objects
         */
        getHashCode(): number;
        /**
         * Stores in the given array from the given starting index the red, green, blue values as successive elements
         * @param array defines the array where to store the r,g,b components
         * @param index defines an optional index in the target array to define where to start storing values
         * @returns the current Color3 object
         */
        toArray(array: FloatArray, index?: number): Color3;
        /**
         * Update the current color with values stored in an array from the starting index of the given array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns the current Color3 object
         */
        fromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Color3;
        /**
         * Returns a new Color4 object from the current Color3 and the given alpha
         * @param alpha defines the alpha component on the new Color4 object (default is 1)
         * @returns a new Color4 object
         */
        toColor4(alpha?: number): Color4;
        /**
         * Returns a new array populated with 3 numeric elements : red, green and blue values
         * @returns the new array
         */
        asArray(): number[];
        /**
         * Returns the luminance value
         * @returns a float value
         */
        toLuminance(): number;
        /**
         * Multiply each Color3 rgb values by the given Color3 rgb values in a new Color3 object
         * @param otherColor defines the second operand
         * @returns the new Color3 object
         */
        multiply(otherColor: DeepImmutable<Color3>): Color3;
        /**
         * Multiply the rgb values of the Color3 and the given Color3 and stores the result in the object "result"
         * @param otherColor defines the second operand
         * @param result defines the Color3 object where to store the result
         * @returns the current Color3
         */
        multiplyToRef(otherColor: DeepImmutable<Color3>, result: Color3): Color3;
        /**
         * Determines equality between Color3 objects
         * @param otherColor defines the second operand
         * @returns true if the rgb values are equal to the given ones
         */
        equals(otherColor: DeepImmutable<Color3>): boolean;
        /**
         * Determines equality between the current Color3 object and a set of r,b,g values
         * @param r defines the red component to check
         * @param g defines the green component to check
         * @param b defines the blue component to check
         * @returns true if the rgb values are equal to the given ones
         */
        equalsFloats(r: number, g: number, b: number): boolean;
        /**
         * Multiplies in place each rgb value by scale
         * @param scale defines the scaling factor
         * @returns the updated Color3
         */
        scale(scale: number): Color3;
        /**
         * Multiplies the rgb values by scale and stores the result into "result"
         * @param scale defines the scaling factor
         * @param result defines the Color3 object where to store the result
         * @returns the unmodified current Color3
         */
        scaleToRef(scale: number, result: Color3): Color3;
        /**
         * Scale the current Color3 values by a factor and add the result to a given Color3
         * @param scale defines the scale factor
         * @param result defines color to store the result into
         * @returns the unmodified current Color3
         */
        scaleAndAddToRef(scale: number, result: Color3): Color3;
        /**
         * Clamps the rgb values by the min and max values and stores the result into "result"
         * @param min defines minimum clamping value (default is 0)
         * @param max defines maximum clamping value (default is 1)
         * @param result defines color to store the result into
         * @returns the original Color3
         */
        clampToRef(min: number | undefined, max: number | undefined, result: Color3): Color3;
        /**
         * Creates a new Color3 set with the added values of the current Color3 and of the given one
         * @param otherColor defines the second operand
         * @returns the new Color3
         */
        add(otherColor: DeepImmutable<Color3>): Color3;
        /**
         * Stores the result of the addition of the current Color3 and given one rgb values into "result"
         * @param otherColor defines the second operand
         * @param result defines Color3 object to store the result into
         * @returns the unmodified current Color3
         */
        addToRef(otherColor: DeepImmutable<Color3>, result: Color3): Color3;
        /**
         * Returns a new Color3 set with the subtracted values of the given one from the current Color3
         * @param otherColor defines the second operand
         * @returns the new Color3
         */
        subtract(otherColor: DeepImmutable<Color3>): Color3;
        /**
         * Stores the result of the subtraction of given one from the current Color3 rgb values into "result"
         * @param otherColor defines the second operand
         * @param result defines Color3 object to store the result into
         * @returns the unmodified current Color3
         */
        subtractToRef(otherColor: DeepImmutable<Color3>, result: Color3): Color3;
        /**
         * Copy the current object
         * @returns a new Color3 copied the current one
         */
        clone(): Color3;
        /**
         * Copies the rgb values from the source in the current Color3
         * @param source defines the source Color3 object
         * @returns the updated Color3 object
         */
        copyFrom(source: DeepImmutable<Color3>): Color3;
        /**
         * Updates the Color3 rgb values from the given floats
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @returns the current Color3 object
         */
        copyFromFloats(r: number, g: number, b: number): Color3;
        /**
         * Updates the Color3 rgb values from the given floats
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @returns the current Color3 object
         */
        set(r: number, g: number, b: number): Color3;
        /**
         * Compute the Color3 hexadecimal code as a string
         * @returns a string containing the hexadecimal representation of the Color3 object
         */
        toHexString(): string;
        /**
         * Computes a new Color3 converted from the current one to linear space
         * @returns a new Color3 object
         */
        toLinearSpace(): Color3;
        /**
         * Converts current color in rgb space to HSV values
         * @returns a new color3 representing the HSV values
         */
        toHSV(): Color3;
        /**
         * Converts current color in rgb space to HSV values
         * @param result defines the Color3 where to store the HSV values
         */
        toHSVToRef(result: Color3): void;
        /**
         * Converts the Color3 values to linear space and stores the result in "convertedColor"
         * @param convertedColor defines the Color3 object where to store the linear space version
         * @returns the unmodified Color3
         */
        toLinearSpaceToRef(convertedColor: Color3): Color3;
        /**
         * Computes a new Color3 converted from the current one to gamma space
         * @returns a new Color3 object
         */
        toGammaSpace(): Color3;
        /**
         * Converts the Color3 values to gamma space and stores the result in "convertedColor"
         * @param convertedColor defines the Color3 object where to store the gamma space version
         * @returns the unmodified Color3
         */
        toGammaSpaceToRef(convertedColor: Color3): Color3;
        private static _BlackReadOnly;
        /**
         * Convert Hue, saturation and value to a Color3 (RGB)
         * @param hue defines the hue
         * @param saturation defines the saturation
         * @param value defines the value
         * @param result defines the Color3 where to store the RGB values
         */
        static HSVtoRGBToRef(hue: number, saturation: number, value: number, result: Color3): void;
        /**
         * Creates a new Color3 from the string containing valid hexadecimal values
         * @param hex defines a string containing valid hexadecimal values
         * @returns a new Color3 object
         */
        static FromHexString(hex: string): Color3;
        /**
         * Creates a new Color3 from the starting index of the given array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns a new Color3 object
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Color3;
        /**
         * Creates a new Color3 from the starting index element of the given array
         * @param array defines the source array to read from
         * @param offset defines the offset in the source array
         * @param result defines the target Color3 object
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number | undefined, result: Color3): void;
        /**
         * Creates a new Color3 from integer values (< 256)
         * @param r defines the red component to read from (value between 0 and 255)
         * @param g defines the green component to read from (value between 0 and 255)
         * @param b defines the blue component to read from (value between 0 and 255)
         * @returns a new Color3 object
         */
        static FromInts(r: number, g: number, b: number): Color3;
        /**
         * Creates a new Color3 with values linearly interpolated of "amount" between the start Color3 and the end Color3
         * @param start defines the start Color3 value
         * @param end defines the end Color3 value
         * @param amount defines the gradient value between start and end
         * @returns a new Color3 object
         */
        static Lerp(start: DeepImmutable<Color3>, end: DeepImmutable<Color3>, amount: number): Color3;
        /**
         * Creates a new Color3 with values linearly interpolated of "amount" between the start Color3 and the end Color3
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @param result defines the Color3 object where to store the result
         */
        static LerpToRef(left: DeepImmutable<Color3>, right: DeepImmutable<Color3>, amount: number, result: Color3): void;
        /**
         * Returns a Color3 value containing a red color
         * @returns a new Color3 object
         */
        static Red(): Color3;
        /**
         * Returns a Color3 value containing a green color
         * @returns a new Color3 object
         */
        static Green(): Color3;
        /**
         * Returns a Color3 value containing a blue color
         * @returns a new Color3 object
         */
        static Blue(): Color3;
        /**
         * Returns a Color3 value containing a black color
         * @returns a new Color3 object
         */
        static Black(): Color3;
        /**
          * Gets a Color3 value containing a black color that must not be updated
          */
        static get BlackReadOnly(): DeepImmutable<Color3>;
        /**
         * Returns a Color3 value containing a white color
         * @returns a new Color3 object
         */
        static White(): Color3;
        /**
         * Returns a Color3 value containing a purple color
         * @returns a new Color3 object
         */
        static Purple(): Color3;
        /**
         * Returns a Color3 value containing a magenta color
         * @returns a new Color3 object
         */
        static Magenta(): Color3;
        /**
         * Returns a Color3 value containing a yellow color
         * @returns a new Color3 object
         */
        static Yellow(): Color3;
        /**
         * Returns a Color3 value containing a gray color
         * @returns a new Color3 object
         */
        static Gray(): Color3;
        /**
         * Returns a Color3 value containing a teal color
         * @returns a new Color3 object
         */
        static Teal(): Color3;
        /**
         * Returns a Color3 value containing a random color
         * @returns a new Color3 object
         */
        static Random(): Color3;
    }
    /**
     * Class used to hold a RBGA color
     */
    export class Color4 {
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r: number;
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g: number;
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b: number;
        /**
         * Defines the alpha component (between 0 and 1, default is 1)
         */
        a: number;
        /**
         * Creates a new Color4 object from red, green, blue values, all between 0 and 1
         * @param r defines the red component (between 0 and 1, default is 0)
         * @param g defines the green component (between 0 and 1, default is 0)
         * @param b defines the blue component (between 0 and 1, default is 0)
         * @param a defines the alpha component (between 0 and 1, default is 1)
         */
        constructor(
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r?: number, 
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g?: number, 
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b?: number, 
        /**
         * Defines the alpha component (between 0 and 1, default is 1)
         */
        a?: number);
        /**
         * Adds in place the given Color4 values to the current Color4 object
         * @param right defines the second operand
         * @returns the current updated Color4 object
         */
        addInPlace(right: DeepImmutable<Color4>): Color4;
        /**
         * Creates a new array populated with 4 numeric elements : red, green, blue, alpha values
         * @returns the new array
         */
        asArray(): number[];
        /**
         * Stores from the starting index in the given array the Color4 successive values
         * @param array defines the array where to store the r,g,b components
         * @param index defines an optional index in the target array to define where to start storing values
         * @returns the current Color4 object
         */
        toArray(array: number[], index?: number): Color4;
        /**
         * Update the current color with values stored in an array from the starting index of the given array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns the current Color4 object
         */
        fromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Color4;
        /**
         * Determines equality between Color4 objects
         * @param otherColor defines the second operand
         * @returns true if the rgba values are equal to the given ones
         */
        equals(otherColor: DeepImmutable<Color4>): boolean;
        /**
         * Creates a new Color4 set with the added values of the current Color4 and of the given one
         * @param right defines the second operand
         * @returns a new Color4 object
         */
        add(right: DeepImmutable<Color4>): Color4;
        /**
         * Creates a new Color4 set with the subtracted values of the given one from the current Color4
         * @param right defines the second operand
         * @returns a new Color4 object
         */
        subtract(right: DeepImmutable<Color4>): Color4;
        /**
         * Subtracts the given ones from the current Color4 values and stores the results in "result"
         * @param right defines the second operand
         * @param result defines the Color4 object where to store the result
         * @returns the current Color4 object
         */
        subtractToRef(right: DeepImmutable<Color4>, result: Color4): Color4;
        /**
         * Creates a new Color4 with the current Color4 values multiplied by scale
         * @param scale defines the scaling factor to apply
         * @returns a new Color4 object
         */
        scale(scale: number): Color4;
        /**
         * Multiplies the current Color4 values by scale and stores the result in "result"
         * @param scale defines the scaling factor to apply
         * @param result defines the Color4 object where to store the result
         * @returns the current unmodified Color4
         */
        scaleToRef(scale: number, result: Color4): Color4;
        /**
         * Scale the current Color4 values by a factor and add the result to a given Color4
         * @param scale defines the scale factor
         * @param result defines the Color4 object where to store the result
         * @returns the unmodified current Color4
         */
        scaleAndAddToRef(scale: number, result: Color4): Color4;
        /**
         * Clamps the rgb values by the min and max values and stores the result into "result"
         * @param min defines minimum clamping value (default is 0)
         * @param max defines maximum clamping value (default is 1)
         * @param result defines color to store the result into.
         * @returns the cuurent Color4
         */
        clampToRef(min: number | undefined, max: number | undefined, result: Color4): Color4;
        /**
          * Multipy an Color4 value by another and return a new Color4 object
          * @param color defines the Color4 value to multiply by
          * @returns a new Color4 object
          */
        multiply(color: Color4): Color4;
        /**
         * Multipy a Color4 value by another and push the result in a reference value
         * @param color defines the Color4 value to multiply by
         * @param result defines the Color4 to fill the result in
         * @returns the result Color4
         */
        multiplyToRef(color: Color4, result: Color4): Color4;
        /**
         * Creates a string with the Color4 current values
         * @returns the string representation of the Color4 object
         */
        toString(): string;
        /**
         * Returns the string "Color4"
         * @returns "Color4"
         */
        getClassName(): string;
        /**
         * Compute the Color4 hash code
         * @returns an unique number that can be used to hash Color4 objects
         */
        getHashCode(): number;
        /**
         * Creates a new Color4 copied from the current one
         * @returns a new Color4 object
         */
        clone(): Color4;
        /**
         * Copies the given Color4 values into the current one
         * @param source defines the source Color4 object
         * @returns the current updated Color4 object
         */
        copyFrom(source: Color4): Color4;
        /**
         * Copies the given float values into the current one
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @param a defines the alpha component to read from
         * @returns the current updated Color4 object
         */
        copyFromFloats(r: number, g: number, b: number, a: number): Color4;
        /**
         * Copies the given float values into the current one
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @param a defines the alpha component to read from
         * @returns the current updated Color4 object
         */
        set(r: number, g: number, b: number, a: number): Color4;
        /**
         * Compute the Color4 hexadecimal code as a string
         * @param returnAsColor3 defines if the string should only contains RGB values (off by default)
         * @returns a string containing the hexadecimal representation of the Color4 object
         */
        toHexString(returnAsColor3?: boolean): string;
        /**
         * Computes a new Color4 converted from the current one to linear space
         * @returns a new Color4 object
         */
        toLinearSpace(): Color4;
        /**
         * Converts the Color4 values to linear space and stores the result in "convertedColor"
         * @param convertedColor defines the Color4 object where to store the linear space version
         * @returns the unmodified Color4
         */
        toLinearSpaceToRef(convertedColor: Color4): Color4;
        /**
         * Computes a new Color4 converted from the current one to gamma space
         * @returns a new Color4 object
         */
        toGammaSpace(): Color4;
        /**
         * Converts the Color4 values to gamma space and stores the result in "convertedColor"
         * @param convertedColor defines the Color4 object where to store the gamma space version
         * @returns the unmodified Color4
         */
        toGammaSpaceToRef(convertedColor: Color4): Color4;
        /**
         * Creates a new Color4 from the string containing valid hexadecimal values
         * @param hex defines a string containing valid hexadecimal values
         * @returns a new Color4 object
         */
        static FromHexString(hex: string): Color4;
        /**
         * Creates a new Color4 object set with the linearly interpolated values of "amount" between the left Color4 object and the right Color4 object
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @returns a new Color4 object
         */
        static Lerp(left: DeepImmutable<Color4>, right: DeepImmutable<Color4>, amount: number): Color4;
        /**
         * Set the given "result" with the linearly interpolated values of "amount" between the left Color4 object and the right Color4 object
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @param result defines the Color4 object where to store data
         */
        static LerpToRef(left: DeepImmutable<Color4>, right: DeepImmutable<Color4>, amount: number, result: Color4): void;
        /**
         * Creates a new Color4 from a Color3 and an alpha value
         * @param color3 defines the source Color3 to read from
         * @param alpha defines the alpha component (1.0 by default)
         * @returns a new Color4 object
         */
        static FromColor3(color3: DeepImmutable<Color3>, alpha?: number): Color4;
        /**
         * Creates a new Color4 from the starting index element of the given array
         * @param array defines the source array to read from
         * @param offset defines the offset in the source array
         * @returns a new Color4 object
         */
        static FromArray(array: DeepImmutable<ArrayLike<number>>, offset?: number): Color4;
        /**
         * Creates a new Color4 from the starting index element of the given array
         * @param array defines the source array to read from
         * @param offset defines the offset in the source array
         * @param result defines the target Color4 object
         */
        static FromArrayToRef(array: DeepImmutable<ArrayLike<number>>, offset: number | undefined, result: Color4): void;
        /**
         * Creates a new Color3 from integer values (< 256)
         * @param r defines the red component to read from (value between 0 and 255)
         * @param g defines the green component to read from (value between 0 and 255)
         * @param b defines the blue component to read from (value between 0 and 255)
         * @param a defines the alpha component to read from (value between 0 and 255)
         * @returns a new Color3 object
         */
        static FromInts(r: number, g: number, b: number, a: number): Color4;
        /**
         * Check the content of a given array and convert it to an array containing RGBA data
         * If the original array was already containing count * 4 values then it is returned directly
         * @param colors defines the array to check
         * @param count defines the number of RGBA data to expect
         * @returns an array containing count * 4 values (RGBA)
         */
        static CheckColors4(colors: number[], count: number): number[];
    }
    /**
     * @hidden
     */
    export class TmpColors {
        static Color3: Color3[];
        static Color4: Color4[];
    }
}
declare module "babylonjs/Animations/animationKey" {
    /**
     * Defines an interface which represents an animation key frame
     */
    export interface IAnimationKey {
        /**
         * Frame of the key frame
         */
        frame: number;
        /**
         * Value at the specifies key frame
         */
        value: any;
        /**
         * The input tangent for the cubic hermite spline
         */
        inTangent?: any;
        /**
         * The output tangent for the cubic hermite spline
         */
        outTangent?: any;
        /**
         * The animation interpolation type
         */
        interpolation?: AnimationKeyInterpolation;
    }
    /**
     * Enum for the animation key frame interpolation type
     */
    export enum AnimationKeyInterpolation {
        /**
         * Do not interpolate between keys and use the start key value only. Tangents are ignored
         */
        STEP = 1
    }
}
declare module "babylonjs/Animations/animationRange" {
    /**
     * Represents the range of an animation
     */
    export class AnimationRange {
        /**The name of the animation range**/
        name: string;
        /**The starting frame of the animation */
        from: number;
        /**The ending frame of the animation*/
        to: number;
        /**
         * Initializes the range of an animation
         * @param name The name of the animation range
         * @param from The starting frame of the animation
         * @param to The ending frame of the animation
         */
        constructor(
        /**The name of the animation range**/
        name: string, 
        /**The starting frame of the animation */
        from: number, 
        /**The ending frame of the animation*/
        to: number);
        /**
         * Makes a copy of the animation range
         * @returns A copy of the animation range
         */
        clone(): AnimationRange;
    }
}
declare module "babylonjs/Animations/animationEvent" {
    /**
     * Composed of a frame, and an action function
     */
    export class AnimationEvent {
        /** The frame for which the event is triggered **/
        frame: number;
        /** The event to perform when triggered **/
        action: (currentFrame: number) => void;
        /** Specifies if the event should be triggered only once**/
        onlyOnce?: boolean | undefined;
        /**
         * Specifies if the animation event is done
         */
        isDone: boolean;
        /**
         * Initializes the animation event
         * @param frame The frame for which the event is triggered
         * @param action The event to perform when triggered
         * @param onlyOnce Specifies if the event should be triggered only once
         */
        constructor(
        /** The frame for which the event is triggered **/
        frame: number, 
        /** The event to perform when triggered **/
        action: (currentFrame: number) => void, 
        /** Specifies if the event should be triggered only once**/
        onlyOnce?: boolean | undefined);
        /** @hidden */
        _clone(): AnimationEvent;
    }
}
declare module "babylonjs/Behaviors/behavior" {
    import { Nullable } from "babylonjs/types";
    /**
     * Interface used to define a behavior
     */
    export interface Behavior<T> {
        /** gets or sets behavior's name */
        name: string;
        /**
         * Function called when the behavior needs to be initialized (after attaching it to a target)
         */
        init(): void;
        /**
         * Called when the behavior is attached to a target
         * @param target defines the target where the behavior is attached to
         */
        attach(target: T): void;
        /**
         * Called when the behavior is detached from its target
         */
        detach(): void;
    }
    /**
     * Interface implemented by classes supporting behaviors
     */
    export interface IBehaviorAware<T> {
        /**
         * Attach a behavior
         * @param behavior defines the behavior to attach
         * @returns the current host
         */
        addBehavior(behavior: Behavior<T>): T;
        /**
         * Remove a behavior from the current object
         * @param behavior defines the behavior to detach
         * @returns the current host
         */
        removeBehavior(behavior: Behavior<T>): T;
        /**
         * Gets a behavior using its name to search
         * @param name defines the name to search
         * @returns the behavior or null if not found
         */
        getBehaviorByName(name: string): Nullable<Behavior<T>>;
    }
}
declare module "babylonjs/Meshes/buffer" {
    import { Nullable, DataArray } from "babylonjs/types";
    import { DataBuffer } from "babylonjs/Meshes/dataBuffer";
    /**
     * Class used to store data that will be store in GPU memory
     */
    export class Buffer {
        private _engine;
        private _buffer;
        /** @hidden */
        _data: Nullable<DataArray>;
        private _updatable;
        private _instanced;
        private _divisor;
        private _isAlreadyOwned;
        /**
         * Gets the byte stride.
         */
        readonly byteStride: number;
        /**
         * Constructor
         * @param engine the engine
         * @param data the data to use for this buffer
         * @param updatable whether the data is updatable
         * @param stride the stride (optional)
         * @param postponeInternalCreation whether to postpone creating the internal WebGL buffer (optional)
         * @param instanced whether the buffer is instanced (optional)
         * @param useBytes set to true if the stride in in bytes (optional)
         * @param divisor sets an optional divisor for instances (1 by default)
         */
        constructor(engine: any, data: DataArray, updatable: boolean, stride?: number, postponeInternalCreation?: boolean, instanced?: boolean, useBytes?: boolean, divisor?: number);
        /**
         * Create a new VertexBuffer based on the current buffer
         * @param kind defines the vertex buffer kind (position, normal, etc.)
         * @param offset defines offset in the buffer (0 by default)
         * @param size defines the size in floats of attributes (position is 3 for instance)
         * @param stride defines the stride size in floats in the buffer (the offset to apply to reach next value when data is interleaved)
         * @param instanced defines if the vertex buffer contains indexed data
         * @param useBytes defines if the offset and stride are in bytes     *
         * @param divisor sets an optional divisor for instances (1 by default)
         * @returns the new vertex buffer
         */
        createVertexBuffer(kind: string, offset: number, size: number, stride?: number, instanced?: boolean, useBytes?: boolean, divisor?: number): VertexBuffer;
        /**
         * Gets a boolean indicating if the Buffer is updatable?
         * @returns true if the buffer is updatable
         */
        isUpdatable(): boolean;
        /**
         * Gets current buffer's data
         * @returns a DataArray or null
         */
        getData(): Nullable<DataArray>;
        /**
         * Gets underlying native buffer
         * @returns underlying native buffer
         */
        getBuffer(): Nullable<DataBuffer>;
        /**
         * Gets the stride in float32 units (i.e. byte stride / 4).
         * May not be an integer if the byte stride is not divisible by 4.
         * @returns the stride in float32 units
         * @deprecated Please use byteStride instead.
         */
        getStrideSize(): number;
        /**
         * Store data into the buffer. If the buffer was already used it will be either recreated or updated depending on isUpdatable property
         * @param data defines the data to store
         */
        create(data?: Nullable<DataArray>): void;
        /** @hidden */
        _rebuild(): void;
        /**
         * Update current buffer data
         * @param data defines the data to store
         */
        update(data: DataArray): void;
        /**
         * Updates the data directly.
         * @param data the new data
         * @param offset the new offset
         * @param vertexCount the vertex count (optional)
         * @param useBytes set to true if the offset is in bytes
         */
        updateDirectly(data: DataArray, offset: number, vertexCount?: number, useBytes?: boolean): void;
        /** @hidden */
        _increaseReferences(): void;
        /**
         * Release all resources
         */
        dispose(): void;
    }
    /**
         * Specialized buffer used to store vertex data
         */
    export class VertexBuffer {
        /** @hidden */
        _buffer: Buffer;
        private _kind;
        private _size;
        private _ownsBuffer;
        private _instanced;
        private _instanceDivisor;
        /**
         * The byte type.
         */
        static readonly BYTE: number;
        /**
         * The unsigned byte type.
         */
        static readonly UNSIGNED_BYTE: number;
        /**
         * The short type.
         */
        static readonly SHORT: number;
        /**
         * The unsigned short type.
         */
        static readonly UNSIGNED_SHORT: number;
        /**
         * The integer type.
         */
        static readonly INT: number;
        /**
         * The unsigned integer type.
         */
        static readonly UNSIGNED_INT: number;
        /**
         * The float type.
         */
        static readonly FLOAT: number;
        /**
         * Gets or sets the instance divisor when in instanced mode
         */
        get instanceDivisor(): number;
        set instanceDivisor(value: number);
        /**
         * Gets the byte stride.
         */
        readonly byteStride: number;
        /**
         * Gets the byte offset.
         */
        readonly byteOffset: number;
        /**
         * Gets whether integer data values should be normalized into a certain range when being casted to a float.
         */
        readonly normalized: boolean;
        /**
         * Gets the data type of each component in the array.
         */
        readonly type: number;
        /**
         * Constructor
         * @param engine the engine
         * @param data the data to use for this vertex buffer
         * @param kind the vertex buffer kind
         * @param updatable whether the data is updatable
         * @param postponeInternalCreation whether to postpone creating the internal WebGL buffer (optional)
         * @param stride the stride (optional)
         * @param instanced whether the buffer is instanced (optional)
         * @param offset the offset of the data (optional)
         * @param size the number of components (optional)
         * @param type the type of the component (optional)
         * @param normalized whether the data contains normalized data (optional)
         * @param useBytes set to true if stride and offset are in bytes (optional)
         * @param divisor defines the instance divisor to use (1 by default)
         * @param takeBufferOwnership defines if the buffer should be released when the vertex buffer is disposed
         */
        constructor(engine: any, data: DataArray | Buffer, kind: string, updatable: boolean, postponeInternalCreation?: boolean, stride?: number, instanced?: boolean, offset?: number, size?: number, type?: number, normalized?: boolean, useBytes?: boolean, divisor?: number, takeBufferOwnership?: boolean);
        /** @hidden */
        _rebuild(): void;
        /**
         * Returns the kind of the VertexBuffer (string)
         * @returns a string
         */
        getKind(): string;
        /**
         * Gets a boolean indicating if the VertexBuffer is updatable?
         * @returns true if the buffer is updatable
         */
        isUpdatable(): boolean;
        /**
         * Gets current buffer's data
         * @returns a DataArray or null
         */
        getData(): Nullable<DataArray>;
        /**
         * Gets underlying native buffer
         * @returns underlying native buffer
         */
        getBuffer(): Nullable<DataBuffer>;
        /**
         * Gets the stride in float32 units (i.e. byte stride / 4).
         * May not be an integer if the byte stride is not divisible by 4.
         * @returns the stride in float32 units
         * @deprecated Please use byteStride instead.
         */
        getStrideSize(): number;
        /**
         * Returns the offset as a multiple of the type byte length.
         * @returns the offset in bytes
         * @deprecated Please use byteOffset instead.
         */
        getOffset(): number;
        /**
         * Returns the number of components per vertex attribute (integer)
         * @returns the size in float
         */
        getSize(): number;
        /**
         * Gets a boolean indicating is the internal buffer of the VertexBuffer is instanced
         * @returns true if this buffer is instanced
         */
        getIsInstanced(): boolean;
        /**
         * Returns the instancing divisor, zero for non-instanced (integer).
         * @returns a number
         */
        getInstanceDivisor(): number;
        /**
         * Store data into the buffer. If the buffer was already used it will be either recreated or updated depending on isUpdatable property
         * @param data defines the data to store
         */
        create(data?: DataArray): void;
        /**
         * Updates the underlying buffer according to the passed numeric array or Float32Array.
         * This function will create a new buffer if the current one is not updatable
         * @param data defines the data to store
         */
        update(data: DataArray): void;
        /**
         * Updates directly the underlying WebGLBuffer according to the passed numeric array or Float32Array.
         * Returns the directly updated WebGLBuffer.
         * @param data the new data
         * @param offset the new offset
         * @param useBytes set to true if the offset is in bytes
         */
        updateDirectly(data: DataArray, offset: number, useBytes?: boolean): void;
        /**
         * Disposes the VertexBuffer and the underlying WebGLBuffer.
         */
        dispose(): void;
        /**
         * Enumerates each value of this vertex buffer as numbers.
         * @param count the number of values to enumerate
         * @param callback the callback function called for each value
         */
        forEach(count: number, callback: (value: number, index: number) => void): void;
        /**
         * Positions
         */
        static readonly PositionKind: string;
        /**
         * Normals
         */
        static readonly NormalKind: string;
        /**
         * Tangents
         */
        static readonly TangentKind: string;
        /**
         * Texture coordinates
         */
        static readonly UVKind: string;
        /**
         * Texture coordinates 2
         */
        static readonly UV2Kind: string;
        /**
         * Texture coordinates 3
         */
        static readonly UV3Kind: string;
        /**
         * Texture coordinates 4
         */
        static readonly UV4Kind: string;
        /**
         * Texture coordinates 5
         */
        static readonly UV5Kind: string;
        /**
         * Texture coordinates 6
         */
        static readonly UV6Kind: string;
        /**
         * Colors
         */
        static readonly ColorKind: string;
        /**
         * Matrix indices (for bones)
         */
        static readonly MatricesIndicesKind: string;
        /**
         * Matrix weights (for bones)
         */
        static readonly MatricesWeightsKind: string;
        /**
         * Additional matrix indices (for bones)
         */
        static readonly MatricesIndicesExtraKind: string;
        /**
         * Additional matrix weights (for bones)
         */
        static readonly MatricesWeightsExtraKind: string;
        /**
         * Deduces the stride given a kind.
         * @param kind The kind string to deduce
         * @returns The deduced stride
         */
        static DeduceStride(kind: string): number;
        /**
         * Gets the byte length of the given type.
         * @param type the type
         * @returns the number of bytes
         */
        static GetTypeByteLength(type: number): number;
        /**
         * Enumerates each value of the given parameters as numbers.
         * @param data the data to enumerate
         * @param byteOffset the byte offset of the data
         * @param byteStride the byte stride of the data
         * @param componentCount the number of components per element
         * @param componentType the type of the component
         * @param count the number of values to enumerate
         * @param normalized whether the data is normalized
         * @param callback the callback function called for each value
         */
        static ForEach(data: DataArray, byteOffset: number, byteStride: number, componentCount: number, componentType: number, count: number, normalized: boolean, callback: (value: number, index: number) => void): void;
        private static _GetFloatValue;
    }
}
declare module "babylonjs/Collisions/intersectionInfo" {
    import { Nullable } from "babylonjs/types";
    /**
     * @hidden
     */
    export class IntersectionInfo {
        bu: Nullable<number>;
        bv: Nullable<number>;
        distance: number;
        faceId: number;
        subMeshId: number;
        constructor(bu: Nullable<number>, bv: Nullable<number>, distance: number);
    }
}
declare module "babylonjs/Culling/boundingSphere" {
    import { DeepImmutable } from "babylonjs/types";
    import { Matrix, Vector3 } from "babylonjs/Maths/math.vector";
    import { Plane } from "babylonjs/Maths/math.plane";
    /**
     * Class used to store bounding sphere information
     */
    export class BoundingSphere {
        /**
         * Gets the center of the bounding sphere in local space
         */
        readonly center: Vector3;
        /**
         * Radius of the bounding sphere in local space
         */
        radius: number;
        /**
         * Gets the center of the bounding sphere in world space
         */
        readonly centerWorld: Vector3;
        /**
         * Radius of the bounding sphere in world space
         */
        radiusWorld: number;
        /**
         * Gets the minimum vector in local space
         */
        readonly minimum: Vector3;
        /**
         * Gets the maximum vector in local space
         */
        readonly maximum: Vector3;
        private _worldMatrix;
        private static readonly TmpVector3;
        /**
         * Creates a new bounding sphere
         * @param min defines the minimum vector (in local space)
         * @param max defines the maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        constructor(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>);
        /**
         * Recreates the entire bounding sphere from scratch as if we call the constructor in place
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        reConstruct(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>): void;
        /**
         * Scale the current bounding sphere by applying a scale factor
         * @param factor defines the scale factor to apply
         * @returns the current bounding box
         */
        scale(factor: number): BoundingSphere;
        /**
         * Gets the world matrix of the bounding box
         * @returns a matrix
         */
        getWorldMatrix(): DeepImmutable<Matrix>;
        /** @hidden */
        _update(worldMatrix: DeepImmutable<Matrix>): void;
        /**
         * Tests if the bounding sphere is intersecting the frustum planes
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an intersection
         */
        isInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if the bounding sphere center is in between the frustum planes.
         * Used for optimistic fast inclusion.
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if the sphere center is in between the frustum planes
         */
        isCenterInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if a point is inside the bounding sphere
         * @param point defines the point to test
         * @returns true if the point is inside the bounding sphere
         */
        intersectsPoint(point: DeepImmutable<Vector3>): boolean;
        /**
         * Checks if two sphere intersct
         * @param sphere0 sphere 0
         * @param sphere1 sphere 1
         * @returns true if the speres intersect
         */
        static Intersects(sphere0: DeepImmutable<BoundingSphere>, sphere1: DeepImmutable<BoundingSphere>): boolean;
    }
}
declare module "babylonjs/Culling/boundingBox" {
    import { DeepImmutable } from "babylonjs/types";
    import { Matrix, Vector3 } from "babylonjs/Maths/math.vector";
    import { BoundingSphere } from "babylonjs/Culling/boundingSphere";
    import { ICullable } from "babylonjs/Culling/boundingInfo";
    import { Plane } from "babylonjs/Maths/math.plane";
    /**
     * Class used to store bounding box information
     */
    export class BoundingBox implements ICullable {
        /**
         * Gets the 8 vectors representing the bounding box in local space
         */
        readonly vectors: Vector3[];
        /**
         * Gets the center of the bounding box in local space
         */
        readonly center: Vector3;
        /**
         * Gets the center of the bounding box in world space
         */
        readonly centerWorld: Vector3;
        /**
         * Gets the extend size in local space
         */
        readonly extendSize: Vector3;
        /**
         * Gets the extend size in world space
         */
        readonly extendSizeWorld: Vector3;
        /**
         * Gets the OBB (object bounding box) directions
         */
        readonly directions: Vector3[];
        /**
         * Gets the 8 vectors representing the bounding box in world space
         */
        readonly vectorsWorld: Vector3[];
        /**
         * Gets the minimum vector in world space
         */
        readonly minimumWorld: Vector3;
        /**
         * Gets the maximum vector in world space
         */
        readonly maximumWorld: Vector3;
        /**
         * Gets the minimum vector in local space
         */
        readonly minimum: Vector3;
        /**
         * Gets the maximum vector in local space
         */
        readonly maximum: Vector3;
        private _worldMatrix;
        private static readonly TmpVector3;
        /**
         * @hidden
         */
        _tag: number;
        /**
         * Creates a new bounding box
         * @param min defines the minimum vector (in local space)
         * @param max defines the maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        constructor(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>);
        /**
         * Recreates the entire bounding box from scratch as if we call the constructor in place
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        reConstruct(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>): void;
        /**
         * Scale the current bounding box by applying a scale factor
         * @param factor defines the scale factor to apply
         * @returns the current bounding box
         */
        scale(factor: number): BoundingBox;
        /**
         * Gets the world matrix of the bounding box
         * @returns a matrix
         */
        getWorldMatrix(): DeepImmutable<Matrix>;
        /** @hidden */
        _update(world: DeepImmutable<Matrix>): void;
        /**
         * Tests if the bounding box is intersecting the frustum planes
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an intersection
         */
        isInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if the bounding box is entirely inside the frustum planes
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an inclusion
         */
        isCompletelyInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if a point is inside the bounding box
         * @param point defines the point to test
         * @returns true if the point is inside the bounding box
         */
        intersectsPoint(point: DeepImmutable<Vector3>): boolean;
        /**
         * Tests if the bounding box intersects with a bounding sphere
         * @param sphere defines the sphere to test
         * @returns true if there is an intersection
         */
        intersectsSphere(sphere: DeepImmutable<BoundingSphere>): boolean;
        /**
         * Tests if the bounding box intersects with a box defined by a min and max vectors
         * @param min defines the min vector to use
         * @param max defines the max vector to use
         * @returns true if there is an intersection
         */
        intersectsMinMax(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>): boolean;
        /**
         * Tests if two bounding boxes are intersections
         * @param box0 defines the first box to test
         * @param box1 defines the second box to test
         * @returns true if there is an intersection
         */
        static Intersects(box0: DeepImmutable<BoundingBox>, box1: DeepImmutable<BoundingBox>): boolean;
        /**
         * Tests if a bounding box defines by a min/max vectors intersects a sphere
         * @param minPoint defines the minimum vector of the bounding box
         * @param maxPoint defines the maximum vector of the bounding box
         * @param sphereCenter defines the sphere center
         * @param sphereRadius defines the sphere radius
         * @returns true if there is an intersection
         */
        static IntersectsSphere(minPoint: DeepImmutable<Vector3>, maxPoint: DeepImmutable<Vector3>, sphereCenter: DeepImmutable<Vector3>, sphereRadius: number): boolean;
        /**
         * Tests if a bounding box defined with 8 vectors is entirely inside frustum planes
         * @param boundingVectors defines an array of 8 vectors representing a bounding box
         * @param frustumPlanes defines the frustum planes to test
         * @return true if there is an inclusion
         */
        static IsCompletelyInFrustum(boundingVectors: Array<DeepImmutable<Vector3>>, frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if a bounding box defined with 8 vectors intersects frustum planes
         * @param boundingVectors defines an array of 8 vectors representing a bounding box
         * @param frustumPlanes defines the frustum planes to test
         * @return true if there is an intersection
         */
        static IsInFrustum(boundingVectors: Array<DeepImmutable<Vector3>>, frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
    }
}
declare module "babylonjs/Collisions/collider" {
    import { Nullable, IndicesArray } from "babylonjs/types";
    import { Vector3 } from "babylonjs/Maths/math.vector";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Plane } from "babylonjs/Maths/math.plane";
    /** @hidden */
    export class Collider {
        /** Define if a collision was found */
        collisionFound: boolean;
        /**
         * Define last intersection point in local space
         */
        intersectionPoint: Vector3;
        /**
         * Define last collided mesh
         */
        collidedMesh: Nullable<AbstractMesh>;
        private _collisionPoint;
        private _planeIntersectionPoint;
        private _tempVector;
        private _tempVector2;
        private _tempVector3;
        private _tempVector4;
        private _edge;
        private _baseToVertex;
        private _destinationPoint;
        private _slidePlaneNormal;
        private _displacementVector;
        /** @hidden */
        _radius: Vector3;
        /** @hidden */
        _retry: number;
        private _velocity;
        private _basePoint;
        private _epsilon;
        /** @hidden */
        _velocityWorldLength: number;
        /** @hidden */
        _basePointWorld: Vector3;
        private _velocityWorld;
        private _normalizedVelocity;
        /** @hidden */
        _initialVelocity: Vector3;
        /** @hidden */
        _initialPosition: Vector3;
        private _nearestDistance;
        private _collisionMask;
        get collisionMask(): number;
        set collisionMask(mask: number);
        /**
         * Gets the plane normal used to compute the sliding response (in local space)
         */
        get slidePlaneNormal(): Vector3;
        /** @hidden */
        _initialize(source: Vector3, dir: Vector3, e: number): void;
        /** @hidden */
        _checkPointInTriangle(point: Vector3, pa: Vector3, pb: Vector3, pc: Vector3, n: Vector3): boolean;
        /** @hidden */
        _canDoCollision(sphereCenter: Vector3, sphereRadius: number, vecMin: Vector3, vecMax: Vector3): boolean;
        /** @hidden */
        _testTriangle(faceIndex: number, trianglePlaneArray: Array<Plane>, p1: Vector3, p2: Vector3, p3: Vector3, hasMaterial: boolean, hostMesh: AbstractMesh): void;
        /** @hidden */
        _collide(trianglePlaneArray: Array<Plane>, pts: Vector3[], indices: IndicesArray, indexStart: number, indexEnd: number, decal: number, hasMaterial: boolean, hostMesh: AbstractMesh): void;
        /** @hidden */
        _getResponse(pos: Vector3, vel: Vector3): void;
    }
}
declare module "babylonjs/Culling/boundingInfo" {
    import { DeepImmutable } from "babylonjs/types";
    import { Matrix, Vector3 } from "babylonjs/Maths/math.vector";
    import { BoundingBox } from "babylonjs/Culling/boundingBox";
    import { BoundingSphere } from "babylonjs/Culling/boundingSphere";
    import { Plane } from "babylonjs/Maths/math.plane";
    import { Collider } from "babylonjs/Collisions/collider";
    /**
     * Interface for cullable objects
     * @see https://doc.babylonjs.com/babylon101/materials#back-face-culling
     */
    export interface ICullable {
        /**
         * Checks if the object or part of the object is in the frustum
         * @param frustumPlanes Camera near/planes
         * @returns true if the object is in frustum otherwise false
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Checks if a cullable object (mesh...) is in the camera frustum
         * Unlike isInFrustum this cheks the full bounding box
         * @param frustumPlanes Camera near/planes
         * @returns true if the object is in frustum otherwise false
         */
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
    }
    /**
     * Info for a bounding data of a mesh
     */
    export class BoundingInfo implements ICullable {
        /**
         * Bounding box for the mesh
         */
        readonly boundingBox: BoundingBox;
        /**
         * Bounding sphere for the mesh
         */
        readonly boundingSphere: BoundingSphere;
        private _isLocked;
        private static readonly TmpVector3;
        /**
         * Constructs bounding info
         * @param minimum min vector of the bounding box/sphere
         * @param maximum max vector of the bounding box/sphere
         * @param worldMatrix defines the new world matrix
         */
        constructor(minimum: DeepImmutable<Vector3>, maximum: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>);
        /**
         * Recreates the entire bounding info from scratch as if we call the constructor in place
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        reConstruct(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>): void;
        /**
         * min vector of the bounding box/sphere
         */
        get minimum(): Vector3;
        /**
         * max vector of the bounding box/sphere
         */
        get maximum(): Vector3;
        /**
         * If the info is locked and won't be updated to avoid perf overhead
         */
        get isLocked(): boolean;
        set isLocked(value: boolean);
        /**
         * Updates the bounding sphere and box
         * @param world world matrix to be used to update
         */
        update(world: DeepImmutable<Matrix>): void;
        /**
         * Recreate the bounding info to be centered around a specific point given a specific extend.
         * @param center New center of the bounding info
         * @param extend New extend of the bounding info
         * @returns the current bounding info
         */
        centerOn(center: DeepImmutable<Vector3>, extend: DeepImmutable<Vector3>): BoundingInfo;
        /**
         * Scale the current bounding info by applying a scale factor
         * @param factor defines the scale factor to apply
         * @returns the current bounding info
         */
        scale(factor: number): BoundingInfo;
        /**
         * Returns `true` if the bounding info is within the frustum defined by the passed array of planes.
         * @param frustumPlanes defines the frustum to test
         * @param strategy defines the strategy to use for the culling (default is BABYLON.AbstractMesh.CULLINGSTRATEGY_STANDARD)
         * @returns true if the bounding info is in the frustum planes
         */
        isInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>, strategy?: number): boolean;
        /**
         * Gets the world distance between the min and max points of the bounding box
         */
        get diagonalLength(): number;
        /**
         * Checks if a cullable object (mesh...) is in the camera frustum
         * Unlike isInFrustum this cheks the full bounding box
         * @param frustumPlanes Camera near/planes
         * @returns true if the object is in frustum otherwise false
         */
        isCompletelyInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /** @hidden */
        _checkCollision(collider: Collider): boolean;
        /**
         * Checks if a point is inside the bounding box and bounding sphere or the mesh
         * @see https://doc.babylonjs.com/babylon101/intersect_collisions_-_mesh
         * @param point the point to check intersection with
         * @returns if the point intersects
         */
        intersectsPoint(point: DeepImmutable<Vector3>): boolean;
        /**
         * Checks if another bounding info intersects the bounding box and bounding sphere or the mesh
         * @see https://doc.babylonjs.com/babylon101/intersect_collisions_-_mesh
         * @param boundingInfo the bounding info to check intersection with
         * @param precise if the intersection should be done using OBB
         * @returns if the bounding info intersects
         */
        intersects(boundingInfo: DeepImmutable<BoundingInfo>, precise: boolean): boolean;
    }
}
declare module "babylonjs/Maths/math.functions" {
    import { FloatArray, Nullable, IndicesArray } from "babylonjs/types";
    import { Vector2, Vector3 } from "babylonjs/Maths/math.vector";
    /**
     * Extracts minimum and maximum values from a list of indexed positions
     * @param positions defines the positions to use
     * @param indices defines the indices to the positions
     * @param indexStart defines the start index
     * @param indexCount defines the end index
     * @param bias defines bias value to add to the result
     * @return minimum and maximum values
     */
    export function extractMinAndMaxIndexed(positions: FloatArray, indices: IndicesArray, indexStart: number, indexCount: number, bias?: Nullable<Vector2>): {
        minimum: Vector3;
        maximum: Vector3;
    };
    /**
     * Extracts minimum and maximum values from a list of positions
     * @param positions defines the positions to use
     * @param start defines the start index in the positions array
     * @param count defines the number of positions to handle
     * @param bias defines bias value to add to the result
     * @param stride defines the stride size to use (distance between two positions in the positions array)
     * @return minimum and maximum values
     */
    export function extractMinAndMax(positions: FloatArray, start: number, count: number, bias?: Nullable<Vector2>, stride?: number): {
        minimum: Vector3;
        maximum: Vector3;
    };
}
declare module "babylonjs/Misc/guid" {
    /**
     * Class used to manipulate GUIDs
     */
    export class GUID {
        /**
         * Implementation from http://stackoverflow.com/questions/105034/how-to-create-a-guid-uuid-in-javascript/2117523#answer-2117523
         * Be aware Math.random() could cause collisions, but:
         * "All but 6 of the 128 bits of the ID are randomly generated, which means that for any two ids, there's a 1 in 2^^122 (or 5.3x10^^36) chance they'll collide"
         * @returns a pseudo random id
         */
        static RandomId(): string;
    }
}
declare module "babylonjs/Materials/Textures/baseTexture" {
    import { Observable } from "babylonjs/Misc/observable";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import "babylonjs/Misc/fileTools";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    import { ThinTexture } from "babylonjs/Materials/Textures/thinTexture";
    /**
     * Base class of all the textures in babylon.
     * It groups all the common properties the materials, post process, lights... might need
     * in order to make a correct use of the texture.
     */
    export class BaseTexture extends ThinTexture implements IAnimatable {
        /**
         * Default anisotropic filtering level for the application.
         * It is set to 4 as a good tradeoff between perf and quality.
         */
        static DEFAULT_ANISOTROPIC_FILTERING_LEVEL: number;
        /**
         * Gets or sets the unique id of the texture
         */
        uniqueId: number;
        /**
         * Define the name of the texture.
         */
        name: string;
        /**
         * Gets or sets an object used to store user defined information.
         */
        metadata: any;
        /**
         * For internal use only. Please do not use.
         */
        reservedDataStore: any;
        private _hasAlpha;
        /**
         * Define if the texture is having a usable alpha value (can be use for transparency or glossiness for instance).
         */
        set hasAlpha(value: boolean);
        get hasAlpha(): boolean;
        /**
         * Defines if the alpha value should be determined via the rgb values.
         * If true the luminance of the pixel might be used to find the corresponding alpha value.
         */
        getAlphaFromRGB: boolean;
        /**
         * Intensity or strength of the texture.
         * It is commonly used by materials to fine tune the intensity of the texture
         */
        level: number;
        /**
         * Define the UV chanel to use starting from 0 and defaulting to 0.
         * This is part of the texture as textures usually maps to one uv set.
         */
        coordinatesIndex: number;
        protected _coordinatesMode: number;
        /**
        * How a texture is mapped.
        *
        * | Value | Type                                | Description |
        * | ----- | ----------------------------------- | ----------- |
        * | 0     | EXPLICIT_MODE                       |             |
        * | 1     | SPHERICAL_MODE                      |             |
        * | 2     | PLANAR_MODE                         |             |
        * | 3     | CUBIC_MODE                          |             |
        * | 4     | PROJECTION_MODE                     |             |
        * | 5     | SKYBOX_MODE                         |             |
        * | 6     | INVCUBIC_MODE                       |             |
        * | 7     | EQUIRECTANGULAR_MODE                |             |
        * | 8     | FIXED_EQUIRECTANGULAR_MODE          |             |
        * | 9     | FIXED_EQUIRECTANGULAR_MIRRORED_MODE |             |
        */
        set coordinatesMode(value: number);
        get coordinatesMode(): number;
        /**
         * | Value | Type               | Description |
         * | ----- | ------------------ | ----------- |
         * | 0     | CLAMP_ADDRESSMODE  |             |
         * | 1     | WRAP_ADDRESSMODE   |             |
         * | 2     | MIRROR_ADDRESSMODE |             |
         */
        get wrapU(): number;
        set wrapU(value: number);
        /**
         * | Value | Type               | Description |
         * | ----- | ------------------ | ----------- |
         * | 0     | CLAMP_ADDRESSMODE  |             |
         * | 1     | WRAP_ADDRESSMODE   |             |
         * | 2     | MIRROR_ADDRESSMODE |             |
         */
        get wrapV(): number;
        set wrapV(value: number);
        /**
        * | Value | Type               | Description |
        * | ----- | ------------------ | ----------- |
        * | 0     | CLAMP_ADDRESSMODE  |             |
        * | 1     | WRAP_ADDRESSMODE   |             |
        * | 2     | MIRROR_ADDRESSMODE |             |
        */
        wrapR: number;
        /**
         * With compliant hardware and browser (supporting anisotropic filtering)
         * this defines the level of anisotropic filtering in the texture.
         * The higher the better but the slower. This defaults to 4 as it seems to be the best tradeoff.
         */
        anisotropicFilteringLevel: number;
        private _isCube;
        /**
         * Define if the texture is a cube texture or if false a 2d texture.
         */
        get isCube(): boolean;
        set isCube(value: boolean);
        /**
         * Define if the texture is a 3d texture (webgl 2) or if false a 2d texture.
         */
        get is3D(): boolean;
        set is3D(value: boolean);
        /**
         * Define if the texture is a 2d array texture (webgl 2) or if false a 2d texture.
         */
        get is2DArray(): boolean;
        set is2DArray(value: boolean);
        private _gammaSpace;
        /**
         * Define if the texture contains data in gamma space (most of the png/jpg aside bump).
         * HDR texture are usually stored in linear space.
         * This only impacts the PBR and Background materials
         */
        get gammaSpace(): boolean;
        set gammaSpace(gamma: boolean);
        /**
         * Gets or sets whether or not the texture contains RGBD data.
         */
        get isRGBD(): boolean;
        set isRGBD(value: boolean);
        /**
         * Is Z inverted in the texture (useful in a cube texture).
         */
        invertZ: boolean;
        /**
         * Are mip maps generated for this texture or not.
         */
        get noMipmap(): boolean;
        /**
         * @hidden
         */
        lodLevelInAlpha: boolean;
        /**
         * With prefiltered texture, defined the offset used during the prefiltering steps.
         */
        get lodGenerationOffset(): number;
        set lodGenerationOffset(value: number);
        /**
         * With prefiltered texture, defined the scale used during the prefiltering steps.
         */
        get lodGenerationScale(): number;
        set lodGenerationScale(value: number);
        /**
         * With prefiltered texture, defined if the specular generation is based on a linear ramp.
         * By default we are using a log2 of the linear roughness helping to keep a better resolution for
         * average roughness values.
         */
        get linearSpecularLOD(): boolean;
        set linearSpecularLOD(value: boolean);
        /**
         * In case a better definition than spherical harmonics is required for the diffuse part of the environment.
         * You can set the irradiance texture to rely on a texture instead of the spherical approach.
         * This texture need to have the same characteristics than its parent (Cube vs 2d, coordinates mode, Gamma/Linear, RGBD).
         */
        get irradianceTexture(): Nullable<BaseTexture>;
        set irradianceTexture(value: Nullable<BaseTexture>);
        /**
         * Define if the texture is a render target.
         */
        isRenderTarget: boolean;
        /**
         * Define the unique id of the texture in the scene.
         */
        get uid(): string;
        /** @hidden */
        _prefiltered: boolean;
        /**
         * Return a string representation of the texture.
         * @returns the texture as a string
         */
        toString(): string;
        /**
         * Get the class name of the texture.
         * @returns "BaseTexture"
         */
        getClassName(): string;
        /**
         * Define the list of animation attached to the texture.
         */
        animations: import("babylonjs/Animations/animation").Animation[];
        /**
        * An event triggered when the texture is disposed.
        */
        onDisposeObservable: Observable<BaseTexture>;
        private _onDisposeObserver;
        /**
         * Callback triggered when the texture has been disposed.
         * Kept for back compatibility, you can use the onDisposeObservable instead.
         */
        set onDispose(callback: () => void);
        protected _scene: Nullable<Scene>;
        /** @hidden */
        _texture: Nullable<InternalTexture>;
        private _uid;
        /**
         * Define if the texture is preventinga material to render or not.
         * If not and the texture is not ready, the engine will use a default black texture instead.
         */
        get isBlocking(): boolean;
        /**
         * Instantiates a new BaseTexture.
         * Base class of all the textures in babylon.
         * It groups all the common properties the materials, post process, lights... might need
         * in order to make a correct use of the texture.
         * @param sceneOrEngine Define the scene or engine the texture blongs to
         */
        constructor(sceneOrEngine: Nullable<Scene | ThinEngine>);
        /**
         * Get the scene the texture belongs to.
         * @returns the scene or null if undefined
         */
        getScene(): Nullable<Scene>;
        /** @hidden */
        protected _getEngine(): Nullable<ThinEngine>;
        /**
         * Checks if the texture has the same transform matrix than another texture
         * @param texture texture to check against
         * @returns true if the transforms are the same, else false
         */
        checkTransformsAreIdentical(texture: Nullable<BaseTexture>): boolean;
        /**
         * Get the texture transform matrix used to offset tile the texture for istance.
         * @returns the transformation matrix
         */
        getTextureMatrix(): Matrix;
        /**
         * Get the texture reflection matrix used to rotate/transform the reflection.
         * @returns the reflection matrix
         */
        getReflectionTextureMatrix(): Matrix;
        /**
         * Get if the texture is ready to be consumed (either it is ready or it is not blocking)
         * @returns true if ready or not blocking
         */
        isReadyOrNotBlocking(): boolean;
        /**
         * Scales the texture if is `canRescale()`
         * @param ratio the resize factor we want to use to rescale
         */
        scale(ratio: number): void;
        /**
         * Get if the texture can rescale.
         */
        get canRescale(): boolean;
        /** @hidden */
        _getFromCache(url: Nullable<string>, noMipmap: boolean, sampling?: number, invertY?: boolean): Nullable<InternalTexture>;
        /** @hidden */
        _rebuild(): void;
        /**
         * Clones the texture.
         * @returns the cloned texture
         */
        clone(): Nullable<BaseTexture>;
        /**
         * Get the texture underlying type (INT, FLOAT...)
         */
        get textureType(): number;
        /**
         * Get the texture underlying format (RGB, RGBA...)
         */
        get textureFormat(): number;
        /**
         * Indicates that textures need to be re-calculated for all materials
         */
        protected _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Reads the pixels stored in the webgl texture and returns them as an ArrayBuffer.
         * This will returns an RGBA array buffer containing either in values (0-255) or
         * float values (0-1) depending of the underlying buffer type.
         * @param faceIndex defines the face of the texture to read (in case of cube texture)
         * @param level defines the LOD level of the texture to read (in case of Mip Maps)
         * @param buffer defines a user defined buffer to fill with data (can be null)
         * @returns The Array buffer containing the pixels data.
         */
        readPixels(faceIndex?: number, level?: number, buffer?: Nullable<ArrayBufferView>): Nullable<ArrayBufferView>;
        /** @hidden */
        get _lodTextureHigh(): Nullable<BaseTexture>;
        /** @hidden */
        get _lodTextureMid(): Nullable<BaseTexture>;
        /** @hidden */
        get _lodTextureLow(): Nullable<BaseTexture>;
        /**
         * Dispose the texture and release its associated resources.
         */
        dispose(): void;
        /**
         * Serialize the texture into a JSON representation that can be parsed later on.
         * @returns the JSON representation of the texture
         */
        serialize(): any;
        /**
         * Helper function to be called back once a list of texture contains only ready textures.
         * @param textures Define the list of textures to wait for
         * @param callback Define the callback triggered once the entire list will be ready
         */
        static WhenAllReady(textures: BaseTexture[], callback: () => void): void;
        private static _isScene;
    }
}
declare module "babylonjs/Meshes/WebGL/webGLDataBuffer" {
    import { DataBuffer } from "babylonjs/Meshes/dataBuffer";
    /** @hidden */
    export class WebGLDataBuffer extends DataBuffer {
        private _buffer;
        constructor(resource: WebGLBuffer);
        get underlyingResource(): any;
    }
}
declare module "babylonjs/Engines/WebGL/webGLPipelineContext" {
    import { IPipelineContext } from "babylonjs/Engines/IPipelineContext";
    import { Nullable } from "babylonjs/types";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    /** @hidden */
    export class WebGLPipelineContext implements IPipelineContext {
        engine: ThinEngine;
        program: Nullable<WebGLProgram>;
        context?: WebGLRenderingContext;
        vertexShader?: WebGLShader;
        fragmentShader?: WebGLShader;
        isParallelCompiled: boolean;
        onCompiled?: () => void;
        transformFeedback?: WebGLTransformFeedback | null;
        vertexCompilationError: Nullable<string>;
        fragmentCompilationError: Nullable<string>;
        programLinkError: Nullable<string>;
        programValidationError: Nullable<string>;
        get isAsync(): boolean;
        get isReady(): boolean;
        _handlesSpectorRebuildCallback(onCompiled: (program: WebGLProgram) => void): void;
        _getVertexShaderCode(): string | null;
        _getFragmentShaderCode(): string | null;
    }
}
declare module "babylonjs/Engines/Extensions/engine.uniformBuffer" {
    import { FloatArray, Nullable } from "babylonjs/types";
    import { DataBuffer } from "babylonjs/Meshes/dataBuffer";
    import { IPipelineContext } from "babylonjs/Engines/IPipelineContext";
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Create an uniform buffer
             * @see https://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
             * @param elements defines the content of the uniform buffer
             * @returns the webGL uniform buffer
             */
            createUniformBuffer(elements: FloatArray): DataBuffer;
            /**
             * Create a dynamic uniform buffer
             * @see https://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
             * @param elements defines the content of the uniform buffer
             * @returns the webGL uniform buffer
             */
            createDynamicUniformBuffer(elements: FloatArray): DataBuffer;
            /**
             * Update an existing uniform buffer
             * @see https://doc.babylonjs.com/features/webgl2#uniform-buffer-objets
             * @param uniformBuffer defines the target uniform buffer
             * @param elements defines the content to update
             * @param offset defines the offset in the uniform buffer where update should start
             * @param count defines the size of the data to update
             */
            updateUniformBuffer(uniformBuffer: DataBuffer, elements: FloatArray, offset?: number, count?: number): void;
            /**
             * Bind an uniform buffer to the current webGL context
             * @param buffer defines the buffer to bind
             */
            bindUniformBuffer(buffer: Nullable<DataBuffer>): void;
            /**
             * Bind a buffer to the current webGL context at a given location
             * @param buffer defines the buffer to bind
             * @param location defines the index where to bind the buffer
             */
            bindUniformBufferBase(buffer: DataBuffer, location: number): void;
            /**
             * Bind a specific block at a given index in a specific shader program
             * @param pipelineContext defines the pipeline context to use
             * @param blockName defines the block name
             * @param index defines the index where to bind the block
             */
            bindUniformBlock(pipelineContext: IPipelineContext, blockName: string, index: number): void;
        }
    }
}
declare module "babylonjs/Materials/uniformBuffer" {
    import { Nullable, FloatArray } from "babylonjs/types";
    import { Matrix, Vector3, Vector4 } from "babylonjs/Maths/math.vector";
    import { Engine } from "babylonjs/Engines/engine";
    import { Effect } from "babylonjs/Materials/effect";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { DataBuffer } from "babylonjs/Meshes/dataBuffer";
    import { Color3 } from "babylonjs/Maths/math.color";
    import "babylonjs/Engines/Extensions/engine.uniformBuffer";
    /**
     * Uniform buffer objects.
     *
     * Handles blocks of uniform on the GPU.
     *
     * If WebGL 2 is not available, this class falls back on traditionnal setUniformXXX calls.
     *
     * For more information, please refer to :
     * https://www.khronos.org/opengl/wiki/Uniform_Buffer_Object
     */
    export class UniformBuffer {
        private _engine;
        private _buffer;
        private _data;
        private _bufferData;
        private _dynamic?;
        private _uniformLocations;
        private _uniformSizes;
        private _uniformLocationPointer;
        private _needSync;
        private _noUBO;
        private _currentEffect;
        /** @hidden */
        _alreadyBound: boolean;
        private static _MAX_UNIFORM_SIZE;
        private static _tempBuffer;
        /**
         * Lambda to Update a 3x3 Matrix in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateMatrix3x3: (name: string, matrix: Float32Array) => void;
        /**
         * Lambda to Update a 2x2 Matrix in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateMatrix2x2: (name: string, matrix: Float32Array) => void;
        /**
         * Lambda to Update a single float in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateFloat: (name: string, x: number) => void;
        /**
         * Lambda to Update a vec2 of float in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateFloat2: (name: string, x: number, y: number, suffix?: string) => void;
        /**
         * Lambda to Update a vec3 of float in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateFloat3: (name: string, x: number, y: number, z: number, suffix?: string) => void;
        /**
         * Lambda to Update a vec4 of float in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateFloat4: (name: string, x: number, y: number, z: number, w: number, suffix?: string) => void;
        /**
         * Lambda to Update a 4x4 Matrix in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateMatrix: (name: string, mat: Matrix) => void;
        /**
         * Lambda to Update vec3 of float from a Vector in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateVector3: (name: string, vector: Vector3) => void;
        /**
         * Lambda to Update vec4 of float from a Vector in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateVector4: (name: string, vector: Vector4) => void;
        /**
         * Lambda to Update vec3 of float from a Color in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateColor3: (name: string, color: Color3, suffix?: string) => void;
        /**
         * Lambda to Update vec4 of float from a Color in a uniform buffer.
         * This is dynamic to allow compat with webgl 1 and 2.
         * You will need to pass the name of the uniform as well as the value.
         */
        updateColor4: (name: string, color: Color3, alpha: number, suffix?: string) => void;
        /**
         * Instantiates a new Uniform buffer objects.
         *
         * Handles blocks of uniform on the GPU.
         *
         * If WebGL 2 is not available, this class falls back on traditionnal setUniformXXX calls.
         *
         * For more information, please refer to :
         * @see https://www.khronos.org/opengl/wiki/Uniform_Buffer_Object
         * @param engine Define the engine the buffer is associated with
         * @param data Define the data contained in the buffer
         * @param dynamic Define if the buffer is updatable
         */
        constructor(engine: Engine, data?: number[], dynamic?: boolean);
        /**
         * Indicates if the buffer is using the WebGL2 UBO implementation,
         * or just falling back on setUniformXXX calls.
         */
        get useUbo(): boolean;
        /**
         * Indicates if the WebGL underlying uniform buffer is in sync
         * with the javascript cache data.
         */
        get isSync(): boolean;
        /**
         * Indicates if the WebGL underlying uniform buffer is dynamic.
         * Also, a dynamic UniformBuffer will disable cache verification and always
         * update the underlying WebGL uniform buffer to the GPU.
         * @returns if Dynamic, otherwise false
         */
        isDynamic(): boolean;
        /**
         * The data cache on JS side.
         * @returns the underlying data as a float array
         */
        getData(): Float32Array;
        /**
         * The underlying WebGL Uniform buffer.
         * @returns the webgl buffer
         */
        getBuffer(): Nullable<DataBuffer>;
        /**
         * std140 layout specifies how to align data within an UBO structure.
         * See https://khronos.org/registry/OpenGL/specs/gl/glspec45.core.pdf#page=159
         * for specs.
         */
        private _fillAlignment;
        /**
         * Adds an uniform in the buffer.
         * Warning : the subsequents calls of this function must be in the same order as declared in the shader
         * for the layout to be correct !
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param size Data size, or data directly.
         */
        addUniform(name: string, size: number | number[]): void;
        /**
         * Adds a Matrix 4x4 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param mat A 4x4 matrix.
         */
        addMatrix(name: string, mat: Matrix): void;
        /**
         * Adds a vec2 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param x Define the x component value of the vec2
         * @param y Define the y component value of the vec2
         */
        addFloat2(name: string, x: number, y: number): void;
        /**
         * Adds a vec3 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param x Define the x component value of the vec3
         * @param y Define the y component value of the vec3
         * @param z Define the z component value of the vec3
         */
        addFloat3(name: string, x: number, y: number, z: number): void;
        /**
         * Adds a vec3 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param color Define the vec3 from a Color
         */
        addColor3(name: string, color: Color3): void;
        /**
         * Adds a vec4 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param color Define the rgb components from a Color
         * @param alpha Define the a component of the vec4
         */
        addColor4(name: string, color: Color3, alpha: number): void;
        /**
         * Adds a vec3 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         * @param vector Define the vec3 components from a Vector
         */
        addVector3(name: string, vector: Vector3): void;
        /**
         * Adds a Matrix 3x3 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         */
        addMatrix3x3(name: string): void;
        /**
         * Adds a Matrix 2x2 to the uniform buffer.
         * @param name Name of the uniform, as used in the uniform block in the shader.
         */
        addMatrix2x2(name: string): void;
        /**
         * Effectively creates the WebGL Uniform Buffer, once layout is completed with `addUniform`.
         */
        create(): void;
        /** @hidden */
        _rebuild(): void;
        /**
         * Updates the WebGL Uniform Buffer on the GPU.
         * If the `dynamic` flag is set to true, no cache comparison is done.
         * Otherwise, the buffer will be updated only if the cache differs.
         */
        update(): void;
        /**
         * Updates the value of an uniform. The `update` method must be called afterwards to make it effective in the GPU.
         * @param uniformName Define the name of the uniform, as used in the uniform block in the shader.
         * @param data Define the flattened data
         * @param size Define the size of the data.
         */
        updateUniform(uniformName: string, data: FloatArray, size: number): void;
        private _valueCache;
        private _cacheMatrix;
        private _updateMatrix3x3ForUniform;
        private _updateMatrix3x3ForEffect;
        private _updateMatrix2x2ForEffect;
        private _updateMatrix2x2ForUniform;
        private _updateFloatForEffect;
        private _updateFloatForUniform;
        private _updateFloat2ForEffect;
        private _updateFloat2ForUniform;
        private _updateFloat3ForEffect;
        private _updateFloat3ForUniform;
        private _updateFloat4ForEffect;
        private _updateFloat4ForUniform;
        private _updateMatrixForEffect;
        private _updateMatrixForUniform;
        private _updateVector3ForEffect;
        private _updateVector3ForUniform;
        private _updateVector4ForEffect;
        private _updateVector4ForUniform;
        private _updateColor3ForEffect;
        private _updateColor3ForUniform;
        private _updateColor4ForEffect;
        private _updateColor4ForUniform;
        /**
         * Sets a sampler uniform on the effect.
         * @param name Define the name of the sampler.
         * @param texture Define the texture to set in the sampler
         */
        setTexture(name: string, texture: Nullable<BaseTexture>): void;
        /**
         * Directly updates the value of the uniform in the cache AND on the GPU.
         * @param uniformName Define the name of the uniform, as used in the uniform block in the shader.
         * @param data Define the flattened data
         */
        updateUniformDirectly(uniformName: string, data: FloatArray): void;
        /**
         * Binds this uniform buffer to an effect.
         * @param effect Define the effect to bind the buffer to
         * @param name Name of the uniform block in the shader.
         */
        bindToEffect(effect: Effect, name: string): void;
        /**
         * Disposes the uniform buffer.
         */
        dispose(): void;
    }
}
declare module "babylonjs/Materials/materialDefines" {
    /**
     * Manages the defines for the Material
     */
    export class MaterialDefines {
        /** @hidden */
        protected _keys: string[];
        private _isDirty;
        /** @hidden */
        _renderId: number;
        /** @hidden */
        _areLightsDirty: boolean;
        /** @hidden */
        _areLightsDisposed: boolean;
        /** @hidden */
        _areAttributesDirty: boolean;
        /** @hidden */
        _areTexturesDirty: boolean;
        /** @hidden */
        _areFresnelDirty: boolean;
        /** @hidden */
        _areMiscDirty: boolean;
        /** @hidden */
        _arePrePassDirty: boolean;
        /** @hidden */
        _areImageProcessingDirty: boolean;
        /** @hidden */
        _normals: boolean;
        /** @hidden */
        _uvs: boolean;
        /** @hidden */
        _needNormals: boolean;
        /** @hidden */
        _needUVs: boolean;
        [id: string]: any;
        /**
         * Specifies if the material needs to be re-calculated
         */
        get isDirty(): boolean;
        /**
         * Marks the material to indicate that it has been re-calculated
         */
        markAsProcessed(): void;
        /**
         * Marks the material to indicate that it needs to be re-calculated
         */
        markAsUnprocessed(): void;
        /**
         * Marks the material to indicate all of its defines need to be re-calculated
         */
        markAllAsDirty(): void;
        /**
         * Marks the material to indicate that image processing needs to be re-calculated
         */
        markAsImageProcessingDirty(): void;
        /**
         * Marks the material to indicate the lights need to be re-calculated
         * @param disposed Defines whether the light is dirty due to dispose or not
         */
        markAsLightDirty(disposed?: boolean): void;
        /**
         * Marks the attribute state as changed
         */
        markAsAttributesDirty(): void;
        /**
         * Marks the texture state as changed
         */
        markAsTexturesDirty(): void;
        /**
         * Marks the fresnel state as changed
         */
        markAsFresnelDirty(): void;
        /**
         * Marks the misc state as changed
         */
        markAsMiscDirty(): void;
        /**
         * Marks the prepass state as changed
         */
        markAsPrePassDirty(): void;
        /**
         * Rebuilds the material defines
         */
        rebuild(): void;
        /**
         * Specifies if two material defines are equal
         * @param other - A material define instance to compare to
         * @returns - Boolean indicating if the material defines are equal (true) or not (false)
         */
        isEqual(other: MaterialDefines): boolean;
        /**
         * Clones this instance's defines to another instance
         * @param other - material defines to clone values to
         */
        cloneTo(other: MaterialDefines): void;
        /**
         * Resets the material define values
         */
        reset(): void;
        /**
         * Converts the material define values to a string
         * @returns - String of material define information
         */
        toString(): string;
    }
}
declare module "babylonjs/Misc/iInspectable" {
    /**
     * Enum that determines the text-wrapping mode to use.
     */
    export enum InspectableType {
        /**
         * Checkbox for booleans
         */
        Checkbox = 0,
        /**
         * Sliders for numbers
         */
        Slider = 1,
        /**
         * Vector3
         */
        Vector3 = 2,
        /**
         * Quaternions
         */
        Quaternion = 3,
        /**
         * Color3
         */
        Color3 = 4,
        /**
         * String
         */
        String = 5
    }
    /**
     * Interface used to define custom inspectable properties.
     * This interface is used by the inspector to display custom property grids
     * @see https://doc.babylonjs.com/how_to/debug_layer#extensibility
     */
    export interface IInspectable {
        /**
         * Gets the label to display
         */
        label: string;
        /**
         * Gets the name of the property to edit
         */
        propertyName: string;
        /**
         * Gets the type of the editor to use
         */
        type: InspectableType;
        /**
         * Gets the minimum value of the property when using in "slider" mode
         */
        min?: number;
        /**
         * Gets the maximum value of the property when using in "slider" mode
         */
        max?: number;
        /**
         * Gets the setp to use when using in "slider" mode
         */
        step?: number;
    }
}
declare module "babylonjs/Lights/light" {
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Vector3 } from "babylonjs/Maths/math.vector";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { Node } from "babylonjs/node";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Effect } from "babylonjs/Materials/effect";
    import { UniformBuffer } from "babylonjs/Materials/uniformBuffer";
    import { IShadowGenerator } from "babylonjs/Lights/Shadows/shadowGenerator";
    /**
     * Base class of all the lights in Babylon. It groups all the generic information about lights.
     * Lights are used, as you would expect, to affect how meshes are seen, in terms of both illumination and colour.
     * All meshes allow light to pass through them unless shadow generation is activated. The default number of lights allowed is four but this can be increased.
     */
    export abstract class Light extends Node {
        /**
         * Falloff Default: light is falling off following the material specification:
         * standard material is using standard falloff whereas pbr material can request special falloff per materials.
         */
        static readonly FALLOFF_DEFAULT: number;
        /**
         * Falloff Physical: light is falling off following the inverse squared distance law.
         */
        static readonly FALLOFF_PHYSICAL: number;
        /**
         * Falloff gltf: light is falling off as described in the gltf moving to PBR document
         * to enhance interoperability with other engines.
         */
        static readonly FALLOFF_GLTF: number;
        /**
         * Falloff Standard: light is falling off like in the standard material
         * to enhance interoperability with other materials.
         */
        static readonly FALLOFF_STANDARD: number;
        /**
         * If every light affecting the material is in this lightmapMode,
         * material.lightmapTexture adds or multiplies
         * (depends on material.useLightmapAsShadowmap)
         * after every other light calculations.
         */
        static readonly LIGHTMAP_DEFAULT: number;
        /**
         * material.lightmapTexture as only diffuse lighting from this light
         * adds only specular lighting from this light
         * adds dynamic shadows
         */
        static readonly LIGHTMAP_SPECULAR: number;
        /**
         * material.lightmapTexture as only lighting
         * no light calculation from this light
         * only adds dynamic shadows from this light
         */
        static readonly LIGHTMAP_SHADOWSONLY: number;
        /**
         * Each light type uses the default quantity according to its type:
         *      point/spot lights use luminous intensity
         *      directional lights use illuminance
         */
        static readonly INTENSITYMODE_AUTOMATIC: number;
        /**
         * lumen (lm)
         */
        static readonly INTENSITYMODE_LUMINOUSPOWER: number;
        /**
         * candela (lm/sr)
         */
        static readonly INTENSITYMODE_LUMINOUSINTENSITY: number;
        /**
         * lux (lm/m^2)
         */
        static readonly INTENSITYMODE_ILLUMINANCE: number;
        /**
         * nit (cd/m^2)
         */
        static readonly INTENSITYMODE_LUMINANCE: number;
        /**
         * Light type const id of the point light.
         */
        static readonly LIGHTTYPEID_POINTLIGHT: number;
        /**
         * Light type const id of the directional light.
         */
        static readonly LIGHTTYPEID_DIRECTIONALLIGHT: number;
        /**
         * Light type const id of the spot light.
         */
        static readonly LIGHTTYPEID_SPOTLIGHT: number;
        /**
         * Light type const id of the hemispheric light.
         */
        static readonly LIGHTTYPEID_HEMISPHERICLIGHT: number;
        /**
         * Diffuse gives the basic color to an object.
         */
        diffuse: Color3;
        /**
         * Specular produces a highlight color on an object.
         * Note: This is note affecting PBR materials.
         */
        specular: Color3;
        /**
         * Defines the falloff type for this light. This lets overrriding how punctual light are
         * falling off base on range or angle.
         * This can be set to any values in Light.FALLOFF_x.
         *
         * Note: This is only useful for PBR Materials at the moment. This could be extended if required to
         * other types of materials.
         */
        falloffType: number;
        /**
         * Strength of the light.
         * Note: By default it is define in the framework own unit.
         * Note: In PBR materials the intensityMode can be use to chose what unit the intensity is defined in.
         */
        intensity: number;
        private _range;
        protected _inverseSquaredRange: number;
        /**
         * Defines how far from the source the light is impacting in scene units.
         * Note: Unused in PBR material as the distance light falloff is defined following the inverse squared falloff.
         */
        get range(): number;
        /**
         * Defines how far from the source the light is impacting in scene units.
         * Note: Unused in PBR material as the distance light falloff is defined following the inverse squared falloff.
         */
        set range(value: number);
        /**
         * Cached photometric scale default to 1.0 as the automatic intensity mode defaults to 1.0 for every type
         * of light.
         */
        private _photometricScale;
        private _intensityMode;
        /**
         * Gets the photometric scale used to interpret the intensity.
         * This is only relevant with PBR Materials where the light intensity can be defined in a physical way.
         */
        get intensityMode(): number;
        /**
         * Sets the photometric scale used to interpret the intensity.
         * This is only relevant with PBR Materials where the light intensity can be defined in a physical way.
         */
        set intensityMode(value: number);
        private _radius;
        /**
         * Gets the light radius used by PBR Materials to simulate soft area lights.
         */
        get radius(): number;
        /**
         * sets the light radius used by PBR Materials to simulate soft area lights.
         */
        set radius(value: number);
        private _renderPriority;
        /**
         * Defines the rendering priority of the lights. It can help in case of fallback or number of lights
         * exceeding the number allowed of the materials.
         */
        renderPriority: number;
        private _shadowEnabled;
        /**
         * Gets wether or not the shadows are enabled for this light. This can help turning off/on shadow without detaching
         * the current shadow generator.
         */
        get shadowEnabled(): boolean;
        /**
         * Sets wether or not the shadows are enabled for this light. This can help turning off/on shadow without detaching
         * the current shadow generator.
         */
        set shadowEnabled(value: boolean);
        private _includedOnlyMeshes;
        /**
         * Gets the only meshes impacted by this light.
         */
        get includedOnlyMeshes(): AbstractMesh[];
        /**
         * Sets the only meshes impacted by this light.
         */
        set includedOnlyMeshes(value: AbstractMesh[]);
        private _excludedMeshes;
        /**
         * Gets the meshes not impacted by this light.
         */
        get excludedMeshes(): AbstractMesh[];
        /**
         * Sets the meshes not impacted by this light.
         */
        set excludedMeshes(value: AbstractMesh[]);
        private _excludeWithLayerMask;
        /**
         * Gets the layer id use to find what meshes are not impacted by the light.
         * Inactive if 0
         */
        get excludeWithLayerMask(): number;
        /**
         * Sets the layer id use to find what meshes are not impacted by the light.
         * Inactive if 0
         */
        set excludeWithLayerMask(value: number);
        private _includeOnlyWithLayerMask;
        /**
         * Gets the layer id use to find what meshes are impacted by the light.
         * Inactive if 0
         */
        get includeOnlyWithLayerMask(): number;
        /**
         * Sets the layer id use to find what meshes are impacted by the light.
         * Inactive if 0
         */
        set includeOnlyWithLayerMask(value: number);
        private _lightmapMode;
        /**
         * Gets the lightmap mode of this light (should be one of the constants defined by Light.LIGHTMAP_x)
         */
        get lightmapMode(): number;
        /**
         * Sets the lightmap mode of this light (should be one of the constants defined by Light.LIGHTMAP_x)
         */
        set lightmapMode(value: number);
        /**
         * Shadow generator associted to the light.
         * @hidden Internal use only.
         */
        _shadowGenerator: Nullable<IShadowGenerator>;
        /**
         * @hidden Internal use only.
         */
        _excludedMeshesIds: string[];
        /**
         * @hidden Internal use only.
         */
        _includedOnlyMeshesIds: string[];
        /**
         * The current light unifom buffer.
         * @hidden Internal use only.
         */
        _uniformBuffer: UniformBuffer;
        /** @hidden */
        _renderId: number;
        /**
         * Creates a Light object in the scene.
         * Documentation : https://doc.babylonjs.com/babylon101/lights
         * @param name The firendly name of the light
         * @param scene The scene the light belongs too
         */
        constructor(name: string, scene: Scene);
        protected abstract _buildUniformLayout(): void;
        /**
         * Sets the passed Effect "effect" with the Light information.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The light
         */
        abstract transferToEffect(effect: Effect, lightIndex: string): Light;
        /**
         * Sets the passed Effect "effect" with the Light textures.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The light
         */
        transferTexturesToEffect(effect: Effect, lightIndex: string): Light;
        /**
         * Binds the lights information from the scene to the effect for the given mesh.
         * @param lightIndex Light index
         * @param scene The scene where the light belongs to
         * @param effect The effect we are binding the data to
         * @param useSpecular Defines if specular is supported
         * @param rebuildInParallel Specifies whether the shader is rebuilding in parallel
         */
        _bindLight(lightIndex: number, scene: Scene, effect: Effect, useSpecular: boolean, rebuildInParallel?: boolean): void;
        /**
         * Sets the passed Effect "effect" with the Light information.
         * @param effect The effect to update
         * @param lightDataUniformName The uniform used to store light data (position or direction)
         * @returns The light
         */
        abstract transferToNodeMaterialEffect(effect: Effect, lightDataUniformName: string): Light;
        /**
         * Returns the string "Light".
         * @returns the class name
         */
        getClassName(): string;
        /** @hidden */
        readonly _isLight: boolean;
        /**
         * Converts the light information to a readable string for debug purpose.
         * @param fullDetails Supports for multiple levels of logging within scene loading
         * @returns the human readable light info
         */
        toString(fullDetails?: boolean): string;
        /** @hidden */
        protected _syncParentEnabledState(): void;
        /**
         * Set the enabled state of this node.
         * @param value - the new enabled state
         */
        setEnabled(value: boolean): void;
        /**
         * Returns the Light associated shadow generator if any.
         * @return the associated shadow generator.
         */
        getShadowGenerator(): Nullable<IShadowGenerator>;
        /**
         * Returns a Vector3, the absolute light position in the World.
         * @returns the world space position of the light
         */
        getAbsolutePosition(): Vector3;
        /**
         * Specifies if the light will affect the passed mesh.
         * @param mesh The mesh to test against the light
         * @return true the mesh is affected otherwise, false.
         */
        canAffectMesh(mesh: AbstractMesh): boolean;
        /**
         * Sort function to order lights for rendering.
         * @param a First Light object to compare to second.
         * @param b Second Light object to compare first.
         * @return -1 to reduce's a's index relative to be, 0 for no change, 1 to increase a's index relative to b.
         */
        static CompareLightsPriority(a: Light, b: Light): number;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Returns the light type ID (integer).
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Returns the intensity scaled by the Photometric Scale according to the light type and intensity mode.
         * @returns the scaled intensity in intensity mode unit
         */
        getScaledIntensity(): number;
        /**
         * Returns a new Light object, named "name", from the current one.
         * @param name The name of the cloned light
         * @param newParent The parent of this light, if it has one
         * @returns the new created light
         */
        clone(name: string, newParent?: Nullable<Node>): Nullable<Light>;
        /**
         * Serializes the current light into a Serialization object.
         * @returns the serialized object.
         */
        serialize(): any;
        /**
         * Creates a new typed light from the passed type (integer) : point light = 0, directional light = 1, spot light = 2, hemispheric light = 3.
         * This new light is named "name" and added to the passed scene.
         * @param type Type according to the types available in Light.LIGHTTYPEID_x
         * @param name The friendly name of the light
         * @param scene The scene the new light will belong to
         * @returns the constructor function
         */
        static GetConstructorFromName(type: number, name: string, scene: Scene): Nullable<() => Light>;
        /**
         * Parses the passed "parsedLight" and returns a new instanced Light from this parsing.
         * @param parsedLight The JSON representation of the light
         * @param scene The scene to create the parsed light in
         * @returns the created light after parsing
         */
        static Parse(parsedLight: any, scene: Scene): Nullable<Light>;
        private _hookArrayForExcluded;
        private _hookArrayForIncludedOnly;
        private _resyncMeshes;
        /**
         * Forces the meshes to update their light related information in their rendering used effects
         * @hidden Internal Use Only
         */
        _markMeshesAsLightDirty(): void;
        /**
         * Recomputes the cached photometric scale if needed.
         */
        private _computePhotometricScale;
        /**
         * Returns the Photometric Scale according to the light type and intensity mode.
         */
        private _getPhotometricScale;
        /**
         * Reorder the light in the scene according to their defined priority.
         * @hidden Internal Use Only
         */
        _reorderLightsInScene(): void;
        /**
         * Prepares the list of defines specific to the light type.
         * @param defines the list of defines
         * @param lightIndex defines the index of the light for the effect
         */
        abstract prepareLightSpecificDefines(defines: any, lightIndex: number): void;
    }
}
declare module "babylonjs/Maths/math.axis" {
    import { Vector3 } from "babylonjs/Maths/math.vector";
    /** Defines supported spaces */
    export enum Space {
        /** Local (object) space */
        LOCAL = 0,
        /** World space */
        WORLD = 1,
        /** Bone space */
        BONE = 2
    }
    /** Defines the 3 main axes */
    export class Axis {
        /** X axis */
        static X: Vector3;
        /** Y axis */
        static Y: Vector3;
        /** Z axis */
        static Z: Vector3;
    }
    /**
     * Defines cartesian components.
     */
    export enum Coordinate {
        /** X axis */
        X = 0,
        /** Y axis */
        Y = 1,
        /** Z axis */
        Z = 2
    }
}
declare module "babylonjs/Lights/shadowLight" {
    import { Camera } from "babylonjs/Cameras/camera";
    import { Scene } from "babylonjs/scene";
    import { Matrix, Vector3 } from "babylonjs/Maths/math.vector";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Light } from "babylonjs/Lights/light";
    /**
     * Interface describing all the common properties and methods a shadow light needs to implement.
     * This helps both the shadow generator and materials to genrate the corresponding shadow maps
     * as well as binding the different shadow properties to the effects.
     */
    export interface IShadowLight extends Light {
        /**
         * The light id in the scene (used in scene.findLighById for instance)
         */
        id: string;
        /**
         * The position the shdow will be casted from.
         */
        position: Vector3;
        /**
         * In 2d mode (needCube being false), the direction used to cast the shadow.
         */
        direction: Vector3;
        /**
         * The transformed position. Position of the light in world space taking parenting in account.
         */
        transformedPosition: Vector3;
        /**
         * The transformed direction. Direction of the light in world space taking parenting in account.
         */
        transformedDirection: Vector3;
        /**
         * The friendly name of the light in the scene.
         */
        name: string;
        /**
         * Defines the shadow projection clipping minimum z value.
         */
        shadowMinZ: number;
        /**
         * Defines the shadow projection clipping maximum z value.
         */
        shadowMaxZ: number;
        /**
         * Computes the transformed information (transformedPosition and transformedDirection in World space) of the current light
         * @returns true if the information has been computed, false if it does not need to (no parenting)
         */
        computeTransformedInformation(): boolean;
        /**
         * Gets the scene the light belongs to.
         * @returns The scene
         */
        getScene(): Scene;
        /**
         * Callback defining a custom Projection Matrix Builder.
         * This can be used to override the default projection matrix computation.
         */
        customProjectionMatrixBuilder: (viewMatrix: Matrix, renderList: Array<AbstractMesh>, result: Matrix) => void;
        /**
         * Sets the shadow projection matrix in parameter to the generated projection matrix.
         * @param matrix The materix to updated with the projection information
         * @param viewMatrix The transform matrix of the light
         * @param renderList The list of mesh to render in the map
         * @returns The current light
         */
        setShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): IShadowLight;
        /**
         * Gets the current depth scale used in ESM.
         * @returns The scale
         */
        getDepthScale(): number;
        /**
         * Returns whether or not the shadow generation require a cube texture or a 2d texture.
         * @returns true if a cube texture needs to be use
         */
        needCube(): boolean;
        /**
         * Detects if the projection matrix requires to be recomputed this frame.
         * @returns true if it requires to be recomputed otherwise, false.
         */
        needProjectionMatrixCompute(): boolean;
        /**
         * Forces the shadow generator to recompute the projection matrix even if position and direction did not changed.
         */
        forceProjectionMatrixCompute(): void;
        /**
         * Get the direction to use to render the shadow map. In case of cube texture, the face index can be passed.
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
    }
    /**
     * Base implementation IShadowLight
     * It groups all the common behaviour in order to reduce dupplication and better follow the DRY pattern.
     */
    export abstract class ShadowLight extends Light implements IShadowLight {
        protected abstract _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _position: Vector3;
        protected _setPosition(value: Vector3): void;
        /**
         * Sets the position the shadow will be casted from. Also use as the light position for both
         * point and spot lights.
         */
        get position(): Vector3;
        /**
         * Sets the position the shadow will be casted from. Also use as the light position for both
         * point and spot lights.
         */
        set position(value: Vector3);
        protected _direction: Vector3;
        protected _setDirection(value: Vector3): void;
        /**
         * In 2d mode (needCube being false), gets the direction used to cast the shadow.
         * Also use as the light direction on spot and directional lights.
         */
        get direction(): Vector3;
        /**
         * In 2d mode (needCube being false), sets the direction used to cast the shadow.
         * Also use as the light direction on spot and directional lights.
         */
        set direction(value: Vector3);
        protected _shadowMinZ: number;
        /**
         * Gets the shadow projection clipping minimum z value.
         */
        get shadowMinZ(): number;
        /**
         * Sets the shadow projection clipping minimum z value.
         */
        set shadowMinZ(value: number);
        protected _shadowMaxZ: number;
        /**
         * Sets the shadow projection clipping maximum z value.
         */
        get shadowMaxZ(): number;
        /**
         * Gets the shadow projection clipping maximum z value.
         */
        set shadowMaxZ(value: number);
        /**
         * Callback defining a custom Projection Matrix Builder.
         * This can be used to override the default projection matrix computation.
         */
        customProjectionMatrixBuilder: (viewMatrix: Matrix, renderList: Array<AbstractMesh>, result: Matrix) => void;
        /**
         * The transformed position. Position of the light in world space taking parenting in account.
         */
        transformedPosition: Vector3;
        /**
         * The transformed direction. Direction of the light in world space taking parenting in account.
         */
        transformedDirection: Vector3;
        private _needProjectionMatrixCompute;
        /**
         * Computes the transformed information (transformedPosition and transformedDirection in World space) of the current light
         * @returns true if the information has been computed, false if it does not need to (no parenting)
         */
        computeTransformedInformation(): boolean;
        /**
         * Return the depth scale used for the shadow map.
         * @returns the depth scale.
         */
        getDepthScale(): number;
        /**
         * Get the direction to use to render the shadow map. In case of cube texture, the face index can be passed.
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Returns the ShadowLight absolute position in the World.
         * @returns the position vector in world space
         */
        getAbsolutePosition(): Vector3;
        /**
         * Sets the ShadowLight direction toward the passed target.
         * @param target The point to target in local space
         * @returns the updated ShadowLight direction
         */
        setDirectionToTarget(target: Vector3): Vector3;
        /**
         * Returns the light rotation in euler definition.
         * @returns the x y z rotation in local space.
         */
        getRotation(): Vector3;
        /**
         * Returns whether or not the shadow generation require a cube texture or a 2d texture.
         * @returns true if a cube texture needs to be use
         */
        needCube(): boolean;
        /**
         * Detects if the projection matrix requires to be recomputed this frame.
         * @returns true if it requires to be recomputed otherwise, false.
         */
        needProjectionMatrixCompute(): boolean;
        /**
         * Forces the shadow generator to recompute the projection matrix even if position and direction did not changed.
         */
        forceProjectionMatrixCompute(): void;
        /** @hidden */
        _initCache(): void;
        /** @hidden */
        _isSynchronized(): boolean;
        /**
         * Computes the world matrix of the node
         * @param force defines if the cache version should be invalidated forcing the world matrix to be created from scratch
         * @returns the world matrix
         */
        computeWorldMatrix(force?: boolean): Matrix;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
        /**
         * Sets the shadow projection matrix in parameter to the generated projection matrix.
         * @param matrix The materix to updated with the projection information
         * @param viewMatrix The transform matrix of the light
         * @param renderList The list of mesh to render in the map
         * @returns The current light
         */
        setShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): IShadowLight;
    }
}
declare module "babylonjs/Materials/prePassConfiguration" {
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { Scene } from "babylonjs/scene";
    import { Effect } from "babylonjs/Materials/effect";
    /**
     * Configuration needed for prepass-capable materials
     */
    export class PrePassConfiguration {
        /**
         * Previous world matrices of meshes carrying this material
         * Used for computing velocity
         */
        previousWorldMatrices: {
            [index: number]: Matrix;
        };
        /**
         * Previous view project matrix
         * Used for computing velocity
         */
        previousViewProjection: Matrix;
        /**
         * Previous bones of meshes carrying this material
         * Used for computing velocity
         */
        previousBones: {
            [index: number]: Float32Array;
        };
        /**
         * Add the required uniforms to the current list.
         * @param uniforms defines the current uniform list.
         */
        static AddUniforms(uniforms: string[]): void;
        /**
         * Add the required samplers to the current list.
         * @param samplers defines the current sampler list.
         */
        static AddSamplers(samplers: string[]): void;
        /**
         * Binds the material data.
         * @param effect defines the effect to update
         * @param scene defines the scene the material belongs to.
         * @param mesh The mesh
         * @param world World matrix of this mesh
         * @param isFrozen Is the material frozen
         */
        bindForSubMesh(effect: Effect, scene: Scene, mesh: Mesh, world: Matrix, isFrozen: boolean): void;
    }
}
declare module "babylonjs/Cameras/targetCamera" {
    import { Nullable } from "babylonjs/types";
    import { Camera } from "babylonjs/Cameras/camera";
    import { Scene } from "babylonjs/scene";
    import { Quaternion, Matrix, Vector3, Vector2 } from "babylonjs/Maths/math.vector";
    /**
     * A target camera takes a mesh or position as a target and continues to look at it while it moves.
     * This is the base of the follow, arc rotate cameras and Free camera
     * @see https://doc.babylonjs.com/features/cameras
     */
    export class TargetCamera extends Camera {
        private static _RigCamTransformMatrix;
        private static _TargetTransformMatrix;
        private static _TargetFocalPoint;
        private _tmpUpVector;
        private _tmpTargetVector;
        /**
         * Define the current direction the camera is moving to
         */
        cameraDirection: Vector3;
        /**
         * Define the current rotation the camera is rotating to
         */
        cameraRotation: Vector2;
        /** Gets or sets a boolean indicating that the scaling of the parent hierarchy will not be taken in account by the camera */
        ignoreParentScaling: boolean;
        /**
         * When set, the up vector of the camera will be updated by the rotation of the camera
         */
        updateUpVectorFromRotation: boolean;
        private _tmpQuaternion;
        /**
         * Define the current rotation of the camera
         */
        rotation: Vector3;
        /**
         * Define the current rotation of the camera as a quaternion to prevent Gimbal lock
         */
        rotationQuaternion: Quaternion;
        /**
         * Define the current speed of the camera
         */
        speed: number;
        /**
         * Add constraint to the camera to prevent it to move freely in all directions and
         * around all axis.
         */
        noRotationConstraint: boolean;
        /**
         * Reverses mouselook direction to 'natural' panning as opposed to traditional direct
         * panning
         */
        invertRotation: boolean;
        /**
         * Speed multiplier for inverse camera panning
         */
        inverseRotationSpeed: number;
        /**
         * Define the current target of the camera as an object or a position.
         */
        lockedTarget: any;
        /** @hidden */
        _currentTarget: Vector3;
        /** @hidden */
        _initialFocalDistance: number;
        /** @hidden */
        _viewMatrix: Matrix;
        /** @hidden */
        _camMatrix: Matrix;
        /** @hidden */
        _cameraTransformMatrix: Matrix;
        /** @hidden */
        _cameraRotationMatrix: Matrix;
        /** @hidden */
        _referencePoint: Vector3;
        /** @hidden */
        _transformedReferencePoint: Vector3;
        /** @hidden */
        _reset: () => void;
        private _defaultUp;
        /**
         * Instantiates a target camera that takes a mesh or position as a target and continues to look at it while it moves.
         * This is the base of the follow, arc rotate cameras and Free camera
         * @see https://doc.babylonjs.com/features/cameras
         * @param name Defines the name of the camera in the scene
         * @param position Defines the start position of the camera in the scene
         * @param scene Defines the scene the camera belongs to
         * @param setActiveOnSceneIfNoneActive Defines wheter the camera should be marked as active if not other active cameras have been defined
         */
        constructor(name: string, position: Vector3, scene: Scene, setActiveOnSceneIfNoneActive?: boolean);
        /**
         * Gets the position in front of the camera at a given distance.
         * @param distance The distance from the camera we want the position to be
         * @returns the position
         */
        getFrontPosition(distance: number): Vector3;
        /** @hidden */
        _getLockedTargetPosition(): Nullable<Vector3>;
        private _storedPosition;
        private _storedRotation;
        private _storedRotationQuaternion;
        /**
         * Store current camera state of the camera (fov, position, rotation, etc..)
         * @returns the camera
         */
        storeState(): Camera;
        /**
         * Restored camera state. You must call storeState() first
         * @returns whether it was successful or not
         * @hidden
         */
        _restoreStateValues(): boolean;
        /** @hidden */
        _initCache(): void;
        /** @hidden */
        _updateCache(ignoreParentClass?: boolean): void;
        /** @hidden */
        _isSynchronizedViewMatrix(): boolean;
        /** @hidden */
        _computeLocalCameraSpeed(): number;
        /**
         * Defines the target the camera should look at.
         * @param target Defines the new target as a Vector or a mesh
         */
        setTarget(target: Vector3): void;
        /**
         * Defines the target point of the camera.
         * The camera looks towards it form the radius distance.
         */
        get target(): Vector3;
        set target(value: Vector3);
        /**
         * Return the current target position of the camera. This value is expressed in local space.
         * @returns the target position
         */
        getTarget(): Vector3;
        /** @hidden */
        _decideIfNeedsToMove(): boolean;
        /** @hidden */
        _updatePosition(): void;
        /** @hidden */
        _checkInputs(): void;
        protected _updateCameraRotationMatrix(): void;
        /**
         * Update the up vector to apply the rotation of the camera (So if you changed the camera rotation.z this will let you update the up vector as well)
         * @returns the current camera
         */
        private _rotateUpVectorWithCameraRotationMatrix;
        private _cachedRotationZ;
        private _cachedQuaternionRotationZ;
        /** @hidden */
        _getViewMatrix(): Matrix;
        protected _computeViewMatrix(position: Vector3, target: Vector3, up: Vector3): void;
        /**
         * @hidden
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * @hidden
         */
        _updateRigCameras(): void;
        private _getRigCamPositionAndTarget;
        /**
         * Gets the current object class name.
         * @return the class name
         */
        getClassName(): string;
    }
}
declare module "babylonjs/Cameras/cameraInputsManager" {
    import { Nullable } from "babylonjs/types";
    import { Camera } from "babylonjs/Cameras/camera";
    /**
     * @ignore
     * This is a list of all the different input types that are available in the application.
     * Fo instance: ArcRotateCameraGamepadInput...
     */
    export var CameraInputTypes: {};
    /**
     * This is the contract to implement in order to create a new input class.
     * Inputs are dealing with listening to user actions and moving the camera accordingly.
     */
    export interface ICameraInput<TCamera extends Camera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: Nullable<TCamera>;
        /**
         * Gets the class name of the current intput.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs?: () => void;
    }
    /**
     * Represents a map of input types to input instance or input index to input instance.
     */
    export interface CameraInputsMap<TCamera extends Camera> {
        /**
         * Accessor to the input by input type.
         */
        [name: string]: ICameraInput<TCamera>;
        /**
         * Accessor to the input by input index.
         */
        [idx: number]: ICameraInput<TCamera>;
    }
    /**
     * This represents the input manager used within a camera.
     * It helps dealing with all the different kind of input attached to a camera.
     * @see https://doc.babylonjs.com/how_to/customizing_camera_inputs
     */
    export class CameraInputsManager<TCamera extends Camera> {
        /**
         * Defines the list of inputs attahed to the camera.
         */
        attached: CameraInputsMap<TCamera>;
        /**
         * Defines the dom element the camera is collecting inputs from.
         * This is null if the controls have not been attached.
         */
        attachedToElement: boolean;
        /**
         * Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        noPreventDefault: boolean;
        /**
         * Defined the camera the input manager belongs to.
         */
        camera: TCamera;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs: () => void;
        /**
         * Instantiate a new Camera Input Manager.
         * @param camera Defines the camera the input manager blongs to
         */
        constructor(camera: TCamera);
        /**
         * Add an input method to a camera
         * @see https://doc.babylonjs.com/how_to/customizing_camera_inputs
         * @param input camera input method
         */
        add(input: ICameraInput<TCamera>): void;
        /**
         * Remove a specific input method from a camera
         * example: camera.inputs.remove(camera.inputs.attached.mouse);
         * @param inputToRemove camera input method
         */
        remove(inputToRemove: ICameraInput<TCamera>): void;
        /**
         * Remove a specific input type from a camera
         * example: camera.inputs.remove("ArcRotateCameraGamepadInput");
         * @param inputType the type of the input to remove
         */
        removeByType(inputType: string): void;
        private _addCheckInputs;
        /**
         * Attach the input controls to the currently attached dom element to listen the events from.
         * @param input Defines the input to attach
         */
        attachInput(input: ICameraInput<TCamera>): void;
        /**
         * Attach the current manager inputs controls to a specific dom element to listen the events from.
         * @param element Defines the dom element to collect the events from
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachElement(noPreventDefault?: boolean): void;
        /**
         * Detach the current manager inputs controls from a specific dom element.
         * @param element Defines the dom element to collect the events from
         * @param disconnect Defines whether the input should be removed from the current list of attached inputs
         */
        detachElement(disconnect?: boolean): void;
        /**
         * Rebuild the dynamic inputCheck function from the current list of
         * defined inputs in the manager.
         */
        rebuildInputCheck(): void;
        /**
         * Remove all attached input methods from a camera
         */
        clear(): void;
        /**
         * Serialize the current input manager attached to a camera.
         * This ensures than once parsed,
         * the input associated to the camera will be identical to the current ones
         * @param serializedCamera Defines the camera serialization JSON the input serialization should write to
         */
        serialize(serializedCamera: any): void;
        /**
         * Parses an input manager serialized JSON to restore the previous list of inputs
         * and states associated to a camera.
         * @param parsedCamera Defines the JSON to parse
         */
        parse(parsedCamera: any): void;
    }
}
declare module "babylonjs/Events/keyboardEvents" {
    /**
     * Gather the list of keyboard event types as constants.
     */
    export class KeyboardEventTypes {
        /**
         * The keydown event is fired when a key becomes active (pressed).
         */
        static readonly KEYDOWN: number;
        /**
         * The keyup event is fired when a key has been released.
         */
        static readonly KEYUP: number;
    }
    /**
     * This class is used to store keyboard related info for the onKeyboardObservable event.
     */
    export class KeyboardInfo {
        /**
         * Defines the type of event (KeyboardEventTypes)
         */
        type: number;
        /**
         * Defines the related dom event
         */
        event: KeyboardEvent;
        /**
         * Instantiates a new keyboard info.
         * This class is used to store keyboard related info for the onKeyboardObservable event.
         * @param type Defines the type of event (KeyboardEventTypes)
         * @param event Defines the related dom event
         */
        constructor(
        /**
         * Defines the type of event (KeyboardEventTypes)
         */
        type: number, 
        /**
         * Defines the related dom event
         */
        event: KeyboardEvent);
    }
    /**
     * This class is used to store keyboard related info for the onPreKeyboardObservable event.
     * Set the skipOnKeyboardObservable property to true if you want the engine to stop any process after this event is triggered, even not calling onKeyboardObservable
     */
    export class KeyboardInfoPre extends KeyboardInfo {
        /**
         * Defines the type of event (KeyboardEventTypes)
         */
        type: number;
        /**
         * Defines the related dom event
         */
        event: KeyboardEvent;
        /**
         * Defines whether the engine should skip the next onKeyboardObservable associated to this pre.
         */
        skipOnPointerObservable: boolean;
        /**
         * Instantiates a new keyboard pre info.
         * This class is used to store keyboard related info for the onPreKeyboardObservable event.
         * @param type Defines the type of event (KeyboardEventTypes)
         * @param event Defines the related dom event
         */
        constructor(
        /**
         * Defines the type of event (KeyboardEventTypes)
         */
        type: number, 
        /**
         * Defines the related dom event
         */
        event: KeyboardEvent);
    }
}
declare module "babylonjs/Cameras/Inputs/freeCameraKeyboardMoveInput" {
    import { ICameraInput } from "babylonjs/Cameras/cameraInputsManager";
    import { FreeCamera } from "babylonjs/Cameras/freeCamera";
    /**
     * Manage the keyboard inputs to control the movement of a free camera.
     * @see https://doc.babylonjs.com/how_to/customizing_camera_inputs
     */
    export class FreeCameraKeyboardMoveInput implements ICameraInput<FreeCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FreeCamera;
        /**
         * Gets or Set the list of keyboard keys used to control the forward move of the camera.
         */
        keysUp: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the upward move of the camera.
         */
        keysUpward: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the backward move of the camera.
         */
        keysDown: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the downward move of the camera.
         */
        keysDownward: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the left strafe move of the camera.
         */
        keysLeft: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the right strafe move of the camera.
         */
        keysRight: number[];
        private _keys;
        private _onCanvasBlurObserver;
        private _onKeyboardObserver;
        private _engine;
        private _scene;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current intput.
         * @returns the class name
         */
        getClassName(): string;
        /** @hidden */
        _onLostFocus(): void;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }
}
declare module "babylonjs/Actions/action" {
    import { Observable } from "babylonjs/Misc/observable";
    import { Condition } from "babylonjs/Actions/condition";
    import { AbstractActionManager } from "babylonjs/Actions/abstractActionManager";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { ActionManager } from "babylonjs/Actions/actionManager";
    import { ActionEvent } from "babylonjs/Actions/actionEvent";
    import { Node } from "babylonjs/node";
    /**
     * Interface used to define Action
     */
    export interface IAction {
        /**
       * Trigger for the action
       */
        trigger: number;
        /** Options of the trigger */
        triggerOptions: any;
        /**
         * Gets the trigger parameters
         * @returns the trigger parameters
         */
        getTriggerParameter(): any;
        /**
         * Internal only - executes current action event
         * @hidden
         */
        _executeCurrent(evt?: ActionEvent): void;
        /**
         * Serialize placeholder for child classes
         * @param parent of child
         * @returns the serialized object
         */
        serialize(parent: any): any;
        /**
        * Internal only
        * @hidden
        */
        _prepare(): void;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: Nullable<AbstractActionManager>;
        /**
         * Adds action to chain of actions, may be a DoNothingAction
         * @param action defines the next action to execute
         * @returns The action passed in
         * @see https://www.babylonjs-playground.com/#1T30HR#0
         */
        then(action: IAction): IAction;
    }
    /**
     * The action to be carried out following a trigger
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions#available-actions
     */
    export class Action implements IAction {
        /** the trigger, with or without parameters, for the action */
        triggerOptions: any;
        /**
         * Trigger for the action
         */
        trigger: number;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        private _nextActiveAction;
        private _child;
        private _condition?;
        private _triggerParameter;
        /**
        * An event triggered prior to action being executed.
        */
        onBeforeExecuteObservable: Observable<Action>;
        /**
         * Creates a new Action
         * @param triggerOptions the trigger, with or without parameters, for the action
         * @param condition an optional determinant of action
         */
        constructor(
        /** the trigger, with or without parameters, for the action */
        triggerOptions: any, condition?: Condition);
        /**
         * Internal only
         * @hidden
         */
        _prepare(): void;
        /**
         * Gets the trigger parameters
         * @returns the trigger parameters
         */
        getTriggerParameter(): any;
        /**
         * Internal only - executes current action event
         * @hidden
         */
        _executeCurrent(evt?: ActionEvent): void;
        /**
         * Execute placeholder for child classes
         * @param evt optional action event
         */
        execute(evt?: ActionEvent): void;
        /**
         * Skips to next active action
         */
        skipToNextActiveAction(): void;
        /**
         * Adds action to chain of actions, may be a DoNothingAction
         * @param action defines the next action to execute
         * @returns The action passed in
         * @see https://www.babylonjs-playground.com/#1T30HR#0
         */
        then(action: Action): Action;
        /**
         * Internal only
         * @hidden
         */
        _getProperty(propertyPath: string): string;
        /**
         * Internal only
         * @hidden
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @param parent of child
         * @returns the serialized object
         */
        serialize(parent: any): any;
        /**
         * Internal only called by serialize
         * @hidden
         */
        protected _serialize(serializedAction: any, parent?: any): any;
        /**
         * Internal only
         * @hidden
         */
        static _SerializeValueAsString: (value: any) => string;
        /**
         * Internal only
         * @hidden
         */
        static _GetTargetProperty: (target: Scene | Node) => {
            name: string;
            targetType: string;
            value: string;
        };
    }
}
declare module "babylonjs/Actions/condition" {
    import { ActionManager } from "babylonjs/Actions/actionManager";
    /**
     * A Condition applied to an Action
     */
    export class Condition {
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @hidden
         */
        _evaluationId: number;
        /**
         * Internal only
         * @hidden
         */
        _currentResult: boolean;
        /**
         * Creates a new Condition
         * @param actionManager the manager of the action the condition is applied to
         */
        constructor(actionManager: ActionManager);
        /**
         * Check if the current condition is valid
         * @returns a boolean
         */
        isValid(): boolean;
        /**
         * Internal only
         * @hidden
         */
        _getProperty(propertyPath: string): string;
        /**
         * Internal only
         * @hidden
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @returns the serialized object
         */
        serialize(): any;
        /**
         * Internal only
         * @hidden
         */
        protected _serialize(serializedCondition: any): any;
    }
    /**
     * Defines specific conditional operators as extensions of Condition
     */
    export class ValueCondition extends Condition {
        /** path to specify the property of the target the conditional operator uses  */
        propertyPath: string;
        /** the value compared by the conditional operator against the current value of the property */
        value: any;
        /** the conditional operator, default ValueCondition.IsEqual */
        operator: number;
        /**
         * Internal only
         * @hidden
         */
        private static _IsEqual;
        /**
         * Internal only
         * @hidden
         */
        private static _IsDifferent;
        /**
         * Internal only
         * @hidden
         */
        private static _IsGreater;
        /**
         * Internal only
         * @hidden
         */
        private static _IsLesser;
        /**
         * returns the number for IsEqual
         */
        static get IsEqual(): number;
        /**
         * Returns the number for IsDifferent
         */
        static get IsDifferent(): number;
        /**
         * Returns the number for IsGreater
         */
        static get IsGreater(): number;
        /**
         * Returns the number for IsLesser
         */
        static get IsLesser(): number;
        /**
         * Internal only The action manager for the condition
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @hidden
         */
        private _target;
        /**
         * Internal only
         * @hidden
         */
        private _effectiveTarget;
        /**
         * Internal only
         * @hidden
         */
        private _property;
        /**
         * Creates a new ValueCondition
         * @param actionManager manager for the action the condition applies to
         * @param target for the action
         * @param propertyPath path to specify the property of the target the conditional operator uses
         * @param value the value compared by the conditional operator against the current value of the property
         * @param operator the conditional operator, default ValueCondition.IsEqual
         */
        constructor(actionManager: ActionManager, target: any, 
        /** path to specify the property of the target the conditional operator uses  */
        propertyPath: string, 
        /** the value compared by the conditional operator against the current value of the property */
        value: any, 
        /** the conditional operator, default ValueCondition.IsEqual */
        operator?: number);
        /**
         * Compares the given value with the property value for the specified conditional operator
         * @returns the result of the comparison
         */
        isValid(): boolean;
        /**
         * Serialize the ValueCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
        /**
         * Gets the name of the conditional operator for the ValueCondition
         * @param operator the conditional operator
         * @returns the name
         */
        static GetOperatorName(operator: number): string;
    }
    /**
     * Defines a predicate condition as an extension of Condition
     */
    export class PredicateCondition extends Condition {
        /** defines the predicate function used to validate the condition */
        predicate: () => boolean;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Creates a new PredicateCondition
         * @param actionManager manager for the action the condition applies to
         * @param predicate defines the predicate function used to validate the condition
         */
        constructor(actionManager: ActionManager, 
        /** defines the predicate function used to validate the condition */
        predicate: () => boolean);
        /**
         * @returns the validity of the predicate condition
         */
        isValid(): boolean;
    }
    /**
     * Defines a state condition as an extension of Condition
     */
    export class StateCondition extends Condition {
        /** Value to compare with target state  */
        value: string;
        /**
         * Internal only - manager for action
         * @hidden
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @hidden
         */
        private _target;
        /**
         * Creates a new StateCondition
         * @param actionManager manager for the action the condition applies to
         * @param target of the condition
         * @param value to compare with target state
         */
        constructor(actionManager: ActionManager, target: any, 
        /** Value to compare with target state  */
        value: string);
        /**
         * Gets a boolean indicating if the current condition is met
         * @returns the validity of the state
         */
        isValid(): boolean;
        /**
         * Serialize the StateCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
    }
}
declare module "babylonjs/Actions/directActions" {
    import { Action } from "babylonjs/Actions/action";
    import { Condition } from "babylonjs/Actions/condition";
    import { ActionEvent } from "babylonjs/Actions/actionEvent";
    /**
     * This defines an action responsible to toggle a boolean once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class SwitchBooleanAction extends Action {
        /**
         * The path to the boolean property in the target object
         */
        propertyPath: string;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the boolean
         * @param propertyPath defines the path to the boolean property in the target object
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action toggle the boolean value.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to set a the state field of the target
     *  to a desired value once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class SetStateAction extends Action {
        /**
         * The value to store in the state field.
         */
        value: string;
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the state property
         * @param value defines the value to store in the state field
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, value: string, condition?: Condition);
        /**
         * Execute the action and store the value on the target state property.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to set a property of the target
     *  to a desired value once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class SetValueAction extends Action {
        /**
         * The path of the property to set in the target.
         */
        propertyPath: string;
        /**
         * The value to set in the property
         */
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the property
         * @param propertyPath defines the path of the property to set in the target
         * @param value defines the value to set in the property
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action and set the targetted property to the desired value.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to increment the target value
     *  to a desired value once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class IncrementValueAction extends Action {
        /**
         * The path of the property to increment in the target.
         */
        propertyPath: string;
        /**
         * The value we should increment the property by.
         */
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the property
         * @param propertyPath defines the path of the property to increment in the target
         * @param value defines the value value we should increment the property by
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action and increment the target of the value amount.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to start an animation once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class PlayAnimationAction extends Action {
        /**
         * Where the animation should start (animation frame)
         */
        from: number;
        /**
         * Where the animation should stop (animation frame)
         */
        to: number;
        /**
         * Define if the animation should loop or stop after the first play.
         */
        loop?: boolean;
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the target animation or animation name
         * @param from defines from where the animation should start (animation frame)
         * @param end defines where the animation should stop (animation frame)
         * @param loop defines if the animation should loop or stop after the first play
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, from: number, to: number, loop?: boolean, condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action and play the animation.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to stop an animation once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class StopAnimationAction extends Action {
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the target animation or animation name
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action and stop the animation.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible that does nothing once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class DoNothingAction extends Action {
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions?: any, condition?: Condition);
        /**
         * Execute the action and do nothing.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to trigger several actions once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class CombineAction extends Action {
        /**
         * The list of aggregated animations to run.
         */
        children: Action[];
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param children defines the list of aggregated animations to run
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, children: Action[], condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action and executes all the aggregated actions.
         */
        execute(evt: ActionEvent): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to run code (external event) once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class ExecuteCodeAction extends Action {
        /**
         * The callback function to run.
         */
        func: (evt: ActionEvent) => void;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param func defines the callback function to run
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, func: (evt: ActionEvent) => void, condition?: Condition);
        /**
         * Execute the action and run the attached code.
         */
        execute(evt: ActionEvent): void;
    }
    /**
     * This defines an action responsible to set the parent property of the target once triggered.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class SetParentAction extends Action {
        private _parent;
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the target containing the parent property
         * @param parent defines from where the animation should start (animation frame)
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, parent: any, condition?: Condition);
        /** @hidden */
        _prepare(): void;
        /**
         * Execute the action and set the parent property.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
}
declare module "babylonjs/Actions/actionManager" {
    import { Nullable } from "babylonjs/types";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Scene } from "babylonjs/scene";
    import { IAction } from "babylonjs/Actions/action";
    import { IActionEvent } from "babylonjs/Actions/actionEvent";
    import { AbstractActionManager } from "babylonjs/Actions/abstractActionManager";
    /**
     * Action Manager manages all events to be triggered on a given mesh or the global scene.
     * A single scene can have many Action Managers to handle predefined actions on specific meshes.
     * @see https://doc.babylonjs.com/how_to/how_to_use_actions
     */
    export class ActionManager extends AbstractActionManager {
        /**
         * Nothing
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly NothingTrigger: number;
        /**
         * On pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickTrigger: number;
        /**
         * On left pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnLeftPickTrigger: number;
        /**
         * On right pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnRightPickTrigger: number;
        /**
         * On center pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnCenterPickTrigger: number;
        /**
         * On pick down
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickDownTrigger: number;
        /**
         * On double pick
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnDoublePickTrigger: number;
        /**
         * On pick up
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickUpTrigger: number;
        /**
         * On pick out.
         * This trigger will only be raised if you also declared a OnPickDown
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPickOutTrigger: number;
        /**
         * On long press
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnLongPressTrigger: number;
        /**
         * On pointer over
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPointerOverTrigger: number;
        /**
         * On pointer out
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnPointerOutTrigger: number;
        /**
         * On every frame
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnEveryFrameTrigger: number;
        /**
         * On intersection enter
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnIntersectionEnterTrigger: number;
        /**
         * On intersection exit
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnIntersectionExitTrigger: number;
        /**
         * On key down
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnKeyDownTrigger: number;
        /**
         * On key up
         * @see https://doc.babylonjs.com/how_to/how_to_use_actions#triggers
         */
        static readonly OnKeyUpTrigger: number;
        private _scene;
        /**
         * Creates a new action manager
         * @param scene defines the hosting scene
         */
        constructor(scene: Scene);
        /**
         * Releases all associated resources
         */
        dispose(): void;
        /**
         * Gets hosting scene
         * @returns the hosting scene
         */
        getScene(): Scene;
        /**
         * Does this action manager handles actions of any of the given triggers
         * @param triggers defines the triggers to be tested
         * @return a boolean indicating whether one (or more) of the triggers is handled
         */
        hasSpecificTriggers(triggers: number[]): boolean;
        /**
         * Does this action manager handles actions of any of the given triggers. This function takes two arguments for
         * speed.
         * @param triggerA defines the trigger to be tested
         * @param triggerB defines the trigger to be tested
         * @return a boolean indicating whether one (or more) of the triggers is handled
         */
        hasSpecificTriggers2(triggerA: number, triggerB: number): boolean;
        /**
         * Does this action manager handles actions of a given trigger
         * @param trigger defines the trigger to be tested
         * @param parameterPredicate defines an optional predicate to filter triggers by parameter
         * @return whether the trigger is handled
         */
        hasSpecificTrigger(trigger: number, parameterPredicate?: (parameter: any) => boolean): boolean;
        /**
         * Does this action manager has pointer triggers
         */
        get hasPointerTriggers(): boolean;
        /**
         * Does this action manager has pick triggers
         */
        get hasPickTriggers(): boolean;
        /**
         * Registers an action to this action manager
         * @param action defines the action to be registered
         * @return the action amended (prepared) after registration
         */
        registerAction(action: IAction): Nullable<IAction>;
        /**
         * Unregisters an action to this action manager
         * @param action defines the action to be unregistered
         * @return a boolean indicating whether the action has been unregistered
         */
        unregisterAction(action: IAction): Boolean;
        /**
         * Process a specific trigger
         * @param trigger defines the trigger to process
         * @param evt defines the event details to be processed
         */
        processTrigger(trigger: number, evt?: IActionEvent): void;
        /** @hidden */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /** @hidden */
        _getProperty(propertyPath: string): string;
        /**
         * Serialize this manager to a JSON object
         * @param name defines the property name to store this manager
         * @returns a JSON representation of this manager
         */
        serialize(name: string): any;
        /**
         * Creates a new ActionManager from a JSON data
         * @param parsedActions defines the JSON data to read from
         * @param object defines the hosting mesh
         * @param scene defines the hosting scene
         */
        static Parse(parsedActions: any, object: Nullable<AbstractMesh>, scene: Scene): void;
        /**
         * Get a trigger name by index
         * @param trigger defines the trigger index
         * @returns a trigger name
         */
        static GetTriggerName(trigger: number): string;
    }
}
declare module "babylonjs/Culling/ray" {
    import { DeepImmutable, Nullable, float } from "babylonjs/types";
    import { Matrix, Vector3 } from "babylonjs/Maths/math.vector";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { PickingInfo } from "babylonjs/Collisions/pickingInfo";
    import { IntersectionInfo } from "babylonjs/Collisions/intersectionInfo";
    import { BoundingBox } from "babylonjs/Culling/boundingBox";
    import { BoundingSphere } from "babylonjs/Culling/boundingSphere";
    import { Plane } from "babylonjs/Maths/math.plane";
    /**
     * Class representing a ray with position and direction
     */
    export class Ray {
        /** origin point */
        origin: Vector3;
        /** direction */
        direction: Vector3;
        /** length of the ray */
        length: number;
        private static readonly _TmpVector3;
        private _tmpRay;
        /**
         * Creates a new ray
         * @param origin origin point
         * @param direction direction
         * @param length length of the ray
         */
        constructor(
        /** origin point */
        origin: Vector3, 
        /** direction */
        direction: Vector3, 
        /** length of the ray */
        length?: number);
        /**
         * Checks if the ray intersects a box
         * This does not account for the ray lenght by design to improve perfs.
         * @param minimum bound of the box
         * @param maximum bound of the box
         * @param intersectionTreshold extra extend to be added to the box in all direction
         * @returns if the box was hit
         */
        intersectsBoxMinMax(minimum: DeepImmutable<Vector3>, maximum: DeepImmutable<Vector3>, intersectionTreshold?: number): boolean;
        /**
         * Checks if the ray intersects a box
         * This does not account for the ray lenght by design to improve perfs.
         * @param box the bounding box to check
         * @param intersectionTreshold extra extend to be added to the BoundingBox in all direction
         * @returns if the box was hit
         */
        intersectsBox(box: DeepImmutable<BoundingBox>, intersectionTreshold?: number): boolean;
        /**
         * If the ray hits a sphere
         * @param sphere the bounding sphere to check
         * @param intersectionTreshold extra extend to be added to the BoundingSphere in all direction
         * @returns true if it hits the sphere
         */
        intersectsSphere(sphere: DeepImmutable<BoundingSphere>, intersectionTreshold?: number): boolean;
        /**
         * If the ray hits a triange
         * @param vertex0 triangle vertex
         * @param vertex1 triangle vertex
         * @param vertex2 triangle vertex
         * @returns intersection information if hit
         */
        intersectsTriangle(vertex0: DeepImmutable<Vector3>, vertex1: DeepImmutable<Vector3>, vertex2: DeepImmutable<Vector3>): Nullable<IntersectionInfo>;
        /**
         * Checks if ray intersects a plane
         * @param plane the plane to check
         * @returns the distance away it was hit
         */
        intersectsPlane(plane: DeepImmutable<Plane>): Nullable<number>;
        /**
         * Calculate the intercept of a ray on a given axis
         * @param axis to check 'x' | 'y' | 'z'
         * @param offset from axis interception (i.e. an offset of 1y is intercepted above ground)
         * @returns a vector containing the coordinates where 'axis' is equal to zero (else offset), or null if there is no intercept.
         */
        intersectsAxis(axis: string, offset?: number): Nullable<Vector3>;
        /**
         * Checks if ray intersects a mesh
         * @param mesh the mesh to check
         * @param fastCheck defines if the first intersection will be used (and not the closest)
         * @returns picking info of the intersecton
         */
        intersectsMesh(mesh: DeepImmutable<AbstractMesh>, fastCheck?: boolean): PickingInfo;
        /**
         * Checks if ray intersects a mesh
         * @param meshes the meshes to check
         * @param fastCheck defines if the first intersection will be used (and not the closest)
         * @param results array to store result in
         * @returns Array of picking infos
         */
        intersectsMeshes(meshes: Array<DeepImmutable<AbstractMesh>>, fastCheck?: boolean, results?: Array<PickingInfo>): Array<PickingInfo>;
        private _comparePickingInfo;
        private static smallnum;
        private static rayl;
        /**
         * Intersection test between the ray and a given segment whithin a given tolerance (threshold)
         * @param sega the first point of the segment to test the intersection against
         * @param segb the second point of the segment to test the intersection against
         * @param threshold the tolerance margin, if the ray doesn't intersect the segment but is close to the given threshold, the intersection is successful
         * @return the distance from the ray origin to the intersection point if there's intersection, or -1 if there's no intersection
         */
        intersectionSegment(sega: DeepImmutable<Vector3>, segb: DeepImmutable<Vector3>, threshold: number): number;
        /**
         * Update the ray from viewport position
         * @param x position
         * @param y y position
         * @param viewportWidth viewport width
         * @param viewportHeight viewport height
         * @param world world matrix
         * @param view view matrix
         * @param projection projection matrix
         * @returns this ray updated
         */
        update(x: number, y: number, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>): Ray;
        /**
         * Creates a ray with origin and direction of 0,0,0
         * @returns the new ray
         */
        static Zero(): Ray;
        /**
         * Creates a new ray from screen space and viewport
         * @param x position
         * @param y y position
         * @param viewportWidth viewport width
         * @param viewportHeight viewport height
         * @param world world matrix
         * @param view view matrix
         * @param projection projection matrix
         * @returns new ray
         */
        static CreateNew(x: number, y: number, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>): Ray;
        /**
         * Function will create a new transformed ray starting from origin and ending at the end point. Ray's length will be set, and ray will be
         * transformed to the given world matrix.
         * @param origin The origin point
         * @param end The end point
         * @param world a matrix to transform the ray to. Default is the identity matrix.
         * @returns the new ray
         */
        static CreateNewFromTo(origin: Vector3, end: Vector3, world?: DeepImmutable<Matrix>): Ray;
        /**
         * Transforms a ray by a matrix
         * @param ray ray to transform
         * @param matrix matrix to apply
         * @returns the resulting new ray
         */
        static Transform(ray: DeepImmutable<Ray>, matrix: DeepImmutable<Matrix>): Ray;
        /**
         * Transforms a ray by a matrix
         * @param ray ray to transform
         * @param matrix matrix to apply
         * @param result ray to store result in
         */
        static TransformToRef(ray: DeepImmutable<Ray>, matrix: DeepImmutable<Matrix>, result: Ray): void;
        /**
         * Unproject a ray from screen space to object space
         * @param sourceX defines the screen space x coordinate to use
         * @param sourceY defines the screen space y coordinate to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         */
        unprojectRayToRef(sourceX: float, sourceY: float, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>): void;
    }
    /**
     * Type used to define predicate used to select faces when a mesh intersection is detected
     */
    export type TrianglePickingPredicate = (p0: Vector3, p1: Vector3, p2: Vector3, ray: Ray) => boolean;
    module "babylonjs/scene" {
        interface Scene {
            /** @hidden */
            _tempPickingRay: Nullable<Ray>;
            /** @hidden */
            _cachedRayForTransform: Ray;
            /** @hidden */
            _pickWithRayInverseMatrix: Matrix;
            /** @hidden */
            _internalPick(rayFunction: (world: Matrix) => Ray, predicate?: (mesh: AbstractMesh) => boolean, fastCheck?: boolean, onlyBoundingInfo?: boolean, trianglePredicate?: TrianglePickingPredicate): Nullable<PickingInfo>;
            /** @hidden */
            _internalMultiPick(rayFunction: (world: Matrix) => Ray, predicate?: (mesh: AbstractMesh) => boolean, trianglePredicate?: TrianglePickingPredicate): Nullable<PickingInfo[]>;
            /** @hidden */
            _internalPickForMesh(pickingInfo: Nullable<PickingInfo>, rayFunction: (world: Matrix) => Ray, mesh: AbstractMesh, world: Matrix, fastCheck?: boolean, onlyBoundingInfo?: boolean, trianglePredicate?: TrianglePickingPredicate, skipBoundingInfo?: boolean): Nullable<PickingInfo>;
        }
    }
}
declare module "babylonjs/sceneComponent" {
    import { Scene } from "babylonjs/scene";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { _InstancesBatch } from "babylonjs/Meshes/mesh";
    import { SmartArrayNoDuplicate } from "babylonjs/Misc/smartArray";
    import { Nullable } from "babylonjs/types";
    import { Camera } from "babylonjs/Cameras/camera";
    import { RenderTargetTexture } from "babylonjs/Materials/Textures/renderTargetTexture";
    import { PickingInfo } from "babylonjs/Collisions/pickingInfo";
    import { AbstractScene } from "babylonjs/abstractScene";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { Effect } from "babylonjs/Materials/effect";
    /**
     * Groups all the scene component constants in one place to ease maintenance.
     * @hidden
     */
    export class SceneComponentConstants {
        static readonly NAME_EFFECTLAYER: string;
        static readonly NAME_LAYER: string;
        static readonly NAME_LENSFLARESYSTEM: string;
        static readonly NAME_BOUNDINGBOXRENDERER: string;
        static readonly NAME_PARTICLESYSTEM: string;
        static readonly NAME_GAMEPAD: string;
        static readonly NAME_SIMPLIFICATIONQUEUE: string;
        static readonly NAME_GEOMETRYBUFFERRENDERER: string;
        static readonly NAME_PREPASSRENDERER: string;
        static readonly NAME_DEPTHRENDERER: string;
        static readonly NAME_POSTPROCESSRENDERPIPELINEMANAGER: string;
        static readonly NAME_SPRITE: string;
        static readonly NAME_SUBSURFACE: string;
        static readonly NAME_OUTLINERENDERER: string;
        static readonly NAME_PROCEDURALTEXTURE: string;
        static readonly NAME_SHADOWGENERATOR: string;
        static readonly NAME_OCTREE: string;
        static readonly NAME_PHYSICSENGINE: string;
        static readonly NAME_AUDIO: string;
        static readonly STEP_ISREADYFORMESH_EFFECTLAYER: number;
        static readonly STEP_BEFOREEVALUATEACTIVEMESH_BOUNDINGBOXRENDERER: number;
        static readonly STEP_EVALUATESUBMESH_BOUNDINGBOXRENDERER: number;
        static readonly STEP_PREACTIVEMESH_BOUNDINGBOXRENDERER: number;
        static readonly STEP_CAMERADRAWRENDERTARGET_EFFECTLAYER: number;
        static readonly STEP_BEFORECAMERADRAW_EFFECTLAYER: number;
        static readonly STEP_BEFORECAMERADRAW_LAYER: number;
        static readonly STEP_BEFORECAMERADRAW_PREPASS: number;
        static readonly STEP_BEFORERENDERTARGETDRAW_LAYER: number;
        static readonly STEP_BEFORERENDERINGMESH_PREPASS: number;
        static readonly STEP_BEFORERENDERINGMESH_OUTLINE: number;
        static readonly STEP_AFTERRENDERINGMESH_PREPASS: number;
        static readonly STEP_AFTERRENDERINGMESH_OUTLINE: number;
        static readonly STEP_AFTERRENDERINGGROUPDRAW_EFFECTLAYER_DRAW: number;
        static readonly STEP_AFTERRENDERINGGROUPDRAW_BOUNDINGBOXRENDERER: number;
        static readonly STEP_BEFORECAMERAUPDATE_SIMPLIFICATIONQUEUE: number;
        static readonly STEP_BEFORECAMERAUPDATE_GAMEPAD: number;
        static readonly STEP_BEFORECLEAR_PROCEDURALTEXTURE: number;
        static readonly STEP_AFTERRENDERTARGETDRAW_LAYER: number;
        static readonly STEP_AFTERCAMERADRAW_EFFECTLAYER: number;
        static readonly STEP_AFTERCAMERADRAW_LENSFLARESYSTEM: number;
        static readonly STEP_AFTERCAMERADRAW_EFFECTLAYER_DRAW: number;
        static readonly STEP_AFTERCAMERADRAW_LAYER: number;
        static readonly STEP_AFTERCAMERADRAW_PREPASS: number;
        static readonly STEP_AFTERRENDER_AUDIO: number;
        static readonly STEP_GATHERRENDERTARGETS_DEPTHRENDERER: number;
        static readonly STEP_GATHERRENDERTARGETS_GEOMETRYBUFFERRENDERER: number;
        static readonly STEP_GATHERRENDERTARGETS_SHADOWGENERATOR: number;
        static readonly STEP_GATHERRENDERTARGETS_POSTPROCESSRENDERPIPELINEMANAGER: number;
        static readonly STEP_GATHERACTIVECAMERARENDERTARGETS_DEPTHRENDERER: number;
        static readonly STEP_BEFORECLEARSTAGE_PREPASS: number;
        static readonly STEP_POINTERMOVE_SPRITE: number;
        static readonly STEP_POINTERDOWN_SPRITE: number;
        static readonly STEP_POINTERUP_SPRITE: number;
    }
    /**
     * This represents a scene component.
     *
     * This is used to decouple the dependency the scene is having on the different workloads like
     * layers, post processes...
     */
    export interface ISceneComponent {
        /**
         * The name of the component. Each component must have a unique name.
         */
        name: string;
        /**
         * The scene the component belongs to.
         */
        scene: Scene;
        /**
         * Register the component to one instance of a scene.
         */
        register(): void;
        /**
         * Rebuilds the elements related to this component in case of
         * context lost for instance.
         */
        rebuild(): void;
        /**
         * Disposes the component and the associated ressources.
         */
        dispose(): void;
    }
    /**
     * This represents a SERIALIZABLE scene component.
     *
     * This extends Scene Component to add Serialization methods on top.
     */
    export interface ISceneSerializableComponent extends ISceneComponent {
        /**
         * Adds all the elements from the container to the scene
         * @param container the container holding the elements
         */
        addFromContainer(container: AbstractScene): void;
        /**
         * Removes all the elements in the container from the scene
         * @param container contains the elements to remove
         * @param dispose if the removed element should be disposed (default: false)
         */
        removeFromContainer(container: AbstractScene, dispose?: boolean): void;
        /**
         * Serializes the component data to the specified json object
         * @param serializationObject The object to serialize to
         */
        serialize(serializationObject: any): void;
    }
    /**
     * Strong typing of a Mesh related stage step action
     */
    export type MeshStageAction = (mesh: AbstractMesh, hardwareInstancedRendering: boolean) => boolean;
    /**
     * Strong typing of a Evaluate Sub Mesh related stage step action
     */
    export type EvaluateSubMeshStageAction = (mesh: AbstractMesh, subMesh: SubMesh) => void;
    /**
     * Strong typing of a pre active Mesh related stage step action
     */
    export type PreActiveMeshStageAction = (mesh: AbstractMesh) => void;
    /**
     * Strong typing of a Camera related stage step action
     */
    export type CameraStageAction = (camera: Camera) => void;
    /**
     * Strong typing of a Camera Frame buffer related stage step action
     */
    export type CameraStageFrameBufferAction = (camera: Camera) => boolean;
    /**
     * Strong typing of a Render Target related stage step action
     */
    export type RenderTargetStageAction = (renderTarget: RenderTargetTexture) => void;
    /**
     * Strong typing of a RenderingGroup related stage step action
     */
    export type RenderingGroupStageAction = (renderingGroupId: number) => void;
    /**
     * Strong typing of a Mesh Render related stage step action
     */
    export type RenderingMeshStageAction = (mesh: Mesh, subMesh: SubMesh, batch: _InstancesBatch, effect: Nullable<Effect>) => void;
    /**
     * Strong typing of a simple stage step action
     */
    export type SimpleStageAction = () => void;
    /**
     * Strong typing of a render target action.
     */
    export type RenderTargetsStageAction = (renderTargets: SmartArrayNoDuplicate<RenderTargetTexture>) => void;
    /**
     * Strong typing of a pointer move action.
     */
    export type PointerMoveStageAction = (unTranslatedPointerX: number, unTranslatedPointerY: number, pickResult: Nullable<PickingInfo>, isMeshPicked: boolean, element: HTMLElement) => Nullable<PickingInfo>;
    /**
     * Strong typing of a pointer up/down action.
     */
    export type PointerUpDownStageAction = (unTranslatedPointerX: number, unTranslatedPointerY: number, pickResult: Nullable<PickingInfo>, evt: PointerEvent) => Nullable<PickingInfo>;
    /**
     * Representation of a stage in the scene (Basically a list of ordered steps)
     * @hidden
     */
    export class Stage<T extends Function> extends Array<{
        index: number;
        component: ISceneComponent;
        action: T;
    }> {
        /**
         * Hide ctor from the rest of the world.
         * @param items The items to add.
         */
        private constructor();
        /**
         * Creates a new Stage.
         * @returns A new instance of a Stage
         */
        static Create<T extends Function>(): Stage<T>;
        /**
         * Registers a step in an ordered way in the targeted stage.
         * @param index Defines the position to register the step in
         * @param component Defines the component attached to the step
         * @param action Defines the action to launch during the step
         */
        registerStep(index: number, component: ISceneComponent, action: T): void;
        /**
         * Clears all the steps from the stage.
         */
        clear(): void;
    }
}
declare module "babylonjs/Sprites/spriteSceneComponent" {
    import { Nullable } from "babylonjs/types";
    import { Observable } from "babylonjs/Misc/observable";
    import { Scene } from "babylonjs/scene";
    import { Sprite } from "babylonjs/Sprites/sprite";
    import { ISpriteManager } from "babylonjs/Sprites/spriteManager";
    import { Ray } from "babylonjs/Culling/ray";
    import { Camera } from "babylonjs/Cameras/camera";
    import { PickingInfo } from "babylonjs/Collisions/pickingInfo";
    import { ISceneComponent } from "babylonjs/sceneComponent";
    module "babylonjs/scene" {
        interface Scene {
            /** @hidden */
            _pointerOverSprite: Nullable<Sprite>;
            /** @hidden */
            _pickedDownSprite: Nullable<Sprite>;
            /** @hidden */
            _tempSpritePickingRay: Nullable<Ray>;
            /**
             * All of the sprite managers added to this scene
             * @see https://doc.babylonjs.com/babylon101/sprites
             */
            spriteManagers: Array<ISpriteManager>;
            /**
             * An event triggered when sprites rendering is about to start
             * Note: This event can be trigger more than once per frame (because sprites can be rendered by render target textures as well)
             */
            onBeforeSpritesRenderingObservable: Observable<Scene>;
            /**
             * An event triggered when sprites rendering is done
             * Note: This event can be trigger more than once per frame (because sprites can be rendered by render target textures as well)
             */
            onAfterSpritesRenderingObservable: Observable<Scene>;
            /** @hidden */
            _internalPickSprites(ray: Ray, predicate?: (sprite: Sprite) => boolean, fastCheck?: boolean, camera?: Camera): Nullable<PickingInfo>;
            /** Launch a ray to try to pick a sprite in the scene
             * @param x position on screen
             * @param y position on screen
             * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
             * @param fastCheck defines if the first intersection will be used (and not the closest)
             * @param camera camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
             * @returns a PickingInfo
             */
            pickSprite(x: number, y: number, predicate?: (sprite: Sprite) => boolean, fastCheck?: boolean, camera?: Camera): Nullable<PickingInfo>;
            /** Use the given ray to pick a sprite in the scene
             * @param ray The ray (in world space) to use to pick meshes
             * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
             * @param fastCheck defines if the first intersection will be used (and not the closest)
             * @param camera camera to use. Can be set to null. In this case, the scene.activeCamera will be used
             * @returns a PickingInfo
             */
            pickSpriteWithRay(ray: Ray, predicate?: (sprite: Sprite) => boolean, fastCheck?: boolean, camera?: Camera): Nullable<PickingInfo>;
            /** @hidden */
            _internalMultiPickSprites(ray: Ray, predicate?: (sprite: Sprite) => boolean, camera?: Camera): Nullable<PickingInfo[]>;
            /** Launch a ray to try to pick sprites in the scene
             * @param x position on screen
             * @param y position on screen
             * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
             * @param camera camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
             * @returns a PickingInfo array
             */
            multiPickSprite(x: number, y: number, predicate?: (sprite: Sprite) => boolean, camera?: Camera): Nullable<PickingInfo[]>;
            /** Use the given ray to pick sprites in the scene
             * @param ray The ray (in world space) to use to pick meshes
             * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
             * @param camera camera to use. Can be set to null. In this case, the scene.activeCamera will be used
             * @returns a PickingInfo array
             */
            multiPickSpriteWithRay(ray: Ray, predicate?: (sprite: Sprite) => boolean, camera?: Camera): Nullable<PickingInfo[]>;
            /**
             * Force the sprite under the pointer
             * @param sprite defines the sprite to use
             */
            setPointerOverSprite(sprite: Nullable<Sprite>): void;
            /**
             * Gets the sprite under the pointer
             * @returns a Sprite or null if no sprite is under the pointer
             */
            getPointerOverSprite(): Nullable<Sprite>;
        }
    }
    /**
     * Defines the sprite scene component responsible to manage sprites
     * in a given scene.
     */
    export class SpriteSceneComponent implements ISceneComponent {
        /**
         * The component name helpfull to identify the component in the list of scene components.
         */
        readonly name: string;
        /**
         * The scene the component belongs to.
         */
        scene: Scene;
        /** @hidden */
        private _spritePredicate;
        /**
         * Creates a new instance of the component for the given scene
         * @param scene Defines the scene to register the component in
         */
        constructor(scene: Scene);
        /**
         * Registers the component in a given scene
         */
        register(): void;
        /**
         * Rebuilds the elements related to this component in case of
         * context lost for instance.
         */
        rebuild(): void;
        /**
         * Disposes the component and the associated ressources.
         */
        dispose(): void;
        private _pickSpriteButKeepRay;
        private _pointerMove;
        private _pointerDown;
        private _pointerUp;
    }
}
declare module "babylonjs/Misc/timingTools" {
    /**
     * Class used to provide helper for timing
     */
    export class TimingTools {
        /**
         * Polyfill for setImmediate
         * @param action defines the action to execute after the current execution block
         */
        static SetImmediate(action: () => void): void;
    }
}
declare module "babylonjs/Misc/instantiationTools" {
    /**
     * Class used to enable instatition of objects by class name
     */
    export class InstantiationTools {
        /**
         * Use this object to register external classes like custom textures or material
         * to allow the laoders to instantiate them
         */
        static RegisteredExternalClasses: {
            [key: string]: Object;
        };
        /**
         * Tries to instantiate a new object from a given class name
         * @param className defines the class name to instantiate
         * @returns the new object or null if the system was not able to do the instantiation
         */
        static Instantiate(className: string): any;
    }
}
declare module "babylonjs/Misc/copyTools" {
    import { Nullable } from "babylonjs/types";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    /**
     * Class used to host copy specific utilities
     */
    export class CopyTools {
        /**
         * Reads the pixels stored in the webgl texture and returns them as a base64 string
         * @param texture defines the texture to read pixels from
         * @param faceIndex defines the face of the texture to read (in case of cube texture)
         * @param level defines the LOD level of the texture to read (in case of Mip Maps)
         * @returns The base64 encoded string or null
         */
        static GenerateBase64StringFromTexture(texture: BaseTexture, faceIndex?: number, level?: number): Nullable<string>;
    }
}
declare module "babylonjs/Engines/depthTextureCreationOptions" {
    /**
     * Define options used to create a depth texture
     */
    export class DepthTextureCreationOptions {
        /** Specifies whether or not a stencil should be allocated in the texture */
        generateStencil?: boolean;
        /** Specifies whether or not bilinear filtering is enable on the texture */
        bilinearFiltering?: boolean;
        /** Specifies the comparison function to set on the texture. If 0 or undefined, the texture is not in comparison mode */
        comparisonFunction?: number;
        /** Specifies if the created texture is a cube texture */
        isCube?: boolean;
    }
}
declare module "babylonjs/Engines/Extensions/engine.cubeTexture" {
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { DepthTextureCreationOptions } from "babylonjs/Engines/depthTextureCreationOptions";
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Creates a depth stencil cube texture.
             * This is only available in WebGL 2.
             * @param size The size of face edge in the cube texture.
             * @param options The options defining the cube texture.
             * @returns The cube texture
             */
            _createDepthStencilCubeTexture(size: number, options: DepthTextureCreationOptions): InternalTexture;
            /**
             * Creates a cube texture
             * @param rootUrl defines the url where the files to load is located
             * @param scene defines the current scene
             * @param files defines the list of files to load (1 per face)
             * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
             * @param onLoad defines an optional callback raised when the texture is loaded
             * @param onError defines an optional callback raised if there is an issue to load the texture
             * @param format defines the format of the data
             * @param forcedExtension defines the extension to use to pick the right loader
             * @param createPolynomials if a polynomial sphere should be created for the cube texture
             * @param lodScale defines the scale applied to environment texture. This manages the range of LOD level used for IBL according to the roughness
             * @param lodOffset defines the offset applied to environment texture. This manages first LOD level used for IBL according to the roughness
             * @param fallback defines texture to use while falling back when (compressed) texture file not found.
             * @param loaderOptions options to be passed to the loader
             * @returns the cube texture as an InternalTexture
             */
            createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean | undefined, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any, createPolynomials: boolean, lodScale: number, lodOffset: number, fallback: Nullable<InternalTexture>, loaderOptions: any): InternalTexture;
            /**
             * Creates a cube texture
             * @param rootUrl defines the url where the files to load is located
             * @param scene defines the current scene
             * @param files defines the list of files to load (1 per face)
             * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
             * @param onLoad defines an optional callback raised when the texture is loaded
             * @param onError defines an optional callback raised if there is an issue to load the texture
             * @param format defines the format of the data
             * @param forcedExtension defines the extension to use to pick the right loader
             * @returns the cube texture as an InternalTexture
             */
            createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any): InternalTexture;
            /**
             * Creates a cube texture
             * @param rootUrl defines the url where the files to load is located
             * @param scene defines the current scene
             * @param files defines the list of files to load (1 per face)
             * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
             * @param onLoad defines an optional callback raised when the texture is loaded
             * @param onError defines an optional callback raised if there is an issue to load the texture
             * @param format defines the format of the data
             * @param forcedExtension defines the extension to use to pick the right loader
             * @param createPolynomials if a polynomial sphere should be created for the cube texture
             * @param lodScale defines the scale applied to environment texture. This manages the range of LOD level used for IBL according to the roughness
             * @param lodOffset defines the offset applied to environment texture. This manages first LOD level used for IBL according to the roughness
             * @returns the cube texture as an InternalTexture
             */
            createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any, createPolynomials: boolean, lodScale: number, lodOffset: number): InternalTexture;
            /** @hidden */
            _partialLoadFile(url: string, index: number, loadedFiles: ArrayBuffer[], onfinish: (files: ArrayBuffer[]) => void, onErrorCallBack: Nullable<(message?: string, exception?: any) => void>): void;
            /** @hidden */
            _cascadeLoadFiles(scene: Nullable<Scene>, onfinish: (images: ArrayBuffer[]) => void, files: string[], onError: Nullable<(message?: string, exception?: any) => void>): void;
            /** @hidden */
            _cascadeLoadImgs(scene: Nullable<Scene>, onfinish: (images: HTMLImageElement[]) => void, files: string[], onError: Nullable<(message?: string, exception?: any) => void>, mimeType?: string): void;
            /** @hidden */
            _partialLoadImg(url: string, index: number, loadedImages: HTMLImageElement[], scene: Nullable<Scene>, onfinish: (images: HTMLImageElement[]) => void, onErrorCallBack: Nullable<(message?: string, exception?: any) => void>, mimeType?: string): void;
            /**
             * @hidden
             */
            _setCubeMapTextureParams(texture: InternalTexture, loadMipmap: boolean): void;
        }
    }
}
declare module "babylonjs/Materials/Textures/cubeTexture" {
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Matrix, Vector3 } from "babylonjs/Maths/math.vector";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    import "babylonjs/Engines/Extensions/engine.cubeTexture";
    import { Observable } from "babylonjs/Misc/observable";
    /**
     * Class for creating a cube texture
     */
    export class CubeTexture extends BaseTexture {
        private _delayedOnLoad;
        /**
         * Observable triggered once the texture has been loaded.
         */
        onLoadObservable: Observable<CubeTexture>;
        /**
         * The url of the texture
         */
        url: string;
        /**
         * Gets or sets the center of the bounding box associated with the cube texture.
         * It must define where the camera used to render the texture was set
         * @see https://doc.babylonjs.com/how_to/reflect#using-local-cubemap-mode
         */
        boundingBoxPosition: Vector3;
        private _boundingBoxSize;
        /**
         * Gets or sets the size of the bounding box associated with the cube texture
         * When defined, the cubemap will switch to local mode
         * @see https://community.arm.com/graphics/b/blog/posts/reflections-based-on-local-cubemaps-in-unity
         * @example https://www.babylonjs-playground.com/#RNASML
         */
        set boundingBoxSize(value: Vector3);
        /**
         * Returns the bounding box size
         * @see https://doc.babylonjs.com/how_to/reflect#using-local-cubemap-mode
         */
        get boundingBoxSize(): Vector3;
        protected _rotationY: number;
        /**
         * Sets texture matrix rotation angle around Y axis in radians.
         */
        set rotationY(value: number);
        /**
         * Gets texture matrix rotation angle around Y axis radians.
         */
        get rotationY(): number;
        /**
         * Are mip maps generated for this texture or not.
         */
        get noMipmap(): boolean;
        private _noMipmap;
        private _files;
        protected _forcedExtension: Nullable<string>;
        private _extensions;
        private _textureMatrix;
        private _format;
        private _createPolynomials;
        private _loaderOptions;
        /**
         * Creates a cube texture from an array of image urls
         * @param files defines an array of image urls
         * @param scene defines the hosting scene
         * @param noMipmap specifies if mip maps are not used
         * @returns a cube texture
         */
        static CreateFromImages(files: string[], scene: Scene, noMipmap?: boolean): CubeTexture;
        /**
         * Creates and return a texture created from prefilterd data by tools like IBL Baker or Lys.
         * @param url defines the url of the prefiltered texture
         * @param scene defines the scene the texture is attached to
         * @param forcedExtension defines the extension of the file if different from the url
         * @param createPolynomials defines whether or not to create polynomial harmonics from the texture data if necessary
         * @return the prefiltered texture
         */
        static CreateFromPrefilteredData(url: string, scene: Scene, forcedExtension?: any, createPolynomials?: boolean): CubeTexture;
        /**
         * Creates a cube texture to use with reflection for instance. It can be based upon dds or six images as well
         * as prefiltered data.
         * @param rootUrl defines the url of the texture or the root name of the six images
         * @param null defines the scene or engine the texture is attached to
         * @param extensions defines the suffixes add to the picture name in case six images are in use like _px.jpg...
         * @param noMipmap defines if mipmaps should be created or not
         * @param files defines the six files to load for the different faces in that order: px, py, pz, nx, ny, nz
         * @param onLoad defines a callback triggered at the end of the file load if no errors occured
         * @param onError defines a callback triggered in case of error during load
         * @param format defines the internal format to use for the texture once loaded
         * @param prefiltered defines whether or not the texture is created from prefiltered data
         * @param forcedExtension defines the extensions to use (force a special type of file to load) in case it is different from the file name
         * @param createPolynomials defines whether or not to create polynomial harmonics from the texture data if necessary
         * @param lodScale defines the scale applied to environment texture. This manages the range of LOD level used for IBL according to the roughness
         * @param lodOffset defines the offset applied to environment texture. This manages first LOD level used for IBL according to the roughness
         * @param loaderOptions options to be passed to the loader
         * @return the cube texture
         */
        constructor(rootUrl: string, sceneOrEngine: Scene | ThinEngine, extensions?: Nullable<string[]>, noMipmap?: boolean, files?: Nullable<string[]>, onLoad?: Nullable<() => void>, onError?: Nullable<(message?: string, exception?: any) => void>, format?: number, prefiltered?: boolean, forcedExtension?: any, createPolynomials?: boolean, lodScale?: number, lodOffset?: number, loaderOptions?: any);
        /**
         * Get the current class name of the texture useful for serialization or dynamic coding.
         * @returns "CubeTexture"
         */
        getClassName(): string;
        /**
         * Update the url (and optional buffer) of this texture if url was null during construction.
         * @param url the url of the texture
         * @param forcedExtension defines the extension to use
         * @param onLoad callback called when the texture is loaded  (defaults to null)
         * @param prefiltered Defines whether the updated texture is prefiltered or not
         */
        updateURL(url: string, forcedExtension?: string, onLoad?: () => void, prefiltered?: boolean): void;
        /**
         * Delays loading of the cube texture
         * @param forcedExtension defines the extension to use
         */
        delayLoad(forcedExtension?: string): void;
        /**
         * Returns the reflection texture matrix
         * @returns the reflection texture matrix
         */
        getReflectionTextureMatrix(): Matrix;
        /**
         * Sets the reflection texture matrix
         * @param value Reflection texture matrix
         */
        setReflectionTextureMatrix(value: Matrix): void;
        /**
         * Parses text to create a cube texture
         * @param parsedTexture define the serialized text to read from
         * @param scene defines the hosting scene
         * @param rootUrl defines the root url of the cube texture
         * @returns a cube texture
         */
        static Parse(parsedTexture: any, scene: Scene, rootUrl: string): CubeTexture;
        /**
         * Makes a clone, or deep copy, of the cube texture
         * @returns a new cube texture
         */
        clone(): CubeTexture;
    }
}
declare module "babylonjs/Materials/colorCurves" {
    import { Effect } from "babylonjs/Materials/effect";
    /**
     * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
     * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
     * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
     * corresponding to low luminance, medium luminance, and high luminance areas respectively.
     */
    export class ColorCurves {
        private _dirty;
        private _tempColor;
        private _globalCurve;
        private _highlightsCurve;
        private _midtonesCurve;
        private _shadowsCurve;
        private _positiveCurve;
        private _negativeCurve;
        private _globalHue;
        private _globalDensity;
        private _globalSaturation;
        private _globalExposure;
        /**
         * Gets the global Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        get globalHue(): number;
        /**
         * Sets the global Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        set globalHue(value: number);
        /**
         * Gets the global Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        get globalDensity(): number;
        /**
         * Sets the global Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        set globalDensity(value: number);
        /**
         * Gets the global Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        get globalSaturation(): number;
        /**
         * Sets the global Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        set globalSaturation(value: number);
        /**
         * Gets the global Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        get globalExposure(): number;
        /**
         * Sets the global Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        set globalExposure(value: number);
        private _highlightsHue;
        private _highlightsDensity;
        private _highlightsSaturation;
        private _highlightsExposure;
        /**
         * Gets the highlights Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        get highlightsHue(): number;
        /**
         * Sets the highlights Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        set highlightsHue(value: number);
        /**
         * Gets the highlights Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        get highlightsDensity(): number;
        /**
         * Sets the highlights Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        set highlightsDensity(value: number);
        /**
         * Gets the highlights Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        get highlightsSaturation(): number;
        /**
         * Sets the highlights Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        set highlightsSaturation(value: number);
        /**
         * Gets the highlights Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        get highlightsExposure(): number;
        /**
         * Sets the highlights Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        set highlightsExposure(value: number);
        private _midtonesHue;
        private _midtonesDensity;
        private _midtonesSaturation;
        private _midtonesExposure;
        /**
         * Gets the midtones Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        get midtonesHue(): number;
        /**
         * Sets the midtones Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        set midtonesHue(value: number);
        /**
         * Gets the midtones Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        get midtonesDensity(): number;
        /**
         * Sets the midtones Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        set midtonesDensity(value: number);
        /**
         * Gets the midtones Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        get midtonesSaturation(): number;
        /**
         * Sets the midtones Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        set midtonesSaturation(value: number);
        /**
         * Gets the midtones Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        get midtonesExposure(): number;
        /**
         * Sets the midtones Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        set midtonesExposure(value: number);
        private _shadowsHue;
        private _shadowsDensity;
        private _shadowsSaturation;
        private _shadowsExposure;
        /**
         * Gets the shadows Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        get shadowsHue(): number;
        /**
         * Sets the shadows Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        set shadowsHue(value: number);
        /**
         * Gets the shadows Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        get shadowsDensity(): number;
        /**
         * Sets the shadows Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        set shadowsDensity(value: number);
        /**
         * Gets the shadows Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        get shadowsSaturation(): number;
        /**
         * Sets the shadows Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        set shadowsSaturation(value: number);
        /**
         * Gets the shadows Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        get shadowsExposure(): number;
        /**
         * Sets the shadows Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        set shadowsExposure(value: number);
        /**
         * Returns the class name
         * @returns The class name
         */
        getClassName(): string;
        /**
         * Binds the color curves to the shader.
         * @param colorCurves The color curve to bind
         * @param effect The effect to bind to
         * @param positiveUniform The positive uniform shader parameter
         * @param neutralUniform The neutral uniform shader parameter
         * @param negativeUniform The negative uniform shader parameter
         */
        static Bind(colorCurves: ColorCurves, effect: Effect, positiveUniform?: string, neutralUniform?: string, negativeUniform?: string): void;
        /**
         * Prepare the list of uniforms associated with the ColorCurves effects.
         * @param uniformsList The list of uniforms used in the effect
         */
        static PrepareUniforms(uniformsList: string[]): void;
        /**
         * Returns color grading data based on a hue, density, saturation and exposure value.
         * @param filterHue The hue of the color filter.
         * @param filterDensity The density of the color filter.
         * @param saturation The saturation.
         * @param exposure The exposure.
         * @param result The result data container.
         */
        private getColorGradingDataToRef;
        /**
         * Takes an input slider value and returns an adjusted value that provides extra control near the centre.
         * @param value The input slider value in range [-100,100].
         * @returns Adjusted value.
         */
        private static applyColorGradingSliderNonlinear;
        /**
         * Returns an RGBA Color4 based on Hue, Saturation and Brightness (also referred to as value, HSV).
         * @param hue The hue (H) input.
         * @param saturation The saturation (S) input.
         * @param brightness The brightness (B) input.
         * @result An RGBA color represented as Vector4.
         */
        private static fromHSBToRef;
        /**
         * Returns a value clamped between min and max
         * @param value The value to clamp
         * @param min The minimum of value
         * @param max The maximum of value
         * @returns The clamped value.
         */
        private static clamp;
        /**
         * Clones the current color curve instance.
         * @return The cloned curves
         */
        clone(): ColorCurves;
        /**
         * Serializes the current color curve instance to a json representation.
         * @return a JSON representation
         */
        serialize(): any;
        /**
         * Parses the color curve from a json representation.
         * @param source the JSON source to parse
         * @return The parsed curves
         */
        static Parse(source: any): ColorCurves;
    }
}
declare module "babylonjs/Materials/imageProcessingConfiguration" {
    import { Observable } from "babylonjs/Misc/observable";
    import { Nullable } from "babylonjs/types";
    import { Color4 } from "babylonjs/Maths/math.color";
    import { MaterialDefines } from "babylonjs/Materials/materialDefines";
    import { Effect } from "babylonjs/Materials/effect";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { ColorCurves } from "babylonjs/Materials/colorCurves";
    /**
     * Interface to follow in your material defines to integrate easily the
     * Image proccessing functions.
     * @hidden
     */
    export interface IImageProcessingConfigurationDefines {
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        TONEMAPPING_ACES: boolean;
        CONTRAST: boolean;
        EXPOSURE: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
    }
    /**
     * @hidden
     */
    export class ImageProcessingConfigurationDefines extends MaterialDefines implements IImageProcessingConfigurationDefines {
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        TONEMAPPING_ACES: boolean;
        CONTRAST: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
        EXPOSURE: boolean;
        constructor();
    }
    /**
     * This groups together the common properties used for image processing either in direct forward pass
     * or through post processing effect depending on the use of the image processing pipeline in your scene
     * or not.
     */
    export class ImageProcessingConfiguration {
        /**
         * Default tone mapping applied in BabylonJS.
         */
        static readonly TONEMAPPING_STANDARD: number;
        /**
         * ACES Tone mapping (used by default in unreal and unity). This can help getting closer
         * to other engines rendering to increase portability.
         */
        static readonly TONEMAPPING_ACES: number;
        /**
         * Color curves setup used in the effect if colorCurvesEnabled is set to true
         */
        colorCurves: Nullable<ColorCurves>;
        private _colorCurvesEnabled;
        /**
         * Gets wether the color curves effect is enabled.
         */
        get colorCurvesEnabled(): boolean;
        /**
         * Sets wether the color curves effect is enabled.
         */
        set colorCurvesEnabled(value: boolean);
        private _colorGradingTexture;
        /**
         * Color grading LUT texture used in the effect if colorGradingEnabled is set to true
         */
        get colorGradingTexture(): Nullable<BaseTexture>;
        /**
         * Color grading LUT texture used in the effect if colorGradingEnabled is set to true
         */
        set colorGradingTexture(value: Nullable<BaseTexture>);
        private _colorGradingEnabled;
        /**
         * Gets wether the color grading effect is enabled.
         */
        get colorGradingEnabled(): boolean;
        /**
         * Sets wether the color grading effect is enabled.
         */
        set colorGradingEnabled(value: boolean);
        private _colorGradingWithGreenDepth;
        /**
         * Gets wether the color grading effect is using a green depth for the 3d Texture.
         */
        get colorGradingWithGreenDepth(): boolean;
        /**
         * Sets wether the color grading effect is using a green depth for the 3d Texture.
         */
        set colorGradingWithGreenDepth(value: boolean);
        private _colorGradingBGR;
        /**
         * Gets wether the color grading texture contains BGR values.
         */
        get colorGradingBGR(): boolean;
        /**
         * Sets wether the color grading texture contains BGR values.
         */
        set colorGradingBGR(value: boolean);
        /** @hidden */
        _exposure: number;
        /**
         * Gets the Exposure used in the effect.
         */
        get exposure(): number;
        /**
         * Sets the Exposure used in the effect.
         */
        set exposure(value: number);
        private _toneMappingEnabled;
        /**
         * Gets wether the tone mapping effect is enabled.
         */
        get toneMappingEnabled(): boolean;
        /**
         * Sets wether the tone mapping effect is enabled.
         */
        set toneMappingEnabled(value: boolean);
        private _toneMappingType;
        /**
         * Gets the type of tone mapping effect.
         */
        get toneMappingType(): number;
        /**
         * Sets the type of tone mapping effect used in BabylonJS.
         */
        set toneMappingType(value: number);
        protected _contrast: number;
        /**
         * Gets the contrast used in the effect.
         */
        get contrast(): number;
        /**
         * Sets the contrast used in the effect.
         */
        set contrast(value: number);
        /**
         * Vignette stretch size.
         */
        vignetteStretch: number;
        /**
         * Vignette centre X Offset.
         */
        vignetteCentreX: number;
        /**
         * Vignette centre Y Offset.
         */
        vignetteCentreY: number;
        /**
         * Vignette weight or intensity of the vignette effect.
         */
        vignetteWeight: number;
        /**
         * Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        vignetteColor: Color4;
        /**
         * Camera field of view used by the Vignette effect.
         */
        vignetteCameraFov: number;
        private _vignetteBlendMode;
        /**
         * Gets the vignette blend mode allowing different kind of effect.
         */
        get vignetteBlendMode(): number;
        /**
         * Sets the vignette blend mode allowing different kind of effect.
         */
        set vignetteBlendMode(value: number);
        private _vignetteEnabled;
        /**
         * Gets wether the vignette effect is enabled.
         */
        get vignetteEnabled(): boolean;
        /**
         * Sets wether the vignette effect is enabled.
         */
        set vignetteEnabled(value: boolean);
        private _applyByPostProcess;
        /**
         * Gets wether the image processing is applied through a post process or not.
         */
        get applyByPostProcess(): boolean;
        /**
         * Sets wether the image processing is applied through a post process or not.
         */
        set applyByPostProcess(value: boolean);
        private _isEnabled;
        /**
         * Gets wether the image processing is enabled or not.
         */
        get isEnabled(): boolean;
        /**
         * Sets wether the image processing is enabled or not.
         */
        set isEnabled(value: boolean);
        /**
        * An event triggered when the configuration changes and requires Shader to Update some parameters.
        */
        onUpdateParameters: Observable<ImageProcessingConfiguration>;
        /**
         * Method called each time the image processing information changes requires to recompile the effect.
         */
        protected _updateParameters(): void;
        /**
         * Gets the current class name.
         * @return "ImageProcessingConfiguration"
         */
        getClassName(): string;
        /**
         * Prepare the list of uniforms associated with the Image Processing effects.
         * @param uniforms The list of uniforms used in the effect
         * @param defines the list of defines currently in use
         */
        static PrepareUniforms(uniforms: string[], defines: IImageProcessingConfigurationDefines): void;
        /**
         * Prepare the list of samplers associated with the Image Processing effects.
         * @param samplersList The list of uniforms used in the effect
         * @param defines the list of defines currently in use
         */
        static PrepareSamplers(samplersList: string[], defines: IImageProcessingConfigurationDefines): void;
        /**
         * Prepare the list of defines associated to the shader.
         * @param defines the list of defines to complete
         * @param forPostProcess Define if we are currently in post process mode or not
         */
        prepareDefines(defines: IImageProcessingConfigurationDefines, forPostProcess?: boolean): void;
        /**
         * Returns true if all the image processing information are ready.
         * @returns True if ready, otherwise, false
         */
        isReady(): boolean;
        /**
         * Binds the image processing to the shader.
         * @param effect The effect to bind to
         * @param overrideAspectRatio Override the aspect ratio of the effect
         */
        bind(effect: Effect, overrideAspectRatio?: number): void;
        /**
         * Clones the current image processing instance.
         * @return The cloned image processing
         */
        clone(): ImageProcessingConfiguration;
        /**
         * Serializes the current image processing instance to a json representation.
         * @return a JSON representation
         */
        serialize(): any;
        /**
         * Parses the image processing from a json representation.
         * @param source the JSON source to parse
         * @return The parsed image processing
         */
        static Parse(source: any): ImageProcessingConfiguration;
        private static _VIGNETTEMODE_MULTIPLY;
        private static _VIGNETTEMODE_OPAQUE;
        /**
         * Used to apply the vignette as a mix with the pixel color.
         */
        static get VIGNETTEMODE_MULTIPLY(): number;
        /**
         * Used to apply the vignette as a replacement of the pixel color.
         */
        static get VIGNETTEMODE_OPAQUE(): number;
    }
}
declare module "babylonjs/Shaders/postprocess.vertex" {
    /** @hidden */
    export var postprocessVertexShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Engines/Extensions/engine.renderTarget" {
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { RenderTargetCreationOptions } from "babylonjs/Materials/Textures/renderTargetCreationOptions";
    import { DepthTextureCreationOptions } from "babylonjs/Engines/depthTextureCreationOptions";
    /**
     * Type used to define a render target texture size (either with a number or with a rect width and height)
     */
    export type RenderTargetTextureSize = number | {
        width: number;
        height: number;
        layers?: number;
    };
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Creates a new render target texture
             * @param size defines the size of the texture
             * @param options defines the options used to create the texture
             * @returns a new render target texture stored in an InternalTexture
             */
            createRenderTargetTexture(size: RenderTargetTextureSize, options: boolean | RenderTargetCreationOptions): InternalTexture;
            /**
             * Creates a depth stencil texture.
             * This is only available in WebGL 2 or with the depth texture extension available.
             * @param size The size of face edge in the texture.
             * @param options The options defining the texture.
             * @returns The texture
             */
            createDepthStencilTexture(size: RenderTargetTextureSize, options: DepthTextureCreationOptions): InternalTexture;
            /** @hidden */
            _createDepthStencilTexture(size: RenderTargetTextureSize, options: DepthTextureCreationOptions): InternalTexture;
        }
    }
}
declare module "babylonjs/Materials/Node/Enums/nodeMaterialBlockConnectionPointTypes" {
    /**
     * Defines the kind of connection point for node based material
     */
    export enum NodeMaterialBlockConnectionPointTypes {
        /** Float */
        Float = 1,
        /** Int */
        Int = 2,
        /** Vector2 */
        Vector2 = 4,
        /** Vector3 */
        Vector3 = 8,
        /** Vector4 */
        Vector4 = 16,
        /** Color3 */
        Color3 = 32,
        /** Color4 */
        Color4 = 64,
        /** Matrix */
        Matrix = 128,
        /** Custom object */
        Object = 256,
        /** Detect type based on connection */
        AutoDetect = 1024,
        /** Output type that will be defined by input type */
        BasedOnInput = 2048
    }
}
declare module "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets" {
    /**
     * Enum used to define the target of a block
     */
    export enum NodeMaterialBlockTargets {
        /** Vertex shader */
        Vertex = 1,
        /** Fragment shader */
        Fragment = 2,
        /** Neutral */
        Neutral = 4,
        /** Vertex and Fragment */
        VertexAndFragment = 3
    }
}
declare module "babylonjs/Materials/Node/Enums/nodeMaterialBlockConnectionPointMode" {
    /**
     * Enum defining the mode of a NodeMaterialBlockConnectionPoint
     */
    export enum NodeMaterialBlockConnectionPointMode {
        /** Value is an uniform */
        Uniform = 0,
        /** Value is a mesh attribute */
        Attribute = 1,
        /** Value is a varying between vertex and fragment shaders */
        Varying = 2,
        /** Mode is undefined */
        Undefined = 3
    }
}
declare module "babylonjs/Materials/Node/Enums/nodeMaterialSystemValues" {
    /**
     * Enum used to define system values e.g. values automatically provided by the system
     */
    export enum NodeMaterialSystemValues {
        /** World */
        World = 1,
        /** View */
        View = 2,
        /** Projection */
        Projection = 3,
        /** ViewProjection */
        ViewProjection = 4,
        /** WorldView */
        WorldView = 5,
        /** WorldViewProjection */
        WorldViewProjection = 6,
        /** CameraPosition */
        CameraPosition = 7,
        /** Fog Color */
        FogColor = 8,
        /** Delta time */
        DeltaTime = 9
    }
}
declare module "babylonjs/Maths/math.frustum" {
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { DeepImmutable } from "babylonjs/types";
    import { Plane } from "babylonjs/Maths/math.plane";
    /**
     * Represents a camera frustum
     */
    export class Frustum {
        /**
         * Gets the planes representing the frustum
         * @param transform matrix to be applied to the returned planes
         * @returns a new array of 6 Frustum planes computed by the given transformation matrix.
         */
        static GetPlanes(transform: DeepImmutable<Matrix>): Plane[];
        /**
         * Gets the near frustum plane transformed by the transform matrix
         * @param transform transformation matrix to be applied to the resulting frustum plane
         * @param frustumPlane the resuling frustum plane
         */
        static GetNearPlaneToRef(transform: DeepImmutable<Matrix>, frustumPlane: Plane): void;
        /**
         * Gets the far frustum plane transformed by the transform matrix
         * @param transform transformation matrix to be applied to the resulting frustum plane
         * @param frustumPlane the resuling frustum plane
         */
        static GetFarPlaneToRef(transform: DeepImmutable<Matrix>, frustumPlane: Plane): void;
        /**
         * Gets the left frustum plane transformed by the transform matrix
         * @param transform transformation matrix to be applied to the resulting frustum plane
         * @param frustumPlane the resuling frustum plane
         */
        static GetLeftPlaneToRef(transform: DeepImmutable<Matrix>, frustumPlane: Plane): void;
        /**
         * Gets the right frustum plane transformed by the transform matrix
         * @param transform transformation matrix to be applied to the resulting frustum plane
         * @param frustumPlane the resuling frustum plane
         */
        static GetRightPlaneToRef(transform: DeepImmutable<Matrix>, frustumPlane: Plane): void;
        /**
         * Gets the top frustum plane transformed by the transform matrix
         * @param transform transformation matrix to be applied to the resulting frustum plane
         * @param frustumPlane the resuling frustum plane
         */
        static GetTopPlaneToRef(transform: DeepImmutable<Matrix>, frustumPlane: Plane): void;
        /**
         * Gets the bottom frustum plane transformed by the transform matrix
         * @param transform transformation matrix to be applied to the resulting frustum plane
         * @param frustumPlane the resuling frustum plane
         */
        static GetBottomPlaneToRef(transform: DeepImmutable<Matrix>, frustumPlane: Plane): void;
        /**
         * Sets the given array "frustumPlanes" with the 6 Frustum planes computed by the given transformation matrix.
         * @param transform transformation matrix to be applied to the resulting frustum planes
         * @param frustumPlanes the resuling frustum planes
         */
        static GetPlanesToRef(transform: DeepImmutable<Matrix>, frustumPlanes: Plane[]): void;
    }
}
declare module "babylonjs/Maths/math.vertexFormat" {
    import { Vector3, Vector2 } from "babylonjs/Maths/math.vector";
    /**
     * Contains position and normal vectors for a vertex
     */
    export class PositionNormalVertex {
        /** the position of the vertex (defaut: 0,0,0) */
        position: Vector3;
        /** the normal of the vertex (defaut: 0,1,0) */
        normal: Vector3;
        /**
         * Creates a PositionNormalVertex
         * @param position the position of the vertex (defaut: 0,0,0)
         * @param normal the normal of the vertex (defaut: 0,1,0)
         */
        constructor(
        /** the position of the vertex (defaut: 0,0,0) */
        position?: Vector3, 
        /** the normal of the vertex (defaut: 0,1,0) */
        normal?: Vector3);
        /**
         * Clones the PositionNormalVertex
         * @returns the cloned PositionNormalVertex
         */
        clone(): PositionNormalVertex;
    }
    /**
     * Contains position, normal and uv vectors for a vertex
     */
    export class PositionNormalTextureVertex {
        /** the position of the vertex (defaut: 0,0,0) */
        position: Vector3;
        /** the normal of the vertex (defaut: 0,1,0) */
        normal: Vector3;
        /** the uv of the vertex (default: 0,0) */
        uv: Vector2;
        /**
         * Creates a PositionNormalTextureVertex
         * @param position the position of the vertex (defaut: 0,0,0)
         * @param normal the normal of the vertex (defaut: 0,1,0)
         * @param uv the uv of the vertex (default: 0,0)
         */
        constructor(
        /** the position of the vertex (defaut: 0,0,0) */
        position?: Vector3, 
        /** the normal of the vertex (defaut: 0,1,0) */
        normal?: Vector3, 
        /** the uv of the vertex (default: 0,0) */
        uv?: Vector2);
        /**
         * Clones the PositionNormalTextureVertex
         * @returns the cloned PositionNormalTextureVertex
         */
        clone(): PositionNormalTextureVertex;
    }
}
declare module "babylonjs/Maths/math" {
    export * from "babylonjs/Maths/math.axis";
    export * from "babylonjs/Maths/math.color";
    export * from "babylonjs/Maths/math.constants";
    export * from "babylonjs/Maths/math.frustum";
    export * from "babylonjs/Maths/math.path";
    export * from "babylonjs/Maths/math.plane";
    export * from "babylonjs/Maths/math.size";
    export * from "babylonjs/Maths/math.vector";
    export * from "babylonjs/Maths/math.vertexFormat";
    export * from "babylonjs/Maths/math.viewport";
}
declare module "babylonjs/Materials/Node/Blocks/Input/animatedInputBlockTypes" {
    /**
     * Enum defining the type of animations supported by InputBlock
     */
    export enum AnimatedInputBlockTypes {
        /** No animation */
        None = 0,
        /** Time based animation. Will only work for floats */
        Time = 1
    }
}
declare module "babylonjs/Materials/Node/Blocks/Input/inputBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBlockConnectionPointTypes } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockConnectionPointTypes";
    import { NodeMaterialSystemValues } from "babylonjs/Materials/Node/Enums/nodeMaterialSystemValues";
    import { Nullable } from "babylonjs/types";
    import { Effect } from "babylonjs/Materials/effect";
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { Scene } from "babylonjs/scene";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialBlockTargets } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets";
    import { AnimatedInputBlockTypes } from "babylonjs/Materials/Node/Blocks/Input/animatedInputBlockTypes";
    import { Observable } from "babylonjs/Misc/observable";
    /**
     * Block used to expose an input value
     */
    export class InputBlock extends NodeMaterialBlock {
        private _mode;
        private _associatedVariableName;
        private _storedValue;
        private _valueCallback;
        private _type;
        private _animationType;
        /** Gets or set a value used to limit the range of float values */
        min: number;
        /** Gets or set a value used to limit the range of float values */
        max: number;
        /** Gets or set a value indicating that this input can only get 0 and 1 values */
        isBoolean: boolean;
        /** Gets or sets a value used by the Node Material editor to determine how to configure the current value if it is a matrix */
        matrixMode: number;
        /** @hidden */
        _systemValue: Nullable<NodeMaterialSystemValues>;
        /** Gets or sets a boolean indicating that the value of this input will not change after a build */
        isConstant: boolean;
        /** Gets or sets the group to use to display this block in the Inspector */
        groupInInspector: string;
        /** Gets an observable raised when the value is changed */
        onValueChangedObservable: Observable<InputBlock>;
        /** Gets or sets a boolean indicating if content needs to be converted to gamma space (for color3/4 only) */
        convertToGammaSpace: boolean;
        /** Gets or sets a boolean indicating if content needs to be converted to linear space (for color3/4 only) */
        convertToLinearSpace: boolean;
        /**
         * Gets or sets the connection point type (default is float)
         */
        get type(): NodeMaterialBlockConnectionPointTypes;
        /**
         * Creates a new InputBlock
         * @param name defines the block name
         * @param target defines the target of that block (Vertex by default)
         * @param type defines the type of the input (can be set to NodeMaterialBlockConnectionPointTypes.AutoDetect)
         */
        constructor(name: string, target?: NodeMaterialBlockTargets, type?: NodeMaterialBlockConnectionPointTypes);
        /**
        * Validates if a name is a reserve word.
        * @param newName the new name to be given to the node.
        * @returns false if the name is a reserve word, else true.
        */
        validateBlockName(newName: string): boolean;
        /**
         * Gets the output component
         */
        get output(): NodeMaterialConnectionPoint;
        /**
         * Set the source of this connection point to a vertex attribute
         * @param attributeName defines the attribute name (position, uv, normal, etc...). If not specified it will take the connection point name
         * @returns the current connection point
         */
        setAsAttribute(attributeName?: string): InputBlock;
        /**
         * Set the source of this connection point to a system value
         * @param value define the system value to use (world, view, etc...) or null to switch to manual value
         * @returns the current connection point
         */
        setAsSystemValue(value: Nullable<NodeMaterialSystemValues>): InputBlock;
        /**
         * Gets or sets the value of that point.
         * Please note that this value will be ignored if valueCallback is defined
         */
        get value(): any;
        set value(value: any);
        /**
         * Gets or sets a callback used to get the value of that point.
         * Please note that setting this value will force the connection point to ignore the value property
         */
        get valueCallback(): () => any;
        set valueCallback(value: () => any);
        /**
         * Gets or sets the associated variable name in the shader
         */
        get associatedVariableName(): string;
        set associatedVariableName(value: string);
        /** Gets or sets the type of animation applied to the input */
        get animationType(): AnimatedInputBlockTypes;
        set animationType(value: AnimatedInputBlockTypes);
        /**
         * Gets a boolean indicating that this connection point not defined yet
         */
        get isUndefined(): boolean;
        /**
         * Gets or sets a boolean indicating that this connection point is coming from an uniform.
         * In this case the connection point name must be the name of the uniform to use.
         * Can only be set on inputs
         */
        get isUniform(): boolean;
        set isUniform(value: boolean);
        /**
         * Gets or sets a boolean indicating that this connection point is coming from an attribute.
         * In this case the connection point name must be the name of the attribute to use
         * Can only be set on inputs
         */
        get isAttribute(): boolean;
        set isAttribute(value: boolean);
        /**
         * Gets or sets a boolean indicating that this connection point is generating a varying variable.
         * Can only be set on exit points
         */
        get isVarying(): boolean;
        set isVarying(value: boolean);
        /**
         * Gets a boolean indicating that the current connection point is a system value
         */
        get isSystemValue(): boolean;
        /**
         * Gets or sets the current well known value or null if not defined as a system value
         */
        get systemValue(): Nullable<NodeMaterialSystemValues>;
        set systemValue(value: Nullable<NodeMaterialSystemValues>);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Animate the input if animationType !== None
         * @param scene defines the rendering scene
         */
        animate(scene: Scene): void;
        private _emitDefine;
        initialize(state: NodeMaterialBuildState): void;
        /**
         * Set the input block to its default value (based on its type)
         */
        setDefaultValue(): void;
        private _emitConstant;
        /** @hidden */
        get _noContextSwitch(): boolean;
        private _emit;
        /** @hidden */
        _transmitWorld(effect: Effect, world: Matrix, worldView: Matrix, worldViewProjection: Matrix): void;
        /** @hidden */
        _transmit(effect: Effect, scene: Scene): void;
        protected _buildBlock(state: NodeMaterialBuildState): void;
        protected _dumpPropertiesCode(): string;
        dispose(): void;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint" {
    import { NodeMaterialBlockConnectionPointTypes } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockConnectionPointTypes";
    import { NodeMaterialBlockTargets } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets";
    import { Nullable } from "babylonjs/types";
    import { InputBlock } from "babylonjs/Materials/Node/Blocks/Input/inputBlock";
    import { Observable } from "babylonjs/Misc/observable";
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    /**
     * Enum used to define the compatibility state between two connection points
     */
    export enum NodeMaterialConnectionPointCompatibilityStates {
        /** Points are compatibles */
        Compatible = 0,
        /** Points are incompatible because of their types */
        TypeIncompatible = 1,
        /** Points are incompatible because of their targets (vertex vs fragment) */
        TargetIncompatible = 2
    }
    /**
     * Defines the direction of a connection point
     */
    export enum NodeMaterialConnectionPointDirection {
        /** Input */
        Input = 0,
        /** Output */
        Output = 1
    }
    /**
     * Defines a connection point for a block
     */
    export class NodeMaterialConnectionPoint {
        /**
         * Checks if two types are equivalent
         * @param type1 type 1 to check
         * @param type2 type 2 to check
         * @returns true if both types are equivalent, else false
         */
        static AreEquivalentTypes(type1: number, type2: number): boolean;
        /** @hidden */
        _ownerBlock: NodeMaterialBlock;
        /** @hidden */
        _connectedPoint: Nullable<NodeMaterialConnectionPoint>;
        private _endpoints;
        private _associatedVariableName;
        private _direction;
        /** @hidden */
        _typeConnectionSource: Nullable<NodeMaterialConnectionPoint>;
        /** @hidden */
        _linkedConnectionSource: Nullable<NodeMaterialConnectionPoint>;
        /** @hidden */
        _acceptedConnectionPointType: Nullable<NodeMaterialConnectionPoint>;
        private _type;
        /** @hidden */
        _enforceAssociatedVariableName: boolean;
        /** Gets the direction of the point */
        get direction(): NodeMaterialConnectionPointDirection;
        /** Indicates that this connection point needs dual validation before being connected to another point */
        needDualDirectionValidation: boolean;
        /**
         * Gets or sets the additional types supported by this connection point
         */
        acceptedConnectionPointTypes: NodeMaterialBlockConnectionPointTypes[];
        /**
         * Gets or sets the additional types excluded by this connection point
         */
        excludedConnectionPointTypes: NodeMaterialBlockConnectionPointTypes[];
        /**
         * Observable triggered when this point is connected
         */
        onConnectionObservable: Observable<NodeMaterialConnectionPoint>;
        /**
         * Gets or sets the associated variable name in the shader
         */
        get associatedVariableName(): string;
        set associatedVariableName(value: string);
        /** Get the inner type (ie AutoDetect for instance instead of the inferred one) */
        get innerType(): NodeMaterialBlockConnectionPointTypes;
        /**
         * Gets or sets the connection point type (default is float)
         */
        get type(): NodeMaterialBlockConnectionPointTypes;
        set type(value: NodeMaterialBlockConnectionPointTypes);
        /**
         * Gets or sets the connection point name
         */
        name: string;
        /**
         * Gets or sets the connection point name
         */
        displayName: string;
        /**
         * Gets or sets a boolean indicating that this connection point can be omitted
         */
        isOptional: boolean;
        /**
         * Gets or sets a boolean indicating that this connection point is exposed on a frame
         */
        isExposedOnFrame: boolean;
        /**
         * Gets or sets number indicating the position that the port is exposed to on a frame
         */
        exposedPortPosition: number;
        /**
         * Gets or sets a string indicating that this uniform must be defined under a #ifdef
         */
        define: string;
        /** @hidden */
        _prioritizeVertex: boolean;
        private _target;
        /** Gets or sets the target of that connection point */
        get target(): NodeMaterialBlockTargets;
        set target(value: NodeMaterialBlockTargets);
        /**
         * Gets a boolean indicating that the current point is connected to another NodeMaterialBlock
         */
        get isConnected(): boolean;
        /**
         * Gets a boolean indicating that the current point is connected to an input block
         */
        get isConnectedToInputBlock(): boolean;
        /**
         * Gets a the connected input block (if any)
         */
        get connectInputBlock(): Nullable<InputBlock>;
        /** Get the other side of the connection (if any) */
        get connectedPoint(): Nullable<NodeMaterialConnectionPoint>;
        /** Get the block that owns this connection point */
        get ownerBlock(): NodeMaterialBlock;
        /** Get the block connected on the other side of this connection (if any) */
        get sourceBlock(): Nullable<NodeMaterialBlock>;
        /** Get the block connected on the endpoints of this connection (if any) */
        get connectedBlocks(): Array<NodeMaterialBlock>;
        /** Gets the list of connected endpoints */
        get endpoints(): NodeMaterialConnectionPoint[];
        /** Gets a boolean indicating if that output point is connected to at least one input */
        get hasEndpoints(): boolean;
        /** Gets a boolean indicating that this connection will be used in the vertex shader */
        get isConnectedInVertexShader(): boolean;
        /** Gets a boolean indicating that this connection will be used in the fragment shader */
        get isConnectedInFragmentShader(): boolean;
        /**
         * Creates a block suitable to be used as an input for this input point.
         * If null is returned, a block based on the point type will be created.
         * @returns The returned string parameter is the name of the output point of NodeMaterialBlock (first parameter of the returned array) that can be connected to the input
         */
        createCustomInputBlock(): Nullable<[NodeMaterialBlock, string]>;
        /**
         * Creates a new connection point
         * @param name defines the connection point name
         * @param ownerBlock defines the block hosting this connection point
         * @param direction defines the direction of the connection point
         */
        constructor(name: string, ownerBlock: NodeMaterialBlock, direction: NodeMaterialConnectionPointDirection);
        /**
         * Gets the current class name e.g. "NodeMaterialConnectionPoint"
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets a boolean indicating if the current point can be connected to another point
         * @param connectionPoint defines the other connection point
         * @returns a boolean
         */
        canConnectTo(connectionPoint: NodeMaterialConnectionPoint): boolean;
        /**
         * Gets a number indicating if the current point can be connected to another point
         * @param connectionPoint defines the other connection point
         * @returns a number defining the compatibility state
         */
        checkCompatibilityState(connectionPoint: NodeMaterialConnectionPoint): NodeMaterialConnectionPointCompatibilityStates;
        /**
         * Connect this point to another connection point
         * @param connectionPoint defines the other connection point
         * @param ignoreConstraints defines if the system will ignore connection type constraints (default is false)
         * @returns the current connection point
         */
        connectTo(connectionPoint: NodeMaterialConnectionPoint, ignoreConstraints?: boolean): NodeMaterialConnectionPoint;
        /**
         * Disconnect this point from one of his endpoint
         * @param endpoint defines the other connection point
         * @returns the current connection point
         */
        disconnectFrom(endpoint: NodeMaterialConnectionPoint): NodeMaterialConnectionPoint;
        /**
         * Serializes this point in a JSON representation
         * @param isInput defines if the connection point is an input (default is true)
         * @returns the serialized point object
         */
        serialize(isInput?: boolean): any;
        /**
         * Release resources
         */
        dispose(): void;
    }
}
declare module "babylonjs/Materials/Node/Enums/nodeMaterialModes" {
    /**
     * Enum used to define the material modes
     */
    export enum NodeMaterialModes {
        /** Regular material */
        Material = 0,
        /** For post process */
        PostProcess = 1,
        /** For particle system */
        Particle = 2,
        /** For procedural texture */
        ProceduralTexture = 3
    }
}
declare module "babylonjs/Shaders/ShadersInclude/helperFunctions" {
    /** @hidden */
    export var helperFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Materials/Node/Blocks/Dual/textureBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialBlockTargets } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { NodeMaterial, NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { Effect } from "babylonjs/Materials/effect";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { Nullable } from "babylonjs/types";
    import { Texture } from "babylonjs/Materials/Textures/texture";
    import { Scene } from "babylonjs/scene";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    /**
     * Block used to read a texture from a sampler
     */
    export class TextureBlock extends NodeMaterialBlock {
        private _defineName;
        private _linearDefineName;
        private _gammaDefineName;
        private _tempTextureRead;
        private _samplerName;
        private _transformedUVName;
        private _textureTransformName;
        private _textureInfoName;
        private _mainUVName;
        private _mainUVDefineName;
        private _fragmentOnly;
        /**
         * Gets or sets the texture associated with the node
         */
        texture: Nullable<Texture>;
        /**
         * Gets or sets a boolean indicating if content needs to be converted to gamma space
         */
        convertToGammaSpace: boolean;
        /**
         * Gets or sets a boolean indicating if content needs to be converted to linear space
         */
        convertToLinearSpace: boolean;
        /**
         * Create a new TextureBlock
         * @param name defines the block name
         */
        constructor(name: string, fragmentOnly?: boolean);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the uv input component
         */
        get uv(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgba output component
         */
        get rgba(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgb output component
         */
        get rgb(): NodeMaterialConnectionPoint;
        /**
         * Gets the r output component
         */
        get r(): NodeMaterialConnectionPoint;
        /**
         * Gets the g output component
         */
        get g(): NodeMaterialConnectionPoint;
        /**
         * Gets the b output component
         */
        get b(): NodeMaterialConnectionPoint;
        /**
         * Gets the a output component
         */
        get a(): NodeMaterialConnectionPoint;
        get target(): NodeMaterialBlockTargets;
        autoConfigure(material: NodeMaterial): void;
        initializeDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines, useInstances?: boolean): void;
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines): void;
        isReady(): boolean;
        bind(effect: Effect, nodeMaterial: NodeMaterial, mesh?: Mesh): void;
        private get _isMixed();
        private _injectVertexCode;
        private _writeTextureRead;
        private _writeOutput;
        protected _buildBlock(state: NodeMaterialBuildState): this | undefined;
        protected _dumpPropertiesCode(): string;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Shaders/ShadersInclude/reflectionFunction" {
    /** @hidden */
    export var reflectionFunction: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Materials/Node/Blocks/Dual/reflectionTextureBaseBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { NodeMaterial, NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { Effect } from "babylonjs/Materials/effect";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import "babylonjs/Shaders/ShadersInclude/reflectionFunction";
    /**
     * Base block used to read a reflection texture from a sampler
     */
    export abstract class ReflectionTextureBaseBlock extends NodeMaterialBlock {
        /** @hidden */
        _define3DName: string;
        /** @hidden */
        _defineCubicName: string;
        /** @hidden */
        _defineExplicitName: string;
        /** @hidden */
        _defineProjectionName: string;
        /** @hidden */
        _defineLocalCubicName: string;
        /** @hidden */
        _defineSphericalName: string;
        /** @hidden */
        _definePlanarName: string;
        /** @hidden */
        _defineEquirectangularName: string;
        /** @hidden */
        _defineMirroredEquirectangularFixedName: string;
        /** @hidden */
        _defineEquirectangularFixedName: string;
        /** @hidden */
        _defineSkyboxName: string;
        /** @hidden */
        _defineOppositeZ: string;
        /** @hidden */
        _cubeSamplerName: string;
        /** @hidden */
        _2DSamplerName: string;
        protected _positionUVWName: string;
        protected _directionWName: string;
        protected _reflectionVectorName: string;
        /** @hidden */
        _reflectionCoordsName: string;
        /** @hidden */
        _reflectionMatrixName: string;
        protected _reflectionColorName: string;
        /**
         * Gets or sets the texture associated with the node
         */
        texture: Nullable<BaseTexture>;
        /**
         * Create a new ReflectionTextureBaseBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the world position input component
         */
        abstract get position(): NodeMaterialConnectionPoint;
        /**
         * Gets the world position input component
         */
        abstract get worldPosition(): NodeMaterialConnectionPoint;
        /**
         * Gets the world normal input component
         */
        abstract get worldNormal(): NodeMaterialConnectionPoint;
        /**
         * Gets the world input component
         */
        abstract get world(): NodeMaterialConnectionPoint;
        /**
        * Gets the camera (or eye) position component
        */
        abstract get cameraPosition(): NodeMaterialConnectionPoint;
        /**
         * Gets the view input component
         */
        abstract get view(): NodeMaterialConnectionPoint;
        protected _getTexture(): Nullable<BaseTexture>;
        autoConfigure(material: NodeMaterial): void;
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines): void;
        isReady(): boolean;
        bind(effect: Effect, nodeMaterial: NodeMaterial, mesh?: Mesh): void;
        /**
         * Gets the code to inject in the vertex shader
         * @param state current state of the node material building
         * @returns the shader code
         */
        handleVertexSide(state: NodeMaterialBuildState): string;
        /**
         * Handles the inits for the fragment code path
         * @param state node material build state
         */
        handleFragmentSideInits(state: NodeMaterialBuildState): void;
        /**
         * Generates the reflection coords code for the fragment code path
         * @param worldNormalVarName name of the world normal variable
         * @param worldPos name of the world position variable. If not provided, will use the world position connected to this block
         * @param onlyReflectionVector if true, generates code only for the reflection vector computation, not for the reflection coordinates
         * @returns the shader code
         */
        handleFragmentSideCodeReflectionCoords(worldNormalVarName: string, worldPos?: string, onlyReflectionVector?: boolean): string;
        /**
         * Generates the reflection color code for the fragment code path
         * @param lodVarName name of the lod variable
         * @param swizzleLookupTexture swizzle to use for the final color variable
         * @returns the shader code
         */
        handleFragmentSideCodeReflectionColor(lodVarName?: string, swizzleLookupTexture?: string): string;
        /**
         * Generates the code corresponding to the connected output points
         * @param state node material build state
         * @param varName name of the variable to output
         * @returns the shader code
         */
        writeOutputs(state: NodeMaterialBuildState, varName: string): string;
        protected _buildBlock(state: NodeMaterialBuildState): this;
        protected _dumpPropertiesCode(): string;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterialConnectionPointCustomObject" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialConnectionPoint, NodeMaterialConnectionPointDirection, NodeMaterialConnectionPointCompatibilityStates } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { Nullable } from "babylonjs/types";
    /**
     * Defines a connection point to be used for points with a custom object type
     */
    export class NodeMaterialConnectionPointCustomObject<T extends NodeMaterialBlock> extends NodeMaterialConnectionPoint {
        private _blockType;
        private _blockName;
        private _nameForCheking?;
        /**
         * Creates a new connection point
         * @param name defines the connection point name
         * @param ownerBlock defines the block hosting this connection point
         * @param direction defines the direction of the connection point
         */
        constructor(name: string, ownerBlock: NodeMaterialBlock, direction: NodeMaterialConnectionPointDirection, _blockType: new (...args: any[]) => T, _blockName: string, _nameForCheking?: string | undefined);
        /**
         * Gets a number indicating if the current point can be connected to another point
         * @param connectionPoint defines the other connection point
         * @returns a number defining the compatibility state
         */
        checkCompatibilityState(connectionPoint: NodeMaterialConnectionPoint): NodeMaterialConnectionPointCompatibilityStates;
        /**
         * Creates a block suitable to be used as an input for this input point.
         * If null is returned, a block based on the point type will be created.
         * @returns The returned string parameter is the name of the output point of NodeMaterialBlock (first parameter of the returned array) that can be connected to the input
         */
        createCustomInputBlock(): Nullable<[NodeMaterialBlock, string]>;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterialDecorator" {
    /**
     * Enum defining the type of properties that can be edited in the property pages in the NME
     */
    export enum PropertyTypeForEdition {
        /** property is a boolean */
        Boolean = 0,
        /** property is a float */
        Float = 1,
        /** property is a Vector2 */
        Vector2 = 2,
        /** property is a list of values */
        List = 3
    }
    /**
     * Interface that defines an option in a variable of type list
     */
    export interface IEditablePropertyListOption {
        /** label of the option */
        "label": string;
        /** value of the option */
        "value": number;
    }
    /**
     * Interface that defines the options available for an editable property
     */
    export interface IEditablePropertyOption {
        /** min value */
        "min"?: number;
        /** max value */
        "max"?: number;
        /** notifiers: indicates which actions to take when the property is changed */
        "notifiers"?: {
            /** the material should be rebuilt */
            "rebuild"?: boolean;
            /** the preview should be updated */
            "update"?: boolean;
        };
        /** list of the options for a variable of type list */
        "options"?: IEditablePropertyListOption[];
    }
    /**
     * Interface that describes an editable property
     */
    export interface IPropertyDescriptionForEdition {
        /** name of the property */
        "propertyName": string;
        /** display name of the property */
        "displayName": string;
        /** type of the property */
        "type": PropertyTypeForEdition;
        /** group of the property - all properties with the same group value will be displayed in a specific section */
        "groupName": string;
        /** options for the property */
        "options": IEditablePropertyOption;
    }
    /**
     * Decorator that flags a property in a node material block as being editable
     */
    export function editableInPropertyPage(displayName: string, propertyType?: PropertyTypeForEdition, groupName?: string, options?: IEditablePropertyOption): (target: any, propertyKey: string) => void;
}
declare module "babylonjs/Materials/Node/Blocks/PBR/refractionBlock" {
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { NodeMaterial, NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Nullable } from "babylonjs/types";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { Effect } from "babylonjs/Materials/effect";
    import { Scene } from "babylonjs/scene";
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    /**
     * Block used to implement the refraction part of the sub surface module of the PBR material
     */
    export class RefractionBlock extends NodeMaterialBlock {
        /** @hidden */
        _define3DName: string;
        /** @hidden */
        _refractionMatrixName: string;
        /** @hidden */
        _defineLODRefractionAlpha: string;
        /** @hidden */
        _defineLinearSpecularRefraction: string;
        /** @hidden */
        _defineOppositeZ: string;
        /** @hidden */
        _cubeSamplerName: string;
        /** @hidden */
        _2DSamplerName: string;
        /** @hidden */
        _vRefractionMicrosurfaceInfosName: string;
        /** @hidden */
        _vRefractionInfosName: string;
        /** @hidden */
        _vRefractionFilteringInfoName: string;
        private _scene;
        /**
         * The properties below are set by the main PBR block prior to calling methods of this class.
         * This is to avoid having to add them as inputs here whereas they are already inputs of the main block, so already known.
         * It's less burden on the user side in the editor part.
        */
        /** @hidden */
        viewConnectionPoint: NodeMaterialConnectionPoint;
        /** @hidden */
        indexOfRefractionConnectionPoint: NodeMaterialConnectionPoint;
        /**
         * This parameters will make the material used its opacity to control how much it is refracting aginst not.
         * Materials half opaque for instance using refraction could benefit from this control.
         */
        linkRefractionWithTransparency: boolean;
        /**
         * Controls if refraction needs to be inverted on Y. This could be useful for procedural texture.
         */
        invertRefractionY: boolean;
        /**
         * Gets or sets the texture associated with the node
         */
        texture: Nullable<BaseTexture>;
        /**
         * Create a new RefractionBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the intensity input component
         */
        get intensity(): NodeMaterialConnectionPoint;
        /**
         * Gets the tint at distance input component
         */
        get tintAtDistance(): NodeMaterialConnectionPoint;
        /**
         * Gets the view input component
         */
        get view(): NodeMaterialConnectionPoint;
        /**
         * Gets the refraction object output component
         */
        get refraction(): NodeMaterialConnectionPoint;
        /**
         * Returns true if the block has a texture
         */
        get hasTexture(): boolean;
        protected _getTexture(): Nullable<BaseTexture>;
        autoConfigure(material: NodeMaterial): void;
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines): void;
        isReady(): boolean;
        bind(effect: Effect, nodeMaterial: NodeMaterial, mesh?: Mesh, subMesh?: SubMesh): void;
        /**
         * Gets the main code of the block (fragment side)
         * @param state current state of the node material building
         * @returns the shader code
         */
        getCode(state: NodeMaterialBuildState): string;
        protected _buildBlock(state: NodeMaterialBuildState): this;
        protected _dumpPropertiesCode(): string;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/Node/Blocks/Dual/currentScreenBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialBlockTargets } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { NodeMaterial } from "babylonjs/Materials/Node/nodeMaterial";
    /**
     * Base block used as input for post process
     */
    export class CurrentScreenBlock extends NodeMaterialBlock {
        private _samplerName;
        private _linearDefineName;
        private _gammaDefineName;
        private _mainUVName;
        private _tempTextureRead;
        /**
         * Gets or sets the texture associated with the node
         */
        texture: Nullable<BaseTexture>;
        /**
         * Gets or sets a boolean indicating if content needs to be converted to gamma space
         */
        convertToGammaSpace: boolean;
        /**
         * Gets or sets a boolean indicating if content needs to be converted to linear space
         */
        convertToLinearSpace: boolean;
        /**
         * Create a new CurrentScreenBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the uv input component
         */
        get uv(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgba output component
         */
        get rgba(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgb output component
         */
        get rgb(): NodeMaterialConnectionPoint;
        /**
         * Gets the r output component
         */
        get r(): NodeMaterialConnectionPoint;
        /**
         * Gets the g output component
         */
        get g(): NodeMaterialConnectionPoint;
        /**
         * Gets the b output component
         */
        get b(): NodeMaterialConnectionPoint;
        /**
         * Gets the a output component
         */
        get a(): NodeMaterialConnectionPoint;
        /**
         * Initialize the block and prepare the context for build
         * @param state defines the state that will be used for the build
         */
        initialize(state: NodeMaterialBuildState): void;
        get target(): NodeMaterialBlockTargets;
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines): void;
        isReady(): boolean;
        private _injectVertexCode;
        private _writeTextureRead;
        private _writeOutput;
        protected _buildBlock(state: NodeMaterialBuildState): this | undefined;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/Node/Blocks/Particle/particleTextureBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { NodeMaterial } from "babylonjs/Materials/Node/nodeMaterial";
    /**
     * Base block used for the particle texture
     */
    export class ParticleTextureBlock extends NodeMaterialBlock {
        private _samplerName;
        private _linearDefineName;
        private _gammaDefineName;
        private _tempTextureRead;
        /**
         * Gets or sets the texture associated with the node
         */
        texture: Nullable<BaseTexture>;
        /**
         * Gets or sets a boolean indicating if content needs to be converted to gamma space
         */
        convertToGammaSpace: boolean;
        /**
         * Gets or sets a boolean indicating if content needs to be converted to linear space
         */
        convertToLinearSpace: boolean;
        /**
         * Create a new ParticleTextureBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the uv input component
         */
        get uv(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgba output component
         */
        get rgba(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgb output component
         */
        get rgb(): NodeMaterialConnectionPoint;
        /**
         * Gets the r output component
         */
        get r(): NodeMaterialConnectionPoint;
        /**
         * Gets the g output component
         */
        get g(): NodeMaterialConnectionPoint;
        /**
         * Gets the b output component
         */
        get b(): NodeMaterialConnectionPoint;
        /**
         * Gets the a output component
         */
        get a(): NodeMaterialConnectionPoint;
        /**
         * Initialize the block and prepare the context for build
         * @param state defines the state that will be used for the build
         */
        initialize(state: NodeMaterialBuildState): void;
        autoConfigure(material: NodeMaterial): void;
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines): void;
        isReady(): boolean;
        private _writeOutput;
        protected _buildBlock(state: NodeMaterialBuildState): this | undefined;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterialBuildStateSharedData" {
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { InputBlock } from "babylonjs/Materials/Node/Blocks/Input/inputBlock";
    import { TextureBlock } from "babylonjs/Materials/Node/Blocks/Dual/textureBlock";
    import { ReflectionTextureBaseBlock } from "babylonjs/Materials/Node/Blocks/Dual/reflectionTextureBaseBlock";
    import { RefractionBlock } from "babylonjs/Materials/Node/Blocks/PBR/refractionBlock";
    import { CurrentScreenBlock } from "babylonjs/Materials/Node/Blocks/Dual/currentScreenBlock";
    import { ParticleTextureBlock } from "babylonjs/Materials/Node/Blocks/Particle/particleTextureBlock";
    import { Scene } from "babylonjs/scene";
    /**
     * Class used to store shared data between 2 NodeMaterialBuildState
     */
    export class NodeMaterialBuildStateSharedData {
        /**
        * Gets the list of emitted varyings
        */
        temps: string[];
        /**
         * Gets the list of emitted varyings
         */
        varyings: string[];
        /**
         * Gets the varying declaration string
         */
        varyingDeclaration: string;
        /**
         * Input blocks
         */
        inputBlocks: InputBlock[];
        /**
         * Input blocks
         */
        textureBlocks: (TextureBlock | ReflectionTextureBaseBlock | RefractionBlock | CurrentScreenBlock | ParticleTextureBlock)[];
        /**
         * Bindable blocks (Blocks that need to set data to the effect)
         */
        bindableBlocks: NodeMaterialBlock[];
        /**
         * List of blocks that can provide a compilation fallback
         */
        blocksWithFallbacks: NodeMaterialBlock[];
        /**
         * List of blocks that can provide a define update
         */
        blocksWithDefines: NodeMaterialBlock[];
        /**
        * List of blocks that can provide a repeatable content
        */
        repeatableContentBlocks: NodeMaterialBlock[];
        /**
        * List of blocks that can provide a dynamic list of uniforms
        */
        dynamicUniformBlocks: NodeMaterialBlock[];
        /**
         * List of blocks that can block the isReady function for the material
         */
        blockingBlocks: NodeMaterialBlock[];
        /**
         * Gets the list of animated inputs
         */
        animatedInputs: InputBlock[];
        /**
         * Build Id used to avoid multiple recompilations
         */
        buildId: number;
        /** List of emitted variables */
        variableNames: {
            [key: string]: number;
        };
        /** List of emitted defines */
        defineNames: {
            [key: string]: number;
        };
        /** Should emit comments? */
        emitComments: boolean;
        /** Emit build activity */
        verbose: boolean;
        /** Gets or sets the hosting scene */
        scene: Scene;
        /**
         * Gets the compilation hints emitted at compilation time
         */
        hints: {
            needWorldViewMatrix: boolean;
            needWorldViewProjectionMatrix: boolean;
            needAlphaBlending: boolean;
            needAlphaTesting: boolean;
        };
        /**
         * List of compilation checks
         */
        checks: {
            emitVertex: boolean;
            emitFragment: boolean;
            notConnectedNonOptionalInputs: NodeMaterialConnectionPoint[];
        };
        /**
         * Is vertex program allowed to be empty?
         */
        allowEmptyVertexProgram: boolean;
        /** Creates a new shared data */
        constructor();
        /**
         * Emits console errors and exceptions if there is a failing check
         */
        emitErrors(): void;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterialBuildState" {
    import { NodeMaterialBlockConnectionPointTypes } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockConnectionPointTypes";
    import { NodeMaterialBlockTargets } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets";
    import { NodeMaterialBuildStateSharedData } from "babylonjs/Materials/Node/nodeMaterialBuildStateSharedData";
    /**
     * Class used to store node based material build state
     */
    export class NodeMaterialBuildState {
        /** Gets or sets a boolean indicating if the current state can emit uniform buffers */
        supportUniformBuffers: boolean;
        /**
         * Gets the list of emitted attributes
         */
        attributes: string[];
        /**
         * Gets the list of emitted uniforms
         */
        uniforms: string[];
        /**
        * Gets the list of emitted constants
        */
        constants: string[];
        /**
         * Gets the list of emitted samplers
         */
        samplers: string[];
        /**
         * Gets the list of emitted functions
         */
        functions: {
            [key: string]: string;
        };
        /**
         * Gets the list of emitted extensions
         */
        extensions: {
            [key: string]: string;
        };
        /**
         * Gets the target of the compilation state
         */
        target: NodeMaterialBlockTargets;
        /**
         * Gets the list of emitted counters
         */
        counters: {
            [key: string]: number;
        };
        /**
         * Shared data between multiple NodeMaterialBuildState instances
         */
        sharedData: NodeMaterialBuildStateSharedData;
        /** @hidden */
        _vertexState: NodeMaterialBuildState;
        /** @hidden */
        _attributeDeclaration: string;
        /** @hidden */
        _uniformDeclaration: string;
        /** @hidden */
        _constantDeclaration: string;
        /** @hidden */
        _samplerDeclaration: string;
        /** @hidden */
        _varyingTransfer: string;
        /** @hidden */
        _injectAtEnd: string;
        private _repeatableContentAnchorIndex;
        /** @hidden */
        _builtCompilationString: string;
        /**
         * Gets the emitted compilation strings
         */
        compilationString: string;
        /**
         * Finalize the compilation strings
         * @param state defines the current compilation state
         */
        finalize(state: NodeMaterialBuildState): void;
        /** @hidden */
        get _repeatableContentAnchor(): string;
        /** @hidden */
        _getFreeVariableName(prefix: string): string;
        /** @hidden */
        _getFreeDefineName(prefix: string): string;
        /** @hidden */
        _excludeVariableName(name: string): void;
        /** @hidden */
        _emit2DSampler(name: string): void;
        /** @hidden */
        _getGLType(type: NodeMaterialBlockConnectionPointTypes): string;
        /** @hidden */
        _emitExtension(name: string, extension: string, define?: string): void;
        /** @hidden */
        _emitFunction(name: string, code: string, comments: string): void;
        /** @hidden */
        _emitCodeFromInclude(includeName: string, comments: string, options?: {
            replaceStrings?: {
                search: RegExp;
                replace: string;
            }[];
            repeatKey?: string;
        }): string;
        /** @hidden */
        _emitFunctionFromInclude(includeName: string, comments: string, options?: {
            repeatKey?: string;
            removeAttributes?: boolean;
            removeUniforms?: boolean;
            removeVaryings?: boolean;
            removeIfDef?: boolean;
            replaceStrings?: {
                search: RegExp;
                replace: string;
            }[];
        }, storeKey?: string): void;
        /** @hidden */
        _registerTempVariable(name: string): boolean;
        /** @hidden */
        _emitVaryingFromString(name: string, type: string, define?: string, notDefine?: boolean): boolean;
        /** @hidden */
        _emitUniformFromString(name: string, type: string, define?: string, notDefine?: boolean): void;
        /** @hidden */
        _emitFloat(value: number): string;
    }
}
declare module "babylonjs/Misc/uniqueIdGenerator" {
    /**
     * Helper class used to generate session unique ID
     */
    export class UniqueIdGenerator {
        private static _UniqueIdCounter;
        /**
         * Gets an unique (relatively to the current scene) Id
         */
        static get UniqueId(): number;
    }
}
declare module "babylonjs/Materials/effectFallbacks" {
    import { IEffectFallbacks } from "babylonjs/Materials/iEffectFallbacks";
    import { Effect } from "babylonjs/Materials/effect";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    /**
     * EffectFallbacks can be used to add fallbacks (properties to disable) to certain properties when desired to improve performance.
     * (Eg. Start at high quality with reflection and fog, if fps is low, remove reflection, if still low remove fog)
     */
    export class EffectFallbacks implements IEffectFallbacks {
        private _defines;
        private _currentRank;
        private _maxRank;
        private _mesh;
        /**
         * Removes the fallback from the bound mesh.
         */
        unBindMesh(): void;
        /**
         * Adds a fallback on the specified property.
         * @param rank The rank of the fallback (Lower ranks will be fallbacked to first)
         * @param define The name of the define in the shader
         */
        addFallback(rank: number, define: string): void;
        /**
         * Sets the mesh to use CPU skinning when needing to fallback.
         * @param rank The rank of the fallback (Lower ranks will be fallbacked to first)
         * @param mesh The mesh to use the fallbacks.
         */
        addCPUSkinningFallback(rank: number, mesh: AbstractMesh): void;
        /**
         * Checks to see if more fallbacks are still availible.
         */
        get hasMoreFallbacks(): boolean;
        /**
         * Removes the defines that should be removed when falling back.
         * @param currentDefines defines the current define statements for the shader.
         * @param effect defines the current effect we try to compile
         * @returns The resulting defines with defines of the current rank removed.
         */
        reduce(currentDefines: string, effect: Effect): string;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterialBlock" {
    import { NodeMaterialBlockConnectionPointTypes } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockConnectionPointTypes";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { Nullable } from "babylonjs/types";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { NodeMaterialBlockTargets } from "babylonjs/Materials/Node/Enums/nodeMaterialBlockTargets";
    import { Effect } from "babylonjs/Materials/effect";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { NodeMaterial, NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { Scene } from "babylonjs/scene";
    import { EffectFallbacks } from "babylonjs/Materials/effectFallbacks";
    /**
     * Defines a block that can be used inside a node based material
     */
    export class NodeMaterialBlock {
        private _buildId;
        private _buildTarget;
        private _target;
        private _isFinalMerger;
        private _isInput;
        private _name;
        protected _isUnique: boolean;
        /** Gets or sets a boolean indicating that only one input can be connected at a time */
        inputsAreExclusive: boolean;
        /** @hidden */
        _codeVariableName: string;
        /** @hidden */
        _inputs: NodeMaterialConnectionPoint[];
        /** @hidden */
        _outputs: NodeMaterialConnectionPoint[];
        /** @hidden */
        _preparationId: number;
        /**
         * Gets the name of the block
         */
        get name(): string;
        /**
         * Sets the name of the block. Will check if the name is valid.
         */
        set name(newName: string);
        /**
         * Gets or sets the unique id of the node
         */
        uniqueId: number;
        /**
         * Gets or sets the comments associated with this block
         */
        comments: string;
        /**
         * Gets a boolean indicating that this block can only be used once per NodeMaterial
         */
        get isUnique(): boolean;
        /**
         * Gets a boolean indicating that this block is an end block (e.g. it is generating a system value)
         */
        get isFinalMerger(): boolean;
        /**
         * Gets a boolean indicating that this block is an input (e.g. it sends data to the shader)
         */
        get isInput(): boolean;
        /**
         * Gets or sets the build Id
         */
        get buildId(): number;
        set buildId(value: number);
        /**
         * Gets or sets the target of the block
         */
        get target(): NodeMaterialBlockTargets;
        set target(value: NodeMaterialBlockTargets);
        /**
         * Gets the list of input points
         */
        get inputs(): NodeMaterialConnectionPoint[];
        /** Gets the list of output points */
        get outputs(): NodeMaterialConnectionPoint[];
        /**
         * Find an input by its name
         * @param name defines the name of the input to look for
         * @returns the input or null if not found
         */
        getInputByName(name: string): Nullable<NodeMaterialConnectionPoint>;
        /**
         * Find an output by its name
         * @param name defines the name of the outputto look for
         * @returns the output or null if not found
         */
        getOutputByName(name: string): Nullable<NodeMaterialConnectionPoint>;
        /** Gets or sets a boolean indicating that this input can be edited in the Inspector (false by default) */
        visibleInInspector: boolean;
        /**
         * Creates a new NodeMaterialBlock
         * @param name defines the block name
         * @param target defines the target of that block (Vertex by default)
         * @param isFinalMerger defines a boolean indicating that this block is an end block (e.g. it is generating a system value). Default is false
         * @param isInput defines a boolean indicating that this block is an input (e.g. it sends data to the shader). Default is false
         */
        constructor(name: string, target?: NodeMaterialBlockTargets, isFinalMerger?: boolean, isInput?: boolean);
        /**
         * Initialize the block and prepare the context for build
         * @param state defines the state that will be used for the build
         */
        initialize(state: NodeMaterialBuildState): void;
        /**
         * Bind data to effect. Will only be called for blocks with isBindable === true
         * @param effect defines the effect to bind data to
         * @param nodeMaterial defines the hosting NodeMaterial
         * @param mesh defines the mesh that will be rendered
         * @param subMesh defines the submesh that will be rendered
         */
        bind(effect: Effect, nodeMaterial: NodeMaterial, mesh?: Mesh, subMesh?: SubMesh): void;
        protected _declareOutput(output: NodeMaterialConnectionPoint, state: NodeMaterialBuildState): string;
        protected _writeVariable(currentPoint: NodeMaterialConnectionPoint): string;
        protected _writeFloat(value: number): string;
        /**
         * Gets the current class name e.g. "NodeMaterialBlock"
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Register a new input. Must be called inside a block constructor
         * @param name defines the connection point name
         * @param type defines the connection point type
         * @param isOptional defines a boolean indicating that this input can be omitted
         * @param target defines the target to use to limit the connection point (will be VertexAndFragment by default)
         * @param point an already created connection point. If not provided, create a new one
         * @returns the current block
         */
        registerInput(name: string, type: NodeMaterialBlockConnectionPointTypes, isOptional?: boolean, target?: NodeMaterialBlockTargets, point?: NodeMaterialConnectionPoint): this;
        /**
         * Register a new output. Must be called inside a block constructor
         * @param name defines the connection point name
         * @param type defines the connection point type
         * @param target defines the target to use to limit the connection point (will be VertexAndFragment by default)
         * @param point an already created connection point. If not provided, create a new one
         * @returns the current block
         */
        registerOutput(name: string, type: NodeMaterialBlockConnectionPointTypes, target?: NodeMaterialBlockTargets, point?: NodeMaterialConnectionPoint): this;
        /**
         * Will return the first available input e.g. the first one which is not an uniform or an attribute
         * @param forOutput defines an optional connection point to check compatibility with
         * @returns the first available input or null
         */
        getFirstAvailableInput(forOutput?: Nullable<NodeMaterialConnectionPoint>): Nullable<NodeMaterialConnectionPoint>;
        /**
         * Will return the first available output e.g. the first one which is not yet connected and not a varying
         * @param forBlock defines an optional block to check compatibility with
         * @returns the first available input or null
         */
        getFirstAvailableOutput(forBlock?: Nullable<NodeMaterialBlock>): Nullable<NodeMaterialConnectionPoint>;
        /**
         * Gets the sibling of the given output
         * @param current defines the current output
         * @returns the next output in the list or null
         */
        getSiblingOutput(current: NodeMaterialConnectionPoint): Nullable<NodeMaterialConnectionPoint>;
        /**
         * Connect current block with another block
         * @param other defines the block to connect with
         * @param options define the various options to help pick the right connections
         * @returns the current block
         */
        connectTo(other: NodeMaterialBlock, options?: {
            input?: string;
            output?: string;
            outputSwizzle?: string;
        }): this | undefined;
        protected _buildBlock(state: NodeMaterialBuildState): void;
        /**
         * Add uniforms, samplers and uniform buffers at compilation time
         * @param state defines the state to update
         * @param nodeMaterial defines the node material requesting the update
         * @param defines defines the material defines to update
         * @param uniformBuffers defines the list of uniform buffer names
         */
        updateUniformsAndSamples(state: NodeMaterialBuildState, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines, uniformBuffers: string[]): void;
        /**
         * Add potential fallbacks if shader compilation fails
         * @param mesh defines the mesh to be rendered
         * @param fallbacks defines the current prioritized list of fallbacks
         */
        provideFallbacks(mesh: AbstractMesh, fallbacks: EffectFallbacks): void;
        /**
         * Initialize defines for shader compilation
         * @param mesh defines the mesh to be rendered
         * @param nodeMaterial defines the node material requesting the update
         * @param defines defines the material defines to update
         * @param useInstances specifies that instances should be used
         */
        initializeDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines, useInstances?: boolean): void;
        /**
         * Update defines for shader compilation
         * @param mesh defines the mesh to be rendered
         * @param nodeMaterial defines the node material requesting the update
         * @param defines defines the material defines to update
         * @param useInstances specifies that instances should be used
         * @param subMesh defines which submesh to render
         */
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines, useInstances?: boolean, subMesh?: SubMesh): void;
        /**
         * Lets the block try to connect some inputs automatically
         * @param material defines the hosting NodeMaterial
         */
        autoConfigure(material: NodeMaterial): void;
        /**
         * Function called when a block is declared as repeatable content generator
         * @param vertexShaderState defines the current compilation state for the vertex shader
         * @param fragmentShaderState defines the current compilation state for the fragment shader
         * @param mesh defines the mesh to be rendered
         * @param defines defines the material defines to update
         */
        replaceRepeatableContent(vertexShaderState: NodeMaterialBuildState, fragmentShaderState: NodeMaterialBuildState, mesh: AbstractMesh, defines: NodeMaterialDefines): void;
        /**
         * Checks if the block is ready
         * @param mesh defines the mesh to be rendered
         * @param nodeMaterial defines the node material requesting the update
         * @param defines defines the material defines to update
         * @param useInstances specifies that instances should be used
         * @returns true if the block is ready
         */
        isReady(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines, useInstances?: boolean): boolean;
        protected _linkConnectionTypes(inputIndex0: number, inputIndex1: number, looseCoupling?: boolean): void;
        private _processBuild;
        /**
        * Validates the new name for the block node.
        * @param newName the new name to be given to the node.
        * @returns false if the name is a reserve word, else true.
        */
        validateBlockName(newName: string): boolean;
        /**
         * Compile the current node and generate the shader code
         * @param state defines the current compilation state (uniforms, samplers, current string)
         * @param activeBlocks defines the list of active blocks (i.e. blocks to compile)
         * @returns true if already built
         */
        build(state: NodeMaterialBuildState, activeBlocks: NodeMaterialBlock[]): boolean;
        protected _inputRename(name: string): string;
        protected _outputRename(name: string): string;
        protected _dumpPropertiesCode(): string;
        /** @hidden */
        _dumpCode(uniqueNames: string[], alreadyDumped: NodeMaterialBlock[]): string;
        /** @hidden */
        _dumpCodeForOutputConnections(alreadyDumped: NodeMaterialBlock[]): string;
        /**
         * Clone the current block to a new identical block
         * @param scene defines the hosting scene
         * @param rootUrl defines the root URL to use to load textures and relative dependencies
         * @returns a copy of the current block
         */
        clone(scene: Scene, rootUrl?: string): Nullable<NodeMaterialBlock>;
        /**
         * Serializes this block in a JSON representation
         * @returns the serialized block object
         */
        serialize(): any;
        /** @hidden */
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
        private _deserializePortDisplayNamesAndExposedOnFrame;
        /**
         * Release resources
         */
        dispose(): void;
    }
}
declare module "babylonjs/Materials/pushMaterial" {
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { Material } from "babylonjs/Materials/material";
    import { Effect } from "babylonjs/Materials/effect";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    /**
     * Base class of materials working in push mode in babylon JS
     * @hidden
     */
    export class PushMaterial extends Material {
        protected _activeEffect: Effect;
        protected _normalMatrix: Matrix;
        constructor(name: string, scene: Scene);
        getEffect(): Effect;
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        protected _isReadyForSubMesh(subMesh: SubMesh): boolean;
        /**
        * Binds the given world matrix to the active effect
        *
        * @param world the matrix to bind
        */
        bindOnlyWorldMatrix(world: Matrix): void;
        /**
         * Binds the given normal matrix to the active effect
         *
         * @param normalMatrix the matrix to bind
         */
        bindOnlyNormalMatrix(normalMatrix: Matrix): void;
        bind(world: Matrix, mesh?: Mesh): void;
        protected _afterBind(mesh: Mesh, effect?: Nullable<Effect>): void;
        protected _mustRebind(scene: Scene, effect: Effect, visibility?: number): boolean;
    }
}
declare module "babylonjs/Materials/Node/Optimizers/nodeMaterialOptimizer" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    /**
     * Root class for all node material optimizers
     */
    export class NodeMaterialOptimizer {
        /**
         * Function used to optimize a NodeMaterial graph
         * @param vertexOutputNodes defines the list of output nodes for the vertex shader
         * @param fragmentOutputNodes defines the list of output nodes for the fragment shader
         */
        optimize(vertexOutputNodes: NodeMaterialBlock[], fragmentOutputNodes: NodeMaterialBlock[]): void;
    }
}
declare module "babylonjs/Materials/Node/Blocks/transformBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { Scene } from "babylonjs/scene";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { NodeMaterial, NodeMaterialDefines } from "babylonjs/Materials/Node/nodeMaterial";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    /**
     * Block used to transform a vector (2, 3 or 4) with a matrix. It will generate a Vector4
     */
    export class TransformBlock extends NodeMaterialBlock {
        /**
         * Defines the value to use to complement W value to transform it to a Vector4
         */
        complementW: number;
        /**
         * Defines the value to use to complement z value to transform it to a Vector4
         */
        complementZ: number;
        /**
         * Creates a new TransformBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the vector input
         */
        get vector(): NodeMaterialConnectionPoint;
        /**
         * Gets the output component
         */
        get output(): NodeMaterialConnectionPoint;
        /**
         * Gets the xyz output component
         */
        get xyz(): NodeMaterialConnectionPoint;
        /**
         * Gets the matrix transform input
         */
        get transform(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
        /**
         * Update defines for shader compilation
         * @param mesh defines the mesh to be rendered
         * @param nodeMaterial defines the node material requesting the update
         * @param defines defines the material defines to update
         * @param useInstances specifies that instances should be used
         * @param subMesh defines which submesh to render
         */
        prepareDefines(mesh: AbstractMesh, nodeMaterial: NodeMaterial, defines: NodeMaterialDefines, useInstances?: boolean, subMesh?: SubMesh): void;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
        protected _dumpPropertiesCode(): string;
    }
}
declare module "babylonjs/Materials/Node/Blocks/Vertex/vertexOutputBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
     * Block used to output the vertex position
     */
    export class VertexOutputBlock extends NodeMaterialBlock {
        /**
         * Creates a new VertexOutputBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the vector input component
         */
        get vector(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
    }
}
declare module "babylonjs/Materials/Node/Blocks/Fragment/fragmentOutputBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
     * Block used to output the final color
     */
    export class FragmentOutputBlock extends NodeMaterialBlock {
        /**
         * Create a new FragmentOutputBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the rgba input component
         */
        get rgba(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgb input component
         */
        get rgb(): NodeMaterialConnectionPoint;
        /**
         * Gets the a input component
         */
        get a(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
    }
}
declare module "babylonjs/Materials/Node/Blocks/Particle/particleRampGradientBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
     * Block used for the particle ramp gradient section
     */
    export class ParticleRampGradientBlock extends NodeMaterialBlock {
        /**
         * Create a new ParticleRampGradientBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the color input component
         */
        get color(): NodeMaterialConnectionPoint;
        /**
         * Gets the rampColor output component
         */
        get rampColor(): NodeMaterialConnectionPoint;
        /**
         * Initialize the block and prepare the context for build
         * @param state defines the state that will be used for the build
         */
        initialize(state: NodeMaterialBuildState): void;
        protected _buildBlock(state: NodeMaterialBuildState): this | undefined;
    }
}
declare module "babylonjs/Materials/Node/Blocks/Particle/particleBlendMultiplyBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
      * Block used for the particle blend multiply section
      */
    export class ParticleBlendMultiplyBlock extends NodeMaterialBlock {
        /**
         * Create a new ParticleBlendMultiplyBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the color input component
         */
        get color(): NodeMaterialConnectionPoint;
        /**
         * Gets the alphaTexture input component
         */
        get alphaTexture(): NodeMaterialConnectionPoint;
        /**
         * Gets the alphaColor input component
         */
        get alphaColor(): NodeMaterialConnectionPoint;
        /**
         * Gets the blendColor output component
         */
        get blendColor(): NodeMaterialConnectionPoint;
        /**
         * Initialize the block and prepare the context for build
         * @param state defines the state that will be used for the build
         */
        initialize(state: NodeMaterialBuildState): void;
        protected _buildBlock(state: NodeMaterialBuildState): this | undefined;
    }
}
declare module "babylonjs/Materials/Node/Blocks/vectorMergerBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
     * Block used to create a Vector2/3/4 out of individual inputs (one for each component)
     */
    export class VectorMergerBlock extends NodeMaterialBlock {
        /**
         * Create a new VectorMergerBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the xyz component (input)
         */
        get xyzIn(): NodeMaterialConnectionPoint;
        /**
         * Gets the xy component (input)
         */
        get xyIn(): NodeMaterialConnectionPoint;
        /**
         * Gets the x component (input)
         */
        get x(): NodeMaterialConnectionPoint;
        /**
         * Gets the y component (input)
         */
        get y(): NodeMaterialConnectionPoint;
        /**
         * Gets the z component (input)
         */
        get z(): NodeMaterialConnectionPoint;
        /**
         * Gets the w component (input)
         */
        get w(): NodeMaterialConnectionPoint;
        /**
         * Gets the xyzw component (output)
         */
        get xyzw(): NodeMaterialConnectionPoint;
        /**
         * Gets the xyz component (output)
         */
        get xyzOut(): NodeMaterialConnectionPoint;
        /**
         * Gets the xy component (output)
         */
        get xyOut(): NodeMaterialConnectionPoint;
        /**
         * Gets the xy component (output)
         * @deprecated Please use xyOut instead.
         */
        get xy(): NodeMaterialConnectionPoint;
        /**
         * Gets the xyz component (output)
         * @deprecated Please use xyzOut instead.
         */
        get xyz(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
    }
}
declare module "babylonjs/Materials/Node/Blocks/remapBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { Vector2 } from "babylonjs/Maths/math.vector";
    import { Scene } from "babylonjs/scene";
    /**
     * Block used to remap a float from a range to a new one
     */
    export class RemapBlock extends NodeMaterialBlock {
        /**
         * Gets or sets the source range
         */
        sourceRange: Vector2;
        /**
         * Gets or sets the target range
         */
        targetRange: Vector2;
        /**
         * Creates a new RemapBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the input component
         */
        get input(): NodeMaterialConnectionPoint;
        /**
         * Gets the source min input component
         */
        get sourceMin(): NodeMaterialConnectionPoint;
        /**
         * Gets the source max input component
         */
        get sourceMax(): NodeMaterialConnectionPoint;
        /**
         * Gets the target min input component
         */
        get targetMin(): NodeMaterialConnectionPoint;
        /**
         * Gets the target max input component
         */
        get targetMax(): NodeMaterialConnectionPoint;
        /**
         * Gets the output component
         */
        get output(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
        protected _dumpPropertiesCode(): string;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/Node/Blocks/multiplyBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
     * Block used to multiply 2 values
     */
    export class MultiplyBlock extends NodeMaterialBlock {
        /**
         * Creates a new MultiplyBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the left operand input component
         */
        get left(): NodeMaterialConnectionPoint;
        /**
         * Gets the right operand input component
         */
        get right(): NodeMaterialConnectionPoint;
        /**
         * Gets the output component
         */
        get output(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
    }
}
declare module "babylonjs/Misc/gradients" {
    import { Color3, Color4 } from "babylonjs/Maths/math.color";
    /** Interface used by value gradients (color, factor, ...) */
    export interface IValueGradient {
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number;
    }
    /** Class used to store color4 gradient */
    export class ColorGradient implements IValueGradient {
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number;
        /**
         * Gets or sets first associated color
         */
        color1: Color4;
        /**
         * Gets or sets second associated color
         */
        color2?: Color4 | undefined;
        /**
         * Creates a new color4 gradient
         * @param gradient gets or sets the gradient value (between 0 and 1)
         * @param color1 gets or sets first associated color
         * @param color2 gets or sets first second color
         */
        constructor(
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number, 
        /**
         * Gets or sets first associated color
         */
        color1: Color4, 
        /**
         * Gets or sets second associated color
         */
        color2?: Color4 | undefined);
        /**
         * Will get a color picked randomly between color1 and color2.
         * If color2 is undefined then color1 will be used
         * @param result defines the target Color4 to store the result in
         */
        getColorToRef(result: Color4): void;
    }
    /** Class used to store color 3 gradient */
    export class Color3Gradient implements IValueGradient {
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number;
        /**
         * Gets or sets the associated color
         */
        color: Color3;
        /**
         * Creates a new color3 gradient
         * @param gradient gets or sets the gradient value (between 0 and 1)
         * @param color gets or sets associated color
         */
        constructor(
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number, 
        /**
         * Gets or sets the associated color
         */
        color: Color3);
    }
    /** Class used to store factor gradient */
    export class FactorGradient implements IValueGradient {
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number;
        /**
         * Gets or sets first associated factor
         */
        factor1: number;
        /**
         * Gets or sets second associated factor
         */
        factor2?: number | undefined;
        /**
         * Creates a new factor gradient
         * @param gradient gets or sets the gradient value (between 0 and 1)
         * @param factor1 gets or sets first associated factor
         * @param factor2 gets or sets second associated factor
         */
        constructor(
        /**
         * Gets or sets the gradient value (between 0 and 1)
         */
        gradient: number, 
        /**
         * Gets or sets first associated factor
         */
        factor1: number, 
        /**
         * Gets or sets second associated factor
         */
        factor2?: number | undefined);
        /**
         * Will get a number picked randomly between factor1 and factor2.
         * If factor2 is undefined then factor1 will be used
         * @returns the picked number
         */
        getFactor(): number;
    }
    /**
     * Helper used to simplify some generic gradient tasks
     */
    export class GradientHelper {
        /**
         * Gets the current gradient from an array of IValueGradient
         * @param ratio defines the current ratio to get
         * @param gradients defines the array of IValueGradient
         * @param updateFunc defines the callback function used to get the final value from the selected gradients
         */
        static GetCurrentGradient(ratio: number, gradients: IValueGradient[], updateFunc: (current: IValueGradient, next: IValueGradient, scale: number) => void): void;
    }
}
declare module "babylonjs/Engines/Extensions/engine.rawTexture" {
    import { Nullable } from "babylonjs/types";
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { Scene } from "babylonjs/scene";
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Creates a raw texture
             * @param data defines the data to store in the texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param format defines the format of the data
             * @param generateMipMaps defines if the engine should generate the mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (Texture.NEAREST_SAMPLINGMODE by default)
             * @param compression defines the compression used (null by default)
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @returns the raw texture inside an InternalTexture
             */
            createRawTexture(data: Nullable<ArrayBufferView>, width: number, height: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>, type: number): InternalTexture;
            /**
             * Update a raw texture
             * @param texture defines the texture to update
             * @param data defines the data to store in the texture
             * @param format defines the format of the data
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean): void;
            /**
             * Update a raw texture
             * @param texture defines the texture to update
             * @param data defines the data to store in the texture
             * @param format defines the format of the data
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the compression used (null by default)
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             */
            updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression: Nullable<string>, type: number): void;
            /**
             * Creates a new raw cube texture
             * @param data defines the array of data to use to create each face
             * @param size defines the size of the textures
             * @param format defines the format of the data
             * @param type defines the type of the data (like Engine.TEXTURETYPE_UNSIGNED_INT)
             * @param generateMipMaps  defines if the engine should generate the mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param compression defines the compression used (null by default)
             * @returns the cube texture as an InternalTexture
             */
            createRawCubeTexture(data: Nullable<ArrayBufferView[]>, size: number, format: number, type: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>): InternalTexture;
            /**
             * Update a raw cube texture
             * @param texture defines the texture to udpdate
             * @param data defines the data to store
             * @param format defines the data format
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean): void;
            /**
             * Update a raw cube texture
             * @param texture defines the texture to udpdate
             * @param data defines the data to store
             * @param format defines the data format
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the compression used (null by default)
             */
            updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean, compression: Nullable<string>): void;
            /**
             * Update a raw cube texture
             * @param texture defines the texture to udpdate
             * @param data defines the data to store
             * @param format defines the data format
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the compression used (null by default)
             * @param level defines which level of the texture to update
             */
            updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean, compression: Nullable<string>, level: number): void;
            /**
             * Creates a new raw cube texture from a specified url
             * @param url defines the url where the data is located
             * @param scene defines the current scene
             * @param size defines the size of the textures
             * @param format defines the format of the data
             * @param type defines the type fo the data (like Engine.TEXTURETYPE_UNSIGNED_INT)
             * @param noMipmap defines if the engine should avoid generating the mip levels
             * @param callback defines a callback used to extract texture data from loaded data
             * @param mipmapGenerator defines to provide an optional tool to generate mip levels
             * @param onLoad defines a callback called when texture is loaded
             * @param onError defines a callback called if there is an error
             * @returns the cube texture as an InternalTexture
             */
            createRawCubeTextureFromUrl(url: string, scene: Nullable<Scene>, size: number, format: number, type: number, noMipmap: boolean, callback: (ArrayBuffer: ArrayBuffer) => Nullable<ArrayBufferView[]>, mipmapGenerator: Nullable<((faces: ArrayBufferView[]) => ArrayBufferView[][])>, onLoad: Nullable<() => void>, onError: Nullable<(message?: string, exception?: any) => void>): InternalTexture;
            /**
             * Creates a new raw cube texture from a specified url
             * @param url defines the url where the data is located
             * @param scene defines the current scene
             * @param size defines the size of the textures
             * @param format defines the format of the data
             * @param type defines the type fo the data (like Engine.TEXTURETYPE_UNSIGNED_INT)
             * @param noMipmap defines if the engine should avoid generating the mip levels
             * @param callback defines a callback used to extract texture data from loaded data
             * @param mipmapGenerator defines to provide an optional tool to generate mip levels
             * @param onLoad defines a callback called when texture is loaded
             * @param onError defines a callback called if there is an error
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param invertY defines if data must be stored with Y axis inverted
             * @returns the cube texture as an InternalTexture
             */
            createRawCubeTextureFromUrl(url: string, scene: Nullable<Scene>, size: number, format: number, type: number, noMipmap: boolean, callback: (ArrayBuffer: ArrayBuffer) => Nullable<ArrayBufferView[]>, mipmapGenerator: Nullable<((faces: ArrayBufferView[]) => ArrayBufferView[][])>, onLoad: Nullable<() => void>, onError: Nullable<(message?: string, exception?: any) => void>, samplingMode: number, invertY: boolean): InternalTexture;
            /**
             * Creates a new raw 3D texture
             * @param data defines the data used to create the texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param depth defines the depth of the texture
             * @param format defines the format of the texture
             * @param generateMipMaps defines if the engine must generate mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param compression defines the compressed used (can be null)
             * @param textureType defines the compressed used (can be null)
             * @returns a new raw 3D texture (stored in an InternalTexture)
             */
            createRawTexture3D(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>, textureType: number): InternalTexture;
            /**
             * Update a raw 3D texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawTexture3D(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean): void;
            /**
             * Update a raw 3D texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the used compression (can be null)
             * @param textureType defines the texture Type (Engine.TEXTURETYPE_UNSIGNED_INT, Engine.TEXTURETYPE_FLOAT...)
             */
            updateRawTexture3D(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression: Nullable<string>, textureType: number): void;
            /**
             * Creates a new raw 2D array texture
             * @param data defines the data used to create the texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param depth defines the number of layers of the texture
             * @param format defines the format of the texture
             * @param generateMipMaps defines if the engine must generate mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param compression defines the compressed used (can be null)
             * @param textureType defines the compressed used (can be null)
             * @returns a new raw 2D array texture (stored in an InternalTexture)
             */
            createRawTexture2DArray(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>, textureType: number): InternalTexture;
            /**
             * Update a raw 2D array texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawTexture2DArray(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean): void;
            /**
             * Update a raw 2D array texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the used compression (can be null)
             * @param textureType defines the texture Type (Engine.TEXTURETYPE_UNSIGNED_INT, Engine.TEXTURETYPE_FLOAT...)
             */
            updateRawTexture2DArray(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression: Nullable<string>, textureType: number): void;
        }
    }
}
declare module "babylonjs/Materials/Textures/rawTexture" {
    import { Texture } from "babylonjs/Materials/Textures/texture";
    import "babylonjs/Engines/Extensions/engine.rawTexture";
    import { Nullable } from "babylonjs/types";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    import { Scene } from "babylonjs/scene";
    /**
     * Raw texture can help creating a texture directly from an array of data.
     * This can be super useful if you either get the data from an uncompressed source or
     * if you wish to create your texture pixel by pixel.
     */
    export class RawTexture extends Texture {
        /**
         * Define the format of the data (RGB, RGBA... Engine.TEXTUREFORMAT_xxx)
         */
        format: number;
        /**
         * Instantiates a new RawTexture.
         * Raw texture can help creating a texture directly from an array of data.
         * This can be super useful if you either get the data from an uncompressed source or
         * if you wish to create your texture pixel by pixel.
         * @param data define the array of data to use to create the texture
         * @param width define the width of the texture
         * @param height define the height of the texture
         * @param format define the format of the data (RGB, RGBA... Engine.TEXTUREFORMAT_xxx)
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps define whether mip maps should be generated or not
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @param type define the format of the data (int, float... Engine.TEXTURETYPE_xxx)
         */
        constructor(data: ArrayBufferView, width: number, height: number, 
        /**
         * Define the format of the data (RGB, RGBA... Engine.TEXTUREFORMAT_xxx)
         */
        format: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number, type?: number);
        /**
         * Updates the texture underlying data.
         * @param data Define the new data of the texture
         */
        update(data: ArrayBufferView): void;
        /**
         * Creates a luminance texture from some data.
         * @param data Define the texture data
         * @param width Define the width of the texture
         * @param height Define the height of the texture
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps Define whether or not to create mip maps for the texture
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @returns the luminance texture
         */
        static CreateLuminanceTexture(data: ArrayBufferView, width: number, height: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number): RawTexture;
        /**
         * Creates a luminance alpha texture from some data.
         * @param data Define the texture data
         * @param width Define the width of the texture
         * @param height Define the height of the texture
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps Define whether or not to create mip maps for the texture
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @returns the luminance alpha texture
         */
        static CreateLuminanceAlphaTexture(data: ArrayBufferView, width: number, height: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number): RawTexture;
        /**
         * Creates an alpha texture from some data.
         * @param data Define the texture data
         * @param width Define the width of the texture
         * @param height Define the height of the texture
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps Define whether or not to create mip maps for the texture
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @returns the alpha texture
         */
        static CreateAlphaTexture(data: ArrayBufferView, width: number, height: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number): RawTexture;
        /**
         * Creates a RGB texture from some data.
         * @param data Define the texture data
         * @param width Define the width of the texture
         * @param height Define the height of the texture
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps Define whether or not to create mip maps for the texture
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @param type define the format of the data (int, float... Engine.TEXTURETYPE_xxx)
         * @returns the RGB alpha texture
         */
        static CreateRGBTexture(data: ArrayBufferView, width: number, height: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number, type?: number): RawTexture;
        /**
         * Creates a RGBA texture from some data.
         * @param data Define the texture data
         * @param width Define the width of the texture
         * @param height Define the height of the texture
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps Define whether or not to create mip maps for the texture
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @param type define the format of the data (int, float... Engine.TEXTURETYPE_xxx)
         * @returns the RGBA texture
         */
        static CreateRGBATexture(data: ArrayBufferView, width: number, height: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number, type?: number): RawTexture;
        /**
         * Creates a R texture from some data.
         * @param data Define the texture data
         * @param width Define the width of the texture
         * @param height Define the height of the texture
         * @param sceneOrEngine defines the scene or engine the texture will belong to
         * @param generateMipMaps Define whether or not to create mip maps for the texture
         * @param invertY define if the data should be flipped on Y when uploaded to the GPU
         * @param samplingMode define the texture sampling mode (Texture.xxx_SAMPLINGMODE)
         * @param type define the format of the data (int, float... Engine.TEXTURETYPE_xxx)
         * @returns the R texture
         */
        static CreateRTexture(data: ArrayBufferView, width: number, height: number, sceneOrEngine: Nullable<Scene | ThinEngine>, generateMipMaps?: boolean, invertY?: boolean, samplingMode?: number, type?: number): RawTexture;
    }
}
declare module "babylonjs/Engines/Extensions/engine.dynamicBuffer" {
    import { DataBuffer } from "babylonjs/Meshes/dataBuffer";
    import { IndicesArray, DataArray } from "babylonjs/types";
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Update a dynamic index buffer
             * @param indexBuffer defines the target index buffer
             * @param indices defines the data to update
             * @param offset defines the offset in the target index buffer where update should start
             */
            updateDynamicIndexBuffer(indexBuffer: DataBuffer, indices: IndicesArray, offset?: number): void;
            /**
             * Updates a dynamic vertex buffer.
             * @param vertexBuffer the vertex buffer to update
             * @param data the data used to update the vertex buffer
             * @param byteOffset the byte offset of the data
             * @param byteLength the byte length of the data
             */
            updateDynamicVertexBuffer(vertexBuffer: DataBuffer, data: DataArray, byteOffset?: number, byteLength?: number): void;
        }
    }
}
declare module "babylonjs/Materials/Textures/Procedurals/proceduralTextureSceneComponent" {
    import { Scene } from "babylonjs/scene";
    import { ISceneComponent } from "babylonjs/sceneComponent";
    import { ProceduralTexture } from "babylonjs/Materials/Textures/Procedurals/proceduralTexture";
    module "babylonjs/abstractScene" {
        interface AbstractScene {
            /**
             * The list of procedural textures added to the scene
             * @see https://doc.babylonjs.com/how_to/how_to_use_procedural_textures
             */
            proceduralTextures: Array<ProceduralTexture>;
        }
    }
    /**
     * Defines the Procedural Texture scene component responsible to manage any Procedural Texture
     * in a given scene.
     */
    export class ProceduralTextureSceneComponent implements ISceneComponent {
        /**
         * The component name helpfull to identify the component in the list of scene components.
         */
        readonly name: string;
        /**
         * The scene the component belongs to.
         */
        scene: Scene;
        /**
         * Creates a new instance of the component for the given scene
         * @param scene Defines the scene to register the component in
         */
        constructor(scene: Scene);
        /**
         * Registers the component in a given scene
         */
        register(): void;
        /**
         * Rebuilds the elements related to this component in case of
         * context lost for instance.
         */
        rebuild(): void;
        /**
         * Disposes the component and the associated ressources.
         */
        dispose(): void;
        private _beforeClear;
    }
}
declare module "babylonjs/Engines/Extensions/engine.renderTargetCube" {
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { RenderTargetCreationOptions } from "babylonjs/Materials/Textures/renderTargetCreationOptions";
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Creates a new render target cube texture
             * @param size defines the size of the texture
             * @param options defines the options used to create the texture
             * @returns a new render target cube texture stored in an InternalTexture
             */
            createRenderTargetCubeTexture(size: number, options?: Partial<RenderTargetCreationOptions>): InternalTexture;
        }
    }
}
declare module "babylonjs/Shaders/procedural.vertex" {
    /** @hidden */
    export var proceduralVertexShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Materials/Textures/Procedurals/proceduralTexture" {
    import { Observable } from "babylonjs/Misc/observable";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Matrix, Vector3, Vector2 } from "babylonjs/Maths/math.vector";
    import { Color4, Color3 } from "babylonjs/Maths/math.color";
    import { Effect } from "babylonjs/Materials/effect";
    import { Texture } from "babylonjs/Materials/Textures/texture";
    import "babylonjs/Engines/Extensions/engine.renderTarget";
    import "babylonjs/Engines/Extensions/engine.renderTargetCube";
    import "babylonjs/Shaders/procedural.vertex";
    import { NodeMaterial } from "babylonjs/Materials/Node/nodeMaterial";
    import { RenderTargetTextureSize } from "babylonjs/Engines/Extensions/engine.renderTarget";
    /**
     * Procedural texturing is a way to programmatically create a texture. There are 2 types of procedural textures: code-only, and code that references some classic 2D images, sometimes calmpler' images.
     * This is the base class of any Procedural texture and contains most of the shareable code.
     * @see https://doc.babylonjs.com/how_to/how_to_use_procedural_textures
     */
    export class ProceduralTexture extends Texture {
        /**
         * Define if the texture is enabled or not (disabled texture will not render)
         */
        isEnabled: boolean;
        /**
         * Define if the texture must be cleared before rendering (default is true)
         */
        autoClear: boolean;
        /**
         * Callback called when the texture is generated
         */
        onGenerated: () => void;
        /**
         * Event raised when the texture is generated
         */
        onGeneratedObservable: Observable<ProceduralTexture>;
        /**
         * Event raised before the texture is generated
         */
        onBeforeGenerationObservable: Observable<ProceduralTexture>;
        /**
         * Gets or sets the node material used to create this texture (null if the texture was manually created)
         */
        nodeMaterialSource: Nullable<NodeMaterial>;
        /** @hidden */
        _generateMipMaps: boolean;
        /** @hidden **/
        _effect: Effect;
        /** @hidden */
        _textures: {
            [key: string]: Texture;
        };
        /** @hidden */
        protected _fallbackTexture: Nullable<Texture>;
        private _size;
        private _currentRefreshId;
        private _frameId;
        private _refreshRate;
        private _vertexBuffers;
        private _indexBuffer;
        private _uniforms;
        private _samplers;
        private _fragment;
        private _floats;
        private _ints;
        private _floatsArrays;
        private _colors3;
        private _colors4;
        private _vectors2;
        private _vectors3;
        private _matrices;
        private _fallbackTextureUsed;
        private _fullEngine;
        private _cachedDefines;
        private _contentUpdateId;
        private _contentData;
        /**
         * Instantiates a new procedural texture.
         * Procedural texturing is a way to programmatically create a texture. There are 2 types of procedural textures: code-only, and code that references some classic 2D images, sometimes called 'refMaps' or 'sampler' images.
         * This is the base class of any Procedural texture and contains most of the shareable code.
         * @see https://doc.babylonjs.com/how_to/how_to_use_procedural_textures
         * @param name  Define the name of the texture
         * @param size Define the size of the texture to create
         * @param fragment Define the fragment shader to use to generate the texture or null if it is defined later
         * @param scene Define the scene the texture belongs to
         * @param fallbackTexture Define a fallback texture in case there were issues to create the custom texture
         * @param generateMipMaps Define if the texture should creates mip maps or not
         * @param isCube Define if the texture is a cube texture or not (this will render each faces of the cube)
         * @param textureType The FBO internal texture type
         */
        constructor(name: string, size: RenderTargetTextureSize, fragment: any, scene: Nullable<Scene>, fallbackTexture?: Nullable<Texture>, generateMipMaps?: boolean, isCube?: boolean, textureType?: number);
        /**
         * The effect that is created when initializing the post process.
         * @returns The created effect corresponding the the postprocess.
         */
        getEffect(): Effect;
        /**
         * Gets texture content (Use this function wisely as reading from a texture can be slow)
         * @returns an ArrayBufferView (Uint8Array or Float32Array)
         */
        getContent(): Nullable<ArrayBufferView>;
        private _createIndexBuffer;
        /** @hidden */
        _rebuild(): void;
        /**
         * Resets the texture in order to recreate its associated resources.
         * This can be called in case of context loss
         */
        reset(): void;
        protected _getDefines(): string;
        /**
         * Is the texture ready to be used ? (rendered at least once)
         * @returns true if ready, otherwise, false.
         */
        isReady(): boolean;
        /**
         * Resets the refresh counter of the texture and start bak from scratch.
         * Could be useful to regenerate the texture if it is setup to render only once.
         */
        resetRefreshCounter(): void;
        /**
         * Set the fragment shader to use in order to render the texture.
         * @param fragment This can be set to a path (into the shader store) or to a json object containing a fragmentElement property.
         */
        setFragment(fragment: any): void;
        /**
         * Define the refresh rate of the texture or the rendering frequency.
         * Use 0 to render just once, 1 to render on every frame, 2 to render every two frames and so on...
         */
        get refreshRate(): number;
        set refreshRate(value: number);
        /** @hidden */
        _shouldRender(): boolean;
        /**
         * Get the size the texture is rendering at.
         * @returns the size (on cube texture it is always squared)
         */
        getRenderSize(): RenderTargetTextureSize;
        /**
         * Resize the texture to new value.
         * @param size Define the new size the texture should have
         * @param generateMipMaps Define whether the new texture should create mip maps
         */
        resize(size: number, generateMipMaps: boolean): void;
        private _checkUniform;
        /**
         * Set a texture in the shader program used to render.
         * @param name Define the name of the uniform samplers as defined in the shader
         * @param texture Define the texture to bind to this sampler
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setTexture(name: string, texture: Texture): ProceduralTexture;
        /**
         * Set a float in the shader.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setFloat(name: string, value: number): ProceduralTexture;
        /**
         * Set a int in the shader.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setInt(name: string, value: number): ProceduralTexture;
        /**
         * Set an array of floats in the shader.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setFloats(name: string, value: number[]): ProceduralTexture;
        /**
         * Set a vec3 in the shader from a Color3.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setColor3(name: string, value: Color3): ProceduralTexture;
        /**
         * Set a vec4 in the shader from a Color4.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setColor4(name: string, value: Color4): ProceduralTexture;
        /**
         * Set a vec2 in the shader from a Vector2.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setVector2(name: string, value: Vector2): ProceduralTexture;
        /**
         * Set a vec3 in the shader from a Vector3.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setVector3(name: string, value: Vector3): ProceduralTexture;
        /**
         * Set a mat4 in the shader from a MAtrix.
         * @param name Define the name of the uniform as defined in the shader
         * @param value Define the value to give to the uniform
         * @return the texture itself allowing "fluent" like uniform updates
         */
        setMatrix(name: string, value: Matrix): ProceduralTexture;
        /**
         * Render the texture to its associated render target.
         * @param useCameraPostProcess Define if camera post process should be applied to the texture
         */
        render(useCameraPostProcess?: boolean): void;
        /**
         * Clone the texture.
         * @returns the cloned texture
         */
        clone(): ProceduralTexture;
        /**
         * Dispose the texture and release its asoociated resources.
         */
        dispose(): void;
    }
}
declare module "babylonjs/Particles/baseParticleSystem" {
    import { Nullable } from "babylonjs/types";
    import { Vector2, Vector3 } from "babylonjs/Maths/math.vector";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { ImageProcessingConfiguration, ImageProcessingConfigurationDefines } from "babylonjs/Materials/imageProcessingConfiguration";
    import { ColorGradient, FactorGradient, Color3Gradient, IValueGradient } from "babylonjs/Misc/gradients";
    import { BoxParticleEmitter, IParticleEmitterType, PointParticleEmitter, HemisphericParticleEmitter, SphereParticleEmitter, SphereDirectedParticleEmitter, CylinderParticleEmitter, CylinderDirectedParticleEmitter, ConeParticleEmitter } from "babylonjs/Particles/EmitterTypes/index";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Color4 } from "babylonjs/Maths/math.color";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    import "babylonjs/Engines/Extensions/engine.dynamicBuffer";
    import { Animation } from "babylonjs/Animations/animation";
    import { Scene } from "babylonjs/scene";
    import { ProceduralTexture } from "babylonjs/Materials/Textures/Procedurals/proceduralTexture";
    import { RawTexture } from "babylonjs/Materials/Textures/rawTexture";
    /**
     * This represents the base class for particle system in Babylon.
     * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
     * Particles can take different shapes while emitted like box, sphere, cone or you can write your custom function.
     * @example https://doc.babylonjs.com/babylon101/particles
     */
    export class BaseParticleSystem {
        /**
         * Source color is added to the destination color without alpha affecting the result
         */
        static BLENDMODE_ONEONE: number;
        /**
         * Blend current color and particle color using particle’s alpha
         */
        static BLENDMODE_STANDARD: number;
        /**
         * Add current color and particle color multiplied by particle’s alpha
         */
        static BLENDMODE_ADD: number;
        /**
         * Multiply current color with particle color
         */
        static BLENDMODE_MULTIPLY: number;
        /**
         * Multiply current color with particle color then add current color and particle color multiplied by particle’s alpha
         */
        static BLENDMODE_MULTIPLYADD: number;
        /**
         * List of animations used by the particle system.
         */
        animations: Animation[];
        /**
         * Gets or sets the unique id of the particle system
         */
        uniqueId: number;
        /**
         * The id of the Particle system.
         */
        id: string;
        /**
         * The friendly name of the Particle system.
         */
        name: string;
        /**
         * Snippet ID if the particle system was created from the snippet server
         */
        snippetId: string;
        /**
         * The rendering group used by the Particle system to chose when to render.
         */
        renderingGroupId: number;
        /**
         * The emitter represents the Mesh or position we are attaching the particle system to.
         */
        emitter: Nullable<AbstractMesh | Vector3>;
        /**
         * The maximum number of particles to emit per frame
         */
        emitRate: number;
        /**
         * If you want to launch only a few particles at once, that can be done, as well.
         */
        manualEmitCount: number;
        /**
         * The overall motion speed (0.01 is default update speed, faster updates = faster animation)
         */
        updateSpeed: number;
        /**
         * The amount of time the particle system is running (depends of the overall update speed).
         */
        targetStopDuration: number;
        /**
         * Specifies whether the particle system will be disposed once it reaches the end of the animation.
         */
        disposeOnStop: boolean;
        /**
         * Minimum power of emitting particles.
         */
        minEmitPower: number;
        /**
         * Maximum power of emitting particles.
         */
        maxEmitPower: number;
        /**
         * Minimum life time of emitting particles.
         */
        minLifeTime: number;
        /**
         * Maximum life time of emitting particles.
         */
        maxLifeTime: number;
        /**
         * Minimum Size of emitting particles.
         */
        minSize: number;
        /**
         * Maximum Size of emitting particles.
         */
        maxSize: number;
        /**
         * Minimum scale of emitting particles on X axis.
         */
        minScaleX: number;
        /**
         * Maximum scale of emitting particles on X axis.
         */
        maxScaleX: number;
        /**
         * Minimum scale of emitting particles on Y axis.
         */
        minScaleY: number;
        /**
         * Maximum scale of emitting particles on Y axis.
         */
        maxScaleY: number;
        /**
         * Gets or sets the minimal initial rotation in radians.
         */
        minInitialRotation: number;
        /**
         * Gets or sets the maximal initial rotation in radians.
         */
        maxInitialRotation: number;
        /**
         * Minimum angular speed of emitting particles (Z-axis rotation for each particle).
         */
        minAngularSpeed: number;
        /**
         * Maximum angular speed of emitting particles (Z-axis rotation for each particle).
         */
        maxAngularSpeed: number;
        /**
         * The texture used to render each particle. (this can be a spritesheet)
         */
        particleTexture: Nullable<BaseTexture>;
        /**
         * The layer mask we are rendering the particles through.
         */
        layerMask: number;
        /**
         * This can help using your own shader to render the particle system.
         * The according effect will be created
         */
        customShader: any;
        /**
         * By default particle system starts as soon as they are created. This prevents the
         * automatic start to happen and let you decide when to start emitting particles.
         */
        preventAutoStart: boolean;
        private _noiseTexture;
        /**
         * Gets or sets a texture used to add random noise to particle positions
         */
        get noiseTexture(): Nullable<ProceduralTexture>;
        set noiseTexture(value: Nullable<ProceduralTexture>);
        /** Gets or sets the strength to apply to the noise value (default is (10, 10, 10)) */
        noiseStrength: Vector3;
        /**
         * Callback triggered when the particle animation is ending.
         */
        onAnimationEnd: Nullable<() => void>;
        /**
         * Blend mode use to render the particle, it can be either ParticleSystem.BLENDMODE_ONEONE or ParticleSystem.BLENDMODE_STANDARD.
         */
        blendMode: number;
        /**
         * Forces the particle to write their depth information to the depth buffer. This can help preventing other draw calls
         * to override the particles.
         */
        forceDepthWrite: boolean;
        /** Gets or sets a value indicating how many cycles (or frames) must be executed before first rendering (this value has to be set before starting the system). Default is 0 */
        preWarmCycles: number;
        /** Gets or sets a value indicating the time step multiplier to use in pre-warm mode (default is 1) */
        preWarmStepOffset: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) defines the speed of the sprite loop (default is 1 meaning the animation will play once during the entire particle lifetime)
         */
        spriteCellChangeSpeed: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) defines the first sprite cell to display
         */
        startSpriteCellID: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) defines the last sprite cell to display
         */
        endSpriteCellID: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines the sprite cell width to use
         */
        spriteCellWidth: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines the sprite cell height to use
         */
        spriteCellHeight: number;
        /**
         * This allows the system to random pick the start cell ID between startSpriteCellID and endSpriteCellID
         */
        spriteRandomStartCell: boolean;
        /** Gets or sets a Vector2 used to move the pivot (by default (0,0)) */
        translationPivot: Vector2;
        /** @hidden */
        protected _isAnimationSheetEnabled: boolean;
        /**
         * Gets or sets a boolean indicating that hosted animations (in the system.animations array) must be started when system.start() is called
         */
        beginAnimationOnStart: boolean;
        /**
         * Gets or sets the frame to start the animation from when beginAnimationOnStart is true
         */
        beginAnimationFrom: number;
        /**
         * Gets or sets the frame to end the animation on when beginAnimationOnStart is true
         */
        beginAnimationTo: number;
        /**
         * Gets or sets a boolean indicating if animations must loop when beginAnimationOnStart is true
         */
        beginAnimationLoop: boolean;
        /**
         * Gets or sets a world offset applied to all particles
         */
        worldOffset: Vector3;
        /**
         * Gets or sets whether an animation sprite sheet is enabled or not on the particle system
         */
        get isAnimationSheetEnabled(): boolean;
        set isAnimationSheetEnabled(value: boolean);
        /**
         * Get hosting scene
         * @returns the scene
         */
        getScene(): Nullable<Scene>;
        /**
         * You can use gravity if you want to give an orientation to your particles.
         */
        gravity: Vector3;
        protected _colorGradients: Nullable<Array<ColorGradient>>;
        protected _sizeGradients: Nullable<Array<FactorGradient>>;
        protected _lifeTimeGradients: Nullable<Array<FactorGradient>>;
        protected _angularSpeedGradients: Nullable<Array<FactorGradient>>;
        protected _velocityGradients: Nullable<Array<FactorGradient>>;
        protected _limitVelocityGradients: Nullable<Array<FactorGradient>>;
        protected _dragGradients: Nullable<Array<FactorGradient>>;
        protected _emitRateGradients: Nullable<Array<FactorGradient>>;
        protected _startSizeGradients: Nullable<Array<FactorGradient>>;
        protected _rampGradients: Nullable<Array<Color3Gradient>>;
        protected _colorRemapGradients: Nullable<Array<FactorGradient>>;
        protected _alphaRemapGradients: Nullable<Array<FactorGradient>>;
        protected _hasTargetStopDurationDependantGradient(): boolean | null;
        /**
         * Defines the delay in milliseconds before starting the system (0 by default)
         */
        startDelay: number;
        /**
         * Gets the current list of drag gradients.
         * You must use addDragGradient and removeDragGradient to udpate this list
         * @returns the list of drag gradients
         */
        getDragGradients(): Nullable<Array<FactorGradient>>;
        /** Gets or sets a value indicating the damping to apply if the limit velocity factor is reached */
        limitVelocityDamping: number;
        /**
         * Gets the current list of limit velocity gradients.
         * You must use addLimitVelocityGradient and removeLimitVelocityGradient to udpate this list
         * @returns the list of limit velocity gradients
         */
        getLimitVelocityGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of color gradients.
         * You must use addColorGradient and removeColorGradient to udpate this list
         * @returns the list of color gradients
         */
        getColorGradients(): Nullable<Array<ColorGradient>>;
        /**
         * Gets the current list of size gradients.
         * You must use addSizeGradient and removeSizeGradient to udpate this list
         * @returns the list of size gradients
         */
        getSizeGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of color remap gradients.
         * You must use addColorRemapGradient and removeColorRemapGradient to udpate this list
         * @returns the list of color remap gradients
         */
        getColorRemapGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of alpha remap gradients.
         * You must use addAlphaRemapGradient and removeAlphaRemapGradient to udpate this list
         * @returns the list of alpha remap gradients
         */
        getAlphaRemapGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of life time gradients.
         * You must use addLifeTimeGradient and removeLifeTimeGradient to udpate this list
         * @returns the list of life time gradients
         */
        getLifeTimeGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of angular speed gradients.
         * You must use addAngularSpeedGradient and removeAngularSpeedGradient to udpate this list
         * @returns the list of angular speed gradients
         */
        getAngularSpeedGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of velocity gradients.
         * You must use addVelocityGradient and removeVelocityGradient to udpate this list
         * @returns the list of velocity gradients
         */
        getVelocityGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of start size gradients.
         * You must use addStartSizeGradient and removeStartSizeGradient to udpate this list
         * @returns the list of start size gradients
         */
        getStartSizeGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of emit rate gradients.
         * You must use addEmitRateGradient and removeEmitRateGradient to udpate this list
         * @returns the list of emit rate gradients
         */
        getEmitRateGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        get direction1(): Vector3;
        set direction1(value: Vector3);
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        get direction2(): Vector3;
        set direction2(value: Vector3);
        /**
         * Minimum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        get minEmitBox(): Vector3;
        set minEmitBox(value: Vector3);
        /**
         * Maximum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        get maxEmitBox(): Vector3;
        set maxEmitBox(value: Vector3);
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors
         */
        color1: Color4;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors
         */
        color2: Color4;
        /**
         * Color the particle will have at the end of its lifetime
         */
        colorDead: Color4;
        /**
         * An optional mask to filter some colors out of the texture, or filter a part of the alpha channel
         */
        textureMask: Color4;
        /**
         * The particle emitter type defines the emitter used by the particle system.
         * It can be for example box, sphere, or cone...
         */
        particleEmitterType: IParticleEmitterType;
        /** @hidden */
        _isSubEmitter: boolean;
        /**
         * Gets or sets the billboard mode to use when isBillboardBased = true.
         * Value can be: ParticleSystem.BILLBOARDMODE_ALL, ParticleSystem.BILLBOARDMODE_Y, ParticleSystem.BILLBOARDMODE_STRETCHED
         */
        billboardMode: number;
        protected _isBillboardBased: boolean;
        /**
         * Gets or sets a boolean indicating if the particles must be rendered as billboard or aligned with the direction
         */
        get isBillboardBased(): boolean;
        set isBillboardBased(value: boolean);
        /**
         * The scene the particle system belongs to.
         */
        protected _scene: Nullable<Scene>;
        /**
         * The engine the particle system belongs to.
         */
        protected _engine: ThinEngine;
        /**
         * Local cache of defines for image processing.
         */
        protected _imageProcessingConfigurationDefines: ImageProcessingConfigurationDefines;
        /**
         * Default configuration related to image processing available in the standard Material.
         */
        protected _imageProcessingConfiguration: Nullable<ImageProcessingConfiguration>;
        /**
         * Gets the image processing configuration used either in this material.
         */
        get imageProcessingConfiguration(): Nullable<ImageProcessingConfiguration>;
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        set imageProcessingConfiguration(value: Nullable<ImageProcessingConfiguration>);
        /**
         * Attaches a new image processing configuration to the Standard Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>): void;
        /** @hidden */
        protected _reset(): void;
        /** @hidden */
        protected _removeGradientAndTexture(gradient: number, gradients: Nullable<IValueGradient[]>, texture: Nullable<RawTexture>): BaseParticleSystem;
        /**
         * Instantiates a particle system.
         * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
         * @param name The name of the particle system
         */
        constructor(name: string);
        /**
         * Creates a Point Emitter for the particle system (emits directly from the emitter position)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the box
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the box
         * @returns the emitter
         */
        createPointEmitter(direction1: Vector3, direction2: Vector3): PointParticleEmitter;
        /**
         * Creates a Hemisphere Emitter for the particle system (emits along the hemisphere radius)
         * @param radius The radius of the hemisphere to emit from
         * @param radiusRange The range of the hemisphere to emit from [0-1] 0 Surface Only, 1 Entire Radius
         * @returns the emitter
         */
        createHemisphericEmitter(radius?: number, radiusRange?: number): HemisphericParticleEmitter;
        /**
         * Creates a Sphere Emitter for the particle system (emits along the sphere radius)
         * @param radius The radius of the sphere to emit from
         * @param radiusRange The range of the sphere to emit from [0-1] 0 Surface Only, 1 Entire Radius
         * @returns the emitter
         */
        createSphereEmitter(radius?: number, radiusRange?: number): SphereParticleEmitter;
        /**
         * Creates a Directed Sphere Emitter for the particle system (emits between direction1 and direction2)
         * @param radius The radius of the sphere to emit from
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the sphere
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the sphere
         * @returns the emitter
         */
        createDirectedSphereEmitter(radius?: number, direction1?: Vector3, direction2?: Vector3): SphereDirectedParticleEmitter;
        /**
         * Creates a Cylinder Emitter for the particle system (emits from the cylinder to the particle position)
         * @param radius The radius of the emission cylinder
         * @param height The height of the emission cylinder
         * @param radiusRange The range of emission [0-1] 0 Surface only, 1 Entire Radius
         * @param directionRandomizer How much to randomize the particle direction [0-1]
         * @returns the emitter
         */
        createCylinderEmitter(radius?: number, height?: number, radiusRange?: number, directionRandomizer?: number): CylinderParticleEmitter;
        /**
         * Creates a Directed Cylinder Emitter for the particle system (emits between direction1 and direction2)
         * @param radius The radius of the cylinder to emit from
         * @param height The height of the emission cylinder
         * @param radiusRange the range of the emission cylinder [0-1] 0 Surface only, 1 Entire Radius (1 by default)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the cylinder
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the cylinder
         * @returns the emitter
         */
        createDirectedCylinderEmitter(radius?: number, height?: number, radiusRange?: number, direction1?: Vector3, direction2?: Vector3): CylinderDirectedParticleEmitter;
        /**
         * Creates a Cone Emitter for the particle system (emits from the cone to the particle position)
         * @param radius The radius of the cone to emit from
         * @param angle The base angle of the cone
         * @returns the emitter
         */
        createConeEmitter(radius?: number, angle?: number): ConeParticleEmitter;
        /**
         * Creates a Box Emitter for the particle system. (emits between direction1 and direction2 from withing the box defined by minEmitBox and maxEmitBox)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the box
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the box
         * @param minEmitBox Particles are emitted from the box between minEmitBox and maxEmitBox
         * @param maxEmitBox  Particles are emitted from the box between minEmitBox and maxEmitBox
         * @returns the emitter
         */
        createBoxEmitter(direction1: Vector3, direction2: Vector3, minEmitBox: Vector3, maxEmitBox: Vector3): BoxParticleEmitter;
    }
}
declare module "babylonjs/Particles/subEmitter" {
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    import { Scene } from "babylonjs/scene";
    import { ParticleSystem } from "babylonjs/Particles/particleSystem";
    /**
         * Type of sub emitter
         */
    export enum SubEmitterType {
        /**
         * Attached to the particle over it's lifetime
         */
        ATTACHED = 0,
        /**
         * Created when the particle dies
         */
        END = 1
    }
    /**
     * Sub emitter class used to emit particles from an existing particle
     */
    export class SubEmitter {
        /**
         * the particle system to be used by the sub emitter
         */
        particleSystem: ParticleSystem;
        /**
         * Type of the submitter (Default: END)
         */
        type: SubEmitterType;
        /**
         * If the particle should inherit the direction from the particle it's attached to. (+Y will face the direction the particle is moving) (Default: false)
         * Note: This only is supported when using an emitter of type Mesh
         */
        inheritDirection: boolean;
        /**
         * How much of the attached particles speed should be added to the sub emitted particle (default: 0)
         */
        inheritedVelocityAmount: number;
        /**
         * Creates a sub emitter
         * @param particleSystem the particle system to be used by the sub emitter
         */
        constructor(
        /**
         * the particle system to be used by the sub emitter
         */
        particleSystem: ParticleSystem);
        /**
         * Clones the sub emitter
         * @returns the cloned sub emitter
         */
        clone(): SubEmitter;
        /**
         * Serialize current object to a JSON object
         * @returns the serialized object
         */
        serialize(): any;
        /** @hidden */
        static _ParseParticleSystem(system: any, sceneOrEngine: Scene | ThinEngine, rootUrl: string): ParticleSystem;
        /**
         * Creates a new SubEmitter from a serialized JSON version
         * @param serializationObject defines the JSON object to read from
         * @param sceneOrEngine defines the hosting scene or the hosting engine
         * @param rootUrl defines the rootUrl for data loading
         * @returns a new SubEmitter
         */
        static Parse(serializationObject: any, sceneOrEngine: Scene | ThinEngine, rootUrl: string): SubEmitter;
        /** Release associated resources */
        dispose(): void;
    }
}
declare module "babylonjs/Shaders/ShadersInclude/clipPlaneFragmentDeclaration" {
    /** @hidden */
    export var clipPlaneFragmentDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/imageProcessingDeclaration" {
    /** @hidden */
    export var imageProcessingDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/imageProcessingFunctions" {
    /** @hidden */
    export var imageProcessingFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/clipPlaneFragment" {
    /** @hidden */
    export var clipPlaneFragment: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/particles.fragment" {
    import "babylonjs/Shaders/ShadersInclude/clipPlaneFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingDeclaration";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingFunctions";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneFragment";
    /** @hidden */
    export var particlesPixelShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/clipPlaneVertexDeclaration" {
    /** @hidden */
    export var clipPlaneVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/clipPlaneVertex" {
    /** @hidden */
    export var clipPlaneVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/particles.vertex" {
    import "babylonjs/Shaders/ShadersInclude/clipPlaneVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneVertex";
    /** @hidden */
    export var particlesVertexShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Misc/interfaces/iClipPlanesHolder" {
    import { Nullable } from "babylonjs/types";
    import { Plane } from "babylonjs/Maths/math";
    /**
     * Interface used to define entities containing multiple clip planes
     */
    export interface IClipPlanesHolder {
        /**
         * Gets or sets the active clipplane 1
         */
        clipPlane: Nullable<Plane>;
        /**
         * Gets or sets the active clipplane 2
         */
        clipPlane2: Nullable<Plane>;
        /**
         * Gets or sets the active clipplane 3
         */
        clipPlane3: Nullable<Plane>;
        /**
         * Gets or sets the active clipplane 4
         */
        clipPlane4: Nullable<Plane>;
        /**
         * Gets or sets the active clipplane 5
         */
        clipPlane5: Nullable<Plane>;
        /**
         * Gets or sets the active clipplane 6
         */
        clipPlane6: Nullable<Plane>;
    }
}
declare module "babylonjs/Materials/thinMaterialHelper" {
    import { Effect } from "babylonjs/Materials/effect";
    import { IClipPlanesHolder } from "babylonjs/Misc/interfaces/iClipPlanesHolder";
    /**
     * "Static Class" containing a few commonly used helper while dealing with material for rendering purpose.
     *
     * It is complementary with MaterialHelper but provides completely independent functions (for tree shaking sake)
     *
     * This works by convention in BabylonJS but is meant to be use only with shader following the in place naming rules and conventions.
     */
    export class ThinMaterialHelper {
        /**
         * Binds the clip plane information from the holder to the effect.
         * @param effect The effect we are binding the data to
         * @param holder The entity containing the clip plane information
         */
        static BindClipPlane(effect: Effect, holder: IClipPlanesHolder): void;
    }
}
declare module "babylonjs/Engines/Extensions/engine.alpha" {
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Sets alpha constants used by some alpha blending modes
             * @param r defines the red component
             * @param g defines the green component
             * @param b defines the blue component
             * @param a defines the alpha component
             */
            setAlphaConstants(r: number, g: number, b: number, a: number): void;
            /**
             * Sets the current alpha mode
             * @param mode defines the mode to use (one of the Engine.ALPHA_XXX)
             * @param noDepthWriteChange defines if depth writing state should remains unchanged (false by default)
             * @see https://doc.babylonjs.com/resources/transparency_and_how_meshes_are_rendered
             */
            setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
            /**
             * Gets the current alpha mode
             * @see https://doc.babylonjs.com/resources/transparency_and_how_meshes_are_rendered
             * @returns the current alpha mode
             */
            getAlphaMode(): number;
            /**
             * Sets the current alpha equation
             * @param equation defines the equation to use (one of the Engine.ALPHA_EQUATION_XXX)
             */
            setAlphaEquation(equation: number): void;
            /**
             * Gets the current alpha equation.
             * @returns the current alpha equation
             */
            getAlphaEquation(): number;
        }
    }
}
declare module "babylonjs/Particles/particleSystem" {
    import { Nullable } from "babylonjs/types";
    import { FactorGradient, Color3Gradient } from "babylonjs/Misc/gradients";
    import { Observable } from "babylonjs/Misc/observable";
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { IDisposable } from "babylonjs/scene";
    import { IParticleSystem } from "babylonjs/Particles/IParticleSystem";
    import { BaseParticleSystem } from "babylonjs/Particles/baseParticleSystem";
    import { Particle } from "babylonjs/Particles/particle";
    import { SubEmitter } from "babylonjs/Particles/subEmitter";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import "babylonjs/Shaders/particles.fragment";
    import "babylonjs/Shaders/particles.vertex";
    import { Color4, Color3 } from "babylonjs/Maths/math.color";
    import { ThinEngine } from "babylonjs/Engines/thinEngine";
    import "babylonjs/Engines/Extensions/engine.alpha";
    import { Scene } from "babylonjs/scene";
    /**
     * This represents a particle system in Babylon.
     * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
     * Particles can take different shapes while emitted like box, sphere, cone or you can write your custom function.
     * @example https://doc.babylonjs.com/babylon101/particles
     */
    export class ParticleSystem extends BaseParticleSystem implements IDisposable, IAnimatable, IParticleSystem {
        /**
         * Billboard mode will only apply to Y axis
         */
        static readonly BILLBOARDMODE_Y: number;
        /**
         * Billboard mode will apply to all axes
         */
        static readonly BILLBOARDMODE_ALL: number;
        /**
         * Special billboard mode where the particle will be biilboard to the camera but rotated to align with direction
         */
        static readonly BILLBOARDMODE_STRETCHED: number;
        /**
         * This function can be defined to provide custom update for active particles.
         * This function will be called instead of regular update (age, position, color, etc.).
         * Do not forget that this function will be called on every frame so try to keep it simple and fast :)
         */
        updateFunction: (particles: Particle[]) => void;
        private _emitterWorldMatrix;
        /**
         * This function can be defined to specify initial direction for every new particle.
         * It by default use the emitterType defined function
         */
        startDirectionFunction: (worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean) => void;
        /**
         * This function can be defined to specify initial position for every new particle.
         * It by default use the emitterType defined function
         */
        startPositionFunction: (worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean) => void;
        /**
         * @hidden
         */
        _inheritedVelocityOffset: Vector3;
        /**
        * An event triggered when the system is disposed
        */
        onDisposeObservable: Observable<IParticleSystem>;
        /**
        * An event triggered when the system is stopped
        */
        onStoppedObservable: Observable<IParticleSystem>;
        private _onDisposeObserver;
        /**
         * Sets a callback that will be triggered when the system is disposed
         */
        set onDispose(callback: () => void);
        private _particles;
        private _epsilon;
        private _capacity;
        private _stockParticles;
        private _newPartsExcess;
        private _vertexData;
        private _vertexBuffer;
        private _vertexBuffers;
        private _spriteBuffer;
        private _indexBuffer;
        private _effect;
        private _customEffect;
        private _cachedDefines;
        private _scaledColorStep;
        private _colorDiff;
        private _scaledDirection;
        private _scaledGravity;
        private _currentRenderId;
        private _alive;
        private _useInstancing;
        private _vertexArrayObject;
        private _started;
        private _stopped;
        private _actualFrame;
        private _scaledUpdateSpeed;
        private _vertexBufferSize;
        /** @hidden */
        _currentEmitRateGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentEmitRate1: number;
        /** @hidden */
        _currentEmitRate2: number;
        /** @hidden */
        _currentStartSizeGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentStartSize1: number;
        /** @hidden */
        _currentStartSize2: number;
        private readonly _rawTextureWidth;
        private _rampGradientsTexture;
        private _useRampGradients;
        /** Gets or sets a matrix to use to compute projection */
        defaultProjectionMatrix: Matrix;
        /** Gets or sets a matrix to use to compute view */
        defaultViewMatrix: Matrix;
        /** Gets or sets a boolean indicating that ramp gradients must be used
         * @see https://doc.babylonjs.com/babylon101/particles#ramp-gradients
         */
        get useRampGradients(): boolean;
        set useRampGradients(value: boolean);
        /**
         * The Sub-emitters templates that will be used to generate the sub particle system to be associated with the system, this property is used by the root particle system only.
         * When a particle is spawned, an array will be chosen at random and all the emitters in that array will be attached to the particle.  (Default: [])
         */
        subEmitters: Array<ParticleSystem | SubEmitter | Array<SubEmitter>>;
        private _subEmitters;
        /**
         * @hidden
         * If the particle systems emitter should be disposed when the particle system is disposed
         */
        _disposeEmitterOnDispose: boolean;
        /**
        * The current active Sub-systems, this property is used by the root particle system only.
        */
        activeSubSystems: Array<ParticleSystem>;
        /**
         * Specifies if the particles are updated in emitter local space or world space
         */
        isLocal: boolean;
        private _rootParticleSystem;
        /**
         * Gets the current list of active particles
         */
        get particles(): Particle[];
        /**
         * Gets the number of particles active at the same time.
         * @returns The number of active particles.
         */
        getActiveCount(): number;
        /**
         * Returns the string "ParticleSystem"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Gets a boolean indicating that the system is stopping
         * @returns true if the system is currently stopping
         */
        isStopping(): boolean;
        /**
         * Gets the custom effect used to render the particles
         * @param blendMode Blend mode for which the effect should be retrieved
         * @returns The effect
         */
        getCustomEffect(blendMode?: number): Nullable<Effect>;
        /**
         * Sets the custom effect used to render the particles
         * @param effect The effect to set
         * @param blendMode Blend mode for which the effect should be set
         */
        setCustomEffect(effect: Nullable<Effect>, blendMode?: number): void;
        /** @hidden */
        private _onBeforeDrawParticlesObservable;
        /**
         * Observable that will be called just before the particles are drawn
         */
        get onBeforeDrawParticlesObservable(): Observable<Nullable<Effect>>;
        /**
         * Gets the name of the particle vertex shader
         */
        get vertexShaderName(): string;
        /**
         * Instantiates a particle system.
         * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
         * @param name The name of the particle system
         * @param capacity The max number of particles alive at the same time
         * @param sceneOrEngine The scene the particle system belongs to or the engine to use if no scene
         * @param customEffect a custom effect used to change the way particles are rendered by default
         * @param isAnimationSheetEnabled Must be true if using a spritesheet to animate the particles texture
         * @param epsilon Offset used to render the particles
         */
        constructor(name: string, capacity: number, sceneOrEngine: Scene | ThinEngine, customEffect?: Nullable<Effect>, isAnimationSheetEnabled?: boolean, epsilon?: number);
        private _addFactorGradient;
        private _removeFactorGradient;
        /**
         * Adds a new life time gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the life time factor to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addLifeTimeGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific life time gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeLifeTimeGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new size gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the size factor to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addSizeGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific size gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeSizeGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new color remap gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param min defines the color remap minimal range
         * @param max defines the color remap maximal range
         * @returns the current particle system
         */
        addColorRemapGradient(gradient: number, min: number, max: number): IParticleSystem;
        /**
         * Remove a specific color remap gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeColorRemapGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new alpha remap gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param min defines the alpha remap minimal range
         * @param max defines the alpha remap maximal range
         * @returns the current particle system
         */
        addAlphaRemapGradient(gradient: number, min: number, max: number): IParticleSystem;
        /**
         * Remove a specific alpha remap gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeAlphaRemapGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new angular speed gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the angular speed  to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addAngularSpeedGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific angular speed gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeAngularSpeedGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new velocity gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the velocity to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addVelocityGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific velocity gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeVelocityGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new limit velocity gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the limit velocity value to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addLimitVelocityGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific limit velocity gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeLimitVelocityGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new drag gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the drag value to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addDragGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific drag gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeDragGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new emit rate gradient (please note that this will only work if you set the targetStopDuration property)
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the emit rate value to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addEmitRateGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific emit rate gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeEmitRateGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new start size gradient (please note that this will only work if you set the targetStopDuration property)
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the start size value to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addStartSizeGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific start size gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeStartSizeGradient(gradient: number): IParticleSystem;
        private _createRampGradientTexture;
        /**
         * Gets the current list of ramp gradients.
         * You must use addRampGradient and removeRampGradient to udpate this list
         * @returns the list of ramp gradients
         */
        getRampGradients(): Nullable<Array<Color3Gradient>>;
        /** Force the system to rebuild all gradients that need to be resync */
        forceRefreshGradients(): void;
        private _syncRampGradientTexture;
        /**
         * Adds a new ramp gradient used to remap particle colors
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param color defines the color to affect to the specified gradient
         * @returns the current particle system
         */
        addRampGradient(gradient: number, color: Color3): ParticleSystem;
        /**
         * Remove a specific ramp gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeRampGradient(gradient: number): ParticleSystem;
        /**
         * Adds a new color gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param color1 defines the color to affect to the specified gradient
         * @param color2 defines an additional color used to define a range ([color, color2]) with main color to pick the final color from
         * @returns this particle system
         */
        addColorGradient(gradient: number, color1: Color4, color2?: Color4): IParticleSystem;
        /**
         * Remove a specific color gradient
         * @param gradient defines the gradient to remove
         * @returns this particle system
         */
        removeColorGradient(gradient: number): IParticleSystem;
        private _fetchR;
        protected _reset(): void;
        private _resetEffect;
        private _createVertexBuffers;
        private _createIndexBuffer;
        /**
         * Gets the maximum number of particles active at the same time.
         * @returns The max number of active particles.
         */
        getCapacity(): number;
        /**
         * Gets whether there are still active particles in the system.
         * @returns True if it is alive, otherwise false.
         */
        isAlive(): boolean;
        /**
         * Gets if the system has been started. (Note: this will still be true after stop is called)
         * @returns True if it has been started, otherwise false.
         */
        isStarted(): boolean;
        private _prepareSubEmitterInternalArray;
        /**
         * Starts the particle system and begins to emit
         * @param delay defines the delay in milliseconds before starting the system (this.startDelay by default)
         */
        start(delay?: number): void;
        /**
         * Stops the particle system.
         * @param stopSubEmitters if true it will stop the current system and all created sub-Systems if false it will stop the current root system only, this param is used by the root particle system only. the default value is true.
         */
        stop(stopSubEmitters?: boolean): void;
        /**
         * Remove all active particles
         */
        reset(): void;
        /**
         * @hidden (for internal use only)
         */
        _appendParticleVertex(index: number, particle: Particle, offsetX: number, offsetY: number): void;
        /**
         * "Recycles" one of the particle by copying it back to the "stock" of particles and removing it from the active list.
         * Its lifetime will start back at 0.
         */
        recycleParticle: (particle: Particle) => void;
        private _stopSubEmitters;
        private _createParticle;
        private _removeFromRoot;
        private _emitFromParticle;
        private _update;
        /** @hidden */
        static _GetAttributeNamesOrOptions(isAnimationSheetEnabled?: boolean, isBillboardBased?: boolean, useRampGradients?: boolean): string[];
        /** @hidden */
        static _GetEffectCreationOptions(isAnimationSheetEnabled?: boolean): string[];
        /**
         * Fill the defines array according to the current settings of the particle system
         * @param defines Array to be updated
         * @param blendMode blend mode to take into account when updating the array
         */
        fillDefines(defines: Array<string>, blendMode: number): void;
        /**
         * Fill the uniforms, attributes and samplers arrays according to the current settings of the particle system
         * @param uniforms Uniforms array to fill
         * @param attributes Attributes array to fill
         * @param samplers Samplers array to fill
         */
        fillUniformsAttributesAndSamplerNames(uniforms: Array<string>, attributes: Array<string>, samplers: Array<string>): void;
        /** @hidden */
        private _getEffect;
        /**
         * Animates the particle system for the current frame by emitting new particles and or animating the living ones.
         * @param preWarmOnly will prevent the system from updating the vertex buffer (default is false)
         */
        animate(preWarmOnly?: boolean): void;
        private _appendParticleVertices;
        /**
         * Rebuilds the particle system.
         */
        rebuild(): void;
        /**
         * Is this system ready to be used/rendered
         * @return true if the system is ready
         */
        isReady(): boolean;
        private _render;
        /**
         * Renders the particle system in its current state.
         * @returns the current number of particles
         */
        render(): number;
        /**
         * Disposes the particle system and free the associated resources
         * @param disposeTexture defines if the particule texture must be disposed as well (true by default)
         */
        dispose(disposeTexture?: boolean): void;
        /**
         * Clones the particle system.
         * @param name The name of the cloned object
         * @param newEmitter The new emitter to use
         * @returns the cloned particle system
         */
        clone(name: string, newEmitter: any): ParticleSystem;
        /**
         * Serializes the particle system to a JSON object
         * @param serializeTexture defines if the texture must be serialized as well
         * @returns the JSON object
         */
        serialize(serializeTexture?: boolean): any;
        /** @hidden */
        static _Serialize(serializationObject: any, particleSystem: IParticleSystem, serializeTexture: boolean): void;
        /** @hidden */
        static _Parse(parsedParticleSystem: any, particleSystem: IParticleSystem, sceneOrEngine: Scene | ThinEngine, rootUrl: string): void;
        /**
         * Parses a JSON object to create a particle system.
         * @param parsedParticleSystem The JSON object to parse
         * @param sceneOrEngine The scene or the engine to create the particle system in
         * @param rootUrl The root url to use to load external dependencies like texture
         * @param doNotStart Ignore the preventAutoStart attribute and does not start
         * @returns the Parsed particle system
         */
        static Parse(parsedParticleSystem: any, sceneOrEngine: Scene | ThinEngine, rootUrl: string, doNotStart?: boolean): ParticleSystem;
    }
}
declare module "babylonjs/Particles/particle" {
    import { Nullable } from "babylonjs/types";
    import { Vector2, Vector3, Vector4 } from "babylonjs/Maths/math.vector";
    import { Color4 } from "babylonjs/Maths/math.color";
    import { ParticleSystem } from "babylonjs/Particles/particleSystem";
    import { SubEmitter } from "babylonjs/Particles/subEmitter";
    import { ColorGradient, FactorGradient } from "babylonjs/Misc/gradients";
    /**
     * A particle represents one of the element emitted by a particle system.
     * This is mainly define by its coordinates, direction, velocity and age.
     */
    export class Particle {
        /**
         * The particle system the particle belongs to.
         */
        particleSystem: ParticleSystem;
        private static _Count;
        /**
         * Unique ID of the particle
         */
        id: number;
        /**
         * The world position of the particle in the scene.
         */
        position: Vector3;
        /**
         * The world direction of the particle in the scene.
         */
        direction: Vector3;
        /**
         * The color of the particle.
         */
        color: Color4;
        /**
         * The color change of the particle per step.
         */
        colorStep: Color4;
        /**
         * Defines how long will the life of the particle be.
         */
        lifeTime: number;
        /**
         * The current age of the particle.
         */
        age: number;
        /**
         * The current size of the particle.
         */
        size: number;
        /**
         * The current scale of the particle.
         */
        scale: Vector2;
        /**
         * The current angle of the particle.
         */
        angle: number;
        /**
         * Defines how fast is the angle changing.
         */
        angularSpeed: number;
        /**
         * Defines the cell index used by the particle to be rendered from a sprite.
         */
        cellIndex: number;
        /**
         * The information required to support color remapping
         */
        remapData: Vector4;
        /** @hidden */
        _randomCellOffset?: number;
        /** @hidden */
        _initialDirection: Nullable<Vector3>;
        /** @hidden */
        _attachedSubEmitters: Nullable<Array<SubEmitter>>;
        /** @hidden */
        _initialStartSpriteCellID: number;
        /** @hidden */
        _initialEndSpriteCellID: number;
        /** @hidden */
        _currentColorGradient: Nullable<ColorGradient>;
        /** @hidden */
        _currentColor1: Color4;
        /** @hidden */
        _currentColor2: Color4;
        /** @hidden */
        _currentSizeGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentSize1: number;
        /** @hidden */
        _currentSize2: number;
        /** @hidden */
        _currentAngularSpeedGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentAngularSpeed1: number;
        /** @hidden */
        _currentAngularSpeed2: number;
        /** @hidden */
        _currentVelocityGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentVelocity1: number;
        /** @hidden */
        _currentVelocity2: number;
        /** @hidden */
        _currentLimitVelocityGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentLimitVelocity1: number;
        /** @hidden */
        _currentLimitVelocity2: number;
        /** @hidden */
        _currentDragGradient: Nullable<FactorGradient>;
        /** @hidden */
        _currentDrag1: number;
        /** @hidden */
        _currentDrag2: number;
        /** @hidden */
        _randomNoiseCoordinates1: Vector3;
        /** @hidden */
        _randomNoiseCoordinates2: Vector3;
        /** @hidden */
        _localPosition?: Vector3;
        /**
         * Creates a new instance Particle
         * @param particleSystem the particle system the particle belongs to
         */
        constructor(
        /**
         * The particle system the particle belongs to.
         */
        particleSystem: ParticleSystem);
        private updateCellInfoFromSystem;
        /**
         * Defines how the sprite cell index is updated for the particle
         */
        updateCellIndex(): void;
        /** @hidden */
        _inheritParticleInfoToSubEmitter(subEmitter: SubEmitter): void;
        /** @hidden */
        _inheritParticleInfoToSubEmitters(): void;
        /** @hidden */
        _reset(): void;
        /**
         * Copy the properties of particle to another one.
         * @param other the particle to copy the information to.
         */
        copyTo(other: Particle): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/IParticleEmitterType" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    /**
     * Particle emitter represents a volume emitting particles.
     * This is the responsibility of the implementation to define the volume shape like cone/sphere/box.
     */
    export interface IParticleEmitterType {
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): IParticleEmitterType;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns the effect defines string
         */
        getEffectDefines(): string;
        /**
         * Returns a string representing the class name
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         * @param scene defines the hosting scene
         */
        parse(serializationObject: any, scene: Nullable<Scene>): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/boxParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    /**
     * Particle emitter emitting particles from the inside of a box.
     * It emits the particles randomly between 2 given directions.
     */
    export class BoxParticleEmitter implements IParticleEmitterType {
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         */
        direction1: Vector3;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         */
        direction2: Vector3;
        /**
         * Minimum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         */
        minEmitBox: Vector3;
        /**
         * Maximum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         */
        maxEmitBox: Vector3;
        /**
         * Creates a new instance BoxParticleEmitter
         */
        constructor();
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): BoxParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "BoxParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/coneParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    /**
     * Particle emitter emitting particles from the inside of a cone.
     * It emits the particles alongside the cone volume from the base to the particle.
     * The emission direction might be randomized.
     */
    export class ConeParticleEmitter implements IParticleEmitterType {
        /** defines how much to randomize the particle direction [0-1] (default is 0) */
        directionRandomizer: number;
        private _radius;
        private _angle;
        private _height;
        /**
         * Gets or sets a value indicating where on the radius the start position should be picked (1 = everywhere, 0 = only surface)
         */
        radiusRange: number;
        /**
         * Gets or sets a value indicating where on the height the start position should be picked (1 = everywhere, 0 = only surface)
         */
        heightRange: number;
        /**
         * Gets or sets a value indicating if all the particles should be emitted from the spawn point only (the base of the cone)
         */
        emitFromSpawnPointOnly: boolean;
        /**
         * Gets or sets the radius of the emission cone
         */
        get radius(): number;
        set radius(value: number);
        /**
         * Gets or sets the angle of the emission cone
         */
        get angle(): number;
        set angle(value: number);
        private _buildHeight;
        /**
         * Creates a new instance ConeParticleEmitter
         * @param radius the radius of the emission cone (1 by default)
         * @param angle the cone base angle (PI by default)
         * @param directionRandomizer defines how much to randomize the particle direction [0-1] (default is 0)
         */
        constructor(radius?: number, angle?: number, 
        /** defines how much to randomize the particle direction [0-1] (default is 0) */
        directionRandomizer?: number);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): ConeParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "ConeParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/cylinderParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    /**
     * Particle emitter emitting particles from the inside of a cylinder.
     * It emits the particles alongside the cylinder radius. The emission direction might be randomized.
     */
    export class CylinderParticleEmitter implements IParticleEmitterType {
        /**
         * The radius of the emission cylinder.
         */
        radius: number;
        /**
         * The height of the emission cylinder.
         */
        height: number;
        /**
         * The range of emission [0-1] 0 Surface only, 1 Entire Radius.
         */
        radiusRange: number;
        /**
         * How much to randomize the particle direction [0-1].
         */
        directionRandomizer: number;
        /**
        * Creates a new instance CylinderParticleEmitter
        * @param radius the radius of the emission cylinder (1 by default)
        * @param height the height of the emission cylinder (1 by default)
        * @param radiusRange the range of the emission cylinder [0-1] 0 Surface only, 1 Entire Radius (1 by default)
        * @param directionRandomizer defines how much to randomize the particle direction [0-1]
        */
        constructor(
        /**
         * The radius of the emission cylinder.
         */
        radius?: number, 
        /**
         * The height of the emission cylinder.
         */
        height?: number, 
        /**
         * The range of emission [0-1] 0 Surface only, 1 Entire Radius.
         */
        radiusRange?: number, 
        /**
         * How much to randomize the particle direction [0-1].
         */
        directionRandomizer?: number);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): CylinderParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "CylinderParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
    /**
     * Particle emitter emitting particles from the inside of a cylinder.
     * It emits the particles randomly between two vectors.
     */
    export class CylinderDirectedParticleEmitter extends CylinderParticleEmitter {
        /**
         * The min limit of the emission direction.
         */
        direction1: Vector3;
        /**
         * The max limit of the emission direction.
         */
        direction2: Vector3;
        /**
         * Creates a new instance CylinderDirectedParticleEmitter
         * @param radius the radius of the emission cylinder (1 by default)
         * @param height the height of the emission cylinder (1 by default)
         * @param radiusRange the range of the emission cylinder [0-1] 0 Surface only, 1 Entire Radius (1 by default)
         * @param direction1 the min limit of the emission direction (up vector by default)
         * @param direction2 the max limit of the emission direction (up vector by default)
         */
        constructor(radius?: number, height?: number, radiusRange?: number, 
        /**
         * The min limit of the emission direction.
         */
        direction1?: Vector3, 
        /**
         * The max limit of the emission direction.
         */
        direction2?: Vector3);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): CylinderDirectedParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "CylinderDirectedParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/hemisphericParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    /**
     * Particle emitter emitting particles from the inside of a hemisphere.
     * It emits the particles alongside the hemisphere radius. The emission direction might be randomized.
     */
    export class HemisphericParticleEmitter implements IParticleEmitterType {
        /**
         * The radius of the emission hemisphere.
         */
        radius: number;
        /**
         * The range of emission [0-1] 0 Surface only, 1 Entire Radius.
         */
        radiusRange: number;
        /**
         * How much to randomize the particle direction [0-1].
         */
        directionRandomizer: number;
        /**
        * Creates a new instance HemisphericParticleEmitter
        * @param radius the radius of the emission hemisphere (1 by default)
        * @param radiusRange the range of the emission hemisphere [0-1] 0 Surface only, 1 Entire Radius (1 by default)
        * @param directionRandomizer defines how much to randomize the particle direction [0-1]
        */
        constructor(
        /**
         * The radius of the emission hemisphere.
         */
        radius?: number, 
        /**
         * The range of emission [0-1] 0 Surface only, 1 Entire Radius.
         */
        radiusRange?: number, 
        /**
         * How much to randomize the particle direction [0-1].
         */
        directionRandomizer?: number);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): HemisphericParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "HemisphericParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/pointParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    /**
     * Particle emitter emitting particles from a point.
     * It emits the particles randomly between 2 given directions.
     */
    export class PointParticleEmitter implements IParticleEmitterType {
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         */
        direction1: Vector3;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         */
        direction2: Vector3;
        /**
         * Creates a new instance PointParticleEmitter
         */
        constructor();
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): PointParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "PointParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/sphereParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    /**
     * Particle emitter emitting particles from the inside of a sphere.
     * It emits the particles alongside the sphere radius. The emission direction might be randomized.
     */
    export class SphereParticleEmitter implements IParticleEmitterType {
        /**
         * The radius of the emission sphere.
         */
        radius: number;
        /**
         * The range of emission [0-1] 0 Surface only, 1 Entire Radius.
         */
        radiusRange: number;
        /**
         * How much to randomize the particle direction [0-1].
         */
        directionRandomizer: number;
        /**
        * Creates a new instance SphereParticleEmitter
        * @param radius the radius of the emission sphere (1 by default)
        * @param radiusRange the range of the emission sphere [0-1] 0 Surface only, 1 Entire Radius (1 by default)
        * @param directionRandomizer defines how much to randomize the particle direction [0-1]
        */
        constructor(
        /**
         * The radius of the emission sphere.
         */
        radius?: number, 
        /**
         * The range of emission [0-1] 0 Surface only, 1 Entire Radius.
         */
        radiusRange?: number, 
        /**
         * How much to randomize the particle direction [0-1].
         */
        directionRandomizer?: number);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): SphereParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "SphereParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
    /**
     * Particle emitter emitting particles from the inside of a sphere.
     * It emits the particles randomly between two vectors.
     */
    export class SphereDirectedParticleEmitter extends SphereParticleEmitter {
        /**
         * The min limit of the emission direction.
         */
        direction1: Vector3;
        /**
         * The max limit of the emission direction.
         */
        direction2: Vector3;
        /**
         * Creates a new instance SphereDirectedParticleEmitter
         * @param radius the radius of the emission sphere (1 by default)
         * @param direction1 the min limit of the emission direction (up vector by default)
         * @param direction2 the max limit of the emission direction (up vector by default)
         */
        constructor(radius?: number, 
        /**
         * The min limit of the emission direction.
         */
        direction1?: Vector3, 
        /**
         * The max limit of the emission direction.
         */
        direction2?: Vector3);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): SphereDirectedParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "SphereDirectedParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/customParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    import { Nullable } from "babylonjs/types";
    /**
     * Particle emitter emitting particles from a custom list of positions.
     */
    export class CustomParticleEmitter implements IParticleEmitterType {
        /**
         * Gets or sets the position generator that will create the inital position of each particle.
         * Index will be provided when used with GPU particle. Particle will be provided when used with CPU particles
         */
        particlePositionGenerator: (index: number, particle: Nullable<Particle>, outPosition: Vector3) => void;
        /**
         * Gets or sets the destination generator that will create the final destination of each particle.
         *  * Index will be provided when used with GPU particle. Particle will be provided when used with CPU particles
         */
        particleDestinationGenerator: (index: number, particle: Nullable<Particle>, outDestination: Vector3) => void;
        /**
         * Creates a new instance CustomParticleEmitter
         */
        constructor();
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): CustomParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "PointParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         */
        parse(serializationObject: any): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/meshParticleEmitter" {
    import { Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Effect } from "babylonjs/Materials/effect";
    import { Particle } from "babylonjs/Particles/particle";
    import { IParticleEmitterType } from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    /**
     * Particle emitter emitting particles from the inside of a box.
     * It emits the particles randomly between 2 given directions.
     */
    export class MeshParticleEmitter implements IParticleEmitterType {
        private _indices;
        private _positions;
        private _normals;
        private _storedNormal;
        private _mesh;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         */
        direction1: Vector3;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         */
        direction2: Vector3;
        /**
         * Gets or sets a boolean indicating that particle directions must be built from mesh face normals
         */
        useMeshNormalsForDirection: boolean;
        /** Defines the mesh to use as source */
        get mesh(): Nullable<AbstractMesh>;
        set mesh(value: Nullable<AbstractMesh>);
        /**
         * Creates a new instance MeshParticleEmitter
         * @param mesh defines the mesh to use as source
         */
        constructor(mesh?: Nullable<AbstractMesh>);
        /**
         * Called by the particle System when the direction is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param directionToUpdate is the direction vector to update with the result
         * @param particle is the particle we are computed the direction for
         * @param isLocal defines if the direction should be set in local space
         */
        startDirectionFunction(worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Called by the particle System when the position is computed for the created particle.
         * @param worldMatrix is the world matrix of the particle system
         * @param positionToUpdate is the position vector to update with the result
         * @param particle is the particle we are computed the position for
         * @param isLocal defines if the position should be set in local space
         */
        startPositionFunction(worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle, isLocal: boolean): void;
        /**
         * Clones the current emitter and returns a copy of it
         * @returns the new emitter
         */
        clone(): MeshParticleEmitter;
        /**
         * Called by the GPUParticleSystem to setup the update shader
         * @param effect defines the update shader
         */
        applyToShader(effect: Effect): void;
        /**
         * Returns a string to use to update the GPU particles update shader
         * @returns a string containng the defines string
         */
        getEffectDefines(): string;
        /**
         * Returns the string "BoxParticleEmitter"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parse properties from a JSON object
         * @param serializationObject defines the JSON object
         * @param scene defines the hosting scene
         */
        parse(serializationObject: any, scene: Nullable<Scene>): void;
    }
}
declare module "babylonjs/Particles/EmitterTypes/index" {
    export * from "babylonjs/Particles/EmitterTypes/boxParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/coneParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/cylinderParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/hemisphericParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/IParticleEmitterType";
    export * from "babylonjs/Particles/EmitterTypes/pointParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/sphereParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/customParticleEmitter";
    export * from "babylonjs/Particles/EmitterTypes/meshParticleEmitter";
}
declare module "babylonjs/Particles/IParticleSystem" {
    import { Nullable } from "babylonjs/types";
    import { Vector2, Vector3, Matrix } from "babylonjs/Maths/math.vector";
    import { Color3, Color4 } from "babylonjs/Maths/math.color";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { BoxParticleEmitter, IParticleEmitterType, PointParticleEmitter, HemisphericParticleEmitter, SphereParticleEmitter, SphereDirectedParticleEmitter, CylinderParticleEmitter, ConeParticleEmitter } from "babylonjs/Particles/EmitterTypes/index";
    import { Scene } from "babylonjs/scene";
    import { ColorGradient, FactorGradient, Color3Gradient } from "babylonjs/Misc/gradients";
    import { Effect } from "babylonjs/Materials/effect";
    import { Observable } from "babylonjs/Misc/observable";
    import { Animation } from "babylonjs/Animations/animation";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    /**
     * Interface representing a particle system in Babylon.js.
     * This groups the common functionalities that needs to be implemented in order to create a particle system.
     * A particle system represents a way to manage particles from their emission to their animation and rendering.
     */
    export interface IParticleSystem {
        /**
         * List of animations used by the particle system.
         */
        animations: Animation[];
        /**
         * The id of the Particle system.
         */
        id: string;
        /**
         * The name of the Particle system.
         */
        name: string;
        /**
         * The emitter represents the Mesh or position we are attaching the particle system to.
         */
        emitter: Nullable<AbstractMesh | Vector3>;
        /**
         * Gets or sets a boolean indicating if the particles must be rendered as billboard or aligned with the direction
         */
        isBillboardBased: boolean;
        /**
         * The rendering group used by the Particle system to chose when to render.
         */
        renderingGroupId: number;
        /**
         * The layer mask we are rendering the particles through.
         */
        layerMask: number;
        /**
        * The overall motion speed (0.01 is default update speed, faster updates = faster animation)
        */
        updateSpeed: number;
        /**
         * The amount of time the particle system is running (depends of the overall update speed).
         */
        targetStopDuration: number;
        /**
         * The texture used to render each particle. (this can be a spritesheet)
         */
        particleTexture: Nullable<BaseTexture>;
        /**
         * Blend mode use to render the particle, it can be either ParticleSystem.BLENDMODE_ONEONE, ParticleSystem.BLENDMODE_STANDARD or ParticleSystem.BLENDMODE_ADD.
         */
        blendMode: number;
        /**
         * Minimum life time of emitting particles.
         */
        minLifeTime: number;
        /**
         * Maximum life time of emitting particles.
         */
        maxLifeTime: number;
        /**
         * Minimum Size of emitting particles.
         */
        minSize: number;
        /**
         * Maximum Size of emitting particles.
         */
        maxSize: number;
        /**
         * Minimum scale of emitting particles on X axis.
         */
        minScaleX: number;
        /**
         * Maximum scale of emitting particles on X axis.
         */
        maxScaleX: number;
        /**
         * Minimum scale of emitting particles on Y axis.
         */
        minScaleY: number;
        /**
         * Maximum scale of emitting particles on Y axis.
         */
        maxScaleY: number;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color1: Color4;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color2: Color4;
        /**
         * Color the particle will have at the end of its lifetime.
         */
        colorDead: Color4;
        /**
         * The maximum number of particles to emit per frame until we reach the activeParticleCount value
         */
        emitRate: number;
        /**
         * You can use gravity if you want to give an orientation to your particles.
         */
        gravity: Vector3;
        /**
         * Minimum power of emitting particles.
         */
        minEmitPower: number;
        /**
         * Maximum power of emitting particles.
         */
        maxEmitPower: number;
        /**
         * Minimum angular speed of emitting particles (Z-axis rotation for each particle).
         */
        minAngularSpeed: number;
        /**
         * Maximum angular speed of emitting particles (Z-axis rotation for each particle).
         */
        maxAngularSpeed: number;
        /**
         * Gets or sets the minimal initial rotation in radians.
         */
        minInitialRotation: number;
        /**
         * Gets or sets the maximal initial rotation in radians.
         */
        maxInitialRotation: number;
        /**
         * The particle emitter type defines the emitter used by the particle system.
         * It can be for example box, sphere, or cone...
         */
        particleEmitterType: Nullable<IParticleEmitterType>;
        /**
         * Defines the delay in milliseconds before starting the system (0 by default)
         */
        startDelay: number;
        /**
         * Gets or sets a value indicating how many cycles (or frames) must be executed before first rendering (this value has to be set before starting the system). Default is 0
         */
        preWarmCycles: number;
        /**
         * Gets or sets a value indicating the time step multiplier to use in pre-warm mode (default is 1)
         */
        preWarmStepOffset: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) defines the speed of the sprite loop (default is 1 meaning the animation will play once during the entire particle lifetime)
         */
        spriteCellChangeSpeed: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) defines the first sprite cell to display
         */
        startSpriteCellID: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) defines the last sprite cell to display
         */
        endSpriteCellID: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines the sprite cell width to use
         */
        spriteCellWidth: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines the sprite cell height to use
         */
        spriteCellHeight: number;
        /**
         * This allows the system to random pick the start cell ID between startSpriteCellID and endSpriteCellID
         */
        spriteRandomStartCell: boolean;
        /**
         * Gets or sets a boolean indicating if a spritesheet is used to animate the particles texture
         */
        isAnimationSheetEnabled: boolean;
        /** Gets or sets a Vector2 used to move the pivot (by default (0,0)) */
        translationPivot: Vector2;
        /**
         * Gets or sets a texture used to add random noise to particle positions
         */
        noiseTexture: Nullable<BaseTexture>;
        /** Gets or sets the strength to apply to the noise value (default is (10, 10, 10)) */
        noiseStrength: Vector3;
        /**
         * Gets or sets the billboard mode to use when isBillboardBased = true.
         * Value can be: ParticleSystem.BILLBOARDMODE_ALL, ParticleSystem.BILLBOARDMODE_Y, ParticleSystem.BILLBOARDMODE_STRETCHED
         */
        billboardMode: number;
        /** Gets or sets a value indicating the damping to apply if the limit velocity factor is reached */
        limitVelocityDamping: number;
        /**
         * Gets or sets a boolean indicating that hosted animations (in the system.animations array) must be started when system.start() is called
         */
        beginAnimationOnStart: boolean;
        /**
         * Gets or sets the frame to start the animation from when beginAnimationOnStart is true
         */
        beginAnimationFrom: number;
        /**
         * Gets or sets the frame to end the animation on when beginAnimationOnStart is true
         */
        beginAnimationTo: number;
        /**
         * Gets or sets a boolean indicating if animations must loop when beginAnimationOnStart is true
         */
        beginAnimationLoop: boolean;
        /**
         * Specifies whether the particle system will be disposed once it reaches the end of the animation.
         */
        disposeOnStop: boolean;
        /**
         * Specifies if the particles are updated in emitter local space or world space
         */
        isLocal: boolean;
        /** Snippet ID if the particle system was created from the snippet server */
        snippetId: string;
        /** Gets or sets a matrix to use to compute projection */
        defaultProjectionMatrix: Matrix;
        /**
         * Gets the maximum number of particles active at the same time.
         * @returns The max number of active particles.
         */
        getCapacity(): number;
        /**
         * Gets the number of particles active at the same time.
         * @returns The number of active particles.
         */
        getActiveCount(): number;
        /**
         * Gets if the system has been started. (Note: this will still be true after stop is called)
         * @returns True if it has been started, otherwise false.
         */
        isStarted(): boolean;
        /**
         * Animates the particle system for this frame.
         */
        animate(): void;
        /**
         * Renders the particle system in its current state.
         * @returns the current number of particles
         */
        render(): number;
        /**
         * Dispose the particle system and frees its associated resources.
         * @param disposeTexture defines if the particule texture must be disposed as well (true by default)
         */
        dispose(disposeTexture?: boolean): void;
        /**
        * An event triggered when the system is disposed
        */
        onDisposeObservable: Observable<IParticleSystem>;
        /**
        * An event triggered when the system is stopped
        */
        onStoppedObservable: Observable<IParticleSystem>;
        /**
         * Clones the particle system.
         * @param name The name of the cloned object
         * @param newEmitter The new emitter to use
         * @returns the cloned particle system
         */
        clone(name: string, newEmitter: any): Nullable<IParticleSystem>;
        /**
         * Serializes the particle system to a JSON object
         * @param serializeTexture defines if the texture must be serialized as well
         * @returns the JSON object
         */
        serialize(serializeTexture: boolean): any;
        /**
         * Rebuild the particle system
         */
        rebuild(): void;
        /** Force the system to rebuild all gradients that need to be resync */
        forceRefreshGradients(): void;
        /**
         * Starts the particle system and begins to emit
         * @param delay defines the delay in milliseconds before starting the system (0 by default)
         */
        start(delay?: number): void;
        /**
         * Stops the particle system.
         */
        stop(): void;
        /**
         * Remove all active particles
         */
        reset(): void;
        /**
         * Gets a boolean indicating that the system is stopping
         * @returns true if the system is currently stopping
         */
        isStopping(): boolean;
        /**
         * Is this system ready to be used/rendered
         * @return true if the system is ready
         */
        isReady(): boolean;
        /**
         * Returns the string "ParticleSystem"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Gets the custom effect used to render the particles
         * @param blendMode Blend mode for which the effect should be retrieved
         * @returns The effect
         */
        getCustomEffect(blendMode: number): Nullable<Effect>;
        /**
         * Sets the custom effect used to render the particles
         * @param effect The effect to set
         * @param blendMode Blend mode for which the effect should be set
         */
        setCustomEffect(effect: Nullable<Effect>, blendMode: number): void;
        /**
         * Fill the defines array according to the current settings of the particle system
         * @param defines Array to be updated
         * @param blendMode blend mode to take into account when updating the array
         */
        fillDefines(defines: Array<string>, blendMode: number): void;
        /**
         * Fill the uniforms, attributes and samplers arrays according to the current settings of the particle system
         * @param uniforms Uniforms array to fill
         * @param attributes Attributes array to fill
         * @param samplers Samplers array to fill
         */
        fillUniformsAttributesAndSamplerNames(uniforms: Array<string>, attributes: Array<string>, samplers: Array<string>): void;
        /**
         * Observable that will be called just before the particles are drawn
         */
        onBeforeDrawParticlesObservable: Observable<Nullable<Effect>>;
        /**
         * Gets the name of the particle vertex shader
         */
        vertexShaderName: string;
        /**
         * Adds a new color gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param color1 defines the color to affect to the specified gradient
         * @param color2 defines an additional color used to define a range ([color, color2]) with main color to pick the final color from
         * @returns the current particle system
         */
        addColorGradient(gradient: number, color1: Color4, color2?: Color4): IParticleSystem;
        /**
         * Remove a specific color gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeColorGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new size gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the size factor to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addSizeGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific size gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeSizeGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of color gradients.
         * You must use addColorGradient and removeColorGradient to udpate this list
         * @returns the list of color gradients
         */
        getColorGradients(): Nullable<Array<ColorGradient>>;
        /**
         * Gets the current list of size gradients.
         * You must use addSizeGradient and removeSizeGradient to udpate this list
         * @returns the list of size gradients
         */
        getSizeGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of angular speed gradients.
         * You must use addAngularSpeedGradient and removeAngularSpeedGradient to udpate this list
         * @returns the list of angular speed gradients
         */
        getAngularSpeedGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new angular speed gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the angular speed to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addAngularSpeedGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific angular speed gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeAngularSpeedGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of velocity gradients.
         * You must use addVelocityGradient and removeVelocityGradient to udpate this list
         * @returns the list of velocity gradients
         */
        getVelocityGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new velocity gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the velocity to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addVelocityGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific velocity gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeVelocityGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of limit velocity gradients.
         * You must use addLimitVelocityGradient and removeLimitVelocityGradient to udpate this list
         * @returns the list of limit velocity gradients
         */
        getLimitVelocityGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new limit velocity gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the limit velocity to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addLimitVelocityGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific limit velocity gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeLimitVelocityGradient(gradient: number): IParticleSystem;
        /**
         * Adds a new drag gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the drag to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addDragGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific drag gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeDragGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of drag gradients.
         * You must use addDragGradient and removeDragGradient to udpate this list
         * @returns the list of drag gradients
         */
        getDragGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new emit rate gradient (please note that this will only work if you set the targetStopDuration property)
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the emit rate to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addEmitRateGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific emit rate gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeEmitRateGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of emit rate gradients.
         * You must use addEmitRateGradient and removeEmitRateGradient to udpate this list
         * @returns the list of emit rate gradients
         */
        getEmitRateGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new start size gradient (please note that this will only work if you set the targetStopDuration property)
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the start size to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addStartSizeGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific start size gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeStartSizeGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of start size gradients.
         * You must use addStartSizeGradient and removeStartSizeGradient to udpate this list
         * @returns the list of start size gradients
         */
        getStartSizeGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new life time gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param factor defines the life time factor to affect to the specified gradient
         * @param factor2 defines an additional factor used to define a range ([factor, factor2]) with main value to pick the final value from
         * @returns the current particle system
         */
        addLifeTimeGradient(gradient: number, factor: number, factor2?: number): IParticleSystem;
        /**
         * Remove a specific life time gradient
         * @param gradient defines the gradient to remove
         * @returns the current particle system
         */
        removeLifeTimeGradient(gradient: number): IParticleSystem;
        /**
         * Gets the current list of life time gradients.
         * You must use addLifeTimeGradient and removeLifeTimeGradient to udpate this list
         * @returns the list of life time gradients
         */
        getLifeTimeGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Gets the current list of color gradients.
         * You must use addColorGradient and removeColorGradient to udpate this list
         * @returns the list of color gradients
         */
        getColorGradients(): Nullable<Array<ColorGradient>>;
        /**
         * Adds a new ramp gradient used to remap particle colors
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param color defines the color to affect to the specified gradient
         * @returns the current particle system
         */
        addRampGradient(gradient: number, color: Color3): IParticleSystem;
        /**
         * Gets the current list of ramp gradients.
         * You must use addRampGradient and removeRampGradient to udpate this list
         * @returns the list of ramp gradients
         */
        getRampGradients(): Nullable<Array<Color3Gradient>>;
        /** Gets or sets a boolean indicating that ramp gradients must be used
         * @see https://doc.babylonjs.com/babylon101/particles#ramp-gradients
         */
        useRampGradients: boolean;
        /**
         * Adds a new color remap gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param min defines the color remap minimal range
         * @param max defines the color remap maximal range
         * @returns the current particle system
         */
        addColorRemapGradient(gradient: number, min: number, max: number): IParticleSystem;
        /**
         * Gets the current list of color remap gradients.
         * You must use addColorRemapGradient and removeColorRemapGradient to udpate this list
         * @returns the list of color remap gradients
         */
        getColorRemapGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Adds a new alpha remap gradient
         * @param gradient defines the gradient to use (between 0 and 1)
         * @param min defines the alpha remap minimal range
         * @param max defines the alpha remap maximal range
         * @returns the current particle system
         */
        addAlphaRemapGradient(gradient: number, min: number, max: number): IParticleSystem;
        /**
         * Gets the current list of alpha remap gradients.
         * You must use addAlphaRemapGradient and removeAlphaRemapGradient to udpate this list
         * @returns the list of alpha remap gradients
         */
        getAlphaRemapGradients(): Nullable<Array<FactorGradient>>;
        /**
         * Creates a Point Emitter for the particle system (emits directly from the emitter position)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the box
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the box
         * @returns the emitter
         */
        createPointEmitter(direction1: Vector3, direction2: Vector3): PointParticleEmitter;
        /**
         * Creates a Hemisphere Emitter for the particle system (emits along the hemisphere radius)
         * @param radius The radius of the hemisphere to emit from
         * @param radiusRange The range of the hemisphere to emit from [0-1] 0 Surface Only, 1 Entire Radius
         * @returns the emitter
         */
        createHemisphericEmitter(radius: number, radiusRange: number): HemisphericParticleEmitter;
        /**
         * Creates a Sphere Emitter for the particle system (emits along the sphere radius)
         * @param radius The radius of the sphere to emit from
         * @param radiusRange The range of the sphere to emit from [0-1] 0 Surface Only, 1 Entire Radius
         * @returns the emitter
         */
        createSphereEmitter(radius: number, radiusRange: number): SphereParticleEmitter;
        /**
         * Creates a Directed Sphere Emitter for the particle system (emits between direction1 and direction2)
         * @param radius The radius of the sphere to emit from
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the sphere
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the sphere
         * @returns the emitter
         */
        createDirectedSphereEmitter(radius: number, direction1: Vector3, direction2: Vector3): SphereDirectedParticleEmitter;
        /**
         * Creates a Cylinder Emitter for the particle system (emits from the cylinder to the particle position)
         * @param radius The radius of the emission cylinder
         * @param height The height of the emission cylinder
         * @param radiusRange The range of emission [0-1] 0 Surface only, 1 Entire Radius
         * @param directionRandomizer How much to randomize the particle direction [0-1]
         * @returns the emitter
         */
        createCylinderEmitter(radius: number, height: number, radiusRange: number, directionRandomizer: number): CylinderParticleEmitter;
        /**
         * Creates a Directed Cylinder Emitter for the particle system (emits between direction1 and direction2)
         * @param radius The radius of the cylinder to emit from
         * @param height The height of the emission cylinder
         * @param radiusRange the range of the emission cylinder [0-1] 0 Surface only, 1 Entire Radius (1 by default)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the cylinder
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the cylinder
         * @returns the emitter
         */
        createDirectedCylinderEmitter(radius: number, height: number, radiusRange: number, direction1: Vector3, direction2: Vector3): SphereDirectedParticleEmitter;
        /**
         * Creates a Cone Emitter for the particle system (emits from the cone to the particle position)
         * @param radius The radius of the cone to emit from
         * @param angle The base angle of the cone
         * @returns the emitter
         */
        createConeEmitter(radius: number, angle: number): ConeParticleEmitter;
        /**
         * Creates a Box Emitter for the particle system. (emits between direction1 and direction2 from withing the box defined by minEmitBox and maxEmitBox)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the box
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the box
         * @param minEmitBox Particles are emitted from the box between minEmitBox and maxEmitBox
         * @param maxEmitBox  Particles are emitted from the box between minEmitBox and maxEmitBox
         * @returns the emitter
         */
        createBoxEmitter(direction1: Vector3, direction2: Vector3, minEmitBox: Vector3, maxEmitBox: Vector3): BoxParticleEmitter;
        /**
         * Get hosting scene
         * @returns the scene
         */
        getScene(): Nullable<Scene>;
    }
}
declare module "babylonjs/Materials/Node/Blocks/colorSplitterBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    /**
     * Block used to expand a Color3/4 into 4 outputs (one for each component)
     */
    export class ColorSplitterBlock extends NodeMaterialBlock {
        /**
         * Create a new ColorSplitterBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the rgba component (input)
         */
        get rgba(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgb component (input)
         */
        get rgbIn(): NodeMaterialConnectionPoint;
        /**
         * Gets the rgb component (output)
         */
        get rgbOut(): NodeMaterialConnectionPoint;
        /**
         * Gets the r component (output)
         */
        get r(): NodeMaterialConnectionPoint;
        /**
         * Gets the g component (output)
         */
        get g(): NodeMaterialConnectionPoint;
        /**
         * Gets the b component (output)
         */
        get b(): NodeMaterialConnectionPoint;
        /**
         * Gets the a component (output)
         */
        get a(): NodeMaterialConnectionPoint;
        protected _inputRename(name: string): string;
        protected _outputRename(name: string): string;
        protected _buildBlock(state: NodeMaterialBuildState): this | undefined;
    }
}
declare module "babylonjs/Materials/Node/Blocks/trigonometryBlock" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { NodeMaterialBuildState } from "babylonjs/Materials/Node/nodeMaterialBuildState";
    import { NodeMaterialConnectionPoint } from "babylonjs/Materials/Node/nodeMaterialBlockConnectionPoint";
    import { Scene } from "babylonjs/scene";
    /**
     * Operations supported by the Trigonometry block
     */
    export enum TrigonometryBlockOperations {
        /** Cos */
        Cos = 0,
        /** Sin */
        Sin = 1,
        /** Abs */
        Abs = 2,
        /** Exp */
        Exp = 3,
        /** Exp2 */
        Exp2 = 4,
        /** Round */
        Round = 5,
        /** Floor */
        Floor = 6,
        /** Ceiling */
        Ceiling = 7,
        /** Square root */
        Sqrt = 8,
        /** Log */
        Log = 9,
        /** Tangent */
        Tan = 10,
        /** Arc tangent */
        ArcTan = 11,
        /** Arc cosinus */
        ArcCos = 12,
        /** Arc sinus */
        ArcSin = 13,
        /** Fraction */
        Fract = 14,
        /** Sign */
        Sign = 15,
        /** To radians (from degrees) */
        Radians = 16,
        /** To degrees (from radians) */
        Degrees = 17
    }
    /**
     * Block used to apply trigonometry operation to floats
     */
    export class TrigonometryBlock extends NodeMaterialBlock {
        /**
         * Gets or sets the operation applied by the block
         */
        operation: TrigonometryBlockOperations;
        /**
         * Creates a new TrigonometryBlock
         * @param name defines the block name
         */
        constructor(name: string);
        /**
         * Gets the current class name
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the input component
         */
        get input(): NodeMaterialConnectionPoint;
        /**
         * Gets the output component
         */
        get output(): NodeMaterialConnectionPoint;
        protected _buildBlock(state: NodeMaterialBuildState): this;
        serialize(): any;
        _deserialize(serializationObject: any, scene: Scene, rootUrl: string): void;
        protected _dumpPropertiesCode(): string;
    }
}
declare module "babylonjs/Materials/Node/nodeMaterial" {
    import { NodeMaterialBlock } from "babylonjs/Materials/Node/nodeMaterialBlock";
    import { PushMaterial } from "babylonjs/Materials/pushMaterial";
    import { Scene } from "babylonjs/scene";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { Engine } from "babylonjs/Engines/engine";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Observable } from "babylonjs/Misc/observable";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { MaterialDefines } from "babylonjs/Materials/materialDefines";
    import { NodeMaterialOptimizer } from "babylonjs/Materials/Node/Optimizers/nodeMaterialOptimizer";
    import { ImageProcessingConfiguration, IImageProcessingConfigurationDefines } from "babylonjs/Materials/imageProcessingConfiguration";
    import { Nullable } from "babylonjs/types";
    import { InputBlock } from "babylonjs/Materials/Node/Blocks/Input/inputBlock";
    import { TextureBlock } from "babylonjs/Materials/Node/Blocks/Dual/textureBlock";
    import { ReflectionTextureBaseBlock } from "babylonjs/Materials/Node/Blocks/Dual/reflectionTextureBaseBlock";
    import { RefractionBlock } from "babylonjs/Materials/Node/Blocks/PBR/refractionBlock";
    import { CurrentScreenBlock } from "babylonjs/Materials/Node/Blocks/Dual/currentScreenBlock";
    import { ParticleTextureBlock } from "babylonjs/Materials/Node/Blocks/Particle/particleTextureBlock";
    import { Effect } from "babylonjs/Materials/effect";
    import { PostProcess, PostProcessOptions } from "babylonjs/PostProcesses/postProcess";
    import { Camera } from "babylonjs/Cameras/camera";
    import { NodeMaterialModes } from "babylonjs/Materials/Node/Enums/nodeMaterialModes";
    import { IParticleSystem } from "babylonjs/Particles/IParticleSystem";
    import { ProceduralTexture } from "babylonjs/Materials/Textures/Procedurals/proceduralTexture";
    /**
     * Interface used to configure the node material editor
     */
    export interface INodeMaterialEditorOptions {
        /** Define the URl to load node editor script */
        editorURL?: string;
    }
    /** @hidden */
    export class NodeMaterialDefines extends MaterialDefines implements IImageProcessingConfigurationDefines {
        NORMAL: boolean;
        TANGENT: boolean;
        UV1: boolean;
        /** BONES */
        NUM_BONE_INFLUENCERS: number;
        BonesPerMesh: number;
        BONETEXTURE: boolean;
        /** MORPH TARGETS */
        MORPHTARGETS: boolean;
        MORPHTARGETS_NORMAL: boolean;
        MORPHTARGETS_TANGENT: boolean;
        MORPHTARGETS_UV: boolean;
        NUM_MORPH_INFLUENCERS: number;
        /** IMAGE PROCESSING */
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        TONEMAPPING_ACES: boolean;
        CONTRAST: boolean;
        EXPOSURE: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
        /** MISC. */
        BUMPDIRECTUV: number;
        constructor();
        setValue(name: string, value: any, markAsUnprocessedIfDirty?: boolean): void;
    }
    /**
     * Class used to configure NodeMaterial
     */
    export interface INodeMaterialOptions {
        /**
         * Defines if blocks should emit comments
         */
        emitComments: boolean;
    }
    /**
     * Class used to create a node based material built by assembling shader blocks
     */
    export class NodeMaterial extends PushMaterial {
        private static _BuildIdGenerator;
        private _options;
        private _vertexCompilationState;
        private _fragmentCompilationState;
        private _sharedData;
        private _buildId;
        private _buildWasSuccessful;
        private _cachedWorldViewMatrix;
        private _cachedWorldViewProjectionMatrix;
        private _optimizers;
        private _animationFrame;
        /** Define the Url to load node editor script */
        static EditorURL: string;
        /** Define the Url to load snippets */
        static SnippetUrl: string;
        /** Gets or sets a boolean indicating that node materials should not deserialize textures from json / snippet content */
        static IgnoreTexturesAtLoadTime: boolean;
        private BJSNODEMATERIALEDITOR;
        /** Get the inspector from bundle or global */
        private _getGlobalNodeMaterialEditor;
        /**
         * Snippet ID if the material was created from the snippet server
         */
        snippetId: string;
        /**
         * Gets or sets data used by visual editor
         * @see https://nme.babylonjs.com
         */
        editorData: any;
        /**
         * Gets or sets a boolean indicating that alpha value must be ignored (This will turn alpha blending off even if an alpha value is produced by the material)
         */
        ignoreAlpha: boolean;
        /**
        * Defines the maximum number of lights that can be used in the material
        */
        maxSimultaneousLights: number;
        /**
         * Observable raised when the material is built
         */
        onBuildObservable: Observable<NodeMaterial>;
        /**
         * Gets or sets the root nodes of the material vertex shader
         */
        _vertexOutputNodes: NodeMaterialBlock[];
        /**
         * Gets or sets the root nodes of the material fragment (pixel) shader
         */
        _fragmentOutputNodes: NodeMaterialBlock[];
        /** Gets or sets options to control the node material overall behavior */
        get options(): INodeMaterialOptions;
        set options(options: INodeMaterialOptions);
        /**
         * Default configuration related to image processing available in the standard Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Gets the image processing configuration used either in this material.
         */
        get imageProcessingConfiguration(): ImageProcessingConfiguration;
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        set imageProcessingConfiguration(value: ImageProcessingConfiguration);
        /**
         * Gets an array of blocks that needs to be serialized even if they are not yet connected
         */
        attachedBlocks: NodeMaterialBlock[];
        /**
         * Specifies the mode of the node material
         * @hidden
         */
        _mode: NodeMaterialModes;
        /**
         * Gets the mode property
         */
        get mode(): NodeMaterialModes;
        /**
         * A free comment about the material
         */
        comment: string;
        /**
         * Create a new node based material
         * @param name defines the material name
         * @param scene defines the hosting scene
         * @param options defines creation option
         */
        constructor(name: string, scene?: Scene, options?: Partial<INodeMaterialOptions>);
        /**
         * Gets the current class name of the material e.g. "NodeMaterial"
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the Standard Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>): void;
        /**
         * Get a block by its name
         * @param name defines the name of the block to retrieve
         * @returns the required block or null if not found
         */
        getBlockByName(name: string): Nullable<NodeMaterialBlock>;
        /**
         * Get a block by its name
         * @param predicate defines the predicate used to find the good candidate
         * @returns the required block or null if not found
         */
        getBlockByPredicate(predicate: (block: NodeMaterialBlock) => boolean): Nullable<NodeMaterialBlock>;
        /**
         * Get an input block by its name
         * @param predicate defines the predicate used to find the good candidate
         * @returns the required input block or null if not found
         */
        getInputBlockByPredicate(predicate: (block: InputBlock) => boolean): Nullable<InputBlock>;
        /**
         * Gets the list of input blocks attached to this material
         * @returns an array of InputBlocks
         */
        getInputBlocks(): InputBlock[];
        /**
         * Adds a new optimizer to the list of optimizers
         * @param optimizer defines the optimizers to add
         * @returns the current material
         */
        registerOptimizer(optimizer: NodeMaterialOptimizer): this | undefined;
        /**
         * Remove an optimizer from the list of optimizers
         * @param optimizer defines the optimizers to remove
         * @returns the current material
         */
        unregisterOptimizer(optimizer: NodeMaterialOptimizer): this | undefined;
        /**
         * Add a new block to the list of output nodes
         * @param node defines the node to add
         * @returns the current material
         */
        addOutputNode(node: NodeMaterialBlock): this;
        /**
         * Remove a block from the list of root nodes
         * @param node defines the node to remove
         * @returns the current material
         */
        removeOutputNode(node: NodeMaterialBlock): this;
        private _addVertexOutputNode;
        private _removeVertexOutputNode;
        private _addFragmentOutputNode;
        private _removeFragmentOutputNode;
        /**
         * Specifies if the material will require alpha blending
         * @returns a boolean specifying if alpha blending is needed
         */
        needAlphaBlending(): boolean;
        /**
         * Specifies if this material should be rendered in alpha test mode
         * @returns a boolean specifying if an alpha test is needed.
         */
        needAlphaTesting(): boolean;
        private _initializeBlock;
        private _resetDualBlocks;
        /**
         * Remove a block from the current node material
         * @param block defines the block to remove
         */
        removeBlock(block: NodeMaterialBlock): void;
        /**
         * Build the material and generates the inner effect
         * @param verbose defines if the build should log activity
         */
        build(verbose?: boolean): void;
        /**
         * Runs an otpimization phase to try to improve the shader code
         */
        optimize(): void;
        private _prepareDefinesForAttributes;
        /**
         * Create a post process from the material
         * @param camera The camera to apply the render pass to.
         * @param options The required width/height ratio to downsize to before computing the render pass. (Use 1.0 for full size)
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param textureFormat Format of textures used when performing the post process. (default: TEXTUREFORMAT_RGBA)
         * @returns the post process created
         */
        createPostProcess(camera: Nullable<Camera>, options?: number | PostProcessOptions, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, textureFormat?: number): Nullable<PostProcess>;
        /**
         * Create the post process effect from the material
         * @param postProcess The post process to create the effect for
         */
        createEffectForPostProcess(postProcess: PostProcess): void;
        private _createEffectForPostProcess;
        /**
         * Create a new procedural texture based on this node material
         * @param size defines the size of the texture
         * @param scene defines the hosting scene
         * @returns the new procedural texture attached to this node material
         */
        createProceduralTexture(size: number | {
            width: number;
            height: number;
            layers?: number;
        }, scene: Scene): Nullable<ProceduralTexture>;
        private _createEffectForParticles;
        private _checkInternals;
        /**
         * Create the effect to be used as the custom effect for a particle system
         * @param particleSystem Particle system to create the effect for
         * @param onCompiled defines a function to call when the effect creation is successful
         * @param onError defines a function to call when the effect creation has failed
         */
        createEffectForParticles(particleSystem: IParticleSystem, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): void;
        private _processDefines;
        /**
          * Get if the submesh is ready to be used and all its information available.
          * Child classes can use it to update shaders
          * @param mesh defines the mesh to check
          * @param subMesh defines which submesh to check
          * @param useInstances specifies that instances should be used
          * @returns a boolean indicating that the submesh is ready or not
          */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: SubMesh, useInstances?: boolean): boolean;
        /**
         * Get a string representing the shaders built by the current node graph
         */
        get compiledShaders(): string;
        /**
         * Binds the world matrix to the material
         * @param world defines the world transformation matrix
         */
        bindOnlyWorldMatrix(world: Matrix): void;
        /**
         * Binds the submesh to this material by preparing the effect and shader to draw
         * @param world defines the world transformation matrix
         * @param mesh defines the mesh containing the submesh
         * @param subMesh defines the submesh to bind the material to
         */
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        /**
         * Gets the active textures from the material
         * @returns an array of textures
         */
        getActiveTextures(): BaseTexture[];
        /**
         * Gets the list of texture blocks
         * @returns an array of texture blocks
         */
        getTextureBlocks(): (TextureBlock | ReflectionTextureBaseBlock | RefractionBlock | CurrentScreenBlock | ParticleTextureBlock)[];
        /**
         * Specifies if the material uses a texture
         * @param texture defines the texture to check against the material
         * @returns a boolean specifying if the material uses the texture
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Disposes the material
         * @param forceDisposeEffect specifies if effects should be forcefully disposed
         * @param forceDisposeTextures specifies if textures should be forcefully disposed
         * @param notBoundToMesh specifies if the material that is being disposed is known to be not bound to any mesh
         */
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean, notBoundToMesh?: boolean): void;
        /** Creates the node editor window. */
        private _createNodeEditor;
        /**
         * Launch the node material editor
         * @param config Define the configuration of the editor
         * @return a promise fulfilled when the node editor is visible
         */
        edit(config?: INodeMaterialEditorOptions): Promise<void>;
        /**
         * Clear the current material
         */
        clear(): void;
        /**
         * Clear the current material and set it to a default state
         */
        setToDefault(): void;
        /**
         * Clear the current material and set it to a default state for post process
         */
        setToDefaultPostProcess(): void;
        /**
         * Clear the current material and set it to a default state for procedural texture
         */
        setToDefaultProceduralTexture(): void;
        /**
         * Clear the current material and set it to a default state for particle
         */
        setToDefaultParticle(): void;
        /**
         * Loads the current Node Material from a url pointing to a file save by the Node Material Editor
         * @param url defines the url to load from
         * @returns a promise that will fullfil when the material is fully loaded
         */
        loadAsync(url: string): Promise<void>;
        private _gatherBlocks;
        /**
         * Generate a string containing the code declaration required to create an equivalent of this material
         * @returns a string
         */
        generateCode(): string;
        /**
         * Serializes this material in a JSON representation
         * @returns the serialized material object
         */
        serialize(selectedBlocks?: NodeMaterialBlock[]): any;
        private _restoreConnections;
        /**
         * Clear the current graph and load a new one from a serialization object
         * @param source defines the JSON representation of the material
         * @param rootUrl defines the root URL to use to load textures and relative dependencies
         * @param merge defines whether or not the source must be merged or replace the current content
         */
        loadFromSerialization(source: any, rootUrl?: string, merge?: boolean): void;
        /**
         * Makes a duplicate of the current material.
         * @param name - name to use for the new material.
         */
        clone(name: string): NodeMaterial;
        /**
         * Creates a node material from parsed material data
         * @param source defines the JSON representation of the material
         * @param scene defines the hosting scene
         * @param rootUrl defines the root URL to use to load textures and relative dependencies
         * @returns a new node material
         */
        static Parse(source: any, scene: Scene, rootUrl?: string): NodeMaterial;
        /**
         * Creates a node material from a snippet saved in a remote file
         * @param name defines the name of the material to create
         * @param url defines the url to load from
         * @param scene defines the hosting scene
         * @returns a promise that will resolve to the new node material
         */
        static ParseFromFileAsync(name: string, url: string, scene: Scene): Promise<NodeMaterial>;
        /**
         * Creates a node material from a snippet saved by the node material editor
         * @param snippetId defines the snippet to load
         * @param scene defines the hosting scene
         * @param rootUrl defines the root URL to use to load textures and relative dependencies
         * @param nodeMaterial defines a node material to update (instead of creating a new one)
         * @returns a promise that will resolve to the new node material
         */
        static ParseFromSnippetAsync(snippetId: string, scene: Scene, rootUrl?: string, nodeMaterial?: NodeMaterial): Promise<NodeMaterial>;
        /**
         * Creates a new node material set to default basic configuration
         * @param name defines the name of the material
         * @param scene defines the hosting scene
         * @returns a new NodeMaterial
         */
        static CreateDefault(name: string, scene?: Scene): NodeMaterial;
    }
}
declare module "babylonjs/Engines/Extensions/engine.multiRender" {
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { IMultiRenderTargetOptions } from "babylonjs/Materials/Textures/multiRenderTarget";
    import { Nullable } from "babylonjs/types";
    module "babylonjs/Engines/thinEngine" {
        interface ThinEngine {
            /**
             * Unbind a list of render target textures from the webGL context
             * This is used only when drawBuffer extension or webGL2 are active
             * @param textures defines the render target textures to unbind
             * @param disableGenerateMipMaps defines a boolean indicating that mipmaps must not be generated
             * @param onBeforeUnbind defines a function which will be called before the effective unbind
             */
            unBindMultiColorAttachmentFramebuffer(textures: InternalTexture[], disableGenerateMipMaps: boolean, onBeforeUnbind?: () => void): void;
            /**
             * Create a multi render target texture
             * @see https://doc.babylonjs.com/features/webgl2#multiple-render-target
             * @param size defines the size of the texture
             * @param options defines the creation options
             * @returns the cube texture as an InternalTexture
             */
            createMultipleRenderTarget(size: any, options: IMultiRenderTargetOptions): InternalTexture[];
            /**
             * Update the sample count for a given multiple render target texture
             * @see https://doc.babylonjs.com/features/webgl2#multisample-render-targets
             * @param textures defines the textures to update
             * @param samples defines the sample count to set
             * @returns the effective sample count (could be 0 if multisample render targets are not supported)
             */
            updateMultipleRenderTargetTextureSampleCount(textures: Nullable<InternalTexture[]>, samples: number): number;
            /**
             * Select a subsets of attachments to draw to.
             * @param attachments gl attachments
             */
            bindAttachments(attachments: number[]): void;
            /**
             * Creates a layout object to draw/clear on specific textures in a MRT
             * @param textureStatus textureStatus[i] indicates if the i-th is active
             * @returns A layout to be fed to the engine, calling `bindAttachments`.
             */
            buildTextureLayout(textureStatus: boolean[]): number[];
            /**
             * Restores the webgl state to only draw on the main color attachment
             */
            restoreSingleAttachment(): void;
        }
    }
}
declare module "babylonjs/Materials/Textures/multiRenderTarget" {
    import { Scene } from "babylonjs/scene";
    import { Engine } from "babylonjs/Engines/engine";
    import { Texture } from "babylonjs/Materials/Textures/texture";
    import { RenderTargetTexture } from "babylonjs/Materials/Textures/renderTargetTexture";
    import "babylonjs/Engines/Extensions/engine.multiRender";
    /**
     * Creation options of the multi render target texture.
     */
    export interface IMultiRenderTargetOptions {
        /**
         * Define if the texture needs to create mip maps after render.
         */
        generateMipMaps?: boolean;
        /**
         * Define the types of all the draw buffers we want to create
         */
        types?: number[];
        /**
         * Define the sampling modes of all the draw buffers we want to create
         */
        samplingModes?: number[];
        /**
         * Define if a depth buffer is required
         */
        generateDepthBuffer?: boolean;
        /**
         * Define if a stencil buffer is required
         */
        generateStencilBuffer?: boolean;
        /**
         * Define if a depth texture is required instead of a depth buffer
         */
        generateDepthTexture?: boolean;
        /**
         * Define the number of desired draw buffers
         */
        textureCount?: number;
        /**
         * Define if aspect ratio should be adapted to the texture or stay the scene one
         */
        doNotChangeAspectRatio?: boolean;
        /**
         * Define the default type of the buffers we are creating
         */
        defaultType?: number;
    }
    /**
     * A multi render target, like a render target provides the ability to render to a texture.
     * Unlike the render target, it can render to several draw buffers in one draw.
     * This is specially interesting in deferred rendering or for any effects requiring more than
     * just one color from a single pass.
     */
    export class MultiRenderTarget extends RenderTargetTexture {
        private _internalTextures;
        private _textures;
        private _multiRenderTargetOptions;
        private _count;
        /**
         * Get if draw buffers are currently supported by the used hardware and browser.
         */
        get isSupported(): boolean;
        /**
         * Get the list of textures generated by the multi render target.
         */
        get textures(): Texture[];
        /**
         * Gets the number of textures in this MRT. This number can be different from `_textures.length` in case a depth texture is generated.
         */
        get count(): number;
        /**
         * Get the depth texture generated by the multi render target if options.generateDepthTexture has been set
         */
        get depthTexture(): Texture;
        /**
         * Set the wrapping mode on U of all the textures we are rendering to.
         * Can be any of the Texture. (CLAMP_ADDRESSMODE, MIRROR_ADDRESSMODE or WRAP_ADDRESSMODE)
         */
        set wrapU(wrap: number);
        /**
         * Set the wrapping mode on V of all the textures we are rendering to.
         * Can be any of the Texture. (CLAMP_ADDRESSMODE, MIRROR_ADDRESSMODE or WRAP_ADDRESSMODE)
         */
        set wrapV(wrap: number);
        /**
         * Instantiate a new multi render target texture.
         * A multi render target, like a render target provides the ability to render to a texture.
         * Unlike the render target, it can render to several draw buffers in one draw.
         * This is specially interesting in deferred rendering or for any effects requiring more than
         * just one color from a single pass.
         * @param name Define the name of the texture
         * @param size Define the size of the buffers to render to
         * @param count Define the number of target we are rendering into
         * @param scene Define the scene the texture belongs to
         * @param options Define the options used to create the multi render target
         */
        constructor(name: string, size: any, count: number, scene: Scene, options?: IMultiRenderTargetOptions);
        private _initTypes;
        /** @hidden */
        _rebuild(forceFullRebuild?: boolean): void;
        private _createInternalTextures;
        private _createTextures;
        /**
         * Replaces a texture within the MRT.
         * @param texture The new texture to insert in the MRT
         * @param index The index of the texture to replace
         */
        replaceTexture(texture: Texture, index: number): void;
        /**
         * Define the number of samples used if MSAA is enabled.
         */
        get samples(): number;
        set samples(value: number);
        /**
         * Resize all the textures in the multi render target.
         * Be careful as it will recreate all the data in the new texture.
         * @param size Define the new size
         */
        resize(size: any): void;
        /**
         * Changes the number of render targets in this MRT
         * Be careful as it will recreate all the data in the new texture.
         * @param count new texture count
         * @param options Specifies texture types and sampling modes for new textures
         */
        updateCount(count: number, options?: IMultiRenderTargetOptions): void;
        protected unbindFrameBuffer(engine: Engine, faceIndex: number): void;
        /**
         * Dispose the render targets and their associated resources
         */
        dispose(): void;
        /**
         * Release all the underlying texture used as draw buffers.
         */
        releaseInternalTextures(): void;
    }
}
declare module "babylonjs/Shaders/imageProcessing.fragment" {
    import "babylonjs/Shaders/ShadersInclude/imageProcessingDeclaration";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingFunctions";
    /** @hidden */
    export var imageProcessingPixelShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/PostProcesses/imageProcessingPostProcess" {
    import { Nullable } from "babylonjs/types";
    import { Color4 } from "babylonjs/Maths/math.color";
    import { Camera } from "babylonjs/Cameras/camera";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { ColorCurves } from "babylonjs/Materials/colorCurves";
    import { ImageProcessingConfiguration } from "babylonjs/Materials/imageProcessingConfiguration";
    import { PostProcess, PostProcessOptions } from "babylonjs/PostProcesses/postProcess";
    import { Engine } from "babylonjs/Engines/engine";
    import "babylonjs/Shaders/imageProcessing.fragment";
    import "babylonjs/Shaders/postprocess.vertex";
    /**
     * ImageProcessingPostProcess
     * @see https://doc.babylonjs.com/how_to/how_to_use_postprocesses#imageprocessing
     */
    export class ImageProcessingPostProcess extends PostProcess {
        /**
         * Default configuration related to image processing available in the PBR Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Gets the image processing configuration used either in this material.
         */
        get imageProcessingConfiguration(): ImageProcessingConfiguration;
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        set imageProcessingConfiguration(value: ImageProcessingConfiguration);
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the PBR Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>, doNotBuild?: boolean): void;
        /**
         * If the post process is supported.
         */
        get isSupported(): boolean;
        /**
         * Gets Color curves setup used in the effect if colorCurvesEnabled is set to true .
         */
        get colorCurves(): Nullable<ColorCurves>;
        /**
         * Sets Color curves setup used in the effect if colorCurvesEnabled is set to true .
         */
        set colorCurves(value: Nullable<ColorCurves>);
        /**
         * Gets wether the color curves effect is enabled.
         */
        get colorCurvesEnabled(): boolean;
        /**
         * Sets wether the color curves effect is enabled.
         */
        set colorCurvesEnabled(value: boolean);
        /**
         * Gets Color grading LUT texture used in the effect if colorGradingEnabled is set to true.
         */
        get colorGradingTexture(): Nullable<BaseTexture>;
        /**
         * Sets Color grading LUT texture used in the effect if colorGradingEnabled is set to true.
         */
        set colorGradingTexture(value: Nullable<BaseTexture>);
        /**
         * Gets wether the color grading effect is enabled.
         */
        get colorGradingEnabled(): boolean;
        /**
         * Gets wether the color grading effect is enabled.
         */
        set colorGradingEnabled(value: boolean);
        /**
         * Gets exposure used in the effect.
         */
        get exposure(): number;
        /**
         * Sets exposure used in the effect.
         */
        set exposure(value: number);
        /**
         * Gets wether tonemapping is enabled or not.
         */
        get toneMappingEnabled(): boolean;
        /**
         * Sets wether tonemapping is enabled or not
         */
        set toneMappingEnabled(value: boolean);
        /**
         * Gets the type of tone mapping effect.
         */
        get toneMappingType(): number;
        /**
         * Sets the type of tone mapping effect.
         */
        set toneMappingType(value: number);
        /**
         * Gets contrast used in the effect.
         */
        get contrast(): number;
        /**
         * Sets contrast used in the effect.
         */
        set contrast(value: number);
        /**
         * Gets Vignette stretch size.
         */
        get vignetteStretch(): number;
        /**
         * Sets Vignette stretch size.
         */
        set vignetteStretch(value: number);
        /**
         * Gets Vignette centre X Offset.
         */
        get vignetteCentreX(): number;
        /**
         * Sets Vignette centre X Offset.
         */
        set vignetteCentreX(value: number);
        /**
         * Gets Vignette centre Y Offset.
         */
        get vignetteCentreY(): number;
        /**
         * Sets Vignette centre Y Offset.
         */
        set vignetteCentreY(value: number);
        /**
         * Gets Vignette weight or intensity of the vignette effect.
         */
        get vignetteWeight(): number;
        /**
         * Sets Vignette weight or intensity of the vignette effect.
         */
        set vignetteWeight(value: number);
        /**
         * Gets Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        get vignetteColor(): Color4;
        /**
         * Sets Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        set vignetteColor(value: Color4);
        /**
         * Gets Camera field of view used by the Vignette effect.
         */
        get vignetteCameraFov(): number;
        /**
         * Sets Camera field of view used by the Vignette effect.
         */
        set vignetteCameraFov(value: number);
        /**
         * Gets the vignette blend mode allowing different kind of effect.
         */
        get vignetteBlendMode(): number;
        /**
         * Sets the vignette blend mode allowing different kind of effect.
         */
        set vignetteBlendMode(value: number);
        /**
         * Gets wether the vignette effect is enabled.
         */
        get vignetteEnabled(): boolean;
        /**
         * Sets wether the vignette effect is enabled.
         */
        set vignetteEnabled(value: boolean);
        private _fromLinearSpace;
        /**
         * Gets wether the input of the processing is in Gamma or Linear Space.
         */
        get fromLinearSpace(): boolean;
        /**
         * Sets wether the input of the processing is in Gamma or Linear Space.
         */
        set fromLinearSpace(value: boolean);
        /**
         * Defines cache preventing GC.
         */
        private _defines;
        constructor(name: string, options: number | PostProcessOptions, camera?: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, imageProcessingConfiguration?: ImageProcessingConfiguration);
        /**
         *  "ImageProcessingPostProcess"
         * @returns "ImageProcessingPostProcess"
         */
        getClassName(): string;
        /**
         * @hidden
         */
        _updateParameters(): void;
        dispose(camera?: Camera): void;
    }
}
declare module "babylonjs/Rendering/prePassEffectConfiguration" {
    import { PostProcess } from "babylonjs/PostProcesses/postProcess";
    /**
     * Interface for defining prepass effects in the prepass post-process pipeline
     */
    export interface PrePassEffectConfiguration {
        /**
         * Name of the effect
         */
        name: string;
        /**
         * Post process to attach for this effect
         */
        postProcess?: PostProcess;
        /**
         * Textures required in the MRT
         */
        texturesRequired: number[];
        /**
         * Is the effect enabled
         */
        enabled: boolean;
        /**
         * Disposes the effect configuration
         */
        dispose?: () => void;
        /**
         * Creates the associated post process
         */
        createPostProcess?: () => PostProcess;
    }
}
declare module "babylonjs/Materials/fresnelParameters" {
    import { DeepImmutable } from "babylonjs/types";
    import { Color3 } from "babylonjs/Maths/math.color";
    /**
     * Options to be used when creating a FresnelParameters.
     */
    export type IFresnelParametersCreationOptions = {
        /**
         * Define the color used on edges (grazing angle)
         */
        leftColor?: Color3;
        /**
         * Define the color used on center
         */
        rightColor?: Color3;
        /**
         * Define bias applied to computed fresnel term
         */
        bias?: number;
        /**
         * Defined the power exponent applied to fresnel term
         */
        power?: number;
        /**
         * Define if the fresnel effect is enable or not.
         */
        isEnabled?: boolean;
    };
    /**
     * Serialized format for FresnelParameters.
     */
    export type IFresnelParametersSerialized = {
        /**
         * Define the color used on edges (grazing angle) [as an array]
         */
        leftColor: number[];
        /**
         * Define the color used on center [as an array]
         */
        rightColor: number[];
        /**
         * Define bias applied to computed fresnel term
         */
        bias: number;
        /**
         * Defined the power exponent applied to fresnel term
         */
        power?: number;
        /**
         * Define if the fresnel effect is enable or not.
         */
        isEnabled: boolean;
    };
    /**
     * This represents all the required information to add a fresnel effect on a material:
     * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
     */
    export class FresnelParameters {
        private _isEnabled;
        /**
         * Define if the fresnel effect is enable or not.
         */
        get isEnabled(): boolean;
        set isEnabled(value: boolean);
        /**
         * Define the color used on edges (grazing angle)
         */
        leftColor: Color3;
        /**
         * Define the color used on center
         */
        rightColor: Color3;
        /**
         * Define bias applied to computed fresnel term
         */
        bias: number;
        /**
         * Defined the power exponent applied to fresnel term
         */
        power: number;
        /**
         * Creates a new FresnelParameters object.
         *
         * @param options provide your own settings to optionally to override defaults
         */
        constructor(options?: IFresnelParametersCreationOptions);
        /**
         * Clones the current fresnel and its valuues
         * @returns a clone fresnel configuration
         */
        clone(): FresnelParameters;
        /**
         * Determines equality between FresnelParameters objects
         * @param otherFresnelParameters defines the second operand
         * @returns true if the power, bias, leftColor, rightColor and isEnabled values are equal to the given ones
         */
        equals(otherFresnelParameters: DeepImmutable<FresnelParameters>): boolean;
        /**
         * Serializes the current fresnel parameters to a JSON representation.
         * @return the JSON serialization
         */
        serialize(): IFresnelParametersSerialized;
        /**
         * Parse a JSON object and deserialize it to a new Fresnel parameter object.
         * @param parsedFresnelParameters Define the JSON representation
         * @returns the parsed parameters
         */
        static Parse(parsedFresnelParameters: IFresnelParametersSerialized): FresnelParameters;
    }
}
declare module "babylonjs/Materials/materialFlags" {
    /**
     * This groups all the flags used to control the materials channel.
     */
    export class MaterialFlags {
        private static _DiffuseTextureEnabled;
        /**
         * Are diffuse textures enabled in the application.
         */
        static get DiffuseTextureEnabled(): boolean;
        static set DiffuseTextureEnabled(value: boolean);
        private static _DetailTextureEnabled;
        /**
         * Are detail textures enabled in the application.
         */
        static get DetailTextureEnabled(): boolean;
        static set DetailTextureEnabled(value: boolean);
        private static _AmbientTextureEnabled;
        /**
         * Are ambient textures enabled in the application.
         */
        static get AmbientTextureEnabled(): boolean;
        static set AmbientTextureEnabled(value: boolean);
        private static _OpacityTextureEnabled;
        /**
         * Are opacity textures enabled in the application.
         */
        static get OpacityTextureEnabled(): boolean;
        static set OpacityTextureEnabled(value: boolean);
        private static _ReflectionTextureEnabled;
        /**
         * Are reflection textures enabled in the application.
         */
        static get ReflectionTextureEnabled(): boolean;
        static set ReflectionTextureEnabled(value: boolean);
        private static _EmissiveTextureEnabled;
        /**
         * Are emissive textures enabled in the application.
         */
        static get EmissiveTextureEnabled(): boolean;
        static set EmissiveTextureEnabled(value: boolean);
        private static _SpecularTextureEnabled;
        /**
         * Are specular textures enabled in the application.
         */
        static get SpecularTextureEnabled(): boolean;
        static set SpecularTextureEnabled(value: boolean);
        private static _BumpTextureEnabled;
        /**
         * Are bump textures enabled in the application.
         */
        static get BumpTextureEnabled(): boolean;
        static set BumpTextureEnabled(value: boolean);
        private static _LightmapTextureEnabled;
        /**
         * Are lightmap textures enabled in the application.
         */
        static get LightmapTextureEnabled(): boolean;
        static set LightmapTextureEnabled(value: boolean);
        private static _RefractionTextureEnabled;
        /**
         * Are refraction textures enabled in the application.
         */
        static get RefractionTextureEnabled(): boolean;
        static set RefractionTextureEnabled(value: boolean);
        private static _ColorGradingTextureEnabled;
        /**
         * Are color grading textures enabled in the application.
         */
        static get ColorGradingTextureEnabled(): boolean;
        static set ColorGradingTextureEnabled(value: boolean);
        private static _FresnelEnabled;
        /**
         * Are fresnels enabled in the application.
         */
        static get FresnelEnabled(): boolean;
        static set FresnelEnabled(value: boolean);
        private static _ClearCoatTextureEnabled;
        /**
         * Are clear coat textures enabled in the application.
         */
        static get ClearCoatTextureEnabled(): boolean;
        static set ClearCoatTextureEnabled(value: boolean);
        private static _ClearCoatBumpTextureEnabled;
        /**
         * Are clear coat bump textures enabled in the application.
         */
        static get ClearCoatBumpTextureEnabled(): boolean;
        static set ClearCoatBumpTextureEnabled(value: boolean);
        private static _ClearCoatTintTextureEnabled;
        /**
         * Are clear coat tint textures enabled in the application.
         */
        static get ClearCoatTintTextureEnabled(): boolean;
        static set ClearCoatTintTextureEnabled(value: boolean);
        private static _SheenTextureEnabled;
        /**
         * Are sheen textures enabled in the application.
         */
        static get SheenTextureEnabled(): boolean;
        static set SheenTextureEnabled(value: boolean);
        private static _AnisotropicTextureEnabled;
        /**
         * Are anisotropic textures enabled in the application.
         */
        static get AnisotropicTextureEnabled(): boolean;
        static set AnisotropicTextureEnabled(value: boolean);
        private static _ThicknessTextureEnabled;
        /**
         * Are thickness textures enabled in the application.
         */
        static get ThicknessTextureEnabled(): boolean;
        static set ThicknessTextureEnabled(value: boolean);
    }
}
declare module "babylonjs/Shaders/ShadersInclude/defaultFragmentDeclaration" {
    /** @hidden */
    export var defaultFragmentDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/defaultUboDeclaration" {
    /** @hidden */
    export var defaultUboDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/prePassDeclaration" {
    /** @hidden */
    export var prePassDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/lightFragmentDeclaration" {
    /** @hidden */
    export var lightFragmentDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/lightUboDeclaration" {
    /** @hidden */
    export var lightUboDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/lightsFragmentFunctions" {
    /** @hidden */
    export var lightsFragmentFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/shadowsFragmentFunctions" {
    /** @hidden */
    export var shadowsFragmentFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/fresnelFunction" {
    /** @hidden */
    export var fresnelFunction: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bumpFragmentMainFunctions" {
    /** @hidden */
    export var bumpFragmentMainFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bumpFragmentFunctions" {
    /** @hidden */
    export var bumpFragmentFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/logDepthDeclaration" {
    /** @hidden */
    export var logDepthDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/fogFragmentDeclaration" {
    /** @hidden */
    export var fogFragmentDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bumpFragment" {
    /** @hidden */
    export var bumpFragment: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/depthPrePass" {
    /** @hidden */
    export var depthPrePass: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/lightFragment" {
    /** @hidden */
    export var lightFragment: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/logDepthFragment" {
    /** @hidden */
    export var logDepthFragment: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/fogFragment" {
    /** @hidden */
    export var fogFragment: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/default.fragment" {
    import "babylonjs/Shaders/ShadersInclude/defaultFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/defaultUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/prePassDeclaration";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    import "babylonjs/Shaders/ShadersInclude/lightFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightsFragmentFunctions";
    import "babylonjs/Shaders/ShadersInclude/shadowsFragmentFunctions";
    import "babylonjs/Shaders/ShadersInclude/fresnelFunction";
    import "babylonjs/Shaders/ShadersInclude/reflectionFunction";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingDeclaration";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpFragmentMainFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpFragmentFunctions";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/logDepthDeclaration";
    import "babylonjs/Shaders/ShadersInclude/fogFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneFragment";
    import "babylonjs/Shaders/ShadersInclude/bumpFragment";
    import "babylonjs/Shaders/ShadersInclude/depthPrePass";
    import "babylonjs/Shaders/ShadersInclude/lightFragment";
    import "babylonjs/Shaders/ShadersInclude/logDepthFragment";
    import "babylonjs/Shaders/ShadersInclude/fogFragment";
    /** @hidden */
    export var defaultPixelShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/defaultVertexDeclaration" {
    /** @hidden */
    export var defaultVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bonesDeclaration" {
    /** @hidden */
    export var bonesDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/instancesDeclaration" {
    /** @hidden */
    export var instancesDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/prePassVertexDeclaration" {
    /** @hidden */
    export var prePassVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bumpVertexDeclaration" {
    /** @hidden */
    export var bumpVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/fogVertexDeclaration" {
    /** @hidden */
    export var fogVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/morphTargetsVertexGlobalDeclaration" {
    /** @hidden */
    export var morphTargetsVertexGlobalDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/morphTargetsVertexDeclaration" {
    /** @hidden */
    export var morphTargetsVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/morphTargetsVertex" {
    /** @hidden */
    export var morphTargetsVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/instancesVertex" {
    /** @hidden */
    export var instancesVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bonesVertex" {
    /** @hidden */
    export var bonesVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/prePassVertex" {
    /** @hidden */
    export var prePassVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/bumpVertex" {
    /** @hidden */
    export var bumpVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/fogVertex" {
    /** @hidden */
    export var fogVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/shadowsVertex" {
    /** @hidden */
    export var shadowsVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pointCloudVertex" {
    /** @hidden */
    export var pointCloudVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/logDepthVertex" {
    /** @hidden */
    export var logDepthVertex: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/default.vertex" {
    import "babylonjs/Shaders/ShadersInclude/defaultVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/defaultUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    import "babylonjs/Shaders/ShadersInclude/bonesDeclaration";
    import "babylonjs/Shaders/ShadersInclude/instancesDeclaration";
    import "babylonjs/Shaders/ShadersInclude/prePassVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/bumpVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/fogVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertexGlobalDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/logDepthDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertex";
    import "babylonjs/Shaders/ShadersInclude/instancesVertex";
    import "babylonjs/Shaders/ShadersInclude/bonesVertex";
    import "babylonjs/Shaders/ShadersInclude/prePassVertex";
    import "babylonjs/Shaders/ShadersInclude/bumpVertex";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneVertex";
    import "babylonjs/Shaders/ShadersInclude/fogVertex";
    import "babylonjs/Shaders/ShadersInclude/shadowsVertex";
    import "babylonjs/Shaders/ShadersInclude/pointCloudVertex";
    import "babylonjs/Shaders/ShadersInclude/logDepthVertex";
    /** @hidden */
    export var defaultVertexShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Materials/material.detailMapConfiguration" {
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { UniformBuffer } from "babylonjs/Materials/uniformBuffer";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    /**
     * @hidden
     */
    export interface IMaterialDetailMapDefines {
        DETAIL: boolean;
        DETAILDIRECTUV: number;
        DETAIL_NORMALBLENDMETHOD: number;
        /** @hidden */
        _areTexturesDirty: boolean;
    }
    /**
     * Define the code related to the detail map parameters of a material
     *
     * Inspired from:
     *   Unity: https://docs.unity3d.com/Packages/com.unity.render-pipelines.high-definition@9.0/manual/Mask-Map-and-Detail-Map.html and https://docs.unity3d.com/Manual/StandardShaderMaterialParameterDetail.html
     *   Unreal: https://docs.unrealengine.com/en-US/Engine/Rendering/Materials/HowTo/DetailTexturing/index.html
     *   Cryengine: https://docs.cryengine.com/display/SDKDOC2/Detail+Maps
     */
    export class DetailMapConfiguration {
        private _texture;
        /**
         * The detail texture of the material.
         */
        texture: Nullable<BaseTexture>;
        /**
         * Defines how strongly the detail diffuse/albedo channel is blended with the regular diffuse/albedo texture
         * Bigger values mean stronger blending
         */
        diffuseBlendLevel: number;
        /**
         * Defines how strongly the detail roughness channel is blended with the regular roughness value
         * Bigger values mean stronger blending. Only used with PBR materials
         */
        roughnessBlendLevel: number;
        /**
         * Defines how strong the bump effect from the detail map is
         * Bigger values mean stronger effect
         */
        bumpLevel: number;
        private _normalBlendMethod;
        /**
         * The method used to blend the bump and detail normals together
         */
        normalBlendMethod: number;
        private _isEnabled;
        /**
         * Enable or disable the detail map on this material
         */
        isEnabled: boolean;
        /** @hidden */
        private _internalMarkAllSubMeshesAsTexturesDirty;
        /** @hidden */
        _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Instantiate a new detail map
         * @param markAllSubMeshesAsTexturesDirty Callback to flag the material to dirty
         */
        constructor(markAllSubMeshesAsTexturesDirty: () => void);
        /**
         * Gets whether the submesh is ready to be used or not.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(defines: IMaterialDetailMapDefines, scene: Scene): boolean;
        /**
         * Update the defines for detail map usage
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         */
        prepareDefines(defines: IMaterialDetailMapDefines, scene: Scene): void;
        /**
         * Binds the material data.
         * @param uniformBuffer defines the Uniform buffer to fill in.
         * @param scene defines the scene the material belongs to.
         * @param isFrozen defines whether the material is frozen or not.
         */
        bindForSubMesh(uniformBuffer: UniformBuffer, scene: Scene, isFrozen: boolean): void;
        /**
         * Checks to see if a texture is used in the material.
         * @param texture - Base texture to use.
         * @returns - Boolean specifying if a texture is used in the material.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Returns an array of the actively used textures.
         * @param activeTextures Array of BaseTextures
         */
        getActiveTextures(activeTextures: BaseTexture[]): void;
        /**
         * Returns the animatable textures.
         * @param animatables Array of animatable textures.
         */
        getAnimatables(animatables: IAnimatable[]): void;
        /**
         * Disposes the resources of the material.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
        * Get the current class name useful for serialization or dynamic coding.
        * @returns "DetailMap"
        */
        getClassName(): string;
        /**
         * Add the required uniforms to the current list.
         * @param uniforms defines the current uniform list.
         */
        static AddUniforms(uniforms: string[]): void;
        /**
         * Add the required samplers to the current list.
         * @param samplers defines the current sampler list.
         */
        static AddSamplers(samplers: string[]): void;
        /**
         * Add the required uniforms to the current buffer.
         * @param uniformBuffer defines the current uniform buffer.
         */
        static PrepareUniformBuffer(uniformBuffer: UniformBuffer): void;
        /**
         * Makes a duplicate of the current instance into another one.
         * @param detailMap define the instance where to copy the info
         */
        copyTo(detailMap: DetailMapConfiguration): void;
        /**
         * Serializes this detail map instance
         * @returns - An object with the serialized instance.
         */
        serialize(): any;
        /**
         * Parses a detail map setting from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/standardMaterial" {
    import { SmartArray } from "babylonjs/Misc/smartArray";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { PrePassConfiguration } from "babylonjs/Materials/prePassConfiguration";
    import { ImageProcessingConfiguration, IImageProcessingConfigurationDefines } from "babylonjs/Materials/imageProcessingConfiguration";
    import { ColorCurves } from "babylonjs/Materials/colorCurves";
    import { FresnelParameters } from "babylonjs/Materials/fresnelParameters";
    import { MaterialDefines } from "babylonjs/Materials/materialDefines";
    import { PushMaterial } from "babylonjs/Materials/pushMaterial";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { RenderTargetTexture } from "babylonjs/Materials/Textures/renderTargetTexture";
    import "babylonjs/Shaders/default.fragment";
    import "babylonjs/Shaders/default.vertex";
    import { IMaterialDetailMapDefines, DetailMapConfiguration } from "babylonjs/Materials/material.detailMapConfiguration";
    /** @hidden */
    export class StandardMaterialDefines extends MaterialDefines implements IImageProcessingConfigurationDefines, IMaterialDetailMapDefines {
        MAINUV1: boolean;
        MAINUV2: boolean;
        DIFFUSE: boolean;
        DIFFUSEDIRECTUV: number;
        DETAIL: boolean;
        DETAILDIRECTUV: number;
        DETAIL_NORMALBLENDMETHOD: number;
        AMBIENT: boolean;
        AMBIENTDIRECTUV: number;
        OPACITY: boolean;
        OPACITYDIRECTUV: number;
        OPACITYRGB: boolean;
        REFLECTION: boolean;
        EMISSIVE: boolean;
        EMISSIVEDIRECTUV: number;
        SPECULAR: boolean;
        SPECULARDIRECTUV: number;
        BUMP: boolean;
        BUMPDIRECTUV: number;
        PARALLAX: boolean;
        PARALLAXOCCLUSION: boolean;
        SPECULAROVERALPHA: boolean;
        CLIPPLANE: boolean;
        CLIPPLANE2: boolean;
        CLIPPLANE3: boolean;
        CLIPPLANE4: boolean;
        CLIPPLANE5: boolean;
        CLIPPLANE6: boolean;
        ALPHATEST: boolean;
        DEPTHPREPASS: boolean;
        ALPHAFROMDIFFUSE: boolean;
        POINTSIZE: boolean;
        FOG: boolean;
        SPECULARTERM: boolean;
        DIFFUSEFRESNEL: boolean;
        OPACITYFRESNEL: boolean;
        REFLECTIONFRESNEL: boolean;
        REFRACTIONFRESNEL: boolean;
        EMISSIVEFRESNEL: boolean;
        FRESNEL: boolean;
        NORMAL: boolean;
        UV1: boolean;
        UV2: boolean;
        VERTEXCOLOR: boolean;
        VERTEXALPHA: boolean;
        NUM_BONE_INFLUENCERS: number;
        BonesPerMesh: number;
        BONETEXTURE: boolean;
        BONES_VELOCITY_ENABLED: boolean;
        INSTANCES: boolean;
        THIN_INSTANCES: boolean;
        GLOSSINESS: boolean;
        ROUGHNESS: boolean;
        EMISSIVEASILLUMINATION: boolean;
        LINKEMISSIVEWITHDIFFUSE: boolean;
        REFLECTIONFRESNELFROMSPECULAR: boolean;
        LIGHTMAP: boolean;
        LIGHTMAPDIRECTUV: number;
        OBJECTSPACE_NORMALMAP: boolean;
        USELIGHTMAPASSHADOWMAP: boolean;
        REFLECTIONMAP_3D: boolean;
        REFLECTIONMAP_SPHERICAL: boolean;
        REFLECTIONMAP_PLANAR: boolean;
        REFLECTIONMAP_CUBIC: boolean;
        USE_LOCAL_REFLECTIONMAP_CUBIC: boolean;
        REFLECTIONMAP_PROJECTION: boolean;
        REFLECTIONMAP_SKYBOX: boolean;
        REFLECTIONMAP_EXPLICIT: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR_FIXED: boolean;
        REFLECTIONMAP_MIRROREDEQUIRECTANGULAR_FIXED: boolean;
        INVERTCUBICMAP: boolean;
        LOGARITHMICDEPTH: boolean;
        REFRACTION: boolean;
        REFRACTIONMAP_3D: boolean;
        REFLECTIONOVERALPHA: boolean;
        TWOSIDEDLIGHTING: boolean;
        SHADOWFLOAT: boolean;
        MORPHTARGETS: boolean;
        MORPHTARGETS_NORMAL: boolean;
        MORPHTARGETS_TANGENT: boolean;
        MORPHTARGETS_UV: boolean;
        NUM_MORPH_INFLUENCERS: number;
        NONUNIFORMSCALING: boolean;
        PREMULTIPLYALPHA: boolean;
        ALPHATEST_AFTERALLALPHACOMPUTATIONS: boolean;
        ALPHABLEND: boolean;
        PREPASS: boolean;
        PREPASS_IRRADIANCE: boolean;
        PREPASS_IRRADIANCE_INDEX: number;
        PREPASS_ALBEDO: boolean;
        PREPASS_ALBEDO_INDEX: number;
        PREPASS_DEPTHNORMAL: boolean;
        PREPASS_DEPTHNORMAL_INDEX: number;
        PREPASS_POSITION: boolean;
        PREPASS_POSITION_INDEX: number;
        PREPASS_VELOCITY: boolean;
        PREPASS_VELOCITY_INDEX: number;
        PREPASS_REFLECTIVITY: boolean;
        PREPASS_REFLECTIVITY_INDEX: number;
        SCENE_MRT_COUNT: number;
        RGBDLIGHTMAP: boolean;
        RGBDREFLECTION: boolean;
        RGBDREFRACTION: boolean;
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        TONEMAPPING_ACES: boolean;
        CONTRAST: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
        MULTIVIEW: boolean;
        /**
         * If the reflection texture on this material is in linear color space
         * @hidden
         */
        IS_REFLECTION_LINEAR: boolean;
        /**
         * If the refraction texture on this material is in linear color space
         * @hidden
         */
        IS_REFRACTION_LINEAR: boolean;
        EXPOSURE: boolean;
        constructor();
        setReflectionMode(modeToEnable: string): void;
    }
    /**
     * This is the default material used in Babylon. It is the best trade off between quality
     * and performances.
     * @see https://doc.babylonjs.com/babylon101/materials
     */
    export class StandardMaterial extends PushMaterial {
        private _diffuseTexture;
        /**
         * The basic texture of the material as viewed under a light.
         */
        diffuseTexture: Nullable<BaseTexture>;
        private _ambientTexture;
        /**
         * AKA Occlusion Texture in other nomenclature, it helps adding baked shadows into your material.
         */
        ambientTexture: Nullable<BaseTexture>;
        private _opacityTexture;
        /**
         * Define the transparency of the material from a texture.
         * The final alpha value can be read either from the red channel (if texture.getAlphaFromRGB is false)
         * or from the luminance or the current texel (if texture.getAlphaFromRGB is true)
         */
        opacityTexture: Nullable<BaseTexture>;
        private _reflectionTexture;
        /**
         * Define the texture used to display the reflection.
         * @see https://doc.babylonjs.com/how_to/reflect#how-to-obtain-reflections-and-refractions
         */
        reflectionTexture: Nullable<BaseTexture>;
        private _emissiveTexture;
        /**
         * Define texture of the material as if self lit.
         * This will be mixed in the final result even in the absence of light.
         */
        emissiveTexture: Nullable<BaseTexture>;
        private _specularTexture;
        /**
         * Define how the color and intensity of the highlight given by the light in the material.
         */
        specularTexture: Nullable<BaseTexture>;
        private _bumpTexture;
        /**
         * Bump mapping is a technique to simulate bump and dents on a rendered surface.
         * These are made by creating a normal map from an image. The means to do this can be found on the web, a search for 'normal map generator' will bring up free and paid for methods of doing this.
         * @see https://doc.babylonjs.com/how_to/more_materials#bump-map
         */
        bumpTexture: Nullable<BaseTexture>;
        private _lightmapTexture;
        /**
         * Complex lighting can be computationally expensive to compute at runtime.
         * To save on computation, lightmaps may be used to store calculated lighting in a texture which will be applied to a given mesh.
         * @see https://doc.babylonjs.com/babylon101/lights#lightmaps
         */
        lightmapTexture: Nullable<BaseTexture>;
        private _refractionTexture;
        /**
         * Define the texture used to display the refraction.
         * @see https://doc.babylonjs.com/how_to/reflect#how-to-obtain-reflections-and-refractions
         */
        refractionTexture: Nullable<BaseTexture>;
        /**
         * The color of the material lit by the environmental background lighting.
         * @see https://doc.babylonjs.com/babylon101/materials#ambient-color-example
         */
        ambientColor: Color3;
        /**
         * The basic color of the material as viewed under a light.
         */
        diffuseColor: Color3;
        /**
         * Define how the color and intensity of the highlight given by the light in the material.
         */
        specularColor: Color3;
        /**
         * Define the color of the material as if self lit.
         * This will be mixed in the final result even in the absence of light.
         */
        emissiveColor: Color3;
        /**
         * Defines how sharp are the highlights in the material.
         * The bigger the value the sharper giving a more glossy feeling to the result.
         * Reversely, the smaller the value the blurrier giving a more rough feeling to the result.
         */
        specularPower: number;
        private _useAlphaFromDiffuseTexture;
        /**
         * Does the transparency come from the diffuse texture alpha channel.
         */
        useAlphaFromDiffuseTexture: boolean;
        private _useEmissiveAsIllumination;
        /**
         * If true, the emissive value is added into the end result, otherwise it is multiplied in.
         */
        useEmissiveAsIllumination: boolean;
        private _linkEmissiveWithDiffuse;
        /**
         * If true, some kind of energy conservation will prevent the end result to be more than 1 by reducing
         * the emissive level when the final color is close to one.
         */
        linkEmissiveWithDiffuse: boolean;
        private _useSpecularOverAlpha;
        /**
         * Specifies that the material will keep the specular highlights over a transparent surface (only the most limunous ones).
         * A car glass is a good exemple of that. When sun reflects on it you can not see what is behind.
         */
        useSpecularOverAlpha: boolean;
        private _useReflectionOverAlpha;
        /**
         * Specifies that the material will keeps the reflection highlights over a transparent surface (only the most limunous ones).
         * A car glass is a good exemple of that. When the street lights reflects on it you can not see what is behind.
         */
        useReflectionOverAlpha: boolean;
        private _disableLighting;
        /**
         * Does lights from the scene impacts this material.
         * It can be a nice trick for performance to disable lighting on a fully emissive material.
         */
        disableLighting: boolean;
        private _useObjectSpaceNormalMap;
        /**
         * Allows using an object space normal map (instead of tangent space).
         */
        useObjectSpaceNormalMap: boolean;
        private _useParallax;
        /**
         * Is parallax enabled or not.
         * @see https://doc.babylonjs.com/how_to/using_parallax_mapping
         */
        useParallax: boolean;
        private _useParallaxOcclusion;
        /**
         * Is parallax occlusion enabled or not.
         * If true, the outcome is way more realistic than traditional Parallax but you can expect a performance hit that worthes consideration.
         * @see https://doc.babylonjs.com/how_to/using_parallax_mapping
         */
        useParallaxOcclusion: boolean;
        /**
         * Apply a scaling factor that determine which "depth" the height map should reprensent. A value between 0.05 and 0.1 is reasonnable in Parallax, you can reach 0.2 using Parallax Occlusion.
         */
        parallaxScaleBias: number;
        private _roughness;
        /**
         * Helps to define how blurry the reflections should appears in the material.
         */
        roughness: number;
        /**
         * In case of refraction, define the value of the index of refraction.
         * @see https://doc.babylonjs.com/how_to/reflect#how-to-obtain-reflections-and-refractions
         */
        indexOfRefraction: number;
        /**
         * Invert the refraction texture alongside the y axis.
         * It can be useful with procedural textures or probe for instance.
         * @see https://doc.babylonjs.com/how_to/reflect#how-to-obtain-reflections-and-refractions
         */
        invertRefractionY: boolean;
        /**
         * Defines the alpha limits in alpha test mode.
         */
        alphaCutOff: number;
        private _useLightmapAsShadowmap;
        /**
         * In case of light mapping, define whether the map contains light or shadow informations.
         */
        useLightmapAsShadowmap: boolean;
        private _diffuseFresnelParameters;
        /**
         * Define the diffuse fresnel parameters of the material.
         * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
         */
        diffuseFresnelParameters: FresnelParameters;
        private _opacityFresnelParameters;
        /**
         * Define the opacity fresnel parameters of the material.
         * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
         */
        opacityFresnelParameters: FresnelParameters;
        private _reflectionFresnelParameters;
        /**
         * Define the reflection fresnel parameters of the material.
         * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
         */
        reflectionFresnelParameters: FresnelParameters;
        private _refractionFresnelParameters;
        /**
         * Define the refraction fresnel parameters of the material.
         * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
         */
        refractionFresnelParameters: FresnelParameters;
        private _emissiveFresnelParameters;
        /**
         * Define the emissive fresnel parameters of the material.
         * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
         */
        emissiveFresnelParameters: FresnelParameters;
        private _useReflectionFresnelFromSpecular;
        /**
         * If true automatically deducts the fresnels values from the material specularity.
         * @see https://doc.babylonjs.com/how_to/how_to_use_fresnelparameters
         */
        useReflectionFresnelFromSpecular: boolean;
        private _useGlossinessFromSpecularMapAlpha;
        /**
         * Defines if the glossiness/roughness of the material should be read from the specular map alpha channel
         */
        useGlossinessFromSpecularMapAlpha: boolean;
        private _maxSimultaneousLights;
        /**
         * Defines the maximum number of lights that can be used in the material
         */
        maxSimultaneousLights: number;
        private _invertNormalMapX;
        /**
         * If sets to true, x component of normal map value will invert (x = 1.0 - x).
         */
        invertNormalMapX: boolean;
        private _invertNormalMapY;
        /**
         * If sets to true, y component of normal map value will invert (y = 1.0 - y).
         */
        invertNormalMapY: boolean;
        private _twoSidedLighting;
        /**
         * If sets to true and backfaceCulling is false, normals will be flipped on the backside.
         */
        twoSidedLighting: boolean;
        /**
         * Default configuration related to image processing available in the standard Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Gets the image processing configuration used either in this material.
         */
        get imageProcessingConfiguration(): ImageProcessingConfiguration;
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        set imageProcessingConfiguration(value: ImageProcessingConfiguration);
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the Standard Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>): void;
        /**
         * Defines additionnal PrePass parameters for the material.
         */
        readonly prePassConfiguration: PrePassConfiguration;
        /**
         * Gets wether the color curves effect is enabled.
         */
        get cameraColorCurvesEnabled(): boolean;
        /**
         * Sets wether the color curves effect is enabled.
         */
        set cameraColorCurvesEnabled(value: boolean);
        /**
         * Gets wether the color grading effect is enabled.
         */
        get cameraColorGradingEnabled(): boolean;
        /**
         * Gets wether the color grading effect is enabled.
         */
        set cameraColorGradingEnabled(value: boolean);
        /**
         * Gets wether tonemapping is enabled or not.
         */
        get cameraToneMappingEnabled(): boolean;
        /**
         * Sets wether tonemapping is enabled or not
         */
        set cameraToneMappingEnabled(value: boolean);
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        get cameraExposure(): number;
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        set cameraExposure(value: number);
        /**
         * Gets The camera contrast used on this material.
         */
        get cameraContrast(): number;
        /**
         * Sets The camera contrast used on this material.
         */
        set cameraContrast(value: number);
        /**
         * Gets the Color Grading 2D Lookup Texture.
         */
        get cameraColorGradingTexture(): Nullable<BaseTexture>;
        /**
         * Sets the Color Grading 2D Lookup Texture.
         */
        set cameraColorGradingTexture(value: Nullable<BaseTexture>);
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        get cameraColorCurves(): Nullable<ColorCurves>;
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        set cameraColorCurves(value: Nullable<ColorCurves>);
        /**
         * Can this material render to several textures at once
         */
        get canRenderToMRT(): boolean;
        /**
         * Defines the detail map parameters for the material.
         */
        readonly detailMap: DetailMapConfiguration;
        protected _renderTargets: SmartArray<RenderTargetTexture>;
        protected _worldViewProjectionMatrix: Matrix;
        protected _globalAmbientColor: Color3;
        protected _useLogarithmicDepth: boolean;
        protected _rebuildInParallel: boolean;
        /**
         * Instantiates a new standard material.
         * This is the default material used in Babylon. It is the best trade off between quality
         * and performances.
         * @see https://doc.babylonjs.com/babylon101/materials
         * @param name Define the name of the material in the scene
         * @param scene Define the scene the material belong to
         */
        constructor(name: string, scene: Scene);
        /**
         * Gets a boolean indicating that current material needs to register RTT
         */
        get hasRenderTargetTextures(): boolean;
        /**
         * Gets the current class name of the material e.g. "StandardMaterial"
         * Mainly use in serialization.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * In case the depth buffer does not allow enough depth precision for your scene (might be the case in large scenes)
         * You can try switching to logarithmic depth.
         * @see https://doc.babylonjs.com/how_to/using_logarithmic_depth_buffer
         */
        get useLogarithmicDepth(): boolean;
        set useLogarithmicDepth(value: boolean);
        /**
         * Specifies if the material will require alpha blending
         * @returns a boolean specifying if alpha blending is needed
         */
        needAlphaBlending(): boolean;
        /**
         * Specifies if this material should be rendered in alpha test mode
         * @returns a boolean specifying if an alpha test is needed.
         */
        needAlphaTesting(): boolean;
        /**
         * Specifies whether or not the alpha value of the diffuse texture should be used for alpha blending.
         */
        protected _shouldUseAlphaFromDiffuseTexture(): boolean;
        /**
         * Specifies whether or not there is a usable alpha channel for transparency.
         */
        protected _hasAlphaChannel(): boolean;
        /**
         * Get the texture used for alpha test purpose.
         * @returns the diffuse texture in case of the standard material.
         */
        getAlphaTestTexture(): Nullable<BaseTexture>;
        /**
         * Get if the submesh is ready to be used and all its information available.
         * Child classes can use it to update shaders
         * @param mesh defines the mesh to check
         * @param subMesh defines which submesh to check
         * @param useInstances specifies that instances should be used
         * @returns a boolean indicating that the submesh is ready or not
         */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: SubMesh, useInstances?: boolean): boolean;
        /**
         * Builds the material UBO layouts.
         * Used internally during the effect preparation.
         */
        buildUniformLayout(): void;
        /**
         * Unbinds the material from the mesh
         */
        unbind(): void;
        /**
         * Binds the submesh to this material by preparing the effect and shader to draw
         * @param world defines the world transformation matrix
         * @param mesh defines the mesh containing the submesh
         * @param subMesh defines the submesh to bind the material to
         */
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        /**
         * Get the list of animatables in the material.
         * @returns the list of animatables object used in the material
         */
        getAnimatables(): IAnimatable[];
        /**
         * Gets the active textures from the material
         * @returns an array of textures
         */
        getActiveTextures(): BaseTexture[];
        /**
         * Specifies if the material uses a texture
         * @param texture defines the texture to check against the material
         * @returns a boolean specifying if the material uses the texture
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Disposes the material
         * @param forceDisposeEffect specifies if effects should be forcefully disposed
         * @param forceDisposeTextures specifies if textures should be forcefully disposed
         */
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        /**
         * Makes a duplicate of the material, and gives it a new name
         * @param name defines the new name for the duplicated material
         * @returns the cloned material
         */
        clone(name: string): StandardMaterial;
        /**
         * Serializes this material in a JSON representation
         * @returns the serialized material object
         */
        serialize(): any;
        /**
         * Creates a standard material from parsed material data
         * @param source defines the JSON representation of the material
         * @param scene defines the hosting scene
         * @param rootUrl defines the root URL to use to load textures and relative dependencies
         * @returns a new standard material
         */
        static Parse(source: any, scene: Scene, rootUrl: string): StandardMaterial;
        /**
         * Are diffuse textures enabled in the application.
         */
        static get DiffuseTextureEnabled(): boolean;
        static set DiffuseTextureEnabled(value: boolean);
        /**
         * Are detail textures enabled in the application.
         */
        static get DetailTextureEnabled(): boolean;
        static set DetailTextureEnabled(value: boolean);
        /**
         * Are ambient textures enabled in the application.
         */
        static get AmbientTextureEnabled(): boolean;
        static set AmbientTextureEnabled(value: boolean);
        /**
         * Are opacity textures enabled in the application.
         */
        static get OpacityTextureEnabled(): boolean;
        static set OpacityTextureEnabled(value: boolean);
        /**
         * Are reflection textures enabled in the application.
         */
        static get ReflectionTextureEnabled(): boolean;
        static set ReflectionTextureEnabled(value: boolean);
        /**
         * Are emissive textures enabled in the application.
         */
        static get EmissiveTextureEnabled(): boolean;
        static set EmissiveTextureEnabled(value: boolean);
        /**
         * Are specular textures enabled in the application.
         */
        static get SpecularTextureEnabled(): boolean;
        static set SpecularTextureEnabled(value: boolean);
        /**
         * Are bump textures enabled in the application.
         */
        static get BumpTextureEnabled(): boolean;
        static set BumpTextureEnabled(value: boolean);
        /**
         * Are lightmap textures enabled in the application.
         */
        static get LightmapTextureEnabled(): boolean;
        static set LightmapTextureEnabled(value: boolean);
        /**
         * Are refraction textures enabled in the application.
         */
        static get RefractionTextureEnabled(): boolean;
        static set RefractionTextureEnabled(value: boolean);
        /**
         * Are color grading textures enabled in the application.
         */
        static get ColorGradingTextureEnabled(): boolean;
        static set ColorGradingTextureEnabled(value: boolean);
        /**
         * Are fresnels enabled in the application.
         */
        static get FresnelEnabled(): boolean;
        static set FresnelEnabled(value: boolean);
    }
}
declare module "babylonjs/Shaders/rgbdDecode.fragment" {
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    /** @hidden */
    export var rgbdDecodePixelShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Misc/rgbdTextureTools" {
    import "babylonjs/Shaders/rgbdDecode.fragment";
    import "babylonjs/Engines/Extensions/engine.renderTarget";
    import { Texture } from "babylonjs/Materials/Textures/texture";
    /**
     * Class used to host RGBD texture specific utilities
     */
    export class RGBDTextureTools {
        /**
         * Expand the RGBD Texture from RGBD to Half Float if possible.
         * @param texture the texture to expand.
         */
        static ExpandRGBDTexture(texture: Texture): void;
    }
}
declare module "babylonjs/Misc/brdfTextureTools" {
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Scene } from "babylonjs/scene";
    /**
     * Class used to host texture specific utilities
     */
    export class BRDFTextureTools {
        /**
         * Prevents texture cache collision
         */
        private static _instanceNumber;
        /**
         * Gets a default environment BRDF for MS-BRDF Height Correlated BRDF
         * @param scene defines the hosting scene
         * @returns the environment BRDF texture
         */
        static GetEnvironmentBRDFTexture(scene: Scene): BaseTexture;
        private static _environmentBRDFBase64Texture;
    }
}
declare module "babylonjs/Materials/PBR/pbrClearCoatConfiguration" {
    import { Nullable } from "babylonjs/types";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { UniformBuffer } from "babylonjs/Materials/uniformBuffer";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import { EffectFallbacks } from "babylonjs/Materials/effectFallbacks";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { Engine } from "babylonjs/Engines/engine";
    import { Scene } from "babylonjs/scene";
    /**
     * @hidden
     */
    export interface IMaterialClearCoatDefines {
        CLEARCOAT: boolean;
        CLEARCOAT_DEFAULTIOR: boolean;
        CLEARCOAT_TEXTURE: boolean;
        CLEARCOAT_TEXTURE_ROUGHNESS: boolean;
        CLEARCOAT_TEXTUREDIRECTUV: number;
        CLEARCOAT_TEXTURE_ROUGHNESSDIRECTUV: number;
        CLEARCOAT_BUMP: boolean;
        CLEARCOAT_BUMPDIRECTUV: number;
        CLEARCOAT_USE_ROUGHNESS_FROM_MAINTEXTURE: boolean;
        CLEARCOAT_TEXTURE_ROUGHNESS_IDENTICAL: boolean;
        CLEARCOAT_REMAP_F0: boolean;
        CLEARCOAT_TINT: boolean;
        CLEARCOAT_TINT_TEXTURE: boolean;
        CLEARCOAT_TINT_TEXTUREDIRECTUV: number;
        /** @hidden */
        _areTexturesDirty: boolean;
    }
    /**
     * Define the code related to the clear coat parameters of the pbr material.
     */
    export class PBRClearCoatConfiguration {
        /**
         * This defaults to 1.5 corresponding to a 0.04 f0 or a 4% reflectance at normal incidence
         * The default fits with a polyurethane material.
         * @hidden
         */
        static readonly _DefaultIndexOfRefraction: number;
        private _isEnabled;
        /**
         * Defines if the clear coat is enabled in the material.
         */
        isEnabled: boolean;
        /**
         * Defines the clear coat layer strength (between 0 and 1) it defaults to 1.
         */
        intensity: number;
        /**
         * Defines the clear coat layer roughness.
         */
        roughness: number;
        private _indexOfRefraction;
        /**
         * Defines the index of refraction of the clear coat.
         * This defaults to 1.5 corresponding to a 0.04 f0 or a 4% reflectance at normal incidence
         * The default fits with a polyurethane material.
         * Changing the default value is more performance intensive.
         */
        indexOfRefraction: number;
        private _texture;
        /**
         * Stores the clear coat values in a texture (red channel is intensity and green channel is roughness)
         * If useRoughnessFromMainTexture is false, the green channel of texture is not used and the green channel of textureRoughness is used instead
         * if textureRoughness is not empty, else no texture roughness is used
         */
        texture: Nullable<BaseTexture>;
        private _useRoughnessFromMainTexture;
        /**
         * Indicates that the green channel of the texture property will be used for roughness (default: true)
         * If false, the green channel from textureRoughness is used for roughness
         */
        useRoughnessFromMainTexture: boolean;
        private _textureRoughness;
        /**
         * Stores the clear coat roughness in a texture (green channel)
         * Not used if useRoughnessFromMainTexture is true
         */
        textureRoughness: Nullable<BaseTexture>;
        private _remapF0OnInterfaceChange;
        /**
         * Defines if the F0 value should be remapped to account for the interface change in the material.
         */
        remapF0OnInterfaceChange: boolean;
        private _bumpTexture;
        /**
         * Define the clear coat specific bump texture.
         */
        bumpTexture: Nullable<BaseTexture>;
        private _isTintEnabled;
        /**
         * Defines if the clear coat tint is enabled in the material.
         */
        isTintEnabled: boolean;
        /**
         * Defines the clear coat tint of the material.
         * This is only use if tint is enabled
         */
        tintColor: Color3;
        /**
         * Defines the distance at which the tint color should be found in the
         * clear coat media.
         * This is only use if tint is enabled
         */
        tintColorAtDistance: number;
        /**
         * Defines the clear coat layer thickness.
         * This is only use if tint is enabled
         */
        tintThickness: number;
        private _tintTexture;
        /**
         * Stores the clear tint values in a texture.
         * rgb is tint
         * a is a thickness factor
         */
        tintTexture: Nullable<BaseTexture>;
        /** @hidden */
        private _internalMarkAllSubMeshesAsTexturesDirty;
        /** @hidden */
        _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Instantiate a new istance of clear coat configuration.
         * @param markAllSubMeshesAsTexturesDirty Callback to flag the material to dirty
         */
        constructor(markAllSubMeshesAsTexturesDirty: () => void);
        /**
         * Gets wehter the submesh is ready to be used or not.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         * @param engine defines the engine the material belongs to.
         * @param disableBumpMap defines wether the material disables bump or not.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(defines: IMaterialClearCoatDefines, scene: Scene, engine: Engine, disableBumpMap: boolean): boolean;
        /**
         * Checks to see if a texture is used in the material.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene to the material belongs to.
         */
        prepareDefines(defines: IMaterialClearCoatDefines, scene: Scene): void;
        /**
         * Binds the material data.
         * @param uniformBuffer defines the Uniform buffer to fill in.
         * @param scene defines the scene the material belongs to.
         * @param engine defines the engine the material belongs to.
         * @param disableBumpMap defines wether the material disables bump or not.
         * @param isFrozen defines wether the material is frozen or not.
         * @param invertNormalMapX If sets to true, x component of normal map value will be inverted (x = 1.0 - x).
         * @param invertNormalMapY If sets to true, y component of normal map value will be inverted (y = 1.0 - y).
         * @param subMesh the submesh to bind data for
         */
        bindForSubMesh(uniformBuffer: UniformBuffer, scene: Scene, engine: Engine, disableBumpMap: boolean, isFrozen: boolean, invertNormalMapX: boolean, invertNormalMapY: boolean, subMesh?: SubMesh): void;
        /**
         * Checks to see if a texture is used in the material.
         * @param texture - Base texture to use.
         * @returns - Boolean specifying if a texture is used in the material.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Returns an array of the actively used textures.
         * @param activeTextures Array of BaseTextures
         */
        getActiveTextures(activeTextures: BaseTexture[]): void;
        /**
         * Returns the animatable textures.
         * @param animatables Array of animatable textures.
         */
        getAnimatables(animatables: IAnimatable[]): void;
        /**
         * Disposes the resources of the material.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
        * Get the current class name of the texture useful for serialization or dynamic coding.
        * @returns "PBRClearCoatConfiguration"
        */
        getClassName(): string;
        /**
         * Add fallbacks to the effect fallbacks list.
         * @param defines defines the Base texture to use.
         * @param fallbacks defines the current fallback list.
         * @param currentRank defines the current fallback rank.
         * @returns the new fallback rank.
         */
        static AddFallbacks(defines: IMaterialClearCoatDefines, fallbacks: EffectFallbacks, currentRank: number): number;
        /**
         * Add the required uniforms to the current list.
         * @param uniforms defines the current uniform list.
         */
        static AddUniforms(uniforms: string[]): void;
        /**
         * Add the required samplers to the current list.
         * @param samplers defines the current sampler list.
         */
        static AddSamplers(samplers: string[]): void;
        /**
         * Add the required uniforms to the current buffer.
         * @param uniformBuffer defines the current uniform buffer.
         */
        static PrepareUniformBuffer(uniformBuffer: UniformBuffer): void;
        /**
         * Makes a duplicate of the current configuration into another one.
         * @param clearCoatConfiguration define the config where to copy the info
         */
        copyTo(clearCoatConfiguration: PBRClearCoatConfiguration): void;
        /**
         * Serializes this clear coat configuration.
         * @returns - An object with the serialized config.
         */
        serialize(): any;
        /**
         * Parses a anisotropy Configuration from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/PBR/pbrAnisotropicConfiguration" {
    import { UniformBuffer } from "babylonjs/Materials/uniformBuffer";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Vector2 } from "babylonjs/Maths/math.vector";
    import { Scene } from "babylonjs/scene";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Nullable } from "babylonjs/types";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import { EffectFallbacks } from "babylonjs/Materials/effectFallbacks";
    /**
     * @hidden
     */
    export interface IMaterialAnisotropicDefines {
        ANISOTROPIC: boolean;
        ANISOTROPIC_TEXTURE: boolean;
        ANISOTROPIC_TEXTUREDIRECTUV: number;
        MAINUV1: boolean;
        _areTexturesDirty: boolean;
        _needUVs: boolean;
    }
    /**
     * Define the code related to the anisotropic parameters of the pbr material.
     */
    export class PBRAnisotropicConfiguration {
        private _isEnabled;
        /**
         * Defines if the anisotropy is enabled in the material.
         */
        isEnabled: boolean;
        /**
         * Defines the anisotropy strength (between 0 and 1) it defaults to 1.
         */
        intensity: number;
        /**
         * Defines if the effect is along the tangents, bitangents or in between.
         * By default, the effect is "strectching" the highlights along the tangents.
         */
        direction: Vector2;
        private _texture;
        /**
         * Stores the anisotropy values in a texture.
         * rg is direction (like normal from -1 to 1)
         * b is a intensity
         */
        texture: Nullable<BaseTexture>;
        /** @hidden */
        private _internalMarkAllSubMeshesAsTexturesDirty;
        /** @hidden */
        _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Instantiate a new istance of anisotropy configuration.
         * @param markAllSubMeshesAsTexturesDirty Callback to flag the material to dirty
         */
        constructor(markAllSubMeshesAsTexturesDirty: () => void);
        /**
         * Specifies that the submesh is ready to be used.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(defines: IMaterialAnisotropicDefines, scene: Scene): boolean;
        /**
         * Checks to see if a texture is used in the material.
         * @param defines the list of "defines" to update.
         * @param mesh the mesh we are preparing the defines for.
         * @param scene defines the scene the material belongs to.
         */
        prepareDefines(defines: IMaterialAnisotropicDefines, mesh: AbstractMesh, scene: Scene): void;
        /**
         * Binds the material data.
         * @param uniformBuffer defines the Uniform buffer to fill in.
         * @param scene defines the scene the material belongs to.
         * @param isFrozen defines wether the material is frozen or not.
         */
        bindForSubMesh(uniformBuffer: UniformBuffer, scene: Scene, isFrozen: boolean): void;
        /**
         * Checks to see if a texture is used in the material.
         * @param texture - Base texture to use.
         * @returns - Boolean specifying if a texture is used in the material.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Returns an array of the actively used textures.
         * @param activeTextures Array of BaseTextures
         */
        getActiveTextures(activeTextures: BaseTexture[]): void;
        /**
         * Returns the animatable textures.
         * @param animatables Array of animatable textures.
         */
        getAnimatables(animatables: IAnimatable[]): void;
        /**
         * Disposes the resources of the material.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
        * Get the current class name of the texture useful for serialization or dynamic coding.
        * @returns "PBRAnisotropicConfiguration"
        */
        getClassName(): string;
        /**
         * Add fallbacks to the effect fallbacks list.
         * @param defines defines the Base texture to use.
         * @param fallbacks defines the current fallback list.
         * @param currentRank defines the current fallback rank.
         * @returns the new fallback rank.
         */
        static AddFallbacks(defines: IMaterialAnisotropicDefines, fallbacks: EffectFallbacks, currentRank: number): number;
        /**
         * Add the required uniforms to the current list.
         * @param uniforms defines the current uniform list.
         */
        static AddUniforms(uniforms: string[]): void;
        /**
         * Add the required uniforms to the current buffer.
         * @param uniformBuffer defines the current uniform buffer.
         */
        static PrepareUniformBuffer(uniformBuffer: UniformBuffer): void;
        /**
         * Add the required samplers to the current list.
         * @param samplers defines the current sampler list.
         */
        static AddSamplers(samplers: string[]): void;
        /**
         * Makes a duplicate of the current configuration into another one.
         * @param anisotropicConfiguration define the config where to copy the info
         */
        copyTo(anisotropicConfiguration: PBRAnisotropicConfiguration): void;
        /**
         * Serializes this anisotropy configuration.
         * @returns - An object with the serialized config.
         */
        serialize(): any;
        /**
         * Parses a anisotropy Configuration from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/PBR/pbrBRDFConfiguration" {
    import { Scene } from "babylonjs/scene";
    /**
     * @hidden
     */
    export interface IMaterialBRDFDefines {
        BRDF_V_HEIGHT_CORRELATED: boolean;
        MS_BRDF_ENERGY_CONSERVATION: boolean;
        SPHERICAL_HARMONICS: boolean;
        SPECULAR_GLOSSINESS_ENERGY_CONSERVATION: boolean;
        /** @hidden */
        _areMiscDirty: boolean;
    }
    /**
     * Define the code related to the BRDF parameters of the pbr material.
     */
    export class PBRBRDFConfiguration {
        /**
         * Default value used for the energy conservation.
         * This should only be changed to adapt to the type of texture in scene.environmentBRDFTexture.
         */
        static DEFAULT_USE_ENERGY_CONSERVATION: boolean;
        /**
         * Default value used for the Smith Visibility Height Correlated mode.
         * This should only be changed to adapt to the type of texture in scene.environmentBRDFTexture.
         */
        static DEFAULT_USE_SMITH_VISIBILITY_HEIGHT_CORRELATED: boolean;
        /**
         * Default value used for the IBL diffuse part.
         * This can help switching back to the polynomials mode globally which is a tiny bit
         * less GPU intensive at the drawback of a lower quality.
         */
        static DEFAULT_USE_SPHERICAL_HARMONICS: boolean;
        /**
         * Default value used for activating energy conservation for the specular workflow.
         * If activated, the albedo color is multiplied with (1. - maxChannel(specular color)).
         * If deactivated, a material is only physically plausible, when (albedo color + specular color) < 1.
         */
        static DEFAULT_USE_SPECULAR_GLOSSINESS_INPUT_ENERGY_CONSERVATION: boolean;
        private _useEnergyConservation;
        /**
         * Defines if the material uses energy conservation.
         */
        useEnergyConservation: boolean;
        private _useSmithVisibilityHeightCorrelated;
        /**
         * LEGACY Mode set to false
         * Defines if the material uses height smith correlated visibility term.
         * If you intent to not use our default BRDF, you need to load a separate BRDF Texture for the PBR
         * You can either load https://assets.babylonjs.com/environments/uncorrelatedBRDF.png
         * or https://assets.babylonjs.com/environments/uncorrelatedBRDF.dds to have more precision
         * Not relying on height correlated will also disable energy conservation.
         */
        useSmithVisibilityHeightCorrelated: boolean;
        private _useSphericalHarmonics;
        /**
         * LEGACY Mode set to false
         * Defines if the material uses spherical harmonics vs spherical polynomials for the
         * diffuse part of the IBL.
         * The harmonics despite a tiny bigger cost has been proven to provide closer results
         * to the ground truth.
         */
        useSphericalHarmonics: boolean;
        private _useSpecularGlossinessInputEnergyConservation;
        /**
         * Defines if the material uses energy conservation, when the specular workflow is active.
         * If activated, the albedo color is multiplied with (1. - maxChannel(specular color)).
         * If deactivated, a material is only physically plausible, when (albedo color + specular color) < 1.
         * In the deactivated case, the material author has to ensure energy conservation, for a physically plausible rendering.
         */
        useSpecularGlossinessInputEnergyConservation: boolean;
        /** @hidden */
        private _internalMarkAllSubMeshesAsMiscDirty;
        /** @hidden */
        _markAllSubMeshesAsMiscDirty(): void;
        /**
         * Instantiate a new istance of clear coat configuration.
         * @param markAllSubMeshesAsMiscDirty Callback to flag the material to dirty
         */
        constructor(markAllSubMeshesAsMiscDirty: () => void);
        /**
         * Checks to see if a texture is used in the material.
         * @param defines the list of "defines" to update.
         */
        prepareDefines(defines: IMaterialBRDFDefines): void;
        /**
        * Get the current class name of the texture useful for serialization or dynamic coding.
        * @returns "PBRClearCoatConfiguration"
        */
        getClassName(): string;
        /**
         * Makes a duplicate of the current configuration into another one.
         * @param brdfConfiguration define the config where to copy the info
         */
        copyTo(brdfConfiguration: PBRBRDFConfiguration): void;
        /**
         * Serializes this BRDF configuration.
         * @returns - An object with the serialized config.
         */
        serialize(): any;
        /**
         * Parses a anisotropy Configuration from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/PBR/pbrSheenConfiguration" {
    import { UniformBuffer } from "babylonjs/Materials/uniformBuffer";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { Scene } from "babylonjs/scene";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Nullable } from "babylonjs/types";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import { EffectFallbacks } from "babylonjs/Materials/effectFallbacks";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    /**
     * @hidden
     */
    export interface IMaterialSheenDefines {
        SHEEN: boolean;
        SHEEN_TEXTURE: boolean;
        SHEEN_TEXTURE_ROUGHNESS: boolean;
        SHEEN_TEXTUREDIRECTUV: number;
        SHEEN_TEXTURE_ROUGHNESSDIRECTUV: number;
        SHEEN_LINKWITHALBEDO: boolean;
        SHEEN_ROUGHNESS: boolean;
        SHEEN_ALBEDOSCALING: boolean;
        SHEEN_USE_ROUGHNESS_FROM_MAINTEXTURE: boolean;
        SHEEN_TEXTURE_ROUGHNESS_IDENTICAL: boolean;
        /** @hidden */
        _areTexturesDirty: boolean;
    }
    /**
     * Define the code related to the Sheen parameters of the pbr material.
     */
    export class PBRSheenConfiguration {
        private _isEnabled;
        /**
         * Defines if the material uses sheen.
         */
        isEnabled: boolean;
        private _linkSheenWithAlbedo;
        /**
         * Defines if the sheen is linked to the sheen color.
         */
        linkSheenWithAlbedo: boolean;
        /**
         * Defines the sheen intensity.
         */
        intensity: number;
        /**
         * Defines the sheen color.
         */
        color: Color3;
        private _texture;
        /**
         * Stores the sheen tint values in a texture.
         * rgb is tint
         * a is a intensity or roughness if the roughness property has been defined and useRoughnessFromTexture is true (in that case, textureRoughness won't be used)
         * If the roughness property has been defined and useRoughnessFromTexture is false then the alpha channel is not used to modulate roughness
         */
        texture: Nullable<BaseTexture>;
        private _useRoughnessFromMainTexture;
        /**
         * Indicates that the alpha channel of the texture property will be used for roughness.
         * Has no effect if the roughness (and texture!) property is not defined
         */
        useRoughnessFromMainTexture: boolean;
        private _roughness;
        /**
         * Defines the sheen roughness.
         * It is not taken into account if linkSheenWithAlbedo is true.
         * To stay backward compatible, material roughness is used instead if sheen roughness = null
         */
        roughness: Nullable<number>;
        private _textureRoughness;
        /**
         * Stores the sheen roughness in a texture.
         * alpha channel is the roughness. This texture won't be used if the texture property is not empty and useRoughnessFromTexture is true
         */
        textureRoughness: Nullable<BaseTexture>;
        private _albedoScaling;
        /**
         * If true, the sheen effect is layered above the base BRDF with the albedo-scaling technique.
         * It allows the strength of the sheen effect to not depend on the base color of the material,
         * making it easier to setup and tweak the effect
         */
        albedoScaling: boolean;
        /** @hidden */
        private _internalMarkAllSubMeshesAsTexturesDirty;
        /** @hidden */
        _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Instantiate a new istance of clear coat configuration.
         * @param markAllSubMeshesAsTexturesDirty Callback to flag the material to dirty
         */
        constructor(markAllSubMeshesAsTexturesDirty: () => void);
        /**
         * Specifies that the submesh is ready to be used.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(defines: IMaterialSheenDefines, scene: Scene): boolean;
        /**
         * Checks to see if a texture is used in the material.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         */
        prepareDefines(defines: IMaterialSheenDefines, scene: Scene): void;
        /**
         * Binds the material data.
         * @param uniformBuffer defines the Uniform buffer to fill in.
         * @param scene defines the scene the material belongs to.
         * @param isFrozen defines wether the material is frozen or not.
         * @param subMesh the submesh to bind data for
         */
        bindForSubMesh(uniformBuffer: UniformBuffer, scene: Scene, isFrozen: boolean, subMesh?: SubMesh): void;
        /**
         * Checks to see if a texture is used in the material.
         * @param texture - Base texture to use.
         * @returns - Boolean specifying if a texture is used in the material.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Returns an array of the actively used textures.
         * @param activeTextures Array of BaseTextures
         */
        getActiveTextures(activeTextures: BaseTexture[]): void;
        /**
         * Returns the animatable textures.
         * @param animatables Array of animatable textures.
         */
        getAnimatables(animatables: IAnimatable[]): void;
        /**
         * Disposes the resources of the material.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
        * Get the current class name of the texture useful for serialization or dynamic coding.
        * @returns "PBRSheenConfiguration"
        */
        getClassName(): string;
        /**
         * Add fallbacks to the effect fallbacks list.
         * @param defines defines the Base texture to use.
         * @param fallbacks defines the current fallback list.
         * @param currentRank defines the current fallback rank.
         * @returns the new fallback rank.
         */
        static AddFallbacks(defines: IMaterialSheenDefines, fallbacks: EffectFallbacks, currentRank: number): number;
        /**
         * Add the required uniforms to the current list.
         * @param uniforms defines the current uniform list.
         */
        static AddUniforms(uniforms: string[]): void;
        /**
         * Add the required uniforms to the current buffer.
         * @param uniformBuffer defines the current uniform buffer.
         */
        static PrepareUniformBuffer(uniformBuffer: UniformBuffer): void;
        /**
         * Add the required samplers to the current list.
         * @param samplers defines the current sampler list.
         */
        static AddSamplers(samplers: string[]): void;
        /**
         * Makes a duplicate of the current configuration into another one.
         * @param sheenConfiguration define the config where to copy the info
         */
        copyTo(sheenConfiguration: PBRSheenConfiguration): void;
        /**
         * Serializes this BRDF configuration.
         * @returns - An object with the serialized config.
         */
        serialize(): any;
        /**
         * Parses a anisotropy Configuration from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Materials/PBR/pbrSubSurfaceConfiguration" {
    import { Nullable } from "babylonjs/types";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { SmartArray } from "babylonjs/Misc/smartArray";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { RenderTargetTexture } from "babylonjs/Materials/Textures/renderTargetTexture";
    import { Effect } from "babylonjs/Materials/effect";
    import { UniformBuffer } from "babylonjs/Materials/uniformBuffer";
    import { EffectFallbacks } from "babylonjs/Materials/effectFallbacks";
    import { Engine } from "babylonjs/Engines/engine";
    import { Scene } from "babylonjs/scene";
    /**
     * @hidden
     */
    export interface IMaterialSubSurfaceDefines {
        SUBSURFACE: boolean;
        SS_REFRACTION: boolean;
        SS_TRANSLUCENCY: boolean;
        SS_SCATTERING: boolean;
        SS_THICKNESSANDMASK_TEXTURE: boolean;
        SS_THICKNESSANDMASK_TEXTUREDIRECTUV: number;
        SS_REFRACTIONMAP_3D: boolean;
        SS_REFRACTIONMAP_OPPOSITEZ: boolean;
        SS_LODINREFRACTIONALPHA: boolean;
        SS_GAMMAREFRACTION: boolean;
        SS_RGBDREFRACTION: boolean;
        SS_LINEARSPECULARREFRACTION: boolean;
        SS_LINKREFRACTIONTOTRANSPARENCY: boolean;
        SS_ALBEDOFORREFRACTIONTINT: boolean;
        SS_MASK_FROM_THICKNESS_TEXTURE: boolean;
        SS_MASK_FROM_THICKNESS_TEXTURE_GLTF: boolean;
        /** @hidden */
        _areTexturesDirty: boolean;
    }
    /**
     * Define the code related to the sub surface parameters of the pbr material.
     */
    export class PBRSubSurfaceConfiguration {
        private _isRefractionEnabled;
        /**
         * Defines if the refraction is enabled in the material.
         */
        isRefractionEnabled: boolean;
        private _isTranslucencyEnabled;
        /**
         * Defines if the translucency is enabled in the material.
         */
        isTranslucencyEnabled: boolean;
        private _isScatteringEnabled;
        /**
         * Defines if the sub surface scattering is enabled in the material.
         */
        isScatteringEnabled: boolean;
        private _scatteringDiffusionProfileIndex;
        /**
         * Diffusion profile for subsurface scattering.
         * Useful for better scattering in the skins or foliages.
         */
        get scatteringDiffusionProfile(): Nullable<Color3>;
        set scatteringDiffusionProfile(c: Nullable<Color3>);
        /**
         * Defines the refraction intensity of the material.
         * The refraction when enabled replaces the Diffuse part of the material.
         * The intensity helps transitionning between diffuse and refraction.
         */
        refractionIntensity: number;
        /**
         * Defines the translucency intensity of the material.
         * When translucency has been enabled, this defines how much of the "translucency"
         * is addded to the diffuse part of the material.
         */
        translucencyIntensity: number;
        /**
         * When enabled, transparent surfaces will be tinted with the albedo colour (independent of thickness)
         */
        useAlbedoToTintRefraction: boolean;
        private _thicknessTexture;
        /**
         * Stores the average thickness of a mesh in a texture (The texture is holding the values linearly).
         * The red channel of the texture should contain the thickness remapped between 0 and 1.
         * 0 would mean minimumThickness
         * 1 would mean maximumThickness
         * The other channels might be use as a mask to vary the different effects intensity.
         */
        thicknessTexture: Nullable<BaseTexture>;
        private _refractionTexture;
        /**
         * Defines the texture to use for refraction.
         */
        refractionTexture: Nullable<BaseTexture>;
        private _indexOfRefraction;
        /**
         * Index of refraction of the material base layer.
         * https://en.wikipedia.org/wiki/List_of_refractive_indices
         *
         * This does not only impact refraction but also the Base F0 of Dielectric Materials.
         *
         * From dielectric fresnel rules: F0 = square((iorT - iorI) / (iorT + iorI))
         */
        indexOfRefraction: number;
        private _volumeIndexOfRefraction;
        /**
         * Index of refraction of the material's volume.
         * https://en.wikipedia.org/wiki/List_of_refractive_indices
         *
         * This ONLY impacts refraction. If not provided or given a non-valid value,
         * the volume will use the same IOR as the surface.
         */
        get volumeIndexOfRefraction(): number;
        set volumeIndexOfRefraction(value: number);
        private _invertRefractionY;
        /**
         * Controls if refraction needs to be inverted on Y. This could be useful for procedural texture.
         */
        invertRefractionY: boolean;
        private _linkRefractionWithTransparency;
        /**
         * This parameters will make the material used its opacity to control how much it is refracting aginst not.
         * Materials half opaque for instance using refraction could benefit from this control.
         */
        linkRefractionWithTransparency: boolean;
        /**
         * Defines the minimum thickness stored in the thickness map.
         * If no thickness map is defined, this value will be used to simulate thickness.
         */
        minimumThickness: number;
        /**
         * Defines the maximum thickness stored in the thickness map.
         */
        maximumThickness: number;
        /**
         * Defines the volume tint of the material.
         * This is used for both translucency and scattering.
         */
        tintColor: Color3;
        /**
         * Defines the distance at which the tint color should be found in the media.
         * This is used for refraction only.
         */
        tintColorAtDistance: number;
        /**
         * Defines how far each channel transmit through the media.
         * It is defined as a color to simplify it selection.
         */
        diffusionDistance: Color3;
        private _useMaskFromThicknessTexture;
        /**
         * Stores the intensity of the different subsurface effects in the thickness texture.
         * * the green channel is the translucency intensity.
         * * the blue channel is the scattering intensity.
         * * the alpha channel is the refraction intensity.
         */
        useMaskFromThicknessTexture: boolean;
        private _scene;
        private _useMaskFromThicknessTextureGltf;
        /**
         * Stores the intensity of the different subsurface effects in the thickness texture. This variation
         * matches the channel-packing that is used by glTF.
         * * the red channel is the transmission/translucency intensity.
         * * the green channel is the thickness.
         */
        useMaskFromThicknessTextureGltf: boolean;
        /** @hidden */
        private _internalMarkAllSubMeshesAsTexturesDirty;
        private _internalMarkScenePrePassDirty;
        /** @hidden */
        _markAllSubMeshesAsTexturesDirty(): void;
        /** @hidden */
        _markScenePrePassDirty(): void;
        /**
         * Instantiate a new istance of sub surface configuration.
         * @param markAllSubMeshesAsTexturesDirty Callback to flag the material to dirty
         * @param markScenePrePassDirty Callback to flag the scene as prepass dirty
         * @param scene The scene
         */
        constructor(markAllSubMeshesAsTexturesDirty: () => void, markScenePrePassDirty: () => void, scene: Scene);
        /**
         * Gets wehter the submesh is ready to be used or not.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene the material belongs to.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(defines: IMaterialSubSurfaceDefines, scene: Scene): boolean;
        /**
         * Checks to see if a texture is used in the material.
         * @param defines the list of "defines" to update.
         * @param scene defines the scene to the material belongs to.
         */
        prepareDefines(defines: IMaterialSubSurfaceDefines, scene: Scene): void;
        /**
         * Binds the material data.
         * @param uniformBuffer defines the Uniform buffer to fill in.
         * @param scene defines the scene the material belongs to.
         * @param engine defines the engine the material belongs to.
         * @param isFrozen defines whether the material is frozen or not.
         * @param lodBasedMicrosurface defines whether the material relies on lod based microsurface or not.
         * @param realTimeFiltering defines whether the textures should be filtered on the fly.
         */
        bindForSubMesh(uniformBuffer: UniformBuffer, scene: Scene, engine: Engine, isFrozen: boolean, lodBasedMicrosurface: boolean, realTimeFiltering: boolean): void;
        /**
         * Unbinds the material from the mesh.
         * @param activeEffect defines the effect that should be unbound from.
         * @returns true if unbound, otherwise false
         */
        unbind(activeEffect: Effect): boolean;
        /**
         * Returns the texture used for refraction or null if none is used.
         * @param scene defines the scene the material belongs to.
         * @returns - Refraction texture if present.  If no refraction texture and refraction
         * is linked with transparency, returns environment texture.  Otherwise, returns null.
         */
        private _getRefractionTexture;
        /**
         * Returns true if alpha blending should be disabled.
         */
        get disableAlphaBlending(): boolean;
        /**
         * Fills the list of render target textures.
         * @param renderTargets the list of render targets to update
         */
        fillRenderTargetTextures(renderTargets: SmartArray<RenderTargetTexture>): void;
        /**
         * Checks to see if a texture is used in the material.
         * @param texture - Base texture to use.
         * @returns - Boolean specifying if a texture is used in the material.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Gets a boolean indicating that current material needs to register RTT
         * @returns true if this uses a render target otherwise false.
         */
        hasRenderTargetTextures(): boolean;
        /**
         * Returns an array of the actively used textures.
         * @param activeTextures Array of BaseTextures
         */
        getActiveTextures(activeTextures: BaseTexture[]): void;
        /**
         * Returns the animatable textures.
         * @param animatables Array of animatable textures.
         */
        getAnimatables(animatables: IAnimatable[]): void;
        /**
         * Disposes the resources of the material.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
        * Get the current class name of the texture useful for serialization or dynamic coding.
        * @returns "PBRSubSurfaceConfiguration"
        */
        getClassName(): string;
        /**
         * Add fallbacks to the effect fallbacks list.
         * @param defines defines the Base texture to use.
         * @param fallbacks defines the current fallback list.
         * @param currentRank defines the current fallback rank.
         * @returns the new fallback rank.
         */
        static AddFallbacks(defines: IMaterialSubSurfaceDefines, fallbacks: EffectFallbacks, currentRank: number): number;
        /**
         * Add the required uniforms to the current list.
         * @param uniforms defines the current uniform list.
         */
        static AddUniforms(uniforms: string[]): void;
        /**
         * Add the required samplers to the current list.
         * @param samplers defines the current sampler list.
         */
        static AddSamplers(samplers: string[]): void;
        /**
         * Add the required uniforms to the current buffer.
         * @param uniformBuffer defines the current uniform buffer.
         */
        static PrepareUniformBuffer(uniformBuffer: UniformBuffer): void;
        /**
         * Makes a duplicate of the current configuration into another one.
         * @param configuration define the config where to copy the info
         */
        copyTo(configuration: PBRSubSurfaceConfiguration): void;
        /**
         * Serializes this Sub Surface configuration.
         * @returns - An object with the serialized config.
         */
        serialize(): any;
        /**
         * Parses a anisotropy Configuration from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }
}
declare module "babylonjs/Maths/sphericalPolynomial" {
    import { Vector3 } from "babylonjs/Maths/math.vector";
    import { Color3 } from "babylonjs/Maths/math.color";
    /**
     * Class representing spherical harmonics coefficients to the 3rd degree
     */
    export class SphericalHarmonics {
        /**
         * Defines whether or not the harmonics have been prescaled for rendering.
         */
        preScaled: boolean;
        /**
         * The l0,0 coefficients of the spherical harmonics
         */
        l00: Vector3;
        /**
         * The l1,-1 coefficients of the spherical harmonics
         */
        l1_1: Vector3;
        /**
         * The l1,0 coefficients of the spherical harmonics
         */
        l10: Vector3;
        /**
         * The l1,1 coefficients of the spherical harmonics
         */
        l11: Vector3;
        /**
         * The l2,-2 coefficients of the spherical harmonics
         */
        l2_2: Vector3;
        /**
         * The l2,-1 coefficients of the spherical harmonics
         */
        l2_1: Vector3;
        /**
         * The l2,0 coefficients of the spherical harmonics
         */
        l20: Vector3;
        /**
         * The l2,1 coefficients of the spherical harmonics
         */
        l21: Vector3;
        /**
         * The l2,2 coefficients of the spherical harmonics
         */
        l22: Vector3;
        /**
         * Adds a light to the spherical harmonics
         * @param direction the direction of the light
         * @param color the color of the light
         * @param deltaSolidAngle the delta solid angle of the light
         */
        addLight(direction: Vector3, color: Color3, deltaSolidAngle: number): void;
        /**
         * Scales the spherical harmonics by the given amount
         * @param scale the amount to scale
         */
        scaleInPlace(scale: number): void;
        /**
         * Convert from incident radiance (Li) to irradiance (E) by applying convolution with the cosine-weighted hemisphere.
         *
         * ```
         * E_lm = A_l * L_lm
         * ```
         *
         * In spherical harmonics this convolution amounts to scaling factors for each frequency band.
         * This corresponds to equation 5 in "An Efficient Representation for Irradiance Environment Maps", where
         * the scaling factors are given in equation 9.
         */
        convertIncidentRadianceToIrradiance(): void;
        /**
         * Convert from irradiance to outgoing radiance for Lambertian BDRF, suitable for efficient shader evaluation.
         *
         * ```
         * L = (1/pi) * E * rho
         * ```
         *
         * This is done by an additional scale by 1/pi, so is a fairly trivial operation but important conceptually.
         */
        convertIrradianceToLambertianRadiance(): void;
        /**
         * Integrates the reconstruction coefficients directly in to the SH preventing further
         * required operations at run time.
         *
         * This is simply done by scaling back the SH with Ylm constants parameter.
         * The trigonometric part being applied by the shader at run time.
         */
        preScaleForRendering(): void;
        /**
         * Constructs a spherical harmonics from an array.
         * @param data defines the 9x3 coefficients (l00, l1-1, l10, l11, l2-2, l2-1, l20, l21, l22)
         * @returns the spherical harmonics
         */
        static FromArray(data: ArrayLike<ArrayLike<number>>): SphericalHarmonics;
        /**
         * Gets the spherical harmonics from polynomial
         * @param polynomial the spherical polynomial
         * @returns the spherical harmonics
         */
        static FromPolynomial(polynomial: SphericalPolynomial): SphericalHarmonics;
    }
    /**
     * Class representing spherical polynomial coefficients to the 3rd degree
     */
    export class SphericalPolynomial {
        private _harmonics;
        /**
         * The spherical harmonics used to create the polynomials.
         */
        get preScaledHarmonics(): SphericalHarmonics;
        /**
         * The x coefficients of the spherical polynomial
         */
        x: Vector3;
        /**
         * The y coefficients of the spherical polynomial
         */
        y: Vector3;
        /**
         * The z coefficients of the spherical polynomial
         */
        z: Vector3;
        /**
         * The xx coefficients of the spherical polynomial
         */
        xx: Vector3;
        /**
         * The yy coefficients of the spherical polynomial
         */
        yy: Vector3;
        /**
         * The zz coefficients of the spherical polynomial
         */
        zz: Vector3;
        /**
         * The xy coefficients of the spherical polynomial
         */
        xy: Vector3;
        /**
         * The yz coefficients of the spherical polynomial
         */
        yz: Vector3;
        /**
         * The zx coefficients of the spherical polynomial
         */
        zx: Vector3;
        /**
         * Adds an ambient color to the spherical polynomial
         * @param color the color to add
         */
        addAmbient(color: Color3): void;
        /**
         * Scales the spherical polynomial by the given amount
         * @param scale the amount to scale
         */
        scaleInPlace(scale: number): void;
        /**
         * Gets the spherical polynomial from harmonics
         * @param harmonics the spherical harmonics
         * @returns the spherical polynomial
         */
        static FromHarmonics(harmonics: SphericalHarmonics): SphericalPolynomial;
        /**
         * Constructs a spherical polynomial from an array.
         * @param data defines the 9x3 coefficients (x, y, z, xx, yy, zz, yz, zx, xy)
         * @returns the spherical polynomial
         */
        static FromArray(data: ArrayLike<ArrayLike<number>>): SphericalPolynomial;
    }
}
declare module "babylonjs/Misc/HighDynamicRange/panoramaToCubemap" {
    import { Nullable } from "babylonjs/types";
    /**
     * CubeMap information grouping all the data for each faces as well as the cubemap size.
     */
    export interface CubeMapInfo {
        /**
         * The pixel array for the front face.
         * This is stored in format, left to right, up to down format.
         */
        front: Nullable<ArrayBufferView>;
        /**
         * The pixel array for the back face.
         * This is stored in format, left to right, up to down format.
         */
        back: Nullable<ArrayBufferView>;
        /**
         * The pixel array for the left face.
         * This is stored in format, left to right, up to down format.
         */
        left: Nullable<ArrayBufferView>;
        /**
         * The pixel array for the right face.
         * This is stored in format, left to right, up to down format.
         */
        right: Nullable<ArrayBufferView>;
        /**
         * The pixel array for the up face.
         * This is stored in format, left to right, up to down format.
         */
        up: Nullable<ArrayBufferView>;
        /**
         * The pixel array for the down face.
         * This is stored in format, left to right, up to down format.
         */
        down: Nullable<ArrayBufferView>;
        /**
         * The size of the cubemap stored.
         *
         * Each faces will be size * size pixels.
         */
        size: number;
        /**
         * The format of the texture.
         *
         * RGBA, RGB.
         */
        format: number;
        /**
         * The type of the texture data.
         *
         * UNSIGNED_INT, FLOAT.
         */
        type: number;
        /**
         * Specifies whether the texture is in gamma space.
         */
        gammaSpace: boolean;
    }
    /**
     * Helper class useful to convert panorama picture to their cubemap representation in 6 faces.
     */
    export class PanoramaToCubeMapTools {
        private static FACE_LEFT;
        private static FACE_RIGHT;
        private static FACE_FRONT;
        private static FACE_BACK;
        private static FACE_DOWN;
        private static FACE_UP;
        /**
         * Converts a panorma stored in RGB right to left up to down format into a cubemap (6 faces).
         *
         * @param float32Array The source data.
         * @param inputWidth The width of the input panorama.
         * @param inputHeight The height of the input panorama.
         * @param size The willing size of the generated cubemap (each faces will be size * size pixels)
         * @return The cubemap data
         */
        static ConvertPanoramaToCubemap(float32Array: Float32Array, inputWidth: number, inputHeight: number, size: number): CubeMapInfo;
        private static CreateCubemapTexture;
        private static CalcProjectionSpherical;
    }
}
declare module "babylonjs/Misc/HighDynamicRange/cubemapToSphericalPolynomial" {
    import { SphericalPolynomial } from "babylonjs/Maths/sphericalPolynomial";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { Nullable } from "babylonjs/types";
    import { CubeMapInfo } from "babylonjs/Misc/HighDynamicRange/panoramaToCubemap";
    /**
     * Helper class dealing with the extraction of spherical polynomial dataArray
     * from a cube map.
     */
    export class CubeMapToSphericalPolynomialTools {
        private static FileFaces;
        /**
         * Converts a texture to the according Spherical Polynomial data.
         * This extracts the first 3 orders only as they are the only one used in the lighting.
         *
         * @param texture The texture to extract the information from.
         * @return The Spherical Polynomial data.
         */
        static ConvertCubeMapTextureToSphericalPolynomial(texture: BaseTexture): Nullable<SphericalPolynomial>;
        /**
         * Converts a cubemap to the according Spherical Polynomial data.
         * This extracts the first 3 orders only as they are the only one used in the lighting.
         *
         * @param cubeInfo The Cube map to extract the information from.
         * @return The Spherical Polynomial data.
         */
        static ConvertCubeMapToSphericalPolynomial(cubeInfo: CubeMapInfo): SphericalPolynomial;
    }
}
declare module "babylonjs/Materials/Textures/baseTexture.polynomial" {
    import { Nullable } from "babylonjs/types";
    import { SphericalPolynomial } from "babylonjs/Maths/sphericalPolynomial";
    module "babylonjs/Materials/Textures/baseTexture" {
        interface BaseTexture {
            /**
             * Get the polynomial representation of the texture data.
             * This is mainly use as a fast way to recover IBL Diffuse irradiance data.
             * @see https://learnopengl.com/PBR/IBL/Diffuse-irradiance
             */
            sphericalPolynomial: Nullable<SphericalPolynomial>;
        }
    }
}
declare module "babylonjs/Shaders/ShadersInclude/pbrFragmentDeclaration" {
    /** @hidden */
    export var pbrFragmentDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrUboDeclaration" {
    /** @hidden */
    export var pbrUboDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrFragmentExtraDeclaration" {
    /** @hidden */
    export var pbrFragmentExtraDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrFragmentSamplersDeclaration" {
    /** @hidden */
    export var pbrFragmentSamplersDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/subSurfaceScatteringFunctions" {
    /** @hidden */
    export var subSurfaceScatteringFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/importanceSampling" {
    /** @hidden */
    export var importanceSampling: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrHelperFunctions" {
    /** @hidden */
    export var pbrHelperFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/harmonicsFunctions" {
    /** @hidden */
    export var harmonicsFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrDirectLightingSetupFunctions" {
    /** @hidden */
    export var pbrDirectLightingSetupFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrDirectLightingFalloffFunctions" {
    /** @hidden */
    export var pbrDirectLightingFalloffFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBRDFFunctions" {
    /** @hidden */
    export var pbrBRDFFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/hdrFilteringFunctions" {
    /** @hidden */
    export var hdrFilteringFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrDirectLightingFunctions" {
    /** @hidden */
    export var pbrDirectLightingFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrIBLFunctions" {
    /** @hidden */
    export var pbrIBLFunctions: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockAlbedoOpacity" {
    /** @hidden */
    export var pbrBlockAlbedoOpacity: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockReflectivity" {
    /** @hidden */
    export var pbrBlockReflectivity: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockAmbientOcclusion" {
    /** @hidden */
    export var pbrBlockAmbientOcclusion: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockAlphaFresnel" {
    /** @hidden */
    export var pbrBlockAlphaFresnel: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockAnisotropic" {
    /** @hidden */
    export var pbrBlockAnisotropic: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockReflection" {
    /** @hidden */
    export var pbrBlockReflection: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockSheen" {
    /** @hidden */
    export var pbrBlockSheen: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockClearcoat" {
    /** @hidden */
    export var pbrBlockClearcoat: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockSubSurface" {
    /** @hidden */
    export var pbrBlockSubSurface: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockNormalGeometric" {
    /** @hidden */
    export var pbrBlockNormalGeometric: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockNormalFinal" {
    /** @hidden */
    export var pbrBlockNormalFinal: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockLightmapInit" {
    /** @hidden */
    export var pbrBlockLightmapInit: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockGeometryInfo" {
    /** @hidden */
    export var pbrBlockGeometryInfo: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockReflectance0" {
    /** @hidden */
    export var pbrBlockReflectance0: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockReflectance" {
    /** @hidden */
    export var pbrBlockReflectance: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockDirectLighting" {
    /** @hidden */
    export var pbrBlockDirectLighting: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockFinalLitComponents" {
    /** @hidden */
    export var pbrBlockFinalLitComponents: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockFinalUnlitComponents" {
    /** @hidden */
    export var pbrBlockFinalUnlitComponents: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockFinalColorComposition" {
    /** @hidden */
    export var pbrBlockFinalColorComposition: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrBlockImageProcessing" {
    /** @hidden */
    export var pbrBlockImageProcessing: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrDebug" {
    /** @hidden */
    export var pbrDebug: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/pbr.fragment" {
    import "babylonjs/Shaders/ShadersInclude/prePassDeclaration";
    import "babylonjs/Shaders/ShadersInclude/pbrFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/pbrUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/pbrFragmentExtraDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/pbrFragmentSamplersDeclaration";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingDeclaration";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/logDepthDeclaration";
    import "babylonjs/Shaders/ShadersInclude/fogFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    import "babylonjs/Shaders/ShadersInclude/subSurfaceScatteringFunctions";
    import "babylonjs/Shaders/ShadersInclude/importanceSampling";
    import "babylonjs/Shaders/ShadersInclude/pbrHelperFunctions";
    import "babylonjs/Shaders/ShadersInclude/imageProcessingFunctions";
    import "babylonjs/Shaders/ShadersInclude/shadowsFragmentFunctions";
    import "babylonjs/Shaders/ShadersInclude/harmonicsFunctions";
    import "babylonjs/Shaders/ShadersInclude/pbrDirectLightingSetupFunctions";
    import "babylonjs/Shaders/ShadersInclude/pbrDirectLightingFalloffFunctions";
    import "babylonjs/Shaders/ShadersInclude/pbrBRDFFunctions";
    import "babylonjs/Shaders/ShadersInclude/hdrFilteringFunctions";
    import "babylonjs/Shaders/ShadersInclude/pbrDirectLightingFunctions";
    import "babylonjs/Shaders/ShadersInclude/pbrIBLFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpFragmentMainFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpFragmentFunctions";
    import "babylonjs/Shaders/ShadersInclude/reflectionFunction";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockAlbedoOpacity";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockReflectivity";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockAmbientOcclusion";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockAlphaFresnel";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockAnisotropic";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockReflection";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockSheen";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockClearcoat";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockSubSurface";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneFragment";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockNormalGeometric";
    import "babylonjs/Shaders/ShadersInclude/bumpFragment";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockNormalFinal";
    import "babylonjs/Shaders/ShadersInclude/depthPrePass";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockLightmapInit";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockGeometryInfo";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockReflectance0";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockReflectance";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockDirectLighting";
    import "babylonjs/Shaders/ShadersInclude/lightFragment";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockFinalLitComponents";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockFinalUnlitComponents";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockFinalColorComposition";
    import "babylonjs/Shaders/ShadersInclude/logDepthFragment";
    import "babylonjs/Shaders/ShadersInclude/fogFragment";
    import "babylonjs/Shaders/ShadersInclude/pbrBlockImageProcessing";
    import "babylonjs/Shaders/ShadersInclude/pbrDebug";
    /** @hidden */
    export var pbrPixelShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/ShadersInclude/pbrVertexDeclaration" {
    /** @hidden */
    export var pbrVertexDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/pbr.vertex" {
    import "babylonjs/Shaders/ShadersInclude/pbrVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/pbrUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/helperFunctions";
    import "babylonjs/Shaders/ShadersInclude/bonesDeclaration";
    import "babylonjs/Shaders/ShadersInclude/instancesDeclaration";
    import "babylonjs/Shaders/ShadersInclude/prePassVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/harmonicsFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/fogVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/lightUboDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertexGlobalDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/logDepthDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertex";
    import "babylonjs/Shaders/ShadersInclude/instancesVertex";
    import "babylonjs/Shaders/ShadersInclude/bonesVertex";
    import "babylonjs/Shaders/ShadersInclude/prePassVertex";
    import "babylonjs/Shaders/ShadersInclude/bumpVertex";
    import "babylonjs/Shaders/ShadersInclude/clipPlaneVertex";
    import "babylonjs/Shaders/ShadersInclude/fogVertex";
    import "babylonjs/Shaders/ShadersInclude/shadowsVertex";
    import "babylonjs/Shaders/ShadersInclude/logDepthVertex";
    /** @hidden */
    export var pbrVertexShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Materials/PBR/pbrBaseMaterial" {
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Mesh } from "babylonjs/Meshes/mesh";
    import { IMaterialClearCoatDefines, PBRClearCoatConfiguration } from "babylonjs/Materials/PBR/pbrClearCoatConfiguration";
    import { IMaterialAnisotropicDefines, PBRAnisotropicConfiguration } from "babylonjs/Materials/PBR/pbrAnisotropicConfiguration";
    import { IMaterialBRDFDefines, PBRBRDFConfiguration } from "babylonjs/Materials/PBR/pbrBRDFConfiguration";
    import { IMaterialSheenDefines, PBRSheenConfiguration } from "babylonjs/Materials/PBR/pbrSheenConfiguration";
    import { IMaterialSubSurfaceDefines, PBRSubSurfaceConfiguration } from "babylonjs/Materials/PBR/pbrSubSurfaceConfiguration";
    import { PrePassConfiguration } from "babylonjs/Materials/prePassConfiguration";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { ImageProcessingConfiguration, IImageProcessingConfigurationDefines } from "babylonjs/Materials/imageProcessingConfiguration";
    import { Material, IMaterialCompilationOptions } from "babylonjs/Materials/material";
    import { MaterialDefines } from "babylonjs/Materials/materialDefines";
    import { PushMaterial } from "babylonjs/Materials/pushMaterial";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { IAnimatable } from "babylonjs/Animations/animatable.interface";
    import "babylonjs/Materials/Textures/baseTexture.polynomial";
    import "babylonjs/Shaders/pbr.fragment";
    import "babylonjs/Shaders/pbr.vertex";
    import { IMaterialDetailMapDefines, DetailMapConfiguration } from "babylonjs/Materials/material.detailMapConfiguration";
    import { PrePassRenderer } from "babylonjs/Rendering/prePassRenderer";
    /**
     * Manages the defines for the PBR Material.
     * @hidden
     */
    export class PBRMaterialDefines extends MaterialDefines implements IImageProcessingConfigurationDefines, IMaterialClearCoatDefines, IMaterialAnisotropicDefines, IMaterialBRDFDefines, IMaterialSheenDefines, IMaterialSubSurfaceDefines, IMaterialDetailMapDefines {
        PBR: boolean;
        NUM_SAMPLES: string;
        REALTIME_FILTERING: boolean;
        MAINUV1: boolean;
        MAINUV2: boolean;
        UV1: boolean;
        UV2: boolean;
        ALBEDO: boolean;
        GAMMAALBEDO: boolean;
        ALBEDODIRECTUV: number;
        VERTEXCOLOR: boolean;
        DETAIL: boolean;
        DETAILDIRECTUV: number;
        DETAIL_NORMALBLENDMETHOD: number;
        AMBIENT: boolean;
        AMBIENTDIRECTUV: number;
        AMBIENTINGRAYSCALE: boolean;
        OPACITY: boolean;
        VERTEXALPHA: boolean;
        OPACITYDIRECTUV: number;
        OPACITYRGB: boolean;
        ALPHATEST: boolean;
        DEPTHPREPASS: boolean;
        ALPHABLEND: boolean;
        ALPHAFROMALBEDO: boolean;
        ALPHATESTVALUE: string;
        SPECULAROVERALPHA: boolean;
        RADIANCEOVERALPHA: boolean;
        ALPHAFRESNEL: boolean;
        LINEARALPHAFRESNEL: boolean;
        PREMULTIPLYALPHA: boolean;
        EMISSIVE: boolean;
        EMISSIVEDIRECTUV: number;
        REFLECTIVITY: boolean;
        REFLECTIVITYDIRECTUV: number;
        SPECULARTERM: boolean;
        MICROSURFACEFROMREFLECTIVITYMAP: boolean;
        MICROSURFACEAUTOMATIC: boolean;
        LODBASEDMICROSFURACE: boolean;
        MICROSURFACEMAP: boolean;
        MICROSURFACEMAPDIRECTUV: number;
        METALLICWORKFLOW: boolean;
        ROUGHNESSSTOREINMETALMAPALPHA: boolean;
        ROUGHNESSSTOREINMETALMAPGREEN: boolean;
        METALLNESSSTOREINMETALMAPBLUE: boolean;
        AOSTOREINMETALMAPRED: boolean;
        METALLIC_REFLECTANCE: boolean;
        METALLIC_REFLECTANCEDIRECTUV: number;
        ENVIRONMENTBRDF: boolean;
        ENVIRONMENTBRDF_RGBD: boolean;
        NORMAL: boolean;
        TANGENT: boolean;
        BUMP: boolean;
        BUMPDIRECTUV: number;
        OBJECTSPACE_NORMALMAP: boolean;
        PARALLAX: boolean;
        PARALLAXOCCLUSION: boolean;
        NORMALXYSCALE: boolean;
        LIGHTMAP: boolean;
        LIGHTMAPDIRECTUV: number;
        USELIGHTMAPASSHADOWMAP: boolean;
        GAMMALIGHTMAP: boolean;
        RGBDLIGHTMAP: boolean;
        REFLECTION: boolean;
        REFLECTIONMAP_3D: boolean;
        REFLECTIONMAP_SPHERICAL: boolean;
        REFLECTIONMAP_PLANAR: boolean;
        REFLECTIONMAP_CUBIC: boolean;
        USE_LOCAL_REFLECTIONMAP_CUBIC: boolean;
        REFLECTIONMAP_PROJECTION: boolean;
        REFLECTIONMAP_SKYBOX: boolean;
        REFLECTIONMAP_EXPLICIT: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR_FIXED: boolean;
        REFLECTIONMAP_MIRROREDEQUIRECTANGULAR_FIXED: boolean;
        INVERTCUBICMAP: boolean;
        USESPHERICALFROMREFLECTIONMAP: boolean;
        USEIRRADIANCEMAP: boolean;
        SPHERICAL_HARMONICS: boolean;
        USESPHERICALINVERTEX: boolean;
        REFLECTIONMAP_OPPOSITEZ: boolean;
        LODINREFLECTIONALPHA: boolean;
        GAMMAREFLECTION: boolean;
        RGBDREFLECTION: boolean;
        LINEARSPECULARREFLECTION: boolean;
        RADIANCEOCCLUSION: boolean;
        HORIZONOCCLUSION: boolean;
        INSTANCES: boolean;
        THIN_INSTANCES: boolean;
        PREPASS: boolean;
        PREPASS_IRRADIANCE: boolean;
        PREPASS_IRRADIANCE_INDEX: number;
        PREPASS_ALBEDO: boolean;
        PREPASS_ALBEDO_INDEX: number;
        PREPASS_DEPTHNORMAL: boolean;
        PREPASS_DEPTHNORMAL_INDEX: number;
        PREPASS_POSITION: boolean;
        PREPASS_POSITION_INDEX: number;
        PREPASS_VELOCITY: boolean;
        PREPASS_VELOCITY_INDEX: number;
        PREPASS_REFLECTIVITY: boolean;
        PREPASS_REFLECTIVITY_INDEX: number;
        SCENE_MRT_COUNT: number;
        NUM_BONE_INFLUENCERS: number;
        BonesPerMesh: number;
        BONETEXTURE: boolean;
        BONES_VELOCITY_ENABLED: boolean;
        NONUNIFORMSCALING: boolean;
        MORPHTARGETS: boolean;
        MORPHTARGETS_NORMAL: boolean;
        MORPHTARGETS_TANGENT: boolean;
        MORPHTARGETS_UV: boolean;
        NUM_MORPH_INFLUENCERS: number;
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        TONEMAPPING_ACES: boolean;
        CONTRAST: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
        EXPOSURE: boolean;
        MULTIVIEW: boolean;
        USEPHYSICALLIGHTFALLOFF: boolean;
        USEGLTFLIGHTFALLOFF: boolean;
        TWOSIDEDLIGHTING: boolean;
        SHADOWFLOAT: boolean;
        CLIPPLANE: boolean;
        CLIPPLANE2: boolean;
        CLIPPLANE3: boolean;
        CLIPPLANE4: boolean;
        CLIPPLANE5: boolean;
        CLIPPLANE6: boolean;
        POINTSIZE: boolean;
        FOG: boolean;
        LOGARITHMICDEPTH: boolean;
        FORCENORMALFORWARD: boolean;
        SPECULARAA: boolean;
        CLEARCOAT: boolean;
        CLEARCOAT_DEFAULTIOR: boolean;
        CLEARCOAT_TEXTURE: boolean;
        CLEARCOAT_TEXTURE_ROUGHNESS: boolean;
        CLEARCOAT_TEXTUREDIRECTUV: number;
        CLEARCOAT_TEXTURE_ROUGHNESSDIRECTUV: number;
        CLEARCOAT_USE_ROUGHNESS_FROM_MAINTEXTURE: boolean;
        CLEARCOAT_TEXTURE_ROUGHNESS_IDENTICAL: boolean;
        CLEARCOAT_BUMP: boolean;
        CLEARCOAT_BUMPDIRECTUV: number;
        CLEARCOAT_REMAP_F0: boolean;
        CLEARCOAT_TINT: boolean;
        CLEARCOAT_TINT_TEXTURE: boolean;
        CLEARCOAT_TINT_TEXTUREDIRECTUV: number;
        ANISOTROPIC: boolean;
        ANISOTROPIC_TEXTURE: boolean;
        ANISOTROPIC_TEXTUREDIRECTUV: number;
        BRDF_V_HEIGHT_CORRELATED: boolean;
        MS_BRDF_ENERGY_CONSERVATION: boolean;
        SPECULAR_GLOSSINESS_ENERGY_CONSERVATION: boolean;
        SHEEN: boolean;
        SHEEN_TEXTURE: boolean;
        SHEEN_TEXTURE_ROUGHNESS: boolean;
        SHEEN_TEXTUREDIRECTUV: number;
        SHEEN_TEXTURE_ROUGHNESSDIRECTUV: number;
        SHEEN_LINKWITHALBEDO: boolean;
        SHEEN_ROUGHNESS: boolean;
        SHEEN_ALBEDOSCALING: boolean;
        SHEEN_USE_ROUGHNESS_FROM_MAINTEXTURE: boolean;
        SHEEN_TEXTURE_ROUGHNESS_IDENTICAL: boolean;
        SUBSURFACE: boolean;
        SS_REFRACTION: boolean;
        SS_TRANSLUCENCY: boolean;
        SS_SCATTERING: boolean;
        SS_THICKNESSANDMASK_TEXTURE: boolean;
        SS_THICKNESSANDMASK_TEXTUREDIRECTUV: number;
        SS_REFRACTIONMAP_3D: boolean;
        SS_REFRACTIONMAP_OPPOSITEZ: boolean;
        SS_LODINREFRACTIONALPHA: boolean;
        SS_GAMMAREFRACTION: boolean;
        SS_RGBDREFRACTION: boolean;
        SS_LINEARSPECULARREFRACTION: boolean;
        SS_LINKREFRACTIONTOTRANSPARENCY: boolean;
        SS_ALBEDOFORREFRACTIONTINT: boolean;
        SS_MASK_FROM_THICKNESS_TEXTURE: boolean;
        SS_MASK_FROM_THICKNESS_TEXTURE_GLTF: boolean;
        UNLIT: boolean;
        DEBUGMODE: number;
        /**
         * Initializes the PBR Material defines.
         */
        constructor();
        /**
         * Resets the PBR Material defines.
         */
        reset(): void;
    }
    /**
     * The Physically based material base class of BJS.
     *
     * This offers the main features of a standard PBR material.
     * For more information, please refer to the documentation :
     * https://doc.babylonjs.com/how_to/physically_based_rendering
     */
    export abstract class PBRBaseMaterial extends PushMaterial {
        /**
         * PBRMaterialTransparencyMode: No transparency mode, Alpha channel is not use.
         */
        static readonly PBRMATERIAL_OPAQUE: number;
        /**
         * PBRMaterialTransparencyMode: Alpha Test mode, pixel are discarded below a certain threshold defined by the alpha cutoff value.
         */
        static readonly PBRMATERIAL_ALPHATEST: number;
        /**
         * PBRMaterialTransparencyMode: Pixels are blended (according to the alpha mode) with the already drawn pixels in the current frame buffer.
         */
        static readonly PBRMATERIAL_ALPHABLEND: number;
        /**
         * PBRMaterialTransparencyMode: Pixels are blended (according to the alpha mode) with the already drawn pixels in the current frame buffer.
         * They are also discarded below the alpha cutoff threshold to improve performances.
         */
        static readonly PBRMATERIAL_ALPHATESTANDBLEND: number;
        /**
         * Defines the default value of how much AO map is occluding the analytical lights
         * (point spot...).
         */
        static DEFAULT_AO_ON_ANALYTICAL_LIGHTS: number;
        /**
         * PBRMaterialLightFalloff Physical: light is falling off following the inverse squared distance law.
         */
        static readonly LIGHTFALLOFF_PHYSICAL: number;
        /**
         * PBRMaterialLightFalloff gltf: light is falling off as described in the gltf moving to PBR document
         * to enhance interoperability with other engines.
         */
        static readonly LIGHTFALLOFF_GLTF: number;
        /**
         * PBRMaterialLightFalloff Standard: light is falling off like in the standard material
         * to enhance interoperability with other materials.
         */
        static readonly LIGHTFALLOFF_STANDARD: number;
        /**
         * Intensity of the direct lights e.g. the four lights available in your scene.
         * This impacts both the direct diffuse and specular highlights.
         */
        protected _directIntensity: number;
        /**
         * Intensity of the emissive part of the material.
         * This helps controlling the emissive effect without modifying the emissive color.
         */
        protected _emissiveIntensity: number;
        /**
         * Intensity of the environment e.g. how much the environment will light the object
         * either through harmonics for rough material or through the refelction for shiny ones.
         */
        protected _environmentIntensity: number;
        /**
         * This is a special control allowing the reduction of the specular highlights coming from the
         * four lights of the scene. Those highlights may not be needed in full environment lighting.
         */
        protected _specularIntensity: number;
        /**
         * This stores the direct, emissive, environment, and specular light intensities into a Vector4.
         */
        private _lightingInfos;
        /**
         * Debug Control allowing disabling the bump map on this material.
         */
        protected _disableBumpMap: boolean;
        /**
         * AKA Diffuse Texture in standard nomenclature.
         */
        protected _albedoTexture: Nullable<BaseTexture>;
        /**
         * AKA Occlusion Texture in other nomenclature.
         */
        protected _ambientTexture: Nullable<BaseTexture>;
        /**
         * AKA Occlusion Texture Intensity in other nomenclature.
         */
        protected _ambientTextureStrength: number;
        /**
         * Defines how much the AO map is occluding the analytical lights (point spot...).
         * 1 means it completely occludes it
         * 0 mean it has no impact
         */
        protected _ambientTextureImpactOnAnalyticalLights: number;
        /**
         * Stores the alpha values in a texture.
         */
        protected _opacityTexture: Nullable<BaseTexture>;
        /**
         * Stores the reflection values in a texture.
         */
        protected _reflectionTexture: Nullable<BaseTexture>;
        /**
         * Stores the emissive values in a texture.
         */
        protected _emissiveTexture: Nullable<BaseTexture>;
        /**
         * AKA Specular texture in other nomenclature.
         */
        protected _reflectivityTexture: Nullable<BaseTexture>;
        /**
         * Used to switch from specular/glossiness to metallic/roughness workflow.
         */
        protected _metallicTexture: Nullable<BaseTexture>;
        /**
         * Specifies the metallic scalar of the metallic/roughness workflow.
         * Can also be used to scale the metalness values of the metallic texture.
         */
        protected _metallic: Nullable<number>;
        /**
         * Specifies the roughness scalar of the metallic/roughness workflow.
         * Can also be used to scale the roughness values of the metallic texture.
         */
        protected _roughness: Nullable<number>;
        /**
         * In metallic workflow, specifies an F0 factor to help configuring the material F0.
         * By default the indexOfrefraction is used to compute F0;
         *
         * This is used as a factor against the default reflectance at normal incidence to tweak it.
         *
         * F0 = defaultF0 * metallicF0Factor * metallicReflectanceColor;
         * F90 = metallicReflectanceColor;
         */
        protected _metallicF0Factor: number;
        /**
         * In metallic workflow, specifies an F90 color to help configuring the material F90.
         * By default the F90 is always 1;
         *
         * Please note that this factor is also used as a factor against the default reflectance at normal incidence.
         *
         * F0 = defaultF0 * metallicF0Factor * metallicReflectanceColor
         * F90 = metallicReflectanceColor;
         */
        protected _metallicReflectanceColor: Color3;
        /**
         * Defines to store metallicReflectanceColor in RGB and metallicF0Factor in A
         * This is multiply against the scalar values defined in the material.
         */
        protected _metallicReflectanceTexture: Nullable<BaseTexture>;
        /**
         * Used to enable roughness/glossiness fetch from a separate channel depending on the current mode.
         * Gray Scale represents roughness in metallic mode and glossiness in specular mode.
         */
        protected _microSurfaceTexture: Nullable<BaseTexture>;
        /**
         * Stores surface normal data used to displace a mesh in a texture.
         */
        protected _bumpTexture: Nullable<BaseTexture>;
        /**
         * Stores the pre-calculated light information of a mesh in a texture.
         */
        protected _lightmapTexture: Nullable<BaseTexture>;
        /**
         * The color of a material in ambient lighting.
         */
        protected _ambientColor: Color3;
        /**
         * AKA Diffuse Color in other nomenclature.
         */
        protected _albedoColor: Color3;
        /**
         * AKA Specular Color in other nomenclature.
         */
        protected _reflectivityColor: Color3;
        /**
         * The color applied when light is reflected from a material.
         */
        protected _reflectionColor: Color3;
        /**
         * The color applied when light is emitted from a material.
         */
        protected _emissiveColor: Color3;
        /**
         * AKA Glossiness in other nomenclature.
         */
        protected _microSurface: number;
        /**
         * Specifies that the material will use the light map as a show map.
         */
        protected _useLightmapAsShadowmap: boolean;
        /**
         * This parameters will enable/disable Horizon occlusion to prevent normal maps to look shiny when the normal
         * makes the reflect vector face the model (under horizon).
         */
        protected _useHorizonOcclusion: boolean;
        /**
         * This parameters will enable/disable radiance occlusion by preventing the radiance to lit
         * too much the area relying on ambient texture to define their ambient occlusion.
         */
        protected _useRadianceOcclusion: boolean;
        /**
         * Specifies that the alpha is coming form the albedo channel alpha channel for alpha blending.
         */
        protected _useAlphaFromAlbedoTexture: boolean;
        /**
         * Specifies that the material will keeps the specular highlights over a transparent surface (only the most limunous ones).
         * A car glass is a good exemple of that. When sun reflects on it you can not see what is behind.
         */
        protected _useSpecularOverAlpha: boolean;
        /**
         * Specifies if the reflectivity texture contains the glossiness information in its alpha channel.
         */
        protected _useMicroSurfaceFromReflectivityMapAlpha: boolean;
        /**
         * Specifies if the metallic texture contains the roughness information in its alpha channel.
         */
        protected _useRoughnessFromMetallicTextureAlpha: boolean;
        /**
         * Specifies if the metallic texture contains the roughness information in its green channel.
         */
        protected _useRoughnessFromMetallicTextureGreen: boolean;
        /**
         * Specifies if the metallic texture contains the metallness information in its blue channel.
         */
        protected _useMetallnessFromMetallicTextureBlue: boolean;
        /**
         * Specifies if the metallic texture contains the ambient occlusion information in its red channel.
         */
        protected _useAmbientOcclusionFromMetallicTextureRed: boolean;
        /**
         * Specifies if the ambient texture contains the ambient occlusion information in its red channel only.
         */
        protected _useAmbientInGrayScale: boolean;
        /**
         * In case the reflectivity map does not contain the microsurface information in its alpha channel,
         * The material will try to infer what glossiness each pixel should be.
         */
        protected _useAutoMicroSurfaceFromReflectivityMap: boolean;
        /**
         * Defines the  falloff type used in this material.
         * It by default is Physical.
         */
        protected _lightFalloff: number;
        /**
         * Specifies that the material will keeps the reflection highlights over a transparent surface (only the most limunous ones).
         * A car glass is a good exemple of that. When the street lights reflects on it you can not see what is behind.
         */
        protected _useRadianceOverAlpha: boolean;
        /**
         * Allows using an object space normal map (instead of tangent space).
         */
        protected _useObjectSpaceNormalMap: boolean;
        /**
         * Allows using the bump map in parallax mode.
         */
        protected _useParallax: boolean;
        /**
         * Allows using the bump map in parallax occlusion mode.
         */
        protected _useParallaxOcclusion: boolean;
        /**
         * Controls the scale bias of the parallax mode.
         */
        protected _parallaxScaleBias: number;
        /**
         * If sets to true, disables all the lights affecting the material.
         */
        protected _disableLighting: boolean;
        /**
         * Number of Simultaneous lights allowed on the material.
         */
        protected _maxSimultaneousLights: number;
        /**
         * If sets to true, x component of normal map value will be inverted (x = 1.0 - x).
         */
        protected _invertNormalMapX: boolean;
        /**
         * If sets to true, y component of normal map value will be inverted (y = 1.0 - y).
         */
        protected _invertNormalMapY: boolean;
        /**
         * If sets to true and backfaceCulling is false, normals will be flipped on the backside.
         */
        protected _twoSidedLighting: boolean;
        /**
         * Defines the alpha limits in alpha test mode.
         */
        protected _alphaCutOff: number;
        /**
         * Enforces alpha test in opaque or blend mode in order to improve the performances of some situations.
         */
        protected _forceAlphaTest: boolean;
        /**
         * A fresnel is applied to the alpha of the model to ensure grazing angles edges are not alpha tested.
         * And/Or occlude the blended part. (alpha is converted to gamma to compute the fresnel)
         */
        protected _useAlphaFresnel: boolean;
        /**
         * A fresnel is applied to the alpha of the model to ensure grazing angles edges are not alpha tested.
         * And/Or occlude the blended part. (alpha stays linear to compute the fresnel)
         */
        protected _useLinearAlphaFresnel: boolean;
        /**
         * Specifies the environment BRDF texture used to comput the scale and offset roughness values
         * from cos thetav and roughness:
         * http://blog.selfshadow.com/publications/s2013-shading-course/karis/s2013_pbs_epic_notes_v2.pdf
         */
        protected _environmentBRDFTexture: Nullable<BaseTexture>;
        /**
         * Force the shader to compute irradiance in the fragment shader in order to take bump in account.
         */
        protected _forceIrradianceInFragment: boolean;
        private _realTimeFiltering;
        /**
         * Enables realtime filtering on the texture.
         */
        get realTimeFiltering(): boolean;
        set realTimeFiltering(b: boolean);
        private _realTimeFilteringQuality;
        /**
         * Quality switch for realtime filtering
         */
        get realTimeFilteringQuality(): number;
        set realTimeFilteringQuality(n: number);
        /**
         * Can this material render to several textures at once
         */
        get canRenderToMRT(): boolean;
        /**
         * Force normal to face away from face.
         */
        protected _forceNormalForward: boolean;
        /**
         * Enables specular anti aliasing in the PBR shader.
         * It will both interacts on the Geometry for analytical and IBL lighting.
         * It also prefilter the roughness map based on the bump values.
         */
        protected _enableSpecularAntiAliasing: boolean;
        /**
         * Default configuration related to image processing available in the PBR Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the PBR Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>): void;
        /**
         * Stores the available render targets.
         */
        private _renderTargets;
        /**
         * Sets the global ambient color for the material used in lighting calculations.
         */
        private _globalAmbientColor;
        /**
         * Enables the use of logarithmic depth buffers, which is good for wide depth buffers.
         */
        private _useLogarithmicDepth;
        /**
         * If set to true, no lighting calculations will be applied.
         */
        private _unlit;
        private _debugMode;
        /**
         * @hidden
         * This is reserved for the inspector.
         * Defines the material debug mode.
         * It helps seeing only some components of the material while troubleshooting.
         */
        debugMode: number;
        /**
         * @hidden
         * This is reserved for the inspector.
         * Specify from where on screen the debug mode should start.
         * The value goes from -1 (full screen) to 1 (not visible)
         * It helps with side by side comparison against the final render
         * This defaults to -1
         */
        private debugLimit;
        /**
         * @hidden
         * This is reserved for the inspector.
         * As the default viewing range might not be enough (if the ambient is really small for instance)
         * You can use the factor to better multiply the final value.
         */
        private debugFactor;
        /**
         * Defines the clear coat layer parameters for the material.
         */
        readonly clearCoat: PBRClearCoatConfiguration;
        /**
         * Defines the anisotropic parameters for the material.
         */
        readonly anisotropy: PBRAnisotropicConfiguration;
        /**
         * Defines the BRDF parameters for the material.
         */
        readonly brdf: PBRBRDFConfiguration;
        /**
         * Defines the Sheen parameters for the material.
         */
        readonly sheen: PBRSheenConfiguration;
        /**
         * Defines the SubSurface parameters for the material.
         */
        readonly subSurface: PBRSubSurfaceConfiguration;
        /**
         * Defines additionnal PrePass parameters for the material.
         */
        readonly prePassConfiguration: PrePassConfiguration;
        /**
         * Defines the detail map parameters for the material.
         */
        readonly detailMap: DetailMapConfiguration;
        protected _rebuildInParallel: boolean;
        /**
         * Instantiates a new PBRMaterial instance.
         *
         * @param name The material name
         * @param scene The scene the material will be use in.
         */
        constructor(name: string, scene: Scene);
        /**
         * Gets a boolean indicating that current material needs to register RTT
         */
        get hasRenderTargetTextures(): boolean;
        /**
         * Gets the name of the material class.
         */
        getClassName(): string;
        /**
         * Enabled the use of logarithmic depth buffers, which is good for wide depth buffers.
         */
        get useLogarithmicDepth(): boolean;
        /**
         * Enabled the use of logarithmic depth buffers, which is good for wide depth buffers.
         */
        set useLogarithmicDepth(value: boolean);
        /**
         * Returns true if alpha blending should be disabled.
         */
        protected get _disableAlphaBlending(): boolean;
        /**
         * Specifies whether or not this material should be rendered in alpha blend mode.
         */
        needAlphaBlending(): boolean;
        /**
         * Specifies whether or not this material should be rendered in alpha test mode.
         */
        needAlphaTesting(): boolean;
        /**
         * Specifies whether or not the alpha value of the albedo texture should be used for alpha blending.
         */
        protected _shouldUseAlphaFromAlbedoTexture(): boolean;
        /**
         * Specifies whether or not there is a usable alpha channel for transparency.
         */
        protected _hasAlphaChannel(): boolean;
        /**
         * Gets the texture used for the alpha test.
         */
        getAlphaTestTexture(): Nullable<BaseTexture>;
        /**
         * Specifies that the submesh is ready to be used.
         * @param mesh - BJS mesh.
         * @param subMesh - A submesh of the BJS mesh.  Used to check if it is ready.
         * @param useInstances - Specifies that instances should be used.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: SubMesh, useInstances?: boolean): boolean;
        /**
         * Specifies if the material uses metallic roughness workflow.
         * @returns boolean specifiying if the material uses metallic roughness workflow.
        */
        isMetallicWorkflow(): boolean;
        private _prepareEffect;
        private _prepareDefines;
        /**
         * Force shader compilation
         */
        forceCompilation(mesh: AbstractMesh, onCompiled?: (material: Material) => void, options?: Partial<IMaterialCompilationOptions>): void;
        /**
         * Initializes the uniform buffer layout for the shader.
         */
        buildUniformLayout(): void;
        /**
         * Unbinds the material from the mesh
         */
        unbind(): void;
        /**
         * Binds the submesh data.
         * @param world - The world matrix.
         * @param mesh - The BJS mesh.
         * @param subMesh - A submesh of the BJS mesh.
         */
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        /**
         * Returns the animatable textures.
         * @returns - Array of animatable textures.
         */
        getAnimatables(): IAnimatable[];
        /**
         * Returns the texture used for reflections.
         * @returns - Reflection texture if present.  Otherwise, returns the environment texture.
         */
        private _getReflectionTexture;
        /**
         * Returns an array of the actively used textures.
         * @returns - Array of BaseTextures
         */
        getActiveTextures(): BaseTexture[];
        /**
         * Checks to see if a texture is used in the material.
         * @param texture - Base texture to use.
         * @returns - Boolean specifying if a texture is used in the material.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Sets the required values to the prepass renderer.
         * @param prePassRenderer defines the prepass renderer to setup
         */
        setPrePassRenderer(prePassRenderer: PrePassRenderer): boolean;
        /**
         * Disposes the resources of the material.
         * @param forceDisposeEffect - Forces the disposal of effects.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
    }
}
declare module "babylonjs/Materials/PBR/pbrMaterial" {
    import { Nullable } from "babylonjs/types";
    import { Scene } from "babylonjs/scene";
    import { Color3 } from "babylonjs/Maths/math.color";
    import { ImageProcessingConfiguration } from "babylonjs/Materials/imageProcessingConfiguration";
    import { ColorCurves } from "babylonjs/Materials/colorCurves";
    import { BaseTexture } from "babylonjs/Materials/Textures/baseTexture";
    import { PBRBaseMaterial } from "babylonjs/Materials/PBR/pbrBaseMaterial";
    /**
     * The Physically based material of BJS.
     *
     * This offers the main features of a standard PBR material.
     * For more information, please refer to the documentation :
     * https://doc.babylonjs.com/how_to/physically_based_rendering
     */
    export class PBRMaterial extends PBRBaseMaterial {
        /**
         * PBRMaterialTransparencyMode: No transparency mode, Alpha channel is not use.
         */
        static readonly PBRMATERIAL_OPAQUE: number;
        /**
         * PBRMaterialTransparencyMode: Alpha Test mode, pixel are discarded below a certain threshold defined by the alpha cutoff value.
         */
        static readonly PBRMATERIAL_ALPHATEST: number;
        /**
         * PBRMaterialTransparencyMode: Pixels are blended (according to the alpha mode) with the already drawn pixels in the current frame buffer.
         */
        static readonly PBRMATERIAL_ALPHABLEND: number;
        /**
         * PBRMaterialTransparencyMode: Pixels are blended (according to the alpha mode) with the already drawn pixels in the current frame buffer.
         * They are also discarded below the alpha cutoff threshold to improve performances.
         */
        static readonly PBRMATERIAL_ALPHATESTANDBLEND: number;
        /**
         * Defines the default value of how much AO map is occluding the analytical lights
         * (point spot...).
         */
        static DEFAULT_AO_ON_ANALYTICAL_LIGHTS: number;
        /**
         * Intensity of the direct lights e.g. the four lights available in your scene.
         * This impacts both the direct diffuse and specular highlights.
         */
        directIntensity: number;
        /**
         * Intensity of the emissive part of the material.
         * This helps controlling the emissive effect without modifying the emissive color.
         */
        emissiveIntensity: number;
        /**
         * Intensity of the environment e.g. how much the environment will light the object
         * either through harmonics for rough material or through the refelction for shiny ones.
         */
        environmentIntensity: number;
        /**
         * This is a special control allowing the reduction of the specular highlights coming from the
         * four lights of the scene. Those highlights may not be needed in full environment lighting.
         */
        specularIntensity: number;
        /**
         * Debug Control allowing disabling the bump map on this material.
         */
        disableBumpMap: boolean;
        /**
         * AKA Diffuse Texture in standard nomenclature.
         */
        albedoTexture: BaseTexture;
        /**
         * AKA Occlusion Texture in other nomenclature.
         */
        ambientTexture: BaseTexture;
        /**
         * AKA Occlusion Texture Intensity in other nomenclature.
         */
        ambientTextureStrength: number;
        /**
         * Defines how much the AO map is occluding the analytical lights (point spot...).
         * 1 means it completely occludes it
         * 0 mean it has no impact
         */
        ambientTextureImpactOnAnalyticalLights: number;
        /**
         * Stores the alpha values in a texture. Use luminance if texture.getAlphaFromRGB is true.
         */
        opacityTexture: BaseTexture;
        /**
         * Stores the reflection values in a texture.
         */
        reflectionTexture: Nullable<BaseTexture>;
        /**
         * Stores the emissive values in a texture.
         */
        emissiveTexture: BaseTexture;
        /**
         * AKA Specular texture in other nomenclature.
         */
        reflectivityTexture: BaseTexture;
        /**
         * Used to switch from specular/glossiness to metallic/roughness workflow.
         */
        metallicTexture: BaseTexture;
        /**
         * Specifies the metallic scalar of the metallic/roughness workflow.
         * Can also be used to scale the metalness values of the metallic texture.
         */
        metallic: Nullable<number>;
        /**
         * Specifies the roughness scalar of the metallic/roughness workflow.
         * Can also be used to scale the roughness values of the metallic texture.
         */
        roughness: Nullable<number>;
        /**
         * In metallic workflow, specifies an F0 factor to help configuring the material F0.
         * By default the indexOfrefraction is used to compute F0;
         *
         * This is used as a factor against the default reflectance at normal incidence to tweak it.
         *
         * F0 = defaultF0 * metallicF0Factor * metallicReflectanceColor;
         * F90 = metallicReflectanceColor;
         */
        metallicF0Factor: number;
        /**
         * In metallic workflow, specifies an F90 color to help configuring the material F90.
         * By default the F90 is always 1;
         *
         * Please note that this factor is also used as a factor against the default reflectance at normal incidence.
         *
         * F0 = defaultF0 * metallicF0Factor * metallicReflectanceColor
         * F90 = metallicReflectanceColor;
         */
        metallicReflectanceColor: Color3;
        /**
         * Defines to store metallicReflectanceColor in RGB and metallicF0Factor in A
         * This is multiply against the scalar values defined in the material.
         */
        metallicReflectanceTexture: Nullable<BaseTexture>;
        /**
         * Used to enable roughness/glossiness fetch from a separate channel depending on the current mode.
         * Gray Scale represents roughness in metallic mode and glossiness in specular mode.
         */
        microSurfaceTexture: BaseTexture;
        /**
         * Stores surface normal data used to displace a mesh in a texture.
         */
        bumpTexture: BaseTexture;
        /**
         * Stores the pre-calculated light information of a mesh in a texture.
         */
        lightmapTexture: BaseTexture;
        /**
         * Stores the refracted light information in a texture.
         */
        get refractionTexture(): Nullable<BaseTexture>;
        set refractionTexture(value: Nullable<BaseTexture>);
        /**
         * The color of a material in ambient lighting.
         */
        ambientColor: Color3;
        /**
         * AKA Diffuse Color in other nomenclature.
         */
        albedoColor: Color3;
        /**
         * AKA Specular Color in other nomenclature.
         */
        reflectivityColor: Color3;
        /**
         * The color reflected from the material.
         */
        reflectionColor: Color3;
        /**
         * The color emitted from the material.
         */
        emissiveColor: Color3;
        /**
         * AKA Glossiness in other nomenclature.
         */
        microSurface: number;
        /**
         * Index of refraction of the material base layer.
         * https://en.wikipedia.org/wiki/List_of_refractive_indices
         *
         * This does not only impact refraction but also the Base F0 of Dielectric Materials.
         *
         * From dielectric fresnel rules: F0 = square((iorT - iorI) / (iorT + iorI))
         */
        get indexOfRefraction(): number;
        set indexOfRefraction(value: number);
        /**
         * Controls if refraction needs to be inverted on Y. This could be useful for procedural texture.
         */
        get invertRefractionY(): boolean;
        set invertRefractionY(value: boolean);
        /**
         * This parameters will make the material used its opacity to control how much it is refracting aginst not.
         * Materials half opaque for instance using refraction could benefit from this control.
         */
        get linkRefractionWithTransparency(): boolean;
        set linkRefractionWithTransparency(value: boolean);
        /**
         * If true, the light map contains occlusion information instead of lighting info.
         */
        useLightmapAsShadowmap: boolean;
        /**
         * Specifies that the alpha is coming form the albedo channel alpha channel for alpha blending.
         */
        useAlphaFromAlbedoTexture: boolean;
        /**
         * Enforces alpha test in opaque or blend mode in order to improve the performances of some situations.
         */
        forceAlphaTest: boolean;
        /**
         * Defines the alpha limits in alpha test mode.
         */
        alphaCutOff: number;
        /**
         * Specifies that the material will keep the specular highlights over a transparent surface (only the most limunous ones).
         * A car glass is a good exemple of that. When sun reflects on it you can not see what is behind.
         */
        useSpecularOverAlpha: boolean;
        /**
         * Specifies if the reflectivity texture contains the glossiness information in its alpha channel.
         */
        useMicroSurfaceFromReflectivityMapAlpha: boolean;
        /**
         * Specifies if the metallic texture contains the roughness information in its alpha channel.
         */
        useRoughnessFromMetallicTextureAlpha: boolean;
        /**
         * Specifies if the metallic texture contains the roughness information in its green channel.
         */
        useRoughnessFromMetallicTextureGreen: boolean;
        /**
         * Specifies if the metallic texture contains the metallness information in its blue channel.
         */
        useMetallnessFromMetallicTextureBlue: boolean;
        /**
         * Specifies if the metallic texture contains the ambient occlusion information in its red channel.
         */
        useAmbientOcclusionFromMetallicTextureRed: boolean;
        /**
         * Specifies if the ambient texture contains the ambient occlusion information in its red channel only.
         */
        useAmbientInGrayScale: boolean;
        /**
         * In case the reflectivity map does not contain the microsurface information in its alpha channel,
         * The material will try to infer what glossiness each pixel should be.
         */
        useAutoMicroSurfaceFromReflectivityMap: boolean;
        /**
         * BJS is using an harcoded light falloff based on a manually sets up range.
         * In PBR, one way to represents the fallof is to use the inverse squared root algorythm.
         * This parameter can help you switch back to the BJS mode in order to create scenes using both materials.
         */
        get usePhysicalLightFalloff(): boolean;
        /**
         * BJS is using an harcoded light falloff based on a manually sets up range.
         * In PBR, one way to represents the fallof is to use the inverse squared root algorythm.
         * This parameter can help you switch back to the BJS mode in order to create scenes using both materials.
         */
        set usePhysicalLightFalloff(value: boolean);
        /**
         * In order to support the falloff compatibility with gltf, a special mode has been added
         * to reproduce the gltf light falloff.
         */
        get useGLTFLightFalloff(): boolean;
        /**
         * In order to support the falloff compatibility with gltf, a special mode has been added
         * to reproduce the gltf light falloff.
         */
        set useGLTFLightFalloff(value: boolean);
        /**
         * Specifies that the material will keeps the reflection highlights over a transparent surface (only the most limunous ones).
         * A car glass is a good exemple of that. When the street lights reflects on it you can not see what is behind.
         */
        useRadianceOverAlpha: boolean;
        /**
         * Allows using an object space normal map (instead of tangent space).
         */
        useObjectSpaceNormalMap: boolean;
        /**
         * Allows using the bump map in parallax mode.
         */
        useParallax: boolean;
        /**
         * Allows using the bump map in parallax occlusion mode.
         */
        useParallaxOcclusion: boolean;
        /**
         * Controls the scale bias of the parallax mode.
         */
        parallaxScaleBias: number;
        /**
         * If sets to true, disables all the lights affecting the material.
         */
        disableLighting: boolean;
        /**
         * Force the shader to compute irradiance in the fragment shader in order to take bump in account.
         */
        forceIrradianceInFragment: boolean;
        /**
         * Number of Simultaneous lights allowed on the material.
         */
        maxSimultaneousLights: number;
        /**
         * If sets to true, x component of normal map value will invert (x = 1.0 - x).
         */
        invertNormalMapX: boolean;
        /**
         * If sets to true, y component of normal map value will invert (y = 1.0 - y).
         */
        invertNormalMapY: boolean;
        /**
         * If sets to true and backfaceCulling is false, normals will be flipped on the backside.
         */
        twoSidedLighting: boolean;
        /**
         * A fresnel is applied to the alpha of the model to ensure grazing angles edges are not alpha tested.
         * And/Or occlude the blended part. (alpha is converted to gamma to compute the fresnel)
         */
        useAlphaFresnel: boolean;
        /**
         * A fresnel is applied to the alpha of the model to ensure grazing angles edges are not alpha tested.
         * And/Or occlude the blended part. (alpha stays linear to compute the fresnel)
         */
        useLinearAlphaFresnel: boolean;
        /**
         * Let user defines the brdf lookup texture used for IBL.
         * A default 8bit version is embedded but you could point at :
         * * Default texture: https://assets.babylonjs.com/environments/correlatedMSBRDF_RGBD.png
         * * Default 16bit pixel depth texture: https://assets.babylonjs.com/environments/correlatedMSBRDF.dds
         * * LEGACY Default None correlated https://assets.babylonjs.com/environments/uncorrelatedBRDF_RGBD.png
         * * LEGACY Default None correlated 16bit pixel depth https://assets.babylonjs.com/environments/uncorrelatedBRDF.dds
         */
        environmentBRDFTexture: Nullable<BaseTexture>;
        /**
         * Force normal to face away from face.
         */
        forceNormalForward: boolean;
        /**
         * Enables specular anti aliasing in the PBR shader.
         * It will both interacts on the Geometry for analytical and IBL lighting.
         * It also prefilter the roughness map based on the bump values.
         */
        enableSpecularAntiAliasing: boolean;
        /**
         * This parameters will enable/disable Horizon occlusion to prevent normal maps to look shiny when the normal
         * makes the reflect vector face the model (under horizon).
         */
        useHorizonOcclusion: boolean;
        /**
         * This parameters will enable/disable radiance occlusion by preventing the radiance to lit
         * too much the area relying on ambient texture to define their ambient occlusion.
         */
        useRadianceOcclusion: boolean;
        /**
         * If set to true, no lighting calculations will be applied.
         */
        unlit: boolean;
        /**
         * Gets the image processing configuration used either in this material.
         */
        get imageProcessingConfiguration(): ImageProcessingConfiguration;
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        set imageProcessingConfiguration(value: ImageProcessingConfiguration);
        /**
         * Gets wether the color curves effect is enabled.
         */
        get cameraColorCurvesEnabled(): boolean;
        /**
         * Sets wether the color curves effect is enabled.
         */
        set cameraColorCurvesEnabled(value: boolean);
        /**
         * Gets wether the color grading effect is enabled.
         */
        get cameraColorGradingEnabled(): boolean;
        /**
         * Gets wether the color grading effect is enabled.
         */
        set cameraColorGradingEnabled(value: boolean);
        /**
         * Gets wether tonemapping is enabled or not.
         */
        get cameraToneMappingEnabled(): boolean;
        /**
         * Sets wether tonemapping is enabled or not
         */
        set cameraToneMappingEnabled(value: boolean);
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        get cameraExposure(): number;
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        set cameraExposure(value: number);
        /**
         * Gets The camera contrast used on this material.
         */
        get cameraContrast(): number;
        /**
         * Sets The camera contrast used on this material.
         */
        set cameraContrast(value: number);
        /**
         * Gets the Color Grading 2D Lookup Texture.
         */
        get cameraColorGradingTexture(): Nullable<BaseTexture>;
        /**
         * Sets the Color Grading 2D Lookup Texture.
         */
        set cameraColorGradingTexture(value: Nullable<BaseTexture>);
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        get cameraColorCurves(): Nullable<ColorCurves>;
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        set cameraColorCurves(value: Nullable<ColorCurves>);
        /**
         * Instantiates a new PBRMaterial instance.
         *
         * @param name The material name
         * @param scene The scene the material will be use in.
         */
        constructor(name: string, scene: Scene);
        /**
         * Returns the name of this material class.
         */
        getClassName(): string;
        /**
         * Makes a duplicate of the current material.
         * @param name - name to use for the new material.
         */
        clone(name: string): PBRMaterial;
        /**
         * Serializes this PBR Material.
         * @returns - An object with the serialized material.
         */
        serialize(): any;
        /**
         * Parses a PBR Material from a serialized object.
         * @param source - Serialized object.
         * @param scene - BJS scene instance.
         * @param rootUrl - url for the scene object
         * @returns - PBRMaterial
         */
        static Parse(source: any, scene: Scene, rootUrl: string): PBRMaterial;
    }
}
declare module "babylonjs/Shaders/ShadersInclude/mrtFragmentDeclaration" {
    /** @hidden */
    export var mrtFragmentDeclaration: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/geometry.fragment" {
    import "babylonjs/Shaders/ShadersInclude/mrtFragmentDeclaration";
    import "babylonjs/Shaders/ShadersInclude/bumpFragmentMainFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpFragmentFunctions";
    import "babylonjs/Shaders/ShadersInclude/bumpFragment";
    /** @hidden */
    export var geometryPixelShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Shaders/geometry.vertex" {
    import "babylonjs/Shaders/ShadersInclude/bonesDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertexGlobalDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertexDeclaration";
    import "babylonjs/Shaders/ShadersInclude/instancesDeclaration";
    import "babylonjs/Shaders/ShadersInclude/morphTargetsVertex";
    import "babylonjs/Shaders/ShadersInclude/instancesVertex";
    import "babylonjs/Shaders/ShadersInclude/bonesVertex";
    import "babylonjs/Shaders/ShadersInclude/bumpVertex";
    /** @hidden */
    export var geometryVertexShader: {
        name: string;
        shader: string;
    };
}
declare module "babylonjs/Rendering/geometryBufferRenderer" {
    import { Matrix } from "babylonjs/Maths/math.vector";
    import { SubMesh } from "babylonjs/Meshes/subMesh";
    import { InternalTexture } from "babylonjs/Materials/Textures/internalTexture";
    import { MultiRenderTarget } from "babylonjs/Materials/Textures/multiRenderTarget";
    import { Effect } from "babylonjs/Materials/effect";
    import { PrePassRenderer } from "babylonjs/Rendering/prePassRenderer";
    import { Scene } from "babylonjs/scene";
    import { AbstractMesh } from "babylonjs/Meshes/abstractMesh";
    import { Nullable } from "babylonjs/types";
    import "babylonjs/Shaders/geometry.fragment";
    import "babylonjs/Shaders/geometry.vertex";
    /** @hidden */
    interface ISavedTransformationMatrix {
        world: Matrix;
        viewProjection: Matrix;
    }
    /**
     * This renderer is helpfull to fill one of the render target with a geometry buffer.
     */
    export class GeometryBufferRenderer {
        /**
         * Constant used to retrieve the depth + normal texture index in the G-Buffer textures array
         * using getIndex(GeometryBufferRenderer.DEPTHNORMAL_TEXTURE_INDEX)
         */
        static readonly DEPTHNORMAL_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the position texture index in the G-Buffer textures array
         * using getIndex(GeometryBufferRenderer.POSITION_TEXTURE_INDEX)
         */
        static readonly POSITION_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the velocity texture index in the G-Buffer textures array
         * using getIndex(GeometryBufferRenderer.VELOCITY_TEXTURE_INDEX)
         */
        static readonly VELOCITY_TEXTURE_TYPE: number;
        /**
         * Constant used to retrieve the reflectivity texture index in the G-Buffer textures array
         * using the getIndex(GeometryBufferRenderer.REFLECTIVITY_TEXTURE_TYPE)
         */
        static readonly REFLECTIVITY_TEXTURE_TYPE: number;
        /**
         * Dictionary used to store the previous transformation matrices of each rendered mesh
         * in order to compute objects velocities when enableVelocity is set to "true"
         * @hidden
         */
        _previousTransformationMatrices: {
            [index: number]: ISavedTransformationMatrix;
        };
        /**
         * Dictionary used to store the previous bones transformation matrices of each rendered mesh
         * in order to compute objects velocities when enableVelocity is set to "true"
         * @hidden
         */
        _previousBonesTransformationMatrices: {
            [index: number]: Float32Array;
        };
        /**
         * Array used to store the ignored skinned meshes while computing velocity map (typically used by the motion blur post-process).
         * Avoids computing bones velocities and computes only mesh's velocity itself (position, rotation, scaling).
         */
        excludedSkinnedMeshesFromVelocity: AbstractMesh[];
        /** Gets or sets a boolean indicating if transparent meshes should be rendered */
        renderTransparentMeshes: boolean;
        private _scene;
        private _resizeObserver;
        private _multiRenderTarget;
        private _ratio;
        private _enablePosition;
        private _enableVelocity;
        private _enableReflectivity;
        private _positionIndex;
        private _velocityIndex;
        private _reflectivityIndex;
        private _depthNormalIndex;
        private _linkedWithPrePass;
        private _prePassRenderer;
        private _attachments;
        protected _effect: Effect;
        protected _cachedDefines: string;
        /**
         * @hidden
         * Sets up internal structures to share outputs with PrePassRenderer
         * This method should only be called by the PrePassRenderer itself
         */
        _linkPrePassRenderer(prePassRenderer: PrePassRenderer): void;
        /**
         * @hidden
         * Separates internal structures from PrePassRenderer so the geometry buffer can now operate by itself.
         * This method should only be called by the PrePassRenderer itself
         */
        _unlinkPrePassRenderer(): void;
        /**
         * @hidden
         * Resets the geometry buffer layout
         */
        _resetLayout(): void;
        /**
         * @hidden
         * Replaces a texture in the geometry buffer renderer
         * Useful when linking textures of the prepass renderer
         */
        _forceTextureType(geometryBufferType: number, index: number): void;
        /**
         * @hidden
         * Sets texture attachments
         * Useful when linking textures of the prepass renderer
         */
        _setAttachments(attachments: number[]): void;
        /**
         * @hidden
         * Replaces the first texture which is hard coded as a depth texture in the geometry buffer
         * Useful when linking textures of the prepass renderer
         */
        _linkInternalTexture(internalTexture: InternalTexture): void;
        /**
         * Gets the render list (meshes to be rendered) used in the G buffer.
         */
        get renderList(): Nullable<AbstractMesh[]>;
        /**
         * Set the render list (meshes to be rendered) used in the G buffer.
         */
        set renderList(meshes: Nullable<AbstractMesh[]>);
        /**
         * Gets wether or not G buffer are supported by the running hardware.
         * This requires draw buffer supports
         */
        get isSupported(): boolean;
        /**
         * Returns the index of the given texture type in the G-Buffer textures array
         * @param textureType The texture type constant. For example GeometryBufferRenderer.POSITION_TEXTURE_INDEX
         * @returns the index of the given texture type in the G-Buffer textures array
         */
        getTextureIndex(textureType: number): number;
        /**
         * Gets a boolean indicating if objects positions are enabled for the G buffer.
         */
        get enablePosition(): boolean;
        /**
         * Sets whether or not objects positions are enabled for the G buffer.
         */
        set enablePosition(enable: boolean);
        /**
         * Gets a boolean indicating if objects velocities are enabled for the G buffer.
         */
        get enableVelocity(): boolean;
        /**
         * Sets wether or not objects velocities are enabled for the G buffer.
         */
        set enableVelocity(enable: boolean);
        /**
         * Gets a boolean indicating if objects roughness are enabled in the G buffer.
         */
        get enableReflectivity(): boolean;
        /**
         * Sets wether or not objects roughness are enabled for the G buffer.
         */
        set enableReflectivity(enable: boolean);
        /**
         * Gets the scene associated with the buffer.
         */
        get scene(): Scene;
        /**
         * Gets the ratio used by the buffer during its creation.
         * How big is the buffer related to the main canvas.
         */
        get ratio(): number;
        /** @hidden */
        static _SceneComponentInitialization: (scene: Scene) => void;
        /**
         * Creates a new G Buffer for the scene
         * @param scene The scene the buffer belongs to
         * @param ratio How big is the buffer related to the main canvas.
         */
        constructor(sc