
declare module BABYLON {
    /**
     * Defines how the parser contract is defined.
     * These parsers are used to parse a list of specific assets (like particle systems, etc..)
     */
    export type BabylonFileParser = (parsedData: any, scene: Scene, container: AssetContainer, rootUrl: string) => void;
    /**
     * Defines how the individual parser contract is defined.
     * These parser can parse an individual asset
     */
    export type IndividualBabylonFileParser = (parsedData: any, scene: Scene, rootUrl: string) => any;
    /**
     * Base class of the scene acting as a container for the different elements composing a scene.
     * This class is dynamically extended by the different components of the scene increasing
     * flexibility and reducing coupling
     */
    export abstract class AbstractScene {
        /**
         * Stores the list of available parsers in the application.
         */
        private static _BabylonFileParsers;
        /**
         * Stores the list of available individual parsers in the application.
         */
        private static _IndividualBabylonFileParsers;
        /**
         * Adds a parser in the list of available ones
         * @param name Defines the name of the parser
         * @param parser Defines the parser to add
         */
        static AddParser(name: string, parser: BabylonFileParser): void;
        /**
         * Gets a general parser from the list of available ones
         * @param name Defines the name of the parser
         * @returns the requested parser or null
         */
        static GetParser(name: string): Nullable<BabylonFileParser>;
        /**
         * Adds n individual parser in the list of available ones
         * @param name Defines the name of the parser
         * @param parser Defines the parser to add
         */
        static AddIndividualParser(name: string, parser: IndividualBabylonFileParser): void;
        /**
         * Gets an individual parser from the list of available ones
         * @param name Defines the name of the parser
         * @returns the requested parser or null
         */
        static GetIndividualParser(name: string): Nullable<IndividualBabylonFileParser>;
        /**
         * Parser json data and populate both a scene and its associated container object
         * @param jsonData Defines the data to parse
         * @param scene Defines the scene to parse the data for
         * @param container Defines the container attached to the parsing sequence
         * @param rootUrl Defines the root url of the data
         */
        static Parse(jsonData: any, scene: Scene, container: AssetContainer, rootUrl: string): void;
        /**
         * Gets the list of root nodes (ie. nodes with no parent)
         */
        rootNodes: Node[];
        /** All of the cameras added to this scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
         */
        cameras: Camera[];
        /**
         * All of the lights added to this scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/lights/lights_introduction
         */
        lights: Light[];
        /**
         * All of the (abstract) meshes added to this scene
         */
        meshes: AbstractMesh[];
        /**
         * The list of skeletons added to the scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/bonesSkeletons
         */
        skeletons: Skeleton[];
        /**
         * All of the particle systems added to this scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/particles/particle_system/particle_system_intro
         */
        particleSystems: IParticleSystem[];
        /**
         * Gets a list of Animations associated with the scene
         */
        animations: Animation[];
        /**
         * All of the animation groups added to this scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/groupAnimations
         */
        animationGroups: AnimationGroup[];
        /**
         * All of the multi-materials added to this scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/using/multiMaterials
         */
        multiMaterials: MultiMaterial[];
        /**
         * All of the materials added to this scene
         * In the context of a Scene, it is not supposed to be modified manually.
         * Any addition or removal should be done using the addMaterial and removeMaterial Scene methods.
         * Note also that the order of the Material within the array is not significant and might change.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/using/materials_introduction
         */
        materials: Material[];
        /**
         * The list of morph target managers added to the scene
         * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/dynamicMeshMorph
         */
        morphTargetManagers: MorphTargetManager[];
        /**
         * The list of geometries used in the scene.
         */
        geometries: Geometry[];
        /**
         * All of the transform nodes added to this scene
         * In the context of a Scene, it is not supposed to be modified manually.
         * Any addition or removal should be done using the addTransformNode and removeTransformNode Scene methods.
         * Note also that the order of the TransformNode within the array is not significant and might change.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/transforms/parent_pivot/transform_node
         */
        transformNodes: TransformNode[];
        /**
         * ActionManagers available on the scene.
         * @deprecated
         */
        actionManagers: AbstractActionManager[];
        /**
         * Textures to keep.
         */
        textures: BaseTexture[];
        /** @internal */
        protected _environmentTexture: Nullable<BaseTexture>;
        /**
         * Texture used in all pbr material as the reflection texture.
         * As in the majority of the scene they are the same (exception for multi room and so on),
         * this is easier to reference from here than from all the materials.
         */
        get environmentTexture(): Nullable<BaseTexture>;
        set environmentTexture(value: Nullable<BaseTexture>);
        /**
         * The list of postprocesses added to the scene
         */
        postProcesses: PostProcess[];
        /**
         * @returns all meshes, lights, cameras, transformNodes and bones
         */
        getNodes(): Array<Node>;
    }


    /**
     * Abstract class used to decouple action Manager from scene and meshes.
     * Do not instantiate.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export abstract class AbstractActionManager implements IDisposable {
        /** Gets the list of active triggers */
        static Triggers: {
            [key: string]: number;
        };
        /** Gets the cursor to use when hovering items */
        hoverCursor: string;
        /** Gets the list of actions */
        actions: IAction[];
        /**
         * Gets or sets a boolean indicating that the manager is recursive meaning that it can trigger action from children
         */
        isRecursive: boolean;
        /**
         * Releases all associated resources
         */
        abstract dispose(): void;
        /**
         * Does this action manager has pointer triggers
         */
        abstract get hasPointerTriggers(): boolean;
        /**
         * Does this action manager has pick triggers
         */
        abstract get hasPickTriggers(): boolean;
        /**
         * Process a specific trigger
         * @param trigger defines the trigger to process
         * @param evt defines the event details to be processed
         */
        abstract processTrigger(trigger: number, evt?: IActionEvent): void;
        /**
         * Does this action manager handles actions of any of the given triggers
         * @param triggers defines the triggers to be tested
         * @returns a boolean indicating whether one (or more) of the triggers is handled
         */
        abstract hasSpecificTriggers(triggers: number[]): boolean;
        /**
         * Does this action manager handles actions of any of the given triggers. This function takes two arguments for
         * speed.
         * @param triggerA defines the trigger to be tested
         * @param triggerB defines the trigger to be tested
         * @returns a boolean indicating whether one (or more) of the triggers is handled
         */
        abstract hasSpecificTriggers2(triggerA: number, triggerB: number): boolean;
        /**
         * Does this action manager handles actions of a given trigger
         * @param trigger defines the trigger to be tested
         * @param parameterPredicate defines an optional predicate to filter triggers by parameter
         * @returns whether the trigger is handled
         */
        abstract hasSpecificTrigger(trigger: number, parameterPredicate?: (parameter: any) => boolean): boolean;
        /**
         * Serialize this manager to a JSON object
         * @param name defines the property name to store this manager
         * @returns a JSON representation of this manager
         */
        abstract serialize(name: string): any;
        /**
         * Registers an action to this action manager
         * @param action defines the action to be registered
         * @returns the action amended (prepared) after registration
         */
        abstract registerAction(action: IAction): Nullable<IAction>;
        /**
         * Unregisters an action to this action manager
         * @param action defines the action to be unregistered
         * @returns a boolean indicating whether the action has been unregistered
         */
        abstract unregisterAction(action: IAction): Boolean;
        /**
         * Does exist one action manager with at least one trigger
         **/
        static get HasTriggers(): boolean;
        /**
         * Does exist one action manager with at least one pick trigger
         **/
        static get HasPickTriggers(): boolean;
        /**
         * Does exist one action manager that handles actions of a given trigger
         * @param trigger defines the trigger to be tested
         * @returns a boolean indicating whether the trigger is handled by at least one action manager
         **/
        static HasSpecificTrigger(trigger: number): boolean;
    }


    /**
     * Interface used to define Action
     */
    export interface IAction {
        /**
         * Trigger for the action
         */
        trigger: number;
        /** Options of the trigger */
        triggerOptions: any;
        /**
         * Gets the trigger parameters
         * @returns the trigger parameters
         */
        getTriggerParameter(): any;
        /**
         * Internal only - executes current action event
         * @internal
         */
        _executeCurrent(evt?: ActionEvent): void;
        /**
         * Serialize placeholder for child classes
         * @param parent of child
         * @returns the serialized object
         */
        serialize(parent: any): any;
        /**
         * Internal only
         * @internal
         */
        _prepare(): void;
        /**
         * Internal only - manager for action
         * @internal
         */
        _actionManager: Nullable<AbstractActionManager>;
        /**
         * Adds action to chain of actions, may be a DoNothingAction
         * @param action defines the next action to execute
         * @returns The action passed in
         * @see https://www.babylonjs-playground.com/#1T30HR#0
         */
        then(action: IAction): IAction;
    }
    /**
     * The action to be carried out following a trigger
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#available-actions
     */
    export class Action implements IAction {
        /** the trigger, with or without parameters, for the action */
        triggerOptions: any;
        /**
         * Trigger for the action
         */
        trigger: number;
        /**
         * Internal only - manager for action
         * @internal
         */
        _actionManager: ActionManager;
        private _nextActiveAction;
        private _child;
        private _condition?;
        private _triggerParameter;
        /**
         * An event triggered prior to action being executed.
         */
        onBeforeExecuteObservable: Observable<Action>;
        /**
         * Creates a new Action
         * @param triggerOptions the trigger, with or without parameters, for the action
         * @param condition an optional determinant of action
         */
        constructor(
        /** the trigger, with or without parameters, for the action */
        triggerOptions: any, condition?: Condition);
        /**
         * Internal only
         * @internal
         */
        _prepare(): void;
        /**
         * Gets the trigger parameter
         * @returns the trigger parameter
         */
        getTriggerParameter(): any;
        /**
         * Sets the trigger parameter
         * @param value defines the new trigger parameter
         */
        setTriggerParameter(value: any): void;
        /**
         * Internal only - Returns if the current condition allows to run the action
         * @internal
         */
        _evaluateConditionForCurrentFrame(): boolean;
        /**
         * Internal only - executes current action event
         * @internal
         */
        _executeCurrent(evt?: ActionEvent): void;
        /**
         * Execute placeholder for child classes
         * @param evt optional action event
         */
        execute(evt?: ActionEvent): void;
        /**
         * Skips to next active action
         */
        skipToNextActiveAction(): void;
        /**
         * Adds action to chain of actions, may be a DoNothingAction
         * @param action defines the next action to execute
         * @returns The action passed in
         * @see https://www.babylonjs-playground.com/#1T30HR#0
         */
        then(action: Action): Action;
        /**
         * Internal only
         * @internal
         */
        _getProperty(propertyPath: string): string;
        /**
         * @internal
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @param parent of child
         * @returns the serialized object
         */
        serialize(parent: any): any;
        /**
         * Internal only called by serialize
         * @internal
         */
        protected _serialize(serializedAction: any, parent?: any): any;
        /**
         * Internal only
         * @internal
         */
        static _SerializeValueAsString: (value: any) => string;
        /**
         * Internal only
         * @internal
         */
        static _GetTargetProperty: (target: Scene | Node | Material) => {
            name: string;
            targetType: string;
            value: string;
        };
    }


    /**
     * Interface used to define ActionEvent
     */
    export interface IActionEvent {
        /** The mesh or sprite that triggered the action */
        source: any;
        /** The X mouse cursor position at the time of the event */
        pointerX: number;
        /** The Y mouse cursor position at the time of the event */
        pointerY: number;
        /** The mesh that is currently pointed at (can be null) */
        meshUnderPointer: Nullable<AbstractMesh>;
        /** the original (browser) event that triggered the ActionEvent */
        sourceEvent?: any;
        /** additional data for the event */
        additionalData?: any;
    }
    /**
     * ActionEvent is the event being sent when an action is triggered.
     */
    export class ActionEvent implements IActionEvent {
        /** The mesh or sprite that triggered the action */
        source: any;
        /** The X mouse cursor position at the time of the event */
        pointerX: number;
        /** The Y mouse cursor position at the time of the event */
        pointerY: number;
        /** The mesh that is currently pointed at (can be null) */
        meshUnderPointer: Nullable<AbstractMesh>;
        /** the original (browser) event that triggered the ActionEvent */
        sourceEvent?: any;
        /** additional data for the event */
        additionalData?: any;
        /**
         * Creates a new ActionEvent
         * @param source The mesh or sprite that triggered the action
         * @param pointerX The X mouse cursor position at the time of the event
         * @param pointerY The Y mouse cursor position at the time of the event
         * @param meshUnderPointer The mesh that is currently pointed at (can be null)
         * @param sourceEvent the original (browser) event that triggered the ActionEvent
         * @param additionalData additional data for the event
         */
        constructor(
        /** The mesh or sprite that triggered the action */
        source: any, 
        /** The X mouse cursor position at the time of the event */
        pointerX: number, 
        /** The Y mouse cursor position at the time of the event */
        pointerY: number, 
        /** The mesh that is currently pointed at (can be null) */
        meshUnderPointer: Nullable<AbstractMesh>, 
        /** the original (browser) event that triggered the ActionEvent */
        sourceEvent?: any, 
        /** additional data for the event */
        additionalData?: any);
        /**
         * Helper function to auto-create an ActionEvent from a source mesh.
         * @param source The source mesh that triggered the event
         * @param evt The original (browser) event
         * @param additionalData additional data for the event
         * @returns the new ActionEvent
         */
        static CreateNew(source: AbstractMesh, evt?: any, additionalData?: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a source sprite
         * @param source The source sprite that triggered the event
         * @param scene Scene associated with the sprite
         * @param evt The original (browser) event
         * @param additionalData additional data for the event
         * @returns the new ActionEvent
         */
        static CreateNewFromSprite(source: Sprite, scene: Scene, evt?: any, additionalData?: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a scene. If triggered by a mesh use ActionEvent.CreateNew
         * @param scene the scene where the event occurred
         * @param evt The original (browser) event
         * @returns the new ActionEvent
         */
        static CreateNewFromScene(scene: Scene, evt: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a primitive
         * @param prim defines the target primitive
         * @param pointerPos defines the pointer position
         * @param evt The original (browser) event
         * @param additionalData additional data for the event
         * @returns the new ActionEvent
         */
        static CreateNewFromPrimitive(prim: any, pointerPos: Vector2, evt?: Event, additionalData?: any): ActionEvent;
    }


    /**
     * Action Manager manages all events to be triggered on a given mesh or the global scene.
     * A single scene can have many Action Managers to handle predefined actions on specific meshes.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class ActionManager extends AbstractActionManager {
        /**
         * Nothing
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly NothingTrigger = 0;
        /**
         * On pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnPickTrigger = 1;
        /**
         * On left pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnLeftPickTrigger = 2;
        /**
         * On right pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnRightPickTrigger = 3;
        /**
         * On center pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnCenterPickTrigger = 4;
        /**
         * On pick down
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnPickDownTrigger = 5;
        /**
         * On double pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnDoublePickTrigger = 6;
        /**
         * On pick up
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnPickUpTrigger = 7;
        /**
         * On pick out.
         * This trigger will only be raised if you also declared a OnPickDown
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnPickOutTrigger = 16;
        /**
         * On long press
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnLongPressTrigger = 8;
        /**
         * On pointer over
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnPointerOverTrigger = 9;
        /**
         * On pointer out
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnPointerOutTrigger = 10;
        /**
         * On every frame
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnEveryFrameTrigger = 11;
        /**
         * On intersection enter
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnIntersectionEnterTrigger = 12;
        /**
         * On intersection exit
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnIntersectionExitTrigger = 13;
        /**
         * On key down
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnKeyDownTrigger = 14;
        /**
         * On key up
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly OnKeyUpTrigger = 15;
        private _scene;
        /**
         * Creates a new action manager
         * @param scene defines the hosting scene
         */
        constructor(scene?: Nullable<Scene>);
        /**
         * Releases all associated resources
         */
        dispose(): void;
        /**
         * Gets hosting scene
         * @returns the hosting scene
         */
        getScene(): Scene;
        /**
         * Does this action manager handles actions of any of the given triggers
         * @param triggers defines the triggers to be tested
         * @returns a boolean indicating whether one (or more) of the triggers is handled
         */
        hasSpecificTriggers(triggers: number[]): boolean;
        /**
         * Does this action manager handles actions of any of the given triggers. This function takes two arguments for
         * speed.
         * @param triggerA defines the trigger to be tested
         * @param triggerB defines the trigger to be tested
         * @returns a boolean indicating whether one (or more) of the triggers is handled
         */
        hasSpecificTriggers2(triggerA: number, triggerB: number): boolean;
        /**
         * Does this action manager handles actions of a given trigger
         * @param trigger defines the trigger to be tested
         * @param parameterPredicate defines an optional predicate to filter triggers by parameter
         * @returns whether the trigger is handled
         */
        hasSpecificTrigger(trigger: number, parameterPredicate?: (parameter: any) => boolean): boolean;
        /**
         * Does this action manager has pointer triggers
         */
        get hasPointerTriggers(): boolean;
        /**
         * Does this action manager has pick triggers
         */
        get hasPickTriggers(): boolean;
        /**
         * Registers an action to this action manager
         * @param action defines the action to be registered
         * @returns the action amended (prepared) after registration
         */
        registerAction(action: IAction): Nullable<IAction>;
        /**
         * Unregisters an action to this action manager
         * @param action defines the action to be unregistered
         * @returns a boolean indicating whether the action has been unregistered
         */
        unregisterAction(action: IAction): Boolean;
        /**
         * Process a specific trigger
         * @param trigger defines the trigger to process
         * @param evt defines the event details to be processed
         */
        processTrigger(trigger: number, evt?: IActionEvent): void;
        /**
         * @internal
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * @internal
         */
        _getProperty(propertyPath: string): string;
        /**
         * Serialize this manager to a JSON object
         * @param name defines the property name to store this manager
         * @returns a JSON representation of this manager
         */
        serialize(name: string): any;
        /**
         * Creates a new ActionManager from a JSON data
         * @param parsedActions defines the JSON data to read from
         * @param object defines the hosting mesh
         * @param scene defines the hosting scene
         */
        static Parse(parsedActions: any, object: Nullable<AbstractMesh>, scene: Scene): void;
        /**
         * Get a trigger name by index
         * @param trigger defines the trigger index
         * @returns a trigger name
         */
        static GetTriggerName(trigger: number): string;
    }


    /**
     * A Condition applied to an Action
     */
    export class Condition {
        /**
         * Internal only - manager for action
         * @internal
         */
        _actionManager: ActionManager;
        /**
         * @internal
         */
        _evaluationId: number;
        /**
         * @internal
         */
        _currentResult: boolean;
        /**
         * Creates a new Condition
         * @param actionManager the manager of the action the condition is applied to
         */
        constructor(actionManager: ActionManager);
        /**
         * Check if the current condition is valid
         * @returns a boolean
         */
        isValid(): boolean;
        /**
         * @internal
         */
        _getProperty(propertyPath: string): string;
        /**
         * @internal
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @returns the serialized object
         */
        serialize(): any;
        /**
         * @internal
         */
        protected _serialize(serializedCondition: any): any;
    }
    /**
     * Defines specific conditional operators as extensions of Condition
     */
    export class ValueCondition extends Condition {
        /** path to specify the property of the target the conditional operator uses  */
        propertyPath: string;
        /** the value compared by the conditional operator against the current value of the property */
        value: any;
        /** the conditional operator, default ValueCondition.IsEqual */
        operator: number;
        private static _IsEqual;
        private static _IsDifferent;
        private static _IsGreater;
        private static _IsLesser;
        /**
         * returns the number for IsEqual
         */
        static get IsEqual(): number;
        /**
         * Returns the number for IsDifferent
         */
        static get IsDifferent(): number;
        /**
         * Returns the number for IsGreater
         */
        static get IsGreater(): number;
        /**
         * Returns the number for IsLesser
         */
        static get IsLesser(): number;
        /**
         * Internal only The action manager for the condition
         * @internal
         */
        _actionManager: ActionManager;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Creates a new ValueCondition
         * @param actionManager manager for the action the condition applies to
         * @param target for the action
         * @param propertyPath path to specify the property of the target the conditional operator uses
         * @param value the value compared by the conditional operator against the current value of the property
         * @param operator the conditional operator, default ValueCondition.IsEqual
         */
        constructor(actionManager: ActionManager, target: any, 
        /** path to specify the property of the target the conditional operator uses  */
        propertyPath: string, 
        /** the value compared by the conditional operator against the current value of the property */
        value: any, 
        /** the conditional operator, default ValueCondition.IsEqual */
        operator?: number);
        /**
         * Compares the given value with the property value for the specified conditional operator
         * @returns the result of the comparison
         */
        isValid(): boolean;
        /**
         * Serialize the ValueCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
        /**
         * Gets the name of the conditional operator for the ValueCondition
         * @param operator the conditional operator
         * @returns the name
         */
        static GetOperatorName(operator: number): string;
    }
    /**
     * Defines a predicate condition as an extension of Condition
     */
    export class PredicateCondition extends Condition {
        /** defines the predicate function used to validate the condition */
        predicate: () => boolean;
        /**
         * Internal only - manager for action
         * @internal
         */
        _actionManager: ActionManager;
        /**
         * Creates a new PredicateCondition
         * @param actionManager manager for the action the condition applies to
         * @param predicate defines the predicate function used to validate the condition
         */
        constructor(actionManager: ActionManager, 
        /** defines the predicate function used to validate the condition */
        predicate: () => boolean);
        /**
         * @returns the validity of the predicate condition
         */
        isValid(): boolean;
    }
    /**
     * Defines a state condition as an extension of Condition
     */
    export class StateCondition extends Condition {
        /** Value to compare with target state  */
        value: string;
        /**
         * Internal only - manager for action
         * @internal
         */
        _actionManager: ActionManager;
        private _target;
        /**
         * Creates a new StateCondition
         * @param actionManager manager for the action the condition applies to
         * @param target of the condition
         * @param value to compare with target state
         */
        constructor(actionManager: ActionManager, target: any, 
        /** Value to compare with target state  */
        value: string);
        /**
         * Gets a boolean indicating if the current condition is met
         * @returns the validity of the state
         */
        isValid(): boolean;
        /**
         * Serialize the StateCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
    }


    /**
     * This defines an action responsible to toggle a boolean once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class SwitchBooleanAction extends Action {
        /**
         * The path to the boolean property in the target object
         */
        propertyPath: string;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the boolean
         * @param propertyPath defines the path to the boolean property in the target object
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action toggle the boolean value.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to set a the state field of the target
     *  to a desired value once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class SetStateAction extends Action {
        /**
         * The value to store in the state field.
         */
        value: string;
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the state property
         * @param value defines the value to store in the state field
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, value: string, condition?: Condition);
        /**
         * Execute the action and store the value on the target state property.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to set a property of the target
     *  to a desired value once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class SetValueAction extends Action {
        /**
         * The path of the property to set in the target.
         */
        propertyPath: string;
        /**
         * The value to set in the property
         */
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the property
         * @param propertyPath defines the path of the property to set in the target
         * @param value defines the value to set in the property
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and set the targeted property to the desired value.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to increment the target value
     *  to a desired value once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class IncrementValueAction extends Action {
        /**
         * The path of the property to increment in the target.
         */
        propertyPath: string;
        /**
         * The value we should increment the property by.
         */
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the property
         * @param propertyPath defines the path of the property to increment in the target
         * @param value defines the value value we should increment the property by
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and increment the target of the value amount.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to start an animation once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class PlayAnimationAction extends Action {
        /**
         * Where the animation should start (animation frame)
         */
        from: number;
        /**
         * Where the animation should stop (animation frame)
         */
        to: number;
        /**
         * Define if the animation should loop or stop after the first play.
         */
        loop?: boolean;
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the target animation or animation name
         * @param from defines from where the animation should start (animation frame)
         * @param to defines where the animation should stop (animation frame)
         * @param loop defines if the animation should loop or stop after the first play
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, from: number, to: number, loop?: boolean, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and play the animation.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to stop an animation once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class StopAnimationAction extends Action {
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the target animation or animation name
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and stop the animation.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible that does nothing once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class DoNothingAction extends Action {
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions?: any, condition?: Condition);
        /**
         * Execute the action and do nothing.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to trigger several actions once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class CombineAction extends Action {
        /**
         * The list of aggregated animations to run.
         */
        children: Action[];
        /**
         * defines if the children actions conditions should be check before execution
         */
        enableChildrenConditions: boolean;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param children defines the list of aggregated animations to run
         * @param condition defines the trigger related conditions
         * @param enableChildrenConditions defines if the children actions conditions should be check before execution
         */
        constructor(triggerOptions: any, children: Action[], condition?: Condition, enableChildrenConditions?: boolean);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and executes all the aggregated actions.
         * @param evt event to execute
         */
        execute(evt: ActionEvent): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action responsible to run code (external event) once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class ExecuteCodeAction extends Action {
        /**
         * The callback function to run.
         */
        func: (evt: ActionEvent) => void;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param func defines the callback function to run
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, func: (evt: ActionEvent) => void, condition?: Condition);
        /**
         * Execute the action and run the attached code.
         * @param evt event to execute
         */
        execute(evt: ActionEvent): void;
    }
    /**
     * This defines an action responsible to set the parent property of the target once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class SetParentAction extends Action {
        private _parent;
        private _target;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the target containing the parent property
         * @param parent defines from where the animation should start (animation frame)
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, target: any, parent: any, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and set the parent property.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }


    /**
     * This defines an action helpful to play a defined sound on a triggered action.
     */
    export class PlaySoundAction extends Action {
        private _sound;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param sound defines the sound to play
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, sound: Sound, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and play the sound.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }
    /**
     * This defines an action helpful to stop a defined sound on a triggered action.
     */
    export class StopSoundAction extends Action {
        private _sound;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param sound defines the sound to stop
         * @param condition defines the trigger related conditions
         */
        constructor(triggerOptions: any, sound: Sound, condition?: Condition);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action and stop the sound.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }




    /**
     * This defines an action responsible to change the value of a property
     * by interpolating between its current value and the newly set one once triggered.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions
     */
    export class InterpolateValueAction extends Action {
        /**
         * Defines the path of the property where the value should be interpolated
         */
        propertyPath: string;
        /**
         * Defines the target value at the end of the interpolation.
         */
        value: any;
        /**
         * Defines the time it will take for the property to interpolate to the value.
         */
        duration: number;
        /**
         * Defines if the other scene animations should be stopped when the action has been triggered
         */
        stopOtherAnimations?: boolean;
        /**
         * Defines a callback raised once the interpolation animation has been done.
         */
        onInterpolationDone?: () => void;
        /**
         * Observable triggered once the interpolation animation has been done.
         */
        onInterpolationDoneObservable: Observable<InterpolateValueAction>;
        private _target;
        private _effectiveTarget;
        private _property;
        /**
         * Instantiate the action
         * @param triggerOptions defines the trigger options
         * @param target defines the object containing the value to interpolate
         * @param propertyPath defines the path to the property in the target object
         * @param value defines the target value at the end of the interpolation
         * @param duration defines the time it will take for the property to interpolate to the value.
         * @param condition defines the trigger related conditions
         * @param stopOtherAnimations defines if the other scene animations should be stopped when the action has been triggered
         * @param onInterpolationDone defines a callback raised once the interpolation animation has been done
         */
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, duration?: number, condition?: Condition, stopOtherAnimations?: boolean, onInterpolationDone?: () => void);
        /** @internal */
        _prepare(): void;
        /**
         * Execute the action starts the value interpolation.
         */
        execute(): void;
        /**
         * Serializes the actions and its related information.
         * @param parent defines the object to serialize in
         * @returns the serialized object
         */
        serialize(parent: any): any;
    }


    /**
     * Class used to store an actual running animation
     */
    export class Animatable {
        /** defines the target object */
        target: any;
        /** defines the starting frame number (default is 0) */
        fromFrame: number;
        /** defines the ending frame number (default is 100) */
        toFrame: number;
        /** defines if the animation must loop (default is false)  */
        loopAnimation: boolean;
        /** defines a callback to call when animation ends if it is not looping */
        onAnimationEnd?: Nullable<() => void> | undefined;
        /** defines a callback to call when animation loops */
        onAnimationLoop?: Nullable<() => void> | undefined;
        /** defines whether the animation should be evaluated additively */
        isAdditive: boolean;
        /** defines the order in which this animatable should be processed in the list of active animatables (default: 0) */
        playOrder: number;
        private _localDelayOffset;
        private _pausedDelay;
        private _manualJumpDelay;
        /** @hidden */
        _runtimeAnimations: RuntimeAnimation[];
        private _paused;
        private _scene;
        private _speedRatio;
        private _weight;
        private _syncRoot;
        private _frameToSyncFromJump;
        private _goToFrame;
        /**
         * Gets or sets a boolean indicating if the animatable must be disposed and removed at the end of the animation.
         * This will only apply for non looping animation (default is true)
         */
        disposeOnEnd: boolean;
        /**
         * Gets a boolean indicating if the animation has started
         */
        animationStarted: boolean;
        /**
         * Observer raised when the animation ends
         */
        onAnimationEndObservable: Observable<Animatable>;
        /**
         * Observer raised when the animation loops
         */
        onAnimationLoopObservable: Observable<Animatable>;
        /**
         * Gets the root Animatable used to synchronize and normalize animations
         */
        get syncRoot(): Nullable<Animatable>;
        /**
         * Gets the current frame of the first RuntimeAnimation
         * Used to synchronize Animatables
         */
        get masterFrame(): number;
        /**
         * Gets or sets the animatable weight (-1.0 by default meaning not weighted)
         */
        get weight(): number;
        set weight(value: number);
        /**
         * Gets or sets the speed ratio to apply to the animatable (1.0 by default)
         */
        get speedRatio(): number;
        set speedRatio(value: number);
        /**
         * Gets the elapsed time since the animatable started in milliseconds
         */
        get elapsedTime(): number;
        /**
         * Creates a new Animatable
         * @param scene defines the hosting scene
         * @param target defines the target object
         * @param fromFrame defines the starting frame number (default is 0)
         * @param toFrame defines the ending frame number (default is 100)
         * @param loopAnimation defines if the animation must loop (default is false)
         * @param speedRatio defines the factor to apply to animation speed (default is 1)
         * @param onAnimationEnd defines a callback to call when animation ends if it is not looping
         * @param animations defines a group of animation to add to the new Animatable
         * @param onAnimationLoop defines a callback to call when animation loops
         * @param isAdditive defines whether the animation should be evaluated additively
         * @param playOrder defines the order in which this animatable should be processed in the list of active animatables (default: 0)
         */
        constructor(scene: Scene, 
        /** defines the target object */
        target: any, 
        /** defines the starting frame number (default is 0) */
        fromFrame?: number, 
        /** defines the ending frame number (default is 100) */
        toFrame?: number, 
        /** defines if the animation must loop (default is false)  */
        loopAnimation?: boolean, speedRatio?: number, 
        /** defines a callback to call when animation ends if it is not looping */
        onAnimationEnd?: Nullable<() => void> | undefined, animations?: Animation[], 
        /** defines a callback to call when animation loops */
        onAnimationLoop?: Nullable<() => void> | undefined, 
        /** defines whether the animation should be evaluated additively */
        isAdditive?: boolean, 
        /** defines the order in which this animatable should be processed in the list of active animatables (default: 0) */
        playOrder?: number);
        /**
         * Synchronize and normalize current Animatable with a source Animatable
         * This is useful when using animation weights and when animations are not of the same length
         * @param root defines the root Animatable to synchronize with (null to stop synchronizing)
         * @returns the current Animatable
         */
        syncWith(root: Nullable<Animatable>): Animatable;
        /**
         * Gets the list of runtime animations
         * @returns an array of RuntimeAnimation
         */
        getAnimations(): RuntimeAnimation[];
        /**
         * Adds more animations to the current animatable
         * @param target defines the target of the animations
         * @param animations defines the new animations to add
         */
        appendAnimations(target: any, animations: Animation[]): void;
        /**
         * Gets the source animation for a specific property
         * @param property defines the property to look for
         * @returns null or the source animation for the given property
         */
        getAnimationByTargetProperty(property: string): Nullable<Animation>;
        /**
         * Gets the runtime animation for a specific property
         * @param property defines the property to look for
         * @returns null or the runtime animation for the given property
         */
        getRuntimeAnimationByTargetProperty(property: string): Nullable<RuntimeAnimation>;
        /**
         * Resets the animatable to its original state
         */
        reset(): void;
        /**
         * Allows the animatable to blend with current running animations
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#animation-blending
         * @param blendingSpeed defines the blending speed to use
         */
        enableBlending(blendingSpeed: number): void;
        /**
         * Disable animation blending
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#animation-blending
         */
        disableBlending(): void;
        /**
         * Jump directly to a given frame
         * @param frame defines the frame to jump to
         */
        goToFrame(frame: number): void;
        /**
         * Returns true if the animations for this animatable are paused
         */
        get paused(): boolean;
        /**
         * Pause the animation
         */
        pause(): void;
        /**
         * Restart the animation
         */
        restart(): void;
        private _raiseOnAnimationEnd;
        /**
         * Stop and delete the current animation
         * @param animationName defines a string used to only stop some of the runtime animations instead of all
         * @param targetMask a function that determines if the animation should be stopped based on its target (all animations will be stopped if both this and animationName are empty)
         * @param useGlobalSplice if true, the animatables will be removed by the caller of this function (false by default)
         */
        stop(animationName?: string, targetMask?: (target: any) => boolean, useGlobalSplice?: boolean): void;
        /**
         * Wait asynchronously for the animation to end
         * @returns a promise which will be fulfilled when the animation ends
         */
        waitAsync(): Promise<Animatable>;
        /**
         * @internal
         */
        _animate(delay: number): boolean;
    }
        interface Scene {
            /** @internal */
            _registerTargetForLateAnimationBinding(runtimeAnimation: RuntimeAnimation, originalValue: any): void;
            /** @internal */
            _processLateAnimationBindingsForMatrices(holder: {
                totalWeight: number;
                totalAdditiveWeight: number;
                animations: RuntimeAnimation[];
                additiveAnimations: RuntimeAnimation[];
                originalValue: Matrix;
            }): any;
            /** @internal */
            _processLateAnimationBindingsForQuaternions(holder: {
                totalWeight: number;
                totalAdditiveWeight: number;
                animations: RuntimeAnimation[];
                additiveAnimations: RuntimeAnimation[];
                originalValue: Quaternion;
            }, refQuaternion: Quaternion): Quaternion;
            /** @internal */
            _processLateAnimationBindings(): void;
            /**
             * Sort active animatables based on their playOrder property
             */
            sortActiveAnimatables(): void;
            /**
             * Will start the animation sequence of a given target
             * @param target defines the target
             * @param from defines from which frame should animation start
             * @param to defines until which frame should animation run.
             * @param weight defines the weight to apply to the animation (1.0 by default)
             * @param loop defines if the animation loops
             * @param speedRatio defines the speed in which to run the animation (1.0 by default)
             * @param onAnimationEnd defines the function to be executed when the animation ends
             * @param animatable defines an animatable object. If not provided a new one will be created from the given params
             * @param targetMask defines if the target should be animated if animations are present (this is called recursively on descendant animatables regardless of return value)
             * @param onAnimationLoop defines the callback to call when an animation loops
             * @param isAdditive defines whether the animation should be evaluated additively (false by default)
             * @returns the animatable object created for this animation
             */
            beginWeightedAnimation(target: any, from: number, to: number, weight: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable, targetMask?: (target: any) => boolean, onAnimationLoop?: () => void, isAdditive?: boolean): Animatable;
            /**
             * Will start the animation sequence of a given target
             * @param target defines the target
             * @param from defines from which frame should animation start
             * @param to defines until which frame should animation run.
             * @param loop defines if the animation loops
             * @param speedRatio defines the speed in which to run the animation (1.0 by default)
             * @param onAnimationEnd defines the function to be executed when the animation ends
             * @param animatable defines an animatable object. If not provided a new one will be created from the given params
             * @param stopCurrent defines if the current animations must be stopped first (true by default)
             * @param targetMask defines if the target should be animate if animations are present (this is called recursively on descendant animatables regardless of return value)
             * @param onAnimationLoop defines the callback to call when an animation loops
             * @param isAdditive defines whether the animation should be evaluated additively (false by default)
             * @returns the animatable object created for this animation
             */
            beginAnimation(target: any, from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable, stopCurrent?: boolean, targetMask?: (target: any) => boolean, onAnimationLoop?: () => void, isAdditive?: boolean): Animatable;
            /**
             * Will start the animation sequence of a given target and its hierarchy
             * @param target defines the target
             * @param directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used.
             * @param from defines from which frame should animation start
             * @param to defines until which frame should animation run.
             * @param loop defines if the animation loops
             * @param speedRatio defines the speed in which to run the animation (1.0 by default)
             * @param onAnimationEnd defines the function to be executed when the animation ends
             * @param animatable defines an animatable object. If not provided a new one will be created from the given params
             * @param stopCurrent defines if the current animations must be stopped first (true by default)
             * @param targetMask defines if the target should be animated if animations are present (this is called recursively on descendant animatables regardless of return value)
             * @param onAnimationLoop defines the callback to call when an animation loops
             * @param isAdditive defines whether the animation should be evaluated additively (false by default)
             * @returns the list of created animatables
             */
            beginHierarchyAnimation(target: any, directDescendantsOnly: boolean, from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable, stopCurrent?: boolean, targetMask?: (target: any) => boolean, onAnimationLoop?: () => void, isAdditive?: boolean): Animatable[];
            /**
             * Begin a new animation on a given node
             * @param target defines the target where the animation will take place
             * @param animations defines the list of animations to start
             * @param from defines the initial value
             * @param to defines the final value
             * @param loop defines if you want animation to loop (off by default)
             * @param speedRatio defines the speed ratio to apply to all animations
             * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
             * @param onAnimationLoop defines the callback to call when an animation loops
             * @param isAdditive defines whether the animation should be evaluated additively (false by default)
             * @returns the list of created animatables
             */
            beginDirectAnimation(target: any, animations: Animation[], from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, onAnimationLoop?: () => void, isAdditive?: boolean): Animatable;
            /**
             * Begin a new animation on a given node and its hierarchy
             * @param target defines the root node where the animation will take place
             * @param directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used.
             * @param animations defines the list of animations to start
             * @param from defines the initial value
             * @param to defines the final value
             * @param loop defines if you want animation to loop (off by default)
             * @param speedRatio defines the speed ratio to apply to all animations
             * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
             * @param onAnimationLoop defines the callback to call when an animation loops
             * @param isAdditive defines whether the animation should be evaluated additively (false by default)
             * @returns the list of animatables created for all nodes
             */
            beginDirectHierarchyAnimation(target: Node, directDescendantsOnly: boolean, animations: Animation[], from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, onAnimationLoop?: () => void, isAdditive?: boolean): Animatable[];
            /**
             * Gets the animatable associated with a specific target
             * @param target defines the target of the animatable
             * @returns the required animatable if found
             */
            getAnimatableByTarget(target: any): Nullable<Animatable>;
            /**
             * Gets all animatables associated with a given target
             * @param target defines the target to look animatables for
             * @returns an array of Animatables
             */
            getAllAnimatablesByTarget(target: any): Array<Animatable>;
            /**
             * Stops and removes all animations that have been applied to the scene
             */
            stopAllAnimations(): void;
            /**
             * Gets the current delta time used by animation engine
             */
            deltaTime: number;
        }
        interface Bone {
            /**
             * Copy an animation range from another bone
             * @param source defines the source bone
             * @param rangeName defines the range name to copy
             * @param frameOffset defines the frame offset
             * @param rescaleAsRequired defines if rescaling must be applied if required
             * @param skelDimensionsRatio defines the scaling ratio
             * @returns true if operation was successful
             */
            copyAnimationRange(source: Bone, rangeName: string, frameOffset: number, rescaleAsRequired: boolean, skelDimensionsRatio: Nullable<Vector3>): boolean;
        }


    /**
     * Interface containing an array of animations
     */
    export interface IAnimatable {
        /**
         * Array of animations
         */
        animations: Nullable<Array<Animation>>;
    }


    export var _staticOffsetValueQuaternion: DeepImmutable<Quaternion>;
    export var _staticOffsetValueVector3: DeepImmutable<Vector3>;
    export var _staticOffsetValueVector2: DeepImmutable<Vector2>;
    export var _staticOffsetValueSize: DeepImmutable<Size>;
    export var _staticOffsetValueColor3: DeepImmutable<Color3>;
    export var _staticOffsetValueColor4: DeepImmutable<Color4>;
    /**
     * Options to be used when creating an additive animation
     */
    export interface IMakeAnimationAdditiveOptions {
        /**
         * The frame that the animation should be relative to (if not provided, 0 will be used)
         */
        referenceFrame?: number;
        /**
         * The name of the animation range to convert to additive. If not provided, fromFrame / toFrame will be used
         * If fromFrame / toFrame are not provided either, the whole animation will be converted to additive
         */
        range?: string;
        /**
         * If true, the original animation will be cloned and converted to additive. If false, the original animation will be converted to additive (default is false)
         */
        cloneOriginalAnimation?: boolean;
        /**
         * The name of the cloned animation if cloneOriginalAnimation is true. If not provided, use the original animation name
         */
        clonedAnimationName?: string;
        /**
         * Together with toFrame, defines the range of the animation to convert to additive. Will only be used if range is not provided
         * If range and fromFrame / toFrame are not provided, the whole animation will be converted to additive
         */
        fromFrame?: number;
        /**
         * Together with fromFrame, defines the range of the animation to convert to additive.
         */
        toFrame?: number;
        /**
         * If true, the key frames will be clipped to the range specified by range or fromFrame / toFrame (default is false)
         */
        clipKeys?: boolean;
    }
    /**
     * @internal
     */
    export interface _IAnimationState {
        key: number;
        repeatCount: number;
        workValue?: any;
        loopMode?: number;
        offsetValue?: any;
        highLimitValue?: any;
    }
    /**
     * Class used to store any kind of animation
     */
    export class Animation {
        /**Name of the animation */
        name: string;
        /**Property to animate */
        targetProperty: string;
        /**The frames per second of the animation */
        framePerSecond: number;
        /**The data type of the animation */
        dataType: number;
        /**The loop mode of the animation */
        loopMode?: number | undefined;
        /**Specifies if blending should be enabled */
        enableBlending?: boolean | undefined;
        private static _UniqueIdGenerator;
        /**
         * Use matrix interpolation instead of using direct key value when animating matrices
         */
        static AllowMatricesInterpolation: boolean;
        /**
         * When matrix interpolation is enabled, this boolean forces the system to use Matrix.DecomposeLerp instead of Matrix.Lerp. Interpolation is more precise but slower
         */
        static AllowMatrixDecomposeForInterpolation: boolean;
        /**
         * Gets or sets the unique id of the animation (the uniqueness is solely among other animations)
         */
        uniqueId: number;
        /** Define the Url to load snippets */
        static SnippetUrl: string;
        /** Snippet ID if the animation was created from the snippet server */
        snippetId: string;
        /**
         * Stores the key frames of the animation
         */
        private _keys;
        /**
         * Stores the easing function of the animation
         */
        private _easingFunction;
        /**
         * @internal Internal use only
         */
        _runtimeAnimations: RuntimeAnimation[];
        /**
         * The set of event that will be linked to this animation
         */
        private _events;
        /**
         * Stores an array of target property paths
         */
        targetPropertyPath: string[];
        /**
         * Stores the blending speed of the animation
         */
        blendingSpeed: number;
        /**
         * Stores the animation ranges for the animation
         */
        private _ranges;
        /**
         * @internal Internal use
         */
        static _PrepareAnimation(name: string, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction): Nullable<Animation>;
        /**
         * Sets up an animation
         * @param property The property to animate
         * @param animationType The animation type to apply
         * @param framePerSecond The frames per second of the animation
         * @param easingFunction The easing function used in the animation
         * @returns The created animation
         */
        static CreateAnimation(property: string, animationType: number, framePerSecond: number, easingFunction: EasingFunction): Animation;
        /**
         * Create and start an animation on a node
         * @param name defines the name of the global animation that will be run on all nodes
         * @param target defines the target where the animation will take place
         * @param targetProperty defines property to animate
         * @param framePerSecond defines the number of frame per second yo use
         * @param totalFrame defines the number of frames in total
         * @param from defines the initial value
         * @param to defines the final value
         * @param loopMode defines which loop mode you want to use (off by default)
         * @param easingFunction defines the easing function to use (linear by default)
         * @param onAnimationEnd defines the callback to call when animation end
         * @param scene defines the hosting scene
         * @returns the animatable created for this animation
         */
        static CreateAndStartAnimation(name: string, target: any, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void, scene?: Scene): Nullable<Animatable>;
        /**
         * Create and start an animation on a node and its descendants
         * @param name defines the name of the global animation that will be run on all nodes
         * @param node defines the root node where the animation will take place
         * @param directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used
         * @param targetProperty defines property to animate
         * @param framePerSecond defines the number of frame per second to use
         * @param totalFrame defines the number of frames in total
         * @param from defines the initial value
         * @param to defines the final value
         * @param loopMode defines which loop mode you want to use (off by default)
         * @param easingFunction defines the easing function to use (linear by default)
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of animatables created for all nodes
         * @example https://www.babylonjs-playground.com/#MH0VLI
         */
        static CreateAndStartHierarchyAnimation(name: string, node: Node, directDescendantsOnly: boolean, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable[]>;
        /**
         * Creates a new animation, merges it with the existing animations and starts it
         * @param name Name of the animation
         * @param node Node which contains the scene that begins the animations
         * @param targetProperty Specifies which property to animate
         * @param framePerSecond The frames per second of the animation
         * @param totalFrame The total number of frames
         * @param from The frame at the beginning of the animation
         * @param to The frame at the end of the animation
         * @param loopMode Specifies the loop mode of the animation
         * @param easingFunction (Optional) The easing function of the animation, which allow custom mathematical formulas for animations
         * @param onAnimationEnd Callback to run once the animation is complete
         * @returns Nullable animation
         */
        static CreateMergeAndStartAnimation(name: string, node: Node, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Convert the keyframes of an animation to be relative to a given reference frame.
         * @param sourceAnimation defines the Animation containing keyframes to convert
         * @param referenceFrame defines the frame that keyframes in the range will be relative to (default: 0)
         * @param range defines the name of the AnimationRange belonging to the Animation to convert
         * @param cloneOriginal defines whether or not to clone the animation and convert the clone or convert the original animation (default is false)
         * @param clonedName defines the name of the resulting cloned Animation if cloneOriginal is true
         * @returns a new Animation if cloneOriginal is true or the original Animation if cloneOriginal is false
         */
        static MakeAnimationAdditive(sourceAnimation: Animation, referenceFrame?: number, range?: string, cloneOriginal?: boolean, clonedName?: string): Animation;
        /**
         * Convert the keyframes of an animation to be relative to a given reference frame.
         * @param sourceAnimation defines the Animation containing keyframes to convert
         * @param options defines the options to use when converting ey keyframes
         * @returns a new Animation if options.cloneOriginalAnimation is true or the original Animation if options.cloneOriginalAnimation is false
         */
        static MakeAnimationAdditive(sourceAnimation: Animation, options?: IMakeAnimationAdditiveOptions): Animation;
        /**
         * Transition property of an host to the target Value
         * @param property The property to transition
         * @param targetValue The target Value of the property
         * @param host The object where the property to animate belongs
         * @param scene Scene used to run the animation
         * @param frameRate Framerate (in frame/s) to use
         * @param transition The transition type we want to use
         * @param duration The duration of the animation, in milliseconds
         * @param onAnimationEnd Callback trigger at the end of the animation
         * @returns Nullable animation
         */
        static TransitionTo(property: string, targetValue: any, host: any, scene: Scene, frameRate: number, transition: Animation, duration: number, onAnimationEnd?: Nullable<() => void>): Nullable<Animatable>;
        /**
         * Return the array of runtime animations currently using this animation
         */
        get runtimeAnimations(): RuntimeAnimation[];
        /**
         * Specifies if any of the runtime animations are currently running
         */
        get hasRunningRuntimeAnimations(): boolean;
        /**
         * Initializes the animation
         * @param name Name of the animation
         * @param targetProperty Property to animate
         * @param framePerSecond The frames per second of the animation
         * @param dataType The data type of the animation
         * @param loopMode The loop mode of the animation
         * @param enableBlending Specifies if blending should be enabled
         */
        constructor(
        /**Name of the animation */
        name: string, 
        /**Property to animate */
        targetProperty: string, 
        /**The frames per second of the animation */
        framePerSecond: number, 
        /**The data type of the animation */
        dataType: number, 
        /**The loop mode of the animation */
        loopMode?: number | undefined, 
        /**Specifies if blending should be enabled */
        enableBlending?: boolean | undefined);
        /**
         * Converts the animation to a string
         * @param fullDetails support for multiple levels of logging within scene loading
         * @returns String form of the animation
         */
        toString(fullDetails?: boolean): string;
        /**
         * Add an event to this animation
         * @param event Event to add
         */
        addEvent(event: AnimationEvent): void;
        /**
         * Remove all events found at the given frame
         * @param frame The frame to remove events from
         */
        removeEvents(frame: number): void;
        /**
         * Retrieves all the events from the animation
         * @returns Events from the animation
         */
        getEvents(): AnimationEvent[];
        /**
         * Creates an animation range
         * @param name Name of the animation range
         * @param from Starting frame of the animation range
         * @param to Ending frame of the animation
         */
        createRange(name: string, from: number, to: number): void;
        /**
         * Deletes an animation range by name
         * @param name Name of the animation range to delete
         * @param deleteFrames Specifies if the key frames for the range should also be deleted (true) or not (false)
         */
        deleteRange(name: string, deleteFrames?: boolean): void;
        /**
         * Gets the animation range by name, or null if not defined
         * @param name Name of the animation range
         * @returns Nullable animation range
         */
        getRange(name: string): Nullable<AnimationRange>;
        /**
         * Gets the key frames from the animation
         * @returns The key frames of the animation
         */
        getKeys(): Array<IAnimationKey>;
        /**
         * Gets the highest frame rate of the animation
         * @returns Highest frame rate of the animation
         */
        getHighestFrame(): number;
        /**
         * Gets the easing function of the animation
         * @returns Easing function of the animation
         */
        getEasingFunction(): Nullable<IEasingFunction>;
        /**
         * Sets the easing function of the animation
         * @param easingFunction A custom mathematical formula for animation
         */
        setEasingFunction(easingFunction: Nullable<IEasingFunction>): void;
        /**
         * Interpolates a scalar linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated scalar value
         */
        floatInterpolateFunction(startValue: number, endValue: number, gradient: number): number;
        /**
         * Interpolates a scalar cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated scalar value
         */
        floatInterpolateFunctionWithTangents(startValue: number, outTangent: number, endValue: number, inTangent: number, gradient: number): number;
        /**
         * Interpolates a quaternion using a spherical linear interpolation
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated quaternion value
         */
        quaternionInterpolateFunction(startValue: Quaternion, endValue: Quaternion, gradient: number): Quaternion;
        /**
         * Interpolates a quaternion cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation curve
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated quaternion value
         */
        quaternionInterpolateFunctionWithTangents(startValue: Quaternion, outTangent: Quaternion, endValue: Quaternion, inTangent: Quaternion, gradient: number): Quaternion;
        /**
         * Interpolates a Vector3 linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate (value between 0 and 1)
         * @returns Interpolated scalar value
         */
        vector3InterpolateFunction(startValue: Vector3, endValue: Vector3, gradient: number): Vector3;
        /**
         * Interpolates a Vector3 cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate (value between 0 and 1)
         * @returns InterpolatedVector3 value
         */
        vector3InterpolateFunctionWithTangents(startValue: Vector3, outTangent: Vector3, endValue: Vector3, inTangent: Vector3, gradient: number): Vector3;
        /**
         * Interpolates a Vector2 linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate (value between 0 and 1)
         * @returns Interpolated Vector2 value
         */
        vector2InterpolateFunction(startValue: Vector2, endValue: Vector2, gradient: number): Vector2;
        /**
         * Interpolates a Vector2 cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate (value between 0 and 1)
         * @returns Interpolated Vector2 value
         */
        vector2InterpolateFunctionWithTangents(startValue: Vector2, outTangent: Vector2, endValue: Vector2, inTangent: Vector2, gradient: number): Vector2;
        /**
         * Interpolates a size linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Size value
         */
        sizeInterpolateFunction(startValue: Size, endValue: Size, gradient: number): Size;
        /**
         * Interpolates a Color3 linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Color3 value
         */
        color3InterpolateFunction(startValue: Color3, endValue: Color3, gradient: number): Color3;
        /**
         * Interpolates a Color3 cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns interpolated value
         */
        color3InterpolateFunctionWithTangents(startValue: Color3, outTangent: Color3, endValue: Color3, inTangent: Color3, gradient: number): Color3;
        /**
         * Interpolates a Color4 linearly
         * @param startValue Start value of the animation curve
         * @param endValue End value of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns Interpolated Color3 value
         */
        color4InterpolateFunction(startValue: Color4, endValue: Color4, gradient: number): Color4;
        /**
         * Interpolates a Color4 cubically
         * @param startValue Start value of the animation curve
         * @param outTangent End tangent of the animation
         * @param endValue End value of the animation curve
         * @param inTangent Start tangent of the animation curve
         * @param gradient Scalar amount to interpolate
         * @returns interpolated value
         */
        color4InterpolateFunctionWithTangents(startValue: Color4, outTangent: Color4, endValue: Color4, inTangent: Color4, gradient: number): Color4;
        /**
         * @internal Internal use only
         */
        _getKeyValue(value: any): any;
        /**
         * Evaluate the animation value at a given frame
         * @param currentFrame defines the frame where we want to evaluate the animation
         * @returns the animation value
         */
        evaluate(currentFrame: number): any;
        /**
         * @internal Internal use only
         */
        _interpolate(currentFrame: number, state: _IAnimationState, searchClosestKeyOnly?: boolean): any;
        /**
         * Defines the function to use to interpolate matrices
         * @param startValue defines the start matrix
         * @param endValue defines the end matrix
         * @param gradient defines the gradient between both matrices
         * @param result defines an optional target matrix where to store the interpolation
         * @returns the interpolated matrix
         */
        matrixInterpolateFunction(startValue: Matrix, endValue: Matrix, gradient: number, result?: Matrix): Matrix;
        /**
         * Makes a copy of the animation
         * @returns Cloned animation
         */
        clone(): Animation;
        /**
         * Sets the key frames of the animation
         * @param values The animation key frames to set
         * @param dontClone Whether to clone the keys or not (default is false, so the array of keys is cloned)
         */
        setKeys(values: Array<IAnimationKey>, dontClone?: boolean): void;
        /**
         * Creates a key for the frame passed as a parameter and adds it to the animation IF a key doesn't already exist for that frame
         * @param frame Frame number
         * @returns The key index if the key was added or the index of the pre existing key if the frame passed as parameter already has a corresponding key
         */
        createKeyForFrame(frame: number): number;
        /**
         * Serializes the animation to an object
         * @returns Serialized object
         */
        serialize(): any;
        /**
         * Float animation type
         */
        static readonly ANIMATIONTYPE_FLOAT = 0;
        /**
         * Vector3 animation type
         */
        static readonly ANIMATIONTYPE_VECTOR3 = 1;
        /**
         * Quaternion animation type
         */
        static readonly ANIMATIONTYPE_QUATERNION = 2;
        /**
         * Matrix animation type
         */
        static readonly ANIMATIONTYPE_MATRIX = 3;
        /**
         * Color3 animation type
         */
        static readonly ANIMATIONTYPE_COLOR3 = 4;
        /**
         * Color3 animation type
         */
        static readonly ANIMATIONTYPE_COLOR4 = 7;
        /**
         * Vector2 animation type
         */
        static readonly ANIMATIONTYPE_VECTOR2 = 5;
        /**
         * Size animation type
         */
        static readonly ANIMATIONTYPE_SIZE = 6;
        /**
         * Relative Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_RELATIVE = 0;
        /**
         * Cycle Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_CYCLE = 1;
        /**
         * Constant Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_CONSTANT = 2;
        /**
         * Yoyo Loop Mode
         */
        static readonly ANIMATIONLOOPMODE_YOYO = 4;
        /**
         * Relative Loop Mode (add to current value of animated object, unlike ANIMATIONLOOPMODE_RELATIVE)
         */
        static readonly ANIMATIONLOOPMODE_RELATIVE_FROM_CURRENT = 5;
        /**
         * @internal
         */
        static _UniversalLerp(left: any, right: any, amount: number): any;
        /**
         * Parses an animation object and creates an animation
         * @param parsedAnimation Parsed animation object
         * @returns Animation object
         */
        static Parse(parsedAnimation: any): Animation;
        /**
         * Appends the serialized animations from the source animations
         * @param source Source containing the animations
         * @param destination Target to store the animations
         */
        static AppendSerializedAnimations(source: IAnimatable, destination: any): void;
        /**
         * Creates a new animation or an array of animations from a snippet saved in a remote file
         * @param name defines the name of the animation to create (can be null or empty to use the one from the json data)
         * @param url defines the url to load from
         * @returns a promise that will resolve to the new animation or an array of animations
         */
        static ParseFromFileAsync(name: Nullable<string>, url: string): Promise<Animation | Array<Animation>>;
        /**
         * Creates an animation or an array of animations from a snippet saved by the Inspector
         * @param snippetId defines the snippet to load
         * @returns a promise that will resolve to the new animation or a new array of animations
         */
        static ParseFromSnippetAsync(snippetId: string): Promise<Animation | Array<Animation>>;
        /**
         * Creates an animation or an array of animations from a snippet saved by the Inspector
         * @deprecated Please use ParseFromSnippetAsync instead
         * @param snippetId defines the snippet to load
         * @returns a promise that will resolve to the new animation or a new array of animations
         */
        static CreateFromSnippetAsync: typeof Animation.ParseFromSnippetAsync;
    }


    /**
     * Composed of a frame, and an action function
     */
    export class AnimationEvent {
        /** The frame for which the event is triggered **/
        frame: number;
        /** The event to perform when triggered **/
        action: (currentFrame: number) => void;
        /** Specifies if the event should be triggered only once**/
        onlyOnce?: boolean | undefined;
        /**
         * Specifies if the animation event is done
         */
        isDone: boolean;
        /**
         * Initializes the animation event
         * @param frame The frame for which the event is triggered
         * @param action The event to perform when triggered
         * @param onlyOnce Specifies if the event should be triggered only once
         */
        constructor(
        /** The frame for which the event is triggered **/
        frame: number, 
        /** The event to perform when triggered **/
        action: (currentFrame: number) => void, 
        /** Specifies if the event should be triggered only once**/
        onlyOnce?: boolean | undefined);
        /** @internal */
        _clone(): AnimationEvent;
    }


    /**
     * This class defines the direct association between an animation and a target
     */
    export class TargetedAnimation {
        /**
         * Animation to perform
         */
        animation: Animation;
        /**
         * Target to animate
         */
        target: any;
        /**
         * Returns the string "TargetedAnimation"
         * @returns "TargetedAnimation"
         */
        getClassName(): string;
        /**
         * Serialize the object
         * @returns the JSON object representing the current entity
         */
        serialize(): any;
    }
    /**
     * Options to be used when creating an additive group animation
     */
    export interface IMakeAnimationGroupAdditiveOptions extends IMakeAnimationAdditiveOptions {
        /**
         * Defines if the animation group should be cloned or not (default is false)
         */
        cloneOriginalAnimationGroup?: boolean;
        /**
         * The name of the cloned animation group if cloneOriginalAnimationGroup is true
         */
        clonedAnimationGroupName?: string;
    }
    /**
     * Use this class to create coordinated animations on multiple targets
     */
    export class AnimationGroup implements IDisposable {
        /** The name of the animation group */
        name: string;
        private _scene;
        private _targetedAnimations;
        private _animatables;
        private _from;
        private _to;
        private _isStarted;
        private _isPaused;
        private _speedRatio;
        private _loopAnimation;
        private _isAdditive;
        private _weight;
        private _playOrder;
        private _enableBlending;
        private _blendingSpeed;
        private _numActiveAnimatables;
        /** @internal */
        _parentContainer: Nullable<AbstractScene>;
        /**
         * Gets or sets the unique id of the node
         */
        uniqueId: number;
        /**
         * This observable will notify when one animation have ended
         */
        onAnimationEndObservable: Observable<TargetedAnimation>;
        /**
         * Observer raised when one animation loops
         */
        onAnimationLoopObservable: Observable<TargetedAnimation>;
        /**
         * Observer raised when all animations have looped
         */
        onAnimationGroupLoopObservable: Observable<AnimationGroup>;
        /**
         * This observable will notify when all animations have ended.
         */
        onAnimationGroupEndObservable: Observable<AnimationGroup>;
        /**
         * This observable will notify when all animations have paused.
         */
        onAnimationGroupPauseObservable: Observable<AnimationGroup>;
        /**
         * This observable will notify when all animations are playing.
         */
        onAnimationGroupPlayObservable: Observable<AnimationGroup>;
        /**
         * Gets or sets an object used to store user defined information for the node
         */
        metadata: any;
        /**
         * Gets or sets the mask associated with this animation group. This mask is used to filter which objects should be animated.
         */
        mask?: AnimationGroupMask;
        /**
         * Makes sure that the animations are either played or stopped according to the animation group mask.
         * Note however that the call won't have any effect if the animation group has not been started yet.
         * You should call this function if you modify the mask after the animation group has been started.
         */
        syncWithMask(): void;
        /**
         * Removes all animations for the targets not retained by the animation group mask.
         * Use this function if you know you won't need those animations anymore and if you want to free memory.
         */
        removeUnmaskedAnimations(): void;
        /**
         * Gets or sets the first frame
         */
        get from(): number;
        set from(value: number);
        /**
         * Gets or sets the last frame
         */
        get to(): number;
        set to(value: number);
        /**
         * Define if the animations are started
         */
        get isStarted(): boolean;
        /**
         * Gets a value indicating that the current group is playing
         */
        get isPlaying(): boolean;
        /**
         * Gets or sets the speed ratio to use for all animations
         */
        get speedRatio(): number;
        /**
         * Gets or sets the speed ratio to use for all animations
         */
        set speedRatio(value: number);
        /**
         * Gets or sets if all animations should loop or not
         */
        get loopAnimation(): boolean;
        set loopAnimation(value: boolean);
        /**
         * Gets or sets if all animations should be evaluated additively
         */
        get isAdditive(): boolean;
        set isAdditive(value: boolean);
        /**
         * Gets or sets the weight to apply to all animations of the group
         */
        get weight(): number;
        set weight(value: number);
        /**
         * Gets the targeted animations for this animation group
         */
        get targetedAnimations(): Array<TargetedAnimation>;
        /**
         * returning the list of animatables controlled by this animation group.
         */
        get animatables(): Array<Animatable>;
        /**
         * Gets the list of target animations
         */
        get children(): TargetedAnimation[];
        /**
         * Gets or sets the order of play of the animation group (default: 0)
         */
        get playOrder(): number;
        set playOrder(value: number);
        /**
         * Allows the animations of the animation group to blend with current running animations
         * Note that a null value means that each animation will use their own existing blending configuration (Animation.enableBlending)
         */
        get enableBlending(): Nullable<boolean>;
        set enableBlending(value: Nullable<boolean>);
        /**
         * Gets or sets the animation blending speed
         * Note that a null value means that each animation will use their own existing blending configuration (Animation.blendingSpeed)
         */
        get blendingSpeed(): Nullable<number>;
        set blendingSpeed(value: Nullable<number>);
        /**
         * Gets the length (in seconds) of the animation group
         * This function assumes that all animations are played at the same framePerSecond speed!
         * Note: you can only call this method after you've added at least one targeted animation!
         * @param from Starting frame range (default is AnimationGroup.from)
         * @param to Ending frame range (default is AnimationGroup.to)
         * @returns The length in seconds
         */
        getLength(from?: number, to?: number): number;
        /**
         * Merge the array of animation groups into a new animation group
         * @param animationGroups List of animation groups to merge
         * @param disposeSource If true, animation groups will be disposed after being merged (default: true)
         * @param normalize If true, animation groups will be normalized before being merged, so that all animations have the same "from" and "to" frame (default: false)
         * @param weight Weight for the new animation group. If not provided, it will inherit the weight from the first animation group of the array
         * @returns The new animation group or null if no animation groups were passed
         */
        static MergeAnimationGroups(animationGroups: Array<AnimationGroup>, disposeSource?: boolean, normalize?: boolean, weight?: number): Nullable<AnimationGroup>;
        /**
         * Instantiates a new Animation Group.
         * This helps managing several animations at once.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/groupAnimations
         * @param name Defines the name of the group
         * @param scene Defines the scene the group belongs to
         * @param weight Defines the weight to use for animations in the group (-1.0 by default, meaning "no weight")
         * @param playOrder Defines the order of play of the animation group (default is 0)
         */
        constructor(
        /** The name of the animation group */
        name: string, scene?: Nullable<Scene>, weight?: number, playOrder?: number);
        /**
         * Add an animation (with its target) in the group
         * @param animation defines the animation we want to add
         * @param target defines the target of the animation
         * @returns the TargetedAnimation object
         */
        addTargetedAnimation(animation: Animation, target: any): TargetedAnimation;
        /**
         * Remove an animation from the group
         * @param animation defines the animation we want to remove
         */
        removeTargetedAnimation(animation: Animation): void;
        /**
         * This function will normalize every animation in the group to make sure they all go from beginFrame to endFrame
         * It can add constant keys at begin or end
         * @param beginFrame defines the new begin frame for all animations or the smallest begin frame of all animations if null (defaults to null)
         * @param endFrame defines the new end frame for all animations or the largest end frame of all animations if null (defaults to null)
         * @returns the animation group
         */
        normalize(beginFrame?: Nullable<number>, endFrame?: Nullable<number>): AnimationGroup;
        private _animationLoopCount;
        private _animationLoopFlags;
        private _processLoop;
        /**
         * Start all animations on given targets
         * @param loop defines if animations must loop
         * @param speedRatio defines the ratio to apply to animation speed (1 by default)
         * @param from defines the from key (optional)
         * @param to defines the to key (optional)
         * @param isAdditive defines the additive state for the resulting animatables (optional)
         * @returns the current animation group
         */
        start(loop?: boolean, speedRatio?: number, from?: number, to?: number, isAdditive?: boolean): AnimationGroup;
        /**
         * Pause all animations
         * @returns the animation group
         */
        pause(): AnimationGroup;
        /**
         * Play all animations to initial state
         * This function will start() the animations if they were not started or will restart() them if they were paused
         * @param loop defines if animations must loop
         * @returns the animation group
         */
        play(loop?: boolean): AnimationGroup;
        /**
         * Reset all animations to initial state
         * @returns the animation group
         */
        reset(): AnimationGroup;
        /**
         * Restart animations from key 0
         * @returns the animation group
         */
        restart(): AnimationGroup;
        /**
         * Stop all animations
         * @returns the animation group
         */
        stop(): AnimationGroup;
        /**
         * Set animation weight for all animatables
         *
         * @since 6.12.4
         *  You can pass the weight to the AnimationGroup constructor, or use the weight property to set it after the group has been created,
         *  making it easier to define the overall animation weight than calling setWeightForAllAnimatables() after the animation group has been started
         * @param weight defines the weight to use
         * @returns the animationGroup
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#animation-weights
         */
        setWeightForAllAnimatables(weight: number): AnimationGroup;
        /**
         * Synchronize and normalize all animatables with a source animatable
         * @param root defines the root animatable to synchronize with (null to stop synchronizing)
         * @returns the animationGroup
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#animation-weights
         */
        syncAllAnimationsWith(root: Nullable<Animatable>): AnimationGroup;
        /**
         * Goes to a specific frame in this animation group
         * @param frame the frame number to go to
         * @returns the animationGroup
         */
        goToFrame(frame: number): AnimationGroup;
        /**
         * Dispose all associated resources
         */
        dispose(): void;
        private _checkAnimationGroupEnded;
        /**
         * Clone the current animation group and returns a copy
         * @param newName defines the name of the new group
         * @param targetConverter defines an optional function used to convert current animation targets to new ones
         * @param cloneAnimations defines if the animations should be cloned or referenced
         * @returns the new animation group
         */
        clone(newName: string, targetConverter?: (oldTarget: any) => any, cloneAnimations?: boolean): AnimationGroup;
        /**
         * Serializes the animationGroup to an object
         * @returns Serialized object
         */
        serialize(): any;
        /**
         * Returns a new AnimationGroup object parsed from the source provided.
         * @param parsedAnimationGroup defines the source
         * @param scene defines the scene that will receive the animationGroup
         * @returns a new AnimationGroup
         */
        static Parse(parsedAnimationGroup: any, scene: Scene): AnimationGroup;
        /**
         * Convert the keyframes for all animations belonging to the group to be relative to a given reference frame.
         * @param sourceAnimationGroup defines the AnimationGroup containing animations to convert
         * @param referenceFrame defines the frame that keyframes in the range will be relative to (default: 0)
         * @param range defines the name of the AnimationRange belonging to the animations in the group to convert
         * @param cloneOriginal defines whether or not to clone the group and convert the clone or convert the original group (default is false)
         * @param clonedName defines the name of the resulting cloned AnimationGroup if cloneOriginal is true
         * @returns a new AnimationGroup if cloneOriginal is true or the original AnimationGroup if cloneOriginal is false
         */
        static MakeAnimationAdditive(sourceAnimationGroup: AnimationGroup, referenceFrame: number, range?: string, cloneOriginal?: boolean, clonedName?: string): AnimationGroup;
        /**
         * Convert the keyframes for all animations belonging to the group to be relative to a given reference frame.
         * @param sourceAnimationGroup defines the AnimationGroup containing animations to convert
         * @param options defines the options to use when converting keyframes
         * @returns a new AnimationGroup if options.cloneOriginalAnimationGroup is true or the original AnimationGroup if options.cloneOriginalAnimationGroup is false
         */
        static MakeAnimationAdditive(sourceAnimationGroup: AnimationGroup, options?: IMakeAnimationGroupAdditiveOptions): AnimationGroup;
        /**
         * Creates a new animation, keeping only the keys that are inside a given key range
         * @param sourceAnimationGroup defines the animation group on which to operate
         * @param fromKey defines the lower bound of the range
         * @param toKey defines the upper bound of the range
         * @param name defines the name of the new animation group. If not provided, use the same name as animationGroup
         * @param dontCloneAnimations defines whether or not the animations should be cloned before clipping the keys. Default is false, so animations will be cloned
         * @returns a new animation group stripped from all the keys outside the given range
         */
        static ClipKeys(sourceAnimationGroup: AnimationGroup, fromKey: number, toKey: number, name?: string, dontCloneAnimations?: boolean): AnimationGroup;
        /**
         * Updates an existing animation, keeping only the keys that are inside a given key range
         * @param animationGroup defines the animation group on which to operate
         * @param fromKey defines the lower bound of the range
         * @param toKey defines the upper bound of the range
         * @param dontCloneAnimations defines whether or not the animations should be cloned before clipping the keys. Default is false, so animations will be cloned
         * @returns the animationGroup stripped from all the keys outside the given range
         */
        static ClipKeysInPlace(animationGroup: AnimationGroup, fromKey: number, toKey: number, dontCloneAnimations?: boolean): AnimationGroup;
        /**
         * Creates a new animation, keeping only the frames that are inside a given frame range
         * @param sourceAnimationGroup defines the animation group on which to operate
         * @param fromFrame defines the lower bound of the range
         * @param toFrame defines the upper bound of the range
         * @param name defines the name of the new animation group. If not provided, use the same name as animationGroup
         * @param dontCloneAnimations defines whether or not the animations should be cloned before clipping the frames. Default is false, so animations will be cloned
         * @returns a new animation group stripped from all the frames outside the given range
         */
        static ClipFrames(sourceAnimationGroup: AnimationGroup, fromFrame: number, toFrame: number, name?: string, dontCloneAnimations?: boolean): AnimationGroup;
        /**
         * Updates an existing animation, keeping only the frames that are inside a given frame range
         * @param animationGroup defines the animation group on which to operate
         * @param fromFrame defines the lower bound of the range
         * @param toFrame defines the upper bound of the range
         * @param dontCloneAnimations defines whether or not the animations should be cloned before clipping the frames. Default is false, so animations will be cloned
         * @returns the animationGroup stripped from all the frames outside the given range
         */
        static ClipFramesInPlace(animationGroup: AnimationGroup, fromFrame: number, toFrame: number, dontCloneAnimations?: boolean): AnimationGroup;
        /**
         * Updates an existing animation, keeping only the keys that are inside a given key or frame range
         * @param animationGroup defines the animation group on which to operate
         * @param start defines the lower bound of the range
         * @param end defines the upper bound of the range
         * @param dontCloneAnimations defines whether or not the animations should be cloned before clipping the keys. Default is false, so animations will be cloned
         * @param useFrame defines if the range is defined by frame numbers or key indices (default is false which means use key indices)
         * @returns the animationGroup stripped from all the keys outside the given range
         */
        static ClipInPlace(animationGroup: AnimationGroup, start: number, end: number, dontCloneAnimations?: boolean, useFrame?: boolean): AnimationGroup;
        /**
         * Returns the string "AnimationGroup"
         * @returns "AnimationGroup"
         */
        getClassName(): string;
        /**
         * Creates a detailed string about the object
         * @param fullDetails defines if the output string will support multiple levels of logging within scene loading
         * @returns a string representing the object
         */
        toString(fullDetails?: boolean): string;
    }


    /**
     * Enum used to define the mode for an animation group mask
     */
    export enum AnimationGroupMaskMode {
        /**
         * The mask defines the animatable target names that should be included
         */
        Include = 0,
        /**
         * The mask defines the animatable target names in a "exclude" mode: all animatable targets will be animated except the ones defined in the mask
         */
        Exclude = 1
    }
    /**
     * Defines a mask used to filter animation targets.
     * If you apply a mask to an animation group (see the AnimationGroup.mask property), only the animations whose target names match the mask will play.
     * Note that a target is defined by its name (string). This means that the same mask can be used for several animation groups, provided that their targets are named in the same way.
     */
    export class AnimationGroupMask {
        /**
         * Defines the mode for the mask
         */
        mode: AnimationGroupMaskMode;
        /**
         * The set of target names included in the mask. If mode is AnimationGroupMaskMode.Exclude, the targets in this set will be excluded from the mask instead.
         */
        private _targetNames;
        /**
         * Gets or sets a boolean indicating if the mask is disabled (default is false)
         */
        disabled: boolean;
        /**
         * Creates a new mask
         * @param names The list of target names to add to the mask (optional)
         * @param mode Defines the mode for the mask (default: AnimationGroupMaskMode.Include)
         */
        constructor(names?: string[], 
        /**
         * Defines the mode for the mask
         */
        mode?: AnimationGroupMaskMode);
        /**
         * Adds one or several target names to the mask
         * @param name The name(s) to add to the mask
         */
        addTargetName(name: string | string[]): void;
        /**
         * Removes one or several target names from the mask
         * @param name The name(s) to remove from the mask
         */
        removeTargetName(name: string | string[]): void;
        /**
         * Checks if the mask includes a target name.
         * This method is intended to know if a given target name is included in the mask, not if the name is actually retained by the mask (see retainsTarget() instead).
         * @param name The name to check with the mask
         * @returns True if the mask includes the name, false otherwise
         */
        hasTarget(name: string): boolean;
        /**
         * Checks if the mask retains a target name.
         * Note that in the "Exclude" mode, this will return false if the mask includes the name, and true otherwise!
         * This method is intended to know if a given target name is retained by the mask, not if the name is in the list of target names.
         * @param name The name to check with the mask
         * @returns True if the mask retains the name, false otherwise
         */
        retainsTarget(name: string): boolean;
    }


    /**
     * Defines an interface which represents an animation key frame
     */
    export interface IAnimationKey {
        /**
         * Frame of the key frame
         */
        frame: number;
        /**
         * Value at the specifies key frame
         */
        value: any;
        /**
         * The input tangent for the cubic hermite spline
         */
        inTangent?: any;
        /**
         * The output tangent for the cubic hermite spline
         */
        outTangent?: any;
        /**
         * The animation interpolation type
         */
        interpolation?: AnimationKeyInterpolation;
        /**
         * Property defined by UI tools to link (or not ) the tangents
         */
        lockedTangent?: boolean;
        /**
         * The easing function associated with the key frame (optional). If not defined, the easing function defined at the animation level (if any) will be used instead
         */
        easingFunction?: IEasingFunction;
    }
    /**
     * Enum for the animation key frame interpolation type
     */
    export enum AnimationKeyInterpolation {
        /**
         * Use tangents to interpolate between start and end values.
         */
        NONE = 0,
        /**
         * Do not interpolate between keys and use the start key value only. Tangents are ignored
         */
        STEP = 1
    }


    /**
     * Class used to override all child animations of a given target
     */
    export class AnimationPropertiesOverride {
        /**
         * Gets or sets a value indicating if animation blending must be used
         */
        enableBlending: boolean;
        /**
         * Gets or sets the blending speed to use when enableBlending is true
         */
        blendingSpeed: number;
        /**
         * Gets or sets the default loop mode to use
         */
        loopMode: number;
    }


    /**
     * Represents the range of an animation
     */
    export class AnimationRange {
        /**The name of the animation range**/
        name: string;
        /**The starting frame of the animation */
        from: number;
        /**The ending frame of the animation*/
        to: number;
        /**
         * Initializes the range of an animation
         * @param name The name of the animation range
         * @param from The starting frame of the animation
         * @param to The ending frame of the animation
         */
        constructor(
        /**The name of the animation range**/
        name: string, 
        /**The starting frame of the animation */
        from: number, 
        /**The ending frame of the animation*/
        to: number);
        /**
         * Makes a copy of the animation range
         * @returns A copy of the animation range
         */
        clone(): AnimationRange;
    }


    /**
     * This represents the main contract an easing function should follow.
     * Easing functions are used throughout the animation system.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export interface IEasingFunction {
        /**
         * Given an input gradient between 0 and 1, this returns the corresponding value
         * of the easing function.
         * The link below provides some of the most common examples of easing functions.
         * @see https://easings.net/
         * @param gradient Defines the value between 0 and 1 we want the easing value for
         * @returns the corresponding value on the curve defined by the easing function
         */
        ease(gradient: number): number;
    }
    /**
     * Base class used for every default easing function.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class EasingFunction implements IEasingFunction {
        /**
         * Interpolation follows the mathematical formula associated with the easing function.
         */
        static readonly EASINGMODE_EASEIN = 0;
        /**
         * Interpolation follows 100% interpolation minus the output of the formula associated with the easing function.
         */
        static readonly EASINGMODE_EASEOUT = 1;
        /**
         * Interpolation uses EaseIn for the first half of the animation and EaseOut for the second half.
         */
        static readonly EASINGMODE_EASEINOUT = 2;
        private _easingMode;
        /**
         * Sets the easing mode of the current function.
         * @param easingMode Defines the willing mode (EASINGMODE_EASEIN, EASINGMODE_EASEOUT or EASINGMODE_EASEINOUT)
         */
        setEasingMode(easingMode: number): void;
        /**
         * Gets the current easing mode.
         * @returns the easing mode
         */
        getEasingMode(): number;
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
        /**
         * Given an input gradient between 0 and 1, this returns the corresponding value
         * of the easing function.
         * @param gradient Defines the value between 0 and 1 we want the easing value for
         * @returns the corresponding value on the curve defined by the easing function
         */
        ease(gradient: number): number;
    }
    /**
     * Easing function with a circle shape (see link below).
     * @see https://easings.net/#easeInCirc
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class CircleEase extends EasingFunction implements IEasingFunction {
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a ease back shape (see link below).
     * @see https://easings.net/#easeInBack
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class BackEase extends EasingFunction implements IEasingFunction {
        /** Defines the amplitude of the function */
        amplitude: number;
        /**
         * Instantiates a back ease easing
         * @see https://easings.net/#easeInBack
         * @param amplitude Defines the amplitude of the function
         */
        constructor(
        /** Defines the amplitude of the function */
        amplitude?: number);
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a bouncing shape (see link below).
     * @see https://easings.net/#easeInBounce
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class BounceEase extends EasingFunction implements IEasingFunction {
        /** Defines the number of bounces */
        bounces: number;
        /** Defines the amplitude of the bounce */
        bounciness: number;
        /**
         * Instantiates a bounce easing
         * @see https://easings.net/#easeInBounce
         * @param bounces Defines the number of bounces
         * @param bounciness Defines the amplitude of the bounce
         */
        constructor(
        /** Defines the number of bounces */
        bounces?: number, 
        /** Defines the amplitude of the bounce */
        bounciness?: number);
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 3 shape (see link below).
     * @see https://easings.net/#easeInCubic
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class CubicEase extends EasingFunction implements IEasingFunction {
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with an elastic shape (see link below).
     * @see https://easings.net/#easeInElastic
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class ElasticEase extends EasingFunction implements IEasingFunction {
        /** Defines the number of oscillations*/
        oscillations: number;
        /** Defines the amplitude of the oscillations*/
        springiness: number;
        /**
         * Instantiates an elastic easing function
         * @see https://easings.net/#easeInElastic
         * @param oscillations Defines the number of oscillations
         * @param springiness Defines the amplitude of the oscillations
         */
        constructor(
        /** Defines the number of oscillations*/
        oscillations?: number, 
        /** Defines the amplitude of the oscillations*/
        springiness?: number);
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with an exponential shape (see link below).
     * @see https://easings.net/#easeInExpo
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class ExponentialEase extends EasingFunction implements IEasingFunction {
        /** Defines the exponent of the function */
        exponent: number;
        /**
         * Instantiates an exponential easing function
         * @see https://easings.net/#easeInExpo
         * @param exponent Defines the exponent of the function
         */
        constructor(
        /** Defines the exponent of the function */
        exponent?: number);
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power shape (see link below).
     * @see https://easings.net/#easeInQuad
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class PowerEase extends EasingFunction implements IEasingFunction {
        /** Defines the power of the function */
        power: number;
        /**
         * Instantiates an power base easing function
         * @see https://easings.net/#easeInQuad
         * @param power Defines the power of the function
         */
        constructor(
        /** Defines the power of the function */
        power?: number);
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 2 shape (see link below).
     * @see https://easings.net/#easeInQuad
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class QuadraticEase extends EasingFunction implements IEasingFunction {
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 4 shape (see link below).
     * @see https://easings.net/#easeInQuart
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class QuarticEase extends EasingFunction implements IEasingFunction {
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a power of 5 shape (see link below).
     * @see https://easings.net/#easeInQuint
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class QuinticEase extends EasingFunction implements IEasingFunction {
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a sin shape (see link below).
     * @see https://easings.net/#easeInSine
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class SineEase extends EasingFunction implements IEasingFunction {
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }
    /**
     * Easing function with a bezier shape (see link below).
     * @see http://cubic-bezier.com/#.17,.67,.83,.67
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#easing-functions
     */
    export class BezierCurveEase extends EasingFunction implements IEasingFunction {
        /** Defines the x component of the start tangent in the bezier curve */
        x1: number;
        /** Defines the y component of the start tangent in the bezier curve */
        y1: number;
        /** Defines the x component of the end tangent in the bezier curve */
        x2: number;
        /** Defines the y component of the end tangent in the bezier curve */
        y2: number;
        /**
         * Instantiates a bezier function
         * @see http://cubic-bezier.com/#.17,.67,.83,.67
         * @param x1 Defines the x component of the start tangent in the bezier curve
         * @param y1 Defines the y component of the start tangent in the bezier curve
         * @param x2 Defines the x component of the end tangent in the bezier curve
         * @param y2 Defines the y component of the end tangent in the bezier curve
         */
        constructor(
        /** Defines the x component of the start tangent in the bezier curve */
        x1?: number, 
        /** Defines the y component of the start tangent in the bezier curve */
        y1?: number, 
        /** Defines the x component of the end tangent in the bezier curve */
        x2?: number, 
        /** Defines the y component of the end tangent in the bezier curve */
        y2?: number);
        /**
         * @internal
         */
        easeInCore(gradient: number): number;
    }




    /**
     * A cursor which tracks a point on a path
     */
    export class PathCursor {
        private _path;
        /**
         * Stores path cursor callbacks for when an onchange event is triggered
         */
        private _onchange;
        /**
         * The value of the path cursor
         */
        value: number;
        /**
         * The animation array of the path cursor
         */
        animations: Animation[];
        /**
         * Initializes the path cursor
         * @param _path The path to track
         */
        constructor(_path: Path2);
        /**
         * Gets the cursor point on the path
         * @returns A point on the path cursor at the cursor location
         */
        getPoint(): Vector3;
        /**
         * Moves the cursor ahead by the step amount
         * @param step The amount to move the cursor forward
         * @returns This path cursor
         */
        moveAhead(step?: number): PathCursor;
        /**
         * Moves the cursor behind by the step amount
         * @param step The amount to move the cursor back
         * @returns This path cursor
         */
        moveBack(step?: number): PathCursor;
        /**
         * Moves the cursor by the step amount
         * If the step amount is greater than one, an exception is thrown
         * @param step The amount to move the cursor
         * @returns This path cursor
         */
        move(step: number): PathCursor;
        /**
         * Ensures that the value is limited between zero and one
         * @returns This path cursor
         */
        private _ensureLimits;
        /**
         * Runs onchange callbacks on change (used by the animation engine)
         * @returns This path cursor
         */
        private _raiseOnChange;
        /**
         * Executes a function on change
         * @param f A path cursor onchange callback
         * @returns This path cursor
         */
        onchange(f: (cursor: PathCursor) => void): PathCursor;
    }


    /**
     * Defines a runtime animation
     */
    export class RuntimeAnimation {
        private _events;
        /**
         * The current frame of the runtime animation
         */
        private _currentFrame;
        /**
         * The animation used by the runtime animation
         */
        private _animation;
        /**
         * The target of the runtime animation
         */
        private _target;
        /**
         * The initiating animatable
         */
        private _host;
        /**
         * The original value of the runtime animation
         */
        private _originalValue;
        /**
         * The original blend value of the runtime animation
         */
        private _originalBlendValue;
        /**
         * The offsets cache of the runtime animation
         */
        private _offsetsCache;
        /**
         * The high limits cache of the runtime animation
         */
        private _highLimitsCache;
        /**
         * Specifies if the runtime animation has been stopped
         */
        private _stopped;
        /**
         * The blending factor of the runtime animation
         */
        private _blendingFactor;
        /**
         * The BabylonJS scene
         */
        private _scene;
        /**
         * The current value of the runtime animation
         */
        private _currentValue;
        /** @internal */
        _animationState: _IAnimationState;
        /**
         * The active target of the runtime animation
         */
        private _activeTargets;
        private _currentActiveTarget;
        private _directTarget;
        /**
         * The target path of the runtime animation
         */
        private _targetPath;
        /**
         * The weight of the runtime animation
         */
        private _weight;
        /**
         * The absolute frame offset of the runtime animation
         */
        private _absoluteFrameOffset;
        /**
         * The previous elapsed time (since start of animation) of the runtime animation
         */
        private _previousElapsedTime;
        /**
         * The previous absolute frame of the runtime animation (meaning, without taking into account the from/to values, only the elapsed time and the fps)
         */
        private _previousAbsoluteFrame;
        private _enableBlending;
        private _keys;
        private _minFrame;
        private _maxFrame;
        private _minValue;
        private _maxValue;
        private _targetIsArray;
        /**
         * Gets the current frame of the runtime animation
         */
        get currentFrame(): number;
        /**
         * Gets the weight of the runtime animation
         */
        get weight(): number;
        /**
         * Gets the current value of the runtime animation
         */
        get currentValue(): any;
        /**
         * Gets or sets the target path of the runtime animation
         */
        get targetPath(): string;
        /**
         * Gets the actual target of the runtime animation
         */
        get target(): any;
        /**
         * Gets the additive state of the runtime animation
         */
        get isAdditive(): boolean;
        /** @internal */
        _onLoop: () => void;
        /**
         * Create a new RuntimeAnimation object
         * @param target defines the target of the animation
         * @param animation defines the source animation object
         * @param scene defines the hosting scene
         * @param host defines the initiating Animatable
         */
        constructor(target: any, animation: Animation, scene: Scene, host: Animatable);
        private _preparePath;
        /**
         * Gets the animation from the runtime animation
         */
        get animation(): Animation;
        /**
         * Resets the runtime animation to the beginning
         * @param restoreOriginal defines whether to restore the target property to the original value
         */
        reset(restoreOriginal?: boolean): void;
        /**
         * Specifies if the runtime animation is stopped
         * @returns Boolean specifying if the runtime animation is stopped
         */
        isStopped(): boolean;
        /**
         * Disposes of the runtime animation
         */
        dispose(): void;
        /**
         * Apply the interpolated value to the target
         * @param currentValue defines the value computed by the animation
         * @param weight defines the weight to apply to this value (Defaults to 1.0)
         */
        setValue(currentValue: any, weight: number): void;
        private _getOriginalValues;
        private _setValue;
        /**
         * Gets the loop pmode of the runtime animation
         * @returns Loop Mode
         */
        private _getCorrectLoopMode;
        /**
         * Move the current animation to a given frame
         * @param frame defines the frame to move to
         */
        goToFrame(frame: number): void;
        /**
         * @internal Internal use only
         */
        _prepareForSpeedRatioChange(newSpeedRatio: number): void;
        /**
         * Execute the current animation
         * @param elapsedTimeSinceAnimationStart defines the elapsed time (in milliseconds) since the animation was started
         * @param from defines the lower frame of the animation range
         * @param to defines the upper frame of the animation range
         * @param loop defines if the current animation must loop
         * @param speedRatio defines the current speed ratio
         * @param weight defines the weight of the animation (default is -1 so no weight)
         * @returns a boolean indicating if the animation is running
         */
        animate(elapsedTimeSinceAnimationStart: number, from: number, to: number, loop: boolean, speedRatio: number, weight?: number): boolean;
    }


    /**
     * Set of assets to keep when moving a scene into an asset container.
     */
    export class KeepAssets extends AbstractScene {
    }
    /**
     * Class used to store the output of the AssetContainer.instantiateAllMeshesToScene function
     */
    export class InstantiatedEntries {
        /**
         * List of new root nodes (eg. nodes with no parent)
         */
        rootNodes: Node[];
        /**
         * List of new skeletons
         */
        skeletons: Skeleton[];
        /**
         * List of new animation groups
         */
        animationGroups: AnimationGroup[];
        /**
         * Disposes the instantiated entries from the scene
         */
        dispose(): void;
    }
    /**
     * Container with a set of assets that can be added or removed from a scene.
     */
    export class AssetContainer extends AbstractScene {
        private _wasAddedToScene;
        private _onContextRestoredObserver;
        /**
         * The scene the AssetContainer belongs to.
         */
        scene: Scene;
        /**
         * Instantiates an AssetContainer.
         * @param scene The scene the AssetContainer belongs to.
         */
        constructor(scene?: Nullable<Scene>);
        /**
         * Given a list of nodes, return a topological sorting of them.
         * @param nodes
         * @returns a sorted array of nodes
         */
        private _topologicalSort;
        private _addNodeAndDescendantsToList;
        /**
         * Check if a specific node is contained in this asset container.
         * @param node the node to check
         * @returns true if the node is contained in this container, otherwise false.
         */
        private _isNodeInContainer;
        /**
         * For every node in the scene, check if its parent node is also in the scene.
         * @returns true if every node's parent is also in the scene, otherwise false.
         */
        private _isValidHierarchy;
        /**
         * Instantiate or clone all meshes and add the new ones to the scene.
         * Skeletons and animation groups will all be cloned
         * @param nameFunction defines an optional function used to get new names for clones
         * @param cloneMaterials defines an optional boolean that defines if materials must be cloned as well (false by default)
         * @param options defines an optional list of options to control how to instantiate / clone models
         * @param options.doNotInstantiate defines if the model must be instantiated or just cloned
         * @param options.predicate defines a predicate used to filter whih mesh to instantiate/clone
         * @returns a list of rootNodes, skeletons and animation groups that were duplicated
         */
        instantiateModelsToScene(nameFunction?: (sourceName: string) => string, cloneMaterials?: boolean, options?: {
            doNotInstantiate?: boolean | ((node: Node) => boolean);
            predicate?: (entity: any) => boolean;
        }): InstantiatedEntries;
        /**
         * Adds all the assets from the container to the scene.
         */
        addAllToScene(): void;
        /**
         * Adds assets from the container to the scene.
         * @param predicate defines a predicate used to select which entity will be added (can be null)
         */
        addToScene(predicate?: Nullable<(entity: any) => boolean>): void;
        /**
         * Removes all the assets in the container from the scene
         */
        removeAllFromScene(): void;
        /**
         * Removes assets in the container from the scene
         * @param predicate defines a predicate used to select which entity will be added (can be null)
         */
        removeFromScene(predicate?: Nullable<(entity: any) => boolean>): void;
        /**
         * Disposes all the assets in the container
         */
        dispose(): void;
        private _moveAssets;
        /**
         * Removes all the assets contained in the scene and adds them to the container.
         * @param keepAssets Set of assets to keep in the scene. (default: empty)
         */
        moveAllFromScene(keepAssets?: KeepAssets): void;
        /**
         * Adds all meshes in the asset container to a root mesh that can be used to position all the contained meshes. The root mesh is then added to the front of the meshes in the assetContainer.
         * @returns the root mesh
         */
        createRootMesh(): Mesh;
        /**
         * Merge animations (direct and animation groups) from this asset container into a scene
         * @param scene is the instance of BABYLON.Scene to append to (default: last created scene)
         * @param animatables set of animatables to retarget to a node from the scene
         * @param targetConverter defines a function used to convert animation targets from the asset container to the scene (default: search node by name)
         * @returns an array of the new AnimationGroup added to the scene (empty array if none)
         */
        mergeAnimationsTo(scene: Nullable<Scene> | undefined, animatables: Animatable[], targetConverter?: Nullable<(target: any) => Nullable<Node>>): AnimationGroup[];
        /**
         * @since 6.15.0
         * This method checks for any node that has no parent
         * and is not in the rootNodes array, and adds the node
         * there, if so.
         */
        populateRootNodes(): void;
        /**
         * @since 6.26.0
         * Given a root asset, this method will traverse its hierarchy and add it, its children and any materials/skeletons/animation groups to the container.
         * @param root root node
         */
        addAllAssetsToContainer(root: Node): void;
    }


    /**
     * Class used to work with sound analyzer using fast fourier transform (FFT)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
     */
    export class Analyser {
        /**
         * Gets or sets the smoothing
         * @ignorenaming
         */
        SMOOTHING: number;
        /**
         * Gets or sets the FFT table size
         * @ignorenaming
         */
        FFT_SIZE: number;
        /**
         * Gets or sets the bar graph amplitude
         * @ignorenaming
         */
        BARGRAPHAMPLITUDE: number;
        /**
         * Gets or sets the position of the debug canvas
         * @ignorenaming
         */
        DEBUGCANVASPOS: {
            x: number;
            y: number;
        };
        /**
         * Gets or sets the debug canvas size
         * @ignorenaming
         */
        DEBUGCANVASSIZE: {
            width: number;
            height: number;
        };
        private _byteFreqs;
        private _byteTime;
        private _floatFreqs;
        private _webAudioAnalyser;
        private _debugCanvas;
        private _debugCanvasContext;
        private _scene;
        private _registerFunc;
        private _audioEngine;
        /**
         * Creates a new analyser
         * @param scene defines hosting scene
         */
        constructor(scene?: Nullable<Scene>);
        /**
         * Get the number of data values you will have to play with for the visualization
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/frequencyBinCount
         * @returns a number
         */
        getFrequencyBinCount(): number;
        /**
         * Gets the current frequency data as a byte array
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/getByteFrequencyData
         * @returns a Uint8Array
         */
        getByteFrequencyData(): Uint8Array;
        /**
         * Gets the current waveform as a byte array
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/getByteTimeDomainData
         * @returns a Uint8Array
         */
        getByteTimeDomainData(): Uint8Array;
        /**
         * Gets the current frequency data as a float array
         * @see https://developer.mozilla.org/en-US/docs/Web/API/AnalyserNode/getByteFrequencyData
         * @returns a Float32Array
         */
        getFloatFrequencyData(): Float32Array;
        /**
         * Renders the debug canvas
         */
        drawDebugCanvas(): void;
        /**
         * Stops rendering the debug canvas and removes it
         */
        stopDebugCanvas(): void;
        /**
         * Connects two audio nodes
         * @param inputAudioNode defines first node to connect
         * @param outputAudioNode defines second node to connect
         */
        connectAudioNodes(inputAudioNode: AudioNode, outputAudioNode: AudioNode): void;
        /**
         * Releases all associated resources
         */
        dispose(): void;
    }


    /**
     * This represents the default audio engine used in babylon.
     * It is responsible to play, synchronize and analyse sounds throughout the  application.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
     */
    export class AudioEngine implements IAudioEngine {
        private _audioContext;
        private _audioContextInitialized;
        private _muteButton;
        private _hostElement;
        private _audioDestination;
        /**
         * Gets whether the current host supports Web Audio and thus could create AudioContexts.
         */
        canUseWebAudio: boolean;
        /**
         * The master gain node defines the global audio volume of your audio engine.
         */
        masterGain: GainNode;
        /**
         * Defines if Babylon should emit a warning if WebAudio is not supported.
         * @ignoreNaming
         */
        WarnedWebAudioUnsupported: boolean;
        /**
         * Gets whether or not mp3 are supported by your browser.
         */
        isMP3supported: boolean;
        /**
         * Gets whether or not ogg are supported by your browser.
         */
        isOGGsupported: boolean;
        /**
         * Gets whether audio has been unlocked on the device.
         * Some Browsers have strong restrictions about Audio and won't autoplay unless
         * a user interaction has happened.
         */
        unlocked: boolean;
        /**
         * Defines if the audio engine relies on a custom unlocked button.
         * In this case, the embedded button will not be displayed.
         */
        useCustomUnlockedButton: boolean;
        /**
         * Event raised when audio has been unlocked on the browser.
         */
        onAudioUnlockedObservable: Observable<IAudioEngine>;
        /**
         * Event raised when audio has been locked on the browser.
         */
        onAudioLockedObservable: Observable<IAudioEngine>;
        /**
         * Gets the current AudioContext if available.
         */
        get audioContext(): Nullable<AudioContext>;
        private _connectedAnalyser;
        /**
         * Instantiates a new audio engine.
         *
         * There should be only one per page as some browsers restrict the number
         * of audio contexts you can create.
         * @param hostElement defines the host element where to display the mute icon if necessary
         * @param audioContext defines the audio context to be used by the audio engine
         * @param audioDestination defines the audio destination node to be used by audio engine
         */
        constructor(hostElement?: Nullable<HTMLElement>, audioContext?: Nullable<AudioContext>, audioDestination?: Nullable<AudioDestinationNode | MediaStreamAudioDestinationNode>);
        /**
         * Flags the audio engine in Locked state.
         * This happens due to new browser policies preventing audio to autoplay.
         */
        lock(): void;
        /**
         * Unlocks the audio engine once a user action has been done on the dom.
         * This is helpful to resume play once browser policies have been satisfied.
         */
        unlock(): void;
        private _resumeAudioContext;
        private _initializeAudioContext;
        private _tryToRun;
        private _triggerRunningState;
        private _triggerSuspendedState;
        private _displayMuteButton;
        private _moveButtonToTopLeft;
        private _onResize;
        private _hideMuteButton;
        /**
         * Destroy and release the resources associated with the audio context.
         */
        dispose(): void;
        /**
         * Gets the global volume sets on the master gain.
         * @returns the global volume if set or -1 otherwise
         */
        getGlobalVolume(): number;
        /**
         * Sets the global volume of your experience (sets on the master gain).
         * @param newVolume Defines the new global volume of the application
         */
        setGlobalVolume(newVolume: number): void;
        /**
         * Connect the audio engine to an audio analyser allowing some amazing
         * synchronization between the sounds/music and your visualization (VuMeter for instance).
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#using-the-analyser
         * @param analyser The analyser to connect to the engine
         */
        connectToAnalyser(analyser: Analyser): void;
    }


        interface AbstractScene {
            /**
             * The list of sounds used in the scene.
             */
            sounds: Nullable<Array<Sound>>;
        }
        interface Scene {
            /**
             * @internal
             * Backing field
             */
            _mainSoundTrack: SoundTrack;
            /**
             * The main sound track played by the scene.
             * It contains your primary collection of sounds.
             */
            mainSoundTrack: SoundTrack;
            /**
             * The list of sound tracks added to the scene
             * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
             */
            soundTracks: Nullable<Array<SoundTrack>>;
            /**
             * Gets a sound using a given name
             * @param name defines the name to search for
             * @returns the found sound or null if not found at all.
             */
            getSoundByName(name: string): Nullable<Sound>;
            /**
             * Gets or sets if audio support is enabled
             * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
             */
            audioEnabled: boolean;
            /**
             * Gets or sets if audio will be output to headphones
             * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
             */
            headphone: boolean;
            /**
             * Gets or sets custom audio listener position provider
             * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
             */
            audioListenerPositionProvider: Nullable<() => Vector3>;
            /**
             * Gets or sets custom audio listener rotation provider
             * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
             */
            audioListenerRotationProvider: Nullable<() => Vector3>;
            /**
             * Gets or sets a refresh rate when using 3D audio positioning
             */
            audioPositioningRefreshRate: number;
        }
    /**
     * Defines the sound scene component responsible to manage any sounds
     * in a given scene.
     */
    export class AudioSceneComponent implements ISceneSerializableComponent {
        private static _CameraDirection;
        /**
         * The component name helpful to identify the component in the list of scene components.
         */
        readonly name = "Audio";
        /**
         * The scene the component belongs to.
         */
        scene: Scene;
        private _audioEnabled;
        /**
         * Gets whether audio is enabled or not.
         * Please use related enable/disable method to switch state.
         */
        get audioEnabled(): boolean;
        private _headphone;
        /**
         * Gets whether audio is outputting to headphone or not.
         * Please use the according Switch methods to change output.
         */
        get headphone(): boolean;
        /**
         * Gets or sets a refresh rate when using 3D audio positioning
         */
        audioPositioningRefreshRate: number;
        /**
         * Gets or Sets a custom listener position for all sounds in the scene
         * By default, this is the position of the first active camera
         */
        audioListenerPositionProvider: Nullable<() => Vector3>;
        /**
         * Gets or Sets a custom listener rotation for all sounds in the scene
         * By default, this is the rotation of the first active camera
         */
        audioListenerRotationProvider: Nullable<() => Vector3>;
        /**
         * Creates a new instance of the component for the given scene
         * @param scene Defines the scene to register the component in
         */
        constructor(scene?: Nullable<Scene>);
        /**
         * Registers the component in a given scene
         */
        register(): void;
        /**
         * Rebuilds the elements related to this component in case of
         * context lost for instance.
         */
        rebuild(): void;
        /**
         * Serializes the component data to the specified json object
         * @param serializationObject The object to serialize to
         */
        serialize(serializationObject: any): void;
        /**
         * Adds all the elements from the container to the scene
         * @param container the container holding the elements
         */
        addFromContainer(container: AbstractScene): void;
        /**
         * Removes all the elements in the container from the scene
         * @param container contains the elements to remove
         * @param dispose if the removed element should be disposed (default: false)
         */
        removeFromContainer(container: AbstractScene, dispose?: boolean): void;
        /**
         * Disposes the component and the associated resources.
         */
        dispose(): void;
        /**
         * Disables audio in the associated scene.
         */
        disableAudio(): void;
        /**
         * Enables audio in the associated scene.
         */
        enableAudio(): void;
        /**
         * Switch audio to headphone output.
         */
        switchAudioModeForHeadphones(): void;
        /**
         * Switch audio to normal speakers.
         */
        switchAudioModeForNormalSpeakers(): void;
        private _cachedCameraDirection;
        private _cachedCameraPosition;
        private _lastCheck;
        private _invertMatrixTemp;
        private _cameraDirectionTemp;
        private _afterRender;
    }




    /**
     * This represents an audio engine and it is responsible
     * to play, synchronize and analyse sounds throughout the application.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
     */
    export interface IAudioEngine extends IDisposable {
        /**
         * Gets whether the current host supports Web Audio and thus could create AudioContexts.
         */
        readonly canUseWebAudio: boolean;
        /**
         * Gets the current AudioContext if available.
         */
        readonly audioContext: Nullable<AudioContext>;
        /**
         * The master gain node defines the global audio volume of your audio engine.
         */
        readonly masterGain: GainNode;
        /**
         * Gets whether or not mp3 are supported by your browser.
         */
        readonly isMP3supported: boolean;
        /**
         * Gets whether or not ogg are supported by your browser.
         */
        readonly isOGGsupported: boolean;
        /**
         * Defines if Babylon should emit a warning if WebAudio is not supported.
         * @ignoreNaming
         */
        WarnedWebAudioUnsupported: boolean;
        /**
         * Defines if the audio engine relies on a custom unlocked button.
         * In this case, the embedded button will not be displayed.
         */
        useCustomUnlockedButton: boolean;
        /**
         * Gets whether or not the audio engine is unlocked (require first a user gesture on some browser).
         */
        readonly unlocked: boolean;
        /**
         * Event raised when audio has been unlocked on the browser.
         */
        onAudioUnlockedObservable: Observable<IAudioEngine>;
        /**
         * Event raised when audio has been locked on the browser.
         */
        onAudioLockedObservable: Observable<IAudioEngine>;
        /**
         * Flags the audio engine in Locked state.
         * This happens due to new browser policies preventing audio to autoplay.
         */
        lock(): void;
        /**
         * Unlocks the audio engine once a user action has been done on the dom.
         * This is helpful to resume play once browser policies have been satisfied.
         */
        unlock(): void;
        /**
         * Gets the global volume sets on the master gain.
         * @returns the global volume if set or -1 otherwise
         */
        getGlobalVolume(): number;
        /**
         * Sets the global volume of your experience (sets on the master gain).
         * @param newVolume Defines the new global volume of the application
         */
        setGlobalVolume(newVolume: number): void;
        /**
         * Connect the audio engine to an audio analyser allowing some amazing
         * synchronization between the sounds/music and your visualization (VuMeter for instance).
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#using-the-analyser
         * @param analyser The analyser to connect to the engine
         */
        connectToAnalyser(analyser: Analyser): void;
    }


    /**
     * Interface used to define options for the Audio Engine
     * @since 5.0.0
     */
    export interface IAudioEngineOptions {
        /**
         * Specifies an existing Audio Context for the audio engine
         */
        audioContext?: AudioContext;
        /**
         * Specifies a destination node for the audio engine
         */
        audioDestination?: AudioDestinationNode | MediaStreamAudioDestinationNode;
    }


    /**
     * Interface used to define options for Sound class
     */
    export interface ISoundOptions {
        /**
         * Does the sound autoplay once loaded.
         */
        autoplay?: boolean;
        /**
         * Does the sound loop after it finishes playing once.
         */
        loop?: boolean;
        /**
         * Sound's volume
         */
        volume?: number;
        /**
         * Is it a spatial sound?
         */
        spatialSound?: boolean;
        /**
         * Maximum distance to hear that sound
         */
        maxDistance?: number;
        /**
         * Uses user defined attenuation function
         */
        useCustomAttenuation?: boolean;
        /**
         * Define the roll off factor of spatial sounds.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        rolloffFactor?: number;
        /**
         * Define the reference distance the sound should be heard perfectly.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        refDistance?: number;
        /**
         * Define the distance attenuation model the sound will follow.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        distanceModel?: string;
        /**
         * Defines the playback speed (1 by default)
         */
        playbackRate?: number;
        /**
         * Defines if the sound is from a streaming source
         */
        streaming?: boolean;
        /**
         * Defines an optional length (in seconds) inside the sound file
         */
        length?: number;
        /**
         * Defines an optional offset (in seconds) inside the sound file
         */
        offset?: number;
        /**
         * If true, URLs will not be required to state the audio file codec to use.
         */
        skipCodecCheck?: boolean;
    }


    /**
     * Defines a sound that can be played in the application.
     * The sound can either be an ambient track or a simple sound played in reaction to a user action.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
     */
    export class Sound {
        /**
         * The name of the sound in the scene.
         */
        name: string;
        /**
         * Does the sound autoplay once loaded.
         */
        autoplay: boolean;
        private _loop;
        /**
         * Does the sound loop after it finishes playing once.
         */
        get loop(): boolean;
        set loop(value: boolean);
        /**
         * Does the sound use a custom attenuation curve to simulate the falloff
         * happening when the source gets further away from the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-your-own-custom-attenuation-function
         */
        useCustomAttenuation: boolean;
        /**
         * The sound track id this sound belongs to.
         */
        soundTrackId: number;
        /**
         * Is this sound currently played.
         */
        isPlaying: boolean;
        /**
         * Is this sound currently paused.
         */
        isPaused: boolean;
        /**
         * Define the reference distance the sound should be heard perfectly.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        refDistance: number;
        /**
         * Define the roll off factor of spatial sounds.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        rolloffFactor: number;
        /**
         * Define the max distance the sound should be heard (intensity just became 0 at this point).
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        maxDistance: number;
        /**
         * Define the distance attenuation model the sound will follow.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        distanceModel: string;
        /**
         * @internal
         * Back Compat
         **/
        onended: () => any;
        /**
         * Gets or sets an object used to store user defined information for the sound.
         */
        metadata: any;
        /**
         * Observable event when the current playing sound finishes.
         */
        onEndedObservable: Observable<Sound>;
        /**
         * Gets the current time for the sound.
         */
        get currentTime(): number;
        /**
         * Does this sound enables spatial sound.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        get spatialSound(): boolean;
        /**
         * Does this sound enables spatial sound.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        set spatialSound(newValue: boolean);
        private _spatialSound;
        private _panningModel;
        private _playbackRate;
        private _streaming;
        private _startTime;
        private _currentTime;
        private _position;
        private _localDirection;
        private _volume;
        private _isReadyToPlay;
        private _isDirectional;
        private _readyToPlayCallback;
        private _audioBuffer;
        private _soundSource;
        private _streamingSource;
        private _soundPanner;
        private _soundGain;
        private _inputAudioNode;
        private _outputAudioNode;
        private _coneInnerAngle;
        private _coneOuterAngle;
        private _coneOuterGain;
        private _scene;
        private _connectedTransformNode;
        private _customAttenuationFunction;
        private _registerFunc;
        private _isOutputConnected;
        private _htmlAudioElement;
        private _urlType;
        private _length?;
        private _offset?;
        private _tryToPlayTimeout;
        private _audioUnlockedObserver?;
        private _url?;
        /**
         * @internal
         */
        static _SceneComponentInitialization: (scene: Scene) => void;
        /**
         * Create a sound and attach it to a scene
         * @param name Name of your sound
         * @param urlOrArrayBuffer Url to the sound to load async or ArrayBuffer, it also works with MediaStreams and AudioBuffers
         * @param scene defines the scene the sound belongs to
         * @param readyToPlayCallback Provide a callback function if you'd like to load your code once the sound is ready to be played
         * @param options Objects to provide with the current available options: autoplay, loop, volume, spatialSound, maxDistance, rolloffFactor, refDistance, distanceModel, panningModel, streaming
         */
        constructor(name: string, urlOrArrayBuffer: any, scene?: Nullable<Scene>, readyToPlayCallback?: Nullable<() => void>, options?: ISoundOptions);
        /**
         * Release the sound and its associated resources
         */
        dispose(): void;
        /**
         * Gets if the sounds is ready to be played or not.
         * @returns true if ready, otherwise false
         */
        isReady(): boolean;
        /**
         * Get the current class name.
         * @returns current class name
         */
        getClassName(): string;
        private _audioBufferLoaded;
        private _soundLoaded;
        /**
         * Sets the data of the sound from an audiobuffer
         * @param audioBuffer The audioBuffer containing the data
         */
        setAudioBuffer(audioBuffer: AudioBuffer): void;
        /**
         * Updates the current sounds options such as maxdistance, loop...
         * @param options A JSON object containing values named as the object properties
         */
        updateOptions(options: ISoundOptions): void;
        private _createSpatialParameters;
        private _disableSpatialSound;
        private _updateSpatialParameters;
        /**
         * Switch the panning model to HRTF:
         * Renders a stereo output of higher quality than equalpower — it uses a convolution with measured impulse responses from human subjects.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        switchPanningModelToHRTF(): void;
        /**
         * Switch the panning model to Equal Power:
         * Represents the equal-power panning algorithm, generally regarded as simple and efficient. equalpower is the default value.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        switchPanningModelToEqualPower(): void;
        private _switchPanningModel;
        /**
         * Connect this sound to a sound track audio node like gain...
         * @param soundTrackAudioNode the sound track audio node to connect to
         */
        connectToSoundTrackAudioNode(soundTrackAudioNode: AudioNode): void;
        /**
         * Transform this sound into a directional source
         * @param coneInnerAngle Size of the inner cone in degree
         * @param coneOuterAngle Size of the outer cone in degree
         * @param coneOuterGain Volume of the sound outside the outer cone (between 0.0 and 1.0)
         */
        setDirectionalCone(coneInnerAngle: number, coneOuterAngle: number, coneOuterGain: number): void;
        /**
         * Gets or sets the inner angle for the directional cone.
         */
        get directionalConeInnerAngle(): number;
        /**
         * Gets or sets the inner angle for the directional cone.
         */
        set directionalConeInnerAngle(value: number);
        /**
         * Gets or sets the outer angle for the directional cone.
         */
        get directionalConeOuterAngle(): number;
        /**
         * Gets or sets the outer angle for the directional cone.
         */
        set directionalConeOuterAngle(value: number);
        /**
         * Sets the position of the emitter if spatial sound is enabled
         * @param newPosition Defines the new position
         */
        setPosition(newPosition: Vector3): void;
        /**
         * Sets the local direction of the emitter if spatial sound is enabled
         * @param newLocalDirection Defines the new local direction
         */
        setLocalDirectionToMesh(newLocalDirection: Vector3): void;
        private _updateDirection;
        /** @internal */
        updateDistanceFromListener(): void;
        /**
         * Sets a new custom attenuation function for the sound.
         * @param callback Defines the function used for the attenuation
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-your-own-custom-attenuation-function
         */
        setAttenuationFunction(callback: (currentVolume: number, currentDistance: number, maxDistance: number, refDistance: number, rolloffFactor: number) => number): void;
        /**
         * Play the sound
         * @param time (optional) Start the sound after X seconds. Start immediately (0) by default.
         * @param offset (optional) Start the sound at a specific time in seconds
         * @param length (optional) Sound duration (in seconds)
         */
        play(time?: number, offset?: number, length?: number): void;
        private _onended;
        /**
         * Stop the sound
         * @param time (optional) Stop the sound after X seconds. Stop immediately (0) by default.
         */
        stop(time?: number): void;
        /**
         * Put the sound in pause
         */
        pause(): void;
        /**
         * Sets a dedicated volume for this sounds
         * @param newVolume Define the new volume of the sound
         * @param time Define time for gradual change to new volume
         */
        setVolume(newVolume: number, time?: number): void;
        /**
         * Set the sound play back rate
         * @param newPlaybackRate Define the playback rate the sound should be played at
         */
        setPlaybackRate(newPlaybackRate: number): void;
        /**
         * Gets the sound play back rate.
         * @returns the  play back rate of the sound
         */
        getPlaybackRate(): number;
        /**
         * Gets the volume of the sound.
         * @returns the volume of the sound
         */
        getVolume(): number;
        /**
         * Attach the sound to a dedicated mesh
         * @param transformNode The transform node to connect the sound with
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#attaching-a-sound-to-a-mesh
         */
        attachToMesh(transformNode: TransformNode): void;
        /**
         * Detach the sound from the previously attached mesh
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#attaching-a-sound-to-a-mesh
         */
        detachFromMesh(): void;
        private _onRegisterAfterWorldMatrixUpdate;
        /**
         * Clone the current sound in the scene.
         * @returns the new sound clone
         */
        clone(): Nullable<Sound>;
        /**
         * Gets the current underlying audio buffer containing the data
         * @returns the audio buffer
         */
        getAudioBuffer(): Nullable<AudioBuffer>;
        /**
         * Gets the WebAudio AudioBufferSourceNode, lets you keep track of and stop instances of this Sound.
         * @returns the source node
         */
        getSoundSource(): Nullable<AudioBufferSourceNode>;
        /**
         * Gets the WebAudio GainNode, gives you precise control over the gain of instances of this Sound.
         * @returns the gain node
         */
        getSoundGain(): Nullable<GainNode>;
        /**
         * Serializes the Sound in a JSON representation
         * @returns the JSON representation of the sound
         */
        serialize(): any;
        /**
         * Parse a JSON representation of a sound to instantiate in a given scene
         * @param parsedSound Define the JSON representation of the sound (usually coming from the serialize method)
         * @param scene Define the scene the new parsed sound should be created in
         * @param rootUrl Define the rooturl of the load in case we need to fetch relative dependencies
         * @param sourceSound Define a sound place holder if do not need to instantiate a new one
         * @returns the newly parsed sound
         */
        static Parse(parsedSound: any, scene: Scene, rootUrl: string, sourceSound?: Sound): Sound;
        private _setOffset;
        private _clearTimeoutsAndObservers;
    }


    /**
     * Options allowed during the creation of a sound track.
     */
    export interface ISoundTrackOptions {
        /**
         * The volume the sound track should take during creation
         */
        volume?: number;
        /**
         * Define if the sound track is the main sound track of the scene
         */
        mainTrack?: boolean;
    }
    /**
     * It could be useful to isolate your music & sounds on several tracks to better manage volume on a grouped instance of sounds.
     * It will be also used in a future release to apply effects on a specific track.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#using-sound-tracks
     */
    export class SoundTrack {
        /**
         * The unique identifier of the sound track in the scene.
         */
        id: number;
        /**
         * The list of sounds included in the sound track.
         */
        soundCollection: Array<Sound>;
        private _outputAudioNode;
        private _scene;
        private _connectedAnalyser;
        private _options;
        private _isInitialized;
        /**
         * Creates a new sound track.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#using-sound-tracks
         * @param scene Define the scene the sound track belongs to
         * @param options
         */
        constructor(scene?: Nullable<Scene>, options?: ISoundTrackOptions);
        private _initializeSoundTrackAudioGraph;
        /**
         * Release the sound track and its associated resources
         */
        dispose(): void;
        /**
         * Adds a sound to this sound track
         * @param sound define the sound to add
         * @ignoreNaming
         */
        addSound(sound: Sound): void;
        /**
         * Removes a sound to this sound track
         * @param sound define the sound to remove
         * @ignoreNaming
         */
        removeSound(sound: Sound): void;
        /**
         * Set a global volume for the full sound track.
         * @param newVolume Define the new volume of the sound track
         */
        setVolume(newVolume: number): void;
        /**
         * Switch the panning model to HRTF:
         * Renders a stereo output of higher quality than equalpower — it uses a convolution with measured impulse responses from human subjects.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        switchPanningModelToHRTF(): void;
        /**
         * Switch the panning model to Equal Power:
         * Represents the equal-power panning algorithm, generally regarded as simple and efficient. equalpower is the default value.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#creating-a-spatial-3d-sound
         */
        switchPanningModelToEqualPower(): void;
        /**
         * Connect the sound track to an audio analyser allowing some amazing
         * synchronization between the sounds/music and your visualization (VuMeter for instance).
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic#using-the-analyser
         * @param analyser The analyser to connect to the engine
         */
        connectToAnalyser(analyser: Analyser): void;
    }


    /**
     * Wraps one or more Sound objects and selects one with random weight for playback.
     */
    export class WeightedSound {
        /** When true a Sound will be selected and played when the current playing Sound completes. */
        loop: boolean;
        private _coneInnerAngle;
        private _coneOuterAngle;
        private _volume;
        /** A Sound is currently playing. */
        isPlaying: boolean;
        /** A Sound is currently paused. */
        isPaused: boolean;
        private _sounds;
        private _weights;
        private _currentIndex?;
        /**
         * Creates a new WeightedSound from the list of sounds given.
         * @param loop When true a Sound will be selected and played when the current playing Sound completes.
         * @param sounds Array of Sounds that will be selected from.
         * @param weights Array of number values for selection weights; length must equal sounds, values will be normalized to 1
         */
        constructor(loop: boolean, sounds: Sound[], weights: number[]);
        /**
         * The size of cone in degrees for a directional sound in which there will be no attenuation.
         */
        get directionalConeInnerAngle(): number;
        /**
         * The size of cone in degrees for a directional sound in which there will be no attenuation.
         */
        set directionalConeInnerAngle(value: number);
        /**
         * Size of cone in degrees for a directional sound outside of which there will be no sound.
         * Listener angles between innerAngle and outerAngle will falloff linearly.
         */
        get directionalConeOuterAngle(): number;
        /**
         * Size of cone in degrees for a directional sound outside of which there will be no sound.
         * Listener angles between innerAngle and outerAngle will falloff linearly.
         */
        set directionalConeOuterAngle(value: number);
        /**
         * Playback volume.
         */
        get volume(): number;
        /**
         * Playback volume.
         */
        set volume(value: number);
        private _onended;
        /**
         * Suspend playback
         */
        pause(): void;
        /**
         * Stop playback
         */
        stop(): void;
        /**
         * Start playback.
         * @param startOffset Position the clip head at a specific time in seconds.
         */
        play(startOffset?: number): void;
    }


    /**
     * Interface for baked vertex animation texture, see BakedVertexAnimationManager
     * @since 5.0
     */
    export interface IBakedVertexAnimationManager {
        /**
         * The vertex animation texture
         */
        texture: Nullable<BaseTexture>;
        /**
         * Gets or sets a boolean indicating if the edgesRenderer is active
         */
        isEnabled: boolean;
        /**
         * The animation parameters for the mesh. See setAnimationParameters()
         */
        animationParameters: Vector4;
        /**
         * The time counter, to pick the correct animation frame.
         */
        time: number;
        /**
         * Binds to the effect.
         * @param effect The effect to bind to.
         * @param useInstances True when it's an instance.
         */
        bind(effect: Effect, useInstances: boolean): void;
        /**
         * Sets animation parameters.
         * @param startFrame The first frame of the animation.
         * @param endFrame The last frame of the animation.
         * @param offset The offset when starting the animation.
         * @param speedFramesPerSecond The frame rate.
         */
        setAnimationParameters(startFrame: number, endFrame: number, offset: number, speedFramesPerSecond: number): void;
        /**
         * Disposes the resources of the manager.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
         * Get the current class name useful for serialization or dynamic coding.
         * @returns "BakedVertexAnimationManager"
         */
        getClassName(): string;
    }
    /**
     * This class is used to animate meshes using a baked vertex animation texture
     * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/baked_texture_animations
     * @since 5.0
     */
    export class BakedVertexAnimationManager implements IBakedVertexAnimationManager {
        private _scene;
        private _texture;
        /**
         * The vertex animation texture
         */
        texture: Nullable<BaseTexture>;
        private _isEnabled;
        /**
         * Enable or disable the vertex animation manager
         */
        isEnabled: boolean;
        /**
         * The animation parameters for the mesh. See setAnimationParameters()
         */
        animationParameters: Vector4;
        /**
         * The time counter, to pick the correct animation frame.
         */
        time: number;
        /**
         * Creates a new BakedVertexAnimationManager
         * @param scene defines the current scene
         */
        constructor(scene?: Nullable<Scene>);
        /** @internal */
        _markSubMeshesAsAttributesDirty(): void;
        /**
         * Binds to the effect.
         * @param effect The effect to bind to.
         * @param useInstances True when it's an instance.
         */
        bind(effect: Effect, useInstances?: boolean): void;
        /**
         * Clone the current manager
         * @returns a new BakedVertexAnimationManager
         */
        clone(): BakedVertexAnimationManager;
        /**
         * Sets animation parameters.
         * @param startFrame The first frame of the animation.
         * @param endFrame The last frame of the animation.
         * @param offset The offset when starting the animation.
         * @param speedFramesPerSecond The frame rate.
         */
        setAnimationParameters(startFrame: number, endFrame: number, offset?: number, speedFramesPerSecond?: number): void;
        /**
         * Disposes the resources of the manager.
         * @param forceDisposeTextures - Forces the disposal of all textures.
         */
        dispose(forceDisposeTextures?: boolean): void;
        /**
         * Get the current class name useful for serialization or dynamic coding.
         * @returns "BakedVertexAnimationManager"
         */
        getClassName(): string;
        /**
         * Makes a duplicate of the current instance into another one.
         * @param vatMap define the instance where to copy the info
         */
        copyTo(vatMap: BakedVertexAnimationManager): void;
        /**
         * Serializes this vertex animation instance
         * @returns - An object with the serialized instance.
         */
        serialize(): any;
        /**
         * Parses a vertex animation setting from a serialized object.
         * @param source - Serialized object.
         * @param scene Defines the scene we are parsing for
         * @param rootUrl Defines the rootUrl to load from
         */
        parse(source: any, scene: Scene, rootUrl: string): void;
    }




    /**
     * Class to bake vertex animation textures.
     * @since 5.0
     */
    export class VertexAnimationBaker {
        private _scene;
        private _mesh;
        /**
         * Create a new VertexAnimationBaker object which can help baking animations into a texture.
         * @param scene Defines the scene the VAT belongs to
         * @param mesh Defines the mesh the VAT belongs to
         */
        constructor(scene: Scene, mesh: Mesh);
        /**
         * Bakes the animation into the texture. This should be called once, when the
         * scene starts, so the VAT is generated and associated to the mesh.
         * @param ranges Defines the ranges in the animation that will be baked.
         * @returns The array of matrix transforms for each vertex (columns) and frame (rows), as a Float32Array.
         */
        bakeVertexData(ranges: AnimationRange[]): Promise<Float32Array>;
        /**
         * Runs an animation frame and stores its vertex data
         *
         * @param vertexData The array to save data to.
         * @param frameIndex Current frame in the skeleton animation to render.
         * @param textureIndex Current index of the texture data.
         */
        private _executeAnimationFrame;
        /**
         * Builds a vertex animation texture given the vertexData in an array.
         * @param vertexData The vertex animation data. You can generate it with bakeVertexData().
         * @returns The vertex animation texture to be used with BakedVertexAnimationManager.
         */
        textureFromBakedVertexData(vertexData: Float32Array): RawTexture;
        /**
         * Serializes our vertexData to an object, with a nice string for the vertexData.
         * @param vertexData The vertex array data.
         * @returns This object serialized to a JS dict.
         */
        serializeBakedVertexDataToObject(vertexData: Float32Array): Record<string, any>;
        /**
         * Loads previously baked data.
         * @param data The object as serialized by serializeBakedVertexDataToObject()
         * @returns The array of matrix transforms for each vertex (columns) and frame (rows), as a Float32Array.
         */
        loadBakedVertexDataFromObject(data: Record<string, any>): Float32Array;
        /**
         * Serializes our vertexData to a JSON string, with a nice string for the vertexData.
         * Should be called right after bakeVertexData().
         * @param vertexData The vertex array data.
         * @returns This object serialized to a safe string.
         */
        serializeBakedVertexDataToJSON(vertexData: Float32Array): string;
        /**
         * Loads previously baked data in string format.
         * @param json The json string as serialized by serializeBakedVertexDataToJSON().
         * @returns The array of matrix transforms for each vertex (columns) and frame (rows), as a Float32Array.
         */
        loadBakedVertexDataFromJSON(json: string): Float32Array;
    }


    /**
     * Interface used to define a behavior
     */
    export interface Behavior<T> {
        /** gets or sets behavior's name */
        name: string;
        /**
         * Function called when the behavior needs to be initialized (before attaching it to a target)
         */
        init(): void;
        /**
         * Called when the behavior is attached to a target
         * @param target defines the target where the behavior is attached to
         */
        attach(target: T): void;
        /**
         * Called when the behavior is detached from its target
         */
        detach(): void;
    }
    /**
     * Interface implemented by classes supporting behaviors
     */
    export interface IBehaviorAware<T> {
        /**
         * Attach a behavior
         * @param behavior defines the behavior to attach
         * @returns the current host
         */
        addBehavior(behavior: Behavior<T>): T;
        /**
         * Remove a behavior from the current object
         * @param behavior defines the behavior to detach
         * @returns the current host
         */
        removeBehavior(behavior: Behavior<T>): T;
        /**
         * Gets a behavior using its name to search
         * @param name defines the name to search
         * @returns the behavior or null if not found
         */
        getBehaviorByName(name: string): Nullable<Behavior<T>>;
    }


    /**
     * The autoRotation behavior (AutoRotationBehavior) is designed to create a smooth rotation of an ArcRotateCamera when there is no user interaction.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#autorotation-behavior
     */
    export class AutoRotationBehavior implements Behavior<ArcRotateCamera> {
        /**
         * Gets the name of the behavior.
         */
        get name(): string;
        private _zoomStopsAnimation;
        private _idleRotationSpeed;
        private _idleRotationWaitTime;
        private _idleRotationSpinupTime;
        /**
         * Target alpha
         */
        targetAlpha: Nullable<number>;
        /**
         * Sets the flag that indicates if user zooming should stop animation.
         */
        set zoomStopsAnimation(flag: boolean);
        /**
         * Gets the flag that indicates if user zooming should stop animation.
         */
        get zoomStopsAnimation(): boolean;
        /**
         * Sets the default speed at which the camera rotates around the model.
         */
        set idleRotationSpeed(speed: number);
        /**
         * Gets the default speed at which the camera rotates around the model.
         */
        get idleRotationSpeed(): number;
        /**
         * Sets the time (in milliseconds) to wait after user interaction before the camera starts rotating.
         */
        set idleRotationWaitTime(time: number);
        /**
         * Gets the time (milliseconds) to wait after user interaction before the camera starts rotating.
         */
        get idleRotationWaitTime(): number;
        /**
         * Sets the time (milliseconds) to take to spin up to the full idle rotation speed.
         */
        set idleRotationSpinupTime(time: number);
        /**
         * Gets the time (milliseconds) to take to spin up to the full idle rotation speed.
         */
        get idleRotationSpinupTime(): number;
        /**
         * Gets a value indicating if the camera is currently rotating because of this behavior
         */
        get rotationInProgress(): boolean;
        private _onPrePointerObservableObserver;
        private _onAfterCheckInputsObserver;
        private _attachedCamera;
        private _isPointerDown;
        private _lastFrameTime;
        private _lastInteractionTime;
        private _cameraRotationSpeed;
        /**
         * Initializes the behavior.
         */
        init(): void;
        /**
         * Attaches the behavior to its arc rotate camera.
         * @param camera Defines the camera to attach the behavior to
         */
        attach(camera: ArcRotateCamera): void;
        /**
         * Detaches the behavior from its current arc rotate camera.
         */
        detach(): void;
        /**
         * Force-reset the last interaction time
         * @param customTime an optional time that will be used instead of the current last interaction time. For example `Date.now()`
         */
        resetLastInteractionTime(customTime?: number): void;
        /**
         * Returns true if camera alpha reaches the target alpha
         * @returns true if camera alpha reaches the target alpha
         */
        private _reachTargetAlpha;
        /**
         * Returns true if user is scrolling.
         * @returns true if user is scrolling.
         */
        private _userIsZooming;
        private _lastFrameRadius;
        private _shouldAnimationStopForInteraction;
        /**
         *  Applies any current user interaction to the camera. Takes into account maximum alpha rotation.
         */
        private _applyUserInteraction;
        private _userIsMoving;
    }


    /**
     * Add a bouncing effect to an ArcRotateCamera when reaching a specified minimum and maximum radius
     * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#bouncing-behavior
     */
    export class BouncingBehavior implements Behavior<ArcRotateCamera> {
        /**
         * Gets the name of the behavior.
         */
        get name(): string;
        /**
         * The easing function used by animations
         */
        static EasingFunction: BackEase;
        /**
         * The easing mode used by animations
         */
        static EasingMode: number;
        /**
         * The duration of the animation, in milliseconds
         */
        transitionDuration: number;
        /**
         * Length of the distance animated by the transition when lower radius is reached
         */
        lowerRadiusTransitionRange: number;
        /**
         * Length of the distance animated by the transition when upper radius is reached
         */
        upperRadiusTransitionRange: number;
        private _autoTransitionRange;
        /**
         * Gets a value indicating if the lowerRadiusTransitionRange and upperRadiusTransitionRange are defined automatically
         */
        get autoTransitionRange(): boolean;
        /**
         * Sets a value indicating if the lowerRadiusTransitionRange and upperRadiusTransitionRange are defined automatically
         * Transition ranges will be set to 5% of the bounding box diagonal in world space
         */
        set autoTransitionRange(value: boolean);
        private _attachedCamera;
        private _onAfterCheckInputsObserver;
        private _onMeshTargetChangedObserver;
        /**
         * Initializes the behavior.
         */
        init(): void;
        /**
         * Attaches the behavior to its arc rotate camera.
         * @param camera Defines the camera to attach the behavior to
         */
        attach(camera: ArcRotateCamera): void;
        /**
         * Detaches the behavior from its current arc rotate camera.
         */
        detach(): void;
        private _radiusIsAnimating;
        private _radiusBounceTransition;
        private _animatables;
        private _cachedWheelPrecision;
        /**
         * Checks if the camera radius is at the specified limit. Takes into account animation locks.
         * @param radiusLimit The limit to check against.
         * @returns Bool to indicate if at limit.
         */
        private _isRadiusAtLimit;
        /**
         * Applies an animation to the radius of the camera, extending by the radiusDelta.
         * @param radiusDelta The delta by which to animate to. Can be negative.
         */
        private _applyBoundRadiusAnimation;
        /**
         * Removes all animation locks. Allows new animations to be added to any of the camera properties.
         */
        protected _clearAnimationLocks(): void;
        /**
         * Stops and removes all animations that have been applied to the camera
         */
        stopAllAnimations(): void;
    }


    /**
     * The framing behavior (FramingBehavior) is designed to automatically position an ArcRotateCamera when its target is set to a mesh. It is also useful if you want to prevent the camera to go under a virtual horizontal plane.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#framing-behavior
     */
    export class FramingBehavior implements Behavior<ArcRotateCamera> {
        /**
         * Gets the name of the behavior.
         */
        get name(): string;
        /**
         * An event triggered when the animation to zoom on target mesh has ended
         */
        onTargetFramingAnimationEndObservable: Observable<void>;
        private _mode;
        private _radiusScale;
        private _positionScale;
        private _defaultElevation;
        private _elevationReturnTime;
        private _elevationReturnWaitTime;
        private _zoomStopsAnimation;
        private _framingTime;
        /**
         * The easing function used by animations
         */
        static EasingFunction: ExponentialEase;
        /**
         * The easing mode used by animations
         */
        static EasingMode: number;
        /**
         * Sets the current mode used by the behavior
         */
        set mode(mode: number);
        /**
         * Gets current mode used by the behavior.
         */
        get mode(): number;
        /**
         * Sets the scale applied to the radius (1 by default)
         */
        set radiusScale(radius: number);
        /**
         * Gets the scale applied to the radius
         */
        get radiusScale(): number;
        /**
         * Sets the scale to apply on Y axis to position camera focus. 0.5 by default which means the center of the bounding box.
         */
        set positionScale(scale: number);
        /**
         * Gets the scale to apply on Y axis to position camera focus. 0.5 by default which means the center of the bounding box.
         */
        get positionScale(): number;
        /**
         * Sets the angle above/below the horizontal plane to return to when the return to default elevation idle
         * behaviour is triggered, in radians.
         */
        set defaultElevation(elevation: number);
        /**
         * Gets the angle above/below the horizontal plane to return to when the return to default elevation idle
         * behaviour is triggered, in radians.
         */
        get defaultElevation(): number;
        /**
         * Sets the time (in milliseconds) taken to return to the default beta position.
         * Negative value indicates camera should not return to default.
         */
        set elevationReturnTime(speed: number);
        /**
         * Gets the time (in milliseconds) taken to return to the default beta position.
         * Negative value indicates camera should not return to default.
         */
        get elevationReturnTime(): number;
        /**
         * Sets the delay (in milliseconds) taken before the camera returns to the default beta position.
         */
        set elevationReturnWaitTime(time: number);
        /**
         * Gets the delay (in milliseconds) taken before the camera returns to the default beta position.
         */
        get elevationReturnWaitTime(): number;
        /**
         * Sets the flag that indicates if user zooming should stop animation.
         */
        set zoomStopsAnimation(flag: boolean);
        /**
         * Gets the flag that indicates if user zooming should stop animation.
         */
        get zoomStopsAnimation(): boolean;
        /**
         * Sets the transition time when framing the mesh, in milliseconds
         */
        set framingTime(time: number);
        /**
         * Gets the transition time when framing the mesh, in milliseconds
         */
        get framingTime(): number;
        /**
         * Define if the behavior should automatically change the configured
         * camera limits and sensibilities.
         */
        autoCorrectCameraLimitsAndSensibility: boolean;
        private _onPrePointerObservableObserver;
        private _onAfterCheckInputsObserver;
        private _onMeshTargetChangedObserver;
        private _attachedCamera;
        private _isPointerDown;
        private _lastInteractionTime;
        /**
         * Initializes the behavior.
         */
        init(): void;
        /**
         * Attaches the behavior to its arc rotate camera.
         * @param camera Defines the camera to attach the behavior to
         */
        attach(camera: ArcRotateCamera): void;
        /**
         * Detaches the behavior from its current arc rotate camera.
         */
        detach(): void;
        private _animatables;
        private _betaIsAnimating;
        private _betaTransition;
        private _radiusTransition;
        private _vectorTransition;
        /**
         * Targets the given mesh and updates zoom level accordingly.
         * @param mesh  The mesh to target.
         * @param focusOnOriginXZ Determines if the camera should focus on 0 in the X and Z axis instead of the mesh
         * @param onAnimationEnd Callback triggered at the end of the framing animation
         */
        zoomOnMesh(mesh: AbstractMesh, focusOnOriginXZ?: boolean, onAnimationEnd?: Nullable<() => void>): void;
        /**
         * Targets the given mesh with its children and updates zoom level accordingly.
         * @param mesh  The mesh to target.
         * @param focusOnOriginXZ Determines if the camera should focus on 0 in the X and Z axis instead of the mesh
         * @param onAnimationEnd Callback triggered at the end of the framing animation
         */
        zoomOnMeshHierarchy(mesh: AbstractMesh, focusOnOriginXZ?: boolean, onAnimationEnd?: Nullable<() => void>): void;
        /**
         * Targets the given meshes with their children and updates zoom level accordingly.
         * @param meshes  The mesh to target.
         * @param focusOnOriginXZ Determines if the camera should focus on 0 in the X and Z axis instead of the mesh
         * @param onAnimationEnd Callback triggered at the end of the framing animation
         */
        zoomOnMeshesHierarchy(meshes: AbstractMesh[], focusOnOriginXZ?: boolean, onAnimationEnd?: Nullable<() => void>): void;
        /**
         * Targets the bounding box info defined by its extends and updates zoom level accordingly.
         * @param minimumWorld Determines the smaller position of the bounding box extend
         * @param maximumWorld Determines the bigger position of the bounding box extend
         * @param focusOnOriginXZ Determines if the camera should focus on 0 in the X and Z axis instead of the mesh
         * @param onAnimationEnd Callback triggered at the end of the framing animation
         * @returns true if the zoom was done
         */
        zoomOnBoundingInfo(minimumWorld: Vector3, maximumWorld: Vector3, focusOnOriginXZ?: boolean, onAnimationEnd?: Nullable<() => void>): boolean;
        /**
         * Calculates the lowest radius for the camera based on the bounding box of the mesh.
         * @param minimumWorld
         * @param maximumWorld
         * @returns The minimum distance from the primary mesh's center point at which the camera must be kept in order
         *		 to fully enclose the mesh in the viewing frustum.
         */
        protected _calculateLowerRadiusFromModelBoundingSphere(minimumWorld: Vector3, maximumWorld: Vector3): number;
        /**
         * Keeps the camera above the ground plane. If the user pulls the camera below the ground plane, the camera
         * is automatically returned to its default position (expected to be above ground plane).
         */
        private _maintainCameraAboveGround;
        /**
         * Removes all animation locks. Allows new animations to be added to any of the arcCamera properties.
         */
        private _clearAnimationLocks;
        /**
         *  Applies any current user interaction to the camera. Takes into account maximum alpha rotation.
         */
        private _applyUserInteraction;
        /**
         * Stops and removes all animations that have been applied to the camera
         */
        stopAllAnimations(): void;
        /**
         * Gets a value indicating if the user is moving the camera
         */
        get isUserIsMoving(): boolean;
        /**
         * The camera can move all the way towards the mesh.
         */
        static IgnoreBoundsSizeMode: number;
        /**
         * The camera is not allowed to zoom closer to the mesh than the point at which the adjusted bounding sphere touches the frustum sides
         */
        static FitFrustumSidesMode: number;
    }






    /**
     * A behavior that when attached to a mesh will will place a specified node on the meshes face pointing towards the camera
     */
    export class AttachToBoxBehavior implements Behavior<Mesh> {
        private _ui;
        /**
         *  The name of the behavior
         */
        name: string;
        /**
         * The distance away from the face of the mesh that the UI should be attached to (default: 0.15)
         */
        distanceAwayFromFace: number;
        /**
         * The distance from the bottom of the face that the UI should be attached to (default: 0.15)
         */
        distanceAwayFromBottomOfFace: number;
        private _faceVectors;
        private _target;
        private _scene;
        private _onRenderObserver;
        private _tmpMatrix;
        private _tmpVector;
        /**
         * Creates the AttachToBoxBehavior, used to attach UI to the closest face of the box to a camera
         * @param _ui The transform node that should be attached to the mesh
         */
        constructor(_ui: TransformNode);
        /**
         *  Initializes the behavior
         */
        init(): void;
        private _closestFace;
        private _zeroVector;
        private _lookAtTmpMatrix;
        private _lookAtToRef;
        /**
         * Attaches the AttachToBoxBehavior to the passed in mesh
         * @param target The mesh that the specified node will be attached to
         */
        attach(target: Mesh): void;
        /**
         *  Detaches the behavior from the mesh
         */
        detach(): void;
    }


    /**
     * Data store to track virtual pointers movement
     */
    type VirtualMeshInfo = {
        dragging: boolean;
        moving: boolean;
        dragMesh: AbstractMesh;
        originMesh: AbstractMesh;
        pivotMesh: AbstractMesh;
        startingPivotPosition: Vector3;
        startingPivotOrientation: Quaternion;
        startingPosition: Vector3;
        startingOrientation: Quaternion;
        lastOriginPosition: Vector3;
        lastDragPosition: Vector3;
    };
    /**
     * Base behavior for six degrees of freedom interactions in XR experiences.
     * Creates virtual meshes that are dragged around
     * And observables for position/rotation changes
     */
    export class BaseSixDofDragBehavior implements Behavior<Mesh> {
        protected static _virtualScene: Scene;
        private _pointerObserver;
        private _attachedToElement;
        protected _virtualMeshesInfo: {
            [id: number]: VirtualMeshInfo;
        };
        private _tmpVector;
        private _tmpQuaternion;
        protected _dragType: {
            NONE: number;
            DRAG: number;
            DRAG_WITH_CONTROLLER: number;
            NEAR_DRAG: number;
        };
        protected _scene: Scene;
        protected _moving: boolean;
        protected _ownerNode: TransformNode;
        protected _dragging: number;
        /**
         * The list of child meshes that can receive drag events
         * If `null`, all child meshes will receive drag event
         */
        draggableMeshes: Nullable<AbstractMesh[]>;
        /**
         * How much faster the object should move when the controller is moving towards it. This is useful to bring objects that are far away from the user to them faster. Set this to 0 to avoid any speed increase. (Default: 3)
         */
        zDragFactor: number;
        /**
         * The id of the pointer that is currently interacting with the behavior (-1 when no pointer is active)
         */
        get currentDraggingPointerId(): number;
        set currentDraggingPointerId(value: number);
        /**
         * In case of multipointer interaction, all pointer ids currently active are stored here
         */
        currentDraggingPointerIds: number[];
        /**
         * Get or set the currentDraggingPointerId
         * @deprecated Please use currentDraggingPointerId instead
         */
        get currentDraggingPointerID(): number;
        set currentDraggingPointerID(currentDraggingPointerID: number);
        /**
        /**
         * If camera controls should be detached during the drag
         */
        detachCameraControls: boolean;
        /**
         * Fires each time a drag starts
         */
        onDragStartObservable: Observable<{
            position: Vector3;
        }>;
        /**
         * Fires each time a drag happens
         */
        onDragObservable: Observable<{
            delta: Vector3;
            position: Vector3;
            pickInfo: PickingInfo;
        }>;
        /**
         *  Fires each time a drag ends (eg. mouse release after drag)
         */
        onDragEndObservable: Observable<{}>;
        /**
         * Should the behavior allow simultaneous pointers to interact with the owner node.
         */
        allowMultiPointer: boolean;
        /**
         *  The name of the behavior
         */
        get name(): string;
        /**
         *  Returns true if the attached mesh is currently moving with this behavior
         */
        get isMoving(): boolean;
        /**
         *  Initializes the behavior
         */
        init(): void;
        /**
         * In the case of multiple active cameras, the cameraToUseForPointers should be used if set instead of active camera
         */
        private get _pointerCamera();
        private _createVirtualMeshInfo;
        protected _resetVirtualMeshesPosition(): void;
        private _pointerUpdate2D;
        private _pointerUpdateXR;
        /**
         * Attaches the scale behavior the passed in mesh
         * @param ownerNode The mesh that will be scaled around once attached
         */
        attach(ownerNode: TransformNode): void;
        private _applyZOffset;
        protected _targetDragStart(worldPosition: Vector3, worldRotation: Quaternion, pointerId: number): void;
        protected _targetDrag(worldDeltaPosition: Vector3, worldDeltaRotation: Quaternion, pointerId: number): void;
        protected _targetDragEnd(pointerId: number): void;
        protected _reattachCameraControls(): void;
        /**
         * Detaches the behavior from the mesh
         */
        detach(): void;
    }


    /**
     * A behavior that when attached to a mesh will allow the mesh to fade in and out
     */
    export class FadeInOutBehavior implements Behavior<Mesh> {
        /**
         * Time in milliseconds to delay before fading in (Default: 0)
         */
        fadeInDelay: number;
        /**
         * Time in milliseconds to delay before fading out (Default: 0)
         */
        fadeOutDelay: number;
        /**
         * Time in milliseconds for the mesh to fade in (Default: 300)
         */
        fadeInTime: number;
        /**
         * Time in milliseconds for the mesh to fade out (Default: 300)
         */
        fadeOutTime: number;
        /**
         * Time in milliseconds to delay before fading in (Default: 0)
         * Will set both fade in and out delay to the same value
         */
        get delay(): number;
        set delay(value: number);
        private _millisecondsPerFrame;
        private _hovered;
        private _hoverValue;
        private _ownerNode;
        private _onBeforeRenderObserver;
        private _delay;
        private _time;
        /**
         * Instantiates the FadeInOutBehavior
         */
        constructor();
        /**
         *  The name of the behavior
         */
        get name(): string;
        /**
         *  Initializes the behavior
         */
        init(): void;
        /**
         * Attaches the fade behavior on the passed in mesh
         * @param ownerNode The mesh that will be faded in/out once attached
         */
        attach(ownerNode: Mesh): void;
        /**
         *  Detaches the behavior from the mesh
         */
        detach(): void;
        /**
         * Triggers the mesh to begin fading in (or out)
         * @param fadeIn if the object should fade in or out (true to fade in)
         */
        fadeIn(fadeIn?: boolean): void;
        /**
         * Triggers the mesh to begin fading out
         */
        fadeOut(): void;
        private _update;
        private _setAllVisibility;
        private _attachObserver;
        private _detachObserver;
    }


    /**
     * A behavior that when attached to a mesh will follow a camera
     * @since 5.0.0
     */
    export class FollowBehavior implements Behavior<TransformNode> {
        private _scene;
        private _tmpQuaternion;
        private _tmpVectors;
        private _tmpMatrix;
        private _tmpInvertView;
        private _tmpForward;
        private _tmpNodeForward;
        private _tmpPosition;
        private _followedCamera;
        private _onBeforeRender;
        private _workingPosition;
        private _workingQuaternion;
        private _lastTick;
        private _recenterNextUpdate;
        /**
         * Attached node of this behavior
         */
        attachedNode: Nullable<TransformNode>;
        /**
         * Set to false if the node should strictly follow the camera without any interpolation time
         */
        interpolatePose: boolean;
        /**
         * Rate of interpolation of position and rotation of the attached node.
         * Higher values will give a slower interpolation.
         */
        lerpTime: number;
        /**
         * If the behavior should ignore the pitch and roll of the camera.
         */
        ignoreCameraPitchAndRoll: boolean;
        /**
         * Pitch offset from camera (relative to Max Distance)
         * Is only effective if `ignoreCameraPitchAndRoll` is set to `true`.
         */
        pitchOffset: number;
        /**
         * The vertical angle from the camera forward axis to the owner will not exceed this value
         */
        maxViewVerticalDegrees: number;
        /**
         * The horizontal angle from the camera forward axis to the owner will not exceed this value
         */
        maxViewHorizontalDegrees: number;
        /**
         * The attached node will not reorient until the angle between its forward vector and the vector to the camera is greater than this value
         */
        orientToCameraDeadzoneDegrees: number;
        /**
         * Option to ignore distance clamping
         */
        ignoreDistanceClamp: boolean;
        /**
         * Option to ignore angle clamping
         */
        ignoreAngleClamp: boolean;
        /**
         * Max vertical distance between the attachedNode and camera
         */
        verticalMaxDistance: number;
        /**
         *  Default distance from eye to attached node, i.e. the sphere radius
         */
        defaultDistance: number;
        /**
         *  Max distance from eye to attached node, i.e. the sphere radius
         */
        maximumDistance: number;
        /**
         *  Min distance from eye to attached node, i.e. the sphere radius
         */
        minimumDistance: number;
        /**
         * Ignore vertical movement and lock the Y position of the object.
         */
        useFixedVerticalOffset: boolean;
        /**
         * Fixed vertical position offset distance.
         */
        fixedVerticalOffset: number;
        /**
         * Enables/disables the behavior
         * @internal
         */
        _enabled: boolean;
        /**
         * The camera that should be followed by this behavior
         */
        get followedCamera(): Nullable<Camera>;
        set followedCamera(camera: Nullable<Camera>);
        /**
         *  The name of the behavior
         */
        get name(): string;
        /**
         *  Initializes the behavior
         */
        init(): void;
        /**
         * Attaches the follow behavior
         * @param ownerNode The mesh that will be following once attached
         * @param followedCamera The camera that should be followed by the node
         */
        attach(ownerNode: TransformNode, followedCamera?: Camera): void;
        /**
         *  Detaches the behavior from the mesh
         */
        detach(): void;
        /**
         * Recenters the attached node in front of the camera on the next update
         */
        recenter(): void;
        private _angleBetweenVectorAndPlane;
        private _length2D;
        private _distanceClamp;
        private _applyVerticalClamp;
        private _toOrientationQuatToRef;
        private _applyPitchOffset;
        private _angularClamp;
        private _orientationClamp;
        private _passedOrientationDeadzone;
        private _updateLeashing;
        private _updateTransformToGoal;
        private _addObservables;
        private _removeObservables;
    }


    /**
     * Zones around the hand
     */
    export enum HandConstraintZone {
        /**
         * Above finger tips
         */
        ABOVE_FINGER_TIPS = 0,
        /**
         * Next to the thumb
         */
        RADIAL_SIDE = 1,
        /**
         * Next to the pinky finger
         */
        ULNAR_SIDE = 2,
        /**
         * Below the wrist
         */
        BELOW_WRIST = 3
    }
    /**
     * Orientations for the hand zones and for the attached node
     */
    export enum HandConstraintOrientation {
        /**
         * Orientation is towards the camera
         */
        LOOK_AT_CAMERA = 0,
        /**
         * Orientation is determined by the rotation of the palm
         */
        HAND_ROTATION = 1
    }
    /**
     * Orientations for the hand zones and for the attached node
     */
    export enum HandConstraintVisibility {
        /**
         * Constraint is always visible
         */
        ALWAYS_VISIBLE = 0,
        /**
         * Constraint is only visible when the palm is up
         */
        PALM_UP = 1,
        /**
         * Constraint is only visible when the user is looking at the constraint.
         * Uses XR Eye Tracking if enabled/available, otherwise uses camera direction
         */
        GAZE_FOCUS = 2,
        /**
         * Constraint is only visible when the palm is up and the user is looking at it
         */
        PALM_AND_GAZE = 3
    }
    /**
     * Hand constraint behavior that makes the attached `TransformNode` follow hands in XR experiences.
     * @since 5.0.0
     */
    export class HandConstraintBehavior implements Behavior<TransformNode> {
        private _scene;
        private _node;
        private _eyeTracking;
        private _handTracking;
        private _sceneRenderObserver;
        private _zoneAxis;
        /**
         * Sets the HandConstraintVisibility level for the hand constraint
         */
        handConstraintVisibility: HandConstraintVisibility;
        /**
         * A number from 0.0 to 1.0, marking how restricted the direction the palm faces is for the attached node to be enabled.
         * A 1 means the palm must be directly facing the user before the node is enabled, a 0 means it is always enabled.
         * Used with HandConstraintVisibility.PALM_UP
         */
        palmUpStrictness: number;
        /**
         * The radius in meters around the center of the hand that the user must gaze inside for the attached node to be enabled and appear.
         * Used with HandConstraintVisibility.GAZE_FOCUS
         */
        gazeProximityRadius: number;
        /**
         * Offset distance from the hand in meters
         */
        targetOffset: number;
        /**
         * Where to place the node regarding the center of the hand.
         */
        targetZone: HandConstraintZone;
        /**
         * Orientation mode of the 4 zones around the hand
         */
        zoneOrientationMode: HandConstraintOrientation;
        /**
         * Orientation mode of the node attached to this behavior
         */
        nodeOrientationMode: HandConstraintOrientation;
        /**
         * Set the hand this behavior should follow. If set to "none", it will follow any visible hand (prioritising the left one).
         */
        handedness: XRHandedness;
        /**
         * Rate of interpolation of position and rotation of the attached node.
         * Higher values will give a slower interpolation.
         */
        lerpTime: number;
        /**
         * Builds a hand constraint behavior
         */
        constructor();
        /** gets or sets behavior's name */
        get name(): string;
        /** Enable the behavior */
        enable(): void;
        /** Disable the behavior */
        disable(): void;
        private _getHandPose;
        /**
         * Initializes the hand constraint behavior
         */
        init(): void;
        /**
         * Attaches the hand constraint to a `TransformNode`
         * @param node defines the node to attach the behavior to
         */
        attach(node: TransformNode): void;
        private _setVisibility;
        /**
         * Detaches the behavior from the `TransformNode`
         */
        detach(): void;
        /**
         * Links the behavior to the XR experience in which to retrieve hand transform information.
         * @param xr xr experience
         */
        linkToXRExperience(xr: WebXRExperienceHelper | WebXRFeaturesManager): void;
    }




    /**
     * A behavior that when attached to a mesh will allow the mesh to be scaled
     */
    export class MultiPointerScaleBehavior implements Behavior<Mesh> {
        private _dragBehaviorA;
        private _dragBehaviorB;
        private _startDistance;
        private _initialScale;
        private _targetScale;
        private _ownerNode;
        private _sceneRenderObserver;
        /**
         * Instantiate a new behavior that when attached to a mesh will allow the mesh to be scaled
         */
        constructor();
        /**
         *  The name of the behavior
         */
        get name(): string;
        /**
         *  Initializes the behavior
         */
        init(): void;
        private _getCurrentDistance;
        /**
         * Attaches the scale behavior the passed in mesh
         * @param ownerNode The mesh that will be scaled around once attached
         */
        attach(ownerNode: Mesh): void;
        /**
         *  Detaches the behavior from the mesh
         */
        detach(): void;
    }


    /**
     * A behavior that when attached to a mesh will allow the mesh to be dragged around the screen based on pointer events
     */
    export class PointerDragBehavior implements Behavior<AbstractMesh> {
        private static _AnyMouseId;
        /**
         * Abstract mesh the behavior is set on
         */
        attachedNode: AbstractMesh;
        protected _dragPlane: Mesh;
        private _scene;
        private _pointerObserver;
        private _beforeRenderObserver;
        private static _PlaneScene;
        private _useAlternatePickedPointAboveMaxDragAngleDragSpeed;
        private _activeDragButton;
        private _activePointerInfo;
        /**
         * The maximum tolerated angle between the drag plane and dragging pointer rays to trigger pointer events. Set to 0 to allow any angle (default: 0)
         */
        maxDragAngle: number;
        /**
         * Butttons that can be used to initiate a drag
         */
        dragButtons: number[];
        /**
         * @internal
         */
        _useAlternatePickedPointAboveMaxDragAngle: boolean;
        /**
         * Get or set the currentDraggingPointerId
         * @deprecated Please use currentDraggingPointerId instead
         */
        get currentDraggingPointerID(): number;
        set currentDraggingPointerID(currentDraggingPointerID: number);
        /**
         * The id of the pointer that is currently interacting with the behavior (-1 when no pointer is active)
         */
        currentDraggingPointerId: number;
        /**
         * The last position where the pointer hit the drag plane in world space
         */
        lastDragPosition: Vector3;
        /**
         * If the behavior is currently in a dragging state
         */
        dragging: boolean;
        /**
         * The distance towards the target drag position to move each frame. This can be useful to avoid jitter. Set this to 1 for no delay. (Default: 0.2)
         */
        dragDeltaRatio: number;
        /**
         * If the drag plane orientation should be updated during the dragging (Default: true)
         */
        updateDragPlane: boolean;
        private _debugMode;
        private _moving;
        /**
         *  Fires each time the attached mesh is dragged with the pointer
         *  * delta between last drag position and current drag position in world space
         *  * dragDistance along the drag axis
         *  * dragPlaneNormal normal of the current drag plane used during the drag
         *  * dragPlanePoint in world space where the drag intersects the drag plane
         *
         *  (if validatedDrag is used, the position of the attached mesh might not equal dragPlanePoint)
         */
        onDragObservable: Observable<{
            delta: Vector3;
            dragPlanePoint: Vector3;
            dragPlaneNormal: Vector3;
            dragDistance: number;
            pointerId: number;
            pointerInfo: Nullable<PointerInfo>;
        }>;
        /**
         *  Fires each time a drag begins (eg. mouse down on mesh)
         *  * dragPlanePoint in world space where the drag intersects the drag plane
         *
         *  (if validatedDrag is used, the position of the attached mesh might not equal dragPlanePoint)
         */
        onDragStartObservable: Observable<{
            dragPlanePoint: Vector3;
            pointerId: number;
            pointerInfo: Nullable<PointerInfo>;
        }>;
        /**
         *  Fires each time a drag ends (eg. mouse release after drag)
         *  * dragPlanePoint in world space where the drag intersects the drag plane
         *
         *  (if validatedDrag is used, the position of the attached mesh might not equal dragPlanePoint)
         */
        onDragEndObservable: Observable<{
            dragPlanePoint: Vector3;
            pointerId: number;
            pointerInfo: Nullable<PointerInfo>;
        }>;
        /**
         *  Fires each time behavior enabled state changes
         */
        onEnabledObservable: Observable<boolean>;
        /**
         *  If the attached mesh should be moved when dragged
         */
        moveAttached: boolean;
        /**
         *  If the drag behavior will react to drag events (Default: true)
         */
        set enabled(value: boolean);
        get enabled(): boolean;
        private _enabled;
        /**
         * If pointer events should start and release the drag (Default: true)
         */
        startAndReleaseDragOnPointerEvents: boolean;
        /**
         * If camera controls should be detached during the drag
         */
        detachCameraControls: boolean;
        /**
         * If set, the drag plane/axis will be rotated based on the attached mesh's world rotation (Default: true)
         */
        useObjectOrientationForDragging: boolean;
        private _options;
        /**
         * Gets the options used by the behavior
         */
        get options(): {
            dragAxis?: Vector3;
            dragPlaneNormal?: Vector3;
        };
        /**
         * Sets the options used by the behavior
         */
        set options(options: {
            dragAxis?: Vector3;
            dragPlaneNormal?: Vector3;
        });
        /**
         * Creates a pointer drag behavior that can be attached to a mesh
         * @param options The drag axis or normal of the plane that will be dragged across. If no options are specified the drag plane will always face the ray's origin (eg. camera)
         * @param options.dragAxis
         * @param options.dragPlaneNormal
         */
        constructor(options?: {
            dragAxis?: Vector3;
            dragPlaneNormal?: Vector3;
        });
        /**
         * Predicate to determine if it is valid to move the object to a new position when it is moved.
         * In the case of rotation gizmo, target contains the angle.
         * @param target destination position or desired angle delta
         * @returns boolean for whether or not it is valid to move
         */
        validateDrag: (target: Vector3) => boolean;
        /**
         *  The name of the behavior
         */
        get name(): string;
        /**
         *  Initializes the behavior
         */
        init(): void;
        private _tmpVector;
        private _alternatePickedPoint;
        private _worldDragAxis;
        private _targetPosition;
        private _attachedToElement;
        /**
         * Attaches the drag behavior the passed in mesh
         * @param ownerNode The mesh that will be dragged around once attached
         * @param predicate Predicate to use for pick filtering
         */
        attach(ownerNode: AbstractMesh, predicate?: (m: AbstractMesh) => boolean): void;
        /**
         * Force release the drag action by code.
         */
        releaseDrag(): void;
        private _startDragRay;
        private _lastPointerRay;
        /**
         * Simulates the start of a pointer drag event on the behavior
         * @param pointerId pointerID of the pointer that should be simulated (Default: Any mouse pointer ID)
         * @param fromRay initial ray of the pointer to be simulated (Default: Ray from camera to attached mesh)
         * @param startPickedPoint picked point of the pointer to be simulated (Default: attached mesh position)
         */
        startDrag(pointerId?: number, fromRay?: Ray, startPickedPoint?: Vector3): void;
        protected _startDrag(pointerId: number, fromRay?: Ray, startPickedPoint?: Vector3): void;
        private _dragDelta;
        protected _moveDrag(ray: Ray): void;
        private _pickWithRayOnDragPlane;
        private _pointA;
        private _pointC;
        private _localAxis;
        private _lookAt;
        private _updateDragPlanePosition;
        /**
         *  Detaches the behavior from the mesh
         */
        detach(): void;
    }


    /**
     * A behavior that when attached to a mesh will allow the mesh to be dragged around based on directions and origin of the pointer's ray
     */
    export class SixDofDragBehavior extends BaseSixDofDragBehavior {
        private _sceneRenderObserver;
        private _virtualTransformNode;
        protected _targetPosition: Vector3;
        protected _targetOrientation: Quaternion;
        protected _targetScaling: Vector3;
        protected _startingPosition: Vector3;
        protected _startingOrientation: Quaternion;
        protected _startingScaling: Vector3;
        /**
         * Fires when position is updated
         */
        onPositionChangedObservable: Observable<{
            position: Vector3;
        }>;
        /**
         * The distance towards the target drag position to move each frame. This can be useful to avoid jitter. Set this to 1 for no delay. (Default: 0.2)
         */
        dragDeltaRatio: number;
        /**
         * If the object should rotate to face the drag origin
         */
        rotateDraggedObject: boolean;
        /**
         * If `rotateDraggedObject` is set to `true`, this parameter determines if we are only rotating around the y axis (yaw)
         */
        rotateAroundYOnly: boolean;
        /**
         * Should the behavior rotate 1:1 with the motion controller, when one is used.
         */
        rotateWithMotionController: boolean;
        /**
         *  The name of the behavior
         */
        get name(): string;
        /**
         * Use this flag to update the target but not move the owner node towards the target
         */
        disableMovement: boolean;
        /**
         * Should the object rotate towards the camera when we start dragging it
         */
        faceCameraOnDragStart: boolean;
        /**
         * Attaches the six DoF drag behavior
         * @param ownerNode The mesh that will be dragged around once attached
         */
        attach(ownerNode: Mesh): void;
        private _getPositionOffsetAround;
        private _onePointerPositionUpdated;
        private _twoPointersPositionUpdated;
        protected _targetDragStart(): void;
        protected _targetDrag(worldDeltaPosition: Vector3, worldDeltaRotation: Quaternion): void;
        protected _targetDragEnd(): void;
        /**
         *  Detaches the behavior from the mesh
         */
        detach(): void;
    }


    /**
     * A behavior that allows a transform node to stick to a surface position/orientation
     * @since 5.0.0
     */
    export class SurfaceMagnetismBehavior implements Behavior<Mesh> {
        private _scene;
        private _attachedMesh;
        private _attachPointLocalOffset;
        private _pointerObserver;
        private _workingPosition;
        private _workingQuaternion;
        private _lastTick;
        private _onBeforeRender;
        private _hit;
        /**
         * Distance offset from the hit point to place the target at, along the hit normal.
         */
        hitNormalOffset: number;
        /**
         * Name of the behavior
         */
        get name(): string;
        /**
         * Spatial mapping meshes to collide with
         */
        meshes: AbstractMesh[];
        /**
         * Function called when the behavior needs to be initialized (after attaching it to a target)
         */
        init(): void;
        /**
         * Set to false if the node should strictly follow the camera without any interpolation time
         */
        interpolatePose: boolean;
        /**
         * Rate of interpolation of position and rotation of the attached node.
         * Higher values will give a slower interpolation.
         */
        lerpTime: number;
        /**
         * If true, pitch and roll are omitted.
         */
        keepOrientationVertical: boolean;
        /**
         * Is this behavior reacting to pointer events
         */
        enabled: boolean;
        /**
         * Maximum distance for the node to stick to the surface
         */
        maxStickingDistance: number;
        /**
         * Attaches the behavior to a transform node
         * @param target defines the target where the behavior is attached to
         * @param scene the scene
         */
        attach(target: Mesh, scene?: Scene): void;
        /**
         * Detaches the behavior
         */
        detach(): void;
        private _getTargetPose;
        /**
         * Updates the attach point with the current geometry extents of the attached mesh
         */
        updateAttachPoint(): void;
        /**
         * Finds the intersection point of the given ray onto the meshes and updates the target.
         * Transformation will be interpolated according to `interpolatePose` and `lerpTime` properties.
         * If no mesh of `meshes` are hit, this does nothing.
         * @param pickInfo The input pickingInfo that will be used to intersect the meshes
         * @returns a boolean indicating if we found a hit to stick to
         */
        findAndUpdateTarget(pickInfo: PickingInfo): boolean;
        private _getAttachPointOffsetToRef;
        private _updateTransformToGoal;
        private _addObservables;
        private _removeObservables;
    }


    /**
     * Class used to store bone information
     * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/bonesSkeletons
     */
    export class Bone extends Node {
        /**
         * defines the bone name
         */
        name: string;
        private static _TmpVecs;
        private static _TmpQuat;
        private static _TmpMats;
        /**
         * Gets the list of child bones
         */
        children: Bone[];
        /** Gets the animations associated with this bone */
        animations: Animation[];
        /**
         * Gets or sets bone length
         */
        length: number;
        /**
         * @internal Internal only
         * Set this value to map this bone to a different index in the transform matrices
         * Set this value to -1 to exclude the bone from the transform matrices
         */
        _index: Nullable<number>;
        private _skeleton;
        private _localMatrix;
        private _absoluteMatrix;
        private _bindMatrix;
        private _absoluteBindMatrix;
        private _absoluteInverseBindMatrix;
        private _finalMatrix;
        private _restMatrix;
        private _scalingDeterminant;
        private _localScaling;
        private _localRotation;
        private _localPosition;
        private _needToDecompose;
        private _needToCompose;
        /** @internal */
        _linkedTransformNode: Nullable<TransformNode>;
        /** @internal */
        _waitingTransformNodeId: Nullable<string>;
        /** @internal */
        get _matrix(): Matrix;
        /** @internal */
        set _matrix(value: Matrix);
        /**
         * Create a new bone
         * @param name defines the bone name
         * @param skeleton defines the parent skeleton
         * @param parentBone defines the parent (can be null if the bone is the root)
         * @param localMatrix defines the local matrix (default: identity)
         * @param restMatrix defines the rest matrix (default: localMatrix)
         * @param bindMatrix defines the bind matrix (default: localMatrix)
         * @param index defines index of the bone in the hierarchy (default: null)
         */
        constructor(
        /**
         * defines the bone name
         */
        name: string, skeleton: Skeleton, parentBone?: Nullable<Bone>, localMatrix?: Nullable<Matrix>, restMatrix?: Nullable<Matrix>, bindMatrix?: Nullable<Matrix>, index?: Nullable<number>);
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Gets the parent skeleton
         * @returns a skeleton
         */
        getSkeleton(): Skeleton;
        get parent(): Bone;
        /**
         * Gets parent bone
         * @returns a bone or null if the bone is the root of the bone hierarchy
         */
        getParent(): Nullable<Bone>;
        /**
         * Returns an array containing the children of the bone
         * @returns an array containing the children of the bone (can be empty if the bone has no children)
         */
        getChildren(): Array<Bone>;
        /**
         * Gets the node index in matrix array generated for rendering
         * @returns the node index
         */
        getIndex(): number;
        set parent(newParent: Nullable<Bone>);
        /**
         * Sets the parent bone
         * @param parent defines the parent (can be null if the bone is the root)
         * @param updateAbsoluteBindMatrices defines if the absolute bind and absolute inverse bind matrices must be updated
         */
        setParent(parent: Nullable<Bone>, updateAbsoluteBindMatrices?: boolean): void;
        /**
         * Gets the local matrix
         * @returns the local matrix
         */
        getLocalMatrix(): Matrix;
        /**
         * Gets the bind matrix
         * @returns the bind matrix
         */
        getBindMatrix(): Matrix;
        /**
         * Gets the bind matrix.
         * @returns the bind matrix
         * @deprecated Please use getBindMatrix instead
         */
        getBaseMatrix(): Matrix;
        /**
         * Gets the rest matrix
         * @returns the rest matrix
         */
        getRestMatrix(): Matrix;
        /**
         * Gets the rest matrix
         * @returns the rest matrix
         * @deprecated Please use getRestMatrix instead
         */
        getRestPose(): Matrix;
        /**
         * Sets the rest matrix
         * @param matrix the local-space rest matrix to set for this bone
         */
        setRestMatrix(matrix: Matrix): void;
        /**
         * Sets the rest matrix
         * @param matrix the local-space rest to set for this bone
         * @deprecated Please use setRestMatrix instead
         */
        setRestPose(matrix: Matrix): void;
        /**
         * Gets the bind matrix
         * @returns the bind matrix
         * @deprecated Please use getBindMatrix instead
         */
        getBindPose(): Matrix;
        /**
         * Sets the bind matrix
         * This will trigger a recomputation of the absolute bind and absolute inverse bind matrices for this bone and its children
         * Note that the local matrix will also be set with the matrix passed in parameter!
         * @param matrix the local-space bind matrix to set for this bone
         */
        setBindMatrix(matrix: Matrix): void;
        /**
         * Sets the bind matrix
         * @param matrix the local-space bind to set for this bone
         * @deprecated Please use setBindMatrix instead
         */
        setBindPose(matrix: Matrix): void;
        /**
         * Gets the matrix used to store the final world transformation of the bone (ie. the matrix sent to shaders)
         * @returns the final world matrix
         */
        getFinalMatrix(): Matrix;
        /**
         * Gets the matrix used to store the final world transformation of the bone (ie. the matrix sent to shaders)
         * @deprecated Please use getFinalMatrix instead
         * @returns the final world matrix
         */
        getWorldMatrix(): Matrix;
        /**
         * Sets the local matrix to the rest matrix
         */
        returnToRest(): void;
        /**
         * Gets the inverse of the bind matrix, in world space (relative to the skeleton root)
         * @returns the inverse bind matrix, in world space
         */
        getAbsoluteInverseBindMatrix(): Matrix;
        /**
         * Gets the inverse of the bind matrix, in world space (relative to the skeleton root)
         * @returns the inverse bind matrix, in world space
         * @deprecated Please use getAbsoluteInverseBindMatrix instead
         */
        getInvertedAbsoluteTransform(): Matrix;
        /**
         * Gets the bone matrix, in world space (relative to the skeleton root)
         * @returns the bone matrix, in world space
         */
        getAbsoluteMatrix(): Matrix;
        /**
         * Gets the bone matrix, in world space (relative to the skeleton root)
         * @returns the bone matrix, in world space
         * @deprecated Please use getAbsoluteMatrix instead
         */
        getAbsoluteTransform(): Matrix;
        /**
         * Links with the given transform node.
         * The local matrix of this bone is overwritten by the transform of the node every frame.
         * @param transformNode defines the transform node to link to
         */
        linkTransformNode(transformNode: Nullable<TransformNode>): void;
        /**
         * Gets the node used to drive the bone's transformation
         * @returns a transform node or null
         */
        getTransformNode(): Nullable<TransformNode>;
        /** Gets or sets current position (in local space) */
        get position(): Vector3;
        set position(newPosition: Vector3);
        /** Gets or sets current rotation (in local space) */
        get rotation(): Vector3;
        set rotation(newRotation: Vector3);
        /** Gets or sets current rotation quaternion (in local space) */
        get rotationQuaternion(): Quaternion;
        set rotationQuaternion(newRotation: Quaternion);
        /** Gets or sets current scaling (in local space) */
        get scaling(): Vector3;
        set scaling(newScaling: Vector3);
        /**
         * Gets the animation properties override
         */
        get animationPropertiesOverride(): Nullable<AnimationPropertiesOverride>;
        private _decompose;
        private _compose;
        /**
         * Update the bind (and optionally the local) matrix
         * @param bindMatrix defines the new matrix to set to the bind/local matrix, in local space
         * @param updateAbsoluteBindMatrices defines if the absolute bind and absolute inverse bind matrices must be recomputed (default: true)
         * @param updateLocalMatrix defines if the local matrix should also be updated with the matrix passed in parameter (default: true)
         */
        updateMatrix(bindMatrix: Matrix, updateAbsoluteBindMatrices?: boolean, updateLocalMatrix?: boolean): void;
        /**
         * @internal
         */
        _updateAbsoluteBindMatrices(bindMatrix?: Matrix, updateChildren?: boolean): void;
        /**
         * Flag the bone as dirty (Forcing it to update everything)
         * @returns this bone
         */
        markAsDirty(): Bone;
        /** @internal */
        _markAsDirtyAndCompose(): void;
        private _markAsDirtyAndDecompose;
        private _updatePosition;
        /**
         * Translate the bone in local or world space
         * @param vec The amount to translate the bone
         * @param space The space that the translation is in (default: Space.LOCAL)
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        translate(vec: Vector3, space?: Space, tNode?: TransformNode): void;
        /**
         * Set the position of the bone in local or world space
         * @param position The position to set the bone
         * @param space The space that the position is in (default: Space.LOCAL)
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setPosition(position: Vector3, space?: Space, tNode?: TransformNode): void;
        /**
         * Set the absolute position of the bone (world space)
         * @param position The position to set the bone
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setAbsolutePosition(position: Vector3, tNode?: TransformNode): void;
        /**
         * Scale the bone on the x, y and z axes (in local space)
         * @param x The amount to scale the bone on the x axis
         * @param y The amount to scale the bone on the y axis
         * @param z The amount to scale the bone on the z axis
         * @param scaleChildren sets this to true if children of the bone should be scaled as well (false by default)
         */
        scale(x: number, y: number, z: number, scaleChildren?: boolean): void;
        /**
         * Set the bone scaling in local space
         * @param scale defines the scaling vector
         */
        setScale(scale: Vector3): void;
        /**
         * Gets the current scaling in local space
         * @returns the current scaling vector
         */
        getScale(): Vector3;
        /**
         * Gets the current scaling in local space and stores it in a target vector
         * @param result defines the target vector
         */
        getScaleToRef(result: Vector3): void;
        /**
         * Set the yaw, pitch, and roll of the bone in local or world space
         * @param yaw The rotation of the bone on the y axis
         * @param pitch The rotation of the bone on the x axis
         * @param roll The rotation of the bone on the z axis
         * @param space The space that the axes of rotation are in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setYawPitchRoll(yaw: number, pitch: number, roll: number, space?: Space, tNode?: TransformNode): void;
        /**
         * Add a rotation to the bone on an axis in local or world space
         * @param axis The axis to rotate the bone on
         * @param amount The amount to rotate the bone
         * @param space The space that the axis is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        rotate(axis: Vector3, amount: number, space?: Space, tNode?: TransformNode): void;
        /**
         * Set the rotation of the bone to a particular axis angle in local or world space
         * @param axis The axis to rotate the bone on
         * @param angle The angle that the bone should be rotated to
         * @param space The space that the axis is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setAxisAngle(axis: Vector3, angle: number, space?: Space, tNode?: TransformNode): void;
        /**
         * Set the euler rotation of the bone in local or world space
         * @param rotation The euler rotation that the bone should be set to
         * @param space The space that the rotation is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setRotation(rotation: Vector3, space?: Space, tNode?: TransformNode): void;
        /**
         * Set the quaternion rotation of the bone in local or world space
         * @param quat The quaternion rotation that the bone should be set to
         * @param space The space that the rotation is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setRotationQuaternion(quat: Quaternion, space?: Space, tNode?: TransformNode): void;
        /**
         * Set the rotation matrix of the bone in local or world space
         * @param rotMat The rotation matrix that the bone should be set to
         * @param space The space that the rotation is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         */
        setRotationMatrix(rotMat: Matrix, space?: Space, tNode?: TransformNode): void;
        private _rotateWithMatrix;
        private _getAbsoluteInverseMatrixUnscaledToRef;
        /**
         * Get the position of the bone in local or world space
         * @param space The space that the returned position is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The position of the bone
         */
        getPosition(space?: Space, tNode?: Nullable<TransformNode>): Vector3;
        /**
         * Copy the position of the bone to a vector3 in local or world space
         * @param space The space that the returned position is in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The vector3 to copy the position to
         */
        getPositionToRef(space: Space | undefined, tNode: Nullable<TransformNode>, result: Vector3): void;
        /**
         * Get the absolute position of the bone (world space)
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The absolute position of the bone
         */
        getAbsolutePosition(tNode?: Nullable<TransformNode>): Vector3;
        /**
         * Copy the absolute position of the bone (world space) to the result param
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The vector3 to copy the absolute position to
         */
        getAbsolutePositionToRef(tNode: TransformNode, result: Vector3): void;
        /**
         * Compute the absolute matrices of this bone and its children
         */
        computeAbsoluteMatrices(): void;
        /**
         * Compute the absolute matrices of this bone and its children
         * @deprecated Please use computeAbsoluteMatrices instead
         */
        computeAbsoluteTransforms(): void;
        /**
         * Get the world direction from an axis that is in the local space of the bone
         * @param localAxis The local direction that is used to compute the world direction
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The world direction
         */
        getDirection(localAxis: Vector3, tNode?: Nullable<TransformNode>): Vector3;
        /**
         * Copy the world direction to a vector3 from an axis that is in the local space of the bone
         * @param localAxis The local direction that is used to compute the world direction
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The vector3 that the world direction will be copied to
         */
        getDirectionToRef(localAxis: Vector3, tNode: Nullable<TransformNode> | undefined, result: Vector3): void;
        /**
         * Get the euler rotation of the bone in local or world space
         * @param space The space that the rotation should be in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The euler rotation
         */
        getRotation(space?: Space, tNode?: Nullable<TransformNode>): Vector3;
        /**
         * Copy the euler rotation of the bone to a vector3.  The rotation can be in either local or world space
         * @param space The space that the rotation should be in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The vector3 that the rotation should be copied to
         */
        getRotationToRef(space: Space | undefined, tNode: Nullable<TransformNode> | undefined, result: Vector3): void;
        /**
         * Get the quaternion rotation of the bone in either local or world space
         * @param space The space that the rotation should be in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The quaternion rotation
         */
        getRotationQuaternion(space?: Space, tNode?: Nullable<TransformNode>): Quaternion;
        /**
         * Copy the quaternion rotation of the bone to a quaternion.  The rotation can be in either local or world space
         * @param space The space that the rotation should be in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The quaternion that the rotation should be copied to
         */
        getRotationQuaternionToRef(space: Space | undefined, tNode: Nullable<TransformNode> | undefined, result: Quaternion): void;
        /**
         * Get the rotation matrix of the bone in local or world space
         * @param space The space that the rotation should be in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The rotation matrix
         */
        getRotationMatrix(space: Space | undefined, tNode: TransformNode): Matrix;
        /**
         * Copy the rotation matrix of the bone to a matrix.  The rotation can be in either local or world space
         * @param space The space that the rotation should be in
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The quaternion that the rotation should be copied to
         */
        getRotationMatrixToRef(space: Space | undefined, tNode: TransformNode, result: Matrix): void;
        /**
         * Get the world position of a point that is in the local space of the bone
         * @param position The local position
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The world position
         */
        getAbsolutePositionFromLocal(position: Vector3, tNode?: Nullable<TransformNode>): Vector3;
        /**
         * Get the world position of a point that is in the local space of the bone and copy it to the result param
         * @param position The local position
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The vector3 that the world position should be copied to
         */
        getAbsolutePositionFromLocalToRef(position: Vector3, tNode: Nullable<TransformNode> | undefined, result: Vector3): void;
        /**
         * Get the local position of a point that is in world space
         * @param position The world position
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @returns The local position
         */
        getLocalPositionFromAbsolute(position: Vector3, tNode?: Nullable<TransformNode>): Vector3;
        /**
         * Get the local position of a point that is in world space and copy it to the result param
         * @param position The world position
         * @param tNode A TransformNode whose world matrix is to be applied to the calculated absolute matrix. In most cases, you'll want to pass the mesh associated with the skeleton from which this bone comes. Used only when space=Space.WORLD
         * @param result The vector3 that the local position should be copied to
         */
        getLocalPositionFromAbsoluteToRef(position: Vector3, tNode: Nullable<TransformNode> | undefined, result: Vector3): void;
        /**
         * Set the current local matrix as the restMatrix for this bone.
         */
        setCurrentPoseAsRest(): void;
    }


    /**
     * Class used to apply inverse kinematics to bones
     * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/bonesSkeletons#boneikcontroller
     */
    export class BoneIKController {
        private static _TmpVecs;
        private static _TmpQuat;
        private static _TmpMats;
        /**
         * Gets or sets the target TransformNode
         * Name kept as mesh for back compatibility
         */
        targetMesh: TransformNode;
        /** Gets or sets the mesh used as pole */
        poleTargetMesh: TransformNode;
        /**
         * Gets or sets the bone used as pole
         */
        poleTargetBone: Nullable<Bone>;
        /**
         * Gets or sets the target position
         */
        targetPosition: Vector3;
        /**
         * Gets or sets the pole target position
         */
        poleTargetPosition: Vector3;
        /**
         * Gets or sets the pole target local offset
         */
        poleTargetLocalOffset: Vector3;
        /**
         * Gets or sets the pole angle
         */
        poleAngle: number;
        /**
         * Gets or sets the TransformNode associated with the controller
         * Name kept as mesh for back compatibility
         */
        mesh: TransformNode;
        /**
         * The amount to slerp (spherical linear interpolation) to the target.  Set this to a value between 0 and 1 (a value of 1 disables slerp)
         */
        slerpAmount: number;
        private _bone1Quat;
        private _bone1Mat;
        private _bone2Ang;
        private _bone1;
        private _bone2;
        private _bone1Length;
        private _bone2Length;
        private _maxAngle;
        private _maxReach;
        private _rightHandedSystem;
        private _bendAxis;
        private _slerping;
        private _adjustRoll;
        private _notEnoughInformation;
        /**
         * Gets or sets maximum allowed angle
         */
        get maxAngle(): number;
        set maxAngle(value: number);
        /**
         * Creates a new BoneIKController
         * @param mesh defines the TransformNode to control
         * @param bone defines the bone to control. The bone needs to have a parent bone. It also needs to have a length greater than 0 or a children we can use to infer its length.
         * @param options defines options to set up the controller
         * @param options.targetMesh
         * @param options.poleTargetMesh
         * @param options.poleTargetBone
         * @param options.poleTargetLocalOffset
         * @param options.poleAngle
         * @param options.bendAxis
         * @param options.maxAngle
         * @param options.slerpAmount
         */
        constructor(mesh: TransformNode, bone: Bone, options?: {
            targetMesh?: TransformNode;
            poleTargetMesh?: TransformNode;
            poleTargetBone?: Bone;
            poleTargetLocalOffset?: Vector3;
            poleAngle?: number;
            bendAxis?: Vector3;
            maxAngle?: number;
            slerpAmount?: number;
        });
        private _setMaxAngle;
        /**
         * Force the controller to update the bones
         */
        update(): void;
        private _updateLinkedTransformRotation;
    }


    /**
     * Class used to make a bone look toward a point in space
     * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/bonesSkeletons#bonelookcontroller
     */
    export class BoneLookController {
        private static _TmpVecs;
        private static _TmpQuat;
        private static _TmpMats;
        /**
         * The target Vector3 that the bone will look at
         */
        target: Vector3;
        /**
         * The TransformNode that the bone is attached to
         * Name kept as mesh for back compatibility
         */
        mesh: TransformNode;
        /**
         * The bone that will be looking to the target
         */
        bone: Bone;
        /**
         * The up axis of the coordinate system that is used when the bone is rotated
         */
        upAxis: Vector3;
        /**
         * The space that the up axis is in - Space.BONE, Space.LOCAL (default), or Space.WORLD
         */
        upAxisSpace: Space;
        /**
         * Used to make an adjustment to the yaw of the bone
         */
        adjustYaw: number;
        /**
         * Used to make an adjustment to the pitch of the bone
         */
        adjustPitch: number;
        /**
         * Used to make an adjustment to the roll of the bone
         */
        adjustRoll: number;
        /**
         * The amount to slerp (spherical linear interpolation) to the target.  Set this to a value between 0 and 1 (a value of 1 disables slerp)
         */
        slerpAmount: number;
        private _minYaw;
        private _maxYaw;
        private _minPitch;
        private _maxPitch;
        private _minYawSin;
        private _minYawCos;
        private _maxYawSin;
        private _maxYawCos;
        private _midYawConstraint;
        private _minPitchTan;
        private _maxPitchTan;
        private _boneQuat;
        private _slerping;
        private _transformYawPitch;
        private _transformYawPitchInv;
        private _firstFrameSkipped;
        private _yawRange;
        private _fowardAxis;
        /**
         * Gets or sets the minimum yaw angle that the bone can look to
         */
        get minYaw(): number;
        set minYaw(value: number);
        /**
         * Gets or sets the maximum yaw angle that the bone can look to
         */
        get maxYaw(): number;
        set maxYaw(value: number);
        /**
         * Use the absolute value for yaw when checking the min/max constraints
         */
        useAbsoluteValueForYaw: boolean;
        /**
         * Gets or sets the minimum pitch angle that the bone can look to
         */
        get minPitch(): number;
        set minPitch(value: number);
        /**
         * Gets or sets the maximum pitch angle that the bone can look to
         */
        get maxPitch(): number;
        set maxPitch(value: number);
        /**
         * Create a BoneLookController
         * @param mesh the TransformNode that the bone belongs to
         * @param bone the bone that will be looking to the target
         * @param target the target Vector3 to look at
         * @param options optional settings:
         * * maxYaw: the maximum angle the bone will yaw to
         * * minYaw: the minimum angle the bone will yaw to
         * * maxPitch: the maximum angle the bone will pitch to
         * * minPitch: the minimum angle the bone will yaw to
         * * slerpAmount: set the between 0 and 1 to make the bone slerp to the target.
         * * upAxis: the up axis of the coordinate system
         * * upAxisSpace: the space that the up axis is in - Space.BONE, Space.LOCAL (default), or Space.WORLD.
         * * yawAxis: set yawAxis if the bone does not yaw on the y axis
         * * pitchAxis: set pitchAxis if the bone does not pitch on the x axis
         * * adjustYaw: used to make an adjustment to the yaw of the bone
         * * adjustPitch: used to make an adjustment to the pitch of the bone
         * * adjustRoll: used to make an adjustment to the roll of the bone
         * @param options.maxYaw
         * @param options.minYaw
         * @param options.maxPitch
         * @param options.minPitch
         * @param options.slerpAmount
         * @param options.upAxis
         * @param options.upAxisSpace
         * @param options.yawAxis
         * @param options.pitchAxis
         * @param options.adjustYaw
         * @param options.adjustPitch
         * @param options.adjustRoll
         **/
        constructor(mesh: TransformNode, bone: Bone, target: Vector3, options?: {
            maxYaw?: number;
            minYaw?: number;
            maxPitch?: number;
            minPitch?: number;
            slerpAmount?: number;
            upAxis?: Vector3;
            upAxisSpace?: Space;
            yawAxis?: Vector3;
            pitchAxis?: Vector3;
            adjustYaw?: number;
            adjustPitch?: number;
            adjustRoll?: number;
            useAbsoluteValueForYaw?: boolean;
        });
        /**
         * Update the bone to look at the target.  This should be called before the scene is rendered (use scene.registerBeforeRender())
         */
        update(): void;
        private _getAngleDiff;
        private _getAngleBetween;
        private _isAngleBetween;
        private _updateLinkedTransformRotation;
    }




    /**
     * Class used to handle skinning animations
     * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/bonesSkeletons
     */
    export class Skeleton implements IAnimatable {
        /** defines the skeleton name */
        name: string;
        /** defines the skeleton Id */
        id: string;
        /**
         * Defines the list of child bones
         */
        bones: Bone[];
        /**
         * Defines an estimate of the dimension of the skeleton at rest
         */
        dimensionsAtRest: Vector3;
        /**
         * Defines a boolean indicating if the root matrix is provided by meshes or by the current skeleton (this is the default value)
         */
        needInitialSkinMatrix: boolean;
        /**
         * Gets the list of animations attached to this skeleton
         */
        animations: Array<Animation>;
        private _scene;
        private _isDirty;
        private _transformMatrices;
        private _transformMatrixTexture;
        private _meshesWithPoseMatrix;
        private _animatables;
        private _identity;
        private _synchronizedWithMesh;
        private _currentRenderId;
        private _ranges;
        private _absoluteTransformIsDirty;
        private _canUseTextureForBones;
        private _uniqueId;
        /** @internal */
        _numBonesWithLinkedTransformNode: number;
        /** @internal */
        _hasWaitingData: Nullable<boolean>;
        /** @internal */
        _parentContainer: Nullable<AbstractScene>;
        /**
         * Specifies if the skeleton should be serialized
         */
        doNotSerialize: boolean;
        private _useTextureToStoreBoneMatrices;
        /**
         * Gets or sets a boolean indicating that bone matrices should be stored as a texture instead of using shader uniforms (default is true).
         * Please note that this option is not available if the hardware does not support it
         */
        get useTextureToStoreBoneMatrices(): boolean;
        set useTextureToStoreBoneMatrices(value: boolean);
        private _animationPropertiesOverride;
        /**
         * Gets or sets the animation properties override
         */
        get animationPropertiesOverride(): Nullable<AnimationPropertiesOverride>;
        set animationPropertiesOverride(value: Nullable<AnimationPropertiesOverride>);
        /**
         * List of inspectable custom properties (used by the Inspector)
         * @see https://doc.babylonjs.com/toolsAndResources/inspector#extensibility
         */
        inspectableCustomProperties: IInspectable[];
        /**
         * An observable triggered before computing the skeleton's matrices
         */
        onBeforeComputeObservable: Observable<Skeleton>;
        /**
         * Gets a boolean indicating that the skeleton effectively stores matrices into a texture
         */
        get isUsingTextureForMatrices(): boolean;
        /**
         * Gets the unique ID of this skeleton
         */
        get uniqueId(): number;
        /**
         * Creates a new skeleton
         * @param name defines the skeleton name
         * @param id defines the skeleton Id
         * @param scene defines the hosting scene
         */
        constructor(
        /** defines the skeleton name */
        name: string, 
        /** defines the skeleton Id */
        id: string, scene: Scene);
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Returns an array containing the root bones
         * @returns an array containing the root bones
         */
        getChildren(): Array<Bone>;
        /**
         * Gets the list of transform matrices to send to shaders (one matrix per bone)
         * @param mesh defines the mesh to use to get the root matrix (if needInitialSkinMatrix === true)
         * @returns a Float32Array containing matrices data
         */
        getTransformMatrices(mesh: AbstractMesh): Float32Array;
        /**
         * Gets the list of transform matrices to send to shaders inside a texture (one matrix per bone)
         * @param mesh defines the mesh to use to get the root matrix (if needInitialSkinMatrix === true)
         * @returns a raw texture containing the data
         */
        getTransformMatrixTexture(mesh: AbstractMesh): Nullable<RawTexture>;
        /**
         * Gets the current hosting scene
         * @returns a scene object
         */
        getScene(): Scene;
        /**
         * Gets a string representing the current skeleton data
         * @param fullDetails defines a boolean indicating if we want a verbose version
         * @returns a string representing the current skeleton data
         */
        toString(fullDetails?: boolean): string;
        /**
         * Get bone's index searching by name
         * @param name defines bone's name to search for
         * @returns the indice of the bone. Returns -1 if not found
         */
        getBoneIndexByName(name: string): number;
        /**
         * Create a new animation range
         * @param name defines the name of the range
         * @param from defines the start key
         * @param to defines the end key
         */
        createAnimationRange(name: string, from: number, to: number): void;
        /**
         * Delete a specific animation range
         * @param name defines the name of the range
         * @param deleteFrames defines if frames must be removed as well
         */
        deleteAnimationRange(name: string, deleteFrames?: boolean): void;
        /**
         * Gets a specific animation range
         * @param name defines the name of the range to look for
         * @returns the requested animation range or null if not found
         */
        getAnimationRange(name: string): Nullable<AnimationRange>;
        /**
         * Gets the list of all animation ranges defined on this skeleton
         * @returns an array
         */
        getAnimationRanges(): Nullable<AnimationRange>[];
        /**
         * Copy animation range from a source skeleton.
         * This is not for a complete retargeting, only between very similar skeleton's with only possible bone length differences
         * @param source defines the source skeleton
         * @param name defines the name of the range to copy
         * @param rescaleAsRequired defines if rescaling must be applied if required
         * @returns true if operation was successful
         */
        copyAnimationRange(source: Skeleton, name: string, rescaleAsRequired?: boolean): boolean;
        /**
         * Forces the skeleton to go to rest pose
         */
        returnToRest(): void;
        private _getHighestAnimationFrame;
        /**
         * Begin a specific animation range
         * @param name defines the name of the range to start
         * @param loop defines if looping must be turned on (false by default)
         * @param speedRatio defines the speed ratio to apply (1 by default)
         * @param onAnimationEnd defines a callback which will be called when animation will end
         * @returns a new animatable
         */
        beginAnimation(name: string, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Convert the keyframes for a range of animation on a skeleton to be relative to a given reference frame.
         * @param skeleton defines the Skeleton containing the animation range to convert
         * @param referenceFrame defines the frame that keyframes in the range will be relative to
         * @param range defines the name of the AnimationRange belonging to the Skeleton to convert
         * @returns the original skeleton
         */
        static MakeAnimationAdditive(skeleton: Skeleton, referenceFrame: number | undefined, range: string): Nullable<Skeleton>;
        /** @internal */
        _markAsDirty(): void;
        /**
         * @internal
         */
        _registerMeshWithPoseMatrix(mesh: AbstractMesh): void;
        /**
         * @internal
         */
        _unregisterMeshWithPoseMatrix(mesh: AbstractMesh): void;
        private _computeTransformMatrices;
        /**
         * Build all resources required to render a skeleton
         * @param dontCheckFrameId defines a boolean indicating if prepare should be run without checking first the current frame id (default: false)
         */
        prepare(dontCheckFrameId?: boolean): void;
        /**
         * Gets the list of animatables currently running for this skeleton
         * @returns an array of animatables
         */
        getAnimatables(): IAnimatable[];
        /**
         * Clone the current skeleton
         * @param name defines the name of the new skeleton
         * @param id defines the id of the new skeleton
         * @returns the new skeleton
         */
        clone(name: string, id?: string): Skeleton;
        /**
         * Enable animation blending for this skeleton
         * @param blendingSpeed defines the blending speed to apply
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#animation-blending
         */
        enableBlending(blendingSpeed?: number): void;
        /**
         * Releases all resources associated with the current skeleton
         */
        dispose(): void;
        /**
         * Serialize the skeleton in a JSON object
         * @returns a JSON object
         */
        serialize(): any;
        /**
         * Creates a new skeleton from serialized data
         * @param parsedSkeleton defines the serialized data
         * @param scene defines the hosting scene
         * @returns a new skeleton
         */
        static Parse(parsedSkeleton: any, scene: Scene): Skeleton;
        /**
         * Compute all node absolute matrices
         * @param forceUpdate defines if computation must be done even if cache is up to date
         */
        computeAbsoluteMatrices(forceUpdate?: boolean): void;
        /**
         * Compute all node absolute matrices
         * @param forceUpdate defines if computation must be done even if cache is up to date
         * @deprecated Please use computeAbsoluteMatrices instead
         */
        computeAbsoluteTransforms(forceUpdate?: boolean): void;
        /**
         * Gets the root pose matrix
         * @returns a matrix
         */
        getPoseMatrix(): Nullable<Matrix>;
        /**
         * Sorts bones per internal index
         */
        sortBones(): void;
        private _sortBones;
        /**
         * Set the current local matrix as the restPose for all bones in the skeleton.
         */
        setCurrentPoseAsRest(): void;
    }


        interface VertexBuffer {
            /**
             * Gets the effective byte stride, that is the byte stride of the buffer that is actually sent to the GPU.
             * It could be different from VertexBuffer.byteStride if a new buffer must be created under the hood because of the forceVertexBufferStrideMultiple4Bytes engine flag.
             */
            effectiveByteStride: number;
            /**
             * Gets the effective byte offset, that is the byte offset of the buffer that is actually sent to the GPU.
             * It could be different from VertexBuffer.byteOffset if a new buffer must be created under the hood because of the forceVertexBufferStrideMultiple4Bytes engine flag.
             */
            effectiveByteOffset: number;
            /**
             * Gets the effective buffer, that is the buffer that is actually sent to the GPU.
             * It could be different from VertexBuffer.getBuffer() if a new buffer must be created under the hood because of the forceVertexBufferStrideMultiple4Bytes engine flag.
             */
            effectiveBuffer: Nullable<DataBuffer>;
            /** @internal */
            _alignBuffer(): void;
            /** @internal */
            _alignedBuffer?: Buffer;
        }


    /**
     * Class used to store data that will be store in GPU memory
     */
    export class Buffer {
        private _engine;
        private _buffer;
        /** @internal */
        _data: Nullable<DataArray>;
        private _updatable;
        private _instanced;
        private _divisor;
        private _isAlreadyOwned;
        private _isDisposed;
        private _label?;
        /**
         * Gets a boolean indicating if the Buffer is disposed
         */
        get isDisposed(): boolean;
        /**
         * Gets the byte stride.
         */
        readonly byteStride: number;
        /**
         * Constructor
         * @param engine the engine
         * @param data the data to use for this buffer
         * @param updatable whether the data is updatable
         * @param stride the stride (optional)
         * @param postponeInternalCreation whether to postpone creating the internal WebGL buffer (optional)
         * @param instanced whether the buffer is instanced (optional)
         * @param useBytes set to true if the stride in in bytes (optional)
         * @param divisor sets an optional divisor for instances (1 by default)
         * @param label defines the label of the buffer (for debug purpose)
         */
        constructor(engine: ThinEngine, data: DataArray | DataBuffer, updatable: boolean, stride?: number, postponeInternalCreation?: boolean, instanced?: boolean, useBytes?: boolean, divisor?: number, label?: string);
        /**
         * Create a new VertexBuffer based on the current buffer
         * @param kind defines the vertex buffer kind (position, normal, etc.)
         * @param offset defines offset in the buffer (0 by default)
         * @param size defines the size in floats of attributes (position is 3 for instance)
         * @param stride defines the stride size in floats in the buffer (the offset to apply to reach next value when data is interleaved)
         * @param instanced defines if the vertex buffer contains indexed data
         * @param useBytes defines if the offset and stride are in bytes     *
         * @param divisor sets an optional divisor for instances (1 by default)
         * @returns the new vertex buffer
         */
        createVertexBuffer(kind: string, offset: number, size: number, stride?: number, instanced?: boolean, useBytes?: boolean, divisor?: number): VertexBuffer;
        /**
         * Gets a boolean indicating if the Buffer is updatable?
         * @returns true if the buffer is updatable
         */
        isUpdatable(): boolean;
        /**
         * Gets current buffer's data
         * @returns a DataArray or null
         */
        getData(): Nullable<DataArray>;
        /**
         * Gets underlying native buffer
         * @returns underlying native buffer
         */
        getBuffer(): Nullable<DataBuffer>;
        /**
         * Gets the stride in float32 units (i.e. byte stride / 4).
         * May not be an integer if the byte stride is not divisible by 4.
         * @returns the stride in float32 units
         * @deprecated Please use byteStride instead.
         */
        getStrideSize(): number;
        /**
         * Store data into the buffer. Creates the buffer if not used already.
         * If the buffer was already used, it will be updated only if it is updatable, otherwise it will do nothing.
         * @param data defines the data to store
         */
        create(data?: Nullable<DataArray>): void;
        /** @internal */
        _rebuild(): void;
        /**
         * Update current buffer data
         * @param data defines the data to store
         */
        update(data: DataArray): void;
        /**
         * Updates the data directly.
         * @param data the new data
         * @param offset the new offset
         * @param vertexCount the vertex count (optional)
         * @param useBytes set to true if the offset is in bytes
         */
        updateDirectly(data: DataArray, offset: number, vertexCount?: number, useBytes?: boolean): void;
        /** @internal */
        _increaseReferences(): void;
        /**
         * Release all resources
         */
        dispose(): void;
    }
    /**
     * Options to be used when creating a vertex buffer
     */
    export interface IVertexBufferOptions {
        /**
         * whether the data is updatable (default: false)
         */
        updatable?: boolean;
        /**
         * whether to postpone creating the internal WebGL buffer (default: false)
         */
        postponeInternalCreation?: boolean;
        /**
         * the stride (will be automatically computed from the kind parameter if not specified)
         */
        stride?: number;
        /**
         * whether the buffer is instanced (default: false)
         */
        instanced?: boolean;
        /**
         * the offset of the data (default: 0)
         */
        offset?: number;
        /**
         * the number of components (will be automatically computed from the kind parameter if not specified)
         */
        size?: number;
        /**
         * the type of the component (will be deduce from the data parameter if not specified)
         */
        type?: number;
        /**
         * whether the data contains normalized data (default: false)
         */
        normalized?: boolean;
        /**
         * set to true if stride and offset are in bytes (default: false)
         */
        useBytes?: boolean;
        /**
         * defines the instance divisor to use (default: 1, only used if instanced is true)
         */
        divisor?: number;
        /**
         * defines if the buffer should be released when the vertex buffer is disposed (default: false)
         */
        takeBufferOwnership?: boolean;
        /**
         * label to use for this vertex buffer (debugging purpose)
         */
        label?: string;
    }
    /**
     * Specialized buffer used to store vertex data
     */
    export class VertexBuffer {
        private static _Counter;
        /** @internal */
        _buffer: Buffer;
        /** @internal */
        _validOffsetRange: boolean;
        private _kind;
        private _size;
        /** @internal */
        _ownsBuffer: boolean;
        private _instanced;
        private _instanceDivisor;
        /** @internal */
        _isDisposed: boolean;
        /** @internal */
        _label?: string;
        /**
         * The byte type.
         */
        static readonly BYTE = 5120;
        /**
         * The unsigned byte type.
         */
        static readonly UNSIGNED_BYTE = 5121;
        /**
         * The short type.
         */
        static readonly SHORT = 5122;
        /**
         * The unsigned short type.
         */
        static readonly UNSIGNED_SHORT = 5123;
        /**
         * The integer type.
         */
        static readonly INT = 5124;
        /**
         * The unsigned integer type.
         */
        static readonly UNSIGNED_INT = 5125;
        /**
         * The float type.
         */
        static readonly FLOAT = 5126;
        /**
         * Gets a boolean indicating if the Buffer is disposed
         */
        get isDisposed(): boolean;
        /**
         * Gets or sets the instance divisor when in instanced mode
         */
        get instanceDivisor(): number;
        set instanceDivisor(value: number);
        /**
         * Gets the byte stride.
         */
        readonly byteStride: number;
        /**
         * Gets the byte offset.
         */
        readonly byteOffset: number;
        /**
         * Gets whether integer data values should be normalized into a certain range when being casted to a float.
         */
        readonly normalized: boolean;
        /**
         * Gets the data type of each component in the array.
         */
        readonly type: number;
        /**
         * Gets the unique id of this vertex buffer
         */
        readonly uniqueId: number;
        /**
         * Gets a hash code representing the format (type, normalized, size, instanced, stride) of this buffer
         * All buffers with the same format will have the same hash code
         */
        readonly hashCode: number;
        /**
         * Gets the engine associated with the buffer
         */
        readonly engine: ThinEngine;
        /**
         * Gets the max possible amount of vertices stored within the current vertex buffer.
         * We do not have the end offset or count so this will be too big for concatenated vertex buffers.
         * @internal
         */
        get _maxVerticesCount(): number;
        /**
         * Constructor
         * @param engine the engine
         * @param data the data to use for this vertex buffer
         * @param kind the vertex buffer kind
         * @param updatable whether the data is updatable
         * @param postponeInternalCreation whether to postpone creating the internal WebGL buffer (optional)
         * @param stride the stride (optional)
         * @param instanced whether the buffer is instanced (optional)
         * @param offset the offset of the data (optional)
         * @param size the number of components (optional)
         * @param type the type of the component (optional)
         * @param normalized whether the data contains normalized data (optional)
         * @param useBytes set to true if stride and offset are in bytes (optional)
         * @param divisor defines the instance divisor to use (1 by default)
         * @param takeBufferOwnership defines if the buffer should be released when the vertex buffer is disposed
         */
        constructor(engine: ThinEngine, data: DataArray | Buffer | DataBuffer, kind: string, updatable: boolean, postponeInternalCreation?: boolean, stride?: number, instanced?: boolean, offset?: number, size?: number, type?: number, normalized?: boolean, useBytes?: boolean, divisor?: number, takeBufferOwnership?: boolean);
        /**
         * Constructor
         * @param engine the engine
         * @param data the data to use for this vertex buffer
         * @param kind the vertex buffer kind
         * @param options defines the rest of the options used to create the buffer
         */
        constructor(engine: ThinEngine, data: DataArray | Buffer | DataBuffer, kind: string, options?: IVertexBufferOptions);
        private _computeHashCode;
        /** @internal */
        _rebuild(): void;
        /**
         * Returns the kind of the VertexBuffer (string)
         * @returns a string
         */
        getKind(): string;
        /**
         * Gets a boolean indicating if the VertexBuffer is updatable?
         * @returns true if the buffer is updatable
         */
        isUpdatable(): boolean;
        /**
         * Gets current buffer's data
         * @returns a DataArray or null
         */
        getData(): Nullable<DataArray>;
        /**
         * Gets current buffer's data as a float array. Float data is constructed if the vertex buffer data cannot be returned directly.
         * @param totalVertices number of vertices in the buffer to take into account
         * @param forceCopy defines a boolean indicating that the returned array must be cloned upon returning it
         * @returns a float array containing vertex data
         */
        getFloatData(totalVertices: number, forceCopy?: boolean): Nullable<FloatArray>;
        /**
         * Gets underlying native buffer
         * @returns underlying native buffer
         */
        getBuffer(): Nullable<DataBuffer>;
        /**
         * Gets the Buffer instance that wraps the native GPU buffer
         * @returns the wrapper buffer
         */
        getWrapperBuffer(): Buffer;
        /**
         * Gets the stride in float32 units (i.e. byte stride / 4).
         * May not be an integer if the byte stride is not divisible by 4.
         * @returns the stride in float32 units
         * @deprecated Please use byteStride instead.
         */
        getStrideSize(): number;
        /**
         * Returns the offset as a multiple of the type byte length.
         * @returns the offset in bytes
         * @deprecated Please use byteOffset instead.
         */
        getOffset(): number;
        /**
         * Returns the number of components or the byte size per vertex attribute
         * @param sizeInBytes If true, returns the size in bytes or else the size in number of components of the vertex attribute (default: false)
         * @returns the number of components
         */
        getSize(sizeInBytes?: boolean): number;
        /**
         * Gets a boolean indicating is the internal buffer of the VertexBuffer is instanced
         * @returns true if this buffer is instanced
         */
        getIsInstanced(): boolean;
        /**
         * Returns the instancing divisor, zero for non-instanced (integer).
         * @returns a number
         */
        getInstanceDivisor(): number;
        /**
         * Store data into the buffer. If the buffer was already used it will be either recreated or updated depending on isUpdatable property
         * @param data defines the data to store
         */
        create(data?: DataArray): void;
        /**
         * Updates the underlying buffer according to the passed numeric array or Float32Array.
         * This function will create a new buffer if the current one is not updatable
         * @param data defines the data to store
         */
        update(data: DataArray): void;
        /**
         * Updates directly the underlying WebGLBuffer according to the passed numeric array or Float32Array.
         * Returns the directly updated WebGLBuffer.
         * @param data the new data
         * @param offset the new offset
         * @param useBytes set to true if the offset is in bytes
         */
        updateDirectly(data: DataArray, offset: number, useBytes?: boolean): void;
        /**
         * Disposes the VertexBuffer and the underlying WebGLBuffer.
         */
        dispose(): void;
        /**
         * Enumerates each value of this vertex buffer as numbers.
         * @param count the number of values to enumerate
         * @param callback the callback function called for each value
         */
        forEach(count: number, callback: (value: number, index: number) => void): void;
        /**
         * Positions
         */
        static readonly PositionKind = "position";
        /**
         * Normals
         */
        static readonly NormalKind = "normal";
        /**
         * Tangents
         */
        static readonly TangentKind = "tangent";
        /**
         * Texture coordinates
         */
        static readonly UVKind = "uv";
        /**
         * Texture coordinates 2
         */
        static readonly UV2Kind = "uv2";
        /**
         * Texture coordinates 3
         */
        static readonly UV3Kind = "uv3";
        /**
         * Texture coordinates 4
         */
        static readonly UV4Kind = "uv4";
        /**
         * Texture coordinates 5
         */
        static readonly UV5Kind = "uv5";
        /**
         * Texture coordinates 6
         */
        static readonly UV6Kind = "uv6";
        /**
         * Colors
         */
        static readonly ColorKind = "color";
        /**
         * Instance Colors
         */
        static readonly ColorInstanceKind = "instanceColor";
        /**
         * Matrix indices (for bones)
         */
        static readonly MatricesIndicesKind = "matricesIndices";
        /**
         * Matrix weights (for bones)
         */
        static readonly MatricesWeightsKind = "matricesWeights";
        /**
         * Additional matrix indices (for bones)
         */
        static readonly MatricesIndicesExtraKind = "matricesIndicesExtra";
        /**
         * Additional matrix weights (for bones)
         */
        static readonly MatricesWeightsExtraKind = "matricesWeightsExtra";
        /**
         * Deduces the stride given a kind.
         * @param kind The kind string to deduce
         * @returns The deduced stride
         */
        static DeduceStride(kind: string): number;
        /**
         * Gets the vertex buffer type of the given data array.
         * @param data the data array
         * @returns the vertex buffer type
         */
        static GetDataType(data: DataArray): number;
        /**
         * Gets the byte length of the given type.
         * @param type the type
         * @returns the number of bytes
         */
        static GetTypeByteLength(type: number): number;
        /**
         * Enumerates each value of the given parameters as numbers.
         * @param data the data to enumerate
         * @param byteOffset the byte offset of the data
         * @param byteStride the byte stride of the data
         * @param componentCount the number of components per element
         * @param componentType the type of the component
         * @param count the number of values to enumerate
         * @param normalized whether the data is normalized
         * @param callback the callback function called for each value
         */
        static ForEach(data: DataArray, byteOffset: number, byteStride: number, componentCount: number, componentType: number, count: number, normalized: boolean, callback: (value: number, index: number) => void): void;
        private static _GetFloatValue;
        /**
         * Gets the given data array as a float array. Float data is constructed if the data array cannot be returned directly.
         * @param data the input data array
         * @param size the number of components
         * @param type the component type
         * @param byteOffset the byte offset of the data
         * @param byteStride the byte stride of the data
         * @param normalized whether the data is normalized
         * @param totalVertices number of vertices in the buffer to take into account
         * @param forceCopy defines a boolean indicating that the returned array must be cloned upon returning it
         * @returns a float array containing vertex data
         */
        static GetFloatData(data: DataArray, size: number, type: number, byteOffset: number, byteStride: number, normalized: boolean, totalVertices: number, forceCopy?: boolean): FloatArray;
    }


    /**
     * Class used to store gfx data (like WebGLBuffer)
     */
    export class DataBuffer {
        private static _Counter;
        /**
         * Gets or sets the number of objects referencing this buffer
         */
        references: number;
        /** Gets or sets the size of the underlying buffer */
        capacity: number;
        /**
         * Gets or sets a boolean indicating if the buffer contains 32bits indices
         */
        is32Bits: boolean;
        /**
         * Gets the underlying buffer
         */
        get underlyingResource(): any;
        /**
         * Gets the unique id of this buffer
         */
        readonly uniqueId: number;
        /**
         * Constructs the buffer
         */
        constructor();
    }




    /**
     * This class is a small wrapper around a native buffer that can be read and/or written
     */
    export class StorageBuffer {
        private _engine;
        private _buffer;
        private _bufferSize;
        private _creationFlags;
        private _label?;
        /**
         * Creates a new storage buffer instance
         * @param engine The engine the buffer will be created inside
         * @param size The size of the buffer in bytes
         * @param creationFlags flags to use when creating the buffer (see Constants.BUFFER_CREATIONFLAG_XXX). The BUFFER_CREATIONFLAG_STORAGE flag will be automatically added.
         * @param label defines the label of the buffer (for debug purpose)
         */
        constructor(engine: ThinEngine, size: number, creationFlags?: number, label?: string);
        private _create;
        /** @internal */
        _rebuild(): void;
        /**
         * Gets underlying native buffer
         * @returns underlying native buffer
         */
        getBuffer(): DataBuffer;
        /**
         * Updates the storage buffer
         * @param data the data used to update the storage buffer
         * @param byteOffset the byte offset of the data (optional)
         * @param byteLength the byte length of the data (optional)
         */
        update(data: DataArray, byteOffset?: number, byteLength?: number): void;
        /**
         * Reads data from the storage buffer
         * @param offset The offset in the storage buffer to start reading from (default: 0)
         * @param size  The number of bytes to read from the storage buffer (default: capacity of the buffer)
         * @param buffer The buffer to write the data we have read from the storage buffer to (optional)
         * @param noDelay If true, a call to flushFramebuffer will be issued so that the data can be read back immediately. This can speed up data retrieval, at the cost of a small perf penalty (default: false).
         * @returns If not undefined, returns the (promise) buffer (as provided by the 4th parameter) filled with the data, else it returns a (promise) Uint8Array with the data read from the storage buffer
         */
        read(offset?: number, size?: number, buffer?: ArrayBufferView, noDelay?: boolean): Promise<ArrayBufferView>;
        /**
         * Disposes the storage buffer
         */
        dispose(): void;
    }


    /**
     * This represents an orbital type of camera.
     *
     * This camera always points towards a given target position and can be rotated around that target with the target as the centre of rotation. It can be controlled with cursors and mouse, or with touch events.
     * Think of this camera as one orbiting its target position, or more imaginatively as a spy satellite orbiting the earth. Its position relative to the target (earth) can be set by three parameters, alpha (radians) the longitudinal rotation, beta (radians) the latitudinal rotation and radius the distance from the target position.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#arc-rotate-camera
     */
    export class ArcRotateCamera extends TargetCamera {
        /**
         * Defines the rotation angle of the camera along the longitudinal axis.
         */
        alpha: number;
        /**
         * Defines the rotation angle of the camera along the latitudinal axis.
         */
        beta: number;
        /**
         * Defines the radius of the camera from it s target point.
         */
        radius: number;
        /**
         * Defines an override value to use as the parameter to setTarget.
         * This allows the parameter to be specified when animating the target (e.g. using FramingBehavior).
         */
        overrideCloneAlphaBetaRadius: Nullable<boolean>;
        protected _target: Vector3;
        protected _targetHost: Nullable<TransformNode>;
        /**
         * Defines the target point of the camera.
         * The camera looks towards it from the radius distance.
         */
        get target(): Vector3;
        set target(value: Vector3);
        /**
         * Defines the target transform node of the camera.
         * The camera looks towards it from the radius distance.
         * Please note that setting a target host will disable panning.
         */
        get targetHost(): Nullable<TransformNode>;
        set targetHost(value: Nullable<TransformNode>);
        /**
         * Return the current target position of the camera. This value is expressed in local space.
         * @returns the target position
         */
        getTarget(): Vector3;
        /**
         * Define the current local position of the camera in the scene
         */
        get position(): Vector3;
        set position(newPosition: Vector3);
        protected _upToYMatrix: Matrix;
        protected _yToUpMatrix: Matrix;
        /**
         * The vector the camera should consider as up. (default is Vector3(0, 1, 0) as returned by Vector3.Up())
         * Setting this will copy the given vector to the camera's upVector, and set rotation matrices to and from Y up.
         * DO NOT set the up vector using copyFrom or copyFromFloats, as this bypasses setting the above matrices.
         */
        set upVector(vec: Vector3);
        get upVector(): Vector3;
        /**
         * Sets the Y-up to camera up-vector rotation matrix, and the up-vector to Y-up rotation matrix.
         */
        setMatUp(): void;
        /**
         * Current inertia value on the longitudinal axis.
         * The bigger this number the longer it will take for the camera to stop.
         */
        inertialAlphaOffset: number;
        /**
         * Current inertia value on the latitudinal axis.
         * The bigger this number the longer it will take for the camera to stop.
         */
        inertialBetaOffset: number;
        /**
         * Current inertia value on the radius axis.
         * The bigger this number the longer it will take for the camera to stop.
         */
        inertialRadiusOffset: number;
        /**
         * Minimum allowed angle on the longitudinal axis.
         * This can help limiting how the Camera is able to move in the scene.
         */
        lowerAlphaLimit: Nullable<number>;
        /**
         * Maximum allowed angle on the longitudinal axis.
         * This can help limiting how the Camera is able to move in the scene.
         */
        upperAlphaLimit: Nullable<number>;
        /**
         * Minimum allowed angle on the latitudinal axis.
         * This can help limiting how the Camera is able to move in the scene.
         */
        lowerBetaLimit: Nullable<number>;
        /**
         * Maximum allowed angle on the latitudinal axis.
         * This can help limiting how the Camera is able to move in the scene.
         */
        upperBetaLimit: Nullable<number>;
        /**
         * Minimum allowed distance of the camera to the target (The camera can not get closer).
         * This can help limiting how the Camera is able to move in the scene.
         */
        lowerRadiusLimit: Nullable<number>;
        /**
         * Maximum allowed distance of the camera to the target (The camera can not get further).
         * This can help limiting how the Camera is able to move in the scene.
         */
        upperRadiusLimit: Nullable<number>;
        /**
         * Defines the current inertia value used during panning of the camera along the X axis.
         */
        inertialPanningX: number;
        /**
         * Defines the current inertia value used during panning of the camera along the Y axis.
         */
        inertialPanningY: number;
        /**
         * Defines the distance used to consider the camera in pan mode vs pinch/zoom.
         * Basically if your fingers moves away from more than this distance you will be considered
         * in pinch mode.
         */
        pinchToPanMaxDistance: number;
        /**
         * Defines the maximum distance the camera can pan.
         * This could help keeping the camera always in your scene.
         */
        panningDistanceLimit: Nullable<number>;
        /**
         * Defines the target of the camera before panning.
         */
        panningOriginTarget: Vector3;
        /**
         * Defines the value of the inertia used during panning.
         * 0 would mean stop inertia and one would mean no deceleration at all.
         */
        panningInertia: number;
        /**
         * Gets or Set the pointer angular sensibility  along the X axis or how fast is the camera rotating.
         */
        get angularSensibilityX(): number;
        set angularSensibilityX(value: number);
        /**
         * Gets or Set the pointer angular sensibility along the Y axis or how fast is the camera rotating.
         */
        get angularSensibilityY(): number;
        set angularSensibilityY(value: number);
        /**
         * Gets or Set the pointer pinch precision or how fast is the camera zooming.
         */
        get pinchPrecision(): number;
        set pinchPrecision(value: number);
        /**
         * Gets or Set the pointer pinch delta percentage or how fast is the camera zooming.
         * It will be used instead of pinchDeltaPrecision if different from 0.
         * It defines the percentage of current camera.radius to use as delta when pinch zoom is used.
         */
        get pinchDeltaPercentage(): number;
        set pinchDeltaPercentage(value: number);
        /**
         * Gets or Set the pointer use natural pinch zoom to override the pinch precision
         * and pinch delta percentage.
         * When useNaturalPinchZoom is true, multi touch zoom will zoom in such
         * that any object in the plane at the camera's target point will scale
         * perfectly with finger motion.
         */
        get useNaturalPinchZoom(): boolean;
        set useNaturalPinchZoom(value: boolean);
        /**
         * Gets or Set the pointer panning sensibility or how fast is the camera moving.
         */
        get panningSensibility(): number;
        set panningSensibility(value: number);
        /**
         * Gets or Set the list of keyboard keys used to control beta angle in a positive direction.
         */
        get keysUp(): number[];
        set keysUp(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control beta angle in a negative direction.
         */
        get keysDown(): number[];
        set keysDown(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control alpha angle in a negative direction.
         */
        get keysLeft(): number[];
        set keysLeft(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control alpha angle in a positive direction.
         */
        get keysRight(): number[];
        set keysRight(value: number[]);
        /**
         * Gets or Set the mouse wheel precision or how fast is the camera zooming.
         */
        get wheelPrecision(): number;
        set wheelPrecision(value: number);
        /**
         * Gets or Set the boolean value that controls whether or not the mouse wheel
         * zooms to the location of the mouse pointer or not.  The default is false.
         */
        get zoomToMouseLocation(): boolean;
        set zoomToMouseLocation(value: boolean);
        /**
         * Gets or Set the mouse wheel delta percentage or how fast is the camera zooming.
         * It will be used instead of pinchDeltaPrecision if different from 0.
         * It defines the percentage of current camera.radius to use as delta when pinch zoom is used.
         */
        get wheelDeltaPercentage(): number;
        set wheelDeltaPercentage(value: number);
        /**
         * Defines how much the radius should be scaled while zooming on a particular mesh (through the zoomOn function)
         */
        zoomOnFactor: number;
        /**
         * Defines a screen offset for the camera position.
         */
        targetScreenOffset: Vector2;
        /**
         * Allows the camera to be completely reversed.
         * If false the camera can not arrive upside down.
         */
        allowUpsideDown: boolean;
        /**
         * Define if double tap/click is used to restore the previously saved state of the camera.
         */
        useInputToRestoreState: boolean;
        /** @internal */
        _viewMatrix: Matrix;
        /** @internal */
        _useCtrlForPanning: boolean;
        /** @internal */
        _panningMouseButton: number;
        /**
         * Defines the input associated to the camera.
         */
        inputs: ArcRotateCameraInputsManager;
        /** @internal */
        _reset: () => void;
        /**
         * Defines the allowed panning axis.
         */
        panningAxis: Vector3;
        protected _transformedDirection: Vector3;
        /**
         * Defines if camera will eliminate transform on y axis.
         */
        mapPanning: boolean;
        private _bouncingBehavior;
        /**
         * Gets the bouncing behavior of the camera if it has been enabled.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#bouncing-behavior
         */
        get bouncingBehavior(): Nullable<BouncingBehavior>;
        /**
         * Defines if the bouncing behavior of the camera is enabled on the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#bouncing-behavior
         */
        get useBouncingBehavior(): boolean;
        set useBouncingBehavior(value: boolean);
        private _framingBehavior;
        /**
         * Gets the framing behavior of the camera if it has been enabled.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#framing-behavior
         */
        get framingBehavior(): Nullable<FramingBehavior>;
        /**
         * Defines if the framing behavior of the camera is enabled on the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#framing-behavior
         */
        get useFramingBehavior(): boolean;
        set useFramingBehavior(value: boolean);
        private _autoRotationBehavior;
        /**
         * Gets the auto rotation behavior of the camera if it has been enabled.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#autorotation-behavior
         */
        get autoRotationBehavior(): Nullable<AutoRotationBehavior>;
        /**
         * Defines if the auto rotation behavior of the camera is enabled on the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/behaviors/cameraBehaviors#autorotation-behavior
         */
        get useAutoRotationBehavior(): boolean;
        set useAutoRotationBehavior(value: boolean);
        /**
         * Observable triggered when the transform node target has been changed on the camera.
         */
        onMeshTargetChangedObservable: Observable<Nullable<TransformNode>>;
        /**
         * Event raised when the camera is colliding with a mesh.
         */
        onCollide: (collidedMesh: AbstractMesh) => void;
        /**
         * Defines whether the camera should check collision with the objects oh the scene.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_collisions#how-can-i-do-this-
         */
        checkCollisions: boolean;
        /**
         * Defines the collision radius of the camera.
         * This simulates a sphere around the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_collisions#arcrotatecamera
         */
        collisionRadius: Vector3;
        protected _collider: Collider;
        protected _previousPosition: Vector3;
        protected _collisionVelocity: Vector3;
        protected _newPosition: Vector3;
        protected _previousAlpha: number;
        protected _previousBeta: number;
        protected _previousRadius: number;
        protected _collisionTriggered: boolean;
        protected _targetBoundingCenter: Nullable<Vector3>;
        private _computationVector;
        /**
         * Instantiates a new ArcRotateCamera in a given scene
         * @param name Defines the name of the camera
         * @param alpha Defines the camera rotation along the longitudinal axis
         * @param beta Defines the camera rotation along the latitudinal axis
         * @param radius Defines the camera distance from its target
         * @param target Defines the camera target
         * @param scene Defines the scene the camera belongs to
         * @param setActiveOnSceneIfNoneActive Defines whether the camera should be marked as active if not other active cameras have been defined
         */
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, scene?: Scene, setActiveOnSceneIfNoneActive?: boolean);
        /** @internal */
        _initCache(): void;
        /**
         * @internal
         */
        _updateCache(ignoreParentClass?: boolean): void;
        protected _getTargetPosition(): Vector3;
        private _storedAlpha;
        private _storedBeta;
        private _storedRadius;
        private _storedTarget;
        private _storedTargetScreenOffset;
        /**
         * Stores the current state of the camera (alpha, beta, radius and target)
         * @returns the camera itself
         */
        storeState(): Camera;
        /**
         * @internal
         * Restored camera state. You must call storeState() first
         */
        _restoreStateValues(): boolean;
        /** @internal */
        _isSynchronizedViewMatrix(): boolean;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param ignored defines an ignored parameter kept for backward compatibility.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(ignored: any, noPreventDefault?: boolean): void;
        /**
         * Attached controls to the current camera.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         * @param useCtrlForPanning  Defines whether ctrl is used for panning within the controls
         */
        attachControl(noPreventDefault: boolean, useCtrlForPanning: boolean): void;
        /**
         * Attached controls to the current camera.
         * @param ignored defines an ignored parameter kept for backward compatibility.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         * @param useCtrlForPanning  Defines whether ctrl is used for panning within the controls
         */
        attachControl(ignored: any, noPreventDefault: boolean, useCtrlForPanning: boolean): void;
        /**
         * Attached controls to the current camera.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         * @param useCtrlForPanning  Defines whether ctrl is used for panning within the controls
         * @param panningMouseButton Defines whether panning is allowed through mouse click button
         */
        attachControl(noPreventDefault: boolean, useCtrlForPanning: boolean, panningMouseButton: number): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /** @internal */
        _checkInputs(): void;
        protected _checkLimits(): void;
        /**
         * Rebuilds angles (alpha, beta) and radius from the give position and target
         */
        rebuildAnglesAndRadius(): void;
        /**
         * Use a position to define the current camera related information like alpha, beta and radius
         * @param position Defines the position to set the camera at
         */
        setPosition(position: Vector3): void;
        /**
         * Defines the target the camera should look at.
         * This will automatically adapt alpha beta and radius to fit within the new target.
         * Please note that setting a target as a mesh will disable panning.
         * @param target Defines the new target as a Vector or a transform node
         * @param toBoundingCenter In case of a mesh target, defines whether to target the mesh position or its bounding information center
         * @param allowSamePosition If false, prevents reapplying the new computed position if it is identical to the current one (optim)
         * @param cloneAlphaBetaRadius If true, replicate the current setup (alpha, beta, radius) on the new target
         */
        setTarget(target: TransformNode | Vector3, toBoundingCenter?: boolean, allowSamePosition?: boolean, cloneAlphaBetaRadius?: boolean): void;
        /** @internal */
        _getViewMatrix(): Matrix;
        protected _onCollisionPositionChange: (collisionId: number, newPosition: Vector3, collidedMesh?: Nullable<AbstractMesh>) => void;
        /**
         * Zooms on a mesh to be at the min distance where we could see it fully in the current viewport.
         * @param meshes Defines the mesh to zoom on
         * @param doNotUpdateMaxZ Defines whether or not maxZ should be updated whilst zooming on the mesh (this can happen if the mesh is big and the maxradius pretty small for instance)
         */
        zoomOn(meshes?: AbstractMesh[], doNotUpdateMaxZ?: boolean): void;
        /**
         * Focus on a mesh or a bounding box. This adapts the target and maxRadius if necessary but does not update the current radius.
         * The target will be changed but the radius
         * @param meshesOrMinMaxVectorAndDistance Defines the mesh or bounding info to focus on
         * @param doNotUpdateMaxZ Defines whether or not maxZ should be updated whilst zooming on the mesh (this can happen if the mesh is big and the maxradius pretty small for instance)
         */
        focusOn(meshesOrMinMaxVectorAndDistance: AbstractMesh[] | {
            min: Vector3;
            max: Vector3;
            distance: number;
        }, doNotUpdateMaxZ?: boolean): void;
        /**
         * @override
         * Override Camera.createRigCamera
         * @param name the name of the camera
         * @param cameraIndex the index of the camera in the rig cameras array
         */
        createRigCamera(name: string, cameraIndex: number): Camera;
        /**
         * @internal
         * @override
         * Override Camera._updateRigCameras
         */
        _updateRigCameras(): void;
        /**
         * @internal
         */
        _calculateLowerRadiusFromModelBoundingSphere(minimumWorld: Vector3, maximumWorld: Vector3, radiusScale?: number): number;
        /**
         * Destroy the camera and release the current resources hold by it.
         */
        dispose(): void;
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
    }


    /**
     * Default Inputs manager for the ArcRotateCamera.
     * It groups all the default supported inputs for ease of use.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class ArcRotateCameraInputsManager extends CameraInputsManager<ArcRotateCamera> {
        /**
         * Instantiates a new ArcRotateCameraInputsManager.
         * @param camera Defines the camera the inputs belong to
         */
        constructor(camera: ArcRotateCamera);
        /**
         * Add mouse wheel input support to the input manager.
         * @returns the current input manager
         */
        addMouseWheel(): ArcRotateCameraInputsManager;
        /**
         * Add pointers input support to the input manager.
         * @returns the current input manager
         */
        addPointers(): ArcRotateCameraInputsManager;
        /**
         * Add keyboard input support to the input manager.
         * @returns the current input manager
         */
        addKeyboard(): ArcRotateCameraInputsManager;
    }


    /**
     * Oblique projection values
     */
    export interface IObliqueParams {
        /** The angle of the plane */
        angle: number;
        /** The length of the plane */
        length: number;
        /** The offset of the plane */
        offset: number;
    }
    /**
     * This is the base class of all the camera used in the application.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class Camera extends Node {
        /**
         * @internal
         */
        static _CreateDefaultParsedCamera: (name: string, scene: Scene) => Camera;
        /**
         * This is the default projection mode used by the cameras.
         * It helps recreating a feeling of perspective and better appreciate depth.
         * This is the best way to simulate real life cameras.
         */
        static readonly PERSPECTIVE_CAMERA = 0;
        /**
         * This helps creating camera with an orthographic mode.
         * Orthographic is commonly used in engineering as a means to produce object specifications that communicate dimensions unambiguously, each line of 1 unit length (cm, meter..whatever) will appear to have the same length everywhere on the drawing. This allows the drafter to dimension only a subset of lines and let the reader know that other lines of that length on the drawing are also that length in reality. Every parallel line in the drawing is also parallel in the object.
         */
        static readonly ORTHOGRAPHIC_CAMERA = 1;
        /**
         * This is the default FOV mode for perspective cameras.
         * This setting aligns the upper and lower bounds of the viewport to the upper and lower bounds of the camera frustum.
         */
        static readonly FOVMODE_VERTICAL_FIXED = 0;
        /**
         * This setting aligns the left and right bounds of the viewport to the left and right bounds of the camera frustum.
         */
        static readonly FOVMODE_HORIZONTAL_FIXED = 1;
        /**
         * This specifies there is no need for a camera rig.
         * Basically only one eye is rendered corresponding to the camera.
         */
        static readonly RIG_MODE_NONE = 0;
        /**
         * Simulates a camera Rig with one blue eye and one red eye.
         * This can be use with 3d blue and red glasses.
         */
        static readonly RIG_MODE_STEREOSCOPIC_ANAGLYPH = 10;
        /**
         * Defines that both eyes of the camera will be rendered side by side with a parallel target.
         */
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_PARALLEL = 11;
        /**
         * Defines that both eyes of the camera will be rendered side by side with a none parallel target.
         */
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_CROSSEYED = 12;
        /**
         * Defines that both eyes of the camera will be rendered over under each other.
         */
        static readonly RIG_MODE_STEREOSCOPIC_OVERUNDER = 13;
        /**
         * Defines that both eyes of the camera will be rendered on successive lines interlaced for passive 3d monitors.
         */
        static readonly RIG_MODE_STEREOSCOPIC_INTERLACED = 14;
        /**
         * Defines that both eyes of the camera should be renderered in a VR mode (carbox).
         */
        static readonly RIG_MODE_VR = 20;
        /**
         * Custom rig mode allowing rig cameras to be populated manually with any number of cameras
         */
        static readonly RIG_MODE_CUSTOM = 22;
        /**
         * Defines if by default attaching controls should prevent the default javascript event to continue.
         */
        static ForceAttachControlToAlwaysPreventDefault: boolean;
        /**
         * Define the input manager associated with the camera.
         */
        inputs: CameraInputsManager<Camera>;
        /** @internal */
        _position: Vector3;
        /**
         * Define the current local position of the camera in the scene
         */
        get position(): Vector3;
        set position(newPosition: Vector3);
        protected _upVector: Vector3;
        /**
         * The vector the camera should consider as up.
         * (default is Vector3(0, 1, 0) aka Vector3.Up())
         */
        set upVector(vec: Vector3);
        get upVector(): Vector3;
        /**
         * Object containing oblique projection values (only used with ORTHOGRAPHIC_CAMERA)
         */
        oblique: Nullable<IObliqueParams>;
        /**
         * The screen area in scene units squared
         */
        get screenArea(): number;
        /**
         * Define the current limit on the left side for an orthographic camera
         * In scene unit
         */
        private _orthoLeft;
        set orthoLeft(value: Nullable<number>);
        get orthoLeft(): Nullable<number>;
        /**
         * Define the current limit on the right side for an orthographic camera
         * In scene unit
         */
        private _orthoRight;
        set orthoRight(value: Nullable<number>);
        get orthoRight(): Nullable<number>;
        /**
         * Define the current limit on the bottom side for an orthographic camera
         * In scene unit
         */
        private _orthoBottom;
        set orthoBottom(value: Nullable<number>);
        get orthoBottom(): Nullable<number>;
        /**
         * Define the current limit on the top side for an orthographic camera
         * In scene unit
         */
        private _orthoTop;
        set orthoTop(value: Nullable<number>);
        get orthoTop(): Nullable<number>;
        /**
         * Field Of View is set in Radians. (default is 0.8)
         */
        fov: number;
        /**
         * Projection plane tilt around the X axis (horizontal), set in Radians. (default is 0)
         * Can be used to make vertical lines in world space actually vertical on the screen.
         * See https://forum.babylonjs.com/t/add-vertical-shift-to-3ds-max-exporter-babylon-cameras/17480
         */
        projectionPlaneTilt: number;
        /**
         * Define the minimum distance the camera can see from.
         * This is important to note that the depth buffer are not infinite and the closer it starts
         * the more your scene might encounter depth fighting issue.
         */
        minZ: number;
        /**
         * Define the maximum distance the camera can see to.
         * This is important to note that the depth buffer are not infinite and the further it end
         * the more your scene might encounter depth fighting issue.
         */
        maxZ: number;
        /**
         * Define the default inertia of the camera.
         * This helps giving a smooth feeling to the camera movement.
         */
        inertia: number;
        /**
         * Define the mode of the camera (Camera.PERSPECTIVE_CAMERA or Camera.ORTHOGRAPHIC_CAMERA)
         */
        private _mode;
        set mode(mode: number);
        get mode(): number;
        /**
         * Define whether the camera is intermediate.
         * This is useful to not present the output directly to the screen in case of rig without post process for instance
         */
        isIntermediate: boolean;
        /**
         * Define the viewport of the camera.
         * This correspond to the portion of the screen the camera will render to in normalized 0 to 1 unit.
         */
        viewport: Viewport;
        /**
         * Restricts the camera to viewing objects with the same layerMask.
         * A camera with a layerMask of 1 will render mesh.layerMask & camera.layerMask!== 0
         */
        layerMask: number;
        /**
         * fovMode sets the camera frustum bounds to the viewport bounds. (default is FOVMODE_VERTICAL_FIXED)
         */
        fovMode: number;
        /**
         * Rig mode of the camera.
         * This is useful to create the camera with two "eyes" instead of one to create VR or stereoscopic scenes.
         * This is normally controlled byt the camera themselves as internal use.
         */
        cameraRigMode: number;
        /**
         * Defines the distance between both "eyes" in case of a RIG
         */
        interaxialDistance: number;
        /**
         * Defines if stereoscopic rendering is done side by side or over under.
         */
        isStereoscopicSideBySide: boolean;
        /**
         * Defines the list of custom render target which are rendered to and then used as the input to this camera's render. Eg. display another camera view on a TV in the main scene
         * This is pretty helpful if you wish to make a camera render to a texture you could reuse somewhere
         * else in the scene. (Eg. security camera)
         *
         * To change the final output target of the camera, camera.outputRenderTarget should be used instead (eg. webXR renders to a render target corresponding to an HMD)
         */
        customRenderTargets: RenderTargetTexture[];
        /**
         * When set, the camera will render to this render target instead of the default canvas
         *
         * If the desire is to use the output of a camera as a texture in the scene consider using camera.customRenderTargets instead
         */
        outputRenderTarget: Nullable<RenderTargetTexture>;
        /**
         * Observable triggered when the camera view matrix has changed.
         */
        onViewMatrixChangedObservable: Observable<Camera>;
        /**
         * Observable triggered when the camera Projection matrix has changed.
         */
        onProjectionMatrixChangedObservable: Observable<Camera>;
        /**
         * Observable triggered when the inputs have been processed.
         */
        onAfterCheckInputsObservable: Observable<Camera>;
        /**
         * Observable triggered when reset has been called and applied to the camera.
         */
        onRestoreStateObservable: Observable<Camera>;
        /**
         * Is this camera a part of a rig system?
         */
        isRigCamera: boolean;
        /**
         * If isRigCamera set to true this will be set with the parent camera.
         * The parent camera is not (!) necessarily the .parent of this camera (like in the case of XR)
         */
        rigParent?: Camera;
        /**
         * Render pass id used by the camera to render into the main framebuffer
         */
        renderPassId: number;
        private _hasMoved;
        /**
         * Gets a flag indicating that the camera has moved in some way since the last call to Camera.update()
         */
        get hasMoved(): boolean;
        /** @internal */
        _cameraRigParams: any;
        /** @internal */
        _rigCameras: Camera[];
        /** @internal */
        _rigPostProcess: Nullable<PostProcess>;
        /** @internal */
        _skipRendering: boolean;
        /** @internal */
        _projectionMatrix: Matrix;
        /** @internal */
        _postProcesses: Nullable<PostProcess>[];
        /** @internal */
        _activeMeshes: SmartArray<AbstractMesh>;
        protected _globalPosition: Vector3;
        /** @internal */
        _computedViewMatrix: Matrix;
        private _doNotComputeProjectionMatrix;
        private _transformMatrix;
        private _frustumPlanes;
        private _refreshFrustumPlanes;
        private _storedFov;
        private _stateStored;
        private _absoluteRotation;
        /**
         * Instantiates a new camera object.
         * This should not be used directly but through the inherited cameras: ArcRotate, Free...
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
         * @param name Defines the name of the camera in the scene
         * @param position Defines the position of the camera
         * @param scene Defines the scene the camera belongs too
         * @param setActiveOnSceneIfNoneActive Defines if the camera should be set as active after creation if no other camera have been defined in the scene
         */
        constructor(name: string, position: Vector3, scene?: Scene, setActiveOnSceneIfNoneActive?: boolean);
        /**
         * Store current camera state (fov, position, etc..)
         * @returns the camera
         */
        storeState(): Camera;
        /**
         * Restores the camera state values if it has been stored. You must call storeState() first
         * @returns true if restored and false otherwise
         */
        protected _restoreStateValues(): boolean;
        /**
         * Restored camera state. You must call storeState() first.
         * @returns true if restored and false otherwise
         */
        restoreState(): boolean;
        /**
         * Gets the class name of the camera.
         * @returns the class name
         */
        getClassName(): string;
        /** @internal */
        readonly _isCamera = true;
        /**
         * Gets a string representation of the camera useful for debug purpose.
         * @param fullDetails Defines that a more verbose level of logging is required
         * @returns the string representation
         */
        toString(fullDetails?: boolean): string;
        /**
         * Automatically tilts the projection plane, using `projectionPlaneTilt`, to correct the perspective effect on vertical lines.
         */
        applyVerticalCorrection(): void;
        /**
         * Gets the current world space position of the camera.
         */
        get globalPosition(): Vector3;
        /**
         * Gets the list of active meshes this frame (meshes no culled or excluded by lod s in the frame)
         * @returns the active meshe list
         */
        getActiveMeshes(): SmartArray<AbstractMesh>;
        /**
         * Check whether a mesh is part of the current active mesh list of the camera
         * @param mesh Defines the mesh to check
         * @returns true if active, false otherwise
         */
        isActiveMesh(mesh: Mesh): boolean;
        /**
         * Is this camera ready to be used/rendered
         * @param completeCheck defines if a complete check (including post processes) has to be done (false by default)
         * @returns true if the camera is ready
         */
        isReady(completeCheck?: boolean): boolean;
        /** @internal */
        _initCache(): void;
        /**
         * @internal
         */
        _updateCache(ignoreParentClass?: boolean): void;
        /** @internal */
        _isSynchronized(): boolean;
        /** @internal */
        _isSynchronizedViewMatrix(): boolean;
        /** @internal */
        _isSynchronizedProjectionMatrix(): boolean;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param ignored defines an ignored parameter kept for backward compatibility.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         * BACK COMPAT SIGNATURE ONLY.
         */
        attachControl(ignored: any, noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Detach the current controls from the specified dom element.
         * @param ignored defines an ignored parameter kept for backward compatibility.
         */
        detachControl(ignored?: any): void;
        /**
         * Update the camera state according to the different inputs gathered during the frame.
         */
        update(): void;
        /** @internal */
        _checkInputs(): void;
        /** @internal */
        get rigCameras(): Camera[];
        /**
         * Gets the post process used by the rig cameras
         */
        get rigPostProcess(): Nullable<PostProcess>;
        /**
         * Internal, gets the first post process.
         * @returns the first post process to be run on this camera.
         */
        _getFirstPostProcess(): Nullable<PostProcess>;
        private _cascadePostProcessesToRigCams;
        /**
         * Attach a post process to the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/postProcesses/usePostProcesses#attach-postprocess
         * @param postProcess The post process to attach to the camera
         * @param insertAt The position of the post process in case several of them are in use in the scene
         * @returns the position the post process has been inserted at
         */
        attachPostProcess(postProcess: PostProcess, insertAt?: Nullable<number>): number;
        /**
         * Detach a post process to the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/postProcesses/usePostProcesses#attach-postprocess
         * @param postProcess The post process to detach from the camera
         */
        detachPostProcess(postProcess: PostProcess): void;
        /**
         * Gets the current world matrix of the camera
         * @returns the world matrix
         */
        getWorldMatrix(): Matrix;
        /** @internal */
        _getViewMatrix(): Matrix;
        /**
         * Gets the current view matrix of the camera.
         * @param force forces the camera to recompute the matrix without looking at the cached state
         * @returns the view matrix
         */
        getViewMatrix(force?: boolean): Matrix;
        /**
         * Freeze the projection matrix.
         * It will prevent the cache check of the camera projection compute and can speed up perf
         * if no parameter of the camera are meant to change
         * @param projection Defines manually a projection if necessary
         */
        freezeProjectionMatrix(projection?: Matrix): void;
        /**
         * Unfreeze the projection matrix if it has previously been freezed by freezeProjectionMatrix.
         */
        unfreezeProjectionMatrix(): void;
        /**
         * Gets the current projection matrix of the camera.
         * @param force forces the camera to recompute the matrix without looking at the cached state
         * @returns the projection matrix
         */
        getProjectionMatrix(force?: boolean): Matrix;
        /**
         * Gets the transformation matrix (ie. the multiplication of view by projection matrices)
         * @returns a Matrix
         */
        getTransformationMatrix(): Matrix;
        private _computeObliqueDistance;
        private _updateFrustumPlanes;
        /**
         * Checks if a cullable object (mesh...) is in the camera frustum
         * This checks the bounding box center. See isCompletelyInFrustum for a full bounding check
         * @param target The object to check
         * @param checkRigCameras If the rig cameras should be checked (eg. with VR camera both eyes should be checked) (Default: false)
         * @returns true if the object is in frustum otherwise false
         */
        isInFrustum(target: ICullable, checkRigCameras?: boolean): boolean;
        /**
         * Checks if a cullable object (mesh...) is in the camera frustum
         * Unlike isInFrustum this checks the full bounding box
         * @param target The object to check
         * @returns true if the object is in frustum otherwise false
         */
        isCompletelyInFrustum(target: ICullable): boolean;
        /**
         * Gets a ray in the forward direction from the camera.
         * @param length Defines the length of the ray to create
         * @param transform Defines the transform to apply to the ray, by default the world matrix is used to create a workd space ray
         * @param origin Defines the start point of the ray which defaults to the camera position
         * @returns the forward ray
         */
        getForwardRay(length?: number, transform?: Matrix, origin?: Vector3): Ray;
        /**
         * Gets a ray in the forward direction from the camera.
         * @param refRay the ray to (re)use when setting the values
         * @param length Defines the length of the ray to create
         * @param transform Defines the transform to apply to the ray, by default the world matrx is used to create a workd space ray
         * @param origin Defines the start point of the ray which defaults to the camera position
         * @returns the forward ray
         */
        getForwardRayToRef(refRay: Ray, length?: number, transform?: Matrix, origin?: Vector3): Ray;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /** @internal */
        _isLeftCamera: boolean;
        /**
         * Gets the left camera of a rig setup in case of Rigged Camera
         */
        get isLeftCamera(): boolean;
        /** @internal */
        _isRightCamera: boolean;
        /**
         * Gets the right camera of a rig setup in case of Rigged Camera
         */
        get isRightCamera(): boolean;
        /**
         * Gets the left camera of a rig setup in case of Rigged Camera
         */
        get leftCamera(): Nullable<FreeCamera>;
        /**
         * Gets the right camera of a rig setup in case of Rigged Camera
         */
        get rightCamera(): Nullable<FreeCamera>;
        /**
         * Gets the left camera target of a rig setup in case of Rigged Camera
         * @returns the target position
         */
        getLeftTarget(): Nullable<Vector3>;
        /**
         * Gets the right camera target of a rig setup in case of Rigged Camera
         * @returns the target position
         */
        getRightTarget(): Nullable<Vector3>;
        /**
         * @internal
         */
        setCameraRigMode(mode: number, rigParams: any): void;
        protected _setRigMode(rigParams: any): void;
        /** @internal */
        _getVRProjectionMatrix(): Matrix;
        /**
         * @internal
         */
        setCameraRigParameter(name: string, value: any): void;
        /**
         * needs to be overridden by children so sub has required properties to be copied
         * @internal
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * May need to be overridden by children
         * @internal
         */
        _updateRigCameras(): void;
        /** @internal */
        _setupInputs(): void;
        /**
         * Serialiaze the camera setup to a json representation
         * @returns the JSON representation
         */
        serialize(): any;
        /**
         * Clones the current camera.
         * @param name The cloned camera name
         * @param newParent The cloned camera's new parent (none by default)
         * @returns the cloned camera
         */
        clone(name: string, newParent?: Nullable<Node>): Camera;
        /**
         * Gets the direction of the camera relative to a given local axis.
         * @param localAxis Defines the reference axis to provide a relative direction.
         * @returns the direction
         */
        getDirection(localAxis: Vector3): Vector3;
        /**
         * Returns the current camera absolute rotation
         */
        get absoluteRotation(): Quaternion;
        /**
         * Gets the direction of the camera relative to a given local axis into a passed vector.
         * @param localAxis Defines the reference axis to provide a relative direction.
         * @param result Defines the vector to store the result in
         */
        getDirectionToRef(localAxis: Vector3, result: Vector3): void;
        /**
         * Gets a camera constructor for a given camera type
         * @param type The type of the camera to construct (should be equal to one of the camera class name)
         * @param name The name of the camera the result will be able to instantiate
         * @param scene The scene the result will construct the camera in
         * @param interaxial_distance In case of stereoscopic setup, the distance between both eyes
         * @param isStereoscopicSideBySide In case of stereoscopic setup, should the sereo be side b side
         * @returns a factory method to construct the camera
         */
        static GetConstructorFromName(type: string, name: string, scene: Scene, interaxial_distance?: number, isStereoscopicSideBySide?: boolean): () => Camera;
        /**
         * Compute the world  matrix of the camera.
         * @returns the camera world matrix
         */
        computeWorldMatrix(): Matrix;
        /**
         * Parse a JSON and creates the camera from the parsed information
         * @param parsedCamera The JSON to parse
         * @param scene The scene to instantiate the camera in
         * @returns the newly constructed camera
         */
        static Parse(parsedCamera: any, scene: Scene): Camera;
        /** @internal */
        _calculateHandednessMultiplier(): number;
    }


    /**
     * @ignore
     * This is a list of all the different input types that are available in the application.
     * Fo instance: ArcRotateCameraGamepadInput...
     */
    export var CameraInputTypes: {};
    /**
     * This is the contract to implement in order to create a new input class.
     * Inputs are dealing with listening to user actions and moving the camera accordingly.
     */
    export interface ICameraInput<TCamera extends Camera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: Nullable<TCamera>;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs?: () => void;
    }
    /**
     * Represents a map of input types to input instance or input index to input instance.
     */
    export interface CameraInputsMap<TCamera extends Camera> {
        /**
         * Accessor to the input by input type.
         */
        [name: string]: ICameraInput<TCamera>;
        /**
         * Accessor to the input by input index.
         */
        [idx: number]: ICameraInput<TCamera>;
    }
    /**
     * This represents the input manager used within a camera.
     * It helps dealing with all the different kind of input attached to a camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class CameraInputsManager<TCamera extends Camera> {
        /**
         * Defines the list of inputs attached to the camera.
         */
        attached: CameraInputsMap<TCamera>;
        /**
         * Defines the dom element the camera is collecting inputs from.
         * This is null if the controls have not been attached.
         */
        attachedToElement: boolean;
        /**
         * Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        noPreventDefault: boolean;
        /**
         * Defined the camera the input manager belongs to.
         */
        camera: TCamera;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs: () => void;
        /**
         * Instantiate a new Camera Input Manager.
         * @param camera Defines the camera the input manager belongs to
         */
        constructor(camera: TCamera);
        /**
         * Add an input method to a camera
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
         * @param input Camera input method
         */
        add(input: ICameraInput<TCamera>): void;
        /**
         * Remove a specific input method from a camera
         * example: camera.inputs.remove(camera.inputs.attached.mouse);
         * @param inputToRemove camera input method
         */
        remove(inputToRemove: ICameraInput<TCamera>): void;
        /**
         * Remove a specific input type from a camera
         * example: camera.inputs.remove("ArcRotateCameraGamepadInput");
         * @param inputType the type of the input to remove
         */
        removeByType(inputType: string): void;
        private _addCheckInputs;
        /**
         * Attach the input controls to the currently attached dom element to listen the events from.
         * @param input Defines the input to attach
         */
        attachInput(input: ICameraInput<TCamera>): void;
        /**
         * Attach the current manager inputs controls to a specific dom element to listen the events from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachElement(noPreventDefault?: boolean): void;
        /**
         * Detach the current manager inputs controls from a specific dom element.
         * @param disconnect Defines whether the input should be removed from the current list of attached inputs
         */
        detachElement(disconnect?: boolean): void;
        /**
         * Rebuild the dynamic inputCheck function from the current list of
         * defined inputs in the manager.
         */
        rebuildInputCheck(): void;
        /**
         * Remove all attached input methods from a camera
         */
        clear(): void;
        /**
         * Serialize the current input manager attached to a camera.
         * This ensures than once parsed,
         * the input associated to the camera will be identical to the current ones
         * @param serializedCamera Defines the camera serialization JSON the input serialization should write to
         */
        serialize(serializedCamera: any): void;
        /**
         * Parses an input manager serialized JSON to restore the previous list of inputs
         * and states associated to a camera.
         * @param parsedCamera Defines the JSON to parse
         */
        parse(parsedCamera: any): void;
    }


    /**
     * This is a camera specifically designed to react to device orientation events such as a modern mobile device
     * being tilted forward or back and left or right.
     */
    export class DeviceOrientationCamera extends FreeCamera {
        private _initialQuaternion;
        private _quaternionCache;
        private _tmpDragQuaternion;
        private _disablePointerInputWhenUsingDeviceOrientation;
        /**
         * Creates a new device orientation camera
         * @param name The name of the camera
         * @param position The start position camera
         * @param scene The scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene?: Scene);
        /**
         * Gets or sets a boolean indicating that pointer input must be disabled on first orientation sensor update (Default: true)
         */
        get disablePointerInputWhenUsingDeviceOrientation(): boolean;
        set disablePointerInputWhenUsingDeviceOrientation(value: boolean);
        private _dragFactor;
        /**
         * Enabled turning on the y axis when the orientation sensor is active
         * @param dragFactor the factor that controls the turn speed (default: 1/300)
         */
        enableHorizontalDragging(dragFactor?: number): void;
        /**
         * Gets the current instance class name ("DeviceOrientationCamera").
         * This helps avoiding instanceof at run time.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * @internal
         * Checks and applies the current values of the inputs to the camera. (Internal use only)
         */
        _checkInputs(): void;
        /**
         * Reset the camera to its default orientation on the specified axis only.
         * @param axis The axis to reset
         */
        resetToCurrentRotation(axis?: Axis): void;
    }


    /**
     * This is a flying camera, designed for 3D movement and rotation in all directions,
     * such as in a 3D Space Shooter or a Flight Simulator.
     */
    export class FlyCamera extends TargetCamera {
        /**
         * Define the collision ellipsoid of the camera.
         * This is helpful for simulating a camera body, like a player's body.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_collisions#arcrotatecamera
         */
        ellipsoid: Vector3;
        /**
         * Define an offset for the position of the ellipsoid around the camera.
         * This can be helpful if the camera is attached away from the player's body center,
         * such as at its head.
         */
        ellipsoidOffset: Vector3;
        /**
         * Enable or disable collisions of the camera with the rest of the scene objects.
         */
        checkCollisions: boolean;
        /**
         * Enable or disable gravity on the camera.
         */
        applyGravity: boolean;
        /**
         * Define the current direction the camera is moving to.
         */
        cameraDirection: Vector3;
        /**
         * Define the current local rotation of the camera as a quaternion to prevent Gimbal lock.
         * This overrides and empties cameraRotation.
         */
        rotationQuaternion: Quaternion;
        /**
         * Track Roll to maintain the wanted Rolling when looking around.
         */
        _trackRoll: number;
        /**
         * Slowly correct the Roll to its original value after a Pitch+Yaw rotation.
         */
        rollCorrect: number;
        /**
         * Mimic a banked turn, Rolling the camera when Yawing.
         * It's recommended to use rollCorrect = 10 for faster banking correction.
         */
        bankedTurn: boolean;
        /**
         * Limit in radians for how much Roll banking will add. (Default: 90°)
         */
        bankedTurnLimit: number;
        /**
         * Value of 0 disables the banked Roll.
         * Value of 1 is equal to the Yaw angle in radians.
         */
        bankedTurnMultiplier: number;
        /**
         * The inputs manager loads all the input sources, such as keyboard and mouse.
         */
        inputs: FlyCameraInputsManager;
        /**
         * Gets the input sensibility for mouse input.
         * Higher values reduce sensitivity.
         */
        get angularSensibility(): number;
        /**
         * Sets the input sensibility for a mouse input.
         * Higher values reduce sensitivity.
         */
        set angularSensibility(value: number);
        /**
         * Get the keys for camera movement forward.
         */
        get keysForward(): number[];
        /**
         * Set the keys for camera movement forward.
         */
        set keysForward(value: number[]);
        /**
         * Get the keys for camera movement backward.
         */
        get keysBackward(): number[];
        set keysBackward(value: number[]);
        /**
         * Get the keys for camera movement up.
         */
        get keysUp(): number[];
        /**
         * Set the keys for camera movement up.
         */
        set keysUp(value: number[]);
        /**
         * Get the keys for camera movement down.
         */
        get keysDown(): number[];
        /**
         * Set the keys for camera movement down.
         */
        set keysDown(value: number[]);
        /**
         * Get the keys for camera movement left.
         */
        get keysLeft(): number[];
        /**
         * Set the keys for camera movement left.
         */
        set keysLeft(value: number[]);
        /**
         * Set the keys for camera movement right.
         */
        get keysRight(): number[];
        /**
         * Set the keys for camera movement right.
         */
        set keysRight(value: number[]);
        /**
         * Event raised when the camera collides with a mesh in the scene.
         */
        onCollide: (collidedMesh: AbstractMesh) => void;
        private _collider;
        private _needMoveForGravity;
        private _oldPosition;
        private _diffPosition;
        private _newPosition;
        /** @internal */
        _localDirection: Vector3;
        /** @internal */
        _transformedDirection: Vector3;
        /**
         * Instantiates a FlyCamera.
         * This is a flying camera, designed for 3D movement and rotation in all directions,
         * such as in a 3D Space Shooter or a Flight Simulator.
         * @param name Define the name of the camera in the scene.
         * @param position Define the starting position of the camera in the scene.
         * @param scene Define the scene the camera belongs to.
         * @param setActiveOnSceneIfNoneActive Defines whether the camera should be marked as active, if no other camera has been defined as active.
         */
        constructor(name: string, position: Vector3, scene?: Scene, setActiveOnSceneIfNoneActive?: boolean);
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach a control from the HTML DOM element.
         * The camera will stop reacting to that input.
         */
        detachControl(): void;
        private _collisionMask;
        /**
         * Get the mask that the camera ignores in collision events.
         */
        get collisionMask(): number;
        /**
         * Set the mask that the camera ignores in collision events.
         */
        set collisionMask(mask: number);
        /**
         * @internal
         */
        _collideWithWorld(displacement: Vector3): void;
        /**
         * @internal
         */
        private _onCollisionPositionChange;
        /** @internal */
        _checkInputs(): void;
        /**
         * Enable movement without a user input. This allows gravity to always be applied.
         */
        set needMoveForGravity(value: boolean);
        /**
         * When true, gravity is applied whether there is user input or not.
         */
        get needMoveForGravity(): boolean;
        /** @internal */
        _decideIfNeedsToMove(): boolean;
        /** @internal */
        _updatePosition(): void;
        /**
         * Restore the Roll to its target value at the rate specified.
         * @param rate - Higher means slower restoring.
         * @internal
         */
        restoreRoll(rate: number): void;
        /**
         * Destroy the camera and release the current resources held by it.
         */
        dispose(): void;
        /**
         * Get the current object class name.
         * @returns the class name.
         */
        getClassName(): string;
    }


    /**
     * Default Inputs manager for the FlyCamera.
     * It groups all the default supported inputs for ease of use.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FlyCameraInputsManager extends CameraInputsManager<FlyCamera> {
        /**
         * Instantiates a new FlyCameraInputsManager.
         * @param camera Defines the camera the inputs belong to.
         */
        constructor(camera: FlyCamera);
        /**
         * Add keyboard input support to the input manager.
         * @returns the new FlyCameraKeyboardMoveInput().
         */
        addKeyboard(): FlyCameraInputsManager;
        /**
         * Add mouse input support to the input manager.
         * @returns the new FlyCameraMouseInput().
         */
        addMouse(): FlyCameraInputsManager;
    }


    /**
     * A follow camera takes a mesh as a target and follows it as it moves. Both a free camera version followCamera and
     * an arc rotate version arcFollowCamera are available.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#followcamera
     */
    export class FollowCamera extends TargetCamera {
        /**
         * Distance the follow camera should follow an object at
         */
        radius: number;
        /**
         * Minimum allowed distance of the camera to the axis of rotation
         * (The camera can not get closer).
         * This can help limiting how the Camera is able to move in the scene.
         */
        lowerRadiusLimit: Nullable<number>;
        /**
         * Maximum allowed distance of the camera to the axis of rotation
         * (The camera can not get further).
         * This can help limiting how the Camera is able to move in the scene.
         */
        upperRadiusLimit: Nullable<number>;
        /**
         * Define a rotation offset between the camera and the object it follows
         */
        rotationOffset: number;
        /**
         * Minimum allowed angle to camera position relative to target object.
         * This can help limiting how the Camera is able to move in the scene.
         */
        lowerRotationOffsetLimit: Nullable<number>;
        /**
         * Maximum allowed angle to camera position relative to target object.
         * This can help limiting how the Camera is able to move in the scene.
         */
        upperRotationOffsetLimit: Nullable<number>;
        /**
         * Define a height offset between the camera and the object it follows.
         * It can help following an object from the top (like a car chasing a plane)
         */
        heightOffset: number;
        /**
         * Minimum allowed height of camera position relative to target object.
         * This can help limiting how the Camera is able to move in the scene.
         */
        lowerHeightOffsetLimit: Nullable<number>;
        /**
         * Maximum allowed height of camera position relative to target object.
         * This can help limiting how the Camera is able to move in the scene.
         */
        upperHeightOffsetLimit: Nullable<number>;
        /**
         * Define how fast the camera can accelerate to follow it s target.
         */
        cameraAcceleration: number;
        /**
         * Define the speed limit of the camera following an object.
         */
        maxCameraSpeed: number;
        /**
         * Define the target of the camera.
         */
        lockedTarget: Nullable<AbstractMesh>;
        /**
         * Defines the input associated with the camera.
         */
        inputs: FollowCameraInputsManager;
        /**
         * Instantiates the follow camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#followcamera
         * @param name Define the name of the camera in the scene
         * @param position Define the position of the camera
         * @param scene Define the scene the camera belong to
         * @param lockedTarget Define the target of the camera
         */
        constructor(name: string, position: Vector3, scene?: Scene, lockedTarget?: Nullable<AbstractMesh>);
        private _follow;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /** @internal */
        _checkInputs(): void;
        private _checkLimits;
        /**
         * Gets the camera class name.
         * @returns the class name
         */
        getClassName(): string;
    }
    /**
     * Arc Rotate version of the follow camera.
     * It still follows a Defined mesh but in an Arc Rotate Camera fashion.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#followcamera
     */
    export class ArcFollowCamera extends TargetCamera {
        /** The longitudinal angle of the camera */
        alpha: number;
        /** The latitudinal angle of the camera */
        beta: number;
        /** The radius of the camera from its target */
        radius: number;
        private _cartesianCoordinates;
        /** Define the camera target (the mesh it should follow) */
        private _meshTarget;
        /**
         * Instantiates a new ArcFollowCamera
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#followcamera
         * @param name Define the name of the camera
         * @param alpha Define the rotation angle of the camera around the longitudinal axis
         * @param beta Define the rotation angle of the camera around the elevation axis
         * @param radius Define the radius of the camera from its target point
         * @param target Define the target of the camera
         * @param scene Define the scene the camera belongs to
         */
        constructor(name: string, 
        /** The longitudinal angle of the camera */
        alpha: number, 
        /** The latitudinal angle of the camera */
        beta: number, 
        /** The radius of the camera from its target */
        radius: number, 
        /** Define the camera target (the mesh it should follow) */
        target: Nullable<AbstractMesh>, scene: Scene);
        /**
         * Sets the mesh to follow with this camera.
         * @param target the target to follow
         */
        setMeshTarget(target: Nullable<AbstractMesh>): void;
        private _follow;
        /** @internal */
        _checkInputs(): void;
        /**
         * Returns the class name of the object.
         * It is mostly used internally for serialization purposes.
         * @returns the class name
         */
        getClassName(): string;
    }


    /**
     * Default Inputs manager for the FollowCamera.
     * It groups all the default supported inputs for ease of use.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FollowCameraInputsManager extends CameraInputsManager<FollowCamera> {
        /**
         * Instantiates a new FollowCameraInputsManager.
         * @param camera Defines the camera the inputs belong to
         */
        constructor(camera: FollowCamera);
        /**
         * Add keyboard input support to the input manager.
         * @returns the current input manager
         */
        addKeyboard(): FollowCameraInputsManager;
        /**
         * Add mouse wheel input support to the input manager.
         * @returns the current input manager
         */
        addMouseWheel(): FollowCameraInputsManager;
        /**
         * Add pointers input support to the input manager.
         * @returns the current input manager
         */
        addPointers(): FollowCameraInputsManager;
        /**
         * Add orientation input support to the input manager.
         * @returns the current input manager
         */
        addVRDeviceOrientation(): FollowCameraInputsManager;
    }


    /**
     * This represents a free type of camera. It can be useful in First Person Shooter game for instance.
     * Please consider using the new UniversalCamera instead as it adds more functionality like the gamepad.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
     */
    export class FreeCamera extends TargetCamera {
        /**
         * Define the collision ellipsoid of the camera.
         * This is helpful to simulate a camera body like the player body around the camera
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_collisions#arcrotatecamera
         */
        ellipsoid: Vector3;
        /**
         * Define an offset for the position of the ellipsoid around the camera.
         * This can be helpful to determine the center of the body near the gravity center of the body
         * instead of its head.
         */
        ellipsoidOffset: Vector3;
        /**
         * Enable or disable collisions of the camera with the rest of the scene objects.
         */
        checkCollisions: boolean;
        /**
         * Enable or disable gravity on the camera.
         */
        applyGravity: boolean;
        /**
         * Define the input manager associated to the camera.
         */
        inputs: FreeCameraInputsManager;
        /**
         * Gets the input sensibility for a mouse input. (default is 2000.0)
         * Higher values reduce sensitivity.
         */
        get angularSensibility(): number;
        /**
         * Sets the input sensibility for a mouse input. (default is 2000.0)
         * Higher values reduce sensitivity.
         */
        set angularSensibility(value: number);
        /**
         * Gets or Set the list of keyboard keys used to control the forward move of the camera.
         */
        get keysUp(): number[];
        set keysUp(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the upward move of the camera.
         */
        get keysUpward(): number[];
        set keysUpward(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the backward move of the camera.
         */
        get keysDown(): number[];
        set keysDown(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the downward move of the camera.
         */
        get keysDownward(): number[];
        set keysDownward(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the left strafe move of the camera.
         */
        get keysLeft(): number[];
        set keysLeft(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the right strafe move of the camera.
         */
        get keysRight(): number[];
        set keysRight(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the left rotation move of the camera.
         */
        get keysRotateLeft(): number[];
        set keysRotateLeft(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the right rotation move of the camera.
         */
        get keysRotateRight(): number[];
        set keysRotateRight(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the up rotation move of the camera.
         */
        get keysRotateUp(): number[];
        set keysRotateUp(value: number[]);
        /**
         * Gets or Set the list of keyboard keys used to control the down rotation move of the camera.
         */
        get keysRotateDown(): number[];
        set keysRotateDown(value: number[]);
        /**
         * Event raised when the camera collide with a mesh in the scene.
         */
        onCollide: (collidedMesh: AbstractMesh) => void;
        private _collider;
        private _needMoveForGravity;
        private _oldPosition;
        private _diffPosition;
        private _newPosition;
        /** @internal */
        _localDirection: Vector3;
        /** @internal */
        _transformedDirection: Vector3;
        /**
         * Instantiates a Free Camera.
         * This represents a free type of camera. It can be useful in First Person Shooter game for instance.
         * Please consider using the new UniversalCamera instead as it adds more functionality like touch to this camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
         * @param name Define the name of the camera in the scene
         * @param position Define the start position of the camera in the scene
         * @param scene Define the scene the camera belongs to
         * @param setActiveOnSceneIfNoneActive Defines whether the camera should be marked as active if not other active cameras have been defined
         */
        constructor(name: string, position: Vector3, scene?: Scene, setActiveOnSceneIfNoneActive?: boolean);
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param ignored defines an ignored parameter kept for backward compatibility.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         * BACK COMPAT SIGNATURE ONLY.
         */
        attachControl(ignored: any, noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        private _collisionMask;
        /**
         * Define a collision mask to limit the list of object the camera can collide with
         */
        get collisionMask(): number;
        set collisionMask(mask: number);
        /**
         * @internal
         */
        _collideWithWorld(displacement: Vector3): void;
        private _onCollisionPositionChange;
        /** @internal */
        _checkInputs(): void;
        /**
         * Enable movement without a user input. This allows gravity to always be applied.
         */
        set needMoveForGravity(value: boolean);
        /**
         * When true, gravity is applied whether there is user input or not.
         */
        get needMoveForGravity(): boolean;
        /** @internal */
        _decideIfNeedsToMove(): boolean;
        /** @internal */
        _updatePosition(): void;
        /**
         * Destroy the camera and release the current resources hold by it.
         */
        dispose(): void;
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
    }


    /**
     * Default Inputs manager for the FreeCamera.
     * It groups all the default supported inputs for ease of use.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraInputsManager extends CameraInputsManager<FreeCamera> {
        /**
         * @internal
         */
        _mouseInput: Nullable<FreeCameraMouseInput>;
        /**
         * @internal
         */
        _mouseWheelInput: Nullable<FreeCameraMouseWheelInput>;
        /**
         * Instantiates a new FreeCameraInputsManager.
         * @param camera Defines the camera the inputs belong to
         */
        constructor(camera: FreeCamera);
        /**
         * Add keyboard input support to the input manager.
         * @returns the current input manager
         */
        addKeyboard(): FreeCameraInputsManager;
        /**
         * Add mouse input support to the input manager.
         * @param touchEnabled if the FreeCameraMouseInput should support touch (default: true)
         * @returns the current input manager
         */
        addMouse(touchEnabled?: boolean): FreeCameraInputsManager;
        /**
         * Removes the mouse input support from the manager
         * @returns the current input manager
         */
        removeMouse(): FreeCameraInputsManager;
        /**
         * Add mouse wheel input support to the input manager.
         * @returns the current input manager
         */
        addMouseWheel(): FreeCameraInputsManager;
        /**
         * Removes the mouse wheel input support from the manager
         * @returns the current input manager
         */
        removeMouseWheel(): FreeCameraInputsManager;
        /**
         * Add touch input support to the input manager.
         * @returns the current input manager
         */
        addTouch(): FreeCameraInputsManager;
        /**
         * Remove all attached input methods from a camera
         */
        clear(): void;
    }


    /**
     * This represents a FPS type of camera. This is only here for back compat purpose.
     * Please use the UniversalCamera instead as both are identical.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
     */
    export class GamepadCamera extends UniversalCamera {
        /**
         * Instantiates a new Gamepad Camera
         * This represents a FPS type of camera. This is only here for back compat purpose.
         * Please use the UniversalCamera instead as both are identical.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
         * @param name Define the name of the camera in the scene
         * @param position Define the start position of the camera in the scene
         * @param scene Define the scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene?: Scene);
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
    }




    /**
     * Manage the gamepad inputs to control an arc rotate camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class ArcRotateCameraGamepadInput implements ICameraInput<ArcRotateCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: ArcRotateCamera;
        /**
         * Defines the gamepad the input is gathering event from.
         */
        gamepad: Nullable<Gamepad>;
        /**
         * Defines the gamepad rotation sensibility.
         * This is the threshold from when rotation starts to be accounted for to prevent jittering.
         */
        gamepadRotationSensibility: number;
        /**
         * Defines the gamepad move sensibility.
         * This is the threshold from when moving starts to be accounted for for to prevent jittering.
         */
        gamepadMoveSensibility: number;
        private _yAxisScale;
        /**
         * Gets or sets a boolean indicating that Yaxis (for right stick) should be inverted
         */
        get invertYAxis(): boolean;
        set invertYAxis(value: boolean);
        private _onGamepadConnectedObserver;
        private _onGamepadDisconnectedObserver;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         */
        attachControl(): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current intput.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Manage the keyboard inputs to control the movement of an arc rotate camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class ArcRotateCameraKeyboardMoveInput implements ICameraInput<ArcRotateCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: ArcRotateCamera;
        /**
         * Defines the list of key codes associated with the up action (increase alpha)
         */
        keysUp: number[];
        /**
         * Defines the list of key codes associated with the down action (decrease alpha)
         */
        keysDown: number[];
        /**
         * Defines the list of key codes associated with the left action (increase beta)
         */
        keysLeft: number[];
        /**
         * Defines the list of key codes associated with the right action (decrease beta)
         */
        keysRight: number[];
        /**
         * Defines the list of key codes associated with the reset action.
         * Those keys reset the camera to its last stored state (with the method camera.storeState())
         */
        keysReset: number[];
        /**
         * Defines the panning sensibility of the inputs.
         * (How fast is the camera panning)
         */
        panningSensibility: number;
        /**
         * Defines the zooming sensibility of the inputs.
         * (How fast is the camera zooming)
         */
        zoomingSensibility: number;
        /**
         * Defines whether maintaining the alt key down switch the movement mode from
         * orientation to zoom.
         */
        useAltToZoom: boolean;
        /**
         * Rotation speed of the camera
         */
        angularSpeed: number;
        private _keys;
        private _ctrlPressed;
        private _altPressed;
        private _onCanvasBlurObserver;
        private _onKeyboardObserver;
        private _engine;
        private _scene;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Manage the mouse wheel inputs to control an arc rotate camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class ArcRotateCameraMouseWheelInput implements ICameraInput<ArcRotateCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: ArcRotateCamera;
        /**
         * Gets or Set the mouse wheel precision or how fast is the camera zooming.
         */
        wheelPrecision: number;
        /**
         * Gets or Set the boolean value that controls whether or not the mouse wheel
         * zooms to the location of the mouse pointer or not.  The default is false.
         */
        zoomToMouseLocation: boolean;
        /**
         * wheelDeltaPercentage will be used instead of wheelPrecision if different from 0.
         * It defines the percentage of current camera.radius to use as delta when wheel is used.
         */
        wheelDeltaPercentage: number;
        /**
         * If set, this function will be used to set the radius delta that will be added to the current camera radius
         */
        customComputeDeltaFromMouseWheel: Nullable<(wheelDelta: number, input: ArcRotateCameraMouseWheelInput, event: IWheelEvent) => number>;
        private _wheel;
        private _observer;
        private _hitPlane;
        private _viewOffset;
        private _globalOffset;
        protected _computeDeltaFromMouseWheelLegacyEvent(mouseWheelDelta: number, radius: number): number;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        private _updateHitPlane;
        private _getPosition;
        private _inertialPanning;
        private _zoomToMouse;
        private _zeroIfClose;
    }


    /**
     * Manage the pointers inputs to control an arc rotate camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class ArcRotateCameraPointersInput extends BaseCameraPointersInput {
        /**
         * Defines the camera the input is attached to.
         */
        camera: ArcRotateCamera;
        /**
         * The minimum radius used for pinch, to avoid radius lock at 0
         */
        static MinimumRadiusForPinch: number;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Defines the buttons associated with the input to handle camera move.
         */
        buttons: number[];
        /**
         * Defines the pointer angular sensibility  along the X axis or how fast is
         * the camera rotating.
         */
        angularSensibilityX: number;
        /**
         * Defines the pointer angular sensibility along the Y axis or how fast is
         * the camera rotating.
         */
        angularSensibilityY: number;
        /**
         * Defines the pointer pinch precision or how fast is the camera zooming.
         */
        pinchPrecision: number;
        /**
         * pinchDeltaPercentage will be used instead of pinchPrecision if different
         * from 0.
         * It defines the percentage of current camera.radius to use as delta when
         * pinch zoom is used.
         */
        pinchDeltaPercentage: number;
        /**
         * When useNaturalPinchZoom is true, multi touch zoom will zoom in such
         * that any object in the plane at the camera's target point will scale
         * perfectly with finger motion.
         * Overrides pinchDeltaPercentage and pinchPrecision.
         */
        useNaturalPinchZoom: boolean;
        /**
         * Defines whether zoom (2 fingers pinch) is enabled through multitouch
         */
        pinchZoom: boolean;
        /**
         * Defines the pointer panning sensibility or how fast is the camera moving.
         */
        panningSensibility: number;
        /**
         * Defines whether panning (2 fingers swipe) is enabled through multitouch.
         */
        multiTouchPanning: boolean;
        /**
         * Defines whether panning is enabled for both pan (2 fingers swipe) and
         * zoom (pinch) through multitouch.
         */
        multiTouchPanAndZoom: boolean;
        /**
         * Revers pinch action direction.
         */
        pinchInwards: boolean;
        private _isPanClick;
        private _twoFingerActivityCount;
        private _isPinching;
        /**
         * Move camera from multi touch panning positions.
         * @param previousMultiTouchPanPosition
         * @param multiTouchPanPosition
         */
        private _computeMultiTouchPanning;
        /**
         * Move camera from pinch zoom distances.
         * @param previousPinchSquaredDistance
         * @param pinchSquaredDistance
         */
        private _computePinchZoom;
        /**
         * Called on pointer POINTERMOVE event if only a single touch is active.
         * @param point current touch point
         * @param offsetX offset on X
         * @param offsetY offset on Y
         */
        onTouch(point: Nullable<PointerTouch>, offsetX: number, offsetY: number): void;
        /**
         * Called on pointer POINTERDOUBLETAP event.
         */
        onDoubleTap(): void;
        /**
         * Called on pointer POINTERMOVE event if multiple touches are active.
         * @param pointA point A
         * @param pointB point B
         * @param previousPinchSquaredDistance distance between points in previous pinch
         * @param pinchSquaredDistance distance between points in current pinch
         * @param previousMultiTouchPanPosition multi-touch position in previous step
         * @param multiTouchPanPosition multi-touch position in current step
         */
        onMultiTouch(pointA: Nullable<PointerTouch>, pointB: Nullable<PointerTouch>, previousPinchSquaredDistance: number, pinchSquaredDistance: number, previousMultiTouchPanPosition: Nullable<PointerTouch>, multiTouchPanPosition: Nullable<PointerTouch>): void;
        /**
         * Called each time a new POINTERDOWN event occurs. Ie, for each button
         * press.
         * @param evt Defines the event to track
         */
        onButtonDown(evt: IPointerEvent): void;
        /**
         * Called each time a new POINTERUP event occurs. Ie, for each button
         * release.
         * @param _evt Defines the event to track
         */
        onButtonUp(_evt: IPointerEvent): void;
        /**
         * Called when window becomes inactive.
         */
        onLostFocus(): void;
    }


        interface ArcRotateCameraInputsManager {
            /**
             * Add orientation input support to the input manager.
             * @returns the current input manager
             */
            addVRDeviceOrientation(): ArcRotateCameraInputsManager;
        }
    /**
     * Manage the device orientation inputs (gyroscope) to control an arc rotate camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class ArcRotateCameraVRDeviceOrientationInput implements ICameraInput<ArcRotateCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: ArcRotateCamera;
        /**
         * Defines a correction factor applied on the alpha value retrieved from the orientation events.
         */
        alphaCorrection: number;
        /**
         * Defines a correction factor applied on the gamma value retrieved from the orientation events.
         */
        gammaCorrection: number;
        private _alpha;
        private _gamma;
        private _dirty;
        private _deviceOrientationHandler;
        /**
         * Instantiate a new ArcRotateCameraVRDeviceOrientationInput.
         */
        constructor();
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * @internal
         */
        _onOrientationEvent(evt: DeviceOrientationEvent): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Base class for mouse wheel input..
     * See FollowCameraMouseWheelInput in src/Cameras/Inputs/freeCameraMouseWheelInput.ts
     * for example usage.
     */
    export abstract class BaseCameraMouseWheelInput implements ICameraInput<Camera> {
        /**
         * Defines the camera the input is attached to.
         */
        abstract camera: Camera;
        /**
         * How fast is the camera moves in relation to X axis mouseWheel events.
         * Use negative value to reverse direction.
         */
        wheelPrecisionX: number;
        /**
         * How fast is the camera moves in relation to Y axis mouseWheel events.
         * Use negative value to reverse direction.
         */
        wheelPrecisionY: number;
        /**
         * How fast is the camera moves in relation to Z axis mouseWheel events.
         * Use negative value to reverse direction.
         */
        wheelPrecisionZ: number;
        /**
         * Observable for when a mouse wheel move event occurs.
         */
        onChangedObservable: Observable<{
            wheelDeltaX: number;
            wheelDeltaY: number;
            wheelDeltaZ: number;
        }>;
        private _wheel;
        private _observer;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls
         *   should call preventdefault().
         *   (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Called for each rendered frame.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        /**
         * Incremental value of multiple mouse wheel movements of the X axis.
         * Should be zero-ed when read.
         */
        protected _wheelDeltaX: number;
        /**
         * Incremental value of multiple mouse wheel movements of the Y axis.
         * Should be zero-ed when read.
         */
        protected _wheelDeltaY: number;
        /**
         * Incremental value of multiple mouse wheel movements of the Z axis.
         * Should be zero-ed when read.
         */
        protected _wheelDeltaZ: number;
        /**
         * Firefox uses a different scheme to report scroll distances to other
         * browsers. Rather than use complicated methods to calculate the exact
         * multiple we need to apply, let's just cheat and use a constant.
         * https://developer.mozilla.org/en-US/docs/Web/API/WheelEvent/deltaMode
         * https://stackoverflow.com/questions/20110224/what-is-the-height-of-a-line-in-a-wheel-event-deltamode-dom-delta-line
         */
        private readonly _ffMultiplier;
        /**
         * Different event attributes for wheel data fall into a few set ranges.
         * Some relevant but dated date here:
         * https://stackoverflow.com/questions/5527601/normalizing-mousewheel-speed-across-browsers
         */
        private readonly _normalize;
    }


    /**
     * Base class for Camera Pointer Inputs.
     * See FollowCameraPointersInput in src/Cameras/Inputs/followCameraPointersInput.ts
     * for example usage.
     */
    export abstract class BaseCameraPointersInput implements ICameraInput<Camera> {
        /**
         * Defines the camera the input is attached to.
         */
        abstract camera: Camera;
        /**
         * Whether keyboard modifier keys are pressed at time of last mouse event.
         */
        protected _altKey: boolean;
        protected _ctrlKey: boolean;
        protected _metaKey: boolean;
        protected _shiftKey: boolean;
        /**
         * Which mouse buttons were pressed at time of last mouse event.
         * https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/buttons
         */
        protected _buttonsPressed: number;
        private _currentActiveButton;
        private _contextMenuBind;
        /**
         * Defines the buttons associated with the input to handle camera move.
         */
        buttons: number[];
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        /**
         * Called on pointer POINTERDOUBLETAP event.
         * Override this method to provide functionality on POINTERDOUBLETAP event.
         * @param type type of event
         */
        onDoubleTap(type: string): void;
        /**
         * Called on pointer POINTERMOVE event if only a single touch is active.
         * Override this method to provide functionality.
         * @param point The current position of the pointer
         * @param offsetX The offsetX of the pointer when the event occurred
         * @param offsetY The offsetY of the pointer when the event occurred
         */
        onTouch(point: Nullable<PointerTouch>, offsetX: number, offsetY: number): void;
        /**
         * Called on pointer POINTERMOVE event if multiple touches are active.
         * Override this method to provide functionality.
         * @param _pointA First point in the pair
         * @param _pointB Second point in the pair
         * @param previousPinchSquaredDistance Sqr Distance between the points the last time this event was fired (by this input)
         * @param pinchSquaredDistance Sqr Distance between the points this time
         * @param previousMultiTouchPanPosition Previous center point between the points
         * @param multiTouchPanPosition Current center point between the points
         */
        onMultiTouch(_pointA: Nullable<PointerTouch>, _pointB: Nullable<PointerTouch>, previousPinchSquaredDistance: number, pinchSquaredDistance: number, previousMultiTouchPanPosition: Nullable<PointerTouch>, multiTouchPanPosition: Nullable<PointerTouch>): void;
        /**
         * Called on JS contextmenu event.
         * Override this method to provide functionality.
         * @param evt the event to be handled
         */
        onContextMenu(evt: PointerEvent): void;
        /**
         * Called each time a new POINTERDOWN event occurs. Ie, for each button
         * press.
         * Override this method to provide functionality.
         * @param _evt Defines the event to track
         */
        onButtonDown(_evt: IPointerEvent): void;
        /**
         * Called each time a new POINTERUP event occurs. Ie, for each button
         * release.
         * Override this method to provide functionality.
         * @param _evt Defines the event to track
         */
        onButtonUp(_evt: IPointerEvent): void;
        /**
         * Called when window becomes inactive.
         * Override this method to provide functionality.
         */
        onLostFocus(): void;
        private _pointerInput;
        private _observer;
        private _onLostFocus;
        private _pointA;
        private _pointB;
    }


    /**
     * Listen to keyboard events to control the camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FlyCameraKeyboardInput implements ICameraInput<FlyCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FlyCamera;
        /**
         * The list of keyboard keys used to control the forward move of the camera.
         */
        keysForward: number[];
        /**
         * The list of keyboard keys used to control the backward move of the camera.
         */
        keysBackward: number[];
        /**
         * The list of keyboard keys used to control the forward move of the camera.
         */
        keysUp: number[];
        /**
         * The list of keyboard keys used to control the backward move of the camera.
         */
        keysDown: number[];
        /**
         * The list of keyboard keys used to control the right strafe move of the camera.
         */
        keysRight: number[];
        /**
         * The list of keyboard keys used to control the left strafe move of the camera.
         */
        keysLeft: number[];
        private _keys;
        private _onCanvasBlurObserver;
        private _onKeyboardObserver;
        private _engine;
        private _scene;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * @internal
         */
        _onLostFocus(): void;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
    }


    /**
     * Listen to mouse events to control the camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FlyCameraMouseInput implements ICameraInput<FlyCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FlyCamera;
        /**
         * Defines if touch is enabled. (Default is true.)
         */
        touchEnabled: boolean;
        /**
         * Defines the buttons associated with the input to handle camera rotation.
         */
        buttons: number[];
        /**
         * Assign buttons for Yaw control.
         */
        buttonsYaw: number[];
        /**
         * Assign buttons for Pitch control.
         */
        buttonsPitch: number[];
        /**
         * Assign buttons for Roll control.
         */
        buttonsRoll: number[];
        /**
         * Detect if any button is being pressed while mouse is moved.
         * -1 = Mouse locked.
         * 0 = Left button.
         * 1 = Middle Button.
         * 2 = Right Button.
         */
        activeButton: number;
        /**
         * Defines the pointer's angular sensibility, to control the camera rotation speed.
         * Higher values reduce its sensitivity.
         */
        angularSensibility: number;
        private _observer;
        private _rollObserver;
        private _previousPosition;
        private _noPreventDefault;
        private _element;
        /**
         * Listen to mouse events to control the camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
         */
        constructor();
        /**
         * Attach the mouse control to the HTML DOM element.
         * @param noPreventDefault Defines whether events caught by the controls should call preventdefault().
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name.
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input's friendly name.
         */
        getSimpleName(): string;
        private _pointerInput;
        private _onMouseMove;
        /**
         * Rotate camera by mouse offset.
         * @param offsetX
         * @param offsetY
         */
        private _rotateCamera;
    }


    /**
     * Manage the keyboard inputs to control the movement of a follow camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FollowCameraKeyboardMoveInput implements ICameraInput<FollowCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FollowCamera;
        /**
         * Defines the list of key codes associated with the up action (increase heightOffset)
         */
        keysHeightOffsetIncr: number[];
        /**
         * Defines the list of key codes associated with the down action (decrease heightOffset)
         */
        keysHeightOffsetDecr: number[];
        /**
         * Defines whether the Alt modifier key is required to move up/down (alter heightOffset)
         */
        keysHeightOffsetModifierAlt: boolean;
        /**
         * Defines whether the Ctrl modifier key is required to move up/down (alter heightOffset)
         */
        keysHeightOffsetModifierCtrl: boolean;
        /**
         * Defines whether the Shift modifier key is required to move up/down (alter heightOffset)
         */
        keysHeightOffsetModifierShift: boolean;
        /**
         * Defines the list of key codes associated with the left action (increase rotationOffset)
         */
        keysRotationOffsetIncr: number[];
        /**
         * Defines the list of key codes associated with the right action (decrease rotationOffset)
         */
        keysRotationOffsetDecr: number[];
        /**
         * Defines whether the Alt modifier key is required to move left/right (alter rotationOffset)
         */
        keysRotationOffsetModifierAlt: boolean;
        /**
         * Defines whether the Ctrl modifier key is required to move left/right (alter rotationOffset)
         */
        keysRotationOffsetModifierCtrl: boolean;
        /**
         * Defines whether the Shift modifier key is required to move left/right (alter rotationOffset)
         */
        keysRotationOffsetModifierShift: boolean;
        /**
         * Defines the list of key codes associated with the zoom-in action (decrease radius)
         */
        keysRadiusIncr: number[];
        /**
         * Defines the list of key codes associated with the zoom-out action (increase radius)
         */
        keysRadiusDecr: number[];
        /**
         * Defines whether the Alt modifier key is required to zoom in/out (alter radius value)
         */
        keysRadiusModifierAlt: boolean;
        /**
         * Defines whether the Ctrl modifier key is required to zoom in/out (alter radius value)
         */
        keysRadiusModifierCtrl: boolean;
        /**
         * Defines whether the Shift modifier key is required to zoom in/out (alter radius value)
         */
        keysRadiusModifierShift: boolean;
        /**
         * Defines the rate of change of heightOffset.
         */
        heightSensibility: number;
        /**
         * Defines the rate of change of rotationOffset.
         */
        rotationSensibility: number;
        /**
         * Defines the rate of change of radius.
         */
        radiusSensibility: number;
        private _keys;
        private _ctrlPressed;
        private _altPressed;
        private _shiftPressed;
        private _onCanvasBlurObserver;
        private _onKeyboardObserver;
        private _engine;
        private _scene;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        /**
         * Check if the pressed modifier keys (Alt/Ctrl/Shift) match those configured to
         * allow modification of the heightOffset value.
         * @returns true if modifier keys match
         */
        private _modifierHeightOffset;
        /**
         * Check if the pressed modifier keys (Alt/Ctrl/Shift) match those configured to
         * allow modification of the rotationOffset value.
         * @returns true if modifier keys match
         */
        private _modifierRotationOffset;
        /**
         * Check if the pressed modifier keys (Alt/Ctrl/Shift) match those configured to
         * allow modification of the radius value.
         * @returns true if modifier keys match
         */
        private _modifierRadius;
    }


    /**
     * Manage the mouse wheel inputs to control a follow camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FollowCameraMouseWheelInput implements ICameraInput<FollowCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FollowCamera;
        /**
         * Moue wheel controls zoom. (Mouse wheel modifies camera.radius value.)
         */
        axisControlRadius: boolean;
        /**
         * Moue wheel controls height. (Mouse wheel modifies camera.heightOffset value.)
         */
        axisControlHeight: boolean;
        /**
         * Moue wheel controls angle. (Mouse wheel modifies camera.rotationOffset value.)
         */
        axisControlRotation: boolean;
        /**
         * Gets or Set the mouse wheel precision or how fast is the camera moves in
         * relation to mouseWheel events.
         */
        wheelPrecision: number;
        /**
         * wheelDeltaPercentage will be used instead of wheelPrecision if different from 0.
         * It defines the percentage of current camera.radius to use as delta when wheel is used.
         */
        wheelDeltaPercentage: number;
        private _wheel;
        private _observer;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Manage the pointers inputs to control an follow camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FollowCameraPointersInput extends BaseCameraPointersInput {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FollowCamera;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Defines the pointer angular sensibility along the X axis or how fast is
         * the camera rotating.
         * A negative number will reverse the axis direction.
         */
        angularSensibilityX: number;
        /**
         * Defines the pointer angular sensibility along the Y axis or how fast is
         * the camera rotating.
         * A negative number will reverse the axis direction.
         */
        angularSensibilityY: number;
        /**
         * Defines the pointer pinch precision or how fast is the camera zooming.
         * A negative number will reverse the axis direction.
         */
        pinchPrecision: number;
        /**
         * pinchDeltaPercentage will be used instead of pinchPrecision if different
         * from 0.
         * It defines the percentage of current camera.radius to use as delta when
         * pinch zoom is used.
         */
        pinchDeltaPercentage: number;
        /**
         * Pointer X axis controls zoom. (X axis modifies camera.radius value.)
         */
        axisXControlRadius: boolean;
        /**
         * Pointer X axis controls height. (X axis modifies camera.heightOffset value.)
         */
        axisXControlHeight: boolean;
        /**
         * Pointer X axis controls angle. (X axis modifies camera.rotationOffset value.)
         */
        axisXControlRotation: boolean;
        /**
         * Pointer Y axis controls zoom. (Y axis modifies camera.radius value.)
         */
        axisYControlRadius: boolean;
        /**
         * Pointer Y axis controls height. (Y axis modifies camera.heightOffset value.)
         */
        axisYControlHeight: boolean;
        /**
         * Pointer Y axis controls angle. (Y axis modifies camera.rotationOffset value.)
         */
        axisYControlRotation: boolean;
        /**
         * Pinch controls zoom. (Pinch modifies camera.radius value.)
         */
        axisPinchControlRadius: boolean;
        /**
         * Pinch controls height. (Pinch modifies camera.heightOffset value.)
         */
        axisPinchControlHeight: boolean;
        /**
         * Pinch controls angle. (Pinch modifies camera.rotationOffset value.)
         */
        axisPinchControlRotation: boolean;
        /**
         * Log error messages if basic misconfiguration has occurred.
         */
        warningEnable: boolean;
        onTouch(pointA: Nullable<PointerTouch>, offsetX: number, offsetY: number): void;
        onMultiTouch(pointA: Nullable<PointerTouch>, pointB: Nullable<PointerTouch>, previousPinchSquaredDistance: number, pinchSquaredDistance: number, previousMultiTouchPanPosition: Nullable<PointerTouch>, multiTouchPanPosition: Nullable<PointerTouch>): void;
        private _warningCounter;
        private _warning;
    }


        interface FreeCameraInputsManager {
            /**
             * @internal
             */
            _deviceOrientationInput: Nullable<FreeCameraDeviceOrientationInput>;
            /**
             * Add orientation input support to the input manager.
             * @param smoothFactor deviceOrientation smoothing. 0: no smoothing, 1: new data ignored, 0.9 recommended for smoothing
             * @returns the current input manager
             */
            addDeviceOrientation(smoothFactor?: number): FreeCameraInputsManager;
        }
    /**
     * Takes information about the orientation of the device as reported by the deviceorientation event to orient the camera.
     * Screen rotation is taken into account.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraDeviceOrientationInput implements ICameraInput<FreeCamera> {
        private _camera;
        private _screenOrientationAngle;
        private _constantTranform;
        private _screenQuaternion;
        private _alpha;
        private _beta;
        private _gamma;
        /** alpha+beta+gamma smoothing. 0: no smoothing, 1: new data ignored, 0.9 recommended for smoothing */
        smoothFactor: number;
        /**
         * Can be used to detect if a device orientation sensor is available on a device
         * @param timeout amount of time in milliseconds to wait for a response from the sensor (default: infinite)
         * @returns a promise that will resolve on orientation change
         */
        static WaitForOrientationChangeAsync(timeout?: number): Promise<void>;
        /**
         * @internal
         */
        _onDeviceOrientationChangedObservable: Observable<void>;
        /**
         * Instantiates a new input
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
         */
        constructor();
        /**
         * Define the camera controlled by the input.
         */
        get camera(): FreeCamera;
        set camera(camera: FreeCamera);
        /**
         * Attach the input controls to a specific dom element to get the input from.
         */
        attachControl(): void;
        private _orientationChanged;
        private _deviceOrientation;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Manage the gamepad inputs to control a free camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraGamepadInput implements ICameraInput<FreeCamera> {
        /**
         * Define the camera the input is attached to.
         */
        camera: FreeCamera;
        /**
         * Define the Gamepad controlling the input
         */
        gamepad: Nullable<Gamepad>;
        /**
         * Defines the gamepad rotation sensibility.
         * This is the threshold from when rotation starts to be accounted for to prevent jittering.
         */
        gamepadAngularSensibility: number;
        /**
         * Defines the gamepad move sensibility.
         * This is the threshold from when moving starts to be accounted for for to prevent jittering.
         */
        gamepadMoveSensibility: number;
        /**
         * Defines the minimum value at which any analog stick input is ignored.
         * Note: This value should only be a value between 0 and 1.
         */
        deadzoneDelta: number;
        private _yAxisScale;
        /**
         * Gets or sets a boolean indicating that Yaxis (for right stick) should be inverted
         */
        get invertYAxis(): boolean;
        set invertYAxis(value: boolean);
        private _onGamepadConnectedObserver;
        private _onGamepadDisconnectedObserver;
        private _cameraTransform;
        private _deltaTransform;
        private _vector3;
        private _vector2;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         */
        attachControl(): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Manage the keyboard inputs to control the movement of a free camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraKeyboardMoveInput implements ICameraInput<FreeCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FreeCamera;
        /**
         * Gets or Set the list of keyboard keys used to control the forward move of the camera.
         */
        keysUp: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the upward move of the camera.
         */
        keysUpward: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the backward move of the camera.
         */
        keysDown: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the downward move of the camera.
         */
        keysDownward: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the left strafe move of the camera.
         */
        keysLeft: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the right strafe move of the camera.
         */
        keysRight: number[];
        /**
         * Defines the pointer angular sensibility  along the X and Y axis or how fast is the camera rotating.
         */
        rotationSpeed: number;
        /**
         * Gets or Set the list of keyboard keys used to control the left rotation move of the camera.
         */
        keysRotateLeft: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the right rotation move of the camera.
         */
        keysRotateRight: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the up rotation move of the camera.
         */
        keysRotateUp: number[];
        /**
         * Gets or Set the list of keyboard keys used to control the down rotation move of the camera.
         */
        keysRotateDown: number[];
        private _keys;
        private _onCanvasBlurObserver;
        private _onKeyboardObserver;
        private _engine;
        private _scene;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /** @internal */
        _onLostFocus(): void;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
        private _getLocalRotation;
    }


    /**
     * Manage the mouse inputs to control the movement of a free camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraMouseInput implements ICameraInput<FreeCamera> {
        /**
         * Define if touch is enabled in the mouse input
         */
        touchEnabled: boolean;
        /**
         * Defines the camera the input is attached to.
         */
        camera: FreeCamera;
        /**
         * Defines the buttons associated with the input to handle camera move.
         */
        buttons: number[];
        /**
         * Defines the pointer angular sensibility  along the X and Y axis or how fast is the camera rotating.
         */
        angularSensibility: number;
        private _pointerInput;
        private _onMouseMove;
        private _observer;
        private _previousPosition;
        /**
         * Observable for when a pointer move event occurs containing the move offset
         */
        onPointerMovedObservable: Observable<{
            offsetX: number;
            offsetY: number;
        }>;
        /**
         * @internal
         * If the camera should be rotated automatically based on pointer movement
         */
        _allowCameraRotation: boolean;
        private _currentActiveButton;
        private _activePointerId;
        private _contextMenuBind;
        /**
         * Manage the mouse inputs to control the movement of a free camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
         * @param touchEnabled Defines if touch is enabled or not
         */
        constructor(
        /**
         * Define if touch is enabled in the mouse input
         */
        touchEnabled?: boolean);
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Called on JS contextmenu event.
         * Override this method to provide functionality.
         * @param evt the context menu event
         */
        onContextMenu(evt: PointerEvent): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


    /**
     * Manage the mouse wheel inputs to control a free camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraMouseWheelInput extends BaseCameraMouseWheelInput {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FreeCamera;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Set which movement axis (relative to camera's orientation) the mouse
         * wheel's X axis controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelXMoveRelative(axis: Nullable<Coordinate>);
        /**
         * Get the configured movement axis (relative to camera's orientation) the
         * mouse wheel's X axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelXMoveRelative(): Nullable<Coordinate>;
        /**
         * Set which movement axis (relative to camera's orientation) the mouse
         * wheel's Y axis controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelYMoveRelative(axis: Nullable<Coordinate>);
        /**
         * Get the configured movement axis (relative to camera's orientation) the
         * mouse wheel's Y axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelYMoveRelative(): Nullable<Coordinate>;
        /**
         * Set which movement axis (relative to camera's orientation) the mouse
         * wheel's Z axis controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelZMoveRelative(axis: Nullable<Coordinate>);
        /**
         * Get the configured movement axis (relative to camera's orientation) the
         * mouse wheel's Z axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelZMoveRelative(): Nullable<Coordinate>;
        /**
         * Set which rotation axis (relative to camera's orientation) the mouse
         * wheel's X axis controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelXRotateRelative(axis: Nullable<Coordinate>);
        /**
         * Get the configured rotation axis (relative to camera's orientation) the
         * mouse wheel's X axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelXRotateRelative(): Nullable<Coordinate>;
        /**
         * Set which rotation axis (relative to camera's orientation) the mouse
         * wheel's Y axis controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelYRotateRelative(axis: Nullable<Coordinate>);
        /**
         * Get the configured rotation axis (relative to camera's orientation) the
         * mouse wheel's Y axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelYRotateRelative(): Nullable<Coordinate>;
        /**
         * Set which rotation axis (relative to camera's orientation) the mouse
         * wheel's Z axis controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelZRotateRelative(axis: Nullable<Coordinate>);
        /**
         * Get the configured rotation axis (relative to camera's orientation) the
         * mouse wheel's Z axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelZRotateRelative(): Nullable<Coordinate>;
        /**
         * Set which movement axis (relative to the scene) the mouse wheel's X axis
         * controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelXMoveScene(axis: Nullable<Coordinate>);
        /**
         * Get the configured movement axis (relative to the scene) the mouse wheel's
         * X axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelXMoveScene(): Nullable<Coordinate>;
        /**
         * Set which movement axis (relative to the scene) the mouse wheel's Y axis
         * controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelYMoveScene(axis: Nullable<Coordinate>);
        /**
         * Get the configured movement axis (relative to the scene) the mouse wheel's
         * Y axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelYMoveScene(): Nullable<Coordinate>;
        /**
         * Set which movement axis (relative to the scene) the mouse wheel's Z axis
         * controls.
         * @param axis The axis to be moved. Set null to clear.
         */
        set wheelZMoveScene(axis: Nullable<Coordinate>);
        /**
         * Get the configured movement axis (relative to the scene) the mouse wheel's
         * Z axis controls.
         * @returns The configured axis or null if none.
         */
        get wheelZMoveScene(): Nullable<Coordinate>;
        /**
         * Called for each rendered frame.
         */
        checkInputs(): void;
        private _moveRelative;
        private _rotateRelative;
        private _moveScene;
        /**
         * These are set to the desired default behaviour.
         */
        private _wheelXAction;
        private _wheelXActionCoordinate;
        private _wheelYAction;
        private _wheelYActionCoordinate;
        private _wheelZAction;
        private _wheelZActionCoordinate;
        /**
         * Update the camera according to any configured properties for the 3
         * mouse-wheel axis.
         */
        private _updateCamera;
        /**
         * Update one property of the camera.
         * @param value
         * @param cameraProperty
         * @param coordinate
         */
        private _updateCameraProperty;
    }


    /**
     * Manage the touch inputs to control the movement of a free camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraTouchInput implements ICameraInput<FreeCamera> {
        /**
         * Define if mouse events can be treated as touch events
         */
        allowMouse: boolean;
        /**
         * Defines the camera the input is attached to.
         */
        camera: FreeCamera;
        /**
         * Defines the touch sensibility for rotation.
         * The lower the faster.
         */
        touchAngularSensibility: number;
        /**
         * Defines the touch sensibility for move.
         * The lower the faster.
         */
        touchMoveSensibility: number;
        /**
         * Swap touch actions so that one touch is used for rotation and multiple for movement
         */
        singleFingerRotate: boolean;
        private _offsetX;
        private _offsetY;
        private _pointerPressed;
        private _pointerInput?;
        private _observer;
        private _onLostFocus;
        private _isSafari;
        /**
         * Manage the touch inputs to control the movement of a free camera.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
         * @param allowMouse Defines if mouse events can be treated as touch events
         */
        constructor(
        /**
         * Define if mouse events can be treated as touch events
         */
        allowMouse?: boolean);
        /**
         * Attach the input controls to a specific dom element to get the input from.
         * @param noPreventDefault Defines whether event caught by the controls should call preventdefault() (https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
         */
        attachControl(noPreventDefault?: boolean): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }


        interface FreeCameraInputsManager {
            /**
             * Add virtual joystick input support to the input manager.
             * @returns the current input manager
             */
            addVirtualJoystick(): FreeCameraInputsManager;
        }
    /**
     * Manage the Virtual Joystick inputs to control the movement of a free camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/customizingCameraInputs
     */
    export class FreeCameraVirtualJoystickInput implements ICameraInput<FreeCamera> {
        /**
         * Defines the camera the input is attached to.
         */
        camera: FreeCamera;
        private _leftjoystick;
        private _rightjoystick;
        /**
         * Gets the left stick of the virtual joystick.
         * @returns The virtual Joystick
         */
        getLeftJoystick(): VirtualJoystick;
        /**
         * Gets the right stick of the virtual joystick.
         * @returns The virtual Joystick
         */
        getRightJoystick(): VirtualJoystick;
        /**
         * Update the current camera state depending on the inputs that have been used this frame.
         * This is a dynamically created lambda to avoid the performance penalty of looping for inputs in the render loop.
         */
        checkInputs(): void;
        /**
         * Attach the input controls to a specific dom element to get the input from.
         */
        attachControl(): void;
        /**
         * Detach the current controls from the specified dom element.
         */
        detachControl(): void;
        /**
         * Gets the class name of the current input.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Get the friendly name associated with the input class.
         * @returns the input friendly name
         */
        getSimpleName(): string;
    }






    /**
     * @internal
     */
    export function setStereoscopicAnaglyphRigMode(camera: Camera): void;


    /**
     * @internal
     */
    export function setStereoscopicRigMode(camera: Camera): void;


    /**
     * @internal
     */
    export function setVRRigMode(camera: Camera, rigParams: any): void;


    /**
     * Camera used to simulate anaglyphic rendering (based on ArcRotateCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#anaglyph-cameras
     */
    export class AnaglyphArcRotateCamera extends ArcRotateCamera {
        /**
         * Creates a new AnaglyphArcRotateCamera
         * @param name defines camera name
         * @param alpha defines alpha angle (in radians)
         * @param beta defines beta angle (in radians)
         * @param radius defines radius
         * @param target defines camera target
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, interaxialDistance: number, scene?: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphArcRotateCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * Camera used to simulate anaglyphic rendering (based on FreeCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#anaglyph-cameras
     */
    export class AnaglyphFreeCamera extends FreeCamera {
        /**
         * Creates a new AnaglyphFreeCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, scene?: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphFreeCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * Camera used to simulate anaglyphic rendering (based on GamepadCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#anaglyph-cameras
     */
    export class AnaglyphGamepadCamera extends GamepadCamera {
        /**
         * Creates a new AnaglyphGamepadCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, scene?: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphGamepadCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * Camera used to simulate anaglyphic rendering (based on UniversalCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#anaglyph-cameras
     */
    export class AnaglyphUniversalCamera extends UniversalCamera {
        /**
         * Creates a new AnaglyphUniversalCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, scene?: Scene);
        /**
         * Gets camera class name
         * @returns AnaglyphUniversalCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }




    /**
     * Camera used to simulate stereoscopic rendering (based on ArcRotateCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class StereoscopicArcRotateCamera extends ArcRotateCamera {
        /**
         * Creates a new StereoscopicArcRotateCamera
         * @param name defines camera name
         * @param alpha defines alpha angle (in radians)
         * @param beta defines beta angle (in radians)
         * @param radius defines radius
         * @param target defines camera target
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene?: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicArcRotateCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * Camera used to simulate stereoscopic rendering (based on FreeCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class StereoscopicFreeCamera extends FreeCamera {
        /**
         * Creates a new StereoscopicFreeCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene?: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicFreeCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * Camera used to simulate stereoscopic rendering (based on GamepadCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class StereoscopicGamepadCamera extends GamepadCamera {
        /**
         * Creates a new StereoscopicGamepadCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene?: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicGamepadCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * Camera used to simulate stereoscopic rendering on real screens (based on UniversalCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class StereoscopicScreenUniversalCamera extends UniversalCamera {
        private _distanceToProjectionPlane;
        private _distanceBetweenEyes;
        set distanceBetweenEyes(newValue: number);
        /**
         * distance between the eyes
         */
        get distanceBetweenEyes(): number;
        set distanceToProjectionPlane(newValue: number);
        /**
         * Distance to projection plane (should be the same units the like distance between the eyes)
         */
        get distanceToProjectionPlane(): number;
        /**
         * Creates a new StereoscopicScreenUniversalCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param scene defines the hosting scene
         * @param distanceToProjectionPlane defines distance between each color axis. The rig cameras will receive this as their negative z position!
         * @param distanceBetweenEyes defines is stereoscopic is done side by side or over under
         */
        constructor(name: string, position: Vector3, scene?: Scene, distanceToProjectionPlane?: number, distanceBetweenEyes?: number);
        /**
         * Gets camera class name
         * @returns StereoscopicScreenUniversalCamera
         */
        getClassName(): string;
        /**
         * @internal
         */
        createRigCamera(name: string): Nullable<Camera>;
        /**
         * @internal
         */
        _updateRigCameras(): void;
        private _updateCamera;
        protected _setRigMode(): void;
    }


    /**
     * Camera used to simulate stereoscopic rendering (based on UniversalCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class StereoscopicUniversalCamera extends UniversalCamera {
        /**
         * Creates a new StereoscopicUniversalCamera
         * @param name defines camera name
         * @param position defines initial position
         * @param interaxialDistance defines distance between each color axis
         * @param isStereoscopicSideBySide defines is stereoscopic is done side by side or over under
         * @param scene defines the hosting scene
         */
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene?: Scene);
        /**
         * Gets camera class name
         * @returns StereoscopicUniversalCamera
         */
        getClassName(): string;
        protected _setRigMode: () => void;
    }


    /**
     * A target camera takes a mesh or position as a target and continues to look at it while it moves.
     * This is the base of the follow, arc rotate cameras and Free camera
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
     */
    export class TargetCamera extends Camera {
        private static _RigCamTransformMatrix;
        private static _TargetTransformMatrix;
        private static _TargetFocalPoint;
        private _tmpUpVector;
        private _tmpTargetVector;
        /**
         * Define the current direction the camera is moving to
         */
        cameraDirection: Vector3;
        /**
         * Define the current rotation the camera is rotating to
         */
        cameraRotation: Vector2;
        /** Gets or sets a boolean indicating that the scaling of the parent hierarchy will not be taken in account by the camera */
        ignoreParentScaling: boolean;
        /**
         * When set, the up vector of the camera will be updated by the rotation of the camera
         */
        updateUpVectorFromRotation: boolean;
        private _tmpQuaternion;
        /**
         * Define the current rotation of the camera
         */
        rotation: Vector3;
        /**
         * Define the current rotation of the camera as a quaternion to prevent Gimbal lock
         */
        rotationQuaternion: Quaternion;
        /**
         * Define the current speed of the camera
         */
        speed: number;
        /**
         * Add constraint to the camera to prevent it to move freely in all directions and
         * around all axis.
         */
        noRotationConstraint: boolean;
        /**
         * Reverses mouselook direction to 'natural' panning as opposed to traditional direct
         * panning
         */
        invertRotation: boolean;
        /**
         * Speed multiplier for inverse camera panning
         */
        inverseRotationSpeed: number;
        /**
         * Define the current target of the camera as an object or a position.
         * Please note that locking a target will disable panning.
         */
        lockedTarget: any;
        /** @internal */
        _currentTarget: Vector3;
        /** @internal */
        _initialFocalDistance: number;
        /** @internal */
        _viewMatrix: Matrix;
        /** @internal */
        _camMatrix: Matrix;
        /** @internal */
        _cameraTransformMatrix: Matrix;
        /** @internal */
        _cameraRotationMatrix: Matrix;
        /** @internal */
        _referencePoint: Vector3;
        /** @internal */
        _transformedReferencePoint: Vector3;
        protected _deferredPositionUpdate: Vector3;
        protected _deferredRotationQuaternionUpdate: Quaternion;
        protected _deferredRotationUpdate: Vector3;
        protected _deferredUpdated: boolean;
        protected _deferOnly: boolean;
        /** @internal */
        _reset: () => void;
        private _defaultUp;
        /**
         * Instantiates a target camera that takes a mesh or position as a target and continues to look at it while it moves.
         * This is the base of the follow, arc rotate cameras and Free camera
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras
         * @param name Defines the name of the camera in the scene
         * @param position Defines the start position of the camera in the scene
         * @param scene Defines the scene the camera belongs to
         * @param setActiveOnSceneIfNoneActive Defines whether the camera should be marked as active if not other active cameras have been defined
         */
        constructor(name: string, position: Vector3, scene?: Scene, setActiveOnSceneIfNoneActive?: boolean);
        /**
         * Gets the position in front of the camera at a given distance.
         * @param distance The distance from the camera we want the position to be
         * @returns the position
         */
        getFrontPosition(distance: number): Vector3;
        /** @internal */
        _getLockedTargetPosition(): Nullable<Vector3>;
        private _storedPosition;
        private _storedRotation;
        private _storedRotationQuaternion;
        /**
         * Store current camera state of the camera (fov, position, rotation, etc..)
         * @returns the camera
         */
        storeState(): Camera;
        /**
         * Restored camera state. You must call storeState() first
         * @returns whether it was successful or not
         * @internal
         */
        _restoreStateValues(): boolean;
        /** @internal */
        _initCache(): void;
        /**
         * @internal
         */
        _updateCache(ignoreParentClass?: boolean): void;
        /** @internal */
        _isSynchronizedViewMatrix(): boolean;
        /** @internal */
        _computeLocalCameraSpeed(): number;
        /**
         * Defines the target the camera should look at.
         * @param target Defines the new target as a Vector
         */
        setTarget(target: Vector3): void;
        /**
         * Defines the target point of the camera.
         * The camera looks towards it form the radius distance.
         */
        get target(): Vector3;
        set target(value: Vector3);
        /**
         * Return the current target position of the camera. This value is expressed in local space.
         * @returns the target position
         */
        getTarget(): Vector3;
        /** @internal */
        _decideIfNeedsToMove(): boolean;
        /** @internal */
        _updatePosition(): void;
        /** @internal */
        _checkInputs(): void;
        protected _updateCameraRotationMatrix(): void;
        /**
         * Update the up vector to apply the rotation of the camera (So if you changed the camera rotation.z this will let you update the up vector as well)
         * @returns the current camera
         */
        private _rotateUpVectorWithCameraRotationMatrix;
        private _cachedRotationZ;
        private _cachedQuaternionRotationZ;
        /** @internal */
        _getViewMatrix(): Matrix;
        protected _computeViewMatrix(position: Vector3, target: Vector3, up: Vector3): void;
        /**
         * @internal
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * @internal
         */
        _updateRigCameras(): void;
        private _getRigCamPositionAndTarget;
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
    }


    /**
     * This represents a FPS type of camera controlled by touch.
     * This is like a universal camera minus the Gamepad controls.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
     */
    export class TouchCamera extends FreeCamera {
        /**
         * Defines the touch sensibility for rotation.
         * The higher the faster.
         */
        get touchAngularSensibility(): number;
        set touchAngularSensibility(value: number);
        /**
         * Defines the touch sensibility for move.
         * The higher the faster.
         */
        get touchMoveSensibility(): number;
        set touchMoveSensibility(value: number);
        /**
         * Instantiates a new touch camera.
         * This represents a FPS type of camera controlled by touch.
         * This is like a universal camera minus the Gamepad controls.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
         * @param name Define the name of the camera in the scene
         * @param position Define the start position of the camera in the scene
         * @param scene Define the scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene?: Scene);
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
        /** @internal */
        _setupInputs(): void;
    }


    /**
     * The Universal Camera is the one to choose for first person shooter type games, and works with all the keyboard, mouse, touch and gamepads. This replaces the earlier Free Camera,
     * which still works and will still be found in many Playgrounds.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
     */
    export class UniversalCamera extends TouchCamera {
        /**
         * Defines the gamepad rotation sensibility.
         * This is the threshold from when rotation starts to be accounted for to prevent jittering.
         */
        get gamepadAngularSensibility(): number;
        set gamepadAngularSensibility(value: number);
        /**
         * Defines the gamepad move sensibility.
         * This is the threshold from when moving starts to be accounted for to prevent jittering.
         */
        get gamepadMoveSensibility(): number;
        set gamepadMoveSensibility(value: number);
        /**
         * The Universal Camera is the one to choose for first person shooter type games, and works with all the keyboard, mouse, touch and gamepads. This replaces the earlier Free Camera,
         * which still works and will still be found in many Playgrounds.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#universal-camera
         * @param name Define the name of the camera in the scene
         * @param position Define the start position of the camera in the scene
         * @param scene Define the scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene?: Scene);
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
    }


    /**
     * This represents a free type of camera. It can be useful in First Person Shooter game for instance.
     * It is identical to the Free Camera and simply adds by default a virtual joystick.
     * Virtual Joysticks are on-screen 2D graphics that are used to control the camera or other scene items.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#virtual-joysticks-camera
     */
    export class VirtualJoysticksCamera extends FreeCamera {
        /**
         * Instantiates a VirtualJoysticksCamera. It can be useful in First Person Shooter game for instance.
         * It is identical to the Free Camera and simply adds by default a virtual joystick.
         * Virtual Joysticks are on-screen 2D graphics that are used to control the camera or other scene items.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#virtual-joysticks-camera
         * @param name Define the name of the camera in the scene
         * @param position Define the start position of the camera in the scene
         * @param scene Define the scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene?: Scene);
        /**
         * Gets the current object class name.
         * @returns the class name
         */
        getClassName(): string;
    }




    /**
     * This represents all the required metrics to create a VR camera.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#device-orientation-camera
     */
    export class VRCameraMetrics {
        /**
         * Define the horizontal resolution off the screen.
         */
        hResolution: number;
        /**
         * Define the vertical resolution off the screen.
         */
        vResolution: number;
        /**
         * Define the horizontal screen size.
         */
        hScreenSize: number;
        /**
         * Define the vertical screen size.
         */
        vScreenSize: number;
        /**
         * Define the vertical screen center position.
         */
        vScreenCenter: number;
        /**
         * Define the distance of the eyes to the screen.
         */
        eyeToScreenDistance: number;
        /**
         * Define the distance between both lenses
         */
        lensSeparationDistance: number;
        /**
         * Define the distance between both viewer's eyes.
         */
        interpupillaryDistance: number;
        /**
         * Define the distortion factor of the VR postprocess.
         * Please, touch with care.
         */
        distortionK: number[];
        /**
         * Define the chromatic aberration correction factors for the VR post process.
         */
        chromaAbCorrection: number[];
        /**
         * Define the scale factor of the post process.
         * The smaller the better but the slower.
         */
        postProcessScaleFactor: number;
        /**
         * Define an offset for the lens center.
         */
        lensCenterOffset: number;
        /**
         * Define if the current vr camera should compensate the distortion of the lens or not.
         */
        compensateDistortion: boolean;
        /**
         * Defines if multiview should be enabled when rendering (Default: false)
         */
        multiviewEnabled: boolean;
        /**
         * Gets the rendering aspect ratio based on the provided resolutions.
         */
        get aspectRatio(): number;
        /**
         * Gets the aspect ratio based on the FOV, scale factors, and real screen sizes.
         */
        get aspectRatioFov(): number;
        /**
         * @internal
         */
        get leftHMatrix(): Matrix;
        /**
         * @internal
         */
        get rightHMatrix(): Matrix;
        /**
         * @internal
         */
        get leftPreViewMatrix(): Matrix;
        /**
         * @internal
         */
        get rightPreViewMatrix(): Matrix;
        /**
         * Get the default VRMetrics based on the most generic setup.
         * @returns the default vr metrics
         */
        static GetDefault(): VRCameraMetrics;
    }


    /**
     * Camera used to simulate VR rendering (based on ArcRotateCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#vr-device-orientation-cameras
     */
    export class VRDeviceOrientationArcRotateCamera extends ArcRotateCamera {
        /**
         * Creates a new VRDeviceOrientationArcRotateCamera
         * @param name defines camera name
         * @param alpha defines the camera rotation along the longitudinal axis
         * @param beta defines the camera rotation along the latitudinal axis
         * @param radius defines the camera distance from its target
         * @param target defines the camera target
         * @param scene defines the scene the camera belongs to
         * @param compensateDistortion defines if the camera needs to compensate the lens distortion
         * @param vrCameraMetrics defines the vr metrics associated to the camera
         */
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, scene?: Scene, compensateDistortion?: boolean, vrCameraMetrics?: VRCameraMetrics);
        /**
         * Gets camera class name
         * @returns VRDeviceOrientationArcRotateCamera
         */
        getClassName(): string;
        protected _setRigMode: (rigParams: any) => void;
    }


    /**
     * Camera used to simulate VR rendering (based on FreeCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#vr-device-orientation-cameras
     */
    export class VRDeviceOrientationFreeCamera extends DeviceOrientationCamera {
        /**
         * Creates a new VRDeviceOrientationFreeCamera
         * @param name defines camera name
         * @param position defines the start position of the camera
         * @param scene defines the scene the camera belongs to
         * @param compensateDistortion defines if the camera needs to compensate the lens distortion
         * @param vrCameraMetrics defines the vr metrics associated to the camera
         */
        constructor(name: string, position: Vector3, scene?: Scene, compensateDistortion?: boolean, vrCameraMetrics?: VRCameraMetrics);
        /**
         * Gets camera class name
         * @returns VRDeviceOrientationFreeCamera
         */
        getClassName(): string;
        protected _setRigMode: (rigParams: any) => void;
    }


    /**
     * Camera used to simulate VR rendering (based on VRDeviceOrientationFreeCamera)
     * @see https://doc.babylonjs.com/features/featuresDeepDive/cameras/camera_introduction#vr-device-orientation-cameras
     */
    export class VRDeviceOrientationGamepadCamera extends VRDeviceOrientationFreeCamera {
        /**
         * Creates a new VRDeviceOrientationGamepadCamera
         * @param name defines camera name
         * @param position defines the start position of the camera
         * @param scene defines the scene the camera belongs to
         * @param compensateDistortion defines if the camera needs to compensate the lens distortion
         * @param vrCameraMetrics defines the vr metrics associated to the camera
         */
        constructor(name: string, position: Vector3, scene?: Scene, compensateDistortion?: boolean, vrCameraMetrics?: VRCameraMetrics);
        /**
         * Gets camera class name
         * @returns VRDeviceOrientationGamepadCamera
         */
        getClassName(): string;
        protected _setRigMode: (rigParams: any) => void;
    }


    /**
     * Options to modify the vr teleportation behavior.
     */
    export interface VRTeleportationOptions {
        /**
         * The name of the mesh which should be used as the teleportation floor. (default: null)
         */
        floorMeshName?: string;
        /**
         * A list of meshes to be used as the teleportation floor. (default: empty)
         */
        floorMeshes?: Mesh[];
        /**
         * The teleportation mode. (default: TELEPORTATIONMODE_CONSTANTTIME)
         */
        teleportationMode?: number;
        /**
         * The duration of the animation in ms, apply when animationMode is TELEPORTATIONMODE_CONSTANTTIME. (default 122ms)
         */
        teleportationTime?: number;
        /**
         * The speed of the animation in distance/sec, apply when animationMode is TELEPORTATIONMODE_CONSTANTSPEED. (default 20 units / sec)
         */
        teleportationSpeed?: number;
        /**
         * The easing function used in the animation or null for Linear. (default CircleEase)
         */
        easingFunction?: EasingFunction;
    }
    /**
     * Options to modify the vr experience helper's behavior.
     */
    export interface VRExperienceHelperOptions {
        /**
         * Create a DeviceOrientationCamera to be used as your out of vr camera. (default: true)
         */
        createDeviceOrientationCamera?: boolean;
        /**
         * Create a VRDeviceOrientationFreeCamera to be used for VR when no external HMD is found. (default: true)
         */
        createFallbackVRDeviceOrientationFreeCamera?: boolean;
        /**
         * Uses the main button on the controller to toggle the laser casted. (default: true)
         */
        laserToggle?: boolean;
        /**
         * A list of meshes to be used as the teleportation floor. If specified, teleportation will be enabled (default: undefined)
         */
        floorMeshes?: Mesh[];
        /**
         * Distortion metrics for the fallback vrDeviceOrientationCamera (default: VRCameraMetrics.Default)
         */
        vrDeviceOrientationCameraMetrics?: VRCameraMetrics;
        /**
         * Defines if WebXR should be used (if available)
         */
        useXR?: boolean;
    }
    /**
     * Event containing information after VR has been entered
     */
    export class OnAfterEnteringVRObservableEvent {
        /**
         * If entering vr was successful
         */
        success: boolean;
    }
    /**
     * Helps to quickly add VR support to an existing scene.
     * See https://doc.babylonjs.com/features/featuresDeepDive/cameras/webVRHelper
     * @deprecated Use WebXR instead!
     */
    export class VRExperienceHelper {
        /** Options to modify the vr experience helper's behavior. */
        webVROptions: VRExperienceHelperOptions;
        private _scene;
        private _position;
        private _btnVR;
        private _btnVRDisplayed;
        private _hasEnteredVR;
        private _fullscreenVRpresenting;
        private _inputElement;
        private _vrDeviceOrientationCamera;
        private _deviceOrientationCamera;
        private _existingCamera;
        private _onKeyDown;
        private _onVrDisplayPresentChangeBind;
        /**
         * Gets or sets a boolean indicating that gaze can be enabled even if pointer lock is not engage (useful on iOS where fullscreen mode and pointer lock are not supported)
         */
        enableGazeEvenWhenNoPointerLock: boolean;
        /**
         * Gets or sets a boolean indicating that the VREXperienceHelper will exit VR if double tap is detected
         */
        exitVROnDoubleTap: boolean;
        /**
         * Observable raised right before entering VR.
         */
        onEnteringVRObservable: Observable<VRExperienceHelper>;
        /**
         * Observable raised when entering VR has completed.
         */
        onAfterEnteringVRObservable: Observable<OnAfterEnteringVRObservableEvent>;
        /**
         * Observable raised when exiting VR.
         */
        onExitingVRObservable: Observable<VRExperienceHelper>;
        /** Return this.onEnteringVRObservable
         * Note: This one is for backward compatibility. Please use onEnteringVRObservable directly
         */
        get onEnteringVR(): Observable<VRExperienceHelper>;
        /** Return this.onExitingVRObservable
         * Note: This one is for backward compatibility. Please use onExitingVRObservable directly
         */
        get onExitingVR(): Observable<VRExperienceHelper>;
        private _useCustomVRButton;
        private _teleportActive;
        private _floorMeshName;
        private _floorMeshesCollection;
        private _teleportationMode;
        private _teleportationTime;
        private _teleportationSpeed;
        private _teleportationEasing;
        private _rotationAllowed;
        private _teleportBackwardsVector;
        private _teleportationTarget;
        private _isDefaultTeleportationTarget;
        private _postProcessMove;
        private _teleportationFillColor;
        private _teleportationBorderColor;
        private _rotationAngle;
        private _haloCenter;
        private _cameraGazer;
        private _padSensibilityUp;
        private _padSensibilityDown;
        private _pickedLaserColor;
        private _pickedGazeColor;
        /**
         * Observable raised when a new mesh is selected based on meshSelectionPredicate
         */
        onNewMeshSelected: Observable<AbstractMesh>;
        /**
         * Observable raised when a new mesh is picked based on meshSelectionPredicate
         */
        onNewMeshPicked: Observable<PickingInfo>;
        private _circleEase;
        /**
         * Observable raised before camera teleportation
         */
        onBeforeCameraTeleport: Observable<Vector3>;
        /**
         *  Observable raised after camera teleportation
         */
        onAfterCameraTeleport: Observable<Vector3>;
        /**
         * Observable raised when current selected mesh gets unselected
         */
        onSelectedMeshUnselected: Observable<AbstractMesh>;
        private _raySelectionPredicate;
        /**
         * To be optionally changed by user to define custom ray selection
         */
        raySelectionPredicate: (mesh: AbstractMesh) => boolean;
        /**
         * To be optionally changed by user to define custom selection logic (after ray selection)
         */
        meshSelectionPredicate: (mesh: AbstractMesh) => boolean;
        /**
         * Set teleportation enabled. If set to false camera teleportation will be disabled but camera rotation will be kept.
         */
        teleportationEnabled: boolean;
        private _defaultHeight;
        private _teleportationInitialized;
        private _interactionsEnabled;
        private _displayGaze;
        private _displayLaserPointer;
        /**
         * The mesh used to display where the user is going to teleport.
         */
        get teleportationTarget(): Mesh;
        /**
         * Sets the mesh to be used to display where the user is going to teleport.
         */
        set teleportationTarget(value: Mesh);
        /**
         * The mesh used to display where the user is selecting, this mesh will be cloned and set as the gazeTracker for the left and right controller
         * when set bakeCurrentTransformIntoVertices will be called on the mesh.
         * See https://doc.babylonjs.com/features/featuresDeepDive/mesh/transforms/center_origin/bakingTransforms
         */
        get gazeTrackerMesh(): Mesh;
        set gazeTrackerMesh(value: Mesh);
        /**
         * If the gaze trackers scale should be updated to be constant size when pointing at near/far meshes
         */
        updateGazeTrackerScale: boolean;
        /**
         * If the gaze trackers color should be updated when selecting meshes
         */
        updateGazeTrackerColor: boolean;
        /**
         * If the controller laser color should be updated when selecting meshes
         */
        updateControllerLaserColor: boolean;
        /**
         * If the ray of the gaze should be displayed.
         */
        get displayGaze(): boolean;
        /**
         * Sets if the ray of the gaze should be displayed.
         */
        set displayGaze(value: boolean);
        /**
         * If the ray of the LaserPointer should be displayed.
         */
        get displayLaserPointer(): boolean;
        /**
         * Sets if the ray of the LaserPointer should be displayed.
         */
        set displayLaserPointer(value: boolean);
        /**
         * The deviceOrientationCamera used as the camera when not in VR.
         */
        get deviceOrientationCamera(): Nullable<DeviceOrientationCamera>;
        /**
         * Based on the current WebVR support, returns the current VR camera used.
         */
        get currentVRCamera(): Nullable<Camera>;
        /**
         * The deviceOrientationCamera that is used as a fallback when vr device is not connected.
         */
        get vrDeviceOrientationCamera(): Nullable<VRDeviceOrientationFreeCamera>;
        /**
         * The html button that is used to trigger entering into VR.
         */
        get vrButton(): Nullable<HTMLButtonElement>;
        private get _teleportationRequestInitiated();
        /**
         * Defines whether or not Pointer lock should be requested when switching to
         * full screen.
         */
        requestPointerLockOnFullScreen: boolean;
        /**
         * If asking to force XR, this will be populated with the default xr experience
         */
        xr: WebXRDefaultExperience;
        /**
         * Was the XR test done already. If this is true AND this.xr exists, xr is initialized.
         * If this is true and no this.xr, xr exists but is not supported, using WebVR.
         */
        xrTestDone: boolean;
        /**
         * Instantiates a VRExperienceHelper.
         * Helps to quickly add VR support to an existing scene.
         * @param scene The scene the VRExperienceHelper belongs to.
         * @param webVROptions Options to modify the vr experience helper's behavior.
         */
        constructor(scene: Scene, 
        /** Options to modify the vr experience helper's behavior. */
        webVROptions?: VRExperienceHelperOptions);
        private _completeVRInit;
        private _onResize;
        private _onFullscreenChange;
        /**
         * Gets a value indicating if we are currently in VR mode.
         */
        get isInVRMode(): boolean;
        private _moveButtonToBottomRight;
        private _displayVRButton;
        private _updateButtonVisibility;
        private _cachedAngularSensibility;
        /**
         * Attempt to enter VR. If a headset is connected and ready, will request present on that.
         * Otherwise, will use the fullscreen API.
         */
        enterVR(): void;
        /**
         * Attempt to exit VR, or fullscreen.
         */
        exitVR(): void;
        /**
         * The position of the vr experience helper.
         */
        get position(): Vector3;
        /**
         * Sets the position of the vr experience helper.
         */
        set position(value: Vector3);
        /**
         * Enables controllers and user interactions such as selecting and object or clicking on an object.
         */
        enableInteractions(): void;
        private _beforeRender;
        private _isTeleportationFloor;
        /**
         * Adds a floor mesh to be used for teleportation.
         * @param floorMesh the mesh to be used for teleportation.
         */
        addFloorMesh(floorMesh: Mesh): void;
        /**
         * Removes a floor mesh from being used for teleportation.
         * @param floorMesh the mesh to be removed.
         */
        removeFloorMesh(floorMesh: Mesh): void;
        /**
         * Enables interactions and teleportation using the VR controllers and gaze.
         * @param vrTeleportationOptions options to modify teleportation behavior.
         */
        enableTeleportation(vrTeleportationOptions?: VRTeleportationOptions): void;
        private _onNewGamepadConnected;
        private _checkTeleportWithRay;
        private _checkRotate;
        private _checkTeleportBackwards;
        private _createTeleportationCircles;
        private _hideTeleportationTarget;
        private _rotateCamera;
        private _workingVector;
        private _workingQuaternion;
        private _workingMatrix;
        /**
         * Time Constant Teleportation Mode
         */
        static readonly TELEPORTATIONMODE_CONSTANTTIME = 0;
        /**
         * Speed Constant Teleportation Mode
         */
        static readonly TELEPORTATIONMODE_CONSTANTSPEED = 1;
        /**
         * Teleports the users feet to the desired location
         * @param location The location where the user's feet should be placed
         */
        teleportCamera(location: Vector3): void;
        /**
         * Permanently set new colors for the laser pointer
         * @param color the new laser color
         * @param pickedColor the new laser color when picked mesh detected
         */
        setLaserColor(color: Color3, pickedColor?: Color3): void;
        /**
         * Set lighting enabled / disabled on the laser pointer of both controllers
         * @param _enabled should the lighting be enabled on the laser pointer
         */
        setLaserLightingState(_enabled?: boolean): void;
        /**
         * Permanently set new colors for the gaze pointer
         * @param color the new gaze color
         * @param pickedColor the new gaze color when picked mesh detected
         */
        setGazeColor(color: Color3, pickedColor?: Color3): void;
        /**
         * Sets the color of the laser ray from the vr controllers.
         * @param _color new color for the ray.
         */
        changeLaserColor(_color: Color3): void;
        /**
         * Sets the color of the ray from the vr headsets gaze.
         * @param color new color for the ray.
         */
        changeGazeColor(color: Color3): void;
        /**
         * Exits VR and disposes of the vr experience helper
         */
        dispose(): void;
        /**
         * Gets the name of the VRExperienceHelper class
         * @returns "VRExperienceHelper"
         */
        getClassName(): string;
    }


    /** @internal */
    export class Collider {
        /** Define if a collision was found */
        collisionFound: boolean;
        /**
         * Define last intersection point in local space
         */
        intersectionPoint: Vector3;
        /**
         * Define last collided mesh
         */
        collidedMesh: Nullable<AbstractMesh>;
        /**
         * If true, it check for double sided faces and only returns 1 collision instead of 2
         */
        static DoubleSidedCheck: boolean;
        private _collisionPoint;
        private _planeIntersectionPoint;
        private _tempVector;
        private _tempVector2;
        private _tempVector3;
        private _tempVector4;
        private _edge;
        private _baseToVertex;
        private _destinationPoint;
        private _slidePlaneNormal;
        private _displacementVector;
        /** @internal */
        _radius: Vector3;
        /** @internal */
        _retry: number;
        private _velocity;
        private _basePoint;
        private _epsilon;
        /** @internal */
        _velocityWorldLength: number;
        /** @internal */
        _basePointWorld: Vector3;
        private _velocityWorld;
        private _normalizedVelocity;
        /** @internal */
        _initialVelocity: Vector3;
        /** @internal */
        _initialPosition: Vector3;
        private _nearestDistance;
        private _collisionMask;
        private _velocitySquaredLength;
        private _nearestDistanceSquared;
        get collisionMask(): number;
        set collisionMask(mask: number);
        /**
         * Gets the plane normal used to compute the sliding response (in local space)
         */
        get slidePlaneNormal(): Vector3;
        /**
         * @internal
         */
        _initialize(source: Vector3, dir: Vector3, e: number): void;
        /**
         * @internal
         */
        _checkPointInTriangle(point: Vector3, pa: Vector3, pb: Vector3, pc: Vector3, n: Vector3): boolean;
        /**
         * @internal
         */
        _canDoCollision(sphereCenter: Vector3, sphereRadius: number, vecMin: Vector3, vecMax: Vector3): boolean;
        /**
         * @internal
         */
        _testTriangle(faceIndex: number, trianglePlaneArray: Array<Plane>, p1: Vector3, p2: Vector3, p3: Vector3, hasMaterial: boolean, hostMesh: AbstractMesh): void;
        /**
         * @internal
         */
        _collide(trianglePlaneArray: Array<Plane>, pts: Vector3[], indices: IndicesArray, indexStart: number, indexEnd: number, decal: number, hasMaterial: boolean, hostMesh: AbstractMesh, invertTriangles?: boolean, triangleStrip?: boolean): void;
        /**
         * @internal
         */
        _getResponse(pos: Vector3, vel: Vector3): void;
    }


    /** @internal */
    export interface ICollisionCoordinator {
        createCollider(): Collider;
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: Nullable<AbstractMesh>, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
    }
    /** @internal */
    export class DefaultCollisionCoordinator implements ICollisionCoordinator {
        private _scene;
        private _scaledPosition;
        private _scaledVelocity;
        private _finalPosition;
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: AbstractMesh, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        createCollider(): Collider;
        init(scene: Scene): void;
        private _collideWithWorld;
    }




    /**
     * @internal
     */
    export class IntersectionInfo {
        bu: Nullable<number>;
        bv: Nullable<number>;
        distance: number;
        faceId: number;
        subMeshId: number;
        constructor(bu: Nullable<number>, bv: Nullable<number>, distance: number);
    }


    /**
     * @internal
     */
    export class _MeshCollisionData {
        _checkCollisions: boolean;
        _collisionMask: number;
        _collisionGroup: number;
        _surroundingMeshes: Nullable<AbstractMesh[]>;
        _collider: Nullable<Collider>;
        _oldPositionForCollisions: Vector3;
        _diffPositionForCollisions: Vector3;
        _onCollideObserver: Nullable<Observer<AbstractMesh>>;
        _onCollisionPositionChangeObserver: Nullable<Observer<Vector3>>;
        _collisionResponse: boolean;
    }


    /**
     * Information about the result of picking within a scene
     * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/interactions/picking_collisions
     */
    export class PickingInfo {
        /**
         * If the pick collided with an object
         */
        hit: boolean;
        /**
         * Distance away where the pick collided
         */
        distance: number;
        /**
         * The location of pick collision
         */
        pickedPoint: Nullable<Vector3>;
        /**
         * The mesh corresponding the pick collision
         */
        pickedMesh: Nullable<AbstractMesh>;
        /** (See getTextureCoordinates) The barycentric U coordinate that is used when calculating the texture coordinates of the collision.*/
        bu: number;
        /** (See getTextureCoordinates) The barycentric V coordinate that is used when calculating the texture coordinates of the collision.*/
        bv: number;
        /** The index of the face on the mesh that was picked, or the index of the Line if the picked Mesh is a LinesMesh */
        faceId: number;
        /** The index of the face on the subMesh that was picked, or the index of the Line if the picked Mesh is a LinesMesh */
        subMeshFaceId: number;
        /** Id of the submesh that was picked */
        subMeshId: number;
        /** If a sprite was picked, this will be the sprite the pick collided with */
        pickedSprite: Nullable<Sprite>;
        /** If we are picking a mesh with thin instance, this will give you the picked thin instance */
        thinInstanceIndex: number;
        /**
         * The ray that was used to perform the picking.
         */
        ray: Nullable<Ray>;
        /**
         * If a mesh was used to do the picking (eg. 6dof controller) as a "near interaction", this will be populated.
         */
        originMesh: Nullable<AbstractMesh>;
        /**
         * The aim-space transform of the input used for picking, if it is an XR input source.
         */
        aimTransform: Nullable<TransformNode>;
        /**
         * The grip-space transform of the input used for picking, if it is an XR input source.
         * Some XR sources, such as input coming from head mounted displays, do not have this.
         */
        gripTransform: Nullable<TransformNode>;
        /**
         * Gets the normal corresponding to the face the pick collided with
         * @param useWorldCoordinates If the resulting normal should be relative to the world (default: false)
         * @param useVerticesNormals If the vertices normals should be used to calculate the normal instead of the normal map (default: true)
         * @returns The normal corresponding to the face the pick collided with
         * @remarks Note that the returned normal will always point towards the picking ray.
         */
        getNormal(useWorldCoordinates?: boolean, useVerticesNormals?: boolean): Nullable<Vector3>;
        /**
         * Gets the texture coordinates of where the pick occurred
         * @param uvSet The UV set to use to calculate the texture coordinates (default: VertexBuffer.UVKind)
         * @returns The vector containing the coordinates of the texture
         */
        getTextureCoordinates(uvSet?: string): Nullable<Vector2>;
    }


    /**
     * Options used to control default behaviors regarding compatibility support
     */
    export class CompatibilityOptions {
        /**
         * Defines if the system should use OpenGL convention for UVs when creating geometry or loading .babylon files (false by default)
         */
        static UseOpenGLOrientationForUV: boolean;
    }




    /**
     * Options to be used when creating a compute effect.
     */
    export interface IComputeEffectCreationOptions {
        /**
         * Define statements that will be set in the shader.
         */
        defines: any;
        /**
         * The name of the entry point in the shader source (default: "main")
         */
        entryPoint?: string;
        /**
         * Callback that will be called when the shader is compiled.
         */
        onCompiled: Nullable<(effect: ComputeEffect) => void>;
        /**
         * Callback that will be called if an error occurs during shader compilation.
         */
        onError: Nullable<(effect: ComputeEffect, errors: string) => void>;
        /**
         * If provided, will be called with the shader code so that this code can be updated before it is compiled by the GPU
         */
        processFinalCode?: Nullable<(code: string) => string>;
    }
    /**
     * Effect wrapping a compute shader and let execute (dispatch) the shader
     */
    export class ComputeEffect {
        private static _UniqueIdSeed;
        /**
         * Enable logging of the shader code when a compilation error occurs
         */
        static LogShaderCodeOnCompilationError: boolean;
        /**
         * Name of the effect.
         */
        name: any;
        /**
         * String container all the define statements that should be set on the shader.
         */
        defines: string;
        /**
         * Callback that will be called when the shader is compiled.
         */
        onCompiled: Nullable<(effect: ComputeEffect) => void>;
        /**
         * Callback that will be called if an error occurs during shader compilation.
         */
        onError: Nullable<(effect: ComputeEffect, errors: string) => void>;
        /**
         * Unique ID of the effect.
         */
        uniqueId: number;
        /**
         * Observable that will be called when the shader is compiled.
         * It is recommended to use executeWhenCompile() or to make sure that scene.isReady() is called to get this observable raised.
         */
        onCompileObservable: Observable<ComputeEffect>;
        /**
         * Observable that will be called if an error occurs during shader compilation.
         */
        onErrorObservable: Observable<ComputeEffect>;
        /**
         * Observable that will be called when effect is bound.
         */
        onBindObservable: Observable<ComputeEffect>;
        /**
         * @internal
         * Specifies if the effect was previously ready
         */
        _wasPreviouslyReady: boolean;
        private _engine;
        private _isReady;
        private _compilationError;
        /** @internal */
        _key: string;
        private _computeSourceCodeOverride;
        /** @internal */
        _pipelineContext: Nullable<IComputePipelineContext>;
        /** @internal */
        _computeSourceCode: string;
        private _rawComputeSourceCode;
        private _entryPoint;
        private _shaderLanguage;
        private _shaderStore;
        private _shaderRepository;
        private _includeShaderStore;
        /**
         * Creates a compute effect that can be used to execute a compute shader
         * @param baseName Name of the effect
         * @param options Set of all options to create the effect
         * @param engine The engine the effect is created for
         * @param key Effect Key identifying uniquely compiled shader variants
         */
        constructor(baseName: any, options: IComputeEffectCreationOptions, engine: Engine, key?: string);
        private _useFinalCode;
        /**
         * Unique key for this effect
         */
        get key(): string;
        /**
         * If the effect has been compiled and prepared.
         * @returns if the effect is compiled and prepared.
         */
        isReady(): boolean;
        private _isReadyInternal;
        /**
         * The engine the effect was initialized with.
         * @returns the engine.
         */
        getEngine(): Engine;
        /**
         * The pipeline context for this effect
         * @returns the associated pipeline context
         */
        getPipelineContext(): Nullable<IComputePipelineContext>;
        /**
         * The error from the last compilation.
         * @returns the error string.
         */
        getCompilationError(): string;
        /**
         * Adds a callback to the onCompiled observable and call the callback immediately if already ready.
         * @param func The callback to be used.
         */
        executeWhenCompiled(func: (effect: ComputeEffect) => void): void;
        private _checkIsReady;
        private _loadShader;
        /**
         * Gets the compute shader source code of this effect
         */
        get computeSourceCode(): string;
        /**
         * Gets the compute shader source code before it has been processed by the preprocessor
         */
        get rawComputeSourceCode(): string;
        /**
         * Prepares the effect
         * @internal
         */
        _prepareEffect(): void;
        private _getShaderCodeAndErrorLine;
        private _processCompilationErrors;
        /**
         * Release all associated resources.
         **/
        dispose(): void;
        /**
         * This function will add a new compute shader to the shader store
         * @param name the name of the shader
         * @param computeShader compute shader content
         */
        static RegisterShader(name: string, computeShader: string): void;
    }


    /**
     * Defines the options associated with the creation of a compute shader.
     */
    export interface IComputeShaderOptions {
        /**
         * list of bindings mapping (key is property name, value is binding location)
         * Must be provided because browsers don't support reflection for wgsl shaders yet (so there's no way to query the binding/group from a variable name)
         * TODO: remove this when browsers support reflection for wgsl shaders
         */
        bindingsMapping: ComputeBindingMapping;
        /**
         * The list of defines used in the shader
         */
        defines?: string[];
        /**
         * The name of the entry point in the shader source (default: "main")
         */
        entryPoint?: string;
        /**
         * If provided, will be called with the shader code so that this code can be updated before it is compiled by the GPU
         */
        processFinalCode?: Nullable<(code: string) => string>;
    }
    /**
     * The ComputeShader object lets you execute a compute shader on your GPU (if supported by the engine)
     */
    export class ComputeShader {
        private _engine;
        private _shaderPath;
        private _options;
        private _effect;
        private _cachedDefines;
        private _bindings;
        private _samplers;
        private _context;
        private _contextIsDirty;
        /**
         * Gets the unique id of the compute shader
         */
        readonly uniqueId: number;
        /**
         * The name of the shader
         */
        name: string;
        /**
         * The options used to create the shader
         */
        get options(): IComputeShaderOptions;
        /**
         * The shaderPath used to create the shader
         */
        get shaderPath(): any;
        /**
         * When set to true, dispatch won't call isReady anymore and won't check if the underlying GPU resources should be (re)created because of a change in the inputs (texture, uniform buffer, etc.)
         * If you know that your inputs did not change since last time dispatch was called and that isReady() returns true, set this flag to true to improve performance
         */
        fastMode: boolean;
        /**
         * Callback triggered when the shader is compiled
         */
        onCompiled: Nullable<(effect: ComputeEffect) => void>;
        /**
         * Callback triggered when an error occurs
         */
        onError: Nullable<(effect: ComputeEffect, errors: string) => void>;
        /**
         * Gets the GPU time spent running the compute shader for the last frame rendered (in nanoseconds).
         * You have to enable the "timestamp-query" extension in the engine constructor options and set engine.enableGPUTimingMeasurements = true.
         */
        readonly gpuTimeInFrame?: WebGPUPerfCounter;
        /**
         * Instantiates a new compute shader.
         * @param name Defines the name of the compute shader in the scene
         * @param engine Defines the engine the compute shader belongs to
         * @param shaderPath Defines  the route to the shader code in one of three ways:
         *  * object: \{ compute: "custom" \}, used with ShaderStore.ShadersStoreWGSL["customComputeShader"]
         *  * object: \{ computeElement: "HTMLElementId" \}, used with shader code in script tags
         *  * object: \{ computeSource: "compute shader code string" \}, where the string contains the shader code
         *  * string: try first to find the code in ShaderStore.ShadersStoreWGSL[shaderPath + "ComputeShader"]. If not, assumes it is a file with name shaderPath.compute.fx in index.html folder.
         * @param options Define the options used to create the shader
         */
        constructor(name: string, engine: ThinEngine, shaderPath: any, options?: Partial<IComputeShaderOptions>);
        /**
         * Gets the current class name of the material e.g. "ComputeShader"
         * Mainly use in serialization.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Binds a texture to the shader
         * @param name Binding name of the texture
         * @param texture Texture to bind
         * @param bindSampler Bind the sampler corresponding to the texture (default: true). The sampler will be bound just before the binding index of the texture
         */
        setTexture(name: string, texture: BaseTexture, bindSampler?: boolean): void;
        /**
         * Binds a storage texture to the shader
         * @param name Binding name of the texture
         * @param texture Texture to bind
         */
        setStorageTexture(name: string, texture: BaseTexture): void;
        /**
         * Binds an external texture to the shader
         * @param name Binding name of the texture
         * @param texture Texture to bind
         */
        setExternalTexture(name: string, texture: ExternalTexture): void;
        /**
         * Binds a video texture to the shader (by binding the external texture attached to this video)
         * @param name Binding name of the texture
         * @param texture Texture to bind
         * @returns true if the video texture was successfully bound, else false. false will be returned if the current engine does not support external textures
         */
        setVideoTexture(name: string, texture: VideoTexture): boolean;
        /**
         * Binds a uniform buffer to the shader
         * @param name Binding name of the buffer
         * @param buffer Buffer to bind
         */
        setUniformBuffer(name: string, buffer: UniformBuffer): void;
        /**
         * Binds a storage buffer to the shader
         * @param name Binding name of the buffer
         * @param buffer Buffer to bind
         */
        setStorageBuffer(name: string, buffer: StorageBuffer): void;
        /**
         * Binds a texture sampler to the shader
         * @param name Binding name of the sampler
         * @param sampler Sampler to bind
         */
        setTextureSampler(name: string, sampler: TextureSampler): void;
        /**
         * Specifies that the compute shader is ready to be executed (the compute effect and all the resources are ready)
         * @returns true if the compute shader is ready to be executed
         */
        isReady(): boolean;
        /**
         * Dispatches (executes) the compute shader
         * @param x Number of workgroups to execute on the X dimension
         * @param y Number of workgroups to execute on the Y dimension (default: 1)
         * @param z Number of workgroups to execute on the Z dimension (default: 1)
         * @returns True if the dispatch could be done, else false (meaning either the compute effect or at least one of the bound resources was not ready)
         */
        dispatch(x: number, y?: number, z?: number): boolean;
        /**
         * Waits for the compute shader to be ready and executes it
         * @param x Number of workgroups to execute on the X dimension
         * @param y Number of workgroups to execute on the Y dimension (default: 1)
         * @param z Number of workgroups to execute on the Z dimension (default: 1)
         * @param delay Delay between the retries while the shader is not ready (in milliseconds - 10 by default)
         * @returns A promise that is resolved once the shader has been sent to the GPU. Note that it does not mean that the shader execution itself is finished!
         */
        dispatchWhenReady(x: number, y?: number, z?: number, delay?: number): Promise<void>;
        /**
         * Serializes this compute shader in a JSON representation
         * @returns the serialized compute shader object
         */
        serialize(): any;
        /**
         * Creates a compute shader from parsed compute shader data
         * @param source defines the JSON representation of the compute shader
         * @param scene defines the hosting scene
         * @param rootUrl defines the root URL to use to load textures and relative dependencies
         * @returns a new compute shader
         */
        static Parse(source: any, scene: Scene, rootUrl: string): ComputeShader;
    }


    /** @internal */
    export interface IComputeContext {
        clear(): void;
    }


    /**
     * Class used to store and describe the pipeline context associated with a compute effect
     */
    export interface IComputePipelineContext {
        /**
         * Gets a boolean indicating that this pipeline context is supporting asynchronous creating
         */
        isAsync: boolean;
        /**
         * Gets a boolean indicating that the context is ready to be used (like shader / pipeline are compiled and ready for instance)
         */
        isReady: boolean;
        /** @internal */
        _name?: string;
        /** @internal */
        _getComputeShaderCode(): string | null;
        /** Releases the resources associated with the pipeline. */
        dispose(): void;
    }




    /**
     * Class used to store bounding box information
     */
    export class BoundingBox implements ICullable {
        /**
         * Gets the 8 vectors representing the bounding box in local space
         */
        readonly vectors: Vector3[];
        /**
         * Gets the center of the bounding box in local space
         */
        readonly center: Vector3;
        /**
         * Gets the center of the bounding box in world space
         */
        readonly centerWorld: Vector3;
        /**
         * Gets the extend size in local space
         */
        readonly extendSize: Vector3;
        /**
         * Gets the extend size in world space
         */
        readonly extendSizeWorld: Vector3;
        /**
         * Gets the OBB (object bounding box) directions
         */
        readonly directions: Vector3[];
        /**
         * Gets the 8 vectors representing the bounding box in world space
         */
        readonly vectorsWorld: Vector3[];
        /**
         * Gets the minimum vector in world space
         */
        readonly minimumWorld: Vector3;
        /**
         * Gets the maximum vector in world space
         */
        readonly maximumWorld: Vector3;
        /**
         * Gets the minimum vector in local space
         */
        readonly minimum: Vector3;
        /**
         * Gets the maximum vector in local space
         */
        readonly maximum: Vector3;
        private _worldMatrix;
        private static readonly _TmpVector3;
        /**
         * @internal
         */
        _tag: number;
        /** @internal */
        _drawWrapperFront: Nullable<DrawWrapper>;
        /** @internal */
        _drawWrapperBack: Nullable<DrawWrapper>;
        /**
         * Creates a new bounding box
         * @param min defines the minimum vector (in local space)
         * @param max defines the maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        constructor(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>);
        /**
         * Recreates the entire bounding box from scratch as if we call the constructor in place
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        reConstruct(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>): void;
        /**
         * Scale the current bounding box by applying a scale factor
         * @param factor defines the scale factor to apply
         * @returns the current bounding box
         */
        scale(factor: number): BoundingBox;
        /**
         * Gets the world matrix of the bounding box
         * @returns a matrix
         */
        getWorldMatrix(): DeepImmutable<Matrix>;
        /**
         * @internal
         */
        _update(world: DeepImmutable<Matrix>): void;
        /**
         * Tests if the bounding box is intersecting the frustum planes
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an intersection
         */
        isInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if the bounding box is entirely inside the frustum planes
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an inclusion
         */
        isCompletelyInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if a point is inside the bounding box
         * @param point defines the point to test
         * @returns true if the point is inside the bounding box
         */
        intersectsPoint(point: DeepImmutable<Vector3>): boolean;
        /**
         * Tests if the bounding box intersects with a bounding sphere
         * @param sphere defines the sphere to test
         * @returns true if there is an intersection
         */
        intersectsSphere(sphere: DeepImmutable<BoundingSphere>): boolean;
        /**
         * Tests if the bounding box intersects with a box defined by a min and max vectors
         * @param min defines the min vector to use
         * @param max defines the max vector to use
         * @returns true if there is an intersection
         */
        intersectsMinMax(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>): boolean;
        /**
         * Disposes the resources of the class
         */
        dispose(): void;
        /**
         * Tests if two bounding boxes are intersections
         * @param box0 defines the first box to test
         * @param box1 defines the second box to test
         * @returns true if there is an intersection
         */
        static Intersects(box0: DeepImmutable<BoundingBox>, box1: DeepImmutable<BoundingBox>): boolean;
        /**
         * Tests if a bounding box defines by a min/max vectors intersects a sphere
         * @param minPoint defines the minimum vector of the bounding box
         * @param maxPoint defines the maximum vector of the bounding box
         * @param sphereCenter defines the sphere center
         * @param sphereRadius defines the sphere radius
         * @returns true if there is an intersection
         */
        static IntersectsSphere(minPoint: DeepImmutable<Vector3>, maxPoint: DeepImmutable<Vector3>, sphereCenter: DeepImmutable<Vector3>, sphereRadius: number): boolean;
        /**
         * Tests if a bounding box defined with 8 vectors is entirely inside frustum planes
         * @param boundingVectors defines an array of 8 vectors representing a bounding box
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an inclusion
         */
        static IsCompletelyInFrustum(boundingVectors: Array<DeepImmutable<Vector3>>, frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if a bounding box defined with 8 vectors intersects frustum planes
         * @param boundingVectors defines an array of 8 vectors representing a bounding box
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an intersection
         */
        static IsInFrustum(boundingVectors: Array<DeepImmutable<Vector3>>, frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
    }


    /**
     * Interface for cullable objects
     * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/using/materials_introduction#back-face-culling
     */
    export interface ICullable {
        /**
         * Checks if the object or part of the object is in the frustum
         * @param frustumPlanes Camera near/planes
         * @returns true if the object is in frustum otherwise false
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Checks if a cullable object (mesh...) is in the camera frustum
         * Unlike isInFrustum this checks the full bounding box
         * @param frustumPlanes Camera near/planes
         * @returns true if the object is in frustum otherwise false
         */
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
    }
    /**
     * Info for a bounding data of a mesh
     */
    export class BoundingInfo implements ICullable {
        /**
         * Bounding box for the mesh
         */
        readonly boundingBox: BoundingBox;
        /**
         * Bounding sphere for the mesh
         */
        readonly boundingSphere: BoundingSphere;
        private _isLocked;
        private static readonly _TmpVector3;
        /**
         * Constructs bounding info
         * @param minimum min vector of the bounding box/sphere
         * @param maximum max vector of the bounding box/sphere
         * @param worldMatrix defines the new world matrix
         */
        constructor(minimum: DeepImmutable<Vector3>, maximum: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>);
        /**
         * Recreates the entire bounding info from scratch as if we call the constructor in place
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        reConstruct(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>): void;
        /**
         * min vector of the bounding box/sphere
         */
        get minimum(): Vector3;
        /**
         * max vector of the bounding box/sphere
         */
        get maximum(): Vector3;
        /**
         * If the info is locked and won't be updated to avoid perf overhead
         */
        get isLocked(): boolean;
        set isLocked(value: boolean);
        /**
         * Updates the bounding sphere and box
         * @param world world matrix to be used to update
         */
        update(world: DeepImmutable<Matrix>): void;
        /**
         * Recreate the bounding info to be centered around a specific point given a specific extend.
         * @param center New center of the bounding info
         * @param extend New extend of the bounding info
         * @returns the current bounding info
         */
        centerOn(center: DeepImmutable<Vector3>, extend: DeepImmutable<Vector3>): BoundingInfo;
        /**
         * Grows the bounding info to include the given point.
         * @param point The point that will be included in the current bounding info (in local space)
         * @returns the current bounding info
         */
        encapsulate(point: Vector3): BoundingInfo;
        /**
         * Grows the bounding info to encapsulate the given bounding info.
         * @param toEncapsulate The bounding info that will be encapsulated in the current bounding info
         * @returns the current bounding info
         */
        encapsulateBoundingInfo(toEncapsulate: BoundingInfo): BoundingInfo;
        /**
         * Scale the current bounding info by applying a scale factor
         * @param factor defines the scale factor to apply
         * @returns the current bounding info
         */
        scale(factor: number): BoundingInfo;
        /**
         * Returns `true` if the bounding info is within the frustum defined by the passed array of planes.
         * @param frustumPlanes defines the frustum to test
         * @param strategy defines the strategy to use for the culling (default is BABYLON.AbstractMesh.CULLINGSTRATEGY_STANDARD)
         * The different strategies available are:
         * * BABYLON.AbstractMesh.CULLINGSTRATEGY_STANDARD most accurate but slower @see https://doc.babylonjs.com/typedoc/classes/BABYLON.AbstractMesh#CULLINGSTRATEGY_STANDARD
         * * BABYLON.AbstractMesh.CULLINGSTRATEGY_BOUNDINGSPHERE_ONLY faster but less accurate @see https://doc.babylonjs.com/typedoc/classes/BABYLON.AbstractMesh#CULLINGSTRATEGY_BOUNDINGSPHERE_ONLY
         * * BABYLON.AbstractMesh.CULLINGSTRATEGY_OPTIMISTIC_INCLUSION can be faster if always visible @see https://doc.babylonjs.com/typedoc/classes/BABYLON.AbstractMesh#CULLINGSTRATEGY_OPTIMISTIC_INCLUSION
         * * BABYLON.AbstractMesh.CULLINGSTRATEGY_OPTIMISTIC_INCLUSION_THEN_BSPHERE_ONLY can be faster if always visible @see https://doc.babylonjs.com/typedoc/classes/BABYLON.AbstractMesh#CULLINGSTRATEGY_OPTIMISTIC_INCLUSION_THEN_BSPHERE_ONLY
         * @returns true if the bounding info is in the frustum planes
         */
        isInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>, strategy?: number): boolean;
        /**
         * Gets the world distance between the min and max points of the bounding box
         */
        get diagonalLength(): number;
        /**
         * Checks if a cullable object (mesh...) is in the camera frustum
         * Unlike isInFrustum this checks the full bounding box
         * @param frustumPlanes Camera near/planes
         * @returns true if the object is in frustum otherwise false
         */
        isCompletelyInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * @internal
         */
        _checkCollision(collider: Collider): boolean;
        /**
         * Checks if a point is inside the bounding box and bounding sphere or the mesh
         * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/interactions/mesh_intersect
         * @param point the point to check intersection with
         * @returns if the point intersects
         */
        intersectsPoint(point: DeepImmutable<Vector3>): boolean;
        /**
         * Checks if another bounding info intersects the bounding box and bounding sphere or the mesh
         * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/interactions/mesh_intersect
         * @param boundingInfo the bounding info to check intersection with
         * @param precise if the intersection should be done using OBB
         * @returns if the bounding info intersects
         */
        intersects(boundingInfo: DeepImmutable<BoundingInfo>, precise: boolean): boolean;
    }


    /**
     * Class used to store bounding sphere information
     */
    export class BoundingSphere {
        /**
         * Gets the center of the bounding sphere in local space
         */
        readonly center: Vector3;
        /**
         * Radius of the bounding sphere in local space
         */
        radius: number;
        /**
         * Gets the center of the bounding sphere in world space
         */
        readonly centerWorld: Vector3;
        /**
         * Radius of the bounding sphere in world space
         */
        radiusWorld: number;
        /**
         * Gets the minimum vector in local space
         */
        readonly minimum: Vector3;
        /**
         * Gets the maximum vector in local space
         */
        readonly maximum: Vector3;
        private _worldMatrix;
        private static readonly _TmpVector3;
        /**
         * Creates a new bounding sphere
         * @param min defines the minimum vector (in local space)
         * @param max defines the maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        constructor(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>);
        /**
         * Recreates the entire bounding sphere from scratch as if we call the constructor in place
         * @param min defines the new minimum vector (in local space)
         * @param max defines the new maximum vector (in local space)
         * @param worldMatrix defines the new world matrix
         */
        reConstruct(min: DeepImmutable<Vector3>, max: DeepImmutable<Vector3>, worldMatrix?: DeepImmutable<Matrix>): void;
        /**
         * Scale the current bounding sphere by applying a scale factor
         * @param factor defines the scale factor to apply
         * @returns the current bounding box
         */
        scale(factor: number): BoundingSphere;
        /**
         * Gets the world matrix of the bounding box
         * @returns a matrix
         */
        getWorldMatrix(): DeepImmutable<Matrix>;
        /**
         * @internal
         */
        _update(worldMatrix: DeepImmutable<Matrix>): void;
        /**
         * Tests if the bounding sphere is intersecting the frustum planes
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if there is an intersection
         */
        isInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if the bounding sphere center is in between the frustum planes.
         * Used for optimistic fast inclusion.
         * @param frustumPlanes defines the frustum planes to test
         * @returns true if the sphere center is in between the frustum planes
         */
        isCenterInFrustum(frustumPlanes: Array<DeepImmutable<Plane>>): boolean;
        /**
         * Tests if a point is inside the bounding sphere
         * @param point defines the point to test
         * @returns true if the point is inside the bounding sphere
         */
        intersectsPoint(point: DeepImmutable<Vector3>): boolean;
        /**
         * Checks if two sphere intersect
         * @param sphere0 sphere 0
         * @param sphere1 sphere 1
         * @returns true if the spheres intersect
         */
        static Intersects(sphere0: DeepImmutable<BoundingSphere>, sphere1: DeepImmutable<BoundingSphere>): boolean;
        /**
         * Creates a sphere from a center and a radius
         * @param center The center
         * @param radius radius
         * @param matrix Optional worldMatrix
         * @returns The sphere
         */
        static CreateFromCenterAndRadius(center: DeepImmutable<Vector3>, radius: number, matrix?: DeepImmutable<Matrix>): BoundingSphere;
    }






    /**
     * Octrees are a really powerful data structure that can quickly select entities based on space coordinates.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimizeOctrees
     */
    export class Octree<T> {
        /** Defines the maximum depth (sub-levels) for your octree. Default value is 2, which means 8 8 8 = 512 blocks :) (This parameter takes precedence over capacity.) */
        maxDepth: number;
        /**
         * Blocks within the octree containing objects
         */
        blocks: Array<OctreeBlock<T>>;
        /**
         * Content stored in the octree
         */
        dynamicContent: T[];
        private _maxBlockCapacity;
        private _selectionContent;
        private _creationFunc;
        /**
         * Creates a octree
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimizeOctrees
         * @param creationFunc function to be used to instantiate the octree
         * @param maxBlockCapacity defines the maximum number of meshes you want on your octree's leaves (default: 64)
         * @param maxDepth defines the maximum depth (sub-levels) for your octree. Default value is 2, which means 8 8 8 = 512 blocks :) (This parameter takes precedence over capacity.)
         */
        constructor(creationFunc: (entry: T, block: OctreeBlock<T>) => void, maxBlockCapacity?: number, 
        /** Defines the maximum depth (sub-levels) for your octree. Default value is 2, which means 8 8 8 = 512 blocks :) (This parameter takes precedence over capacity.) */
        maxDepth?: number);
        /**
         * Updates the octree by adding blocks for the passed in meshes within the min and max world parameters
         * @param worldMin worldMin for the octree blocks var blockSize = new Vector3((worldMax.x - worldMin.x) / 2, (worldMax.y - worldMin.y) / 2, (worldMax.z - worldMin.z) / 2);
         * @param worldMax worldMax for the octree blocks var blockSize = new Vector3((worldMax.x - worldMin.x) / 2, (worldMax.y - worldMin.y) / 2, (worldMax.z - worldMin.z) / 2);
         * @param entries meshes to be added to the octree blocks
         */
        update(worldMin: Vector3, worldMax: Vector3, entries: T[]): void;
        /**
         * Adds a mesh to the octree
         * @param entry Mesh to add to the octree
         */
        addMesh(entry: T): void;
        /**
         * Remove an element from the octree
         * @param entry defines the element to remove
         */
        removeMesh(entry: T): void;
        /**
         * Selects an array of meshes within the frustum
         * @param frustumPlanes The frustum planes to use which will select all meshes within it
         * @param allowDuplicate If duplicate objects are allowed in the resulting object array
         * @returns array of meshes within the frustum
         */
        select(frustumPlanes: Plane[], allowDuplicate?: boolean): SmartArray<T>;
        /**
         * Test if the octree intersect with the given bounding sphere and if yes, then add its content to the selection array
         * @param sphereCenter defines the bounding sphere center
         * @param sphereRadius defines the bounding sphere radius
         * @param allowDuplicate defines if the selection array can contains duplicated entries
         * @returns an array of objects that intersect the sphere
         */
        intersects(sphereCenter: Vector3, sphereRadius: number, allowDuplicate?: boolean): SmartArray<T>;
        /**
         * Test if the octree intersect with the given ray and if yes, then add its content to resulting array
         * @param ray defines the ray to test with
         * @returns array of intersected objects
         */
        intersectsRay(ray: Ray): SmartArray<T>;
        /**
         * Adds a mesh into the octree block if it intersects the block
         * @param entry defines the mesh to try to add to the block
         * @param block defines the block where the mesh should be added
         */
        static CreationFuncForMeshes: (entry: AbstractMesh, block: OctreeBlock<AbstractMesh>) => void;
        /**
         * Adds a submesh into the octree block if it intersects the block
         * @param entry defines the submesh to try to add to the block
         * @param block defines the block where the submesh should be added
         */
        static CreationFuncForSubMeshes: (entry: SubMesh, block: OctreeBlock<SubMesh>) => void;
    }


    /**
     * Contains an array of blocks representing the octree
     */
    export interface IOctreeContainer<T> {
        /**
         * Blocks within the octree
         */
        blocks: Array<OctreeBlock<T>>;
    }
    /**
     * Class used to store a cell in an octree
     * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimizeOctrees
     */
    export class OctreeBlock<T> {
        /**
         * Gets the content of the current block
         */
        entries: T[];
        /**
         * Gets the list of block children
         */
        blocks: Array<OctreeBlock<T>>;
        private _depth;
        private _maxDepth;
        private _capacity;
        private _minPoint;
        private _maxPoint;
        private _boundingVectors;
        private _creationFunc;
        /**
         * Creates a new block
         * @param minPoint defines the minimum vector (in world space) of the block's bounding box
         * @param maxPoint defines the maximum vector (in world space) of the block's bounding box
         * @param capacity defines the maximum capacity of this block (if capacity is reached the block will be split into sub blocks)
         * @param depth defines the current depth of this block in the octree
         * @param maxDepth defines the maximal depth allowed (beyond this value, the capacity is ignored)
         * @param creationFunc defines a callback to call when an element is added to the block
         */
        constructor(minPoint: Vector3, maxPoint: Vector3, capacity: number, depth: number, maxDepth: number, creationFunc: (entry: T, block: OctreeBlock<T>) => void);
        /**
         * Gets the maximum capacity of this block (if capacity is reached the block will be split into sub blocks)
         */
        get capacity(): number;
        /**
         * Gets the minimum vector (in world space) of the block's bounding box
         */
        get minPoint(): Vector3;
        /**
         * Gets the maximum vector (in world space) of the block's bounding box
         */
        get maxPoint(): Vector3;
        /**
         * Add a new element to this block
         * @param entry defines the element to add
         */
        addEntry(entry: T): void;
        /**
         * Remove an element from this block
         * @param entry defines the element to remove
         */
        removeEntry(entry: T): void;
        /**
         * Add an array of elements to this block
         * @param entries defines the array of elements to add
         */
        addEntries(entries: T[]): void;
        /**
         * Test if the current block intersects the frustum planes and if yes, then add its content to the selection array
         * @param frustumPlanes defines the frustum planes to test
         * @param selection defines the array to store current content if selection is positive
         * @param allowDuplicate defines if the selection array can contains duplicated entries
         */
        select(frustumPlanes: Plane[], selection: SmartArrayNoDuplicate<T>, allowDuplicate?: boolean): void;
        /**
         * Test if the current block intersect with the given bounding sphere and if yes, then add its content to the selection array
         * @param sphereCenter defines the bounding sphere center
         * @param sphereRadius defines the bounding sphere radius
         * @param selection defines the array to store current content if selection is positive
         * @param allowDuplicate defines if the selection array can contains duplicated entries
         */
        intersects(sphereCenter: Vector3, sphereRadius: number, selection: SmartArrayNoDuplicate<T>, allowDuplicate?: boolean): void;
        /**
         * Test if the current block intersect with the given ray and if yes, then add its content to the selection array
         * @param ray defines the ray to test with
         * @param selection defines the array to store current content if selection is positive
         */
        intersectsRay(ray: Ray, selection: SmartArrayNoDuplicate<T>): void;
        /**
         * Subdivide the content into child blocks (this block will then be empty)
         */
        createInnerBlocks(): void;
        /**
         * @internal
         */
        static _CreateBlocks<T>(worldMin: Vector3, worldMax: Vector3, entries: T[], maxBlockCapacity: number, currentDepth: number, maxDepth: number, target: IOctreeContainer<T>, creationFunc: (entry: T, block: OctreeBlock<T>) => void): void;
    }


        interface Scene {
            /**
             * @internal
             * Backing Filed
             */
            _selectionOctree: Octree<AbstractMesh>;
            /**
             * Gets the octree used to boost mesh selection (picking)
             * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimizeOctrees
             */
            selectionOctree: Octree<AbstractMesh>;
            /**
             * Creates or updates the octree used to boost selection (picking)
             * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimizeOctrees
             * @param maxCapacity defines the maximum capacity per leaf
             * @param maxDepth defines the maximum depth of the octree
             * @returns an octree of AbstractMesh
             */
            createOrUpdateSelectionOctree(maxCapacity?: number, maxDepth?: number): Octree<AbstractMesh>;
        }
        interface AbstractMesh {
            /**
             * @internal
             * Backing Field
             */
            _submeshesOctree: Octree<SubMesh>;
            /**
             * This function will create an octree to help to select the right submeshes for rendering, picking and collision computations.
             * Please note that you must have a decent number of submeshes to get performance improvements when using an octree
             * @param maxCapacity defines the maximum size of each block (64 by default)
             * @param maxDepth defines the maximum depth to use (no more than 2 levels by default)
             * @returns the new octree
             * @see https://www.babylonjs-playground.com/#NA4OQ#12
             * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimizeOctrees
             */
            createOrUpdateSubmeshesOctree(maxCapacity?: number, maxDepth?: number): Octree<SubMesh>;
        }
    /**
     * Defines the octree scene component responsible to manage any octrees
     * in a given scene.
     */
    export class OctreeSceneComponent {
        /**
         * The component name help to identify the component in the list of scene components.
         */
        readonly name = "Octree";
        /**
         * The scene the component belongs to.
         */
        scene: Scene;
        /**
         * Indicates if the meshes have been checked to make sure they are isEnabled()
         */
        readonly checksIsEnabled = true;
        /**
         * Creates a new instance of the component for the given scene
         * @param scene Defines the scene to register the component in
         */
        constructor(scene?: Scene);
        /**
         * Registers the component in a given scene
         */
        register(): void;
        /**
         * Return the list of active meshes
         * @returns the list of active meshes
         */
        getActiveMeshCandidates(): ISmartArrayLike<AbstractMesh>;
        /**
         * Return the list of active sub meshes
         * @param mesh The mesh to get the candidates sub meshes from
         * @returns the list of active sub meshes
         */
        getActiveSubMeshCandidates(mesh: AbstractMesh): ISmartArrayLike<SubMesh>;
        private _tempRay;
        /**
         * Return the list of sub meshes intersecting with a given local ray
         * @param mesh defines the mesh to find the submesh for
         * @param localRay defines the ray in local space
         * @returns the list of intersecting sub meshes
         */
        getIntersectingSubMeshCandidates(mesh: AbstractMesh, localRay: Ray): ISmartArrayLike<SubMesh>;
        /**
         * Return the list of sub meshes colliding with a collider
         * @param mesh defines the mesh to find the submesh for
         * @param collider defines the collider to evaluate the collision against
         * @returns the list of colliding sub meshes
         */
        getCollidingSubMeshCandidates(mesh: AbstractMesh, collider: Collider): ISmartArrayLike<SubMesh>;
        /**
         * Rebuilds the elements related to this component in case of
         * context lost for instance.
         */
        rebuild(): void;
        /**
         * Disposes the component and the associated resources.
         */
        dispose(): void;
    }


    /**
     * Class representing a ray with position and direction
     */
    export class Ray {
        /** origin point */
        origin: Vector3;
        /** direction */
        direction: Vector3;
        /** length of the ray */
        length: number;
        /** The epsilon value to use when calculating the ray/triangle intersection (default: Epsilon from math constants) */
        epsilon: number;
        private static readonly _TmpVector3;
        private static _RayDistant;
        private _tmpRay;
        /**
         * Creates a new ray
         * @param origin origin point
         * @param direction direction
         * @param length length of the ray
         * @param epsilon The epsilon value to use when calculating the ray/triangle intersection (default: 0)
         */
        constructor(
        /** origin point */
        origin: Vector3, 
        /** direction */
        direction: Vector3, 
        /** length of the ray */
        length?: number, 
        /** The epsilon value to use when calculating the ray/triangle intersection (default: Epsilon from math constants) */
        epsilon?: number);
        /**
         * Clone the current ray
         * @returns a new ray
         */
        clone(): Ray;
        /**
         * Checks if the ray intersects a box
         * This does not account for the ray length by design to improve perfs.
         * @param minimum bound of the box
         * @param maximum bound of the box
         * @param intersectionTreshold extra extend to be added to the box in all direction
         * @returns if the box was hit
         */
        intersectsBoxMinMax(minimum: DeepImmutable<Vector3>, maximum: DeepImmutable<Vector3>, intersectionTreshold?: number): boolean;
        /**
         * Checks if the ray intersects a box
         * This does not account for the ray lenght by design to improve perfs.
         * @param box the bounding box to check
         * @param intersectionTreshold extra extend to be added to the BoundingBox in all direction
         * @returns if the box was hit
         */
        intersectsBox(box: DeepImmutable<BoundingBox>, intersectionTreshold?: number): boolean;
        /**
         * If the ray hits a sphere
         * @param sphere the bounding sphere to check
         * @param intersectionTreshold extra extend to be added to the BoundingSphere in all direction
         * @returns true if it hits the sphere
         */
        intersectsSphere(sphere: DeepImmutable<BoundingSphere>, intersectionTreshold?: number): boolean;
        /**
         * If the ray hits a triange
         * @param vertex0 triangle vertex
         * @param vertex1 triangle vertex
         * @param vertex2 triangle vertex
         * @returns intersection information if hit
         */
        intersectsTriangle(vertex0: DeepImmutable<Vector3>, vertex1: DeepImmutable<Vector3>, vertex2: DeepImmutable<Vector3>): Nullable<IntersectionInfo>;
        /**
         * Checks if ray intersects a plane
         * @param plane the plane to check
         * @returns the distance away it was hit
         */
        intersectsPlane(plane: DeepImmutable<Plane>): Nullable<number>;
        /**
         * Calculate the intercept of a ray on a given axis
         * @param axis to check 'x' | 'y' | 'z'
         * @param offset from axis interception (i.e. an offset of 1y is intercepted above ground)
         * @returns a vector containing the coordinates where 'axis' is equal to zero (else offset), or null if there is no intercept.
         */
        intersectsAxis(axis: string, offset?: number): Nullable<Vector3>;
        /**
         * Checks if ray intersects a mesh. The ray is defined in WORLD space. A mesh triangle can be picked both from its front and back sides,
         * irrespective of orientation.
         * @param mesh the mesh to check
         * @param fastCheck defines if the first intersection will be used (and not the closest)
         * @param trianglePredicate defines an optional predicate used to select faces when a mesh intersection is detected
         * @param onlyBoundingInfo defines a boolean indicating if picking should only happen using bounding info (false by default)
         * @param worldToUse defines the world matrix to use to get the world coordinate of the intersection point
         * @param skipBoundingInfo a boolean indicating if we should skip the bounding info check
         * @returns picking info of the intersection
         */
        intersectsMesh(mesh: DeepImmutable<AbstractMesh>, fastCheck?: boolean, trianglePredicate?: TrianglePickingPredicate, onlyBoundingInfo?: boolean, worldToUse?: Matrix, skipBoundingInfo?: boolean): PickingInfo;
        /**
         * Checks if ray intersects a mesh
         * @param meshes the meshes to check
         * @param fastCheck defines if the first intersection will be used (and not the closest)
         * @param results array to store result in
         * @returns Array of picking infos
         */
        intersectsMeshes(meshes: Array<DeepImmutable<AbstractMesh>>, fastCheck?: boolean, results?: Array<PickingInfo>): Array<PickingInfo>;
        private _comparePickingInfo;
        private static _Smallnum;
        private static _Rayl;
        /**
         * Intersection test between the ray and a given segment within a given tolerance (threshold)
         * @param sega the first point of the segment to test the intersection against
         * @param segb the second point of the segment to test the intersection against
         * @param threshold the tolerance margin, if the ray doesn't intersect the segment but is close to the given threshold, the intersection is successful
         * @returns the distance from the ray origin to the intersection point if there's intersection, or -1 if there's no intersection
         */
        intersectionSegment(sega: DeepImmutable<Vector3>, segb: DeepImmutable<Vector3>, threshold: number): number;
        /**
         * Update the ray from viewport position
         * @param x position
         * @param y y position
         * @param viewportWidth viewport width
         * @param viewportHeight viewport height
         * @param world world matrix
         * @param view view matrix
         * @param projection projection matrix
         * @param enableDistantPicking defines if picking should handle large values for mesh position/scaling (false by default)
         * @returns this ray updated
         */
        update(x: number, y: number, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>, enableDistantPicking?: boolean): Ray;
        /**
         * Creates a ray with origin and direction of 0,0,0
         * @returns the new ray
         */
        static Zero(): Ray;
        /**
         * Creates a new ray from screen space and viewport
         * @param x position
         * @param y y position
         * @param viewportWidth viewport width
         * @param viewportHeight viewport height
         * @param world world matrix
         * @param view view matrix
         * @param projection projection matrix
         * @returns new ray
         */
        static CreateNew(x: number, y: number, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>): Ray;
        /**
         * Function will create a new transformed ray starting from origin and ending at the end point. Ray's length will be set, and ray will be
         * transformed to the given world matrix.
         * @param origin The origin point
         * @param end The end point
         * @param world a matrix to transform the ray to. Default is the identity matrix.
         * @returns the new ray
         */
        static CreateNewFromTo(origin: Vector3, end: Vector3, world?: DeepImmutable<Matrix>): Ray;
        /**
         * Function will update a transformed ray starting from origin and ending at the end point. Ray's length will be set, and ray will be
         * transformed to the given world matrix.
         * @param origin The origin point
         * @param end The end point
         * @param result the object to store the result
         * @param world a matrix to transform the ray to. Default is the identity matrix.
         * @returns the ref ray
         */
        static CreateFromToToRef(origin: Vector3, end: Vector3, result: Ray, world?: DeepImmutable<Matrix>): Ray;
        /**
         * Transforms a ray by a matrix
         * @param ray ray to transform
         * @param matrix matrix to apply
         * @returns the resulting new ray
         */
        static Transform(ray: DeepImmutable<Ray>, matrix: DeepImmutable<Matrix>): Ray;
        /**
         * Transforms a ray by a matrix
         * @param ray ray to transform
         * @param matrix matrix to apply
         * @param result ray to store result in
         * @returns the updated result ray
         */
        static TransformToRef(ray: DeepImmutable<Ray>, matrix: DeepImmutable<Matrix>, result: Ray): Ray;
        /**
         * Unproject a ray from screen space to object space
         * @param sourceX defines the screen space x coordinate to use
         * @param sourceY defines the screen space y coordinate to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         */
        unprojectRayToRef(sourceX: float, sourceY: float, viewportWidth: number, viewportHeight: number, world: DeepImmutable<Matrix>, view: DeepImmutable<Matrix>, projection: DeepImmutable<Matrix>): void;
    }
    /**
     * Type used to define predicate used to select faces when a mesh intersection is detected
     */
    export type TrianglePickingPredicate = (p0: Vector3, p1: Vector3, p2: Vector3, ray: Ray, i0: number, i1: number, i2: number) => boolean;
        interface Scene {
            /** @internal */
            _tempPickingRay: Nullable<Ray>;
            /** @internal */
            _cachedRayForTransform: Ray;
            /** @internal */
            _pickWithRayInverseMatrix: Matrix;
            /** @internal */
            _internalPick(rayFunction: (world: Matrix, enableDistantPicking: boolean) => Ray, predicate?: (mesh: AbstractMesh) => boolean, fastCheck?: boolean, onlyBoundingInfo?: boolean, trianglePredicate?: TrianglePickingPredicate): PickingInfo;
            /** @internal */
            _internalMultiPick(rayFunction: (world: Matrix, enableDistantPicking: boolean) => Ray, predicate?: (mesh: AbstractMesh) => boolean, trianglePredicate?: TrianglePickingPredicate): Nullable<PickingInfo[]>;
            /** @internal */
            _internalPickForMesh(pickingInfo: Nullable<PickingInfo>, rayFunction: (world: Matrix, enableDistantPicking: boolean) => Ray, mesh: AbstractMesh, world: Matrix, fastCheck?: boolean, onlyBoundingInfo?: boolean, trianglePredicate?: TrianglePickingPredicate, skipBoundingInfo?: boolean): Nullable<PickingInfo>;
        }



}
declare module BABYLON.Debug {
        /**
     * The Axes viewer will show 3 axes in a specific point in space
     * @see https://doc.babylonjs.com/toolsAndResources/utilities/World_Axes
     */
    export class AxesViewer {
        private _xAxis;
        private _yAxis;
        private _zAxis;
        private _scaleLinesFactor;
        private _instanced;
        /**
         * Gets the hosting scene
         */
        scene: Nullable<Scene>;
        private _scaleLines;
        /**
         * Gets or sets a number used to scale line length
         */
        get scaleLines(): number;
        set scaleLines(value: number);
        /** Gets the node hierarchy used to render x-axis */
        get xAxis(): TransformNode;
        /** Gets the node hierarchy used to render y-axis */
        get yAxis(): TransformNode;
        /** Gets the node hierarchy used to render z-axis */
        get zAxis(): TransformNode;
        /**
         * Creates a new AxesViewer
         * @param scene defines the hosting scene
         * @param scaleLines defines a number used to scale line length (1 by default)
         * @param renderingGroupId defines a number used to set the renderingGroupId of the meshes (2 by default)
         * @param xAxis defines the node hierarchy used to render the x-axis
         * @param yAxis defines the node hierarchy used to render the y-axis
         * @param zAxis defines the node hierarchy used to render the z-axis
         * @param lineThickness The line thickness to use when creating the arrow. defaults to 1.
         */
        constructor(scene?: Scene, scaleLines?: number, renderingGroupId?: Nullable<number>, xAxis?: TransformNode, yAxis?: TransformNode, zAxis?: TransformNode, lineThickness?: number);
        /**
         * Force the viewer to update
         * @param position defines the position of the viewer
         * @param xaxis defines the x axis of the viewer
         * @param yaxis defines the y axis of the viewer
         * @param zaxis defines the z axis of the viewer
         */
        update(position: Vector3, xaxis: Vector3, yaxis: Vector3, zaxis: Vector3): void;
        /**
         * Creates an instance of this axes viewer.
         * @returns a new axes viewer with instanced meshes
         */
        createInstance(): AxesViewer;
        /** Releases resources */
        dispose(): void;
        private static _SetRenderingGroupId;
    }



}
declare module BABYLON {

}
declare module BABYLON.Debug {
        /**
     * The BoneAxesViewer will attach 3 axes to a specific bone of a specific mesh
     * @see demo here: https://www.babylonjs-playground.com/#0DE8F4#8
     */
    export class BoneAxesViewer extends BABYLON.Debug.AxesViewer {
        /**
         * Gets or sets the target mesh where to display the axes viewer
         */
        mesh: Nullable<Mesh>;
        /**
         * Gets or sets the target bone where to display the axes viewer
         */
        bone: Nullable<Bone>;
        /** Gets current position */
        pos: Vector3;
        /** Gets direction of X axis */
        xaxis: Vector3;
        /** Gets direction of Y axis */
        yaxis: Vector3;
        /** Gets direction of Z axis */
        zaxis: Vector3;
        /**
         * Creates a new BoneAxesViewer
         * @param scene defines the hosting scene
         * @param bone defines the target bone
         * @param mesh defines the target mesh
         * @param scaleLines defines a scaling factor for line length (1 by default)
         */
        constructor(scene: Scene, bone: Bone, mesh: Mesh, scaleLines?: number);
        /**
         * Force the viewer to update
         */
        update(): void;
        /** Releases resources */
        dispose(): void;
    }



}
declare module BABYLON {
    /**
     * Interface used to define scene explorer extensibility option
     */
    export interface IExplorerExtensibilityOption {
        /**
         * Define the option label
         */
        label: string;
        /**
         * Defines the action to execute on click
         */
        action: (entity: any) => void;
        /**
         * Keep popup open after click
         */
        keepOpenAfterClick?: boolean;
    }
    /**
     * Defines a group of actions associated with a predicate to use when extending the Inspector scene explorer
     */
    export interface IExplorerExtensibilityGroup {
        /**
         * Defines a predicate to test if a given type mut be extended
         */
        predicate: (entity: any) => boolean;
        /**
         * Gets the list of options added to a type
         */
        entries: IExplorerExtensibilityOption[];
    }
    /**
     * Defines a new node that will be displayed as top level node in the explorer
     */
    export interface IExplorerAdditionalChild {
        /**
         * Gets the name of the additional node
         */
        name: string;
        /**
         * Function used to return the class name of the child node
         */
        getClassName(): string;
        /**
         * List of inspectable custom properties (used by the Inspector)
         * @see https://doc.babylonjs.com/toolsAndResources/inspector#extensibility
         */
        inspectableCustomProperties: IInspectable[];
    }
    /**
     * Defines a new node that will be displayed as top level node in the explorer
     */
    export interface IExplorerAdditionalNode {
        /**
         * Gets the name of the additional node
         */
        name: string;
        /**
         * Function used to return the list of child entries
         */
        getContent(): IExplorerAdditionalChild[];
    }
    export type IInspectorContextMenuType = "pipeline" | "node" | "materials" | "spriteManagers" | "particleSystems";
    /**
     * Context menu item
     */
    export interface IInspectorContextMenuItem {
        /**
         * Display label - menu item
         */
        label: string;
        /**
         * Callback function that will be called when the menu item is selected
         * @param entity the entity that is currently selected in the scene explorer
         */
        action: (entity?: unknown) => void;
    }
    /**
     * Interface used to define the options to use to create the Inspector
     */
    export interface IInspectorOptions {
        /**
         * Display in overlay mode (default: false)
         */
        overlay?: boolean;
        /**
         * HTML element to use as root (the parent of the rendering canvas will be used as default value)
         */
        globalRoot?: HTMLElement;
        /**
         * Display the Scene explorer
         */
        showExplorer?: boolean;
        /**
         * Display the property inspector
         */
        showInspector?: boolean;
        /**
         * Display in embed mode (both panes on the right)
         */
        embedMode?: boolean;
        /**
         * let the Inspector handles resize of the canvas when panes are resized (default to true)
         */
        handleResize?: boolean;
        /**
         * Allow the panes to popup (default: true)
         */
        enablePopup?: boolean;
        /**
         * Allow the panes to be closed by users (default: true)
         */
        enableClose?: boolean;
        /**
         * Optional list of extensibility entries
         */
        explorerExtensibility?: IExplorerExtensibilityGroup[];
        /**
         * Optional list of additional top level nodes
         */
        additionalNodes?: IExplorerAdditionalNode[];
        /**
         * Optional URL to get the inspector script from (by default it uses the babylonjs CDN).
         */
        inspectorURL?: string;
        /**
         * Optional initial tab (default to DebugLayerTab.Properties)
         */
        initialTab?: DebugLayerTab;
        /**
         * Optional camera to use to render the gizmos from the inspector (default to the scene.activeCamera or the latest from scene.activeCameras)
         */
        gizmoCamera?: Camera;
        /**
         * Context menu for inspector tools such as "Post Process", "Nodes", "Materials", etc.
         */
        contextMenu?: Partial<Record<IInspectorContextMenuType, IInspectorContextMenuItem[]>>;
        /**
         * List of context menu items that should be completely overridden by custom items from the contextMenu property.
         */
        contextMenuOverride?: IInspectorContextMenuType[];
    }
        interface Scene {
            /**
             * @internal
             * Backing field
             */
            _debugLayer: DebugLayer;
            /**
             * Gets the debug layer (aka Inspector) associated with the scene
             * @see https://doc.babylonjs.com/toolsAndResources/inspector
             */
            debugLayer: DebugLayer;
        }
    /**
     * Enum of inspector action tab
     */
    export enum DebugLayerTab {
        /**
         * Properties tag (default)
         */
        Properties = 0,
        /**
         * Debug tab
         */
        Debug = 1,
        /**
         * Statistics tab
         */
        Statistics = 2,
        /**
         * Tools tab
         */
        Tools = 3,
        /**
         * Settings tab
         */
        Settings = 4
    }
    /**
     * The debug layer (aka Inspector) is the go to tool in order to better understand
     * what is happening in your scene
     * @see https://doc.babylonjs.com/toolsAndResources/inspector
     */
    export class DebugLayer {
        /**
         * Define the url to get the inspector script from.
         * By default it uses the babylonjs CDN.
         * @ignoreNaming
         */
        static InspectorURL: string;
        /**
         * The default configuration of the inspector
         */
        static Config: IInspectorOptions;
        private _scene;
        private BJSINSPECTOR;
        private _onPropertyChangedObservable?;
        /**
         * Observable triggered when a property is changed through the inspector.
         */
        get onPropertyChangedObservable(): any;
        private _onSelectionChangedObservable?;
        /**
         * Observable triggered when the selection is changed through the inspector.
         */
        get onSelectionChangedObservable(): any;
        /**
         * Instantiates a new debug layer.
         * The debug layer (aka Inspector) is the go to tool in order to better understand
         * what is happening in your scene
         * @see https://doc.babylonjs.com/toolsAndResources/inspector
         * @param scene Defines the scene to inspect
         */
        constructor(scene?: Scene);
        /**
         * Creates the inspector window.
         * @param config
         */
        private _createInspector;
        /**
         * Select a specific entity in the scene explorer and highlight a specific block in that entity property grid
         * @param entity defines the entity to select
         * @param lineContainerTitles defines the specific blocks to highlight (could be a string or an array of strings)
         */
        select(entity: any, lineContainerTitles?: string | string[]): void;
        /**
         * Get the inspector from bundle or global
         * @returns the inspector instance if found otherwise, null
         */
        private _getGlobalInspector;
        /**
         * Get if the inspector is visible or not.
         * @returns true if visible otherwise, false
         */
        isVisible(): boolean;
        /**
         * Hide the inspector and close its window.
         */
        hide(): void;
        /**
         * Update the scene in the inspector
         */
        setAsActiveScene(): void;
        /**
         * Launch the debugLayer.
         * @param config Define the configuration of the inspector
         * @returns a promise fulfilled when the debug layer is visible
         */
        show(config?: IInspectorOptions): Promise<DebugLayer>;
    }


    /**
     * Class used to render a debug view of the frustum for a directional light
     * @see https://playground.babylonjs.com/#7EFGSG#4
     * @since 5.0.0
     */
    export class DirectionalLightFrustumViewer {
        private _scene;
        private _light;
        private _camera;
        private _inverseViewMatrix;
        private _visible;
        private _rootNode;
        private _lightHelperFrustumMeshes;
        private _nearLinesPoints;
        private _farLinesPoints;
        private _trLinesPoints;
        private _brLinesPoints;
        private _tlLinesPoints;
        private _blLinesPoints;
        private _nearPlaneVertices;
        private _farPlaneVertices;
        private _rightPlaneVertices;
        private _leftPlaneVertices;
        private _topPlaneVertices;
        private _bottomPlaneVertices;
        private _oldPosition;
        private _oldDirection;
        private _oldAutoCalc;
        private _oldMinZ;
        private _oldMaxZ;
        private _transparency;
        /**
         * Gets or sets the transparency of the frustum planes
         */
        get transparency(): number;
        set transparency(alpha: number);
        private _showLines;
        /**
         * true to display the edges of the frustum
         */
        get showLines(): boolean;
        set showLines(show: boolean);
        private _showPlanes;
        /**
         * true to display the planes of the frustum
         */
        get showPlanes(): boolean;
        set showPlanes(show: boolean);
        /**
         * Creates a new frustum viewer
         * @param light directional light to display the frustum for
         * @param camera camera used to retrieve the minZ / maxZ values if the shadowMinZ/shadowMaxZ values of the light are not setup
         */
        constructor(light: DirectionalLight, camera: Camera);
        /**
         * Shows the frustum
         */
        show(): void;
        /**
         * Hides the frustum
         */
        hide(): void;
        /**
         * Updates the frustum.
         * Call this method to update the frustum view if the light has changed position/direction
         */
        update(): void;
        /**
         * Dispose of the class / remove the frustum view
         */
        dispose(): void;
        protected _createGeometry(): void;
        protected _getInvertViewMatrix(): Matrix;
    }




    /**
     * Defines the options associated with the creation of a SkeletonViewer.
     */
    export interface ISkeletonViewerOptions {
        /** Should the system pause animations before building the Viewer? */
        pauseAnimations: boolean;
        /** Should the system return the skeleton to rest before building? */
        returnToRest: boolean;
        /** public Display Mode of the Viewer */
        displayMode: number;
        /** Flag to toggle if the Viewer should use the CPU for animations or not? */
        displayOptions: ISkeletonViewerDisplayOptions;
        /** Flag to toggle if the Viewer should use the CPU for animations or not? */
        computeBonesUsingShaders: boolean;
        /** Flag ignore non weighted bones */
        useAllBones: boolean;
    }
    /**
     * Defines how to display the various bone meshes for the viewer.
     */
    export interface ISkeletonViewerDisplayOptions {
        /** How far down to start tapering the bone spurs */
        midStep?: number;
        /** How big is the midStep? */
        midStepFactor?: number;
        /** Base for the Sphere Size */
        sphereBaseSize?: number;
        /** The ratio of the sphere to the longest bone in units */
        sphereScaleUnit?: number;
        /** Ratio for the Sphere Size */
        sphereFactor?: number;
        /** Whether a spur should attach its far end to the child bone position */
        spurFollowsChild?: boolean;
        /** Whether to show local axes or not  */
        showLocalAxes?: boolean;
        /** Length of each local axis */
        localAxesSize?: number;
    }
    /**
     * Defines the constructor options for the BoneWeight Shader.
     */
    export interface IBoneWeightShaderOptions {
        /** Skeleton to Map */
        skeleton: Skeleton;
        /** Colors for Uninfluenced bones */
        colorBase?: Color3;
        /** Colors for 0.0-0.25 Weight bones */
        colorZero?: Color3;
        /** Color for 0.25-0.5 Weight Influence */
        colorQuarter?: Color3;
        /** Color for 0.5-0.75 Weight Influence */
        colorHalf?: Color3;
        /** Color for 0.75-1 Weight Influence */
        colorFull?: Color3;
        /** Color for Zero Weight Influence */
        targetBoneIndex?: number;
    }
    /**
     * Simple structure of the gradient steps for the Color Map.
     */
    export interface ISkeletonMapShaderColorMapKnot {
        /** Color of the Knot */
        color: Color3;
        /** Location of the Knot */
        location: number;
    }
    /**
     * Defines the constructor options for the SkeletonMap Shader.
     */
    export interface ISkeletonMapShaderOptions {
        /** Skeleton to Map */
        skeleton: Skeleton;
        /** Array of ColorMapKnots that make the gradient must be ordered with knot[i].location < knot[i+1].location*/
        colorMap?: ISkeletonMapShaderColorMapKnot[];
    }



}
declare module BABYLON.Debug {
        /**
     * Used to show the physics impostor around the specific mesh
     */
    export class PhysicsViewer {
        /** @internal */
        protected _impostors: Array<Nullable<PhysicsImpostor>>;
        /** @internal */
        protected _meshes: Array<Nullable<AbstractMesh>>;
        /** @internal */
        protected _bodies: Array<Nullable<PhysicsBody>>;
        /** @internal */
        protected _inertiaBodies: Array<Nullable<PhysicsBody>>;
        /** @internal */
        protected _constraints: Array<Nullable<PhysicsConstraint>>;
        /** @internal */
        protected _bodyMeshes: Array<Nullable<AbstractMesh>>;
        /** @internal */
        protected _inertiaMeshes: Array<Nullable<AbstractMesh>>;
        /** @internal */
        protected _constraintMeshes: Array<Nullable<AbstractMesh>>;
        /** @internal */
        protected _scene: Nullable<Scene>;
        /** @internal */
        protected _numMeshes: number;
        /** @internal */
        protected _numBodies: number;
        /** @internal */
        protected _numInertiaBodies: number;
        /** @internal */
        protected _numConstraints: number;
        /** @internal */
        protected _physicsEnginePlugin: BABYLON.IPhysicsEnginePlugin | IPhysicsEnginePluginV2 | null;
        private _renderFunction;
        private _inertiaRenderFunction;
        private _constraintRenderFunction;
        private _utilityLayer;
        private _debugBoxMesh;
        private _debugSphereMesh;
        private _debugCapsuleMesh;
        private _debugCylinderMesh;
        private _debugMaterial;
        private _debugInertiaMaterial;
        private _debugMeshMeshes;
        private _constraintAxesSize;
        /**
         * Creates a new PhysicsViewer
         * @param scene defines the hosting scene
         */
        constructor(scene?: Scene);
        /**
         * Updates the debug meshes of the physics engine.
         *
         * This code is useful for synchronizing the debug meshes of the physics engine with the physics impostor and mesh.
         * It checks if the impostor is disposed and if the plugin version is 1, then it syncs the mesh with the impostor.
         * This ensures that the debug meshes are up to date with the physics engine.
         */
        protected _updateDebugMeshes(): void;
        /**
         * Updates the debug meshes of the physics engine.
         *
         * This method is useful for synchronizing the debug meshes with the physics impostors.
         * It iterates through the impostors and meshes, and if the plugin version is 1, it syncs the mesh with the impostor.
         * This ensures that the debug meshes accurately reflect the physics impostors, which is important for debugging the physics engine.
         */
        protected _updateDebugMeshesV1(): void;
        /**
         * Updates the debug meshes of the physics engine for V2 plugin.
         *
         * This method is useful for synchronizing the debug meshes of the physics engine with the current state of the bodies.
         * It iterates through the bodies array and updates the debug meshes with the current transform of each body.
         * This ensures that the debug meshes accurately reflect the current state of the physics engine.
         */
        protected _updateDebugMeshesV2(): void;
        protected _updateInertiaMeshes(): void;
        protected _updateDebugInertia(body: PhysicsBody, inertiaMesh: AbstractMesh): void;
        protected _updateDebugConstraints(): void;
        /**
         * Given a scaling vector, make all of its components
         * 1, preserving the sign
         * @param scaling
         */
        protected _makeScalingUnitInPlace(scaling: Vector3): void;
        protected _updateDebugConstraint(constraint: PhysicsConstraint, parentingMesh: AbstractMesh): void;
        /**
         * Renders a specified physic impostor
         * @param impostor defines the impostor to render
         * @param targetMesh defines the mesh represented by the impostor
         * @returns the new debug mesh used to render the impostor
         */
        showImpostor(impostor: PhysicsImpostor, targetMesh?: Mesh): Nullable<AbstractMesh>;
        /**
         * Shows a debug mesh for a given physics body.
         * @param body The physics body to show.
         * @returns The debug mesh, or null if the body is already shown.
         *
         * This function is useful for visualizing the physics body in the scene.
         * It creates a debug mesh for the given body and adds it to the scene.
         * It also registers a before render function to update the debug mesh position and rotation.
         */
        showBody(body: PhysicsBody): Nullable<AbstractMesh>;
        /**
         * Shows a debug box corresponding to the inertia of a given body
         * @param body the physics body used to get the inertia
         * @returns the debug mesh used to show the inertia, or null if the body is already shown
         */
        showInertia(body: PhysicsBody): Nullable<AbstractMesh>;
        /**
         * Shows a debug mesh for a given physics constraint.
         * @param constraint the physics constraint to show
         * @returns the debug mesh, or null if the constraint is already shown
         */
        showConstraint(constraint: PhysicsConstraint): Nullable<AbstractMesh>;
        /**
         * Hides an impostor from the scene.
         * @param impostor - The impostor to hide.
         *
         * This method is useful for hiding an impostor from the scene. It removes the
         * impostor from the utility layer scene, disposes the mesh, and removes the
         * impostor from the list of impostors. If the impostor is the last one in the
         * list, it also unregisters the render function.
         */
        hideImpostor(impostor: Nullable<PhysicsImpostor>): void;
        /**
         * Hides a body from the physics engine.
         * @param body - The body to hide.
         *
         * This function is useful for hiding a body from the physics engine.
         * It removes the body from the utility layer scene and disposes the mesh associated with it.
         * It also unregisters the render function if the number of bodies is 0.
         * This is useful for hiding a body from the physics engine without deleting it.
         */
        hideBody(body: Nullable<PhysicsBody>): void;
        /**
         * Hides a body's inertia from the viewer utility layer
         * @param body the body to hide
         */
        hideInertia(body: Nullable<PhysicsBody>): void;
        /**
         * Hide a physics constraint from the viewer utility layer
         * @param constraint the constraint to hide
         */
        hideConstraint(constraint: Nullable<PhysicsConstraint>): void;
        private _getDebugMaterial;
        private _getDebugInertiaMaterial;
        private _getDebugBoxMesh;
        private _getDebugSphereMesh;
        private _getDebugCapsuleMesh;
        private _getDebugCylinderMesh;
        private _getDebugMeshMesh;
        private _getDebugMesh;
        /**
         * Creates a debug mesh for a given physics body
         * @param body The physics body to create the debug mesh for
         * @returns The created debug mesh or null if the utility layer is not available
         *
         * This code is useful for creating a debug mesh for a given physics body.
         * It creates a Mesh object with a VertexData object containing the positions and indices
         * of the geometry of the body. The mesh is then assigned a debug material from the utility layer scene.
         * This allows for visualizing the physics body in the scene.
         */
        private _getDebugBodyMesh;
        private _getMeshDebugInertiaMatrixToRef;
        private _getDebugInertiaMesh;
        private _getTransformFromBodyToRef;
        private _getDebugConstraintMesh;
        /**
         * Clean up physics debug display
         */
        dispose(): void;
    }



}
declare module BABYLON {
    /**
     * As raycast might be hard to debug, the RayHelper can help rendering the different rays
     * in order to better appreciate the issue one might have.
     * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/interactions/picking_collisions#debugging
     */
    export class RayHelper {
        /**
         * Defines the ray we are currently trying to visualize.
         */
        ray: Nullable<Ray>;
        private _renderPoints;
        private _renderLine;
        private _renderFunction;
        private _scene;
        private _onAfterRenderObserver;
        private _onAfterStepObserver;
        private _attachedToMesh;
        private _meshSpaceDirection;
        private _meshSpaceOrigin;
        /**
         * Helper function to create a colored helper in a scene in one line.
         * @param ray Defines the ray we are currently trying to visualize
         * @param scene Defines the scene the ray is used in
         * @param color Defines the color we want to see the ray in
         * @returns The newly created ray helper.
         */
        static CreateAndShow(ray: Ray, scene: Scene, color: Color3): RayHelper;
        /**
         * Instantiate a new ray helper.
         * As raycast might be hard to debug, the RayHelper can help rendering the different rays
         * in order to better appreciate the issue one might have.
         * @see https://doc.babylonjs.com/features/featuresDeepDive/mesh/interactions/picking_collisions#debugging
         * @param ray Defines the ray we are currently trying to visualize
         */
        constructor(ray: Ray);
        /**
         * Shows the ray we are willing to debug.
         * @param scene Defines the scene the ray needs to be rendered in
         * @param color Defines the color the ray needs to be rendered in
         */
        show(scene: Scene, color?: Color3): void;
        /**
         * Hides the ray we are debugging.
         */
        hide(): void;
        private _render;
        /**
         * Attach a ray helper to a mesh so that we can easily see its orientation for instance or information like its normals.
         * @param mesh Defines the mesh we want the helper attached to
         * @param meshSpaceDirection Defines the direction of the Ray in mesh space (local space of the mesh node)
         * @param meshSpaceOrigin Defines the origin of the Ray in mesh space (local space of the mesh node)
         * @param length Defines the length of the ray
         */
        attachToMesh(mesh: AbstractMesh, meshSpaceDirection?: Vector3, meshSpaceOrigin?: Vector3, length?: number): void;
        /**
         * Detach the ray helper from the mesh it has previously been attached to.
         */
        detachFromMesh(): void;
        private _updateToMesh;
        /**
         * Dispose the helper and release its associated resources.
         */
        dispose(): void;
    }



}
declare module BABYLON.Debug {
        /**
     * Class used to render a debug view of a given skeleton
     * @see http://www.babylonjs-playground.com/#1BZJVJ#8
     */
    export class SkeletonViewer {
        /** defines the skeleton to render */
        skeleton: Skeleton;
        /** defines the mesh attached to the skeleton */
        mesh: Nullable<AbstractMesh>;
        /** defines a boolean indicating if bones matrices must be forced to update before rendering (true by default)  */
        autoUpdateBonesMatrices: boolean;
        /** defines the rendering group id to use with the viewer */
        renderingGroupId: number;
        /** is the options for the viewer */
        options: Partial<ISkeletonViewerOptions>;
        /** public Display constants BABYLON.SkeletonViewer.DISPLAY_LINES */
        static readonly DISPLAY_LINES = 0;
        /** public Display constants BABYLON.SkeletonViewer.DISPLAY_SPHERES */
        static readonly DISPLAY_SPHERES = 1;
        /** public Display constants BABYLON.SkeletonViewer.DISPLAY_SPHERE_AND_SPURS */
        static readonly DISPLAY_SPHERE_AND_SPURS = 2;
        /** public static method to create a BoneWeight Shader
         * @param options The constructor options
         * @param scene The scene that the shader is scoped to
         * @returns The created ShaderMaterial
         * @see http://www.babylonjs-playground.com/#1BZJVJ#395
         */
        static CreateBoneWeightShader(options: IBoneWeightShaderOptions, scene: Scene): ShaderMaterial;
        /** public static method to create a BoneWeight Shader
         * @param options The constructor options
         * @param scene The scene that the shader is scoped to
         * @returns The created ShaderMaterial
         */
        static CreateSkeletonMapShader(options: ISkeletonMapShaderOptions, scene: Scene): ShaderMaterial;
        /** private static method to create a BoneWeight Shader
         * @param size The size of the buffer to create (usually the bone count)
         * @param colorMap The gradient data to generate
         * @param scene The scene that the shader is scoped to
         * @returns an Array of floats from the color gradient values
         */
        private static _CreateBoneMapColorBuffer;
        /** If SkeletonViewer scene scope. */
        private _scene;
        /** Gets or sets the color used to render the skeleton */
        color: Color3;
        /** Array of the points of the skeleton fo the line view. */
        private _debugLines;
        /** The SkeletonViewers Mesh. */
        private _debugMesh;
        /** The local axes Meshes. */
        private _localAxes;
        /** If SkeletonViewer is enabled. */
        private _isEnabled;
        /** If SkeletonViewer is ready. */
        private _ready;
        /** SkeletonViewer render observable. */
        private _obs;
        /** The Utility Layer to render the gizmos in. */
        private _utilityLayer;
        private _boneIndices;
        /** Gets the Scene. */
        get scene(): Scene;
        /** Gets the utilityLayer. */
        get utilityLayer(): Nullable<UtilityLayerRenderer>;
        /** Checks Ready Status. */
        get isReady(): Boolean;
        /** Sets Ready Status. */
        set ready(value: boolean);
        /** Gets the debugMesh */
        get debugMesh(): Nullable<AbstractMesh> | Nullable<LinesMesh>;
        /** Sets the debugMesh */
        set debugMesh(value: Nullable<AbstractMesh> | Nullable<LinesMesh>);
        /** Gets the displayMode */
        get displayMode(): number;
        /** Sets the displayMode */
        set displayMode(value: number);
        /**
         * Creates a new SkeletonViewer
         * @param skeleton defines the skeleton to render
         * @param mesh defines the mesh attached to the skeleton
         * @param scene defines the hosting scene
         * @param autoUpdateBonesMatrices defines a boolean indicating if bones matrices must be forced to update before rendering (true by default)
         * @param renderingGroupId defines the rendering group id to use with the viewer
         * @param options All of the extra constructor options for the SkeletonViewer
         */
        constructor(
        /** defines the skeleton to render */
        skeleton: Skeleton, 
        /** defines the mesh attached to the skeleton */
        mesh: Nullable<AbstractMesh>, 
        /** The Scene scope*/
        scene: Scene, 
        /** defines a boolean indicating if bones matrices must be forced to update before rendering (true by default)  */
        autoUpdateBonesMatrices?: boolean, 
        /** defines the rendering group id to use with the viewer */
        renderingGroupId?: number, 
        /** is the options for the viewer */
        options?: Partial<ISkeletonViewerOptions>);
        /** The Dynamic bindings for the update functions */
        private _bindObs;
        /** Update the viewer to sync with current skeleton state, only used to manually update. */
        update(): void;
        /** Gets or sets a boolean indicating if the viewer is enabled */
        set isEnabled(value: boolean);
        get isEnabled(): boolean;
        private _getBonePosition;
        private _getLinesForBonesWithLength;
        private _getLinesForBonesNoLength;
        /**
         * function to revert the mesh and scene back to the initial state.
         * @param animationState
         */
        private _revert;
        /**
         * function to get the absolute bind pose of a bone by accumulating transformations up the bone hierarchy.
         * @param bone
         * @param matrix
         */
        private _getAbsoluteBindPoseToRef;
        private _createSpur;
        private _getBoundingSphereForBone;
        /**
         * function to build and bind sphere joint points and spur bone representations.
         * @param spheresOnly
         */
        private _buildSpheresAndSpurs;
        private _buildLocalAxes;
        /** Update the viewer to sync with current skeleton state, only used for the line display. */
        private _displayLinesUpdate;
        /** Changes the displayMode of the skeleton viewer
         * @param mode The displayMode numerical value
         */
        changeDisplayMode(mode: number): void;
        /** Sets a display option of the skeleton viewer
         *
         * | Option           | Type    | Default | Description |
         * | ---------------- | ------- | ------- | ----------- |
         * | midStep          | float   | 0.235   | A percentage between a bone and its child that determines the widest part of a spur. Only used when `displayMode` is set to `DISPLAY_SPHERE_AND_SPURS`. |
         * | midStepFactor    | float   | 0.15    | Mid step width expressed as a factor of the length. A value of 0.5 makes the spur width half of the spur length. Only used when `displayMode` is set to `DISPLAY_SPHERE_AND_SPURS`. |
         * | sphereBaseSize   | float   | 2       | Sphere base size. Only used when `displayMode` is set to `DISPLAY_SPHERE_AND_SPURS`. |
         * | sphereScaleUnit  | float   | 0.865   | Sphere scale factor used to scale spheres in relation to the longest bone. Only used when `displayMode` is set to `DISPLAY_SPHERE_AND_SPURS`. |
         * | spurFollowsChild | boolean | false   | Whether a spur should attach its far end to the child bone. |
         * | showLocalAxes    | boolean | false   | Displays local axes on all bones. |
         * | localAxesSize    | float   | 0.075   | Determines the length of each local axis. |
         *
         * @param option String of the option name
         * @param value The numerical option value
         */
        changeDisplayOptions(option: string, value: number): void;
        /** Release associated resources */
        dispose(): void;
    }



}
declare module BABYLON {
    /**
     * Enum defining the type of properties that can be edited in the property pages in the node editor
     */
    export enum PropertyTypeForEdition {
        /** property is a boolean */
        Boolean = 0,
        /** property is a float */
        Float = 1,
        /** property is a int */
        Int = 2,
        /** property is a Vector2 */
        Vector2 = 3,
        /** property is a list of values */
        List = 4
    }
    /**
     * Interface that defines an option in a variable of type list
     */
    export interface IEditablePropertyListOption {
        /** label of the option */
        label: string;
        /** value of the option */
        value: number;
    }
    /**
     * Interface that defines the options available for an editable property
     */
    export interface IEditablePropertyOption {
        /** min value */
        min?: number;
        /** max value */
        max?: number;
        /** notifiers: indicates which actions to take when the property is changed */
        notifiers?: {
            /** the entity should be rebuilt */
            rebuild?: boolean;
            /** the preview should be updated */
            update?: boolean;
            /** the onPreviewCommandActivated observer of the preview manager should be triggered */
            activatePreviewCommand?: boolean;
            /** a callback to trigger */
            callback?: (scene: Nullable<Scene>, block: any) => boolean | undefined | void;
            /** a callback to validate the property. Returns true if the property is ok, else false. If false, the rebuild/update/callback events won't be called */
            onValidation?: (block: any, propertyName: string) => boolean;
        };
        /** list of the options for a variable of type list */
        options?: IEditablePropertyListOption[];
    }
    /**
     * Interface that describes an editable property
     */
    export interface IPropertyDescriptionForEdition {
        /** name of the property */
        propertyName: string;
        /** display name of the property */
        displayName: string;
        /** type of the property */
        type: PropertyTypeForEdition;
        /** group of the property - all properties with the same group value will be displayed in a specific section */
        groupName: string;
        /** options for the property */
        options: IEditablePropertyOption;
    }
    /**
     * Decorator that flags a property in a node block as being editable
     * @param displayName the display name of the property
     * @param propertyType the type of the property
     * @param groupName the group name of the property
     * @param options the options of the property
     * @returns the decorator
     */
    export function editableInPropertyPage(displayName: string, propertyType?: PropertyTypeForEdition, groupName?: string, options?: IEditablePropertyOption): (target: any, propertyKey: string) => void;


    /**
     * Class to wrap DeviceInputSystem data into an event object
     */
    export class DeviceEventFactory {
        /**
         * Create device input events based on provided type and slot
         *
         * @param deviceType Type of device
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @param currentState Current value for given input
         * @param deviceInputSystem Reference to DeviceInputSystem
         * @param elementToAttachTo HTMLElement to reference as target for inputs
         * @param pointerId PointerId to use for pointer events
         * @returns IUIEvent object
         */
        static CreateDeviceEvent(deviceType: DeviceType, deviceSlot: number, inputIndex: number, currentState: Nullable<number>, deviceInputSystem: IDeviceInputSystem, elementToAttachTo?: any, pointerId?: number): IUIEvent;
        /**
         * Creates pointer event
         *
         * @param deviceType Type of device
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @param currentState Current value for given input
         * @param deviceInputSystem Reference to DeviceInputSystem
         * @param elementToAttachTo HTMLElement to reference as target for inputs
         * @param pointerId PointerId to use for pointer events
         * @returns IUIEvent object (Pointer)
         */
        private static _CreatePointerEvent;
        /**
         * Create Mouse Wheel Event
         * @param deviceType Type of device
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @param currentState Current value for given input
         * @param deviceInputSystem Reference to DeviceInputSystem
         * @param elementToAttachTo HTMLElement to reference as target for inputs
         * @returns IUIEvent object (Wheel)
         */
        private static _CreateWheelEvent;
        /**
         * Create Mouse Event
         * @param deviceType Type of device
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @param currentState Current value for given input
         * @param deviceInputSystem Reference to DeviceInputSystem
         * @param elementToAttachTo HTMLElement to reference as target for inputs
         * @returns IUIEvent object (Mouse)
         */
        private static _CreateMouseEvent;
        /**
         * Create Keyboard Event
         * @param inputIndex Id of input to be checked
         * @param currentState Current value for given input
         * @param deviceInputSystem Reference to DeviceInputSystem
         * @param elementToAttachTo HTMLElement to reference as target for inputs
         * @returns IEvent object (Keyboard)
         */
        private static _CreateKeyboardEvent;
        /**
         * Add parameters for non-character keys (Ctrl, Alt, Meta, Shift)
         * @param evt Event object to add parameters to
         * @param deviceInputSystem DeviceInputSystem to pull values from
         */
        private static _CheckNonCharacterKeys;
        /**
         * Create base event object
         * @param elementToAttachTo Value to use as event target
         * @returns
         */
        private static _CreateEvent;
    }




    /**
     * Enum for Device Types
     */
    export enum DeviceType {
        /** Generic */
        Generic = 0,
        /** Keyboard */
        Keyboard = 1,
        /** Mouse */
        Mouse = 2,
        /** Touch Pointers */
        Touch = 3,
        /** PS4 Dual Shock */
        DualShock = 4,
        /** Xbox */
        Xbox = 5,
        /** Switch Controller */
        Switch = 6,
        /** PS5 DualSense */
        DualSense = 7
    }
    /**
     * Enum for All Pointers (Touch/Mouse)
     */
    export enum PointerInput {
        /** Horizontal Axis (Not used in events/observables; only in polling) */
        Horizontal = 0,
        /** Vertical Axis (Not used in events/observables; only in polling) */
        Vertical = 1,
        /** Left Click or Touch */
        LeftClick = 2,
        /** Middle Click */
        MiddleClick = 3,
        /** Right Click */
        RightClick = 4,
        /** Browser Back */
        BrowserBack = 5,
        /** Browser Forward */
        BrowserForward = 6,
        /** Mouse Wheel X */
        MouseWheelX = 7,
        /** Mouse Wheel Y */
        MouseWheelY = 8,
        /** Mouse Wheel Z */
        MouseWheelZ = 9,
        /** Used in events/observables to identify if x/y changes occurred */
        Move = 12
    }
    /** @internal */
    export enum NativePointerInput {
        /** Horizontal Axis */
        Horizontal = 0,
        /** Vertical Axis */
        Vertical = 1,
        /** Left Click or Touch */
        LeftClick = 2,
        /** Middle Click */
        MiddleClick = 3,
        /** Right Click */
        RightClick = 4,
        /** Browser Back */
        BrowserBack = 5,
        /** Browser Forward */
        BrowserForward = 6,
        /** Mouse Wheel X */
        MouseWheelX = 7,
        /** Mouse Wheel Y */
        MouseWheelY = 8,
        /** Mouse Wheel Z */
        MouseWheelZ = 9,
        /** Delta X */
        DeltaHorizontal = 10,
        /** Delta Y */
        DeltaVertical = 11
    }
    /**
     * Enum for Dual Shock Gamepad
     */
    export enum DualShockInput {
        /** Cross */
        Cross = 0,
        /** Circle */
        Circle = 1,
        /** Square */
        Square = 2,
        /** Triangle */
        Triangle = 3,
        /** L1 */
        L1 = 4,
        /** R1 */
        R1 = 5,
        /** L2 */
        L2 = 6,
        /** R2 */
        R2 = 7,
        /** Share */
        Share = 8,
        /** Options */
        Options = 9,
        /** L3 */
        L3 = 10,
        /** R3 */
        R3 = 11,
        /** DPadUp */
        DPadUp = 12,
        /** DPadDown */
        DPadDown = 13,
        /** DPadLeft */
        DPadLeft = 14,
        /** DRight */
        DPadRight = 15,
        /** Home */
        Home = 16,
        /** TouchPad */
        TouchPad = 17,
        /** LStickXAxis */
        LStickXAxis = 18,
        /** LStickYAxis */
        LStickYAxis = 19,
        /** RStickXAxis */
        RStickXAxis = 20,
        /** RStickYAxis */
        RStickYAxis = 21
    }
    /**
     * Enum for Dual Sense Gamepad
     */
    export enum DualSenseInput {
        /** Cross */
        Cross = 0,
        /** Circle */
        Circle = 1,
        /** Square */
        Square = 2,
        /** Triangle */
        Triangle = 3,
        /** L1 */
        L1 = 4,
        /** R1 */
        R1 = 5,
        /** L2 */
        L2 = 6,
        /** R2 */
        R2 = 7,
        /** Create */
        Create = 8,
        /** Options */
        Options = 9,
        /** L3 */
        L3 = 10,
        /** R3 */
        R3 = 11,
        /** DPadUp */
        DPadUp = 12,
        /** DPadDown */
        DPadDown = 13,
        /** DPadLeft */
        DPadLeft = 14,
        /** DRight */
        DPadRight = 15,
        /** Home */
        Home = 16,
        /** TouchPad */
        TouchPad = 17,
        /** LStickXAxis */
        LStickXAxis = 18,
        /** LStickYAxis */
        LStickYAxis = 19,
        /** RStickXAxis */
        RStickXAxis = 20,
        /** RStickYAxis */
        RStickYAxis = 21
    }
    /**
     * Enum for Xbox Gamepad
     */
    export enum XboxInput {
        /** A */
        A = 0,
        /** B */
        B = 1,
        /** X */
        X = 2,
        /** Y */
        Y = 3,
        /** LB */
        LB = 4,
        /** RB */
        RB = 5,
        /** LT */
        LT = 6,
        /** RT */
        RT = 7,
        /** Back */
        Back = 8,
        /** Start */
        Start = 9,
        /** LS */
        LS = 10,
        /** RS */
        RS = 11,
        /** DPadUp */
        DPadUp = 12,
        /** DPadDown */
        DPadDown = 13,
        /** DPadLeft */
        DPadLeft = 14,
        /** DRight */
        DPadRight = 15,
        /** Home */
        Home = 16,
        /** LStickXAxis */
        LStickXAxis = 17,
        /** LStickYAxis */
        LStickYAxis = 18,
        /** RStickXAxis */
        RStickXAxis = 19,
        /** RStickYAxis */
        RStickYAxis = 20
    }
    /**
     * Enum for Switch (Pro/JoyCon L+R) Gamepad
     */
    export enum SwitchInput {
        /** B */
        B = 0,
        /** A */
        A = 1,
        /** Y */
        Y = 2,
        /** X */
        X = 3,
        /** L */
        L = 4,
        /** R */
        R = 5,
        /** ZL */
        ZL = 6,
        /** ZR */
        ZR = 7,
        /** Minus */
        Minus = 8,
        /** Plus */
        Plus = 9,
        /** LS */
        LS = 10,
        /** RS */
        RS = 11,
        /** DPadUp */
        DPadUp = 12,
        /** DPadDown */
        DPadDown = 13,
        /** DPadLeft */
        DPadLeft = 14,
        /** DRight */
        DPadRight = 15,
        /** Home */
        Home = 16,
        /** Capture */
        Capture = 17,
        /** LStickXAxis */
        LStickXAxis = 18,
        /** LStickYAxis */
        LStickYAxis = 19,
        /** RStickXAxis */
        RStickXAxis = 20,
        /** RStickYAxis */
        RStickYAxis = 21
    }


    /**
     * Subset of DeviceInput that only handles pointers and keyboard
     */
    export type DeviceSourceEvent<T extends DeviceType> = T extends DeviceType.Keyboard ? IKeyboardEvent : T extends DeviceType.Mouse ? IWheelEvent | IPointerEvent : T extends DeviceType.Touch ? IPointerEvent : never;
    /**
     * Class that handles all input for a specific device
     */
    export class DeviceSource<T extends DeviceType> {
        /** Type of device */
        readonly deviceType: T;
        /** "Slot" or index that device is referenced in */
        readonly deviceSlot: number;
        /**
         * Observable to handle device input changes per device
         */
        readonly onInputChangedObservable: Observable<DeviceSourceEvent<T>>;
        private readonly _deviceInputSystem;
        /**
         * Default Constructor
         * @param deviceInputSystem - Reference to DeviceInputSystem
         * @param deviceType - Type of device
         * @param deviceSlot - "Slot" or index that device is referenced in
         */
        constructor(deviceInputSystem: IDeviceInputSystem, 
        /** Type of device */
        deviceType: T, 
        /** "Slot" or index that device is referenced in */
        deviceSlot?: number);
        /**
         * Get input for specific input
         * @param inputIndex - index of specific input on device
         * @returns Input value from DeviceInputSystem
         */
        getInput(inputIndex: DeviceInput<T>): number;
    }


    /**
     * Class to keep track of devices
     */
    export class DeviceSourceManager implements IDisposable, IObservableManager {
        /**
         * Observable to be triggered when after a device is connected, any new observers added will be triggered against already connected devices
         */
        readonly onDeviceConnectedObservable: Observable<DeviceSourceType>;
        /**
         * Observable to be triggered when after a device is disconnected
         */
        readonly onDeviceDisconnectedObservable: Observable<DeviceSourceType>;
        private _engine;
        private _onDisposeObserver;
        private readonly _devices;
        private readonly _firstDevice;
        /**
         * Gets a DeviceSource, given a type and slot
         * @param deviceType - Type of Device
         * @param deviceSlot - Slot or ID of device
         * @returns DeviceSource
         */
        getDeviceSource<T extends DeviceType>(deviceType: T, deviceSlot?: number): Nullable<DeviceSource<T>>;
        /**
         * Gets an array of DeviceSource objects for a given device type
         * @param deviceType - Type of Device
         * @returns All available DeviceSources of a given type
         */
        getDeviceSources<T extends DeviceType>(deviceType: T): ReadonlyArray<DeviceSource<T>>;
        /**
         * Default constructor
         * @param engine - Used to get canvas (if applicable)
         */
        constructor(engine: Engine);
        /**
         * Dispose of DeviceSourceManager
         */
        dispose(): void;
        /**
         * @param deviceSource - Source to add
         * @internal
         */
        _addDevice(deviceSource: DeviceSourceType): void;
        /**
         * @param deviceType - DeviceType
         * @param deviceSlot - DeviceSlot
         * @internal
         */
        _removeDevice(deviceType: DeviceType, deviceSlot: number): void;
        /**
         * @param deviceType - DeviceType
         * @param deviceSlot - DeviceSlot
         * @param eventData - Event
         * @internal
         */
        _onInputChanged<T extends DeviceType>(deviceType: T, deviceSlot: number, eventData: IUIEvent): void;
        private _updateFirstDevices;
    }


    /**
     * Type to handle enforcement of inputs
     */
    export type DeviceInput<T extends DeviceType> = T extends DeviceType.Keyboard | DeviceType.Generic ? number : T extends DeviceType.Mouse | DeviceType.Touch ? Exclude<PointerInput, PointerInput.Move | PointerInput.MouseWheelX | PointerInput.MouseWheelY | PointerInput.MouseWheelZ> : T extends DeviceType.DualShock ? DualShockInput : T extends DeviceType.Xbox ? XboxInput : T extends DeviceType.Switch ? SwitchInput : T extends DeviceType.DualSense ? DualSenseInput : never;


    /**
     * Interface for DeviceInputSystem implementations (JS and Native)
     */
    export interface IDeviceInputSystem extends IDisposable {
        /**
         * Checks for current device input value, given an id and input index. Throws exception if requested device not initialized.
         * @param deviceType Enum specifying device type
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @returns Current value of input
         */
        pollInput(deviceType: DeviceType, deviceSlot: number, inputIndex: number): number;
        /**
         * Check for a specific device in the DeviceInputSystem
         * @param deviceType Type of device to check for
         * @returns bool with status of device's existence
         */
        isDeviceAvailable(deviceType: DeviceType): boolean;
    }


    type Distribute<T> = T extends DeviceType ? DeviceSource<T> : never;
    export type DeviceSourceType = Distribute<DeviceType>;
        interface Engine {
            /** @internal */
            _deviceSourceManager?: InternalDeviceSourceManager;
        }
    /** @internal */
    export interface IObservableManager {
        onDeviceConnectedObservable: Observable<DeviceSourceType>;
        onDeviceDisconnectedObservable: Observable<DeviceSourceType>;
        _onInputChanged(deviceType: DeviceType, deviceSlot: number, eventData: IUIEvent): void;
        _addDevice(deviceSource: DeviceSource<DeviceType>): void;
        _removeDevice(deviceType: DeviceType, deviceSlot: number): void;
    }
    /** @internal */
    export class InternalDeviceSourceManager implements IDisposable {
        readonly _deviceInputSystem: IDeviceInputSystem;
        private readonly _devices;
        private readonly _registeredManagers;
        _refCount: number;
        constructor(engine: Engine);
        readonly registerManager: (manager: IObservableManager) => void;
        readonly unregisterManager: (manager: IObservableManager) => void;
        dispose(): void;
    }


    /** @internal */
    export class NativeDeviceInputSystem implements IDeviceInputSystem {
        private readonly _nativeInput;
        constructor(onDeviceConnected: (deviceType: DeviceType, deviceSlot: number) => void, onDeviceDisconnected: (deviceType: DeviceType, deviceSlot: number) => void, onInputChanged: (deviceType: DeviceType, deviceSlot: number, eventData: IUIEvent) => void);
        /**
         * Checks for current device input value, given an id and input index. Throws exception if requested device not initialized.
         * @param deviceType Enum specifying device type
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @returns Current value of input
         */
        pollInput(deviceType: DeviceType, deviceSlot: number, inputIndex: number): number;
        /**
         * Check for a specific device in the DeviceInputSystem
         * @param deviceType Type of device to check for
         * @returns bool with status of device's existence
         */
        isDeviceAvailable(deviceType: DeviceType): boolean;
        /**
         * Dispose of all the observables
         */
        dispose(): void;
        /**
         * For versions of BabylonNative that don't have the NativeInput plugin initialized, create a dummy version
         * @returns Object with dummy functions
         */
        private _createDummyNativeInput;
    }


    /** @internal */
    export class WebDeviceInputSystem implements IDeviceInputSystem {
        private _inputs;
        private _gamepads;
        private _keyboardActive;
        private _pointerActive;
        private _elementToAttachTo;
        private _metaKeys;
        private readonly _engine;
        private readonly _usingSafari;
        private readonly _usingMacOS;
        private _onDeviceConnected;
        private _onDeviceDisconnected;
        private _onInputChanged;
        private _keyboardDownEvent;
        private _keyboardUpEvent;
        private _keyboardBlurEvent;
        private _pointerMoveEvent;
        private _pointerDownEvent;
        private _pointerUpEvent;
        private _pointerCancelEvent;
        private _pointerWheelEvent;
        private _pointerBlurEvent;
        private _pointerMacOSChromeOutEvent;
        private _wheelEventName;
        private _eventsAttached;
        private _mouseId;
        private readonly _isUsingFirefox;
        private readonly _isUsingChromium;
        private _activeTouchIds;
        private _maxTouchPoints;
        private _pointerInputClearObserver;
        private _gamepadConnectedEvent;
        private _gamepadDisconnectedEvent;
        private _eventPrefix;
        /**
         * Constructor for the WebDeviceInputSystem
         * @param engine Engine to reference
         * @param onDeviceConnected Callback to execute when device is connected
         * @param onDeviceDisconnected Callback to execute when device is disconnected
         * @param onInputChanged Callback to execute when input changes on device
         */
        constructor(engine: Engine, onDeviceConnected: (deviceType: DeviceType, deviceSlot: number) => void, onDeviceDisconnected: (deviceType: DeviceType, deviceSlot: number) => void, onInputChanged: (deviceType: DeviceType, deviceSlot: number, eventData: IUIEvent) => void);
        /**
         * Checks for current device input value, given an id and input index. Throws exception if requested device not initialized.
         * @param deviceType Enum specifying device type
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         * @returns Current value of input
         */
        pollInput(deviceType: DeviceType, deviceSlot: number, inputIndex: number): number;
        /**
         * Check for a specific device in the DeviceInputSystem
         * @param deviceType Type of device to check for
         * @returns bool with status of device's existence
         */
        isDeviceAvailable(deviceType: DeviceType): boolean;
        /**
         * Dispose of all the eventlisteners
         */
        dispose(): void;
        /**
         * Enable listening for user input events
         */
        private _enableEvents;
        /**
         * Disable listening for user input events
         */
        private _disableEvents;
        /**
         * Checks for existing connections to devices and register them, if necessary
         * Currently handles gamepads and mouse
         */
        private _checkForConnectedDevices;
        /**
         * Add a gamepad to the DeviceInputSystem
         * @param gamepad A single DOM Gamepad object
         */
        private _addGamePad;
        /**
         * Add pointer device to DeviceInputSystem
         * @param deviceType Type of Pointer to add
         * @param deviceSlot Pointer ID (0 for mouse, pointerId for Touch)
         * @param currentX Current X at point of adding
         * @param currentY Current Y at point of adding
         */
        private _addPointerDevice;
        /**
         * Add device and inputs to device array
         * @param deviceType Enum specifying device type
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param numberOfInputs Number of input entries to create for given device
         */
        private _registerDevice;
        /**
         * Given a specific device name, remove that device from the device map
         * @param deviceType Enum specifying device type
         * @param deviceSlot "Slot" or index that device is referenced in
         */
        private _unregisterDevice;
        /**
         * Handle all actions that come from keyboard interaction
         */
        private _handleKeyActions;
        /**
         * Handle all actions that come from pointer interaction
         */
        private _handlePointerActions;
        /**
         * Handle all actions that come from gamepad interaction
         */
        private _handleGamepadActions;
        /**
         * Update all non-event based devices with each frame
         * @param deviceType Enum specifying device type
         * @param deviceSlot "Slot" or index that device is referenced in
         * @param inputIndex Id of input to be checked
         */
        private _updateDevice;
        /**
         * Gets DeviceType from the device name
         * @param deviceName Name of Device from DeviceInputSystem
         * @returns DeviceType enum value
         */
        private _getGamepadDeviceType;
        /**
         * Get DeviceType from a given pointer/mouse/touch event.
         * @param evt PointerEvent to evaluate
         * @returns DeviceType interpreted from event
         */
        private _getPointerType;
    }


    /** Defines the cross module used constants to avoid circular dependencies */
    export class Constants {
        /** Defines that alpha blending is disabled */
        static readonly ALPHA_DISABLE = 0;
        /** Defines that alpha blending is SRC ALPHA * SRC + DEST */
        static readonly ALPHA_ADD = 1;
        /** Defines that alpha blending is SRC ALPHA * SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_COMBINE = 2;
        /** Defines that alpha blending is DEST - SRC * DEST */
        static readonly ALPHA_SUBTRACT = 3;
        /** Defines that alpha blending is SRC * DEST */
        static readonly ALPHA_MULTIPLY = 4;
        /** Defines that alpha blending is SRC ALPHA * SRC + (1 - SRC) * DEST */
        static readonly ALPHA_MAXIMIZED = 5;
        /** Defines that alpha blending is SRC + DEST */
        static readonly ALPHA_ONEONE = 6;
        /** Defines that alpha blending is SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_PREMULTIPLIED = 7;
        /**
         * Defines that alpha blending is SRC + (1 - SRC ALPHA) * DEST
         * Alpha will be set to (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_PREMULTIPLIED_PORTERDUFF = 8;
        /** Defines that alpha blending is CST * SRC + (1 - CST) * DEST */
        static readonly ALPHA_INTERPOLATE = 9;
        /**
         * Defines that alpha blending is SRC + (1 - SRC) * DEST
         * Alpha will be set to SRC ALPHA + (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_SCREENMODE = 10;
        /**
         * Defines that alpha blending is SRC + DST
         * Alpha will be set to SRC ALPHA + DST ALPHA
         */
        static readonly ALPHA_ONEONE_ONEONE = 11;
        /**
         * Defines that alpha blending is SRC * DST ALPHA + DST
         * Alpha will be set to 0
         */
        static readonly ALPHA_ALPHATOCOLOR = 12;
        /**
         * Defines that alpha blending is SRC * (1 - DST) + DST * (1 - SRC)
         */
        static readonly ALPHA_REVERSEONEMINUS = 13;
        /**
         * Defines that alpha blending is SRC + DST * (1 - SRC ALPHA)
         * Alpha will be set to SRC ALPHA + DST ALPHA * (1 - SRC ALPHA)
         */
        static readonly ALPHA_SRC_DSTONEMINUSSRCALPHA = 14;
        /**
         * Defines that alpha blending is SRC + DST
         * Alpha will be set to SRC ALPHA
         */
        static readonly ALPHA_ONEONE_ONEZERO = 15;
        /**
         * Defines that alpha blending is SRC * (1 - DST) + DST * (1 - SRC)
         * Alpha will be set to DST ALPHA
         */
        static readonly ALPHA_EXCLUSION = 16;
        /**
         * Defines that alpha blending is SRC * SRC ALPHA + DST * (1 - SRC ALPHA)
         * Alpha will be set to SRC ALPHA + (1 - SRC ALPHA) * DST ALPHA
         */
        static readonly ALPHA_LAYER_ACCUMULATE = 17;
        /** Defines that alpha blending equation a SUM */
        static readonly ALPHA_EQUATION_ADD = 0;
        /** Defines that alpha blending equation a SUBSTRACTION */
        static readonly ALPHA_EQUATION_SUBSTRACT = 1;
        /** Defines that alpha blending equation a REVERSE SUBSTRACTION */
        static readonly ALPHA_EQUATION_REVERSE_SUBTRACT = 2;
        /** Defines that alpha blending equation a MAX operation */
        static readonly ALPHA_EQUATION_MAX = 3;
        /** Defines that alpha blending equation a MIN operation */
        static readonly ALPHA_EQUATION_MIN = 4;
        /**
         * Defines that alpha blending equation a DARKEN operation:
         * It takes the min of the src and sums the alpha channels.
         */
        static readonly ALPHA_EQUATION_DARKEN = 5;
        /** Defines that the resource is not delayed*/
        static readonly DELAYLOADSTATE_NONE = 0;
        /** Defines that the resource was successfully delay loaded */
        static readonly DELAYLOADSTATE_LOADED = 1;
        /** Defines that the resource is currently delay loading */
        static readonly DELAYLOADSTATE_LOADING = 2;
        /** Defines that the resource is delayed and has not started loading */
        static readonly DELAYLOADSTATE_NOTLOADED = 4;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will never pass. i.e. Nothing will be drawn */
        static readonly NEVER = 512;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will always pass. i.e. Pixels will be drawn in the order they are drawn */
        static readonly ALWAYS = 519;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than the stored value */
        static readonly LESS = 513;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is equals to the stored value */
        static readonly EQUAL = 514;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than or equal to the stored value */
        static readonly LEQUAL = 515;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than the stored value */
        static readonly GREATER = 516;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than or equal to the stored value */
        static readonly GEQUAL = 518;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is not equal to the stored value */
        static readonly NOTEQUAL = 517;
        /** Passed to stencilOperation to specify that stencil value must be kept */
        static readonly KEEP = 7680;
        /** Passed to stencilOperation to specify that stencil value must be zero */
        static readonly ZERO = 0;
        /** Passed to stencilOperation to specify that stencil value must be replaced */
        static readonly REPLACE = 7681;
        /** Passed to stencilOperation to specify that stencil value must be incremented */
        static readonly INCR = 7682;
        /** Passed to stencilOperation to specify that stencil value must be decremented */
        static readonly DECR = 7683;
        /** Passed to stencilOperation to specify that stencil value must be inverted */
        static readonly INVERT = 5386;
        /** Passed to stencilOperation to specify that stencil value must be incremented with wrapping */
        static readonly INCR_WRAP = 34055;
        /** Passed to stencilOperation to specify that stencil value must be decremented with wrapping */
        static readonly DECR_WRAP = 34056;
        /** Texture is not repeating outside of 0..1 UVs */
        static readonly TEXTURE_CLAMP_ADDRESSMODE = 0;
        /** Texture is repeating outside of 0..1 UVs */
        static readonly TEXTURE_WRAP_ADDRESSMODE = 1;
        /** Texture is repeating and mirrored */
        static readonly TEXTURE_MIRROR_ADDRESSMODE = 2;
        /** Flag to create a storage texture */
        static readonly TEXTURE_CREATIONFLAG_STORAGE = 1;
        /** ALPHA */
        static readonly TEXTUREFORMAT_ALPHA = 0;
        /** LUMINANCE */
        static readonly TEXTUREFORMAT_LUMINANCE = 1;
        /** LUMINANCE_ALPHA */
        static readonly TEXTUREFORMAT_LUMINANCE_ALPHA = 2;
        /** RGB */
        static readonly TEXTUREFORMAT_RGB = 4;
        /** RGBA */
        static readonly TEXTUREFORMAT_RGBA = 5;
        /** RED */
        static readonly TEXTUREFORMAT_RED = 6;
        /** RED (2nd reference) */
        static readonly TEXTUREFORMAT_R = 6;
        /** RG */
        static readonly TEXTUREFORMAT_RG = 7;
        /** RED_INTEGER */
        static readonly TEXTUREFORMAT_RED_INTEGER = 8;
        /** RED_INTEGER (2nd reference) */
        static readonly TEXTUREFORMAT_R_INTEGER = 8;
        /** RG_INTEGER */
        static readonly TEXTUREFORMAT_RG_INTEGER = 9;
        /** RGB_INTEGER */
        static readonly TEXTUREFORMAT_RGB_INTEGER = 10;
        /** RGBA_INTEGER */
        static readonly TEXTUREFORMAT_RGBA_INTEGER = 11;
        /** BGRA */
        static readonly TEXTUREFORMAT_BGRA = 12;
        /** Depth 24 bits + Stencil 8 bits */
        static readonly TEXTUREFORMAT_DEPTH24_STENCIL8 = 13;
        /** Depth 32 bits float */
        static readonly TEXTUREFORMAT_DEPTH32_FLOAT = 14;
        /** Depth 16 bits */
        static readonly TEXTUREFORMAT_DEPTH16 = 15;
        /** Depth 24 bits */
        static readonly TEXTUREFORMAT_DEPTH24 = 16;
        /** Depth 24 bits unorm + Stencil 8 bits */
        static readonly TEXTUREFORMAT_DEPTH24UNORM_STENCIL8 = 17;
        /** Depth 32 bits float + Stencil 8 bits */
        static readonly TEXTUREFORMAT_DEPTH32FLOAT_STENCIL8 = 18;
        /** Stencil 8 bits */
        static readonly TEXTUREFORMAT_STENCIL8 = 19;
        /** UNDEFINED */
        static readonly TEXTUREFORMAT_UNDEFINED = 4294967295;
        /** Compressed BC7 */
        static readonly TEXTUREFORMAT_COMPRESSED_RGBA_BPTC_UNORM = 36492;
        /** Compressed BC7 (SRGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB_ALPHA_BPTC_UNORM = 36493;
        /** Compressed BC6 unsigned float */
        static readonly TEXTUREFORMAT_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT = 36495;
        /** Compressed BC6 signed float */
        static readonly TEXTUREFORMAT_COMPRESSED_RGB_BPTC_SIGNED_FLOAT = 36494;
        /** Compressed BC3 */
        static readonly TEXTUREFORMAT_COMPRESSED_RGBA_S3TC_DXT5 = 33779;
        /** Compressed BC3 (SRGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT = 35919;
        /** Compressed BC2 */
        static readonly TEXTUREFORMAT_COMPRESSED_RGBA_S3TC_DXT3 = 33778;
        /** Compressed BC2 (SRGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT = 35918;
        /** Compressed BC1 (RGBA) */
        static readonly TEXTUREFORMAT_COMPRESSED_RGBA_S3TC_DXT1 = 33777;
        /** Compressed BC1 (RGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_RGB_S3TC_DXT1 = 33776;
        /** Compressed BC1 (SRGB+A) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT = 35917;
        /** Compressed BC1 (SRGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB_S3TC_DXT1_EXT = 35916;
        /** Compressed ASTC 4x4 */
        static readonly TEXTUREFORMAT_COMPRESSED_RGBA_ASTC_4x4 = 37808;
        /** Compressed ASTC 4x4 (SRGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB8_ALPHA8_ASTC_4x4_KHR = 37840;
        /** Compressed ETC1 (RGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_RGB_ETC1_WEBGL = 36196;
        /** Compressed ETC2 (RGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_RGB8_ETC2 = 37492;
        /** Compressed ETC2 (SRGB) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB8_ETC2 = 37493;
        /** Compressed ETC2 (RGB+A1) */
        static readonly TEXTUREFORMAT_COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2 = 37494;
        /** Compressed ETC2 (SRGB+A1)*/
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2 = 37495;
        /** Compressed ETC2 (RGB+A) */
        static readonly TEXTUREFORMAT_COMPRESSED_RGBA8_ETC2_EAC = 37496;
        /** Compressed ETC2 (SRGB+1) */
        static readonly TEXTUREFORMAT_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC = 37497;
        /** UNSIGNED_BYTE */
        static readonly TEXTURETYPE_UNSIGNED_BYTE = 0;
        /** UNSIGNED_BYTE (2nd reference) */
        static readonly TEXTURETYPE_UNSIGNED_INT = 0;
        /** FLOAT */
        static readonly TEXTURETYPE_FLOAT = 1;
        /** HALF_FLOAT */
        static readonly TEXTURETYPE_HALF_FLOAT = 2;
        /** BYTE */
        static readonly TEXTURETYPE_BYTE = 3;
        /** SHORT */
        static readonly TEXTURETYPE_SHORT = 4;
        /** UNSIGNED_SHORT */
        static readonly TEXTURETYPE_UNSIGNED_SHORT = 5;
        /** INT */
        static readonly TEXTURETYPE_INT = 6;
        /** UNSIGNED_INT */
        static readonly TEXTURETYPE_UNSIGNED_INTEGER = 7;
        /** UNSIGNED_SHORT_4_4_4_4 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_4_4_4_4 = 8;
        /** UNSIGNED_SHORT_5_5_5_1 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_5_5_5_1 = 9;
        /** UNSIGNED_SHORT_5_6_5 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_5_6_5 = 10;
        /** UNSIGNED_INT_2_10_10_10_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_2_10_10_10_REV = 11;
        /** UNSIGNED_INT_24_8 */
        static readonly TEXTURETYPE_UNSIGNED_INT_24_8 = 12;
        /** UNSIGNED_INT_10F_11F_11F_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_10F_11F_11F_REV = 13;
        /** UNSIGNED_INT_5_9_9_9_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_5_9_9_9_REV = 14;
        /** FLOAT_32_UNSIGNED_INT_24_8_REV */
        static readonly TEXTURETYPE_FLOAT_32_UNSIGNED_INT_24_8_REV = 15;
        /** UNDEFINED */
        static readonly TEXTURETYPE_UNDEFINED = 16;
        /** 2D Texture target*/
        static readonly TEXTURE_2D = 3553;
        /** 2D Array Texture target */
        static readonly TEXTURE_2D_ARRAY = 35866;
        /** Cube Map Texture target */
        static readonly TEXTURE_CUBE_MAP = 34067;
        /** Cube Map Array Texture target */
        static readonly TEXTURE_CUBE_MAP_ARRAY = 3735928559;
        /** 3D Texture target */
        static readonly TEXTURE_3D = 32879;
        /** nearest is mag = nearest and min = nearest and no mip */
        static readonly TEXTURE_NEAREST_SAMPLINGMODE = 1;
        /** mag = nearest and min = nearest and mip = none */
        static readonly TEXTURE_NEAREST_NEAREST = 1;
        /** Bilinear is mag = linear and min = linear and no mip */
        static readonly TEXTURE_BILINEAR_SAMPLINGMODE = 2;
        /** mag = linear and min = linear and mip = none */
        static readonly TEXTURE_LINEAR_LINEAR = 2;
        /** Trilinear is mag = linear and min = linear and mip = linear */
        static readonly TEXTURE_TRILINEAR_SAMPLINGMODE = 3;
        /** Trilinear is mag = linear and min = linear and mip = linear */
        static readonly TEXTURE_LINEAR_LINEAR_MIPLINEAR = 3;
        /** mag = nearest and min = nearest and mip = nearest */
        static readonly TEXTURE_NEAREST_NEAREST_MIPNEAREST = 4;
        /** mag = nearest and min = linear and mip = nearest */
        static readonly TEXTURE_NEAREST_LINEAR_MIPNEAREST = 5;
        /** mag = nearest and min = linear and mip = linear */
        static readonly TEXTURE_NEAREST_LINEAR_MIPLINEAR = 6;
        /** mag = nearest and min = linear and mip = none */
        static readonly TEXTURE_NEAREST_LINEAR = 7;
        /** nearest is mag = nearest and min = nearest and mip = linear */
        static readonly TEXTURE_NEAREST_NEAREST_MIPLINEAR = 8;
        /** mag = linear and min = nearest and mip = nearest */
        static readonly TEXTURE_LINEAR_NEAREST_MIPNEAREST = 9;
        /** mag = linear and min = nearest and mip = linear */
        static readonly TEXTURE_LINEAR_NEAREST_MIPLINEAR = 10;
        /** Bilinear is mag = linear and min = linear and mip = nearest */
        static readonly TEXTURE_LINEAR_LINEAR_MIPNEAREST = 11;
        /** mag = linear and min = nearest and mip = none */
        static readonly TEXTURE_LINEAR_NEAREST = 12;
        /** Explicit coordinates mode */
        static readonly TEXTURE_EXPLICIT_MODE = 0;
        /** Spherical coordinates mode */
        static readonly TEXTURE_SPHERICAL_MODE = 1;
        /** Planar coordinates mode */
        static readonly TEXTURE_PLANAR_MODE = 2;
        /** Cubic coordinates mode */
        static readonly TEXTURE_CUBIC_MODE = 3;
        /** Projection coordinates mode */
        static readonly TEXTURE_PROJECTION_MODE = 4;
        /** Skybox coordinates mode */
        static readonly TEXTURE_SKYBOX_MODE = 5;
        /** Inverse Cubic coordinates mode */
        static readonly TEXTURE_INVCUBIC_MODE = 6;
        /** Equirectangular coordinates mode */
        static readonly TEXTURE_EQUIRECTANGULAR_MODE = 7;
        /** Equirectangular Fixed coordinates mode */
        static readonly TEXTURE_FIXED_EQUIRECTANGULAR_MODE = 8;
        /** Equirectangular Fixed Mirrored coordinates mode */
        static readonly TEXTURE_FIXED_EQUIRECTANGULAR_MIRRORED_MODE = 9;
        /** Offline (baking) quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_OFFLINE = 4096;
        /** High quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_HIGH = 64;
        /** Medium quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_MEDIUM = 16;
        /** Low quality for texture filtering */
        static readonly TEXTURE_FILTERING_QUALITY_LOW = 8;
        /** Defines that texture rescaling will use a floor to find the closer power of 2 size */
        static readonly SCALEMODE_FLOOR = 1;
        /** Defines that texture rescaling will look for the nearest power of 2 size */
        static readonly SCALEMODE_NEAREST = 2;
        /** Defines that texture rescaling will use a ceil to find the closer power of 2 size */
        static readonly SCALEMODE_CEILING = 3;
        /**
         * The dirty texture flag value
         */
        static readonly MATERIAL_TextureDirtyFlag = 1;
        /**
         * The dirty light flag value
         */
        static readonly MATERIAL_LightDirtyFlag = 2;
        /**
         * The dirty fresnel flag value
         */
        static readonly MATERIAL_FresnelDirtyFlag = 4;
        /**
         * The dirty attribute flag value
         */
        static readonly MATERIAL_AttributesDirtyFlag = 8;
        /**
         * The dirty misc flag value
         */
        static readonly MATERIAL_MiscDirtyFlag = 16;
        /**
         * The dirty prepass flag value
         */
        static readonly MATERIAL_PrePassDirtyFlag = 32;
        /**
         * The all dirty flag value
         */
        static readonly MATERIAL_AllDirtyFlag = 63;
        /**
         * Returns the triangle fill mode
         */
        static readonly MATERIAL_TriangleFillMode = 0;
        /**
         * Returns the wireframe mode
         */
        static readonly MATERIAL_WireFrameFillMode = 1;
        /**
         * Returns the point fill mode
         */
        static readonly MATERIAL_PointFillMode = 2;
        /**
         * Returns the point list draw mode
         */
        static readonly MATERIAL_PointListDrawMode = 3;
        /**
         * Returns the line list draw mode
         */
        static readonly MATERIAL_LineListDrawMode = 4;
        /**
         * Returns the line loop draw mode
         */
        static readonly MATERIAL_LineLoopDrawMode = 5;
        /**
         * Returns the line strip draw mode
         */
        static readonly MATERIAL_LineStripDrawMode = 6;
        /**
         * Returns the triangle strip draw mode
         */
        static readonly MATERIAL_TriangleStripDrawMode = 7;
        /**
         * Returns the triangle fan draw mode
         */
        static readonly MATERIAL_TriangleFanDrawMode = 8;
        /**
         * Stores the clock-wise side orientation
         */
        static readonly MATERIAL_ClockWiseSideOrientation = 0;
        /**
         * Stores the counter clock-wise side orientation
         */
        static readonly MATERIAL_CounterClockWiseSideOrientation = 1;
        /**
         * Nothing
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_NothingTrigger = 0;
        /**
         * On pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnPickTrigger = 1;
        /**
         * On left pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnLeftPickTrigger = 2;
        /**
         * On right pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnRightPickTrigger = 3;
        /**
         * On center pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnCenterPickTrigger = 4;
        /**
         * On pick down
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnPickDownTrigger = 5;
        /**
         * On double pick
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnDoublePickTrigger = 6;
        /**
         * On pick up
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnPickUpTrigger = 7;
        /**
         * On pick out.
         * This trigger will only be raised if you also declared a OnPickDown
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnPickOutTrigger = 16;
        /**
         * On long press
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnLongPressTrigger = 8;
        /**
         * On pointer over
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnPointerOverTrigger = 9;
        /**
         * On pointer out
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnPointerOutTrigger = 10;
        /**
         * On every frame
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnEveryFrameTrigger = 11;
        /**
         * On intersection enter
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnIntersectionEnterTrigger = 12;
        /**
         * On intersection exit
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnIntersectionExitTrigger = 13;
        /**
         * On key down
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnKeyDownTrigger = 14;
        /**
         * On key up
         * @see https://doc.babylonjs.com/features/featuresDeepDive/events/actions#triggers
         */
        static readonly ACTION_OnKeyUpTrigger = 15;
        /**
         * Billboard mode will only apply to Y axis
         */
        static readonly PARTICLES_BILLBOARDMODE_Y = 2;
        /**
         * Billboard mode will apply to all axes
         */
        static readonly PARTICLES_BILLBOARDMODE_ALL = 7;
        /**
         * Special billboard mode where the particle will be biilboard to the camera but rotated to align with direction
         */
        static readonly PARTICLES_BILLBOARDMODE_STRETCHED = 8;
        /**
         * Special billboard mode where the particle will be billboard to the camera but only around the axis of the direction of particle emission
         */
        static readonly PARTICLES_BILLBOARDMODE_STRETCHED_LOCAL = 9;
        /** Default culling strategy : this is an exclusion test and it's the more accurate.
         *  Test order :
         *  Is the bounding sphere outside the frustum ?
         *  If not, are the bounding box vertices outside the frustum ?
         *  It not, then the cullable object is in the frustum.
         */
        static readonly MESHES_CULLINGSTRATEGY_STANDARD = 0;
        /** Culling strategy : Bounding Sphere Only.
         *  This is an exclusion test. It's faster than the standard strategy because the bounding box is not tested.
         *  It's also less accurate than the standard because some not visible objects can still be selected.
         *  Test : is the bounding sphere outside the frustum ?
         *  If not, then the cullable object is in the frustum.
         */
        static readonly MESHES_CULLINGSTRATEGY_BOUNDINGSPHERE_ONLY = 1;
        /** Culling strategy : Optimistic Inclusion.
         *  This in an inclusion test first, then the standard exclusion test.
         *  This can be faster when a cullable object is expected to be almost always in the camera frustum.
         *  This could also be a little slower than the standard test when the tested object center is not the frustum but one of its bounding box vertex is still inside.
         *  Anyway, it's as accurate as the standard strategy.
         *  Test :
         *  Is the cullable object bounding sphere center in the frustum ?
         *  If not, apply the default culling strategy.
         */
        static readonly MESHES_CULLINGSTRATEGY_OPTIMISTIC_INCLUSION = 2;
        /** Culling strategy : Optimistic Inclusion then Bounding Sphere Only.
         *  This in an inclusion test first, then the bounding sphere only exclusion test.
         *  This can be the fastest test when a cullable object is expected to be almost always in the camera frustum.
         *  This could also be a little slower than the BoundingSphereOnly strategy when the tested object center is not in the frustum but its bounding sphere still intersects it.
         *  It's less accurate than the standard strategy and as accurate as the BoundingSphereOnly strategy.
         *  Test :
         *  Is the cullable object bounding sphere center in the frustum ?
         *  If not, apply the Bounding Sphere Only strategy. No Bounding Box is tested here.
         */
        static readonly MESHES_CULLINGSTRATEGY_OPTIMISTIC_INCLUSION_THEN_BSPHERE_ONLY = 3;
        /**
         * No logging while loading
         */
        static readonly SCENELOADER_NO_LOGGING = 0;
        /**
         * Minimal logging while loading
         */
        static readonly SCENELOADER_MINIMAL_LOGGING = 1;
        /**
         * Summary logging while loading
         */
        static readonly SCENELOADER_SUMMARY_LOGGING = 2;
        /**
         * Detailed logging while loading
         */
        static readonly SCENELOADER_DETAILED_LOGGING = 3;
        /**
         * Constant used to retrieve the irradiance texture index in the textures array in the prepass
         * using getIndex(Constants.PREPASS_IRRADIANCE_TEXTURE_TYPE)
         */
        static readonly PREPASS_IRRADIANCE_TEXTURE_TYPE = 0;
        /**
         * Constant used to retrieve the position texture index in the textures array in the prepass
         * using getIndex(Constants.PREPASS_POSITION_TEXTURE_INDEX)
         */
        static readonly PREPASS_POSITION_TEXTURE_TYPE = 1;
        /**
         * Constant used to retrieve the velocity texture index in the textures array in the prepass
         * using getIndex(Constants.PREPASS_VELOCITY_TEXTURE_INDEX)
         */
        static readonly PREPASS_VELOCITY_TEXTURE_TYPE = 2;
        /**
         * Constant used to retrieve the reflectivity texture index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_REFLECTIVITY_TEXTURE_TYPE)
         */
        static readonly PREPASS_REFLECTIVITY_TEXTURE_TYPE = 3;
        /**
         * Constant used to retrieve the lit color texture index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_COLOR_TEXTURE_TYPE)
         */
        static readonly PREPASS_COLOR_TEXTURE_TYPE = 4;
        /**
         * Constant used to retrieve depth index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_DEPTH_TEXTURE_TYPE)
         */
        static readonly PREPASS_DEPTH_TEXTURE_TYPE = 5;
        /**
         * Constant used to retrieve normal index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_NORMAL_TEXTURE_TYPE)
         */
        static readonly PREPASS_NORMAL_TEXTURE_TYPE = 6;
        /**
         * Constant used to retrieve albedo index in the textures array in the prepass
         * using the getIndex(Constants.PREPASS_ALBEDO_SQRT_TEXTURE_TYPE)
         */
        static readonly PREPASS_ALBEDO_SQRT_TEXTURE_TYPE = 7;
        /** Flag to create a readable buffer (the buffer can be the source of a copy) */
        static readonly BUFFER_CREATIONFLAG_READ = 1;
        /** Flag to create a writable buffer (the buffer can be the destination of a copy) */
        static readonly BUFFER_CREATIONFLAG_WRITE = 2;
        /** Flag to create a readable and writable buffer */
        static readonly BUFFER_CREATIONFLAG_READWRITE = 3;
        /** Flag to create a buffer suitable to be used as a uniform buffer */
        static readonly BUFFER_CREATIONFLAG_UNIFORM = 4;
        /** Flag to create a buffer suitable to be used as a vertex buffer */
        static readonly BUFFER_CREATIONFLAG_VERTEX = 8;
        /** Flag to create a buffer suitable to be used as an index buffer */
        static readonly BUFFER_CREATIONFLAG_INDEX = 16;
        /** Flag to create a buffer suitable to be used as a storage buffer */
        static readonly BUFFER_CREATIONFLAG_STORAGE = 32;
        /**
         * Prefixes used by the engine for sub mesh draw wrappers
         */
        /** @internal */
        static readonly RENDERPASS_MAIN = 0;
        /**
         * Constant used as key code for Alt key
         */
        static readonly INPUT_ALT_KEY = 18;
        /**
         * Constant used as key code for Ctrl key
         */
        static readonly INPUT_CTRL_KEY = 17;
        /**
         * Constant used as key code for Meta key (Left Win, Left Cmd)
         */
        static readonly INPUT_META_KEY1 = 91;
        /**
         * Constant used as key code for Meta key (Right Win)
         */
        static readonly INPUT_META_KEY2 = 92;
        /**
         * Constant used as key code for Meta key (Right Win, Right Cmd)
         */
        static readonly INPUT_META_KEY3 = 93;
        /**
         * Constant used as key code for Shift key
         */
        static readonly INPUT_SHIFT_KEY = 16;
        /** Standard snapshot rendering. In this mode, some form of dynamic behavior is possible (for eg, uniform buffers are still updated) */
        static readonly SNAPSHOTRENDERING_STANDARD = 0;
        /** Fast snapshot rendering. In this mode, everything is static and only some limited form of dynamic behaviour is possible */
        static readonly SNAPSHOTRENDERING_FAST = 1;
        /**
         * This is the default projection mode used by the cameras.
         * It helps recreating a feeling of perspective and better appreciate depth.
         * This is the best way to simulate real life cameras.
         */
        static readonly PERSPECTIVE_CAMERA = 0;
        /**
         * This helps creating camera with an orthographic mode.
         * Orthographic is commonly used in engineering as a means to produce object specifications that communicate dimensions unambiguously, each line of 1 unit length (cm, meter..whatever) will appear to have the same length everywhere on the drawing. This allows the drafter to dimension only a subset of lines and let the reader know that other lines of that length on the drawing are also that length in reality. Every parallel line in the drawing is also parallel in the object.
         */
        static readonly ORTHOGRAPHIC_CAMERA = 1;
        /**
         * This is the default FOV mode for perspective cameras.
         * This setting aligns the upper and lower bounds of the viewport to the upper and lower bounds of the camera frustum.
         */
        static readonly FOVMODE_VERTICAL_FIXED = 0;
        /**
         * This setting aligns the left and right bounds of the viewport to the left and right bounds of the camera frustum.
         */
        static readonly FOVMODE_HORIZONTAL_FIXED = 1;
        /**
         * This specifies there is no need for a camera rig.
         * Basically only one eye is rendered corresponding to the camera.
         */
        static readonly RIG_MODE_NONE = 0;
        /**
         * Simulates a camera Rig with one blue eye and one red eye.
         * This can be use with 3d blue and red glasses.
         */
        static readonly RIG_MODE_STEREOSCOPIC_ANAGLYPH = 10;
        /**
         * Defines that both eyes of the camera will be rendered side by side with a parallel target.
         */
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_PARALLEL = 11;
        /**
         * Defines that both eyes of the camera will be rendered side by side with a none parallel target.
         */
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_CROSSEYED = 12;
        /**
         * Defines that both eyes of the camera will be rendered over under each other.
         */
        static readonly RIG_MODE_STEREOSCOPIC_OVERUNDER = 13;
        /**
         * Defines that both eyes of the camera will be rendered on successive lines interlaced for passive 3d monitors.
         */
        static readonly RIG_MODE_STEREOSCOPIC_INTERLACED = 14;
        /**
         * Defines that both eyes of the camera should be renderered in a VR mode (carbox).
         */
        static readonly RIG_MODE_VR = 20;
        /**
         * Custom rig mode allowing rig cameras to be populated manually with any number of cameras
         */
        static readonly RIG_MODE_CUSTOM = 22;
        /**
         * Maximum number of uv sets supported
         */
        static readonly MAX_SUPPORTED_UV_SETS = 6;
        /**
         * GL constants
         */
        /** Alpha blend equation: ADD */
        static readonly GL_ALPHA_EQUATION_ADD = 32774;
        /** Alpha equation: MIN */
        static readonly GL_ALPHA_EQUATION_MIN = 32775;
        /** Alpha equation: MAX */
        static readonly GL_ALPHA_EQUATION_MAX = 32776;
        /** Alpha equation: SUBTRACT */
        static readonly GL_ALPHA_EQUATION_SUBTRACT = 32778;
        /** Alpha equation: REVERSE_SUBTRACT */
        static readonly GL_ALPHA_EQUATION_REVERSE_SUBTRACT = 32779;
        /** Alpha blend function: SRC */
        static readonly GL_ALPHA_FUNCTION_SRC = 768;
        /** Alpha blend function: ONE_MINUS_SRC */
        static readonly GL_ALPHA_FUNCTION_ONE_MINUS_SRC_COLOR = 769;
        /** Alpha blend function: SRC_ALPHA */
        static readonly GL_ALPHA_FUNCTION_SRC_ALPHA = 770;
        /** Alpha blend function: ONE_MINUS_SRC_ALPHA */
        static readonly GL_ALPHA_FUNCTION_ONE_MINUS_SRC_ALPHA = 771;
        /** Alpha blend function: DST_ALPHA */
        static readonly GL_ALPHA_FUNCTION_DST_ALPHA = 772;
        /** Alpha blend function: ONE_MINUS_DST_ALPHA */
        static readonly GL_ALPHA_FUNCTION_ONE_MINUS_DST_ALPHA = 773;
        /** Alpha blend function: ONE_MINUS_DST */
        static readonly GL_ALPHA_FUNCTION_DST_COLOR = 774;
        /** Alpha blend function: ONE_MINUS_DST */
        static readonly GL_ALPHA_FUNCTION_ONE_MINUS_DST_COLOR = 775;
        /** Alpha blend function: SRC_ALPHA_SATURATED */
        static readonly GL_ALPHA_FUNCTION_SRC_ALPHA_SATURATED = 776;
        /** Alpha blend function: CONSTANT */
        static readonly GL_ALPHA_FUNCTION_CONSTANT_COLOR = 32769;
        /** Alpha blend function: ONE_MINUS_CONSTANT */
        static readonly GL_ALPHA_FUNCTION_ONE_MINUS_CONSTANT_COLOR = 32770;
        /** Alpha blend function: CONSTANT_ALPHA */
        static readonly GL_ALPHA_FUNCTION_CONSTANT_ALPHA = 32771;
        /** Alpha blend function: ONE_MINUS_CONSTANT_ALPHA */
        static readonly GL_ALPHA_FUNCTION_ONE_MINUS_CONSTANT_ALPHA = 32772;
        /** URL to the snippet server. Points to the public snippet server by default */
        static SnippetUrl: string;
    }


    /**
     * Defines the interface used by objects containing a viewport (like a camera)
     */
    interface IViewportOwnerLike {
        /**
         * Gets or sets the viewport
         */
        viewport: IViewportLike;
    }
    /**
     * The engine class is responsible for interfacing with all lower-level APIs such as WebGL and Audio
     */
    export class Engine extends ThinEngine {
        /** Defines that alpha blending is disabled */
        static readonly ALPHA_DISABLE = 0;
        /** Defines that alpha blending to SRC ALPHA * SRC + DEST */
        static readonly ALPHA_ADD = 1;
        /** Defines that alpha blending to SRC ALPHA * SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_COMBINE = 2;
        /** Defines that alpha blending to DEST - SRC * DEST */
        static readonly ALPHA_SUBTRACT = 3;
        /** Defines that alpha blending to SRC * DEST */
        static readonly ALPHA_MULTIPLY = 4;
        /** Defines that alpha blending to SRC ALPHA * SRC + (1 - SRC) * DEST */
        static readonly ALPHA_MAXIMIZED = 5;
        /** Defines that alpha blending to SRC + DEST */
        static readonly ALPHA_ONEONE = 6;
        /** Defines that alpha blending to SRC + (1 - SRC ALPHA) * DEST */
        static readonly ALPHA_PREMULTIPLIED = 7;
        /**
         * Defines that alpha blending to SRC + (1 - SRC ALPHA) * DEST
         * Alpha will be set to (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_PREMULTIPLIED_PORTERDUFF = 8;
        /** Defines that alpha blending to CST * SRC + (1 - CST) * DEST */
        static readonly ALPHA_INTERPOLATE = 9;
        /**
         * Defines that alpha blending to SRC + (1 - SRC) * DEST
         * Alpha will be set to SRC ALPHA + (1 - SRC ALPHA) * DEST ALPHA
         */
        static readonly ALPHA_SCREENMODE = 10;
        /** Defines that the resource is not delayed*/
        static readonly DELAYLOADSTATE_NONE = 0;
        /** Defines that the resource was successfully delay loaded */
        static readonly DELAYLOADSTATE_LOADED = 1;
        /** Defines that the resource is currently delay loading */
        static readonly DELAYLOADSTATE_LOADING = 2;
        /** Defines that the resource is delayed and has not started loading */
        static readonly DELAYLOADSTATE_NOTLOADED = 4;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will never pass. i.e. Nothing will be drawn */
        static readonly NEVER = 512;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will always pass. i.e. Pixels will be drawn in the order they are drawn */
        static readonly ALWAYS = 519;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than the stored value */
        static readonly LESS = 513;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is equals to the stored value */
        static readonly EQUAL = 514;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is less than or equal to the stored value */
        static readonly LEQUAL = 515;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than the stored value */
        static readonly GREATER = 516;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is greater than or equal to the stored value */
        static readonly GEQUAL = 518;
        /** Passed to depthFunction or stencilFunction to specify depth or stencil tests will pass if the new depth value is not equal to the stored value */
        static readonly NOTEQUAL = 517;
        /** Passed to stencilOperation to specify that stencil value must be kept */
        static readonly KEEP = 7680;
        /** Passed to stencilOperation to specify that stencil value must be replaced */
        static readonly REPLACE = 7681;
        /** Passed to stencilOperation to specify that stencil value must be incremented */
        static readonly INCR = 7682;
        /** Passed to stencilOperation to specify that stencil value must be decremented */
        static readonly DECR = 7683;
        /** Passed to stencilOperation to specify that stencil value must be inverted */
        static readonly INVERT = 5386;
        /** Passed to stencilOperation to specify that stencil value must be incremented with wrapping */
        static readonly INCR_WRAP = 34055;
        /** Passed to stencilOperation to specify that stencil value must be decremented with wrapping */
        static readonly DECR_WRAP = 34056;
        /** Texture is not repeating outside of 0..1 UVs */
        static readonly TEXTURE_CLAMP_ADDRESSMODE = 0;
        /** Texture is repeating outside of 0..1 UVs */
        static readonly TEXTURE_WRAP_ADDRESSMODE = 1;
        /** Texture is repeating and mirrored */
        static readonly TEXTURE_MIRROR_ADDRESSMODE = 2;
        /** ALPHA */
        static readonly TEXTUREFORMAT_ALPHA = 0;
        /** LUMINANCE */
        static readonly TEXTUREFORMAT_LUMINANCE = 1;
        /** LUMINANCE_ALPHA */
        static readonly TEXTUREFORMAT_LUMINANCE_ALPHA = 2;
        /** RGB */
        static readonly TEXTUREFORMAT_RGB = 4;
        /** RGBA */
        static readonly TEXTUREFORMAT_RGBA = 5;
        /** RED */
        static readonly TEXTUREFORMAT_RED = 6;
        /** RED (2nd reference) */
        static readonly TEXTUREFORMAT_R = 6;
        /** RG */
        static readonly TEXTUREFORMAT_RG = 7;
        /** RED_INTEGER */
        static readonly TEXTUREFORMAT_RED_INTEGER = 8;
        /** RED_INTEGER (2nd reference) */
        static readonly TEXTUREFORMAT_R_INTEGER = 8;
        /** RG_INTEGER */
        static readonly TEXTUREFORMAT_RG_INTEGER = 9;
        /** RGB_INTEGER */
        static readonly TEXTUREFORMAT_RGB_INTEGER = 10;
        /** RGBA_INTEGER */
        static readonly TEXTUREFORMAT_RGBA_INTEGER = 11;
        /** UNSIGNED_BYTE */
        static readonly TEXTURETYPE_UNSIGNED_BYTE = 0;
        /** UNSIGNED_BYTE (2nd reference) */
        static readonly TEXTURETYPE_UNSIGNED_INT = 0;
        /** FLOAT */
        static readonly TEXTURETYPE_FLOAT = 1;
        /** HALF_FLOAT */
        static readonly TEXTURETYPE_HALF_FLOAT = 2;
        /** BYTE */
        static readonly TEXTURETYPE_BYTE = 3;
        /** SHORT */
        static readonly TEXTURETYPE_SHORT = 4;
        /** UNSIGNED_SHORT */
        static readonly TEXTURETYPE_UNSIGNED_SHORT = 5;
        /** INT */
        static readonly TEXTURETYPE_INT = 6;
        /** UNSIGNED_INT */
        static readonly TEXTURETYPE_UNSIGNED_INTEGER = 7;
        /** UNSIGNED_SHORT_4_4_4_4 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_4_4_4_4 = 8;
        /** UNSIGNED_SHORT_5_5_5_1 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_5_5_5_1 = 9;
        /** UNSIGNED_SHORT_5_6_5 */
        static readonly TEXTURETYPE_UNSIGNED_SHORT_5_6_5 = 10;
        /** UNSIGNED_INT_2_10_10_10_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_2_10_10_10_REV = 11;
        /** UNSIGNED_INT_24_8 */
        static readonly TEXTURETYPE_UNSIGNED_INT_24_8 = 12;
        /** UNSIGNED_INT_10F_11F_11F_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_10F_11F_11F_REV = 13;
        /** UNSIGNED_INT_5_9_9_9_REV */
        static readonly TEXTURETYPE_UNSIGNED_INT_5_9_9_9_REV = 14;
        /** FLOAT_32_UNSIGNED_INT_24_8_REV */
        static readonly TEXTURETYPE_FLOAT_32_UNSIGNED_INT_24_8_REV = 15;
        /** nearest is mag = nearest and min = nearest and mip = none */
        static readonly TEXTURE_NEAREST_SAMPLINGMODE = 1;
        /** Bilinear is mag = linear and min = linear and mip = nearest */
        static readonly TEXTURE_BILINEAR_SAMPLINGMODE = 2;
        /** Trilinear is mag = linear and min = linear and mip = linear */
        static readonly TEXTURE_TRILINEAR_SAMPLINGMODE = 3;
        /** nearest is mag = nearest and min = nearest and mip = linear */
        static readonly TEXTURE_NEAREST_NEAREST_MIPLINEAR = 8;
        /** Bilinear is mag = linear and min = linear and mip = nearest */
        static readonly TEXTURE_LINEAR_LINEAR_MIPNEAREST = 11;
        /** Trilinear is mag = linear and min = linear and mip = linear */
        static readonly TEXTURE_LINEAR_LINEAR_MIPLINEAR = 3;
        /** mag = nearest and min = nearest and mip = nearest */
        static readonly TEXTURE_NEAREST_NEAREST_MIPNEAREST = 4;
        /** mag = nearest and min = linear and mip = nearest */
        static readonly TEXTURE_NEAREST_LINEAR_MIPNEAREST = 5;
        /** mag = nearest and min = linear and mip = linear */
        static readonly TEXTURE_NEAREST_LINEAR_MIPLINEAR = 6;
        /** mag = nearest and min = linear and mip = none */
        static readonly TEXTURE_NEAREST_LINEAR = 7;
        /** mag = nearest and min = nearest and mip = none */
        static readonly TEXTURE_NEAREST_NEAREST = 1;
        /** mag = linear and min = nearest and mip = nearest */
        static readonly TEXTURE_LINEAR_NEAREST_MIPNEAREST = 9;
        /** mag = linear and min = nearest and mip = linear */
        static readonly TEXTURE_LINEAR_NEAREST_MIPLINEAR = 10;
        /** mag = linear and min = linear and mip = none */
        static readonly TEXTURE_LINEAR_LINEAR = 2;
        /** mag = linear and min = nearest and mip = none */
        static readonly TEXTURE_LINEAR_NEAREST = 12;
        /** Explicit coordinates mode */
        static readonly TEXTURE_EXPLICIT_MODE = 0;
        /** Spherical coordinates mode */
        static readonly TEXTURE_SPHERICAL_MODE = 1;
        /** Planar coordinates mode */
        static readonly TEXTURE_PLANAR_MODE = 2;
        /** Cubic coordinates mode */
        static readonly TEXTURE_CUBIC_MODE = 3;
        /** Projection coordinates mode */
        static readonly TEXTURE_PROJECTION_MODE = 4;
        /** Skybox coordinates mode */
        static readonly TEXTURE_SKYBOX_MODE = 5;
        /** Inverse Cubic coordinates mode */
        static readonly TEXTURE_INVCUBIC_MODE = 6;
        /** Equirectangular coordinates mode */
        static readonly TEXTURE_EQUIRECTANGULAR_MODE = 7;
        /** Equirectangular Fixed coordinates mode */
        static readonly TEXTURE_FIXED_EQUIRECTANGULAR_MODE = 8;
        /** Equirectangular Fixed Mirrored coordinates mode */
        static readonly TEXTURE_FIXED_EQUIRECTANGULAR_MIRRORED_MODE = 9;
        /** Defines that texture rescaling will use a floor to find the closer power of 2 size */
        static readonly SCALEMODE_FLOOR = 1;
        /** Defines that texture rescaling will look for the nearest power of 2 size */
        static readonly SCALEMODE_NEAREST = 2;
        /** Defines that texture rescaling will use a ceil to find the closer power of 2 size */
        static readonly SCALEMODE_CEILING = 3;
        /**
         * Returns the current npm package of the sdk
         */
        static get NpmPackage(): string;
        /**
         * Returns the current version of the framework
         */
        static get Version(): string;
        /** Gets the list of created engines */
        static get Instances(): Engine[];
        /**
         * Gets the latest created engine
         */
        static get LastCreatedEngine(): Nullable<Engine>;
        /**
         * Gets the latest created scene
         */
        static get LastCreatedScene(): Nullable<Scene>;
        /** @internal */
        /**
         * Engine abstraction for loading and creating an image bitmap from a given source string.
         * @param imageSource source to load the image from.
         * @param options An object that sets options for the image's extraction.
         * @returns ImageBitmap.
         */
        _createImageBitmapFromSource(imageSource: string, options?: ImageBitmapOptions): Promise<ImageBitmap>;
        /**
         * Engine abstraction for createImageBitmap
         * @param image source for image
         * @param options An object that sets options for the image's extraction.
         * @returns ImageBitmap
         */
        createImageBitmap(image: ImageBitmapSource, options?: ImageBitmapOptions): Promise<ImageBitmap>;
        /**
         * Resize an image and returns the image data as an uint8array
         * @param image image to resize
         * @param bufferWidth destination buffer width
         * @param bufferHeight destination buffer height
         * @returns an uint8array containing RGBA values of bufferWidth * bufferHeight size
         */
        resizeImageBitmap(image: HTMLImageElement | ImageBitmap, bufferWidth: number, bufferHeight: number): Uint8Array;
        /**
         * Will flag all materials in all scenes in all engines as dirty to trigger new shader compilation
         * @param flag defines which part of the materials must be marked as dirty
         * @param predicate defines a predicate used to filter which materials should be affected
         */
        static MarkAllMaterialsAsDirty(flag: number, predicate?: (mat: Material) => boolean): void;
        /**
         * Method called to create the default loading screen.
         * This can be overridden in your own app.
         * @param canvas The rendering canvas element
         * @returns The loading screen
         */
        static DefaultLoadingScreenFactory(canvas: HTMLCanvasElement): ILoadingScreen;
        /**
         * Method called to create the default rescale post process on each engine.
         */
        static _RescalePostProcessFactory: Nullable<(engine: Engine) => PostProcess>;
        /**
         * Gets or sets a boolean to enable/disable IndexedDB support and avoid XHR on .manifest
         **/
        enableOfflineSupport: boolean;
        /**
         * Gets or sets a boolean to enable/disable checking manifest if IndexedDB support is enabled (js will always consider the database is up to date)
         **/
        disableManifestCheck: boolean;
        /**
         * Gets or sets a boolean to enable/disable the context menu (right-click) from appearing on the main canvas
         */
        disableContextMenu: boolean;
        /**
         * Gets the list of created scenes
         */
        scenes: Scene[];
        /** @internal */
        _virtualScenes: Scene[];
        /**
         * Event raised when a new scene is created
         */
        onNewSceneAddedObservable: Observable<Scene>;
        /**
         * Gets the list of created postprocesses
         */
        postProcesses: PostProcess[];
        /**
         * Gets a boolean indicating if the pointer is currently locked
         */
        isPointerLock: boolean;
        /**
         * Observable event triggered each time the rendering canvas is resized
         */
        onResizeObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas loses focus
         */
        onCanvasBlurObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas gains focus
         */
        onCanvasFocusObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas receives pointerout event
         */
        onCanvasPointerOutObservable: Observable<PointerEvent>;
        /**
         * Observable raised when the engine begins a new frame
         */
        onBeginFrameObservable: Observable<Engine>;
        /**
         * If set, will be used to request the next animation frame for the render loop
         */
        customAnimationFrameRequester: Nullable<ICustomAnimationFrameRequester>;
        /**
         * Observable raised when the engine ends the current frame
         */
        onEndFrameObservable: Observable<Engine>;
        /**
         * Observable raised when the engine is about to compile a shader
         */
        onBeforeShaderCompilationObservable: Observable<Engine>;
        /**
         * Observable raised when the engine has just compiled a shader
         */
        onAfterShaderCompilationObservable: Observable<Engine>;
        /**
         * Gets the audio engine
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
         * @ignorenaming
         */
        static audioEngine: Nullable<IAudioEngine>;
        /**
         * Default AudioEngine factory responsible of creating the Audio Engine.
         * By default, this will create a BabylonJS Audio Engine if the workload has been embedded.
         */
        static AudioEngineFactory: (hostElement: Nullable<HTMLElement>, audioContext: Nullable<AudioContext>, audioDestination: Nullable<AudioDestinationNode | MediaStreamAudioDestinationNode>) => IAudioEngine;
        /**
         * Default offline support factory responsible of creating a tool used to store data locally.
         * By default, this will create a Database object if the workload has been embedded.
         */
        static OfflineProviderFactory: (urlToScene: string, callbackManifestChecked: (checked: boolean) => any, disableManifestCheck: boolean) => IOfflineProvider;
        private _loadingScreen;
        private _pointerLockRequested;
        private _rescalePostProcess;
        protected _deterministicLockstep: boolean;
        protected _lockstepMaxSteps: number;
        protected _timeStep: number;
        protected get _supportsHardwareTextureRescaling(): boolean;
        private _fps;
        private _deltaTime;
        /** @internal */
        _drawCalls: PerfCounter;
        /** Gets or sets the tab index to set to the rendering canvas. 1 is the minimum value to set to be able to capture keyboard events */
        canvasTabIndex: number;
        /**
         * Turn this value on if you want to pause FPS computation when in background
         */
        disablePerformanceMonitorInBackground: boolean;
        private _performanceMonitor;
        /**
         * Gets the performance monitor attached to this engine
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimize_your_scene#engineinstrumentation
         */
        get performanceMonitor(): PerformanceMonitor;
        private _onFocus;
        private _onBlur;
        private _onCanvasPointerOut;
        private _onCanvasBlur;
        private _onCanvasFocus;
        private _onCanvasContextMenu;
        private _onFullscreenChange;
        private _onPointerLockChange;
        protected _compatibilityMode: boolean;
        /**
         * (WebGPU only) True (default) to be in compatibility mode, meaning rendering all existing scenes without artifacts (same rendering than WebGL).
         * Setting the property to false will improve performances but may not work in some scenes if some precautions are not taken.
         * See https://doc.babylonjs.com/setup/support/webGPU/webGPUOptimization/webGPUNonCompatibilityMode for more details
         */
        get compatibilityMode(): boolean;
        set compatibilityMode(mode: boolean);
        /**
         * Gets the HTML element used to attach event listeners
         * @returns a HTML element
         */
        getInputElement(): Nullable<HTMLElement>;
        /**
         * Creates a new engine
         * @param canvasOrContext defines the canvas or WebGL context to use for rendering. If you provide a WebGL context, Babylon.js will not hook events on the canvas (like pointers, keyboards, etc...) so no event observables will be available. This is mostly used when Babylon.js is used as a plugin on a system which already used the WebGL context
         * @param antialias defines enable antialiasing (default: false)
         * @param options defines further options to be sent to the getContext() function
         * @param adaptToDeviceRatio defines whether to adapt to the device's viewport characteristics (default: false)
         */
        constructor(canvasOrContext: Nullable<HTMLCanvasElement | OffscreenCanvas | WebGLRenderingContext | WebGL2RenderingContext>, antialias?: boolean, options?: EngineOptions, adaptToDeviceRatio?: boolean);
        protected _initGLContext(): void;
        /**
         * Shared initialization across engines types.
         * @param canvas The canvas associated with this instance of the engine.
         */
        protected _sharedInit(canvas: HTMLCanvasElement): void;
        /** @internal */
        _verifyPointerLock(): void;
        /**
         * Gets current aspect ratio
         * @param viewportOwner defines the camera to use to get the aspect ratio
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the aspect ratio
         */
        getAspectRatio(viewportOwner: IViewportOwnerLike, useScreen?: boolean): number;
        /**
         * Gets current screen aspect ratio
         * @returns a number defining the aspect ratio
         */
        getScreenAspectRatio(): number;
        /**
         * Gets the client rect of the HTML canvas attached with the current webGL context
         * @returns a client rectangle
         */
        getRenderingCanvasClientRect(): Nullable<ClientRect>;
        /**
         * Gets the client rect of the HTML element used for events
         * @returns a client rectangle
         */
        getInputElementClientRect(): Nullable<ClientRect>;
        /**
         * Gets a boolean indicating that the engine is running in deterministic lock step mode
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         * @returns true if engine is in deterministic lock step mode
         */
        isDeterministicLockStep(): boolean;
        /**
         * Gets the max steps when engine is running in deterministic lock step
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         * @returns the max steps
         */
        getLockstepMaxSteps(): number;
        /**
         * Returns the time in ms between steps when using deterministic lock step.
         * @returns time step in (ms)
         */
        getTimeStep(): number;
        /**
         * Force the mipmap generation for the given render target texture
         * @param texture defines the render target texture to use
         * @param unbind defines whether or not to unbind the texture after generation. Defaults to true.
         */
        generateMipMapsForCubemap(texture: InternalTexture, unbind?: boolean): void;
        /** States */
        /**
         * Gets a boolean indicating if depth writing is enabled
         * @returns the current depth writing state
         */
        getDepthWrite(): boolean;
        /**
         * Enable or disable depth writing
         * @param enable defines the state to set
         */
        setDepthWrite(enable: boolean): void;
        /**
         * Gets a boolean indicating if stencil buffer is enabled
         * @returns the current stencil buffer state
         */
        getStencilBuffer(): boolean;
        /**
         * Enable or disable the stencil buffer
         * @param enable defines if the stencil buffer must be enabled or disabled
         */
        setStencilBuffer(enable: boolean): void;
        /**
         * Gets the current stencil mask
         * @returns a number defining the new stencil mask to use
         */
        getStencilMask(): number;
        /**
         * Sets the current stencil mask
         * @param mask defines the new stencil mask to use
         */
        setStencilMask(mask: number): void;
        /**
         * Gets the current stencil function
         * @returns a number defining the stencil function to use
         */
        getStencilFunction(): number;
        /**
         * Gets the current stencil reference value
         * @returns a number defining the stencil reference value to use
         */
        getStencilFunctionReference(): number;
        /**
         * Gets the current stencil mask
         * @returns a number defining the stencil mask to use
         */
        getStencilFunctionMask(): number;
        /**
         * Sets the current stencil function
         * @param stencilFunc defines the new stencil function to use
         */
        setStencilFunction(stencilFunc: number): void;
        /**
         * Sets the current stencil reference
         * @param reference defines the new stencil reference to use
         */
        setStencilFunctionReference(reference: number): void;
        /**
         * Sets the current stencil mask
         * @param mask defines the new stencil mask to use
         */
        setStencilFunctionMask(mask: number): void;
        /**
         * Gets the current stencil operation when stencil fails
         * @returns a number defining stencil operation to use when stencil fails
         */
        getStencilOperationFail(): number;
        /**
         * Gets the current stencil operation when depth fails
         * @returns a number defining stencil operation to use when depth fails
         */
        getStencilOperationDepthFail(): number;
        /**
         * Gets the current stencil operation when stencil passes
         * @returns a number defining stencil operation to use when stencil passes
         */
        getStencilOperationPass(): number;
        /**
         * Sets the stencil operation to use when stencil fails
         * @param operation defines the stencil operation to use when stencil fails
         */
        setStencilOperationFail(operation: number): void;
        /**
         * Sets the stencil operation to use when depth fails
         * @param operation defines the stencil operation to use when depth fails
         */
        setStencilOperationDepthFail(operation: number): void;
        /**
         * Sets the stencil operation to use when stencil passes
         * @param operation defines the stencil operation to use when stencil passes
         */
        setStencilOperationPass(operation: number): void;
        /**
         * Sets a boolean indicating if the dithering state is enabled or disabled
         * @param value defines the dithering state
         */
        setDitheringState(value: boolean): void;
        /**
         * Sets a boolean indicating if the rasterizer state is enabled or disabled
         * @param value defines the rasterizer state
         */
        setRasterizerState(value: boolean): void;
        /**
         * Gets the current depth function
         * @returns a number defining the depth function
         */
        getDepthFunction(): Nullable<number>;
        /**
         * Sets the current depth function
         * @param depthFunc defines the function to use
         */
        setDepthFunction(depthFunc: number): void;
        /**
         * Sets the current depth function to GREATER
         */
        setDepthFunctionToGreater(): void;
        /**
         * Sets the current depth function to GEQUAL
         */
        setDepthFunctionToGreaterOrEqual(): void;
        /**
         * Sets the current depth function to LESS
         */
        setDepthFunctionToLess(): void;
        /**
         * Sets the current depth function to LEQUAL
         */
        setDepthFunctionToLessOrEqual(): void;
        private _cachedStencilBuffer;
        private _cachedStencilFunction;
        private _cachedStencilMask;
        private _cachedStencilOperationPass;
        private _cachedStencilOperationFail;
        private _cachedStencilOperationDepthFail;
        private _cachedStencilReference;
        /**
         * Caches the state of the stencil buffer
         */
        cacheStencilState(): void;
        /**
         * Restores the state of the stencil buffer
         */
        restoreStencilState(): void;
        /**
         * Directly set the WebGL Viewport
         * @param x defines the x coordinate of the viewport (in screen space)
         * @param y defines the y coordinate of the viewport (in screen space)
         * @param width defines the width of the viewport (in screen space)
         * @param height defines the height of the viewport (in screen space)
         * @returns the current viewport Object (if any) that is being replaced by this call. You can restore this viewport later on to go back to the original state
         */
        setDirectViewport(x: number, y: number, width: number, height: number): Nullable<IViewportLike>;
        /**
         * Executes a scissor clear (ie. a clear on a specific portion of the screen)
         * @param x defines the x-coordinate of the bottom left corner of the clear rectangle
         * @param y defines the y-coordinate of the corner of the clear rectangle
         * @param width defines the width of the clear rectangle
         * @param height defines the height of the clear rectangle
         * @param clearColor defines the clear color
         */
        scissorClear(x: number, y: number, width: number, height: number, clearColor: IColor4Like): void;
        /**
         * Enable scissor test on a specific rectangle (ie. render will only be executed on a specific portion of the screen)
         * @param x defines the x-coordinate of the bottom left corner of the clear rectangle
         * @param y defines the y-coordinate of the corner of the clear rectangle
         * @param width defines the width of the clear rectangle
         * @param height defines the height of the clear rectangle
         */
        enableScissor(x: number, y: number, width: number, height: number): void;
        /**
         * Disable previously set scissor test rectangle
         */
        disableScissor(): void;
        /**
         * @internal
         */
        _reportDrawCall(numDrawCalls?: number): void;
        _loadFileAsync(url: string, offlineProvider?: IOfflineProvider, useArrayBuffer?: false): Promise<string>;
        _loadFileAsync(url: string, offlineProvider?: IOfflineProvider, useArrayBuffer?: true): Promise<ArrayBuffer>;
        /**
         * Gets the source code of the vertex shader associated with a specific webGL program
         * @param program defines the program to use
         * @returns a string containing the source code of the vertex shader associated with the program
         */
        getVertexShaderSource(program: WebGLProgram): Nullable<string>;
        /**
         * Gets the source code of the fragment shader associated with a specific webGL program
         * @param program defines the program to use
         * @returns a string containing the source code of the fragment shader associated with the program
         */
        getFragmentShaderSource(program: WebGLProgram): Nullable<string>;
        /**
         * Sets a depth stencil texture from a render target to the according uniform.
         * @param channel The texture channel
         * @param uniform The uniform to set
         * @param texture The render target texture containing the depth stencil texture to apply
         * @param name The texture name
         */
        setDepthStencilTexture(channel: number, uniform: Nullable<WebGLUniformLocation>, texture: Nullable<RenderTargetTexture>, name?: string): void;
        /**
         * Sets a texture to the webGL context from a postprocess
         * @param channel defines the channel to use
         * @param postProcess defines the source postprocess
         * @param name name of the channel
         */
        setTextureFromPostProcess(channel: number, postProcess: Nullable<PostProcess>, name: string): void;
        /**
         * Binds the output of the passed in post process to the texture channel specified
         * @param channel The channel the texture should be bound to
         * @param postProcess The post process which's output should be bound
         * @param name name of the channel
         */
        setTextureFromPostProcessOutput(channel: number, postProcess: Nullable<PostProcess>, name: string): void;
        /**
         * sets the object from which width and height will be taken from when getting render width and height
         * Will fallback to the gl object
         * @param dimensions the framebuffer width and height that will be used.
         */
        set framebufferDimensionsObject(dimensions: Nullable<{
            framebufferWidth: number;
            framebufferHeight: number;
        }>);
        protected _rebuildBuffers(): void;
        protected _rebuildTextures(): void;
        /** @internal */
        _renderFrame(): void;
        protected _cancelFrame(): void;
        _renderLoop(): void;
        /** @internal */
        _renderViews(): boolean;
        /**
         * Toggle full screen mode
         * @param requestPointerLock defines if a pointer lock should be requested from the user
         */
        switchFullscreen(requestPointerLock: boolean): void;
        /**
         * Enters full screen mode
         * @param requestPointerLock defines if a pointer lock should be requested from the user
         */
        enterFullscreen(requestPointerLock: boolean): void;
        /**
         * Exits full screen mode
         */
        exitFullscreen(): void;
        /**
         * Enters Pointerlock mode
         */
        enterPointerlock(): void;
        /**
         * Exits Pointerlock mode
         */
        exitPointerlock(): void;
        /**
         * Begin a new frame
         */
        beginFrame(): void;
        /**
         * End the current frame
         */
        endFrame(): void;
        /**
         * Force a specific size of the canvas
         * @param width defines the new canvas' width
         * @param height defines the new canvas' height
         * @param forceSetSize true to force setting the sizes of the underlying canvas
         * @returns true if the size was changed
         */
        setSize(width: number, height: number, forceSetSize?: boolean): boolean;
        _deletePipelineContext(pipelineContext: IPipelineContext): void;
        createShaderProgram(pipelineContext: IPipelineContext, vertexCode: string, fragmentCode: string, defines: Nullable<string>, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        protected _createShaderProgram(pipelineContext: WebGLPipelineContext, vertexShader: WebGLShader, fragmentShader: WebGLShader, context: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        /**
         * @internal
         */
        _releaseTexture(texture: InternalTexture): void;
        /**
         * @internal
         */
        _releaseRenderTargetWrapper(rtWrapper: RenderTargetWrapper): void;
        protected static _RenderPassIdCounter: number;
        /**
         * Gets or sets the current render pass id
         */
        currentRenderPassId: number;
        private _renderPassNames;
        /**
         * Gets the names of the render passes that are currently created
         * @returns list of the render pass names
         */
        getRenderPassNames(): string[];
        /**
         * Gets the name of the current render pass
         * @returns name of the current render pass
         */
        getCurrentRenderPassName(): string;
        /**
         * Creates a render pass id
         * @param name Name of the render pass (for debug purpose only)
         * @returns the id of the new render pass
         */
        createRenderPassId(name?: string): number;
        /**
         * Releases a render pass id
         * @param id id of the render pass to release
         */
        releaseRenderPassId(id: number): void;
        /**
         * @internal
         * Rescales a texture
         * @param source input texture
         * @param destination destination texture
         * @param scene scene to use to render the resize
         * @param internalFormat format to use when resizing
         * @param onComplete callback to be called when resize has completed
         */
        _rescaleTexture(source: InternalTexture, destination: InternalTexture, scene: Nullable<any>, internalFormat: number, onComplete: () => void): void;
        /**
         * Gets the current framerate
         * @returns a number representing the framerate
         */
        getFps(): number;
        /**
         * Gets the time spent between current and previous frame
         * @returns a number representing the delta time in ms
         */
        getDeltaTime(): number;
        private _measureFps;
        /**
         * Wraps an external web gl texture in a Babylon texture.
         * @param texture defines the external texture
         * @param hasMipMaps defines whether the external texture has mip maps (default: false)
         * @param samplingMode defines the sampling mode for the external texture (default: Constants.TEXTURE_TRILINEAR_SAMPLINGMODE)
         * @param width defines the width for the external texture (default: 0)
         * @param height defines the height for the external texture (default: 0)
         * @returns the babylon internal texture
         */
        wrapWebGLTexture(texture: WebGLTexture, hasMipMaps?: boolean, samplingMode?: number, width?: number, height?: number): InternalTexture;
        /**
         * @internal
         */
        _uploadImageToTexture(texture: InternalTexture, image: HTMLImageElement | ImageBitmap, faceIndex?: number, lod?: number): void;
        /**
         * Updates a depth texture Comparison Mode and Function.
         * If the comparison Function is equal to 0, the mode will be set to none.
         * Otherwise, this only works in webgl 2 and requires a shadow sampler in the shader.
         * @param texture The texture to set the comparison function for
         * @param comparisonFunction The comparison function to set, 0 if no comparison required
         */
        updateTextureComparisonFunction(texture: InternalTexture, comparisonFunction: number): void;
        /**
         * Creates a webGL buffer to use with instantiation
         * @param capacity defines the size of the buffer
         * @returns the webGL buffer
         */
        createInstancesBuffer(capacity: number): DataBuffer;
        /**
         * Delete a webGL buffer used with instantiation
         * @param buffer defines the webGL buffer to delete
         */
        deleteInstancesBuffer(buffer: WebGLBuffer): void;
        private _clientWaitAsync;
        /**
         * @internal
         */
        _readPixelsAsync(x: number, y: number, w: number, h: number, format: number, type: number, outputBuffer: ArrayBufferView): Promise<ArrayBufferView> | null;
        dispose(): void;
        private _disableTouchAction;
        /**
         * Display the loading screen
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/customLoadingScreen
         */
        displayLoadingUI(): void;
        /**
         * Hide the loading screen
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/customLoadingScreen
         */
        hideLoadingUI(): void;
        /**
         * Gets the current loading screen object
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/customLoadingScreen
         */
        get loadingScreen(): ILoadingScreen;
        /**
         * Sets the current loading screen object
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/customLoadingScreen
         */
        set loadingScreen(loadingScreen: ILoadingScreen);
        /**
         * Sets the current loading screen text
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/customLoadingScreen
         */
        set loadingUIText(text: string);
        /**
         * Sets the current loading screen background color
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/customLoadingScreen
         */
        set loadingUIBackgroundColor(color: string);
        /**
         * creates and returns a new video element
         * @param constraints video constraints
         * @returns video element
         */
        createVideoElement(constraints: MediaTrackConstraints): any;
        /** Pointerlock and fullscreen */
        /**
         * Ask the browser to promote the current element to pointerlock mode
         * @param element defines the DOM element to promote
         */
        static _RequestPointerlock(element: HTMLElement): void;
        /**
         * Asks the browser to exit pointerlock mode
         */
        static _ExitPointerlock(): void;
        /**
         * Ask the browser to promote the current element to fullscreen rendering mode
         * @param element defines the DOM element to promote
         */
        static _RequestFullscreen(element: HTMLElement): void;
        /**
         * Asks the browser to exit fullscreen mode
         */
        static _ExitFullscreen(): void;
        /**
         * Get Font size information
         * @param font font name
         * @returns an object containing ascent, height and descent
         */
        getFontOffset(font: string): {
            ascent: number;
            height: number;
            descent: number;
        };
    }


    /**
     * Interface used to describe the capabilities of the engine relatively to the current browser
     */
    export interface EngineCapabilities {
        /** Maximum textures units per fragment shader */
        maxTexturesImageUnits: number;
        /** Maximum texture units per vertex shader */
        maxVertexTextureImageUnits: number;
        /** Maximum textures units in the entire pipeline */
        maxCombinedTexturesImageUnits: number;
        /** Maximum texture size */
        maxTextureSize: number;
        /** Maximum texture samples */
        maxSamples?: number;
        /** Maximum cube texture size */
        maxCubemapTextureSize: number;
        /** Maximum render texture size */
        maxRenderTextureSize: number;
        /** Maximum number of vertex attributes */
        maxVertexAttribs: number;
        /** Maximum number of varyings */
        maxVaryingVectors: number;
        /** Maximum number of uniforms per vertex shader */
        maxVertexUniformVectors: number;
        /** Maximum number of uniforms per fragment shader */
        maxFragmentUniformVectors: number;
        /** Defines if standard derivatives (dx/dy) are supported */
        standardDerivatives: boolean;
        /** Defines if s3tc texture compression is supported */
        s3tc?: WEBGL_compressed_texture_s3tc;
        /** Defines if s3tc sRGB texture compression is supported */
        s3tc_srgb?: WEBGL_compressed_texture_s3tc_srgb;
        /** Defines if pvrtc texture compression is supported */
        pvrtc: any;
        /** Defines if etc1 texture compression is supported */
        etc1: any;
        /** Defines if etc2 texture compression is supported */
        etc2: any;
        /** Defines if astc texture compression is supported */
        astc: any;
        /** Defines if bptc texture compression is supported */
        bptc: any;
        /** Defines if float textures are supported */
        textureFloat: boolean;
        /** Defines if vertex array objects are supported */
        vertexArrayObject: boolean;
        /** Gets the webgl extension for anisotropic filtering (null if not supported) */
        textureAnisotropicFilterExtension?: EXT_texture_filter_anisotropic;
        /** Gets the maximum level of anisotropy supported */
        maxAnisotropy: number;
        /** Defines if instancing is supported */
        instancedArrays: boolean;
        /** Defines if 32 bits indices are supported */
        uintIndices: boolean;
        /** Defines if high precision shaders are supported */
        highPrecisionShaderSupported: boolean;
        /** Defines if depth reading in the fragment shader is supported */
        fragmentDepthSupported: boolean;
        /** Defines if float texture linear filtering is supported*/
        textureFloatLinearFiltering: boolean;
        /** Defines if rendering to float textures is supported */
        textureFloatRender: boolean;
        /** Defines if half float textures are supported*/
        textureHalfFloat: boolean;
        /** Defines if half float texture linear filtering is supported*/
        textureHalfFloatLinearFiltering: boolean;
        /** Defines if rendering to half float textures is supported */
        textureHalfFloatRender: boolean;
        /** Defines if textureLOD shader command is supported */
        textureLOD: boolean;
        /** Defines if texelFetch shader command is supported */
        texelFetch: boolean;
        /** Defines if draw buffers extension is supported */
        drawBuffersExtension: boolean;
        /** Defines if depth textures are supported */
        depthTextureExtension: boolean;
        /** Defines if float color buffer are supported */
        colorBufferFloat: boolean;
        /** Defines if half float color buffer are supported */
        colorBufferHalfFloat?: boolean;
        /** Gets disjoint timer query extension (null if not supported) */
        timerQuery?: EXT_disjoint_timer_query;
        /** Defines if timestamp can be used with timer query */
        canUseTimestampForTimerQuery: boolean;
        /** Defines if occlusion queries are supported by the engine */
        supportOcclusionQuery: boolean;
        /** Defines if multiview is supported (https://www.khronos.org/registry/webgl/extensions/WEBGL_multiview/) */
        multiview?: any;
        /** Defines if oculus multiview is supported (https://developer.oculus.com/documentation/oculus-browser/latest/concepts/browser-multiview/) */
        oculusMultiview?: any;
        /** Function used to let the system compiles shaders in background */
        parallelShaderCompile?: {
            COMPLETION_STATUS_KHR: number;
        };
        /** Max number of texture samples for MSAA */
        maxMSAASamples: number;
        /** Defines if the blend min max extension is supported */
        blendMinMax: boolean;
        /** In some iOS + WebGL1, gl_InstanceID (and gl_InstanceIDEXT) is undefined even if instancedArrays is true. So don't use gl_InstanceID in those cases */
        canUseGLInstanceID: boolean;
        /** Defines if gl_vertexID is available */
        canUseGLVertexID: boolean;
        /** Defines if compute shaders are supported by the engine */
        supportComputeShaders: boolean;
        /** Defines if sRGB texture formats are supported */
        supportSRGBBuffers: boolean;
        /** Defines if transform feedbacks are supported */
        supportTransformFeedbacks: boolean;
        /** Defines if texture max level are supported */
        textureMaxLevel: boolean;
        /** Defines the maximum layer count for a 2D Texture array. */
        texture2DArrayMaxLayerCount: number;
        /** Defines if the morph target texture is supported. */
        disableMorphTargetTexture: boolean;
        /** Defines if float textures like r32f, rg32f or rgba32f support being used as a resolve target */
        supportFloatTexturesResolve: boolean;
        /** Defines if RG11B10UFloat texture format is color renderable */
        rg11b10ufColorRenderable: boolean;
    }


    /**
     * Helper class to create the best engine depending on the current hardware
     */
    export class EngineFactory {
        /**
         * Creates an engine based on the capabilities of the underlying hardware
         * @param canvas Defines the canvas to use to display the result
         * @param options Defines the options passed to the engine to create the context dependencies
         * @returns a promise that resolves with the created engine
         */
        static CreateAsync(canvas: HTMLCanvasElement, options: any): Promise<Engine>;
    }


    /** @internal */
    export interface EngineFeatures {
        /** Force using Bitmap when Bitmap or HTMLImageElement can be used */
        forceBitmapOverHTMLImageElement: boolean;
        /** Indicates that the engine support rendering to as well as copying to lod float textures */
        supportRenderAndCopyToLodForFloatTextures: boolean;
        /** Indicates that the engine support handling depth/stencil textures */
        supportDepthStencilTexture: boolean;
        /** Indicates that the engine support shadow samplers */
        supportShadowSamplers: boolean;
        /** Indicates to check the matrix bytes per bytes to know if it has changed or not. If false, only the updateFlag of the matrix is checked */
        uniformBufferHardCheckMatrix: boolean;
        /** Indicates that prefiltered mipmaps can be generated in some processes (for eg when loading an HDR cube texture) */
        allowTexturePrefiltering: boolean;
        /** Indicates to track the usage of ubos and to create new ones as necessary during a frame duration */
        trackUbosInFrame: boolean;
        /** Indicates that the current content of a ubo should be compared to the content of the corresponding GPU buffer and the GPU buffer updated only if different. Requires trackUbosInFrame to be true */
        checkUbosContentBeforeUpload: boolean;
        /** Indicates that the Cascaded Shadow Map technic is supported */
        supportCSM: boolean;
        /** Indicates that the textures transcoded by the basis transcoder must have power of 2 width and height */
        basisNeedsPOT: boolean;
        /** Indicates that the engine supports 3D textures */
        support3DTextures: boolean;
        /** Indicates that constants need a type suffix in shaders (used by realtime filtering...) */
        needTypeSuffixInShaderConstants: boolean;
        /** Indicates that MSAA is supported */
        supportMSAA: boolean;
        /** Indicates that SSAO2 is supported */
        supportSSAO2: boolean;
        /** Indicates that some additional texture formats are supported (like TEXTUREFORMAT_R for eg) */
        supportExtendedTextureFormats: boolean;
        /** Indicates that the switch/case construct is supported in shaders */
        supportSwitchCaseInShader: boolean;
        /** Indicates that synchronous texture reading is supported */
        supportSyncTextureRead: boolean;
        /** Indicates that y should be inverted when dealing with bitmaps (notably in environment tools) */
        needsInvertingBitmap: boolean;
        /** Indicates that the engine should cache the bound UBO */
        useUBOBindingCache: boolean;
        /** Indicates that the inliner should be run over every shader code */
        needShaderCodeInlining: boolean;
        /** Indicates that even if we don't have to update the properties of a uniform buffer (because of some optimzations in the material) we still need to bind the uniform buffer themselves */
        needToAlwaysBindUniformBuffers: boolean;
        /**  Indicates that the engine supports render passes */
        supportRenderPasses: boolean;
        /**  Indicates that the engine supports sprite instancing */
        supportSpriteInstancing: boolean;
        /** Indicates that the stride of a vertex buffer must always be a multiple of 4 bytes */
        forceVertexBufferStrideMultiple4Bytes: boolean;
        /** @internal */
        _collectUbosUpdatedInFrame: boolean;
    }


    /**
     * The engine store class is responsible to hold all the instances of Engine and Scene created
     * during the life time of the application.
     */
    export class EngineStore {
        /** Gets the list of created engines */
        static Instances: Engine[];
        /**
         * Notifies when an engine was disposed.
         * Mainly used for static/cache cleanup
         */
        static OnEnginesDisposedObservable: Observable<Engine>;
        /** @internal */
        static _LastCreatedScene: Nullable<Scene>;
        /**
         * Gets the latest created engine
         */
        static get LastCreatedEngine(): Nullable<Engine>;
        /**
         * Gets the latest created scene
         */
        static get LastCreatedScene(): Nullable<Scene>;
        /**
         * Gets or sets a global variable indicating if fallback texture must be used when a texture cannot be loaded
         * @ignorenaming
         */
        static UseFallbackTexture: boolean;
        /**
         * Texture content used if a texture cannot loaded
         * @ignorenaming
         */
        static FallbackTexture: string;
    }


        interface ThinEngine {
            /**
             * Sets alpha constants used by some alpha blending modes
             * @param r defines the red component
             * @param g defines the green component
             * @param b defines the blue component
             * @param a defines the alpha component
             */
            setAlphaConstants(r: number, g: number, b: number, a: number): void;
            /**
             * Sets the current alpha mode
             * @param mode defines the mode to use (one of the Engine.ALPHA_XXX)
             * @param noDepthWriteChange defines if depth writing state should remains unchanged (false by default)
             * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/advanced/transparent_rendering
             */
            setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
            /**
             * Gets the current alpha mode
             * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/advanced/transparent_rendering
             * @returns the current alpha mode
             */
            getAlphaMode(): number;
            /**
             * Sets the current alpha equation
             * @param equation defines the equation to use (one of the Engine.ALPHA_EQUATION_XXX)
             */
            setAlphaEquation(equation: number): void;
            /**
             * Gets the current alpha equation.
             * @returns the current alpha equation
             */
            getAlphaEquation(): number;
        }


    /**
     * Type used to locate a resource in a compute shader.
     * TODO: remove this when browsers support reflection for wgsl shaders
     */
    export type ComputeBindingLocation = {
        group: number;
        binding: number;
    };
    /**
     * Type used to lookup a resource and retrieve its binding location
     * TODO: remove this when browsers support reflection for wgsl shaders
     */
    export type ComputeBindingMapping = {
        [key: string]: ComputeBindingLocation;
    };
    /** @internal */
    export enum ComputeBindingType {
        Texture = 0,
        StorageTexture = 1,
        UniformBuffer = 2,
        StorageBuffer = 3,
        TextureWithoutSampler = 4,
        Sampler = 5,
        ExternalTexture = 6
    }
    /** @internal */
    export type ComputeBindingList = {
        [key: string]: {
            type: ComputeBindingType;
            object: any;
            indexInGroupEntries?: number;
        };
    };
        interface ThinEngine {
            /**
             * Creates a new compute effect
             * @param baseName Name of the effect
             * @param options Options used to create the effect
             * @returns The new compute effect
             */
            createComputeEffect(baseName: any, options: IComputeEffectCreationOptions): ComputeEffect;
            /**
             * Creates a new compute pipeline context
             * @returns the new pipeline
             */
            createComputePipelineContext(): IComputePipelineContext;
            /**
             * Creates a new compute context
             * @returns the new context
             */
            createComputeContext(): IComputeContext | undefined;
            /**
             * Dispatches a compute shader
             * @param effect The compute effect
             * @param context The compute context
             * @param bindings The list of resources to bind to the shader
             * @param x The number of workgroups to execute on the X dimension
             * @param y The number of workgroups to execute on the Y dimension
             * @param z The number of workgroups to execute on the Z dimension
             * @param bindingsMapping list of bindings mapping (key is property name, value is binding location)
             * @param gpuPerfCounter GPU time computed for the compute shader will be assigned to this object
             */
            computeDispatch(effect: ComputeEffect, context: IComputeContext, bindings: ComputeBindingList, x: number, y?: number, z?: number, bindingsMapping?: ComputeBindingMapping, gpuPerfCounter?: WebGPUPerfCounter): void;
            /**
             * Gets a boolean indicating if all created compute effects are ready
             * @returns true if all effects are ready
             */
            areAllComputeEffectsReady(): boolean;
            /**
             * Forces the engine to release all cached compute effects. This means that next effect compilation will have to be done completely even if a similar effect was already compiled
             */
            releaseComputeEffects(): void;
            /** @internal */
            _prepareComputePipelineContext(pipelineContext: IComputePipelineContext, computeSourceCode: string, rawComputeSourceCode: string, defines: Nullable<string>, entryPoint: string): void;
            /** @internal */
            _rebuildComputeEffects(): void;
            /** @internal */
            _executeWhenComputeStateIsCompiled(pipelineContext: IComputePipelineContext, action: () => void): void;
            /** @internal */
            _releaseComputeEffect(effect: ComputeEffect): void;
            /** @internal */
            _deleteComputePipelineContext(pipelineContext: IComputePipelineContext): void;
        }


        interface ThinEngine {
            /**
             * Creates a depth stencil cube texture.
             * This is only available in WebGL 2.
             * @param size The size of face edge in the cube texture.
             * @param options The options defining the cube texture.
             * @returns The cube texture
             */
            _createDepthStencilCubeTexture(size: number, options: DepthTextureCreationOptions): InternalTexture;
            /**
             * Creates a cube texture
             * @param rootUrl defines the url where the files to load is located
             * @param scene defines the current scene
             * @param files defines the list of files to load (1 per face)
             * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
             * @param onLoad defines an optional callback raised when the texture is loaded
             * @param onError defines an optional callback raised if there is an issue to load the texture
             * @param format defines the format of the data
             * @param forcedExtension defines the extension to use to pick the right loader
             * @param createPolynomials if a polynomial sphere should be created for the cube texture
             * @param lodScale defines the scale applied to environment texture. This manages the range of LOD level used for IBL according to the roughness
             * @param lodOffset defines the offset applied to environment texture. This manages first LOD level used for IBL according to the roughness
             * @param fallback defines texture to use while falling back when (compressed) texture file not found.
             * @param loaderOptions options to be passed to the loader
             * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
             * @returns the cube texture as an InternalTexture
             */
            createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean | undefined, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any, createPolynomials: boolean, lodScale: number, lodOffset: number, fallback: Nullable<InternalTexture>, loaderOptions: any, useSRGBBuffer: boolean): InternalTexture;
            /**
             * Creates a cube texture
             * @param rootUrl defines the url where the files to load is located
             * @param scene defines the current scene
             * @param files defines the list of files to load (1 per face)
             * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
             * @param onLoad defines an optional callback raised when the texture is loaded
             * @param onError defines an optional callback raised if there is an issue to load the texture
             * @param format defines the format of the data
             * @param forcedExtension defines the extension to use to pick the right loader
             * @returns the cube texture as an InternalTexture
             */
            createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any): InternalTexture;
            /**
             * Creates a cube texture
             * @param rootUrl defines the url where the files to load is located
             * @param scene defines the current scene
             * @param files defines the list of files to load (1 per face)
             * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
             * @param onLoad defines an optional callback raised when the texture is loaded
             * @param onError defines an optional callback raised if there is an issue to load the texture
             * @param format defines the format of the data
             * @param forcedExtension defines the extension to use to pick the right loader
             * @param createPolynomials if a polynomial sphere should be created for the cube texture
             * @param lodScale defines the scale applied to environment texture. This manages the range of LOD level used for IBL according to the roughness
             * @param lodOffset defines the offset applied to environment texture. This manages first LOD level used for IBL according to the roughness
             * @returns the cube texture as an InternalTexture
             */
            createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any, createPolynomials: boolean, lodScale: number, lodOffset: number): InternalTexture;
            /** @internal */
            createCubeTextureBase(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap: boolean, onLoad: Nullable<(data?: any) => void>, onError: Nullable<(message?: string, exception?: any) => void>, format: number | undefined, forcedExtension: any, createPolynomials: boolean, lodScale: number, lodOffset: number, fallback: Nullable<InternalTexture>, beforeLoadCubeDataCallback: Nullable<(texture: InternalTexture, data: ArrayBufferView | ArrayBufferView[]) => void>, imageHandler: Nullable<(texture: InternalTexture, imgs: HTMLImageElement[] | ImageBitmap[]) => void>, useSRGBBuffer: boolean): InternalTexture;
            /** @internal */
            _partialLoadFile(url: string, index: number, loadedFiles: ArrayBuffer[], onfinish: (files: ArrayBuffer[]) => void, onErrorCallBack: Nullable<(message?: string, exception?: any) => void>): void;
            /** @internal */
            _cascadeLoadFiles(scene: Nullable<Scene>, onfinish: (images: ArrayBuffer[]) => void, files: string[], onError: Nullable<(message?: string, exception?: any) => void>): void;
            /** @internal */
            _cascadeLoadImgs(scene: Nullable<Scene>, texture: InternalTexture, onfinish: Nullable<(texture: InternalTexture, images: HTMLImageElement[] | ImageBitmap[]) => void>, files: string[], onError: Nullable<(message?: string, exception?: any) => void>, mimeType?: string): void;
            /** @internal */
            _partialLoadImg(url: string, index: number, loadedImages: HTMLImageElement[] | ImageBitmap[], scene: Nullable<Scene>, texture: InternalTexture, onfinish: Nullable<(texture: InternalTexture, images: HTMLImageElement[] | ImageBitmap[]) => void>, onErrorCallBack: Nullable<(message?: string, exception?: any) => void>, mimeType?: string): void;
            /**
             * @internal
             */
            _setCubeMapTextureParams(texture: InternalTexture, loadMipmap: boolean, maxLevel?: number): void;
        }


        interface ThinEngine {
            /** @internal */
            _debugPushGroup(groupName: string, targetObject?: number): void;
            /** @internal */
            _debugPopGroup(targetObject?: number): void;
            /** @internal */
            _debugInsertMarker(text: string, targetObject?: number): void;
            /** @internal */
            _debugFlushPendingCommands(): void;
        }


        interface ThinEngine {
            /**
             * Update a dynamic index buffer
             * @param indexBuffer defines the target index buffer
             * @param indices defines the data to update
             * @param offset defines the offset in the target index buffer where update should start
             */
            updateDynamicIndexBuffer(indexBuffer: DataBuffer, indices: IndicesArray, offset?: number): void;
            /**
             * Updates a dynamic vertex buffer.
             * @param vertexBuffer the vertex buffer to update
             * @param data the data used to update the vertex buffer
             * @param byteOffset the byte offset of the data
             * @param byteLength the byte length of the data
             */
            updateDynamicVertexBuffer(vertexBuffer: DataBuffer, data: DataArray, byteOffset?: number, byteLength?: number): void;
        }


        interface ThinEngine {
            /**
             * Creates a dynamic texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param generateMipMaps defines if the engine should generate the mip levels
             * @param samplingMode defines the required sampling mode (Texture.NEAREST_SAMPLINGMODE by default)
             * @returns the dynamic texture inside an InternalTexture
             */
            createDynamicTexture(width: number, height: number, generateMipMaps: boolean, samplingMode: number): InternalTexture;
            /**
             * Update the content of a dynamic texture
             * @param texture defines the texture to update
             * @param source defines the source containing the data
             * @param invertY defines if data must be stored with Y axis inverted
             * @param premulAlpha defines if alpha is stored as premultiplied
             * @param format defines the format of the data
             * @param forceBindTexture if the texture should be forced to be bound eg. after a graphics context loss (Default: false)
             * @param allowGPUOptimization true to allow some specific GPU optimizations (subject to engine feature "allowGPUOptimizationsForGUI" being true)
             */
            updateDynamicTexture(texture: Nullable<InternalTexture>, source: ImageSource | ICanvas, invertY?: boolean, premulAlpha?: boolean, format?: number, forceBindTexture?: boolean, allowGPUOptimization?: boolean): void;
        }


        interface ThinEngine {
            /**
             * Creates an external texture
             * @param video video element
             * @returns the external texture, or null if external textures are not supported by the engine
             */
            createExternalTexture(video: HTMLVideoElement): Nullable<ExternalTexture>;
            /**
             * Sets an internal texture to the according uniform.
             * @param name The name of the uniform in the effect
             * @param texture The texture to apply
             */
            setExternalTexture(name: string, texture: Nullable<ExternalTexture>): void;
        }


        interface ThinEngine {
            /**
             * Unbind a list of render target textures from the webGL context
             * This is used only when drawBuffer extension or webGL2 are active
             * @param rtWrapper defines the render target wrapper to unbind
             * @param disableGenerateMipMaps defines a boolean indicating that mipmaps must not be generated
             * @param onBeforeUnbind defines a function which will be called before the effective unbind
             */
            unBindMultiColorAttachmentFramebuffer(rtWrapper: RenderTargetWrapper, disableGenerateMipMaps: boolean, onBeforeUnbind?: () => void): void;
            /**
             * Create a multi render target texture
             * @see https://doc.babylonjs.com/setup/support/webGL2#multiple-render-target
             * @param size defines the size of the texture
             * @param options defines the creation options
             * @param initializeBuffers if set to true, the engine will make an initializing call of drawBuffers
             * @returns a new render target wrapper ready to render textures
             */
            createMultipleRenderTarget(size: TextureSize, options: IMultiRenderTargetOptions, initializeBuffers?: boolean): RenderTargetWrapper;
            /**
             * Update the sample count for a given multiple render target texture
             * @see https://doc.babylonjs.com/setup/support/webGL2#multisample-render-targets
             * @param rtWrapper defines the render target wrapper to update
             * @param samples defines the sample count to set
             * @param initializeBuffers if set to true, the engine will make an initializing call of drawBuffers
             * @returns the effective sample count (could be 0 if multisample render targets are not supported)
             */
            updateMultipleRenderTargetTextureSampleCount(rtWrapper: Nullable<RenderTargetWrapper>, samples: number, initializeBuffers?: boolean): number;
            /**
             * Select a subsets of attachments to draw to.
             * @param attachments gl attachments
             */
            bindAttachments(attachments: number[]): void;
            /**
             * Creates a layout object to draw/clear on specific textures in a MRT
             * @param textureStatus textureStatus[i] indicates if the i-th is active
             * @returns A layout to be fed to the engine, calling `bindAttachments`.
             */
            buildTextureLayout(textureStatus: boolean[]): number[];
            /**
             * Restores the webgl state to only draw on the main color attachment
             * when the frame buffer associated is the canvas frame buffer
             */
            restoreSingleAttachment(): void;
            /**
             * Restores the webgl state to only draw on the main color attachment
             * when the frame buffer associated is not the canvas frame buffer
             */
            restoreSingleAttachmentForRenderTarget(): void;
        }


        interface Engine {
            /**
             * Creates a new multiview render target
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @returns the created multiview render target wrapper
             */
            createMultiviewRenderTargetTexture(width: number, height: number, colorTexture?: WebGLTexture, depthStencilTexture?: WebGLTexture): RenderTargetWrapper;
            /**
             * Binds a multiview render target wrapper to be drawn to
             * @param multiviewTexture render target wrapper to bind
             */
            bindMultiviewFramebuffer(multiviewTexture: RenderTargetWrapper): void;
            /**
             * Binds a Space Warp render target wrapper to be drawn to
             * @param spaceWarpTexture render target wrapper to bind
             */
            bindSpaceWarpFramebuffer(spaceWarpTexture: RenderTargetWrapper): void;
        }
        interface Camera {
            /**
             * @internal
             * For cameras that cannot use multiview images to display directly. (e.g. webVR camera will render to multiview texture, then copy to each eye texture and go from there)
             */
            _useMultiviewToSingleView: boolean;
            /**
             * @internal
             * For cameras that cannot use multiview images to display directly. (e.g. webVR camera will render to multiview texture, then copy to each eye texture and go from there)
             */
            _multiviewTexture: Nullable<RenderTargetTexture>;
            /**
             * @internal
             * For WebXR cameras that are rendering to multiview texture arrays.
             */
            _renderingMultiview: boolean;
            /**
             * @internal
             * ensures the multiview texture of the camera exists and has the specified width/height
             * @param width height to set on the multiview texture
             * @param height width to set on the multiview texture
             */
            _resizeOrCreateMultiviewTexture(width: number, height: number): void;
        }
        interface Scene {
            /** @internal */
            _transformMatrixR: Matrix;
            /** @internal */
            _multiviewSceneUbo: Nullable<UniformBuffer>;
            /** @internal */
            _createMultiviewUbo(): void;
            /** @internal */
            _updateMultiviewUbo(viewR?: Matrix, projectionR?: Matrix): void;
            /** @internal */
            _renderMultiviewToSingleView(camera: Camera): void;
        }


    /** @internal */
    export type OcclusionQuery = WebGLQuery | number;
    /** @internal */
    export class _OcclusionDataStorage {
        /** @internal */
        occlusionInternalRetryCounter: number;
        /** @internal */
        isOcclusionQueryInProgress: boolean;
        /** @internal */
        isOccluded: boolean;
        /** @internal */
        occlusionRetryCount: number;
        /** @internal */
        occlusionType: number;
        /** @internal */
        occlusionQueryAlgorithmType: number;
        /** @internal */
        forceRenderingWhenOccluded: boolean;
    }
        interface Engine {
            /**
             * Create a new webGL query (you must be sure that queries are supported by checking getCaps() function)
             * @returns the new query
             */
            createQuery(): OcclusionQuery;
            /**
             * Delete and release a webGL query
             * @param query defines the query to delete
             * @returns the current engine
             */
            deleteQuery(query: OcclusionQuery): Engine;
            /**
             * Check if a given query has resolved and got its value
             * @param query defines the query to check
             * @returns true if the query got its value
             */
            isQueryResultAvailable(query: OcclusionQuery): boolean;
            /**
             * Gets the value of a given query
             * @param query defines the query to check
             * @returns the value of the query
             */
            getQueryResult(query: OcclusionQuery): number;
            /**
             * Initiates an occlusion query
             * @param algorithmType defines the algorithm to use
             * @param query defines the query to use
             * @returns the current engine
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            beginOcclusionQuery(algorithmType: number, query: OcclusionQuery): boolean;
            /**
             * Ends an occlusion query
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             * @param algorithmType defines the algorithm to use
             * @returns the current engine
             */
            endOcclusionQuery(algorithmType: number): Engine;
            /**
             * Starts a time query (used to measure time spent by the GPU on a specific frame)
             * Please note that only one query can be issued at a time
             * @returns a time token used to track the time span
             */
            startTimeQuery(): Nullable<_TimeToken>;
            /**
             * Ends a time query
             * @param token defines the token used to measure the time span
             * @returns the time spent (in ns)
             */
            endTimeQuery(token: _TimeToken): int;
            /**
             * Get the performance counter associated with the frame time computation
             * @returns the perf counter
             */
            getGPUFrameTimeCounter(): PerfCounter;
            /**
             * Enable or disable the GPU frame time capture
             * @param value True to enable, false to disable
             */
            captureGPUFrameTime(value: boolean): void;
            /** @internal */
            _currentNonTimestampToken: Nullable<_TimeToken>;
            /** @internal */
            _captureGPUFrameTime: boolean;
            /** @internal */
            _gpuFrameTimeToken: Nullable<_TimeToken>;
            /** @internal */
            _gpuFrameTime: PerfCounter;
            /** @internal */
            _onBeginFrameObserver: Nullable<Observer<Engine>>;
            /** @internal */
            _onEndFrameObserver: Nullable<Observer<Engine>>;
            /** @internal */
            _createTimeQuery(): WebGLQuery;
            /** @internal */
            _deleteTimeQuery(query: WebGLQuery): void;
            /** @internal */
            _getGlAlgorithmType(algorithmType: number): number;
            /** @internal */
            _getTimeQueryResult(query: WebGLQuery): any;
            /** @internal */
            _getTimeQueryAvailability(query: WebGLQuery): any;
        }
        interface AbstractMesh {
            /**
             * Backing filed
             * @internal
             */
            __occlusionDataStorage: _OcclusionDataStorage;
            /**
             * Access property
             * @internal
             */
            _occlusionDataStorage: _OcclusionDataStorage;
            /**
             * This number indicates the number of allowed retries before stop the occlusion query, this is useful if the occlusion query is taking long time before to the query result is retrieved, the query result indicates if the object is visible within the scene or not and based on that Babylon.Js engine decides to show or hide the object.
             * The default value is -1 which means don't break the query and wait till the result
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            occlusionRetryCount: number;
            /**
             * This property is responsible for starting the occlusion query within the Mesh or not, this property is also used to determine what should happen when the occlusionRetryCount is reached. It has supports 3 values:
             * * OCCLUSION_TYPE_NONE (Default Value): this option means no occlusion query within the Mesh.
             * * OCCLUSION_TYPE_OPTIMISTIC: this option is means use occlusion query and if occlusionRetryCount is reached and the query is broken show the mesh.
             * * OCCLUSION_TYPE_STRICT: this option is means use occlusion query and if occlusionRetryCount is reached and the query is broken restore the last state of the mesh occlusion if the mesh was visible then show the mesh if was hidden then hide don't show.
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            occlusionType: number;
            /**
             * This property determines the type of occlusion query algorithm to run in WebGl, you can use:
             * * AbstractMesh.OCCLUSION_ALGORITHM_TYPE_ACCURATE which is mapped to GL_ANY_SAMPLES_PASSED.
             * * AbstractMesh.OCCLUSION_ALGORITHM_TYPE_CONSERVATIVE (Default Value) which is mapped to GL_ANY_SAMPLES_PASSED_CONSERVATIVE which is a false positive algorithm that is faster than GL_ANY_SAMPLES_PASSED but less accurate.
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            occlusionQueryAlgorithmType: number;
            /**
             * Gets or sets whether the mesh is occluded or not, it is used also to set the initial state of the mesh to be occluded or not
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            isOccluded: boolean;
            /**
             * Flag to check the progress status of the query
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            isOcclusionQueryInProgress: boolean;
            /**
             * Flag to force rendering the mesh even if occluded
             * @see https://doc.babylonjs.com/features/featuresDeepDive/occlusionQueries
             */
            forceRenderingWhenOccluded: boolean;
        }


        interface ThinEngine {
            /**
             * Creates a raw texture
             * @param data defines the data to store in the texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param format defines the format of the data
             * @param generateMipMaps defines if the engine should generate the mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (Texture.NEAREST_SAMPLINGMODE by default)
             * @param compression defines the compression used (null by default)
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param creationFlags specific flags to use when creating the texture (Constants.TEXTURE_CREATIONFLAG_STORAGE for storage textures, for eg)
             * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
             * @returns the raw texture inside an InternalTexture
             */
            createRawTexture(data: Nullable<ArrayBufferView>, width: number, height: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>, type: number, creationFlags?: number, useSRGBBuffer?: boolean): InternalTexture;
            /**
             * Update a raw texture
             * @param texture defines the texture to update
             * @param data defines the data to store in the texture
             * @param format defines the format of the data
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean): void;
            /**
             * Update a raw texture
             * @param texture defines the texture to update
             * @param data defines the data to store in the texture
             * @param format defines the format of the data
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the compression used (null by default)
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
             */
            updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression: Nullable<string>, type: number, useSRGBBuffer: boolean): void;
            /**
             * Creates a new raw cube texture
             * @param data defines the array of data to use to create each face
             * @param size defines the size of the textures
             * @param format defines the format of the data
             * @param type defines the type of the data (like Engine.TEXTURETYPE_UNSIGNED_INT)
             * @param generateMipMaps  defines if the engine should generate the mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param compression defines the compression used (null by default)
             * @returns the cube texture as an InternalTexture
             */
            createRawCubeTexture(data: Nullable<ArrayBufferView[]>, size: number, format: number, type: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>): InternalTexture;
            /**
             * Update a raw cube texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean): void;
            /**
             * Update a raw cube texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the compression used (null by default)
             */
            updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean, compression: Nullable<string>): void;
            /**
             * Update a raw cube texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the compression used (null by default)
             * @param level defines which level of the texture to update
             */
            updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean, compression: Nullable<string>, level: number): void;
            /**
             * Creates a new raw cube texture from a specified url
             * @param url defines the url where the data is located
             * @param scene defines the current scene
             * @param size defines the size of the textures
             * @param format defines the format of the data
             * @param type defines the type fo the data (like Engine.TEXTURETYPE_UNSIGNED_INT)
             * @param noMipmap defines if the engine should avoid generating the mip levels
             * @param callback defines a callback used to extract texture data from loaded data
             * @param mipmapGenerator defines to provide an optional tool to generate mip levels
             * @param onLoad defines a callback called when texture is loaded
             * @param onError defines a callback called if there is an error
             * @returns the cube texture as an InternalTexture
             */
            createRawCubeTextureFromUrl(url: string, scene: Nullable<Scene>, size: number, format: number, type: number, noMipmap: boolean, callback: (ArrayBuffer: ArrayBuffer) => Nullable<ArrayBufferView[]>, mipmapGenerator: Nullable<(faces: ArrayBufferView[]) => ArrayBufferView[][]>, onLoad: Nullable<() => void>, onError: Nullable<(message?: string, exception?: any) => void>): InternalTexture;
            /**
             * Creates a new raw cube texture from a specified url
             * @param url defines the url where the data is located
             * @param scene defines the current scene
             * @param size defines the size of the textures
             * @param format defines the format of the data
             * @param type defines the type fo the data (like Engine.TEXTURETYPE_UNSIGNED_INT)
             * @param noMipmap defines if the engine should avoid generating the mip levels
             * @param callback defines a callback used to extract texture data from loaded data
             * @param mipmapGenerator defines to provide an optional tool to generate mip levels
             * @param onLoad defines a callback called when texture is loaded
             * @param onError defines a callback called if there is an error
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param invertY defines if data must be stored with Y axis inverted
             * @returns the cube texture as an InternalTexture
             */
            createRawCubeTextureFromUrl(url: string, scene: Nullable<Scene>, size: number, format: number, type: number, noMipmap: boolean, callback: (ArrayBuffer: ArrayBuffer) => Nullable<ArrayBufferView[]>, mipmapGenerator: Nullable<(faces: ArrayBufferView[]) => ArrayBufferView[][]>, onLoad: Nullable<() => void>, onError: Nullable<(message?: string, exception?: any) => void>, samplingMode: number, invertY: boolean): InternalTexture;
            /**
             * Creates a new raw 3D texture
             * @param data defines the data used to create the texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param depth defines the depth of the texture
             * @param format defines the format of the texture
             * @param generateMipMaps defines if the engine must generate mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param compression defines the compressed used (can be null)
             * @param textureType defines the compressed used (can be null)
             * @param creationFlags specific flags to use when creating the texture (Constants.TEXTURE_CREATIONFLAG_STORAGE for storage textures, for eg)
             * @returns a new raw 3D texture (stored in an InternalTexture)
             */
            createRawTexture3D(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>, textureType: number, creationFlags?: number): InternalTexture;
            /**
             * Update a raw 3D texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawTexture3D(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean): void;
            /**
             * Update a raw 3D texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the used compression (can be null)
             * @param textureType defines the texture Type (Engine.TEXTURETYPE_UNSIGNED_INT, Engine.TEXTURETYPE_FLOAT...)
             */
            updateRawTexture3D(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression: Nullable<string>, textureType: number): void;
            /**
             * Creates a new raw 2D array texture
             * @param data defines the data used to create the texture
             * @param width defines the width of the texture
             * @param height defines the height of the texture
             * @param depth defines the number of layers of the texture
             * @param format defines the format of the texture
             * @param generateMipMaps defines if the engine must generate mip levels
             * @param invertY defines if data must be stored with Y axis inverted
             * @param samplingMode defines the required sampling mode (like Texture.NEAREST_SAMPLINGMODE)
             * @param compression defines the compressed used (can be null)
             * @param textureType defines the compressed used (can be null)
             * @param creationFlags specific flags to use when creating the texture (Constants.TEXTURE_CREATIONFLAG_STORAGE for storage textures, for eg)
             * @returns a new raw 2D array texture (stored in an InternalTexture)
             */
            createRawTexture2DArray(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression: Nullable<string>, textureType: number, creationFlags?: number): InternalTexture;
            /**
             * Update a raw 2D array texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateRawTexture2DArray(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean): void;
            /**
             * Update a raw 2D array texture
             * @param texture defines the texture to update
             * @param data defines the data to store
             * @param format defines the data format
             * @param invertY defines if data must be stored with Y axis inverted
             * @param compression defines the used compression (can be null)
             * @param textureType defines the texture Type (Engine.TEXTURETYPE_UNSIGNED_INT, Engine.TEXTURETYPE_FLOAT...)
             */
            updateRawTexture2DArray(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression: Nullable<string>, textureType: number): void;
        }


        interface ThinEngine {
            /** @internal */
            _readTexturePixels(texture: InternalTexture, width: number, height: number, faceIndex?: number, level?: number, buffer?: Nullable<ArrayBufferView>, flushRenderer?: boolean, noDataConversion?: boolean, x?: number, y?: number): Promise<ArrayBufferView>;
            /** @internal */
            _readTexturePixelsSync(texture: InternalTexture, width: number, height: number, faceIndex?: number, level?: number, buffer?: Nullable<ArrayBufferView>, flushRenderer?: boolean, noDataConversion?: boolean, x?: number, y?: number): ArrayBufferView;
        }
    /**
     * Allocate a typed array depending on a texture type. Optionally can copy existing data in the buffer.
     * @param type type of the texture
     * @param sizeOrDstBuffer size of the array OR an existing buffer that will be used as the destination of the copy (if copyBuffer is provided)
     * @param sizeInBytes true if the size of the array is given in bytes, false if it is the number of elements of the array
     * @param copyBuffer if provided, buffer to copy into the destination buffer (either a newly allocated buffer if sizeOrDstBuffer is a number or use sizeOrDstBuffer as the destination buffer otherwise)
     * @returns the allocated buffer or sizeOrDstBuffer if the latter is an ArrayBuffer
     */
    export function allocateAndCopyTypedBuffer(type: number, sizeOrDstBuffer: number | ArrayBuffer, sizeInBytes?: boolean, copyBuffer?: ArrayBuffer): ArrayBufferView;


    /**
     * Type used to define a texture size (either with a number or with a rect width and height)
     * @deprecated please use TextureSize instead
     */
    export type RenderTargetTextureSize = TextureSize;
        interface ThinEngine {
            /**
             * Creates a new render target texture
             * @param size defines the size of the texture
             * @param options defines the options used to create the texture
             * @returns a new render target wrapper ready to render texture
             */
            createRenderTargetTexture(size: TextureSize, options: boolean | RenderTargetCreationOptions): RenderTargetWrapper;
            /**
             * Creates a depth stencil texture.
             * This is only available in WebGL 2 or with the depth texture extension available.
             * @param size The size of face edge in the texture.
             * @param options The options defining the texture.
             * @param rtWrapper The render target wrapper for which the depth/stencil texture must be created
             * @returns The texture
             */
            createDepthStencilTexture(size: TextureSize, options: DepthTextureCreationOptions, rtWrapper: RenderTargetWrapper): InternalTexture;
            /**
             * Updates the sample count of a render target texture
             * @see https://doc.babylonjs.com/setup/support/webGL2#multisample-render-targets
             * @param rtWrapper defines the render target wrapper to update
             * @param samples defines the sample count to set
             * @returns the effective sample count (could be 0 if multisample render targets are not supported)
             */
            updateRenderTargetTextureSampleCount(rtWrapper: Nullable<RenderTargetWrapper>, samples: number): number;
            /** @internal */
            _createDepthStencilTexture(size: TextureSize, options: DepthTextureCreationOptions, rtWrapper: RenderTargetWrapper): InternalTexture;
            /** @internal */
            _createHardwareRenderTargetWrapper(isMulti: boolean, isCube: boolean, size: TextureSize): RenderTargetWrapper;
        }


        interface ThinEngine {
            /**
             * Creates a new render target cube wrapper
             * @param size defines the size of the texture
             * @param options defines the options used to create the texture
             * @returns a new render target cube wrapper
             */
            createRenderTargetCubeTexture(size: number, options?: RenderTargetCreationOptions): RenderTargetWrapper;
        }


        interface ThinEngine {
            /**
             * Creates a storage buffer
             * @param data the data for the storage buffer or the size of the buffer
             * @param creationFlags flags to use when creating the buffer (see Constants.BUFFER_CREATIONFLAG_XXX). The BUFFER_CREATIONFLAG_STORAGE flag will be automatically added
             * @param label defines the label of the buffer (for debug purpose)
             * @returns the new buffer
             */
            createStorageBuffer(data: DataArray | number, creationFlags: number, label?: string): DataBuffer;
            /**
             * Updates a storage buffer
             * @param buffer the storage buffer to update
             * @param data the data used to update the storage buffer
             * @param byteOffset the byte offset of the data
             * @param byteLength the byte length of the data
             */
            updateStorageBuffer(buffer: DataBuffer, data: DataArray, byteOffset?: number, byteLength?: number): void;
            /**
             * Read data from a storage buffer
             * @param storageBuffer The storage buffer to read from
             * @param offset The offset in the storage buffer to start reading from (default: 0)
             * @param size  The number of bytes to read from the storage buffer (default: capacity of the buffer)
             * @param buffer The buffer to write the data we have read from the storage buffer to (optional)
             * @param noDelay If true, a call to flushFramebuffer will be issued so that the data can be read back immediately and not in engine.onEndFrameObservable. This can speed up data retrieval, at the cost of a small perf penalty (default: false).
             * @returns If not undefined, returns the (promise) buffer (as provided by the 4th parameter) filled with the data, else it returns a (promise) Uint8Array with the data read from the storage buffer
             */
            readFromStorageBuffer(storageBuffer: DataBuffer, offset?: number, size?: number, buffer?: ArrayBufferView, noDelay?: boolean): Promise<ArrayBufferView>;
            /**
             * Sets a storage buffer in the shader
             * @param name Defines the name of the storage buffer as defined in the shader
             * @param buffer Defines the value to give to the uniform
             */
            setStorageBuffer(name: string, buffer: Nullable<StorageBuffer>): void;
        }


        interface ThinEngine {
            /**
             * Sets a texture sampler to the according uniform.
             * @param name The name of the uniform in the effect
             * @param sampler The sampler to apply
             */
            setTextureSampler(name: string, sampler: Nullable<TextureSampler>): void;
        }


        interface Engine {
            /** @internal */
            _excludedCompressedTextures: string[];
            /** @internal */
            _textureFormatInUse: string;
            /**
             * Gets the list of texture formats supported
             */
            readonly texturesSupported: Array<string>;
            /**
             * Gets the texture format in use
             */
            readonly textureFormatInUse: Nullable<string>;
            /**
             * Set the compressed texture extensions or file names to skip.
             *
             * @param skippedFiles defines the list of those texture files you want to skip
             * Example: [".dds", ".env", "myfile.png"]
             */
            setCompressedTextureExclusions(skippedFiles: Array<string>): void;
            /**
             * Set the compressed texture format to use, based on the formats you have, and the formats
             * supported by the hardware / browser.
             *
             * Khronos Texture Container (.ktx) files are used to support this.  This format has the
             * advantage of being specifically designed for OpenGL.  Header elements directly correspond
             * to API arguments needed to compressed textures.  This puts the burden on the container
             * generator to house the arcane code for determining these for current & future formats.
             *
             * for description see https://www.khronos.org/opengles/sdk/tools/KTX/
             * for file layout see https://www.khronos.org/opengles/sdk/tools/KTX/file_format_spec/
             *
             * Note: The result of this call is not taken into account when a texture is base64.
             *
             * @param formatsAvailable defines the list of those format families you have created
             * on your server.  Syntax: '-' + format family + '.ktx'.  (Case and order do not matter.)
             *
             * Current families are astc, dxt, pvrtc, etc2, & etc1.
             * @returns The extension selected.
             */
            setTextureFormatToUse(formatsAvailable: Array<string>): Nullable<string>;
        }


    /** @internal */
    export var _forceTransformFeedbackToBundle: boolean;
        interface Engine {
            /**
             * Creates a webGL transform feedback object
             * Please makes sure to check webGLVersion property to check if you are running webGL 2+
             * @returns the webGL transform feedback object
             */
            createTransformFeedback(): WebGLTransformFeedback;
            /**
             * Delete a webGL transform feedback object
             * @param value defines the webGL transform feedback object to delete
             */
            deleteTransformFeedback(value: WebGLTransformFeedback): void;
            /**
             * Bind a webGL transform feedback object to the webgl context
             * @param value defines the webGL transform feedback object to bind
             */
            bindTransformFeedback(value: Nullable<WebGLTransformFeedback>): void;
            /**
             * Begins a transform feedback operation
             * @param usePoints defines if points or triangles must be used
             */
            beginTransformFeedback(usePoints: boolean): void;
            /**
             * Ends a transform feedback operation
             */
            endTransformFeedback(): void;
            /**
             * Specify the varyings to use with transform feedback
             * @param program defines the associated webGL program
             * @param value defines the list of strings representing the varying names
             */
            setTranformFeedbackVaryings(program: WebGLProgram, value: string[]): void;
            /**
             * Bind a webGL buffer for a transform feedback operation
             * @param value defines the webGL buffer to bind
             */
            bindTransformFeedbackBuffer(value: Nullable<DataBuffer>): void;
        }


        interface ThinEngine {
            /**
             * Create an uniform buffer
             * @see https://doc.babylonjs.com/setup/support/webGL2#uniform-buffer-objets
             * @param elements defines the content of the uniform buffer
             * @param label defines a name for the buffer (for debugging purpose)
             * @returns the webGL uniform buffer
             */
            createUniformBuffer(elements: FloatArray, label?: string): DataBuffer;
            /**
             * Create a dynamic uniform buffer
             * @see https://doc.babylonjs.com/setup/support/webGL2#uniform-buffer-objets
             * @param elements defines the content of the uniform buffer
             * @param label defines a name for the buffer (for debugging purpose)
             * @returns the webGL uniform buffer
             */
            createDynamicUniformBuffer(elements: FloatArray, label?: string): DataBuffer;
            /**
             * Update an existing uniform buffer
             * @see https://doc.babylonjs.com/setup/support/webGL2#uniform-buffer-objets
             * @param uniformBuffer defines the target uniform buffer
             * @param elements defines the content to update
             * @param offset defines the offset in the uniform buffer where update should start
             * @param count defines the size of the data to update
             */
            updateUniformBuffer(uniformBuffer: DataBuffer, elements: FloatArray, offset?: number, count?: number): void;
            /**
             * Bind an uniform buffer to the current webGL context
             * @param buffer defines the buffer to bind
             */
            bindUniformBuffer(buffer: Nullable<DataBuffer>): void;
            /**
             * Bind a buffer to the current webGL context at a given location
             * @param buffer defines the buffer to bind
             * @param location defines the index where to bind the buffer
             * @param name Name of the uniform variable to bind
             */
            bindUniformBufferBase(buffer: DataBuffer, location: number, name: string): void;
            /**
             * Bind a specific block at a given index in a specific shader program
             * @param pipelineContext defines the pipeline context to use
             * @param blockName defines the block name
             * @param index defines the index where to bind the block
             */
            bindUniformBlock(pipelineContext: IPipelineContext, blockName: string, index: number): void;
        }


        interface ThinEngine {
            /**
             * Update a video texture
             * @param texture defines the texture to update
             * @param video defines the video element to use
             * @param invertY defines if data must be stored with Y axis inverted
             */
            updateVideoTexture(texture: Nullable<InternalTexture>, video: HTMLVideoElement | Nullable<ExternalTexture>, invertY: boolean): void;
        }


    /**
     * Class used to define an additional view for the engine
     * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/multiCanvas
     */
    export class EngineView {
        /**
         * A randomly generated unique id
         */
        readonly id: string;
        /** Defines the canvas where to render the view */
        target: HTMLCanvasElement;
        /**
         * Defines an optional camera or array of cameras used to render the view (will use active camera / cameras else)
         * Support for array of cameras @since
         */
        camera?: Camera | Camera[];
        /** Indicates if the destination view canvas should be cleared before copying the parent canvas. Can help if the scene clear color has alpha < 1 */
        clearBeforeCopy?: boolean;
        /** Indicates if the view is enabled (true by default) */
        enabled: boolean;
        /** Defines a custom function to handle canvas size changes. (the canvas to render into is provided to the callback) */
        customResize?: (canvas: HTMLCanvasElement) => void;
    }
        interface Engine {
            /** @internal */
            _inputElement: Nullable<HTMLElement>;
            /**
             * Gets or sets the  HTML element to use for attaching events
             */
            inputElement: Nullable<HTMLElement>;
            /**
             * Observable to handle when a change to inputElement occurs
             * @internal
             */
            _onEngineViewChanged?: () => void;
            /**
             * Will be triggered before the view renders
             */
            readonly onBeforeViewRenderObservable: Observable<EngineView>;
            /**
             * Will be triggered after the view rendered
             */
            readonly onAfterViewRenderObservable: Observable<EngineView>;
            /**
             * Gets the current engine view
             * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/multiCanvas
             */
            activeView: Nullable<EngineView>;
            /** Gets or sets the list of views */
            views: EngineView[];
            /**
             * Register a new child canvas
             * @param canvas defines the canvas to register
             * @param camera defines an optional camera or array of cameras to use with this canvas (it will overwrite the scene.activeCamera / scene.activeCameras for this view). Support for array of cameras @since
             * @param clearBeforeCopy Indicates if the destination view canvas should be cleared before copying the parent canvas. Can help if the scene clear color has alpha \< 1
             * @returns the associated view
             */
            registerView(canvas: HTMLCanvasElement, camera?: Camera | Camera[], clearBeforeCopy?: boolean): EngineView;
            /**
             * Remove a registered child canvas
             * @param canvas defines the canvas to remove
             * @returns the current engine
             */
            unRegisterView(canvas: HTMLCanvasElement): Engine;
            /**
             * @internal
             */
            _renderViewStep(view: EngineView): boolean;
        }




    /**
     * Class used to abstract a canvas
     */
    export interface ICanvas {
        /**
         * Canvas width.
         */
        width: number;
        /**
         * Canvas height.
         */
        height: number;
        /**
         * returns a drawing context on the canvas.
         * @param contextType context identifier.
         * @param contextAttributes context attributes.
         * @returns ICanvasRenderingContext object.
         */
        getContext(contextType: string, contextAttributes?: any): ICanvasRenderingContext;
        /**
         * returns a data URI containing a representation of the image in the format specified by the type parameter.
         * @param mime the image format.
         * @returns string containing the requested data URI.
         */
        toDataURL(mime: string): string;
        /**
         * Removes the canvas from the document.
         */
        remove(): void;
    }
    /**
     * Class used to abstract am image to use with the canvas and its context
     */
    export interface IImage {
        /**
         * onload callback.
         */
        onload: ((this: GlobalEventHandlers, ev: Event) => any) | null;
        /**
         * Error callback.
         */
        onerror: ((this: GlobalEventHandlers, ev: Event) => any) | null;
        /**
         * Image source.
         */
        src: string;
        /**
         * Image width.
         */
        readonly width: number;
        /**
         * Image height.
         */
        readonly height: number;
        /**
         * The original height of the image resource before sizing.
         */
        readonly naturalHeight: number;
        /**
         * The original width of the image resource before sizing.
         */
        readonly naturalWidth: number;
        /**
         * provides support for CORS, defining how the element handles crossorigin requests,
         * thereby enabling the configuration of the CORS requests for the element's fetched data.
         */
        crossOrigin: string | null;
        /**
         * provides support for referrer policy on xhr load request,
         * it is used to control the request header.
         */
        referrerPolicy: string;
    }
    /**
     * Class used to abstract a canvas gradient
     */
    export interface ICanvasGradient {
        /**
         * adds a new color stop, defined by an offset and a color, to a given canvas gradient.
         * @param offset A number between 0 and 1, inclusive, representing the position of the color stop. 0 represents the start of the gradient and 1 represents the end.
         * @param color value representing the color of the stop.
         */
        addColorStop(offset: number, color: string): void;
    }
    /**
     * Class used to abstract a text measurement
     */
    export interface ITextMetrics {
        /**
         * Text width.
         */
        readonly width: number;
        /**
         * distance (in pixels) parallel to the baseline from the alignment point given by the CanvasRenderingContext2D.textAlign
         * property to the left side of the bounding rectangle of the given text
         */
        readonly actualBoundingBoxLeft: number;
        /**
         * distance (in pixels) parallel to the baseline from the alignment point given by the CanvasRenderingContext2D.textAlign
         * property to the right side of the bounding rectangle of the given text
         */
        readonly actualBoundingBoxRight: number;
    }
    /**
     * Class used to abstract a matrix
     */
    export interface DOMMatrix {
        /**
         * A Boolean flag whose value is true if the matrix was initialized as a 2D matrix. If false, the matrix is 3D.
         */
        is2D: boolean;
        /**
         * A Boolean whose value is true if the matrix is the identity matrix. The identity matrix is one in which every value is 0 except those on the main diagonal from top-left to bottom-right corner (in other words, where the offsets in each direction are equal).
         */
        isIdentity: boolean;
        /**
         * The following double-precision floating-point values represent the components of a matrix which are required in order to perform 2D rotations and translations.
         */
        a: number;
        /**
         * The following double-precision floating-point values represent the components of a matrix which are required in order to perform 2D rotations and translations.
         */
        b: number;
        /**
         * The following double-precision floating-point values represent the components of a matrix which are required in order to perform 2D rotations and translations.
         */
        c: number;
        /**
         * The following double-precision floating-point values represent the components of a matrix which are required in order to perform 2D rotations and translations.
         */
        d: number;
        /**
         * The following double-precision floating-point values represent the components of a matrix which are required in order to perform 2D rotations and translations.
         */
        e: number;
        /**
         * The following double-precision floating-point values represent the components of a matrix which are required in order to perform 2D rotations and translations.
         */
        f: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m11: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m12: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m13: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m14: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m21: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m22: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m23: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m24: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m31: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m32: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m33: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m34: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m41: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m42: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m43: number;
        /**
         * The following are double-precision floating-point values representing each component of a 4×4 matrix, where m11 through m14 are the first column, m21 through m24 are the second column, and so forth.
         */
        m44: number;
    }
    /**
     * Class used to abstract canvas rendering
     */
    export interface ICanvasRenderingContext {
        /**
         * Defines the type of corners where two lines meet. Possible values: round, bevel, miter (default).
         */
        lineJoin: string;
        /**
         * Miter limit ratio. Default 10.
         */
        miterLimit: number;
        /**
         * Font setting. Default value 10px sans-serif.
         */
        font: string;
        /**
         * Color or style to use for the lines around shapes. Default #000 (black).
         */
        strokeStyle: string | ICanvasGradient;
        /**
         * Color or style to use inside shapes. Default #000 (black).
         */
        fillStyle: string | ICanvasGradient;
        /**
         * Alpha value that is applied to shapes and images before they are composited onto the canvas. Default 1.0 (opaque).
         */
        globalAlpha: number;
        /**
         * Color of the shadow. Default: fully-transparent black.
         */
        shadowColor: string;
        /**
         * Specifies the blurring effect. Default: 0.
         */
        shadowBlur: number;
        /**
         * Horizontal distance the shadow will be offset. Default: 0.
         */
        shadowOffsetX: number;
        /**
         * Vertical distance the shadow will be offset. Default: 0.
         */
        shadowOffsetY: number;
        /**
         * Width of lines. Default 1.0.
         */
        lineWidth: number;
        /**
         * canvas is a read-only reference to ICanvas.
         */
        readonly canvas: ICanvas;
        /**
         * Sets all pixels in the rectangle defined by starting point (x, y) and size (width, height) to transparent black, erasing any previously drawn content.
         * @param x The x-axis coordinate of the rectangle's starting point.
         * @param y The y-axis coordinate of the rectangle's starting point.
         * @param width The rectangle's width. Positive values are to the right, and negative to the left.
         * @param height The rectangle's height. Positive values are down, and negative are up.
         */
        clearRect(x: number, y: number, width: number, height: number): void;
        /**
         * Saves the current drawing style state using a stack so you can revert any change you make to it using restore().
         */
        save(): void;
        /**
         * Restores the drawing style state to the last element on the 'state stack' saved by save().
         */
        restore(): void;
        /**
         * Draws a filled rectangle at (x, y) position whose size is determined by width and height.
         * @param x The x-axis coordinate of the rectangle's starting point.
         * @param y The y-axis coordinate of the rectangle's starting point.
         * @param width The rectangle's width. Positive values are to the right, and negative to the left.
         * @param height The rectangle's height. Positive values are down, and negative are up.
         */
        fillRect(x: number, y: number, width: number, height: number): void;
        /**
         * Adds a scaling transformation to the canvas units by x horizontally and by y vertically.
         * @param x Scaling factor in the horizontal direction. A negative value flips pixels across the vertical axis. A value of 1 results in no horizontal scaling.
         * @param y Scaling factor in the vertical direction. A negative value flips pixels across the horizontal axis. A value of 1 results in no vertical scaling.
         */
        scale(x: number, y: number): void;
        /**
         * Adds a rotation to the transformation matrix. The angle argument represents a clockwise rotation angle and is expressed in radians.
         * @param angle The rotation angle, clockwise in radians. You can use degree * Math.PI / 180 to calculate a radian from a degree.
         */
        rotate(angle: number): void;
        /**
         * Adds a translation transformation by moving the canvas and its origin x horizontally and y vertically on the grid.
         * @param x Distance to move in the horizontal direction. Positive values are to the right, and negative to the left.
         * @param y Distance to move in the vertical direction. Positive values are down, and negative are up.
         */
        translate(x: number, y: number): void;
        /**
         * Paints a rectangle which has a starting point at (x, y) and has a w width and an h height onto the canvas, using the current stroke style.
         * @param x The x-axis coordinate of the rectangle's starting point.
         * @param y The y-axis coordinate of the rectangle's starting point.
         * @param width The rectangle's width. Positive values are to the right, and negative to the left.
         * @param height The rectangle's height. Positive values are down, and negative are up.
         */
        strokeRect(x: number, y: number, width: number, height: number): void;
        /**
         * Creates a path for a rectangle at position (x, y) with a size that is determined by width and height.
         * @param x The x-axis coordinate of the rectangle's starting point.
         * @param y The y-axis coordinate of the rectangle's starting point.
         * @param width The rectangle's width. Positive values are to the right, and negative to the left.
         * @param height The rectangle's height. Positive values are down, and negative are up.
         */
        rect(x: number, y: number, width: number, height: number): void;
        /**
         * Creates a clipping path from the current sub-paths. Everything drawn after clip() is called appears inside the clipping path only.
         */
        clip(): void;
        /**
         * Paints data from the given ImageData object onto the bitmap. If a dirty rectangle is provided, only the pixels from that rectangle are painted.
         * @param imageData An ImageData object containing the array of pixel values.
         * @param dx Horizontal position (x coordinate) at which to place the image data in the destination canvas.
         * @param dy Vertical position (y coordinate) at which to place the image data in the destination canvas.
         */
        putImageData(imageData: ImageData, dx: number, dy: number): void;
        /**
         * Adds a circular arc to the current path.
         * @param x The horizontal coordinate of the arc's center.
         * @param y The vertical coordinate of the arc's center.
         * @param radius The arc's radius. Must be positive.
         * @param startAngle The angle at which the arc starts in radians, measured from the positive x-axis.
         * @param endAngle The angle at which the arc ends in radians, measured from the positive x-axis.
         * @param anticlockwise An optional Boolean. If true, draws the arc counter-clockwise between the start and end angles. The default is false (clockwise).
         */
        arc(x: number, y: number, radius: number, startAngle: number, endAngle: number, anticlockwise?: boolean): void;
        /**
         * Starts a new path by emptying the list of sub-paths. Call this method when you want to create a new path.
         */
        beginPath(): void;
        /**
         * Causes the point of the pen to move back to the start of the current sub-path. It tries to draw a straight line from the current point to the start.
         * If the shape has already been closed or has only one point, this function does nothing.
         */
        closePath(): void;
        /**
         * Moves the starting point of a new sub-path to the (x, y) coordinates.
         * @param x The x-axis (horizontal) coordinate of the point.
         * @param y The y-axis (vertical) coordinate of the point.
         */
        moveTo(x: number, y: number): void;
        /**
         * Connects the last point in the current sub-path to the specified (x, y) coordinates with a straight line.
         * @param x The x-axis coordinate of the line's end point.
         * @param y The y-axis coordinate of the line's end point.
         */
        lineTo(x: number, y: number): void;
        /**
         * Adds a quadratic Bézier curve to the current path.
         * @param cpx The x-axis coordinate of the control point.
         * @param cpy The y-axis coordinate of the control point.
         * @param x The x-axis coordinate of the end point.
         * @param y The y-axis coordinate of the end point.
         */
        quadraticCurveTo(cpx: number, cpy: number, x: number, y: number): void;
        /**
         * Returns a TextMetrics object.
         * @param text The text String to measure.
         * @returns ITextMetrics A ITextMetrics object.
         */
        measureText(text: string): ITextMetrics;
        /**
         * Strokes the current sub-paths with the current stroke style.
         */
        stroke(): void;
        /**
         * Fills the current sub-paths with the current fill style.
         */
        fill(): void;
        /**
         * Draws the specified image. This method is available in multiple formats, providing a great deal of flexibility in its use.
         * @param image An element to draw into the context.
         * @param sx The x-axis coordinate of the top left corner of the sub-rectangle of the source image to draw into the destination context.
         * @param sy The y-axis coordinate of the top left corner of the sub-rectangle of the source image to draw into the destination context.
         * @param sWidth The width of the sub-rectangle of the source image to draw into the destination context. If not specified, the entire rectangle from the coordinates specified by sx and sy to the bottom-right corner of the image is used.
         * @param sHeight The height of the sub-rectangle of the source image to draw into the destination context.
         * @param dx The x-axis coordinate in the destination canvas at which to place the top-left corner of the source image.
         * @param dy The y-axis coordinate in the destination canvas at which to place the top-left corner of the source image.
         * @param dWidth The width to draw the image in the destination canvas. This allows scaling of the drawn image. If not specified, the image is not scaled in width when drawn.
         * @param dHeight The height to draw the image in the destination canvas. This allows scaling of the drawn image. If not specified, the image is not scaled in height when drawn.
         */
        drawImage(image: any, sx: number, sy: number, sWidth: number, sHeight: number, dx: number, dy: number, dWidth: number, dHeight: number): void;
        /**
         * Draws the specified image. This method is available in multiple formats, providing a great deal of flexibility in its use.
         * @param image An element to draw into the context.
         * @param dx The x-axis coordinate in the destination canvas at which to place the top-left corner of the source image.
         * @param dy The y-axis coordinate in the destination canvas at which to place the top-left corner of the source image.
         * @param dWidth The width to draw the image in the destination canvas. This allows scaling of the drawn image. If not specified, the image is not scaled in width when drawn.
         * @param dHeight The height to draw the image in the destination canvas. This allows scaling of the drawn image. If not specified, the image is not scaled in height when drawn.
         */
        drawImage(image: any, dx: number, dy: number, dWidth: number, dHeight: number): void;
        /**
         * Draws the specified image. This method is available in multiple formats, providing a great deal of flexibility in its use.
         * @param image An element to draw into the context.
         * @param dx The x-axis coordinate in the destination canvas at which to place the top-left corner of the source image.
         * @param dy The y-axis coordinate in the destination canvas at which to place the top-left corner of the source image.
         */
        drawImage(image: any, dx: number, dy: number): void;
        /**
         * Returns an ImageData object representing the underlying pixel data for the area of the canvas denoted by the rectangle which starts at (sx, sy) and has an sw width and sh height.
         * @param sx The x-axis coordinate of the top-left corner of the rectangle from which the ImageData will be extracted.
         * @param sy The y-axis coordinate of the top-left corner of the rectangle from which the ImageData will be extracted.
         * @param sw The width of the rectangle from which the ImageData will be extracted. Positive values are to the right, and negative to the left.
         * @param sh The height of the rectangle from which the ImageData will be extracted. Positive values are down, and negative are up.
         * @returns ImageData An ImageData object containing the image data for the rectangle of the canvas specified.
         */
        getImageData(sx: number, sy: number, sw: number, sh: number): ImageData;
        /**
         * Sets the current line dash pattern.
         * @param segments An Array of numbers that specify distances to alternately draw a line and a gap (in coordinate space units).
         */
        setLineDash(segments: Array<number>): void;
        /**
         * Draws (fills) a given text at the given (x, y) position.
         * @param text A String specifying the text string to render into the context. The text is rendered using the settings specified by font, textAlign, textBaseline, and direction.
         * @param x The x-axis coordinate of the point at which to begin drawing the text, in pixels.
         * @param y The y-axis coordinate of the baseline on which to begin drawing the text, in pixels.
         * @param maxWidth The maximum number of pixels wide the text may be once rendered. If not specified, there is no limit to the width of the text.
         */
        fillText(text: string, x: number, y: number, maxWidth?: number): void;
        /**
         * Draws (strokes) a given text at the given (x, y) position.
         * @param text A String specifying the text string to render into the context. The text is rendered using the settings specified by font, textAlign, textBaseline, and direction.
         * @param x The x-axis coordinate of the point at which to begin drawing the text, in pixels.
         * @param y The y-axis coordinate of the baseline on which to begin drawing the text, in pixels.
         * @param maxWidth The maximum number of pixels wide the text may be once rendered. If not specified, there is no limit to the width of the text.
         */
        strokeText(text: string, x: number, y: number, maxWidth?: number): void;
        /**
         * Creates a linear gradient along the line given by the coordinates represented by the parameters.
         * @param x0 The x-axis coordinate of the start point.
         * @param y0 The y-axis coordinate of the start point.
         * @param x1 The x-axis coordinate of the end point.
         * @param y1 The y-axis coordinate of the end point.
         * @returns ICanvasGradient A linear ICanvasGradient initialized with the specified line.
         */
        createLinearGradient(x0: number, y0: number, x1: number, y1: number): ICanvasGradient;
        /**
         * Creates a linear gradient along the line given by the coordinates represented by the parameters.
         * @param x0 The x-axis coordinate of the start circle.
         * @param y0 The y-axis coordinate of the start circle.
         * @param r0 The radius of the start circle. Must be non-negative and finite.
         * @param x1 The x-axis coordinate of the end point.
         * @param y1 The y-axis coordinate of the end point.
         * @param r1 The radius of the end circle. Must be non-negative and finite.
         * @returns ICanvasGradient A linear ICanvasGradient initialized with the two specified circles.
         */
        createRadialGradient(x0: number, y0: number, r0: number, x1: number, y1: number, r1: number): ICanvasGradient;
        /**
         * Resets the current transform to matrix composed with a, b, c, d, e, f.
         * @param a Horizontal scaling. A value of 1 results in no scaling.
         * @param b Vertical skewing.
         * @param c Horizontal skewing.
         * @param d Vertical scaling. A value of 1 results in no scaling.
         * @param e Horizontal translation (moving).
         * @param f Vertical translation (moving).
         */
        setTransform(a: number, b: number, c: number, d: number, e: number, f: number): void;
        /**
         * Retrieves the current transformation matrix being applied to the context.
         */
        getTransform(): DOMMatrix;
    }


    /** @internal */
    export interface IDrawContext {
        uniqueId: number;
        useInstancing: boolean;
        reset(): void;
        dispose(): void;
    }


    /** @internal */
    export interface IMaterialContext {
        uniqueId: number;
        reset(): void;
    }




    /**
     * Interface for attribute information associated with buffer instantiation
     */
    export interface InstancingAttributeInfo {
        /**
         * Name of the GLSL attribute
         * if attribute index is not specified, this is used to retrieve the index from the effect
         */
        attributeName: string;
        /**
         * Index/offset of the attribute in the vertex shader
         * if not specified, this will be computes from the name.
         */
        index?: number;
        /**
         * size of the attribute, 1, 2, 3 or 4
         */
        attributeSize: number;
        /**
         * Offset of the data in the Vertex Buffer acting as the instancing buffer
         */
        offset: number;
        /**
         * Modifies the rate at which generic vertex attributes advance when rendering multiple instances
         * default to 1
         */
        divisor?: number;
        /**
         * type of the attribute, gl.BYTE, gl.UNSIGNED_BYTE, gl.SHORT, gl.UNSIGNED_SHORT, gl.FIXED, gl.FLOAT.
         * default is FLOAT
         */
        attributeType?: number;
        /**
         * normalization of fixed-point data. behavior unclear, use FALSE, default is FALSE
         */
        normalized?: boolean;
    }


    /**
     * Class used to store and describe the pipeline context associated with an effect
     */
    export interface IPipelineContext {
        /**
         * Gets a boolean indicating that this pipeline context is supporting asynchronous creating
         */
        readonly isAsync: boolean;
        /**
         * Gets a boolean indicating that the context is ready to be used (like shaders / pipelines are compiled and ready for instance)
         */
        readonly isReady: boolean;
        /** @internal */
        _name?: string;
        /** @internal */
        _getVertexShaderCode(): string | null;
        /** @internal */
        _getFragmentShaderCode(): string | null;
        /** @internal */
        _handlesSpectorRebuildCallback?(onCompiled: (compiledObject: any) => void): void;
        /** @internal */
        _fillEffectInformation(effect: Effect, uniformBuffersNames: {
            [key: string]: number;
        }, uniformsNames: string[], uniforms: {
            [key: string]: Nullable<WebGLUniformLocation>;
        }, samplerList: string[], samplers: {
            [key: string]: number;
        }, attributesNames: string[], attributes: number[]): void;
        /** Releases the resources associated with the pipeline. */
        dispose(): void;
        /**
         * Sets an integer value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value Value to be set.
         */
        setInt(uniformName: string, value: number): void;
        /**
         * Sets an int2 value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First int in int2.
         * @param y Second int in int2.
         */
        setInt2(uniformName: string, x: number, y: number): void;
        /**
         * Sets an int3 value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First int in int3.
         * @param y Second int in int3.
         * @param z Third int in int3.
         */
        setInt3(uniformName: string, x: number, y: number, z: number): void;
        /**
         * Sets an int4 value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First int in int4.
         * @param y Second int in int4.
         * @param z Third int in int4.
         * @param w Fourth int in int4.
         */
        setInt4(uniformName: string, x: number, y: number, z: number, w: number): void;
        /**
         * Sets an int array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray(uniformName: string, array: Int32Array): void;
        /**
         * Sets an int array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray2(uniformName: string, array: Int32Array): void;
        /**
         * Sets an int array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray3(uniformName: string, array: Int32Array): void;
        /**
         * Sets an int array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray4(uniformName: string, array: Int32Array): void;
        /**
         * Sets an unsigned integer value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value Value to be set.
         */
        setUInt(uniformName: string, value: number): void;
        /**
         * Sets an unsigned int2 value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First unsigned int in uint2.
         * @param y Second unsigned int in uint2.
         */
        setUInt2(uniformName: string, x: number, y: number): void;
        /**
         * Sets an unsigned int3 value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First unsigned int in uint3.
         * @param y Second unsigned int in uint3.
         * @param z Third unsigned int in uint3.
         */
        setUInt3(uniformName: string, x: number, y: number, z: number): void;
        /**
         * Sets an unsigned int4 value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First unsigned int in uint4.
         * @param y Second unsigned int in uint4.
         * @param z Third unsigned int in uint4.
         * @param w Fourth unsigned int in uint4.
         */
        setUInt4(uniformName: string, x: number, y: number, z: number, w: number): void;
        /**
         * Sets an unsigned int array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an unsigned int array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray2(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an unsigned int array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray3(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an unsigned int array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray4(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray(uniformName: string, array: number[] | Float32Array): void;
        /**
         * Sets an array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray2(uniformName: string, array: number[] | Float32Array): void;
        /**
         * Sets an array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray3(uniformName: string, array: number[] | Float32Array): void;
        /**
         * Sets an array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray4(uniformName: string, array: number[] | Float32Array): void;
        /**
         * Sets matrices on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrices matrices to be set.
         */
        setMatrices(uniformName: string, matrices: Float32Array): void;
        /**
         * Sets matrix on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         */
        setMatrix(uniformName: string, matrix: IMatrixLike): void;
        /**
         * Sets a 3x3 matrix on a uniform variable. (Specified as [1,2,3,4,5,6,7,8,9] will result in [1,2,3][4,5,6][7,8,9] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         */
        setMatrix3x3(uniformName: string, matrix: Float32Array): void;
        /**
         * Sets a 2x2 matrix on a uniform variable. (Specified as [1,2,3,4] will result in [1,2][3,4] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         */
        setMatrix2x2(uniformName: string, matrix: Float32Array): void;
        /**
         * Sets a float on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value value to be set.
         */
        setFloat(uniformName: string, value: number): void;
        /**
         * Sets a Vector2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector2 vector2 to be set.
         */
        setVector2(uniformName: string, vector2: IVector2Like): void;
        /**
         * Sets a float2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float2.
         * @param y Second float in float2.
         */
        setFloat2(uniformName: string, x: number, y: number): void;
        /**
         * Sets a Vector3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector3 Value to be set.
         */
        setVector3(uniformName: string, vector3: IVector3Like): void;
        /**
         * Sets a float3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float3.
         * @param y Second float in float3.
         * @param z Third float in float3.
         */
        setFloat3(uniformName: string, x: number, y: number, z: number): void;
        /**
         * Sets a Vector4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector4 Value to be set.
         */
        setVector4(uniformName: string, vector4: IVector4Like): void;
        /**
         * Sets a Quaternion on a uniform variable.
         * @param uniformName Name of the variable.
         * @param quaternion Value to be set.
         */
        setQuaternion(uniformName: string, quaternion: IQuaternionLike): void;
        /**
         * Sets a float4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float4.
         * @param y Second float in float4.
         * @param z Third float in float4.
         * @param w Fourth float in float4.
         */
        setFloat4(uniformName: string, x: number, y: number, z: number, w: number): void;
        /**
         * Sets a Color3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         */
        setColor3(uniformName: string, color3: IColor3Like): void;
        /**
         * Sets a Color4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         * @param alpha Alpha value to be set.
         */
        setColor4(uniformName: string, color3: IColor3Like, alpha: number): void;
        /**
         * Sets a Color4 on a uniform variable
         * @param uniformName defines the name of the variable
         * @param color4 defines the value to be set
         */
        setDirectColor4(uniformName: string, color4: IColor4Like): void;
    }




    /** @internal */
    export type NativeData = Uint32Array;
    /** @internal */
    export class NativeDataStream {
        private readonly _uint32s;
        private readonly _int32s;
        private readonly _float32s;
        private readonly _length;
        private _position;
        private readonly _nativeDataStream;
        private static readonly DEFAULT_BUFFER_SIZE;
        constructor();
        /**
         * Writes a uint32 to the stream
         * @param value the value to write
         */
        writeUint32(value: number): void;
        /**
         * Writes an int32 to the stream
         * @param value the value to write
         */
        writeInt32(value: number): void;
        /**
         * Writes a float32 to the stream
         * @param value the value to write
         */
        writeFloat32(value: number): void;
        /**
         * Writes a uint32 array to the stream
         * @param values the values to write
         */
        writeUint32Array(values: Uint32Array): void;
        /**
         * Writes an int32 array to the stream
         * @param values the values to write
         */
        writeInt32Array(values: Int32Array): void;
        /**
         * Writes a float32 array to the stream
         * @param values the values to write
         */
        writeFloat32Array(values: Float32Array): void;
        /**
         * Writes native data to the stream
         * @param handle the handle to the native data
         */
        writeNativeData(handle: NativeData): void;
        /**
         * Writes a boolean to the stream
         * @param value the value to write
         */
        writeBoolean(value: boolean): void;
        private _flushIfNecessary;
        private _flush;
    }


    /** @internal */
    export class NativeHardwareTexture implements HardwareTextureWrapper {
        private readonly _engine;
        private _nativeTexture;
        get underlyingResource(): Nullable<NativeTexture>;
        constructor(existingTexture: NativeTexture, engine: INativeEngine);
        setUsage(): void;
        set(hardwareTexture: NativeTexture): void;
        reset(): void;
        release(): void;
    }


    export function getNativeTextureFormat(format: number, type: number): number;
    export function getNativeSamplingMode(samplingMode: number): number;
    export function getNativeAddressMode(wrapMode: number): number;
    export function getNativeStencilFunc(func: number): number;
    export function getNativeStencilOpFail(opFail: number): number;
    export function getNativeStencilDepthFail(depthFail: number): number;
    export function getNativeStencilDepthPass(opPass: number): number;
    export function getNativeAlphaMode(mode: number): number;
    export function getNativeAttribType(type: number): number;


    export type NativeTexture = NativeData;
    export type NativeFramebuffer = NativeData;
    export type NativeVertexArrayObject = NativeData;
    export type NativeProgram = NativeData;
    export type NativeUniform = NativeData;
    /** @internal */
    export interface INativeEngine {
        dispose(): void;
        requestAnimationFrame(callback: () => void): void;
        createVertexArray(): NativeData;
        createIndexBuffer(bytes: ArrayBuffer, byteOffset: number, byteLength: number, is32Bits: boolean, dynamic: boolean): NativeData;
        recordIndexBuffer(vertexArray: NativeData, indexBuffer: NativeData): void;
        updateDynamicIndexBuffer(buffer: NativeData, bytes: ArrayBuffer, byteOffset: number, byteLength: number, startIndex: number): void;
        createVertexBuffer(bytes: ArrayBuffer, byteOffset: number, byteLength: number, dynamic: boolean): NativeData;
        recordVertexBuffer(vertexArray: NativeData, vertexBuffer: NativeData, location: number, byteOffset: number, byteStride: number, numElements: number, type: number, normalized: boolean, instanceDivisor: number): void;
        updateDynamicVertexBuffer(vertexBuffer: NativeData, bytes: ArrayBuffer, byteOffset: number, byteLength: number): void;
        createProgram(vertexShader: string, fragmentShader: string): NativeProgram;
        createProgramAsync(vertexShader: string, fragmentShader: string, onSuccess: () => void, onError: (error: Error) => void): NativeProgram;
        getUniforms(shaderProgram: NativeProgram, uniformsNames: string[]): WebGLUniformLocation[];
        getAttributes(shaderProgram: NativeProgram, attributeNames: string[]): number[];
        createTexture(): NativeTexture;
        initializeTexture(texture: NativeTexture, width: number, height: number, hasMips: boolean, format: number, renderTarget: boolean, srgb: boolean, samples: number): void;
        loadTexture(texture: NativeTexture, data: ArrayBufferView, generateMips: boolean, invertY: boolean, srgb: boolean, onSuccess: () => void, onError: () => void): void;
        loadRawTexture(texture: NativeTexture, data: ArrayBufferView, width: number, height: number, format: number, generateMips: boolean, invertY: boolean): void;
        loadRawTexture2DArray(texture: NativeTexture, data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean): void;
        loadCubeTexture(texture: NativeTexture, data: Array<ArrayBufferView>, generateMips: boolean, invertY: boolean, srgb: boolean, onSuccess: () => void, onError: () => void): void;
        loadCubeTextureWithMips(texture: NativeTexture, data: Array<Array<ArrayBufferView>>, invertY: boolean, srgb: boolean, onSuccess: () => void, onError: () => void): void;
        getTextureWidth(texture: NativeTexture): number;
        getTextureHeight(texture: NativeTexture): number;
        copyTexture(desination: NativeTexture, source: NativeTexture): void;
        deleteTexture(texture: NativeTexture): void;
        readTexture(texture: NativeTexture, mipLevel: number, x: number, y: number, width: number, height: number, buffer: Nullable<ArrayBuffer>, bufferOffset: number, bufferLength: number): Promise<ArrayBuffer>;
        createImageBitmap(data: ArrayBufferView | IImage): ImageBitmap;
        resizeImageBitmap(image: ImageBitmap, bufferWidth: number, bufferHeight: number): Uint8Array;
        createFrameBuffer(texture: Nullable<NativeTexture>, width: number, height: number, generateStencilBuffer: boolean, generateDepthBuffer: boolean, samples: number): NativeFramebuffer;
        getRenderWidth(): number;
        getRenderHeight(): number;
        setHardwareScalingLevel(level: number): void;
        setViewPort(x: number, y: number, width: number, height: number): void;
        setCommandDataStream(dataStream: NativeDataStream): void;
        submitCommands(): void;
    }
    /** @internal */
    interface INativeEngineConstructor {
        prototype: INativeEngine;
        new (): INativeEngine;
        readonly PROTOCOL_VERSION: number;
        readonly CAPS_LIMITS_MAX_TEXTURE_SIZE: number;
        readonly CAPS_LIMITS_MAX_TEXTURE_LAYERS: number;
        readonly TEXTURE_NEAREST_NEAREST: number;
        readonly TEXTURE_LINEAR_LINEAR: number;
        readonly TEXTURE_LINEAR_LINEAR_MIPLINEAR: number;
        readonly TEXTURE_NEAREST_NEAREST_MIPNEAREST: number;
        readonly TEXTURE_NEAREST_LINEAR_MIPNEAREST: number;
        readonly TEXTURE_NEAREST_LINEAR_MIPLINEAR: number;
        readonly TEXTURE_NEAREST_LINEAR: number;
        readonly TEXTURE_NEAREST_NEAREST_MIPLINEAR: number;
        readonly TEXTURE_LINEAR_NEAREST_MIPNEAREST: number;
        readonly TEXTURE_LINEAR_NEAREST_MIPLINEAR: number;
        readonly TEXTURE_LINEAR_LINEAR_MIPNEAREST: number;
        readonly TEXTURE_LINEAR_NEAREST: number;
        readonly DEPTH_TEST_LESS: number;
        readonly DEPTH_TEST_LEQUAL: number;
        readonly DEPTH_TEST_EQUAL: number;
        readonly DEPTH_TEST_GEQUAL: number;
        readonly DEPTH_TEST_GREATER: number;
        readonly DEPTH_TEST_NOTEQUAL: number;
        readonly DEPTH_TEST_NEVER: number;
        readonly DEPTH_TEST_ALWAYS: number;
        readonly ADDRESS_MODE_WRAP: number;
        readonly ADDRESS_MODE_MIRROR: number;
        readonly ADDRESS_MODE_CLAMP: number;
        readonly ADDRESS_MODE_BORDER: number;
        readonly ADDRESS_MODE_MIRROR_ONCE: number;
        readonly TEXTURE_FORMAT_BC1: number;
        readonly TEXTURE_FORMAT_BC2: number;
        readonly TEXTURE_FORMAT_BC3: number;
        readonly TEXTURE_FORMAT_BC4: number;
        readonly TEXTURE_FORMAT_BC5: number;
        readonly TEXTURE_FORMAT_BC6H: number;
        readonly TEXTURE_FORMAT_BC7: number;
        readonly TEXTURE_FORMAT_ETC1: number;
        readonly TEXTURE_FORMAT_ETC2: number;
        readonly TEXTURE_FORMAT_ETC2A: number;
        readonly TEXTURE_FORMAT_ETC2A1: number;
        readonly TEXTURE_FORMAT_PTC12: number;
        readonly TEXTURE_FORMAT_PTC14: number;
        readonly TEXTURE_FORMAT_PTC12A: number;
        readonly TEXTURE_FORMAT_PTC14A: number;
        readonly TEXTURE_FORMAT_PTC22: number;
        readonly TEXTURE_FORMAT_PTC24: number;
        readonly TEXTURE_FORMAT_ATC: number;
        readonly TEXTURE_FORMAT_ATCE: number;
        readonly TEXTURE_FORMAT_ATCI: number;
        readonly TEXTURE_FORMAT_ASTC4x4: number;
        readonly TEXTURE_FORMAT_ASTC5x4: number;
        readonly TEXTURE_FORMAT_ASTC5x5: number;
        readonly TEXTURE_FORMAT_ASTC6x5: number;
        readonly TEXTURE_FORMAT_ASTC6x6: number;
        readonly TEXTURE_FORMAT_ASTC8x5: number;
        readonly TEXTURE_FORMAT_ASTC8x6: number;
        readonly TEXTURE_FORMAT_ASTC8x8: number;
        readonly TEXTURE_FORMAT_ASTC10x5: number;
        readonly TEXTURE_FORMAT_ASTC10x6: number;
        readonly TEXTURE_FORMAT_ASTC10x8: number;
        readonly TEXTURE_FORMAT_ASTC10x10: number;
        readonly TEXTURE_FORMAT_ASTC12x10: number;
        readonly TEXTURE_FORMAT_ASTC12x12: number;
        readonly TEXTURE_FORMAT_R1: number;
        readonly TEXTURE_FORMAT_A8: number;
        readonly TEXTURE_FORMAT_R8: number;
        readonly TEXTURE_FORMAT_R8I: number;
        readonly TEXTURE_FORMAT_R8U: number;
        readonly TEXTURE_FORMAT_R8S: number;
        readonly TEXTURE_FORMAT_R16: number;
        readonly TEXTURE_FORMAT_R16I: number;
        readonly TEXTURE_FORMAT_R16U: number;
        readonly TEXTURE_FORMAT_R16F: number;
        readonly TEXTURE_FORMAT_R16S: number;
        readonly TEXTURE_FORMAT_R32I: number;
        readonly TEXTURE_FORMAT_R32U: number;
        readonly TEXTURE_FORMAT_R32F: number;
        readonly TEXTURE_FORMAT_RG8: number;
        readonly TEXTURE_FORMAT_RG8I: number;
        readonly TEXTURE_FORMAT_RG8U: number;
        readonly TEXTURE_FORMAT_RG8S: number;
        readonly TEXTURE_FORMAT_RG16: number;
        readonly TEXTURE_FORMAT_RG16I: number;
        readonly TEXTURE_FORMAT_RG16U: number;
        readonly TEXTURE_FORMAT_RG16F: number;
        readonly TEXTURE_FORMAT_RG16S: number;
        readonly TEXTURE_FORMAT_RG32I: number;
        readonly TEXTURE_FORMAT_RG32U: number;
        readonly TEXTURE_FORMAT_RG32F: number;
        readonly TEXTURE_FORMAT_RGB8: number;
        readonly TEXTURE_FORMAT_RGB8I: number;
        readonly TEXTURE_FORMAT_RGB8U: number;
        readonly TEXTURE_FORMAT_RGB8S: number;
        readonly TEXTURE_FORMAT_RGB9E5F: number;
        readonly TEXTURE_FORMAT_BGRA8: number;
        readonly TEXTURE_FORMAT_RGBA8: number;
        readonly TEXTURE_FORMAT_RGBA8I: number;
        readonly TEXTURE_FORMAT_RGBA8U: number;
        readonly TEXTURE_FORMAT_RGBA8S: number;
        readonly TEXTURE_FORMAT_RGBA16: number;
        readonly TEXTURE_FORMAT_RGBA16I: number;
        readonly TEXTURE_FORMAT_RGBA16U: number;
        readonly TEXTURE_FORMAT_RGBA16F: number;
        readonly TEXTURE_FORMAT_RGBA16S: number;
        readonly TEXTURE_FORMAT_RGBA32I: number;
        readonly TEXTURE_FORMAT_RGBA32U: number;
        readonly TEXTURE_FORMAT_RGBA32F: number;
        readonly TEXTURE_FORMAT_B5G6R5: number;
        readonly TEXTURE_FORMAT_R5G6B5: number;
        readonly TEXTURE_FORMAT_BGRA4: number;
        readonly TEXTURE_FORMAT_RGBA4: number;
        readonly TEXTURE_FORMAT_BGR5A1: number;
        readonly TEXTURE_FORMAT_RGB5A1: number;
        readonly TEXTURE_FORMAT_RGB10A2: number;
        readonly TEXTURE_FORMAT_RG11B10F: number;
        readonly TEXTURE_FORMAT_D16: number;
        readonly TEXTURE_FORMAT_D24: number;
        readonly TEXTURE_FORMAT_D24S8: number;
        readonly TEXTURE_FORMAT_D32: number;
        readonly TEXTURE_FORMAT_D16F: number;
        readonly TEXTURE_FORMAT_D24F: number;
        readonly TEXTURE_FORMAT_D32F: number;
        readonly TEXTURE_FORMAT_D0S8: number;
        readonly ATTRIB_TYPE_INT8: number;
        readonly ATTRIB_TYPE_UINT8: number;
        readonly ATTRIB_TYPE_INT16: number;
        readonly ATTRIB_TYPE_UINT16: number;
        readonly ATTRIB_TYPE_FLOAT: number;
        readonly ALPHA_DISABLE: number;
        readonly ALPHA_ADD: number;
        readonly ALPHA_COMBINE: number;
        readonly ALPHA_SUBTRACT: number;
        readonly ALPHA_MULTIPLY: number;
        readonly ALPHA_MAXIMIZED: number;
        readonly ALPHA_ONEONE: number;
        readonly ALPHA_PREMULTIPLIED: number;
        readonly ALPHA_PREMULTIPLIED_PORTERDUFF: number;
        readonly ALPHA_INTERPOLATE: number;
        readonly ALPHA_SCREENMODE: number;
        readonly STENCIL_TEST_LESS: number;
        readonly STENCIL_TEST_LEQUAL: number;
        readonly STENCIL_TEST_EQUAL: number;
        readonly STENCIL_TEST_GEQUAL: number;
        readonly STENCIL_TEST_GREATER: number;
        readonly STENCIL_TEST_NOTEQUAL: number;
        readonly STENCIL_TEST_NEVER: number;
        readonly STENCIL_TEST_ALWAYS: number;
        readonly STENCIL_OP_FAIL_S_ZERO: number;
        readonly STENCIL_OP_FAIL_S_KEEP: number;
        readonly STENCIL_OP_FAIL_S_REPLACE: number;
        readonly STENCIL_OP_FAIL_S_INCR: number;
        readonly STENCIL_OP_FAIL_S_INCRSAT: number;
        readonly STENCIL_OP_FAIL_S_DECR: number;
        readonly STENCIL_OP_FAIL_S_DECRSAT: number;
        readonly STENCIL_OP_FAIL_S_INVERT: number;
        readonly STENCIL_OP_FAIL_Z_ZERO: number;
        readonly STENCIL_OP_FAIL_Z_KEEP: number;
        readonly STENCIL_OP_FAIL_Z_REPLACE: number;
        readonly STENCIL_OP_FAIL_Z_INCR: number;
        readonly STENCIL_OP_FAIL_Z_INCRSAT: number;
        readonly STENCIL_OP_FAIL_Z_DECR: number;
        readonly STENCIL_OP_FAIL_Z_DECRSAT: number;
        readonly STENCIL_OP_FAIL_Z_INVERT: number;
        readonly STENCIL_OP_PASS_Z_ZERO: number;
        readonly STENCIL_OP_PASS_Z_KEEP: number;
        readonly STENCIL_OP_PASS_Z_REPLACE: number;
        readonly STENCIL_OP_PASS_Z_INCR: number;
        readonly STENCIL_OP_PASS_Z_INCRSAT: number;
        readonly STENCIL_OP_PASS_Z_DECR: number;
        readonly STENCIL_OP_PASS_Z_DECRSAT: number;
        readonly STENCIL_OP_PASS_Z_INVERT: number;
        readonly COMMAND_DELETEVERTEXARRAY: NativeData;
        readonly COMMAND_DELETEINDEXBUFFER: NativeData;
        readonly COMMAND_DELETEVERTEXBUFFER: NativeData;
        readonly COMMAND_SETPROGRAM: NativeData;
        readonly COMMAND_SETMATRIX: NativeData;
        readonly COMMAND_SETMATRIX3X3: NativeData;
        readonly COMMAND_SETMATRIX2X2: NativeData;
        readonly COMMAND_SETMATRICES: NativeData;
        readonly COMMAND_SETINT: NativeData;
        readonly COMMAND_SETINTARRAY: NativeData;
        readonly COMMAND_SETINTARRAY2: NativeData;
        readonly COMMAND_SETINTARRAY3: NativeData;
        readonly COMMAND_SETINTARRAY4: NativeData;
        readonly COMMAND_SETFLOATARRAY: NativeData;
        readonly COMMAND_SETFLOATARRAY2: NativeData;
        readonly COMMAND_SETFLOATARRAY3: NativeData;
        readonly COMMAND_SETFLOATARRAY4: NativeData;
        readonly COMMAND_SETTEXTURESAMPLING: NativeData;
        readonly COMMAND_SETTEXTUREWRAPMODE: NativeData;
        readonly COMMAND_SETTEXTUREANISOTROPICLEVEL: NativeData;
        readonly COMMAND_SETTEXTURE: NativeData;
        readonly COMMAND_BINDVERTEXARRAY: NativeData;
        readonly COMMAND_SETSTATE: NativeData;
        readonly COMMAND_DELETEPROGRAM: NativeData;
        readonly COMMAND_SETZOFFSET: NativeData;
        readonly COMMAND_SETZOFFSETUNITS: NativeData;
        readonly COMMAND_SETDEPTHTEST: NativeData;
        readonly COMMAND_SETDEPTHWRITE: NativeData;
        readonly COMMAND_SETCOLORWRITE: NativeData;
        readonly COMMAND_SETBLENDMODE: NativeData;
        readonly COMMAND_SETFLOAT: NativeData;
        readonly COMMAND_SETFLOAT2: NativeData;
        readonly COMMAND_SETFLOAT3: NativeData;
        readonly COMMAND_SETFLOAT4: NativeData;
        readonly COMMAND_BINDFRAMEBUFFER: NativeData;
        readonly COMMAND_UNBINDFRAMEBUFFER: NativeData;
        readonly COMMAND_DELETEFRAMEBUFFER: NativeData;
        readonly COMMAND_DRAWINDEXED: NativeData;
        readonly COMMAND_DRAWINDEXEDINSTANCED?: NativeData;
        readonly COMMAND_DRAW: NativeData;
        readonly COMMAND_DRAWINSTANCED?: NativeData;
        readonly COMMAND_CLEAR: NativeData;
        readonly COMMAND_SETSTENCIL: NativeData;
        readonly COMMAND_SETVIEWPORT: NativeData;
        readonly COMMAND_SETSCISSOR: NativeData;
    }
    /** @internal */
    export interface INativeCamera {
        createVideo(constraints: MediaTrackConstraints): any;
        updateVideoTexture(texture: Nullable<InternalTexture>, video: HTMLVideoElement, invertY: boolean): void;
    }
    /** @internal */
    interface INativeCameraConstructor {
        prototype: INativeCamera;
        new (): INativeCamera;
    }
    /** @internal */
    interface INativeCanvasConstructor {
        prototype: ICanvas;
        new (): ICanvas;
        loadTTFAsync(fontName: string, buffer: ArrayBuffer): void;
    }
    /** @internal */
    interface INativeImageConstructor {
        prototype: IImage;
        new (): IImage;
    }
    /** @internal */
    interface IDeviceInputSystemConstructor {
        prototype: IDeviceInputSystem;
        new (onDeviceConnected: (deviceType: DeviceType, deviceSlot: number) => void, onDeviceDisconnected: (deviceType: DeviceType, deviceSlot: number) => void, onInputChanged: (deviceType: DeviceType, deviceSlot: number, inputIndex: number, currentState: number) => void): IDeviceInputSystem;
    }
    /** @internal */
    export interface INativeDataStream {
        writeBuffer(buffer: ArrayBuffer, length: number): void;
    }
    /** @internal */
    interface INativeDataStreamConstructor {
        prototype: INativeDataStream;
        new (requestFlushCallback: () => void): INativeDataStream;
        readonly VALIDATION_ENABLED: boolean;
        readonly VALIDATION_UINT_32: number;
        readonly VALIDATION_INT_32: number;
        readonly VALIDATION_FLOAT_32: number;
        readonly VALIDATION_UINT_32_ARRAY: number;
        readonly VALIDATION_INT_32_ARRAY: number;
        readonly VALIDATION_FLOAT_32_ARRAY: number;
        readonly VALIDATION_NATIVE_DATA: number;
        readonly VALIDATION_BOOLEAN: number;
    }
    /** @internal */
    export interface INative {
        Engine: INativeEngineConstructor;
        Camera: INativeCameraConstructor;
        Canvas: INativeCanvasConstructor;
        Image: INativeImageConstructor;
        XMLHttpRequest: any;
        DeviceInputSystem: IDeviceInputSystemConstructor;
        NativeDataStream: INativeDataStreamConstructor;
    }


    export class NativePipelineContext implements IPipelineContext {
        isCompiled: boolean;
        compilationError?: Error;
        readonly isAsync: boolean;
        program: NativeProgram;
        get isReady(): boolean;
        onCompiled?: () => void;
        _getVertexShaderCode(): string | null;
        _getFragmentShaderCode(): string | null;
        private _engine;
        private _valueCache;
        private _uniforms;
        constructor(engine: NativeEngine, isAsync: boolean);
        _fillEffectInformation(effect: Effect, uniformBuffersNames: {
            [key: string]: number;
        }, uniformsNames: string[], uniforms: {
            [key: string]: Nullable<WebGLUniformLocation>;
        }, samplerList: string[], samplers: {
            [key: string]: number;
        }, attributesNames: string[], attributes: number[]): void;
        /**
         * Release all associated resources.
         **/
        dispose(): void;
        /**
         * @internal
         */
        _cacheMatrix(uniformName: string, matrix: IMatrixLike): boolean;
        /**
         * @internal
         */
        _cacheFloat2(uniformName: string, x: number, y: number): boolean;
        /**
         * @internal
         */
        _cacheFloat3(uniformName: string, x: number, y: number, z: number): boolean;
        /**
         * @internal
         */
        _cacheFloat4(uniformName: string, x: number, y: number, z: number, w: number): boolean;
        /**
         * Sets an integer value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value Value to be set.
         */
        setInt(uniformName: string, value: number): void;
        /**
         * Sets a int2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First int in int2.
         * @param y Second int in int2.
         */
        setInt2(uniformName: string, x: number, y: number): void;
        /**
         * Sets a int3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First int in int3.
         * @param y Second int in int3.
         * @param z Third int in int3.
         */
        setInt3(uniformName: string, x: number, y: number, z: number): void;
        /**
         * Sets a int4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First int in int4.
         * @param y Second int in int4.
         * @param z Third int in int4.
         * @param w Fourth int in int4.
         */
        setInt4(uniformName: string, x: number, y: number, z: number, w: number): void;
        /**
         * Sets an int array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray(uniformName: string, array: Int32Array): void;
        /**
         * Sets an int array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray2(uniformName: string, array: Int32Array): void;
        /**
         * Sets an int array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray3(uniformName: string, array: Int32Array): void;
        /**
         * Sets an int array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setIntArray4(uniformName: string, array: Int32Array): void;
        /**
         * Sets an unsigned integer value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value Value to be set.
         */
        setUInt(uniformName: string, value: number): void;
        /**
         * Sets a unsigned int2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First unsigned int in uint2.
         * @param y Second unsigned int in uint2.
         */
        setUInt2(uniformName: string, x: number, y: number): void;
        /**
         * Sets a unsigned int3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First unsigned int in uint3.
         * @param y Second unsigned int in uint3.
         * @param z Third unsigned int in uint3.
         */
        setUInt3(uniformName: string, x: number, y: number, z: number): void;
        /**
         * Sets a unsigned int4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First unsigned int in uint4.
         * @param y Second unsigned int in uint4.
         * @param z Third unsigned int in uint4.
         * @param w Fourth unsigned int in uint4.
         */
        setUInt4(uniformName: string, x: number, y: number, z: number, w: number): void;
        /**
         * Sets an unsigned int array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an unsigned int array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray2(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an unsigned int array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray3(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an unsigned int array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setUIntArray4(uniformName: string, array: Uint32Array): void;
        /**
         * Sets an float array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setFloatArray(uniformName: string, array: Float32Array): void;
        /**
         * Sets an float array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setFloatArray2(uniformName: string, array: Float32Array): void;
        /**
         * Sets an float array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setFloatArray3(uniformName: string, array: Float32Array): void;
        /**
         * Sets an float array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setFloatArray4(uniformName: string, array: Float32Array): void;
        /**
         * Sets an array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray(uniformName: string, array: number[]): void;
        /**
         * Sets an array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray2(uniformName: string, array: number[]): void;
        /**
         * Sets an array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray3(uniformName: string, array: number[]): void;
        /**
         * Sets an array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         */
        setArray4(uniformName: string, array: number[]): void;
        /**
         * Sets matrices on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrices matrices to be set.
         */
        setMatrices(uniformName: string, matrices: Float32Array): void;
        /**
         * Sets matrix on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         */
        setMatrix(uniformName: string, matrix: IMatrixLike): void;
        /**
         * Sets a 3x3 matrix on a uniform variable. (Specified as [1,2,3,4,5,6,7,8,9] will result in [1,2,3][4,5,6][7,8,9] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         */
        setMatrix3x3(uniformName: string, matrix: Float32Array): void;
        /**
         * Sets a 2x2 matrix on a uniform variable. (Specified as [1,2,3,4] will result in [1,2][3,4] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         */
        setMatrix2x2(uniformName: string, matrix: Float32Array): void;
        /**
         * Sets a float on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value value to be set.
         */
        setFloat(uniformName: string, value: number): void;
        /**
         * Sets a boolean on a uniform variable.
         * @param uniformName Name of the variable.
         * @param bool value to be set.
         */
        setBool(uniformName: string, bool: boolean): void;
        /**
         * Sets a Vector2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector2 vector2 to be set.
         */
        setVector2(uniformName: string, vector2: IVector2Like): void;
        /**
         * Sets a float2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float2.
         * @param y Second float in float2.
         */
        setFloat2(uniformName: string, x: number, y: number): void;
        /**
         * Sets a Vector3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector3 Value to be set.
         */
        setVector3(uniformName: string, vector3: IVector3Like): void;
        /**
         * Sets a float3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float3.
         * @param y Second float in float3.
         * @param z Third float in float3.
         */
        setFloat3(uniformName: string, x: number, y: number, z: number): void;
        /**
         * Sets a Vector4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector4 Value to be set.
         */
        setVector4(uniformName: string, vector4: IVector4Like): void;
        /**
         * Sets a Quaternion on a uniform variable.
         * @param uniformName Name of the variable.
         * @param quaternion Value to be set.
         */
        setQuaternion(uniformName: string, quaternion: IQuaternionLike): void;
        /**
         * Sets a float4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float4.
         * @param y Second float in float4.
         * @param z Third float in float4.
         * @param w Fourth float in float4.
         */
        setFloat4(uniformName: string, x: number, y: number, z: number, w: number): void;
        /**
         * Sets a Color3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         */
        setColor3(uniformName: string, color3: IColor3Like): void;
        /**
         * Sets a Color4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         * @param alpha Alpha value to be set.
         */
        setColor4(uniformName: string, color3: IColor3Like, alpha: number): void;
        /**
         * Sets a Color4 on a uniform variable
         * @param uniformName defines the name of the variable
         * @param color4 defines the value to be set
         */
        setDirectColor4(uniformName: string, color4: IColor4Like): void;
    }


    export class NativeRenderTargetWrapper extends RenderTargetWrapper {
        readonly _engine: NativeEngine;
        private __framebuffer;
        private __framebufferDepthStencil;
        get _framebuffer(): Nullable<NativeFramebuffer>;
        set _framebuffer(framebuffer: Nullable<NativeFramebuffer>);
        get _framebufferDepthStencil(): Nullable<NativeFramebuffer>;
        set _framebufferDepthStencil(framebufferDepthStencil: Nullable<NativeFramebuffer>);
        constructor(isMulti: boolean, isCube: boolean, size: TextureSize, engine: NativeEngine);
        dispose(disposeOnlyFramebuffers?: boolean): void;
    }


    /**
     * Validated Native Data Stream
     */
    export class ValidatedNativeDataStream extends NativeDataStream {
        constructor();
        writeUint32(value: number): void;
        writeInt32(value: number): void;
        writeFloat32(value: number): void;
        writeUint32Array(values: Uint32Array): void;
        writeInt32Array(values: Int32Array): void;
        writeFloat32Array(values: Float32Array): void;
        writeNativeData(handle: NativeData): void;
        writeBoolean(value: boolean): void;
    }


    /**
     * Returns _native only after it has been defined by BabylonNative.
     * @internal
     */
    export function AcquireNativeObjectAsync(): Promise<INative>;
    /**
     * Registers a constructor on the _native object. See NativeXRFrame for an example.
     * @internal
     */
    export function RegisterNativeTypeAsync<Type>(typeName: string, constructor: Type): Promise<void>;
    /**
     * Container for accessors for natively-stored mesh data buffers.
     */
    class NativeDataBuffer extends DataBuffer {
        /**
         * Accessor value used to identify/retrieve a natively-stored index buffer.
         */
        nativeIndexBuffer?: NativeData;
        /**
         * Accessor value used to identify/retrieve a natively-stored vertex buffer.
         */
        nativeVertexBuffer?: NativeData;
    }
    /**
     * Options to create the Native engine
     */
    export interface NativeEngineOptions {
        /**
         * defines whether to adapt to the device's viewport characteristics (default: false)
         */
        adaptToDeviceRatio?: boolean;
    }
    /** @internal */
    export class NativeEngine extends Engine {
        private static readonly PROTOCOL_VERSION;
        private readonly _engine;
        private readonly _camera;
        private readonly _commandBufferEncoder;
        private _boundBuffersVertexArray;
        private _currentDepthTest;
        private _stencilTest;
        private _stencilMask;
        private _stencilFunc;
        private _stencilFuncRef;
        private _stencilFuncMask;
        private _stencilOpStencilFail;
        private _stencilOpDepthFail;
        private _stencilOpStencilDepthPass;
        private _zOffset;
        private _zOffsetUnits;
        private _depthWrite;
        setHardwareScalingLevel(level: number): void;
        constructor(options?: NativeEngineOptions);
        dispose(): void;
        /** @internal */
        static _createNativeDataStream(): NativeDataStream;
        /**
         * Can be used to override the current requestAnimationFrame requester.
         * @internal
         */
        protected _queueNewFrame(bindedRenderFunction: any, requester?: any): number;
        /**
         * Override default engine behavior.
         * @param framebuffer
         */
        _bindUnboundFramebuffer(framebuffer: Nullable<WebGLFramebuffer>): void;
        /**
         * Gets host document
         * @returns the host document object
         */
        getHostDocument(): Nullable<Document>;
        clear(color: Nullable<IColor4Like>, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        createIndexBuffer(indices: IndicesArray, updateable?: boolean, _label?: string): NativeDataBuffer;
        createVertexBuffer(vertices: DataArray, updateable?: boolean, _label?: string): NativeDataBuffer;
        protected _recordVertexArrayObject(vertexArray: any, vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: Nullable<NativeDataBuffer>, effect: Effect, overrideVertexBuffers?: {
            [kind: string]: Nullable<VertexBuffer>;
        }): void;
        bindBuffers(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: Nullable<NativeDataBuffer>, effect: Effect): void;
        recordVertexArrayObject(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: Nullable<NativeDataBuffer>, effect: Effect, overrideVertexBuffers?: {
            [kind: string]: Nullable<VertexBuffer>;
        }): WebGLVertexArrayObject;
        private _deleteVertexArray;
        bindVertexArrayObject(vertexArray: WebGLVertexArrayObject): void;
        releaseVertexArrayObject(vertexArray: WebGLVertexArrayObject): void;
        getAttributes(pipelineContext: IPipelineContext, attributesNames: string[]): number[];
        /**
         * Draw a list of indexed primitives
         * @param fillMode defines the primitive to use
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of unindexed primitives
         * @param fillMode defines the primitive to use
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        createPipelineContext(): IPipelineContext;
        createMaterialContext(): IMaterialContext | undefined;
        createDrawContext(): IDrawContext | undefined;
        /**
         * @internal
         */
        _preparePipelineContext(pipelineContext: IPipelineContext, vertexSourceCode: string, fragmentSourceCode: string, createAsRaw: boolean, _rawVertexSourceCode: string, _rawFragmentSourceCode: string, _rebuildRebind: any, defines: Nullable<string>): void;
        /**
         * @internal
         */
        _executeWhenRenderingStateIsCompiled(pipelineContext: IPipelineContext, action: () => void): void;
        createRawShaderProgram(): WebGLProgram;
        createShaderProgram(pipelineContext: IPipelineContext, vertexCode: string, fragmentCode: string, defines: Nullable<string>): WebGLProgram;
        /**
         * Inline functions in shader code that are marked to be inlined
         * @param code code to inline
         * @returns inlined code
         */
        inlineShaderCode(code: string): string;
        protected _setProgram(program: WebGLProgram): void;
        _deletePipelineContext(pipelineContext: IPipelineContext): void;
        getUniforms(pipelineContext: IPipelineContext, uniformsNames: string[]): WebGLUniformLocation[];
        bindUniformBlock(pipelineContext: IPipelineContext, blockName: string, index: number): void;
        bindSamplers(effect: Effect): void;
        getRenderWidth(useScreen?: boolean): number;
        getRenderHeight(useScreen?: boolean): number;
        setViewport(viewport: IViewportLike, requiredWidth?: number, requiredHeight?: number): void;
        enableScissor(x: number, y: number, width: number, height: number): void;
        disableScissor(): void;
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean, cullBackFaces?: boolean, stencil?: IStencilState, zOffsetUnits?: number): void;
        /**
         * Gets the client rect of native canvas.  Needed for InputManager.
         * @returns a client rectangle
         */
        getInputElementClientRect(): Nullable<DOMRect>;
        /**
         * Set the z offset Factor to apply to current rendering
         * @param value defines the offset to apply
         */
        setZOffset(value: number): void;
        /**
         * Gets the current value of the zOffset Factor
         * @returns the current zOffset Factor state
         */
        getZOffset(): number;
        /**
         * Set the z offset Units to apply to current rendering
         * @param value defines the offset to apply
         */
        setZOffsetUnits(value: number): void;
        /**
         * Gets the current value of the zOffset Units
         * @returns the current zOffset Units state
         */
        getZOffsetUnits(): number;
        /**
         * Enable or disable depth buffering
         * @param enable defines the state to set
         */
        setDepthBuffer(enable: boolean): void;
        /**
         * Gets a boolean indicating if depth writing is enabled
         * @returns the current depth writing state
         */
        getDepthWrite(): boolean;
        getDepthFunction(): Nullable<number>;
        setDepthFunction(depthFunc: number): void;
        /**
         * Enable or disable depth writing
         * @param enable defines the state to set
         */
        setDepthWrite(enable: boolean): void;
        /**
         * Enable or disable color writing
         * @param enable defines the state to set
         */
        setColorWrite(enable: boolean): void;
        /**
         * Gets a boolean indicating if color writing is enabled
         * @returns the current color writing state
         */
        getColorWrite(): boolean;
        private applyStencil;
        private _setStencil;
        /**
         * Enable or disable the stencil buffer
         * @param enable defines if the stencil buffer must be enabled or disabled
         */
        setStencilBuffer(enable: boolean): void;
        /**
         * Gets a boolean indicating if stencil buffer is enabled
         * @returns the current stencil buffer state
         */
        getStencilBuffer(): boolean;
        /**
         * Gets the current stencil operation when stencil passes
         * @returns a number defining stencil operation to use when stencil passes
         */
        getStencilOperationPass(): number;
        /**
         * Sets the stencil operation to use when stencil passes
         * @param operation defines the stencil operation to use when stencil passes
         */
        setStencilOperationPass(operation: number): void;
        /**
         * Sets the current stencil mask
         * @param mask defines the new stencil mask to use
         */
        setStencilMask(mask: number): void;
        /**
         * Sets the current stencil function
         * @param stencilFunc defines the new stencil function to use
         */
        setStencilFunction(stencilFunc: number): void;
        /**
         * Sets the current stencil reference
         * @param reference defines the new stencil reference to use
         */
        setStencilFunctionReference(reference: number): void;
        /**
         * Sets the current stencil mask
         * @param mask defines the new stencil mask to use
         */
        setStencilFunctionMask(mask: number): void;
        /**
         * Sets the stencil operation to use when stencil fails
         * @param operation defines the stencil operation to use when stencil fails
         */
        setStencilOperationFail(operation: number): void;
        /**
         * Sets the stencil operation to use when depth fails
         * @param operation defines the stencil operation to use when depth fails
         */
        setStencilOperationDepthFail(operation: number): void;
        /**
         * Gets the current stencil mask
         * @returns a number defining the new stencil mask to use
         */
        getStencilMask(): number;
        /**
         * Gets the current stencil function
         * @returns a number defining the stencil function to use
         */
        getStencilFunction(): number;
        /**
         * Gets the current stencil reference value
         * @returns a number defining the stencil reference value to use
         */
        getStencilFunctionReference(): number;
        /**
         * Gets the current stencil mask
         * @returns a number defining the stencil mask to use
         */
        getStencilFunctionMask(): number;
        /**
         * Gets the current stencil operation when stencil fails
         * @returns a number defining stencil operation to use when stencil fails
         */
        getStencilOperationFail(): number;
        /**
         * Gets the current stencil operation when depth fails
         * @returns a number defining stencil operation to use when depth fails
         */
        getStencilOperationDepthFail(): number;
        /**
         * Sets alpha constants used by some alpha blending modes
         * @param r defines the red component
         * @param g defines the green component
         * @param b defines the blue component
         * @param a defines the alpha component
         */
        setAlphaConstants(r: number, g: number, b: number, a: number): void;
        /**
         * Sets the current alpha mode
         * @param mode defines the mode to use (one of the BABYLON.Constants.ALPHA_XXX)
         * @param noDepthWriteChange defines if depth writing state should remains unchanged (false by default)
         * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/advanced/transparent_rendering
         */
        setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
        /**
         * Gets the current alpha mode
         * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/advanced/transparent_rendering
         * @returns the current alpha mode
         */
        getAlphaMode(): number;
        setInt(uniform: WebGLUniformLocation, int: number): boolean;
        setIntArray(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        setIntArray2(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        setIntArray3(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        setIntArray4(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        setFloatArray(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        setFloatArray2(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        setFloatArray3(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        setFloatArray4(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        setArray(uniform: WebGLUniformLocation, array: number[]): boolean;
        setArray2(uniform: WebGLUniformLocation, array: number[]): boolean;
        setArray3(uniform: WebGLUniformLocation, array: number[]): boolean;
        setArray4(uniform: WebGLUniformLocation, array: number[]): boolean;
        setMatrices(uniform: WebGLUniformLocation, matrices: Float32Array): boolean;
        setMatrix3x3(uniform: WebGLUniformLocation, matrix: Float32Array): boolean;
        setMatrix2x2(uniform: WebGLUniformLocation, matrix: Float32Array): boolean;
        setFloat(uniform: WebGLUniformLocation, value: number): boolean;
        setFloat2(uniform: WebGLUniformLocation, x: number, y: number): boolean;
        setFloat3(uniform: WebGLUniformLocation, x: number, y: number, z: number): boolean;
        setFloat4(uniform: WebGLUniformLocation, x: number, y: number, z: number, w: number): boolean;
        setColor3(uniform: WebGLUniformLocation, color3: IColor3Like): boolean;
        setColor4(uniform: WebGLUniformLocation, color3: IColor3Like, alpha: number): boolean;
        wipeCaches(bruteForce?: boolean): void;
        protected _createTexture(): WebGLTexture;
        protected _deleteTexture(texture: Nullable<WebGLTexture>): void;
        /**
         * Update the content of a dynamic texture
         * @param texture defines the texture to update
         * @param canvas defines the canvas containing the source
         * @param invertY defines if data must be stored with Y axis inverted
         * @param premulAlpha defines if alpha is stored as premultiplied
         * @param format defines the format of the data
         */
        updateDynamicTexture(texture: Nullable<InternalTexture>, canvas: any, invertY: boolean, premulAlpha?: boolean, format?: number): void;
        createDynamicTexture(width: number, height: number, generateMipMaps: boolean, samplingMode: number): InternalTexture;
        createVideoElement(constraints: MediaTrackConstraints): any;
        updateVideoTexture(texture: Nullable<InternalTexture>, video: HTMLVideoElement, invertY: boolean): void;
        createRawTexture(data: Nullable<ArrayBufferView>, width: number, height: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>, type?: number, creationFlags?: number, useSRGBBuffer?: boolean): InternalTexture;
        createRawTexture2DArray(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>, textureType?: number): InternalTexture;
        updateRawTexture(texture: Nullable<InternalTexture>, bufferView: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression?: Nullable<string>, type?: number, useSRGBBuffer?: boolean): void;
        /**
         * Usually called from Texture.ts.
         * Passed information to create a NativeTexture
         * @param url defines a value which contains one of the following:
         * * A conventional http URL, e.g. 'http://...' or 'file://...'
         * * A base64 string of in-line texture data, e.g. 'data:image/jpg;base64,/...'
         * * An indicator that data being passed using the buffer parameter, e.g. 'data:mytexture.jpg'
         * @param noMipmap defines a boolean indicating that no mipmaps shall be generated.  Ignored for compressed textures.  They must be in the file
         * @param invertY when true, image is flipped when loaded.  You probably want true. Certain compressed textures may invert this if their default is inverted (eg. ktx)
         * @param scene needed for loading to the correct scene
         * @param samplingMode mode with should be used sample / access the texture (Default: Texture.TRILINEAR_SAMPLINGMODE)
         * @param onLoad optional callback to be called upon successful completion
         * @param onError optional callback to be called upon failure
         * @param buffer a source of a file previously fetched as either a base64 string, an ArrayBuffer (compressed or image format), HTMLImageElement (image format), or a Blob
         * @param fallback an internal argument in case the function must be called again, due to etc1 not having alpha capabilities
         * @param format internal format.  Default: RGB when extension is '.jpg' else RGBA.  Ignored for compressed textures
         * @param forcedExtension defines the extension to use to pick the right loader
         * @param mimeType defines an optional mime type
         * @param loaderOptions options to be passed to the loader
         * @param creationFlags specific flags to use when creating the texture (Constants.TEXTURE_CREATIONFLAG_STORAGE for storage textures, for eg)
         * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
         * @returns a InternalTexture for assignment back into BABYLON.Texture
         */
        createTexture(url: Nullable<string>, noMipmap: boolean, invertY: boolean, scene: Nullable<ISceneLike>, samplingMode?: number, onLoad?: Nullable<(texture: InternalTexture) => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<string | ArrayBuffer | ArrayBufferView | HTMLImageElement | Blob | ImageBitmap>, fallback?: Nullable<InternalTexture>, format?: Nullable<number>, forcedExtension?: Nullable<string>, mimeType?: string, loaderOptions?: any, creationFlags?: number, useSRGBBuffer?: boolean): InternalTexture;
        /**
         * Wraps an external native texture in a Babylon texture.
         * @param texture defines the external texture
         * @param hasMipMaps defines whether the external texture has mip maps
         * @param samplingMode defines the sampling mode for the external texture (default: Constants.TEXTURE_TRILINEAR_SAMPLINGMODE)
         * @returns the babylon internal texture
         */
        wrapNativeTexture(texture: NativeTexture, hasMipMaps?: boolean, samplingMode?: number): InternalTexture;
        /**
         * Wraps an external web gl texture in a Babylon texture.
         * @returns the babylon internal texture
         */
        wrapWebGLTexture(): InternalTexture;
        _createDepthStencilTexture(size: TextureSize, options: DepthTextureCreationOptions, rtWrapper: RenderTargetWrapper): InternalTexture;
        /**
         * @internal
         */
        _releaseFramebufferObjects(framebuffer: Nullable<NativeFramebuffer>): void;
        /**
         * @internal Engine abstraction for loading and creating an image bitmap from a given source string.
         * @param imageSource source to load the image from.
         * @param options An object that sets options for the image's extraction.
         * @returns ImageBitmap
         */
        _createImageBitmapFromSource(imageSource: string, options?: ImageBitmapOptions): Promise<ImageBitmap>;
        /**
         * Engine abstraction for createImageBitmap
         * @param image source for image
         * @param options An object that sets options for the image's extraction.
         * @returns ImageBitmap
         */
        createImageBitmap(image: ImageBitmapSource, options?: ImageBitmapOptions): Promise<ImageBitmap>;
        /**
         * Resize an image and returns the image data as an uint8array
         * @param image image to resize
         * @param bufferWidth destination buffer width
         * @param bufferHeight destination buffer height
         * @returns an uint8array containing RGBA values of bufferWidth * bufferHeight size
         */
        resizeImageBitmap(image: ImageBitmap, bufferWidth: number, bufferHeight: number): Uint8Array;
        /**
         * Creates a cube texture
         * @param rootUrl defines the url where the files to load is located
         * @param scene defines the current scene
         * @param files defines the list of files to load (1 per face)
         * @param noMipmap defines a boolean indicating that no mipmaps shall be generated (false by default)
         * @param onLoad defines an optional callback raised when the texture is loaded
         * @param onError defines an optional callback raised if there is an issue to load the texture
         * @param format defines the format of the data
         * @param forcedExtension defines the extension to use to pick the right loader
         * @param createPolynomials if a polynomial sphere should be created for the cube texture
         * @param lodScale defines the scale applied to environment texture. This manages the range of LOD level used for IBL according to the roughness
         * @param lodOffset defines the offset applied to environment texture. This manages first LOD level used for IBL according to the roughness
         * @param fallback defines texture to use while falling back when (compressed) texture file not found.
         * @param loaderOptions options to be passed to the loader
         * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
         * @returns the cube texture as an InternalTexture
         */
        createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap?: boolean, onLoad?: Nullable<(data?: any) => void>, onError?: Nullable<(message?: string, exception?: any) => void>, format?: number, forcedExtension?: any, createPolynomials?: boolean, lodScale?: number, lodOffset?: number, fallback?: Nullable<InternalTexture>, loaderOptions?: any, useSRGBBuffer?: boolean): InternalTexture;
        /** @internal */
        _createHardwareTexture(): HardwareTextureWrapper;
        /** @internal */
        _createHardwareRenderTargetWrapper(isMulti: boolean, isCube: boolean, size: TextureSize): RenderTargetWrapper;
        /** @internal */
        _createInternalTexture(size: TextureSize, options: boolean | InternalTextureCreationOptions, _delayGPUTextureCreation?: boolean, source?: InternalTextureSource): InternalTexture;
        createRenderTargetTexture(size: number | {
            width: number;
            height: number;
        }, options: boolean | RenderTargetCreationOptions): RenderTargetWrapper;
        updateRenderTargetTextureSampleCount(rtWrapper: RenderTargetWrapper, samples: number): number;
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture): void;
        bindFramebuffer(texture: RenderTargetWrapper, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean): void;
        unBindFramebuffer(texture: RenderTargetWrapper, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        createDynamicVertexBuffer(data: DataArray): DataBuffer;
        updateDynamicIndexBuffer(indexBuffer: DataBuffer, indices: IndicesArray, offset?: number): void;
        updateDynamicVertexBuffer(vertexBuffer: DataBuffer, verticies: DataArray, byteOffset?: number, byteLength?: number): void;
        protected _setTexture(channel: number, texture: Nullable<BaseTexture>, isPartOfTextureArray?: boolean, depthStencilTexture?: boolean): boolean;
        private _setTextureSampling;
        private _setTextureWrapMode;
        private _setTextureCore;
        private _updateAnisotropicLevel;
        /**
         * @internal
         */
        _bindTexture(channel: number, texture: InternalTexture): void;
        protected _deleteBuffer(buffer: NativeDataBuffer): void;
        /**
         * Create a canvas
         * @param width width
         * @param height height
         * @returns ICanvas interface
         */
        createCanvas(width: number, height: number): ICanvas;
        /**
         * Create an image to use with canvas
         * @returns IImage interface
         */
        createCanvasImage(): IImage;
        /**
         * Update a portion of an internal texture
         * @param texture defines the texture to update
         * @param imageData defines the data to store into the texture
         * @param xOffset defines the x coordinates of the update rectangle
         * @param yOffset defines the y coordinates of the update rectangle
         * @param width defines the width of the update rectangle
         * @param height defines the height of the update rectangle
         * @param faceIndex defines the face index if texture is a cube (0 by default)
         * @param lod defines the lod level to update (0 by default)
         * @param generateMipMaps defines whether to generate mipmaps or not
         */
        updateTextureData(texture: InternalTexture, imageData: ArrayBufferView, xOffset: number, yOffset: number, width: number, height: number, faceIndex?: number, lod?: number, generateMipMaps?: boolean): void;
        /**
         * @internal
         */
        _uploadCompressedDataToTextureDirectly(texture: InternalTexture, internalFormat: number, width: number, height: number, data: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadDataToTextureDirectly(texture: InternalTexture, imageData: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadArrayBufferViewToTexture(texture: InternalTexture, imageData: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadImageToTexture(texture: InternalTexture, image: HTMLImageElement, faceIndex?: number, lod?: number): void;
        getFontOffset(font: string): {
            ascent: number;
            height: number;
            descent: number;
        };
        /**
         * No equivalent for native. Do nothing.
         */
        flushFramebuffer(): void;
        _readTexturePixels(texture: InternalTexture, width: number, height: number, faceIndex?: number, level?: number, buffer?: Nullable<ArrayBufferView>, _flushRenderer?: boolean, _noDataConversion?: boolean, x?: number, y?: number): Promise<ArrayBufferView>;
    }


    /**
     * Options to create the null engine
     */
    export class NullEngineOptions {
        /**
         * Render width (Default: 512)
         */
        renderWidth: number;
        /**
         * Render height (Default: 256)
         */
        renderHeight: number;
        /**
         * Texture size (Default: 512)
         */
        textureSize: number;
        /**
         * If delta time between frames should be constant
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         */
        deterministicLockstep: boolean;
        /** Defines the seconds between each deterministic lock step */
        timeStep?: number;
        /**
         * Maximum about of steps between frames (Default: 4)
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         */
        lockstepMaxSteps: number;
        /**
         * Make the matrix computations to be performed in 64 bits instead of 32 bits. False by default
         */
        useHighPrecisionMatrix?: boolean;
    }
    /**
     * The null engine class provides support for headless version of babylon.js.
     * This can be used in server side scenario or for testing purposes
     */
    export class NullEngine extends Engine {
        private _options;
        /**
         * Gets a boolean indicating that the engine is running in deterministic lock step mode
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         * @returns true if engine is in deterministic lock step mode
         */
        isDeterministicLockStep(): boolean;
        /**
         * Gets the max steps when engine is running in deterministic lock step
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         * @returns the max steps
         */
        getLockstepMaxSteps(): number;
        /**
         * Gets the current hardware scaling level.
         * By default the hardware scaling level is computed from the window device ratio.
         * if level = 1 then the engine will render at the exact resolution of the canvas. If level = 0.5 then the engine will render at twice the size of the canvas.
         * @returns a number indicating the current hardware scaling level
         */
        getHardwareScalingLevel(): number;
        constructor(options?: NullEngineOptions);
        /**
         * Creates a vertex buffer
         * @param vertices the data for the vertex buffer
         * @returns the new WebGL static buffer
         */
        createVertexBuffer(vertices: FloatArray): DataBuffer;
        /**
         * Creates a new index buffer
         * @param indices defines the content of the index buffer
         * @returns a new webGL buffer
         */
        createIndexBuffer(indices: IndicesArray): DataBuffer;
        /**
         * Clear the current render buffer or the current render target (if any is set up)
         * @param color defines the color to use
         * @param backBuffer defines if the back buffer must be cleared
         * @param depth defines if the depth buffer must be cleared
         * @param stencil defines if the stencil buffer must be cleared
         */
        clear(color: IColor4Like, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        /**
         * Gets the current render width
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the current render width
         */
        getRenderWidth(useScreen?: boolean): number;
        /**
         * Gets the current render height
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the current render height
         */
        getRenderHeight(useScreen?: boolean): number;
        /**
         * Set the WebGL's viewport
         * @param viewport defines the viewport element to be used
         * @param requiredWidth defines the width required for rendering. If not provided the rendering canvas' width is used
         * @param requiredHeight defines the height required for rendering. If not provided the rendering canvas' height is used
         */
        setViewport(viewport: IViewportLike, requiredWidth?: number, requiredHeight?: number): void;
        createShaderProgram(pipelineContext: IPipelineContext, vertexCode: string, fragmentCode: string, defines: string, context?: WebGLRenderingContext): WebGLProgram;
        /**
         * Gets the list of webGL uniform locations associated with a specific program based on a list of uniform names
         * @param pipelineContext defines the pipeline context to use
         * @param uniformsNames defines the list of uniform names
         * @returns an array of webGL uniform locations
         */
        getUniforms(pipelineContext: IPipelineContext, uniformsNames: string[]): Nullable<WebGLUniformLocation>[];
        /**
         * Gets the lsit of active attributes for a given webGL program
         * @param pipelineContext defines the pipeline context to use
         * @param attributesNames defines the list of attribute names to get
         * @returns an array of indices indicating the offset of each attribute
         */
        getAttributes(pipelineContext: IPipelineContext, attributesNames: string[]): number[];
        /**
         * Binds an effect to the webGL context
         * @param effect defines the effect to bind
         */
        bindSamplers(effect: Effect): void;
        /**
         * Activates an effect, making it the current one (ie. the one used for rendering)
         * @param effect defines the effect to activate
         */
        enableEffect(effect: Nullable<Effect | DrawWrapper>): void;
        /**
         * Set various states to the webGL context
         * @param culling defines culling state: true to enable culling, false to disable it
         * @param zOffset defines the value to apply to zOffset (0 by default)
         * @param force defines if states must be applied even if cache is up to date
         * @param reverseSide defines if culling must be reversed (CCW if false, CW if true)
         * @param cullBackFaces true to cull back faces, false to cull front faces (if culling is enabled)
         * @param stencil stencil states to set
         * @param zOffsetUnits defines the value to apply to zOffsetUnits (0 by default)
         */
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean, cullBackFaces?: boolean, stencil?: IStencilState, zOffsetUnits?: number): void;
        /**
         * Set the value of an uniform to an array of int32
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if value was set
         */
        setIntArray(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if value was set
         */
        setIntArray2(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if value was set
         */
        setIntArray3(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if value was set
         */
        setIntArray4(uniform: WebGLUniformLocation, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of float32
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         * @returns true if value was set
         */
        setFloatArray(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of float32 (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         * @returns true if value was set
         */
        setFloatArray2(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of float32 (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         * @returns true if value was set
         */
        setFloatArray3(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of float32 (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of float32 to store
         * @returns true if value was set
         */
        setFloatArray4(uniform: WebGLUniformLocation, array: Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of number
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if value was set
         */
        setArray(uniform: WebGLUniformLocation, array: number[]): boolean;
        /**
         * Set the value of an uniform to an array of number (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if value was set
         */
        setArray2(uniform: WebGLUniformLocation, array: number[]): boolean;
        /**
         * Set the value of an uniform to an array of number (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if value was set
         */
        setArray3(uniform: WebGLUniformLocation, array: number[]): boolean;
        /**
         * Set the value of an uniform to an array of number (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if value was set
         */
        setArray4(uniform: WebGLUniformLocation, array: number[]): boolean;
        /**
         * Set the value of an uniform to an array of float32 (stored as matrices)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrices defines the array of float32 to store
         * @returns true if value was set
         */
        setMatrices(uniform: WebGLUniformLocation, matrices: Float32Array): boolean;
        /**
         * Set the value of an uniform to a matrix (3x3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the Float32Array representing the 3x3 matrix to store
         * @returns true if value was set
         */
        setMatrix3x3(uniform: WebGLUniformLocation, matrix: Float32Array): boolean;
        /**
         * Set the value of an uniform to a matrix (2x2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the Float32Array representing the 2x2 matrix to store
         * @returns true if value was set
         */
        setMatrix2x2(uniform: WebGLUniformLocation, matrix: Float32Array): boolean;
        /**
         * Set the value of an uniform to a number (float)
         * @param uniform defines the webGL uniform location where to store the value
         * @param value defines the float number to store
         * @returns true if value was set
         */
        setFloat(uniform: WebGLUniformLocation, value: number): boolean;
        /**
         * Set the value of an uniform to a vec2
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @returns true if value was set
         */
        setFloat2(uniform: WebGLUniformLocation, x: number, y: number): boolean;
        /**
         * Set the value of an uniform to a vec3
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @returns true if value was set
         */
        setFloat3(uniform: WebGLUniformLocation, x: number, y: number, z: number): boolean;
        /**
         * Set the value of an uniform to a boolean
         * @param uniform defines the webGL uniform location where to store the value
         * @param bool defines the boolean to store
         * @returns true if value was set
         */
        setBool(uniform: WebGLUniformLocation, bool: number): boolean;
        /**
         * Set the value of an uniform to a vec4
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @param w defines the 4th component of the value
         * @returns true if value was set
         */
        setFloat4(uniform: WebGLUniformLocation, x: number, y: number, z: number, w: number): boolean;
        /**
         * Sets the current alpha mode
         * @param mode defines the mode to use (one of the Engine.ALPHA_XXX)
         * @param noDepthWriteChange defines if depth writing state should remains unchanged (false by default)
         * @see https://doc.babylonjs.com/features/featuresDeepDive/materials/advanced/transparent_rendering
         */
        setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
        /**
         * Bind webGl buffers directly to the webGL context
         * @param vertexBuffers defines the vertex buffer to bind
         * @param indexBuffer defines the index buffer to bind
         * @param effect defines the effect associated with the vertex buffer
         */
        bindBuffers(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: DataBuffer, effect: Effect): void;
        /**
         * Force the entire cache to be cleared
         * You should not have to use this function unless your engine needs to share the webGL context with another engine
         * @param bruteForce defines a boolean to force clearing ALL caches (including stencil, detoh and alpha states)
         */
        wipeCaches(bruteForce?: boolean): void;
        /**
         * Send a draw order
         * @param useTriangles defines if triangles must be used to draw (else wireframe will be used)
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        draw(useTriangles: boolean, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of indexed primitives
         * @param fillMode defines the primitive to use
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of unindexed primitives
         * @param fillMode defines the primitive to use
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        /** @internal */
        protected _createTexture(): WebGLTexture;
        /**
         * @internal
         */
        _releaseTexture(texture: InternalTexture): void;
        /**
         * Usually called from Texture.ts.
         * Passed information to create a WebGLTexture
         * @param urlArg defines a value which contains one of the following:
         * * A conventional http URL, e.g. 'http://...' or 'file://...'
         * * A base64 string of in-line texture data, e.g. 'data:image/jpg;base64,/...'
         * * An indicator that data being passed using the buffer parameter, e.g. 'data:mytexture.jpg'
         * @param noMipmap defines a boolean indicating that no mipmaps shall be generated.  Ignored for compressed textures.  They must be in the file
         * @param invertY when true, image is flipped when loaded.  You probably want true. Certain compressed textures may invert this if their default is inverted (eg. ktx)
         * @param scene needed for loading to the correct scene
         * @param samplingMode mode with should be used sample / access the texture (Default: Texture.TRILINEAR_SAMPLINGMODE)
         * @param onLoad optional callback to be called upon successful completion
         * @param onError optional callback to be called upon failure
         * @param buffer a source of a file previously fetched as either a base64 string, an ArrayBuffer (compressed or image format), HTMLImageElement (image format), or a Blob
         * @param fallback an internal argument in case the function must be called again, due to etc1 not having alpha capabilities
         * @param format internal format.  Default: RGB when extension is '.jpg' else RGBA.  Ignored for compressed textures
         * @param forcedExtension defines the extension to use to pick the right loader
         * @param mimeType defines an optional mime type
         * @returns a InternalTexture for assignment back into BABYLON.Texture
         */
        createTexture(urlArg: Nullable<string>, noMipmap: boolean, invertY: boolean, scene: Nullable<ISceneLike>, samplingMode?: number, onLoad?: Nullable<(texture: InternalTexture) => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<string | ArrayBuffer | ArrayBufferView | HTMLImageElement | Blob | ImageBitmap>, fallback?: Nullable<InternalTexture>, format?: Nullable<number>, forcedExtension?: Nullable<string>, mimeType?: string): InternalTexture;
        /**
         * @internal
         */
        _createHardwareRenderTargetWrapper(isMulti: boolean, isCube: boolean, size: number | {
            width: number;
            height: number;
            layers?: number;
        }): RenderTargetWrapper;
        /**
         * Creates a new render target wrapper
         * @param size defines the size of the texture
         * @param options defines the options used to create the texture
         * @returns a new render target wrapper
         */
        createRenderTargetTexture(size: any, options: boolean | RenderTargetCreationOptions): RenderTargetWrapper;
        /**
         * Creates a new render target wrapper
         * @param size defines the size of the texture
         * @param options defines the options used to create the texture
         * @returns a new render target wrapper
         */
        createRenderTargetCubeTexture(size: number, options?: RenderTargetCreationOptions): RenderTargetWrapper;
        /**
         * Update the sampling mode of a given texture
         * @param samplingMode defines the required sampling mode
         * @param texture defines the texture to update
         */
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture): void;
        /**
         * Creates a raw texture
         * @param data defines the data to store in the texture
         * @param width defines the width of the texture
         * @param height defines the height of the texture
         * @param format defines the format of the data
         * @param generateMipMaps defines if the engine should generate the mip levels
         * @param invertY defines if data must be stored with Y axis inverted
         * @param samplingMode defines the required sampling mode (Texture.NEAREST_SAMPLINGMODE by default)
         * @param compression defines the compression used (null by default)
         * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
         * @param creationFlags specific flags to use when creating the texture (Constants.TEXTURE_CREATIONFLAG_STORAGE for storage textures, for eg)
         * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
         * @returns the raw texture inside an InternalTexture
         */
        createRawTexture(data: Nullable<ArrayBufferView>, width: number, height: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>, type?: number, creationFlags?: number, useSRGBBuffer?: boolean): InternalTexture;
        /**
         * Update a raw texture
         * @param texture defines the texture to update
         * @param data defines the data to store in the texture
         * @param format defines the format of the data
         * @param invertY defines if data must be stored with Y axis inverted
         * @param compression defines the compression used (null by default)
         * @param type defines the type fo the data (Engine.TEXTURETYPE_UNSIGNED_INT by default)
         * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
         */
        updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression?: Nullable<string>, type?: number, useSRGBBuffer?: boolean): void;
        /**
         * Binds the frame buffer to the specified texture.
         * @param rtWrapper The render target wrapper to render to
         * @param faceIndex The face of the texture to render to in case of cube texture
         * @param requiredWidth The width of the target to render to
         * @param requiredHeight The height of the target to render to
         * @param forceFullscreenViewport Forces the viewport to be the entire texture/screen if true
         */
        bindFramebuffer(rtWrapper: RenderTargetWrapper, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean): void;
        /**
         * Unbind the current render target texture from the webGL context
         * @param rtWrapper defines the render target wrapper to unbind
         * @param disableGenerateMipMaps defines a boolean indicating that mipmaps must not be generated
         * @param onBeforeUnbind defines a function which will be called before the effective unbind
         */
        unBindFramebuffer(rtWrapper: RenderTargetWrapper, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        /**
         * Creates a dynamic vertex buffer
         * @param vertices the data for the dynamic vertex buffer
         * @returns the new WebGL dynamic buffer
         */
        createDynamicVertexBuffer(vertices: FloatArray): DataBuffer;
        /**
         * Update the content of a dynamic texture
         * @param texture defines the texture to update
         * @param canvas defines the canvas containing the source
         * @param invertY defines if data must be stored with Y axis inverted
         * @param premulAlpha defines if alpha is stored as premultiplied
         * @param format defines the format of the data
         */
        updateDynamicTexture(texture: Nullable<InternalTexture>, canvas: HTMLCanvasElement, invertY: boolean, premulAlpha?: boolean, format?: number): void;
        /**
         * Gets a boolean indicating if all created effects are ready
         * @returns true if all effects are ready
         */
        areAllEffectsReady(): boolean;
        /**
         * @internal
         * Get the current error code of the webGL context
         * @returns the error code
         * @see https://developer.mozilla.org/en-US/docs/Web/API/WebGLRenderingContext/getError
         */
        getError(): number;
        /** @internal */
        _getUnpackAlignement(): number;
        /**
         * @internal
         */
        _unpackFlipY(value: boolean): void;
        /**
         * Update a dynamic index buffer
         * @param indexBuffer defines the target index buffer
         * @param indices defines the data to update
         * @param offset defines the offset in the target index buffer where update should start
         */
        updateDynamicIndexBuffer(indexBuffer: WebGLBuffer, indices: IndicesArray, offset?: number): void;
        /**
         * Updates a dynamic vertex buffer.
         * @param vertexBuffer the vertex buffer to update
         * @param vertices the data used to update the vertex buffer
         * @param byteOffset the byte offset of the data (optional)
         * @param byteLength the byte length of the data (optional)
         */
        updateDynamicVertexBuffer(vertexBuffer: WebGLBuffer, vertices: FloatArray, byteOffset?: number, byteLength?: number): void;
        /**
         * @internal
         */
        _bindTextureDirectly(target: number, texture: InternalTexture): boolean;
        /**
         * @internal
         */
        _bindTexture(channel: number, texture: InternalTexture): void;
        protected _deleteBuffer(buffer: WebGLBuffer): void;
        /**
         * Force the engine to release all cached effects. This means that next effect compilation will have to be done completely even if a similar effect was already compiled
         */
        releaseEffects(): void;
        displayLoadingUI(): void;
        hideLoadingUI(): void;
        set loadingUIText(_: string);
        /**
         * @internal
         */
        _uploadCompressedDataToTextureDirectly(texture: InternalTexture, internalFormat: number, width: number, height: number, data: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadDataToTextureDirectly(texture: InternalTexture, imageData: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadArrayBufferViewToTexture(texture: InternalTexture, imageData: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadImageToTexture(texture: InternalTexture, image: HTMLImageElement, faceIndex?: number, lod?: number): void;
    }


    /** @internal */
    export class PerformanceConfigurator {
        /** @internal */
        static MatrixUse64Bits: boolean;
        /** @internal */
        static MatrixTrackPrecisionChange: boolean;
        /** @internal */
        static MatrixCurrentType: any;
        /** @internal */
        static MatrixTrackedMatrices: Array<any> | null;
        /**
         * @internal
         */
        static SetMatrixPrecision(use64bits: boolean): void;
    }






    /** @internal */
    export class ShaderDefineAndOperator extends ShaderDefineExpression {
        leftOperand: ShaderDefineExpression;
        rightOperand: ShaderDefineExpression;
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }


    /** @internal */
    export class ShaderDefineArithmeticOperator extends ShaderDefineExpression {
        define: string;
        operand: string;
        testValue: string;
        constructor(define: string, operand: string, testValue: string);
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }


    /** @internal */
    export class ShaderDefineIsDefinedOperator extends ShaderDefineExpression {
        define: string;
        not: boolean;
        constructor(define: string, not?: boolean);
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }


    /** @internal */
    export class ShaderDefineOrOperator extends ShaderDefineExpression {
        leftOperand: ShaderDefineExpression;
        rightOperand: ShaderDefineExpression;
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
    }


    /** @internal */
    export class ShaderDefineExpression {
        /**
         * Cache items count limit for the InfixToPostfix cache.
         * It uses to improve the performance of the shader compilation.
         * For details see PR: https://github.com/BabylonJS/Babylon.js/pull/13936
         */
        static InfixToPostfixCacheLimitSize: number;
        /**
         * When the cache size is exceeded, a cache cleanup will be triggered
         * and the cache will be reduced by the size specified
         * in the InfixToPostfixCacheCleanupSize variable, removing entries
         * that have not been accessed the longest.
         */
        static InfixToPostfixCacheCleanupSize: number;
        protected static _InfixToPostfixCache: Map<string, {
            accessTime: number;
            result: string[];
        }>;
        isTrue(preprocessors: {
            [key: string]: string;
        }): boolean;
        private static _OperatorPriority;
        private static _Stack;
        static postfixToInfix(postfix: string[]): string;
        /**
         * Converts an infix expression to a postfix expression.
         *
         * This method is used to transform infix expressions, which are more human-readable,
         * into postfix expressions, also known as Reverse Polish Notation (RPN), that can be
         * evaluated more efficiently by a computer. The conversion is based on the operator
         * priority defined in _OperatorPriority.
         *
         * The function employs a stack-based algorithm for the conversion and caches the result
         * to improve performance. The cache keeps track of each converted expression's access time
         * to manage the cache size and optimize memory usage. When the cache size exceeds a specified
         * limit, the least recently accessed items in the cache are deleted.
         *
         * The cache mechanism is particularly helpful for shader compilation, where the same infix
         * expressions might be encountered repeatedly, hence the caching can speed up the process.
         *
         * @param infix - The infix expression to be converted.
         * @returns The postfix expression as an array of strings.
         */
        static infixToPostfix(infix: string): string[];
        private static ClearCache;
    }




    /** @internal */
    export interface IShaderProcessor {
        shaderLanguage: ShaderLanguage;
        uniformRegexp?: RegExp;
        uniformBufferRegexp?: RegExp;
        textureRegexp?: RegExp;
        noPrecision?: boolean;
        parseGLES3?: boolean;
        attributeKeywordName?: string;
        varyingVertexKeywordName?: string;
        varyingFragmentKeywordName?: string;
        preProcessShaderCode?: (code: string, isFragment: boolean) => string;
        attributeProcessor?: (attribute: string, preProcessors: {
            [key: string]: string;
        }, processingContext: Nullable<ShaderProcessingContext>) => string;
        varyingCheck?: (varying: string, isFragment: boolean) => boolean;
        varyingProcessor?: (varying: string, isFragment: boolean, preProcessors: {
            [key: string]: string;
        }, processingContext: Nullable<ShaderProcessingContext>) => string;
        uniformProcessor?: (uniform: string, isFragment: boolean, preProcessors: {
            [key: string]: string;
        }, processingContext: Nullable<ShaderProcessingContext>) => string;
        uniformBufferProcessor?: (uniformBuffer: string, isFragment: boolean, processingContext: Nullable<ShaderProcessingContext>) => string;
        textureProcessor?: (texture: string, isFragment: boolean, preProcessors: {
            [key: string]: string;
        }, processingContext: Nullable<ShaderProcessingContext>) => string;
        endOfUniformBufferProcessor?: (closingBracketLine: string, isFragment: boolean, processingContext: Nullable<ShaderProcessingContext>) => string;
        lineProcessor?: (line: string, isFragment: boolean, processingContext: Nullable<ShaderProcessingContext>) => string;
        preProcessor?: (code: string, defines: string[], isFragment: boolean, processingContext: Nullable<ShaderProcessingContext>) => string;
        postProcessor?: (code: string, defines: string[], isFragment: boolean, processingContext: Nullable<ShaderProcessingContext>, engine: ThinEngine) => string;
        initializeShaders?: (processingContext: Nullable<ShaderProcessingContext>) => void;
        finalizeShaders?: (vertexCode: string, fragmentCode: string, processingContext: Nullable<ShaderProcessingContext>) => {
            vertexCode: string;
            fragmentCode: string;
        };
    }


    /** @internal */
    export class ShaderCodeConditionNode extends ShaderCodeNode {
        process(preprocessors: {
            [key: string]: string;
        }, options: ProcessingOptions): string;
    }


    /** @internal */
    export class ShaderCodeCursor {
        private _lines;
        lineIndex: number;
        get currentLine(): string;
        get canRead(): boolean;
        set lines(value: string[]);
    }


    /**
     * Class used to inline functions in shader code
     */
    export class ShaderCodeInliner {
        private static readonly _RegexpFindFunctionNameAndType;
        private _sourceCode;
        private _functionDescr;
        private _numMaxIterations;
        /** Gets or sets the token used to mark the functions to inline */
        inlineToken: string;
        /** Gets or sets the debug mode */
        debug: boolean;
        /** Gets the code after the inlining process */
        get code(): string;
        /**
         * Initializes the inliner
         * @param sourceCode shader code source to inline
         * @param numMaxIterations maximum number of iterations (used to detect recursive calls)
         */
        constructor(sourceCode: string, numMaxIterations?: number);
        /**
         * Start the processing of the shader code
         */
        processCode(): void;
        private _collectFunctions;
        private _processInlining;
        private _replaceFunctionCallsByCode;
        private _replaceNames;
    }


    /** @internal */
    export class ShaderCodeNode {
        line: string;
        children: ShaderCodeNode[];
        additionalDefineKey?: string;
        additionalDefineValue?: string;
        isValid(preprocessors: {
            [key: string]: string;
        }): boolean;
        process(preprocessors: {
            [key: string]: string;
        }, options: ProcessingOptions): string;
    }


    /** @internal */
    export class ShaderCodeTestNode extends ShaderCodeNode {
        testExpression: ShaderDefineExpression;
        isValid(preprocessors: {
            [key: string]: string;
        }): boolean;
    }


    /**
     * Function for custom code generation
     */
    export type ShaderCustomProcessingFunction = (shaderType: string, code: string) => string;
    /** @internal */
    export interface ShaderProcessingContext {
    }
    /** @internal */
    export interface ProcessingOptions {
        defines: string[];
        indexParameters: any;
        isFragment: boolean;
        shouldUseHighPrecisionShader: boolean;
        supportsUniformBuffers: boolean;
        shadersRepository: string;
        includesShadersStore: {
            [key: string]: string;
        };
        processor: Nullable<IShaderProcessor>;
        version: string;
        platformName: string;
        lookForClosingBracketForUniformBuffer?: boolean;
        processingContext: Nullable<ShaderProcessingContext>;
        isNDCHalfZRange: boolean;
        useReverseDepthBuffer: boolean;
        processCodeAfterIncludes?: ShaderCustomProcessingFunction;
    }


    /** @internal */
    export class ShaderProcessor {
        private static _MoveCursorRegex;
        static Initialize(options: ProcessingOptions): void;
        static Process(sourceCode: string, options: ProcessingOptions, callback: (migratedCode: string, codeBeforeMigration: string) => void, engine: ThinEngine): void;
        static PreProcess(sourceCode: string, options: ProcessingOptions, callback: (migratedCode: string, codeBeforeMigration: string) => void, engine: ThinEngine): void;
        static Finalize(vertexCode: string, fragmentCode: string, options: ProcessingOptions): {
            vertexCode: string;
            fragmentCode: string;
        };
        private static _ProcessPrecision;
        private static _ExtractOperation;
        private static _BuildSubExpression;
        private static _BuildExpression;
        private static _MoveCursorWithinIf;
        private static _MoveCursor;
        private static _EvaluatePreProcessors;
        private static _PreparePreProcessors;
        private static _ProcessShaderConversion;
        private static _ApplyPreProcessing;
        /** @internal */
        static _ProcessIncludes(sourceCode: string, options: ProcessingOptions, callback: (data: any) => void): void;
        /**
         * Loads a file from a url
         * @param url url to load
         * @param onSuccess callback called when the file successfully loads
         * @param onProgress callback called while file is loading (if the server supports this mode)
         * @param offlineProvider defines the offline provider for caching
         * @param useArrayBuffer defines a boolean indicating that date must be returned as ArrayBuffer
         * @param onError callback called when the file fails to load
         * @returns a file request object
         * @internal
         */
        static _FileToolsLoadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (ev: ProgressEvent) => void, offlineProvider?: IOfflineProvider, useArrayBuffer?: boolean, onError?: (request?: WebRequest, exception?: LoadFileError) => void): IFileRequest;
    }


    /**
     * An interface enforcing the renderTarget accessor to used by render target textures.
     */
    export interface IRenderTargetTexture {
        /**
         * Entry point to access the wrapper on a texture.
         */
        renderTarget: Nullable<RenderTargetWrapper>;
    }
    /**
     * Wrapper around a render target (either single or multi textures)
     */
    export class RenderTargetWrapper {
        protected _engine: ThinEngine;
        private _size;
        private _isCube;
        private _isMulti;
        private _textures;
        private _faceIndices;
        private _layerIndices;
        private _depthStencilTextureLabel?;
        /** @internal */
        _samples: number;
        /** @internal */
        _attachments: Nullable<number[]>;
        /** @internal */
        _generateStencilBuffer: boolean;
        /** @internal */
        _generateDepthBuffer: boolean;
        /** @internal */
        _depthStencilTexture: Nullable<InternalTexture>;
        /** @internal */
        _depthStencilTextureWithStencil: boolean;
        /**
         * Gets or sets the label of the render target wrapper (optional, for debugging purpose)
         */
        label?: string;
        /**
         * Gets the depth/stencil texture (if created by a createDepthStencilTexture() call)
         */
        get depthStencilTexture(): Nullable<InternalTexture>;
        /**
         * Indicates if the depth/stencil texture has a stencil aspect
         */
        get depthStencilTextureWithStencil(): boolean;
        /**
         * Defines if the render target wrapper is for a cube texture or if false a 2d texture
         */
        get isCube(): boolean;
        /**
         * Defines if the render target wrapper is for a single or multi target render wrapper
         */
        get isMulti(): boolean;
        /**
         * Defines if the render target wrapper is for a single or an array of textures
         */
        get is2DArray(): boolean;
        /**
         * Gets the size of the render target wrapper (used for cubes, as width=height in this case)
         */
        get size(): number;
        /**
         * Gets the width of the render target wrapper
         */
        get width(): number;
        /**
         * Gets the height of the render target wrapper
         */
        get height(): number;
        /**
         * Gets the number of layers of the render target wrapper (only used if is2DArray is true and wrapper is not a multi render target)
         */
        get layers(): number;
        /**
         * Gets the render texture. If this is a multi render target, gets the first texture
         */
        get texture(): Nullable<InternalTexture>;
        /**
         * Gets the list of render textures. If we are not in a multi render target, the list will be null (use the texture getter instead)
         */
        get textures(): Nullable<InternalTexture[]>;
        /**
         * Gets the face indices that correspond to the list of render textures. If we are not in a multi render target, the list will be null
         */
        get faceIndices(): Nullable<number[]>;
        /**
         * Gets the layer indices that correspond to the list of render textures. If we are not in a multi render target, the list will be null
         */
        get layerIndices(): Nullable<number[]>;
        /**
         * Gets the sample count of the render target
         */
        get samples(): number;
        /**
         * Sets the sample count of the render target
         * @param value sample count
         * @param initializeBuffers If set to true, the engine will make an initializing call to drawBuffers (only used when isMulti=true).
         * @param force true to force calling the update sample count engine function even if the current sample count is equal to value
         * @returns the sample count that has been set
         */
        setSamples(value: number, initializeBuffers?: boolean, force?: boolean): number;
        /**
         * Initializes the render target wrapper
         * @param isMulti true if the wrapper is a multi render target
         * @param isCube true if the wrapper should render to a cube texture
         * @param size size of the render target (width/height/layers)
         * @param engine engine used to create the render target
         * @param label defines the label to use for the wrapper (for debugging purpose only)
         */
        constructor(isMulti: boolean, isCube: boolean, size: TextureSize, engine: ThinEngine, label?: string);
        /**
         * Sets the render target texture(s)
         * @param textures texture(s) to set
         */
        setTextures(textures: Nullable<InternalTexture> | Nullable<InternalTexture[]>): void;
        /**
         * Set a texture in the textures array
         * @param texture The texture to set
         * @param index The index in the textures array to set
         * @param disposePrevious If this function should dispose the previous texture
         */
        setTexture(texture: InternalTexture, index?: number, disposePrevious?: boolean): void;
        /**
         * Sets the layer and face indices of every render target texture bound to each color attachment
         * @param layers The layers of each texture to be set
         * @param faces The faces of each texture to be set
         */
        setLayerAndFaceIndices(layers: number[], faces: number[]): void;
        /**
         * Sets the layer and face indices of a texture in the textures array that should be bound to each color attachment
         * @param index The index of the texture in the textures array to modify
         * @param layer The layer of the texture to be set
         * @param face The face of the texture to be set
         */
        setLayerAndFaceIndex(index?: number, layer?: number, face?: number): void;
        /**
         * Creates the depth/stencil texture
         * @param comparisonFunction Comparison function to use for the texture
         * @param bilinearFiltering true if bilinear filtering should be used when sampling the texture
         * @param generateStencil true if the stencil aspect should also be created
         * @param samples sample count to use when creating the texture
         * @param format format of the depth texture
         * @param label defines the label to use for the texture (for debugging purpose only)
         * @returns the depth/stencil created texture
         */
        createDepthStencilTexture(comparisonFunction?: number, bilinearFiltering?: boolean, generateStencil?: boolean, samples?: number, format?: number, label?: string): InternalTexture;
        /**
         * Shares the depth buffer of this render target with another render target.
         * @internal
         * @param renderTarget Destination renderTarget
         */
        _shareDepth(renderTarget: RenderTargetWrapper): void;
        /**
         * @internal
         */
        _swapAndDie(target: InternalTexture): void;
        protected _cloneRenderTargetWrapper(): Nullable<RenderTargetWrapper>;
        protected _swapRenderTargetWrapper(target: RenderTargetWrapper): void;
        /** @internal */
        _rebuild(): void;
        /**
         * Releases the internal render textures
         */
        releaseTextures(): void;
        /**
         * Disposes the whole render target wrapper
         * @param disposeOnlyFramebuffers true if only the frame buffers should be released (used for the WebGL engine). If false, all the textures will also be released
         */
        dispose(disposeOnlyFramebuffers?: boolean): void;
    }


    /**
     * Defines the shader related stores and directory
     */
    export class ShaderStore {
        /**
         * Gets or sets the relative url used to load shaders if using the engine in non-minified mode
         */
        static ShadersRepository: string;
        /**
         * Store of each shader (The can be looked up using effect.key)
         */
        static ShadersStore: {
            [key: string]: string;
        };
        /**
         * Store of each included file for a shader (The can be looked up using effect.key)
         */
        static IncludesShadersStore: {
            [key: string]: string;
        };
        /**
         * Gets or sets the relative url used to load shaders (WGSL) if using the engine in non-minified mode
         */
        static ShadersRepositoryWGSL: string;
        /**
         * Store of each shader  (WGSL)
         */
        static ShadersStoreWGSL: {
            [key: string]: string;
        };
        /**
         * Store of each included file for a shader (WGSL)
         */
        static IncludesShadersStoreWGSL: {
            [key: string]: string;
        };
        /**
         * Gets the shaders repository path for a given shader language
         * @param shaderLanguage the shader language
         * @returns the path to the shaders repository
         */
        static GetShadersRepository(shaderLanguage?: ShaderLanguage): string;
        /**
         * Gets the shaders store of a given shader language
         * @param shaderLanguage the shader language
         * @returns the shaders store
         */
        static GetShadersStore(shaderLanguage?: ShaderLanguage): {
            [key: string]: string;
        };
        /**
         * Gets the include shaders store of a given shader language
         * @param shaderLanguage the shader language
         * @returns the include shaders store
         */
        static GetIncludesShadersStore(shaderLanguage?: ShaderLanguage): {
            [key: string]: string;
        };
    }


    /**
     * Defines the interface used by objects working like Scene
     * @internal
     */
    export interface ISceneLike {
        addPendingData(data: any): void;
        removePendingData(data: any): void;
        offlineProvider: IOfflineProvider;
    }
    /**
     * Information about the current host
     */
    export interface HostInformation {
        /**
         * Defines if the current host is a mobile
         */
        isMobile: boolean;
    }
    /** Interface defining initialization parameters for ThinEngine class */
    export interface ThinEngineOptions {
        /**
         * Defines if the engine should no exceed a specified device ratio
         * @see https://developer.mozilla.org/en-US/docs/Web/API/Window/devicePixelRatio
         */
        limitDeviceRatio?: number;
        /**
         * Defines if webaudio should be initialized as well
         * @see https://doc.babylonjs.com/features/featuresDeepDive/audio/playingSoundsMusic
         */
        audioEngine?: boolean;
        /**
         * Specifies options for the audio engine
         */
        audioEngineOptions?: IAudioEngineOptions;
        /**
         * Defines if animations should run using a deterministic lock step
         * @see https://doc.babylonjs.com/features/featuresDeepDive/animation/advanced_animations#deterministic-lockstep
         */
        deterministicLockstep?: boolean;
        /** Defines the maximum steps to use with deterministic lock step mode */
        lockstepMaxSteps?: number;
        /** Defines the seconds between each deterministic lock step */
        timeStep?: number;
        /**
         * Defines that engine should ignore context lost events
         * If this event happens when this parameter is true, you will have to reload the page to restore rendering
         */
        doNotHandleContextLost?: boolean;
        /**
         * Defines that engine should ignore modifying touch action attribute and style
         * If not handle, you might need to set it up on your side for expected touch devices behavior.
         */
        doNotHandleTouchAction?: boolean;
        /**
         * Make the matrix computations to be performed in 64 bits instead of 32 bits. False by default
         */
        useHighPrecisionMatrix?: boolean;
        /**
         * Defines whether to adapt to the device's viewport characteristics (default: false)
         */
        adaptToDeviceRatio?: boolean;
        /**
         * True if the more expensive but exact conversions should be used for transforming colors to and from linear space within shaders.
         * Otherwise, the default is to use a cheaper approximation.
         */
        useExactSrgbConversions?: boolean;
        /**
         * Defines whether MSAA is enabled on the canvas.
         */
        antialias?: boolean;
        /**
         * Defines whether the stencil buffer should be enabled.
         */
        stencil?: boolean;
        /**
         * Defines whether the canvas should be created in "premultiplied" mode (if false, the canvas is created in the "opaque" mode) (true by default)
         */
        premultipliedAlpha?: boolean;
    }
    /** Interface defining initialization parameters for Engine class */
    export interface EngineOptions extends ThinEngineOptions, WebGLContextAttributes {
        /**
         * Defines if webgl2 should be turned off even if supported
         * @see https://doc.babylonjs.com/setup/support/webGL2
         */
        disableWebGL2Support?: boolean;
        /**
         * Defines that engine should compile shaders with high precision floats (if supported). True by default
         */
        useHighPrecisionFloats?: boolean;
        /**
         * Make the canvas XR Compatible for XR sessions
         */
        xrCompatible?: boolean;
        /**
         * Will prevent the system from falling back to software implementation if a hardware device cannot be created
         */
        failIfMajorPerformanceCaveat?: boolean;
        /**
         * If sRGB Buffer support is not set during construction, use this value to force a specific state
         * This is added due to an issue when processing textures in chrome/edge/firefox
         * This will not influence NativeEngine and WebGPUEngine which set the behavior to true during construction.
         */
        forceSRGBBufferSupportState?: boolean;
        /**
         * Defines if the gl context should be released.
         * It's false by default for backward compatibility, but you should probably pass true (see https://registry.khronos.org/webgl/extensions/WEBGL_lose_context/)
         */
        loseContextOnDispose?: boolean;
    }
    /**
     * The base engine class (root of all engines)
     */
    export class ThinEngine {
        private static _TempClearColorUint32;
        private static _TempClearColorInt32;
        /** Use this array to turn off some WebGL2 features on known buggy browsers version */
        static ExceptionList: ({
            key: string;
            capture: string;
            captureConstraint: number;
            targets: string[];
        } | {
            key: string;
            capture: null;
            captureConstraint: null;
            targets: string[];
        })[];
        /** @internal */
        static _TextureLoaders: IInternalTextureLoader[];
        /**
         * Returns the current npm package of the sdk
         */
        static get NpmPackage(): string;
        /**
         * Returns the current version of the framework
         */
        static get Version(): string;
        /**
         * Returns a string describing the current engine
         */
        get description(): string;
        /** @internal */
        protected _name: string;
        /**
         * Gets or sets the name of the engine
         */
        get name(): string;
        set name(value: string);
        /**
         * Returns the version of the engine
         */
        get version(): number;
        protected _isDisposed: boolean;
        get isDisposed(): boolean;
        /**
         * Gets or sets the epsilon value used by collision engine
         */
        static CollisionsEpsilon: number;
        /**
         * Gets or sets the relative url used to load shaders if using the engine in non-minified mode
         */
        static get ShadersRepository(): string;
        static set ShadersRepository(value: string);
        protected _shaderProcessor: Nullable<IShaderProcessor>;
        /**
         * @internal
         */
        _getShaderProcessor(shaderLanguage: ShaderLanguage): Nullable<IShaderProcessor>;
        /**
         * Gets or sets a boolean that indicates if textures must be forced to power of 2 size even if not required
         */
        forcePOTTextures: boolean;
        /**
         * Gets a boolean indicating if the engine is currently rendering in fullscreen mode
         */
        isFullscreen: boolean;
        /**
         * Gets or sets a boolean indicating if back faces must be culled. If false, front faces are culled instead (true by default)
         * If non null, this takes precedence over the value from the material
         */
        cullBackFaces: Nullable<boolean>;
        /**
         * Gets or sets a boolean indicating if the engine must keep rendering even if the window is not in foreground
         */
        renderEvenInBackground: boolean;
        /**
         * Gets or sets a boolean indicating that cache can be kept between frames
         */
        preventCacheWipeBetweenFrames: boolean;
        /** Gets or sets a boolean indicating if the engine should validate programs after compilation */
        validateShaderPrograms: boolean;
        private _useReverseDepthBuffer;
        /**
         * Gets or sets a boolean indicating if depth buffer should be reverse, going from far to near.
         * This can provide greater z depth for distant objects.
         */
        get useReverseDepthBuffer(): boolean;
        set useReverseDepthBuffer(useReverse: boolean);
        /**
         * Indicates if the z range in NDC space is 0..1 (value: true) or -1..1 (value: false)
         */
        readonly isNDCHalfZRange: boolean;
        /**
         * Indicates that the origin of the texture/framebuffer space is the bottom left corner. If false, the origin is top left
         */
        readonly hasOriginBottomLeft: boolean;
        /**
         * Gets or sets a boolean indicating that uniform buffers must be disabled even if they are supported
         */
        disableUniformBuffers: boolean;
        /**
         * An event triggered when the engine is disposed.
         */
        readonly onDisposeObservable: Observable<ThinEngine>;
        private _frameId;
        /**
         * Gets the current frame id
         */
        get frameId(): number;
        /**
         * The time (in milliseconds elapsed since the current page has been loaded) when the engine was initialized
         */
        readonly startTime: number;
        /** @internal */
        _uniformBuffers: UniformBuffer[];
        /** @internal */
        _storageBuffers: StorageBuffer[];
        /**
         * Gets a boolean indicating that the engine supports uniform buffers
         * @see https://doc.babylonjs.com/setup/support/webGL2#uniform-buffer-objets
         */
        get supportsUniformBuffers(): boolean;
        /** @internal */
        _gl: WebGL2RenderingContext;
        /** @internal */
        _webGLVersion: number;
        protected _renderingCanvas: Nullable<HTMLCanvasElement>;
        protected _windowIsBackground: boolean;
        protected _creationOptions: EngineOptions;
        protected _audioContext: Nullable<AudioContext>;
        protected _audioDestination: Nullable<AudioDestinationNode | MediaStreamAudioDestinationNode>;
        /** @internal */
        _glSRGBExtensionValues: {
            SRGB: typeof WebGL2RenderingContext.SRGB;
            SRGB8: typeof WebGL2RenderingContext.SRGB8 | EXT_sRGB["SRGB_ALPHA_EXT"];
            SRGB8_ALPHA8: typeof WebGL2RenderingContext.SRGB8_ALPHA8 | EXT_sRGB["SRGB_ALPHA_EXT"];
        };
        /**
         * Gets the options used for engine creation
         * @returns EngineOptions object
         */
        getCreationOptions(): EngineOptions;
        protected _highPrecisionShadersAllowed: boolean;
        /** @internal */
        get _shouldUseHighPrecisionShader(): boolean;
        /**
         * Gets a boolean indicating that only power of 2 textures are supported
         * Please note that you can still use non power of 2 textures but in this case the engine will forcefully convert them
         */
        get needPOTTextures(): boolean;
        /** @internal */
        _badOS: boolean;
        /** @internal */
        _badDesktopOS: boolean;
        /** @internal */
        _hardwareScalingLevel: number;
        /** @internal */
        _caps: EngineCapabilities;
        /** @internal */
        _features: EngineFeatures;
        protected _isStencilEnable: boolean;
        private _glVersion;
        private _glRenderer;
        private _glVendor;
        /** @internal */
        _videoTextureSupported: boolean;
        protected _renderingQueueLaunched: boolean;
        protected _activeRenderLoops: (() => void)[];
        /**
         * Gets the list of current active render loop functions
         * @returns an array with the current render loop functions
         */
        get activeRenderLoops(): Array<() => void>;
        /**
         * Observable signaled when a context lost event is raised
         */
        onContextLostObservable: Observable<ThinEngine>;
        /**
         * Observable signaled when a context restored event is raised
         */
        onContextRestoredObservable: Observable<ThinEngine>;
        private _onContextLost;
        private _onContextRestored;
        protected _contextWasLost: boolean;
        /** @internal */
        _doNotHandleContextLost: boolean;
        /**
         * Gets or sets a boolean indicating if resources should be retained to be able to handle context lost events
         * @see https://doc.babylonjs.com/features/featuresDeepDive/scene/optimize_your_scene#handling-webgl-context-lost
         */
        get doNotHandleContextLost(): boolean;
        set doNotHandleContextLost(value: boolean);
        /**
         * Gets or sets a boolean indicating that vertex array object must be disabled even if they are supported
         */
        disableVertexArrayObjects: boolean;
        /** @internal */
        protected _colorWrite: boolean;
        /** @internal */
        protected _colorWriteChanged: boolean;
        /** @internal */
        protected _depthCullingState: DepthCullingState;
        /** @internal */
        protected _stencilStateComposer: StencilStateComposer;
        /** @internal */
        protected _stencilState: StencilState;
        /** @internal */
        _alphaState: AlphaState;
        /** @internal */
        _alphaMode: number;
        /** @internal */
        _alphaEquation: number;
        /** @internal */
        _internalTexturesCache: InternalTexture[];
        /** @internal */
        _renderTargetWrapperCache: RenderTargetWrapper[];
        /** @internal */
        protected _activeChannel: number;
        private _currentTextureChannel;
        /** @internal */
        protected _boundTexturesCache: {
            [key: string]: Nullable<InternalTexture>;
        };
        protected _currentEffect: Nullable<Effect>;
        /** @internal */
        _currentDrawContext: IDrawContext;
        /** @internal */
        _currentMaterialContext: IMaterialContext;
        /** @internal */
        protected _currentProgram: Nullable<WebGLProgram>;
        protected _compiledEffects: {
            [key: string]: Effect;
        };
        private _vertexAttribArraysEnabled;
        /** @internal */
        protected _cachedViewport: Nullable<IViewportLike>;
        private _cachedVertexArrayObject;
        /** @internal */
        protected _cachedVertexBuffers: any;
        /** @internal */
        protected _cachedIndexBuffer: Nullable<DataBuffer>;
        /** @internal */
        protected _cachedEffectForVertexBuffers: Nullable<Effect>;
        /** @internal */
        _currentRenderTarget: Nullable<RenderTargetWrapper>;
        private _uintIndicesCurrentlySet;
        protected _currentBoundBuffer: Nullable<DataBuffer>[];
        /** @internal */
        _currentFramebuffer: Nullable<WebGLFramebuffer>;
        /** @internal */
        _dummyFramebuffer: Nullable<WebGLFramebuffer>;
        private _currentBufferPointers;
        private _currentInstanceLocations;
        private _currentInstanceBuffers;
        private _textureUnits;
        /** @internal */
        _workingCanvas: Nullable<ICanvas>;
        /** @internal */
        _workingContext: Nullable<ICanvasRenderingContext>;
        /** @internal */
        _boundRenderFunction: any;
        private _vaoRecordInProgress;
        private _mustWipeVertexAttributes;
        private _emptyTexture;
        private _emptyCubeTexture;
        private _emptyTexture3D;
        private _emptyTexture2DArray;
        /** @internal */
        _frameHandler: number;
        private _nextFreeTextureSlots;
        private _maxSimultaneousTextures;
        private _maxMSAASamplesOverride;
        private _activeRequests;
        /**
         * If set to true zooming in and out in the browser will rescale the hardware-scaling correctly.
         */
        adaptToDeviceRatio: boolean;
        /** @internal */
        protected _lastDevicePixelRatio: number;
        /** @internal */
        _transformTextureUrl: Nullable<(url: string) => string>;
        /**
         * Gets information about the current host
         */
        hostInformation: HostInformation;
        protected get _supportsHardwareTextureRescaling(): boolean;
        protected _framebufferDimensionsObject: Nullable<{
            framebufferWidth: number;
            framebufferHeight: number;
        }>;
        /**
         * sets the object from which width and height will be taken from when getting render width and height
         * Will fallback to the gl object
         * @param dimensions the framebuffer width and height that will be used.
         */
        set framebufferDimensionsObject(dimensions: Nullable<{
            framebufferWidth: number;
            framebufferHeight: number;
        }>);
        /**
         * Gets the current viewport
         */
        get currentViewport(): Nullable<IViewportLike>;
        /**
         * Gets the default empty texture
         */
        get emptyTexture(): InternalTexture;
        /**
         * Gets the default empty 3D texture
         */
        get emptyTexture3D(): InternalTexture;
        /**
         * Gets the default empty 2D array texture
         */
        get emptyTexture2DArray(): InternalTexture;
        /**
         * Gets the default empty cube texture
         */
        get emptyCubeTexture(): InternalTexture;
        /**
         * Defines whether the engine has been created with the premultipliedAlpha option on or not.
         */
        premultipliedAlpha: boolean;
        /**
         * Observable event triggered before each texture is initialized
         */
        onBeforeTextureInitObservable: Observable<Texture>;
        /** @internal */
        protected _isWebGPU: boolean;
        /**
         * Gets a boolean indicating if the engine runs in WebGPU or not.
         */
        get isWebGPU(): boolean;
        /** @internal */
        protected _shaderPlatformName: string;
        /**
         * Gets the shader platform name used by the effects.
         */
        get shaderPlatformName(): string;
        /**
         * Enables or disables the snapshot rendering mode
         * Note that the WebGL engine does not support snapshot rendering so setting the value won't have any effect for this engine
         */
        get snapshotRendering(): boolean;
        set snapshotRendering(activate: boolean);
        protected _snapshotRenderingMode: number;
        /**
         * Gets or sets the snapshot rendering mode
         */
        get snapshotRenderingMode(): number;
        set snapshotRenderingMode(mode: number);
        /**
         * Gets a boolean indicating if the exact sRGB conversions or faster approximations are used for converting to and from linear space.
         */
        readonly useExactSrgbConversions: boolean;
        /**
         * Creates a new snapshot at the next frame using the current snapshotRenderingMode
         */
        snapshotRenderingReset(): void;
        private _checkForMobile;
        private static _CreateCanvas;
        /**
         * Create a canvas. This method is overridden by other engines
         * @param width width
         * @param height height
         * @returns ICanvas interface
         */
        createCanvas(width: number, height: number): ICanvas;
        /**
         * Create an image to use with canvas
         * @returns IImage interface
         */
        createCanvasImage(): IImage;
        /**
         * Creates a new engine
         * @param canvasOrContext defines the canvas or WebGL context to use for rendering. If you provide a WebGL context, Babylon.js will not hook events on the canvas (like pointers, keyboards, etc...) so no event observables will be available. This is mostly used when Babylon.js is used as a plugin on a system which already used the WebGL context
         * @param antialias defines enable antialiasing (default: false)
         * @param options defines further options to be sent to the getContext() function
         * @param adaptToDeviceRatio defines whether to adapt to the device's viewport characteristics (default: false)
         */
        constructor(canvasOrContext: Nullable<HTMLCanvasElement | OffscreenCanvas | WebGLRenderingContext | WebGL2RenderingContext>, antialias?: boolean, options?: EngineOptions, adaptToDeviceRatio?: boolean);
        protected _setupMobileChecks(): void;
        protected _restoreEngineAfterContextLost(initEngine: () => void): void;
        /**
         * Shared initialization across engines types.
         * @param canvas The canvas associated with this instance of the engine.
         */
        protected _sharedInit(canvas: HTMLCanvasElement): void;
        /**
         * @internal
         */
        _getShaderProcessingContext(shaderLanguage: ShaderLanguage): Nullable<ShaderProcessingContext>;
        private _rebuildInternalTextures;
        private _rebuildRenderTargetWrappers;
        private _rebuildEffects;
        /**
         * Gets a boolean indicating if all created effects are ready
         * @returns true if all effects are ready
         */
        areAllEffectsReady(): boolean;
        protected _rebuildBuffers(): void;
        protected _rebuildTextures(): void;
        protected _initGLContext(): void;
        protected _initFeatures(): void;
        /**
         * Gets version of the current webGL context
         * Keep it for back compat - use version instead
         */
        get webGLVersion(): number;
        /**
         * Gets a string identifying the name of the class
         * @returns "Engine" string
         */
        getClassName(): string;
        /**
         * Returns true if the stencil buffer has been enabled through the creation option of the context.
         */
        get isStencilEnable(): boolean;
        /** @internal */
        _prepareWorkingCanvas(): void;
        /**
         * Reset the texture cache to empty state
         */
        resetTextureCache(): void;
        /**
         * Gets an object containing information about the current engine context
         * @returns an object containing the vendor, the renderer and the version of the current engine context
         */
        getInfo(): {
            vendor: string;
            renderer: string;
            version: string;
        };
        /**
         * Gets an object containing information about the current webGL context
         * @returns an object containing the vendor, the renderer and the version of the current webGL context
         */
        getGlInfo(): {
            vendor: string;
            renderer: string;
            version: string;
        };
        /**
         * Defines the hardware scaling level.
         * By default the hardware scaling level is computed from the window device ratio.
         * if level = 1 then the engine will render at the exact resolution of the canvas. If level = 0.5 then the engine will render at twice the size of the canvas.
         * @param level defines the level to use
         */
        setHardwareScalingLevel(level: number): void;
        /**
         * Gets the current hardware scaling level.
         * By default the hardware scaling level is computed from the window device ratio.
         * if level = 1 then the engine will render at the exact resolution of the canvas. If level = 0.5 then the engine will render at twice the size of the canvas.
         * @returns a number indicating the current hardware scaling level
         */
        getHardwareScalingLevel(): number;
        /**
         * Gets the list of loaded textures
         * @returns an array containing all loaded textures
         */
        getLoadedTexturesCache(): InternalTexture[];
        /**
         * Gets the object containing all engine capabilities
         * @returns the EngineCapabilities object
         */
        getCaps(): EngineCapabilities;
        /**
         * stop executing a render loop function and remove it from the execution array
         * @param renderFunction defines the function to be removed. If not provided all functions will be removed.
         */
        stopRenderLoop(renderFunction?: () => void): void;
        protected _cancelFrame(): void;
        /** @internal */
        _renderLoop(): void;
        /**
         * Gets the HTML canvas attached with the current webGL context
         * @returns a HTML canvas
         */
        getRenderingCanvas(): Nullable<HTMLCanvasElement>;
        /**
         * Gets the audio context specified in engine initialization options
         * @returns an Audio Context
         */
        getAudioContext(): Nullable<AudioContext>;
        /**
         * Gets the audio destination specified in engine initialization options
         * @returns an audio destination node
         */
        getAudioDestination(): Nullable<AudioDestinationNode | MediaStreamAudioDestinationNode>;
        /**
         * Gets host window
         * @returns the host window object
         */
        getHostWindow(): Nullable<Window>;
        /**
         * Gets the current render width
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the current render width
         */
        getRenderWidth(useScreen?: boolean): number;
        /**
         * Gets the current render height
         * @param useScreen defines if screen size must be used (or the current render target if any)
         * @returns a number defining the current render height
         */
        getRenderHeight(useScreen?: boolean): number;
        /**
         * Can be used to override the current requestAnimationFrame requester.
         * @internal
         */
        protected _queueNewFrame(bindedRenderFunction: any, requester?: any): number;
        /**
         * Register and execute a render loop. The engine can have more than one render function
         * @param renderFunction defines the function to continuously execute
         */
        runRenderLoop(renderFunction: () => void): void;
        /**
         * Clear the current render buffer or the current render target (if any is set up)
         * @param color defines the color to use
         * @param backBuffer defines if the back buffer must be cleared
         * @param depth defines if the depth buffer must be cleared
         * @param stencil defines if the stencil buffer must be cleared
         */
        clear(color: Nullable<IColor4Like>, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        protected _viewportCached: {
            x: number;
            y: number;
            z: number;
            w: number;
        };
        /**
         * @internal
         */
        _viewport(x: number, y: number, width: number, height: number): void;
        /**
         * Set the WebGL's viewport
         * @param viewport defines the viewport element to be used
         * @param requiredWidth defines the width required for rendering. If not provided the rendering canvas' width is used
         * @param requiredHeight defines the height required for rendering. If not provided the rendering canvas' height is used
         */
        setViewport(viewport: IViewportLike, requiredWidth?: number, requiredHeight?: number): void;
        /**
         * Begin a new frame
         */
        beginFrame(): void;
        /**
         * Enf the current frame
         */
        endFrame(): void;
        /**
         * Resize the view according to the canvas' size
         * @param forceSetSize true to force setting the sizes of the underlying canvas
         */
        resize(forceSetSize?: boolean): void;
        /**
         * Force a specific size of the canvas
         * @param width defines the new canvas' width
         * @param height defines the new canvas' height
         * @param forceSetSize true to force setting the sizes of the underlying canvas
         * @returns true if the size was changed
         */
        setSize(width: number, height: number, forceSetSize?: boolean): boolean;
        /**
         * Binds the frame buffer to the specified texture.
         * @param rtWrapper The render target wrapper to render to
         * @param faceIndex The face of the texture to render to in case of cube texture and if the render target wrapper is not a multi render target
         * @param requiredWidth The width of the target to render to
         * @param requiredHeight The height of the target to render to
         * @param forceFullscreenViewport Forces the viewport to be the entire texture/screen if true
         * @param lodLevel Defines the lod level to bind to the frame buffer
         * @param layer Defines the 2d array index to bind to the frame buffer if the render target wrapper is not a multi render target
         */
        bindFramebuffer(rtWrapper: RenderTargetWrapper, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean, lodLevel?: number, layer?: number): void;
        /**
         * Set various states to the webGL context
         * @param culling defines culling state: true to enable culling, false to disable it
         * @param zOffset defines the value to apply to zOffset (0 by default)
         * @param force defines if states must be applied even if cache is up to date
         * @param reverseSide defines if culling must be reversed (CCW if false, CW if true)
         * @param cullBackFaces true to cull back faces, false to cull front faces (if culling is enabled)
         * @param stencil stencil states to set
         * @param zOffsetUnits defines the value to apply to zOffsetUnits (0 by default)
         */
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean, cullBackFaces?: boolean, stencil?: IStencilState, zOffsetUnits?: number): void;
        /**
         * Gets a boolean indicating if depth testing is enabled
         * @returns the current state
         */
        getDepthBuffer(): boolean;
        /**
         * Enable or disable depth buffering
         * @param enable defines the state to set
         */
        setDepthBuffer(enable: boolean): void;
        /**
         * Set the z offset Factor to apply to current rendering
         * @param value defines the offset to apply
         */
        setZOffset(value: number): void;
        /**
         * Gets the current value of the zOffset Factor
         * @returns the current zOffset Factor state
         */
        getZOffset(): number;
        /**
         * Set the z offset Units to apply to current rendering
         * @param value defines the offset to apply
         */
        setZOffsetUnits(value: number): void;
        /**
         * Gets the current value of the zOffset Units
         * @returns the current zOffset Units state
         */
        getZOffsetUnits(): number;
        /**
         * @internal
         */
        _bindUnboundFramebuffer(framebuffer: Nullable<WebGLFramebuffer>): void;
        /** @internal */
        _currentFrameBufferIsDefaultFrameBuffer(): boolean;
        /**
         * Generates the mipmaps for a texture
         * @param texture texture to generate the mipmaps for
         */
        generateMipmaps(texture: InternalTexture): void;
        /**
         * Unbind the current render target texture from the webGL context
         * @param texture defines the render target wrapper to unbind
         * @param disableGenerateMipMaps defines a boolean indicating that mipmaps must not be generated
         * @param onBeforeUnbind defines a function which will be called before the effective unbind
         */
        unBindFramebuffer(texture: RenderTargetWrapper, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        /**
         * Force a webGL flush (ie. a flush of all waiting webGL commands)
         */
        flushFramebuffer(): void;
        /**
         * Unbind the current render target and bind the default framebuffer
         */
        restoreDefaultFramebuffer(): void;
        /** @internal */
        protected _resetVertexBufferBinding(): void;
        /**
         * Creates a vertex buffer
         * @param data the data or size for the vertex buffer
         * @param _updatable whether the buffer should be created as updatable
         * @param _label defines the label of the buffer (for debug purpose)
         * @returns the new WebGL static buffer
         */
        createVertexBuffer(data: DataArray | number, _updatable?: boolean, _label?: string): DataBuffer;
        private _createVertexBuffer;
        /**
         * Creates a dynamic vertex buffer
         * @param data the data for the dynamic vertex buffer
         * @param _label defines the label of the buffer (for debug purpose)
         * @returns the new WebGL dynamic buffer
         */
        createDynamicVertexBuffer(data: DataArray | number, _label?: string): DataBuffer;
        protected _resetIndexBufferBinding(): void;
        /**
         * Creates a new index buffer
         * @param indices defines the content of the index buffer
         * @param updatable defines if the index buffer must be updatable
         * @param _label defines the label of the buffer (for debug purpose)
         * @returns a new webGL buffer
         */
        createIndexBuffer(indices: IndicesArray, updatable?: boolean, _label?: string): DataBuffer;
        protected _normalizeIndexData(indices: IndicesArray): Uint16Array | Uint32Array;
        /**
         * Bind a webGL buffer to the webGL context
         * @param buffer defines the buffer to bind
         */
        bindArrayBuffer(buffer: Nullable<DataBuffer>): void;
        protected bindIndexBuffer(buffer: Nullable<DataBuffer>): void;
        private _bindBuffer;
        /**
         * update the bound buffer with the given data
         * @param data defines the data to update
         */
        updateArrayBuffer(data: Float32Array): void;
        private _vertexAttribPointer;
        /**
         * @internal
         */
        _bindIndexBufferWithCache(indexBuffer: Nullable<DataBuffer>): void;
        private _bindVertexBuffersAttributes;
        /**
         * Records a vertex array object
         * @see https://doc.babylonjs.com/setup/support/webGL2#vertex-array-objects
         * @param vertexBuffers defines the list of vertex buffers to store
         * @param indexBuffer defines the index buffer to store
         * @param effect defines the effect to store
         * @param overrideVertexBuffers defines optional list of avertex buffers that overrides the entries in vertexBuffers
         * @returns the new vertex array object
         */
        recordVertexArrayObject(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: Nullable<DataBuffer>, effect: Effect, overrideVertexBuffers?: {
            [kind: string]: Nullable<VertexBuffer>;
        }): WebGLVertexArrayObject;
        /**
         * Bind a specific vertex array object
         * @see https://doc.babylonjs.com/setup/support/webGL2#vertex-array-objects
         * @param vertexArrayObject defines the vertex array object to bind
         * @param indexBuffer defines the index buffer to bind
         */
        bindVertexArrayObject(vertexArrayObject: WebGLVertexArrayObject, indexBuffer: Nullable<DataBuffer>): void;
        /**
         * Bind webGl buffers directly to the webGL context
         * @param vertexBuffer defines the vertex buffer to bind
         * @param indexBuffer defines the index buffer to bind
         * @param vertexDeclaration defines the vertex declaration to use with the vertex buffer
         * @param vertexStrideSize defines the vertex stride of the vertex buffer
         * @param effect defines the effect associated with the vertex buffer
         */
        bindBuffersDirectly(vertexBuffer: DataBuffer, indexBuffer: DataBuffer, vertexDeclaration: number[], vertexStrideSize: number, effect: Effect): void;
        private _unbindVertexArrayObject;
        /**
         * Bind a list of vertex buffers to the webGL context
         * @param vertexBuffers defines the list of vertex buffers to bind
         * @param indexBuffer defines the index buffer to bind
         * @param effect defines the effect associated with the vertex buffers
         * @param overrideVertexBuffers defines optional list of avertex buffers that overrides the entries in vertexBuffers
         */
        bindBuffers(vertexBuffers: {
            [key: string]: Nullable<VertexBuffer>;
        }, indexBuffer: Nullable<DataBuffer>, effect: Effect, overrideVertexBuffers?: {
            [kind: string]: Nullable<VertexBuffer>;
        }): void;
        /**
         * Unbind all instance attributes
         */
        unbindInstanceAttributes(): void;
        /**
         * Release and free the memory of a vertex array object
         * @param vao defines the vertex array object to delete
         */
        releaseVertexArrayObject(vao: WebGLVertexArrayObject): void;
        /**
         * @internal
         */
        _releaseBuffer(buffer: DataBuffer): boolean;
        protected _deleteBuffer(buffer: DataBuffer): void;
        /**
         * Update the content of a webGL buffer used with instantiation and bind it to the webGL context
         * @param instancesBuffer defines the webGL buffer to update and bind
         * @param data defines the data to store in the buffer
         * @param offsetLocations defines the offsets or attributes information used to determine where data must be stored in the buffer
         */
        updateAndBindInstancesBuffer(instancesBuffer: DataBuffer, data: Float32Array, offsetLocations: number[] | InstancingAttributeInfo[]): void;
        /**
         * Bind the content of a webGL buffer used with instantiation
         * @param instancesBuffer defines the webGL buffer to bind
         * @param attributesInfo defines the offsets or attributes information used to determine where data must be stored in the buffer
         * @param computeStride defines Whether to compute the strides from the info or use the default 0
         */
        bindInstancesBuffer(instancesBuffer: DataBuffer, attributesInfo: InstancingAttributeInfo[], computeStride?: boolean): void;
        /**
         * Disable the instance attribute corresponding to the name in parameter
         * @param name defines the name of the attribute to disable
         */
        disableInstanceAttributeByName(name: string): void;
        /**
         * Disable the instance attribute corresponding to the location in parameter
         * @param attributeLocation defines the attribute location of the attribute to disable
         */
        disableInstanceAttribute(attributeLocation: number): void;
        /**
         * Disable the attribute corresponding to the location in parameter
         * @param attributeLocation defines the attribute location of the attribute to disable
         */
        disableAttributeByIndex(attributeLocation: number): void;
        /**
         * Send a draw order
         * @param useTriangles defines if triangles must be used to draw (else wireframe will be used)
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        draw(useTriangles: boolean, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of points
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawPointClouds(verticesStart: number, verticesCount: number, instancesCount?: number): void;
        /**
         * Draw a list of unindexed primitives
         * @param useTriangles defines if triangles must be used to draw (else wireframe will be used)
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawUnIndexed(useTriangles: boolean, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        /**
         * Draw a list of indexed primitives
         * @param fillMode defines the primitive to use
         * @param indexStart defines the starting index
         * @param indexCount defines the number of index to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        /**
         * Draw a list of unindexed primitives
         * @param fillMode defines the primitive to use
         * @param verticesStart defines the index of first vertex to draw
         * @param verticesCount defines the count of vertices to draw
         * @param instancesCount defines the number of instances to draw (if instantiation is enabled)
         */
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        private _drawMode;
        /** @internal */
        protected _reportDrawCall(): void;
        /**
         * @internal
         */
        _releaseEffect(effect: Effect): void;
        /**
         * @internal
         */
        _deletePipelineContext(pipelineContext: IPipelineContext): void;
        /** @internal */
        _getGlobalDefines(defines?: {
            [key: string]: string;
        }): string | undefined;
        /**
         * Create a new effect (used to store vertex/fragment shaders)
         * @param baseName defines the base name of the effect (The name of file without .fragment.fx or .vertex.fx)
         * @param attributesNamesOrOptions defines either a list of attribute names or an IEffectCreationOptions object
         * @param uniformsNamesOrEngine defines either a list of uniform names or the engine to use
         * @param samplers defines an array of string used to represent textures
         * @param defines defines the string containing the defines to use to compile the shaders
         * @param fallbacks defines the list of potential fallbacks to use if shader compilation fails
         * @param onCompiled defines a function to call when the effect creation is successful
         * @param onError defines a function to call when the effect creation has failed
         * @param indexParameters defines an object containing the index values to use to compile shaders (like the maximum number of simultaneous lights)
         * @param shaderLanguage the language the shader is written in (default: GLSL)
         * @returns the new Effect
         */
        createEffect(baseName: any, attributesNamesOrOptions: string[] | IEffectCreationOptions, uniformsNamesOrEngine: string[] | ThinEngine, samplers?: string[], defines?: string, fallbacks?: IEffectFallbacks, onCompiled?: Nullable<(effect: Effect) => void>, onError?: Nullable<(effect: Effect, errors: string) => void>, indexParameters?: any, shaderLanguage?: ShaderLanguage): Effect;
        protected static _ConcatenateShader(source: string, defines: Nullable<string>, shaderVersion?: string): string;
        private _compileShader;
        private _compileRawShader;
        /**
         * @internal
         */
        _getShaderSource(shader: WebGLShader): Nullable<string>;
        /**
         * Directly creates a webGL program
         * @param pipelineContext  defines the pipeline context to attach to
         * @param vertexCode defines the vertex shader code to use
         * @param fragmentCode defines the fragment shader code to use
         * @param context defines the webGL context to use (if not set, the current one will be used)
         * @param transformFeedbackVaryings defines the list of transform feedback varyings to use
         * @returns the new webGL program
         */
        createRawShaderProgram(pipelineContext: IPipelineContext, vertexCode: string, fragmentCode: string, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        /**
         * Creates a webGL program
         * @param pipelineContext  defines the pipeline context to attach to
         * @param vertexCode  defines the vertex shader code to use
         * @param fragmentCode defines the fragment shader code to use
         * @param defines defines the string containing the defines to use to compile the shaders
         * @param context defines the webGL context to use (if not set, the current one will be used)
         * @param transformFeedbackVaryings defines the list of transform feedback varyings to use
         * @returns the new webGL program
         */
        createShaderProgram(pipelineContext: IPipelineContext, vertexCode: string, fragmentCode: string, defines: Nullable<string>, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        /**
         * Inline functions in shader code that are marked to be inlined
         * @param code code to inline
         * @returns inlined code
         */
        inlineShaderCode(code: string): string;
        /**
         * Creates a new pipeline context
         * @param shaderProcessingContext defines the shader processing context used during the processing if available
         * @returns the new pipeline
         */
        createPipelineContext(shaderProcessingContext: Nullable<ShaderProcessingContext>): IPipelineContext;
        /**
         * Creates a new material context
         * @returns the new context
         */
        createMaterialContext(): IMaterialContext | undefined;
        /**
         * Creates a new draw context
         * @returns the new context
         */
        createDrawContext(): IDrawContext | undefined;
        protected _createShaderProgram(pipelineContext: WebGLPipelineContext, vertexShader: WebGLShader, fragmentShader: WebGLShader, context: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        protected _finalizePipelineContext(pipelineContext: WebGLPipelineContext): void;
        /**
         * @internal
         */
        _preparePipelineContext(pipelineContext: IPipelineContext, vertexSourceCode: string, fragmentSourceCode: string, createAsRaw: boolean, rawVertexSourceCode: string, rawFragmentSourceCode: string, rebuildRebind: any, defines: Nullable<string>, transformFeedbackVaryings: Nullable<string[]>, key: string): void;
        /**
         * @internal
         */
        _isRenderingStateCompiled(pipelineContext: IPipelineContext): boolean;
        /**
         * @internal
         */
        _executeWhenRenderingStateIsCompiled(pipelineContext: IPipelineContext, action: () => void): void;
        /**
         * Gets the list of webGL uniform locations associated with a specific program based on a list of uniform names
         * @param pipelineContext defines the pipeline context to use
         * @param uniformsNames defines the list of uniform names
         * @returns an array of webGL uniform locations
         */
        getUniforms(pipelineContext: IPipelineContext, uniformsNames: string[]): Nullable<WebGLUniformLocation>[];
        /**
         * Gets the list of active attributes for a given webGL program
         * @param pipelineContext defines the pipeline context to use
         * @param attributesNames defines the list of attribute names to get
         * @returns an array of indices indicating the offset of each attribute
         */
        getAttributes(pipelineContext: IPipelineContext, attributesNames: string[]): number[];
        /**
         * Activates an effect, making it the current one (ie. the one used for rendering)
         * @param effect defines the effect to activate
         */
        enableEffect(effect: Nullable<Effect | DrawWrapper>): void;
        /**
         * Set the value of an uniform to a number (int)
         * @param uniform defines the webGL uniform location where to store the value
         * @param value defines the int number to store
         * @returns true if the value was set
         */
        setInt(uniform: Nullable<WebGLUniformLocation>, value: number): boolean;
        /**
         * Set the value of an uniform to a int2
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @returns true if the value was set
         */
        setInt2(uniform: Nullable<WebGLUniformLocation>, x: number, y: number): boolean;
        /**
         * Set the value of an uniform to a int3
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @returns true if the value was set
         */
        setInt3(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number): boolean;
        /**
         * Set the value of an uniform to a int4
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @param w defines the 4th component of the value
         * @returns true if the value was set
         */
        setInt4(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number, w: number): boolean;
        /**
         * Set the value of an uniform to an array of int32
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if the value was set
         */
        setIntArray(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if the value was set
         */
        setIntArray2(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if the value was set
         */
        setIntArray3(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to an array of int32 (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of int32 to store
         * @returns true if the value was set
         */
        setIntArray4(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): boolean;
        /**
         * Set the value of an uniform to a number (unsigned int)
         * @param uniform defines the webGL uniform location where to store the value
         * @param value defines the unsigned int number to store
         * @returns true if the value was set
         */
        setUInt(uniform: Nullable<WebGLUniformLocation>, value: number): boolean;
        /**
         * Set the value of an uniform to a unsigned int2
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @returns true if the value was set
         */
        setUInt2(uniform: Nullable<WebGLUniformLocation>, x: number, y: number): boolean;
        /**
         * Set the value of an uniform to a unsigned int3
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @returns true if the value was set
         */
        setUInt3(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number): boolean;
        /**
         * Set the value of an uniform to a unsigned int4
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @param w defines the 4th component of the value
         * @returns true if the value was set
         */
        setUInt4(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number, w: number): boolean;
        /**
         * Set the value of an uniform to an array of unsigned int32
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of unsigned int32 to store
         * @returns true if the value was set
         */
        setUIntArray(uniform: Nullable<WebGLUniformLocation>, array: Uint32Array): boolean;
        /**
         * Set the value of an uniform to an array of unsigned int32 (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of unsigned int32 to store
         * @returns true if the value was set
         */
        setUIntArray2(uniform: Nullable<WebGLUniformLocation>, array: Uint32Array): boolean;
        /**
         * Set the value of an uniform to an array of unsigned int32 (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of unsigned int32 to store
         * @returns true if the value was set
         */
        setUIntArray3(uniform: Nullable<WebGLUniformLocation>, array: Uint32Array): boolean;
        /**
         * Set the value of an uniform to an array of unsigned int32 (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of unsigned int32 to store
         * @returns true if the value was set
         */
        setUIntArray4(uniform: Nullable<WebGLUniformLocation>, array: Uint32Array): boolean;
        /**
         * Set the value of an uniform to an array of number
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if the value was set
         */
        setArray(uniform: Nullable<WebGLUniformLocation>, array: number[] | Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of number (stored as vec2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if the value was set
         */
        setArray2(uniform: Nullable<WebGLUniformLocation>, array: number[] | Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of number (stored as vec3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if the value was set
         */
        setArray3(uniform: Nullable<WebGLUniformLocation>, array: number[] | Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of number (stored as vec4)
         * @param uniform defines the webGL uniform location where to store the value
         * @param array defines the array of number to store
         * @returns true if the value was set
         */
        setArray4(uniform: Nullable<WebGLUniformLocation>, array: number[] | Float32Array): boolean;
        /**
         * Set the value of an uniform to an array of float32 (stored as matrices)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrices defines the array of float32 to store
         * @returns true if the value was set
         */
        setMatrices(uniform: Nullable<WebGLUniformLocation>, matrices: Float32Array): boolean;
        /**
         * Set the value of an uniform to a matrix (3x3)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the Float32Array representing the 3x3 matrix to store
         * @returns true if the value was set
         */
        setMatrix3x3(uniform: Nullable<WebGLUniformLocation>, matrix: Float32Array): boolean;
        /**
         * Set the value of an uniform to a matrix (2x2)
         * @param uniform defines the webGL uniform location where to store the value
         * @param matrix defines the Float32Array representing the 2x2 matrix to store
         * @returns true if the value was set
         */
        setMatrix2x2(uniform: Nullable<WebGLUniformLocation>, matrix: Float32Array): boolean;
        /**
         * Set the value of an uniform to a number (float)
         * @param uniform defines the webGL uniform location where to store the value
         * @param value defines the float number to store
         * @returns true if the value was transferred
         */
        setFloat(uniform: Nullable<WebGLUniformLocation>, value: number): boolean;
        /**
         * Set the value of an uniform to a vec2
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @returns true if the value was set
         */
        setFloat2(uniform: Nullable<WebGLUniformLocation>, x: number, y: number): boolean;
        /**
         * Set the value of an uniform to a vec3
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @returns true if the value was set
         */
        setFloat3(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number): boolean;
        /**
         * Set the value of an uniform to a vec4
         * @param uniform defines the webGL uniform location where to store the value
         * @param x defines the 1st component of the value
         * @param y defines the 2nd component of the value
         * @param z defines the 3rd component of the value
         * @param w defines the 4th component of the value
         * @returns true if the value was set
         */
        setFloat4(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number, w: number): boolean;
        /**
         * Apply all cached states (depth, culling, stencil and alpha)
         */
        applyStates(): void;
        /**
         * Enable or disable color writing
         * @param enable defines the state to set
         */
        setColorWrite(enable: boolean): void;
        /**
         * Gets a boolean indicating if color writing is enabled
         * @returns the current color writing state
         */
        getColorWrite(): boolean;
        /**
         * Gets the depth culling state manager
         */
        get depthCullingState(): DepthCullingState;
        /**
         * Gets the alpha state manager
         */
        get alphaState(): AlphaState;
        /**
         * Gets the stencil state manager
         */
        get stencilState(): StencilState;
        /**
         * Gets the stencil state composer
         */
        get stencilStateComposer(): StencilStateComposer;
        /**
         * Clears the list of texture accessible through engine.
         * This can help preventing texture load conflict due to name collision.
         */
        clearInternalTexturesCache(): void;
        /**
         * Force the entire cache to be cleared
         * You should not have to use this function unless your engine needs to share the webGL context with another engine
         * @param bruteForce defines a boolean to force clearing ALL caches (including stencil, detoh and alpha states)
         */
        wipeCaches(bruteForce?: boolean): void;
        /**
         * @internal
         */
        _getSamplingParameters(samplingMode: number, generateMipMaps: boolean): {
            min: number;
            mag: number;
        };
        /** @internal */
        protected _createTexture(): WebGLTexture;
        /** @internal */
        _createHardwareTexture(): HardwareTextureWrapper;
        /**
         * Creates an internal texture without binding it to a framebuffer
         * @internal
         * @param size defines the size of the texture
         * @param options defines the options used to create the texture
         * @param delayGPUTextureCreation true to delay the texture creation the first time it is really needed. false to create it right away
         * @param source source type of the texture
         * @returns a new internal texture
         */
        _createInternalTexture(size: TextureSize, options: boolean | InternalTextureCreationOptions, delayGPUTextureCreation?: boolean, source?: InternalTextureSource): InternalTexture;
        /**
         * @internal
         */
        _getUseSRGBBuffer(useSRGBBuffer: boolean, noMipmap: boolean): boolean;
        protected _createTextureBase(url: Nullable<string>, noMipmap: boolean, invertY: boolean, scene: Nullable<ISceneLike>, samplingMode: number | undefined, onLoad: Nullable<(texture: InternalTexture) => void> | undefined, onError: Nullable<(message: string, exception: any) => void> | undefined, prepareTexture: (texture: InternalTexture, extension: string, scene: Nullable<ISceneLike>, img: HTMLImageElement | ImageBitmap | {
            width: number;
            height: number;
        }, invertY: boolean, noMipmap: boolean, isCompressed: boolean, processFunction: (width: number, height: number, img: HTMLImageElement | ImageBitmap | {
            width: number;
            height: number;
        }, extension: string, texture: InternalTexture, continuationCallback: () => void) => boolean, samplingMode: number) => void, prepareTextureProcessFunction: (width: number, height: number, img: HTMLImageElement | ImageBitmap | {
            width: number;
            height: number;
        }, extension: string, texture: InternalTexture, continuationCallback: () => void) => boolean, buffer?: Nullable<string | ArrayBuffer | ArrayBufferView | HTMLImageElement | Blob | ImageBitmap>, fallback?: Nullable<InternalTexture>, format?: Nullable<number>, forcedExtension?: Nullable<string>, mimeType?: string, loaderOptions?: any, useSRGBBuffer?: boolean): InternalTexture;
        /**
         * Usually called from Texture.ts.
         * Passed information to create a WebGLTexture
         * @param url defines a value which contains one of the following:
         * * A conventional http URL, e.g. 'http://...' or 'file://...'
         * * A base64 string of in-line texture data, e.g. 'data:image/jpg;base64,/...'
         * * An indicator that data being passed using the buffer parameter, e.g. 'data:mytexture.jpg'
         * @param noMipmap defines a boolean indicating that no mipmaps shall be generated.  Ignored for compressed textures.  They must be in the file
         * @param invertY when true, image is flipped when loaded.  You probably want true. Certain compressed textures may invert this if their default is inverted (eg. ktx)
         * @param scene needed for loading to the correct scene
         * @param samplingMode mode with should be used sample / access the texture (Default: Texture.TRILINEAR_SAMPLINGMODE)
         * @param onLoad optional callback to be called upon successful completion
         * @param onError optional callback to be called upon failure
         * @param buffer a source of a file previously fetched as either a base64 string, an ArrayBuffer (compressed or image format), HTMLImageElement (image format), or a Blob
         * @param fallback an internal argument in case the function must be called again, due to etc1 not having alpha capabilities
         * @param format internal format.  Default: RGB when extension is '.jpg' else RGBA.  Ignored for compressed textures
         * @param forcedExtension defines the extension to use to pick the right loader
         * @param mimeType defines an optional mime type
         * @param loaderOptions options to be passed to the loader
         * @param creationFlags specific flags to use when creating the texture (Constants.TEXTURE_CREATIONFLAG_STORAGE for storage textures, for eg)
         * @param useSRGBBuffer defines if the texture must be loaded in a sRGB GPU buffer (if supported by the GPU).
         * @returns a InternalTexture for assignment back into BABYLON.Texture
         */
        createTexture(url: Nullable<string>, noMipmap: boolean, invertY: boolean, scene: Nullable<ISceneLike>, samplingMode?: number, onLoad?: Nullable<(texture: InternalTexture) => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<string | ArrayBuffer | ArrayBufferView | HTMLImageElement | Blob | ImageBitmap>, fallback?: Nullable<InternalTexture>, format?: Nullable<number>, forcedExtension?: Nullable<string>, mimeType?: string, loaderOptions?: any, creationFlags?: number, useSRGBBuffer?: boolean): InternalTexture;
        /**
         * Calls to the GL texImage2D and texImage3D functions require three arguments describing the pixel format of the texture.
         * createTexture derives these from the babylonFormat and useSRGBBuffer arguments and also the file extension of the URL it's working with.
         * This function encapsulates that derivation for easy unit testing.
         * @param babylonFormat Babylon's format enum, as specified in ITextureCreationOptions.
         * @param fileExtension The file extension including the dot, e.g. .jpg.
         * @param useSRGBBuffer Use SRGB not linear.
         * @returns The options to pass to texImage2D or texImage3D calls.
         * @internal
         */
        _getTexImageParametersForCreateTexture(babylonFormat: Nullable<number>, fileExtension: string, useSRGBBuffer: boolean): TexImageParameters;
        /**
         * Loads an image as an HTMLImageElement.
         * @param input url string, ArrayBuffer, or Blob to load
         * @param onLoad callback called when the image successfully loads
         * @param onError callback called when the image fails to load
         * @param offlineProvider offline provider for caching
         * @param mimeType optional mime type
         * @param imageBitmapOptions optional the options to use when creating an ImageBitmap
         * @returns the HTMLImageElement of the loaded image
         * @internal
         */
        static _FileToolsLoadImage(input: string | ArrayBuffer | ArrayBufferView | Blob, onLoad: (img: HTMLImageElement | ImageBitmap) => void, onError: (message?: string, exception?: any) => void, offlineProvider: Nullable<IOfflineProvider>, mimeType?: string, imageBitmapOptions?: ImageBitmapOptions): Nullable<HTMLImageElement>;
        /**
         * @internal
         */
        _rescaleTexture(source: InternalTexture, destination: InternalTexture, scene: Nullable<any>, internalFormat: number, onComplete: () => void): void;
        private _unpackFlipYCached;
        /**
         * In case you are sharing the context with other applications, it might
         * be interested to not cache the unpack flip y state to ensure a consistent
         * value would be set.
         */
        enableUnpackFlipYCached: boolean;
        /**
         * @internal
         */
        _unpackFlipY(value: boolean): void;
        /** @internal */
        _getUnpackAlignement(): number;
        private _getTextureTarget;
        /**
         * Update the sampling mode of a given texture
         * @param samplingMode defines the required sampling mode
         * @param texture defines the texture to update
         * @param generateMipMaps defines whether to generate mipmaps for the texture
         */
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture, generateMipMaps?: boolean): void;
        /**
         * Update the dimensions of a texture
         * @param texture texture to update
         * @param width new width of the texture
         * @param height new height of the texture
         * @param depth new depth of the texture
         */
        updateTextureDimensions(texture: InternalTexture, width: number, height: number, depth?: number): void;
        /**
         * Update the sampling mode of a given texture
         * @param texture defines the texture to update
         * @param wrapU defines the texture wrap mode of the u coordinates
         * @param wrapV defines the texture wrap mode of the v coordinates
         * @param wrapR defines the texture wrap mode of the r coordinates
         */
        updateTextureWrappingMode(texture: InternalTexture, wrapU: Nullable<number>, wrapV?: Nullable<number>, wrapR?: Nullable<number>): void;
        /**
         * @internal
         */
        _setupDepthStencilTexture(internalTexture: InternalTexture, size: number | {
            width: number;
            height: number;
            layers?: number;
        }, generateStencil: boolean, bilinearFiltering: boolean, comparisonFunction: number, samples?: number): void;
        /**
         * @internal
         */
        _uploadCompressedDataToTextureDirectly(texture: InternalTexture, internalFormat: number, width: number, height: number, data: ArrayBufferView, faceIndex?: number, lod?: number): void;
        /**
         * @internal
         */
        _uploadDataToTextureDirectly(texture: InternalTexture, imageData: ArrayBufferView, faceIndex?: number, lod?: number, babylonInternalFormat?: number, useTextureWidthAndHeight?: boolean): void;
        /**
         * Update a portion of an internal texture
         * @param texture defines the texture to update
         * @param imageData defines the data to store into the texture
         * @param xOffset defines the x coordinates of the update rectangle
         * @param yOffset defines the y coordinates of the update rectangle
         * @param width defines the width of the update rectangle
         * @param height defines the height of the update rectangle
         * @param faceIndex defines the face index if texture is a cube (0 by default)
         * @param lod defines the lod level to update (0 by default)
         * @param generateMipMaps defines whether to generate mipmaps or not
         */
        updateTextureData(texture: InternalTexture, imageData: ArrayBufferView, xOffset: number, yOffset: number, width: number, height: number, faceIndex?: number, lod?: number, generateMipMaps?: boolean): void;
        /**
         * @internal
         */
        _uploadArrayBufferViewToTexture(texture: InternalTexture, imageData: ArrayBufferView, faceIndex?: number, lod?: number): void;
        protected _prepareWebGLTextureContinuation(texture: InternalTexture, scene: Nullable<ISceneLike>, noMipmap: boolean, isCompressed: boolean, samplingMode: number): void;
        private _prepareWebGLTexture;
        /**
         * @internal
         */
        _setupFramebufferDepthAttachments(generateStencilBuffer: boolean, generateDepthBuffer: boolean, width: number, height: number, samples?: number): Nullable<WebGLRenderbuffer>;
        /**
         * @internal
         */
        _createRenderBuffer(width: number, height: number, samples: number, internalFormat: number, msInternalFormat: number, attachment: number, unbindBuffer?: boolean): Nullable<WebGLRenderbuffer>;
        _updateRenderBuffer(renderBuffer: Nullable<WebGLRenderbuffer>, width: number, height: number, samples: number, internalFormat: number, msInternalFormat: number, attachment: number, unbindBuffer?: boolean): Nullable<WebGLRenderbuffer>;
        /**
         * @internal
         */
        _releaseTexture(texture: InternalTexture): void;
        /**
         * @internal
         */
        _releaseRenderTargetWrapper(rtWrapper: RenderTargetWrapper): void;
        protected _deleteTexture(texture: Nullable<WebGLTexture>): void;
        protected _setProgram(program: WebGLProgram): void;
        protected _boundUniforms: {
            [key: number]: WebGLUniformLocation;
        };
        /**
         * Binds an effect to the webGL context
         * @param effect defines the effect to bind
         */
        bindSamplers(effect: Effect): void;
        private _activateCurrentTexture;
        /**
         * @internal
         */
        _bindTextureDirectly(target: number, texture: Nullable<InternalTexture>, forTextureDataUpdate?: boolean, force?: boolean): boolean;
        /**
         * @internal
         */
        _bindTexture(channel: number, texture: Nullable<InternalTexture>, name: string): void;
        /**
         * Unbind all textures from the webGL context
         */
        unbindAllTextures(): void;
        /**
         * Sets a texture to the according uniform.
         * @param channel The texture channel
         * @param uniform The uniform to set
         * @param texture The texture to apply
         * @param name The name of the uniform in the effect
         */
        setTexture(channel: number, uniform: Nullable<WebGLUniformLocation>, texture: Nullable<ThinTexture>, name: string): void;
        private _bindSamplerUniformToChannel;
        private _getTextureWrapMode;
        protected _setTexture(channel: number, texture: Nullable<ThinTexture>, isPartOfTextureArray?: boolean, depthStencilTexture?: boolean, name?: string): boolean;
        /**
         * Sets an array of texture to the webGL context
         * @param channel defines the channel where the texture array must be set
         * @param uniform defines the associated uniform location
         * @param textures defines the array of textures to bind
         * @param name name of the channel
         */
        setTextureArray(channel: number, uniform: Nullable<WebGLUniformLocation>, textures: ThinTexture[], name: string): void;
        /**
         * @internal
         */
        _setAnisotropicLevel(target: number, internalTexture: InternalTexture, anisotropicFilteringLevel: number): void;
        private _setTextureParameterFloat;
        private _setTextureParameterInteger;
        /**
         * Unbind all vertex attributes from the webGL context
         */
        unbindAllAttributes(): void;
        /**
         * Force the engine to release all cached effects. This means that next effect compilation will have to be done completely even if a similar effect was already compiled
         */
        releaseEffects(): void;
        /**
         * Dispose and release all associated resources
         */
        dispose(): void;
        /**
         * Attach a new callback raised when context lost event is fired
         * @param callback defines the callback to call
         */
        attachContextLostEvent(callback: (event: WebGLContextEvent) => void): void;
        /**
         * Attach a new callback raised when context restored event is fired
         * @param callback defines the callback to call
         */
        attachContextRestoredEvent(callback: (event: WebGLContextEvent) => void): void;
        /**
         * Get the current error code of the webGL context
         * @returns the error code
         * @see https://developer.mozilla.org/en-US/docs/Web/API/WebGLRenderingContext/getError
         */
        getError(): number;
        private _canRenderToFloatFramebuffer;
        private _canRenderToHalfFloatFramebuffer;
        private _canRenderToFramebuffer;
        /**
         * @internal
         */
        _getWebGLTextureType(type: number): number;
        /**
         * @internal
         */
        _getInternalFormat(format: number, useSRGBBuffer?: boolean): number;
        /**
         * @internal
         */
        _getRGBABufferInternalSizedFormat(type: number, format?: number, useSRGBBuffer?: boolean): number;
        /**
         * @internal
         */
        _loadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (data: any) => void, offlineProvider?: IOfflineProvider, useArrayBuffer?: boolean, onError?: (request?: IWebRequest, exception?: any) => void): IFileRequest;
        /**
         * Loads a file from a url
         * @param url url to load
         * @param onSuccess callback called when the file successfully loads
         * @param onProgress callback called while file is loading (if the server supports this mode)
         * @param offlineProvider defines the offline provider for caching
         * @param useArrayBuffer defines a boolean indicating that date must be returned as ArrayBuffer
         * @param onError callback called when the file fails to load
         * @returns a file request object
         * @internal
         */
        static _FileToolsLoadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (ev: ProgressEvent) => void, offlineProvider?: IOfflineProvider, useArrayBuffer?: boolean, onError?: (request?: WebRequest, exception?: LoadFileError) => void): IFileRequest;
        /**
         * Reads pixels from the current frame buffer. Please note that this function can be slow
         * @param x defines the x coordinate of the rectangle where pixels must be read
         * @param y defines the y coordinate of the rectangle where pixels must be read
         * @param width defines the width of the rectangle where pixels must be read
         * @param height defines the height of the rectangle where pixels must be read
         * @param hasAlpha defines whether the output should have alpha or not (defaults to true)
         * @param flushRenderer true to flush the renderer from the pending commands before reading the pixels
         * @returns a ArrayBufferView promise (Uint8Array) containing RGBA colors
         */
        readPixels(x: number, y: number, width: number, height: number, hasAlpha?: boolean, flushRenderer?: boolean): Promise<ArrayBufferView>;
        private static _IsSupported;
        private static _HasMajorPerformanceCaveat;
        /**
         * Gets a Promise<boolean> indicating if the engine can be instantiated (ie. if a webGL context can be found)
         */
        static get IsSupportedAsync(): Promise<boolean>;
        /**
         * Gets a boolean indicating if the engine can be instantiated (ie. if a webGL context can be found)
         */
        static get IsSupported(): boolean;
        /**
         * Gets a boolean indicating if the engine can be instantiated (ie. if a webGL context can be found)
         * @returns true if the engine can be created
         * @ignorenaming
         */
        static isSupported(): boolean;
        /**
         * Gets a boolean indicating if the engine can be instantiated on a performant device (ie. if a webGL context can be found and it does not use a slow implementation)
         */
        static get HasMajorPerformanceCaveat(): boolean;
        /**
         * Find the next highest power of two.
         * @param x Number to start search from.
         * @returns Next highest power of two.
         */
        static CeilingPOT(x: number): number;
        /**
         * Find the next lowest power of two.
         * @param x Number to start search from.
         * @returns Next lowest power of two.
         */
        static FloorPOT(x: number): number;
        /**
         * Find the nearest power of two.
         * @param x Number to start search from.
         * @returns Next nearest power of two.
         */
        static NearestPOT(x: number): number;
        /**
         * Get the closest exponent of two
         * @param value defines the value to approximate
         * @param max defines the maximum value to return
         * @param mode defines how to define the closest value
         * @returns closest exponent of two of the given value
         */
        static GetExponentOfTwo(value: number, max: number, mode?: number): number;
        /**
         * Queue a new function into the requested animation frame pool (ie. this function will be executed by the browser (or the javascript engine) for the next frame)
         * @param func - the function to be called
         * @param requester - the object that will request the next frame. Falls back to window.
         * @returns frame number
         */
        static QueueNewFrame(func: () => void, requester?: any): number;
        /**
         * Gets host document
         * @returns the host document object
         */
        getHostDocument(): Nullable<Document>;
    }
    interface TexImageParameters {
        internalFormat: number;
        format: number;
        type: number;
    }


    /** @internal */
    export class WebGL2ShaderProcessor implements IShaderProcessor {
        shaderLanguage: ShaderLanguage;
        attributeProcessor(attribute: string): string;
        varyingCheck(varying: string, _isFragment: boolean): boolean;
        varyingProcessor(varying: string, isFragment: boolean): string;
        postProcessor(code: string, defines: string[], isFragment: boolean): string;
    }


    /** @internal */
    export class WebGLHardwareTexture implements HardwareTextureWrapper {
        private _webGLTexture;
        private _context;
        private _MSAARenderBuffers;
        get underlyingResource(): Nullable<WebGLTexture>;
        constructor(existingTexture: Nullable<WebGLTexture> | undefined, context: WebGLRenderingContext);
        setUsage(): void;
        set(hardwareTexture: WebGLTexture): void;
        reset(): void;
        addMSAARenderBuffer(buffer: WebGLRenderbuffer): void;
        releaseMSAARenderBuffers(): void;
        getMSAARenderBuffer(index?: number): WebGLRenderbuffer | null;
   