# Marionette.Region

Regions provide consistent methods to manage, show and close
views in your applications and layouts. They use a jQuery selector
to show your views in the correct place.

Using the `Layout` class you can create nested regions.

## Documentation Index

* [Defining An Application Region](#defining-an-application-region)
* [Initialize A Region With An `el`](#initialize-a-region-with-an-el)
* [Basic Use](#basic-use)
* [Showing a view](#showing-a-view)
* [`reset` A Region](#reset-a-region)
* [Set How View's `el` Is Attached](#set-how-views-el-is-attached)
* [Attach Existing View](#attach-existing-view)
  * [Set `currentView` On Initialization](#set-currentview-on-initialization)
  * [Call `attachView` On Region](#call-attachview-on-region)
* [Region Events And Callbacks](#region-events-and-callbacks)
  * [Events raised during `show`](#events-raised-during-show)
* [Custom Region Types](#custom-region-types)
  * [Attaching Custom Region Types](#attaching-custom-region-types)
  * [Instantiate Your Own Region](#instantiate-your-own-region)

## Defining An Application Region

You can add regions to your applications by calling the `addRegions` method on
your application instance. This method expects a single hash parameter, with
named regions and either jQuery selectors or `Region` objects. You may
call this method as many times as you like, and it will continue adding regions
to the app.

```js
MyApp.addRegions({
  mainRegion: "#main-content",
  navigationRegion: "#navigation"
});
```

As soon as you call `addRegions`, your regions are available on your
app object. In the above, example `MyApp.mainRegion` and `MyApp.navigationRegion`
would be available for use immediately.

If you specify the same region name twice, the last one in wins.

You can also add regions via `Layouts`:

```js
AppLayout = Backbone.Marionette.Layout.extend({
  template: "#layout-template",

  regions: {
    menu: "#menu",
    content: "#content"
  }
});
var layout = new AppLayout();
layout.render();
layout.menu.show(new MenuView());
layout.content.show(new MainContentView());
```


## Initialize A Region With An `el`

You can specify an `el` for the region to manage at the time
that the region is instantiated:

```js
var mgr = new Backbone.Marionette.Region({
  el: "#someElement"
});
```

## Basic Use

### Showing a View

Once a region is defined, you can call its `show`
and `close` methods to display and shut-down a view:

```js
var myView = new MyView();

// render and display the view
MyApp.mainRegion.show(myView);

// closes the current view
MyApp.mainRegion.close();
```

If you replace the current view with a new view by calling `show`,
by default it will automatically close the previous view.
You can prevent this behavior by passing `{preventClose: true}` in the options
parameter. Several events will also be triggered on the views; see
[Region Events And Callbacks](#region-events-and-callbacks) for details.

```js
// Show the first view.
var myView = new MyView();
MyApp.mainRegion.show(myView);

// Replace the view with another. The
// `close` method is called for you
var anotherView = new AnotherView();
MyApp.mainRegion.show(anotherView);

// Replace the view with another.
// Prevent `close` from being called
var anotherView2 = new AnotherView();
MyApp.mainRegion.show(anotherView2, { preventClose: true });
```

NOTE: When using `preventClose: true` you must be careful to cleanup your old views
manually to prevent memory leaks.

### `reset` A Region

A region can be `reset` at any time. This closes any existing view
being displayed, and deletes the cached `el`. The next time the
region shows a view, the region's `el` is queried from
the DOM.

```js
myRegion.reset();
```

This is useful when regions are re-used across view
instances, and in unit testing.

### Set How View's `el` Is Attached

Override the region's `open` method to change how the view is attached
to the DOM. This method receives one parameter - the view to show.

The default implementation of `open` is:

```js
Marionette.Region.prototype.open = function(view){
  this.$el.empty().append(view.el);
}
```

This replaces the contents of the region with the view's
`el` / content. You can override `open` for transition effects and more.

```js
Marionette.Region.prototype.open = function(view){
  this.$el.hide();
  this.$el.html(view.el);
  this.$el.slideDown("fast");
}
```

This example will cause a view to slide down from the top
of the region, instead of just appearing in place.

### Attach Existing View

There are some scenarios where it's desirable to attach an existing
view to a region , without rendering or showing the view, and
without replacing the HTML content of the region. For example, SEO and
accessibiliy often need HTML to be generated by the server, and progressive
enhancement of the HTML.

There are two ways to accomplish this:

* set the `currentView` in the region's constructor
* call `attachView` on the region instance

#### Set `currentView` On Initialization

```js
var myView = new MyView({
  el: $("#existing-view-stuff")
});

var region = new Backbone.Marionette.Region({
  el: "#content",
  currentView: myView
});
```

#### Call `attachView` On Region

```js
MyApp.addRegions({
  someRegion: "#content"
});

var myView = new MyView({
  el: $("#existing-view-stuff")
});

MyApp.someRegion.attachView(myView);
```

## Region Events And Callbacks

### Events raised during `show`:
A region will raise a few events when showing
and closing views:

* "before:show" / `onBeforeShow` - Called on the view instance after the view has been rendered, but before its been displayed.
* "before:show" / `onBeforeShow` - Called on the region instance after the view has been rendered, but before its been displayed.
* "show" / `onShow` - Called on the view instance when the view has been rendered and displayed.
* "show" / `onShow` - Called on the region instance when the view has been rendered and displayed.
* "close" / `onClose` - Called when the view has been closed.

These events can be used to run code when your region
opens and closes views.

```js
MyApp.mainRegion.on("before:show", function(view){
  // manipulate the `view` or do something extra
  // with the region via `this`
});

MyApp.mainRegion.on("show", function(view){
  // manipulate the `view` or do something extra
  // with the region via `this`
});

MyApp.mainRegion.on("close", function(view){
  // manipulate the `view` or do something extra
  // with the region via `this`
});

MyRegion = Backbone.Marionette.Region.extend({
  // ...

  onBeforeShow: function(view) {
    // the `view` has not been shown yet
  },

  onShow: function(view){
    // the `view` has been shown
  }
});

MyView = Marionette.ItemView.extend({
  onBeforeShow: function() {
    // called before the view has been shown
  },
  onShow: function(){
    // called when the view has been shown
  }
});
```

## Custom Region Types

You can define a custom region by extending from
`Region`. This allows you to create new functionality,
or provide a base set of functionality for your app.

### Attaching Custom Region Types

Once you define a region type, you can attach the
new region type by specifying the region type as the
value. In this case, `addRegions` expects the constructor itself, not an instance.

```js
var FooterRegion = Backbone.Marionette.Region.extend({
  el: "#footer"
});

MyApp.addRegions({
  footerRegion: FooterRegion
});
```

You can also specify a selector for the region by using
an object literal for the configuration.

```js
var FooterRegion = Backbone.Marionette.Region.extend({
  el: "#footer"
});

MyApp.addRegions({
  footerRegion: {
    selector: "#footer",
    regionType: FooterRegion
  }
});
```

Note that a region must have an element to attach itself to. If you
do not specify a selector when attaching the region instance to your
Application or Layout, the region must provide an `el` either in its
definition or constructor options.

### Instantiate Your Own Region

There may be times when you want to add a region to your
application after your app is up and running. To do this, you'll
need to extend from `Region` as shown above and then use
that constructor function on your own:

```js
var SomeRegion = Backbone.Marionette.Region.extend({
  el: "#some-div",

  initialize: function(options){
    // your init code, here
  }
});

MyApp.someRegion = new SomeRegion();

MyApp.someRegion.show(someView);
```

You can optionally add an `initialize` function to your Region
definition as shown in this example. It receives the `options`
that were passed to the constructor of the Region, similar to
a Backbone.View.
