import { isEmpty, formatDateString } from '@baloise/web-app-utils';
import capitalize from 'lodash.capitalize';
import round from 'lodash.round';
import isArray from 'lodash.isarray';
import isString from 'lodash.isstring';

/**
 * Transforms the given blob parameter to object URL string.
 *
 * For more information look up the documentation about
 * [URL.createObjectURL](https://developer.mozilla.org/en-US/docs/Web/API/URL/createObjectURL)
 */
function balBlobToUrl(value) {
    return value != null ? URL.createObjectURL(value) : '';
}

/**
 * Transforms the given string parameter to capitalize string.
 *
 * ```typescript
 * balCapitalize('baloise') // Baloise
 * ```
 */
function balCapitalize(value) {
    if (isEmpty(value)) {
        return '';
    }
    else {
        return capitalize(value);
    }
}

/**
 * Transforms the given string into the correct claim-number format.
 *
 * ```typescript
 * balClaimNumber('73001217169') // 73/001217/16.9
 * ```
 */
function balClaimNumber(value) {
    if (!value) {
        return '';
    }
    value = `${value}`;
    const parts = value.match(/^(\d{2})(\d{6})(\d{2})(\w{1})$/);
    if (!parts) {
        return value;
    }
    return `${parts[1]}/${parts[2]}/${parts[3]}.${parts[4]}`;
}

/**
 * Transforms the given string into the correct police-number format.
 *
 * ```typescript
 * balContractNumber('501222333') // 50/1.222.333
 * ```
 */
function balContractNumber(value) {
    if (!value) {
        return '';
    }
    let newValue = `${value}`.trim();
    if (newValue[0] !== '0') {
        newValue = `0${value}`;
    }
    const parts = [
        newValue.substring(1, 3),
        newValue.substring(3, 4),
        newValue.substring(4, 7),
        newValue.substring(7, 10),
    ].filter(val => val.length > 0);
    if (!parts || parts.length < 4) {
        return `${value}`;
    }
    return `${parts[0]}/${parts[1]}.${parts[2]}.${parts[3]}`;
}

/**
 * Formats the number into a human readable currency string.
 *
 * ```typescript
 * balCurrency(1234567.89) // 1'234'567.89
 * ```
 */
function balCurrency(value, currencySign = 'CHF', showZero = false, decimalLength = 2) {
    if (!value && !(showZero && value === 0)) {
        return '';
    }
    const chunkDelimiter = "'";
    const decimalDelimiter = '.';
    const chunkLength = 3;
    const result = '\\d(?=(\\d{' + chunkLength + '})+' + (decimalLength > 0 ? '\\D' : '$') + ')';
    const num = value.toFixed(Math.max(0, ~~decimalLength));
    // CharCode 160 is a non-breaking space (&nbsp;) (https://www.w3schools.com/charsets/ref_utf_latin1_supplement.asp)
    return (currencySign +
        String.fromCharCode(160) +
        (num.replace('.', decimalDelimiter) ).replace(new RegExp(result, 'g'), '$&' + chunkDelimiter));
}

/**
 * Transforms the given string parameter to capitalize string.
 *
 * ```typescript
 * balDateString(new Date(2022, 11, 31)) // '2022-12-31'
 * ```
 */
function balDateString(value) {
    if (isEmpty(value)) {
        return '';
    }
    else {
        return formatDateString(value);
    }
}

/**
 * If the value is empty it shows a dash ('-').
 *
 * ```typescript
 * balDefaultString('') // -\n
 * balDefaultString('text') // text
 * ```
 */
function balDefaultString(value, defaultString = '-') {
    return isEmpty(value) ? defaultString : value;
}

/**
 * Transforms the filesize in human readable string.
 *
 * ```typescript
 * balFileSize(86956565) // 82.9 MB
 * ```
 */
function balFileSize(value) {
    if (value < 1024) {
        return `${value} bytes`;
    }
    else if (value < 1024 * 1024) {
        return `${round(value / 1024)} kB`;
    }
    else {
        return `${round(value / (1024 * 1024), 1)} MB`;
    }
}

/**
 * Transforms the given text into a highlighted html content.
 *
 * ```typescript
 * balHighlight('Some Text') // <span class="bal-highlight">Some Text</span>
 * ```
 */
function balHighlight(value, search, cssClass = 'bal-highlight') {
    if (search && value) {
        const hrefTag = '(<(.*?)>)|(';
        let pattern = hrefTag.concat(search.replace(/[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, '\\$&')).concat(')');
        pattern = pattern
            .split(' ')
            .filter(t => {
            return t.length > 0;
        })
            .join('|');
        const regex = new RegExp(pattern, 'gi');
        return value.replace(regex, match => {
            return match.includes('<') && match.includes('>') ? match : `<span class="${cssClass}">${match}</span>`;
        });
    }
    else {
        return value;
    }
}

/**
 * Transforms the given text into a highlighted html content.
 *
 * ```typescript
 * balHighlightStartsWith('Some Text', 'Some') // <span class="bal-highlight">Some</span> Text
 * ```
 */
function balHighlightStartsWith(value, search, cssClass = 'bal-highlight') {
    if (search && value && value.startsWith(search)) {
        return `<span class="${cssClass}">${search}</span>${value.replace(search, '')}`;
    }
    else {
        return value;
    }
}

/**
 * Transforms the given string array in to a string.
 *
 * ```typescript
 * balJoinArray(['Apple', 'Potato', 'Bacon']) // Apple, Potato, Bacon
 * ```
 */
function balJoinArray(value, delimiter = ', ') {
    if (value === null || value === undefined) {
        return '';
    }
    if (isArray(value)) {
        return value.join(delimiter);
    }
    return `${value}`;
}

/**
 * Limits the input string.
 *
 * ```typescript
 * balLimit('Some string that is ways to long to be rendered') // Some string that is ...
 * ```
 */
function balLimit(value, limit = 20) {
    if (value === null || value === undefined) {
        return '';
    }
    if (value.length > limit && limit > 0) {
        const TRAIL_STRING = '...';
        return value.substring(0, limit) + TRAIL_STRING;
    }
    return value;
}

/**
 * Transforms the input string into a offer number.
 *
 * ```typescript
 * balOfferNumber('987654321') // 98/7.654.321
 * ```
 */
function balOfferNumber(value, varianteNr) {
    if (!isEmpty(value)) {
        const offertNrNoLeadingZeros = parseInt(value, 10).toString(10);
        if (offertNrNoLeadingZeros && offertNrNoLeadingZeros.length === 9) {
            return (offertNrNoLeadingZeros.substr(0, 2) +
                '/' +
                offertNrNoLeadingZeros.substr(2, 1) +
                '.' +
                offertNrNoLeadingZeros.substr(3, 3) +
                '.' +
                offertNrNoLeadingZeros.substr(6, 3) +
                (varianteNr ? ' / ' + varianteNr : ''));
        }
    }
    return '';
}

/**
 * Formats the given phone.
 *
 * ```typescript
 * balPhoneNumber('41,41564410808') // +41 56 441 08 08
 * balPhoneNumber({ countryCode: '41', phoneNumber: '564410808' }) // +41 56 441 08 08
 * ```
 */
function balPhoneNumber(value) {
    if (value === null || value === undefined) {
        return '';
    }
    let prefix;
    let number;
    if (isString(value)) {
        const split = value.split(',');
        prefix = split[0];
        number = split[1];
    }
    else {
        prefix = value.countryCode;
        number = value.phoneNumber;
    }
    if (!number || !prefix) {
        return '';
    }
    const swissPhoneNumberPattern = new RegExp(`(?:(?:|0{0,2}|\\+{0,2})${prefix}(?:|\\(0\\))|0{0,1})\\s*([1-9]\\d)\\s*(\\d{3})\\s*(\\d{2})\\s*(\\d{2})(.*)`);
    const match = number.match(swissPhoneNumberPattern);
    function removeLeadingZero(phone) {
        return phone.startsWith('0') ? phone.substring(1) : phone;
    }
    if (!match) {
        return '';
    }
    else if (match[5] !== '') {
        return `+${prefix} ${removeLeadingZero(number)}`;
    }
    else {
        return `+${prefix} ${match[1]} ${match[2]} ${match[3]} ${match[4]}`;
    }
}

export { balBlobToUrl, balCapitalize, balClaimNumber, balContractNumber, balCurrency, balDateString, balDefaultString, balFileSize, balHighlight, balHighlightStartsWith, balJoinArray, balLimit, balOfferNumber, balPhoneNumber };
//# sourceMappingURL=index.esm.js.map
