import isNil from 'lodash.isnil';
import isEqual from 'lodash.isequal';
import { isValid, formatISO, isMatch, parseISO, parse as parse$1, getYear, getMonth, getDate, setYear } from 'date-fns';
import padStart from 'lodash.padstart';
import isString from 'lodash.isstring';
import isBoolean from 'lodash.isboolean';
import isObject from 'lodash.isobject';
import isDate from 'lodash.isdate';
import isNaN from 'lodash.isnan';

/**
 * Returns `true` if the arrays are equal
 *
 * ```typescript
 * areArraysEqual(['a', 'b'], ['b', 'a']) // true
 * ```
 */
function areArraysEqual(a, b) {
    if (isNil(a) || isNil(b)) {
        return false;
    }
    const copyA = [...a].sort();
    const copyB = [...b].sort();
    return isEqual(copyA, copyB);
}

/**
 * Parses the locale to the correct display locale
 *
 * ```typescript
 * numberLocale('de-CH')
 * // 'de-CH'
 * ```
 */
function numberLocale(locale = 'de-CH') {
    const [, region] = locale.split('-');
    if (region === 'CH') {
        return 'de-CH';
    }
    if (region === 'BE') {
        return 'fr-BE';
    }
    if (region === 'LU') {
        return 'fr-LU';
    }
    return locale;
}
/**
 * Parses the locale to the correct display locale
 *
 * ```typescript
 * numberLocale('de-CH')
 * // 'fr-CH'
 * ```
 */
function dateLocale(locale = 'de-CH') {
    const [, region] = locale.split('-');
    if (region === 'CH') {
        return 'fr-CH';
    }
    if (region === 'BE') {
        return 'fr-BE';
    }
    if (region === 'LU') {
        return 'fr-LU';
    }
    return locale;
}

const ISO_PATTERN = 'yyyy-MM-dd';
const DATE_PATTERN = 'dd-MM-yyyy';
const TIMEZONE = 'T00:00:00.000Z';
/**
 * Returns a JS Date instance of the exact moment
 *
 * ```typescript
 * const date = now()
 * // Wed Mar 10 2021 20:30:32 GMT+0100 (Central European Standard Time)
 * ```
 */
function now() {
    return new Date();
}
/**
 * Returns a JS Date instance of today with time being set to 0
 *
 * ```typescript
 * const date = today()
 * // Wed Mar 10 2021 00:00:00 GMT+0100 (Central European Standard Time)
 * ```
 */
function today() {
    return floorTime(now());
}
/**
 * Returns a JS Date instance with time being set to 0
 *
 * ```typescript
 * const date = floorTime(new Date())
 * // Wed Mar 10 2021 00:00:00 GMT+0100 (Central European Standard Time)
 * ```
 */
function floorTime(date) {
    const result = new Date(date);
    result.setHours(0, 0, 0, 0);
    return result;
}
/**
 * Returns a JS Date instance with the time set to the possible end
 *
 * ```typescript
 * const date = ceilTime(new Date())
 * // Wed Mar 10 2021 23:59:59 GMT+0100 (Central European Standard Time)
 * ```
 */
function ceilTime(date) {
    const result = new Date(date);
    result.setHours(23, 59, 59, 999);
    return result;
}
/**
 * Return the formatted date string in ISO 8601 format. Options may be passed to control the parts and notations of the date.
 *
 * ```typescript
 * const dateString = formatDateString(new Date())
 * // '2022-02-14'
 * ```
 */
function formatDateString(date) {
    return date && isValid(date) ? formatISO(date, { representation: 'date' }) : '';
}
/**
 * Validates if the given date string matches the iso date format.
 *
 * ```typescript
 * isValidIsoString('2022-02-14')
 * // 'true'
 * ```
 */
function isValidIsoString(dateString) {
    return !!dateString ? isMatch(dateString, ISO_PATTERN) : false;
}
/**
 * Formats the dates according to the given locale.
 *
 * ```typescript
 * format('de-CH', new Date())
 * // '14.2.2022'
 * ```
 */
function format(locale = 'de-CH', date) {
    return isValid(date) ? intlFormat(dateLocale(locale), date) : '';
}
/**
 * Parses the iso date string into a javascript date object.
 *
 * ```typescript
 * const dateString = parse('2021-03-10')
 * // Wed Mar 10 2021 00:00:00 GMT+0100 (Central European Standard Time)
 * ```
 */
function parse(dateString, locale = 'de-CH') {
    if (isMatch(dateString, ISO_PATTERN)) {
        const d = parseISO(dateString + TIMEZONE);
        if (d && isValid(d)) {
            return validateYear(d);
        }
        const [year, month, day] = `${dateString}`.split('-').map(d => parseInt(d, 10));
        return generateIsoDate([year, month, day]);
    }
    if (isMatch(dateString, getDatePattern(locale))) {
        const d = parse$1(dateString, getDatePattern(locale), now());
        return generateIsoDate([getYear(d), getMonth(d) + 1, getDate(d)]);
    }
    return undefined;
}
/**
 * Returns the char which separates day form month and year.
 *
 * ```typescript
 * dateSeparator('de-CH')
 * // .
 * ```
 */
function dateSeparator(locale = 'de-CH') {
    return new Intl.DateTimeFormat(dateLocale(locale))
        .format(now())
        .replace(/\p{Number}/gu, '')
        .charAt(0);
}
/**************************************************************
 * PRIVATE
 **************************************************************/
function getDatePattern(locale = 'de-CH') {
    return DATE_PATTERN.split('-').join(dateSeparator(locale));
}
function intlFormat(locale = 'de-CH', date) {
    const intl = new Intl.DateTimeFormat(dateLocale(locale));
    return intl.format(date);
}
function pad(value) {
    return padStart(`${value}`, 2, '0');
}
function validateYear(date) {
    if (date && isValid(date)) {
        if (getYear(date) < 1000) {
            return setYear(date, getYear(date) + 2000);
        }
        return date;
    }
    return undefined;
}
function generateIsoDate([year, month, day]) {
    if (year > 0 && month > 0 && day > 0) {
        return parseISO(`${year < 1000 ? year + 2000 : year}-${pad(month)}-${pad(day)}` + TIMEZONE);
    }
    return undefined;
}

/**
 * The debounce function wait a certain amount of time before running the inner function again.
 * This should reduce the number of times a function is called.
 *
 * ```typescript
 * const debounceFunction = debounce(function() {
 *   // The function's code
 * }, 250);
 *
 * window.addEventListener('resize', debounceFunction);
 * ```
 */
function debounce(func, wait = 300) {
    let timeout;
    return (...args) => {
        clearTimeout(timeout);
        timeout = setTimeout(() => {
            func.apply(this, args);
        }, wait);
    };
}

/**
 * Returns `true` if the value is empty
 */
function isEmpty(value) {
    if (isNil(value)) {
        return true;
    }
    if (isString(value) && value === '') {
        return true;
    }
    if (isBoolean(value)) {
        return false;
    }
    if (isDate(value)) {
        return false;
    }
    if (Array.isArray(value) && value.length === 0) {
        return true;
    }
    if (isObject(value) && Object.keys(value).length === 0) {
        return true;
    }
    return false;
}

/**
 * Returns `true` if the keyboard event was triggered by the `Enter` key
 */
function isEnterKey(event) {
    return event.key === 'Enter';
}
/**
 * Returns `true` if the keyboard event was triggered by the `Space` key
 */
function isSpaceKey(event) {
    return event.key === ' ';
}
/**
 * Returns `true` if the keyboard event was triggered by the `Escape` key
 */
function isEscapeKey(event) {
    return event.key === 'Escape';
}
/**
 * Returns `true` if the keyboard event was triggered by the `Backspace` key
 */
function isBackspaceKey(event) {
    return event.key === 'Backspace';
}
/**
 * Returns `true` if the keyboard event was triggered by the `ArrowDown` key
 */
function isArrowDownKey(event) {
    return event.key === 'ArrowDown' || event.key === 'Down';
}
/**
 * Returns `true` if the keyboard event was triggered by the `ArrowUp` key
 */
function isArrowUpKey(event) {
    return event.key === 'ArrowUp' || event.key === 'Up';
}

/**
 * Returns `true` if the arrays are equal
 *
 * ```typescript
 * isValidMonetaryNumber(`1'000.99`) // true
 * ```
 */
function isValidMonetaryNumber(stringValue) {
    if (isNil(stringValue) && !isString(stringValue)) {
        return false;
    }
    stringValue = stringValue.replace(/'/g, '');
    stringValue = stringValue.replace(/‘/g, '');
    stringValue = stringValue.replace(/’/g, '');
    stringValue = stringValue.replace(/,/g, '.');
    return !isNaN(parseFloat(stringValue));
}
/**
 * Returns the decimal separator of the given locale
 *
 * ```typescript
 * getDecimalSeparator('de-ch') // .
 * ```
 */
function getDecimalSeparator(locale = 'de-CH') {
    return Intl.NumberFormat(numberLocale(locale))
        .format(1.1)
        .replace(/\p{Number}/gu, '');
}
/**
 * Returns the thousand separator of the given locale
 *
 * ```typescript
 * getThousandSeparator('de-ch') // '
 * ```
 */
function getThousandSeparator(locale = 'de-CH') {
    return Intl.NumberFormat(numberLocale(locale))
        .format(11111)
        .replace(/\p{Number}/gu, '');
}
/**
 * Formats the number into the given locale
 *
 * ```typescript
 * formatLocaleNumber('de-ch', 1000.42) // 1'000.42
 * ```
 */
function formatLocaleNumber(locale = 'de-CH', number, minimumFractionDigits) {
    const options = minimumFractionDigits !== undefined ? { minimumFractionDigits, maximumFractionDigits: minimumFractionDigits } : {};
    const formattedNumber = Intl.NumberFormat(numberLocale(locale), Object.assign({}, options)).format(number);
    if (formattedNumber === 'NaN') {
        return '';
    }
    return formattedNumber;
}
/**
 * Parses the locale formatted number into a native number
 *
 * ```typescript
 * parseLocaleNumber('de-ch', '1'000.42') // 1000.42
 * ```
 */
function parseLocaleNumber(locale = 'de-CH', stringNumber) {
    const thousandSeparator = getThousandSeparator(numberLocale(locale));
    const decimalSeparator = getDecimalSeparator(numberLocale(locale));
    return parseFloat(stringNumber
        .replace(new RegExp('\\' + thousandSeparator, 'g'), '')
        .replace(new RegExp('\\' + decimalSeparator), '.'));
}

var RequestState;
(function (RequestState) {
    RequestState[RequestState["INIT"] = 0] = "INIT";
    RequestState[RequestState["RUNNING"] = 1] = "RUNNING";
    RequestState[RequestState["SUCCESS"] = 2] = "SUCCESS";
    RequestState[RequestState["ERROR"] = 3] = "ERROR";
})(RequestState || (RequestState = {}));

const scrollToAnchorElement = (referenceName) => {
    const nodes = document.querySelectorAll(`[data-ref="${referenceName}"]`);
    if (nodes.length > 0) {
        nodes[0].scrollIntoView();
    }
};
const scrollToTopOfBody = () => {
    window.scrollTo(0, 0);
};
const open = (url) => {
    return openWindowSafely(url, '_self');
};
const openInNewWindow = (url) => {
    return openWindowSafely(url, '_blank');
};
const openWindowSafely = (url, target) => {
    const newWindow = window.open(url, target);
    // prevent a security hazard: the new window is able to access window.opener (e.g. replace window.opener.location)
    if (newWindow != null && newWindow.opener != null) {
        newWindow.opener = undefined;
    }
    return newWindow;
};
const getBrowserLanguage = () => {
    // e.g. de-CH
    const browserLocale = window.navigator.language ||
        // userLanguage is only for IE11
        // eslint-disable-next-line
        window.navigator['userLanguage'];
    if (browserLocale.indexOf('-') === 2) {
        return browserLocale.substr(0, 2);
    }
    else {
        return browserLocale;
    }
};
const disableDragAndDropFiles = () => {
    window.addEventListener('dragover', (e) => e.preventDefault(), false);
    window.addEventListener('drop', (e) => e.preventDefault(), false);
};
const getBrowserInfo = () => {
    return {
        cookiesEnabled: window.navigator.cookieEnabled,
        language: window.navigator.language,
        userAgent: window.navigator.userAgent,
        timezone: getBrowserTimezone(),
    };
};
function getBrowserTimezone() {
    const timezoneOffset = new Date().getTimezoneOffset();
    return timezoneOffset >= 0 ? `+${timezoneOffset}` : `-${timezoneOffset * -1}`;
}

class Language {
    constructor(key) {
        this.key = key;
    }
}

class LanguageModel {
    constructor(all, defaultLanguage) {
        this.all = all;
        this.default = defaultLanguage;
    }
    valueOf(key) {
        return this.all.find(language => language.key === key);
    }
    valueOfOrDefault(key) {
        return this.valueOf(key) || this.default;
    }
    isValidKey(key) {
        return this.valueOf(key) !== null;
    }
}

const GERMAN = new Language('de');
const FRENCH = new Language('fr');
const ITALIAN = new Language('it');
const ENGLISH = new Language('en');
const DUTCH = new Language('nl');
const LUXEMBOURGISH = new Language('lb');
const LanguagesOfSwitzerland = new LanguageModel([GERMAN, FRENCH, ITALIAN, ENGLISH], GERMAN);
const LanguagesOfGermany = new LanguageModel([GERMAN, ENGLISH], GERMAN);
const LanguagesOfBelgium = new LanguageModel([DUTCH, FRENCH, GERMAN, ENGLISH], DUTCH);
const LanguagesOfLuxembourg = new LanguageModel([LUXEMBOURGISH, FRENCH, GERMAN, ENGLISH], LUXEMBOURGISH);

const INTEGRATION_BASE_URL = 'https://www.baloise.ch';
const INTEGRATION_BASE_PATH = '/app-integration';
const getIntegrationBaseUrl = () => {
    const host = getHost();
    if (host && host.indexOf('www.baloise.ch') >= 0) {
        return `https://${host}`;
    }
    return INTEGRATION_BASE_URL;
};
const getIntegrationUrl = (lang, region = 'CH', type = IntegrationType.FOOTER, version = 'v2') => {
    let langPath = lang.key;
    if (type === IntegrationType.ONETRUST) {
        langPath = 'all';
    }
    else {
        if (region === 'DE') {
            langPath = 'de';
        }
    }
    return `${getIntegrationBaseUrl()}${INTEGRATION_BASE_PATH}/${version}/${region.toLowerCase()}/${type}/${langPath}.json`;
};
const getHost = () => { var _a; return ((_a = window === null || window === void 0 ? void 0 : window.location) === null || _a === void 0 ? void 0 : _a.host) || undefined; };
var IntegrationType;
(function (IntegrationType) {
    IntegrationType["FOOTER"] = "footer";
    IntegrationType["SOCIAL_MEDIA"] = "socialmediachannels";
    IntegrationType["ONETRUST"] = "onetrust";
})(IntegrationType || (IntegrationType = {}));

const loadFooterLinks = (lang, region = 'CH') => {
    const url = getIntegrationUrl(lang, region, IntegrationType.FOOTER);
    return fetch(url)
        .then(res => res.json())
        .then(res => res);
};

const loadSocialMediaLinks = (lang, region = 'CH') => {
    const url = getIntegrationUrl(lang, region, IntegrationType.SOCIAL_MEDIA, 'v1');
    return fetch(url)
        .then(res => res.json())
        .then(res => res);
};

const loadOneTrustBaloiseSwitzerland = (lang) => {
    return loadOnetrustDataSwitzerland().then((cmsData) => {
        const effectiveLang = LanguagesOfSwitzerland.valueOfOrDefault(lang ? lang.key : undefined);
        const oneTrustData = cmsData.find(entry => entry.lang === effectiveLang.key);
        if (oneTrustData) {
            const oneTrustScript = oneTrustData.script;
            includeScriptsFromString(oneTrustScript);
        }
        return;
    });
};
const includeScriptsFromString = (scriptAsString) => {
    const sandbox = document.implementation.createHTMLDocument();
    sandbox.body.innerHTML = scriptAsString;
    const scripts = sandbox.getElementsByTagName('script');
    for (let i = 0; i < scripts.length; i++) {
        const scriptElement = document.createElement('script');
        scriptElement.type = scripts[i].type;
        scriptElement.src = scripts[i].src;
        scriptElement.innerHTML = scripts[i].innerHTML;
        for (let j = 0; j < scripts[i].attributes.length; j++) {
            scriptElement.setAttribute(scripts[i].attributes[j].name, scripts[i].attributes[j].value);
        }
        document.getElementsByTagName('head')[0].appendChild(scriptElement);
    }
};
const loadOnetrustDataSwitzerland = (lang = GERMAN, region = 'CH') => {
    const url = getIntegrationUrl(lang, region, IntegrationType.ONETRUST);
    return fetch(url)
        .then(res => res.json())
        .then(res => res);
};

export { DATE_PATTERN, DUTCH, ENGLISH, FRENCH, GERMAN, ISO_PATTERN, ITALIAN, LUXEMBOURGISH, Language, LanguageModel, LanguagesOfBelgium, LanguagesOfGermany, LanguagesOfLuxembourg, LanguagesOfSwitzerland, RequestState, TIMEZONE, areArraysEqual, ceilTime, dateLocale, dateSeparator, debounce, disableDragAndDropFiles, floorTime, format, formatDateString, formatLocaleNumber, getBrowserInfo, getBrowserLanguage, getDecimalSeparator, getThousandSeparator, isArrowDownKey, isArrowUpKey, isBackspaceKey, isEmpty, isEnterKey, isEscapeKey, isSpaceKey, isValidIsoString, isValidMonetaryNumber, loadFooterLinks, loadOneTrustBaloiseSwitzerland, loadSocialMediaLinks, now, numberLocale, open, openInNewWindow, parse, parseLocaleNumber, scrollToAnchorElement, scrollToTopOfBody, today };
//# sourceMappingURL=index.esm.js.map
