import * as BalUtils from '@baloise/web-app-utils';
import { isEmpty, isValidMonetaryNumber } from '@baloise/web-app-utils';
import * as DateFns from 'date-fns';
import isString from 'lodash.isstring';
import _isNumber from 'lodash.isnumber';
import isNil from 'lodash.isnil';

/**
 * Returns `true` if the value date is before the given date.
 *
 * ```typescript
 * BalValidators.isCustom((value) => value > 2)(3) // true
 * ```
 */
function isCustom(validatorFn) {
    return function (value) {
        return validatorFn(value);
    };
}

/**
 * Returns `true` if the value date is before the given date
 *
 * ```typescript
 * BalValidators.isBefore('2000-01-02')('2000-01-01') // true
 * BalValidators.isBefore(new Date(2020, 0, 2))(new Date(2020, 0, 1)) // true
 * ```
 */
function isBefore(date) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        if (isString(value)) {
            value = BalUtils.parse(value);
        }
        if (isString(date)) {
            date = BalUtils.parse(date);
        }
        return DateFns.isBefore(value, date);
    };
}
/**
 * Returns `true` if the value date is before the given date
 *
 * ```typescript
 * BalValidators.isAfter('2000-01-01')('2000-01-02') // true
 * BalValidators.isAfter(new Date(2020, 0, 1))(new Date(2020, 0, 2)) // true
 * ```
 */
function isAfter(date) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        if (isString(value)) {
            value = BalUtils.parse(value);
        }
        if (isString(date)) {
            date = BalUtils.parse(date);
        }
        return DateFns.isAfter(value, date);
    };
}
/**
 * Returns `true` if the value is valid date
 *
 * ```typescript
 * BalValidators.isDate()('2000-01-02') // true
 * BalValidators.isDate()(new Date(2000, 0, 1)) // true
 * ```
 */
function isDate() {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        if (isString(value) && BalUtils.isValidIsoString(value)) {
            return true;
        }
        return DateFns.isDate(value);
    };
}

/**
 * Returns `true` if the number is bigger or equal than the min number
 *
 * ```typescript
 * BalValidators.isMin(10)(10) // true
 * BalValidators.isMin(10)(11) // true
 * BalValidators.isMin(10)(9) // false
 * ```
 */
function isMin(min) {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        const num = parseInt(value, 10);
        if (num === undefined) {
            return false;
        }
        return num >= min;
    };
}
/**
 * Returns `true` if the number is smaller or equal than the max number
 *
 * ```typescript
 * BalValidators.isMax(10)(10) // true
 * BalValidators.isMax(10)(9) // true
 * BalValidators.isMax(10)(11) // false
 * ```
 */
function isMax(max) {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        const num = parseInt(value, 10);
        if (num === undefined) {
            return false;
        }
        return num <= max;
    };
}
/**
 * Returns `true` if the number is valid
 *
 * ```typescript
 * BalValidators.isNumber()(10) // true
 * BalValidators.isNumber()('10') // true
 * BalValidators.isNumber()('a') // false
 * ```
 */
function isNumber() {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        if (typeof value === 'string' && new RegExp(/^\d*$/).test(value)) {
            return _isNumber(+value);
        }
        return _isNumber(value);
    };
}
/**
 * Returns `true` if the value is a valid formatted number
 *
 * ```typescript
 * BalValidators.isMonetaryNumber()(10) // true
 * BalValidators.isMonetaryNumber()(`1'000.99`) // true
 * BalValidators.isMonetaryNumber()(`a`) // false
 * ```
 */
function isMonetaryNumber() {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        if (_isNumber(value)) {
            return true;
        }
        return isValidMonetaryNumber(value);
    };
}

/**
 * Returns `true` if the value matches the regex
 *
 * ```typescript
 * BalValidators.matchesRegex(new RegExp('^\\d+$'))('1') // true
 * ```
 */
function matchesRegex(regex) {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        return validateRegex(regex, value);
    };
}
/**
 * Returns `true` if the value matches the regex
 *
 * ```typescript
 * BalValidators.isEmail()('peter@baloise.ch') // true
 * ```
 */
function isEmail() {
    const regex = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        return validateRegex(regex, value);
    };
}
/**
 * Returns `true` if the value matches the regex
 *
 * ```typescript
 * BalValidators.isPhone()('123 456 78 90') // true
 * ```
 */
function isPhone() {
    const regex = /^\+?[0-9 ]*$/;
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        return validateRegex(regex, value);
    };
}
function validateRegex(regex, value) {
    if (isNil(value)) {
        return false;
    }
    if (typeof value !== 'string') {
        return false;
    }
    return regex.test(value);
}

/**
 * Returns `true` if the value is a non-empty value
 *
 * ```typescript
 * BalValidators.isRequired()('foo') // true
 * BalValidators.isRequired()('') // false
 * ```
 */
function isRequired() {
    return function (value) {
        return !isEmpty(value);
    };
}
/**
 * Returns `true` if the value is true. This validator is commonly used for required checkboxes.
 *
 * ```typescript
 * BalValidators.isRequiredTrue()(true) // true
 * BalValidators.isRequiredTrue()('') // false
 * ```
 */
function isRequiredTrue() {
    return function (value) {
        return value === true;
    };
}

/**
 * Returns `true` if the string is bigger or equal than the min length
 *
 * ```typescript
 * BalValidators.isMinLength(3)('123') // true
 * BalValidators.isMinLength(3)('12') // false
 * ```
 */
function isMinLength(minLength) {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        let text = `${value}`;
        if (isNil(value)) {
            text = '';
        }
        return text.length >= minLength;
    };
}
/**
 * Returns `true` if the string is smaller or equal than the max length
 *
 * ```typescript
 * BalValidators.isMaxLength(3)('123') // true
 * BalValidators.isMaxLength(3)('1234') // false
 * ```
 */
function isMaxLength(maxLength) {
    return function (value) {
        if (isEmpty(value)) {
            return true;
        }
        let text = `${value}`;
        if (isNil(value)) {
            text = '';
        }
        return text.length <= maxLength;
    };
}

/**
 * Returns `true` if the condition is true and the validations is true too.
 *
 * ```typescript
 * BalValidators.validateConditionally((value) => value > 2, () => true)(3) // true
 * BalValidators.validateConditionally((value) => value > 2, () => false)(3) // undefined
 * ```
 */
function validateConditionally(validatorFn, conditionFn) {
    return function (value) {
        return conditionFn(value) === true ? validatorFn(value) : undefined;
    };
}

// generated file by .build/validators.index.js
const BalValidators = {
    isCustom,
    isBefore,
    isAfter,
    isDate,
    isMin,
    isMax,
    isNumber,
    isMonetaryNumber,
    matchesRegex,
    isEmail,
    isPhone,
    isRequired,
    isRequiredTrue,
    isMinLength,
    isMaxLength,
    validateConditionally,
};

export { BalValidators };
//# sourceMappingURL=index.esm.js.map
