'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const BalUtils = require('@baloise/web-app-utils');
const DateFns = require('date-fns');
const isString = require('lodash.isstring');
const _isNumber = require('lodash.isnumber');
const isNil = require('lodash.isnil');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

function _interopNamespace(e) {
    if (e && e.__esModule) return e;
    const n = Object.create(null);
    if (e) {
        for (const k in e) {
            if (k !== 'default') {
                const d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(n, k, d.get ? d : {
                    enumerable: true,
                    get: function () { return e[k]; }
                });
            }
        }
    }
    n["default"] = e;
    return Object.freeze(n);
}

const BalUtils__namespace = /*#__PURE__*/_interopNamespace(BalUtils);
const DateFns__namespace = /*#__PURE__*/_interopNamespace(DateFns);
const isString__default = /*#__PURE__*/_interopDefaultLegacy(isString);
const _isNumber__default = /*#__PURE__*/_interopDefaultLegacy(_isNumber);
const isNil__default = /*#__PURE__*/_interopDefaultLegacy(isNil);

/**
 * Returns `true` if the value date is before the given date.
 *
 * ```typescript
 * BalValidators.isCustom((value) => value > 2)(3) // true
 * ```
 */
function isCustom(validatorFn) {
    return function (value) {
        return validatorFn(value);
    };
}

/**
 * Returns `true` if the value date is before the given date
 *
 * ```typescript
 * BalValidators.isBefore('2000-01-02')('2000-01-01') // true
 * BalValidators.isBefore(new Date(2020, 0, 2))(new Date(2020, 0, 1)) // true
 * ```
 */
function isBefore(date) {
    return function (value) {
        if (BalUtils__namespace.isEmpty(value)) {
            return true;
        }
        if (isString__default["default"](value)) {
            value = BalUtils__namespace.parse(value);
        }
        if (isString__default["default"](date)) {
            date = BalUtils__namespace.parse(date);
        }
        return DateFns__namespace.isBefore(value, date);
    };
}
/**
 * Returns `true` if the value date is before the given date
 *
 * ```typescript
 * BalValidators.isAfter('2000-01-01')('2000-01-02') // true
 * BalValidators.isAfter(new Date(2020, 0, 1))(new Date(2020, 0, 2)) // true
 * ```
 */
function isAfter(date) {
    return function (value) {
        if (BalUtils__namespace.isEmpty(value)) {
            return true;
        }
        if (isString__default["default"](value)) {
            value = BalUtils__namespace.parse(value);
        }
        if (isString__default["default"](date)) {
            date = BalUtils__namespace.parse(date);
        }
        return DateFns__namespace.isAfter(value, date);
    };
}
/**
 * Returns `true` if the value is valid date
 *
 * ```typescript
 * BalValidators.isDate()('2000-01-02') // true
 * BalValidators.isDate()(new Date(2000, 0, 1)) // true
 * ```
 */
function isDate() {
    return function (value) {
        if (BalUtils__namespace.isEmpty(value)) {
            return true;
        }
        if (isString__default["default"](value) && BalUtils__namespace.isValidIsoString(value)) {
            return true;
        }
        return DateFns__namespace.isDate(value);
    };
}

/**
 * Returns `true` if the number is bigger or equal than the min number
 *
 * ```typescript
 * BalValidators.isMin(10)(10) // true
 * BalValidators.isMin(10)(11) // true
 * BalValidators.isMin(10)(9) // false
 * ```
 */
function isMin(min) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        const num = parseInt(value, 10);
        if (num === undefined) {
            return false;
        }
        return num >= min;
    };
}
/**
 * Returns `true` if the number is smaller or equal than the max number
 *
 * ```typescript
 * BalValidators.isMax(10)(10) // true
 * BalValidators.isMax(10)(9) // true
 * BalValidators.isMax(10)(11) // false
 * ```
 */
function isMax(max) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        const num = parseInt(value, 10);
        if (num === undefined) {
            return false;
        }
        return num <= max;
    };
}
/**
 * Returns `true` if the number is valid
 *
 * ```typescript
 * BalValidators.isNumber()(10) // true
 * BalValidators.isNumber()('10') // true
 * BalValidators.isNumber()('a') // false
 * ```
 */
function isNumber() {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        if (typeof value === 'string' && new RegExp(/^\d*$/).test(value)) {
            return _isNumber__default["default"](+value);
        }
        return _isNumber__default["default"](value);
    };
}
/**
 * Returns `true` if the value is a valid formatted number
 *
 * ```typescript
 * BalValidators.isMonetaryNumber()(10) // true
 * BalValidators.isMonetaryNumber()(`1'000.99`) // true
 * BalValidators.isMonetaryNumber()(`a`) // false
 * ```
 */
function isMonetaryNumber() {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        if (_isNumber__default["default"](value)) {
            return true;
        }
        return BalUtils.isValidMonetaryNumber(value);
    };
}

/**
 * Returns `true` if the value matches the regex
 *
 * ```typescript
 * BalValidators.matchesRegex(new RegExp('^\\d+$'))('1') // true
 * ```
 */
function matchesRegex(regex) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        return validateRegex(regex, value);
    };
}
/**
 * Returns `true` if the value matches the regex
 *
 * ```typescript
 * BalValidators.isEmail()('peter@baloise.ch') // true
 * ```
 */
function isEmail() {
    const regex = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        return validateRegex(regex, value);
    };
}
/**
 * Returns `true` if the value matches the regex
 *
 * ```typescript
 * BalValidators.isPhone()('123 456 78 90') // true
 * ```
 */
function isPhone() {
    const regex = /^\+?[0-9 ]*$/;
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        return validateRegex(regex, value);
    };
}
function validateRegex(regex, value) {
    if (isNil__default["default"](value)) {
        return false;
    }
    if (typeof value !== 'string') {
        return false;
    }
    return regex.test(value);
}

/**
 * Returns `true` if the value is a non-empty value
 *
 * ```typescript
 * BalValidators.isRequired()('foo') // true
 * BalValidators.isRequired()('') // false
 * ```
 */
function isRequired() {
    return function (value) {
        return !BalUtils.isEmpty(value);
    };
}
/**
 * Returns `true` if the value is true. This validator is commonly used for required checkboxes.
 *
 * ```typescript
 * BalValidators.isRequiredTrue()(true) // true
 * BalValidators.isRequiredTrue()('') // false
 * ```
 */
function isRequiredTrue() {
    return function (value) {
        return value === true;
    };
}

/**
 * Returns `true` if the string is bigger or equal than the min length
 *
 * ```typescript
 * BalValidators.isMinLength(3)('123') // true
 * BalValidators.isMinLength(3)('12') // false
 * ```
 */
function isMinLength(minLength) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        let text = `${value}`;
        if (isNil__default["default"](value)) {
            text = '';
        }
        return text.length >= minLength;
    };
}
/**
 * Returns `true` if the string is smaller or equal than the max length
 *
 * ```typescript
 * BalValidators.isMaxLength(3)('123') // true
 * BalValidators.isMaxLength(3)('1234') // false
 * ```
 */
function isMaxLength(maxLength) {
    return function (value) {
        if (BalUtils.isEmpty(value)) {
            return true;
        }
        let text = `${value}`;
        if (isNil__default["default"](value)) {
            text = '';
        }
        return text.length <= maxLength;
    };
}

/**
 * Returns `true` if the condition is true and the validations is true too.
 *
 * ```typescript
 * BalValidators.validateConditionally((value) => value > 2, () => true)(3) // true
 * BalValidators.validateConditionally((value) => value > 2, () => false)(3) // undefined
 * ```
 */
function validateConditionally(validatorFn, conditionFn) {
    return function (value) {
        return conditionFn(value) === true ? validatorFn(value) : undefined;
    };
}

// generated file by .build/validators.index.js
const BalValidators = {
    isCustom,
    isBefore,
    isAfter,
    isDate,
    isMin,
    isMax,
    isNumber,
    isMonetaryNumber,
    matchesRegex,
    isEmail,
    isPhone,
    isRequired,
    isRequiredTrue,
    isMinLength,
    isMaxLength,
    validateConditionally,
};

exports.BalValidators = BalValidators;
//# sourceMappingURL=index.js.map
