## Building functional reactive Angular apps with BangJS

create snippet/excerpt of article or other type of web page.

simple enough to oversee easily, complex/asynchronous enough to quickly end up in event spaghetti without proper care.

attempts to grab some suggested images from the page.

measure and filter out images that do not meet minimum size.

previous two steps:
  * are being shortcut it they take longer that 3 sec total;
  * always take at least 1 sec.

uploads on submit


## Getting started

All you need to get a basic BangJS app running are AngularJS (latest 1.2 or up), Bacon.js (latest 0.7 or up) and BangJS (latest 0.3 or up):

`npm install angular baconjs bangjs`

Create a file `app.js` and have it define your application module:

```js
angular.module('linkSnippetComposer', ['bang']);
```

Next throw up a simple HTML page that includes the following scripts:

```html
<script src="node_modules/angular/angular.js"></script>
<script src="node_modules/baconjs/dist/Bacon.js"></script>
<script src="node_modules/bangjs/dist/bang.js"></script>
<script src="app.js"></script>
```

Also make sure that your app is being bootstrapped by including `ng-app="linkSnippetComposer"` on the `body` or another appropriately high level element.

Now we are ready to start working on our actual link snippet composer.


## Version 1: the basics

### View

Let’s see which variables the view needs to achieve its desired behavior:

```html
<div ng-controller="compose">
  <form ng-submit="scrape.start(input.url)">
    <input type="text" ng-model="input.url"
           placeholder="Type your URL here and hit Enter" autofocus>
  </form>
  <div id="snippet" ng-show="scrape.result" ng-cloak>
    <div class="image" showcase-image="image.selected"
         ng-show="image.selected" ng-cloak>
        <button class="prev" type="button" ng-click="image.select(-1)"
                ng-show="image.selectables.length > 1" ng-cloak>&laquo;</button>
        <button class="next" type="button" ng-click="image.select(1)"
                ng-show="image.selectables.length > 1" ng-cloak>&raquo;</button>
    </div>
    <input class="title" type="text" ng-model="input.title"
           placeholder="Snippet title">
    <textarea class="description" ng-model="input.description"
              placeholder="Snippet description"></textarea>
  </div>
</div>
```

We grouped the variables in nested "namespaces". Two reasons:

* nesting all values that go into `ngModel` inside `input` keeps us clear from the dotted binding gotcha;
* it helps us keeping things organized.

Note that none of this nesting is required in any way. It is entirely up to you to decide how you want to name and organize your variables.

### Controller

Now list these variables as keys of a BangJS controller component as follows, with a short description of what each resulting observable is to represent:

```js
angular.module('linkSnippetComposer').
  controller('compose', Controller);

function Controller($scope, $http, Bacon, bang) {
  bang.component($scope, {
    scrape: {
      start: /* scrape commands */,
      result: /* scraped data */
    },
    image: {
      selected: /* currently selected snippet image */,
      selectables: /* all selectable snippet images */,
      select: /* change selected snippet image */
    },
    input: {
      title: /* current snippet title */,
      description: /* current snippet description */
      // `url` is assigned by the view but we don't care about it here so we
      // simply don't mention it.
    }
  });
}
```

For each of these keys we are going to create a _field_, which is essentially a factory for an observable. The type of field that we will be needing can be determined by answering the following three questions:

1. should the observable have a current value or should it merely be a stream of values? in other words; does it represent a state or an event.
2. how do we want it to manifest itself on the scope;
3. how should the observable behave -- when does it issue which value?

The answer to the first question determines the first element of the field definition: the type of the observable -- either a property (`Bacon.Property`) or a stream (`Bacon.EventStream`).

The answer to the second question determines the second part of the field definition: how the observable connects to and manifests itself on the scope.

The answer to the third question defines the implementation of the field's setup function. This is where dependencies on other fields may be specified. Notice that some field factory methods define some behavior out-of-the-box, making defining an explicit setup function option.

Notice how we have nicely divided the task at hand into a manageable set of smaller problems, which can be solved one by one by answering two very concrete questions.

Let's give it a shot.

#### `scrape.start`

1. we are representing an action so no current value is needed.
2. should be invokable with a URL as its single argument.
3. stream each invocation represented by the to-be-scraped URL.

```js
bang.stream.method(function (url) {
  return url;
})
```

Or simply:

```js
bang.stream.method(angular.identity)
```

#### `scrape.result`

1. latest scraped data should be available to anyone who requests it at any point in time.
2. scraped data should be made available as scope property.
3. scrape request should be sent and parsed upon every URL coming in over `scrape.start`. we are only interested in the last scrape request the came in.

```js
bang.property.digest(function () {
  return this.scrape.start.flatMapLatest(function (url) {
    return Bacon.fromPromise(
      $http.get('http://scraper.nouncy.be', { url: url })
    ).map('.metadata');
  });
})
```

#### `image.selectables`

1. represents state so maintain current value.
2. selectable images should be made available as scope property.
3. every time data is scraped, images should be extracted and represented here.

```js
bang.property.digest(function (sink) {
  this.scrape.result.onValue(function (data) {
    sink(data.image || []);
  });
})
```

#### `input.title`

1. represents current value in text input.
2. should expose a scope property that guides the value of the observable and vice versa (value of observable is reflected in scope property). a.k.a. two-way binding.
3. every time user changes value of text input _and_ very time data is scraped, title value should be updated.

```js
bang.property.watch(function () {
  return this.scrape.result.map('.title.0');
})
```

### Result

```js
angular.module('linkSnippetComposer').
  controller('compose', Controller);

function Controller($scope, $http, Bacon, bang) {
  bang.component($scope, {

    scrape: {

      start: bang.stream.method(angular.identity),

      result: bang.property.digest(function () {
        return this.scrape.start.flatMapLatest(function (url) {
          return Bacon.fromPromise(
            $http.get('http://scraper.nouncy.be', { url: url })
          ).map('.metadata');
        });
      })

    },

    image: {

      selected: bang.property.digest(function (sink, me) {
        var current;
        me.onValue(function (url) {
          current = url;
        });

        sink(undefined);
        this.image.selectables.onValue(function (urls) {
          sink(urls[0] || null);
        });

        return this.image.select.flatMapLatest(function (delta) {
          return this.image.selectables.map(function (urls) {
            var index = urls.indexOf(current) + delta;
            index %= urls.length;
            if (index < 0) index += urls.length;
            return urls[index];
          }).take(1);
        });
      }),

      selectables: bang.property.digest(function (sink) {
        this.scrape.result.onValue(function (data) {
          sink(data.image || []);
        });
      }),

      select: bang.stream.method(angular.identity)

    },

    input: {

      title: bang.property.watch(function () {
        return this.scrape.result.map('.title.0');
      }),

      description: bang.property.watch(function (sink) {
        this.scrape.result.onValue(function (data) {
          sink(data.title[0]);
        });
      })

      // `$scope.input.url` is created by the view (`ngModel` on the text input
      // to be precise), but since we do not need a property representation of
      // it here we simply don't mention it.

    }

  });
}
```

Notice how some fields are implemented in a purely FRP way by generating a observable and returning it, while others take a style more akin to traditional event handling using `onValue` and `sink`. Consider `input.title` versus `input.description`. Both styles can be used interchangeably, and can even be combined (as in `image.selected`).

For instance `scrape.result` could be implemented as follows:

```js
// use sink to implement scrape.result
```

While `image.selectables` could be implemented as such:

```js
// use FRP-based approach
```

Similarly, we can replace the somewhat verbose implementation of `image.selected` with a full-blown FRP version (_ahh that's better_):

```js
bang.property.digest(function () {
  return this.image.selectables.flatMapLatest(function (urls) {
    return this.image.select.
      scan(0, function (cursor, delta) {
        return cursor + delta;
      }).
      map(function (cursor) {
        var index = cursor % urls.length;
        if (index < 0) index += urls.length;
        return urls[index];
      });
  }.bind(this));
})
```

This is where BangJS truly shines. You get all the benefits of FRP, without _requiring_ that you use it everywhere and everytime. You can quickly prototype a field using a lot of `onValue`s and `sink`s, and then implement it more concisely in a FRP fashion at a later stage if you prefer. Or you can use `onValue` and `sink` in fields where you feel a FRP-based implementation isn't very readable. You can even have your not-so-FRP-savvy colleague work with you on an implementation without giving in on "reactiveness". Bliss!

For an exhaustive overview of all the FRP tools readily available to you check out the [Bacon.js API reference](https://baconjs.github.io/api.html).

Check out the app we just built, running live [here](http://bangjs.org/showcase/v1.html).


## Version 2: filter out invalid images

filter out invalid images


## Version 3: from good to great

finishing ux touches

* control latency: 1s < user wait time < 3s
* add progress feedback


## Alternative way of thinking about this

TODO: this is vague babble

For any piece of logic or behavior one should ask the question: is it part of the behavior of any of the observables that are used in the view? If yes, then that answers the question where it should be implemented. If no, then it is really a side-effect of one of these observables (TODO: which means that it should be implemented where?)

In principle it is never necessary to have observables that are not used in the view, but there could be reasons to introduce those anyway. These reasons tend to be similar to reasons for creating functions in non-FRP code bases:

* facilitate code reuse;
* code readability.


## Debugging

Spot errors without really looking for them. The built-in debug logs give a zero-cost instant insight in the behavior and state of your application.

Can be enabled and disabled via `$logProvider`:

```js
angular.module('linkSnippetComposer').
  config(function ($logProvider) {
    $logProvider.debugEnabled(/* boolean */);
  });
```


## Testing

TODO


## Arrows

As a side note; if instead of EcmaScript 5 JavaScript you use a language that supports arrows (such as EcmaScript 6, CoffeeScript, TypeScript), the syntax can be compressed significantly.

Take for example the FRP implementation of `image.selected` once again. In ES6 it can be formulated as follows:

```js
bang.property.digest(function () {
  return this.image.selectables.flatMapLatest(urls =>
    this.image.select.
      scan(0, (cursor, delta) => cursor + delta).
      map(cursor => {
        let index = cursor % urls.length;
        if (index < 0) index += urls.length;
        return urls[index];
      })
  );
})
```

And in CoffeeScript we can undress it even more:

```coffee
bang.property.digest ->
  @image.selectables.flatMapLatest (urls) =>
    @image.select.
      scan(0, (cursor, delta) -> cursor + delta).
      map (cursor) ->
        index = cursor % urls.length
        index += urls.length if index < 0
        urls[index]
```