## problem

the nature of user interfaces and front-end apps in general is inherently asynchronous. servers and users have one thing in common: you never know when they decide to talk to you. the typical response to this challenge: events. and more events. and even more events. you soon reach the point where you find yourself facing a giant smelly bowl of event spaghetti and the ominous sensation that it is impossible to exclude the possibility of edge case scenarios in which your entire user interface behavior falls to pieces.

imagine you are building a single page web app. the first thing you probably want to do is load some of the user data based on a user id or maybe just a session key. as long as this data is not in yet, the user interface can’t do its job and thus the user should not be allowed to interact with it. so you hide the user interface and you register a callback (a.k.a. event handler) to unhide it as soon as all data is in place. your load times are not too great but hey as long as you’re not dealing with too large amounts of data; it works.

TODO: include code sample

### example 1 

now imagine that part of the user interface is not only dependent on having user data, but also on the location hash being in a certain state. to account for this second asynchronous conditionality you would have to register a second event handler, and a slew of horrific bookkeeping follows in its footsteps:

```js
var isDataLoaded = false;
emitter.on('userDataLoaded', function () {
  isDataLoaded = true;
  refreshVisiblity();
});
var locationHashMatch = false;
emitter.on('locationHashChange', function () {
  locationHashMatch = location.hash.match(/yep/);
  refreshVisiblity();
});
function refreshVisibility() {
  if (isDataLoaded && locationHashMatch)
    ui.show();
  else
    ui.hide();
}
```

### example 2

imagine that part of your user interface requires some other piece of data to work. this data may need to be loaded from server, but it may also be already available (loaded earlier in the session, or cached in localstorage). now as soon as your user interface wants to do something with this data, it will have to explicitly support the two scenarios. the resulting code is not pretty. mutable state galore:

```js
function doSomethingWith(dataX) {
  // follow user orders
}
var loadedDataX = localstorage['loadedDataX'];
ui.on('userActsOnDataX', function () {
  if (loadedDataX)
    doSomethingWith(loadedDataX)
  else
    emitter.on('dataXLoaded', function (e, dataX) {
      emitter.off('dataXLoaded', arguments.callee);
      localstorage['loadedDataX'] = dataX;
      doSomethingWith(dataX);
    });
});
```

## solution

### asynchronous, without having to think about it

divide component logic in a set of observables. the individual observables are interconnected via FRP operators, which makes asynchronous communication a first-class citizen.

### functional reactive programming, where needed

the behavior of each individual observable on the other hand can be implemented either with pure FRP (no mutable state) or entirely imperative (local mutable state) or something in between. this freedom of style without losing any of the core benefits of an FRP-based architecture is what makes BangJS truly enjoyable to work with.

### work from the outside inward, implementation follows interface

each observable can have an equally named representation on the external interface, as either a getter property, a getter/setter property, a function or as the observable itself. an initial impression of a useful set of observables can very easily be compiled by simply listing the values that are needed in the view: _work from the outside inward_

### composable and maintainable, every part describes its own behavior

another useful rule of thumb is to have each observable define all of its own behavior and nothing but its own behavior. that way you can easily get an complete overview of how an observable (often corresponding directly to a view value) behaves and which others it depends on. this greatly improves composability and therefore maintainability.

TODO?: explain that behavior of each observable is guaranteed to be entirely described by its own implementation, which makes for a wonderful separation of concerns and an inherent scalable and maintainable code base as a result. it's a great mental model to subscribe to, as it means that whenever you wonder why a certain observable behaves the way it does, you can just look at the observable's code in isolation, and you will be certain that it can provide you the answer.