# Copyright 2017 The Bazel Authors. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""NodeJS testing

These rules let you run tests outside of a browser. This is typically faster
than launching a test in Karma, for example.
"""

load("@build_bazel_rules_nodejs//:providers.bzl", "JSNamedModuleInfo")
load("@build_bazel_rules_nodejs//internal/node:node.bzl", "nodejs_test")

def _devmode_js_sources_impl(ctx):
    depsets = []
    for src in ctx.attr.srcs:
        if JSNamedModuleInfo in src:
            depsets.append(src[JSNamedModuleInfo].sources)
        if hasattr(src, "files"):
            depsets.append(src.files)
    sources = depset(transitive = depsets)

    ctx.actions.write(ctx.outputs.manifest, "".join([
        f.short_path + "\n"
        for f in sources.to_list()
        if f.path.endswith(".js") or f.path.endswith(".mjs")
    ]))

    return [DefaultInfo(files = sources)]

"""Rule to get devmode js sources from deps.

Outputs a manifest file with the sources listed.
"""
_devmode_js_sources = rule(
    implementation = _devmode_js_sources_impl,
    attrs = {
        "srcs": attr.label_list(
            allow_files = True,
        ),
    },
    outputs = {
        "manifest": "%{name}.MF",
    },
)

def jasmine_node_test(
        name,
        srcs = [],
        data = [],
        deps = [],
        expected_exit_code = 0,
        tags = [],
        config_file = None,
        coverage = False,
        jasmine = "@npm//@bazel/jasmine",
        jasmine_entry_point = "@npm//:node_modules/@bazel/jasmine/jasmine_runner.js",
        **kwargs):
    """Runs tests in NodeJS using the Jasmine test runner.

    Detailed XML test results are found in the standard `bazel-testlogs`
    directory. This may be symlinked in your workspace.
    See https://docs.bazel.build/versions/master/output_directories.html

    To debug the test, see debugging notes in `nodejs_test`.

    Args:
      name: Name of the resulting label
      srcs: JavaScript source files containing Jasmine specs
      data: Runtime dependencies which will be loaded while the test executes
      deps: Other targets which produce JavaScript, such as ts_library
      expected_exit_code: The expected exit code for the test.
      tags: Bazel tags applied to test
      config_file: (experimental) label of a file containing Jasmine JSON config.

        Note that not all configuration options are honored, and
        we expect some strange feature interations.
        For example, the filter for which files are instrumented for
        code coverage doesn't understand the spec_files setting in the config.

        See https://jasmine.github.io/setup/nodejs.html#configuration

      coverage: Enables code coverage collection and reporting.
      jasmine: A label providing the `@bazel/jasmine` npm dependency.
      jasmine_entry_point: A label providing the `@bazel/jasmine` entry point.
      **kwargs: Remaining arguments are passed to the test rule
    """
    _devmode_js_sources(
        name = "%s_devmode_srcs" % name,
        srcs = srcs + deps,
        testonly = 1,
        tags = tags,
    )

    all_data = data + srcs + deps + [Label(jasmine)]

    all_data += [":%s_devmode_srcs.MF" % name]
    all_data += [Label("@build_bazel_rules_nodejs//third_party/github.com/bazelbuild/bazel/tools/bash/runfiles")]

    # jasmine_runner.js consumes the first 3 args.
    # The remaining target templated_args will be passed through to jasmine or
    # specs to consume.
    templated_args = [
        "$(rootpath :%s_devmode_srcs.MF)" % name,
        "--coverage" if coverage else "--nocoverage",
        "$(rootpath %s)" % config_file if config_file else "--noconfig",
    ] + kwargs.pop("templated_args", [])

    if config_file:
        # Calculate a label relative to the user's BUILD file
        pkg = Label("%s//%s:__pkg__" % (native.repository_name(), native.package_name()))
        all_data.append(pkg.relative(config_file))

    nodejs_test(
        name = name,
        data = all_data,
        entry_point = jasmine_entry_point,
        templated_args = templated_args,
        testonly = 1,
        expected_exit_code = expected_exit_code,
        tags = tags,
        **kwargs
    )
