"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var utils_1 = require("../helpers/utils");
var JsJsonNodeInfo = /** @class */ (function () {
    function JsJsonNodeInfo(ref, path) {
        this.path = [];
        this.ref = ref;
        this.path = path;
        var jsType = typeof ref;
        if (jsType === 'undefined') {
            this.type = 'undefined';
        }
        if (jsType === 'symbol') {
            this.type = 'symbol';
        }
        if (jsType === 'function') {
            this.type = 'function';
        }
        if (jsType === 'object' && ref === null) {
            this.type = 'null';
        }
        else if (jsType === 'object' && Array.isArray(ref)) {
            this.type = 'array';
        }
        else {
            this.type = jsType;
        }
        if (this.type === 'object') {
            this.length = Object.keys(ref).length;
        }
        if (this.type === 'array' || this.type === 'string') {
            this.length = ref.length;
        }
    }
    /**
     * Returns the list of keys in case of an object for the defined range
     * @param {number} start
     * @param {number} limit
     */
    JsJsonNodeInfo.prototype.getObjectKeys = function (start, limit) {
        if (start === void 0) { start = 0; }
        if (this.type !== 'object') {
            throw new Error("Unsupported method on non-object " + this.type);
        }
        utils_1.assertStartLimit(start, limit);
        var keys = Object.keys(this.ref);
        if (limit) {
            return keys.slice(start, start + limit);
        }
        return keys.slice(start);
    };
    /**
     * Return the NodeInfo at the defined position.
     * Use the index from getObjectKeys
     * @param index
     */
    JsJsonNodeInfo.prototype.getByIndex = function (index) {
        if (this.type === 'object') {
            var nodes = this.getObjectNodes(index, 1);
            if (nodes.length) {
                return nodes[0];
            }
        }
        if (this.type === 'array') {
            var nodes = this.getArrayNodes(index, 1);
            if (nodes.length) {
                return nodes[0];
            }
        }
        return undefined;
    };
    /**
     * Return the NodeInfo for the specified key
     * Use the index from getObjectKeys
     * @param key
     */
    JsJsonNodeInfo.prototype.getByKey = function (key) {
        if (this.type === 'object' && this.ref.hasOwnProperty(key)) {
            return new JsJsonNodeInfo(this.ref[key], this.path.concat([key]));
        }
        if (this.type === 'array') {
            return this.getByIndex(parseInt(key));
        }
        return undefined;
    };
    /**
     * Find the information for a given path
     * @param {string[]} path
     */
    JsJsonNodeInfo.prototype.getByPath = function (path) {
        if (!path) {
            return undefined;
        }
        if (!path.length) {
            return this;
        }
        var p = path.slice();
        var key;
        var node = this;
        while ((key = p.shift()) !== undefined && node) {
            node = node.getByKey(key);
        }
        return node;
    };
    /**
     * Returns a list with the NodeInfo objects for the defined range
     * @param {number} start
     * @param {number} limit
     */
    JsJsonNodeInfo.prototype.getObjectNodes = function (start, limit) {
        var _this = this;
        if (start === void 0) { start = 0; }
        if (this.type !== 'object') {
            throw new Error("Unsupported method on non-object " + this.type);
        }
        utils_1.assertStartLimit(start, limit);
        var nodes = {};
        return this.getObjectKeys(start, limit).map(function (key) { return new JsJsonNodeInfo(_this.ref[key], _this.path.concat([key])); });
    };
    /**
     * Returns a list of NodeInfo for the defined range
     * @param {number} start
     * @param {number} limit
     */
    JsJsonNodeInfo.prototype.getArrayNodes = function (start, limit) {
        var _this = this;
        if (start === void 0) { start = 0; }
        if (this.type !== 'array') {
            throw new Error("Unsupported method on non-array " + this.type);
        }
        utils_1.assertStartLimit(start, limit);
        var elements = limit
            ? this.ref.slice(start, start + limit)
            : this.ref.slice(start);
        return elements.map(function (ref, i) { return new JsJsonNodeInfo(ref, _this.path.concat([String(start + i)])); });
    };
    /**
     * Get the natively parsed value
     */
    JsJsonNodeInfo.prototype.getValue = function () {
        return this.ref;
    };
    return JsJsonNodeInfo;
}());
exports.JsJsonNodeInfo = JsJsonNodeInfo;
var JsParser = /** @class */ (function () {
    function JsParser(data) {
        this.data = data;
    }
    JsParser.prototype.getRootNodeInfo = function () {
        if (this.data === undefined) {
            return null;
        }
        return new JsJsonNodeInfo(this.data, []);
    };
    return JsParser;
}());
exports.JsParser = JsParser;
//# sourceMappingURL=js-parser.js.map