/**
 * @license
 * Copyright 2019 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import type { BlockSvg } from '../../block_svg.js';
import type { Connection } from '../../connection.js';
import { RenderedConnection } from '../../rendered_connection.js';
import type { BlockStyle } from '../../theme.js';
import { Coordinate } from '../../utils/coordinate.js';
import type { ConstantProvider } from './constants.js';
import type { IPathObject } from './i_path_object.js';
/**
 * An object that handles creating and setting each of the SVG elements
 * used by the renderer.
 */
export declare class PathObject implements IPathObject {
    svgRoot: SVGElement;
    svgPath: SVGElement;
    /**
     * Holds the cursors svg element when the cursor is attached to the block.
     * This is null if there is no cursor on the block.
     */
    cursorSvg: SVGElement | null;
    /**
     * Holds the markers svg element when the marker is attached to the block.
     * This is null if there is no marker on the block.
     */
    markerSvg: SVGElement | null;
    constants: ConstantProvider;
    style: BlockStyle;
    /** Highlight paths associated with connections. */
    private connectionHighlights;
    /** Locations of connection highlights. */
    private highlightOffsets;
    /**
     * @param root The root SVG element.
     * @param style The style object to use for colouring.
     * @param constants The renderer's constants.
     */
    constructor(root: SVGElement, style: BlockStyle, constants: ConstantProvider);
    /**
     * Set the path generated by the renderer onto the respective SVG element.
     *
     * @param pathString The path.
     */
    setPath(pathString: string): void;
    /**
     * Flip the SVG paths in RTL.
     */
    flipRTL(): void;
    /**
     * Add the cursor SVG to this block's SVG group.
     *
     * @param cursorSvg The SVG root of the cursor to be added to the block SVG
     *     group.
     */
    setCursorSvg(cursorSvg: SVGElement): void;
    /**
     * Add the marker SVG to this block's SVG group.
     *
     * @param markerSvg The SVG root of the marker to be added to the block SVG
     *     group.
     */
    setMarkerSvg(markerSvg: SVGElement): void;
    /**
     * Apply the stored colours to the block's path, taking into account whether
     * the paths belong to a shadow block.
     *
     * @param block The source block.
     */
    applyColour(block: BlockSvg): void;
    /**
     * Set the style.
     *
     * @param blockStyle The block style to use.
     */
    setStyle(blockStyle: BlockStyle): void;
    /**
     * Add or remove the given CSS class on the path object's root SVG element.
     *
     * @param className The name of the class to add or remove
     * @param add True if the class should be added.  False if it should be
     *     removed.
     */
    protected setClass_(className: string, add: boolean): void;
    /**
     * Set whether the block shows a highlight or not.  Block highlighting is
     * often used to visually mark blocks currently being executed.
     *
     * @param enable True if highlighted.
     */
    updateHighlighted(enable: boolean): void;
    /**
     * Updates the look of the block to reflect a shadow state.
     *
     * @param shadow True if the block is a shadow block.
     */
    protected updateShadow_(shadow: boolean): void;
    /**
     * Updates the look of the block to reflect a disabled state.
     *
     * @param disabled True if disabled.
     */
    protected updateDisabled_(disabled: boolean): void;
    /**
     * Add or remove styling showing that a block is selected.
     *
     * @param enable True if selection is enabled, false otherwise.
     */
    updateSelected(enable: boolean): void;
    /**
     * Add or remove styling showing that a block is dragged over a delete area.
     *
     * @param enable True if the block is being dragged over a delete area, false
     *     otherwise.
     */
    updateDraggingDelete(enable: boolean): void;
    /**
     * Add or remove styling showing that a block is an insertion marker.
     *
     * @param enable True if the block is an insertion marker, false otherwise.
     */
    updateInsertionMarker(enable: boolean): void;
    /**
     * Add or remove styling showing that a block is movable.
     *
     * @param enable True if the block is movable, false otherwise.
     */
    updateMovable(enable: boolean): void;
    /**
     * Add or remove styling that shows that if the dragging block is dropped,
     * this block will be replaced.  If a shadow block, it will disappear.
     * Otherwise it will bump.
     *
     * @param enable True if styling should be added.
     */
    updateReplacementFade(enable: boolean): void;
    /**
     * Add or remove styling that shows that if the dragging block is dropped,
     * this block will be connected to the input.
     *
     * @param _conn The connection on the input to highlight.
     * @param _enable True if styling should be added.
     */
    updateShapeForInputHighlight(_conn: Connection, _enable: boolean): void;
    /** Adds the given path as a connection highlight for the given connection. */
    addConnectionHighlight(connection: RenderedConnection, connectionPath: string, offset: Coordinate, rtl: boolean): void;
    private currentHighlightMatchesNew;
    /**
     * Removes any highlight associated with the given connection, if it exists.
     */
    removeConnectionHighlight(connection: RenderedConnection): void;
}
//# sourceMappingURL=path_object.d.ts.map